"""
Surviaval mini-wrapper on unisim files
"""

import os
import shutil
from string import Template
import pandas as pd
from win32com.client import GetObject

STRIPCHART_EXTRACTION_TEMPLATE = Template("""Message "/" "OpenDocument"
FileNamePrompt Yes "$case"
Message "DataLoggerManager.300/DataLogger.200($stripchart)" "View HistoryView"
Message "DataLoggerManager.300/DataLogger.200($stripchart)" "DumpToCSVFile"
FileNamePrompt Yes "$target"
""")

PROFILE_KEYS = {'gas_hol': 'HistoricalVapHoldupValues',
                'oil_hol': 'HistoricalLiq1HoldupValues',
                'wat_hol': 'HistoricalLiq2HoldupValues',
                'gas_massflow': 'HistoricalVapFlowValues',
                'oil_massflow': 'HistoricalLiq1FlowValues',
                'wat_massflow': 'HistoricalLiq2FlowValues',
                'gas_velocity': 'HistoricalVapVelocityValues',
                'oil_velocity': 'HistoricalLiq1VelocityValues',
                'wat_velocity': 'HistoricalLiq2VelocityValues',
                'flow_pattern': 'HistoricalFlowPatternValues',
                'pressure': 'HistoricalTemperatureValues',
                'temperature': 'HistoricalPressureValues'
               }

PROFILE_LENGTH_NON_ST = 'CellAccumulatedLengthString'
PROFILE_LENGTH_ST = 'CellAccumulatedLength1String'
PROFILE_TIME = 'HistoricalTimesString'


def unisim_csv_formatting(path, fname):
    """
    Remove some useless stuff from the head of a csv file generated by unisim
    and returns a pandas dataframe
    """
    with open(path+fname, 'r') as fobj:
        data = fobj.readlines()
        header = data[9].split(",")[:-1]
        unit_of_measures = data[10].split(",")[:-1]
    data = pd.read_csv(path+fname,
                       skiprows=10,
                       index_col=0,
                       usecols=(range(0, len(header))),
                       na_values=('Shutdown', 'Bad',
                                  'I/O Timeout', 'Scan Timeout'))
    data.columns = header[1:]
    data.unit = unit_of_measures[1:]
    return data


class Usc:
    """
    Survival mini-wrapper for a Unisim case
    """
    def __init__(self, fname):
        self.case = GetObject(os.path.abspath(fname))
        self.fname = self.case.FullName
        self.integrator = self.case.solver.Integrator
        self.solver = self.case.solver
        self.current_time = self.case.solver.Integrator.currenttimevalue
        self.path = self.case.path
        self.flsh = self.case.flowsheet
        self.stream_names = [self.flsh.Streams[i]() for i in self.flsh.Streams]
        self.streams = self.flsh.Streams
        self.ops = [self.flsh.Operations[i]() for i in self.flsh.Operations]
        self.stripcharts = {}
        self.profiles = {}
        self.pipes = {}

    def extract_stripchart(self, stripchart='overall', expose_data=True):
        """
        Extract a specific stripchard and exposes the data in the namespace
        """
        csv_fname = self.fname.split(os.sep)[-1].replace(".usc", ".csv")
        scp_fname = self.fname.split(os.sep)[-1].replace(".usc", ".SCP")
        case_details = {'case': self.fname.__repr__()[1:-1],
                        'stripchart': stripchart,
                        'target': self.path.__repr__()[1:-1] + csv_fname}
        script = STRIPCHART_EXTRACTION_TEMPLATE.substitute(case_details)
        with open(self.path + scp_fname, 'w') as fobj:
            fobj.write(script)
        self.case.visible = True
        self.case.application.playscript(self.path + scp_fname)
        self.case.visible = False
        os.remove(self.path + scp_fname)
        if expose_data is True:
            self.stripcharts[stripchart] = unisim_csv_formatting(self.path,
                                                                 csv_fname)
        if os.path.isdir(self.path+'trends') is not True:
            os.mkdir(self.path + 'trends')
        shutil.copy(self.path + csv_fname,
                    self.path + 'trends\\{}.csv'.format(stripchart))
        os.remove(self.path + csv_fname)

    def extract_profiles(self, pipeline_name, expose_data=True):
        """
        Extract all the profiles of a specific pipeline and exposes the data
        in the namespace
        """
        compas_pipe = self.__profile_definition(pipeline_name)
        get_variable = compas_pipe.GEtUserVariable
        if os.path.isdir(self.path+'profiles') is not True:
            os.mkdir(self.path + 'profiles')
        target_dir = self.path + 'profiles'
        if expose_data is True:
            self.profiles[pipeline_name] = {}
        for key in PROFILE_KEYS:
            pipe = self.pipes[pipeline_name]
            pipe['data'][key] = get_variable(PROFILE_KEYS[key]).Variable()
            temp = {key: val for (key, val) in enumerate(pipe['data'][key])}
            try:
                data = pd.DataFrame(temp, index=pipe['grid'])
            except ValueError:
                data = pd.DataFrame(temp, index=pipe['non_st_grid'])
            data.columns = self.pipes[pipeline_name]['timesteps']
            data.to_csv('{}/{}-{}.csv'.format(target_dir, pipeline_name, key))
            if expose_data is True:
                self.profiles[pipeline_name][key] = data

    def __profile_definition(self, pipeline_name):
        """
        Prepare the profiles extraction from a specific profile
        """
        pipe = self.flsh.Operations[pipeline_name]
        x_st = pipe.GetUserVariable(PROFILE_LENGTH_ST).Variable()
        x_non_st = pipe.GetUserVariable(PROFILE_LENGTH_NON_ST).Variable()
        timesteps = pipe.GetUserVariable(PROFILE_TIME).Variable()
        self.pipes[pipeline_name] = {'grid': x_st,
                                     'non_st_grid': x_non_st,
                                     'timesteps': timesteps,
                                     'data': {}
                                    }
        return pipe

    def run_until(self, endtime, timeunit='minutes', save=True):
        """
        Run a case untile the specifiend endtime
        """
        integrator = self.case.solver.Integrator
        integrator.rununtil(endtime, timeunit)
        if save is True:
            self.case.save()

    def save(self, fname=''):
        """
        Save the current case
        """
        if fname is '':
            self.case.save()
        else:
            self.case.SaveAs(self.path+os.sep+fname)

    def close(self):
        """
        Close the current case
        """
        self.case.close()
