# imports and constants
import os
import sys
import ctypes
import pandas as pd
import numpy as np

IO_COLNAMES = ['SUPERF. GAS VELOCITY [m/s]', 
               'SUPERF. LIQUID VEL [m/s]', 
               'WATERCUT (WC) [fraction]', 
               'DENSITY GAS [kg/m3]', 
               'DENSITY OIL [kg/m3]', 
               'DENSITY WATER [kg/m3]', 
               'VISCOSITY GAS [Ns/m2]', 
               'VISCOSITY OIL [Ns/m2]', 
               'VISCOSITY WATER [Ns/m2]', 
               'SURFACE TENSION O/G [N/m]', 
               'SURFACE TENSION W/O [N/m]', 
               'PIPE DIAMETER [m]', 
               'PIPE INCLINATION [deg]', 
               'WALL ROUGHNESS [m]', 
               'ENTRAINMENT [fraction of liquid]', 
               'PHASE INVERSION [-]', 
               'LINEAR EMULSION FACTOR  [-]', 
               'QUADRATIC EMULSION FACTOR [-]', 
               'SMOOTHING FACTOR [-]', 
               'POWER LAW / HERSCHEL-BULKLEY EXP (N)', 
               'POWER LAW CONSISTENCY', 
               'YIELD STRESS', 
               'Liquid holdup [fraction]', 
               'Frictional pressure gradient (> 0 for dp_f/dx < 0) [N/m3]', 
               'Hydrostatic head pressure gradient (>0 for dp_h/dx < 0) [N/m3]', 
               'Accelerational pressure gradient, not due to expansion effect [N/m3]', 
               'Liquid volume fraction [fraction]', 
               'Mixture velocity [m/s]', 
               'Slug nose velocity [m/s]', 
               'Film tail end velocity [m/s]', 
               'Film start velocity [m/s]', 
               'Film end gas velocity [m/s]', 
               'Film start gas velocity [m/s]', 
               'Slug liquid holdup [None]', 
               'Film tail end holdup [None]', 
               'Film start holdup [None]', 
               'Film average holdup [None]', 
               'Unit length [m]', 
               'Film length to unit length ratio [None]', 
               'Friction factor liquid phase [None]', 
               'Slug frequency [1/s]', 
               'Gas Froude number [None]', 
               'Film minimum holdup [None]', 
               'Bubble rise velocity [m/s]', 
               'Drift flux distribution parameter [None]', 
               'Liquid entrainment fraction [None]', 
               'Taylor bubble rise velocity [m/s]', 
               'Film holdup [None]', 
               'Slug flow shedding parameter [None]', 
               'Frictional pressure gradient for film [N/m3]', 
               'Frictional pressure gradient for slug [N/m3]', 
               'Mixture Froude number [None]', 
               'Relative film height [None]', 
               'Gas friction factor [None]', 
               'Interface friction factor [None]', 
               'Relative interface velocity [None]', 
               'Relative RMS interface wave amplitude [None]', 
               'Dimensionless liquid layer depth [None]', 
               'Actual (average) liquid velocity [m/s]', 
               'Actual (average) gas velocity [m/s]', 
               'Relative interface hydraulic roughness [None]', 
               'Dimensionless gas layer depth [None]', 
               'Lockhart-Martinelli X parameter [None]', 
               'Gravity Y parameter [None]', 
               'Wallis C parameter [None]', 
               'Correction factor for laminar film flow [None]', 
               'Superficial liquid Reynolds number [None]', 
               'Superficial gas Reynolds number [None]', 
               'Liquid Froude number [None]', 
               'Superficial liquid friction factor [None]', 
               'Watercut (volume fraction) [fraction]', 
               'Superficial gas friction factor [None]', 
               'Drift flux C1 parameter [None]', 
               'Zukoski surface tension group [None]', 
               'Mixture Reynolds number [None]', 
               'Collins S parameter [None]', 
               'Flow pattern [None]', 
               'Slug growth rate [None]', 
               'Water density [kg/m3]', 
               'Water viscosity [kg/ms]', 
               'Water hold-up (sum of water in oil and water film layers) [Fraction]', 
               'Oil hold-up (sum of oil in oil and water film layers) [Fraction]', 
               'Relative water film thickness [None]', 
               'Relative oil film thickness [None]', 
               'Actual water film velocity [m/s]', 
               'Actual oil film velocity [m/s]', 
               'Water friction factor [Fraction]', 
               'Oil friction factor [Fraction]', 
               'Water-oil interfacial friction factor [None]', 
               'Water film hold-up [Fraction]', 
               'Oil film hold-up [Fraction]', 
               'Version number [None]', 
               'Water surface tension [N/m]', 
               'Initial/terminal value for HLS in subroutine THREEP [None]', 
               'Initial/terminal value for HWS in subroutine THREEP [None]', 
               'Calculated liquid density [kg/me]', 
               'Calculated liquid viscosity [Ns/m2]', 
               'Water in oil holdup [Fraction]', 
               'Oil in water holdup [Fraction]', 
               'Superficial water film velocity [m/s]', 
               'Superficial water in oil velocity [m/s]', 
               'Superficial oil film velocity [m/s]', 
               'Superficial oil in water velocity [m/s]', 
               'Water film density [kg/m2]', 
               'Water film viscosity [Ns/m2]', 
               'Oil film density [kg/m3]', 
               'Oil film viscosity [Ns/m2]', 
               'Stability of water-oil interface', 
               'Superficial water velocity [m/s]', 
               'Superficial oil velocity [m/s]', 
               'Superficial liquid velocity [m/s]', 
               'Ratio of the water and liquid hold-up [Fraction]', 
               'Water velocity in the film of a slug [m/s]', 
               'Oil velocity in the film of a slug [m/s]', 
               'Water holdup in the film of a slug [Fraction]', 
               'Oil holdup in the film of a slug [Fraction]', 
               'Initial/end value for HLS in 3phase slug model [None]', 
               'Initial/end value for HWS in 3phase slug model [None]', 
               'Parameter in 3phase slug model', 
               'Fluid index N in Power Law and Herschel-Buckley models [None]', 
               'Consistency factor in Non-Newtonian models Pa [sn]', 
               'Yield stress in Bingham and Herschel-Buckley [Pa]', 
               'Wall shear stress for gas flow (FP = 1, 2, 5, 6, 11) [Pa]', 
               'Wall shear stress for liquid flow (2-phase) (FP = 1, 2, 5, 6) [Pa]', 
               'Wall shear stress for mixture (homogeneous) flow (FP = 9, 10) [Pa]', 
               'Wall shear stress for oil flow (3-phase) (FP = 11) [Pa]', 
               'Wall shear stress for slug body (FP = 6) [Pa]', 
               'Wall shear stress for water (3-phase) (FP = 11) [Pa]', 
               'Interfacial shear stress between gas and oil or gas and liquid [Pa]', 
               'Interfacial shear stress between oil and water [Pa]', 
               'Perimeter of pipe for gas flow [m]', 
               'Perimeter of pipe for liquid flow (2-phase) [m]', 
               'Perimeter of pipe for oil flow (3-phase) [m]', 
               'Perimeter of pipe for water flow (3-phase) [m]', 
               'Interfacial length for gas-liquid or gas-oil interface [m]', 
               'Interfacial length for oil-water interface [m]', 
               'Friction factor for homogeneous/bubbly flow [None]', 
               'Friction factor for slug body [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               'Not yet defined [None]', 
               ]

#---------------------------------------------------------------------------------------------------
# Interface with SFC (in the same folder)

class SFC:
    def __init__(self, version=64, dll_path=""):
        try:
            if version is 64:
                self.dll = ctypes.windll.LoadLibrary(dll_path+"./SFC_FRONT64.dll") 
            if version is 32:
                self.dll = ctypes.windll.LoadLibrary(dll_path+"./SFC_FRONT.dll") 
        except OSError:
            print("\nSFC_FRONT dll not found in {}".format(dll_path))
            sys.exit(1)
            
    def run(self, phase_number=3, **kwargs):
        """
        Input data description: FLOW PATTERN (FREE=0), SUPERF. GAS VELOCITY, [m/s],
        SUPERF. LIQUID VEL, [m/s], WATERCUT (WC), [fraction], DENSITY GAS, [kg/m3],
        DENSITY OIL, [kg/m3], DENSITY WATER, [kg/m3], VISCOSITY GAS, [Ns/m2],
        VISCOSITY OIL, [Ns/m2], VISCOSITY WATER, [Ns/m2], SURFACE TENSION O/G, [N/m],
        SURFACE TENSION W/O, [N/m], PIPE DIAMETER, [m], PIPE INCLINATION, [deg],
        WALL ROUGHNESS,[m], ENTRAINMENT, [fraction of liquid], 
        PHASE INVERSION, [-], LINEAR EMULSION FACTOR, [-],
        QUADRATIC EMULSION FACTOR, [-], SMOOTHING FACTOR, [-],
        POWER LAW / HERSCHEL-BULKLEY EXP (N), POWER LAW CONSISTENCY, YIELD STRESS
        """
        input_data = {'FP':   0,
                      'SGV':  kwargs['SGV'],
                      'SLV':  kwargs['SLV'],
                      'WC':   kwargs['WC'],
                      'RO_G': kwargs['RO_G'],
                      'RO_O': kwargs['RO_O'],
                      'RO_W': kwargs['RO_W'],
                      'MU_G': kwargs['MU_G'],
                      'MU_O': kwargs['MU_O'],
                      'MU_W': kwargs['MU_W'],
                      'S_OG': kwargs['S_OG'],
                      'S_OW': kwargs['S_OW'],
                      'ID':   kwargs['ID'],
                      'INCL': kwargs['INCL'],
                      'Ra':   kwargs['Ra'],
                      'ENTRAINMENT': 0,
                      'PHASE INVERSION': 0.67,
                      'LINEAR EMULSION FACTOR': 1,
                      'QUADRATIC EMULSION FACTOR': 1,
                      'SMOOTHING FACTOR': 1,
                      'POWER LAW / HERSCHEL-BULKLEY EXP': 0,
                      'POWER LAW CONSISTENCY': 0,
                      'YIELD STRESS': 0,}
        input_arr = [v for v in input_data.values()]
        arr = np.zeros(200)
        if phase_number is 2:
            indx = [4,2,1,1,6,0,0,1,2,0,2,0,0,0,0,0,0,1,0,0] # SFC 2-phase option
        if phase_number is 3:
            indx = [4,2,1,1,6,0,0,1,2,0,2,0,0,0,0,1,0,1,0,0] # SFC 3-phase option
        input_arr = (ctypes.c_double * len(input_arr))(*input_arr)
        arr = (ctypes.c_double * len(arr))(*arr)
        indx = (ctypes.c_short * len(indx))(*indx)
        self.dll.sfc_front(input_arr, indx, arr)
        results = pd.DataFrame(np.hstack([input_arr[1:], arr])).T
        results.columns = IO_COLNAMES
        return results
        
    def default_input(self):
        """
        It returns a dictionary with some default values for the SFC input data.
        It should be used as base to create your own input.
        """
        return {'FP':   0,
                'SGV':  1,
                'SLV':  0.1,
                'WC':   0,
                'RO_G': 50,
                'RO_O': 700,
                'RO_W': 1000,
                'MU_G': 1.845E-005,
                'MU_O': 0.001,
                'MU_W': 0.001,
                'S_OG': 0.072,
                'S_OW': 0,
                'ID':   0.1,
                'INCL': 0,
                'Ra':   1e-5,
                'ENTRAINMENT': 0,
                'PHASE INVERSION': 0.67,
                'LINEAR EMULSION FACTOR': 1,
                'QUADRATIC EMULSION FACTOR': 1,
                'SMOOTHING FACTOR': 1,
                'POWER LAW / HERSCHEL-BULKLEY EXP': 0,
                'POWER LAW CONSISTENCY': 0,
                'YIELD STRESS': 0,}
