"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const glue = require("../lib");
function createTable(props) {
    const stack = new cdk.Stack();
    new glue.Table(stack, 'table', {
        ...props,
        database: new glue.Database(stack, 'db', {
            databaseName: 'database_name',
        }),
        dataFormat: glue.DataFormat.Json
    });
}
module.exports = {
    'unpartitioned JSON table'(test) {
        const app = new cdk.App();
        const dbStack = new cdk.Stack(app, 'db');
        const database = new glue.Database(dbStack, 'Database', {
            databaseName: 'database',
        });
        const tableStack = new cdk.Stack(app, 'table');
        const table = new glue.Table(tableStack, 'Table', {
            database,
            tableName: 'table',
            columns: [{
                    name: 'col',
                    type: glue.Schema.STRING
                }],
            dataFormat: glue.DataFormat.Json,
        });
        test.equals(table.encryption, glue.TableEncryption.UNENCRYPTED);
        assert_1.expect(tableStack).to(assert_1.haveResource('AWS::S3::Bucket', {
            Type: 'AWS::S3::Bucket',
            DeletionPolicy: 'Retain',
            UpdateReplacePolicy: 'Retain'
        }, assert_1.ResourcePart.CompleteDefinition));
        assert_1.expect(tableStack).to(assert_1.haveResource('AWS::Glue::Table', {
            CatalogId: {
                Ref: 'AWS::AccountId'
            },
            DatabaseName: {
                'Fn::ImportValue': 'db:ExportsOutputRefDatabaseB269D8BB88F4B1C4'
            },
            TableInput: {
                Name: 'table',
                Description: 'table generated by CDK',
                Parameters: {
                    has_encrypted_data: false
                },
                StorageDescriptor: {
                    Columns: [
                        {
                            Name: 'col',
                            Type: 'string'
                        }
                    ],
                    Compressed: false,
                    InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                    Location: {
                        'Fn::Join': [
                            '',
                            [
                                's3://',
                                {
                                    Ref: 'TableBucketDA42407C'
                                },
                                '/data/'
                            ]
                        ]
                    },
                    OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                    SerdeInfo: {
                        SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe'
                    },
                    StoredAsSubDirectories: false
                },
                TableType: 'EXTERNAL_TABLE'
            }
        }));
        test.done();
    },
    'partitioned JSON table'(test) {
        const app = new cdk.App();
        const dbStack = new cdk.Stack(app, 'db');
        const database = new glue.Database(dbStack, 'Database', {
            databaseName: 'database',
        });
        const tableStack = new cdk.Stack(app, 'table');
        const table = new glue.Table(tableStack, 'Table', {
            database,
            tableName: 'table',
            columns: [{
                    name: 'col',
                    type: glue.Schema.STRING
                }],
            partitionKeys: [{
                    name: 'year',
                    type: glue.Schema.SMALL_INT
                }],
            dataFormat: glue.DataFormat.Json,
        });
        test.equals(table.encryption, glue.TableEncryption.UNENCRYPTED);
        test.equals(table.encryptionKey, undefined);
        test.equals(table.bucket.encryptionKey, undefined);
        assert_1.expect(tableStack).to(assert_1.haveResource('AWS::Glue::Table', {
            CatalogId: {
                Ref: 'AWS::AccountId'
            },
            DatabaseName: {
                'Fn::ImportValue': 'db:ExportsOutputRefDatabaseB269D8BB88F4B1C4'
            },
            TableInput: {
                Name: 'table',
                Description: 'table generated by CDK',
                Parameters: {
                    has_encrypted_data: false
                },
                PartitionKeys: [
                    {
                        Name: 'year',
                        Type: 'smallint'
                    }
                ],
                StorageDescriptor: {
                    Columns: [
                        {
                            Name: 'col',
                            Type: 'string'
                        }
                    ],
                    Compressed: false,
                    InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                    Location: {
                        'Fn::Join': [
                            '',
                            [
                                's3://',
                                {
                                    Ref: 'TableBucketDA42407C'
                                },
                                '/data/'
                            ]
                        ]
                    },
                    OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                    SerdeInfo: {
                        SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe'
                    },
                    StoredAsSubDirectories: false
                },
                TableType: 'EXTERNAL_TABLE'
            }
        }));
        test.done();
    },
    'compressed table'(test) {
        const stack = new cdk.Stack();
        const database = new glue.Database(stack, 'Database', {
            databaseName: 'database',
        });
        const table = new glue.Table(stack, 'Table', {
            database,
            tableName: 'table',
            columns: [{
                    name: 'col',
                    type: glue.Schema.STRING
                }],
            compressed: true,
            dataFormat: glue.DataFormat.Json,
        });
        test.equals(table.encryptionKey, undefined);
        test.equals(table.bucket.encryptionKey, undefined);
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
            CatalogId: {
                Ref: 'AWS::AccountId'
            },
            DatabaseName: {
                Ref: 'DatabaseB269D8BB'
            },
            TableInput: {
                Name: 'table',
                Description: 'table generated by CDK',
                Parameters: {
                    has_encrypted_data: false
                },
                StorageDescriptor: {
                    Columns: [
                        {
                            Name: 'col',
                            Type: 'string'
                        }
                    ],
                    Compressed: true,
                    InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                    Location: {
                        'Fn::Join': [
                            '',
                            [
                                's3://',
                                {
                                    Ref: 'TableBucketDA42407C'
                                },
                                '/data/'
                            ]
                        ]
                    },
                    OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                    SerdeInfo: {
                        SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe'
                    },
                    StoredAsSubDirectories: false
                },
                TableType: 'EXTERNAL_TABLE'
            }
        }));
        test.done();
    },
    'table.node.defaultChild'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const database = new glue.Database(stack, 'Database', {
            databaseName: 'database',
        });
        // WHEN
        const table = new glue.Table(stack, 'Table', {
            database,
            tableName: 'table',
            columns: [{
                    name: 'col',
                    type: glue.Schema.STRING
                }],
            compressed: true,
            dataFormat: glue.DataFormat.Json,
        });
        // THEN
        test.ok(table.node.defaultChild instanceof glue.CfnTable);
        test.done();
    },
    'encrypted table': {
        'SSE-S3'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                encryption: glue.TableEncryption.S3_MANAGED,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.S3_MANAGED);
            test.equals(table.encryptionKey, undefined);
            test.equals(table.bucket.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: 'AWS::AccountId'
                },
                DatabaseName: {
                    Ref: 'DatabaseB269D8BB'
                },
                TableInput: {
                    Name: 'table',
                    Description: 'table generated by CDK',
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: 'col',
                                Type: 'string'
                            }
                        ],
                        Compressed: false,
                        InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                        Location: {
                            'Fn::Join': [
                                '',
                                [
                                    's3://',
                                    {
                                        Ref: 'TableBucketDA42407C'
                                    },
                                    '/data/'
                                ]
                            ]
                        },
                        OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                        SerdeInfo: {
                            SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe'
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: 'EXTERNAL_TABLE'
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                BucketEncryption: {
                    ServerSideEncryptionConfiguration: [
                        {
                            ServerSideEncryptionByDefault: {
                                SSEAlgorithm: 'AES256'
                            }
                        }
                    ]
                }
            }));
            test.done();
        },
        'SSE-KMS (implicitly created key)'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                encryption: glue.TableEncryption.KMS,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.KMS);
            test.equals(table.encryptionKey, table.bucket.encryptionKey);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                KeyPolicy: {
                    Statement: [
                        {
                            Action: [
                                'kms:Create*',
                                'kms:Describe*',
                                'kms:Enable*',
                                'kms:List*',
                                'kms:Put*',
                                'kms:Update*',
                                'kms:Revoke*',
                                'kms:Disable*',
                                'kms:Get*',
                                'kms:Delete*',
                                'kms:ScheduleKeyDeletion',
                                'kms:CancelKeyDeletion',
                                'kms:GenerateDataKey',
                                'kms:TagResource',
                                'kms:UntagResource'
                            ],
                            Effect: 'Allow',
                            Principal: {
                                AWS: {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                Ref: 'AWS::Partition'
                                            },
                                            ':iam::',
                                            {
                                                Ref: 'AWS::AccountId'
                                            },
                                            ':root'
                                        ]
                                    ]
                                }
                            },
                            Resource: '*'
                        }
                    ],
                    Version: '2012-10-17'
                },
                Description: 'Created by Table/Bucket'
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                BucketEncryption: {
                    ServerSideEncryptionConfiguration: [
                        {
                            ServerSideEncryptionByDefault: {
                                KMSMasterKeyID: {
                                    'Fn::GetAtt': [
                                        'TableBucketKey3E9F984A',
                                        'Arn'
                                    ]
                                },
                                SSEAlgorithm: 'aws:kms'
                            }
                        }
                    ]
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: 'AWS::AccountId'
                },
                DatabaseName: {
                    Ref: 'DatabaseB269D8BB'
                },
                TableInput: {
                    Name: 'table',
                    Description: 'table generated by CDK',
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: 'col',
                                Type: 'string'
                            }
                        ],
                        Compressed: false,
                        InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                        Location: {
                            'Fn::Join': [
                                '',
                                [
                                    's3://',
                                    {
                                        Ref: 'TableBucketDA42407C'
                                    },
                                    '/data/'
                                ]
                            ]
                        },
                        OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                        SerdeInfo: {
                            SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe'
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: 'EXTERNAL_TABLE'
                }
            }));
            test.done();
        },
        'SSE-KMS (explicitly created key)'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const encryptionKey = new kms.Key(stack, 'MyKey');
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                encryption: glue.TableEncryption.KMS,
                encryptionKey,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.KMS);
            test.equals(table.encryptionKey, table.bucket.encryptionKey);
            test.notEqual(table.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                KeyPolicy: {
                    Statement: [
                        {
                            Action: [
                                'kms:Create*',
                                'kms:Describe*',
                                'kms:Enable*',
                                'kms:List*',
                                'kms:Put*',
                                'kms:Update*',
                                'kms:Revoke*',
                                'kms:Disable*',
                                'kms:Get*',
                                'kms:Delete*',
                                'kms:ScheduleKeyDeletion',
                                'kms:CancelKeyDeletion',
                                'kms:GenerateDataKey',
                                'kms:TagResource',
                                'kms:UntagResource'
                            ],
                            Effect: 'Allow',
                            Principal: {
                                AWS: {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                Ref: 'AWS::Partition'
                                            },
                                            ':iam::',
                                            {
                                                Ref: 'AWS::AccountId'
                                            },
                                            ':root'
                                        ]
                                    ]
                                }
                            },
                            Resource: '*'
                        }
                    ],
                    Version: '2012-10-17'
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                BucketEncryption: {
                    ServerSideEncryptionConfiguration: [
                        {
                            ServerSideEncryptionByDefault: {
                                KMSMasterKeyID: {
                                    'Fn::GetAtt': [
                                        'MyKey6AB29FA6',
                                        'Arn'
                                    ]
                                },
                                SSEAlgorithm: 'aws:kms'
                            }
                        }
                    ]
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: 'AWS::AccountId'
                },
                DatabaseName: {
                    Ref: 'DatabaseB269D8BB'
                },
                TableInput: {
                    Description: 'table generated by CDK',
                    Name: 'table',
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: 'col',
                                Type: 'string'
                            }
                        ],
                        Compressed: false,
                        InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                        Location: {
                            'Fn::Join': [
                                '',
                                [
                                    's3://',
                                    {
                                        Ref: 'TableBucketDA42407C'
                                    },
                                    '/data/'
                                ]
                            ]
                        },
                        OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                        SerdeInfo: {
                            SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe'
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: 'EXTERNAL_TABLE'
                }
            }));
            test.done();
        },
        'SSE-KMS_MANAGED'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                encryption: glue.TableEncryption.KMS_MANAGED,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.KMS_MANAGED);
            test.equals(table.encryptionKey, undefined);
            test.equals(table.bucket.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket', {
                BucketEncryption: {
                    ServerSideEncryptionConfiguration: [
                        {
                            ServerSideEncryptionByDefault: {
                                SSEAlgorithm: 'aws:kms'
                            }
                        }
                    ]
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: 'AWS::AccountId'
                },
                DatabaseName: {
                    Ref: 'DatabaseB269D8BB'
                },
                TableInput: {
                    Name: 'table',
                    Description: 'table generated by CDK',
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: 'col',
                                Type: 'string'
                            }
                        ],
                        Compressed: false,
                        InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                        Location: {
                            'Fn::Join': [
                                '',
                                [
                                    's3://',
                                    {
                                        Ref: 'TableBucketDA42407C'
                                    },
                                    '/data/'
                                ]
                            ]
                        },
                        OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                        SerdeInfo: {
                            SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe'
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: 'EXTERNAL_TABLE'
                }
            }));
            test.done();
        },
        'CSE-KMS (implicitly created key)'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                encryption: glue.TableEncryption.CLIENT_SIDE_KMS,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.CLIENT_SIDE_KMS);
            test.notEqual(table.encryptionKey, undefined);
            test.equals(table.bucket.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                KeyPolicy: {
                    Statement: [
                        {
                            Action: [
                                'kms:Create*',
                                'kms:Describe*',
                                'kms:Enable*',
                                'kms:List*',
                                'kms:Put*',
                                'kms:Update*',
                                'kms:Revoke*',
                                'kms:Disable*',
                                'kms:Get*',
                                'kms:Delete*',
                                'kms:ScheduleKeyDeletion',
                                'kms:CancelKeyDeletion',
                                'kms:GenerateDataKey',
                                'kms:TagResource',
                                'kms:UntagResource'
                            ],
                            Effect: 'Allow',
                            Principal: {
                                AWS: {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                Ref: 'AWS::Partition'
                                            },
                                            ':iam::',
                                            {
                                                Ref: 'AWS::AccountId'
                                            },
                                            ':root'
                                        ]
                                    ]
                                }
                            },
                            Resource: '*'
                        }
                    ],
                    Version: '2012-10-17'
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: 'AWS::AccountId'
                },
                DatabaseName: {
                    Ref: 'DatabaseB269D8BB'
                },
                TableInput: {
                    Description: 'table generated by CDK',
                    Name: 'table',
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: 'col',
                                Type: 'string'
                            }
                        ],
                        Compressed: false,
                        InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                        Location: {
                            'Fn::Join': [
                                '',
                                [
                                    's3://',
                                    {
                                        Ref: 'TableBucketDA42407C'
                                    },
                                    '/data/'
                                ]
                            ]
                        },
                        OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                        SerdeInfo: {
                            SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe'
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: 'EXTERNAL_TABLE'
                }
            }));
            test.done();
        },
        'CSE-KMS (explicitly created key)'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const encryptionKey = new kms.Key(stack, 'MyKey');
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                encryption: glue.TableEncryption.CLIENT_SIDE_KMS,
                encryptionKey,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.CLIENT_SIDE_KMS);
            test.notEqual(table.encryptionKey, undefined);
            test.equals(table.bucket.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                KeyPolicy: {
                    Statement: [
                        {
                            Action: [
                                'kms:Create*',
                                'kms:Describe*',
                                'kms:Enable*',
                                'kms:List*',
                                'kms:Put*',
                                'kms:Update*',
                                'kms:Revoke*',
                                'kms:Disable*',
                                'kms:Get*',
                                'kms:Delete*',
                                'kms:ScheduleKeyDeletion',
                                'kms:CancelKeyDeletion',
                                'kms:GenerateDataKey',
                                'kms:TagResource',
                                'kms:UntagResource'
                            ],
                            Effect: 'Allow',
                            Principal: {
                                AWS: {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                Ref: 'AWS::Partition'
                                            },
                                            ':iam::',
                                            {
                                                Ref: 'AWS::AccountId'
                                            },
                                            ':root'
                                        ]
                                    ]
                                }
                            },
                            Resource: '*'
                        }
                    ],
                    Version: '2012-10-17'
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: 'AWS::AccountId'
                },
                DatabaseName: {
                    Ref: 'DatabaseB269D8BB'
                },
                TableInput: {
                    Description: 'table generated by CDK',
                    Name: 'table',
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: 'col',
                                Type: 'string'
                            }
                        ],
                        Compressed: false,
                        InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                        Location: {
                            'Fn::Join': [
                                '',
                                [
                                    's3://',
                                    {
                                        Ref: 'TableBucketDA42407C'
                                    },
                                    '/data/'
                                ]
                            ]
                        },
                        OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                        SerdeInfo: {
                            SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe'
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: 'EXTERNAL_TABLE'
                }
            }));
            test.done();
        },
        'CSE-KMS (explicitly passed bucket and key)'(test) {
            const stack = new cdk.Stack();
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const bucket = new s3.Bucket(stack, 'Bucket');
            const encryptionKey = new kms.Key(stack, 'MyKey');
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                bucket,
                encryption: glue.TableEncryption.CLIENT_SIDE_KMS,
                encryptionKey,
                dataFormat: glue.DataFormat.Json,
            });
            test.equals(table.encryption, glue.TableEncryption.CLIENT_SIDE_KMS);
            test.notEqual(table.encryptionKey, undefined);
            test.equals(table.bucket.encryptionKey, undefined);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
                KeyPolicy: {
                    Statement: [
                        {
                            Action: [
                                'kms:Create*',
                                'kms:Describe*',
                                'kms:Enable*',
                                'kms:List*',
                                'kms:Put*',
                                'kms:Update*',
                                'kms:Revoke*',
                                'kms:Disable*',
                                'kms:Get*',
                                'kms:Delete*',
                                'kms:ScheduleKeyDeletion',
                                'kms:CancelKeyDeletion',
                                'kms:GenerateDataKey',
                                'kms:TagResource',
                                'kms:UntagResource'
                            ],
                            Effect: 'Allow',
                            Principal: {
                                AWS: {
                                    'Fn::Join': [
                                        '',
                                        [
                                            'arn:',
                                            {
                                                Ref: 'AWS::Partition'
                                            },
                                            ':iam::',
                                            {
                                                Ref: 'AWS::AccountId'
                                            },
                                            ':root'
                                        ]
                                    ]
                                }
                            },
                            Resource: '*'
                        }
                    ],
                    Version: '2012-10-17'
                }
            }));
            assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
                CatalogId: {
                    Ref: 'AWS::AccountId'
                },
                DatabaseName: {
                    Ref: 'DatabaseB269D8BB'
                },
                TableInput: {
                    Description: 'table generated by CDK',
                    Name: 'table',
                    Parameters: {
                        has_encrypted_data: true
                    },
                    StorageDescriptor: {
                        Columns: [
                            {
                                Name: 'col',
                                Type: 'string'
                            }
                        ],
                        Compressed: false,
                        InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                        Location: {
                            'Fn::Join': [
                                '',
                                [
                                    's3://',
                                    {
                                        Ref: 'Bucket83908E77'
                                    },
                                    '/data/'
                                ]
                            ]
                        },
                        OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                        SerdeInfo: {
                            SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe'
                        },
                        StoredAsSubDirectories: false
                    },
                    TableType: 'EXTERNAL_TABLE'
                }
            }));
            test.done();
        }
    },
    'explicit s3 bucket and prefix'(test) {
        const app = new cdk.App();
        const dbStack = new cdk.Stack(app, 'db');
        const stack = new cdk.Stack(app, 'app');
        const bucket = new s3.Bucket(stack, 'ExplicitBucket');
        const database = new glue.Database(dbStack, 'Database', {
            databaseName: 'database',
        });
        new glue.Table(stack, 'Table', {
            database,
            bucket,
            s3Prefix: 'prefix/',
            tableName: 'table',
            columns: [{
                    name: 'col',
                    type: glue.Schema.STRING
                }],
            dataFormat: glue.DataFormat.Json,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
            CatalogId: {
                Ref: 'AWS::AccountId'
            },
            DatabaseName: {
                'Fn::ImportValue': 'db:ExportsOutputRefDatabaseB269D8BB88F4B1C4'
            },
            TableInput: {
                Description: 'table generated by CDK',
                Name: 'table',
                Parameters: {
                    has_encrypted_data: false
                },
                StorageDescriptor: {
                    Columns: [
                        {
                            Name: 'col',
                            Type: 'string'
                        }
                    ],
                    Compressed: false,
                    InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                    Location: {
                        'Fn::Join': [
                            '',
                            [
                                's3://',
                                {
                                    Ref: 'ExplicitBucket0AA51A3F'
                                },
                                '/prefix/'
                            ]
                        ]
                    },
                    OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                    SerdeInfo: {
                        SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe'
                    },
                    StoredAsSubDirectories: false
                },
                TableType: 'EXTERNAL_TABLE'
            }
        }));
        test.done();
    },
    'explicit s3 bucket and with empty prefix'(test) {
        const app = new cdk.App();
        const dbStack = new cdk.Stack(app, 'db');
        const stack = new cdk.Stack(app, 'app');
        const bucket = new s3.Bucket(stack, 'ExplicitBucket');
        const database = new glue.Database(dbStack, 'Database', {
            databaseName: 'database',
        });
        new glue.Table(stack, 'Table', {
            database,
            bucket,
            s3Prefix: '',
            tableName: 'table',
            columns: [{
                    name: 'col',
                    type: glue.Schema.STRING
                }],
            dataFormat: glue.DataFormat.Json,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Glue::Table', {
            CatalogId: {
                Ref: 'AWS::AccountId'
            },
            DatabaseName: {
                'Fn::ImportValue': 'db:ExportsOutputRefDatabaseB269D8BB88F4B1C4'
            },
            TableInput: {
                Description: 'table generated by CDK',
                Name: 'table',
                Parameters: {
                    has_encrypted_data: false
                },
                StorageDescriptor: {
                    Columns: [
                        {
                            Name: 'col',
                            Type: 'string'
                        }
                    ],
                    Compressed: false,
                    InputFormat: 'org.apache.hadoop.mapred.TextInputFormat',
                    Location: {
                        'Fn::Join': [
                            '',
                            [
                                's3://',
                                {
                                    Ref: 'ExplicitBucket0AA51A3F'
                                },
                                '/'
                            ]
                        ]
                    },
                    OutputFormat: 'org.apache.hadoop.hive.ql.io.HiveIgnoreKeyTextOutputFormat',
                    SerdeInfo: {
                        SerializationLibrary: 'org.openx.data.jsonserde.JsonSerDe'
                    },
                    StoredAsSubDirectories: false
                },
                TableType: 'EXTERNAL_TABLE'
            }
        }));
        test.done();
    },
    'grants': {
        'read only'(test) {
            const stack = new cdk.Stack();
            const user = new iam.User(stack, 'User');
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                compressed: true,
                dataFormat: glue.DataFormat.Json,
            });
            table.grantRead(user);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'glue:BatchDeletePartition',
                                'glue:BatchGetPartition',
                                'glue:GetPartition',
                                'glue:GetPartitions',
                                'glue:GetTable',
                                'glue:GetTables',
                                'glue:GetTableVersions'
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition'
                                        },
                                        ':glue:',
                                        {
                                            Ref: 'AWS::Region'
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId'
                                        },
                                        ':table/',
                                        {
                                            Ref: 'DatabaseB269D8BB'
                                        },
                                        '/',
                                        {
                                            Ref: 'Table4C2D914F'
                                        }
                                    ]
                                ]
                            }
                        },
                        {
                            Action: [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*'
                            ],
                            Effect: 'Allow',
                            Resource: [
                                {
                                    'Fn::GetAtt': [
                                        'TableBucketDA42407C',
                                        'Arn'
                                    ]
                                },
                                {
                                    'Fn::Join': [
                                        '',
                                        [
                                            {
                                                'Fn::GetAtt': [
                                                    'TableBucketDA42407C',
                                                    'Arn'
                                                ]
                                            },
                                            '/data/'
                                        ]
                                    ]
                                }
                            ]
                        }
                    ],
                    Version: '2012-10-17'
                },
                PolicyName: 'UserDefaultPolicy1F97781E',
                Users: [
                    {
                        Ref: 'User00B015A1'
                    }
                ]
            }));
            test.done();
        },
        'write only'(test) {
            const stack = new cdk.Stack();
            const user = new iam.User(stack, 'User');
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                compressed: true,
                dataFormat: glue.DataFormat.Json,
            });
            table.grantWrite(user);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'glue:BatchCreatePartition',
                                'glue:BatchDeletePartition',
                                'glue:CreatePartition',
                                'glue:DeletePartition',
                                'glue:UpdatePartition'
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition'
                                        },
                                        ':glue:',
                                        {
                                            Ref: 'AWS::Region'
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId'
                                        },
                                        ':table/',
                                        {
                                            Ref: 'DatabaseB269D8BB'
                                        },
                                        '/',
                                        {
                                            Ref: 'Table4C2D914F'
                                        }
                                    ]
                                ]
                            }
                        },
                        {
                            Action: [
                                's3:DeleteObject*',
                                's3:PutObject*',
                                's3:Abort*'
                            ],
                            Effect: 'Allow',
                            Resource: [
                                {
                                    'Fn::GetAtt': [
                                        'TableBucketDA42407C',
                                        'Arn'
                                    ]
                                },
                                {
                                    'Fn::Join': [
                                        '',
                                        [
                                            {
                                                'Fn::GetAtt': [
                                                    'TableBucketDA42407C',
                                                    'Arn'
                                                ]
                                            },
                                            '/data/'
                                        ]
                                    ]
                                }
                            ]
                        }
                    ],
                    Version: '2012-10-17'
                },
                PolicyName: 'UserDefaultPolicy1F97781E',
                Users: [
                    {
                        Ref: 'User00B015A1'
                    }
                ]
            }));
            test.done();
        },
        'read and write'(test) {
            const stack = new cdk.Stack();
            const user = new iam.User(stack, 'User');
            const database = new glue.Database(stack, 'Database', {
                databaseName: 'database',
            });
            const table = new glue.Table(stack, 'Table', {
                database,
                tableName: 'table',
                columns: [{
                        name: 'col',
                        type: glue.Schema.STRING
                    }],
                compressed: true,
                dataFormat: glue.DataFormat.Json,
            });
            table.grantReadWrite(user);
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'glue:BatchDeletePartition',
                                'glue:BatchGetPartition',
                                'glue:GetPartition',
                                'glue:GetPartitions',
                                'glue:GetTable',
                                'glue:GetTables',
                                'glue:GetTableVersions',
                                'glue:BatchCreatePartition',
                                'glue:BatchDeletePartition',
                                'glue:CreatePartition',
                                'glue:DeletePartition',
                                'glue:UpdatePartition'
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition'
                                        },
                                        ':glue:',
                                        {
                                            Ref: 'AWS::Region'
                                        },
                                        ':',
                                        {
                                            Ref: 'AWS::AccountId'
                                        },
                                        ':table/',
                                        {
                                            Ref: 'DatabaseB269D8BB'
                                        },
                                        '/',
                                        {
                                            Ref: 'Table4C2D914F'
                                        }
                                    ]
                                ]
                            }
                        },
                        {
                            Action: [
                                's3:GetObject*',
                                's3:GetBucket*',
                                's3:List*',
                                's3:DeleteObject*',
                                's3:PutObject*',
                                's3:Abort*'
                            ],
                            Effect: 'Allow',
                            Resource: [
                                {
                                    'Fn::GetAtt': [
                                        'TableBucketDA42407C',
                                        'Arn'
                                    ]
                                },
                                {
                                    'Fn::Join': [
                                        '',
                                        [
                                            {
                                                'Fn::GetAtt': [
                                                    'TableBucketDA42407C',
                                                    'Arn'
                                                ]
                                            },
                                            '/data/'
                                        ]
                                    ]
                                }
                            ]
                        }
                    ],
                    Version: '2012-10-17'
                },
                PolicyName: 'UserDefaultPolicy1F97781E',
                Users: [
                    {
                        Ref: 'User00B015A1'
                    }
                ]
            }));
            test.done();
        }
    },
    'validate': {
        'at least one column'(test) {
            test.throws(() => {
                createTable({
                    columns: [],
                    tableName: 'name',
                });
            }, undefined, 'you must specify at least one column for the table');
            test.done();
        },
        'unique column names'(test) {
            test.throws(() => {
                createTable({
                    tableName: 'name',
                    columns: [{
                            name: 'col1',
                            type: glue.Schema.STRING
                        }, {
                            name: 'col1',
                            type: glue.Schema.STRING
                        }]
                });
            }, undefined, "column names and partition keys must be unique, but 'col1' is duplicated.");
            test.done();
        },
        'unique partition keys'(test) {
            test.throws(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.STRING
                    }],
                partitionKeys: [{
                        name: 'p1',
                        type: glue.Schema.STRING
                    }, {
                        name: 'p1',
                        type: glue.Schema.STRING
                    }]
            }), undefined, "column names and partition keys must be unique, but 'p1' is duplicated");
            test.done();
        },
        'column names and partition keys are all unique'(test) {
            test.throws(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.STRING
                    }],
                partitionKeys: [{
                        name: 'col1',
                        type: glue.Schema.STRING
                    }]
            }), "column names and partition keys must be unique, but 'col1' is duplicated");
            test.done();
        },
        'can not specify an explicit bucket and encryption'(test) {
            test.throws(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.STRING
                    }],
                bucket: new s3.Bucket(new cdk.Stack(), 'Bucket'),
                encryption: glue.TableEncryption.KMS
            }), undefined, 'you can not specify encryption settings if you also provide a bucket');
            test.done();
        },
        'can explicitly pass bucket if Encryption undefined'(test) {
            test.doesNotThrow(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.STRING
                    }],
                bucket: new s3.Bucket(new cdk.Stack(), 'Bucket'),
                encryption: undefined
            }));
            test.done();
        },
        'can explicitly pass bucket if Unencrypted'(test) {
            test.doesNotThrow(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.STRING
                    }],
                bucket: new s3.Bucket(new cdk.Stack(), 'Bucket'),
                encryption: undefined
            }));
            test.done();
        },
        'can explicitly pass bucket if ClientSideKms'(test) {
            test.doesNotThrow(() => createTable({
                tableName: 'name',
                columns: [{
                        name: 'col1',
                        type: glue.Schema.STRING
                    }],
                bucket: new s3.Bucket(new cdk.Stack(), 'Bucket'),
                encryption: glue.TableEncryption.CLIENT_SIDE_KMS
            }));
            test.done();
        }
    },
    'Table.fromTableArn'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        const table = glue.Table.fromTableArn(stack, 'boom', 'arn:aws:glue:us-east-1:123456789012:table/db1/tbl1');
        // THEN
        test.deepEqual(table.tableArn, 'arn:aws:glue:us-east-1:123456789012:table/db1/tbl1');
        test.deepEqual(table.tableName, 'tbl1');
        test.done();
    }
};
//# sourceMappingURL=data:application/json;base64,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