from tqdm import tqdm
from ..console_utils import TQDM_BAR_FORMAT, warning, error
from collections import OrderedDict
from tabulate import tabulate
import numpy as np
from time import time
from .image import Image
from pathlib import Path
from functools import partial
import multiprocessing as mp
from ..blocks.utils import DataBlock
import sys
import yaml
from ..utils import full_class_name
from ..citations import _all_citations

def progress(name, x, **kwargs):
    return tqdm(
        x,
        desc=name,
        unit="images",
        ncols=80,
        bar_format=TQDM_BAR_FORMAT,
        **kwargs
    )

class Sequence:
    def __init__(self, blocks, name=None):
        """A sequence of :py:class:`Block` objects to sequentially process images

        Parameters
        ----------
        blocks : list
            list of :py:class:`Block` objects
        name : str, optional
            name of the sequence, by default None
        """
        self.name = name
        self.images = []
        self.blocks = blocks

        self.data = {}
        self.n_processed_images = None

    def __getattr__(self, item):
        return self.blocks_dict[item]

    @property
    def blocks(self):
        """list of :py:class:`Block` objects

        Returns
        -------
        _type_
            _description_
        """
        return list(self.blocks_dict.values())

    @blocks.setter
    def blocks(self, blocks):
        self.blocks_dict = OrderedDict({
            block.name if block.name is not None else "block{}".format(i): block
            for i, block in enumerate(blocks)
        })

    def _set_blocks_in_sequence(self, in_sequence):
        for b in self.blocks:
            b.in_sequence = in_sequence

    def run(self, images, terminate=True, show_progress=True, loader=Image):
        """Run the sequence

        Parameters
        ----------
        images : list, str, :py:class:`Image`
            :py:class:`Image` object or path (single or as a list) to be processed by the sequence
        terminate : bool, optional
            whether to run :py:class:`Sequence.terminate` at the end of the sequence, by default True
        show_progress : bool, optional
            whether to show a progress bar, by default True
        loader : Image sub-class, optional
            An Image sub-class to load images path(s) of provided as inputs, by default py:class:`Image`
        """
        self._set_blocks_in_sequence(True)
        self.images = images if not isinstance(images, (str, Path, Image)) else [images]
        assert len(self.images) != 0, "Empty array or no images provided"

        if not show_progress:
            def _p(x, **kwargs): return x
            self.progress = _p
        else:
            self.progress = partial(progress, self.name)

        if isinstance(self.images, list):
            if len(self.images) == 0:
                raise ValueError("No images to process")
        elif self.images is None:
            raise ValueError("No images to process")

        # run
        self.n_processed_images = 0
        self.discards = {}
        self._run(loader=loader)

        if terminate:
            self.terminate()
        
        for block_name, discarded in self.discards.items():
            warning(f"{block_name} discarded image{'s' if len(discarded)>1 else ''} {', '.join(discarded)}")

    def _run(self, loader=Image):
        for i, image in enumerate(self.progress(self.images)):
            if isinstance(image, (str, Path)):
                image = loader(image)

            image.i = i

            for block in self.blocks:
                block._run(image)
                # This allows to discard image in any Block
                if image.discard:
                    self._add_discard(type(block).__name__, i)
                    break

            del image
            self.n_processed_images += 1

    def terminate(self):
        """Run the :py:class:`Block.terminate` method of all blocks
        """
        for block in self.blocks:
            block.terminate()
        self._set_blocks_in_sequence(False)

    def __str__(self):
        rows = [[
            i, block.name, block.__class__.__name__, f"{block.processing_time:.3f} s ({(block.processing_time/self.processing_time)*100:.0f}%)"] 
            for i, block in enumerate(self.blocks)
            ]
        headers = ["index", "name", "type", "processing"]

        return tabulate(rows, headers, tablefmt="fancy_grid")

    def __repr__(self) -> str:
        return self.__str__()

    @property
    def processing_time(self):
        """Total processing time of the sequence last run
        """
        return np.sum([block.processing_time for block in self.blocks])

    def __getitem__(self, item):
        return self.blocks[item]

    # io
    # --

    def _add_discard(self, discard_block, i):
        if discard_block not in self.discards:
            self.discards[discard_block] = []
        self.discards[discard_block].append(str(i))


    @property
    def args(self):
        blocks = []
        for block in self.blocks:        
            blocks.append({
                'block': full_class_name(block),
                **block.args
            })

        return blocks

    @classmethod
    def from_args(cls, args):
        import prose
        
        blocks = []
        for block_dict in args:
            block_class = block_dict["block"]
            del block_dict["block"]
            block = eval(block_class).from_args(block_dict)
            blocks.append(block)
            
        return cls(blocks)

    @property
    def params_str(self):
        return yaml.safe_dump(self.args, sort_keys=False)

    def citations(self):
        """Citations aggregated from all blocks

        Returns
        -------
        str
            LaTex string to cite sequence block dependencies
        str
            LaTex bibtex items

        Example
        -------
        .. jupyter-execute::

            from prose import Sequence, blocks

            sequence = Sequence([              
                blocks.psf.Moffat2D(),
                blocks.detection.LimitStars(min=3),
                blocks.Set(stars_coords=None),
                blocks.AffineTransform(data=False, inverse=True),
                blocks.BalletCentroid(),                           
                blocks.PhotutilsAperturePhotometry(scale=1.),
                blocks.Peaks(),
                blocks.XArray(
                    ("time", "jd_utc"),
                    ("time", "bjd_tdb"),
                    ("time", "flip"),
                    ("time", "fwhm"),
                    ("time", "fwhmx"),
                    name="xarray"
                ),
            ])

            tex, bib = sequence.citations()

            print(tex)
            print(bib[0:500], "...")
        """
        citations = [block.citations for block in self.blocks]
        citations = [*citations, "astropy", "prose"]
        cites = {}

        def add_citation(c):
            if isinstance(c, str):
                if c.strip("\n").strip(" ")[0] == "@":
                    name = c.split("{")[1].split(",")[0]
                    cites[name] = c
                else:
                    if c in _all_citations:
                        cites[c] = _all_citations[c]
                    else:
                        raise KeyError(f"{c} not in defautls citations, please provide it as a dict")

            elif isinstance(c, dict):
                cites.update(c)

        for c in citations:
            if isinstance(c, (list, tuple)):
                for cc in c:
                    add_citation(cc)
            else:
                add_citation(c)
                
        tex_citep = ", ".join([f"{name} \citep{{{name}}}" for name in cites.keys() if name not in ["prose", "astropy"]])
        tex_citep += " and astropy \citep{astropy}"
        tex = f"This research made use of \\textsf{{prose}} \citep{{prose}} and its dependencies ({tex_citep})."""

        return tex, "\n\n".join(cites.values())


class MPSequence(Sequence):

    def __init__(self, blocks, data_blocks=None, name="", loader=Image):
        super().__init__(blocks, name=name, loader=loader)
        if data_blocks is None:
            self.data = None
            self._has_data = False
        else:
            self.data = Sequence(data_blocks)
            self._has_data = True

    def check_data_blocks(self):
        bad_blocks = []
        for b in self.blocks: 
            if isinstance(b, DataBlock):
                bad_blocks.append(f"{b.__class__.__name__}")
        if len(bad_blocks) > 0:
            bad_blocks = ', '.join(list(np.unique(bad_blocks)))
            error(f"Data blocks [{bad_blocks}] cannot be used in MPSequence\n\nConsider using the data_blocks kwargs")
            sys.exit()
    
    def _run(self, telescope=None):
        self.check_data_blocks()

        self.n_processed_images = 0
        n = len(self.images)
        processed_blocks = mp.Manager().list(self.blocks)
        images_i = list(enumerate(self.images))

        with mp.Pool() as pool:
            for image in self.progress(pool.imap(partial(
                _run_all,
                blocks=processed_blocks,
                loader=self.loader
            ), images_i), total=n):
                if not image.discard:
                    if self._has_data:
                        self.data.run(image, terminate=False, show_progress=False)
                else:
                    self._add_discard(image.discard_block, image.i)

    def terminate(self):
        if self._has_data:
            self.data.terminate()

def _run_all(image_i, blocks=None, loader=None):

    i, image = image_i

    if isinstance(image, (str, Path)):
        image = loader(image)
    
    image.i = i

    for block in blocks:
        # This allows to discard image in any Block
        if image.discard:
            return image
        else:
            block._run(image)
            if image.discard:
                image.discard_block = type(block).__name__

    return image