"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationDeleteStackAction = exports.CloudFormationCreateUpdateStackAction = exports.CloudFormationCreateReplaceChangeSetAction = exports.CloudFormationExecuteChangeSetAction = void 0;
const cloudformation = require("@aws-cdk/aws-cloudformation");
const codepipeline = require("@aws-cdk/aws-codepipeline");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const action_1 = require("../action");
/**
 * Base class for Actions that execute CloudFormation
 */
class CloudFormationAction extends action_1.Action {
    constructor(props, inputs) {
        super({
            ...props,
            provider: 'CloudFormation',
            category: codepipeline.ActionCategory.DEPLOY,
            artifactBounds: {
                minInputs: 0,
                maxInputs: 10,
                minOutputs: 0,
                maxOutputs: 1,
            },
            inputs,
            outputs: props.outputFileName
                ? [props.output || new codepipeline.Artifact(`${props.actionName}_${props.stackName}_Artifact`)]
                : undefined,
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        const singletonPolicy = SingletonPolicy.forRole(options.role);
        if ((this.actionProperties.outputs || []).length > 0) {
            options.bucket.grantReadWrite(singletonPolicy);
        }
        else if ((this.actionProperties.inputs || []).length > 0) {
            options.bucket.grantRead(singletonPolicy);
        }
        return {
            configuration: {
                StackName: this.props.stackName,
                OutputFileName: this.props.outputFileName,
            },
        };
    }
}
/**
 * CodePipeline action to execute a prepared change set.
 */
class CloudFormationExecuteChangeSetAction extends CloudFormationAction {
    constructor(props) {
        super(props, undefined);
        this.props2 = props;
    }
    bound(scope, stage, options) {
        SingletonPolicy.forRole(options.role).grantExecuteChangeSet(this.props2);
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_EXECUTE',
                ChangeSetName: this.props2.changeSetName,
            },
        };
    }
}
exports.CloudFormationExecuteChangeSetAction = CloudFormationExecuteChangeSetAction;
// tslint:enable:max-line-length
/**
 * Base class for all CloudFormation actions that execute or stage deployments.
 */
class CloudFormationDeployAction extends CloudFormationAction {
    constructor(props, inputs) {
        super(props, (props.extraInputs || []).concat(inputs || []));
        this.props2 = props;
    }
    /**
     * Add statement to the service role assumed by CloudFormation while executing this action.
     */
    addToDeploymentRolePolicy(statement) {
        return this.getDeploymentRole('method addToRolePolicy()').addToPolicy(statement);
    }
    get deploymentRole() {
        return this.getDeploymentRole('property role()');
    }
    bound(scope, stage, options) {
        if (this.props2.deploymentRole) {
            this._deploymentRole = this.props2.deploymentRole;
        }
        else {
            const roleStack = cdk.Stack.of(options.role);
            const pipelineStack = cdk.Stack.of(scope);
            if (roleStack.account !== pipelineStack.account) {
                // pass role is not allowed for cross-account access - so,
                // create the deployment Role in the other account!
                this._deploymentRole = new iam.Role(roleStack, `${stage.pipeline.node.uniqueId}-${stage.stageName}-${this.actionProperties.actionName}-DeploymentRole`, {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
                    roleName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                });
            }
            else {
                this._deploymentRole = new iam.Role(scope, 'Role', {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
                });
            }
            // the deployment role might need read access to the pipeline's bucket
            // (for example, if it's deploying a Lambda function),
            // and even if it has admin permissions, it won't be enough,
            // as it needs to be added to the key's resource policy
            // (and the bucket's, if the access is cross-account)
            options.bucket.grantRead(this._deploymentRole);
            if (this.props2.adminPermissions) {
                this._deploymentRole.addToPolicy(new iam.PolicyStatement({
                    actions: ['*'],
                    resources: ['*'],
                }));
            }
        }
        SingletonPolicy.forRole(options.role).grantPassRole(this._deploymentRole);
        const capabilities = this.props2.adminPermissions && this.props2.capabilities === undefined
            ? [cloudformation.CloudFormationCapabilities.NAMED_IAM]
            : this.props2.capabilities;
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                // None evaluates to empty string which is falsey and results in undefined
                Capabilities: parseCapabilities(capabilities),
                RoleArn: this.deploymentRole.roleArn,
                ParameterOverrides: cdk.Stack.of(scope).toJsonString(this.props2.parameterOverrides),
                TemplateConfiguration: this.props2.templateConfiguration
                    ? this.props2.templateConfiguration.location
                    : undefined,
                StackName: this.props2.stackName,
            },
        };
    }
    getDeploymentRole(member) {
        if (this._deploymentRole) {
            return this._deploymentRole;
        }
        else {
            throw new Error(`Cannot use the ${member} before the Action has been added to a Pipeline`);
        }
    }
}
/**
 * CodePipeline action to prepare a change set.
 *
 * Creates the change set if it doesn't exist based on the stack name and template that you submit.
 * If the change set exists, AWS CloudFormation deletes it, and then creates a new one.
 */
class CloudFormationCreateReplaceChangeSetAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantCreateReplaceChangeSet(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_REPLACE',
                ChangeSetName: this.props3.changeSetName,
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateReplaceChangeSetAction = CloudFormationCreateReplaceChangeSetAction;
/**
 * CodePipeline action to deploy a stack.
 *
 * Creates the stack if the specified stack doesn't exist. If the stack exists,
 * AWS CloudFormation updates the stack. Use this action to update existing
 * stacks.
 *
 * AWS CodePipeline won't replace the stack, and will fail deployment if the
 * stack is in a failed state. Use `ReplaceOnFailure` for an action that
 * will delete and recreate the stack to try and recover from failed states.
 *
 * Use this action to automatically replace failed stacks without recovering or
 * troubleshooting them. You would typically choose this mode for testing.
 */
class CloudFormationCreateUpdateStackAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantCreateUpdateStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: this.props3.replaceOnFailure ? 'REPLACE_ON_FAILURE' : 'CREATE_UPDATE',
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateUpdateStackAction = CloudFormationCreateUpdateStackAction;
/**
 * CodePipeline action to delete a stack.
 *
 * Deletes a stack. If you specify a stack that doesn't exist, the action completes successfully
 * without deleting a stack.
 */
class CloudFormationDeleteStackAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, undefined);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantDeleteStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'DELETE_ONLY',
            },
        };
    }
}
exports.CloudFormationDeleteStackAction = CloudFormationDeleteStackAction;
/**
 * Manages a bunch of singleton-y statements on the policy of an IAM Role.
 * Dedicated methods can be used to add specific permissions to the role policy
 * using as few statements as possible (adding resources to existing compatible
 * statements instead of adding new statements whenever possible).
 *
 * Statements created outside of this class are not considered when adding new
 * permissions.
 */
class SingletonPolicy extends cdk.Construct {
    constructor(role) {
        super(role, SingletonPolicy.UUID);
        this.role = role;
        this.statements = {};
        this.grantPrincipal = role;
    }
    /**
     * Obtain a SingletonPolicy for a given role.
     * @param role the Role this policy is bound to.
     * @returns the SingletonPolicy for this role.
     */
    static forRole(role) {
        const found = role.node.tryFindChild(SingletonPolicy.UUID);
        return found || new SingletonPolicy(role);
    }
    grantExecuteChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStacks',
                'cloudformation:DescribeChangeSet',
                'cloudformation:ExecuteChangeSet',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateReplaceChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:CreateChangeSet',
                'cloudformation:DeleteChangeSet',
                'cloudformation:DescribeChangeSet',
                'cloudformation:DescribeStacks',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateUpdateStack(props) {
        const actions = [
            'cloudformation:DescribeStack*',
            'cloudformation:CreateStack',
            'cloudformation:UpdateStack',
            'cloudformation:GetTemplate*',
            'cloudformation:ValidateTemplate',
            'cloudformation:GetStackPolicy',
            'cloudformation:SetStackPolicy',
        ];
        if (props.replaceOnFailure) {
            actions.push('cloudformation:DeleteStack');
        }
        this.statementFor({ actions }).addResources(this.stackArnFromProps(props));
    }
    grantDeleteStack(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStack*',
                'cloudformation:DeleteStack',
            ],
        }).addResources(this.stackArnFromProps(props));
    }
    grantPassRole(role) {
        this.statementFor({ actions: ['iam:PassRole'] }).addResources(role.roleArn);
    }
    statementFor(template) {
        const key = keyFor(template);
        if (!(key in this.statements)) {
            this.statements[key] = new iam.PolicyStatement({ actions: template.actions });
            if (template.conditions) {
                this.statements[key].addConditions(template.conditions);
            }
            this.role.addToPolicy(this.statements[key]);
        }
        return this.statements[key];
        function keyFor(props) {
            const actions = `${props.actions.sort().join('\x1F')}`;
            const conditions = formatConditions(props.conditions);
            return `${actions}\x1D${conditions}`;
            function formatConditions(cond) {
                if (cond == null) {
                    return '';
                }
                let result = '';
                for (const op of Object.keys(cond).sort()) {
                    result += `${op}\x1E`;
                    const condition = cond[op];
                    for (const attribute of Object.keys(condition).sort()) {
                        const value = condition[attribute];
                        result += `${value}\x1F`;
                    }
                }
                return result;
            }
        }
    }
    stackArnFromProps(props) {
        return cdk.Stack.of(this).formatArn({
            region: props.region,
            service: 'cloudformation',
            resource: 'stack',
            resourceName: `${props.stackName}/*`,
        });
    }
}
SingletonPolicy.UUID = '8389e75f-0810-4838-bf64-d6f85a95cf83';
function parseCapabilities(capabilities) {
    if (capabilities === undefined) {
        return undefined;
    }
    else if (capabilities.length === 1) {
        const capability = capabilities.toString();
        return (capability === '') ? undefined : capability;
    }
    else if (capabilities.length > 1) {
        return capabilities.join(',');
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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