"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3Code = exports.InlineCode = exports.AssetCode = exports.Code = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const s3_assets = require("aws-cdk-lib/aws-s3-assets");
const runtime_1 = require("./runtime");
/**
 * The code the canary should execute
 */
class Code {
    /**
     * Specify code inline.
     *
     * @param code The actual handler code (limited to 4KiB)
     *
     * @returns `InlineCode` with inline code.
     */
    static fromInline(code) {
        return new InlineCode(code);
    }
    /**
     * Specify code from a local path. Path must include the folder structure `nodejs/node_modules/myCanaryFilename.js`.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     *
     * @param assetPath Either a directory or a .zip file
     *
     * @returns `AssetCode` associated with the specified path.
     */
    static fromAsset(assetPath, options) {
        return new AssetCode(assetPath, options);
    }
    /**
     * Specify code from an s3 bucket. The object in the s3 bucket must be a .zip file that contains
     * the structure `nodejs/node_modules/myCanaryFilename.js`.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html#CloudWatch_Synthetics_Canaries_write_from_scratch
     *
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     *
     * @returns `S3Code` associated with the specified S3 object.
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Code(bucket, key, objectVersion);
    }
}
exports.Code = Code;
_a = JSII_RTTI_SYMBOL_1;
Code[_a] = { fqn: "@aws-cdk/aws-synthetics-alpha.Code", version: "2.19.0-alpha.0" };
/**
 * Canary code from an Asset
 */
class AssetCode extends Code {
    /**
     * @param assetPath The path to the asset file or directory.
     */
    constructor(assetPath, options) {
        super();
        this.assetPath = assetPath;
        this.options = options;
        if (!fs.existsSync(this.assetPath)) {
            throw new Error(`${this.assetPath} is not a valid path`);
        }
    }
    bind(scope, handler, family) {
        jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_RuntimeFamily(family);
        this.validateCanaryAsset(handler, family);
        // If the same AssetCode is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new s3_assets.Asset(scope, 'Code', {
                path: this.assetPath,
                ...this.options,
            });
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
    /**
     * Validates requirements specified by the canary resource. For example, the canary code with handler `index.handler`
     * must be found in the file structure `nodejs/node_modules/index.js`.
     *
     * Requires path to be either zip file or directory.
     * Requires asset directory to have the structure 'nodejs/node_modules'.
     * Requires canary file to be directly inside node_modules folder.
     * Requires canary file name matches the handler name.
     * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary.html
     *
     * @param handler the canary handler
     */
    validateCanaryAsset(handler, family) {
        if (path.extname(this.assetPath) !== '.zip') {
            if (!fs.lstatSync(this.assetPath).isDirectory()) {
                throw new Error(`Asset must be a .zip file or a directory (${this.assetPath})`);
            }
            const filename = handler.split('.')[0];
            const nodeFilename = `${filename}.js`;
            const pythonFilename = `${filename}.py`;
            if (family === runtime_1.RuntimeFamily.NODEJS && !fs.existsSync(path.join(this.assetPath, 'nodejs', 'node_modules', nodeFilename))) {
                throw new Error(`The canary resource requires that the handler is present at "nodejs/node_modules/${nodeFilename}" but not found at ${this.assetPath} (https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary_Nodejs.html)`);
            }
            if (family === runtime_1.RuntimeFamily.PYTHON && !fs.existsSync(path.join(this.assetPath, 'python', pythonFilename))) {
                throw new Error(`The canary resource requires that the handler is present at "python/${pythonFilename}" but not found at ${this.assetPath} (https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Canaries_WritingCanary_Python.html)`);
            }
        }
    }
}
exports.AssetCode = AssetCode;
_b = JSII_RTTI_SYMBOL_1;
AssetCode[_b] = { fqn: "@aws-cdk/aws-synthetics-alpha.AssetCode", version: "2.19.0-alpha.0" };
/**
 * Canary code from an inline string.
 */
class InlineCode extends Code {
    constructor(code) {
        super();
        this.code = code;
        if (code.length === 0) {
            throw new Error('Canary inline code cannot be empty');
        }
    }
    bind(_scope, handler, _family) {
        jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_RuntimeFamily(_family);
        if (handler !== 'index.handler') {
            throw new Error(`The handler for inline code must be "index.handler" (got "${handler}")`);
        }
        return {
            inlineCode: this.code,
        };
    }
}
exports.InlineCode = InlineCode;
_c = JSII_RTTI_SYMBOL_1;
InlineCode[_c] = { fqn: "@aws-cdk/aws-synthetics-alpha.InlineCode", version: "2.19.0-alpha.0" };
/**
 * S3 bucket path to the code zip file
 */
class S3Code extends Code {
    constructor(bucket, key, objectVersion) {
        super();
        this.bucket = bucket;
        this.key = key;
        this.objectVersion = objectVersion;
    }
    bind(_scope, _handler, _family) {
        jsiiDeprecationWarnings._aws_cdk_aws_synthetics_alpha_RuntimeFamily(_family);
        return {
            s3Location: {
                bucketName: this.bucket.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Code = S3Code;
_d = JSII_RTTI_SYMBOL_1;
S3Code[_d] = { fqn: "@aws-cdk/aws-synthetics-alpha.S3Code", version: "2.19.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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