"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceNamespace = exports.ScalableTarget = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
/**
 * Define a scalable target
 */
class ScalableTarget extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.actions = new Array();
        core_1.withResolved(props.maxCapacity, max => {
            if (max < 0) {
                throw new RangeError(`maxCapacity cannot be negative, got: ${props.maxCapacity}`);
            }
        });
        core_1.withResolved(props.minCapacity, min => {
            if (min < 0) {
                throw new RangeError(`minCapacity cannot be negative, got: ${props.minCapacity}`);
            }
        });
        core_1.withResolved(props.minCapacity, props.maxCapacity, (min, max) => {
            if (max < min) {
                throw new RangeError(`minCapacity (${props.minCapacity}) should be lower than maxCapacity (${props.maxCapacity})`);
            }
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('application-autoscaling.amazonaws.com'),
        });
        const resource = new applicationautoscaling_generated_1.CfnScalableTarget(this, 'Resource', {
            maxCapacity: props.maxCapacity,
            minCapacity: props.minCapacity,
            resourceId: props.resourceId,
            roleArn: this.role.roleArn,
            scalableDimension: props.scalableDimension,
            scheduledActions: core_1.Lazy.anyValue({ produce: () => this.actions }, { omitEmptyArray: true }),
            serviceNamespace: props.serviceNamespace,
        });
        this.scalableTargetId = resource.ref;
    }
    static fromScalableTargetId(scope, id, scalableTargetId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.scalableTargetId = scalableTargetId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add a policy statement to the role's policy
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, action) {
        if (action.minCapacity === undefined && action.maxCapacity === undefined) {
            throw new Error(`You must supply at least one of minCapacity or maxCapacity, got ${JSON.stringify(action)}`);
        }
        this.actions.push({
            scheduledActionName: id,
            schedule: action.schedule.expressionString,
            startTime: action.startTime,
            endTime: action.endTime,
            scalableTargetAction: {
                maxCapacity: action.maxCapacity,
                minCapacity: action.minCapacity,
            },
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, scalingTarget: this });
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, id, { ...props, scalingTarget: this });
    }
}
exports.ScalableTarget = ScalableTarget;
/**
 * The service that supports Application AutoScaling
 */
var ServiceNamespace;
(function (ServiceNamespace) {
    /**
     * Elastic Container Service
     */
    ServiceNamespace["ECS"] = "ecs";
    /**
     * Elastic Map Reduce
     */
    ServiceNamespace["ELASTIC_MAP_REDUCE"] = "elasticmapreduce";
    /**
     * Elastic Compute Cloud
     */
    ServiceNamespace["EC2"] = "ec2";
    /**
     * App Stream
     */
    ServiceNamespace["APPSTREAM"] = "appstream";
    /**
     * Dynamo DB
     */
    ServiceNamespace["DYNAMODB"] = "dynamodb";
    /**
     * Relational Database Service
     */
    ServiceNamespace["RDS"] = "rds";
    /**
     * SageMaker
     */
    ServiceNamespace["SAGEMAKER"] = "sagemaker";
    /**
     * Custom Resource
     */
    ServiceNamespace["CUSTOM_RESOURCE"] = "custom-resource";
    /**
     * Lambda
     */
    ServiceNamespace["LAMBDA"] = "lambda";
    /**
     * Comprehend
     */
    ServiceNamespace["COMPREHEND"] = "comprehend";
})(ServiceNamespace = exports.ServiceNamespace || (exports.ServiceNamespace = {}));
//# sourceMappingURL=data:application/json;base64,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