
import os
import sys
import logging
import logging.handlers
from argparse import ArgumentParser, RawTextHelpFormatter, FileType, SUPPRESS
from .gitlab_tree import GitlabTree

from . import __version__ as VERSION

logging.basicConfig(level=logging.INFO, format='%(asctime)s - %(name)s - %(levelname)s - %(message)s')
log = logging.getLogger(__name__)

def main():
    args = parse_args(argv=None if sys.argv[1:] else ['--help'])

    if args.version:
        print(VERSION)
        sys.exit(1) 

    config_logging(args)
    includes=split(args.include)
    excludes=split(args.exclude)
    tree = GitlabTree(args.url, args.token, includes, excludes, args.file, args.out)
    tree.load_tree()

    if tree.is_empty():
        log.error("The tree is empty, check your include/exclude patterns or run with the -v for debugging")
        sys.exit(1) 

    if(args.print):
        tree.print_tree()
    else:
        tree.sync_tree(args.dest)


def split(arg):
    return arg.split(",") if arg != "" else None

def config_logging(args):
    if args.verbose:
        handler = logging.StreamHandler(sys.stdout)
        handler.setFormatter(logging.Formatter('%(asctime)s - %(name)s - %(levelname)s - %(message)s'))
        log.addHandler(handler)
        log.setLevel(logging.DEBUG)
        os.environ["GIT_PYTHON_TRACE"] = 'full'


def parse_args(argv=None):
    example_text = '''examples:

    clone an entire gitlab tree using a url and a token:
    gitlabber -t <personal access token> -u <gitlab url>

    only print the gitlab tree:
    gitlabber -p

    clone only projects under subgroup 'MySubGroup':
    gitlabber -i '/MyGroup/MySubGroup**'

    clone only projects under group 'MyGroup' excluding any projects under subgroup 'MySubGroup':
    gitlabber -i '/MyGroup**' -x '/MyGroup/MySubGroup**'

    clone an entire gitlab tree except projects under groups named 'ArchiveGroup':
    gitlabber -x '/ArchiveGroup**'

    clone projects that start with a case insensitive 'w' using a regular expression:
    gitlabber -i '/{[w].*}'
    '''

    parser = ArgumentParser(
        description='Gitlabber - clones or pulls entire groups/projects tree from gitlab',
        prog="gitlabber",
        epilog=example_text,
        formatter_class=RawTextHelpFormatter)    
    parser.add_argument(
        '-d',
        '--dest',
        metavar=('path'),
        type=validate_path,
        default=os.environ.get('GITLABBER_LOCAL_DIR', '.'),
        help='destination path for the cloned tree (created if doesn\'t exist), if omitted cloned to the current path')
    parser.add_argument(
        '-t',
        '--token',
        metavar=('token'),
        default=os.environ.get('GITLAB_TOKEN'),
        help='gitlab personal access token https://docs.gitlab.com/ee/user/profile/personal_access_tokens.html')
    parser.add_argument(
        '-u',
        '--url',
        metavar=('url'),
        default=os.environ.get('GITLAB_URL'),
        help='gitlab url (e.g.: \'http://gitlab.mycompany.com\')')
    parser.add_argument(
        '-v',
        '--verbose',
        action='store_true')
    parser.add_argument(
        '-o',
        '--out',
        metavar=('file'),
        help=SUPPRESS)
        # help='dump the binary tree structure to file for local debugging')
    parser.add_argument(
        '-f',
        '--file',
        metavar=('file'),
        help=SUPPRESS)
        # help='load the binary tree structure from a file for local debugging')
    parser.add_argument(
        '-p',
        '--print',
        action='store_true',
        help='print the tree without cloning')
    parser.add_argument(
        '-i',
        '--include',
        metavar=('csv'),
        default=os.environ.get('GITLABBER_INCLUDE', ""),
        help='comma delimited list of glob patterns of paths to projects or groups to clone/pull')
    parser.add_argument(
        '-x',
        '--exclude',
        metavar=('csv'),
        default=os.environ.get('GITLABBER_EXCLUDE', ""),
        help='comma delimited list of glob patterns of paths to projects or groups to exclude from clone/pull')
    parser.add_argument(
        '--version',
        action='store_true',
        help='print the version')

    args = parser.parse_args(argv)
   
    return args

def validate_path(value):
    if value.endswith('/'):
        return value[:-1]
    return value

