"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SnsPublish = exports.MessageAttributeDataType = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * The data type set for the SNS message attributes.
 *
 * @see https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
 * @stability stable
 */
var MessageAttributeDataType;
(function (MessageAttributeDataType) {
    MessageAttributeDataType["STRING"] = "String";
    MessageAttributeDataType["STRING_ARRAY"] = "String.Array";
    MessageAttributeDataType["NUMBER"] = "Number";
    MessageAttributeDataType["BINARY"] = "Binary";
})(MessageAttributeDataType = exports.MessageAttributeDataType || (exports.MessageAttributeDataType = {}));
/**
 * A Step Functions Task to publish messages to SNS topic.
 *
 * @stability stable
 */
class SnsPublish extends sfn.TaskStateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id, props);
        this.props = props;
        jsiiDeprecationWarnings._aws_cdk_aws_stepfunctions_tasks_SnsPublishProps(props);
        this.integrationPattern = (_b = props.integrationPattern) !== null && _b !== void 0 ? _b : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SnsPublish.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN) {
            if (!sfn.FieldUtils.containsTaskToken(props.message)) {
                throw new Error('Task Token is required in `message` Use JsonPath.taskToken to set the token.');
            }
        }
        this.taskPolicies = [
            new iam.PolicyStatement({
                actions: ['sns:Publish'],
                resources: [this.props.topic.topicArn],
            }),
        ];
    }
    /**
     * Provides the SNS Publish service integration task configuration
     */
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sns', 'publish', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                TopicArn: this.props.topic.topicArn,
                Message: this.props.message.value,
                MessageStructure: this.props.messagePerSubscriptionType ? 'json' : undefined,
                MessageAttributes: renderMessageAttributes(this.props.messageAttributes),
                Subject: this.props.subject,
            }),
        };
    }
}
exports.SnsPublish = SnsPublish;
_a = JSII_RTTI_SYMBOL_1;
SnsPublish[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.SnsPublish", version: "1.141.0" };
SnsPublish.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
function renderMessageAttributes(attributes) {
    if (attributes === undefined) {
        return undefined;
    }
    const renderedAttributes = {};
    Object.entries(attributes).map(([key, val]) => {
        renderedAttributes[key] = renderMessageAttributeValue(val);
    });
    return sfn.TaskInput.fromObject(renderedAttributes).value;
}
function renderMessageAttributeValue(attribute) {
    const dataType = attribute.dataType;
    if (attribute.value instanceof sfn.TaskInput) {
        return {
            DataType: dataType !== null && dataType !== void 0 ? dataType : MessageAttributeDataType.STRING,
            StringValue: dataType !== MessageAttributeDataType.BINARY ? attribute.value.value : undefined,
            BinaryValue: dataType === MessageAttributeDataType.BINARY ? attribute.value.value : undefined,
        };
    }
    if (dataType === MessageAttributeDataType.BINARY) {
        return { DataType: dataType, BinaryValue: `${attribute.value}` };
    }
    if (core_1.Token.isUnresolved(attribute.value)) {
        return { DataType: dataType !== null && dataType !== void 0 ? dataType : MessageAttributeDataType.STRING, StringValue: attribute.value };
    }
    validateMessageAttribute(attribute);
    if (Array.isArray(attribute.value)) {
        return { DataType: MessageAttributeDataType.STRING_ARRAY, StringValue: JSON.stringify(attribute.value) };
    }
    const value = attribute.value;
    if (typeof value === 'number') {
        return { DataType: MessageAttributeDataType.NUMBER, StringValue: `${value}` };
    }
    else {
        return { DataType: MessageAttributeDataType.STRING, StringValue: `${value}` };
    }
}
function validateMessageAttribute(attribute) {
    const dataType = attribute.dataType;
    const value = attribute.value;
    if (dataType === undefined) {
        return;
    }
    if (Array.isArray(value)) {
        if (dataType !== MessageAttributeDataType.STRING_ARRAY) {
            throw new Error(`Requested SNS message attribute type was ${dataType} but ${value} was of type Array`);
        }
        const validArrayTypes = ['string', 'boolean', 'number'];
        value.forEach((v) => {
            if (v !== null || !validArrayTypes.includes(typeof v)) {
                throw new Error(`Requested SNS message attribute type was ${typeof value} but Array values must be one of ${validArrayTypes}`);
            }
        });
        return;
    }
    const error = new Error(`Requested SNS message attribute type was ${dataType} but ${value} was of type ${typeof value}`);
    switch (typeof value) {
        case 'string':
            // trust the user or will default to string
            if (sfn.JsonPath.isEncodedJsonPath(attribute.value)) {
                return;
            }
            if (dataType === MessageAttributeDataType.STRING ||
                dataType === MessageAttributeDataType.BINARY) {
                return;
            }
            throw error;
        case 'number':
            if (dataType === MessageAttributeDataType.NUMBER) {
                return;
            }
            throw error;
        case 'boolean':
            if (dataType === MessageAttributeDataType.STRING) {
                return;
            }
            throw error;
        default:
            throw error;
    }
}
//# sourceMappingURL=data:application/json;base64,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