"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'CodeCommit Repositories': {
        'add an SNS trigger to repository'(test) {
            const stack = new core_1.Stack();
            const props = {
                repositoryName: 'MyRepository',
            };
            const snsArn = 'arn:aws:sns:*:123456789012:my_topic';
            new lib_1.Repository(stack, 'MyRepository', props).notify(snsArn);
            assert_1.expect(stack).toMatch({
                Resources: {
                    MyRepository4C4BD5FC: {
                        Type: 'AWS::CodeCommit::Repository',
                        Properties: {
                            RepositoryName: 'MyRepository',
                            Triggers: [
                                {
                                    Events: [
                                        'all',
                                    ],
                                    DestinationArn: 'arn:aws:sns:*:123456789012:my_topic',
                                    Name: 'Default/MyRepository/arn:aws:sns:*:123456789012:my_topic',
                                },
                            ],
                        },
                    },
                },
            });
            test.done();
        },
        'fails when triggers have duplicate names'(test) {
            const stack = new core_1.Stack();
            const myRepository = new lib_1.Repository(stack, 'MyRepository', {
                repositoryName: 'MyRepository',
            }).notify('myTrigger');
            test.throws(() => myRepository.notify('myTrigger'));
            test.done();
        },
        'can be imported using a Repository ARN'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const repositoryArn = 'arn:aws:codecommit:us-east-1:585695036304:my-repo';
            // WHEN
            const repo = lib_1.Repository.fromRepositoryArn(stack, 'ImportedRepo', repositoryArn);
            // THEN
            test.deepEqual(stack.resolve(repo.repositoryArn), repositoryArn);
            test.deepEqual(stack.resolve(repo.repositoryName), 'my-repo');
            test.done();
        },
        /**
         * Fix for https://github.com/aws/aws-cdk/issues/10630
         */
        'can be imported using a Repository ARN and respect the region in clone urls'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const repositoryArn = 'arn:aws:codecommit:us-west-2:585695036304:my-repo';
            // WHEN
            const repo = lib_1.Repository.fromRepositoryArn(stack, 'ImportedRepo', repositoryArn);
            // THEN
            // a fully qualified arn should use the region from the arn
            test.deepEqual(stack.resolve(repo.repositoryCloneUrlHttp), {
                'Fn::Join': [
                    '',
                    [
                        'https://git-codecommit.us-west-2.',
                        { Ref: 'AWS::URLSuffix' },
                        '/v1/repos/my-repo',
                    ],
                ],
            });
            test.deepEqual(stack.resolve(repo.repositoryCloneUrlSsh), {
                'Fn::Join': [
                    '',
                    [
                        'ssh://git-codecommit.us-west-2.',
                        { Ref: 'AWS::URLSuffix' },
                        '/v1/repos/my-repo',
                    ],
                ],
            });
            test.done();
        },
        'can be imported using just a Repository name (the ARN is deduced)'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            // WHEN
            const repo = lib_1.Repository.fromRepositoryName(stack, 'ImportedRepo', 'my-repo');
            // THEN
            test.deepEqual(stack.resolve(repo.repositoryArn), {
                'Fn::Join': ['', [
                        'arn:',
                        { Ref: 'AWS::Partition' },
                        ':codecommit:',
                        { Ref: 'AWS::Region' },
                        ':',
                        { Ref: 'AWS::AccountId' },
                        ':my-repo',
                    ]],
            });
            test.deepEqual(stack.resolve(repo.repositoryName), 'my-repo');
            //local name resolution should use stack region
            test.deepEqual(stack.resolve(repo.repositoryCloneUrlHttp), {
                'Fn::Join': [
                    '',
                    [
                        'https://git-codecommit.',
                        { Ref: 'AWS::Region' },
                        '.',
                        { Ref: 'AWS::URLSuffix' },
                        '/v1/repos/my-repo',
                    ],
                ],
            });
            test.done();
        },
        'grant push'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            const repository = new lib_1.Repository(stack, 'Repo', {
                repositoryName: 'repo-name',
            });
            const role = new aws_iam_1.Role(stack, 'Role', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            });
            // WHEN
            repository.grantPullPush(role);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: 'codecommit:GitPull',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    'Repo02AC86CF',
                                    'Arn',
                                ],
                            },
                        },
                        {
                            Action: 'codecommit:GitPush',
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    'Repo02AC86CF',
                                    'Arn',
                                ],
                            },
                        },
                    ],
                    Version: '2012-10-17',
                },
            }));
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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