from __future__ import division

from functools import partial

from numpy import allclose, isclose

from funcsim.addist import ad_unif, ad_unif_inf, ad_unif_fix

# Reset default tolerance settings not to forget to specify some later.
allclose = partial(allclose, atol=0, rtol=0)
isclose = partial(isclose, atol=0, rtol=0)

# Values from doi:10.1214/aoms/1177704850, table 1. These were generated by a
# Monte Carlo simulation in 1960s. Only about a million repetitions with small
# sample counts was possible at that time, resulting in an accuracy of about
# 3 decimal digits; tuples of (sample count, statistic value, probability).
lewis_values = (
    (1, .5, .328), (1, 1., .677), (1, 1.5, .820), (1, 2., .895),
    (1, 2.5, .938), (1, 3., .963), (1, 3.5, .978), (1, 4., .986),
    (1, 5., .995), (1, 6., .998), (1, 7., .998), (1, 8., .999),

    (3, .5, .271), (3, 1., .652), (3, 1.5, .823), (3, 2., .904),
    (3, 2.5, .947), (3, 3., .970), (3, 3.5, .983), (3, 4., .990),
    (3, 5., .996), (3, 6., .998), (3, 7., .999), (3, 8., 1.00),

    (5, .5, .261), (5, 1., .647), (5, 1.5, .824), (5, 2., .907),
    (5, 2.5, .949), (5, 3., .971), (5, 3.5, .983), (5, 4., .990),
    (5, 5., .996), (5, 6., .999), (5, 7., .999), (5, 8., 1.00),

    (8, .5, .259), (8, 1., .646), (8, 1.5, .824), (8, 2., .907),
    (8, 2.5, .949), (8, 3., .971), (8, 3.5, .984), (8, 4., .990),
    (8, 5., .997), (8, 6., .999), (8, 7., .999), (8, 8., 1.00))


def test_cdf():
    # The high tolerance is more due to the imprecision of the test values.
    for sp, st, pr in lewis_values:
        assert isclose(ad_unif.cdf(st, sp), pr, rtol=.05)

    # Compared to R package "goftest" (based on the same approximation).
    ps = ad_unif(5).cdf((.5, 1, 2, 3))
    assert allclose(ps, (.2617, .6474, .9057, .9708), rtol=.5e-3)
    ps = ad_unif(10).cdf((.5, 1, 2, 3))
    assert allclose(ps, (.2574, .6449, .9069, .9717), rtol=.5e-3)
    ps = ad_unif(20).cdf((.5, 1, 2, 3))
    assert allclose(ps, (.2553, .6438, .9075, .9722), rtol=.5e-3)

def test_one():
    # For n = 1, p = sqrt(1 - 4 exp(-1 - z)) or 0.
    ps = ad_unif(1).cdf((.1, 1, 3))
    assert allclose(ps, (0, .677243580297, .962672033688), rtol=.5e-12)

def test_inf():
    # Values for pinf from doi:10.18637/jss.v009.i02 (the original 1954
    # doi:10.2307/2281537 gives respectively 1.933, 2.492, and 3.857).
    ps = [ad_unif_inf(s) for s in (1.932957833, 2.492367160, 3.878125022)]
    assert allclose(ps, (.90, .95, .99), rtol=.5e-4)

    # A few values from R (using the slower, more accurate approximation).
    ps = [ad_unif_inf(s) for s in (.5, 1, 2, 3)]
    assert allclose(ps, (.253186, .642733, .908163, .972635), rtol=.5e-4)

def test_fix():
    # Compared with a fragment of Marsaglia code.
    ps = [ad_unif_fix(3, pinf) for pinf in (.1, .5, .9)]
    assert allclose(ps, (.003673680, .01346629, -.003937215), rtol=.5e-6)
    ps = [ad_unif_fix(8, pinf) for pinf in (.1, .5, .9)]
    assert allclose(ps, (.002521423, .004600524, -.001476456), rtol=.5e-6)
