import shutil
import os
import random

from celery.decorators import task
from celery import Task
import yaml
from sqlalchemy.exc import IntegrityError

from conda_store_server.worker.app import CondaStoreWorker
from conda_store_server import api, environment, utils, orm
from conda_store_server.build import (
    build_conda_environment,
    build_conda_env_export,
    build_conda_pack,
    build_conda_docker,
)


class WorkerTask(Task):
    _worker = None

    def after_return(self, *args, **kwargs):
        if self._worker is not None:
            self._worker.conda_store.session_factory.remove()

    @property
    def worker(self):
        if self._worker is None:
            self._worker = CondaStoreWorker()
            self._worker.initialize()
        return self._worker


@task(base=WorkerTask, name="task_watch_paths", bind=True)
def task_watch_paths(self):
    conda_store = self.worker.conda_store

    environment_paths = environment.discover_environments(self.worker.watch_paths)
    for path in environment_paths:
        with open(path) as f:
            conda_store.register_environment(
                specification=yaml.safe_load(f), namespace="filesystem"
            )


@task(base=WorkerTask, name="task_update_storage_metrics", bind=True)
def task_update_storage_metrics(self):
    conda_store = self.worker.conda_store
    conda_store.configuration.update_storage_metrics(
        conda_store.db, conda_store.store_directory
    )


@task(base=WorkerTask, name="task_update_conda_channels", bind=True)
def task_update_conda_channels(self):
    conda_store = self.worker.conda_store
    try:
        conda_store.ensure_conda_channels()

        for channel in api.list_conda_channels(conda_store.db):
            channel.update_packages(conda_store.db)
    except IntegrityError as exc:
        # there is a persistent error on startup
        # that when the conda channels are out of data
        # and two tasks try to add the same packages
        # it runs into integrity errors
        # the solution is to let one of them finish
        # and the other try again at a later time
        self.retry(exc=exc, countdown=random.randrange(15, 30))


@task(base=WorkerTask, name="task_build_conda_environment", bind=True)
def task_build_conda_environment(self, build_id):
    conda_store = self.worker.conda_store
    build = api.get_build(conda_store.db, build_id)
    build_conda_environment(conda_store, build)


@task(base=WorkerTask, name="task_build_conda_env_export", bind=True)
def task_build_conda_env_export(self, build_id):
    conda_store = self.worker.conda_store
    build = api.get_build(conda_store.db, build_id)
    build_conda_env_export(conda_store, build)


@task(base=WorkerTask, name="task_build_conda_pack", bind=True)
def task_build_conda_pack(self, build_id):
    conda_store = self.worker.conda_store
    build = api.get_build(conda_store.db, build_id)
    build_conda_pack(conda_store, build)


@task(base=WorkerTask, name="task_build_conda_docker", bind=True)
def task_build_conda_docker(self, build_id):
    conda_store = self.worker.conda_store
    build = api.get_build(conda_store.db, build_id)
    build_conda_docker(conda_store, build)


@task(base=WorkerTask, name="task_update_environment_build", bind=True)
def task_update_environment_build(self, environment_id):
    conda_store = self.worker.conda_store
    environment = api.get_environment(conda_store.db, id=environment_id)

    conda_prefix = environment.build.build_path(conda_store.store_directory)
    environment_prefix = environment.build.environment_path(
        conda_store.environment_directory
    )

    utils.symlink(conda_prefix, environment_prefix)


@task(base=WorkerTask, name="task_delete_build", bind=True)
def task_delete_build(self, build_id):
    conda_store = self.worker.conda_store
    build = api.get_build(conda_store.db, build_id)

    conda_store.log.error("deleting artifacts")
    for build_artifact in api.list_build_artifacts(
        conda_store.db,
        limit=None,
        build_id=build_id,
        excluded_artifact_types=conda_store.build_artifacts_kept_on_deletion,
    ):
        if build_artifact.artifact_type == orm.BuildArtifactType.DIRECTORY:
            # ignore key
            conda_prefix = build.build_path(conda_store.store_directory)
            # be REALLY sure this is a directory within store directory
            if conda_prefix.startswith(conda_store.store_directory) and os.path.isdir(
                conda_prefix
            ):
                shutil.rmtree(conda_prefix)
                conda_store.db.delete(build_artifact)
        elif build_artifact.artifact_type == orm.BuildArtifactType.LOCKFILE:
            pass
        else:
            conda_store.log.error(f"deleting {build_artifact.key}")
            conda_store.storage.delete(conda_store.db, build_id, build_artifact.key)

    conda_store.db.commit()
