"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StringStep = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const choice_1 = require("../../domain/choice");
const data_type_1 = require("../../domain/data-type");
const operation_1 = require("../../domain/operation");
const string_variable_1 = require("../../interface/variables/string-variable");
const approve_step_1 = require("../../parent-steps/automation/approve-step");
const assert_aws_resource_step_1 = require("../../parent-steps/automation/assert-aws-resource-step");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const branch_step_1 = require("../../parent-steps/automation/branch-step");
const execute_script_step_1 = require("../../parent-steps/automation/execute-script-step");
const pause_step_1 = require("../../parent-steps/automation/pause-step");
const sleep_step_1 = require("../../parent-steps/automation/sleep-step");
const wait_for_resource_step_1 = require("../../parent-steps/automation/wait-for-resource-step");
const composite_step_1 = require("./composite-step");
// eslint-disable-next-line
const yaml = require('js-yaml');
/**
 * StringStep allows for including a step from an existing automation document verbatim.
 * This is useful in that it allows developers to integrate with existing document steps.
 * This step can be used just as you would use any other Step including simulation and deployment.
 */
class StringStep extends composite_step_1.CompositeAutomationStep {
    constructor(scope, id, props) {
        super(scope, id);
        const sharedProps = {};
        sharedProps.name = props.name;
        sharedProps.action = props.action;
        if (props.maxAttempts) {
            sharedProps.maxAttempts = props.maxAttempts;
        }
        if (props.timeoutSeconds) {
            sharedProps.timeoutSeconds = props.timeoutSeconds;
        }
        if (props.isEnd != undefined) {
            sharedProps.isEnd = props.isEnd;
        }
        if (props.isCritical != undefined) {
            sharedProps.isCritical = props.isCritical;
        }
        if (props.outputs != undefined) {
            sharedProps.outputs = this.toOutputs(props.outputs);
        }
        const { Service, Api, PropertySelector, DesiredValues, ...restParams } = props.inputs;
        switch (props.action) {
            case 'aws:executeAwsApi':
                this.automationStep = new aws_api_step_1.AwsApiStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    outputs: sharedProps.outputs,
                    ...sharedProps,
                });
                break;
            case 'aws:waitForAwsResourceProperty':
                this.automationStep = new wait_for_resource_step_1.WaitForResourceStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    selector: PropertySelector,
                    desiredValues: DesiredValues,
                    ...sharedProps,
                });
                break;
            case 'aws:assertAwsResourceProperty':
                this.automationStep = new assert_aws_resource_step_1.AssertAwsResourceStep(this, props.name, {
                    service: Service,
                    pascalCaseApi: Api,
                    apiParams: restParams,
                    selector: PropertySelector,
                    desiredValues: DesiredValues,
                    ...sharedProps,
                });
                break;
            case 'aws:pause':
                this.automationStep = new pause_step_1.PauseStep(this, props.name, {
                    ...sharedProps,
                });
                break;
            case 'aws:sleep':
                this.automationStep = new sleep_step_1.SleepStep(this, props.name, {
                    sleepSeconds: this.parseDurationToSeconds(restParams.Duration),
                    ...sharedProps,
                });
                break;
            case 'aws:executeScript':
                const inputs = {};
                Object.entries(restParams.InputPayload ?? {}).forEach(([key, value]) => inputs[key] = this.toVariable(value));
                const handler = restParams.Handler ? restParams.Handler.replace('function.', '') : undefined;
                this.automationStep = new execute_script_step_1.ExecuteScriptStep(this, props.name, {
                    language: execute_script_step_1.ScriptLanguage.fromRuntime(restParams.Runtime, handler),
                    inputPayload: inputs,
                    code: execute_script_step_1.ScriptCode.inline(restParams.Script),
                    ...sharedProps,
                });
                break;
            case 'aws:branch':
                this.automationStep = new branch_step_1.BranchStep(this, props.name, {
                    choices: this.toChoices(restParams.Choices),
                    defaultStepName: restParams.Default,
                    ...sharedProps,
                });
                break;
            case 'aws:approve':
                this.automationStep = new approve_step_1.ApproveStep(this, props.name, {
                    approvers: restParams.Approvers,
                    minRequiredApprovals: restParams.MinRequiredApprovals,
                    message: restParams.Message,
                    notificationArn: restParams.NotificationArn,
                    ...sharedProps,
                });
                break;
            default:
                throw new Error('Action not implemented as StringStep ' + props.action);
        }
    }
    /**
       * Builds a step object from a yaml declaration.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromYaml(scope, inputYaml) {
        return StringStep.fromObject(scope, yaml.load(inputYaml));
    }
    /**
       * Builds a step object from a json declaration.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromJson(scope, json) {
        return StringStep.fromObject(scope, JSON.parse(json));
    }
    /**
       * Builds a step object from an object.
       * You may cast the step afterword to the associated Step for this action
       * if you wish to gain access to action specific functionality,
       * @returns Step associated with the provided activity.
       */
    static fromObject(scope, props) {
        return new StringStep(scope, props.name, props);
    }
    addToDocument(doc) {
        doc.addStep(this.automationStep);
    }
    toOutputs(declaredOutputs) {
        return declaredOutputs.map(declaredOutput => this.toOutput(declaredOutput));
    }
    toOutput(declaredOutput) {
        return {
            name: declaredOutput.Name,
            selector: declaredOutput.Selector,
            outputType: data_type_1.DataType.fromDataType(declaredOutput.Type).dataTypeEnum,
        };
    }
    toChoices(declaredChoices) {
        return declaredChoices.map(declaredChoice => this.toChoice(declaredChoice));
    }
    toChoice(declaredChoice) {
        const operationEntry = Object.entries(declaredChoice)
            .filter(entry => operation_1.OperationEvaluator.STRING_TO_OPERATION[entry[0]] != undefined)[0];
        return new choice_1.Choice({
            jumpToStepName: declaredChoice.NextStep,
            variable: this.toVariable(declaredChoice.Variable),
            operation: operation_1.OperationEvaluator.fromOperationName(operationEntry[0]),
            constant: operationEntry[1],
        });
    }
    toVariable(variable) {
        if (variable.includes('{{')) {
            return new string_variable_1.StringVariable(variable.replace('{{', '').replace('}}', '').trim());
        }
        return new string_variable_1.HardCodedString(variable.trim());
    }
    parseDurationToSeconds(iso8601Duration) {
        // https://stackoverflow.com/a/29153059
        const iso8601DurationRegex = /(-)?P(?:([.,\d]+)Y)?(?:([.,\d]+)M)?(?:([.,\d]+)W)?(?:([.,\d]+)D)?T(?:([.,\d]+)H)?(?:([.,\d]+)M)?(?:([.,\d]+)S)?/;
        const matches = iso8601Duration.match(iso8601DurationRegex);
        if (matches == undefined) {
            throw new Error('Could not parse Duration' + iso8601Duration);
        }
        const periods = {
            years: matches[2] === undefined ? 0 : Number.parseInt(matches[2]),
            months: matches[3] === undefined ? 0 : Number.parseInt(matches[3]),
            weeks: matches[4] === undefined ? 0 : Number.parseInt(matches[4]),
            days: matches[5] === undefined ? 0 : Number.parseInt(matches[5]),
            hours: matches[6] === undefined ? 0 : Number.parseInt(matches[6]),
            minutes: matches[7] === undefined ? 0 : Number.parseInt(matches[7]),
            seconds: matches[8] === undefined ? 0 : Number.parseInt(matches[8]),
        };
        return periods.seconds +
            periods.minutes * 60 +
            periods.hours * 60 * 60 +
            periods.days * 60 * 60 * 24 +
            periods.weeks * 60 * 60 * 24 * 7 +
            periods.months * 60 * 60 * 24 * 30.5 +
            periods.years * 60 * 60 * 24 * 365;
    }
    ;
}
exports.StringStep = StringStep;
_a = JSII_RTTI_SYMBOL_1;
StringStep[_a] = { fqn: "@cdklabs/cdk-ssm-documents.StringStep", version: "0.0.24" };
//# sourceMappingURL=data:application/json;base64,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