"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Input = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const data_type_1 = require("./data-type");
class Input {
    constructor(props) {
        this.name = props.name;
        this.inputType = props.inputType;
        this.description = props.description;
        this.defaultValue = props.defaultValue;
        this.allowedValues = props.allowedValues;
        this.allowedPattern = props.allowedPattern;
        this.minItems = props.minItems;
        this.maxItems = props.maxItems;
        this.minChars = props.minChars;
        this.maxChars = props.maxChars;
    }
    static ofTypeString(name, props) {
        return new StringInput(name, props);
    }
    static ofTypeInteger(name, props) {
        return new IntegerInput(name, props);
    }
    static ofTypeBoolean(name, props) {
        return new BooleanInput(name, props);
    }
    static ofTypeStringList(name, props) {
        return new StringListInput(name, props);
    }
    static ofTypeMapList(name, props) {
        return new MapListInput(name, props);
    }
    static ofTypeStringMap(name, props) {
        return new StringMapInput(name, props);
    }
    static ofSpecifiedType(type, inputName, props) {
        switch (type) {
            case data_type_1.DataTypeEnum.STRING:
                return this.ofTypeString(inputName, props);
            case data_type_1.DataTypeEnum.INTEGER:
                return this.ofTypeInteger(inputName, props);
            case data_type_1.DataTypeEnum.BOOLEAN:
                return this.ofTypeBoolean(inputName, props);
            case data_type_1.DataTypeEnum.STRING_LIST:
                return this.ofTypeStringList(inputName, props);
            case data_type_1.DataTypeEnum.MAP_LIST:
                return this.ofTypeMapList(inputName, props);
            case data_type_1.DataTypeEnum.STRING_MAP:
                return this.ofTypeStringMap(inputName, props);
            default:
                throw new Error('DataTypeEnum not supported: ' + type);
        }
    }
    toSsm() {
        const nested = {
            type: new data_type_1.DataType(this.inputType).toSsmString(),
        };
        if (this.description != undefined) {
            nested.description = this.description;
        }
        if (this.defaultValue != undefined) {
            nested.default = this.defaultValue;
        }
        if (this.allowedValues != undefined) {
            nested.allowedValues = this.allowedValues;
        }
        if (this.allowedPattern != undefined) {
            nested.allowedPattern = this.allowedPattern;
        }
        if (this.minItems != undefined) {
            nested.minItems = this.minItems;
        }
        if (this.maxItems != undefined) {
            nested.maxItems = this.maxItems;
        }
        return nested;
    }
    validate(value) {
        if (!(new data_type_1.DataType(this.inputType).validateType(value))) {
            throw new Error(`Value ${value} does not match type ${this.inputType}`);
        }
        if (this.maxItems && Array.isArray(value) && this.maxItems < value.length) {
            throw new Error(`Values ${value} is greater than maxItems: ${this.maxItems}`);
        }
        if (this.minItems && Array.isArray(value) && this.minItems > value.length) {
            throw new Error(`Values ${value} is less than minItems: ${this.maxItems}`);
        }
        if (this.allowedPattern && !value.match(this.allowedPattern)) {
            throw new Error(`Value ${value} does not match regex ${this.allowedPattern}`);
        }
        if (this.allowedValues && !this.allowedValues.includes(value)) {
            throw new Error(`Value ${value} is not among allowedValues ${this.allowedValues}`);
        }
    }
}
exports.Input = Input;
_a = JSII_RTTI_SYMBOL_1;
Input[_a] = { fqn: "@cdklabs/cdk-ssm-documents.Input", version: "0.0.24" };
class StringInput extends Input {
    constructor(name, props) {
        super({ name: name, inputType: data_type_1.DataTypeEnum.STRING, ...props });
    }
    validate(value) {
        super.validate(value);
        const stringVal = value;
        if (this.minChars != undefined && this.minChars > stringVal.length) {
            new Error(`Value ${value} is less than minimum specified ${this.minChars}`);
        }
        if (this.maxChars != undefined && this.maxChars < stringVal.length) {
            new Error(`Value ${value} is greater than minimum specified ${this.maxChars}`);
        }
    }
}
class IntegerInput extends Input {
    constructor(name, props) {
        super({ name: name, inputType: data_type_1.DataTypeEnum.INTEGER, ...props });
    }
}
class BooleanInput extends Input {
    constructor(name, props) {
        super({ name: name, inputType: data_type_1.DataTypeEnum.BOOLEAN, ...props });
    }
}
class StringListInput extends Input {
    constructor(name, props) {
        super({ name: name, inputType: data_type_1.DataTypeEnum.STRING_LIST, ...props });
    }
}
class MapListInput extends Input {
    constructor(name, props) {
        super({ name: name, inputType: data_type_1.DataTypeEnum.MAP_LIST, ...props });
    }
}
class StringMapInput extends Input {
    constructor(name, props) {
        super({ name: name, inputType: data_type_1.DataTypeEnum.STRING_MAP, ...props });
    }
}
//# sourceMappingURL=data:application/json;base64,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