"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkMultipleTargetGroupsServiceBase = void 0;
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const core_1 = require("@aws-cdk/core");
/**
 * The base class for NetworkMultipleTargetGroupsEc2Service and NetworkMultipleTargetGroupsFargateService classes.
 */
class NetworkMultipleTargetGroupsServiceBase extends core_1.Construct {
    /**
     * Constructs a new instance of the NetworkMultipleTargetGroupsServiceBase class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.listeners = new Array();
        this.targetGroups = new Array();
        this.loadBalancers = new Array();
        this.validateInput(props);
        this.cluster = props.cluster || this.getDefaultCluster(this, props.vpc);
        this.desiredCount = props.desiredCount || 1;
        if (props.taskImageOptions) {
            this.logDriver = this.createLogDriver(props.taskImageOptions.enableLogging, props.taskImageOptions.logDriver);
        }
        if (props.loadBalancers) {
            for (const lbProps of props.loadBalancers) {
                const lb = this.createLoadBalancer(lbProps.name, lbProps.publicLoadBalancer);
                this.loadBalancers.push(lb);
                for (const listenerProps of lbProps.listeners) {
                    const listener = this.createListener(listenerProps.name, lb, listenerProps.port || 80);
                    this.listeners.push(listener);
                }
                this.createDomainName(lb, lbProps.domainName, lbProps.domainZone);
                new core_1.CfnOutput(this, `LoadBalancerDNS${lb.node.id}`, { value: lb.loadBalancerDnsName });
            }
            // set up default load balancer and listener.
            this.loadBalancer = this.loadBalancers[0];
            this.listener = this.listeners[0];
        }
        else {
            this.loadBalancer = this.createLoadBalancer('LB');
            this.listener = this.createListener('PublicListener', this.loadBalancer, 80);
            this.createDomainName(this.loadBalancer);
            new core_1.CfnOutput(this, 'LoadBalancerDNS', { value: this.loadBalancer.loadBalancerDnsName });
        }
    }
    /**
     * Returns the default cluster.
     */
    getDefaultCluster(scope, vpc) {
        // magic string to avoid collision with user-defined constructs.
        const DEFAULT_CLUSTER_ID = `EcsDefaultClusterMnL3mNNYN${vpc ? vpc.node.id : ''}`;
        const stack = core_1.Stack.of(scope);
        return stack.node.tryFindChild(DEFAULT_CLUSTER_ID) || new aws_ecs_1.Cluster(stack, DEFAULT_CLUSTER_ID, { vpc });
    }
    createAWSLogDriver(prefix) {
        return new aws_ecs_1.AwsLogDriver({ streamPrefix: prefix });
    }
    findListener(name) {
        if (!name) {
            return this.listener;
        }
        for (const listener of this.listeners) {
            if (listener.node.id === name) {
                return listener;
            }
        }
        throw new Error(`Listener ${name} is not defined. Did you define listener with name ${name}?`);
    }
    registerECSTargets(service, container, targets) {
        for (const targetProps of targets) {
            const targetGroup = this.findListener(targetProps.listener).addTargets(`ECSTargetGroup${container.containerName}${targetProps.containerPort}`, {
                port: 80,
                targets: [
                    service.loadBalancerTarget({
                        containerName: container.containerName,
                        containerPort: targetProps.containerPort,
                    }),
                ],
            });
            this.targetGroups.push(targetGroup);
        }
        if (this.targetGroups.length === 0) {
            throw new Error('At least one target group should be specified.');
        }
        return this.targetGroups[0];
    }
    addPortMappingForTargets(container, targets) {
        for (const target of targets) {
            if (!container.findPortMapping(target.containerPort, aws_ecs_1.Protocol.TCP)) {
                container.addPortMappings({
                    containerPort: target.containerPort,
                });
            }
        }
    }
    /**
     * Create log driver if logging is enabled.
     */
    createLogDriver(enableLoggingProp, logDriverProp) {
        const enableLogging = enableLoggingProp !== undefined ? enableLoggingProp : true;
        const logDriver = logDriverProp !== undefined
            ? logDriverProp : enableLogging
            ? this.createAWSLogDriver(this.node.id) : undefined;
        return logDriver;
    }
    validateInput(props) {
        if (props.cluster && props.vpc) {
            throw new Error('You can only specify either vpc or cluster. Alternatively, you can leave both blank');
        }
        if (props.desiredCount !== undefined && props.desiredCount < 1) {
            throw new Error('You must specify a desiredCount greater than 0');
        }
        if (props.loadBalancers) {
            if (props.loadBalancers.length === 0) {
                throw new Error('At least one load balancer must be specified');
            }
            for (const lbProps of props.loadBalancers) {
                if (lbProps.listeners.length === 0) {
                    throw new Error('At least one listener must be specified');
                }
            }
        }
    }
    createLoadBalancer(name, publicLoadBalancer) {
        const internetFacing = publicLoadBalancer !== undefined ? publicLoadBalancer : true;
        const lbProps = {
            vpc: this.cluster.vpc,
            internetFacing,
        };
        return new aws_elasticloadbalancingv2_1.NetworkLoadBalancer(this, name, lbProps);
    }
    createListener(name, lb, port) {
        return lb.addListener(name, {
            port,
        });
    }
    createDomainName(loadBalancer, name, zone) {
        if (typeof name !== 'undefined') {
            if (typeof zone === 'undefined') {
                throw new Error('A Route53 hosted domain zone name is required to configure the specified domain name');
            }
            new aws_route53_1.ARecord(this, `DNS${loadBalancer.node.id}`, {
                zone,
                recordName: name,
                target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.LoadBalancerTarget(loadBalancer)),
            });
        }
    }
}
exports.NetworkMultipleTargetGroupsServiceBase = NetworkMultipleTargetGroupsServiceBase;
//# sourceMappingURL=data:application/json;base64,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