"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SPADeploy = void 0;
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_route53_patterns_1 = require("@aws-cdk/aws-route53-patterns");
const aws_route53_targets_1 = require("@aws-cdk/aws-route53-targets");
const cdk = require("@aws-cdk/core");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const s3 = require("@aws-cdk/aws-s3");
class SPADeploy extends cdk.Construct {
    constructor(scope, id, config) {
        super(scope, id);
        if (typeof config !== 'undefined') {
            this.globalConfig = config;
        }
        else {
            this.globalConfig = {
                encryptBucket: false,
                ipFilter: false,
            };
        }
    }
    /**
     * Helper method to provide a configured s3 bucket
     */
    getS3Bucket(config, isForCloudFront) {
        const bucketConfig = {
            websiteIndexDocument: config.indexDoc,
            websiteErrorDocument: config.errorDoc,
            publicReadAccess: true,
        };
        if (this.globalConfig.encryptBucket === true) {
            bucketConfig.encryption = s3.BucketEncryption.S3_MANAGED;
        }
        if (this.globalConfig.ipFilter === true || isForCloudFront === true) {
            bucketConfig.publicReadAccess = false;
        }
        const bucket = new s3.Bucket(this, 'WebsiteBucket', bucketConfig);
        if (this.globalConfig.ipFilter === true && isForCloudFront === false) {
            if (typeof this.globalConfig.ipList === 'undefined') {
                this.node.addError('When IP Filter is true then the IP List is required');
            }
            const bucketPolicy = new aws_iam_1.PolicyStatement();
            bucketPolicy.addAnyPrincipal();
            bucketPolicy.addActions('s3:GetObject');
            bucketPolicy.addResources(`${bucket.bucketArn}/*`);
            bucketPolicy.addCondition('IpAddress', {
                'aws:SourceIp': this.globalConfig.ipList,
            });
            bucket.addToResourcePolicy(bucketPolicy);
        }
        return bucket;
    }
    /**
     * Helper method to provide configuration for cloudfront
     */
    getCFConfig(websiteBucket, config, accessIdentity, cert) {
        const cfConfig = {
            originConfigs: [
                {
                    s3OriginSource: {
                        s3BucketSource: websiteBucket,
                        originAccessIdentity: accessIdentity,
                    },
                    behaviors: [{ isDefaultBehavior: true }],
                },
            ],
            // We need to redirect all unknown routes back to index.html for angular routing to work
            errorConfigurations: [{
                    errorCode: 403,
                    responsePagePath: (config.errorDoc ? `/${config.errorDoc}` : `/${config.indexDoc}`),
                    responseCode: 200,
                },
                {
                    errorCode: 404,
                    responsePagePath: (config.errorDoc ? `/${config.errorDoc}` : `/${config.indexDoc}`),
                    responseCode: 200,
                }],
        };
        if (typeof config.certificateARN !== 'undefined' && typeof config.cfAliases !== 'undefined') {
            cfConfig.aliasConfiguration = {
                acmCertRef: config.certificateARN,
                names: config.cfAliases,
            };
        }
        if (typeof config.zoneName !== 'undefined' && typeof cert !== 'undefined') {
            cfConfig.viewerCertificate = aws_cloudfront_1.ViewerCertificate.fromAcmCertificate(cert, {
                aliases: [config.zoneName],
            });
        }
        return cfConfig;
    }
    /**
     * Basic setup needed for a non-ssl, non vanity url, non cached s3 website
     */
    createBasicSite(config) {
        const websiteBucket = this.getS3Bucket(config, false);
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
        });
        const cfnOutputConfig = {
            description: 'The url of the website',
            value: websiteBucket.bucketWebsiteUrl,
        };
        if (config.exportWebsiteUrlOutput === true) {
            if (typeof config.exportWebsiteUrlName === 'undefined' || config.exportWebsiteUrlName === '') {
                this.node.addError('When Output URL as AWS Export property is true then the output name is required');
            }
            cfnOutputConfig.exportName = config.exportWebsiteUrlName;
        }
        new cdk.CfnOutput(this, 'URL', cfnOutputConfig);
        return { websiteBucket };
    }
    /**
     * This will create an s3 deployment fronted by a cloudfront distribution
     * It will also setup error forwarding and unauth forwarding back to indexDoc
     */
    createSiteWithCloudfront(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            // Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution,
            distributionPaths: ['/', `/${config.indexDoc}`],
        });
        new cdk.CfnOutput(this, 'cloudfront domain', {
            description: 'The domain of the website',
            value: distribution.domainName,
        });
        return { websiteBucket };
    }
    /**
     * S3 Deployment, cloudfront distribution, ssl cert and error forwarding auto
     * configured by using the details in the hosted zone provided
     */
    createSiteFromHostedZone(config) {
        const websiteBucket = this.getS3Bucket(config, true);
        const zone = aws_route53_1.HostedZone.fromLookup(this, 'HostedZone', { domainName: config.zoneName });
        const cert = new aws_certificatemanager_1.DnsValidatedCertificate(this, 'Certificate', {
            hostedZone: zone,
            domainName: config.zoneName,
            region: 'us-east-1',
        });
        const accessIdentity = new aws_cloudfront_1.OriginAccessIdentity(this, 'OriginAccessIdentity', { comment: `${websiteBucket.bucketName}-access-identity` });
        const distribution = new aws_cloudfront_1.CloudFrontWebDistribution(this, 'cloudfrontDistribution', this.getCFConfig(websiteBucket, config, accessIdentity, cert));
        new s3deploy.BucketDeployment(this, 'BucketDeployment', {
            sources: [s3deploy.Source.asset(config.websiteFolder)],
            destinationBucket: websiteBucket,
            // Invalidate the cache for / and index.html when we deploy so that cloudfront serves latest site
            distribution,
            distributionPaths: ['/', `/${config.indexDoc}`],
        });
        new aws_route53_1.ARecord(this, 'Alias', {
            zone,
            recordName: config.zoneName,
            target: aws_route53_1.RecordTarget.fromAlias(new aws_route53_targets_1.CloudFrontTarget(distribution)),
        });
        new aws_route53_patterns_1.HttpsRedirect(this, 'Redirect', {
            zone,
            recordNames: [`www.${config.zoneName}`],
            targetDomain: config.zoneName,
        });
        return { websiteBucket };
    }
}
exports.SPADeploy = SPADeploy;
//# sourceMappingURL=data:application/json;base64,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