
from scipy.interpolate import griddata
import astropy.units as u
import numba
import pandas as pd
import numpy as np
import os
import sys

from .relations import  scale_to_local_lf, teff_to_spt_kirkpatrick, \
spt_to_teff_kirkpatrick, teff_to_spt_pecaut, spt_to_teff_pecaut
from .core_tools import sample_from_powerlaw, fast_2d_interpolation
from .evol_models import EVOL_MODELS

#DATA_FOLDER=os.environ['POPSIMS_DATA_FOLDER']

#CODE_FOLDER=os.path.abspath(os.path.join(os.path.dirname(__file__),os.pardir))
#sys.path.append(CODE_FOLDER)
#DATA_FOLDER= CODE_FOLDER+'/data/'

def _read_bintemplates():
    from .binaries import BINARIES 
    #must be a pickle file with columns prim, sec, sys all floats
    tbl=pd.DataFrame(BINARIES)
    return [tbl.prim.values, tbl.sec.values, tbl.sys.values]


def get_system_type(pr, sc):
    """
    Determine the type of a system based on its primary and secondary attributes.

    This function takes the primary and secondary attributes of a system and
    uses interpolation to determine the type of the system based on these
    attributes. The system type is returned as a floating point value.

    Args:
        pr: The primary attribute of the system, as a floating point value.
        sc: The secondary attribute of the system, as a floating point value.

    Returns:
        The type of the system, as a floating point value.

    #generated by chatGPT
    """
    # Function implementation goes here

    interpolators= _read_bintemplates()
    #where secondary are nans set to primaries
    sc[np.isnan(sc)]=pr[np.isnan(sc)]
    #interpolate
    interpoints=np.array([interpolators[0], interpolators[1] ]).T
    comb=griddata(interpoints, interpolators[-1] , (pr, sc), method='linear')
    #rember to assign <15 =primary and > 39= primary
    return comb

# Cache for DataFrames
EVOL_MODEL_CACHE = {}
#print (EVOL_MODEL_CACHE)
def evolutionary_model_interpolator(mass, age, model, subset=None):
    # Check if DataFrame is in the cache, create it otherwise
    if model not in EVOL_MODEL_CACHE.keys():
        evol_model_data = EVOL_MODELS[model]
        EVOL_MODEL_CACHE[model] = evol_model_data
    else:
        evol_model_data = EVOL_MODEL_CACHE[model]

    # Filter data if subset is provided --> pending issue (hybrid cloud not working)
    #if subset is not None:
    #    column_index = EVOL_MODELS[model].index(subset[0])
    #    evol_model_data = evol_model_data[evol_model_data[:, column_index] == subset[1]]
    #print (model)
    #print (evol_model_data.keys())
    values_t = np.log10(evol_model_data['temperature'])  # Temperature values 'age', 'mass', 'luminosity', 'temperature', 'gravity', 'radius'
    values_lumn = evol_model_data['luminosity']         # Luminosity values
    values_m = np.log10(evol_model_data['mass'])  # Mass values
    values_ag = np.log10(evol_model_data['age']) # Age values

    evol_points = np.column_stack((values_m, values_ag))

    teffs = griddata(evol_points, values_t, (np.log10(mass), np.log10(age)), method='linear')
    lumn = griddata(evol_points, values_lumn, (np.log10(mass), np.log10(age)), method='linear')

    return {'mass': mass * u.Msun, 'age': age * u.Gyr, 'temperature': 10**teffs * u.Kelvin, 'luminosity': lumn * u.Lsun}

#need an evolutionary model class that automatically does the interpolations across mass, age and metallicity upon intialization
class EvolutionaryModel:
    def __init__(self,  dataframe):
        #initialize model 
        req_columns=['mass', 'age']
        assert all(column in dataframe.columns for column in req_columns), "DataFrame is missing required columns"
        self.columns=np.array(dataframe.columns)
        self.data = dataframe[self.columns].values

    def interpolate(self, x_axis, y_axis, x_values, y_values, \
                    logscale=['mass', 'age', 'temperature'], interp_columns=['temperature']):
        
        assert x_axis in self.columns, f"x_axis '{x_axis}' not found in DataFrame columns"
        assert y_axis in self.columns, f"y_axis '{y_axis}' not found in DataFrame columns"
        
        #make a copy of data 
        #put stuff on logscale
        dt= self.data.copy()
        for c in logscale:
            col_idx = np.where(self.columns == c)[0][0]
            dt[:, col_idx] =np.log10(np.array(dt[:, col_idx]).astype(float))
            
        points = dt[:, [np.where(self.columns == x_axis)[0][0], np.where(self.columns == y_axis)[0][0]]]
        remaining_columns = [col for col in interp_columns if col not in [x_axis, y_axis]]
        results = {}
        
        for col in remaining_columns:
            col_idx = np.where(self.columns == col)[0][0]
            values =dt[:, col_idx]
            #result = np.empty_like(y_values)
            #interp_values= interpolate_2d(points, values, x_values, y_values, result)
            results[col] = fast_2d_interpolation(points, values, x_values, y_values)
            
        interp_df= pd.DataFrame(results)
        interp_df[x_axis]=x_values
        interp_df[y_axis]=y_values
        return interp_df