# Graphdoc - Generate docs for your GraphQL API

Graphdoc uses your GraphQL schema to generate an HTML page documenting your API.
It works with Graphene and Ariadne and any other framework generating `GraphQLSchema`
instances from the `graphql-core` package.

You can see an example of autogenerated docs using the [SWAPI](https://swapi.dev/)
schema [in this page](https://wallee94.github.io/graphdoc/)

## Installation

Install using pip:

    pip install graphdoc

## Usage

Use `graphdoc.to_doc` to create HTML documentation 
from a `GraphQLSchema` instance or a string with your schema in SDL.
Use `graphdoc.to_md` to create Markdown documentation.

## Examples

### Django and Graphene

```python
# view.py
from django.http import HttpResponse
from graphene_django.views import GraphQLView
import graphdoc

def graphql_docs(request):
    # For graphene>=3 use schema.graphql_schema
    # html = graphdoc.to_doc(GraphQLView().schema.graphql_schema)
    html = graphdoc.to_doc(GraphQLView().schema)
    return HttpResponse(html, content_type='text/html')
    

# urls.py
from graphene_django.views import GraphQLView
from .views import graphql_docs

urlpatterns = [
    path('graphql', GraphQLView.as_view(), name='graphql_endpoint'),
    path('docs', graphql_docs, name='graphql_docs'),
]
```

### FastAPI and Ariadne

```python
# schema.py
from ariadne import load_schema_from_path

schema = load_schema_from_path("schema.graphql")


# main.py
from fastapi import FastAPI, Response
from schema import schema
import graphdoc

app = FastAPI()


@app.get("/docs")
async def graphql_docs():
    html = graphdoc.to_doc(schema)
    return Response(content=html, media_type="text/html")
```

## API

#### `graphdoc.to_doc(schema, templates_folder=None, context=None)`

**schema**: `GraphQLSchema` instance or a str with a schema in SDL.

**templates_folder**: Optional path to a templates folder. If received,
Jinja will look for templates in this folder before falling back to the defaults
[here](https://github.com/wallee94/graphdoc/tree/main/graphdoc/templates).

**context**: An optional custom dictionary to be passed to the templates variables.

## Custom templates

You can find more example apps using custom templates and documenting multiple schemas 
in the [examples folder](https://github.com/wallee94/graphdoc/tree/main/examples)