"""more fixtures"""
import pytest
import pandas as pd
import scanpy as sc
import scipy.sparse


@pytest.fixture
def database_paths_mouse():
    """Database paths for tests."""
    db = {
        "igblast_db": "container/database/igblast/",
        "germline": "container/database/germlines/imgt/mouse/vdj/",
        "blastdb": "container/database/blast/mouse/",
        "blastdb_fasta": "container/database/blast/mouse/mouse_BCR_C.fasta",
    }
    return db


@pytest.fixture
def dummy_adata_mouse():
    """Dummy anndata for tests."""
    barcodes = [
        "AAACGGGAGTTACCCA-1",
        "AAAGATGGTCTAACGT-1",
        "AAAGATGGTTACGACT-1",
        "AAAGTAGCAAAGTGCG-1",
        "AAAGTAGCAAATCCGT-1",
        "AAATGCCGTACCGGCT-1",
        "AACACGTGTTCGGGCT-1",
        "AACCGCGAGAAACGCC-1",
        "AACCGCGCAAGGTTTC-1",
        "AACGTTGGTATTCGTG-1",
        "AACGTTGGTTATGCGT-1",
        "AACGTTGTCTGCAAGT-1",
        "AACTCAGCACATGACT-1",
        "AACTCAGCACGGTAGA-1",
        "AACTCAGGTAGAGCTG-1",
        "AACTCAGTCCAGAGGA-1",
        "AACTCCCTCAGCACAT-1",
        "AACTCCCTCTTCATGT-1",
        "AACTGGTAGGACTGGT-1",
        "AACTTTCCAGCGAACA-1",
        "AACTTTCGTAGCTGCC-1",
        "AAGCCGCCAGCTCCGA-1",
        "AAGGAGCGTAAAGTCA-1",
        "AAGGAGCGTCATATGC-1",
        "AAGGCAGCACGGTTTA-1",
        "AAGGTTCTCTTGAGGT-1",
        "AAGTCTGGTATAGGGC-1",
        "AAGTCTGTCCATGAAC-1",
        "AATCCAGAGATGCGAC-1",
        "AATCCAGGTCAACATC-1",
        "AATCGGTCAGGATTGG-1",
        "AATCGGTCATACAGCT-1",
        "ACACCAAGTTCGGGCT-1",
        "ACACCCTAGGAATCGC-1",
        "ACACCCTTCAGCTCGG-1",
        "ACAGCCGCACCGCTAG-1",
        "ACAGCCGCACGAAAGC-1",
        "ACAGCCGGTAGCGTAG-1",
        "ACAGCCGGTCCGAGTC-1",
        "ACATGGTAGATGCCTT-1",
        "ACCAGTATCAATACCG-1",
        "ACCGTAACAATCCGAT-1",
        "ACCGTAAGTGTAACGG-1",
        "ACCGTAATCTCTGCTG-1",
        "ACCTTTATCCTAGGGC-1",
        "ACGAGCCAGCCGATTT-1",
        "ACGAGCCCACAAGTAA-1",
        "ACGAGCCCATGGTTGT-1",
        "ACGAGCCGTCATATCG-1",
        "ACGAGCCGTCCAAGTT-1",
        "ACGAGCCTCTGCAAGT-1",
        "ACGATACCACGAAAGC-1",
        "ACGATACCATCGGACC-1",
        "ACGCAGCTCACGGTTA-1",
        "ACGCCAGCACCCTATC-1",
        "ACGCCAGGTACCGGCT-1",
        "ACGCCAGGTGTCAATC-1",
        "ACGCCAGTCGTGGTCG-1",
        "ACGTCAACACCACCAG-1",
        "ACGTCAAGTCTCCACT-1",
        "ACGTCAAGTGTGACCC-1",
        "ACTATCTAGAGGTTAT-1",
        "ACTATCTCAGTGGAGT-1",
        "ACTGAACAGGACGAAA-1",
        "ACTGAACGTACCGGCT-1",
        "ACTGAGTCATGGTCAT-1",
        "ACTGATGCAGCAGTTT-1",
        "ACTGCTCGTACCGCTG-1",
        "ACTGTCCTCGGAAATA-1",
        "AGAATAGCAGTATGCT-1",
        "AGAATAGTCTCGATGA-1",
        "AGAGCGAAGGTGCTTT-1",
        "AGAGCTTCAATCTACG-1",
        "AGAGTGGCACAGGCCT-1",
        "AGAGTGGTCTCCAACC-1",
        "AGCAGCCCATCGGACC-1",
        "AGCAGCCGTGGTGTAG-1",
        "AGCATACGTAGCGTAG-1",
        "AGCATACGTCCAGTGC-1",
        "AGCATACTCTCCGGTT-1",
        "AGCCTAAAGTACCGGA-1",
        "AGCCTAACACTGTCGG-1",
        "AGCCTAACATCGATGT-1",
        "AGCCTAAGTGTTTGTG-1",
        "AGCCTAATCTGAAAGA-1",
        "AGCGTCGGTCTGATCA-1",
        "AGCTCCTTCCGCTGTT-1",
        "AGCTCCTTCTGGCGTG-1",
        "AGGCCACCATAGACTC-1",
        "AGGCCGTAGATAGCAT-1",
        "AGGCCGTAGATCCGAG-1",
        "AGGGAGTAGTGTGAAT-1",
        "AGGGAGTCAAGTAATG-1",
        "AGGGAGTTCATCACCC-1",
        "AGGTCATCACGGTGTC-1",
        "AGGTCATGTAGTGAAT-1",
        "AGTGAGGGTTGTCTTT-1",
        "ATAACGCGTATAAACG-1",
        "ATAAGAGGTACCGTAT-1",
        "ATCATCTAGTGGGCTA-1",
        "ATCATCTGTGCTCTTC-1",
        "ATCCACCCACCGTTGG-1",
        "ATCCGAAAGGACAGCT-1",
        "ATCCGAAGTCGTGGCT-1",
        "ATCGAGTGTCATATCG-1",
        "ATCTGCCAGGAATCGC-1",
        "ATGGGAGGTCGTCTTC-1",
        "ATGTGTGAGCCCAGCT-1",
        "ATGTGTGAGCTAAGAT-1",
        "ATTACTCGTACTCAAC-1",
        "ATTCTACAGTCTCCTC-1",
        "ATTCTACAGTGCGATG-1",
        "ATTGGACAGCAATCTC-1",
        "ATTGGTGCAGACAAGC-1",
        "ATTTCTGTCCCGGATG-1",
        "CAACCAAGTTGGGACA-1",
        "CAACCTCCATGCTAGT-1",
        "CAAGAAACACGAAACG-1",
        "CAAGATCCAGCGTAAG-1",
        "CAAGATCCATACCATG-1",
        "CAAGTTGGTATAGGGC-1",
        "CACAAACCAACGATGG-1",
        "CACACAAAGCATGGCA-1",
        "CACACAACAGTTTACG-1",
        "CACAGGCGTCTCCACT-1",
        "CACAGGCGTTGGTTTG-1",
        "CACAGTAAGGCGATAC-1",
        "CACATAGAGATTACCC-1",
        "CACATTTAGCTCTCGG-1",
        "CACCAGGAGGTGACCA-1",
        "CACCTTGCAGTTCATG-1",
        "CAGAATCGTGTTCGAT-1",
        "CAGATCAGTCTGGTCG-1",
        "CAGCAGCTCCAAGTAC-1",
        "CAGCAGCTCCTATTCA-1",
        "CAGCATAGTTGCTCCT-1",
        "CAGCATATCCCTAACC-1",
        "CAGCCGATCTACTATC-1",
        "CAGCGACAGTGTTGAA-1",
        "CAGCTAAAGTCCAGGA-1",
        "CAGCTAACAAACGCGA-1",
        "CAGCTGGCACATCTTT-1",
        "CAGCTGGGTTCGGCAC-1",
        "CAGGTGCCACCTCGGA-1",
        "CAGTAACCAAACCTAC-1",
        "CAGTAACCATCCCATC-1",
        "CAGTAACGTTATGTGC-1",
        "CATATGGCAGCTTAAC-1",
        "CATATTCCAGTCACTA-1",
        "CATATTCTCTGTCTAT-1",
        "CATCAAGCACCTTGTC-1",
        "CATCAAGTCTGAAAGA-1",
        "CATCAGAAGCACAGGT-1",
        "CATCAGACAGGACGTA-1",
        "CATCGGGCAAATACAG-1",
        "CATCGGGCACTGTGTA-1",
        "CATGACAAGCCAGGAT-1",
        "CATGACACACGAAGCA-1",
        "CATGACATCAGTGCAT-1",
        "CATGGCGGTGTCGCTG-1",
        "CATTATCGTGCCTGCA-1",
        "CCAATCCTCAAACAAG-1",
        "CCACGGAGTTATCCGA-1",
        "CCAGCGAAGTCCGGTC-1",
        "CCAGCGATCCTAGAAC-1",
        "CCATGTCTCACCTTAT-1",
        "CCATTCGAGTGTGAAT-1",
        "CCATTCGCATCAGTAC-1",
        "CCATTCGGTACCTACA-1",
        "CCATTCGTCAACTCTT-1",
        "CCCAATCAGTGGTAGC-1",
        "CCCAATCCAAGCGTAG-1",
        "CCCAATCGTGTGTGCC-1",
        "CCCAGTTTCATACGGT-1",
        "CCGGGATGTCTTCTCG-1",
        "CCGGGATTCTTGTCAT-1",
        "CCGGTAGTCACTCTTA-1",
        "CCGTACTCACGACTCG-1",
        "CCGTACTTCACAAACC-1",
        "CCGTTCAAGAGCTGCA-1",
        "CCGTTCACACTCGACG-1",
        "CCGTTCAGTTATGTGC-1",
        "CCTAAAGTCGACAGCC-1",
        "CCTACCAGTTTAGGAA-1",
        "CCTACCATCTTTAGTC-1",
        "CCTAGCTCACGAAATA-1",
        "CCTCAGTGTGAAATCA-1",
        "CCTCTGAAGAGGTAGA-1",
        "CCTCTGAGTTGTGGAG-1",
        "CCTTACGAGGAGTTTA-1",
        "CCTTCGAAGCACCGTC-1",
        "CCTTTCTAGATGTGGC-1",
        "CCTTTCTTCCGTTGCT-1",
        "CGAACATAGTCATGCT-1",
        "CGAACATCATTGCGGC-1",
        "CGACTTCAGTGGCACA-1",
        "CGACTTCGTCTTTCAT-1",
        "CGACTTCGTTTGACTG-1",
        "CGAGCACCAAATACAG-1",
        "CGAGCACGTCATGCCG-1",
        "CGAGCACGTTCAGTAC-1",
        "CGAGCCACACGGCTAC-1",
        "CGAGCCACAGCAGTTT-1",
        "CGAGCCACATGGTAGG-1",
        "CGAGCCATCTGCCAGG-1",
        "CGATGGCCAAGCCGTC-1",
        "CGATGTAGTAACGCGA-1",
        "CGCCAAGAGTCGCCGT-1",
        "CGCTATCCACCGGAAA-1",
        "CGCTTCACATCCCATC-1",
        "CGGACGTGTCATTAGC-1",
        "CGGACTGCACATCCAA-1",
        "CGGACTGGTTTGACTG-1",
        "CGGACTGTCACATAGC-1",
        "CGGACTGTCACTCTTA-1",
        "CGGAGCTAGGCATTGG-1",
        "CGGAGCTCACACGCTG-1",
        "CGGCTAGCACGACTCG-1",
        "CGGCTAGCAGGAATGC-1",
        "CGGCTAGCATATGAGA-1",
        "CGGCTAGCATTTCACT-1",
        "CGGCTAGGTGACTACT-1",
        "CGGGTCATCAGGCAAG-1",
        "CGGTTAAGTATATCCG-1",
        "CGGTTAATCTGCCAGG-1",
        "CGTAGGCCACTTAACG-1",
        "CGTAGGCTCGTCCAGG-1",
        "CGTCACTAGGGTGTGT-1",
        "CGTCACTCAAGAGGCT-1",
        "CGTCAGGAGCGAGAAA-1",
        "CGTCAGGTCTGCAAGT-1",
        "CGTCCATAGGTTCCTA-1",
        "CGTGAGCGTACCGAGA-1",
        "CGTGTCTAGCTGATAA-1",
        "CGTTCTGAGCACAGGT-1",
        "CGTTCTGCACGTCAGC-1",
        "CTAACTTGTTCCATGA-1",
        "CTAAGACAGGCGCTCT-1",
        "CTACATTCACTAGTAC-1",
        "CTACATTGTATTCTCT-1",
        "CTACCCACAACTGCTA-1",
        "CTACCCACACCTTGTC-1",
        "CTACGTCGTGCACTTA-1",
        "CTACGTCGTTACAGAA-1",
        "CTACGTCGTTCCGTCT-1",
        "CTAGAGTGTCACTGGC-1",
        "CTAGCCTCATTCCTCG-1",
        "CTAGTGATCTGCTGCT-1",
        "CTCACACCAGATAATG-1",
        "CTCATTAGTTAAGTAG-1",
        "CTCGAAAGTCGTTGTA-1",
        "CTCGAAAGTTCGGGCT-1",
        "CTCGGAGCAATGGAAT-1",
        "CTCGGAGCACCAGTTA-1",
        "CTCGGAGGTATTAGCC-1",
        "CTCGGGAGTGACGGTA-1",
        "CTCGTACAGACACGAC-1",
        "CTCGTACCAATCAGAA-1",
        "CTGAAACTCCTCATTA-1",
        "CTGATAGCATGGATGG-1",
        "CTGATAGGTCTTTCAT-1",
        "CTGATCCAGTAGCGGT-1",
        "CTGATCCCAGAAGCAC-1",
        "CTGATCCGTACTTCTT-1",
        "CTGCCTAGTGGGTCAA-1",
        "CTGCCTATCAAACAAG-1",
        "CTGCGGAAGCCAGTTT-1",
        "CTGCGGATCACCGTAA-1",
        "CTGGTCTTCCGCGGTA-1",
        "CTGTGCTAGCCCAGCT-1",
        "CTGTGCTAGTACATGA-1",
        "CTGTGCTAGTGTCCCG-1",
        "CTTAACTAGGGCACTA-1",
        "CTTAACTCACTGTTAG-1",
        "CTTAACTTCATCACCC-1",
        "CTTACCGGTCTAGTCA-1",
        "CTTACCGGTGGCCCTA-1",
        "CTTAGGAAGATAGTCA-1",
        "CTTAGGAAGTGACATA-1",
        "CTTTGCGGTCATCGGC-1",
        "CTTTGCGTCTTGCAAG-1",
        "GAAATGAAGGTGCAAC-1",
        "GAACATCAGACCTAGG-1",
        "GAACATCGTGACCAAG-1",
        "GAACCTAAGATCGGGT-1",
        "GAACCTAGTGGTCTCG-1",
        "GAACGGAGTCGGGTCT-1",
        "GAAGCAGCACCAGTTA-1",
        "GAAGCAGCATGGTCAT-1",
        "GAAGCAGTCGGTCCGA-1",
        "GACCAATCATCGTCGG-1",
        "GACCTGGAGATCTGCT-1",
        "GACGCGTCAGTAGAGC-1",
        "GACGGCTTCACATAGC-1",
        "GACGGCTTCGCCGTGA-1",
        "GACTAACCATGCCTTC-1",
        "GACTAACGTGTGGTTT-1",
        "GACTACAAGTTTCCTT-1",
        "GACTACACAGGATTGG-1",
        "GACTACATCATCATTC-1",
        "GACTACATCTGCGTAA-1",
        "GACTGCGGTACAGTGG-1",
        "GACTGCGGTCGTGGCT-1",
        "GACTGCGTCCTATTCA-1",
        "GAGCAGATCAAAGTAG-1",
        "GAGCAGATCTGGCGTG-1",
        "GAGGTGATCCGTTGTC-1",
        "GATCAGTGTCCAACTA-1",
        "GATCAGTTCGGGAGTA-1",
        "GATCGATGTCCTAGCG-1",
        "GATCGCGGTCGGCTCA-1",
        "GATCGCGTCTATCCCG-1",
        "GATCTAGCATTGCGGC-1",
        "GATCTAGGTCACCCAG-1",
        "GATGAAATCCTGCTTG-1",
        "GATGAGGGTTACCAGT-1",
        "GATGAGGTCCCTAACC-1",
        "GATGCTACACTAAGTC-1",
        "GATGCTACATGTTCCC-1",
        "GATGCTAGTTGTCTTT-1",
        "GATTCAGCACTCTGTC-1",
        "GATTCAGTCTACTATC-1",
        "GATTCAGTCTTGCCGT-1",
        "GCAAACTCAAACGCGA-1",
        "GCAAACTCATGCAATC-1",
        "GCAAACTTCGCTTAGA-1",
        "GCAATCAGTGGCGAAT-1",
        "GCACATAGTAGAGTGC-1",
        "GCACTCTAGGTCATCT-1",
        "GCACTCTCAGGGATTG-1",
        "GCACTCTTCCCATTTA-1",
        "GCAGCCACACTTACGA-1",
        "GCAGCCAGTGTGCGTC-1",
        "GCAGCCATCTACTCAT-1",
        "GCAGTTAAGTCATGCT-1",
        "GCAGTTAGTCTAGTCA-1",
        "GCATACAAGACCTTTG-1",
        "GCATACAAGGCAGTCA-1",
        "GCATGATAGACAGGCT-1",
        "GCATGATAGCTAACAA-1",
        "GCATGATAGGGCATGT-1",
        "GCATGCGGTATATGAG-1",
        "GCATGCGTCGCATGGC-1",
        "GCCAAATGTGACCAAG-1",
        "GCCAAATTCCTGTAGA-1",
        "GCCTCTAGTAGAGGAA-1",
        "GCGACCAGTTGGTGGA-1",
        "GCGAGAAAGACACTAA-1",
        "GCGAGAACAGTCGATT-1",
        "GCGAGAAGTCGAACAG-1",
        "GCGAGAATCGTATCAG-1",
        "GCGCAACAGTGTGGCA-1",
        "GCGCAGTAGTTGTAGA-1",
        "GCGCAGTCAGCGAACA-1",
        "GCGCCAAAGAGTGAGA-1",
        "GCGCCAAGTTACTGAC-1",
        "GCGCGATCACAACGTT-1",
        "GCGCGATTCAACGGGA-1",
        "GCGGGTTTCGCACTCT-1",
        "GCTCTGTCAGGTTTCA-1",
        "GCTCTGTGTCTAGCGC-1",
        "GCTGCAGAGCACCGCT-1",
        "GCTGCGAGTCACAAGG-1",
        "GCTGCTTGTCAGATAA-1",
        "GCTGGGTGTCCGAGTC-1",
        "GCTGGGTTCGTGGACC-1",
        "GCTTCCAGTTATCACG-1",
        "GCTTGAAGTAACGACG-1",
        "GGAAAGCCATACCATG-1",
        "GGAAAGCGTCCTGCTT-1",
        "GGAACTTGTAAACCTC-1",
        "GGAACTTTCTGGAGCC-1",
        "GGAATAACAAGGTGTG-1",
        "GGAATAATCGACCAGC-1",
        "GGACAAGCAGTATGCT-1",
        "GGACAGAAGAACAACT-1",
        "GGATGTTGTATCGCAT-1",
        "GGCCGATCATGTTCCC-1",
        "GGCGACTAGCGATATA-1",
        "GGCGACTGTCCTAGCG-1",
        "GGCTCGATCCCATTAT-1",
        "GGCTGGTCAACACCTA-1",
        "GGCTGGTCATCACCCT-1",
        "GGGAATGGTAAGCACG-1",
        "GGGACCTCAGCCACCA-1",
        "GGGAGATTCACTTCAT-1",
        "GGGCACTAGGAGTAGA-1",
        "GGGCACTTCACCGTAA-1",
        "GGGTTGCGTACAGCAG-1",
        "GGTATTGAGAGTACAT-1",
        "GGTATTGAGCTTTGGT-1",
        "GGTGCGTTCTCGATGA-1",
        "GGTGTTATCCGTAGGC-1",
        "GTAACGTCAGGGTATG-1",
        "GTAACTGAGGATTCGG-1",
        "GTAACTGGTAGGGTAC-1",
        "GTAACTGGTTGGGACA-1",
        "GTACGTACAAGCGATG-1",
        "GTAGTCATCTCCTATA-1",
        "GTATCTTAGTGACTCT-1",
        "GTCAAGTAGGCATTGG-1",
        "GTCAAGTGTAAGTGGC-1",
        "GTCAAGTGTTCCACAA-1",
        "GTCAAGTTCGACCAGC-1",
        "GTCAAGTTCGGAAACG-1",
        "GTCACAAAGCTAGTGG-1",
        "GTCACAACATTTGCCC-1",
        "GTCATTTGTTAAGACA-1",
        "GTCCTCAGTCATCCCT-1",
        "GTCCTCATCAGCCTAA-1",
        "GTCGTAAAGACACGAC-1",
        "GTCTCGTGTCTCCACT-1",
        "GTCTCGTTCTACCTGC-1",
        "GTCTTCGTCTAACTGG-1",
        "GTGCAGCCACAACTGT-1",
        "GTGCGGTAGACGCAAC-1",
        "GTGCTTCAGGGCTTGA-1",
        "GTGCTTCCAGGTCGTC-1",
        "GTGCTTCGTTACCAGT-1",
        "GTGGGTCTCGGCCGAT-1",
        "GTGTGCGCAGTAGAGC-1",
        "GTTAAGCCAGGTTTCA-1",
        "GTTAAGCTCGCCGTGA-1",
        "GTTCATTGTTCACGGC-1",
        "GTTTCTAAGACAAAGG-1",
        "GTTTCTACAGCTCGCA-1",
        "TAAACCGAGTTCGATC-1",
        "TAAACCGGTCGACTGC-1",
        "TAAACCGTCATGCATG-1",
        "TAAGCGTAGGATGGTC-1",
        "TAAGCGTTCACTGGGC-1",
        "TACAGTGAGATGCCTT-1",
        "TACAGTGGTAAACACA-1",
        "TACAGTGGTTATGTGC-1",
        "TACCTTACATACGCTA-1",
        "TACCTTAGTACTTGAC-1",
        "TACGGATAGACTACAA-1",
        "TACGGGCTCACCTCGT-1",
        "TACGGGCTCGTATCAG-1",
        "TACGGGCTCGTCGTTC-1",
        "TACTCATAGTGTGGCA-1",
        "TACTCATCATCTACGA-1",
        "TACTCATTCATTCACT-1",
        "TACTCGCAGGATGTAT-1",
        "TACTCGCCATCTATGG-1",
        "TACTTACGTACGCACC-1",
        "TAGACCATCGCGCCAA-1",
        "TAGACCATCTGTCCGT-1",
        "TAGGCATGTTACGGAG-1",
        "TAGTGGTCATCACAAC-1",
        "TAGTGGTTCAGCGATT-1",
        "TAGTTGGTCTGGTATG-1",
        "TATCAGGTCACAATGC-1",
        "TATCTCAGTATCGCAT-1",
        "TATGCCCAGGCCCTCA-1",
        "TATGCCCAGTCAAGGC-1",
        "TATGCCCGTGTGGCTC-1",
        "TATGCCCTCAAGAAGT-1",
        "TATTACCGTAAGGGAA-1",
        "TATTACCGTTGGTAAA-1",
        "TCAATCTAGCACAGGT-1",
        "TCAGCAATCTTTAGTC-1",
        "TCAGCTCAGAGTGACC-1",
        "TCAGCTCGTCGCGAAA-1",
        "TCAGGATGTAGAAGGA-1",
        "TCATTACTCTTGTATC-1",
        "TCATTTGAGGGCTTCC-1",
        "TCCACACAGCCTCGTG-1",
        "TCCCGATTCAACTCTT-1",
        "TCGAGGCAGTGTGGCA-1",
        "TCGCGTTAGGAATTAC-1",
        "TCGCGTTCAGATCTGT-1",
        "TCGGGACTCCGTACAA-1",
        "TCGGTAACAATACGCT-1",
        "TCTATTGCAAGCCCAC-1",
        "TCTCATAGTCTCTTAT-1",
        "TCTCATATCTCCAGGG-1",
        "TCTCTAAAGGAATCGC-1",
        "TCTGAGAGTCACTGGC-1",
        "TCTGAGAGTTCTGGTA-1",
        "TCTGGAAGTAGCTAAA-1",
        "TGAAAGAAGATGGCGT-1",
        "TGACAACAGGCGTACA-1",
        "TGACAACGTCCGCTGA-1",
        "TGACGGCAGGACAGCT-1",
        "TGACGGCGTGTGACCC-1",
        "TGACGGCTCCTTGGTC-1",
        "TGACTAGAGTGGGCTA-1",
        "TGACTAGCAGATGGCA-1",
        "TGACTTTGTTAAGGGC-1",
        "TGAGAGGAGGTGCTTT-1",
        "TGAGCATTCGGATGGA-1",
        "TGAGCCGTCGTATCAG-1",
        "TGAGGGAGTCCGAGTC-1",
        "TGAGGGATCCGTTGTC-1",
        "TGATTTCTCTTTACAC-1",
        "TGCACCTGTTCCGGCA-1",
        "TGCCAAAGTCCAGTTA-1",
        "TGCCAAAGTGTAACGG-1",
        "TGCCCTACAGTGGAGT-1",
        "TGCCCTAGTAGAGGAA-1",
        "TGCGGGTGTAGGGACT-1",
        "TGCTACCGTACATCCA-1",
        "TGGACGCCACACATGT-1",
        "TGGCTGGAGCACCGCT-1",
        "TGGCTGGGTAGCCTAT-1",
        "TGGGAAGAGAGCAATT-1",
        "TGGGAAGCACGTCAGC-1",
        "TGGTTAGAGAGTACAT-1",
        "TGGTTCCAGATGTAAC-1",
        "TGGTTCCGTCTCCACT-1",
        "TGTGGTACATCCAACA-1",
        "TGTGGTATCACCCGAG-1",
        "TGTGTTTTCAGGCGAA-1",
        "TGTTCCGCACCTCGGA-1",
        "TGTTCCGGTGGTAACG-1",
        "TTAGTTCCAGATTGCT-1",
        "TTATGCTAGGCTCAGA-1",
        "TTATGCTCACTTCGAA-1",
        "TTATGCTGTCGAGATG-1",
        "TTCCCAGAGACTACAA-1",
        "TTCCCAGCAGGATCGA-1",
        "TTCGAAGAGCTAACAA-1",
        "TTCGAAGCATCCTTGC-1",
        "TTCGGTCCAAAGGAAG-1",
        "TTCTACACATGCCCGA-1",
        "TTCTCAAGTACCAGTT-1",
        "TTCTCCTAGGTGCTTT-1",
        "TTCTCCTAGTTGTCGT-1",
        "TTCTTAGAGCTAGTCT-1",
        "TTCTTAGCACATGACT-1",
        "TTCTTAGGTGATAAAC-1",
        "TTGAACGGTCATCCCT-1",
        "TTGACTTCAAGAAGAG-1",
        "TTGACTTCAATTCCTT-1",
        "TTGCCGTCATCCAACA-1",
        "TTGCCGTTCATCGATG-1",
        "TTGCCGTTCCACTCCA-1",
        "TTGCGTCCACTTACGA-1",
        "TTGGAACGTAAGGATT-1",
        "TTGGAACGTTTCGCTC-1",
        "TTGGAACTCCACGAAT-1",
        "TTGGCAATCACATGCA-1",
        "TTGTAGGAGCCCAACC-1",
        "TTTGGTTCATTTCACT-1",
        "TTTGTCACAAGCGTAG-1",
        "TTTGTCATCGCTTGTC-1",
    ]
    obs = pd.DataFrame(index=barcodes)
    obs["sample_id"] = "sample_test"

    n = obs.shape[0]

    # just create a random matrix
    adata = sc.AnnData(X=scipy.sparse.random(n, 100, format="csr"), obs=obs)

    # this is just to populate the neighbors slot
    sc.pp.neighbors(adata, use_rep="X", n_neighbors=10)

    return adata


@pytest.fixture
def fasta_10x_mouse():
    """Standard cellranger fasta file to test the preprocessing."""
    seq = {
        "GAACCTAAGATCGGGT-1_contig_8": "GGTTCACACCACTGGGGCAAGGACTGTGAATCCTGGAAGGATGAGGCAGAGCTCAGAGCTGGGCTGTTAGAAGCACAGTGACAAACAAAAAGAGCAGCAGTCTTCTTCACGCGGCTCCACATGTGAGTGTGGTCAGCTGAGAGACTGGAGAGAGCCACAGCCTAGGGAGAGCACTGGGCCTTTCCAGAACTAATAGATGGAAGATCACAGGAGGATGGGGAGGAGTTGGCAGATCAGCCCCCCAAAAAGGGCCTGTGTGTTGCAGAGAAAAGGTGCCTACCTGCAACCTGGTGCCAACCAGGAAGAGTCCAGAGTTCTCACACACAGAAGAATGGACCAGTAAGCTGTGCACACCTACAGACAACCAGACCAAAACAACACCCCCATCAGTCTATCCACTGGCCCCTGGGTGTGGAGATACAACTGGTTCCTCCGTGACTCTGGGATGCCTGGTCAAGGGCTACTTCCCTGAGTCAGTGACTGTGACTTGGAACTCTGGATCCCTGTCC",
        "AGGCCGTAGATCCGAG-1_contig_1": "GCAGCCAGTTTCTTATATGGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTCAGCTCCAGCTTGGTCCCAGCACCGAACGTGAGCGGA",
        "TGCCAAAGTCCAGTTA-1_contig_3": "GGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCTCAGAGGAAGGTGGAAACACGGTGAGAGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACCGCCGAAAACATAACACAATGACTAAAACCCAACCCAAAACCTATGGGGCCAGCACCTGGTCC",
        "TGAGAGGAGGTGCTTT-1_contig_11": "GAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGG",
        "CCATTCGTCAACTCTT-1_contig_5": "GGGGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TTCTTAGAGCTAGTCT-1_contig_2": "TGGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTGATTTCC",
        "ACACCCTTCAGCTCGG-1_contig_5": "ATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCGGGGTTACGACGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CACAGGCGTTGGTTTG-1_contig_1": "GGCAGTTCAGACATAGGAGCTTGGCTCTGGTTCCCAAGACCTCTCACTCACTTCTCAACATGGAGTGGGAACTGAGCTTAATTTTCATTTTTGCTCTTTTAAAAGATGTCCAGTGTGAAGTGCAGCTGTTGGAGACTGGAGGAGGCTTGGTGCAACCTGGGGGGTCACGGGGACTCTCTTGTGAAGGCTCAGGGTTTACTTTTAGTGGCTTCTGGATGAGCTGGGTTCGACAGACACCTGGGAAGACCCTGGAGTGGATTGGAGACATTAATTCTGATGGCAGTGCAATAAACTACGCACCATCCATAAAGGATCGATTCACTATCTTCAGAGACAATGACAAGAGCACCCTGTACCTGCAGATGAGCAATGTGCGATCTGAGGACACAGCCACGTATTTCTGTATGAGATATGATGGTTACTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "TACAGTGGTAAACACA-1_contig_6": "GGGGCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACT",
        "TGTGTTTTCAGGCGAA-1_contig_3": "AAAACAACATATGTCCAATGTCCTCTCCACAGACACTGAACACACTGACTCTAACCATGGGATGGAGCTGGATCTTTCTCTTCCTCCTGTCAGGAACTGCAGGCGTCCACTCTGAGGTCCAGCTTCAGCAGTCAGGACCTGAGCTGGTGAAACCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTTCTGGATACACATTCACTGACTACAACATGCACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGATATATTTATCCTTACAATGGTGGTACTGGCTACAACCAGAAGTTCAAGAGCAAGGCCACATTGACTGTAGACAATTCCTCCAGCACAGCCTACATGGAGCTCCGCAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGAAGGGGCTACGATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CTGCGGATCACCGTAA-1_contig_3": "GCAAATAGGGCCTCTTTCTCCTCATGAAACGCAGACCAACCTATCCTTGCAGTTCAGACATAGGAGCTTGGCTCTGGTTCCCAAGACCTCTCACTCACTTCTCAACATGGAGTGGGAACTGAGCTTAATTTTCATTTTTGCTCTTTTAAAAGATGTCCAGTGTGAAGTGCAGCTGTTGGAGACTGGAGGAGGCTTGGTGCAACCTGGGGGGTCACGGGGACTCTCTTGTGAAGGCTCAGGGTTTACTTTTAGTGGCTTCTGGATGAGCTGGGTTCGACAGACACCTGGGAAGACCCTGGAGTGGATTGGAGACATTAATTCTGATGGCAGTGCAATAAACTACGCACCATCCATAAAGGATCGATTCACTATCTTCAGAGACAATGACAAGAGCACCCTGTACCTGCAGATGAGCAATGTGCGATCTGAGGACACAGCCACGTATTTCTGTATGAGATATGGTAACTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGGACTGGTTTGACTG-1_contig_1": "TGGGGAGCTGCCAGGAGCCTAAGAAGCATCCTCTCATCTAGTTCTCAGAGATGGAGACAGACACAATCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGCTCCACTGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAAGGCCAGCCAAAGTGTTGATTATGATGGTGATAGTTATATGAACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAATCTAGAATCTGGGATCCCAGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "AAAGATGGTTACGACT-1_contig_3": "GGAGATATGAAATGCATCACACCAGCATGGGCATCAAAATGGAGTCACAGATTCAGGTCTTTGTATTCGTGTTTCTCTGGTTGTCTGGTGTTGACGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGAGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGACAATCTCCTAAACTACTGATTTACTCGGCATCCTACCGGTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATCTGGGACGGATTTCACTTTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAACATTATAGTACTCCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCAAGTACCTAGCCACCTCGCA",
        "CATGACACACGAAGCA-1_contig_1": "GATCCAGTGCAGCAATCAGTTCCTGCCAGGACACAGTTTAGATATGAGGTTCCAGGTTCAGGTTCTGGGGCTCCTTCTGCTCTGGATATCAGGTGCCCAGTGTGATGTCCAGATAACCCAGTCTCCATCTTATCTTGCTGCATCTCCTGGAGAAACCATTACTATTAATTGCAGGGCAAGTAAGAGCATTAGCAAATATTTAGCCTGGTATCAAGAGAAACCTGGGAAAACTAATAAGCTTCTTATCTACTCTGGATCCACTTTGCAATCTGGAATTCCATCAAGGTTCAGTGGCAGTGGATCTGGTACAGATTTCACTCTCACCATCAGTAGCCTGGAGCCTGAAGATTTTGCAATGTATTACTGTCAACAGCATAATGAATACCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGG",
        "AGAATAGTCTCGATGA-1_contig_1": "GGTCAACCGACGATCAGTGTCCTCTCCAAAGTCCCTGAACACACTGACTCTAACCATGGAATGGAGTTGGATATTTCTCTTTCTCCTGTCAAGAACTGCAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTAAAGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGATACACATTCACTAGCTATGTTATGCACTGGGTGAAGCAGAAGCCTGGGCAGGGCCTTGAGTGGATTGGATATATTAATCCTTACAATGATGGTACTAAGTACAATGAGAAGTTCAAAGGCAAGGCCACACTGACTTCAGACAAATCCTCCAGCACAGCCTACATGGAGCTCAGCAGCCTGACCTCTGAGGACTCTGCGGTCTATTACTGTGCAAGAGGGGGGTTACTACATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GGCGACTGTCCTAGCG-1_contig_1": "GGGCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CGACTTCGTCTTTCAT-1_contig_2": "AGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCC",
        "CCGGTAGTCACTCTTA-1_contig_1": "TGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGCGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATTTTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTC",
        "CGTCAGGTCTGCAAGT-1_contig_3": "ATCATCATCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGCATTACTCTTCTGCCTGGTAACATTCCCAAGCTGTATCCTTTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCGTCTCAGGGTTCTCATTAACCGGCTATGGTGTAAACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAATGATATGGGGTGATGGAAGCACAGACTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCAGGTACTACTGTGCCGGCTATGGTAACCTTCTTCCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TTCTCAAGTACCAGTT-1_contig_1": "GTTTCAGTGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "TGCCAAAGTCCAGTTA-1_contig_10": "GACATACCAGCAAGGGAGTGACCAGTTAGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGTTATACCTTCACAGACTATTCAATGCACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACTGAGACTGGTGAGCCAACATATGCAGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCTAGAAAAGACGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "TCATTACTCTTGTATC-1_contig_1": "GGATTGTGTATCCAGTCTAGTTCAATGTGTCTTAGGAAGCCCAGTCATATGCAAATCTAGAGAAGACTTTAGAGTAGAAATCTGAGGCTCACCTCACATACCAGCAAGGGAGTGACCAGTTAGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGTTATACCTTCACAGACTATTCAATGCACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACTGAGACTGGTGAGCCAACATATGCAGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCTAGAATATACTATGGTAACTCGATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CACATAGAGATTACCC-1_contig_4": "GGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCTCCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATG",
        "AAGGTTCTCTTGAGGT-1_contig_2": "GCACATCCAGTGCAGTAATCAATGTCAGGTCACAGCAGAAACATGAAGTTTCCTTCTCAACTTCTGCTCTTACTGCTGTTTGGAATCCCAGGCATGATATGTGACATCCAGATGACACAATCTTCATCCTCCTTTTCTGTATCTCTAGGAGACAGAGTCACCATTACTTGCAAGGCAAGTGAGGACATATATAATCGGTTAGCCTGGTATCAGCAGAAACCAGGAAATGCTCCTAGGCTCTTAATATCTGGTGCAACCAGTTTGGAAACTGGGGTTCCTTCAAGATTCAGTGGCAGTGGATCTGGAAAGGATTACACTCTCAGCATTACCAGTCTTCAGACTGAAGATGTTGCTACTTATTACTGTCAACAGTATTGGAGTACTGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGA",
        "ATGTGTGAGCCCAGCT-1_contig_5": "TGGGGCTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "CATGGCGGTGTCGCTG-1_contig_1": "GGGGATCAGACAGGCAGGGGAAGCAAGATGGATTCACAGGCCCAGGTTCTTATGTTACTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTAGCTGTGTCAGTTGGAGAGAAGGTTACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAG",
        "CAGGTGCCACCTCGGA-1_contig_1": "TCTGGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGCA",
        "CCATTCGTCAACTCTT-1_contig_2": "GGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGG",
        "TACAGTGAGATGCCTT-1_contig_2": "TGGGGGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "GCTTCCAGTTATCACG-1_contig_1": "GGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTTATTTCCAGCTTGGTCCCCCCTCCGAACGTGTACA",
        "TTGAACGGTCATCCCT-1_contig_5": "GATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGAGACAGACTCTTCTCCACAGTGTCCCCTTCATG",
        "GACTAACGTGTGGTTT-1_contig_3": "AAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGG",
        "CACAGGCGTCTCCACT-1_contig_9": "GAACATATGTCCAATGTCCTCTCCACAGTCCCTGAAGACACTGACTATAACTATGGGATGGACCTGGATCTTTATTTTAATCCTGTCAGTAACTACAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGAGAAGCCTGGCGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGCTACAACATGAACTGGGTGAAGCAGAGCAATGGAAAGAGCCTTGAGTGGATTGGAAATATTGATCCTTACTATGGTGGTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAAGAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGAAACTATAGGTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GAAATGAAGGTGCAAC-1_contig_7": "GGACATCTCTCTCATTAGAGGTTGATCTTTGAGGAAAACAGGGTGTTGCCTAAAGGATGAAAGTGTTGAGTCTGTTGTACCTGTTGACAGCCATTCCTGGTATCCTGTCTGATGTACAGCTTCAGGAGTCAGGACCTGGCCTCGTGAAACCTTCTCAGTCTCTGTCTCTCACCTGCTCTGTCACTGGCTACTCCATCACCAGTGGTTATTACTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATAAGCTACGACGGTAGCAATAACTACAACCCATCTCTCAAAAATCGAATCTCCATCACTCGTGACACATCTAAGAACCAGTTTTTCCTGAAGTTGAATTCTGTGACTACTGAGGACACAGCTACATATTACTGTGCAAGAGGTAACTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "ACGTCAAGTGTGACCC-1_contig_5": "GGGGGAGCCCTATCTTCTCTACAGACACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTTATCTTCTTCCTGATGGCAGTGGTTACAGGGGTCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCAGAGCTTGTGAAGCCAGGGGCCTCAGTCAAGTTGTCCTGGAAAGTTTCTGGCTTCAACATTAAAGACACCTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGAAGGATTGATCCTGCGAATGGTAATACTAAATATGACCCGAAGTTCCAGGACAAGGCCACTATAACAGCAGACACATCCTCCAATACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTGCTTTAACTGGGCGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGCCAAAACAACACCCCCATCAGTCTATCCACTGGCCCCTGGGTGTGGAGATACAACTGGTTCCTCCGTGACTCTGGGATGCCTGGTCAAGGGCTACTTCCCTGAGTCAGTGACTGTGACTTGGAACTCTGGATCCCTG",
        "CGGACTGCACATCCAA-1_contig_6": "GGGATGGTCCACACAAACTCAGGGAAAGTTTGAAGATGGTATCCACACCTCAGTTCCTTGTATTTTTGCTTTTCTGGATTCCAGCCTCCAGAGGTGACATCTTGCTGACTCAGTCTCCAGCCATCCTGTCTGTGAGTCCAGGAGAAAGAGTCAGTTTCTCCTGCAGGGCCAGTCAGAGCATTGGCACAAGCATACACTGGTATCAGCAAAGAACAAATGGTTCTCCAAGGCTTCTCATAAAGTATGCTTCTGAGTCTATCTCTGGGATCCCTTCCAGGTTTAGTGGCAGTGGATCAGGGACAGATTTTACTCTTAGCATCAACAGTGTGGAGTCTGAAGATATTGCAGATTATTACTGTCAACAAAGTAATAGCTGGCCAACGCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GTGCTTCGTTACCAGT-1_contig_5": "TTGGCTCTTATATGGGGACTTATGAGCACTGTTTTCTCTACAGTCACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTCATCTTCTTCCTGATGGCAGTGGTTACAGGGGTCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCTGAGCTTGTGAGGCCAGGGGCCTTAGTCAAGTTGTCCTGCAAAGCTTCTGGCTTCAACATTAAAGACTACTATATGCACTGGGTGAAACAGAGGCCTGAACAGGGCCTGGAGTGGATTGGATGGTTTAATCCTGAGAATGGTAATAATATATATGACCCGAAGTTCCAGGGCAAGGCCAGTATAACAGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCCGTCTATTACTGTGCTAGAGAGGGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGTCTGCGAGAAATCCCACCATCTACCCACTGACACTCCCACCAGCTCTGTCAAGTGACCCAGTGATAATCGGCTGCCTGATTCACGATTACTTCCCTTCCGGCACGATGAATGTGACCTGGGGAAAGAGTGGGAAGGATATAACCACCGTAAACTTCCCACCTGCCCTGGCCTCTGGGGGACGGTACACCATGAGCAGCCAGTT",
        "TGGTTCCAGATGTAAC-1_contig_5": "CTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGA",
        "CTCGGAGCACCAGTTA-1_contig_1": "GAATGCAATTATCTAGACTTATTTCGGTTGAACATGCTGGTTGGTGGTTGAGAGGACACTCAGTCAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACCGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTGATTTCCAGCTTGGTGCCTCCACCGAACGTCCGAGGAA",
        "TGAGCCGTCGTATCAG-1_contig_2": "TGAGCCGTCGTATCAGTTTAACCCGCTATATGGGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "CTGTGCTAGTGTCCCG-1_contig_2": "TTACTGTTTTCTCTTTCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "AACGTTGGTTATGCGT-1_contig_4": "GCTCTGACAGAGGAGGCCAGTCCTGGATTTGAGTTCCTCACATTCAGTGATGAGCACTGAACACGGACCCCTCACCATGAATTTCGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCTGTGTGAAGTGAAGCTGGTGGAGTCTGGGGGAGGTTTAGTGCAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATACCATGTCTTGGGTTCGCCAGACTCCAGAGAAGAGGCTGGAGTGGGTCGCATACATTAGTAATGGTGGTGGTAGCACCTACTATCCAGACACTGTAAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACGGCCATGTATTACTGTGCAAGACATTACTACGGTAGTAGCTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TGTTCCGCACCTCGGA-1_contig_1": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTAC",
        "CAACCTCCATGCTAGT-1_contig_6": "GATCATCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGCATTACTCTTCTGCCTGGTAACATTCCCAAGCTGTATCCTTTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCGTCTCAGGGTTCTCATTAACCGGCTATGGTGTAAACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAATGATATGGGGTGATGGAAGCACAGACTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCAGGTACTACTGTGCCAGAGGGGATGCGATTACCCTGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCAC",
        "GGAACTTTCTGGAGCC-1_contig_1": "TGGGGGACTGATCAGTCTCCTCAGGCTCCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGCTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGGTCTAGTCAGAGCCTTGAAAACAGTAATGGAAACACCTATTTGAACTGGTACCTCCAGAAACCAGGCCAGTCTCCACAGCTCCTGATCTACAGGGTTTCCAACCGATTTTCTGGGGTCCTAGACAGGTTCAGTGGTAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTTCTGCCTCCAAGTTACACATGTCCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CGCTATCCACCGGAAA-1_contig_5": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TATCAGGTCACAATGC-1_contig_2": "TGGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TGGTTCCAGATGTAAC-1_contig_3": "GGGGGCCAGGAGCCTAAGAAGCATTCTCTCTTCCAGTTCTCAGAGATGGAGAAAGACACACTCCTGCTATGGGTCCTGCTTCTCTGGGTTCCAGGTTCCACAGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAGAGCCAGCGAAAGTGTTGATAATTATGGCATTAGTTTTATGAACTGGTTCCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAACCAAGGATCCGGGGTCCCTGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCAGCCTCAACATCCATCCTATGGAGGAGGATGATACTGCAATGTATTTCTGTCAGCAAAGTAAGGAGGTTCCTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GATCTAGCATTGCGGC-1_contig_2": "TGGTGTTTAGGACAATGGCAGAAAGTCACTCTCAGTGAGGATACACCATCAGCATGAGGGTCCTTGCTGAGCTCCTGGGGCTGCTGCTGTTCTGCTTTTTAGGTGTGAGATGTGACATCCAGATGAACCAGTCTCCATCCAGTCTGTCTGCATCCCTTGGAGACACAATTACCATCACTTGCCATGCCAGTCAGAACATTAATGTTTGGTTAAGCTGGTACCAGCAGAAACCAGGAAATATTCCTAAACTATTGATCTATAAGGCTTCCAACTTGCACACAGGCGTCCCATCAAGGTTTAGTGGCAGTGGATCTGGAACAGGTTTCACATTAACCATCAGCAGCCTGCAGCCTGAAGACATTGCCACTTACTACTGTCAACAGGGTCAAAGTTATCCTCTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AGGCCACCATAGACTC-1_contig_5": "CTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GTCACAACATTTGCCC-1_contig_1": "GATAGCAAAGTACTTATGAGAATAGTAGTAATTAGCTAGGGACCAAAGTTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATACTGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAGACTCCTGATTTATGACACATCCAACCTGGCTTCTGGAGTCCCTGTTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTTACCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTC",
        "CATGACAAGCCAGGAT-1_contig_7": "GATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "ACACCCTAGGAATCGC-1_contig_6": "GACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "CAGTAACGTTATGTGC-1_contig_4": "GAGGGGTCACATTGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTTGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGCCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCAGGACCTGACCTGGTGAAACCTTCTCAGTCACTTTCACTCACCTGCACTGTCACTGGCTACTCCATCACCAGTGGTTATAGCTGGCACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATACACTACAGTGGTAGCACTAACTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGAGAATCTACGGCTACCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TAGGCATGTTACGGAG-1_contig_4": "GAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "ACTGAACAGGACGAAA-1_contig_1": "GGGACACCCTGTGCTGGAGTCAGCCTCACACAGAACACACACAGATATGGGTGTACCCACTCAGCTCCTGTTGCTGTGGCTTACAGTCGTAGTTGTCAGATGTGACATCCAGATGACTCAGTCTCCAGCTTCACTGTCTGCATCTGTGGGAGAAACTGTCACCATCACATGTGGAGCAAGTGAGAATATTTACGGTGCTTTAAATTGGTATCAGCGGAAACAGGGAAAATCTCCTCAGCTCCTGATCTATGGTGCAACCAACTTGGCAGATGGCATGTCATCGAGGTTCAGTGGCAGTGGATCTGGTAGACAGTATTCTCTCAAGATCAGTAGCCTGCATCCTGACGATGTTGCAACGTATTACTGTCAAAATGTGTTAAGTACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CATCGGGCAAATACAG-1_contig_2": "GGGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TACTTACGTACGCACC-1_contig_4": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CAGCCGATCTACTATC-1_contig_2": "TGGGGGAAATACATCAGACCAGCATGGGCTTCAAGATGGAGTCTCATACTCAGGCCTTTGTATTCGCGTTTCTCTGGTTGTCTGGTGTTGATGGAGTCATTGTGATGACCCAGTCTCAAAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGAATGTGGGTACTTCTGTAGCCTGGTATCAACAGAAACCAGGACAATCTCCTAAACTACTGATTTACTCGGCATCCAATCGGTACACTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAATATGCAGTCTGAAGACCTGGCAGATTATTTCTGCCAGCAATATAGCAGATTTCCTCTCACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCTCAGAGGAAGGTGGAAACACGGTGAGAGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACCGCCGAAAACATAACACAATGACTAAAACCCAACCCAAAACCTATGGGGCCAGC",
        "CTGCGGATCACCGTAA-1_contig_2": "GGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGC",
        "TGGGAAGCACGTCAGC-1_contig_5": "GGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCTTGGGGCTGCTCTTCTGCCTGGTGACATTCCCAAGCTGTGTCCTATCCCAGGTGCAGCTGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATCACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGTGGTGGAAGCACAGACTATAATGCAGCTTTCATATCCAGACTGAGCATCAGCAAGGACAATTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTAATGACACAGCCATATATTACTGTGCCAGAAATTACGACTACGGTAGTACCTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GACTACACAGGATTGG-1_contig_1": "GAGCTGCCAGGAGCCTAAGAAGCATCCTCTCATCTAGTTCTCAGAGATGGAGACAGACACAATCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGCTCCACTGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAAGGCCAGCCAAAGTGTTGATTATGATGGTGATAGTTATATGAACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAATCTAGAATCTGGGATCCCAGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCTCACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CAGCAGCTCCAAGTAC-1_contig_1": "GGGACAAATCCAGTGCAAGAATCAATGTCAGGTCACAGCAGAAACATGAAGTTTCCTTCTCAACTTCTGCTCTTCCTGCTGTTCAGAATCACAGGCATAATATGTGACATCCAGATGACACAATCTTCATCCTACTTGTCTGTATCTCTAGGAGGCAGAGTCACCATTACTTGCAAGGCAAGTGACCACATTAATAATTGGTTAGCCTGGTATCAGCAGAAACCAGGAAATGCTCCTAGGCTCTTAATATCTGGTGCAACCAGTTTGGAAACTGGGGTTCCTTCAAGATTCAGTGGCAGTGGATCTGGAAAGGATTACACTCTCAGCATTACCAGTCTTCAGACTGAAGATGTTGCTACTTATTACTGTCAACAGTATTGGAGTACTCCTCCTACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "TGACTAGAGTGGGCTA-1_contig_2": "TGGGGATTGTCATTGCAGCCAGGACTCAGCATGGACATGAGGACCCCTGCTCAGTTTCTTGGAATCTTGTTGCTCTGGTTTCCAGGTATCAAATGTGACATCAAGATGACCCAGTCTCCATCTTCCATGTATGCATCTCTAGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAATAGCTATTTAAGCTGGTTCCAGCAGAAACCAGGGAAATCTCCTAAGACCCTGATCTATCGTGCAAACAGATTGGTAGATGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTCACCATCAGCAGCCTGGAGTATGAAGATATGGGAATTTATTATTGTCTACAGTATGATGAGTTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAG",
        "CTTACCGGTCTAGTCA-1_contig_7": "GGGCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTG",
        "TAGACCATCTGTCCGT-1_contig_1": "TGGGGGAAATACATCAGACCAGCATGGGCATCAAGATGGAGACACATTCTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGAAGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGGGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAACTACTGATTTACTGGGCATCCACCCGGCACACTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATTAGCAATGTGCAGTCTGAAGACTTGGCAGATTATTTCTGTCAGCAATATAGCAGCTATCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CGTCCATAGGTTCCTA-1_contig_4": "GACACAAGTGTGCAGACATGGGCAGGCTTACTTTTTCATTCTTGCTACTGATTGTCCCTGCATATGTCCTGTGCCAGGTTACTCTGAAAGAGTCTGGCCCTGGGATATTGCAGCCCTCCCAGACCCTCAGTCTGACTTGTTCTTTCTCTGGGTTTTCACTGAGCACTTCTAATATGGGTGTAGGCTGGATTCGTCAGCCTTCAGGGAAGGGTCTGGAGTGGCTGTTACACATTTTGTGGAATGATAGTAAGTACTATAACCCAGCCCTGAAGAGCCGGCTCACAATCTCCAAGGATACCTACAACAACCAGGTATTCCTCAAGATCGCCAATGTGGACACTGCAGATACTGCCACATACTACTGTGCTCGAATAGCGGGGTACTACGGTAGTAGCTACGACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCAAGAGCATCCTTGAAGG",
        "CGACTTCGTTTGACTG-1_contig_3": "TGGGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TGAGCATTCGGATGGA-1_contig_1": "GGACAGTTGTTTCAAAGTGATGTTCTTATCAGACTAGCACTAATTAGCCAGAGACCAAAATCCAAATACACAATGGACTTTCGGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGTCACAGTGTCCAGAGGAGAAATTGTACTCACTCAGTCTCCAGTCATCACAACTGCATCTCTGGGGCAAAAGGTCACCATCACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAGGCACCTCCCCCAAACCATGGATTTATGAAATATCCAAACTGGCTTCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTTCTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCATTTATTACTGCCAGCAGTGGAATTATCCTCTTATCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAGACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TGACAACGTCCGCTGA-1_contig_1": "GGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCACTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACCCAAGT",
        "CTGATCCAGTAGCGGT-1_contig_1": "GGAGCTGCCAGGAGCCTAAGAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGTTATGGGTACTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCAAAAGTGTCAGTACATCTGGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCACAGTAGGGAGCTTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTG",
        "CCCAATCCAAGCGTAG-1_contig_1": "ACACCTCGGCGCTGACCCTGCAGCTCTGACAGAGGAGGCAGGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACGGACCCCTCACCATGAACTTTGTGCTCAGCTTGATTTTCCTTGCCCTCATTTTAAAAGGTGTCCAGTGTGAAGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGCCATGTCTTGGGTTCGCCAGACTCCGGAGAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGTCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGACATTACTACGGTAGTAGCTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CAGCGACAGTGTTGAA-1_contig_4": "GATATGAACACTGTTTTCTCTACAGTCACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTCATCTTCTTCCTGATGGCAGTGGTTATAGGAATCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCAGAGCTTGTGAGGTCAGGGGCCTCAGTCAAGTTGTCCTGCACAGCTTCTGGCTTCAACATTAAAGACTACTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGATGGATTGATCCTGAGAATGGTGATACTGAATATGCCCCGAAGTTCCAGGGCAAGGCCACTATGACTGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTAATGCAGTCTACTATGATTCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TATTACCGTTGGTAAA-1_contig_3": "GGGAGCGTATGAACCTAGCCCTGATTTCCCCAGCCTTCAGTTCCCAGATTCAGTGATCAGCCTTGAACACAGACCTGTCACCATGAAGTTGTGGCTGAACTGGATTTTCCTTGTAACACTTTTAAATGGTATCCAGTGTGAGGTGAAGCTGGTGGAGTCTGGAGGAGGCTTGGTACAGCCTGGGGGTTCTCTGAGACTCTCCTGTGCAACTTCTGGGTTCACCTTCACTGATTACTACATGAGCTGGGTCCGCCAGCCTCCAGGAAAGGCACTTGAGTGGTTGGGTTTTATTAGAAACAAAGCTAATGGTTACACAACAGAGTACAGTGCATCTGTGAAGGGTCGGTTCACCATCTCCAGAGATAATTCCCAAAGCATCCTCTATCTTCAAATGAACACCCTGAGAGCTGAGGACAGTGCCACTTATTACTGTGCAAGAGATGGTAACTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GATGCTAGTTGTCTTT-1_contig_7": "GAGCTCTGACAGAGGAGGCCGGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACGGACCCCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGCCCTCATTTTAAAAGGTGTCCAGTGTGAGGTGCAGCTGGTGGAGTCTGGGGGAGACTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGGCATGTCTTGGGTTCGCCAGACTCCAGACAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGGCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTGCAAGACTAACTGGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "CACATTTAGCTCTCGG-1_contig_3": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCAC",
        "CGTGTCTAGCTGATAA-1_contig_5": "AAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GTCAAGTTCGACCAGC-1_contig_1": "GAAAAGCAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCTTCAGTCATAATGTCCAGAGGACAAATTGTTCTCTCCCAGTCTCCAGCAATCCTGTCTGCATCTCCAGGGGAGAAGGTCACAATGACTTGCAGGGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAAACCCTGGATTTATGCCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGAGTGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCACCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "ATTGGACAGCAATCTC-1_contig_3": "ATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATC",
        "CACCTTGCAGTTCATG-1_contig_1": "GAGATTGTGAGCCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "ACGTCAAGTGTGACCC-1_contig_3": "GAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGG",
        "CGATGGCCAAGCCGTC-1_contig_1": "GCATTCAACAAAGCCTCCATAACCACTATCATATTTGGTACAATTTGCATGCCCCCATTACCACATCAGCCTCTCTGATGACATATAATGCATGCCATCAAGCTGCATGGGAAGCTGGCTATCACTCAAGGCCATGACCATGTTCTCACTAGCTCTTCTCCTCAGTCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCATGGCTATAGGAGAAAAAGTCACCATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGTACAGATTTTGTTTTTACAATTGAAAACATGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACTTGCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CAGATCAGTCTGGTCG-1_contig_4": "TGGGGATCTACATCTGAAAGGCAGGTGGAGCAAGATGGAATCACAGACTCAGGTCCTCATGTCCCTGCTGTTCTGGGTATCTGGTACCTGTGGGGACATTGTGATGACACAGTCTCCATCCTCCCTGACTGTGACAGCAGGAGAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGTCTGTTAAACAGTGGAAATCAAAAGAACTACTTGACCTGGTACCAGCAGAAACCAGGGCAGCCTCCTAAACTGTTGATCTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGAACAGATTTCACTCTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGAATGATTATAGTTATCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CTTTGCGTCTTGCAAG-1_contig_3": "GGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CGTTCTGCACGTCAGC-1_contig_1": "GCCATTCAACAAAGCCTCCATAACCACTATCATATTTGGTACAATTTGCATGCCCCCATTACCACATCAGCCTCTCTGATGACATATAATGCATGCCATCAAGCTGCATGGGAAGCTGGCTATCACTCAAGGCCATGACCATGTTCTCACTAGCTCTTCTCCTCAGTCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCATGGCTATAGGAGAAAAAGTCACCATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGTACAGATTTTGTTTTTACAATTGAAAACATGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACTTGCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CGAACATCATTGCGGC-1_contig_1": "TGGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CCGTACTCACGACTCG-1_contig_1": "TGGGGGCCATTGTCATTGCAGTCAGGACTCAGCATGGACATGAGGGCTCCTGCACAGATTTTTGGCTTCTTGTTGCTCTTGTTTCCAGGTACCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCTTATCTGCCTCTCTGGGAGAAAGAGTCAGTCTCACTTGTCGGGCAAGTCAGGACATTGGTAGTAGCTTAAACTGGCTTCAGCAGGAACCAGATGGAACTATTAAACGCCTGATCTACGCCACATCCAGTTTAGATTCTGGTGTCCCCAAAAGGTTCAGTGGCAGTAGGTCTGGGTCAGATTATTCTCTCACCATCAGCAGCCTTGAGTCTGAAGATTTTGTAGACTATTACTGTCTACAATATGCTAGTTCTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CTCGTACCAATCAGAA-1_contig_4": "GCTGACCCTGCAGCTCTGACAGAGGAGGCCAGTCCTGGAATTGATTCCCAGTTCCTCACGTTCAGTGATGAGCACTGAACACAGACACCTCACCATGAACTTTGGGCTCAGATTGATTTTCCTTGTCCTTACTTTAAAAGGTGTGAAGTGTGAAGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCGCTTTCAGTAGCTATGACATGTCTTGGGTTCGCCAGACTCCGGAGAAGAGGCTGGAGTGGGTCGCATACATTAGTAGTGGTGGTGGTAGCACCTACTATCCAGACACTGTGAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTGCAAGACTAGTAATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCAC",
        "TGGCTGGAGCACCGCT-1_contig_1": "TGGGGGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TAAGCGTAGGATGGTC-1_contig_5": "AGATCAGTGGCCTCTCCACAGTCCCTGAACACACTGACTCAAACCATGGACTGGAGTTGGGTCTTTCTCTTCCTCCTGTCAGTAAATGAAGGTGTCTACTGTCAGGTCCAGCTGCAGCAGTCTGGAGATGATCTGGTAAAGCCTGGGGCCTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTGGATTAACTGGATAAAACAGAGGCCTGGACAGGGCCTTGAGTGGATAGGACGTATTGCTCCTGGAAGTGGTAGTACTTACTACAATGAAATGTTCAAGGGCAAGGCAACACTGACTGTAGACACATCCTCCAGCACAGCCTACATTCAGCTCAGCAGCCTGTCATCTGAGGACTCTGCTGTCTATTTCTGTGCAAAAGAGGGGGGGGACTACTATAGGTACGACGGAGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CTCGGGAGTGACGGTA-1_contig_2": "GGGACATACCAGCAAGCGAGTGACCAGTTAGTCTTAAGGCACCACTTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCAAGCTACTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCT",
        "CATCAAGCACCTTGTC-1_contig_3": "ATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGG",
        "GACTGCGGTACAGTGG-1_contig_2": "GGGACACCCTGTGCTGGAGTCAGCCTCACACTGATCACACACAGTCATGAGTATTTCCACTCAGCTCCTGGGGTTGCTGCTGCTGTGGCTTACAGATGCCAGATGTGACATTCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCAAGCAAGTGAGAATATTGCCAGTGATTTAGCATGGTATTAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATGATGCGAGAAACTTAGCAGATGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACACTATTCTCTCAATATCCACAGCCTGCAGTCTGAAGATGTTGCGAGATATTACTGTCAACATTATTATGGTACTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAA",
        "GAAGCAGTCGGTCCGA-1_contig_2": "TGGGGTGAAAGACTCGCCAGAGCTAGACCTCCTTGGTGTGTGTTAACTCAGGAAGAAAAAGACGACATAAAACAACAGCCAGGACTCAGCATGGACATGAGGACCCCTGCTCAGTTTCTTGGAATCTTGTTGCTCTGGTTTCCAGGTATCAAATGTGACATCAAGATGACCCAGTCTCCATCTTCCATGTATGCATCTCTAGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAATAGCTATTTAAGCTGGTTCCAGCAGAAACCAGGGAAATCTCCTAAGACCCTGATCTATCGTGCAAACAGATTGGTAGATGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTCACCATCAGCAGCCTGGAGTATGAAGATATGGGAATTTATTATTGTCTACAGTATGATGAGTTTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TACCTTACATACGCTA-1_contig_1": "ATCTGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "AATCCAGAGATGCGAC-1_contig_11": "GACACAGAAAAACATGAGATCACAGTTCTCTCTACAGTTACTGAGCACACAGGACCTCACCATGGGATGGAGCTATATCATTTTCTTTTTGGTAGCAACAGCTACAGGTGTCCACTCCCAGGTCCAACTCCAGCAACCTGGGGCTGAACTGGTGAAGCCTGGGGCTTCAGTGAAGTTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTGGATGCACTGGGTGAAGCTGAGGCCTGGACAAGGCTTTGAGTGGATTGGAGAGATTAATCCTAGCAATGGTGGTACTAACTACAATGAGAAGTTCAAGAGAAAGGCCACACTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTCCTGTACAATAAATGGGGGATATGGTAACTACGGGTTTGTTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "CCTAAAGTCGACAGCC-1_contig_1": "GCATGCCATCAAGCTGCATTGGAAGCTGGCTATCACTCAAGGCCATGACCATGCTCTCACTAGCTCCTCTCCTCAGCCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCGTGGCTACAGGAGAAAAAGTCACTATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGCACAGATTTTGTTTTTACAATTGAAAACACGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACATGCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GCAGCCAGTGTGCGTC-1_contig_2": "ACATTGTGAGCCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGTGCAAGTCAGGGCATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTATTACACATCAAGTTTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGGACAGATTATTCTCTCACCATCAGCAACCTGGAACCTGAAGATATTGCCACTTACTATTGTCAGCAGTATAGTAAGCTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GATGCTACACTAAGTC-1_contig_1": "TGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GGGAATGGTAAGCACG-1_contig_2": "GAGCACTGTCCAGATGAAGCTGATTTGCATGTGCTGAGATCATATTCTACTGCCCCAGAGATTTAATAATCTGATCATACACACTCCAACAGTCATTCTTGGTCAGGAGACGTTGTAGAAATGAGACCGTCTATTCAGTTCCTGGGGCTCTTGTTGTTCTGGCTTCATGGTGCTCAGTGTGACATCCAGATGACACAGTCTCCATCCTCACTGTCTGCATCTCTGGGAGGCAAAGTCACCATCACTTGCAAGGCAAGCCAAGACATTAACAAGTATATAGCTTGGTACCAACACAAGCCTGGAAAAGGTCCTAGGCTGCTCATACATTACACATCTACATTACAGCCAGGCATCCCATCAAGGTTCAGTGGAAGTGGGTCTGGGAGAGATTATTCCTTCAGCATCAGCAACCTGGAGCCTGAAGATATTGCAACTTATTATTGTCTACAGTATGATAATCTTGGTCTGGGACAGATTATTCTCTCACCATCAGCAACCTGGAACCTGAAGATATTGCCACTTACTATTGTCAGCAGTATAGTAAGCTTCCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CACAGGCGTCTCCACT-1_contig_8": "GACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCC",
        "GAGGTGATCCGTTGTC-1_contig_2": "GCTGACAGAGGAGCCAAGCCCTGGATTCCCAGGTCCTCACATTCAGTGATCAGCACTGAACACAGACCACTCACCATGGACTCCAGGCTCAATTTAGTTTTCCTTGTCCTTATTTTAAAAGGTGTCCAGTGTGATGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGCAGCCTGGAGGGTCCCGGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTTTGGAATGCACTGGGTTCGTCAGGCTCCAGAGAAGGGGCTGGAGTGGGTCGCATACATTAGTAGTGGCAGTAGTACCATCTACTATGCAGACACAGTGAAGGGCCGATTCACCATCTCCAGAGACAATCCCAAGAACACCCTGTTCCTGCAAATGACCAGTCTAAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGATACTACGGTAGTAGCTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GAGGTGATCCGTTGTC-1_contig_1": "GGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CCATTCGCATCAGTAC-1_contig_2": "ACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GACTGCGTCCTATTCA-1_contig_1": "TGGGGCTCTTTCTGTCTTGATTACTTTTGACTCACCATATCAAGTTCGCAGAATGAGGTTCTCTGCTCAGCTTCTGGGGCTGCTTGTGCTCTGGATCCCTGGATCCACTGCAGATATTGTGATGACGCAGGCTGCATTCTCCAATCCAGTCACTCTTGGAACATCAGCTTCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTACATAGTAATGGCATCACTTATTTGTATTGGTATCTGCAGAAGCCAGGCCAGTCTCCTCAGCTCCTGATTTATCAGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTAGCAGTGGGTCAGGAACTGATTTCACACTGAGAATCAGCAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTGCTCAAAATCTAGAACTTCCTCCCACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACT",
        "CGGAGCTAGGCATTGG-1_contig_3": "GCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATAAAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "CCTAGCTCACGAAATA-1_contig_2": "GACAGTTGTCTTAAAGCAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGCATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATAACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTTCCAGCAGAAGCCAGGCACTTCTCCCAAACTCTGGATTTATAGCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGATCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAAAGGAGTAGTTACCCATATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CATCAAGCACCTTGTC-1_contig_5": "GAAAAACATGACATCACTGTTCTCTCTACAGTTACTGAGCACACAGGACCTCGCCATGGGATGGAGCTGTATCATCCTCTTCTTGGTAGCAACAGCTACAGGTGTCCACTCCCAGGTCCAACTGCAGCAGCCTGGGGCTGAGCTTGTGAAGCCTGGGGCTCCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTGGATGAACTGGGTGAAGCAGAGGCCTGGACGAGGCCTCGAGTGGATTGGAAGGATTGATCCTTCCGATAGTGAAACTCACTACAATCAAAAGTTCAAGGACAAGGCCACACTGACTGTAGACAAATCCTCCAGCACAGCCTACATCCAACTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGATCGGGCCTCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GGACAGAAGAACAACT-1_contig_3": "GGAGCGTATGAACCTAGCCCTGATTTCCCCAGCCTTCAGTTCCCAGATTCAGTGATCAGCCTTGAACACAGACCTGTCACCATGAAGTTGTGGCTGAACTGGATTTTCCTTGTAACACTTTTAAATGGTATCCAGTGTGAGGTGAAGCTGGTGGAGTCTGGAGGAGGCTTGGTACAGCCTGGGGGTTCTCTGAGACTCTCCTGTGCAACTTCTGGGTTCACCTTCACTGATTACTACATGAGCTGGGTCCGCCAGCCTCCAGGAAAGGCACTTGAGTGGTTGGGTTTTATTAGAAACAAAGCTAATGGTTACACAACAGGGTACAGTGCATCTGTGAAGGGTCGGTTCACCATCTCCAGAGATAATTCCCAAAGCATCCTCTATCTTCAAATGAACACCCTGAGAGCTGAGGACAGTGCCACTTATTACTGTGCAAGAGCCTCTATTACTACGGCTACCTATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GCATACAAGACCTTTG-1_contig_1": "GAACATAAGGTCACTGTTCTCACTATAGTTACTGAGCACACAGACCTCACCATGGGATGGAGCTCTATCATCCTCTTCTTGGTAGCAACAGCTACAGGTGTCCACTCCCAGGTCCAACTGCAGCAGCCTGGGGCTGAGCTGGTGAGGCCTGGAGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACTCCTTCACCAGCTACTGGATGAACTGGGTGAAGCAGAGGCCTGGACAAGGCCTTGAGTGGATTGGCATGATTCATCCTTCCGATAGTGAAACTAGGTTAAATCAGAAGTTCAAGGACAAGGCCACATTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAACTCAGCAGCCCGACATCTGAGGACTCTGCGGTCTATTACTGTGCAACCCTGTATGGTAACTACGATGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AGTGAGGGTTGTCTTT-1_contig_1": "GGACCACATCAGCCTCTCTGATGACATATAATGCATGCCATCAAGCTGCATGGGAAGCTGGCTATCACTCAAGGCCATGACCATGTTCTCACTAGCTCTTCTCCTCAGTCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCATGGCTATAGGAGAAAAAGTCACCATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGTACAGATTTTGTTTTTACAATTGAAAACATGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACTTGCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TGCTACCGTACATCCA-1_contig_1": "TGGGGGTCAGCTTCACACATGTTACACAGAGACATGAACATGCTCACTCAGCTCCTGGGATTACTGCTGCTCTGGTTTGCAGGTGGTAAATGTGACATTCAGATGACCCAGTCTCCTGCCTCCCAGTCTGCATCTCTGGGAGAAAGTGTCACCATCACATGCCTGGCAAGTCAGACCATTGGTACATGGTTAGCATGGTATCAGCAGAAACCAGGGAAATCTCCTCAGCTCCTGATTTATGCTGCAACCAGCTTGGCAGATGGGGTCCCATCAAGGTTCAGTGGTAGTGGATCTGGCACAAAATTTTCTTTCAAGATCAGCAGCCTACAGGCTGAAGATTTTGTAAGTTATTACTGTCAACAACTTTACAGTACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCTCAGAGGAAGGTGGAAACACGGTGAGAGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACCGCCGAAAACATAACACAATGACTAAAACCCAACCCAAAACCTATGGGGCCAGCACCTGGTCCTGTTCTC",
        "AACTCAGGTAGAGCTG-1_contig_3": "AGATCCAAGGTGAGTGTGAGAGGACAGGGGCTGGGGTATGGATACGCAGAAGGAAGGCCACAGCTGTACAGAATTGAGAAAGAATAGAGACCTGCAGTTGAGGCCAGCAGGTCGGCTGGACTAACTCTCCAGCCACAGTAATGACCCAGACAGAGAAGGCCAGACTCATAAAGCTTGCTGAGCAAAATTAAGGGAACAAGGTTGAGAGCCCTAGTAAGCGAGGCTCTAAAAAGCATGGCTGAGCTGAGATGGGTGGGCTTCTCTGAGCGCTTCTAAAATGCGCTAAACTGAGGTGATTACTCTGAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CTGATCCAGTAGCGGT-1_contig_3": "TGGGGGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GTCCTCAGTCATCCCT-1_contig_1": "TGGGGAATCAACCTCCTTAAAATGAATTTGCCTGTTCATCTCTTGGTGCTTCTGTTGTTCTGGATTCCTGCTTCCAGAGGTGATGTTGTGGTGACTCAAACTCCACTCTCCCTGCCTGTCAGCTTTGGAGATCAAGTTTCTATCTCTTGCAGGTCTAGTCAGAGTCTTGCAAACAGTTATGGGAACACCTATTTGTCTTGGTACCTGCACAAGCCTGGCCAGTCTCCACAGCTCCTCATCTATGGGATTTCCAACAGATTTTCTGGGGTGCCAGACAGGTTCAGTGGCAGTGGTTCAGGGACAGATTTCACACTCAAGATCAGCACAATAAAGCCTGAGGACTTGGGAATGTATTACTGCTTACAAGGTACACATCAGCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "ATCTGCCAGGAATCGC-1_contig_1": "TGGGGAACAGCATCAGACAGGCTGGGCAGCAAGATGGAATCACAGACCCAGGTCCTCATGTTTCTTCTGCTCTGGGTATCTGGTGCCTGTGCAGACATTGTGATGACACAGTCTCCATCCTCCCTGGCTATGTCAGTAGGACAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTAAATAGTAGCAATCAAAAGAACTATTTGGCCTGGTACCAGCAGAAACCAGGACAGTCTCCTAAACTTCTGGTATACTTTGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCATAGGCAGTGGATCTGGGACAGATTTCACTCTTACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGATTACTTCTGTCAGCAACATTATAGCACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCA",
        "GAGCAGATCTGGCGTG-1_contig_3": "GCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "AGAATAGCAGTATGCT-1_contig_1": "GAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CCTTCGAAGCACCGTC-1_contig_4": "AGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATC",
        "CGTCAGGTCTGCAAGT-1_contig_5": "GGGGACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATACTGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTATAAGTTACATGCACTGGTACCAGCAGAAGCCAGGCACCTCCCCCAAAAGATGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTATTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCATCAGCGGAGTAGTTACCCATCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GATCTAGGTCACCCAG-1_contig_3": "GGACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGTACCTGGCCTGCTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGATCGGGTAACTACGCCGCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGT",
        "GCTGCTTGTCAGATAA-1_contig_7": "GGGCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTG",
        "AGGTCATCACGGTGTC-1_contig_3": "GCACTTACTGTTTTCTCTTTCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CTGATAGGTCTTTCAT-1_contig_3": "GGGACAAGTGTGCAGCCATGGGCAGGCTTACTTCTTCATTCTTGCTACTGATTGTCCCTGCATATGTCCTGTCCCAGGTTACTCTGAAAGAGTCTGGCCCTGGGATATTGCAGCCCTCCCAGACCCTCAGTCTGACTTGTTCTTTCTCTGGGTTTTCACTGAGCACTTCTGGTATGGGTGTAGGCTGGATTCGTCAGCCTTCAGGGAAGGGTCTGGAGTGGCTGGCACACATTTGGTGGGATGATGACAAGCGCTATAACCCAGCCCTGAAGAGCCGACTGACAATCTCCAAGGATACCTCCAGCAACCAGGTATTCCTCAAGATCGCCAGTGTGGACACTGCAGATACTGCCACATACTACTGTGCTCGAATCAGGAACTATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CTCGGAGGTATTAGCC-1_contig_4": "GACTAGTGTGCAGATATGAACAGGCTTACTTCCTCATTGCTGCTGCTGATTGTCCCTGCATATGTCCTGTCCCAGGTTACTCTGAAAGAGTCTGGCCCTGGGATATTGCAGCCCTCCCAGACCCTCAGTCTGACTTGTTCTTTCTCTGGGTTTTCACTGAGCACTTCTGGTATGGGTGTGAGCTGGATTCGTCAGCCTTCAGGAAAGGGTCTGGAGTGGCTGGCACACATTTACTGGGATGATGACAAGCGCTATAACCCATCCCTGAAGAGCCGGCTCACAATCTCCAAGGATACCTCCAGCAACCAGGTATTCCTCAAGATCACCAGTGTGGACACTGCAGATACTGCCACATACTACTGTGCTCGAAGAGCTATACTACGGAGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GATCGATGTCCTAGCG-1_contig_5": "GGGACAGACGCACAACCCTGGACTCACAAGTCTTTCTCTTCAGTGACAAACACAGAAATAGAACATTCACCATGTACTTGGGACTGAACTGTGTATTCATAGTTTTTCTCTTAAAAGGTGTCCAGAGTGAAGTGAAGCTTGAGGAGTCTGGAGGAGGCTTGGTGCAACCTGGAGGATCCATGAAACTCTCCTGTGTTGCCTCTGGATTCACTTTCAGTAACTACTGGATGAACTGGGTCCGCCAGTCTCCAGAGAAGGGGCTTGAGTGGGTTGCTGAAATTAGATTGAAATCTAATAATTATGCAACACATTATGCGGAGTCTGTGAAAGGGAGGTTCACCATCTCAAGAGATGATTCCAAAAGTAGTGTCTACCTGCAAATGAACAACTTAAGAGCTGAAGACACTGGCATTTATTACTGTACCAGCCTTTACTACGGTAGTAGCCCAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CAGTAACGTTATGTGC-1_contig_1": "GATAGCAAAGTACTTATGAGAATAGTAGTAATTAGCTAGGGACCAAAGTTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATACTGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAGACTCCTGATTTATGACACATCCAACCTGGCTTCTGGAGTCCCTGTTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTTACCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TCTGAGAGTTCTGGTA-1_contig_4": "GGGCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGT",
        "TCGCGTTCAGATCTGT-1_contig_7": "GCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTC",
        "GTGCTTCAGGGCTTGA-1_contig_2": "GGGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCT",
        "GATGAAATCCTGCTTG-1_contig_2": "GATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCTTGGGGCTGCTCTTCTGCCTGGTGACATTCCCAAGCTGTGTCCTATCCCAGGTGCAGCTGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATCACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGTGGTGGAAGCACAGACTATAATGCAGCTTTCATATCCAGACTGAGCATCAGCAAGGACAATTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTAATGACACAGCCATATATTACTGTGCCAGAAATCCCTACTATGGTAATCTCGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GGGTTGCGTACAGCAG-1_contig_1": "GGGGGTCACAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTG",
        "CGAGCCACACGGCTAC-1_contig_2": "GTGTTTCTTATATGGGGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACAC",
        "CTGGTCTTCCGCGGTA-1_contig_2": "GAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATC",
        "TATGCCCGTGTGGCTC-1_contig_2": "AAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACAGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATATCCTGCAGAGCCAGTGAAAGTGTTGATAGTTATGGCAATAGTTTTATGCACTGGTACCAGCAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCGTGCATCCAACCTAGAATCTGGGATCCCTGCCAGGTTCAGTGGCAGTGGGTCTAGGACAGACTTCACCCTCACCATTAATCCTGTGGAGGCTGATGATGTTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCA",
        "TGGCTGGGTAGCCTAT-1_contig_4": "ACTGTTCTCTCTACAGTTACTGAGCACACAGGATCTCACCATGGGATGGAGCTATATCATCCTCTTCTTGTTAGCAACAGCTACATGTGTCCACTCCCAGGTCCAACTGCAGCAGCCTGGGGCTGAGCTGGTGAGGCCTGGGGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACGTTCACCAGCTACTGGATGAACTGGGTTAAGCAGAGGCCTGAGCAAGGCCTTGAGTGGATTGGAAGGATTGATCCTTACGATAGTGAAACTCACTACAATCAAAAGTTCAAGGACAAGGCCATATTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTACAATACTATCCCTCTACTATAGGTACGAAGGCTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CCGTACTCACGACTCG-1_contig_7": "CAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "ACTGAACGTACCGGCT-1_contig_5": "ATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TTGGAACGTAAGGATT-1_contig_3": "AGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTG",
        "GATTCAGTCTTGCCGT-1_contig_5": "TGGGGAGCTCTGACAGAGGAGCCAAGCCCTGGATTCCCAGGTCCTCACATTCAGTGATCAGCACTGAACACAGACCACTCACCATGGACTCCAGGCTCAATTTAGTTTTCCTTGTCCTTATTTTAAAAGGTGTCCAGTGTGATGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGCAGCCTGGAGGGTCCCGGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTTTGGAATGCACTGGGTTCGTCAGGCTCCAGAGAAGGGGCTGGAGTGGGTCGCATACATTAGTAGTGGCAGTAGTACCATCTACTATGCAGACACAGTGAAGGGCCGATTCACCATCTCCAGAGACAATCCCAAGAACACCCTGTTCCTGCAAATGACCAGTCTAAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGAACCTCGGTAGCCCCCTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GATGCTAGTTGTCTTT-1_contig_2": "GGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GACGGCTTCGCCGTGA-1_contig_5": "GACTTATGAGAATAGCAGTAATTAGCTAGAGATCAAAATTCAAAGAGAAAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCTGGGGAACGGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTCCAGCTACTTGTACTGGTACCAGCAGAAGTCAGGATCCTCCCCAAAACTCTGGATTTATAGCATATCCAACCTGGCTTCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAACAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCACCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "ACCTTTATCCTAGGGC-1_contig_2": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGTGTAGATCTCGGTGGTCGCCGTATCATT",
        "CCAATCCTCAAACAAG-1_contig_4": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "ATCATCTGTGCTCTTC-1_contig_5": "GATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGAATTGAAGTCTCAACTATGAAAATCAGCAGTTCTCTGTCAGAGAAGCCCAAGCGCTTCCACGCATGCTTGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CGACTTCAGTGGCACA-1_contig_3": "GGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "AACTCAGTCCAGAGGA-1_contig_3": "GGGGAACCTATGATCACTGTCCTCTCCACAGTCCCTGAAGACACTGACTCTAACCATGGGATGGAGCTGTATCTTTCTCTTTCTCCTGTCAGTAACTGTAGGTGTGTTCTCTGAGGTTCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTTACTGGCTACTTTATGAACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGACGTATTAATCCTTACAATGGTGATACTTTCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAATCCTCTAGCACAGCCCACATGGAGCTCCTGAGCCTGACATCTGAGGACTCTGCAGTCTATTATTGTGGAAGAGACTACGGTAGTAGCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "AGCAGCCGTGGTGTAG-1_contig_6": "GCTCTCACCTGCTCTGTCACTGGCTACTCCATCACCAGTGGTTATTACTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATAAGCTACGACGGTAGCAATAACTACAACCCATCTCTCAAAAATCGAATCTCCATCACTCGTGACACATCTAAGAACCAGTTTTTCCTGAAGTTGAATTCTGTGACTACTGAGGACACAGCTACATATTACTGTGCAAGAGGTTACGTTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGTCCATAGGTTCCTA-1_contig_3": "GGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGG",
        "CATCGGGCACTGTGTA-1_contig_1": "TGGGAGTCAGCTTCACACATGTTACACAGAGACATGAACATGCTCACTCAGCTCCTGGGATTACTGCTGCTCTGGTTTGCAGGTGGTAAATGTGACATTCAGATGACCCAGTCTCCTGCCTCCCAGTCTGCATCTCTGGGAGAAAGTGTCACCATCACATGCCTGGCAAGTCAGACCATTGGTACATGGTTAGCATGGTATCAGCAGAAACCAGGGAAATCTCCTCAGCTCCTGATTTATGCTGCAACCAGCTTGGCAGATGGGGTCCCATCAAGGTTCAGTGGTAGTGGATCTGGCACAAAATTTTCTTTCAAGATCAGCAGCCTACAGGCTGAAGATTTTGTAAGTTATTACTGTCAACAACTTTACAGTACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCTCAGAGGAAGGTGGAAACACGGTGAGAGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACTGCCGAAAATAAACCACAGTGACTGAAACCC",
        "GGACAAGCAGTATGCT-1_contig_1": "GGTGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AACCGCGAGAAACGCC-1_contig_2": "GGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCTCCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "ACATGGTAGATGCCTT-1_contig_3": "GAGCTCTGACAGAGGAGCCAAGCCCTGGATTCCCAGGTCCTCACATTCAGTGATCAGCACTGAACACAGACCACTCACCATGGACTCCAGGCTCAATTTAGTTTTCCTTGTCCTTATTTTAAAAGGTGTCCAGTGTGATGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGCAGCCTGGAGGGTCCCGGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTTTGGAATGCACTGGGTTCGTCAGGCTCCAGAGAAGGGGCTGGAGTGGGTCGCATACATTAGTAGTGGCAGTAGTACCATCTACTATGCAGACACAGTGAAGGGCCGATTCACCATCTCCAGAGACAATCCCAAGAACACCCTGTTCCTGCAAATGACCAGTCTAAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGAGAGGACTACGGGGTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAG",
        "CGGACTGTCACATAGC-1_contig_1": "TGGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACCA",
        "GCACTCTTCCCATTTA-1_contig_2": "GGGACATACCAGCAAGCGAGTGACCAGTTAGTCTTAAGGCACCACTTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCAAGAGAGGGTAACTACTCATATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TCAGCTCAGAGTGACC-1_contig_3": "GGGGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTTATTTCCAGCTTGGTCCCCCCTCCGAACGTGTACGG",
        "ATAACGCGTATAAACG-1_contig_1": "GAAGGGCAGCTGCCAGGAGCCTAAGAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGTTATGGGTACTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCAAAAGTGTCAGTACATCTGGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCACAGTAGGGAGCTTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAA",
        "TTGGCAATCACATGCA-1_contig_1": "GAAAAGTGATGTACTTATGAGAAGAGAAGTAATTAGCCAGGGAACAAAATTCAAATACACAATGGATTTTCTGGTGCAGATTTTCAGCTTCTTGCTAATCAGTGCCTCAGTTGCAATGTCCAGAGGAGAAAATGTGCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAAAAGGTCACCATGACCTGCAGGGCCAGCTCAAGTGTAAGTTCCAGTTACTTGCACTGGTACCAGCAGAAGTCAGGTGCCTCCCCCAAACTCTGGATTTATAGCACATCCAACTTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGTGTGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTACAGTGGTTACCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TGTGGTATCACCCGAG-1_contig_2": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "ACGAGCCGTCCAAGTT-1_contig_4": "GGGAGCTGCCAGGAGCCTAAGAAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACAGGTAACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATATCCTGCAGAGCCAGTGAAAGTGTTGATAGTTATGGCAATAGTTTTATGCACTGGTACCAGCAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTAGGACAGACTTCACCCTCACCATTGATCCTGTGGAGGCTGATGATGCTGCAACCTATTACTGTCAGCAAAATAATGAGGATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "AAGGAGCGTCATATGC-1_contig_1": "CATATGCACGGCACCATTTTCTTATATGGGGATTGTCATTGCAGTCAGGACTCAGCATGGACATGAGGGCTCCTGCACAGATTTTTGGCTTCTTGTTGCTCTTGTTTCCAGGTACCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCTTATCTGCCTCTCTGGGAGAAAGAGTCAGTCTCACTTGTCGGGCAAGTCAGGACATTGGTAGTAGCTTAAACTGGCTTCAGCAGGAACCAGATGGAACTATTAAACGCCTGATCTACGCCACATCCAGTTTAGATTCTGGTGTCCCCAAAAGGTTCAGTGGCAGTAGGTCTGGGTCAGATTATTCTCTCACCATCAGCAGCCTTGAGTCTGAAGATTTTGTAGACTATTACTGTCTACAATATGCTAGTTCTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "TACAGTGAGATGCCTT-1_contig_1": "GGTCCATTCAACAAAGCCTCCATACCCACTATCATATTTGGTACAATTTGCATGCCCCCATTACCACATCAGCCTCCCTGATGACATATAATGCATGCCATCAAGCTGCATTGGAAGCTGGCTATCACTCAAGGCCATGACCATGCTCTCACTAGCTCCTCTCCTCAGCCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCGTGGCTACAGGAGAAAAAGTCACTATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGCACAGATTTTGTTTTTACAATTGAAAACACGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACATGCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCACCATCTCATGCAGGGCCAGCCA",
        "CATATTCTCTGTCTAT-1_contig_3": "GGACATAACAGCAAGAGAGTGTCCGGTTAGTCTCAAGGAAGACTGAGACACAGTCTTAGATATCATGGAATGGCTGTGGAACTTGCTATTTCTCATGGCAGCAGCTCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAGGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAACTGCTGGAATGCAGTGGGTGCAAAAGATGCCAGGAAAGGGTTTGAAGTGGATTGGCTGGATAAACACCCACTCTGGAGTGCCAAAATATGCAGAAGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCATATTTACAGATAAGCAACCTCAAAAATGAGGACACGGCTACGTATTTCTGTGCGAGAAGGGGGTATCGGTACGACGCCTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAG",
        "ACTGAGTCATGGTCAT-1_contig_2": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "GTGCGGTAGACGCAAC-1_contig_2": "GGGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "AACGTTGGTATTCGTG-1_contig_3": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAG",
        "TGACTTTGTTAAGGGC-1_contig_1": "GAGGTCACAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTG",
        "GCACTCTAGGTCATCT-1_contig_7": "GGGTGAAAGCTCACATCTCTCTCATTAGAGGTTGATCTTTGAGGAAAACAGGGTGTTGCCTAAAGGATGAAAGTGTTGAGTCTGTTGTACCTGTTGACAGCCATTCCTGGTATCCTGTCTGATGTACAGCTTCAGGAGTCAGGACCTGGCCTCGTGAAACCTTCTCAGTCTCTGTCTCTCACCTGCTCTGTCACTGGCTACTCCATCACCAGTGGTTATTACTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATAAGCTACGACGGTAGCAATAACTACAACCCATCTCTCAAAAATCGAATCTCCATCACTCGTGACACATCTAAGAACCAGTTTTTCCTGAAGTTGAATTCTGTGACTACTGAGGACACAGCTACATATTACTGTGCAAATAGGTACGACGAGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TGTTCCGGTGGTAACG-1_contig_1": "GATGTTGTAGTGTCTGACCAGTCTCCTAAAGGCGCCTCATCAAAATGAAGCTGCCTGTTCTGCTAGTGGTGCTGCTATTGTTCACGAGTCCAGCCTCAAGCAGTGATGTTGTTCTGACCCAAACTCCACTCTCTCTGCCTGTCAATATTGGAGATCAAGCCTCTATCTCTTGCAAGTCTACTAAGAGTCTTCTGAATAGTGATGGATTCACTTATTTGGACTGGTACCTGCAGAAGCCAGGCCAGTCTCCACAGCTCCTAATATATTTGGTTTCTAATCGATTTTCTGGAGTTCCAGACAGGTTCAGTGGCAGTGGGTCAGGAACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTTCCAGAGTAACTATCTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCT",
        "CGGACTGGTTTGACTG-1_contig_3": "AAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TTGCGTCCACTTACGA-1_contig_4": "GGGAGCTCTGACAGAGGAGGCCAGTCCTGGATTTGAGTTCCTCACATTCAGTGATGAGCACTGAACACGGACCCCTCACCATGAATTTCGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCTGTGTGAAGTGAAGCTGGTGGAGTCTGGGGGAGGTTTAGTGCAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATACCATGTCTTGGGTTCGCCAGACTCCAGAGAAGAGGCTGGAGTGGGTCGCATACATTAGTAATGGTGGTGGTAGCACCTACTATCCAGACACTGTAAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACGGCCATGTATTACTGTGCAAGACATGATGGTTACTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GGAAAGCGTCCTGCTT-1_contig_1": "GGGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGG",
        "ACTGTCCTCGGAAATA-1_contig_2": "GGTAGACTGTAATGAACTGGAATGAGCTGGGCCGCTAAGCTAAACTAGGCTGGCTTAACCGAGATGAGCCAAACTGGAATGAACTTCATTAATCTAGGTTGAATAGAGCTAAACTCTACTGCCTACACTGGACTGTTCTGAGCTGAGATGAGCTGGGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "ACGAGCCGTCCAAGTT-1_contig_2": "GCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCC",
        "GTCGTAAAGACACGAC-1_contig_2": "GAGAAAAACATGAGATCACAGTTCTCTCTACAGTTACTGAGCACACAGGACCTCACCATGGGATGGAGCTATATCATCCTCTTTTTGGTAGCAACAGCTACAGATGTCCACTCCCAGGTCCAACTGCAGCAGCCTGGGGCTGAACTGGTGAAGCCTGGGGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTGGATGCACTGGGTGAAGCAGAGGCCTGGACAAGGCCTTGAGTGGATTGGAGAGATTAATCCTAGCAACGGTCGTACTAACTACAATGAGAAGTTCAAGAGCAAGGCCACACTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGATATGGTAACTACGGGGTATATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAG",
        "CATGACAAGCCAGGAT-1_contig_5": "GGTATGAACCTAGCCCTGATTTCCCCAGCCTTCAGTTCCCAGATTCAGTGATCAGCCTTGAACACAGACCTGTCACCATGAAGTTGTGGCTGAACTGGATTTTCCTTGTAACACTTTTAAATGGTATCCAGTGTGAGGTGAAGCTGGTGGAGTCTGGAGGAGGCTTGGTACAGCCTGGGGGTTCTCTGAGACTCTCCTGTGCAACTTCTGGGTTCACCTTCACTGATTACTACATGAGCTGGGTCCGCCAGCCTCCAGGAAAGGCTCTTGAGTGGTTGGGTTTTATTAGGAACAAAGCTAATGGTTACACAACAGAGTATAGTGCATCTGTGAAGGGTCGGTTCACCATCTCCAGAGATAATTCCCAAAGCATCCTCTATCTTCAAATGAACACCCTGCGAGCTGAGGACAGTGCCACTTATTACTGTGCAAGAGATTACTCGTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCC",
        "CACATTTAGCTCTCGG-1_contig_4": "GACTTATGAGCACTGTTTTCTCTACAGTCACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTCATCTTCTTCCTGATGGCAGTGGTTACAGGGGTCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCTGAGCTTGTGAGGCCAGGGGCCTTAGTCAAGTTGTCCTGCAAAGCTTCTGGCTTCAACATTAAAGACTACTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGATGGATTGATCCTGAGAATGGTAATACTATATATGACCCGAAGTTCCAGGGCAAGGCCAGTATAACAGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTGCTATTACTACGGTAGAGGCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CATCGGGCACTGTGTA-1_contig_2": "TCTGACAGAGGAGGCCAGCCCTGGATTCGATTCCCAGTTCCTCACATTCAGTGATGAGCACTGAACACAGACCTCTCAACATGAACTTAGGGCTCAGCTTCATTTTCCTTGCCCTTATTTTAAAAGGTGTCCAGTGTGAGGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGCAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGGCATGTCTTGGGTTCGCCAGACTCCAGACAAGAGGCTGGAGTTGGTCGCAACCATTAATAGTAATGGTGGTAGCACCTATTATCCAGACAGTGTGAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTGCAATCTACTATGGTTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GTATCTTAGTGACTCT-1_contig_7": "TGGGGATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGAATGGGTTACTACGGCCCCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCCTTGTCAAAGGCTACTTCCCTGAGCCGGTAACTGTAAAATGGAACTATGGAGCCCTGTCCAGC",
        "GAGCAGATCTGGCGTG-1_contig_1": "GAGCTGCCAGGAGCCTAAGAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGTTATGGGTACTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCAAAAGTGTCAGTACATCTGGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCACAGTAGGGAGCTTCCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GCAAACTCATGCAATC-1_contig_2": "GGGATCATCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGCATTACTCTTCTGCCTGGTAACATTCCCAAGCTGTATCCTTTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCGTCTCAGGGTTCTCATTAACCGGCTATGGTGTAAACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAATGATATGGGGTGATGGAAGCACAGACTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCAGGTACTACTGTGCCCCACGGGGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCAC",
        "AACTTTCCAGCGAACA-1_contig_2": "GGAGAAATAGTGCAGAGAGTACATGGTGTTGAATCAACTCCATGGGGTACCTGCCTGACAAGAAGACAAGCCTATAAAAGGATTACCACCCACTGCTTCTCAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GATTCAGTCTACTATC-1_contig_3": "GGGGGGCAGAAAGTCACTCTCAGTGAGGATACACCATCAGCATGAGGGTCCTTGCTGAGCTCCTGGGGCTGCTGCTGTTCTGCTTTTTAGGTGTGAGATGTGACATCCAGATGAACCAGTCTCCATCCAGTCTGTCTGCATCCCTTGGAGACACAATTACAATCACTTGCCATGCCAGTCAGAACATTAATGTTTGGTTAAGCTGGTACCAGCAGAAACCAGGAAATATTCCTAAACTATTGATCTATAAGGCTTCCAACTTGCACACAGGCGTCCCATCAAGGTTTAGTGGCAGTGGATCTGGAACAGGTTTCACATTAACCATCAGCAGCCTGCAGCCTGAAGACATTGCCACTTACTACTGTCAACAGGGTCAAAGTTATCCTCTAACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCA",
        "GATTCAGTCTACTATC-1_contig_2": "GACCTCTCCAATGTTTCTTATGACACCTCCCCCAGCAGCCACTGTCCAGGCACTCAGAGAGCAAGGGATGTTGGGAAGACAGACTTGGGACAGCCCCAGGTATATTTACTTATATAGGACCTCCCTGGGAGATAGAATCCCAGGAGAAAAAAGGTTGGATGTATATTAAGGATGGCCCCTAACACTCTGCACTGCTACCTCTGGCCCCACCAGACAATGTTCCTGCAGAACCTGTTACCTTACTTGGCAGGGATTTTTGTCAAGGGATCTACTGCCCCTACGGCTGGCCTACTATAGGCTCATGGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGCCAAAACAACACCCCCATCAGTCTATCCACTGGCCCCTGGGTGTGGAGATACAACTGGTTCCTCCGTGACTCTGGGATGCCTGGTCAAGGG",
        "GGTGTTATCCGTAGGC-1_contig_5": "GGGGACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "TACTCATCATCTACGA-1_contig_2": "TGGGGAAATGCATCACACCAGCATGGGCATCAAAATGGAGTCACAGATTCAGGTCTTTGTATTCGTGTTTCTCTGGTTGTCTGGTGTTGACGGAGACATTGTGATGACCCAGTCTCACAATTTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGAGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGACAATCTCCTAAATTACTGATTTACTCGGCATCCTACCGGTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATCTGGGACGGATTTCACTTTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCACTTTATTACTGTCAGCAGCATTATAGTACTCCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TTTGTCATCGCTTGTC-1_contig_5": "GGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGG",
        "TAGGCATGTTACGGAG-1_contig_1": "TGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCGGAGCCTTGTACACAATAATGGAAACACCTATTTACATTGGTACCTGCAGAGGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGGGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCC",
        "TACTCGCAGGATGTAT-1_contig_3": "GATGTACTTATGAGAAGAGAAGTAATTAGCCAGGGAACAAAATTCAAATACACAATGGATTTTCTGGTGCAGATTTTCAGCTTCTTGCTAATCAGTGCCTCAGTTGCAATGTCCAGAGGAGAAAATGTGCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAAAAGGTCACCATGACCTGCAGGGCCAGCTCAAGTGTAAGTTCCAGTTACTTGCACTGGTACCAGCAGAAGTCAGGTGCCTCCCCCAAACTCTGGATTTATAGCACATCCAACTTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGTGTGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTACAGTGGTTACCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GATGAGGGTTACCAGT-1_contig_1": "GGGGGAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGTTATGGGTACTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCAAAAGTGTCAGTACATCTGGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCACAGTAGGGAGCTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGAC",
        "GTGCTTCGTTACCAGT-1_contig_1": "GAGACGCCCCGCTACCATATACAGGAAGATACGATTTAAATTTTGATAGGTGGGTCACAGTCAACGGCTATAAAGTGTTGTACAGATCCCTCCCCTTTCGTGAAAGACTCGCCAGAGCTAGACCTCCTTGGTGTGTGTTAACTCAGGAAGAAAAAGACGACATAAAACAACAGCCAGGACTCAGCATGGACATGAGGACCCCTGCTCAGTTTCTTGGAATCTTGTTGCTCTGGTTTCCAGGTATCAAATGTGACATCAAGATGACCCAGTCTCCATCTTCCATGTATGTATCTCTAGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAATAGCTATTTAAGCTGGTTCCGGCAGAAACCAGGGAAATCTCCTAAGACCCTGATCTATCGTGCAAACAGATTGGTGGATGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTCACCATCAGCAGCCTGGAGTATGAAGATATGGGAATTTATTATTGTCTACAGTATGATGAGTTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCTCAGAGGAAGGTGGAAACACGGTGAGAGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACCGCCGAAAACATAACACAATGACTAAAACCCAACCCAAAACCTATGGGGCCAGCACCTGGTCCT",
        "CGTAGGCTCGTCCAGG-1_contig_1": "TGGGGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTCCTCAGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "CTAGCCTCATTCCTCG-1_contig_9": "GGAGTTCAACCTTTTAATTTTAGCTTGAGTAGTTCTAGTTTCCCCAAACTTAAGTTTATCGACTTCTAAAATGTATTTAGAATTCATTTTCAAAATTAGGTTATGTAAGAAATTGAAGGACTTTAGTGTCTTTAATTTCTAATATATTTAGAAAACTTCTTAAAATTACTCTATTATTCTTCCCTCTGATTATTGGTCTCCATTCAATTCTTTTCCAATACCCGAAGCATTTACAGTGACTTTGTTCATGATCTTTTTTAGTTGTTTGTTTTGCCTTACTATTAAGACTTTGACATTCTGGTCAAAACGGCTTCACAAATCTTTTTCAAGACCACTTTCTGAGTATTCATTTTAGGAGAAATACTTTTTTTTTAAATGAATGCAATTATCTAGACTTATTTCGGTTGAACATGCTGGTTGGTGGTTGAGAGGACACTCAGTCAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TGTGGTATCACCCGAG-1_contig_1": "TGGGGAGCTGCCAGGAGCCTAATAAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACAGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATATCCTGCAGAGCCAGTGAAAGTGTTGATAGTTATGGCAATAGTTTTATGCACTGGTACCAGCAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCGTGCATCCAACCTAGAATCTGGGATCCCTGCCAGGTTCAGTGGCAGTGGGTCTAGGACAGACTTCACCCTCACCATTAATCCTGTGGAGGCTGATGATGTTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CGAGCACGTCATGCCG-1_contig_14": "GGGCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAG",
        "TAAGCGTAGGATGGTC-1_contig_2": "GAGTATGAACTGAGCAGCATCAGACAGGCAGGGGAAGCAAGATGGATTCACAGGCCCAGGTTCTTATGTTACTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTAGCTGTGTCAGTTGGAGAGAAGGTTACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "ATAAGAGGTACCGTAT-1_contig_1": "GGGACTACTCAAGACTTTTTGTATCAAGTTCTCAGAATGAGGTGCCTAGCTGAGTTCCTGGGGCTGCTTGTGCTCTGGATCCCTGGAGCCATTGGGGATATTGTGATGACTCAGGCTGCACCCTCTGTACCTGTCACTCCTGGAGAGTCAGTATCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTGCATAGTAATGGCAACACTTACTTGTATTGGTTCCTGCAGAGGCCAGGCCAGTCTCCTCAGCTCCTGATATATCGGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTGGCAGTGGGTCAGGAACTGCTTTCACACTGAGAATCAGTAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTATGCAACATCTAGAATATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GAAGCAGCATGGTCAT-1_contig_3": "GAGTTGATGGTGACAATTTCAGGGTCAGTGACTGTCTGGTTTCTCTGAGGTGAGGCTGGAATATAGGTCACCTTGAAGACTTAAGAGGGGTCCAGGGGGCTTCTGCACAGGCAGGGAACAGAATGTGGAACAATGACTTGAATGGTTGATTCTTGTGTGACACCAGGAATTGGCATAATGTCTGAGTTGCCCAGGGGTGATTCTAGTCAGACTCTGGGGTTTTTGTCGGGTATAGAGGAAAAATCCACTATTGTGATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "ACGTCAACACCACCAG-1_contig_2": "GGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATGAGTGCCTCAGTCATAATGTCCAGGGGACAAATTGTTCTCACCCAGTCTCCAGCACTCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACTGGTACCAGCAGAAGCCAAGATCCTCCCCCAAACCCTGGATTTATCTCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAA",
        "AAGGAGCGTCATATGC-1_contig_8": "AGGGGTCACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAATATATGATTACGACGAGGACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CTGCCTATCAAACAAG-1_contig_5": "GAGGCAGAGAACTTTAGCCCTGTCTTCTTTTTTAGTGTTCAGCACTGACAATATGACATTGAACATGCTGTTGGGGCTGAAGTGGGTTTTCTTTGTTGTTTTTTATCAAGGTGTGCATTGTGAGGTGCAGCTTGTTGAGTCTGGTGGAGGATTGGTGCAGCCTAAAGGGTCATTGAAACTCTCATGTGCAGCCTCTGGATTCACCTTCAATACCTACGCCATGAACTGGGTCCGCCAGGCTCCAGGAAAGGGTTTGGAATGGGTTGCTCGCATAAGAAGTAAAAGTAATAATTATGCAACATATTATGCCGATTCAGTGAAAGACAGGTTCACCATCTCCAGAGATGATTCACAAAGCATGCTCTATCTGCAAATGAACAACTTGAAAACTGAGGACACAGCCATGTATTACTGTGTGAGACAAGGGACAGCTCGGGCCCCATATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AGGCCGTAGATCCGAG-1_contig_7": "GAGTTCTAGAGCTTCCAGGTGTGCTGTCAGGCTGCTAGTGTATGCTCTCTCTAGTTTCTTTTTGGAGGCACTCAGAGCTATGAGTTTTCCTCTTAGGACTGCCTTCATTGTGTCCCATAAGTTTGGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GTCCTCATCAGCCTAA-1_contig_2": "TGGGGAAATACATCAGATCAGCATGGGCATCAAGATGGAGTCACAGACTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGATGGAGACATTGTGATGACCCAGTCTCAAAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCGTCACCTGCAAGGCCAGTCAGAATGTGGGTACTAATGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAAGCACTGATTTACTCGGCATCCTACCGGTACAGTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAATGTGCAGTCTGAAGACTTGGCAGAGTATTTCTGTCAGCAATATAACAGCTATCCTTGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "ACGTCAAGTCTCCACT-1_contig_6": "GGGCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTG",
        "CACAGGCGTTGGTTTG-1_contig_8": "GGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CTCGGAGCAATGGAAT-1_contig_1": "GGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "ATGTGTGAGCTAAGAT-1_contig_9": "GGTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "CAGCCGATCTACTATC-1_contig_1": "TCTGGGGAAGCTGGCTATCACTCAAGGCCATGACCATGTTCTCACTAGCTCTTCTCCTCAGTCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCATGGCTATAGGAGAAAGAGTCACCATCAGATGCATAACCAGCACTGATATTAATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCTTCCCGATTCTCCAGCAGTGGCTTTGGTACAGATTTTGTTTTTACAATTGAAAACATGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACTTGCCGGTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCTCAGAGGAAGGTGGAAACACGGTGAGAGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACCGCCGAAAACATAACACAATGACTAAAACCCAACCCAAAACCTATGGGGCCAGC",
        "AGAATAGCAGTATGCT-1_contig_3": "GAACCTTGAATCTGAAGCAAAGGGGATCAGCCCGAGATTCTCATTCAGTGATCAACACTGAACACACATCCCTTACCATGGATTTTGGGCTGATTTTTTTTATTGTTGCTCTTTTAAAAGGGGTCCAGTGTGAGGTGAAGCTTCTCGAGTCTGGAGGTGGCCTGGTGCAGCCTGGAGGATCCCTGAAACTCTCCTGTGCAGCCTCAGGATTCGATTTTAGTAGATACTGGATGAGTTGGGTCCGGCAGGCTCCAGGGAAAGGGCTAGAATGGATTGGAGAAATTAATCCAGATAGCAGTACGATAAACTATACGCCATCTCTAAAGGATAAATTCATCATCTCCAGAGACAACGCCAAAAATACGCTGTACCTGCAAATGAGCAAAGTGAGATCTGAGGACACAGCCCTTTATTACTGTGCAAGAGGGGACGGTAGTAGCTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AGGCCGTAGATCCGAG-1_contig_4": "GGGATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGAGATTACTACGGTAGTAGCTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CGTCAGGTCTGCAAGT-1_contig_6": "GGGAAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGCCTCCTAAACTGTTGATCTACGGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGAACCGATTTCACTCTTACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGAATGATCATAGTTATCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TCGCGTTCAGATCTGT-1_contig_6": "AGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGG",
        "ACGCCAGGTGTCAATC-1_contig_2": "GATGTTGGTCAGGAAGTCATCCAGAAACAGACCTCTCCAATGTTTCTTATGACACCTCCCCCAGCAGCCACTGTCCAGGCACTCAGAGAGCAAGGGATGTTGGGAAGACAGACTTGGGACAGCCCCAGACAATGTTCCTGCAGAACCTGTTACCTTACTTGGCAGGGATTTTTGTCAAGGGATCTACTACTGTGCCTACTATAGGTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AGCATACTCTCCGGTT-1_contig_1": "GGGTCACAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGG",
        "GACTACAAGTTTCCTT-1_contig_1": "GGGGATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGTGCAAGTCAGGGCATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTATTACACATCAAGTTTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGGACAGATTATTCTCTCACCATCAGCAACCTGGAACCTGAAGATATTGCCACTTACTATTGTCAGCAGTATAGTAAGCTTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAG",
        "CGTGAGCGTACCGAGA-1_contig_4": "GGGGCTTTCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AACTGGTAGGACTGGT-1_contig_2": "GGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAAC",
        "TAAACCGGTCGACTGC-1_contig_4": "GCTCTGACAGAGGAGGCCTGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACGGACCCCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCAGTGTGAAGTGATGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGCCATGTCTTGGGTTCGCCAGACTCCGGAGAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGGCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGACATGGGGCGACGAACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACC",
        "GACCTGGAGATCTGCT-1_contig_1": "GATCAGCATCCTCTCCACAGACACTGAAAACTCTGACTCACAATGGAAAGGCACTGGATCTTTCTCTTCCTGTTTTCAGTAACTGCAGGTGTCCACTCCCAGGTCCAGCTTCAGCAGTCTGGGGCTGAACTGGCAAAACCTGGGGCCTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACCTTTACTAGCTACTGGATGCACTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATTGGATACATTAATCCTAGCACTGGTTATACTGAGTACAATCAGAAGTTCAAGGACAAGGCCACATTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAACTGAGCAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGGGGGGGCTACTATGGTAACTACGTGGGCTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACCA",
        "TGAGAGGAGGTGCTTT-1_contig_5": "GAGCACTGTCCAGATGAAGCTGATTTGCATGTGCTGAGATCATATTCTACTGCCCCAGAGATTTAATAATCTGATCATACACACTCCAACAGTCATTCTTGGTCAGGAGACGTTGTAGAAATGAGACCGTCTATTCAGTTCCTGGGGCTCTTGTTGTTCTGGCTTCATGGTGCTCAGTGTGACATCCAGATGACACAGTCTCCATCCTCACTGTCTGCATCTCTGGGAGGCAAAGTCACCATCACTTGCAAGGCAAGCCAAGACATTAACAAGTATATAGCTTGGTACCAACACAAGCCTGGAAAAGGTCCTAGGCTGCTCATACATTACACATCTACATTACAGCCAGGCATCCCATCAAGGTTCAGTGGAAGTGGGTCTGGGAGAGATTATTCCTTCAGCATCAGCAACCTGGAGCCTGAAGATATTGCAACTTATTATTGTCTACAGTATGATAATCTTGGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCACA",
        "TATGCCCAGGCCCTCA-1_contig_4": "GGGCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "ATAAGAGGTACCGTAT-1_contig_2": "GGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTCCTGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GCGAGAATCGTATCAG-1_contig_2": "GGAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTTATTTCCAGCTTGGTCCCCCCTCCGAACGTGTACGGGCTCTCACCATGCTGTAGAC",
        "GATCGCGTCTATCCCG-1_contig_3": "TGGGATCCTCTTCTCATAGAGCCTCCATCAGACCATGGCTGTCCTGGCACTGCTCCTCTGCCTGGTGACATTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACTGTCTCAGGGTTCTCATTAACCAGCTATGGTGTAAGCTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGGTGACGGGAGCACAAATTATCATTCAGCTCTCATATCCAGACTGAGCATCAGCAAGGATAACTCCAAGAGCCAAGTTTTCTTAAAACTGAACAGTCTGCAAACTGATGACACAGCCACGTACTACTGTGCCAAGATGAGTAAGACAGCTCGGGCTCAGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAA",
        "GGAAAGCCATACCATG-1_contig_1": "GGGGACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGATCTGGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GTCTTCGTCTAACTGG-1_contig_4": "GGCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GGGAATGGTAAGCACG-1_contig_7": "GGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "AAAGTAGCAAATCCGT-1_contig_3": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TGTGGTACATCCAACA-1_contig_4": "GACATACCAGCAAGGGAGTGACCAGTTAGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGTTATACCTTCACAGACTATTCAATGCACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACTGAGACTGGTGAGCCAACATATGCAGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCCCCTACGGTAGTAGCTAGGGGGTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "ACGCCAGGTACCGGCT-1_contig_1": "TGGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "AGCATACGTAGCGTAG-1_contig_1": "GGGGATCAGACAGGCAGTGGGAGCAAGATGGATTCACAGGCCCAGGTTCTTATATTGCTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTGGCTGTGTCAGCAGGAGAGAAGGTCACTATGAGCTGCAAATCCAGTCAGAGTCTGCTCAACAGTAGAACCCGAAAGAACTACTTGGCTTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATCTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGCAAGCAATCTTATAATCTATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "AGCATACTCTCCGGTT-1_contig_3": "GCTCTTAGAGTTTTATAAGAAGGTCTTTCTCAGCTCTGTGTCATAATCCACAGAAGCTCAGGGAAAATTTGAAGATGGTGTCCACTTCTCAGCTCCTTGGACTTTTGCTTTTCTGGATTTCAGCCTCCAGATGTGACATTGTGATGACTCAGTCTCCAGCCACCCTGTCTGTGACTCCAGGAGATAGAGTCTCTCTTTCCTGCAGGGCCAGCCAGAGTATTAGCGACTACTTACACTGGTATCAACAAAAATCACATGAGTCTCCAAGGCTTCTCATCAAATATGCTTCCCAATCCATCTCTGGGATCCCCTCCAGGTTCAGTGGCAGTGGATCAGGGTCAGATTTCACTCTCAGTATCAACAGTGTGGAACCTGAAGATGTTGGAGTGTATTACTGTCAAAATGGTCACAGCTTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCT",
        "CTCGAAAGTCGTTGTA-1_contig_3": "GCACTTCAAGTGAGGTCACAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGG",
        "CGTTCTGCACGTCAGC-1_contig_4": "GGCATATGATCAGTATCCTCTTCACAGTCACTGAAAACATTGACTCTAATCATGGAATGTAACTGGATACTTCCTTTTATTCTGTCGGTAACTTCAGGGGTCTACTCAGAGGTTCAGCTCCAGCAGTCTGGGACTGTGCTGGCAAGGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACCTTTACCAGCTACTGGATGCACTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATTGGCGCTATTTATCCTGGAAATAGTGATACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCAAACTGACTGCAGTCACATCCACCAGCACTGCCTACATGGAGCTCAGCAGCCTGACAAATGAGGACTCTGCGGTCTATTACTGTACAAGAGGGGGCGACTGGAGGGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGCCAAAACAACACCCCCATCAGTCTATCCACTGGCCCCTGGGTGTGGAGATACAACTGGTTCCTCCGTGACTCTGGGATGCCTGGTCAAGGG",
        "GCGAGAACAGTCGATT-1_contig_3": "CTGGTTTGCCTTAGTTAAGAGACCATGTGTTTATAAGCCAAGAACTGCGTAGCCTATATCTGGATATTTGCATACTTCATCTTCAGTAGCCACAAATATCTCACAGTTGTTTTAAAGTGATGTACTTATGAGAAGAGAAGTAATTAGCCAGGGAACAAAATTCAAATACACAATGGATTTTCTGGTGCAGATTTTCAGCTTCTTGCTAATCAGTGCCTCAGTTGCAATGTCCAGAGGAGAAAATGTGCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAAAAGGTCACCATGACCTGCAGGGCCAGCTCAAGTGTAAGTTCCAGTTACTTGCACTGGTACCAGCAGAAGTCAGGTGCCTCCCCCAAACTCTGGATTTATAGCACATCCAACTTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGTGTGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTACAGTGGTTACCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CGTCACTCAAGAGGCT-1_contig_4": "GGGGCCTGTGAACTCACAAGTGTGCAGACATGGACAGGCTTACTTCTTCATTCCTGCTGCTGATTGTCCCTGCATATGTCATGTCCCAAGTTACTCTAAAAGAGTCTGGCCCTGGGATATTGAAGCCCTCACAGACCCTCAGTCTGACTTGTTCTTTCTCTGGGTTTTCACTGAGCACTTCTGGTATGGGTGTAGGCTGGATTCGTCAGCCTTCAGGGAAGGGTCTGGAGTGGCTGGCACACATTTGGTGGAATGATGATAAGTACTATAACCCATTCCTGAAGAGCCAGCTCACAATCTCCAAGGATACCTCCAGAAACCAGGTATTCCTCAAGATCACCAGTGTGGACACTGGAGATACTGCCACTTACTACTGTGCTCGAGGAGAAGGACTAAAGGCGTACTACTTTAACTACTGGGTCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GCGCGATTCAACGGGA-1_contig_7": "CTGTGATCGTTTCTTATATGGGGATCCTCTTCTCATAGAGCCTCCATCAGACCATGGCTGTCCTGGCACTGCTCCTCTGCCTGGTGACATTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACTGTCTCAGGGTTCTCATTAACCAGCTATGGTGTAAGCTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGGTGACGGGAGCACAAATTATCATTCAGCTCTCATATCCAGACTGAGCATCAGCAAGGATAACTCCAAGAGCCAAGTTTTCTTAAAACTGAACAGTCTGCAAACTGATGACACAGCCACGTACTACTGTGCCAAATCCCGTCCCTGGTATGGTAACTACGACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GGCCGATCATGTTCCC-1_contig_7": "GACTGAAAACTCTGACTCAACATGGAAAGGCACTGGATCTTTCTCTTCCTGTTGTCAGTAACTGCAGGTGTCCACTCCCAGGTCCAGCTGCAGCAGTCTGCAGCTGAACTGGCAAGACCTGGGGCCTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACCTTTACTAGCTACACGATGCACTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATTGGATACATTAATCCTAGCAGTGGATATACTGAGTACAATCAGAAGTTCAAGGACAAGACCACATTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAACTGAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGATCGGAGGGCTACTATGGTTACGACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AATCCAGAGATGCGAC-1_contig_3": "TGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCTCAGAGGAAGGTGGAAACACGGTGAGAGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACCGCCGAAAACATAACACAATGACTAAAACCCAACCCAAAACCTATGGGGCCAGCACCTGGTCCT",
        "AACGTTGGTTATGCGT-1_contig_6": "ATCCATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCTTGGGGCTGCTCTTCTGCCTGGTGACATTCCCAAGCTGTGTCCTATCCCAGGTGCAGCTGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATCACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGTGGTGGAAGCACAGACTATAATGCAGCTTTCATATCCAGACTGAGCATCAGCAAGGACAATTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTAATGACACAGCCATATATTACTGTGCCATATGATGGTTACTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AACTTTCCAGCGAACA-1_contig_1": "TGGGAGCTGCCAGGAGCCTAAGAAGCATTCTCTCTTCCAGTTCTCAGAGATGGAGAAAGACACACTCCTGCTATGGGTCCTGCTTCTCTGGGTTCCAGGTTCCACAGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAGAGCCAGCGAAAGTGTTGATAATTATGGCATTAGTTTTATGAACTGGTTCCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAACCAAGGATCCGGGGTCCCTGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCAGCCTCAACATCCATCCTATGGAGGAGGATGATACTGCAATGTATTTCTGTCAGCAAAGTAAGGAGGTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GATGCTACACTAAGTC-1_contig_8": "TGGGGACATACCAGCAAGGGAGTGACCAGTTAGTCTTAAGGCACCATTGAGCCCAAGTCTTAGACATCATGGATTGGCTGTGGAACTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCAACACTGGAGAGCCAACATATGCTGAAGAGTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCAAGATGGGGGGACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CAGCTAAAGTCCAGGA-1_contig_1": "TGGGGCTTCCTCGAATAATCAGAACTCTTCACACCAAGTTCTCAGAATGAGGTGCTCTCTTCAGTTCCTGGGGGTGCTTATGTTCTGGATCTCTGGAGTCAGTGGGGATATTGTGATAACCCAGGATGAACTCTCCAATCCTGTCACTTCTGGAGAATCAGTTTCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTATATAAGGATGGGAAGACATACTTGAATTGGTTTCTGCAGAGACCAGGACAATCTCCTCAGCTCCTGATCTATTTGATGTCCACCCGTGCATCAGGAGTCTCAGACCGGTTTAGTGGCAGTGGGTCAGGAACAGATTTCACCCTGGAAATCAGTAGAGTGAAGGCTGAGGATGTGGGTGTGTATTACTGTCAACAACTTGTAGAGTATCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCTCAGAGGAAGGTGGAAACACGGTGAGAGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACCGCCGAAAACATAATGGGTGCTGTACC",
        "GACTGCGGTCGTGGCT-1_contig_1": "ATGGGGAAGCTGGCTATCACTCAAGGCCATGACCATGCTCTCACTAGCTCCTCTCCTCAGCCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCGTGGCTACAGGAGAAAAAGTCACTATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGCACAGATTTTGTTTTTACAATTGAAAACACGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACATGCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "ACGTCAAGTCTCCACT-1_contig_3": "GGCTTGATATTTGCATACTTCATCTTCAGTAGTCACAAATATCTCACAGCTGTTTTAAAGCGATGTACTTAGGAGAAGAGCAGTAATTAGCCAGGGAACAAAATTCAAATACACAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTATCTCAGTTGTAATGTCCAGAGGAGAAAATGTGCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCTAGGGGAGAAGGTCACCATGAGCTGCAGGGCCAGCTCAAGTGTAAATTACATGTACTGGTACCAGCAGAAGTCAGATGCCTCCCCCAAACTATGGATTTATTACACATCCAACCTGGCTCCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGAACTCTTATTCTCTCACAATCAGCAGCATGGAGGGTGAAGATGCTGCCACTTATTACTGCCAGCAGTTTACTAGTTCCCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GCTGCTTGTCAGATAA-1_contig_5": "GGGATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGAGATTACTACGGTAGTAGCTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GTGCTTCCAGGTCGTC-1_contig_5": "GAAAACATAAGGTCACTGTTCTCACTACAGTTACTGAGCACACAGACCTCACCATGGGATGGAGCTCTATCATCCTCTTCTTGGTAGCAACAGCTACAGGTGTCCACTCCCAGGTCCAACTGCAGCAGCCTGGGTCTGTGCTGGTGAGGCCTGGAGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTCCTGGATGCACTGGGCGAAGCAGAGGCCTGGACAAGGCCTTGAGTGGATTGGAGAGATTCATCCTAATAGTGGTAATACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGTAGACACATCCTCCAGCACAGCCTACGTGGATCTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGAGGGGGGAGGTACGGCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "ACCGTAATCTCTGCTG-1_contig_7": "GACATACCAGCAAGGGAGTGACCAGTTAGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGTTATACCTTCACAGACTATTCAATGCACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACTGAGACTGGTGAGCCAACATATGCAGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCTTCAGGATTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ACTGAACAGGACGAAA-1_contig_2": "TGGGGAAACATAAGGTCACTGTTCTCACTACAGTTACTGAGCACACAGACCTCACCATGGGATGGAGCTCTATCATCCTCTTCTTGGTAGCAACAGCTACAGGTGTCCACTCCCAGGTCCAACTGCAGCAGCCTGGGTCTGTGCTGGTGAGGCCTGGAGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTCCTGGATGCACTGGGCGAAGCAGAGGCCTGGACAAGGCCTTGAGTGGATTGGAGAGATTCATCCTAATAGTGGTAATACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGTAGACACATCCTCCAGCACAGCCTACGTGGATCTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAACTGGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AGGGAGTCAAGTAATG-1_contig_2": "GAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TAAACCGGTCGACTGC-1_contig_1": "CTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "CTAAGACAGGCGCTCT-1_contig_2": "GGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "AGCCTAAGTGTTTGTG-1_contig_1": "GAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGT",
        "ACGAGCCTCTGCAAGT-1_contig_1": "GAGCTGCCAGGAAGCTAAGAAGCATCCTCTCTTCTAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCCAAAGTGTCAGTACATCTAGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCAAGTATGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATACTGCAACATATTACTGTCAGCACAGTTGGGAGATTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCCTGGAGATGCAGACATGATTGCTGGAGACTGGGTGAGAACAATTTGTCCTCTGGACATTAT",
        "AGGTCATGTAGTGAAT-1_contig_3": "AAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CTAGTGATCTGCTGCT-1_contig_1": "GGGGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCTCCCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCAAC",
        "GCATGCGTCGCATGGC-1_contig_2": "GGGGATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AACTTTCCAGCGAACA-1_contig_3": "GGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AGAGTGGCACAGGCCT-1_contig_1": "TGGGGAATCAACCTCCTTAAAATGAATTTGCCTGTTCATCTCTTGGTGCTTCTGTTGTTCTGGATTCCTGCTTCCAGAGGTGATGTTGTGGTGACTCAAACTCCACTCTCCCTGCCTGTCAGCTTTGGAGATCAAGTTTCTATCTCTTGCAGGTCTAGTCAGAGTCTTGCAAACAGTTATGGGAACACCTATTTGTCTTGGTACCTGCACAAGCCTGGCCAGTCTCCACAGCTCCTCATCTATGGGATTTCCAACAGATTTTCTGGGGTGCCAGACAGGTTCAGTGGCAGTGGTTCAGGGACAGATTTCACACTCAAGATCAGCACAATAAAGCCTGAGGACTTGGGAATGTATTACTGCTTACAAGGTACACATCAGCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CACATAGAGATTACCC-1_contig_7": "AGCTCTGACAGAGGAGGCCTGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACGGACCCCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCAGTGTGAAGTGATGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGCCATGTCTTGGGTTCGCCAGACTCCGGAGAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGGCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGACAGGGTTACTACGGTAGTAGCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAA",
        "TTCTACACATGCCCGA-1_contig_6": "GTATAGGACCTCCCTGGGGAGATAGAATCCCAGGAGTAAAAAGGTTGGATGTATATTAAGGATGGCCCCTGACACTCTGCACTGCTACCTCTGGCCCCACCAATGTTCCTGCAGAACCTGTTACCTTACTTGGCAGGGATTTTTGTCAAGGGATCTACTACTGTGTCTACTATGGTTACGACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACC",
        "GACGGCTTCACATAGC-1_contig_3": "GGGGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CGTAGGCCACTTAACG-1_contig_1": "ACAGGACACTTTCTTATATGGGAGGACAATGGCAGAAAGTCACTCTCAGTGAGGATACACCATCAGCATGAGGGTCCTTGCTGAGCTCCTGGGGCTGCTGCTGTTCTGCTTTTTAGGTGTGAGATGTGACATCCAGATGAACCAGTCTCCATCCAGTCTGTCTGCATCCCTTGGAGACACAATTACCATCACTTGCCATGCCAGTCAGAACATTAATGTTTGGTTAAGCTGGTACCAGCAGAAACCAGGAAATATTCCTAAACTATTGATCTATAAGGCTTCCAACTTGCACACAGGCGTCCCATCAAGGTTTAGTGGCAGTGGATCTGGAACAGGTTTCACATTAACCATCAGCAGCCTGCAGCCTGAAGACATTGCCACTTACTACTGTCAACAGGGTCAAAGTTATCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AGCATACGTCCAGTGC-1_contig_4": "GGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGCTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CCTACCATCTTTAGTC-1_contig_7": "GACATCTCTCTCATTAGAGGTTGATCTTTGAGGAAAACAGGGTGTTGCCTAAAGGATGAAAGTGTTGAGTCTGTTGTACCTGTTGACAGCCATTCCTGGTATCCTGTCTGATGTACAGCTTCAGGAGTCAGGACCTGGCCTCGTGAAACCTTCTCAGTCTCTGTCTCTCACCTGCTCTGTCACTGGCTACTCCATCACCAGTGGTTATTACTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATAAGCTACGACGGTAGCAATAACTACAACCCATCTCTCAAAAATCGAATCTCCATCACTCGTGACACATCTAAGAACCAGTTTTTCCTGAAGTTGAATTCTGTGACTACTGAGGACACAGCTACATATTACTGTGCAAGAGAGGGCCTCTTACGTGAGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GGTATTGAGAGTACAT-1_contig_3": "GGGGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGG",
        "TATGCCCTCAAGAAGT-1_contig_1": "TGGGGCGTTGTCATTGCAGTCAGGACTCAGCATGGACATGAGGGCCCCTGCTCAGTTTTTTGGGATCTTGTTGCTCTGGTTTCCAGGTATCAGATGTGACATCAAGATGACCCAGTCTCCATCCTCCATGTATGCATCGCTGGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAAAAGCTATTTAAGCTGGTACCAGCAGAAACCATGGAAATCTCCTAAGACCCTGATCTATTATGCAACAAGCTTGGCAGATGGGGTCCCATCAAGATTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTAACCATCAGCAGCCTGGAGTCTGACGATACAGCAACTTATTACTGTCTACAGCATGGTGAGAGCCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "CGAGCACCAAATACAG-1_contig_2": "TGGGGGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GTCCTCAGTCATCCCT-1_contig_7": "GGGAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCGTACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "ACGAGCCAGCCGATTT-1_contig_1": "TCACCGCTTTCTTATATGGGGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "TTGCCGTTCCACTCCA-1_contig_3": "ATGCAAATAGGGCCTCTTTCTCCTCATGAAACGCAGACCAACCTATCCTTGCAGTTCAGACATAGGAGCTTGGCTCTGGTTCCCAAGACCTCTCACTCACTTCTCAACATGGAGTGGGAACTGAGCTTAATTTTCATTTTTGCTCTTTTAAAAGATGTCCAGTGTGAAGTGCAGCTGTTGGAGACTGGAGGAGGCTTGGTGCAACCTGGGGGGTCACGGGGACTCTCTTGTGAAGGCTCAGGGTTTACTTTTAGTGGCTTCTGGATGAGCTGGGTTCGACAGACACCTGGGAAGACCCTGGAGTGGATTGGAGACATTAATTCTGATGGCAGTGCAATAAACTACGCACCATCCATAAAGGATCGATTCACTATCTTCAGAGACAATGACAAGAGCACCCTGTACCTGCAGATGAGCAATGTGCGATCTGAGGACACAGCCACGTATTTCTGTATGAGATATGATGGTTACTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CGCTTCACATCCCATC-1_contig_3": "GGGGAGCTCTGACAGAGGAGGCCTGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACGGACCCCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCAGTGTGAAGTGATGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGCCATGTCTTGGGTTCGCCAGACTCCGGAGAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGGCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGACAGGATGACGCAGGAGCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTG",
        "GCATGATAGCTAACAA-1_contig_5": "GAGGCAGAGAACTTTAGCCCTGTCTTCCTTTTTGTGTTCAGCACTGACATTATAACATTGAACATGCTGTTGGGGCTGAAGTGGGTTTTCTTTGTTGTTTTTTATCAAGGTGTGCATTGTGAGGTGCAGCTTGTTGAGACTGGTGGAGGATTGGTGCAGCCTAAAGGGTCATTGAAACTCTCATGTGCAGCCTCTGGATTCACCTTCAATACCAATGCCATGAACTGGGTCCGCCAGGCTCCAGGAAAGGGTTTGGAATGGGTTGCTCGCATAAGAAGTAAAAGTAATAATTATGCAACATATTATGCCGATTCAGTGAAAGACAGGTTCACCATCTCCAGAGATGATTCACAAAGCATGCTCTATCTGCAAATGAACAACTTGAAAACTGAGGACACAGCCATGTATTACTGTGTGAGAGACTATGGTAACTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GCCAAATTCCTGTAGA-1_contig_4": "GTAGGAAGATGAGATGTGGATAGATACTGATGACTGTGTATGTGTTCTCCAACTGCTCTCTCCTGAAGTGCCTCAGATGTTTCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GACTGCGTCCTATTCA-1_contig_2": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGC",
        "GGCTCGATCCCATTAT-1_contig_3": "GGGACATACCAGCAAGCGAGTGACCAGTTAGTCTTAAGGCACCACTTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCAAGAGAGTTTCTGGGACCGGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CACAGTAAGGCGATAC-1_contig_3": "GATAAGACGATCTCTATCAGGACAGTGTCATGAGCCACACAAACTCAGGGAAAGCTCGAAGATGGTTTTCACACCTCAGATACTTGGACTTATGCTTTTTTGGATTTCAGCCTCCAGAGGTGATATTGTGCTAACTCAGTCTCCAGCCACCCTGTCTGTGACTCCAGGAGATAGCGTCAGTCTTTCCTGCAGGGCCAGCCAAAGTATTAGCAACAACCTACACTGGTATCAACAAAAATCACATGAGTCTCCAAGGCTTCTCATCAAGTATGCTTCCCAGTCCATCTCTGGGATCCCCTCCAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACTCTCAGTATCAACAGTGTGGAGACTGAAGATTTTGGAATGTATTTCTGTCAACAGAGTAACAGCTGGCCTCACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCTCCAGGTCACTGTGCTGGCCTC",
        "CAACCAAGTTGGGACA-1_contig_3": "GAAAACAACCTATGATCAGTGTCCTCTCTACACAGTCCCTGACAACACTGACTCTAACCATGGGATGGAGCCGGATCTTTCTCTTCCTCCTGTCAATAATTGCAGGTGTCCATTGCCAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAGGATATCCTGCAAGGCTTCTGGCTACACCTTCACAAGCTACTATATACACTGGGTGAAGCAGAGGCCTGGACAGGGACTTGAGTGGATTGGATGGATTTATCCTGGAAATGTTAATACTAAGTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACCTCTGAGGACTCTGCGGTCTATTTCTGTGCAAGAGGCCCTTTTACTACGGTAGTAGCTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AAGGTTCTCTTGAGGT-1_contig_4": "GAGCTCTGACAGAGGAGGCCAGTCCTGGATTTGAGTTCCTCACATTCAGTGATGAGCACTGAACACGGACCCCTCACCATGAATTTCGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCTGTGTGAAGTGAAGCTGGTGGAGTCTGGGGGAGGTTTAGTGCAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATACCATGTCTTGGGTTCGCCAGACTCCAGAGAAGAGGCTGGAGTGGGTCGCATACATTAGTAATGGTGGTGGTAGCACCTACTATCCAGACACTGTAAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACGGCCATGTATTACTGTGCAAGACTGGGGGATCTAGGGGACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTC",
        "CGAACATCATTGCGGC-1_contig_5": "AGATGTTTCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTG",
        "CAAGAAACACGAAACG-1_contig_3": "TCTATTGATCCTTTCTCTTAGAGTTTTATAAGAAGGTCTTTCTCAGCTCTGTGTCATAATCCACAGAAGCTCAGGGAAAATTTGAAGATGGTGTCCACTTCTCAGCTCCTTGGACTTTTGCTTTTCTGGACTTCAGCCTCCAGATGTGACATTGTGATGACTCAGTCTCCAGCCACCCTGTCTGTGACTCCAGGAGATAGAGTCTCTCTTTCCTGCAGGGCCAGCCAGAGTATTAGCGACTACTTACACTGGTATCAACAAAAATCACATGAGTCTCCAAGGCTTCTCATCAAATATGCTTCCCAATCCATCTCTGGGATCCCCTCCAGGTTCAGTGGCAGTGGATCAGGGTCAGATTTCACTCTCAGTATCAACAGTGTGGAACCTGAAGATGTTGGAGTGTATTACTGTCAAAATGGTCACAGCTTTTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GACTACAAGTTTCCTT-1_contig_2": "GGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCA",
        "CCGTTCAGTTATGTGC-1_contig_3": "GAAACAACCTATGATCAGTGTCCTCTCCACAGTCCCTGAACACACTGACTCTCACCATGGAATGGAGCGGGGTCTTTATCTTTCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTCCAGCTGCAGCAGTCTGGAGCTGAGCTGGTAAGGCCTGGGACTTCAGTGAAGATGTCCTGCAAGGCTGCTGGATACACCTTCACTAACTACTGGATAGGTTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGATATTTACCCTGGAGGTGGTTATACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACACATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCCATCTATTACTGTGCAAGGGGTCCGGCCATTACTACGGCTACGGGAGATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GCTGCAGAGCACCGCT-1_contig_3": "GCTGCTGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TGGCTGGAGCACCGCT-1_contig_2": "TGGGGGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TTCGGTCCAAAGGAAG-1_contig_2": "GTGATCATACACACTCCAACAGTCATTCTTGGTCAGGAGACGTTGTAGAAATGAGACCGTCTATTCAGTTCCTGGGGCTCTTGTTGTTCTGGCTTCATGGTGCTCAGTGTGACATCCAGATGACACAGTCTCCATCCTCACTGTCTGCATCTCTGGGAGGCAAAGTCACCATCACTTGCAAGGCAAGCCAAGACATTAACAAGTATATAGCTTGGTACCAACACAAGCCTGGAAAAGGTCCTAGGCTGCTCATACATTACACATCTACATTACAGCCAGGCATCCCATCAAGGTTCAGTGGAAGTGGGTCTGGGAGAGATTATTCCTTCAGCATCAGCAACCTGGAGCCTGAAGATATTGCAACTTATTATTGTCTACAGTATGATAATCTGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "AACTCAGTCCAGAGGA-1_contig_1": "CTCTTAGAGTTTTATAAGAAGGTCTTTCTCAGCTCTGTGTCATAATCCACAGAAGCTCAGGGAAAATTTGAAGATGGTGTCCACTTCTCAGCTCCTTGGACTTTTGCTTTTCTGGACTTCAGCCTCCAGATGTGACATTGTGATGACTCAGTCTCCAGCCACCCTGTCTGTGACTCCAGGAGATAGAGTCTCTCTTTCCTGCAGGGCCAGCCAGAGTATTAGCGACTACTTACACTGGTATCAACAAAAATCACATGAGTCTCCAAGGCTTCTCATCAAATATGCTTCCCAATCCATCTCTGGGATCCCCTCCAGGTTCAGTGGCAGTGGATCAGGGTCAGATTTCACTCTCAGTATCAACAGTGTGGAACCTGAAGATGTTGGAGTGTATTACTGTCAAAATGGTCACAGCTTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCC",
        "TCGCGTTCAGATCTGT-1_contig_1": "GGGGATCAGACAGGCAGGGGAAGCAAGATGGATTCACAGGCCCAGGTTCTTATGTTACTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTAGCTGTGTCAGTTGGAGAGAAGGTTACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GACTAACCATGCCTTC-1_contig_1": "TGGGGGAAATGCATCAGACCAGCATGGGCATCAAGATGGAATCACAGACTCTGGTCTTCATATCCATACTGCTCTGGTTATATGGTGCTGATGGGAACATTGTAATGACCCAATCTCCCAAATCCATGTCCATGTCAGTAGGAGAGAGGGTCACCTTGAGCTGCAAGGCCAGTGAGAATGTGGGTACTTATGTATCCTGGTATCAACAGAAACCAGAGCAGTCTCCTAAACTGCTGATATACGGGGCATCCAACCGGTACACTGGGGTCCCCGATCGCTTCACAGGCAGTGGATCTGCAACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGACCTTGCAGATTATCACTGTGGACAGAGTTACAGCTATCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GAGCAGATCAAAGTAG-1_contig_1": "TGGGGGAAATACATCAGATCAGCATGGGCATCAAGATGGAGTCACAGACTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGATGGAGACATTGTGATGACCCAGTCTCAAAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCGTCACCTGCAAGGCCAGTCAGAATGTGGGTACTAATGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAAGCACTGATTTACTCGGCATCCTACCGGTACAGTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAATGTGCAGTCTGAAGACTTGGCAGAGTATTTCTGTCAGCAATATAACAGCTATCCTCTATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCGC",
        "CGATGGCCAAGCCGTC-1_contig_3": "AGGTCACAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACCTAGCCACCTCGCA",
        "CATTATCGTGCCTGCA-1_contig_6": "GAGGTAGTACCTTCTGTGAACCAGTTTCAATCCTGTTCTTAGCCTGTGCAGAGAGATGGGTGTGGGGTAGCATTCAGACCTTTGTGACAATTGGGTTGCAACTCCTAAGAGGAAAGATCCCAAAGCTAAGCTCCTGGAAGCTATGGACCGTCACAAAAAAGAAGGCAGTCACTTTCATATTTCCCATGCATGAGTTGTGTCAGCCGGTAAAGGAGAGTACAGTGGAAGAAGGCCTTGAGACCTGAGTCAGCAGTGGATATCAGCAAAACAGAACAACCAACAACCCATGGAAGAGTGGGAAGAAGTCGAAATACCAGACTGTGGAGGGGGTCTCTCTGATGCACAGGTAAGGTGATGGCCTGAAGAATGTCATCCAGACAGGGTCCAAGGATGCCACAGGCTGGAACATGGGCAAGTGGATCTGAACACACACAACAAAAACAACAACTACTGCTACCACCACCACCAGAGTCAGCCTCAAGGAGATGATGGGATATATCAGGATACCTAAGATAAGATGAACTTGGTCTGTCTGGAAGCTGGCAGGACCAAATTCGCTGAGTCATCAGAGAAGAGGTGGCCAGAGGAGCAAGATCTCTGCAGCAGAAATCTGCAGGACTAACAAAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCCTTGTCAAAGGCTACTTCCCTGAGCCGGTAACTGTAAAATGGAACTATGGAGCCCTGTCCAGC",
        "CTCGAAAGTTCGGGCT-1_contig_5": "TGGGGAAGCTGGCTATCACTCAAGGCCATGACCATGCTCTCACTAGCTCCTCTCCTCAGCCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCGTGGCTACAGGAGAAAAAGTCACTATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGCACAGATTTTGTTTTTACAATTGAAAACACGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACATGCCTCTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GAACCTAGTGGTCTCG-1_contig_1": "GGGGACTTATGAGAATAGTAGTAATTAGCTAGGGACCAAAGTTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATACTGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAGACTCCTGATTTATGACACATCCAACCTGGCTTCTGGAGTCCCTGTTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTTACCCACTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGCTGCTGCTCGGGCTGGTGCCATCGCCCCGTGTGAGGTCACTGTGCTGGCCTC",
        "GATGCTACACTAAGTC-1_contig_9": "GAGGAGAAATACTTTTTTTTTAAATGAATGCAATTATCTAGACTTATTTCGGTTGAACATGCTGGTTGGTGGTTGAGAGGACACTCAGTCAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GTCACAAAGCTAGTGG-1_contig_3": "GGACATAACAGCAAGAGAGTGTCCGGTTAGTCTCAAGGAAGACTGAGACACAGTCTTAGATATCATGGAATGGCTGTGGAACTTGCTATTTCTCATGGCAGCAGCTCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAGGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAACTGCTGGAATGCAGTGGGTGCAAAAGATGCCAGGAAAGGGTTTGAAGTGGATTGGCTGGATAAACACCCACTCTGGAGTGCCAAAATATGCAGAAGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCATATTTACAGATAAGCAACCTCAAAAATGAGGACACGGCTACGTATTTCTGTGCGAGATCAGATGATGGTTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "TCTGGAAGTAGCTAAA-1_contig_8": "GCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "CCAGCGAAGTCCGGTC-1_contig_2": "GAACATGCTGGTTGGTGGTTGAGAGGACACTCAGTCAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CAGCCGATCTACTATC-1_contig_9": "GAACAGACACACAAACCTGGACTCACAAGTTTTTCTCTTCAGTGACAGACACAGACATAGAACATTCACGATGTACTTGGGACTGAACTATGTATTCATAGTTTTTCTCCTAAATGGTGTCCAGAGTGAAGTGAAGCTTGAGGAGTCTGGAGGAGGCTTGGTGCAACCTGGAGGATCCATGAAACTCTCTTGTGCAGCCTCTGGATTCACTTTTAGTGACGCCTGGATGGACTGGGTCCGCCAGTCTCCAGAGAAGGGGCTTGAGTGGGTTGCTGAAATTAGAAGTAAAGCTAATAATCATGCAACATACTATGCTGAGTCTGTGAAAGGGAGGTTCACCATCTCAAGAGATGATTCCAAAAGTAGTGTCTACCTGCAAATGAACAGGTTAAGAGCTGAAGACACTGGCATTTATTACTGTAGGGGTTCCGCTATGGGCTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AGAGCTTCAATCTACG-1_contig_1": "GGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "ACGCCAGTCGTGGTCG-1_contig_3": "AGTTCAAAATCTGAAGAAGACAAATGACCTAGGATGTCACTGATATACACTCTTGTTTCTCTTCGCAGGCATCCACTCTAAGGTCAAGCTGCAGCAGTCTGGACCTGAGCTGGTGAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTCATTACTCATTCACTGGCTACTACATGTACTGGGTGAAGCAGAGCCATGGAAATAGCCTTGAGTAGATTGGACTTATTATTCCTTACAATGGTGATACTGGCTACAACCAGAAGTTTAAGGGCAAGGCCACATTGACTGTAGACTAGTCCTTCAGCACAGCCTACATGGAGCTCCACAGCCTGAAATCTTAGGACTCTGTGGTCTATTACTGTACAAGACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGC",
        "CGGACTGCACATCCAA-1_contig_5": "GATATTTGCATACTTCATCTTCAGTAGTCACAAATATCTCACAGCTGTTTTAAAGCGATGTACTTAGGAGAAGAGCAGTAATTAGCCAGGGAACAAAATTCAAATACACAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTATCTCAGTTGTAATGTCCAGAGGAGAAAATGTGCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCTAGGGGAGAAGGTCACCATGAGCTGCAGGGCCAGCTCAAGTGTAAATTACATGTACTGGTACCAGCAGAAGTCAGATGCCTCCCCCAAACTATGGATTTATTACACATCCAACCTGGCTCCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGAACTCTTATTCTCTCACAATCAGCAGCATGGAGGGTGAAGATGCTGCCACTTATTACTGCCAGCAGTTTACTAGTTCCCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GCGGGTTTCGCACTCT-1_contig_1": "GCTCTTAGAGTTTTATAAGAAGGTCTTTCTCAGCTCTGTGTCATAATCCACAGAAGCTCAGGGAAAATTTGAAGATGGTGTCCACTTCTCAGCTCCTTGGACTTTTGCTTTTCTGGACTTCAGCCTCCAGATGTGACATTGTGATGACTCAGTCTCCAGCCACCCTGTCTGTGACTCCAGGAGATAGAGTCTCTCTTTCCTGCAGGGCCAGCCAGAGTATTAGCGACTACTTACACTGGTATCAACAAAAATCACATGAGTCTCCAAGGCTTCTCATCAAATATGCTTCCCAATCCATCTCTGGGATCCCCTCCAGGTTCAGTGGCAGTGGATCAGGGTCAGATTTCACTCTCAGTATCAACAGTGTGGAACCTGAAGATGTTGGAGTGTATTACTGTCAAAATGGTCACAGCTTTCCTCCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCA",
        "AGCCTAACACTGTCGG-1_contig_3": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CCCAATCGTGTGTGCC-1_contig_9": "GACAGACCTTTCTCCATAGTTGATAACTCAGACACAAGTGAATGACAGATGGACCTCCATCTACTCTTATTTTAAAAAGAAGACAAACCCCACAGGCTCGAGAACTTTAGCGACTGTTTTGAGAGAAATCATTGGTCCCTGACTCAAGAGATGACTGGCAGATTGGGGATCAGAATACCCATACTCTGTGGCTAGTGTGAGGTTTAAGCCTCAGAGTCCCTGTGGTCTCTGACTGGTGCAAGGTTTTGACTAAGCGGAGCACCACAGTGCTAACTGGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CTTTGCGTCTTGCAAG-1_contig_1": "CTTTGCGTCTTGCAAGGCCCCCGGGTTATATGGGGAGAAATACATCAGGCAGGCAAGGGCATCAAGATGAAGTCACAGACCCAGGTCTTCGTATTTCTACTGCTCTGTGTGTCTGGTGCTCATGGGAGTATTGTGATGACCCAGACTCCCAAATTCCTGCTTGTATCAGCAGGAGACAGGGTTACCATAACCTGCAAGGCCAGTCAGAGTGTGAGTAATGATGTAGCTTGGTACCAACAGAAGCCAGGGCAGTCTCCTAAACTGCTGATATACTATGCATCCAATCGCTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATATGGGACGGATTTCACTTTCACCATCAGCACTGTGCAGGCTGAAGACCTGGCAGTTTATTTCTGTCAGCAGGATTATAGCTCTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "CATATGGCAGCTTAAC-1_contig_3": "GATCATCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGCATTACTCTTCTGCCTGGTAACATTCCCAAGCTGTATCCTTTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCGTCTCAGGGTTCTCATTAACCGGCTATGGTGTAAACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAATGATATGGGGTGATGGAAGCACAGACTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCAGGTACTACTGTGCCAGAGATCCCGGTAGTAGTGGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACCA",
        "TGAGAGGAGGTGCTTT-1_contig_6": "GGGGATTGTAATTGAAGTCAAGACTCAGCATGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTTTGTTTTCAAGGTACCAGATATGATATCCAGATGACACAGACTACTTCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTGAGGACATTAGCACTTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTATTACACATCAGGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGGGCAGATTATTCTCTCACCATCAGCAACCTGGAACCTGAAGATATTGCCACTTACTATTGTCAGCAGTATAGTAAGCTTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCACA",
        "ACAGCCGGTAGCGTAG-1_contig_3": "GAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGCCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CTGATCCGTACTTCTT-1_contig_1": "GAGTGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TCTCATAGTCTCTTAT-1_contig_1": "GGGGAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATGAGTGCCTCAGTCATAATGTCCAGGGGACAAATTGTTCTCACCCAGTCTCCAGCACTCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACTGGTACCAGCAGAAGCCAAGATCCTCCCCCAAACCCTGGATTTATCTCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CCATTCGGTACCTACA-1_contig_7": "GGGGAATCTTGTCCAGGGTCTATCGGACTCTTGTGAGAATTAGGGGCTGACAGTTGATGGTGACAATTTCAGGGTCAGTGACTGTCTGGTTTCTCTGAGGTGAGGCTGGAATATAGGTCACCTTGAAGACTTAAGAGGGGTCCAGGGGGCTTCTGCACAGGCAGGGAACAGAATGTGGAACAATGACTTGAATGGTTGATTCTTGTGTGACACCAGGAATTGGCATAATGTCTGAGTTGCCCAGGGGTGATTCTAGTCAGACTCTGGGGTTTTTGTCGGGTATAGAGGAAAAATCCACTATTGTGATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "AATCCAGAGATGCGAC-1_contig_12": "ATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGAGCCCTATACGATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "CCTACCATCTTTAGTC-1_contig_2": "GGGCTGCTGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCACTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGAC",
        "AACTTTCCAGCGAACA-1_contig_10": "GGGGATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TTCGAAGAGCTAACAA-1_contig_2": "GGTACTTCACATATACTTACTCTGCCATGTTCATTGCTTTTGTGGTGGTGACCTGAAGGAGAGTTGGACAGAAGAAGGACATGGACCACAGCCTATCTGGATCCTAATGTGGCTCACCATACCCTGGACAGCTAATACCAGCAATGAAGCAAAAAAAGTTGAGCAGATGACAGAGAAGATAGCAAGACCTTCTTCTGTTGGCTTAGAGCCTGACTCCAAATAATTCCTGTGCTGAAAAATATCAGGCACTGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "ACACCAAGTTCGGGCT-1_contig_3": "GGTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTC",
        "AACACGTGTTCGGGCT-1_contig_3": "GGAGCTCTGACAGAGGAGCCAAGCCCTGGATTCCCAGGTCCTCACATTCAGTGATCAGCACTGAACACAGACCACTCACCATGGACTCCAGGCTCAATTTAGTTTTCCTTGTCCTTATTTTAAAAGGTGTCCAGTGTGATGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGCAGCCTGGAGGGTCCCGGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTTTGGAATGCACTGGGTTCGTCAGGCTCCAGAGAAGGGGCTGGAGTGGGTCGCATACATTAGTAGTGGCAGTAGTACCATCTACTATGCAGACACAGTGAAGGGCCGATTCACCATCTCCAGAGACAATCCCAAGAACACCCTGTTCCTGCAAATGACCAGTCTAAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGAAAGGGGGGTAACTACGTGGACTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CGTCAGGTCTGCAAGT-1_contig_2": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGATCTCCAAGACTGACAGGCAGGGAGAGTGGAGTTTGGGTCATCAAAACATCACT",
        "GAAGCAGCACCAGTTA-1_contig_1": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GCTTCCAGTTATCACG-1_contig_4": "GTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGAGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTAC",
        "CCTCTGAAGAGGTAGA-1_contig_4": "GACAGTTACTGAGCACACAGGACCTCACCATGGGATGGAGCTGTATCATGCTCTTCTTGGCAGCAACAGCTACAGGTGTCCACTCCCCGGTCCAACTGCAGCAGCCTGGGGCTGAGCTTGTGAAGCCTGGGGCTTCAGTAAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTGGATGCACTGGGTGAAGCAGAGGCCTGGACGAGGCCTTGAGTGGATTGGAAGGATTGATCCTAATAGTGGTGGTACTAAGTACAATGAGAAGTTCAAGAGCAAGGCCACACTGACTGTAGACAAACCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTACAAGATCGGCTGGGACGAATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AGCATACGTCCAGTGC-1_contig_1": "TGGGCTGAAATGCATCAGACCAGCATGGGCATCAAGATGGAATCACAGACTCTGGTCTTCATATCCATACTGCTCTGGTTATATGGTGCTGATGGGAACATTGTAATGACCCAATCTCCCAAATCCATGTCCATGTCAGTAGGAGAGAGGGTCACCTTGAGCTGCAAGGCCAGTGAGAATGTGGGTACTTATGTATCCTGGTATCAACAGAAACCAGAGCAGTCTCCTAAACTGCTGATATACGGGGCATCCAACCGGTACACTGGGGTCCCCGATCGCTTCACAGGCAGTGGATCTGCAACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGACCTTGCAGATTATCACTGTGGACAGAGTTACAGCTATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GCGGGTTTCGCACTCT-1_contig_2": "GACAAATATCTCACAGTTCTTTTAAAGCAAAGTACTTGTGAGAATAGGAGTAATTAGCCAGGAACTAAAATTCAAAGACACAATGGATTTACAGGTGCAGATTATCAGCTTCCTGCTAATCAGTGTCACAGTCATAATGTCCAGAGGAGAAAATGTGCTCACCCAGTCTCCAGCAATAATGGCTGCCTCTCTGGGGCAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTCCAGTTACTTGCACTGGTACCAGCAGAAGTCAGGCGCTTCCCCCAAACCCTTGATTCATAGGACATCCAACCTGGCTTCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCGTGGAGGCTGAAGATGATGCAACTTATTACTGCCAGCAGTGGAGTGGTTACCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCA",
        "CATCGGGCAAATACAG-1_contig_4": "GGAGTATACAGAACTTCATTATGGGATGGAGCCAGATCACCCTCTTTCTGGTGGCAGCAATTACATGTGCCTCCTCCCAGATTCAGCTGCAGCAGTCAGGAGCTGAGCTGGCGAGTCCTGGGGCATCAGTGACACTGTCCTGCAAGGCTTCTGGCTACACATTTACTGACCATATTATGAATTGGGTAAAAAAGAGGCCTGGACAGGGCCTTGAGTGGATTGGAAGGATTTATCCAGTAAGTGGTGAAACTAACTACAATCAAAAGTTCATGGGCAAGGCCACATTCTCTGTAGACCGGTCCTCCAGCACAGTGTACATGGTGTTGAACAGCCTGACATCTGAGGACCCTGCTGTCTATTACTGTGGAACATAGAGGGTATCCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CAAGAAACACGAAACG-1_contig_1": "GGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACTAACAAGGCCACACTGGTGTGTA",
        "GTGCTTCAGGGCTTGA-1_contig_3": "AAAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACAGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATATCCTGCAGAGCCAGTGAAAGTGTTGATAGTTATGGCAATAGTTTTATGCACTGGTACCAGCAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCGTGCATCCAACCTAGAATCTGGGATCCCTGCCAGGTTCAGTGGCAGTGGGTCTAGGACAGACTTCACCCTCACCATTAATCCTGTGGAGGCTGATGATGTTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TTTGGTTCATTTCACT-1_contig_4": "GAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "ATTCTACAGTGCGATG-1_contig_1": "GAGCATATGATCAGTGTCCTCTCCAAAGTCCTTGAACATAGACTCTAACCATGGAATGGACCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGAGCTGAGCTGATGAAGCCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTACTGGCTACACATTCAGTAGCTACTGGATAGAGTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGAGATTTTACCTGGAAGTGGTAGTACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACATTCACTGCAGATACATCCTCCAACACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCCGTCTATTACTGTGCAAGATGGGTCTACTATGGTAACCACGAGGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "CGAACATAGTCATGCT-1_contig_2": "TGGGGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTCCCCACGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCC",
        "GGAATAACAAGGTGTG-1_contig_3": "AAACAGCATATGATCAGTGTCCTCTCCAAAGTCCTTGAACATAGACTCTAACCATGGAATGGACCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGAGCTGAGCTGATGAAGCCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTACTGGCTACACATTCAGTAGCTACTGGATAGAGTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGAGATTTTACCTGGAAGTGGTAGTACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACATTCACTGCAGATACATCCTCCAACACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCCGTCTATTACTGTGCAAGAAACAAGTATGGTAACTCTTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TATGCCCAGTCAAGGC-1_contig_6": "GGGGGAAGAGGAAACAGAATCTGAGGTGACAGTGATACCTACTATGCCTTTCTGTCCACAGGTGTCCACTCCCAGGTCCAGCTTCAGCAGTCTGGGGCTGAACTGGCAAAACCTGGGGCCTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACCTTTACTAGCTACTGGATGCACTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATTGGATACATTAATCCTAGCACTGGTTATACTGAGTACAATCAGAAGTTCAAGGACAAGGCCACATTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAACTGAGCAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGATGGGGGACTCTACTATGGTTACGACGGTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TGCCAAAGTGTAACGG-1_contig_5": "GGGAACAACATATGTCCAATGTCTTCTCCACAGACACTGAACACACTGACTCTAACCATGGGATGGAGCTGGATCTTTCTCTTTCTCCTGTCAGGAACTGCAGGTGTCCTCTCTGAGGTCCAGCTGCAACAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGCTACTACATGCACTGGGTGAAGCAAAGCCATGTAAAGAGCCTTGAGTGGATTGGACGTATTAATCCTTACAATGGTGCTACTAGCTACAACCAGAATTTCAAGGACAAGGCCAGCTTGACTGTAGATAAGTCCTCCAGCACAGCCTACATGGAGCTCCACAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGAAAGTATGGTAACTACGCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGTCCATAGGTTCCTA-1_contig_2": "GGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCTCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GTCCTCATCAGCCTAA-1_contig_1": "GGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "CGTGTCTAGCTGATAA-1_contig_7": "GCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TCGGGACTCCGTACAA-1_contig_2": "GGGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCTTGGGGCTGCTCTTCTGCCTGGTGACATTCCCAAGCTGTGTCCTATCCCAGGTGCAGCTGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATCACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGTGGTGGAAGCACAGACTATAATGCAGCTTTCATATCCAGACTGAGCATCAGCAAGGACAATTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTAATGACACAGCCATATATTACTGTGCCAGAAATGAGGGGGATGATTACGACGCCGGGGCTTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TGTTCCGCACCTCGGA-1_contig_3": "TGGGGATTGTCATTGCAGCCAGGACTCAGCATGGACATGAGGACCCCTGCTCAGTTTCTTGGAATCTTGTTGCTCTGGTTTCCAGGTATCAAATGTGACATCAAGATGACCCAGTCTCCATCTTCCATGTATGCATCTCTAGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAATAGCTATTTAAGCTGGTTCCAGCAGAAACCAGGGAAATCTCCTAAGACCCTGATCTATCGTGCAAACAGATTGGTAGATGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTCACCATCAGCAGCCTGGAGTATGAAGATATGGGAATTTATTATTGTCTACAGTATGATGAGTTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GAACGGAGTCGGGTCT-1_contig_3": "GGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTTATTTCCAGCTTG",
        "ACACCCTAGGAATCGC-1_contig_4": "AAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGCATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TGGTTAGAGAGTACAT-1_contig_3": "CTTATATGGGGGTGACTGAACAGCATCAGACAGGCAGGGGAGCAATATGGTATCAGAGACCCATGTCCTCATATTTTTGCTGCTATGGGTGTCTGGTGCTTGTGGGGAAATTGTGTTGACCCAGTCTATACCATCCCTGACTGTGTCAGCAGGAGAGAGGGTCACTATCAGCTGCAAATCCAATCAGAATCTTTTATGGAGTGGAAACCAAAGGTACTGTTTGGTCTGGCACCAGTGGAAACCAGGGCAAACTCCTACACCGTTGATCACCTGGACATCTGATAGGTACTCTGGAGTCCCTGATCGTTTCATAGGCAGTGGATCTGTGACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGATGTGGCAGTTTATTTCTGTCAGCAGCATTTACACATTCCTCCCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCC",
        "CCTCTGAAGAGGTAGA-1_contig_2": "TGGGACTACTCAAGACTTTTTGTATCAAGTTCTCAGAATGAGGTGCCTAGCTGAGTTCCTGGGGCTGCTTGTGCTCTGGATCCCTGGAGCCATTGGGGATATTGTGATGACTCAGGCTGCACCCTCTGTACCTGTCACTCCTGGAGAGTCAGTATCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTGCATAGTAATGGCAACACTTACTTGTATTGGTTCCTGCAGAGGCCAGGCCAGTCTCCTCAGCTCCTGATATATCGGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTGGCAGTGGGTCAGGAACTGCTTTCACACTGAGAATCAGTAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTATGCAACATCTAGAATATCCTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCATCCAGTGAGCAG",
        "CGTAGGCTCGTCCAGG-1_contig_5": "ACCTATGATCAGTATCCTCTCAACAGTCCCTGAACACACTGACTCTCACCATGGAATGGAGCGGAGTCTTTATCTTTCTCCTGTCAGTAACTGCAGGTGTTCACTCCCAGGTCCAGCTGCAGCAGTCTGGAGCTGAGCTGGTAAGGCCTGGGACTTCAGTGAAGGTGTCCTGCAAGGCTTCTGGATACGCCTTCACTAATTACTTGATAGAGTGGGTAAAGCAGAGGCCTGGACAGGGCCTTGAGTGGATTGGAGTGATTAATCCTGGAAGTGGTGGTACTAACTACAATGAGAAGTTCAAGGGCAAGGCAACACTGACTGCAGACAAATCCTCCAGCACTGCCTACATGCAGCTCAGCAGCCTGACATCTGATGACTCTGCGGTCTATTTCTGTGCAATTGATTACGACTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GCATGCGGTATATGAG-1_contig_1": "GGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CTTAACTAGGGCACTA-1_contig_2": "GGAAAACTGAGCAGTGTCAAACAGCAGGGGGAGCAGGATGGAGTTTCAGACCCAGGTACTCATGTCCCTGCTGCTCTGCATGTCTGGTGCCTGTGCAGACATTGTGATGACTCAGTCTCCAACTTTCCTTGCTGTGACAGCAAGTAAGAAGGTCACCATTAGTTGCACGGCCAGTGAGAGCCTTTATTCAAGCAAACACAAGGTGCACTACTTGGCTTGGTACCAGAAGAAACCAGAGCAATCTCCTAAACTGCTGATATACGGGGCATCCAACCGATACATTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTGACCATCAGCAGTGTACAGGTTGAAGACCTCACACATTATTACTGTGCACAGTTTTACAGCTATCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCT",
        "CATGACACACGAAGCA-1_contig_7": "GGCTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAG",
        "AAGGAGCGTCATATGC-1_contig_2": "TATGCTTAGGTCGTTTTTCTTATATGGGGGAAATACATCAGACCAGCATGGGCATCAAGATGGAGACACATTCTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGAAGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGGGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAACTACTGATTTACTGGGCATCCACCCGGCACACTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATTAGCAATGTGCAGTCTGAAGACTTGGCAGATTATTTCTGTCAGCAATATAGCAGCTATCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GCTGGGTTCGTGGACC-1_contig_2": "TGGGGAGCTGCCAGGAGCCTAATAAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACAGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATATCCTGCAGAGCCAGTGAAAGTGTTGATAGTTATGGCAATAGTTTTATGCACTGGTACCAGCAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCGTGCATCCAACCTAGAATCTGGGATCCCTGCCAGGTTCAGTGGCAGTGGGTCTAGGACAGACTTCACCCTCACCATTAATCCTGTGGAGGCTGATGATGTTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GTCAAGTTCGGAAACG-1_contig_1": "GGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTAC",
        "TGAAAGAAGATGGCGT-1_contig_1": "TGGGGAAGGGCAGCTGCCAGGAAGCTAAGAAGCATCCTCTCTTCTAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCCAAAGTGTCAGTACATCTAGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCAAGTATGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATACTGCAACATATTACTGTCAGCACAGTTGGGAGATTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "TTATGCTCACTTCGAA-1_contig_4": "CTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCC",
        "GCTCTGTCAGGTTTCA-1_contig_6": "TGGGGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGGGCTGCTTCTCTGCCTGGTGACTTTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCATCTCAGGGTTCTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGTAGTGATATGGAGTGATGGAAGCACAACCTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTCCAAACTGATGACACAGCCATGTACTACTGTGCCAAACATGGAAACAATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CATCAAGCACCTTGTC-1_contig_1": "GAATCAATCAACCTCCTTAAAATGAATTTGCCTGTTCATCTCTTGGTGCTTCTGTTGTTCTGGATTCCTGCTTCCAGAGGTGATGTTGTGGTGACTCAAACTCCACTCTCCCTGCCTGTCAGCTTTGGAGATCAAGTTTCTATCTCTTGCAGGTCTAGTCAGAGTCTTGCAAACAGTTATGGGAACACCTATTTGTCTTGGTACCTGCACAAGCCTGGCCAGTCTCCACAGCTCCTCATCTATGGGATTTCCAACAGATTTTCTGGGGTGCCAGACAGGTTCAGTGGCAGTGGTTCAGGGACAGATTTCACACTCAAGATCAGCACAATAAAGCCTGAGGACTTGGGAATGTATTACTGCTTACAAGGTACACATCAGCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACA",
        "CCGTACTTCACAAACC-1_contig_1": "GGACATGAGGATTTCTTCATAGTCAGGTCACACCCTGTGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "CGAGCACGTTCAGTAC-1_contig_1": "GGGACATATCCTGTACCGTTGTCATTGCAGTCAGGACTCAGCATGGACATGAGGGCCCCTGCTCAGTTTTTTGGGATCTTGTTGCTCTGGTTTCCAGGTATCAGATGTGACATCAAGATGACCCAGTCTCCATCCTCCATGTATGCATCGCTGGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAAAAGCTATTTAAGCTGGTACCAGCAGAAACCATGGAAATCTCCTAAGACCCTGATCTATTATGCAACAAGCTTGGCAGATGGGGTCCCATCAAGATTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTAACCATCAGCAGCCTGGAGTCTGACGATACAGCAACTTATTACTGTCTACAGCATGGTGAGAGCCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GATCGCGTCTATCCCG-1_contig_5": "CAAATACATCTTCTACCTCTAAGTTAAATCTCCTCCTTGTCTATGGCAGTTCCTATCTCTCTCACTGGAGGCTGATTTTTGAAGAAAGGGGTTGTAGCCTAAAAGATGATGGTGTTAAGTCTTCTGTACCTGTTGACAGCCCTTCCGGGTATCCTGTCAGAGGTGCAGCTTCAGGAGTCAGGACCTAGCCTCGTGAAACCTTCTCAGACTCTGTCCCTCACCTGTTCTGTCACTGGCGACTCCATCACCAGTGGTTACTGGAACTGGATCCGGAAATTCCCAGGGAATAAACTTGAGTACATGGGGTACATAAGCTACAGTGGTAGCACTTACTACAATCCATCTCTCAAAAGTCGAATCTCCATCACTCGAGACACATCCAAGAACCAGTACTACCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGATCGGTACTACGGTAGTAGCTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAA",
        "CGAACATAGTCATGCT-1_contig_6": "GGGAAGCTCTGAGAGGATTTTGAACAAGTTACTGTCACAGTGAGACAGCTCGGGCTCCTTTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "ATAAGAGGTACCGTAT-1_contig_4": "TGGGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGGGCTGCTTCTCTGCCTGGTGACGTTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACTGTCTCAGGGTTCTCATTAACCGACTATGGTGTAAGCTGGATTCGCCAGCTTACAGGAAAGGGTCTGGACTGGCTGGGAGTAATATGGGGTGGTGGAAGCACATACTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAAACATGGATACTACGGTAGCCTAGTGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCCTTGTCAAAGGCTACTTCCCTGAGCCGGTAACTGTAAAATGGAACT",
        "TCTGAGAGTTCTGGTA-1_contig_3": "GAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCA",
        "ACGCAGCTCACGGTTA-1_contig_3": "GAAACAACCAATGATCAGTGTCTTCTCCACAGTCCCTGAAGACACTGACTCTAAACATGGAATGGAGCTGCATCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCACTCTGAGGTCCAGCTTCAGCAGTCTGGAGCTGAGCTGGGGAGGCCTGGGTCCTCAGTGAAGCTGTCCTGCAAGACTTCTGGATATACATTCACAAGCTATGGTATAAAGTGGGTGAAACAGAGGCCTGGACAGGGCCTGGAATGGATTGGATATATTTATCCTGGAAATGGTTATACTGTGTACAATGAGAAGTTCCAGGGCAAGGCCACACTGACTTCAGACACATCCTCCAGCACAGCCTACATGCAGCTCAGAAGCCTGACATCTGAGGACTCTGCAATCTATTTCTGTGCGGGGTACGCGGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GCTGCTTGTCAGATAA-1_contig_1": "GGGAAACCAGGTCTTTGCAGTGAGATCTGAAATACATCAGACCAGCATGGGCATCAAGATGGAGACACATTCTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGAAGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGGGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAACTACTGATTTACTGGGCATCCACCCGGCACACTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATTAGCAATGTGCAGTCTGAAGACTTGGCAGATTATTTCTGTCAGCAATATAGCAGCTATCCTCTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTG",
        "CGAGCACGTCATGCCG-1_contig_8": "GAAAAGCAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATGAGTGCCTCAGTCATAATGTCCAGGGGACAAATTGTTCTCACCCAGTCTCCAGCACTCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACTGGTACCAGCAGAAGCCAAGATCCTCCCCCAAACCCTGGATTTATCTCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCACCATCCAGTGAGCAGTTAACATCTGGA",
        "ACGATACCACGAAAGC-1_contig_5": "GAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGCTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCT",
        "CTCATTAGTTAAGTAG-1_contig_1": "GAATAATCTGATCATACACACTCCAACAGTCATTCTTGGTCAGGAGACGTTGTAGAAATGAGACCGTCTATTCAGTTCCTGGGGCTCTTGTTGTTCTGGCTTCATGGTGCTCAGTGTGACATCCAGATGACACAGTCTCCATCCTCACTGTCTGCATCTCTGGGAGGCAAAGTCACCATCACTTGCAAGGCAAGCCAAGACATTAACAAGTATATAGCTTGGTACCAACACAAGCCTGGAAAAGGTCCTAGGCTGCTCATACATTACACATCTACATTACAGCCAGGCATCCCATCAAGGTTCAGTGGAAGTGGGTCTGGGAGAGATTATTCCTTCAGCATCAGCAACCTGGAGCCTGAAGATATTGCAACTTATTATTGTCTACAGTATGATAATCTGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TCGCGTTCAGATCTGT-1_contig_4": "GGGGACATACCAGCAAGGGAGTGACCAGTTAGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGTTATACCTTCACAGACTATTCAATGCACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACTGAGACTGGTGAGCCAACATATGCAGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCTAGAGGGGCCTTATTACTACGGTTCTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "CCACGGAGTTATCCGA-1_contig_5": "CTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TGGACGCCACACATGT-1_contig_3": "CATTTGGTTTCTTATATGGGACAAATCTTTTTCAAGACCACTTTCTGAGTATTCATTTTAGGAGAAATACTTTTTTTTTAAATGAATGCAATTATCTAGACTTATTTCGGTTGAACATGCTGGTTGGTGGTTGAGAGGACACTCAGTCAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCAC",
        "ACGAGCCGTCCAAGTT-1_contig_1": "GAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "AACTGGTAGGACTGGT-1_contig_6": "TGGGCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTG",
        "TGAGGGATCCGTTGTC-1_contig_5": "TGGGGCCAGGGGGGCTTCTGCACAGGCAGGGAACAGAATGTGGAACAATGACTTGAATGGTTGATTCTTGTGTGACACCAGGAATTGGCATAATGTCTGAGTTGCCCAGGGGTGATTCTAGTCAGACTCTGGGGTTTTTGTCGGGTATAGAGGAAAAATCCACTATTGTGATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "TAGTGGTCATCACAAC-1_contig_1": "GGGGTACTTAGGAGAAGAGCAGTAATTAGCCAGGGAACAAAATTCAAATACACAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTATCTCAGTTGTAATGTCCAGAGGAGAAAATGTGCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCTAGGGGAGAAGGTCACCATGAGCTGCAGGGCCAGCTCAAGTGTAAATTACATGTACTGGTACCAGCAGAAGTCAGATGCCTCCCCCAAACTATGGATTTATTACACATCCAACCTGGCTCCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGAACTCTTATTCTCTCACAATCAGCAGCATGGAGGGTGAAGATGCTGCCACTTATTACTGCCAGCAGTTTACTAGTTCCCCATGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCA",
        "AATCCAGAGATGCGAC-1_contig_5": "TGGGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CCAATCCTCAAACAAG-1_contig_1": "TGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "CATCAGACAGGACGTA-1_contig_1": "GATCAGACAGGCAGGGGAAGCAAGATGGATTCACAGGCCCAGGTTCTTATGTTACTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTAGCTGTGTCAGTTGGAGAGAAGGTTACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GCAATCAGTGGCGAAT-1_contig_3": "GTAGAGCTTCCAGGTGTGCTGTCAGGCTGCTAGTGTATGCTCTCTCTAGTTTCTTTTTGGAGGCACTCAGAGCTATGAGTTTTCCTCTTAGGACTGCCTTCATTGTGTCCCATAAGTTTGGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGAC",
        "TCTCATAGTCTCTTAT-1_contig_9": "GACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGGTTAATCTGCCAGG-1_contig_1": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GCAGCCACACTTACGA-1_contig_1": "GGATTTATAAACCAGGTCTTTGCAGTGAGATATGAAATGCATCACACCAGCATGGGCATCAAAATGGAGTCACAGATTCAGGTCTTTGTATTCGTGTTTCTCTGGTTGTCTGGTGTTGACGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGAGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGACAATCTCCTAAACTACTGATTTACTCGGCATCCTACCGGTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATCTGGGACGGATTTCACTTTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAACATTATAGTACTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAG",
        "TCTGAGAGTCACTGGC-1_contig_5": "CTTAAAAACCGCATATGATCAGTATCCTCTTCACAGTCATTGAAAACACTGACTCTAATCATGGAATGTAACTGGATACTTCCTTTTATTCTGTCGGTAATTTCAGGGGTCTACTCAGAGGTTCAGCTCCAGCAGTCTGGGACTGTGCTGGCAAGGCCTGGGGCTTCCGTGAAGATGTCCTGCAAGGCTTCTGGCTACAGCTTTACCAGCTACTGGATGCACTGGGTAAAACAGAGGCCTGGACAGGGTCTAGAATGGATTGGTGCTATTTATCCTGGAAATAGTGATACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCAAACTGACTGCAGTCACATCCGCCAGCACTGCCTACATGGAGCTCAGCAGCCTGACAAATGAGGACTCTGCGGTCTATTACTGTACAAGATGTTACTACGGTAGTAGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TTGGAACGTAAGGATT-1_contig_1": "TGGGGGAAATACATCAGACCAGCATGGGCATCAAGATGGAGACACATTCTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGAAGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGGGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAACTACTGATTTACTGGGCATCCACCCGGCACACTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATTAGCAATGTGCAGTCTGAAGACTTGGCAGATTATTTCTGTCAGCAATATAGCAGCTATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCC",
        "CTAGAGTGTCACTGGC-1_contig_3": "GGGGAAGCAACATGTGTCCAATGTCCTCTCCACAGACACTGAACACACTGACTCCAACTATGGGATGGAGCTGGATCTTTCTCTTCCTCCTGTCAGGAACTGCAGGTGTCCACTCTGAGGTCCAGCTGCAACAGTCTGGACCTGAGCTGGTGAAGCCTGGAGCTTCAATGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGCTACACCATGAACTGGGTGAAGCAGAGCCATGGAAAGAACCTTGAGTGGATTGGACTTATTAATCCTTACAATGGTGGTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTAACTGTAGACAAGTCATCCAGCACAGCCTACATGGAGCTCCTCAGTCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGACCCTACGGTAGTAGCTACCAGGACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCC",
        "ACGAGCCGTCCAAGTT-1_contig_3": "GGACAGACGCACAACCCTGGACTCACAAGTCTTTCTCTTCAGTGACAAACACAGAAATAGAACATTCACCATGTACTTGGGACTGAACTGTGTATTCATAGTTTTTCTCTTAAAAGGTGTCCAGAGTGAAGTGAAGCTTGAGGAGTCTGGAGGAGGCTTGGTGCAACCTGGAGGATCCATGAAACTCTCCTGTGTTGCCTCTGGATTCACTTTCAGTAACTACTGGATGAACTGGGTCCGCCAGTCTCCAGAGAAGGGGCTTGAGTGGGTTGCTGAAATTAGATTGAAATCTAATAATTATGCAACACATTATGCGGAGTCTGTGAAAGGGAGGTTCACCATCTCAAGAGATGATTCCAAAAGTAGTGTCTACCTGCAAATGAACAACTTAAGAGCTGAAGACACTGGCATTTATTACTGTACCAGGCCCCCCCCTGACTACGGTAGTAGCCACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TCGCGTTAGGAATTAC-1_contig_2": "AAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTG",
        "CACAGGCGTTGGTTTG-1_contig_5": "TGGGGGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CAAGATCCATACCATG-1_contig_1": "GGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCTCCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GAAGCAGCACCAGTTA-1_contig_7": "TGGGGACTTACTGTTTTCTCTTTCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGAATTGAAGTCTCAACTATGAAAATCAGCAGTTCTCTGTCAGAGAAGCCCAAGCGCTTCCACGCATGCTTGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "CGAGCACGTCATGCCG-1_contig_13": "TGGGGATCAGACAGGCAGGGGAAGCAAGATGGATTCACAGGCCCAGGTTCTTATGTTACTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTAGCTGTGTCAGTTGGAGAGAAGGTTACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCACCATCCAGTGAGCAGTTAACATCTGGA",
        "GCAGTTAAGTCATGCT-1_contig_2": "TCTGGGGGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACT",
        "CCATTCGTCAACTCTT-1_contig_1": "GACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGGGTACGACTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TTCTTAGGTGATAAAC-1_contig_5": "TGGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTT",
        "CATGGCGGTGTCGCTG-1_contig_2": "GTTACTTTTCTTATATGGGGAGTGGGATCCTGTCCTGAGTTCCCCAATCTTCACATTCAGAAATCAGCACTCAGTCCTGTCACTATGAAGTTGTGGTTAAACTGGGTTTTTCTTTTAACACTTTTACATGGTATCCAGTGTGAGGTGAAGCTGGTGGAATCTGGAGGAGGCTTGGTACAGCCTGGGGGTTCTCTGAGACTCTCCTGTGCAACTTCTGGGTTCACCTTCAGTGATTTCTACATGGAGTGGGTCCGCCAGCCTCCAGGGAAGAGACTGGAGTGGATTGCTGCAAGTAGAAACAAAGCTAATGATTATACAACAGAGTACAGTGCATCTGTGAAGGGTCGGTTCATCGTCTCCAGAGACACTTCCCAAAGCATCCTCTACCTTCAGATGAATGCCCTGAGAGCTGAGGACACTGCCATTTATTACTGTGCAAGAGATGGGTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "AACGTTGGTTATGCGT-1_contig_5": "TGGGCAATTATCTAGACTTATTTCGGTTGAACATGCTGGTTGGTGGTTGAGAGGACACTCAGTCAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AGTGAGGGTTGTCTTT-1_contig_9": "TGGGGATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGAGATTACTACGGTAGTAGCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CTGCGGATCACCGTAA-1_contig_1": "TGGGGTTGTCATTGCAGTCAGGACTCAGCATGGACATGAGGGCCCCTGCTCAGTTTTTTGGGATCTTGTTGCTCTGGTTTCCAGGTATCAGATGTGACATCAAGATGACCCAGTCTCCATCCTCCATGTATGCATCGCTGGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAAAAGCTATTTAAGCTGGTACCAGCAGAAACCATGGAAATCTCCTAAGACCCTGATCTATTATGCAACAAGCTTGGCAGATGGGGTCCCATCAAGATTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTAACCATCAGCAGCCTGGAGTCTGACGATACAGCAACTTATTACTGTCTACAGCATGGTGAGAGCCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TTCCCAGCAGGATCGA-1_contig_3": "AGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "AGCAGCCCATCGGACC-1_contig_7": "GTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCC",
        "GGCTGGTCATCACCCT-1_contig_1": "TGGGGGAAATGCATCAGACCAGCATGGGCATCAAGATGGAATCACAGACTCTGGTCTTCATATCCATACTGCTCTGGTTATATGGTGCTGATGGGAACATTGTAATGACCCAATCTCCCAAATCCATGTCCATGTCAGTAGGAGAGAGGGTCACCTTGAGCTGCAAGGCCAGTGAGAATGTGGGTACTTATGTATCCTGGTATCAACAGAAACCAGAGCAGTCTCCTAAACTGCTGATATACGGGGCATCCAACCGGTACACTGGGGTCCCCGATCGCTTCACAGGCAGTGGATCTGCAACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGACCTTGCAGATTATCACTGTGGACAGAGTTACAGCTATCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CTGCCTAGTGGGTCAA-1_contig_2": "GGGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGG",
        "CAGCTAACAAACGCGA-1_contig_2": "GGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGCCCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GCTGCAGAGCACCGCT-1_contig_2": "GGGATCATCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGCATTACTCTTCTGCCTGGTAACATTCCCAAGCTGTATCCTTTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCGTCTCAGGGTTCTCATTAACCGGCTATGGTGTAAACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAATGATATGGGGTGATGGAAGCACAGACTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCAGGTACTACTGTCATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AACTCAGGTAGAGCTG-1_contig_10": "GGGATCATACACACTCCAACAGTCATTCTTGGTCAGGAGACGTTGTAGAAATGAGACCGTCTATTCAGTTCCTGGGGCTCTTGTTGTTCTGGCTTCATGGTGCTCAGTGTGACATCCAGATGACACAGTCTCCATCCTCACTGTCTGCATCTCTGGGAGGCAAAGTCACCATCACTTGCAAGGCAAGCCAAGACATTAACAAGTATATAGCTTGGTACCAACACAAGCCTGGAAAAGGTCCTAGGCTGCTCATACATTACACATCTACATTACAGCCAGGCATCCCATCAAGGTTCAGTGGAAGTGGGTCTGGGAGAGATTATTCCTTCAGCATCAGCAACCTGGAGCCTGAAGATATTGCAACTTATTATTGTCTACAGTATGATAATCTTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACC",
        "AATCCAGGTCAACATC-1_contig_3": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGAGACAGACTCTTCTCCACAGTGTCCCCTTCATGTG",
        "GCTGCTTGTCAGATAA-1_contig_6": "GACAACCTATGATCACTGTCCTCTCCACAGTCCCTGAAGACACTAACTCTAACCATGGGATGGAGCCGGATCTTTCTCTTTCTCCTGTCAGTAATTGCAGGTGTCCTCTCTGAGGTTCAGCTGCAACAGTCTGGACCTGAGCTGGTGAAGCCTGGGCTTCAGTGAAGATGTCCTGCAAGCCTTCTGGTTACTCATTTACTGACTACTTTATGAACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGACTGGATTGGACGTATTAATCCTTACAATGGTGATACTTTCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAATCCTCTAGCACAGCCCACATGGAGCTCCGGAGCCTGACATCTGAGGACTCTGCAGTCTATTATTGTGCAAGATCACTACGGTAGTAGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CTGTGCTAGCCCAGCT-1_contig_2": "GGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGCCCC",
        "AACTCAGGTAGAGCTG-1_contig_2": "GGCATATGATCAGTATCCTCTTCACAGTCACTGAAAACATTGACTCTAATCATGGAATGTAACTGGATACTTCCTTTTATTCTGTCGGTAACTTCAGGGGTCTACTCAGAGGTTCAGCTCCAGCAGTCTGGGACTGTGCTGGCAAGGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACCTTTACCAGCTACTGGATGCACTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATTGGCGCTATTTATCCTGGAAATAGTGATACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCAAACTGACTGCAGTCACATCCACCAGCACTGCCTACATGGAGCTCAGCAGCCTGACAAATGAGGACTCTGCGGTCTATTACTGTACAAGATCCTTAACTGGGTTCTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ACTATCTAGAGGTTAT-1_contig_4": "AGAACTGAAGCTTGAAGTCTGAGGCAGAATCTTGTCCAGGGTCTATCGGACTCTTGTGAGAATTAGGGGCTGACAGTTGATGGTGACAATTTCAGGGTCAGTGACTGTCTGGTTTCTCTGAGGTGAGGCTGGAATATAGGTCACCTTGAAGACTTAAGAGGGGTCCAGGGGGCTTCTGCACAGGCAGGGAACAGAATGTGGAACAATGACTTGAATGGTTGATTCTTGTGTGACACCAGGAATTGGCATAATGTCTGAGTTGCCCAGGGGTGATTCTAGTCAGACTCTGGGGTTTTTGTCGGGTATAGAGGAAAAATCCACTATTGTGATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CTGCCTAGTGGGTCAA-1_contig_5": "GAGCTCTGACAGAGGAGGCCAGTCCTGGAATTGATTCCCAGTTCCTCACGTTCAGTGATGAGCAGTGAACACAGACCCCTCACCATGAACTTCGGGCTCAGATTGATTTTCCTTGTCCTTACTTTAAAAGGTGTCCAGTGTGACGTGAAGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATACCATGTCTTGGGTTCGCCAGACTCCGGAGAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTACAAGAGTTAAGTATGGTAACGGCCCCCCGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AGCATACGTAGCGTAG-1_contig_4": "TCATATGAGCCCTATCTTCTCTACAGACACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTTATCTTCTTCCTGATGGCAGTGGTTACAGGGGTCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCAGAGCTTGTGAAGCCAGGGGCCTCAGTCAAGTTGTCCTGCACAGCTTCTGGCTTCAACATTAAAGACACCTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGAAGGATTGATCCTGCGAATGGTAATACTAAATATGACCCGAAGTTCCAGGGCAAGGCCACTATAACAGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTGCTAGATGGTTACTACGTTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TAAGCGTAGGATGGTC-1_contig_1": "GGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TTGGCAATCACATGCA-1_contig_2": "GGAGAATCCCAGGAGTAAAAATGCTGGATGTACACTAAGGATGGCCCCTGACACTCTGCACTGCTATCTCTGGCCCCACCAGGCAATGTTCCTGTAGAATCTCTTACCTTACTTGGCAGAGATTTTTGTCAAGGGATCTACTACTGTGTCTACTATGATTACGACTGTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "CTACATTCACTAGTAC-1_contig_7": "GGGCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCAC",
        "CGGACTGTCACATAGC-1_contig_2": "GGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CGTCACTCAAGAGGCT-1_contig_2": "GGGGCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAG",
        "GCCAAATTCCTGTAGA-1_contig_1": "ACAGTTGTTTTAAAGTGATGTACTTATGAGAAGAGAAGTAATTAGCCAGGGAACAAAATTCAAATACACAATGGATTTTCTGGTGCAGATTTTCAGCTTCTTGCTAATCAGTGCCTCAGTTGCAATGTCCAGAGGAGAAAATGTGCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAAAAGGTCACCATGACCTGCAGGGCCAGCTCAAGTGTAAGTTCCAGTTACTTGCACTGGTACCAGCAGAAGTCAGGTGCCTCCCCCAAACTCTGGATTTATAGCACATCCAACTTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGTGTGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTACAGTGGTTACCCACGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TGCTACCGTACATCCA-1_contig_5": "GGGGCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTAT",
        "TACTCATAGTGTGGCA-1_contig_2": "TATGGAGGACAATGGCAGAAAGTCACTCTCAGTGAGGATACACCATCAGCATGAGGGTCCTTGCTGAGCTCCTGGGGCTGCTGCTGTTCTGCTTTTTAGGTGTGAGATGTGACATCCAGATGAACCAGTCTCCATCCAGTCTGTCTGCATCCCTTGGAGACACAATTACCATCACTTGCCATGCCAGTCAGAACATTAATGTTTGGTTAAGCTGGTACCAGCAGAAACCAGGAAATATTCCTAAACTATTGATCTATAAGGCTTCCAACTTGCACACAGGCGTCCCATCAAGGTTTAGTGGCAGTGGATCTGGAACAGGTTTCACATTAACCATCAGCAGCCTGCAGCCTGAAGACATTGCCACTTACTACTGTCAACAGGGTCAAAGTTATCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GTCAAGTTCGACCAGC-1_contig_8": "GCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCAC",
        "GCAGCCACACTTACGA-1_contig_6": "GGACATAGAGGCACCTAACTCTGGACTCACAAGTCTTTCCCTTCAGTGACCAAAAGTGACACAGATCATTCATCATGTACTTGGGACTGAGCTGTGTATTCATTGTTTTTCTCTTAAAAGGTGTCCAGTGTGAGGTGAAGCTGGATGAGACTGGAGGAGGCTTGGTGCAACCTGGGAGGCCCATGAAACTCTCCTGTGTTGCCTCTGGATTCACTTTTAGTGACTACTGGATGAACTGGGTCCGCCAGTCTCCAGAGAAAGGACTGGAGTGGGTAGCACAAATTAGAAACAAACCTTATAATTATGAAACATATTATTCAGATTCTGTGAAAGGCAGATTCACCATCTCAAGAGATGATTCCAAAAGTAGTGTCTACCTGCAAATGAACAACTTAAGAGCTGAAGACATGGGTATCTATTACTGTACATGGTCGGCTACTGCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GGGAATGGTAAGCACG-1_contig_1": "GGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TTGGAACGTTTCGCTC-1_contig_5": "GCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TGATTTCTCTTTACAC-1_contig_7": "GGGAGAGCTTCCAGGTGTGCTGTCAGGCTGCTAGTGTATGCTCTCTCTAGTTTCTTTTTGGAGGCACTCAGAGCTATGAGTTTTCCTCTTAGGACTGCCTTCATTGTGTCCCATAAGTTTGGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GGCGACTAGCGATATA-1_contig_12": "GAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGAC",
        "TACTCGCCATCTATGG-1_contig_3": "GACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGA",
        "AAGGAGCGTAAAGTCA-1_contig_2": "GGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GTTTCTAAGACAAAGG-1_contig_3": "GAAGGCCCTGTCCATACAGTAGGTTTAGCTTGGCTACACCAAAGGAAGCCATACAGAGGCTAATATCAGAGTATTCTTGGAAGAGACAGGAGAAAATGAAAGCCAGTTTCTGCTCTTACCTTATGTGCTTGTGTTCAGACTCCCAAACATCAGGAGTGTCAGATAAACTGGTCTGAATCTCTGTCTGAAGCATGGAACTGAAAAGAATGTAGTTTCAGGGAAGAAAGGCAATAGAAGGAAGCCTGAGAATATCTTCAAAGGGTCAGACTCAATTTACTTTCTAAAGAAGTAGCTAGGAACTAGGGAATAACTTAGAAACAACAAGATTGTATATATGTGCATCCTGGCCCCATTGTTCCTTATCTGTAGGGATAAGCGTGCTTTTTTGTGT",
        "TGGTTAGAGAGTACAT-1_contig_5": "TGGGGACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCACGGGACAAACTCTTCTCCACAGTGTGACCTTCATGAGTGACCTGGCAGCTGTAACTGCTATGCCTTTCCCATGCTCTTGCTG",
        "CTAACTTGTTCCATGA-1_contig_3": "TGGGGGATATTTGCATACTTCATCTTCAGTAGTCACAAATATCTCACAGCTGTTTTAAAGCGATGTACTTAGGAGAAGAGCAGTAATTAGCCAGGGAACAAAATTCAAATACACAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTATCTCAGTTGTAATGTCCAGAGGAGAAAATGTGCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCTAGGGGAGAAGGTCACCATGAGCTGCAGGGCCAGCTCAAGTGTAAATTACATGTACTGGTACCAGCAGAAGTCAGATGCCTCCCCCAAACTATGGATTTATTACACATCCAACCTGGCTCCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGAACTCTTATTCTCTCACAATCAGCAGCATGGAGGGTGAAGATGCTGCCACTTATTACTGCCAGCAGTTTACTAGTTCCCCATTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCA",
        "ACGATACCATCGGACC-1_contig_2": "GATATATGAGCCCTATCTTCTCTACAGACACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTTATCTTCTTCCTGATGGCAGTGGTTACAGGGGTCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCAGAGCTTGTGAAGCCAGGGGCCTCAGTCAAGTTGTCCTGCACAGCTTCTGGCTTCAACATTAAAGACACCTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGAAGGATTGATCCTGCGAATGGTAATACTAAATATGACCCGAAGTTCCAGGGCAAGGCCACTATAACAGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTGCTAGAGCCCTAAGGGGTTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGC",
        "CAGCATAGTTGCTCCT-1_contig_1": "GGGGAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATATCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAACTTACATGCACTGGTACCAGCAGAAGTCAGGCACCTCCCCCAAAAGATGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACCTATTACTGCCAGCAGTGGAGTAGTAACCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACC",
        "GTGCGGTAGACGCAAC-1_contig_7": "GGGGACATCTCTCTCACACGGGACACTGAACTCTGAGGAGAAGGGTGTGAAGCTTAGAAGATGAGAGTGCTGGGATTTTTGTGCCTGGTGACAGTCCTTCCTGGTAGCCTGTCTCAGATGCAGCTTCAGGAGTCAGGACCTGGCCTGGTGAAACCCTCACAGTCACTCTTCCTCGCCTGCTCTATTACTGGTTTCCCCATCACCAGTGGTTACTACTGGATCTGGATCCGTCAGTCACCTGGGAAACCCCTAGAATGGATGGGGTACATCACTCATAGTGGGGAAACTTTCTACAACCCATCCCTCCAGAGCCCCATCTCCATTACTAGAGAAACATCCAAGAACCAGTTCTTTCTGCAATTGAACTCTGTGACCACAGAGGACACAGCCATGTATTACTGTGCAGGAGACAGATGGGGCTACGGCTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "GTCATTTGTTAAGACA-1_contig_6": "TGGGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACCTAGCCACCTCGCA",
        "AACCGCGAGAAACGCC-1_contig_1": "GGGGGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACAC",
        "CATGACAAGCCAGGAT-1_contig_6": "GCTATCTCTCTCACTGGAGGCTGATTTTTGAAGAAAGGGGTTGTAGCCTAAAAGATGATGGTGTTAAGTCTTCTGTACCTGTTGACAGCCCTTCCGGGTATCCTGTCAGAGGTGCAGCTTCAGGAGTCAGGACCTAGCCTCGTGAAACCTTCTCAGACTCTGTCCCTCACCTGTTCTGTCACTGGCGACTCCATCACCAGTGGTTACTGGAACTGGATCCGGAAATTCCCAGGGAATAAACTTGAGTACATGGGGTACATAAGCTACAGTGGTAGCACTTACTACAATCCATCTCTCAAAAGTCGAATCTCCATCACTCGAGACACATCCAAGAACCAGTACTACCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGATTCCTTTATGATTACGACTGGGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "ACGTCAAGTGTGACCC-1_contig_1": "GGGGGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CCAATCCTCAAACAAG-1_contig_3": "GGCTTGCTCTTTTCTTATATGGGACATACCAGCAAGGGAGTGACCAGTTAGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGTTATACCTTCACAGACTATTCAATGCACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACTGAGACTGGTGAGCCAACATATGCAGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCTAGAAGAGAGGGGGATGGTTACTACGTAGACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CCTACCATCTTTAGTC-1_contig_4": "GAGCATCAGACAGGCTGGGCAGCAAGATGGAATCACAGACCCAGGTCCTCATGTTTCTTCTGCTCTGGGTATCTGGTGCCTGTGCAGACATTGTGATGACACAGTCTCCATCCTCCCTGGCTATGTCAGTAGGACAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTAAATAGTAGCAATCAAAAGAACTATTTGGCCTGGTACCAGCAGAAACCAGGACAGTCTCCTAAACTTCTGGTATACTTTGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCATAGGCAGTGGATCTGGGACAGATTTCACTCTTACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGATTACTTCTGTCAGCAACATTATAGCACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TTCTCCTAGGTGCTTT-1_contig_4": "GGAGCTGGGTGAGCTGAGCTGAGCTGAGCTGGGTGAGCTGAGCTGGGCTGAGCTGAGCTGGGTGAGCTGAGCTGAACTGAGCTGAGCTGGGTGAGCTGAGCTGAGCTGAGCTGGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CCGTACTCACGACTCG-1_contig_2": "GATAAACCAGGCCTTTGGATTGTGACTGAACAGCATCAGACAGGCAGGGGAGCAATATGGTATCAGAGACCCATGTCCTCATATTTTTGCTGCTATGGGTGTCTGGTGCTTGTGGGGAAATTGTGTTGACCCAGTCTATACCATCCCTGACTGTGTCAGCAGGAGAGAGGGTCACTATCAGCTGCAAATCCAATCAGAATCTTTTATGGAGTGGAAACCAAAGGTACTGTTTGGTCTGGCACCAGTGGAAACCAGGGCAAACTCCTACACCGTTGATCACCTGGACATCTGATAGGTACTCTGGAGTCCCTGATCGTTTCATAGGCAGTGGATCTGTGACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGATGTGGCAGTTTATTTCTGTCAGCAGCATTTACACATTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CGAGCACGTCATGCCG-1_contig_6": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CCTCAGTGTGAAATCA-1_contig_2": "GATTGTCATTGCAGCCAGGACTCAGCATGGACATGAGGACCCCTGCTCAGTTTCTTGGAATCTTGTTGCTCTGGTTTCCAGGTATCAAATGTGACATCAAGATGACCCAGTCTCCATCTTCCATGTATGCATCTCTAGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAATAGCTATTTAAGCTGGTTCCAGCAGAAACCAGGGAAATCTCCTAAGACCCTGATCTATCGTGCAAACAGATTGGTAGATGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTCACCATCAGCAGCCTGGAGTATGAAGATATGGGAATTTATTATTGTCTACAGTATGATGAGTTTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TTATGCTGTCGAGATG-1_contig_5": "AGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "ACGTCAAGTCTCCACT-1_contig_5": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTAT",
        "ATTCTACAGTGCGATG-1_contig_6": "TGGGGCTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTG",
        "ACGCCAGGTACCGGCT-1_contig_5": "GACTGGTCTCTCTGCAGTTACTAGGCACACAGGACCTCACCATGAGATGGAGCTGTATCATCCTCTTCTTGGTAGCAACAGCTACAGGTGTCAACTCCCAGGTCCAACTGCAGCAGCCTGGGGCTGAGCTTGTGATGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACATTCACTGACTACTGGATGCACTGGGTGAAGCAGAGGCCTGGACAAGGCCTTGAGTGGATCGGAGCGATTGATACTTCTGATAGTTATACTAGCTACAATCAAAAGTTCAAGGGCAAGGCCACATTGACTGTAGACGAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGATGGGGGGGGAATTTTTATGAGGACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CGGACTGTCACTCTTA-1_contig_1": "GGTTTAAAGCAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATATCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAGGCACCTCCCCCAAAAGATGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGC",
        "TATGCCCAGTCAAGGC-1_contig_5": "GGGGAACCGACGATCAGTGTCCTCTCCAAAGTCCCTGAACACACTGACTCTAACCATGGAATGGAGTTGGATATTTCTCTTTCTCCTGTCAGGAACTGCAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTAAAGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGATACACATTCACTAGCTATGTTATGCACTGGGTGAAGCAGAAGCCTGGGCAGGGCCTTGAGTGGATTGGATATATTAATCCTTACAATGATGGTACTAAGTACAATGAGAAGTTCAAAGGCAAGGCCACACTGACTTCAGACAAATCCTCCAGCACAGCCTACATGGAGCTCAGCAGCCTGACCTCTGAGGACTCTGCGGTCTATTACTGTGCAAGACCTGGGAAGGTCCTGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GAACATCAGACCTAGG-1_contig_4": "TGGGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GGCCGATCATGTTCCC-1_contig_2": "TGGGGAAGCTGGCTATCACTCAAGGCCATGACCATGTTCTCACTAGCTCTTCTCCTCAGTCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCATGGCTATAGGAGAAAAAGTCACCATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGTACAGATTTTGTTTTTACAATTGAAAACATGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACTTGCCATGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCT",
        "GCGAGAAGTCGAACAG-1_contig_1": "GAATCTGATCATACACACTCCAACAGTCATTCTTGGTCAGGAGACGTTGTAGAAATGAGACCGTCTATTCAGTTCCTGGGGCTCTTGTTGTTCTGGCTTCATGGTGCTCAGTGTGACATCCAGATGACACAGTCTCCATCCTCACTGTCTGCATCTCTGGGAGGCAAAGTCACCATCACTTGCAAGGCAAGCCAAGACATTAACAAGTATATAGCTTGGTACCAACACAAGCCTGGAAAAGGTCCTAGGCTGCTCATACATTACACATCTACATTACAGCCAGGCATCCCATCAAGGTTCAGTGGAAGTGGGTCTGGGAGAGATTATTCCTTCAGCATCAGCAACCTGGAGCCTGAAGATATTGCAACTTATTATTGTCTACAGTATGATAATCTGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GCGCAACAGTGTGGCA-1_contig_4": "GACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTAC",
        "ACTATCTCAGTGGAGT-1_contig_3": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TCTGGAAGTAGCTAAA-1_contig_1": "GGGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGGGCTGCTTCTCTGCCTGGTGACTTTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCATCTCAGGGTTCTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGTAGTGATATGGAGTGATGGAAGCACAACCTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTCCAAACTGATGACACAGCCATGTACTACTGTGCCAGAAATTACTACGGTAGTAGCTACGGATACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "ATTGGTGCAGACAAGC-1_contig_3": "GGACATACCAGCAAGCGAGTGACCAGTTAGTCTTAAGGCACCACTTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCAATTTATAGGTACGACGGCTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "GCAAACTCAAACGCGA-1_contig_4": "GGGAACCGACGATCAGTGTCCTCTCCAAAGTCCCTGAACACACTGACTCTAACCATGGAATGGAGTTGGATATTTCTCTTTCTCCTGTCAGGAACTGCAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTAAAGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGATACACATTCTCTAGCTATGTTATACACTGGGTGAAGCAGAAGCCTGGGCAGGGCCTTGAGTGGATTGGATATATTAATCCTTACAATGATGGTACTAAGTACAATGAGAAGTTCAAAGGCAAGGCCACACTGACTTCAGACAAATCCTCCAGCACAGCCTACATGGAGCTCAGCAGCCTGACCTCTGAGGACTCTGCGGTCTATTACTGTGCAAGTGGGTACTACGGTAGTAGCTACTGGGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "TGAAAGAAGATGGCGT-1_contig_7": "GGGGAAAACAGCATATGATCAGTGTCCTCTCCAAAGTCCTTGAACATAGACTCTAACCATGGAATGGACCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGAGCTGAGCTGATGAAGCCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTACTGGCTACACATTCAGTAGCTACTGGATAGAGTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGAGATTTTACCTGGAAGTGGTAGTACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACATTCACTGCAGATACATCCTCCAACACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCCGTCTATTACTGTGCAAGACGGTATGATGGTTACTCCTTGTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "TGTTCCGGTGGTAACG-1_contig_4": "GACTAGTGTGCAGATATGGACAGGCTTACTTCCTCATTGCTGCTGCTGATTGTCCCTGCATATGTCCTGTCCCAGGTTACTCTGAAAGAGTCTGGCCCTGGGATATTGCAGCCCTCCCAGACCCTCAGTCTGACTTGTTCTTTCTCTGGGTTTTCACTGAGCACTTCTGGTATGGGTGTGAGCTGGATTCGTCAGCCTTCAGGAAAGGGTCTGGAGTGGCTGGCACACATTTACTGGGATGATGACAAGCGCTATAACCCATCCCTGAAGAGCCGGCTCACAATCTCCAAGGATACCTCCAGCAACCAGGTATTCCTCAAGATCACCAGTGTGGACACTGCAGATACTGCCACATACTACTGTGCTCCTCGTAGTTACCATAGTCGGTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "ATTTCTGTCCCGGATG-1_contig_3": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGCCATGAAGCCAGAACAACAAGAGCCCCAGGAACTGAATAGACGGTCT",
        "GCATGATAGCTAACAA-1_contig_1": "GGGGATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGTGCAAGTCAGGGCATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTATTACACATCAAGTTTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGGACAGATTATTCTCTCACCATCAGCAACCTGGAACCTGAAGATATTGCCACTTACTATTGTCAGCAGTATAGTAAGCTTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACA",
        "TCCCGATTCAACTCTT-1_contig_6": "GGGAACCACAGCCCCTGAAGACACTGACTCTAACTATGGAATGGAGCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTACAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGACCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGCTACTACATGCACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGACGTGTTAATCCTAACAATGGTGGTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCATATTAACTGTAGACAAGTCATCCAGCACAGCCTACATGGAGCTCCGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGATGGAACTACGGCCCATCATATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACC",
        "TTCGAAGCATCCTTGC-1_contig_7": "GGCAGCTCTGACAGAGGAGGCCGGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACGGACCCCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGCCCTCATTTTAAAAGGTGTCCAGTGTGAGGTGCAGCTGGTGGAGTCTGGGGGAGACTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGGCATGTCTTGGGTTCGCCAGACTCCAGACAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGGCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTGCAAGACATTACTACGGCTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TGACTTTGTTAAGGGC-1_contig_6": "GAACATTTGTCCAATGTCCTCTCCACAGTCCCTGCAGACACTGACCCTAAGCATGGAATGGAGATGGATCTTTCTCTTCCTCCTGTCAGGAACTACAGGTGTCCACTCTGAGATCCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGGTATCCTGCAAGGCTTCTGGTTATGCATTCACTAGCTACAACATGTACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGATATATTGATCCTTACAATGGTGGTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTTGACAAGTCCTCCAGCACAGCCTACATGCATCTCAACAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGATCGGACTATGATTACGTCCTCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGCCAAAACAACACCCCCATCAGTCTATCCACTGGCCCCTGGGTGTGGAGATACAACTGGTTCCTCCGTGACTCTGGGATGCCTGGTCAAGGG",
        "TGAGAGGAGGTGCTTT-1_contig_12": "GATCCTCTTCTCATAGAGCATCCATCAGAGCATGGCTGTCCTGGGGCTGCTTCTCTGCCTGGTGACTTTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCGTCTCAGGGTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGTAGTGATATGGAGTGATGGAAGCACAACCTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTCCAAACTGATGACACAGCCATGTACTACTGTGCCACGTATGGTAACTTCGCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAG",
        "TGGTTCCGTCTCCACT-1_contig_2": "AATCCCAGGAGAAAAAATGCTGGATGTACACTAAGGATGGCCCCTGACACTCTGCACTGCTACCTCTGGCCCCACCAGACAATGTTCCTGCAGAACCTGTTACCTTACTTGGCAGGGATTTTTGTCAAGGTATCTACTACTGTGTCTACTATGGTAACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "TAAACCGTCATGCATG-1_contig_2": "GGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGAC",
        "TTCGGTCCAAAGGAAG-1_contig_9": "GGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGTGCTGCTCCTCTGCCTGGTGACATTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGCAGTCAGGACCTAGCCTAGTGCAGCCCTCACAGAGCCTGTCCATAACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGAGGTGGAAGCACAGACTACAATGCAGCTTTCATGTCCAGACTGAGCATCACCAAGGACAACTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTGATGACACTGCCATATACTACTGTGCCAAAGGGGGTAACCCCTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTG",
        "ACTGCTCGTACCGCTG-1_contig_3": "GAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGAGACAGACTCTTCTCCACAGTGTCCCCTTCATGTGTAACTT",
        "TGAGCCGTCGTATCAG-1_contig_1": "GACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAACTGGCTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CGACTTCGTTTGACTG-1_contig_7": "GGGACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGATACTATAGGTACGACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CTACATTGTATTCTCT-1_contig_5": "GAGTTGGTTTAAAGCAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATATCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCACCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAGGCACCTCCCCCAAAAGATGGATTTATGACACATCCAAACTGCCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGACTAGTAACCCATGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TATCTCAGTATCGCAT-1_contig_1": "TGGGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CTGCCTAGTGGGTCAA-1_contig_6": "GATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCTTGGGGCTGCTCTTCTGCCTGGTGACATTCCCAAGCTGTGTCCTATCCCAGGTGCAGCTGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATCACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGTGGTGGAAGCACAGACTATAATGCAGCTTTCATATCCAGACTGAGCATCAGCAAGGACAATTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTAATGACACAGCCATATATTACTGTGCCAGCATGATCACGTTCTGGTCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CTAGAGTGTCACTGGC-1_contig_1": "GGGGAAAGTACTCATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGAGGAGAAAATGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAAAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAAGCACCTCCCCCAAACTCTGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCAGGTCGCTTCAGTGGCAGTGGGTCTGGAAACTCTTACTCTCTCACGATCAGCAGCATGGAGGCTGAAGATGTTGCCACTTATTACTGTTTTCAGGGGAGTGGGTACCCACTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GACGGCTTCACATAGC-1_contig_7": "TCTCATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCACATATGGTTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCC",
        "GATGCTACACTAAGTC-1_contig_3": "AGTTAAGAACTGACTAGACTCTATCTTGCTATTTGCATATTACATTTTCAGTAACCACAAATATCTCACAGTTGGTTTATAGCAAAGTACTTATGAGAATAGTAGTAATTAGCTAGGGACCAAAGTTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATACTGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAGACTCCTGATTTATGACACATCCAACCTGGCTTCTGGAGTCCCTGTTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTTACCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TTCTTAGGTGATAAAC-1_contig_4": "GAGGCACTCAGAGTGCAAGGGATGTTGGGAAGTCAGCCTTGTGACAGCCCCCGGTACATTTACTTATATAGAACCTCCCTGGGGAGATAGAATCCCAGGAGAAAAAATGCTGGATGTACACTAAGGATGGCCCCTGACACTCTGCACTGCTACCTCTGGCCCCACCAGACAATGTTCCTGCAGAACCTGTTACCTTACTTGGCAGGGATTTTTGTCAAGGTATCTACTACTGTGTCTACTATACCCTTCGTGGTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTG",
        "AAAGATGGTCTAACGT-1_contig_5": "GACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GCCTCTAGTAGAGGAA-1_contig_3": "AGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TAAGCGTAGGATGGTC-1_contig_4": "GACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "AAAGATGGTCTAACGT-1_contig_4": "TGGGGATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGATTATATTACTACGGTAGTGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCACCAGATGTCAGGCTGCTGAGTTGCATGTATGCTGTGCTGGAGGATTTGTCTA",
        "GAACATCAGACCTAGG-1_contig_7": "GGAGACATAGGAGCTTGGCTCTGGTTCCCAAGACCTCTCACTCACTTCTCAACATGGAGTGGGAACTGAGCTTAATTTTCATTTTTGCTCTTTTAAAAGATGTCCAGTGTGAAGTGCAGCTGTTGGAGACTGGAGGAGGCTTGGTGCAACCTGGGGGGTCACGGGGACTCTCTTGTGAAGGCTCAGGGTTTACTTTTAGTGGCTTCTGGATGAGCTGGGTTCGACAGACACCTGGGAAGACCCTGGAGTGGATTGGAGACATTAATTCTGATGGCAGTGCAATAAACTACGCACCATCCATAAAGGATCGATTCACTATCTTCAGAGACAATGACAAGAGCACCCTGTACCTGCAGATGAGCAATGTGCGATCTGAGGACACAGCCACGTATTTCTGTATGAGATACGCGGGTCACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GACTGCGTCCTATTCA-1_contig_4": "GGATATATGAGCCCTATCTTCTCTACAGACACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTTATCTTCTTCCTGATGGCAGTGGTTACAGGGGTCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCAGAGCTTGTGAAGCCAGGGGCCTCAGTCAAGTTGTCCTGCACAGCTTCTGGCTTCAACATTAAAGACACCTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGAAGGATTGATCCTGCGAATGGTAATACTAAATATGACCCGAAGTTCCAGGGCAAGGCCACTATAACAGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTGCTTTGTATGGTAACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "TTCCCAGAGACTACAA-1_contig_1": "GGGGGAAATGCATCACACCAGCATGGGCATCAAAATGGAGTCACAGATTCAGGTCTTTGTATTCGTGTTTCTCTGGTTGTCTGGTGTTGACGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGAGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGACAATCTCCTAAACTACTGATTTACTCGGCATCCTACCGGTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATCTGGGACGGATTTCACTTTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAACATTATAGTACTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAA",
        "CTCGAAAGTCGTTGTA-1_contig_1": "GATGAACTGAGCAGCATCAGACAGGCAGGGGAAGCAAGATGGATTCACAGGCCCAGGTTCTTATGTTACTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTAGCTGTGTCAGTTGGAGAGAAGGTTACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CTGCGGAAGCCAGTTT-1_contig_2": "GGAGAACTCTTCACACCAAGTTCTCAGAATGAGGTGCTCTCTTCAGTTCCTGGGGGTGCTTATGTTCTGGATCTCTGGAGTCAGTGGGGATATTGTGATAACCCAGGATGAACTCTCCAATCCTGTCACTTCTGGAGAATCAGTTTCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTATATAAGGATGGGAAGACATACTTGAATTGGTTTCTGCAGAGACCAGGACAATCTCCTCAGCTCCTGATCTATTTGATGTCCACCCGTGCATCAGGAGTCTCAGACCGGTTTAGTGGCAGTGGGTCAGGAACAGATTTCACCCTGGAAATCAGTAGAGTGAAGGCTGAGGATGTGGGTGTGTATTACTGTCAACAACTTGTAGAGTATCCTATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "ACATGGTAGATGCCTT-1_contig_2": "GGACATCTCTCTCATTAGAGGTTGATCTTTGAGGAAAACAGGGTGTTGCCTAAAGGATGAAAGTGTTGAGTCTGTTGTACCTGTTGACAGCCATTCCTGGTATCCTGTCTGATGTACAGCTTCAGGAGTCAGGACCTGGCCTCGTGAAACCTTCTCAGTCTCTGTCTCTCACCTGCTCTGTCACTGGCTACTCCATCACCAGTGGTTATTACTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATAAGCTACGACGGTAGCAATAACTACAACCCATCTCTCAAAAATCGAATCTCCATCACTCGTGACACATCTAAGAACCAGTTTTTCCTGAAGTTGAATTCTGTGACTACTGAGGACACAGCTACATATTACTGTGCAAGGGGTGGTTACGTCCCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAG",
        "GATCAGTTCGGGAGTA-1_contig_2": "GCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GCATACAAGGCAGTCA-1_contig_1": "GGGGATCCTCTTCTCATAGAGCATCCATCAGAGCATGGCTGTCCTGGGGCTGCTTCTCTGCCTGGTGACTTTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCGTCTCAGGGTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGTAGTGATATGGAGTGATGGAAGCACAACCTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTCCAAACTGATGACACAGCCATGTACTACTGTGCCAGAGGGAGCTACTATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "TGGGAAGAGAGCAATT-1_contig_2": "GGGTGACTGAACAGCATCAGACAGGCAGGGGAGCAATATGGTATCAGAGACCCATGTCCTCATATTTTTGCTGCTATGGGTGTCTGGTGCTTGTGGGGAAATTGTGTTGACCCAGTCTATACCATCCCTGACTGTGTCAGCAGGAGAGAGGGTCACTATCAGCTGCAAATCCAATCAGAATCTTTTATGGAGTGGAAACCAAAGGTACTGTTTGGTCTGGCACCAGTGGAAACCAGGGCAAACTCCTACACCGTTGATCACCTGGACATCTGATAGGTACTCTGGAGTCCCTGATCGTTTCATAGGCAGTGGATCTGTGACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGATGTGGCAGTTTATTTCTGTCAGCAGCATTTACACATTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCTCAGAGGAAGGTGGAAACACGGTGAGAGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACTGCCGAAAATAAACCACAGTGACTGAAACCCAACCC",
        "CCATGTCTCACCTTAT-1_contig_1": "CACTCCTCTTTCTGTCTTGATTACTTTTGACTCACCATATCAAGTTCGCAGAATGAGGTTCTCTGCTCAGCTTCTGGGGCTGCTTGTGCTCTGGATCCCTGGATCCACTGCAGATATTGTGATGACGCAGGCTGCATTCTCCAATCCAGTCACTCTTGGAACATCAGCTTCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTACATAGTAATGGCATCACTTATTTGTATTGGTATCTGCAGAAGCCAGGCCAGTCTCCTCAGCTCCTGATTTATCAGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTAGCAGTGGGTCAGGAACTGATTTCACACTGAGAATCAGCAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTGCTCAAAATCTAGAACTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "AAGGAGCGTCATATGC-1_contig_3": "GGGGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "ATTACTCGTACTCAAC-1_contig_6": "GACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TTTGTCATCGCTTGTC-1_contig_2": "GGGGAGGACAATGGCAGAAAGTCACTCTCAGTGAGGATACACCATCAGCATGAGGGTCCTTGCTGAGCTCCTGGGGCTGCTGCTGTTCTGCTTTTTAGGTGTGAGATGTGACATCCAGATGAACCAGTCTCCATCCAGTCTGTCTGCATCCCTTGGAGACACAATTACCATCACTTGCCATGCCAGTCAGAACATTAATGTTTGGTTAAGCTGGTACCAGCAGAAACCAGGAAATATTCCTAAACTATTGATCTATAAGGCTTCCAACTTGCACACAGGCGTCCCATCAAGGTTTAGTGGCAGTGGATCTGGAACAGGTTTCACATTAACCATCAGCAGCCTGCAGCCTGAAGACATTGCCACTTACTACTGTCAACAGGGTCAAAGTTATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GTGCTTCAGGGCTTGA-1_contig_1": "GGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GGCGACTGTCCTAGCG-1_contig_5": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTC",
        "GTCAAGTAGGCATTGG-1_contig_7": "GATAACATAAGATCACTGTTCTCTCTACAGTTACTGAGCACACAGGAACTCACCATGGGATGGAGCTGTATCATCCTCTTCTTGGCAGCAACAGCTACAGGTGTCCACTCCCAGGTCCAACTGCAGCAGTCTGGGCCTGAGCTGGTGAGGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCAGGCTATACCTTCACCAGCTACTGGATGCACTGGGTGAAACAGAGGCCTGGACAAGGCCTTGAGTGGATTGGCATGATTGATCCTTCCAATAGTGAAACTAGGTTAAATCAGAAGTTCAAGGACAAGGCCACATTGAATGTAGACAAATCCTCCAACACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGAGGGGGCTCGGGCCCTTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTG",
        "CAGGTGCCACCTCGGA-1_contig_4": "GGCAGCACTTAGGAGAGCAGGCCCTGCACCTCGCCCAGCTAGCACCGTGGAGCGGGCCCTGGTGGCAAGTGAGCAAGCCCCATGAGTGTGGGAGAGTTGACCCCTCCTTCTTCTGCAGTGGGCTGGCACAGGTGCAGGGGTGATGCCCCCTCCCACCCCTCACTACCTCAGCAGTCAGAAAAGTAACCACAGGGTCAGGAGAGTTGGAGGACACAACAGAGTTGACCTTGGTGGTGGGGACACAGATGAGCCACCACCAAGATCTGAGTCCACCACAAGTCTGCTACAAGGTGGTATATGTACATGGTTGATGTCCCCTCCACACCTCCCCCTCAACGCCTTCAGTAGTCAGGAAAGAAAATGAAGATGTGTGGACAGAGGGACATACTTTGGAACACACACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GACGGCTTCGCCGTGA-1_contig_4": "TGGGATCAGACAGGCAGTGGGAGCAAGATGGATTCACAGGCCCAGGTTCTTATATTGCTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTGGCTGTGTCAGCAGGAGAGAAGGTCACTATGAGCTGCAAATCCAGTCAGAGTCTGCTCAACAGTAGAACCCGAAAGAACTACTTGGCTTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATCTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGCAAGCAATCTTATAATCTTCCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CGAGCACGTTCAGTAC-1_contig_5": "GAGTTCTAGAGCTTCCAGGTGTGCTGTCAGGCTGCTAGTGTATGCTCTCTCTAGTTTCTTTTTGGAGGCACTCAGAGCTATGAGTTTTCCTCTTAGGACTGCCTTCATTGTGTCCCATAAGTTTGGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTG",
        "ACTGTCCTCGGAAATA-1_contig_3": "GAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCTTCAGTCATAATGTCCAGAGGACAAATTGTTCTCTCCCAGTCTCCAGCAATCCTGTCTGCATCTCCAGGGGAGAAGGTCACAATGACTTGCAGGGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAAACCCTGGATTTATGCCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGAGTGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "ATCGAGTGTCATATCG-1_contig_3": "GAGCTCTGACAGAGGAGGCCGGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACGGACCCCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGCCCTCATTTTAAAAGGTGTCCAGTGTGAGGTGCAGCTGGTGGAGTCTGGGGGAGACTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGGCATGTCTTGGGTTCGCCAGACTCCAGACAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGGCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTGCAAGACAGTATGGTAACTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGAGCCACAGCAGTTT-1_contig_1": "AGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGG",
        "TCTCTAAAGGAATCGC-1_contig_1": "TTATCGTTTCTTATATGGGGAAGCTGGCTATCACTCAAGGCCATGACCATGTTCTCACTAGCTCTTCTCCTCAGTCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCATGGCTATAGGAGAAAAAGTCACCATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGTACAGATTTTGTTTTTACAATTGAAAACATGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACTTGCCTCTCACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TTCTCCTAGGTGCTTT-1_contig_3": "GATCATCATCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGCATTACTCTTCTGCCTGGTAACATTCCCAAGCTGTATCCTTTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCGTCTCAGGGTTCTCATTAACCGGCTATGGTGTAAACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAATGATATGGGGTGATGGAAGCACAGACTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCAGGTACTACTGTGCCAGAGATGGTTACTACGACGTTTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ACGCAGCTCACGGTTA-1_contig_2": "GACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATATCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAGGCACCTCCCCCAAAAGATGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAA",
        "TCCACACAGCCTCGTG-1_contig_1": "GAAAGTGATGTTCTTATAAGACTAGCACTAATTAGCCAGAGACCAAAATCCAAATACACAATGGACTTTCGGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGTCACAGTGTCCAGAGGAGAAATTGTGCTCACTCAGTCTCCAGCCATCACAGCTGCATCTCTGGGGCAAAAGGTCACCATCACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAGGCACCTCCCCCAAACCATGGATTTATGAAATATCCAAACTGGCTTCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCATTTATTACTGCCAGCAGTGGAATTATCCTCTTACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "ACTGCTCGTACCGCTG-1_contig_2": "GATGCACTTATAGGAAGAGCAATAATTAGTCAGAGACCAGGATCAAAAACACAATGGATTTTCATGTGCAGATTTTCAGCTTCATGCTAATCAGTGTCACAGTCATATTGTCCAGTGGAGAAATTGTGCTCACCCAGTCTCCAGCACTCATGGCTGCATCTCCAGGGGAGAAGGTCACCATCACCTGCAGTGTCAGCTCAAGTATAAGTTCCAGCAACTTGCACTGGTACCAGCAGAAGTCAGAAACCTCCCCCAAACCCTGGATTTATGGCACATCCAACCTGGCTTCTGGAGTCCCTGTTCGCTTCAGTGGCAGTGGATCTGGGACCTCTTATTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGTCAACAGTGGAGTAGTTACCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAAAGGAACCATAACTGATTTAATGAGCCATTCGCAGTTTCACTGCCCATATAAGAAACGCTCGCTTGTAAGAGCCTCAGACCCAGA",
        "CCTACCATCTTTAGTC-1_contig_5": "GGAACAGCATCAGACAGGCAGGGGAGCAATATGGTATCAGAGACCCATGTCCTCATATTTTTGCTGCCATGGGTGTCTGGTGCTTGTGGGGAAATTGTGCTGACTGTGTCAGCAGGAGAGAGGGTCACTATCAGCTGCAAATCCAATCAGAATCTTTTATGGAGTGGAAACCAAAGGTACTGTTTGGTCTGGCACCAGTGGAAACCAGGGCAAACTCCTACACCGTTGATCACCTGGACATCTGATAGGTACTCTGGAGTCCCTGATCGTTTCATAGGCAGTGGATCTGTGACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGATGTGGCAGTTTATTTCTGTCAGCAGCATTTACACATTCCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CATGACAAGCCAGGAT-1_contig_2": "GGGATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCTCCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TGCTACCGTACATCCA-1_contig_6": "TGGGGATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGTATGATTACGACAGGGTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CCATTCGCATCAGTAC-1_contig_1": "GACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGATTTCCGATCTATGATGGTTACTACCTTCTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TACTCGCCATCTATGG-1_contig_1": "AGTGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGGTGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTTCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCATTCACGTTCGGCTCGGGGACAAACTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATG",
        "ACTGTCCTCGGAAATA-1_contig_1": "GAGCATATGATCAGTGTCCTCTCCAAAGTCCTTGAACATAGACTCTAACCATGGAATGGACCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGAGCTGAGCTGATGAAGCCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTACTGGCTACACATTCAGTAGCTACTGGATAGAGTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGAGATTTTACCTGGAAGTGGTAGTACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACATTCACTGCAGATACATCCTCCAACACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCCGTCTATTACTGTGCAAGAAAGTATGGTAACTACCCTTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CTGATCCAGTAGCGGT-1_contig_6": "GGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTAC",
        "CAGTAACGTTATGTGC-1_contig_3": "GAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATC",
        "GCGAGAATCGTATCAG-1_contig_1": "GAGTCACACATATCCTGTACCGTTGTCATTGCAGTCAGGACTCAGCATGGACATGAGGGCCCCTGCTCAGTTTTTTGGGATCTTGTTGCTCTGGTTTCCAGGTATCAGATGTGACATCAAGATGACCCAGTCTCCATCCTCCATGTATGCATCGCTGGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAAAAGCTATTTAAGCTGGTACCAGCAGAAACCATGGAAATCTCCTAAGACCCTGATCTATTATGCAACAAGCTTGGCAGATGGGGTCCCATCAAGATTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTAACCATCAGCAGCCTGGAGTCTGACGATACAGCAACTTATTACTGTCTACAGCATGGTGAGAGCCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCA",
        "CTGTGCTAGTACATGA-1_contig_2": "TGGGGAATCAACCTCCTTAAAATGAATTTGCCTGTTCATCTCTTGGTGCTTCTGTTGTTCTGGATTCCTGCTTCCAGAGGTGATGTTGTGGTGACTCAAACTCCACTCTCCCTGCCTGTCAGCTTTGGAGATCAAGTTTCTATCTCTTGCAGGTCTAGTCAGAGTCTTGCAAACAGTTATGGGAACACCTATTTGTCTTGGTACCTGCACAAGCCTGGCCAGTCTCCACAGCTCCTCATCTATGGGATTTCCAACAGATTTTCTGGGGTGCCAGACAGGTTCAGTGGCAGTGGTTCAGGGACAGATTTCACACTCAAGATCAGCACAATAAAGCCTGAGGACTTGGGAATGTATTACTGCTTACAAGGTACACATCAGCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TTGTAGGAGCCCAACC-1_contig_3": "GACTCAGTCAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "ACTGAGTCATGGTCAT-1_contig_3": "GACAACCTATGATCACTGTCCTCTCCACAGTCCCTGAAGACACTGACTCTAACCATGGGATGGAGCTGTATCTTTCTCTTTCTCCTGTCAGTAACTGTAGGTGTGTTCTCTGAGGTTCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTTACTGGCTACTTTATGAACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGACGTATTAATCCTTACAATGGTGATACTTTCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAATCCTCTAGCACAGCCCACATGGAGCTCCTGAGCCTGACATCTGAGGACTCTGCAGTCTATTATTGTGGGAGGGGTAACTACTTCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AAATGCCGTACCGGCT-1_contig_4": "GGGGAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCAAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AAGTCTGTCCATGAAC-1_contig_2": "GGGACATATCCTGTGCCATTGTCATTGCATTCAGAACTCAGCATGGACATCAGGGCTCCTGCTCAGTTTCTTGGCATCTTGTTGCTCTGGTTTCCAGGTGCCAGATGTGACATCCAGATGATTCAGTCTCCATCGTCCATGTTTGCCTCTCTGGGAAACAGAGTCAGTCTCTCTTGTCGGGCTAGTCAGGGCATTAGAGGTAATTTAGACTGGTATCAGCAGAAACCAGGTGGAACTATTAAGCTCCTGATCTACTCCACATCCAATTTAAATTCTGGTGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGGTCAGATTATTCTCTCACCATCAGCAGCCTAGAGTCTGAAGATTTTGCAGACTATTACTGTCTACAGCGTAATGCGTTTCCTCTGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CACACAACAGTTTACG-1_contig_8": "GGGCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTGA",
        "CTGAAACTCCTCATTA-1_contig_2": "GACTGTTTTCTCTTTCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGAATTGAAGTCTCAACTATGAAAATCAGCAGTTCTCTGTCAGAGAAGCCCAAGCGCTTCCACGCATGCTTGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GCGAGAAAGACACTAA-1_contig_8": "GAACTTTAGGTAGTACCTTCTGTGAACCAGTTTCAATCCTGTTCTTAGCCTGTGCAGAGAGATGGGTGTGGGGTAGCATTCAGACCTTTGTGACAATTGGGTTGCAACTCCTAAGAGGAAAGATCCCAAAGCTAAGCTCCTGGAAGCTATGGACCGTCACAAAAAAGAAGGCAGTCACTTTCATATTTCCCATGCATGAGTTGTGTCAGCCGGTAAAGGAGAGTACAGTGGAAGAAGGCCTTGAGACCTGAGTCAGCAGTGGATATCAGCAAAACAGAACAACCAACAACCCATGGAAGAGTGGGAAGAAGTCGAAATACCAGACTGTGGAGGGGGTCTCTCTGATGCACAGGTAAGGTGATGGCCTGAAGAATGTCATCCAGACAGGGTCCAAGGATGCCACAGGCTGGAACATGGGCAAGTGGATCTGAACACACACAACAAAAACAACAACTACTGCTACCACCACCACCAGAGTCAGCCTCAAGGAGATGATGGGATATATCAGGATACCTAAGATAAGATGAACTTGGTCTGTCTGGAAGCTGGCAGGACCAAATTCGCTGAGTCATCAGAGAAGAGGTGGCCAGAGGAGCAAGATCTCTGCAGCAGAAATCTGCAGGACTAACAAAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCC",
        "ATCTGCCAGGAATCGC-1_contig_2": "AGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGACAGACTCTTCTCCAC",
        "ACGCCAGTCGTGGTCG-1_contig_2": "GATCATATGAGCCCTATCTTCTCTACAGACACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTTATCTTCTTCCTGATGGCAGTGGTTACAGGGGTCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCAGAGCTTGTGAAGCCAGGGGCCTCAGTCAAGTTGTCCTGCACAGCTTCTGGCTTCAACATTAAAGACACCTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGAAGGATTGATCCTGCGAATGGTAATACTAAATATGACCCGAAGTTCCAGGGCAAGGCCACTATAACAGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTGCTAGAACTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGC",
        "GCGAGAACAGTCGATT-1_contig_4": "ACATTTTTCTTATATGGGCAGTCGATTGTTGAGTCGTTTTCTTATATGGGGAGCTCTGACAGAGGAGGCCGGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACGGACCCCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGCCCTCATTTTAAAAGGTGTCCAGTGTGAGGTGCAGCTGGTGGAGTCTGGGGGAGACTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGGCATGTCTTGGGTTCGCCAGACTCCAGACAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGGCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTGCAAGACAATATGATTACGACGGCTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GCATGATAGGGCATGT-1_contig_5": "GCTTGCAGTTCAGACATAGGAGCTTGGCTCTGGTTCCCAAGACCTCTCACTCACTTCTCAACATGGAGTGGGAACTGAGCTTAATTTTCATTTTTGCTCTTTTAAAAGATGTCCAGTGTGAAGTGCAGCTGTTGGAGACTGGAGGAGGCTTGGTGCAACCTGGGGGGTCACGGGGACTCTCTTGTGAAGGCTCAGGGTTTACTTTTAGTGGCTTCTGGATGAGCTGGGTTCGACAGACACCTGGGAAGACCCTGGAGTGGATTGGAGACATTAATTCTGATGGCAGTGCAATAAACTACGCACCATCCATAAAGGATCGATTCACTATCTTCAGAGACAATGACAAGAGCACCCTGTACCTGCAGATGAGCAATGTGCGATCTGAGGACACAGCCACGTATTTCTGTATGAGATATGGTAACTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAAGA",
        "TGACGGCTCCTTGGTC-1_contig_1": "GAACATTTGTCCAATGTCCTCTCCACAGTCCCTGCAGACACTGACCCTAAGCATGGAATGGAGATGGATCTTTCTCTTCCTCCTGTCAGGAACTACAGGTGTCCACTCTGAGATCCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGGTATCCTGCAAGGCTTCTGGTTATGCATTCACTAGCTACAACATGTACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGATATATTGATCCTTACAATGGTGGTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTTGACAAGTCCTCCAGCACAGCCTACATGCATCTCAACAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGAAACTACGGTAGTAGCTACCACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CCGTTCACACTCGACG-1_contig_2": "GGGACTCAGTCAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CACCTTGCAGTTCATG-1_contig_4": "AGTCAGTGTTGTGACAGCCCCAGGTACATTTACTTATATAGGACCTCCCTGGGGAGATAGAATCCCAGGAGTAAAAATGCTGGATGTACACTAAGGATGGCCCCTGACACTCTGCACTGCTACCTCTGGCCCCACCAGGCAATGTTCCTGCAGAATCTCTTACCTTACTTGGCAGGGATTTTTGTCAAGGGATCTACTACTGTGTCTACTATGATTACGACTACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCC",
        "AAGTCTGGTATAGGGC-1_contig_1": "GGGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTG",
        "TTGCCGTTCCACTCCA-1_contig_2": "TGGGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTGA",
        "GCATACAAGGCAGTCA-1_contig_5": "GAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAAC",
        "TTGGAACGTTTCGCTC-1_contig_6": "GCTATGATCAGTGTCCTCTCTACACAGTCCCTGACAACACTGACTCTAACCATGGGATGGAGCCGGATCTTTCTCTTCCTCCTGTCAATAATTGCAGGTGTCCATTGCCAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAGGATATCCTGCAAGGCTTCTGGCTACACCTTCACAAGCTACTATATACACTGGGTGAAGCAGAGGCCTGGACAGGGACTTGAGTGGATTGGATGGATTTATCCTGGAAATGTTAATACTAAGTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACCTCTGAGGACTCTGCGGTCTATTTCTGTGCAAGAGGCTATAGGTACGACGGGGTCTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACC",
        "ACAGCCGCACGAAAGC-1_contig_1": "TGGGGAGCTGCCAGGAGCCTAAGAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGTTATGGGTACTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCAAAAGTGTCAGTACATCTGGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCACAGTAGGGAGCTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "GTGGGTCTCGGCCGAT-1_contig_1": "TGGGGGTTCTGCATCAGAAAGGCAGGGGGATCAAGATGGAATCACAGACTCAGGTCTTCCTCTCCCTGCTGCTCTGGGTATCTGGTACCTGTGGGAACATTATGATGACACAGTCGCCATCATCTCTGGCTGTGTCTGCAGGAGAAAAGGTCACTATGAGCTGTAAGTCCAGTCAAAGTGTTTTATACAGTTCAAATCAGAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATCTACTGGGCATCCACTAGGGAATCTGGTGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTTACTCTTACCATCAGCAGTGTACAAGCTGAAGACCTGGCAGTTTATTACTGTCATCAATACCTCTCCTCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "ATCCGAAAGGACAGCT-1_contig_6": "GACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAAC",
        "TCTCATAGTCTCTTAT-1_contig_8": "GAGTTCCTATCTCTCTCACTGGAGGCTGATTTTTGAAGAAAGGGGTTGTAGCCTAAAAGATGATGGTGTTAAGTCTTCTGTACCTGTTGACAGCCCTTCCGGGTATCCTGTCAGAGGTGCAGCTTCAGGAGTCAGGACCTAGCCTCGTGAAACCTTCTCAGACTCTGTCCCTCACCTGTTCTGTCACTGGCGACTCCATCACCAGTGGTTACTGGAACTGGATCCGGAAATTCCCAGGGAATAAACTTGAGTACATGGGGTACATAAGCTACAGTGGTAGCACTTACTACAATCCATCTCTCAAAAGTCGAATCTCCATCACTCGAGACACATCCAAGAACCAGTACTACCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGATCCCTCTATGATGGTTACTACGTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGGCTAGCAGGAATGC-1_contig_2": "GGGAGCTGCCAGGAGCCTAAGAAGCATCCTCTCATCTAGTTCTCAGAGATGGAGACAGACACAATCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGCTCCACTGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAAGGCCAGCCAAAGTGTTGATTATGATGGTGATAGTTATATGAACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAATCTAGAATCTGGGATCCCAGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCACGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAA",
        "GCATGCGGTATATGAG-1_contig_6": "TGCAACTGTTTCTTATATGGGGAGCTCTGACAGAGGAGGCCAGTCCTGGAATTGATTCCCAGTTCCTCACGTTCAGTGATGAGCAGTGAACACAGACCCCTCACCATGAACTTCGGGCTCAGATTGATTTTCCTTGTCCTTACTTTAAAAGGTGTCCAGTGTGACGTGAAGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATACCATGTCTTGGGTTCGCCAGACTCCGGAGAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTACAAGAATCTATAGGTACGACTCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ACTGAACGTACCGGCT-1_contig_1": "GAGGACAATGGCAGAAAGTCACTCTCAGTGAGGATACACCATCAGCATGAGGGTCCTTGCTGAGCTCCTGGGGCTGCTGCTGTTCTGCTTTTTAGGTGTGAGATGTGACATCCAGATGAACCAGTCTCCATCCAGTCTGTCTGCATCCCTTGGAGACACAATTACCATCACTTGCCATGCCAGTCAGAACATTAATGTTTGGTTAAGCTGGTACCAGCAGAAACCAGGAAATATTCCTAAACTATTGATCTATAAGGCTTCCAACTTGCACACAGGCGTCCCATCAAGGTTTAGTGGCAGTGGATCTGGAACAGGTTTCACATTAACCATCAGCAGCCTGCAGCCTGAAGACATTGCCACTTACTACTGTCAACAGGGTCAAAGTTATCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GCTGGGTGTCCGAGTC-1_contig_4": "GACTTGGAAGGGTCTCAAGGCATCACCACAGCGCCAACCACTAGAGGAGGAAGGAAAGCAATCTGTCAGTGAGGAAGTCACCTAGAGACAGATCTTTCCAGCAATTCTTATCACTCCTCCCTCAGCAGCCTCCATCCAGGCATTAAGAGAACAAGAGATGTTGAAATGTCAGCCTTGTGGCTGGCACAGACCCAGCATTCCATCCAGAGCATTCCTGAAGAGTTAGAAGCCCAGGAGGAAAAAGGCTGGACCTGCACTCAGGATGACCCCTGGACTCGGCCCTGCTGCCTGTGCCTCAAAGGTCCCTTACTCATGCAAAACCAGAGTCCTTACTCATCTGGGCTTTTTGTGAAGGGATCTACTACTGTGTTCATTACTACGGCTACGTAAGATTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "GACCAATCATCGTCGG-1_contig_4": "ACCAATCATCGTCGGTGCCGTCGGCTTTCTTTTGGGGCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGG",
        "TGACGGCTCCTTGGTC-1_contig_3": "GAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATATCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAGGCACCTCCCCCAAAAGATGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TGACGGCGTGTGACCC-1_contig_1": "GGGGGAAATGCATCACACCAGCATGGGCATCAAAATGGAGTCACAGATTCAGGTCTTTGTATTCGTGTTTCTCTGGTTGTCTGGTGTTGACGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGAGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGACAATCTCCTAAACTACTGATTTACTCGGCATCCTACCGGTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATCTGGGACGGATTTCACTTTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAACATTATAGTACTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCA",
        "CGTCAGGAGCGAGAAA-1_contig_3": "GCTTTCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TTCGGTCCAAAGGAAG-1_contig_1": "TGGGGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CGATGGCCAAGCCGTC-1_contig_6": "GAATAGAGACCTGCAGTTGAGGCCAGCAGGTCGGCTGGACTAACTCTCCAGCCACAGTAATGACCCAGACAGAGAAGGCCAGACTCATAAAGCTTGCTGAGCAAAATTAAGGGAACAAGGTTGAGAGCCCTAGTAAGCGAGGCTCTAAAAAGCATGGCTGAGCTGAGATGGGTGGGCTTCTCTGAGCGCTTCTAAAATGCGCTAAACTGAGGTGATTACTCTGAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGT",
        "TTTGGTTCATTTCACT-1_contig_8": "TATCTGGTATCATCAGTGGCATCAGCCTAGGTCCAACCCCGAGCATGGTATAGCAGGAAGACATGCAAATAGGTCTTCTCTGTGCCCATGAAAAACACCTCGGCCCTGACCCTGTAGCTCTGACAGAGGAGGCCAGTCCTGGAATTGATTCCCAGTTCCTCACGTTCAGTGATGAGCAGTGAACACAGACCCCTCACCATGAACTTCGGGCTCAGATTGATTTTCCTTGTCCTTACTTTAAAAGGTGTCCAGTGTGACGTGAAGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATACCATGTCTTGGGTTCGCCAGACTCCGGAGAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTACAAGAGATCGAAGGGGCCCCCGTGGTTACTCCTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "ATCCACCCACCGTTGG-1_contig_5": "GACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TTGCCGTTCATCGATG-1_contig_1": "GGACAAATATCTCACAGTTGTCTTAAAGCAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGCATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATAACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTTCCAGCAGAAGCCAGGCACTTCTCCCAAACTCTGGATTTATAGCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGATCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAAAGGAGTAGTTACCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTAC",
        "TCAGCTCGTCGCGAAA-1_contig_2": "TCTTATATGGGGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGCA",
        "GTCTCGTGTCTCCACT-1_contig_5": "GAAAACAACATATGTCCAATGTCTTCTCCACAGACACTGAACACACTGACTCTAACCATGGGATGGAGCTGGATCTTTCTCTTTCTCCTGTCAGGAACTGCAGGTGTCCTCTCTGAGGTCCAGCTGCAACAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGCTACTACATGCACTGGGTGAAGCAAAGCCATGTAAAGAGCCTTGAGTGGATTGGACGTATTAATCCTTACAATGGTGCTACTAGCTACAACCAGAATTTCAAGGACAAGGCCAGCTTGACTGTAGATAAGTCCTCCAGCACAGCCTACATGGAGCTCCACAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGAGGGGGCTACGGTAGTAGCTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAA",
        "CACACAACAGTTTACG-1_contig_5": "TGGGGATCTACATCTGAAAGGCAGGTGGAGCAAGATGGAATCACAGACTCAGGTCCTCATCTCCTTGCTGTTCTGGGTATCTGGTACCTGTGGGGACATTGTGATGACACAGTCTCCATCCTCCCTGAGTGTGTCAGCAGGAGAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGTCTGTTAAACAGTGGAAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGCCTCCTAAACTGTTGATCTACGGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGAACCGATTTCACTCTTACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGAATGATCATAGTTATCCGCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGTTAACATCTGGAGGTGCCTCAGTCGTG",
        "GCGAGAATCGTATCAG-1_contig_4": "GGGCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAAC",
        "GACCTGGAGATCTGCT-1_contig_5": "GGGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAAAGAGTCTGTCTCCTGC",
        "CACATTTAGCTCTCGG-1_contig_2": "TGGGGGGAAGCTGGCTATCACTCAAGGCCATGACCATGCTCTCACTAGCTCCTCTCCTCAGCCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCGTGGCTACAGGAGAAAAAGTCACTATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGCACAGATTTTGTTTTTACAATTGAAAACACGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTCCAGTGAGCAGTTAACATCTGGAGGTGCCT",
        "TTGAACGGTCATCCCT-1_contig_1": "GGGGAGCTGCCAGGAGCCTAAGAAGCATCCTCTCATCTAGTTCTCAGAGATGGAGACAGACACAATCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGCTCCACTGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAAGGCCAGCCAAAGTGTTGATTATGATGGTGATAGTTATATGAACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAATCTAGAATCTGGGATCCCAGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CACATAGAGATTACCC-1_contig_1": "AGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CTTACCGGTCTAGTCA-1_contig_3": "AGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "AGAGCTTCAATCTACG-1_contig_6": "GGGGAGCTGCCAGGAGCCTAAGAAGCATCCTCTCATCTAGTTCTCAGAGATGGAGACAGACACAATCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGCTCCACTGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAAGGCCAGCCAAAGTGTTGATTATGATGGTGATAGTTATATGAACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAATCTAGAATCTGGGATCCCAGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCTTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGT",
        "TGCCCTAGTAGAGGAA-1_contig_2": "GCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GCAATCAGTGGCGAAT-1_contig_5": "GAAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACC",
        "ACCGTAATCTCTGCTG-1_contig_8": "GGGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGT",
        "ATTTCTGTCCCGGATG-1_contig_4": "GAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CGGAGCTAGGCATTGG-1_contig_4": "CTTGAATCTGAAGCAAAGGGGATCAGCCCGAGATTCTCATTCAGTGATCAACACTGAACACACATCCCTTACCATGGATTTTGGGCTGATTTTTTTTATTGTTGCTCTTTTAAAAGGGGTCCAGTGTGAGGTGAAGCTTCTCGAGTCTGGAGGTGGCCTGGTGCAGCCTGGAGGATCCCTGAAACTCTCCTGTGCAGCCTCAGGATTCGATTTTAGTAGATACTGGATGAGTTGGGTCCGGCAGGCTCCAGGGAAAGGGCTAGAATGGATTGGAGAAATTAATCCAGATAGCAGTACGATAAACTATACGCCATCTCTAAAGGATAAATTCATCATCTCCAGAGACAACGCCAAAAATACGCTGTACCTGCAAATGAGCAAAGTGAGATCTGAGGACACAGCCCTTTATTACTGTGCAAGACCCTACTATCCGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CTACATTCACTAGTAC-1_contig_3": "CACACACCTGAAACAAGATTTGTTTGGCCAATAAAATTTGCATGAAGGGCCCTCTCATTTCTCTGAGAATTTTGAAGGATAAGAGAGAACTACAACCTGTCTGTCTCAGCAGAGATCAGTAGTACCTGCATTATGGCCTGGACTCCTCTCTTCTTCTTCTTTGTTCTTCATTGCTCAGGTTCTTTCTCCCAACTTGTGCTCACTCAGTCATCTTCAGCCTCTTTCTCCCTGGGAGCCTCAGCAAAACTCACGTGCACCTTGAGTAGTCAGCACAGTACGTACACCATTGAATGGTATCAGCAACAGCCACTCAAGCCTCCTAAGTATGTGATGGAGCTTAAGAAAGATGGAAGCCACAGCACAGGTGATGGGATTCCTGATCGCTTCTCTGGATCCAGCTCTGGTGCTGATCGCTACCTTAGCATTTCCAACATCCAGCCTGAAGATGAAGCAATATACATCTGTGGTGTGGGTGATACAATTAAGGAACAATTTGTGATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGCCTGTGGGGACATTGTGATGTCACAGTCTCCATCC",
        "CATTATCGTGCCTGCA-1_contig_5": "GAGGCAGAGAACTTTAGCCCTGTCTTCTTTTTTAGTGTTCAGCACTGACAATATGACATTGAACATGCTGTTGGGGCTGAAGTGGGTTTTCTTTGTTGTTTTTTATCAAGGTGTGCATTGTGAGGTGCAGCTTGTTGAGTCTGGTGGAGGATTGGTGCAGCCTAAAGGGTCATTGAAACTCTCATGTGCAGCCTCTGGATTCACCTTCAATACCTACGCCATGAACTGGGTCCGCCAGGCTCCAGGAAAGGGTTTGGAATGGGTTGCTCGCATAAGAAGTAAAAGTAATAATTATGCAACATATTATGCCGATTCAGTGAAAGACAGGTTCACCATCTCCAGAGATGATTCACAAAGCATGCTCTATCTGCAAATGAACAACTTGAAAACTGAGGACACAGCCATGTATTACTGTGTGGCCAACTGGGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AACACGTGTTCGGGCT-1_contig_1": "GGAATTAGCTAGGGACCAAAGTTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATACTGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAGACTCCTGATTTATGACACATCCAACCTGGCTTCTGGAGTCCCTGTTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTTACCCACTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACT",
        "CCGTTCACACTCGACG-1_contig_3": "GGGACTCAGTCAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCAC",
        "TGGTTCCGTCTCCACT-1_contig_3": "GACATACCAGCAAGGGAGTGACCAGTTAGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGTTATACCTTCACAGACTATTCAATGCACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACTGAGACTGGTGAGCCAACATATGCAGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCTAGAAATCTCTACTATGATTACGCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "TGGCTGGAGCACCGCT-1_contig_6": "GAAGATCACAGTAGTCTCTACAGTCACAGGAGTACACAGGGCATTGCCATGGGTTGGAGCTGTATCATCTTCTTTCTGGTAGCAACAGCTACAGGTGTGCACTCCCAGGTCCAGCTGCAGCAGTCTGGGCCTGAGGTGGTGAGGCCTGGGGTCTCAGTGAAGATTTCCTGCAAGGGTTCCGGCTACACATTCACTGATTATGCTATGCACTGGGTGAAGCAGAGTCATGCAAAGAGTCTAGAGTGGATTGGAGTTATTAGTACTTACAATGGTAATACAAACTACAACCAGAAGTTTAAGGGCAAGGCCACAATGACTGTAGACAAATCCTCCAGCACAGCCTATATGGAACTTGCCAGATTGACATCTGAGGATTCTGCCATCTATTACTGTGCAAGATACGGTAGTAGCTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "CCCAATCAGTGGTAGC-1_contig_1": "GATCAGACAGGCAGGGGAAGCAAGATGGATTCACAGGCCCAGGTTCTTATGTTACTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTAGCTGTGTCAGTTGGAGAGAAGGTTACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TGAGAGGAGGTGCTTT-1_contig_1": "GGGGGAGAACTACAACCTGTCTGTCTCAGCAGAGATCAGTAGTACCTGCATTATGGCCTGGACTCCTCTCTTCTTCTTCTTTGTTCTTCATTGCTCAGGTTCTTTCTCCCAACTTGTGCTCACTCAGTCATCTTCAGCCTCTTTCTCCCTGGGAGCCTCAGCAAAACTCACGTGCACCTTGAGTAGTCAGCACAGTACGTACACCATTGAATGGTATCAGCAACAGCCACTCAAGCCTCCTAAGTATGTGATGGAGCTTAAGAAAGATGGAAGCCACAGCACAGGTGATGGGATTCCTGATCGCTTCTCTGGATCCAGCTCTGGTGCTGATCGCTACCTTAGCATTTCCAACATCCAGCCTGAAGATGAAGCAATATACATCTGTGGTGTGGGTGATACAATTAAGGAACAATTTGTGTAATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GATGCTACACTAAGTC-1_contig_7": "GAGCATATGATCAGTGTCCTCTCCAAAGTCCTTGAACATAGACTCTAACCATGGAATGGACCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGAGCTGAGCTGATGAAGCCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTACTGGCTACACATTCAGTAGCTACTGGATAGAGTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGAGATTTTACCTGGAAGTGGTAGTACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACATTCACTGCAGATACATCCTCCAACACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCCGTCTATTACTGTGCAAGATTCCCCTATGGTAACTACGGGGCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ACGCCAGCACCCTATC-1_contig_1": "CATTACAAAGGTTTCTTATATGGGGAAAAGTGATGTTCTTATAAGACTAGCACTAATTAGCCAGAGACCAAAATCCAAATACACAATGGACTTTCGGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGTCACAGTGTCCAGAGGAGAAATTGTGCTCACTCAGTCTCCAGCCATCACAGCTGCATCTCTGGGGCAAAAGGTCACCATCACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAGGCACCTCCCCCAAACCATGGATTTATGAAATATCCAAACTGGCTTCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCATTTATTACTGCCAGCAGTGGAATTATCCTCTTATCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CAGCTGGGTTCGGCAC-1_contig_1": "TGGGGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTCCTCACACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "AGAGCGAAGGTGCTTT-1_contig_4": "GGGGATAAACGACATATGACCAATATCCTCTCCACATTCCCTTAAGACACTGACTCTAAACATGGGATGGATCTGGATCTTTCTCTTCCTCCTGTCAGGAACTGCAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTAGTGAAGACTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGTTACTACATGCACTGGGTCAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGATATATTAGTTGTTACAATGGTGCTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTTACTGTAGACACATCCTCCAGCACAGCCTACATGCAGTTCAACAGCCTGACATCTGAAGACTCTGCGGTCTATTACTGTGCAAGAAGATGGTTACTAGGTTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "AACTGGTAGGACTGGT-1_contig_7": "GCATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGAGATGGTTACTACGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CGAGCCACAGCAGTTT-1_contig_3": "GACACCATGTGCTGGAGTCAGCCTCACACTGATCACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGATGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGTATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTAATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATGCTGCAACAAACTTAGCAGATGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTATTCCCTCAAGATCAACAGCCTGCAGTCTGAAGATTTTGGGAGTTATTACTGTCAACATTTTTGGGGTACTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGAC",
        "CTCGTACAGACACGAC-1_contig_1": "ATGGCAGTTCCTATCTCTCTCACTGGAGGCTGATTTTTGAAGAAAGGGGTTGTAGCCTAAAAGATGATGGTGTTAAGTCTTCTGTACCTGTTGACAGCCCTTCCGGGTATCCTGTCAGAGGTGCAGCTTCAGGAGTCAGGACCTAGCCTCGTGAAACCTTCTCAGACTCTGTCCCTCACCTGTTCTGTCACTGGCGACTCCATCACCAGTGGTTACTGGAACTGGATCCGGAAATTCCCAGGGAATAAACTTGAGTACATGGGGTACATAAGCTACAGTGGTAGCACTTACTACAATCCATCTCTCAAAAGTCGAATCTCCATCACTCGAGACACATCCAAGAACCAGTACTACCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGATACTATGGTAACTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "TTTGTCATCGCTTGTC-1_contig_1": "TGGGGGAAATACATCAGATCAGCATGGGCATCAAGATGGAGTCACAGACTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGATGGAGACATTGTGATGACCCAGTCTCAAAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCGTCACCTGCAAGGCCAGTCAGAATGTGGGTACTAATGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAAGCACTGATTTACTCGGCATCCTACCGGTACAGTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAATGTGCAGTCTGAAGACTTGGCAGAGTATTTCTGTCAGCAATATAACAGCTATCCTCTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TACGGGCTCGTCGTTC-1_contig_8": "GCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGT",
        "GAGGTGATCCGTTGTC-1_contig_3": "GCACTTACTGTTTTCTCTTTCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CTGGTCTTCCGCGGTA-1_contig_1": "AGATGGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAACTACACATGTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GGAACTTGTAAACCTC-1_contig_2": "ATATTGCTCCCTTGGGAGGATCTCTCCTGCAGGTCCAAGACAAAAGCTAGCCTCCATTTCTGTCTTGACTACTCAAGACTTTTTGTATCAAGTTCTCAGAATGAGGTGCCTAGCTGAGTTCCTGGGGCTGCTTGTGCTCTGGATCCCTGGAGCCATTGGGGATATTGTGATGACTCAGGCTGCACCCTCTGTACCTGTCACTCCTGGAGAGTCAGTATCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTGCATAGTAATGGCAACACTTACTTGTATTGGTTCCTGCAGAGGCCAGGCCAGTCTCCTCAGCTCCTGATATATCGGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTGGCAGTGGGTCAGGAACTGCTTTCACACTGAGAATCAGTAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTATGCAACATCTAGAATATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAA",
        "AGCTCCTTCTGGCGTG-1_contig_2": "GGACAGAGGAGGCCGGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACGGACCCCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGTCCTAATTTTAAAAGGTGTCCAGTGTGAAGTGAAGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGGCATGTCTTGGGTTCGCCAGACTCCGGAGAAGAGGCTGGAGTGGGTCGCAACCATTAGTGGTGGTGGTAGTTACACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GGTATTGAGCTTTGGT-1_contig_1": "TGGGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TGAGCATTCGGATGGA-1_contig_7": "GAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "AGCAGCCCATCGGACC-1_contig_8": "GCCCATCCTCTTCTCATAGAGCCTCCATCAGACCATGGCTGTCCTGGCACTGCTCCTCTGCCTGGTGACATTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACTGTCTCAGGGTTCTCATTAACCAGCTATGGTGTAAGCTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGGTGACGGGAGCACAAATTATCATTCAGCTCTCATATCCAGACTGAGCATCAGCAAGGATAACTCCAAGAGCCAAGTTTTCTTAAAACTGAACAGTCTGCAAACTGATGACACAGCCACGTACTACTGTGCCAACTATGATGGTTACTACGTGAAGGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GGTGCGTTCTCGATGA-1_contig_7": "GGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTAC",
        "ACCGTAAGTGTAACGG-1_contig_1": "GGGGAGCTGCCAGGAGCCTAAGAAGCATCCTCTCATCTAGTTCTCAGAGATGGAGACAGACACAATCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGCTCCACTGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAAGGCCAGCCAAAGTGTTGATTATGATGGTGATAGTTATATGAACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAATCTAGAATCTGGGATCCCAGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CCTTTCTTCCGTTGCT-1_contig_5": "TGGGGACAGACGCACAACCCTGGACTCACAAGTCTTTCTCTTCAGTGACAAACACAGAAATAGAACATTCACCATGTACTTGGGACTGAACTGTGTATTCATAGTTTTTCTCTTAAAAGGTGTCCAGAGTGAAGTGAAGCTTGAGGAGTCTGGAGGAGGCTTGGTGCAACCTGGAGGATCCATGAAACTCTCCTGTGTTGCCTCTGGATTCACTTTCAGTAACTACTGGATGAACTGGGTCCGCCAGTCTCCAGAGAAGGGGCTTGAGTGGGTTGCTGAAATTAGATTGAAATCTAATAATTATGCAACACATTATGCGGAGTCTGTGAAAGGGAGGTTCACCATCTCAAGAGATGATTCCAAAAGTAGTGTCTACCTGCAAATGAACAACTTAAGAGCTGAAGACACTGGCATTTATTACTGTACCAGGGACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GACTACATCTGCGTAA-1_contig_2": "GGGAACCTATGATCAGTATCCTCTCAACAGTCCCTGAACACACTGACTCTCACCATGGAATGGAGCGGAGTCTTTATCTTTCTCCTGTCAGTAACTGCAGGTGTTCACTCCCAGGTCCAGCTGCAGCAGTCTGGAGCTGAGCTGGTAAGGCCTGGGACTTCAGTGAAGGTGTCCTGCAAGGCTTCTGGATACGCCTTCACTAATTACTTGATAGAGTGGGTAAAGCAGAGGCCTGGACAGGGCCTTGAGTGGATTGGAGTGATTAATCCTGGAAGTGGTGGTACTAACTACAATGAGAAGTTCAAGGGCAAGGCAACACTGACTGCAGACAAATCCTCCAGCACTGCCTACATGCAGCTCAGCAGCCTGACATCTGATGACTCTGCGGTCTATTTCTGTGCAAGAGAGGGTGGTTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "ACGATACCACGAAAGC-1_contig_1": "TGGGGACTTTTGACTCACCATATCAAGTTCGCAGAATGAGGTTCTCTGCTCAGCTTCTGGGGCTGCTTGTGCTCTGGATCCCTGGATCCACTGCAGATATTGTGATGACGCAGGCTGCATTCTCCAATCCAGTCACTCTTGGAACATCAGCTTCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTACATAGTAATGGCATCACTTATTTGTATTGGTATCTGCAGAAGCCAGGCCAGTCTCCTCAGCTCCTGATTTATCAGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTAGCAGTGGGTCAGGAACTGATTTCACACTGAGAATCAGCAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTGCTCAAAATCTAGAACTGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CTTTGCGGTCATCGGC-1_contig_2": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "GGGACCTCAGCCACCA-1_contig_1": "TGGGGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TAGACCATCGCGCCAA-1_contig_2": "GGGGAAAGTACTTATGATAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGGAAAAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCTAGGGGAACGGGTCACCATGACCTGCACTGCCAGCTCAAGTGTAAGTTCCAGTTACTTGCACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAAACTCTGGATTTATAGCACATCCAACCTGGCTTCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCACCAGTATCATCGTTCCCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CAGCTGGCACATCTTT-1_contig_1": "TGGCGGTTTTCTTATATGGGGCCATCATCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGCATTACTCTTCTGCCTGGTAACATTCCCAAGCTGTATCCTTTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCGTCTCAGGGTTCTCATTAACCGGCTATGGTGTAAACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAATGATATGGGGTGATGGAAGCACAGACTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCAGGTACTACTGTGCCAGGAGCATTACTACGGCCCCATATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GCGCGATCACAACGTT-1_contig_2": "TGGGGACACATACCAGCAAGGGAGTGACCAGTTTGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGATTGGCTGTGGAACTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACATGGCTACATATTTCTGTGCAAGAAGGGATGGTTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ACGCCAGGTGTCAATC-1_contig_5": "GTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCACGGGACAAACTCTTCTCCACAGTGTGACCTTCATGAGTGACCTGGCAGCTGTAACTGCTATGCCTTTCCCATGCTCTTGCTG",
        "CCCAATCGTGTGTGCC-1_contig_4": "TGGGGCTGAAATGCATCAGACCAGCATGGGCATCAAGATGGAATCACAGACTCTGGTCTTCATATCCATACTGCTCTGGTTATATGGTGCTGATGGGAACATTGTAATGACCCAATCTCCCAAATCCATGTCCATGTCAGTAGGAGAGAGGGTCACCTTGAGCTGCAAGGCCAGTGAGAATGTGGGTACTTATGTATCCTGGTATCAACAGAAACCAGAGCAGTCTCCTAAACTGCTGATATACGGGGCATCCAACCGGTACACTGGGGTCCCCGATCGCTTCACAGGCAGTGGATCTGCAACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGACCTTGCAGATTATCACTGTGGACAGAGTTACAGCTATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCACC",
        "CCGGTAGTCACTCTTA-1_contig_4": "GACACATCTCTCTCATTAGAGGTTGATCTTTGAGGAAAACAGGGTGTTGCCTAAAGGATGAAAGTGTTGAGTCTGTTGTACCTGTTGACAGCCATTCCTGGTATCCTGTCTGATGTACAGCTTCAGGAGTCAGGACCTGGCCTCGTGAAACCTTCTCAGTCTCTGTCTCTCACCTGCTCTGTCACTGGCTACTCCATCACCAGTAATTATTACTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATAAGCTACGACGGTAGCAATAACTACAACCCATCTCTCAAAAATCGAATCTCCATCACTCGTGACACATCTAAGAACCAGTTTTTCCTGAAGTTGAATTCTGTGACTACTGAGGACACAGCTACATATTACTGTGCAAGAGTGTATGGTAACTACCTCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGCCAAAACAACACCCCCATCAGTCTATCCACTGGCCCCTGGGTGTGGAGATACAACTGGTTCCTCCGTGACTCTGGGATGCCTGGTCAAGGGCTACTTCCCTGAGTCAGTGACTGTGACTTGGAACTCTGGATCCCTG",
        "TTGCCGTTCATCGATG-1_contig_4": "GATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCGGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGAATTGAAGTCTCAACTATGAAAATCAGCAGTTCTCTGTCAGAGAAGCCCAAGCGCTTCCACGCATGCTTGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCAC",
        "ACGAGCCGTCATATCG-1_contig_3": "GGGGAGCTGCCAGGAGCCTAAGAAGCATCCTCTCATCTAGTTCTCAGAGATGGAGACAGACACAATCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGCTCCACTGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAAGGCCAGCCAAAGTGTTGATTATGATGGTGATAGTTATATGAACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAATCTAGAATCTGGGATCCCAGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCA",
        "CGAACATAGTCATGCT-1_contig_3": "TGGGGACTGCTTCTCAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGT",
        "CTGCGGAAGCCAGTTT-1_contig_1": "GACAAATATCTCACAGCTGTTTTAAAGCGATGTACTTAGGAGAAGAGCAGTAATTAGCCAGGGAACAAAATTCAAATACACAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTATCTCAGTTGTAATGTCCAGAGGAGAAAATGTGCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCTAGGGGAGAAGGTCACCATGAGCTGCAGGGCCAGCTCAAGTGTAAATTACATGTACTGGTACCAGCAGAAGTCAGATGCCTCCCCCAAACTATGGATTTATTACACATCCAACCTGGCTCCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGAACTCTTATTCTCTCACAATCAGCAGCATGGAGGGTGAAGATGCTGCCACTTATTACTGCCAGCAGTTTACTAGTTCCCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AGCATACGTAGCGTAG-1_contig_5": "GACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGATGGCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AAGGAGCGTAAAGTCA-1_contig_3": "GGTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "CGGAGCTAGGCATTGG-1_contig_1": "GGCTCCATTTCTGTCTTGACTACTCAAGACTTTTTGTATCAAGTTCTCAGAATGAGGTGCCTAGCTGAGTTCCTGGGGCTGCTTGTGCTCTGGATCCCTGGAGCCATTGGGGATATTGTGATGACTCAGGCTGCACCCTCTGTACCTGTCACTCCTGGAGAGTCAGTATCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTGCATAGTAATGGCAACACTTACTTGTATTGGTTCCTGCAGAGGCCAGGCCAGTCTCCTCAGCTCCTGATATATCGGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTGGCAGTGGGTCAGGAACTGCTTTCACACTGAGAATCAGTAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTATGCAACATCTAGAATATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCT",
        "CCAGCGATCCTAGAAC-1_contig_2": "GAAACCGCATATGATCAGTATCCTCTTCACAGTCATTGAAAACACTGACTCTAATCATGGAATGTAACTGGATACTTCCTTTTATTCTGTCGGTAATTTCAGGGGTCTACTCAGAGGTTCAGCTCCAGCAGTCTGGGACTGTGCTGGCAAGGCCTGGGGCTTCCGTGAAGATGTCCTGCAAGGCTTCTGGCTACAGCTTTACCAGCTACTGGATGCACTGGGTAAAACAGAGGCCTGGACAGGGTCTAGAATGGATTGGTGCTATTTATCCTGGAAATAGTGATACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCAAACTGACTGCAGTCACATCCGCCAGCACTGCCTACATGGAGCTCAGCAGCCTGACAAATGAGGACTCTGCGGTCTATTACTGTACAAGATCAGGGTACTATGATTACGAAAACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TTTGTCACAAGCGTAG-1_contig_2": "TTATGGGGATTGTCTTTACAATCAGGACTCAGCATGGACATGATGGTCCTTGCTCAGTTTCTTGCATTCTTGTTGCTTTGGTTTCCAGGTGCAAGATGTGACATCCTGATGACCCAATCTCCATCCTCCATGTCTGTATCTCTGGGAGACACAGTCAGCATCACTTGCCATGCAAGTCAGGGCATTAGCAGTAATATAGGGTGGTTGCAGCAGAAACCAGGGAAATCATTTAAGGGCCTGATCTATCATGGAACCAACTTGGAAGATGGAGTTCCATCAAGGTTCAGTGGCAGTGGATCTGGAGCAGATTATTCTCTCACCATCAGCAGCCTGGAATCTGAAGATTTTGCAGACTATTACTGTGTACAGTATGCTCAGTTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCATATAAG",
        "CTACGTCGTTACAGAA-1_contig_7": "GCATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGAGATAGGGCTACTATGATTACTTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TCTGAGAGTCACTGGC-1_contig_1": "CAAACTCCAAATTTCTTATATGGGGGAAATACATCAGATCAGCATGGGCATCAAGATGGAGTCACAGACTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGATGGAGACATTGTGATGACCCAGTCTCAAAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCGTCACCTGCAAGGCCAGTCAGAATGTGGGTACTAATGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAAGCACTGATTTACTCGGCATCCTACCGGTACAGTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAATGTGCAGTCTGAAGACTTGGCAGAGTATTTCTGTCAGCAATATAACAGCTATTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCAGATCGGAAGAGC",
        "GAACATCGTGACCAAG-1_contig_1": "GATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGTGCTGCTCTTCTGCCTGGTGACATTCCCAAGCTGTGTCCTATCCCAGGTGCAGCTGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATCACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGTGGTGGAAGCACAGACTATAATGCTGCTTTCATATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTGATGACACAGCCATATACTACTGTGCCAGGCCCTATGATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AGCCTAACACTGTCGG-1_contig_7": "GGAAAAATACGATCAGCGTCCTCTCCACAGACACTGAAAACTCTGACTCAACATGGAAAGGCACTGGATCTTTCTCTTCCTGTTGTCAGTAACTGCAGGTGTCCACTCCCAGGTCCAGCTGCAGCAGTCTGCAGCTGAACTGGCAAGACCTGGGGCCTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACCTTTACTAGCTACACGATGCACTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATTGGATACATTAATCCTAGCAGTGGATATACTGAGTACAATCAGAAGTTCAAGGACAAGACCACATTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAACTGAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGTAACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GCATACAAGGCAGTCA-1_contig_6": "GGGACTGTTTTCTCTTTCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAAC",
        "ACCGTAAGTGTAACGG-1_contig_3": "GGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "CAGCTGGGTTCGGCAC-1_contig_2": "GGGGAGCTGCCAGGAGCCTAAGAAGCATTCTCTCTTCCAGTTCTCAGAGATGGAGAAAGACACACTCCTGCTATGGGTCCTGCTTCTCTGGGTTCCAGGTTCCACAGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAGAGCCAGCGAAAGTGTTGATAATTATGGCATTAGTTTTATGAACTGGTTCCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAACCAAGGATCCGGGGTCCCTGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCAGCCTCAACATCCATCCTATGGAGGAGGATGATACTGCAATGTATTTCTGTCAGCAAAGTAAGGAGGTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GGGTTGCGTACAGCAG-1_contig_2": "GGTTTCAGTGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "ATTTCTGTCCCGGATG-1_contig_8": "GGGGAACCTATGATCAGTGTCCTCTCTACACAGTCCCTGACAACACTGACTCTAACCATGGGATGGAGCCGGATCTTTCTCTTCCTCCTGTCAATAATTGCAGGTGTCCATTGCCAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAGGATATCCTGCAAGGCTTCTGGCTACACCTTCACAAGCTACTATATACACTGGGTGAAGCAGAGGCCTGGACAGGGACTTGAGTGGATTGGATGGATTTATCCTGGAAATGTTAATACTAAGTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACCTCTGAGGACTCTGCGGTCTATTTCTGTGCAATCTACTATGGTAACTACGTGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "TACCTTACATACGCTA-1_contig_5": "GGGGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TACTCGCCATCTATGG-1_contig_2": "GACACATAACAACAAGAGAGTGTCCGGTTAGTCTCAAGGAAGACTGAGACACAGTCTTAGATATCATGGAATGGCTGTGGAACTTGCTATTTCTCATGGCAGCAGCTCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAGGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAACTGCTGGAATGCAGTGGGTGCAAAAGATGCCAGGAAAGGGTTTGAAGTGGATTGGCTGGATAAACACCCACTCTGGAGTGCCAATATATGCAGAAGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAAGGTCTGCCAGCGCTGCATATTTACAGATAAGCAACCTCAAAAATGAGGACACGGCTACGTATTTCTGTGCGAGAGAGCCCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GCGCAGTCAGCGAACA-1_contig_3": "TGGGGAGTTCTCTCTACAGTTACTGAGCACACAGGACCTCACCATGGGATGGAGCTATATCATCCTCTTTTTGGTAGCAACAGCTACAGATGTCCACTCCCAGGTCCAACTGCAGCAGCCTGGGGCTGAACTGGTGAAGCCTGGGGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTGGATGCACTGGGTGAAGCAGAGGCCTGGACAAGGCCTTGAGTGGATTGGAGAGATTAATCCTAGCAACGGTCGTACTAACTACAATGAGAAGTTCAAGAGCAAGGCCACACTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGAGAGGGAAGGTACGACTTCCTGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "GAAGCAGCACCAGTTA-1_contig_8": "GGGGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGGGCTGCTTCTCTGCCTGGTGACGTTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACTGTCTCAGGGTTCTCATTAACCGACTATGGTGTAAGCTGGATTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGGTGGTGGAAGCACATACTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAAACATGATGGTAACTACCCCCCTTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CCTTACGAGGAGTTTA-1_contig_3": "GAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCC",
        "GCGAGAAAGACACTAA-1_contig_1": "TGGGGACAGTTGCTTTAAAGAGATGTACTTATAGGAAGCACAATAATTAGCCAGAGACTGAGATGGAAAACAAAATGGATTTTCAGATGCAGATTATCAGCTTGCTGCTAATCAGTGTCACAGTCATAGTGTCTAATGGAGAAATTGTGCTCACCCAGTCTCCAACCACCATGGCTGCATCTCCCGGGGAGAAGATCACTATCACCTGCAGTGCCAGCTCAAGTATAAGTTCCAATTACTTGCATTGGTATCAGCAGAAGCCAGGATTCTCCCCTAAACTCTTGATTTATAGGACATCCAATCTGGCTTCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATTGGCACCATGGAGGCTGAAGATGTTGCCACTTACTACTGCCAGCAGGGTAGTAGTATACCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAA",
        "CTTACCGGTCTAGTCA-1_contig_2": "TGGGGAGAAATACATCAGGCAGGCAAGGGCATCAAGATGAAGTCACAGACCCAGGTCTTCGTATTTCTACTGCTCTGTGTGTCTGGTGCTCATGGGAGTATTGTGATGACCCAGACTCCCAAATTCCTGCTTGTATCAGCAGGAGACAGGGTTACCATAACCTGCAAGGCCAGTCAGAGTGTGAGTAATGATGTAGCTTGGTACCAACAGAAGCCAGGGCAGTCTCCTAAACTGCTGATATACTATGCATCCAATCGCTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATATGGGACGGATTTCACTTTCACCATCAGCACTGTGCAGGCTGAAGACCTGGCAGTTTATTTCTGTCAGCAGGATTATAGCTCTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGA",
        "GTCAAGTAGGCATTGG-1_contig_3": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TGACAACAGGCGTACA-1_contig_3": "AGATCACAGTAGTCTCTACAGTCACAGGAGTACACAGGGCATTGCCATGGGTTGGAGCTGTATCATCTTCTTTCTGGTAGCAACAGCTACAGGTGTGCACTCCCAGGTCCAGCTGCAGCAGTCTGGGCCTGAGGTGGTGAGGCCTGGGGTCTCAGTGAAGATTTCCTGCAAGGGTTCCGGCTACACATTCACTGATTATGCTATGCACTGGGTGAAGCAGAGTCATGCAAAGAGTCTAGAGTGGATTGGAGTTATTAGTACTTACAATGGTAATACAAACTACAACCAGAAGTTTAAGGGCAAGGCCACAATGACTGTAGACAAATCCTCCAGCACAGCCTATATGGAACTTGCCAGATTGACATCTGAGGATTCTGCCATCTATTACTGTGCAGTGCAGACTGATAGGTACGACGTAGGGGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CTCGGAGGTATTAGCC-1_contig_1": "TGGGGACTTATGAGAATAGTAGTAATTAGCTAGGGACCAAAGTTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATACTGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAGACTCCTGATTTATGACACATCCAACCTGGCTTCTGGAGTCCCTGTTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTTACCCACCCACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GCTGGGTTCGTGGACC-1_contig_10": "GACAGTCACTGAAAACATTGACTAATCATGGAAAGGAACTGGATACTTCCTTTTATTATGTCAGTAACTGCAGGTGTCTACTCACAGGTTCAGCTCCAGCAGTCTGGGCCTGAGCTGGCAAGGCCTTGGGCTTCAGTGAAGATATCCTGCCAGGCTTTCTACACCTTTTCCAGAAGGGTGTACTTTGCCATTAGGGATACCAACTACTGGATGCAGTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATCGGGGCTATTTATCCTGGAAATGGTGATACTAGTTACAATCAGAAGTTCAAGGGCAAGGCCACATTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCATACTACTATAGGTACGACGAGAACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "ATCCGAAAGGACAGCT-1_contig_1": "GGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GAAATGAAGGTGCAAC-1_contig_3": "GGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "ATCGAGTGTCATATCG-1_contig_2": "TGGGATCAGACAGGCAGGGGAAGCAAGATGGATTCACAGGCCCAGGTTCTTATGTTACTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTAGCTGTGTCAGTTGGAGAGAAGGTTACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GCTTGAAGTAACGACG-1_contig_1": "GAACATATGAACACTGTTTTCTCTACAGTCACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTCATCTTCTTCCTGATGGCAGTGGTTATAGGAATCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCAGAGCTTGTGAGGTCAGGGGCCTCAGTCAAGTTGTCCTGCACAGCTTCTGGCTTCAACATTAAAGACTACTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGATGGATTGATCCTGAGAATGGTGATACTGAATATGCCCCGAAGTTCCAGGGCAAGGCCACTATGACTGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTAATGCCTATTACTACGGTAGTAGCTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GCAATCAGTGGCGAAT-1_contig_4": "CTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAATCTCTACTATGATTACGACGGGGGACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGCCAAAACAACACCCCCATCAGTCTATCCACTGGCCCCTGGGTGTGGAGATACAACTGGTTCCTCCGTGACTCTGGGATGCCTGGTCAAGGG",
        "TTGGCAATCACATGCA-1_contig_5": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CGGAGCTCACACGCTG-1_contig_5": "GCTCTCCAACTGCTCTCTCCTGAAGTGCCTCAGATGTTTCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTGA",
        "AGAGCTTCAATCTACG-1_contig_7": "TGGGGAATCCTCTTCTCATAGAGCCTCCATCAGAACATGGCTGTCCTGGCACTACTCCTCTGCCTGGTGGCTTTCCCAAGCTGTACCCTGTCCCAGGTGCAACTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATTACCTGCACTGTCTCTGGGTTCTCATTAACCAGCTATGATATAAGCTGGATTCGCCAGCCACCAGGAAAGGGTCTGGAGTGGCTTGGAGTAATATGGACTGGTGGAGGCACAAATTATAATTCAGCTTTCATGTCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATATATTACTGTGTAAGGAGGGGTTATGGTTACGACGGGTCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CCTCAGTGTGAAATCA-1_contig_1": "GGGGTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GACTACATCATCATTC-1_contig_2": "GGAATACATCAGACCAGCATGGGCATCAAGATGGAGTTTCAGACCCAGGTCTTTGTATTCGTGTTGCTCTGGTTGTCTGGTGTTGATGGAGACATTGTGATGACCCAGTCTCAAAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGAATGTTCGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGGCAGTCTCCTAAAGCACTGATTTACTTGGCATCCAACCGGCACACTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATTAGCAATGTGCAATCTGAAGACCTGGCAGATTATTTCTGTCTGCAACATTGGAATTATTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AATCCAGGTCAACATC-1_contig_1": "TCTGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGACTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TTCCCAGCAGGATCGA-1_contig_1": "GGTATAAACCAGGCCTTTGGATTGTGACTGAACAGCATCAGACAGGCAGGGGAGCAATATGGTATCAGAGACCCATGTCCTCATATTTTTGCTGCTATGGGTGTCTGGTGCTTGTGGGGAAATTGTGCTGACTGTGTCAGCAGGAGAGAGGGTCACTATCAGCTGCAAATCCAATCAGAATCTTTTATGGAGTGGAAACCAAAGGTACTGTTTGGTCTGGCACCAGTGGAAACCAGGGCAAACTCCTACACCGTTGATCACCTGGACATCTGATAGGTACTCTGGAGTCCCTGATCGTTTCATAGGCAGTGGATCTGTGACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGATGTGGCAGTTTATTTCTGTCAGCAGCATTTACACATTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCC",
        "CTGCCTATCAAACAAG-1_contig_1": "GAAAGCAAAGTATTTATGAGAAGAGCAGTAATTAGCTAGGGACCAAAATTCAAAGAGAAAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCTAGGGGAGGAGATCACCCTAACCTGCAGTGCCAGCTCGAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAGGCACTTCTCCCAAACTCTTGATTTATAGCACATCCAACCTGGCTTCTGGAGTCCCTTCTCGCTTCAGTGGCAGTGGGTCTGGGACCTTTTATTCTCTCACAATCAGCAGTGTGGAGGCTGAAGATGCTGCCGATTATTACTGCCATCAGTGGAGTAGTTATCCCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "AAAGTAGCAAAGTGCG-1_contig_4": "GAGTAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GGACAGAAGAACAACT-1_contig_4": "GGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "ATCATCTAGTGGGCTA-1_contig_3": "GGGCTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TCCACACAGCCTCGTG-1_contig_2": "GGACATACCAGCAAGGGAGTGACCAGTTAGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGTTATACCTTCACAGACTATTCAATGCACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACTGAGACTGGTGAGCCAACATATGCAGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCTTGGTATGATGGTTACCGCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CTAACTTGTTCCATGA-1_contig_4": "GAAGGTCACTGTTCTCACTACAGTTACTGAGCACACAGACCTCACCATGGGATGGAGCTCTATCATCCTCTTCTTGGTAGCAACAGCCTCAGGTGTCCACTCCCAGGTCCAACTGCAGCAACCTGGGTCTGAGCTGGTGAGGCCTGGAGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACATTCACCAGCTACTGGATGCACTGGGTGAAGCAGAGGCCTGGACAAGGCCTTGAGTGGATTGGAAATATTTATCCTGGTAGTGGTAGTACTAACTACGATGAGAAGTTCAAGAGCAAGGCCACACTGACTGTAGACACATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTACAAGAGGACTATGGTCATCGGGGGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CATCAGAAGCACAGGT-1_contig_3": "GCTGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGG",
        "CAGCCGATCTACTATC-1_contig_10": "GAGAACAGGCCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "AACTCAGCACGGTAGA-1_contig_1": "TAACAAATCAGAGGTTGTTACGCTTTGGTTGAGTTACTAAAATAAGCCTATTCTGCAGCTGTGCCCAGCACTGTCCAGATGAAGCTGATTTGCATGTGCTGAGATCATATTCTACTGCCCCAGAGATTTAATAATCTGATCATACACACTCCAACAGTCATTCTTGGTCAGGAGACGTTGTAGAAATGAGACCGTCTATTCAGTTCCTGGGGCTCTTGTTGTTCTGGCTTCATGGTGCTCAGTGTGACATCCAGATGACACAGTCTCCATCCTCACTGTCTGCATCTCTGGGAGGCAAAGTCACCATCACTTGCAAGGCAAGCCAAGACATTAACAAGTATATAGCTTGGTACCAACACAAGCCTGGAAAAGGTCCTAGGCTGCTCATACATTACACATCTACATTACAGCCAGGCATCCCATCAAGGTTCAGTGGAAGTGGGTCTGGGAGAGATTATTCCTTCAGCATCAGCAACCTGGAGCCTGAAGATATTGCAACTTATTATTGTCTACAGTATGATAATCTTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "CCATTCGGTACCTACA-1_contig_1": "TGGGGCATTGTCATTGCAGTCAGGACTCAGCATGGACATGAGGGCTCCTGCACAGATTTTTGGCTTCTTGTTGCTCTTGTTTCCAGGTACCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCTTATCTGCCTCTCTGGGAGAAAGAGTCAGTCTCACTTGTCGGGCAAGTCAGGACATTGGTAGTAGCTTAAACTGGCTTCAGCAGGAACCAGATGGAACTATTAAACGCCTGATCTACGCCACATCCAGTTTAGATTCTGGTGTCCCCAAAAGGTTCAGTGGCAGTAGGTCTGGGTCAGATTATTCTCTCACCATCAGCAGCCTTGAGTCTGAAGATTTTGTAGACTATTACTGTCTACAATATGCTAGTTCTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "CGTTCTGAGCACAGGT-1_contig_3": "GGACCCTGCAGCTCTGACAGAGGAGGCAGGTCCTAGATTTGAGTTCCTCACATTCAGTGATCAGCACTGAACACGGAACCCTCACCATGAACTTTGGGCTGAGCTTGATTTTCCTTGTCCTAATTTTAAAAGGTGTCCAGTGTGAAGTGATGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATACCATGTCTTGGGTTCGCCAGACTCCGGAGAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTGGTAACACCTACTATCCAGACAGTGTGAAGGGTCGATTCACCATCTCCAGAGACAATGCCAAGAACAACCTGTACCTGCAAATGAGCAGTCTGAGGTCTGAGGACACGGCCTTGTATTACTGTGCAAGACGGAAGGTACGACGGTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "TTGGAACTCCACGAAT-1_contig_1": "TGGGGGAAATGCATCGCACCAGCATGGGCATCAAGATGGAGTCACAGATTCAGGCATTTGTATTCGTGTTTCTCTGGTTGTCTGGTGTTGACGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGAGTACTGCTGTAGCCTGGTATCAACAAAAACCAGGGCAATCTCCTAAACTACTGATTTACTGGGCATCCACCCGGCACACTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTATACTCTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCACTTTATTACTGTCAGCAACATTATAGCACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "CCGTTCAAGAGCTGCA-1_contig_8": "GGACATATGACCAATATCCTCTCCACATTCCCTTAAGACACTGACTCTAAACATGGGATGGATCTGGATCTTTCTCTTCCTCCTGTCAGGAACTGCAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTAGTGAAGACTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGTTACTACATGCACTGGGTCAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGATATATTAGTTGTTACAATGGTGCTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTTACTGTAGACACATCCTCCAGCACAGCCTACATGCAGTTCAACAGCCTGACATCTGAAGACTCTGCGGTCTATTACTGTGCAAGATACTACTACGGTAGTAGCTACGATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CTCGGAGCACCAGTTA-1_contig_5": "AGTTGGTTTATAGCAAAGTACTTATGAGAATAGTAGTAATTAGCTAGGGACCAAAGTTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATACTGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAGACTCCTGATTTATGACACATCCAACCTGGCTTCTGGAGTCCCTGTTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTTACCCACTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "TAGTGGTTCAGCGATT-1_contig_4": "GGGACAGTTCTCTCTACAGTTACTGAGCACACAGGACCTCACCATGGGATGGAGCTATATCATCCTCTTTTTGGTAGCAACAGCTACAGATGTCCACTCCCAGGTCCAACTGCAGCAGCCTGGGGCTGAACTGGTGAAGCCTGGGGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTGGATGCACTGGGTGAAGCAGAGGCCTGGACAAGGCCTTGAGTGGATTGGAGAGATTAATCCTAGCAACGGTCGTACTAACTACAATGAGAAGTTCAAGAGCAAGGCCACACTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGAGATGATGGTTACCACTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGCCAAAACAACACCCCCATCAGTCTATCCACTGGCCCCTGGGTGTGGAGATACAACTGGTTCCTCCGTGACTCTGGGATGCCTGGTCAAGGGCTACTTCCCTGAGTCAGTGACTGTGACTTGGAACTCTGGATCCCTGTC",
        "ATTACTCGTACTCAAC-1_contig_8": "GCAGAAAACAGCATATGATCAGTGTCCTCTCCAAAGTCCTTGAACATAGACTCTAACCATGGAATGGACCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGAGCTGAGCTGATGAAGCCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTACTGGCTACACATTCAGTAGCTACTGGATAGAGTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGAGATTTTACCTGGAAGTGGTAGTACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACATTCACTGCAGATACATCCTCCAACACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCCGTCTATTACTGTGCAAGAGGGATTCTCCATTACTACGGTAGTAGCTACGATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AGCTCCTTCCGCTGTT-1_contig_2": "TGGGGACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGCATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATAACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTTCCAGCAGAAGCCAGGCACTTCTCCCAAACTCTGGATTTATAGCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGATCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAAAGGAGTAGTTACCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCAATGATACGGCGACCAC",
        "TAGTTGGTCTGGTATG-1_contig_3": "GAGCGTATGAACCTAGCCCTGATTTCCCCAGCCTTCAGTTCCCAGATTCAGTGATCAGCCTTGAACACAGACCTGTCACCATGAAGTTGTGGCTGAACTGGATTTTCCTTGTAACACTTTTAAATGGTATCCAGTGTGAGGTGAAGCTGGTGGAGTCTGGAGGAGGCTTGGTACAGCCTGGGGGTTCTCTGAGACTCTCCTGTGCAACTTCTGGGTTCACCTTCACTGATTACTACATGAGCTGGGTCCGCCAGCCTCCAGGAAAGGCACTTGAGTGGTTGGGTTTTATTAGAAACAAAGCTAATGGTTACACAACAGAGTACAGTGCATCTGTGAAGGGTCGGTTCACCATCTCCAGAGATAATTCCCAAAGCATCCTCTATCTTCAAATGAACACCCTGAGAGCTGAGGACAGTGCCACTTATTACTGTGCAAGAGGCTCCTACTATAGGTACGACGTGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTG",
        "GATTCAGCACTCTGTC-1_contig_2": "GGGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGCCC",
        "ACGTCAACACCACCAG-1_contig_4": "GAAATACGATCAGCATCCTCTCCACAGACACTGAAAACTCTGACTCACAATGGAAAGGCACTGGATCTTTCTCTTCCTGTTTTCAGTAACTGCAGGTGTCCACTCCCAGGTCCAGCTTCAGCAGTCTGGGGCTGAACTGGCAAAACCTGGGGCCTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACCTTTACTAGCTACTGGATGCACTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATTGGATACATTAATCCTAGCACTGGTTATACTGAGTACAATCAGAAGTTCAAGGACAAGGCCACATTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAACTGAGCAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGATTCCATGGTAACTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAAC",
        "CATCAAGTCTGAAAGA-1_contig_1": "GAGTTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGCCCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CGAGCACCAAATACAG-1_contig_4": "GGAGAAAAACATGAGATCACTGTTCTCTCTACAGTTACTGAGCACACAGGACCTCACCATGGGATGGAGTTGTATCATCCTCTTCTTGGTAGCAACAGCTACAGGTGTCCACTCCCAGGTGCAACTGCAGCAGCCTGGGGCTGAGCTGGTGAAGCCTGGGGCCTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACATTTACCAGTTACAATATGCACTGGGTAAAGCAGACACCTGGACAGGGCCTGGAATGGATTGGAGCTATTTATCCAGGAAATGGTGATACTTCCTACAATCAGAAGTTCAAAGGCAAGGCCACATTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGAGAGGGGGGCCCCTACTATGGTAACTCCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TCTATTGCAAGCCCAC-1_contig_2": "GGAGGTCACAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGC",
        "ACGAGCCTCTGCAAGT-1_contig_2": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACC",
        "AGCCTAAGTGTTTGTG-1_contig_3": "GGGGACTTGAATGGTTGATTCTTGTGTGACACCAGGAATTGGCATAATGTCTGAGTTGCCCAGGGGTGATTCTAGTCAGACTCTGGGGTTTTTGTCGGGTATAGAGGAAAAATCCACTATTGTGATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GATCGCGTCTATCCCG-1_contig_1": "AGCCTTTTTCTTATATGGGGGAAATACATCAGATCAGCATGGGCATCAAGATGGAGTCACAGACTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGATGGAGACATTGTGATGACCCAGTCTCAAAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCGTCACCTGCAAGGCCAGTCAGAATGTGGGTACTAATGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAAGCACTGATTTACTCGGCATCCTACCGGTACAGTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAATGTGCAGTCTGAAGACTTGGCAGAGTATTTCTGTCAGCAATATAACAGCTATCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAA",
        "GCCAAATGTGACCAAG-1_contig_3": "GGGCTATTTGCATACTTCATCTTCTGTAACCACAAATATCTCACAGTTCTTTTAAAGCAAAGTACTTGTGAGAATAGGAGTAATTAGCCAGGAACTAAAATTCAAAGACACAATGGATTTACAGGTGCAGATTATCAGCTTCCTGCTAATCAGTGTCACAGTCATAATGTCCAGAGGAGAAAATGTGCTCACCCAGTCTCCAGCAATAATGGCTGCCTCTCTGGGGCAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTCCAGTTACTTGCACTGGTACCAGCAGAAGTCAGGCGCTTCCCCCAAACCCTTGATTCATAGGACATCCAACCTGGCTTCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCGTGGAGGCTGAAGATGATGCAACTTATTACTGCCAGCAGTGGAGTGGTTACCCATTCACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CGTAGGCCACTTAACG-1_contig_2": "TACCAGCAAGCGAGTGACCAGTTAGTCTTAAGGCACCACTTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTATATATTTCTGTGCAAGAAATGGTAACTACCTCGCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGCCAAAACAACACCCCCATCAGTCTATCCACTGGCCCCTGGGTGTGGAGATACAACTGGTTCCTCCGTGACTCTGGGATGCCTGGTCAAGGG",
        "TTGGAACGTTTCGCTC-1_contig_4": "GGGAGAAATAGTGCAGAGAGTACATGGTGTTGAATCAACTCCATGGGGTACCTGCCTGACAAGAAGACAAGCCTATAAAAGGATTACCACCCACTGCTTCTCAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GACTGCGTCCTATTCA-1_contig_3": "GAGACAGATCTTTCCAGCAATTCTTATCACTCCTCCCTCAGCAGCCTCCATCCAGGCATTAAGAGAACAAGAGATGTTGAAATGTCAGCCTTGTGGCTGGCACAGACCCAGCATTCCATCCAGAGCATTCCTGAAGAGTTAGAAGCCCAGGAGGAAAAAGGCTGGACCTGCACTCAGGATGACCCCTGGACTCGGCCCTGCTGCCTGTGCCTCAAAGGTCCCTTACTCATGCAAAACCAGAGTCCTTACTCATCTGGGCTTTTTGTGAAGGGATCTACTACTGTGTTCATTACTAAGGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "AACCGCGCAAGGTTTC-1_contig_3": "GGGTAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GAGCAGATCAAAGTAG-1_contig_5": "TCTCATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGAGATTACTACGGTAGTTATTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "TGATTTCTCTTTACAC-1_contig_6": "GGGGAACCTATGATCAGTATCCTCTCCACAGTCCCTGAACACACTGACTCAAAACATGGAATGGAGTTGGGTCTTTCTCTTCCTCCTGTCATTAACTTCAGGTGTCCACAGCCACGTCCAGCTGCAGCAATCTGGACCTGAGCTGGTGAGGCCTGGGGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTATATCTTCATCACCTACTGGATGAACTGGGTGAAGCAGAGGCCTGGACAGGGCCTTGAGTGGATTGGACAGATTTTTCCTGCAAGTGGTAGTACTAACTACAATGAGATGTTCGAGGGCAAGGCCACATTGACTGTAGACACATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGAGGGTACTACGGTAGTAGCCTCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "CATCAGAAGCACAGGT-1_contig_4": "GACATAACAGCAAGAGAGTGTCCGGTTAGTCTCAAGGAAGACTGAGACACAGTCTTAGATATCATGGAATGGCTGTGGAACTTGCTATTTCTCATGGCAGCAGCTCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAGGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAACTGCTGGAATGCAGTGGGTGCAAAAGATGCCAGGAAAGGGTTTGAAGTGGATTGGCTGGATAAACACCCACTCTGGAGTGCCAAAATATGCAGAAGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCATATTTACAGATAAGCAACCTCAAAAATGAGGACACGGCTACGTATTTCTGTGCGAGATCTGATGGTTACTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ACTGAACAGGACGAAA-1_contig_3": "GGACAGTAATGACCCAGACAGAGAAGGCCAGACTCATAAAGCTTGCTGAGCAAAATTAAGGGAACAAGGTTGAGAGCCCTAGTAAGCGAGGCTCTAAAAAGCATGGCTGAGCTGAGATGGGTGGGCTTCTCTGAGCGCTTCTAAAATGCGCTAAACTGAGGTGATTACTCTGAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CAACCTCCATGCTAGT-1_contig_2": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTC",
        "GCATGATAGACAGGCT-1_contig_1": "GGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCTGTACCAACTGTATCCATCTTCCCACCATCCAGT",
        "GGAACTTGTAAACCTC-1_contig_9": "GGAGTCCTGAACTAACCTTGAATCTGAAGCAAAGGGGATCAGCCCGAGATTCTCATTCAGTGATCAACACTGAACACACATCCCTTACCATGGATTTTGGGCTGATTTTTTTTATTGTTGCTCTTTTAAAAGGGGTCCAGTGTGAGGTGAAGCTTCTCGAGTCTGGAGGTGGCCTGGTGCAGCCTGGAGGATCCCTGAAACTCTCCTGTGCAGCCTCAGGATTCGATTTTAGTAGATACTGGATGAGTTGGGTCCGGCAGGCTCCAGGGAAAGGGCTAGAATGGATTGGAGAAATTAATCCAGATAGCAGTACGATAAACTATACGCCATCTCTAAAGGATAAATTCATCATCTCCAGAGACAACGCCAAAAATACGCTGTACCTGCAAATGAGCAAAGTGAGATCTGAGGACACAGCCCTTTATTACTGTGCAAGACGAGGTAATCCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "GACCAATCATCGTCGG-1_contig_2": "TTTCAATGAACAAATAATAAGCAGTCCTATGTGACATGCTTCTAAAGCAAAAGATATAATATTTAGTATTATATACATTAATAATAAAATACATTATCTTCTAAGAATTGAAGTCTCAACTATGAAAATCAGCAGTTCTCTGTCAGAGAAGCCCAAGCGCTTCCACGCATGCTTGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TTGGAACGTTTCGCTC-1_contig_2": "GGGGACATTGTGAGCCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGTGCAAGTCAGGGCATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTATTACACATCAAGTTTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGGACAGATTATTCTCTCACCATCAGCAACCTGGAACCTGAAGATATTGCCACTTACTATTGTCAGCAGTATAGTAAGCTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TTGCCGTCATCCAACA-1_contig_4": "GCTTATGAGAATAGCAGTAATTAACTAGGGACCAAAATTCAAAGAGAAAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCTGGGGAGAAGGTCACCATGACCTGCAGTGCCAGATCAAGTGTAAGTTCCAGCTACTTGTACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAAACTCTGGATTTATAGCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTATTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTTTTACTGCCAGCAGTACAGTGGTTACCCATCCATCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "CACAGTAAGGCGATAC-1_contig_4": "AGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCTCCAGGTCACTGTGCTGGCCTC",
        "GGAACTTGTAAACCTC-1_contig_8": "CTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTG",
        "GATTCAGCACTCTGTC-1_contig_6": "GGGGGTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTG",
        "TTCTTAGGTGATAAAC-1_contig_1": "TGGGGATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTAC",
        "TGCCCTACAGTGGAGT-1_contig_1": "GGGGATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCTTCCCACCATCCAGTGAGCAGTTAACATCT",
        "TATGCCCAGGCCCTCA-1_contig_5": "GATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGTGCTGCTCCTCTGCCTGGTGACATTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATAACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGAGGTGGAAGCACAGACTACAATGCAGCTTTCATGTCCAGACTGAGCATCACCAAGGACAACTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTGATGACACTGCCATATACTACTGTGCCAAAAATTACTACGGTAGTAGCTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CCGTTCACACTCGACG-1_contig_1": "GATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCT",
        "CAAGATCCAGCGTAAG-1_contig_10": "GACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "AGCGTCGGTCTGATCA-1_contig_2": "GAGTGTCCTCTCCACAGTCCCTGACGACACTGACTCTAGCCATGGAATGGAGCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCAATCCCAGGTTCAACTGCAGCAGTCTGGGGCTGAGCTGGTGAGGCCTGGGGCTTCAGTGAAGCTGTCCTGCAAGGCTTTGGGCTACACATTTACTGACTATGAAATGCACTGGGTGAAGCAGACACCTGTGCATGGCCTGGAATGGATTGGAGCTATTCATCCAGGAAGTGGTGGTACTGCCTACAATCAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACAGCCTACATGGAGCTCAGCAGCCTGACATCTGAGGACTCTGCTGTCTATTACTGTACAAGAGGGGGGACGAGATACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TACGGGCTCGTCGTTC-1_contig_1": "GGAGTAGTTATCTTGTGAATAATCAACCTCCTTAAAATGAATTTGCCTGTTCATCTCTTGGTGCTTCTGTTGTTCTGGATTCCTGCTTCCAGAGGTGATGTTGTGGTGACTCAAACTCCACTCTCCCTGCCTGTCAGCTTTGGAGATCAAGTTTCTATCTCTTGCAGGTCTAGTCAGAGTCTTGCAAACAGTTATGGGAACACCTATTTGTCTTGGTACCTGCACAAGCCTGGCCAGTCTCCACAGCTCCTCATCTATGGGATTTCCAACAGATTTTCTGGGGTGCCAGACAGGTTCAGTGGCAGTGGTTCAGGGACAGATTTCACACTCAAGATCAGCACAATAAAGCCTGAGGACTTGGGAATGTATTACTGCTTACAAGGTACACATCAGCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "TGCCAAAGTGTAACGG-1_contig_1": "GAGCTGCCAGGAGCCTAAGAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGTTATGGGTACTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCAAAAGTGTCAGTACATCTGGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCACAGTAGGGAGCTTCCTCCCACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACC",
        "ACGAGCCGTCATATCG-1_contig_4": "GGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGAGACAGACTCTTCTCCACAGTGTCCCCTTCATGT",
        "GTTAAGCTCGCCGTGA-1_contig_1": "AAACAACCTATGATCAGTGTCCTCTCTACACAGTCCCTGACGACACTGACTCTAACCATGGGATGGAGCCGGATCTTTCTCTTCCTCCTGTCAATAACTGCAGGTGTCCATTGCCAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCCTCAGTGAAGATTTCCTGCAAAGCTTCTGGCTACGCATTCAGTAGCTCTTGGATGAACTGGGTGAAGCAGAGGCCTGGACAGGGTCTTGAGTGGATTGGACGGATTTATCCTGGAGATGGAGATACTAACTACAATGGGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACCTCTGTGGACTCTGCGGTCTATTTCTGTGCAAGATTGAACTATGATTACGACGGGTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TATTACCGTAAGGGAA-1_contig_1": "GGGGGAAATGCATCAGACCAGCATGGGCATCAAGATGGAATCACAGACTCTGGTCTTCATATCCATACTGCTCTGGTTATATGGTGCTGATGGGAACATTGTAATGACCCAATCTCCCAAATCCATGTCCATGTCAGTAGGAGAGAGGGTCACCTTGAGCTGCAAGGCCAGTGAGAATGTGGGTACTTATGTATCCTGGTATCAACAGAAACCAGAGCAGTCTCCTAAACTGCTGATATACGGGGCATCCAACCGGTACACTGGGGTCCCCGATCGCTTCACAGGCAGTGGATCTGCAACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGACCTTGCAGATTATCACTGTGGACAGAGTTACAGCTATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTG",
        "GGCTGGTCAACACCTA-1_contig_4": "GGAGCTCTGACAGAGGAGCCAAGCCCTGGATTCCCAGGTCCTCACATTCAGTGATCAGCACTGAACACAGACCACTCACCATGGACTCCAGGCTCAATTTAGTTTTCCTTGTCCTTATTTTAAAAGGTGTCCAGTGTGATGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGCAGCCTGGAGGGTCCCGGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTTTGGAATGCACTGGGTTCGTCAGGCTCCAGAGAAGGGGCTGGAGTGGGTCGCATACATTAGTAGTGGCAGTAGTACCATCTACTATGCAGACACAGTGAAGGGCCGATTCACCATCTCCAGAGACAATCCCAAGAACACCCTGTTCCTGCAAATGACCAGTCTAAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGATACGACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GACGCGTCAGTAGAGC-1_contig_1": "GTTTTCTCTTTCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGAATTGAAGTCTCAACTATGAAAATCAGCAGTTCTCTGTCAGAGAAGCCCAAGCGCTTCCACGCATGCTTGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GTAACTGGTAGGGTAC-1_contig_1": "GGGGATAGTCAGGTCACACCCTGTGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCTCCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCT",
        "CTAGTGATCTGCTGCT-1_contig_2": "AGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAA",
        "AAGGCAGCACGGTTTA-1_contig_1": "TGGGGGAAATGCATCACACCAGCATGGGCATCAAAATGGAGTCACAGATTCAGGTCTTTGTATTCGTGTTTCTCTGGTTGTCTGGTGTTGACGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGAGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGACAATCTCCTAAACTACTGATTTACTCGGCATCCTACCGGTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATCTGGGACGGATTTCACTTTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAACATTATAGTACTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCAG",
        "TACAGTGAGATGCCTT-1_contig_3": "GAACCTATGATCAGTGTCCTCTCCACAATCTCTGAAGACACTGACTCTAACCATGGAATGGAGCTGGGTCTTTCTGTTCCTCCTGTCAGTAACTTCAGGTGTCCACTCCCAGGTCCAGCTGCAGCAGTCTGGAGCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGCTGTCCTGCAAGACTTCTGGCTACACCTTCACCAGCTACTGGATTCAGTGGGTAAAACAGAGGCCTGGACAGGGCCTTGGGTGGATTGGAGAGATATTTCCTGGAACTGGCACTACTTACTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTATAGACACATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACCTCTGAGGACTCTGCTGTCTATTTCTGTGCAAGAAGGTTAACTGGGACGTCCGGCTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACCA",
        "CCTTCGAAGCACCGTC-1_contig_5": "GAACATAAGATCACTGTTCTCTCTACAGTTACTGAGCACACAGGAACTCACCATGGGATGGAGCTGTATCATCCTCTTCTTGGCAGCAACAGCTACAGGTGTCCACTCCCAGGTCCAACTGCAGCAGTCTGGGCCTGAGCTGGTGAGGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCAGGCTATACCTTCACCAGCTACTGGATGCACTGGGTGAAACAGAGGCCTGGACAAGGCCTTGAGTGGATTGGCATGATTGATCCTTCCAATAGTGAAACTAGGTTAAATCAGAAGTTCAAGGACAAGGCCACATTGAATGTAGACAAATCCTCCAACACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGAGTAGGACGCTACTATGGTTACGACGACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GATGCTAGTTGTCTTT-1_contig_1": "GGGGATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "GAACCTAAGATCGGGT-1_contig_1": "GGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "ACCGTAAGTGTAACGG-1_contig_4": "GAAGATCACTGGTCTCTCTGCAGTTACTAGGCACACAGGACCTCACCATGAGATGGAGCTGTATCATCCTCTTCTTGGTAGCAACAGCTACAGGTGTCAACTCCCAGGTCCAACTGCAGCAGCCTGGGGCTGAGCTTGTGATGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACATTCACTGACTACTGGATGCACTGGGTGAAGCAGAGGCCTGGACAAGGCCTTGAGTGGATCGGAGCGATTGATACTTCTGATAGTTATACTAGCTACAATCAAAAGTTCAAGGGCAAGGCCACATTGACTGTAGACGAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGAGAGGATAGGTATGATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "AGGGAGTAGTGTGAAT-1_contig_4": "GTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCA",
        "GCACTCTAGGTCATCT-1_contig_6": "GACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "CTTACCGGTCTAGTCA-1_contig_1": "GGGGATATCTCTCACTGGAGCGTGATCTCTGAAGGAAGGAGGTGTAGCCTAAAAGATGAGAGTGTTGATTCTTGTGTACCTGTTGACAGTCCTTCCTGGTATACTGTCTGATGTACAGCTTCAGGAGTCAGGACCTGGCCTGGTGAAGCCTTCTCAGACAGTGTCCCTCACCTGCACTGTCACTGGCTACTCTATCACTAATGGTAATCACTGGTGGAACTGGATCCGGCAGGTTTCAGGAAACAAACTGGAGTGGATGGGGTACATAAGCTCCAGTGGTAGCACTGACAGCAATCCATCTCTCAAAAGTCAAATCTCCATCACTAGAGACACTTCCAAGAACCAGTTATTCCTGCAGTTGAACTCTGTGACTATTGAAGATATAGCCACATATTACTGTGCAAGAGATTACGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GGTATTGAGAGTACAT-1_contig_4": "GGGATCCTCTTCTCATAGAGCCTCCATCAGACCATGGCTGTCCTGGCACTGCTCCTCTGCCTGGTGACATTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACTGTCTCAGGGTTCTCATTAACCAGCTATGGTGTAAGCTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGGTGACGGGAGCACAAATTATCATTCAGCTCTCATATCCAGACTGAGCATCAGCAAGGATAACTCCAAGAGCCAAGTTTTCTTAAAACTGAACAGTCTGCAAACTGATGACACAGCCACGTACTACTGTGCCAAATGCTACTACGGTAGTAGCTACCCTTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TGCCCTAGTAGAGGAA-1_contig_1": "ATAGTCAGATCACACCATGTGCTGGAGTCAGCCTCACACTGATCACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGATGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGTATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTAATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATGCTGCAACAAACTTAGCAGATGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTATTCCCTCAAGATCAACAGCCTGCAGTCTGAAGATTTTGGGAGTTATTACTGTCAACATTTTTGGGGTACTTCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACAT",
        "TCGGGACTCCGTACAA-1_contig_1": "GGGTTTAAAGCAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATATCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAGGCACCTCCCCCAAAAGATGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GACGGCTTCACATAGC-1_contig_2": "TGGGGAGCTGCCAGGAGCCTAAGAAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACAGGTAACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATATCCTGCAGAGCCAGTGAAAGTGTTGATAGTTATGGCAATAGTTTTATGCACTGGTACCAGCAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTAGGACAGACTTCACCCTCACCATTGATCCTGTGGAGGCTGATGATGCTGCAACCTATTACTGTCAGCAAAATAATGAGGATCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TACTCATAGTGTGGCA-1_contig_4": "GAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "AAAGTAGCAAATCCGT-1_contig_2": "GGGGTTCTGCATCAGAAAGGCAGGGGGATCAAGATGGAATCACAGACTCAGGTCTTCCTCTCCCTGCTGCTCTGGGTATCTGGTACCTGTGGGAACATTATGATGACACAGTCGCCATCATCTCTGGCTGTGTCTGCAGGAGAAAAGGTCACTATGAGCTGTAAGTCCAGTCAAAGTGTTTTATACAGTTCAAATCAGAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATCTACTGGGCATCCACTAGGGAATCTGGTGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTTACTCTTACCATCAGCAGTGTACAAGCTGAAGACCTGGCAGTTTATTACTGTCATCAATACCTCTCCTCGCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "CGAGCCACACGGCTAC-1_contig_1": "GGGGAGCTGCCAGGAAGCTAAGAAGCATCCTCTCTTCTAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCCAAAGTGTCAGTACATCTAGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCAAGTATGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATACTGCAACATATTACTGTCAGCACAGTTGGGAGATTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCC",
        "GATCAGTGTCCAACTA-1_contig_3": "GAAAAATACGATCAGCATCCTCTCCACAGACACTGAAAACTCTGACTCACAATGGAAAGGCACTGGATCTTTCTCTTCCTGTTTTCAGTAACTGCAGGTGTCCACTCCCAGGTCCAGCTTCAGCAGTCTGGGGCTGAACTGGCAAAACCTGGGGCCTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACCTTTACTAGCTACTGGATGCACTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATTGGATACATTAATCCTAGCACTGGTTATACTGAGTACAATCAGAAGTTCAAGGACAAGGCCACATTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAACTGAGCAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGAGGGGGATTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "ACAGCCGGTCCGAGTC-1_contig_6": "GGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TCCCGATTCAACTCTT-1_contig_4": "TGGGGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTCCTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "TGAGGGATCCGTTGTC-1_contig_3": "GGGGATTGTCATTGCAGCCAGGACTCAGCATGGACATGAGGACCCCTGCTCAGTTTCTTGGAATCTTGTTGCTCTGGTTTCCAGGTATCAAATGTGACATCAAGATGACCCAGTCTCCATCTTCCATGTATGCATCTCTAGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAATAGCTATTTAAGCTGGTTCCAGCAGAAACCAGGGAAATCTCCTAAGACCCTGATCTATCGTGCAAACAGATTGGTAGATGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTCACCATCAGCAGCCTGGAGTATGAAGATATGGGAATTTATTATTGTCTACAGTATGATGAGTTTCCTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCA",
        "AGCTCCTTCCGCTGTT-1_contig_1": "TGGGGAAATACATCAGATCAGCATGGGCATCAAGATGGAGTCACAGACTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGATGGAGACATTGTGATGACCCAGTCTCAAAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCGTCACCTGCAAGGCCAGTCAGAATGTGGGTACTAATGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAAGCACTGATTTACTCGGCATCCTACCGGTACAGTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAATGTGCAGTCTGAAGACTTGGCAGAGTATTTCTGTCAGCAATATAACAGCTATCCTCTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCAATGATACGGCGACCAC",
        "GAAATGAAGGTGCAAC-1_contig_1": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGCCCCAGGCCCAGGCCCAGGCTCCAGCTCCGGCTCCGGCTCCAGCTCCAGCTCCTGCCACTGCTTCAG",
        "CACAAACCAACGATGG-1_contig_1": "GAGGCTGATCATATCCAGTGCAGCAATCAGTTCCTGCCAGGACACAGTTTAGATATGAGGTTCCAGGTTCAGGTTCTGGGGCTCCTTCTGCTCTGGATATCAGGTGCCCAGTGTGATGTCCAGATAACCCAGTCTCCATCTTATCTTGCTGCATCTCCTGGAGAAACCATTACTATTAATTGCAGGGCAAGTAAGAGCATTAGCAAATATTTAGCCTGGTATCAAGAGAAACCTGGGAAAACTAATAAGCTTCTTATCTACTCTGGATCCACTTTGCAATCTGGAATTCCATCAAGGTTCAGTGGCAGTGGATCTGGTACAGATTTCACTCTCACCATCAGTAGCCTGGAGCCTGAAGATTTTGCAATGTATTACTGTCAACAGCATAATGAATACCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CGGCTAGCATTTCACT-1_contig_3": "TGGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTAC",
        "CAGCGACAGTGTTGAA-1_contig_3": "GTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "ACGAGCCCATGGTTGT-1_contig_1": "TGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "CGGCTAGGTGACTACT-1_contig_2": "TGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CGGTTAAGTATATCCG-1_contig_1": "GACATACCAGCAAGCGAGTGACCAGTTAGTCTTAAGGCACCACTTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCAAGAGAGGATACTACGGCCTATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AACTCAGCACATGACT-1_contig_5": "GGGGCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "CCCAATCGTGTGTGCC-1_contig_11": "GGACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGATACTATGATTACGATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GCAATCAGTGGCGAAT-1_contig_1": "GGGAGTCAGATCACACCATGTGCTGGAGTCAGCCTCACACTGATCACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGATGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGTATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTAATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATGCTGCAACAAACTTAGCAGATGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTATTCCCTCAAGATCAACAGCCTGCAGTCTGAAGATTTTGGGAGTTATTACTGTCAACATTTTTGGGGTACTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCT",
        "AGGGAGTCAAGTAATG-1_contig_3": "GACCCTGAAAACAACCTATGATCAGTGTCCTCTCCACAGTTCCTGAAGACACTGACTCTAACCATGGGATGGAGCTGGATCTTTCTTTTCCTCCTGTCAGGAACTGCAGGTGTCCATTGCCAGATCCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGCTACACCTTCACTGACTACTATATAAACTGGGTGAAGCAGAAGCCTGGACAGGGACTTGAGTGGATTGGATGGATTTATCCTGGAAGCGGTAATACTAAGTACAATGAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACACATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACACTGCTGTCTATTTCTGTGCAAGAGGGAAGGGTAGTAGCTACGGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CACAGGCGTTGGTTTG-1_contig_9": "TGGGGAATCAACCTCCTTAAAATGAATTTGCCTGTTCATCTCTTGGTGCTTCTGTTGTTCTGGATTCCTGCTTCCAGAGGTGATGTTGTGGTGACTCAAACTCCACTCTCCCTGCCTGTCAGCTTTGGAGATCAAGTTTCTATCTCTTGCAGGTCTAGTCAGAGTCTTGCAAACAGTTATGGGAACACCTATTTGTCTTGGTACCTGCACAAGCCTGGCCAGTCTCCACAGCTCCTCATCTATGGGATTTCCAACAGATTTTCTGGGGTGCCAGACAGGTTCAGTGGCAGTGGTTCAGGGACAGATTTCACACTCAAGATCAGCACAATAAAGCCTGAGGACTTGGGAATGTATTACTGCTTACAAGGTACACATCAGCCTCCGACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CAAGATCCAGCGTAAG-1_contig_6": "GGAGGTCACAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GTGCTTCAGGGCTTGA-1_contig_4": "GAGCTCTGACAGAGGAGGCCGGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACGGACCCCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGCCCTCATTTTAAAAGGTGTCCAGTGTGAGGTGCAGCTGGTGGAGTCTGGGGGAGACTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGGCATGTCTTGGGTTCGCCAGACTCCAGACAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGGCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTGCAAGACGCCTCGGTAGTAGCTACCCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CTCACACCAGATAATG-1_contig_2": "GGGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "ACTGCTCGTACCGCTG-1_contig_4": "ACTTTTGACTCACCATATCAAGTTCGCAGAATGAGGTTCTCTGCTCAGCTTCGGGGGCTGCTTGTGCTCTGGATCCCTGGATCCACTGCAGATATTGTGATGACGCAGGCTGCATTCTCCAATCCAGTCACTCTTGGAACATCAGCTTCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTACATAGTAATGGCATCACTTATTTGTATTGGTATCTGCAGAAGCCAGGCCAGTCTCCTCAGCTCCTGATTTATCAGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTAGCAGTGGGTCAGGAACTGATTTCACACTGAGAATCAGCAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTGCTCAAAATCTAGAACTTGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAAAGGAACCATAACTGATTTAATGAGCCATTCGCAGTTTCACTGCCCATATAAGAAACGCTCGCTTGTAAGAGCCTCAGACCCAGA",
        "AAACGGGAGTTACCCA-1_contig_3": "GACATACCAGCAAGCGAGTGACCAGTTAGTCTTAAGGCACCACTTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCAACTGGGGCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AACCGCGAGAAACGCC-1_contig_4": "GGGGAAGCAACATGTGTCCAATGTCCTCTCCACAGACACTGAACACACTGACTCCAACTATGGGATGGAGCTGGATCTTTCTCTTCCTCCTGTCAGGAACTGCAGGTGTCCACTCTGAGGTCCAGCTGCAACAGTCTGGACCTGAGCTGGTGAAGCCTGGAGCTTCAATGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGCTACACCATGAACTGGGTGAAGCAGAGCCATGGAAAGAACCTTGAGTGGATTGGACTTATTAATCCTTACAATGGTGGTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTAACTGTAGACAAGTCATCCAGCACAGCCTACATGGAGCTCCTCAGTCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGATCCGGAGGTAACTACGCATCCGTATATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCATG",
        "AGTGAGGGTTGTCTTT-1_contig_8": "GGAGCAACATGTGTCCAATGTCCTCTCCACAGACACTGAACACACTGACTCCAACTATGGGATGGAGCTGGATCTTTCTCTTCCTCCTGTCAGGAACTGCAGGTGTCCACTCTGAGGTCCAGCTGCAACAGTCTGGACCTGAGCTGGTGAAGCCTGGAGCTTCAATGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGCTACACCATGAACTGGGTGAAGCAGAGCCATGGAAAGAACCTTGAGTGGATTGGACTTATTAATCCTTACAATGGTGGTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTAACTGTAGACAAGTCATCCAGCACAGCCTACATGGAGCTCCTCAGTCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGATGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CTCGAAAGTTCGGGCT-1_contig_2": "GGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACC",
        "AACTTTCCAGCGAACA-1_contig_8": "GAAAACAACATATGTCCAATGTCCTCTCCTCAGACACTGAACACACTGACTCTAACCATGGGATGGAGCTGGATCTTTCTCTTTCTCCTGTCAGGAACTGCAGGTGTCCTCTCTGAGGTCCAGCTGCAACAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGACTTCTGGATACACATTCACTGAATACACCATGCACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGAGGTATTAATCCTAACAATGGTGGTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAGTCCTCCAGCACAGCCTACATGGAGCTCCGCAGCCTGACATCTGAGGATTCTGCAGTCTATTACTGTGCAAGATTGGACTATAGGTACGACCCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AGCGTCGGTCTGATCA-1_contig_1": "GAGTCAGGTCACACCCTTTGCTGGAGTCAGAATCACACTGATCACACACAGTCATGAGTGTGCTCACTCAGGTCCTGGCGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGGGAATATTCACAATTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGATGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGAACACAATATTCTCTCAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATTTTTGGAGTACTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AGCTCCTTCCGCTGTT-1_contig_5": "TGGGGAAATACATCAGATCAGCATGGGCATCAAGATGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAAC",
        "CCTCTGAAGAGGTAGA-1_contig_5": "GGGTCACTGGGGGCTCTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTA",
        "CGAGCCATCTGCCAGG-1_contig_8": "GGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTG",
        "CAGCAGCTCCTATTCA-1_contig_1": "GGGCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AGAGCGAAGGTGCTTT-1_contig_2": "TGGGGAGCTGCCAGGAGCCTAAGAAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACAGGTAACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATATCCTGCAGAGCCAGTGAAAGTGTTGATAGTTATGGCAATAGTTTTATGCACTGGTACCAGCAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTAGGACAGACTTCACCCTCACCATTGATCCTGTGGAGGCTGATGATGCTGCAACCTATTACTGTCAGCAAAATAATGAGGATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GACGCGTCAGTAGAGC-1_contig_3": "GGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "AATCGGTCATACAGCT-1_contig_1": "GTTTTGTGGTTTCTTATATGGGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTGA",
        "GCCTCTAGTAGAGGAA-1_contig_5": "GTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTGA",
        "CCCAATCCAAGCGTAG-1_contig_3": "GATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GGACAGAAGAACAACT-1_contig_2": "GAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CGGTTAATCTGCCAGG-1_contig_4": "GCTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCAC",
        "CGTAGGCCACTTAACG-1_contig_3": "AGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGG",
        "CGGGTCATCAGGCAAG-1_contig_1": "GACATATGAACACTGTTTTCTCTACAGTCACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTCATCTTCTTCCTGATGGCAGTGGTTATAGGAATCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCAGAGCTTGTGAGGTCAGGGGCCTCAGTCAAGTTGTCCTGCACAGCTTCTGGCTTCAACATTAAAGACTACTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGATGGATTGATCCTGAGAATGGTGATACTGAATATGCCCCGAAGTTCCAGGGCAAGGCCACTATGACTGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTAATGCCCTCCCCTATAGGTACGACGGGGGGTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAG",
        "ATTCTACAGTCTCCTC-1_contig_1": "GGGGAACAGCATCAGACAGGCTGGGCAGCAAGATGGAATCACAGACCCAGGTCCTCATGTTTCTTCTGCTCTGGGTATCTGGTGCCTGTGCAGACATTGTGATGACACAGTCTCCATCCTCCCTGGCTATGTCAGTAGGACAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTAAATAGTAGCAATCAAAAGAACTATTTGGCCTGGTACCAGCAGAAACCAGGACAGTCTCCTAAACTTCTGGTATACTTTGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCATAGGCAGTGGATCTGGGACAGATTTCACTCTTACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGATTACTTCTGTCAGCAACATTATAGCACTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GATCGCGGTCGGCTCA-1_contig_1": "GGGGTCATTGTCTTTACAATCAGGACTCAGCATGGACATGATGGTCCTTGCTCAGTTTCTTGCATTCTTGTTGCTTTGGTTTCCAGGTGCAAGATGTGACATCCTGATGACCCAATCTCCATCCTCCATGTCTGTATCTCTGGGAGACACAGTCAGCATCACTTGCCATGCAAGTCAGGGCATTAGCAGTAATATAGGGTGGTTGCAGCAGAAACCAGGGAAATCATTTAAGGGCCTGATCTATCATGGAACCAACTTGGAAGATGGAGTTCCATCAAGGTTCAGTGGCAGTGGATCTGGAGCAGATTATTCTCTCACCATCAGCAGCCTGGAATCTGAAGATTTTGCAGACTATTACTGTGTACAGTATGCTCAGTTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCATCTTCCCACCATCCAGTGAGCAGTTAAC",
        "AAGGAGCGTAAAGTCA-1_contig_4": "GGAAATCTGAGGCTCACCTCACATACCAGCAAGGGAGTGACCAGTTAGTCTTAAGGCACCATTGAGCCCAAGTCTTAGACATCATGGATTGGCTGTGGAACTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCAACACTGGAGAGCCAACATATGCTGAAGAGTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCAAGAAGGATTACGACGAGAGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AAGGTTCTCTTGAGGT-1_contig_3": "GTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGACAGACTCTTCTCCAC",
        "CTTAGGAAGTGACATA-1_contig_4": "GATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGAGACAGACTCTTCTCCACAGTGTCCCCTT",
        "ATTACTCGTACTCAAC-1_contig_3": "GGAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGCATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATAACCTGCAGTGCCAGCTCAAGTGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCTCAGTCGTGTGCTT",
        "CGCTTCACATCCCATC-1_contig_2": "AGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGCCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GCACATAGTAGAGTGC-1_contig_3": "GGCAGAAAACAGCATATGATCAGTGTCCTCTCCAAAGTCCTTGAACATAGACTCTAACCATGGAATGGACCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGAGCTGAGCTGATGAAGCCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTACTGGCTACACATTCAGTAGCTACTGGATAGAGTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGAGATTTTACCTGGAAGTGGTAGTACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACATTCACTGCAGATACATCCTCCAACACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCCGTCTATTACTGTGCAAGATGGGGGGGACGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CCGGGATGTCTTCTCG-1_contig_1": "GAATCTGATCATACACACTCCAACAGTCATTCTTGGTCAGGAGACGTTGTAGAAATGAGACCGTCTATTCAGTTCCTGGGGCTCTTGTTGTTCTGGCTTCATGGTGCTCAGTGTGACATCCAGATGACACAGTCTCCATCCTCACTGTCTGCATCTCTGGGAGGCAAAGTCACCATCACTTGCAAGGCAAGCCAAGACATTAACAAGTATATAGCTTGGTACCAACACAAGCCTGGAAAAGGTCCTAGGCTGCTCATACATTACACATCTACATTACAGCCAGGCATCCCATCAAGGTTCAGTGGAAGTGGGTCTGGGAGAGATTATTCCTTCAGCATCAGCAACCTGGAGCCTGAAGATATTGCAACTTATTATTGTCTACAGTATGATAATCTTTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAAC",
        "GCAAACTCAAACGCGA-1_contig_6": "GGGGGGCAGCCTGAGCTATGAGGAAGTAGAGAGGCTTGAGAAATCTGAGGAAGAAAAGAGTAGATCTGAGAGGAAAGGTAGCTTTCTGGAGGTCAGGAGACAGTGCAGAGAAGAACGAGTTACTGTGGACAGGTCTTAGATGGGGAAAGAATGAGCAAATGCAAGCATCAGAAGGGTGGATGCAATGTCCTGCCAAGGACTTACCAAGAGGATCCCCGGACAGAGCAGGCAGGTGGAGTTGACTGAGAGGACAGGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "GACTACATCATCATTC-1_contig_1": "TGGGGAGTGCAGAGAGTACATGGTGTTGAATCAACTCCATGGGGTACCTGCCTGACAAGAAGACAAGCCTATAAAAGGATTACCACCCACTGCTTCTCAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CCCAATCAGTGGTAGC-1_contig_3": "GAAAACAACCTATGATCAGTGTCCTCTCTACACAGTCCCTGACGACACTGACTCTAACCATGGGATGGAGCCGGATCTTTCTCTTCCTCCTGTCAATAACTGCAGGTGTCCATTGCCAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCCTCAGTGAAGATTTCCTGCAAAGCTTCTGGCTACGCATTCAGTAGCTCTTGGATGAACTGGGTGAAGCAGAGGCCTGGACAGGGTCTTGAGTGGATTGGACGGATTTATCCTGGAGATGGAGATACTAACTACAATGGGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACCTCTGTGGACTCTGCGGTCTATTTCTGTGCAAGAAGGGGCCCTTATGATTACGACGGTGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GGGAGATTCACTTCAT-1_contig_2": "GAGAAAAACATGAGATCACAGTTCTCTCTACAGTTACTGAGCACACAGGACCTCACCATGGGATGGAGCTGTATCATCCTCTTCTTGGTAGCAACAGCTACAGGTGTCCACTCCCAGGTCCAACTGCAGCAGCCTGGGGCTGAGCTGGTGAGGCCTGGGGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTGGATAAACTGGGTGAAGCAGAGGCCTGGACAAGGCCTTGAGTGGATCGGAAATATTTATCCTTCTGATAGTTATACTAACTACAATCAAAAGTTCAAGGACAAGGCCACATTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCCGACATCTGAGGACTCTGCGGTCTATTACTGTACCATGGGACACGCGCGCTACTATGGTAACTACGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TTATGCTAGGCTCAGA-1_contig_1": "GGGAGCTGCCAGGAGCCTAAGAAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACAGGTAACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATATCCTGCAGAGCCAGTGAAAGTGTTGATAGTTATGGCAATAGTTTTATGCACTGGTACCAGCAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTAGGACAGACTTCACCCTCACCATTGATCCTGTGGAGGCTGATGATGCTGCAACCTATTACTGTCAGCAAAATAATGAGGATCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GCTCTGTCAGGTTTCA-1_contig_10": "GGGCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGT",
        "TACTCATTCATTCACT-1_contig_3": "CTCTGGGACAAAGGCTGTGACTCTGGGAAAGACAAGAGAAGGGCAGGACCAAAACAGGAACAGAGACGGCTGCTTTCACAGCTTCCACATGTGAGTGGGGTCAGCAGGGAAAGGAGCTGCACGAAGAGGCCATACAAACAGCACGCATCTGTGGCCCTTCCAGATCTTTGAGTCATCCTATCACGGGAGATTGGGAAGGAGTTGACAGACCAGCCCAGGCAGTGGAAGCCTCTGTGTTAAAGAGTAAAGGTGCTTGCCTACAGCCTGGTGTCAACTAGGCAGGCCCTGGGGGGCCGGGAAGGGGCCTCCTAGACAAGCACAGGCATGTAGAGCTGCACACCCCACAGACAAACCTGAGCCCCGAGGATATCATGGAATATATCGAGAAGCCTGAGGAATGTGTTTGGCATGGACTACAGGTTGAGAGAACCAAGGAAGCTGAGCCCTGCGCCAAAACGACACCCCCATCTGTCTATCCACTGGCCCCTGGATCTGCTGCCCAAACTAACTCCATGGTGACCCTGGGATGCCTGGTCAAGGG",
        "CTGCCTAGTGGGTCAA-1_contig_1": "TTGCGTACTTTTCTTATATGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCT",
        "GACTAACGTGTGGTTT-1_contig_5": "CCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "CAACCTCCATGCTAGT-1_contig_9": "GGAGTGTGAGAGGACAGGGGCTGGGGTATGGATACGCAGAAGGAAGGCCACAGCTGTACAGAATTGAGAAAGAATAGAGACCTGCAGTTGAGGCCAGCAGGTCGGCTGGACTAACTCTCCAGCCACAGTAATGACCCAGACAGAGAAGGCCAGACTCATAAAGCTTGCTGAGCAAAATTAAGGGAACAAGGTTGAGAGCCCTAGTAAGCGAGGCTCTAAAAAGCATGGCTGAGCTGAGATGGGTGGGCTTCTCTGAGCGCTTCTAAAATGCGCTAAACTGAGGTGATTACTCTGAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CTTAGGAAGATAGTCA-1_contig_1": "GGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCACTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTG",
        "ACTATCTCAGTGGAGT-1_contig_4": "GGGGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCTTGGGGCTGCTCTTCTGCCTGGTGACATTCCCAAGCTGTGTCCTATCCCAGGTGCAGCTGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATCACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGTGGTGGAAGCACAGACTATAATGCAGCTTTCATATCCAGACTGAGCATCAGCAAGGACAATTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTAATGACACAGCCATATATTACTGTGCCAGAAAGTATGGTAACTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TGAAAGAAGATGGCGT-1_contig_2": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CATATGGCAGCTTAAC-1_contig_2": "GAGATTGTGAGCCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCTCTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CAGAATCGTGTTCGAT-1_contig_1": "GGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CTAACTTGTTCCATGA-1_contig_1": "GGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTG",
        "GCGAGAATCGTATCAG-1_contig_3": "GCAAATAGGGCCTCTTTCTCCTCATGAAACGCAGACCAACCTATCCTTGCAGTTCAGACATAGGAGCTTGGCTCTGGTTCCCAAGACCTCTCACTCACTTCTCAACATGGAGTGGGAACTGAGCTTAATTTTCATTTTTGCTCTTTTAAAAGATGTCCAGTGTGAAGTGCAGCTGTTGGAGACTGGAGGAGGCTTGGTGCAACCTGGGGGGTCACGGGGACTCTCTTGTGAAGGCTCAGGGTTTACTTTTAGTGGCTTCTGGATGAGCTGGGTTCGACAGACACCTGGGAAGACCCTGGAGTGGATTGGAGACATTAATTCTGATGGCAGTGCAATAAACTACGCACCATCCATAAAGGATCGATTCACTATCTTCAGAGACAATGACAAGAGCACCCTGTACCTGCAGATGAGCAATGTGCGATCTGAGGACACAGCCACGTATTTCTGTATGAGATATGGTGATTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GTGCAGCCACAACTGT-1_contig_2": "ATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGAGACGGGGGCTGGAACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACC",
        "ATGGGAGGTCGTCTTC-1_contig_3": "GGCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TAAACCGGTCGACTGC-1_contig_2": "GGGGTAATTAGCCAGGAACTAAAATTCAAAGACACAATGGATTTACAGGTGCAGATTATCAGCTTCCTGCTAATCAGTGTCACAGTCATAATGTCCAGAGGAGAAAATGTGCTCACCCAGTCTCCAGCAATAATGGCTGCCTCTCTGGGGCAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTCCAGTTACTTGCACTGGTACCAGCAGAAGTCAGGCGCTTCCCCCAAACCCTTGATTCATAGGACATCCAACCTGGCTTCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCGTGGAGGCTGAAGATGATGCAACTTATTACTGCCAGCAGTGGAGTGGTTACCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CTCGAAAGTTCGGGCT-1_contig_3": "GAAAAATACGATCAGCATCCTCTCCACAGACACTGAAAACTCTGACTCACAATGGAAAGGCACTGGATCTTTCTCTTCCTGTTTTCAGTAACTGCAGGTGTCCACTCCCAGGTCCAGCTTCAGCAGTCTGGGGCTGAACTGGCAAAACCTGGGGCCTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACCTTTACTAGCTACTGGATGCACTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATTGGATACATTAATCCTAGCACTGGTTATACTGAGTACAATCAGAAGTTCAAGGACAAGGCCACATTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAACTGAGCAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGACGGGCGTATGGTAACTTACCTCTCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TATCTCAGTATCGCAT-1_contig_3": "GCACTCCTCTTTCTGTCTTGATTACTTTTGACTCACCATATCAAGTTCGCAGAATGAGGTTCTCTGCTCAGCTTCTGGGGCTGCTTGTGCTCTGGATCCCTGGATCCACTGCAGATATTGTGATGACGCAGGCTGCATTCTCCAATCCAGTCACTCTTGGAACATCAGCTTCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTACATAGTAATGGCATCACTTATTTGTATTGGTATCTGCAGAAGCCAGGCCAGTCTCCTCAGCTCCTGATTTATCAGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTAGCAGTGGGTCAGGAACTGATTTCACACTGAGAATCAGCAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTGCTCAAAATCTAGAACTTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "ACGATACCACGAAAGC-1_contig_3": "TGGGGATATGAGCCCTATCTTCTCTACAGACACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTTATCTTCTTCCTGATGGCAGTGGTTACAGGGGTCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCAGAGCTTGTGAAGCCAGGGGCCTCAGTCAAGTTGTCCTGCACAGCTTCTGGCTTCAACATTAAAGACACCTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGAAGGATTGATCCTGCGAATGGTAATACTAAATATGACCCGAAGTTCCAGGGCAAGGCCACTATAACAGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTGCTAGATTAGATTACGACGCGATTTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TTGGAACGTTTCGCTC-1_contig_1": "GGGAGCTGCCAGGAGCCTAAGAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGTTATGGGTACTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCAAAAGTGTCAGTACATCTGGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCACAGTAGGGAGCTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TGAAAGAAGATGGCGT-1_contig_8": "GAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "ACGAGCCTCTGCAAGT-1_contig_4": "TGGGGAACCACAGCCCCTGAAGACACTGACTCTAACTATGGAATGGAGCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTACAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGACCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGCTACTACATGCACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGACGTGTTAATCCTAACAATGGTGGTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCATATTAACTGTAGACAAGTCATCCAGCACAGCCTACATGGAGCTCCGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGAGGAGGTTACTACGGTAGTAGCGAGGGTAAGGATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCT",
        "CCCAATCGTGTGTGCC-1_contig_12": "GGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCACGGGACAAACTCTTCTCCACAGTGTGACCTTCATGAGTGACCTGGCAGCTGTAACTGCTATGCCTTTCCCATGCTCTTGCTGT",
        "TCGGTAACAATACGCT-1_contig_2": "GGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CACCAGGAGGTGACCA-1_contig_1": "TGGGGGATCTACATCTGAAAGGCAGGTGGAGCAAGATGGAATCACAGACTCAGGTCCTCATGTCCCTGCTGTTCTGGGTATCTGGTACCTGTGGGGACATTGTGATGACACAGTCTCCATCCTCCCTGACTGTGACAGCAGGAGAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGTCTGTTAAACAGTGGAAATCAAAAGAACTACTTGACCTGGTACCAGCAGAAACCAGGGCAGCCTCCTAAACTGTTGATCTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGAACAGATTTCACTCTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGAATGATTATAGTTATCCTCTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "AATCGGTCAGGATTGG-1_contig_3": "GATCCATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCTTGGGGCTGCTCTTCTGCCTGGTGACATTCCCAAGCTGTGTCCTATCCCAGGTGCAGCTGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATCACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGTGGTGGAAGCACAGACTATAATGCAGCTTTCATATCCAGACTGAGCATCAGCAAGGACAATTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTAATGACACAGCCATATATTACTGTGCCAGAAATTTCCTCTACTATAGGTACCTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGT",
        "GGCGACTAGCGATATA-1_contig_2": "CAGCCTCTTTCTTATATGGGGAGCATCAGACAGGCAGGGGAGCAATATGGTATCAGAGACCCATGTCCTCATATTTTTGCTGCTATGGGTGTCTGGTGCTTGTGGGGAAATTGTGCTGACTGTGTCAGCAGGAGAGAGGGTCACTATCAGCTGCAAATCCAATCAGAATCTTTTATGGAGTGGAAACCAAAGGTACTGTTTGGTCTGGCACCAGTGGAAACCAGGGCAAACTCCTACACCGTTGATCACCTGGACATCTGATAGGTACTCTGGAGTCCCTGATCGTTTCATAGGCAGTGGATCTGTGACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGATGTGGCAGTTTATTTCTGTCAGCAGCATTTACACATTCCTTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACT",
        "GCAGTTAAGTCATGCT-1_contig_1": "AGCTCTGACAGAGGAGCCAAGCCCTGGATTCCCAGGTCCTCACATTCAGTGATCAGCACTGAACACAGACCACTCACCATGGACTCCAGGCTCAATTTAGTTTTCCTTGTCCTTATTTTAAAAGGTGTCCAGTGTGATGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGCAGCCTGGAGGGTCCCGGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTTTGGAATGCACTGGGTTCGTCAGGCTCCAGAGAAGGGGCTGGAGTGGGTCGCATACATTAGTAGTGGCAGTAGTACCATCTACTATGCAGACACAGTGAAGGGCCGATTCACCATCTCCAGAGACAATCCCAAGAACACCCTGTTCCTGCAAATGACCAGTCTAAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGAGAGGGGTACTATGGTTACGACTCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "AATCCAGAGATGCGAC-1_contig_10": "GGACATCTCTCTCATTAGAGGTTGATCTTTGAGGAAAACAGGGTGTTGCCTAAAGGATGAAAGTGTTGAGTCTGTTGTACCTGTTGACAGCCATTCCTGGTATCCTGTCTGATGTACAGCTTCAGGAGTCAGGACCTGGCCTCGTGAAACCTTCTCAGTCTCTGTCTCTCACCTGCTCTGTCACTGGCTACTCCATCACCAGTGGTTATTACTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATAAGCTACGACGGTAGCAATAACTACAACCCATCTCTCAAAAATCGAATCTCCATCACTCGTGACACATCTAAGAACCAGTTTTTCCTGAAGTTGAATTCTGTGACTACTGAGGACACAGCTACATATTACTGTGCATATGGTAACTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "GCGCAGTAGTTGTAGA-1_contig_2": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GTCAAGTGTAAGTGGC-1_contig_5": "GACAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TGGACGCCACACATGT-1_contig_1": "TGGGGGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GCGACCAGTTGGTGGA-1_contig_1": "GAGTTGTCATGTTGCAGTGACTGATCAGTCTCCTCAGGCTCCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGCTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGGTCTAGTCAGAGCCTTGAAAACAGTAATGGAAACACCTATTTGAACTGGTACCTCCAGAAACCAGGCCAGTCTCCACAGCTCCTGATCTACAGGGTTTCCAACCGATTTTCTGGGGTCCTAGACAGGTTCAGTGGTAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTTCTGCCTCCAAGTTACACATGTCCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "AGCATACGTCCAGTGC-1_contig_6": "GCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTG",
        "GCGAGAAAGACACTAA-1_contig_7": "GCTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACAGGCCACTGTCACACCACTTGGGGAAAAATTGGAAATCAGACACACGAGTGTGGCTTTGTTTTCCTGGAGCTCCTCAGGGGAAGGTGGAAACATGGTGAGTGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACCGCCGAAAACATAA",
        "CCGTACTCACGACTCG-1_contig_5": "GGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTGATTTCCAGCTTGGTGCCTCCACCGAACGTCCACGGAA",
        "GTTTCTAAGACAAAGG-1_contig_4": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "GATGAGGTCCCTAACC-1_contig_4": "AACATTTGTCCAATGTCCTCGCCACAGTCCCTGCAGACACTGACTCTAACCATGGGATGGAGCTGGATCTTTCTCTTCCTTCTGTCAGGAACTGCAGGTGTCCACTCTGAGATCCAGCTGCAGCAGTCTGGACCTGAGCTGATGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTAGCTACTACATGCACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGATATATTGATCCTTTCAATGGTGGTACTAGCTACAACCAGAAATTCAAGGGCAAGGCCACATTGACTGTAGACAAATCTTCCAGCACAGCCTACATGCATCTCAGCAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGACCAGCTAACTGGGACGTCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CGGAGCTCACACGCTG-1_contig_2": "TGGGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GCAAACTTCGCTTAGA-1_contig_1": "AAACATATCCTGTGCCATTGTCATTGCAGTCAGGACTCAGCATGGACATGAGGGTTCCTGCTCACGTTTTTGGCTTCTTGTTGCTCTGGTTTCCAGGTACCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCTTATCTGCCTCTCTGGGAGAAAGAGTCAGTCTCACTTGTCGGGCAAGTCAGGAAATTAGTGGTTACTTAAGCTGGCTTCAGCAGAAACCAGATGGAACTATTAAACGCCTGATCTACGCCGCATCCACTTTAGATTCTGGTGTCCCAAAAAGGTTCAGTGGCAGTAGGTCTGGGTCAGATTATTCTCTCACCATCAGCAGCCTTGAGTCTGAAGATTTTGCAGACTATTACTGTCTACAATATGCTAGTTATCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TTCTCAAGTACCAGTT-1_contig_2": "GGAAACCAGTATGAACTGAGCAGCATCAGACAGGCAGGGGAAGCAAGATGGATTCACAGGCCCAGGTTCTTATGTTACTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTAGCTGTGTCAGTTGGAGAGAAGGTTACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTTCTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "CATCAAGTCTGAAAGA-1_contig_5": "GAGGAGAAAAAATGCTGGATGTACACTAAGGATGGCCCCTGACACTCTGCACTGCTACCTCTGGCCCCACCAGACAATGTTCCTGCAGAACCTGTTACCTTACTTGGCAGGGATTTTTGTCAAGGTATCTACTACTGTGTCTACTATCCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GCAGCCAGTGTGCGTC-1_contig_1": "GGGAAGCTGGCTATCACTCAAGGCCATGACCATGTTCTCACTAGCTCTTCTCCTCAGTCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCATGGCTATAGGAGAAAAAGTCACCATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGTACAGATTTTGTTTTTACAATTGAAAACATGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACTTGCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TGTTCCGCACCTCGGA-1_contig_2": "GATGTTGAGTCCATATGATGAGCCCAGCTAATACTGCAGGTGGCTCTAGTCATGCATTCCCCTGCTGATTTGCATGTTACCAGAGCACAGCCCACATCTAAGATTTCTTCAGCTGGTGTTTAGGACAATGGCAGAAAGTCACTCTCAGTGAGGATACACCATCAGCATGAGGGTCCTTGCTGAGCTCCTGGGGCTGCTGCTGTTCTGCTTTTTAGGTGTGAGATGTGACATCCAGATGAACCAGTCTCCATCCAGTCTGTCTGCATCCCTTGGAGACACAATTACCATCACTTGCCATGCCAGTCAGAACATTAATGTTTGGTTAAGCTGGTACCAGCAGAAACCAGGAAATATTCCTAAACTATTGATCTATAAGGCTTCCAACTTGCACACAGGCGTCCCATCAAGGTTTAGTGGCAGTGGATCTGGAACAGGTTTCACATTAACCATCAGCAGCCTGCAGCCTGAAGACATTGCCACTTACTACTGTCAACAGGGTCAAAGTTATCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CTTTGCGTCTTGCAAG-1_contig_4": "GGGAACACATGCCCAATGTCCTCTCCACAGACACTGAACACACTGACTCCAACCATGGGATGGAGCTGGATCTTTCTCTTTCTCTTGTCAGGAACTGGAGGTGTCCTCTCTGAGGTCCAGCTGCAACAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGATACACCTTCACTGACTACTACATGAAGTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGAGATATTAATCCTAACAATGGTGATACTTTCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAACAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGAGAAGGCTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AAAGATGGTTACGACT-1_contig_5": "CTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TAAGCGTTCACTGGGC-1_contig_2": "GGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CAAGAAACACGAAACG-1_contig_2": "AGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CGAGCACGTTCAGTAC-1_contig_4": "AAATAGGGCCTCTTTCTCCTCATGAAACGCAGACCAACCTATCCTTGCAGTTCAGACATAGGAGCTTGGCTCTGGTTCCCAAGACCTCTCACTCACTTCTCAACATGGAGTGGGAACTGAGCTTAATTTTCATTTTTGCTCTTTTAAAAGATGTCCAGTGTGAAGTGCAGCTGTTGGAGACTGGAGGAGGCTTGGTGCAACCTGGGGGGTCACGGGGACTCTCTTGTGAAGGCTCAGGGTTTACTTTTAGTGGCTTCTGGATGAGCTGGGTTCGACAGACACCTGGGAAGACCCTGGAGTGGATTGGAGACATTAATTCTGATGGCAGTGCAATAAACTACGCACCATCCATAAAGGATCGATTCACTATCTTCAGAGACAATGACAAGAGCACCCTGTACCTGCAGATGAGCAATGTGCGATCTGAGGACACAGCCACGTATTTCTGTATGAGATACGGTAGTAGCTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TACGGGCTCACCTCGT-1_contig_1": "GGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "GGGAGATTCACTTCAT-1_contig_1": "ATATACCAGTCAGATTGTGAGCCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GTCTCGTGTCTCCACT-1_contig_6": "GCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAA",
        "CATGACAAGCCAGGAT-1_contig_4": "GGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGAC",
        "CTGTGCTAGTACATGA-1_contig_4": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GCATGATAGACAGGCT-1_contig_6": "GATATATGATCAGTGTCTTCTCCGCTATCCCTGGACACACTGACTCTAACCATGGAATGGAGCTGGGTCTTTCTCTTCTTCCTGTCAGTAACTACAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGACGCTGAGTTGGTGAAACCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGCTACACCTTCACTGACCATGCTATTCACTGGGTGAAGCAGAAGCCTGAACAGGGCCTGGAATGGATTGGATATATTTCTCCCGGAAATGGTGATATTAAGTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACTGCCTACATGCAGCTCAACAGCCTGACATCTGAGGATTCTGCAGTGTATTTCTGTAAAAGATTTGATTACTACGGTAGTAGCAGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GACTGCGGTACAGTGG-1_contig_4": "ATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGAGAGAGGGATGGTCCGTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GATGCTACATGTTCCC-1_contig_4": "TGAGTTTGATTATTTCGTGCCATCTACTCCTGTTGGGTGAATTTGCTTCCTTTAGTTCTAGAGCTTCCAGGTGTGCTGTCAGGCTGCTAGTGTATGCTCTCTCTAGTTTCTTTTTGGAGGCACTCAGAGCTATGAGTTTTCCTCTTAGGACTGCCTTCATTGTGTCCCATAAGTTTGGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GGTGCGTTCTCGATGA-1_contig_8": "TGGGGGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACCTAGCCACCTCGCA",
        "TGCCAAAGTCCAGTTA-1_contig_1": "GGAAATCTGAAATGCATCAGACCAGCATGGGCATCAAGATGGAATCACAGACTCTGGTCTTCATATCCATACTGCTCTGGTTATATGGTGCTGATGGGAACATTGTAATGACCCAATCTCCCAAATCCATGTCCATGTCAGTAGGAGAGAGGGTCACCTTGAGCTGCAAGGCCAGTGAGAATGTGGGTACTTATGTATCCTGGTATCAACAGAAACCAGAGCAGTCTCCTAAACTGCTGATATACGGGGCATCCAACCGGTACACTGGGGTCCCCGATCGCTTCACAGGCAGTGGATCTGCAACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGACCTTGCAGATTATCACTGTGGACAGAGTTACAGCTATCCCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCTCAGAGGAAGGTGGAAACACGGTGAGAGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACCGCCGAAAACATAACACAATGACTAAAACCCAACCCAAAACCTATGGGGCCAGCACCTGGTCC",
        "TTTGTCATCGCTTGTC-1_contig_6": "GGAGCTCTGACAGAGGAGCCAAGCCCTGGATTCCCAGGTCCTCACATTCAGTGATCAGCACTGAACACAGACCACTCACCATGGACTCCAGGCTCAATTTAGTTTTCCTTGTCCTTATTTTAAAAGGTGTCCAGTGTGATGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGCAGCCTGGAGGGTCCCGGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTTTGGAATGCACTGGGTTCGTCAGGCTCCAGAGAAGGGGCTGGAGTGGGTCGCATACATTAGTAGTGGCAGTAGTACCATCTACTATGCAGACACAGTGAAGGGCCGATTCACCATCTCCAGAGACAATCCCAAGAACACCCTGTTCCTGCAAATGACCAGTCTAAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGACTCGATGGTAACTACGACGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGA",
        "TTGGCAATCACATGCA-1_contig_4": "TGGGGAGCTCTGACAGAGGAGGCCAGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTGATCAGCACTGAACACGGACCCCTCACCATGAACTTGGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCAGTGTGAAGTGAAGCTGGTGGAGTCTGGGGGAGGCTTAGTGCAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAACCTCTGGATTCACTTTCAGTGACTATTACATGTATTGGGTTCGCCAGACTCCAGAGAAGAGGCTGGAGTGGGTCGCATACATTAGTAATGGTGGTGGTAGCACCTATTATCCAGACACTGTAAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCCGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTGCAAGACACCTACTATGGTTACGACGACGGGATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "AGGCCGTAGATAGCAT-1_contig_3": "AAATCTGATTGTGTATCCAGTCTAGTTCAATGTGACTTAGGAAGCCCAGTCATATGCAAATCTAGAGAAGACTTTAGAGTAGAAATCTGAGGCTCACCTCACATACCAGCAAGCGAGTGACCAGTTAGTCTTAAGGCACCACTTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCAAGAGAGGGTTACCCCCATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TGGTTCCGTCTCCACT-1_contig_1": "GGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGC",
        "CTTACCGGTGGCCCTA-1_contig_1": "GAGATCACACCATGTGCTGGAGTCAGCCTCACACTGATCACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGATGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGTATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTAATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATGCTGCAACAAACTTAGCAGATGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTATTCCCTCAAGATCAACAGCCTGCAGTCTGAAGATTTTGGGAGTTATTACTGTCAACATTTTTGGGGTACTACGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGG",
        "CCTTCGAAGCACCGTC-1_contig_1": "GGGTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GCGCCAAGTTACTGAC-1_contig_1": "GGGAACTGCATCAGTCAGGCAGGGGGGAGAAAGATGGAATCACAGACCCATGTCCTCATGTTCTTGCTGCTTTGGGTATCTGATACCTGTGGGGACATTGTGATGACCCAGTCTCCATCCTCCCTGGCTGTGACAGCAGGAGAGAAGGTCACTATGAGATGCAAGTCCAGTCAGAGTCTTTTGTGGAGTGTAAACCAAAATAACTACTTATCCTGGTACCAGCAGAAACAAGGGCAGCCTCCTAAACTGCTTATCTATGGGGCATCCATTAGAGAATCTTGGGTCCCTGATCGATTCACAGGAAGTGGATCTGGGACAGACTTCACTCTCACCATTAGCAATGTGCATGCTGAAGACCTAGCAGTTTATTACTGTCAGCACAATCATGGCAGCTTTCTCCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "CTAAGACAGGCGCTCT-1_contig_1": "TCGAGTGCAGTACTATCTTTCTTATATGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GTGCAGCCACAACTGT-1_contig_4": "GGAGCTGAGCTGGGGTGAGCTGAGCTGAGCTGAGCTGGGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAA",
        "CAGATCAGTCTGGTCG-1_contig_3": "CTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTATGAAGAGCCTGGGCCATGAAGAGTTTGCCTAAGTTTTGGGCAGTGAACTCCTTGATGTTCTGGCAGTAAGTGTTAATCTGGCCTGCAATGAGCAGT",
        "TTCTCAAGTACCAGTT-1_contig_6": "GAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGAGACAGACTCTTCTCCACAGTGTCCCCTTCATGTGTAACT",
        "AGCAGCCGTGGTGTAG-1_contig_7": "TCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "AATCCAGAGATGCGAC-1_contig_2": "GGGCTTCCTCGAATAATCAGAACTCTTCACACCAAGTTCTCAGAATGAGGTGCTCTCTTCAGTTCCTGGGGGTGCTTATGTTCTGGATCTCTGGAGTCAGTGGGGATATTGTGATAACCCAGGATGAACTCTCCAATCCTGTCACTTCTGGAGAATCAGTTTCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTATATAAGGATGGGAAGACATACTTGAATTGGTTTCTGCAGAGACCAGGACAATCTCCTCAGCTCCTGATCTATTTGATGTCCACCCGTGCATCAGGAGTCTCAGACCGGTTTAGTGGCAGTGGGTCAGGAACAGATTTCACCCTGGAAATCAGTAGAGTGAAGGCTGAGGATGTGGGTGTGTATTACTGTCAACAACTTGTAGAGTATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCTCAGAGGAAGGTGGAAACACGGTGAGAGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACCGCCGAAAACATAACACAATGACTAAAACCCAACCCAAAACCTATGGGGCCAGCACCTGGTCCT",
        "CTTAGGAAGATAGTCA-1_contig_3": "TTACGGCTTTCTTATATGGGGAACAGCATCAGACAGGCTGGGCAGCAAGATGGAATCACAGACCCAGGTCCTCATGTTTCTTCTGCTCTGGGTATCTGGTGCCTGTGCAGACATTGTGATGACACAGTCTCCATCCTCCCTGGCTATGTCAGTAGGACAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACA",
        "CAGCGACAGTGTTGAA-1_contig_2": "AGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGT",
        "GATCGATGTCCTAGCG-1_contig_4": "GGGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGTGCTACTCCTCTGCCTGCTGACTTTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATCACCTGCACATTCTCTGGTTTCTGATTAACCAGCTATGGTGTACACTGGGAGCGCCATTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGTGGTGTACACACAGACTATAATGCAGCTTTCATATCCAGATTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTGATGACACAGCCATATACTACTGTGCCAGAAGATTACTAGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "TCTCATAGTCTCTTAT-1_contig_7": "GGAGCTCTGACAGAGGAGGCCAGTCCTGGATTTGAGTTCCTCACATTCAGTGATGAGCACTGAACACGGACCCCTCACCATGAATTTCGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCTGTGTGAAGTGAAGCTGGTGGAGTCTGGGGGAGGTTTAGTGCAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATACCATGTCTTGGGTTCGCCAGACTCCAGAGAAGAGGCTGGAGTGGGTCGCATACATTAGTAATGGTGGTGGTAGCACCTACTATCCAGACACTGTAAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACGGCCATGTATTACTGTGCAAGACACTATGGTTACGACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TTAGTTCCAGATTGCT-1_contig_3": "GGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTGATTTCCAGCTTGGTGCCTCCACCGAACGTCC",
        "AGGCCACCATAGACTC-1_contig_4": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "TACTCGCAGGATGTAT-1_contig_4": "GGGACTTACTGTTTTCTCTTTCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGAATTGAAGTCTCAACTATGAAAATCAGCAGTTCTCTGTCAGAGAAGCCCAAGCGCTTCCACGCATGCTTGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GGAAAGCCATACCATG-1_contig_2": "TCTGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CAGCGACAGTGTTGAA-1_contig_1": "GGGAGGACAATGGCAGAAAGTCACTCTCAGTGAGGATACACCATCAGCATGAGGGTCCTTGCTGAGCTCCTGGGGCTGCTGCTGTTCTGCTTTTTAGGTGTGAGATGTGACATCCAGATGAACCAGTCTCCATCCAGTCTGTCTGCATCCCTTGGAGACACAATTACCATCACTTGCCATGCCAGTCAGAACATTAATGTTTGGTTAAGCTGGTACCAGCAGAAACCAGGAAATATTCCTAAACTATTGATCTATAAGGCTTCCAACTTGCACACAGGCGTCCCATCAAGGTTTAGTGGCAGTGGATCTGGAACAGGTTTCACATTAACCATCAGCAGCCTGCAGCCTGAAGACATTGCCACTTACTACTGTCAACAGGGTCAAAGTTATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "ATCGAGTGTCATATCG-1_contig_1": "TGGGAAGCATCCTCTCTTCTAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCCAAAGTGTCAGTACATCTAGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCAAGTATGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATACTGCAACATATTACTGTCAGCACAGTTGGGAGATTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GTTAAGCCAGGTTTCA-1_contig_1": "GGGCCACATCCAGTGCAGTAATCAATGTCAGGTCACAGCAGAAACATGAAGTTTCCTTCTCAACTTCTGCTCTTACTGCTGTTTGGAATCCCAGGCATGATATGTGACATCCAGATGACACAATCTTCATCCTCCTTTTCTGTATCTCTAGGAGACAGAGTCACCATTACTTGCAAGGCAAGTGAGGACATATATAATCGGTTAGCCTGGTATCAGCAGAAACCAGGAAATGCTCCTAGGCTCTTAATATCTGGTGCAACCAGTTTGGAAACTGGGGTTCCTTCAAGATTCAGTGGCAGTGGATCTGGAAAGGATTACACTCTCAGCATTACCAGTCTTCAGACTGAAGATGTTGCTACTTATTACTGTCAACAGTATTGGAGTACTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCAGA",
        "TACTCGCAGGATGTAT-1_contig_2": "GAGTGTGAGGTTTAAGCCTCAGAGTCCCTGTGGTCTCTGACTGGTGCAAGGTTTTGACTAAGCGGAGCACCACAGTGCTAACTGGGGGTTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GTCGTAAAGACACGAC-1_contig_1": "GAGGCTGATCATATCCAGTGCAGCAATCAGTTCCTGCCAGGACACAGTTTAGATATGAGGTTCCAGGTTCAGGTTCTGGGGCTCCTTCTGCTCTGGATATCAGGTGCCCAGTGTGATGTCCAGATAACCCAGTCTCCATCTTATCTTGCTGCATCTCCTGGAGAAACCATTACTATTAATTGCAGGGCAAGTAAGAGCATTAGCAAATATTTAGCCTGGTATCAAGAGAAACCTGGGAAAACTAATAAGCTTCTTATCTACTCTGGATCCACTTTGCAATCTGGAATTCCATCAAGGTTCAGTGGCAGTGGATCTGGTACAGATTTCACTCTCACCATCAGTAGCCTGGAGCCTGAAGATTTTGCAATGTATTACTGTCAACAGCATAATGAATACCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GTGTGCGCAGTAGAGC-1_contig_2": "AGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTGATTTCCAGCTTGGTGCCTCCACC",
        "GCTCTGTCAGGTTTCA-1_contig_4": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCC",
        "TATGCCCAGGCCCTCA-1_contig_1": "TGGGGAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCTTCAGTCATAATGTCCAGAGGACAAATTGTTCTCTCCCAGTCTCCAGCAATCCTGTCTGCATCTCCAGGGGAGAAGGTCACAATGACTTGCAGGGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAAACCCTGGATTTATGCCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGAGTGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCACGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCA",
        "AGCATACGTCCAGTGC-1_contig_2": "GATGCTTGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GTCAAGTAGGCATTGG-1_contig_1": "GAGATCACACCATGTGCTGGAGTCAGCCTCACACTGATCACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGATGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGTATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTAATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATGCTGCAACAAACTTAGCAGATGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTATTCCCTCAAGATCAACAGCCTGCAGTCTGAAGATTTTGGGAGTTATTACTGTCAACATTTTTGGGGTACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "TATCTCAGTATCGCAT-1_contig_5": "GGGGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGTGCTGCTCCTCTGCCTGGTGACATTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATAACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGAGGTGGAAGCACAGACTACAATGCAGCTTTCATGTCCAGACTGAGCATCACCAAGGACAACTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTGATGACACTGCCATATACTACTGTGCCAAAATGGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GACTAACCATGCCTTC-1_contig_3": "GGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCA",
        "ACTGATGCAGCAGTTT-1_contig_4": "CTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GCATGCGGTATATGAG-1_contig_3": "GGGGATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCC",
        "TGGTTAGAGAGTACAT-1_contig_1": "GAAAACATGAGATCACTGTTCTCTTTACAGTTACTGAGCACACAGGACCTCACCATGGGATGGAGCTGTATCATGCTCTTCTTGGCAGCAACAGCTACAGGTGTCCACTCCCCGGTCCAACTGCAGCAGCCTGGGGCTGAGCTTGTGAAGCCTGGGGCTTCAGTAAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTGGATGCACTGGGTGAAGCAGAGGCCTGGACGAGGCCTTGAGTGGATTGGAAGGATTGATCCTAATAGTGGTGGTACTAAGTACAATGAGAAGTTCAAGAGCAAGGCCACACTGACTGTAGACAAACCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTACAAGAGGGGGCTACGGTAGTGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGTTCTGAGCACAGGT-1_contig_1": "GGCATATGAAATAATTTTATAACAGCCCAGGCTTCTTTAAGGGCAGCTGCCAGGAGCCTAAGAAGCATCCTCTCATCTAGTTCTCAGAGATGGAGACAGACACAATCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGCTCCACTGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAAGGCCAGCCAAAGTGTTGATTATGATGGTGATAGTTATATGAACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAATCTAGAATCTGGGATCCCAGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "ACGAGCCCACAAGTAA-1_contig_1": "GCCATATGATGAGCCCAGCTAATACTGCAGGTGGCTCTAGTCATGCATTCCCCTGCTGATTTGCATGTTACCAGAGCACAGCCCACATCTAAGATTTCTTCAGCTGGTGTTTAGGACAATGGCAGAAAGTCACTCTCAGTGAGGATACACCATCAGCATGAGGGTCCTTGCTGAGCTCCTGGGGCTGCTGCTGTTCTGCTTTTTAGGTGTGAGATGTGACATCCAGATGAACCAGTCTCCATCCAGTCTGTCTGCATCCCTTGGAGACACAATTACCATCACTTGCCATGCCAGTCAGAACATTAATGTTTGGTTAAGCTGGTACCAGCAGAAACCAGGAAATATTCCTAAACTATTGATCTATAAGGCTTCCAACTTGCACACAGGCGTCCCATCAAGGTTTAGTGGCAGTGGATCTGGAACAGGTTTCACATTAACCATCAGCAGCCTGCAGCCTGAAGACATTGCCTCTTACTACTGTCAACAGGGTCAAAGTTATCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GAACGGAGTCGGGTCT-1_contig_1": "TGGGGGAAATGCATCAGACCAGCATGGGCATCAAGATGGAATCACAGACTCTGGTCTTCATATCCATACTGCTCTGGTTATATGGTGCTGATGGGAACATTGTAATGACCCAATCTCCCAAATCCATGTCCATGTCAGTAGGAGAGAGGGTCACCTTGAGCTGCAAGGCCAGTGAGAATGTGGGTACTTATGTATCCTGGTATCAACAGAAACCAGAGCAGTCTCCTAAACTGCTGATATACGGGGCATCCAACCGGTACACTGGGGTCCCCGATCGCTTCACAGGCAGTGGATCTGCAACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGACCTTGCAGATTATCACTGTGGACAGAGTTACAGCTATCCTCCCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCT",
        "CCGTACTCACGACTCG-1_contig_6": "AACCGCATATGATCAGTAACCTCTTCACAGTCACTGAAAACACTGACTCTAATCATGGAATGTAACTGGATACTTCCTTTTATTCTGTCAGTAACTTCAGGTGTCTACTCACAGGTTCAGCTCCAGCAGTCTGGGGCTGAGCTGGCAAGACCTGGGGCTTCAGTGAAGTTGTCCTGCAAGGCTTCTGGCTACACCTTTACTAGCTACTGGATGCAGTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATTGGGGCTATTTATCCTGGAGATGGTGATACTAGGTACACTCAGAAGTTCAAGGGCAAGGCCACATTGACTGCAGATAAATCCTCCAGCACAGCCTACATGCAACTCAGCAGCTTGGCATCTGAGGACTCTGCGGTCTATTACTGTGCACCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GTGCTTCCAGGTCGTC-1_contig_3": "GGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "ATCCGAAAGGACAGCT-1_contig_5": "GGGGATCCTCTTCTCATAGAGCCTCCATCAGACCATGGCTGTCCTGGCACTGCTCCTCTGCCTGGTGACATTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACTGTCTCAGGGTTCTCATTAACCAGCTATGGTGTAAGCTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGGTGACGGGAGCACAAATTATCATTCAGCTCTCATATCCAGACTGAGCATCAGCAAGGATAACTCCAAGAGCCAAGTTTTCTTAAAACTGAACAGTCTGCAAACTGATGACACAGCCACGTACTACTGTGCCAAAGGGGGCTACTATGGTTACTCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TTGCCGTCATCCAACA-1_contig_1": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CGAGCCATCTGCCAGG-1_contig_9": "GGACATACCAGCAAGGGAGTGACCAGTTAGTCTTAAGGCACCATTGAGCCCAAGTCTTAGACATCATGGATTGGCTGTGGAACTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCAACACTGGAGAGCCAACATATGCTGAAGAGTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCAAGTGGTAACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GCTCTGTCAGGTTTCA-1_contig_1": "TGGGCTTGACTACTCAAGACTTTTTGTATCAAGTTCTCAGAATGAGGTGCCTAGCTGAGTTCCTGGGGCTGCTTGTGCTCTGGATCCCTGGAGCCATTGGGGATATTGTGATGACTCAGGCTGCACCCTCTGTACCTGTCACTCCTGGAGAGTCAGTATCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTGCATAGTAATGGCAACACTTACTTGTATTGGTTCCTGCAGAGGCCAGGCCAGTCTCCTCAGCTCCTGATATATCGGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTGGCAGTGGGTCAGGAACTGCTTTCACACTGAGAATCAGTAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTATGCAACATCTAGAATATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TAGTGGTCATCACAAC-1_contig_5": "ACAGAATTGAGAAAGAATAGAGACCTGCAGTTGAGGCCAGCAGGTCGGCTGGACTAACTCTCCAGCCACAGTAATGACCCAGACAGAGAAGGCCAGACTCATAAAGCTTGCTGAGCAAAATTAAGGGAACAAGGTTGAGAGCCCTAGTAAGCGAGGCTCTAAAAAGCATGGCTGAGCTGAGATGGGTGGGCTTCTCTGAGCGCTTCTAAAATGCGCTAAACTGAGGTGATTACTCTGAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "ATCCACCCACCGTTGG-1_contig_6": "GGGAGCATATGATCAGTGTCCTCTCCAAAGTCCTTGAACATAGACTCTAACCATGGAATGGACCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGAGCTGAGCTGATGAAGCCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTACTGGCTACACATTCAGTAGCTACTGGATAGAGTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGAGATTTTACCTGGAAGTGGTAGTACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACATTCACTGCAGATACATCCTCCAACACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCCGTCTATTACTGTGCAAGAGCGCAATTTATTACTACGGTAGTAGAGGGCTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCGTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GTAACTGGTAGGGTAC-1_contig_5": "TCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ACACCCTAGGAATCGC-1_contig_1": "TGGGGATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGAGATAGGGACTACGGTTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCCTTGTCAAAGGCTACTTCCCTGAGCCGGTAACTGTAAAATGGAACT",
        "TGATTTCTCTTTACAC-1_contig_3": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TGCACCTGTTCCGGCA-1_contig_1": "TCTTGCGATTTGCATATTACATTTTCAGTAACCACAAATATCTCACAGTTGGTTTAAAGCAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCTTCAGTCATAATGTCCAGAGGACAAATTGTTCTCTCCCAGTCTCCAGCAATCCTGTCTGCATCTCCAGGGGAGAAGGTCACAATGACTTGCAGGGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAAACCCTGGATTTATGCCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGAGTGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCACCCACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "ACCGTAATCTCTGCTG-1_contig_3": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "ACACCAAGTTCGGGCT-1_contig_2": "GGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "ACTGCTCGTACCGCTG-1_contig_10": "CTATTAAGACTTTGACATTCTGGTCAAAACGGCTTCACAAATCTTTTTCAAGACCACTTTCTGAGTATTCATTTTAGGAGAAATACTTTTTTTTTAAATGAATGCAATTATCTAGACTTATTTCGGTTGAACATGCTGGTTGGTGGTTGAGAGGACACTCAGTCAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ACTGAACGTACCGGCT-1_contig_7": "GGGGGGGATCCTGTCCTGAGTTCCCCAATCTTCACATTCAGAAATCAGCACTCAGTCCTGTCACTATGAAGTTGTGGTTAAACTGGGTTTTTCTTTTAACACTTTTACATGGTATCCAGTGTGAGGTGAAGCTGGTGGAATCTGGAGGAGGCTTGGTACAGCCTGGGGGTTCTCTGAGACTCTCCTGTGCAACTTCTGGGTTCACCTTCAGTGATTTCTACATGGAGTGGGTCCGCCAGCCTCCAGGGAAGAGACTGGAGTGGATTGCTGCAAGTAGAAACAAAGCTAATGATTATACAACAGAGTACAGTGCATCTGTGAAGGGTCGGTTCATCGTCTCCAGAGACACTTCCCAAAGCATCCTCTACCTTCAGATGAATGCCCTGAGAGCTGAGGACACTGCCATTTATTACTGTGCAAGAGATGCCTGGGACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGTCACTAGGGTGTGT-1_contig_3": "GGATCCTCTTCTCATAGAGCATCCATCAGAGCATGGCTGTCCTGGGGCTGCTTCTCTGCCTGGTGACTTTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCGTCTCAGGGTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGTAGTGATATGGAGTGATGGAAGCACAACCTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTCCAAACTGATGACACAGCCATGTACTACTGTGCCAGAAATGAGATGGGTATGATTACGCTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACAC",
        "CCTCTGAAGAGGTAGA-1_contig_1": "GAATGCATGCCATCAAGCTGCATTGGAAGCTGGCTATCACTCAAGGCCATGACCATGCTCTCACTAGCTCCTCTCCTCAGCCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCGTGGCTACAGGAGAAAAAGTCACTATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGCACAGATTTTGTTTTTACAATTGAAAACACGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACATGCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCATCCAGTGAGCAG",
        "GATGAGGTCCCTAACC-1_contig_1": "GGAGTCAGATCACACCATGTGCTGGAGTCAGCCTCACACTGATCACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGATGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGTATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTAATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATGCTGCAACAAACTTAGCAGATGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTATTCCCTCAAGATCAACAGCCTGCAGTCTGAAGATTTTGGGAGTTATTACTGTCAACATTTTTGGGGTACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TTCTTAGGTGATAAAC-1_contig_6": "GCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAG",
        "CTGCCTATCAAACAAG-1_contig_2": "TCTACTCGTTTTCTTATATGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGCCTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "ATCCACCCACCGTTGG-1_contig_2": "GATAAGAAGGTCTTTCTCAGCTCTGTGTCATAATCCACAGAAGCTCAGGGAAAATTTGAAGATGGTGTCCACTTCTCAGCTCCTTGGACTTTTGCTTTTCTGGACTTCAGCCTCCAGATGTGACATTGTGATGACTCAGTCTCCAGCCACCCTGTCTGTGACTCCAGGAGATAGAGTCTCTCTTTCCTGCAGGGCCAGCCAGAGTATTAGCGACTACTTACACTGGTATCAACAAAAATCACATGAGTCTCCAAGGCTTCTCATCAAATATGCTTCCCAATCCATCTCTGGGATCCCCTCCAGGTTCAGTGGCAGTGGATCAGGGTCAGATTTCACTCTCAGTATCAACAGTGTGGAACCTGAAGATGTTGGAGTGTATTACTGTCAAAATGGTCACAGCTTTCCTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "TCTCATAGTCTCTTAT-1_contig_2": "TGGGGGAAATACATCAGACCAGCATGGGCATCAAGATGGAGACACATTCTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGAAGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGGGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAACTACTGATTTACTGGGCATCCACCCGGCACACTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATTAGCAATGTGCAGTCTGAAGACTTGGCAGATTATTTCTGTCAGCAATATAGCAGCTATCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CTAGCCTCATTCCTCG-1_contig_2": "TGGGAGCTGCCAGGAGCCTAAGAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGTTATGGGTACTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCAAAAGTGTCAGTACATCTGGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCACAGTAGGGAGCTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAG",
        "CAGTAACCAAACCTAC-1_contig_1": "TGGGGGAAATGCATCACACCAGCATGGGCATCAAAATGGAGTCACAGATTCAGGTCTTTGTATTCGTGTTTCTCTGGTTGTCTGGTGTTGACGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGAGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGACAATCTCCTAAACTACTGATTTACTCGGCATCCTACCGGTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATCTGGGACGGATTTCACTTTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAACATTATAGTACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAA",
        "GCATGATAGGGCATGT-1_contig_2": "ACCAGTCACACATATCCTGTACCGTTGTCATTGCAGTCAGGACTCAGCATGGACATGAGGGCCCCTGCTCAGTTTTTTGGGATCTTGTTGCTCTGGTTTCCAGGTATCAGATGTGACATCAAGATGACCCAGTCTCCATCCTCCATGTATGCATCGCTGGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAAAAGCTATTTAAGCTGGTACCAGCAGAAACCATGGAAATCTCCTAAGACCCTGATCTATTATGCAACAAGCTTGGCAGATGGGGTCCCATCAAGATTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTAACCATCAGCAGCCTGGAGTCTGACGATACAGCAACTTATTACTGTCTACAGCATGGTGAGAGCCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CTTAGGAAGTGACATA-1_contig_6": "GAGTGTCCTCTCCAAAGTCCTTGAACATAGACTCTAACCATGGAATGGACCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGAGCTGAGCTGATGAAGCCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTACTGGCTACACATTCAGTAGCTACTGGATAGAGTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGAGATTTTACCTGGAAGTGGTAGTACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACATTCACTGCAGATACATCCTCCAACACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCCGTCTATTACTGTGCAAGAAGGTATGATGGTTACTACGATAACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GCACTCTAGGTCATCT-1_contig_2": "TGGGAGACAGGCAGGGGAAGCAAGATGGATTCACAGGCCCAGGTTCTTATGTTACTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTAGCTGTGTCAGTTGGAGAGAAGGTTACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTTACTCTTACCATCAGCAGTGTACAAGCTGAAGACCTGGCAGTTTATTACTGTCATCAATACCTCTCCTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TGGGAAGCACGTCAGC-1_contig_2": "GGGGAAATACATCAGACCAGCATGGGCTTCAAGATGGAGTCTCATACTCAGGCCTTTGTATTCGCGTTTCTCTGGTTGTCTGGTGTTGATGGAGACATTGTGATGACCCAGTCTCAAAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGAATGTGGGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGACAATCTCCTAAACTACTGATTTACTCGGCATCCAATCGGTACACTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAATATGCAGTCTGAAGACCTGGCAGATTATTTCTGCCAGCAATATAGCAGCTATCCTCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TTATGCTGTCGAGATG-1_contig_7": "GGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGTGCTGCTCCTCTGCCTGGTGACATTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGCAGTCAGGACCTAGCCTAGTGCAGCCCTCACAGAGCCTGTCCATAACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGAGGTGGAAGCACAGACTACAATGCAGCTTTCATGTCCAGACTGAGCATCACCAAGGACAACTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTGATGACACTGCCATATACTACTGTGCCAAAAATTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TAGGCATGTTACGGAG-1_contig_10": "GGGGAGGGACCCCACCTAGGGGCCAGACACACAGTCATGCAGCCCCCTCACCGCTCAGTCTGACCCATCCACAGGCAACCACACACAGAGAAAAACAGCTTCCAGCCAAGGCTTCGCCCAGTCCTAAGCTCTCTACCATAGGGAAGATAGCCTTTAGCTATTATGAGAATGGATCTTCAGTTGGGACCACCTGGCCACAGGAGGCCTGGCTGTTCCCCTATGAAGGACACTCAACAGCATTGAGCAGAGTCTGCGAGAAATCCCACCATCTACCCACTGACACTCCCACCAGCTCTGTCAAGTGACCCAGTGATAATCGGCTGCCTGATTCACGATTACTTCCCTTCCGGCACGATGAATGTGACCTGGGGAAAGAGTGGGAAGGATATAACCACCGTAAACTTCCCACCTGCCCTGGCCTCTGGGGGACGGTACACCATGAGCAGCCAGT",
        "CAGCTAACAAACGCGA-1_contig_1": "GGACTACTCAAGACTTTTTGTATCAAGTTCTCAGAATGAGGTGCCTAGCTGAGTTCCTGGGGCTGCTTGTGCTCTGGATCCCTGGAGCCATTGGGGATATTGTGATGACTCAGGCTGCACCCTCTGTACCTGTCACTCCTGGAGAGTCAGTATCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTGCATAGTAATGGCAACACTTACTTGTATTGGTTCCTGCAGAGGCCAGGCCAGTCTCCTCAGCTCCTGATATATCGGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTGGCAGTGGGTCAGGAACTGCTTTCACACTGAGAATCAGTAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTATGCAACATCTAGAATATCCTTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CGTCACTCAAGAGGCT-1_contig_5": "GAACAGACACACAAACCTGGACTCACAAGTTTTTCTCTTCAGTGACAGACACAGACATAGAACATTCACGATGTACTTGGGACTGAACTATGTATTCATAGTTTTTCTCTTAAATGGTGTCCAGAGTGAAGTGAAGCTTGAGGAGTCTGGAGGAGGCTTGGTGCAACCTGGAGGATCCATGAAACTCTCTTGTGCTGCCTCTGGATTCACTTTTAGTGACGCCTGGATGGACTGGGTCCGCCAGTCTCCAGAGAAGGGGCTTGAGTGGGTTGCTGAAATTAGAAGCAAAGCTAATAATCATGCAACATACTATACTGAGTCTGTGAAAGGGAGGTTCACCATCTCAAGAGATGATTCCAAAAGTAGTGTCTACCTGCAAATGAACAGCTTAAGAGCTGAAGACACTGGCATTTATTACTGTACCAGGCGGTCCTATGACCCTTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GCTGCAGAGCACCGCT-1_contig_4": "TGGGAAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GTAGTCATCTCCTATA-1_contig_2": "GGGCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GTCTTCGTCTAACTGG-1_contig_1": "GGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GTGTGCGCAGTAGAGC-1_contig_3": "GAGACTCTGGGGTTTTTGTCGGGTATAGAGGAAAAATCCACTATTGTGATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCACCGAGTGGTGTGACAGTGGCCTG",
        "AAAGTAGCAAAGTGCG-1_contig_1": "GGGTTTCAGTGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "AGCATACGTAGCGTAG-1_contig_3": "GAGTTTGATTCAGCTCAGCTGTGAAAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAG",
        "CGTGTCTAGCTGATAA-1_contig_1": "GCTAAAATACATCAGACCAGCATGGGCATCAAGATGGAGTTTCAGACCCAGGTCTTTGTATTCGTGTTGCTCTGGTTGTCTGGTGTTGATGGAGACATTGTGATGACCCAGTCTCAAAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGAATGTTCGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGGCAGTCTCCTAAAGCACTGATTTACTTGGCATCCAACCGGCACACTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATTAGCAATGTGCAATCTGAAGACCTGGCAGATTATTTCTGTCTGCAACATTGGAATTATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GCTGGGTTCGTGGACC-1_contig_1": "GCCCCCTTAGAGTATTATAAGACGATCTCTATCAGGACAGTGTCATGAGCCACACAAACTCAGGGAAAGCTCGAAGATGGTTTTCACACCTCAGATACTTGGACTTATGCTTTTTTGGATTTCAGCCTCCAGAGGTGATATTGTGCTAACTCAGTCTCCAGCCACCCTGTCTGTGACTCCAGGAGATAGCGTCAGTCTTTCCTGCAGGGCCAGCCAAAGTATTAGCAACAACCTACACTGGTATCAACAAAAATCACATGAGTCTCCAAGGCTTCTCATCAAGTATGCTTCCCAGTCCATCTCTGGGATCCCCTCCAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACTCTCAGTATCAACAGTGTGGAGACTGAAGATTTTGGAATGTATTTCTGTCAACAGAGTAACAGCTGGCCTCAGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GCTCTGTCAGGTTTCA-1_contig_7": "TGGGGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGGGCTGCTTCTCTGCCTGGTGACTTTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCATCTCAGGGTTCTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGTAGTGATATGGAGTGATGGAAGCACAACCTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTGATGACACAGCCATATACTACTGTGCCAGAAGGGGGTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TTGACTTCAAGAAGAG-1_contig_2": "TGGGGATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "CCGGTAGTCACTCTTA-1_contig_2": "GGGGATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGTGCAAGTCAGGGCATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTATTACACATCAAGTTTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGGACAGATTATTCTCTCACCATCAGCAACCTGGAACCTGAAGATATTGCCACTTACTATTGTCAGCAGTATAGTAAGCTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTC",
        "TGCCAAAGTCCAGTTA-1_contig_5": "TGGGGAAAGTACTTATGATAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGGAAAAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCTAGGGGAACGGGTCACCATGACCTGCACTGCCAGCTCAAGTGTAAGTTCCAGTTACTTGCACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAAACTCTGGATTTATAGCACATCCAACCTGGCTTCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCACCAGTATCATCGTTCCCCACCCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCTCAGAGGAAGGTGGAAACACGGTGAGAGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACCGCCGAAAACATAACACAATGACTAAAACCCAACCCAAAACCTATGGGGCCAGCACCTGGTCC",
        "AAGGAGCGTCATATGC-1_contig_6": "GAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TACAGTGGTTATGTGC-1_contig_3": "GACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTATTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAGCTGGAGTGGATTGGCTACATAATCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGCTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGTGGTTACGACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TATGCCCAGGCCCTCA-1_contig_2": "AGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATC",
        "ACGATACCATCGGACC-1_contig_1": "GGGGATAAGAAGATCTTTCTCAGGGATGTGTCATGGTCCACACAAACTCAGGGAAAGTTTGAAGATGGTATCCACACCTCAGTTCCTTGTATTTTTGCTTTTCTGGATTCCAGCCTCCAGAGGTGACATCTTGCTGACTCAGTCTCCAGCCATCCTGTCTGTGAGTCCAGGAGAAAGAGTCAGTTTCTCCTGCAGGGCCAGTCAGAGCATTGGCACAAGCATACACTGGTATCAGCAAAGAACAAATGGTTCTCCAAGGCTTCTCATAAAGTATGCTTCTGAGTCTATCTCTGGGATCCCTTCCAGGTTTAGTGGCAGTGGATCAGGGACAGATTTTACTCTTAGCATCAACAGTGTGGAGTCTGAAGATATTGCAGATTATTACTGTCAACAAAGTAATAGCTGGCCAACTACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GGCTGGTCATCACCCT-1_contig_4": "GATCCTGTCCTGAGTTCCCCAATCTTCACATTCAGAAATCAGCACTCAGTCCTGTCACTATGAAGTTGTGGTTAAACTGGGTTTTTCTTTTAACACTTTTACATGGTATCCAGTGTGAGGTGAAGCTGGTGGAATCTGGAGGAGGCTTGGTACAGCCTGGGGGTTCTCTGAGACTCTCCTGTGCAACTTCTGGGTTCACCTTCAGTGATTTCTACATGGAGTGGGTCCGCCAGCCTCCAGGGAAGAGACTGGAGTGGATTGCTGCAAGTAGAAACAAAGCTAATGATTATACAACAGAGTACAGTGCATCTGTGAAGGGTCGGTTCATCGTCTCCAGAGACACTTCCCAAAGCATCCTCTACCTTCAGATGAATGCCCTGAGAGCTGAGGACACTGCCATTTATTACTGTGCAAGAGATGCCCTTTACTACGGTGTCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GGGACCTCAGCCACCA-1_contig_2": "GGTGAAAGACTCGCCAGAGCTAGACCTCCTTGGTGTGTGTTAACTCAGGAAGAAAAAGACGACATAAAACAACAGCCAGGACTCAGCATGGACATGAGGACCCCTGCTCAGTTTCTTGGAATCTTGTTGCTCTGGTTTCCAGGTATCAAATGTGACATCAAGATGACCCAGTCTCCATCTTCCATGTATGCATCTCTAGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAATAGCTATTTAAGCTGGTTCCAGCAGAAACCAGGGAAATCTCCTAAGACCCTGATCTATCGTGCAAACAGATTGGTAGATGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTCACCATCAGCAGCCTGGAGTATGAAGATATGGGAATTTATTATTGTCTACAGTATGATGAGTTTCCACGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TCATTTGAGGGCTTCC-1_contig_1": "GAAACCTACCTATGATCAGTGTGCTCTCCACAGTCTCTGAAGACACTGACTCTAACCATGGGATGGAGATGGATCTTTCTTTTCCTCCTGTCAGGAACTGCAGGTGTCCATTGCCAGGTTCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTTAGTGAAGATATCCTGCAAGGCTTCTGGTTACACCTTCACAAGCTACGATATAAACTGGGTGAAGCAGAGGCCTGGACAGGGACTTGAGTGGATTGGATGGATTTATCCTGGAGATGGTAGTACTAAGTACAATGAGAAATTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACTTCTGAGAACTCTGCAGTCTATTTCTGTGCAAGAGGTCACTATGATTACGATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AACTGGTAGGACTGGT-1_contig_1": "GGGGATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAAC",
        "TAGTGGTTCAGCGATT-1_contig_3": "TCTTTTAACTTATTGAGTTCAACCTTTTAATTTTAGCTTGAGTAGTTCTAGTTTCCCCAAACTTAAGTTTATCGACTTCTAAAATGTATTTAGAATTCATTTTCAAAATTAGGTTATGTAAGAAATTGAAGGACTTTAGTGTCTTTAATTTCTAATATATTTAGAAAACTTCTTAAAATTACTCTATTATTCTTCCCTCTGATTATTGGTCTCCATTCAATTCTTTTCCAATACCCGAAGCATTTACAGTGACTTTGTTCATGATCTTTTTTAGTTGTTTGTTTTGCCTTACTATTAAGACTTTGACATTCTGGTCAAAACGGCTTCACAAATCTTTTTCAAGACCACTTTCTGAGTATTCATTTTAGGAGAAATACTTTTTTTTTAAATGAATGCAATTATCTAGACTTATTTCGGTTGAACATGCTGGTTGGTGGTTGAGAGGACACTCAGTCAGT",
        "GATTCAGCACTCTGTC-1_contig_1": "GGGGAGCTGCCAGGAAGCTAAGAAGCATCCTCTCTTCTAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCCAAAGTGTCAGTACATCTAGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCAAGTATGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATACTGCAACATATTACTGTCAGCACAGTTGGGAGATTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCA",
        "CTGCGGATCACCGTAA-1_contig_4": "GGGGGGAGCTATGGGGGCCAGGCTGGACAGCTCTGGGGGTAGCTAGGAGTCGGATGCACAGAGGGAAGGAAGAAGGAGGCACATCAGGGTCTGAGCAGTGGAAGCCAGCAGCTGTGGTTCGAAGAAGGATGACCCTGAAGCTGAGAGGAGACCTGGGTTGGAGAAAAGACAGCCTGACAGCCAGTTTTCTCATGAATGGCAGCATATCCTAAACTCAAGGGCTCATGGCTTCTAGAAAGAAATGGGGAACTCATTGGAATCTGGAGACACTGGCACCTGTGACATAGGAGAAGGCAGGGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCC",
        "AAAGTAGCAAAGTGCG-1_contig_5": "GATCAACATTTGTCCAATGTCCTCGCCACAGTCCCTGCAGACACTGACTCTAACCATGGGATGGAGCTGGATCTTTCTCTTCCTTCTGTCAGGAACTGCAGGTGTCCACTCTGAGATCCAGCTGCAGCAGTCTGGACCTGAGCTGATGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTAGCTACTACATGCACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGATATATTGATCCTTTCAATGGTGGTACTAGCTACAACCAGAAATTCAAGGGCAAGGCCACATTGACTGTAGACAAATCTTCCAGCACAGCCTACATGCATCTCAGCAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGATCGAAGAAGTATGGTAACTACGTAGGAACTGGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ATTACTCGTACTCAAC-1_contig_4": "AGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GCTCTGTGTCTAGCGC-1_contig_2": "GATGATCAGTGTCCTCTCCACAGTTCCTGAAGACACTGACTCTAACCATGGGATGGAGCTGGATCTTTCTTTTCCTCCTGTCAGGAACTGCAGGTGTCCATTGCCAGATCCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGCTACACCTTCACTGACTACTATATAAACTGGGTGAAGCAGAAGCCTGGACAGGGACTTGAGTGGATTGGATGGATTTATCCTGGAAGCGGTAATACTAAGTACAATGAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACACATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACACTGCTGTCTATTTCTGTGCAAGATGGCGGTATTACTACGGTAGTAGCTACGTTACGTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACCA",
        "AGCTCCTTCCGCTGTT-1_contig_6": "AGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CGGCTAGCATATGAGA-1_contig_2": "CGGCTAGCATATGAGATCCATGTGTTTTCTTGGGATCCTCTTCTCATAGAGCATCCATCAGAGCATGGCTGTCCTGGGGCTGCTTCTCTGCCTGGTGACTTTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCGTCTCAGGGTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGTAGTGATATGGAGTGATGGAAGCACAACCTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTCCAAACTGATGACACAGCCATGTACTACTGTGCCAGAAATAGGGATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "ATCATCTGTGCTCTTC-1_contig_4": "GATCAGACAGGCAGGGGAAGCAAGATGGATTCACAGGCCCAGGTTCTTATGTTACTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTAGCTGTGTCAGTTGGAGAGAAGGTTACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CCTTTCTAGATGTGGC-1_contig_3": "GGTTTTCTCTTTCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGAATTGAAGTCTCAACTATGAAAATCAGCAGTTCTCTGTCAGAGAAGCCCAAGCGCTTCCACGCATGCTTGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGG",
        "TCAATCTAGCACAGGT-1_contig_5": "GGGGAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGC",
        "CAAGATCCAGCGTAAG-1_contig_1": "TTTGAAATCTTTCTTATATGGGAGACAGACAGGCAGTGGGAGCAAGATGGATTCACAGGCCCAGGTTCTTATATTGCTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTGGCTGTGTCAGCAGGAGAGAAGGTCACTATGAGCTGCAAATCCAGTCAGAGTCTGCTCAACAGTAGAACCCGAAAGAACTACTTGGCTTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATCTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGCAAGCAATCTTATAATCTGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "TGCTACCGTACATCCA-1_contig_2": "TGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTGATTTCCAGCTTGGTGCCTCCACCGAACGTCCACGGAGTACTATA",
        "CGGCTAGCACGACTCG-1_contig_3": "GACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGCCC",
        "CTGTGCTAGTACATGA-1_contig_10": "TGGGGAACCTAAGATCAGTGGCCTCTCCACAGTCCCTGAACACACTGACTCAAACCATGGACTGGAGTTGGGTCTTTCTCTTCCTCCTGTCAGTAAATGAAGGTGTCTACTGTCAGGTCCAGCTGCAGCAGTCTGGAGATGATCTGGTAAAGCCTGGGGCCTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTGGATTAACTGGATAAAACAGAGGCCTGGACAGGGCCTTGAGTGGATAGGACGTATTGCTCCTGGAAGTGGTAGTACTTACTACAATGAAATGTTCAAGGGCAAGGCAACACTGACTGTAGACACATCCTCCAGCACAGCCTACATTCAGCTCAGCAGCCTGTCATCTGAGGACTCTGCTGTCTATTTCTGTGCAAGAGGGGGGCCTAGACTAACTGGATACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGATGTAGTAACGCGA-1_contig_4": "GACAACCTATGATCAGTGTCCTCTCTACACAGTCCCTGACAACACTGACTCTAACCATGGGATGGAGCCGGATCTTTCTCTTCCTCCTGTCAATAATTGCAGGTGTCCATTGCCAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAGGATATCCTGCAAGGCTTCTGGCTACACCTTCACAAGCTACTATATACACTGGGTGAAGCAGAGGCCTGGACAGGGACTTGAGTGGATTGGATGGATTTATCCTGGAAATGTTAATACTAAGTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACCTCTGAGGACTCTGCGGTCTATTTCTGTGCAAGATGGGCTTACTACGGTAGTAGCCGCCGTGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "ACACCCTAGGAATCGC-1_contig_3": "TGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCTCCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TAGACCATCGCGCCAA-1_contig_1": "TGGGGAATCAACCTCCTTAAAATGAATTTGCCTGTTCATCTCTTGGTGCTTCTGTTGTTCTGGATTCCTGCTTCCAGAGGTGATGTTGTGGTGACTCAAACTCCACTCTCCCTGCCTGTCAGCTTTGGAGATCAAGTTTCTATCTCTTGCAGGTCTAGTCAGAGTCTTGCAAACAGTTATGGGAACACCTATTTGTCTTGGTACCTGCACAAGCCTGGCCAGTCTCCACAGCTCCTCATCTATGGGATTTCCAACAGATTTTCTGGGGTGCCAGACAGGTTCAGTGGCAGTGGTTCAGGGACAGATTTCACACTCAAGATCAGCACAATAAAGCCTGAGGACTTGGGAATGTATTACTGCTTACAAGGTACACATCAGCCATGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CATCGGGCACTGTGTA-1_contig_3": "CCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCA",
        "GTCTCGTGTCTCCACT-1_contig_1": "GAGATTGTGAGCCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAG",
        "TTGACTTCAAGAAGAG-1_contig_3": "GCCTTTCTCTTAGAGTTTTATAAGAAGGTCTTTCTCAGCTCTGTGTCATAATCCACAGAAGCTCAGGGAAAATTTGAAGATGGTGTCCACTTCTCAGCTCCTTGGACTTTTGCTTTTCTGGACTTCAGCCTCCAGATGTGACATTGTGATGACTCAGTCTCCAGCCACCCTGTCTGTGACTCCAGGAGATAGAGTCTCTCTTTCCTGCAGGGCCAGCCAGAGTATTAGCGACTACTTACACTGGTATCAACAAAAATCACATGAGTCTCCAAGGCTTCTCATCAAATATGCTTCCCAATCCATCTCTGGGATCCCCTCCAGGTTCAGTGGCAGTGGATCAGGGTCAGATTTCACTCTCAGTATCAACAGTGTGGAACCTGAAGATGTTGGAGTGTATTACTGTCAAAATGGTCACAGCTTTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "TGCGGGTGTAGGGACT-1_contig_3": "GTCCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCAC",
        "CTACATTCACTAGTAC-1_contig_4": "TGGGGAATCAACCTCCTTAAAATGAATTTGCCTGTTCATCTCTTGGTGCTTCTGTTGTTCTGGATTCCTGCTTCCAGAGGTGATGTTGTGGTGACTCAAACTCCACTCTCCCTGCCTGTCAGCTTTGGAGATCAAGTTTCTATCTCTTGCAGGTCTAGTCAGAGTCTTGCAAACAGTTATGGGAACACCTATTTGTCTTGGTACCTGCACAAGCCTGGCCAGTCTCCACAGCTCCTCATCTATGGGATTTCCAACAGATTTTCTGGGGTGCCAGACAGGTTCAGTGGCAGTGGTTCAGGGACAGATTTCACACTCAAGATCAGCACAATAAAGCCTGAGGACTTGGGAATGTATTACTGCTTACAAGGTACACATCAGCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GAAGCAGCACCAGTTA-1_contig_6": "TGGGGACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATACTGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTATAAGTTACATGCACTGGTACCAGCAGAAGCCAGGCACCTCCCCCAAAAGATGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTATTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCTCTTATTTCTGCCATCAGTGGAGTAGTTACCCACCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "GTTCATTGTTCACGGC-1_contig_6": "GGGAGTTCTAGAGCTTCCAGGTGTGCTGTCAGGCTGCTAGTGTATGCTCTCTCTAGTTTCTTTTTGGAGGCACTCAGAGCTATGAGTTTTCCTCTTAGGACTGCCTTCATTGTGTCCCATAAGTTTGGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACC",
        "AAGTCTGTCCATGAAC-1_contig_4": "GGAAGCAAAGGGGATCAGCCCGAGATTCTCATTCAGTGATCAACACTGAACACACATCCCTTACCATGGATTTTGGGCTGATTTTTTTTATTGTTGCTCTTTTAAAAGGGGTCCAGTGTGAGGTGAAGGTTTTCGAGTCTGGAGGTGGCCTGGTGCAGCCTGGAGGATCCCTGAAACTCTCCTGTGCAGCCTCAGGATTCGATTTTAGTAGATACTGGATGAATTGGGTCCGGCAGGCTCCAGGGAAAGGGCTAGAATGGATTGGAGAAATTAATCCAGATAGCAATACGATAAACTATACGCCATCTCTAAAGGATAAATTCATCATCTCCAGAGACAACGCCAAAAATACGCTGTACCTGCAAATGAGCAAAGTGAGATCTGAGGACACAGCCCTTTATTACTGTGCAAGACGCGATAATTACCACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCCTTGTCAAAGGCTACTTCCCTGAGCCGGTAACTGTAAAATGGAACTATGGAGCCCTGTCCAGC",
        "CCTACCAGTTTAGGAA-1_contig_1": "GACATTGTGAGCCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGTGCAAGTCAGGGCATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTATTACACATCAAGTTTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGGACAGATTATTCTCTCACCATCAGCAACCTGGAACCTGAAGATATTGCCACTTACTATTGTCAGCAGTATAGTAAGCTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TTGGAACGTAAGGATT-1_contig_4": "GGGATCCTCTTCTCATAGAGCCTCCATCAGACCATGGCTGTCCTGGCACTGCTCCTCTGCCTGGTGACATTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACTGTCTCAGGGTTCTCATTAACCAGCTATGGTGTAAGCTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGGTGACGGGAGCACAAATTATCATTCAGCTCTCATATCCAGACTGAGCATCAGCAAGGATAACTCCAAGAGCCAAGTTTTCTTAAAACTGAACAGTCTGCAAACTGATGACACAGCCACGTACTACTGTGCCAAACCCCTCTATGGTAAAGGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GCATACAAGGCAGTCA-1_contig_3": "TTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "GCATGATAGCTAACAA-1_contig_2": "GAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TGCCAAAGTGTAACGG-1_contig_4": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGCAAGTACCTAGCCACCTCGC",
        "GCAGCCATCTACTCAT-1_contig_1": "GAGCTGCCAGGAGCCTAAGAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGTTATGGGTACTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCAAAAGTGTCAGTACATCTGGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCACAGTAGGGAGCTTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCT",
        "CGAGCACCAAATACAG-1_contig_3": "GACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GGGACCTCAGCCACCA-1_contig_3": "ACATACCAGCAAGGGAGTGACCAGTTTGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGATTGGCTGTGGAACTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACATGGCTACATATTTCTGTGCAAGAGGGGGTTACTACGGTAGTAGAGTATATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AGGGAGTAGTGTGAAT-1_contig_2": "AGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TATTACCGTAAGGGAA-1_contig_3": "TGGGGAGACATAGGAGCTTGGCTCTGGTTCCCAAGACCTCTCACTCACTTCTCAACATGGAGTGGGAACTGAGCTTAATTTTCATTTTTGCTCTTTTAAAAGATGTCCAGTGTGAAGTGCAGCTGTTGGAGACTGGAGGAGGCTTGGTGCAACCTGGGGGGTCACGGGGACTCTCTTGTGAAGGCTCAGGGTTTACTTTTAGTGGCTTCTGGATGAGCTGGGTTCGACAGACACCTGGGAAGACCCTGGAGTGGATTGGAGACATTAATTCTGATGGCAGTGCAATAAACTACGCACCATCCATAAAGGATCGATTCACTATCTTCAGAGACAATGACAAGAGCACCCTGTACCTGCAGATGAGCAATGTGCGATCTGAGGACACAGCCACGTATTTCTGTATGAGATATGATTACGACGGGGTTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TGAGGGATCCGTTGTC-1_contig_4": "GATATATGAGCCCTATCTTCTCTACAGACACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTTATCTTCTTCCTGATGGCAGTGGTTACAGGGGTCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCAGAGCTTGTGAAGCCAGGGGCCTCAGTCAAGTTGTCCTGCACAGCTTCTGGCTTCAACATTAAAGACACCTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGAAGGATTGATCCTGCGAATGGTAATACTAAATATGACCCGAAGTTCCAGGGCAAGGCCACTATAACAGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTTCCGGATGTGCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CTACCCACAACTGCTA-1_contig_3": "GAAGGGAGTGACCAGTTAGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGTTATACCTTCACAGACTATTCAATGCACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACTGAGACTGGTGAGCCAACATATGCAGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCTAGGGGGAATTCATTACTACGGCTACTGGATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCT",
        "GGCGACTAGCGATATA-1_contig_11": "GAGCTCTGACAGAGGAGCCAAGCCCTGGATTCCCAGGTCCTCACATTCAGTGATCAGCACTGAACACAGACCACTCACCATGGACTCCAGGCTCAATTTAGTTTTCCTTGTCCTTATTTTAAAAGGTGTCCAGTGTGATGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGCAGCCTGGAGGGTCCCGGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTTTGGAATGCACTGGGTTCGTCAGGCTCCAGAGAAGGGGCTGGAGTGGGTCGCATACATTAGTAGTGGCAGTAGTACCATCTACTATGCAGACACAGTGAAGGGCCGATTCACCATCTCCAGAGACAATCCCAAGAACACCCTGTTCCTGCAAATGACCAGTCTAAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGCTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "TATGCCCTCAAGAAGT-1_contig_3": "GGCCTTGCAGTTCAGACATAGGAGCTTGGCTCTGGTTCCCAAGACCTCTCACTCACTTCTCAACATGGAGTGGGAACTGAGCTTAATTTTCATTTTTGCTCTTTTAAAAGATGTCCAGTGTGAAGTGCAGCTGTTGGAGACTGGAGGAGGCTTGGTGCAACCTGGGGGGTCACGGGGACTCTCTTGTGAAGGCTCAGGGTTTACTTTTAGTGGCTTCTGGATGAGCTGGGTTCGACAGACACCTGGGAAGACCCTGGAGTGGATTGGAGACATTAATTCTGATGGCAGTGCAATAAACTACGCACCATCCATAAAGGATCGATTCACTATCTTCAGAGACAATGACAAGAGCACCCTGTACCTGCAGATGAGCAATGTGCGATCTGAGGACACAGCCACGTATTTCTGTATGAGGTACGGTAGTAGCTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CTACATTCACTAGTAC-1_contig_1": "TACTCGTGTCTCTTTTCTTATATGGGGGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGCCTGTGGGGACATTGTGATGTCACAGTCTCCATCC",
        "CATCAAGTCTGAAAGA-1_contig_2": "CATAGTTTTTCTTATATGGGCTCTTTCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGAATTGAAGTCTCAACTATGAAAATCAGCAGTTCTCTGTCAGAGAAGCCCAAGCGCTTCCACGCATGCTTGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TCTGGAAGTAGCTAAA-1_contig_6": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTTATTTCCAGCTTGGTCCCCCCTCCGAACGTGTACGGATAGCTGT",
        "TCAGCAATCTTTAGTC-1_contig_1": "GGATCAGACAGGCAGGGGAAGCAAGATGGATTCACAGGCCCAGGTTCTTATGTTACTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTAGCTGTGTCAGTTGGAGAGAAGGTTACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "ACGAGCCCACAAGTAA-1_contig_3": "AAGGTGTCCAGTGTGAAGTGAAGCTGGTGGAGTCTGGGGGAGGCTTAGTGCAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAACCTCTGGATTCACTTTCAGTGACTATTACATGTATTGGGTTCGCCAGACTCCAGAGAAGAGGCTGGAGTGGGTCGCATACATTAGTAATGGTGGTGGTAGCACCTATTATCCAGACACTGTAAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCCGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTGCAAGAGGGATGATTACGCCCGGATATTACTATATTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGCCAAAACAACACCCCCATCAGTCTATCCACTGGCCCCTGGGTGTGGAGATACAACTGGTTCCTCCGTGACTCTGGGATGCCTGGTCAAGGG",
        "GGGTTGCGTACAGCAG-1_contig_3": "ACATACCAGCAAGGGAGTGACCAGTTTGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGATTGGCTGTGGAACTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACATGGCTACATATTTCTGTGCAAGAGGGGGGGTTATATACTACGGTAGTATCTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GAACGGAGTCGGGTCT-1_contig_4": "GGACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGAAGTGGTAACTATTTGTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CCAGCGAAGTCCGGTC-1_contig_3": "GGGGATGAAATGCATCACACCAGCATGGGCATCAAAATGGAGTCACAGATTCAGGTCTTTGTATTCGTGTTTCTCTGGTTGTCTGGTGTTGACGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGAGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGACAATCTCCTAAACTACTGATTTACTCGGCATCCTACCGGTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATCTGGGACGGATTTCACTTTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAACATTATAGTACTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACC",
        "GCGGGTTTCGCACTCT-1_contig_3": "GGATCATATGAGCCCTATCTTCTCTACAGACACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTTATCTTCTTCCTGATGGCAGTGGTTACAGGGGTCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCAGAGCTTGTGAAGCCAGGGGCCTCAGTCAAGTTGTCCTGCACAGCTTCTGGCTTCAACATTAAAGACACCTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGAAGGATTGATCCTGCGAATGGTAATACTAAATATGACCCGAAGTTCCAGGGCAAGGCCACTATAACAGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTGCTAGAGGGGGACGTTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAG",
        "CGTCACTCAAGAGGCT-1_contig_1": "GCTTGACTACTCAAGACTTTTTGTATCAAGTTCTCAGAATGAGGTGCCTAGCTGAGTTCCTGGGGCTGCTTGTGCTCTGGATCCCTGGAGCCATTGGGGATATTGTGATGACTCAGGCTGCACCCTCTGTACCTGTCACTCCTGGAGAGTCAGTATCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTGCATAGTAATGGCAACACTTACTTGTATTGGTTCCTGCAGAGGCCAGGCCAGTCTCCTCAGCTCCTGATATATCGGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTGGCAGTGGGTCAGGAACTGCTTTCACACTGAGAATCAGTAGAGTGGAGGCTGAGGATGTGGGTATTTATTACTGTATACAACATCTAGAATATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TTGTAGGAGCCCAACC-1_contig_5": "AACATATGTCCAATGTCCTCTCCACAGTCCCTGAAGACACTGACTATAACTATGGGATGGACCTGGATCTTTATTTTAATCCTGTCAGTAACTACAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGAGAAGCCTGGCGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGCTACAACATGAACTGGGTGAAGCAGAGCAATGGAAAGAGCCTTGAGTGGATTGGAAATATTGATCCTTACTATGGTGGTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAAGAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGAAGGGGGCTACGGCTTGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCT",
        "GACTACATCTGCGTAA-1_contig_4": "GACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GAGCAGATCTGGCGTG-1_contig_4": "GAGCTCTGACAGAGGAGGCCAGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTGATCAGCACTGAACACGGACCCCTCACCATGAACTTGGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCAGTGTGAAGTGAAGCTGGTGGAGTCTGGGGGAGGCTTAGTGCAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAACCTCTGGATTCACTTTCAGTGACTATTACATGTATTGGGTTCGCCAGACTCCAGAGAAGAGGCTGGAGTGGGTCGCATACATTAGTAATGGTGGTGGTAGCACCTATTATCCAGACACTGTAAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCCGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTGCAAGACATGGGGGCGGTAGTAGCCCCAACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GATGCTACATGTTCCC-1_contig_5": "GGAGAAAACAGCATATGATCAGTGTCCTCTCCAAAGTCCTTGAACATAGACTCTAACCATGGAATGGACCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGAGCTGAGCTGATGAAGCCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTACTGGCTACACATTCAGTAGCTACTGGATAGAGTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGAGATTTTACCTGGAAGTGGTAGTACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACATTCACTGCAGATACATCCTCCAACACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCCGTCTATTACTGTACTACGGCTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GCTTCCAGTTATCACG-1_contig_2": "CATTTCTGTCTTGACTACTCAAGACTTTTTGTATCAAGTTCTCAGAATGAGGTGCCTAGCTGAGTTCCTGGGGCTGCTTGTGCTCTGGATCCCTGGAGCCATTGGGGATATTGTGATGACTCAGGCTGCACCCTCTGTACCTGTCACTCCTGGAGAGTCAGTATCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTGCATAGTAATGGCAACACTTACTTGTATTGGTTCCTGCAGAGGCCAGGCCAGTCTCCTCAGCTCCTGATATATCGGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTGGCAGTGGGTCAGGAACTGCTTTCACACTGAGAATCAGTAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTATGCAACATCTAGAATATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GTAACTGGTTGGGACA-1_contig_3": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGT",
        "GACTGCGGTCGTGGCT-1_contig_5": "GAAGATCACAGTAGTCTCTACAGTCACAGGAGTACACCGGGCATTGACATGGGTTGGAGCTGTATCATCTTCTTTCTGGTAGCAACAGCTACAGGTGTGCACTCCCAGGTCCAGCTGCAGCAGTCTGGGGCTGAGCTGGTGAGGCCTGGGGTCTCAGTGAAGATTTCCTGCAAGGGTTCTGGCTACACATTCACTGATTATGCTATGCACTGGGTGAAGCAGAGTCATGCAAAGAGTCTAGAGTGGATTGGAGTTATTAGTACTTACTATGGTGATGCTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACAATGACTGTAGACAAATCCTCCAGCACAGCCTATATGGAACTTGCCAGACTGACATCTGAGGATTCTGCCATCTATTACTGTGCAAGAGGGTCTATCTACTATGGTAACTATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CACCAGGAGGTGACCA-1_contig_2": "GAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATT",
        "TGACAACGTCCGCTGA-1_contig_3": "GGGACTCAAGACTTTTTGTATCAAGTTCTCAGAATGAGGTGCCTAGCTGAGTTCCTGGGGCTGCTTGTGCTCTGGATCCCTGGAGCCATTGGGGATATTGTGATGACTCAGGCTGCACCCTCTGTACCTGTCACTCCTGGAGAGTCAGTATCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTGCATAGTAATGGCAACACTTACTTGTATTGGTTCCTGCAGAGGCCAGGCCAGTCTCCTCAGCTCCTGATATATCGGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTGGCAGTGGGTCAGGAACTGCTTTCACACTGAGAATCAGTAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTATGCAACATCTAGAATATCCTATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "CGTTCTGCACGTCAGC-1_contig_3": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTTATTTCCAGCTTGGTCCCCCCTCCGAACGTGTACGGA",
        "CGTGTCTAGCTGATAA-1_contig_3": "GAGCTCTGACAGAGGAGGCCAGTCCTGGATTTGAGTTCCTCACATTCAGTGATGAGCACTGAACACGGACCCCTCACCATGAATTTCGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCTGTGTGAAGTGAAGCTGGTGGAGTCTGGGGGAGGTTTAGTGCAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATACCATGTCTTGGGTTCGCCAGACTCCAGAGAAGAGGCTGGAGTGGGTCGCATACATTAGTAATGGTGGTGGTAGCACCTACTATCCAGACACTGTAAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACGGCCATGTATTACTGTGCAAGACATATTACTACGGTAGTATATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CCGGGATTCTTGTCAT-1_contig_3": "GGGGACTGTAGATAAGTCCTCCAGCACAGCCTACATGGAGCTCCACAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGGGAAAGGGGGGATTACGTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TTAGTTCCAGATTGCT-1_contig_1": "GGGGTCATTGTCTTTACAATCAGGACTCAGCATGGACATGATGGTCCTTGCTCAGTTTCTTGCATTCTTGTTGCTTTGGTTTCCAGGTGCAAGATGTGACATCCTGATGACCCAATCTCCATCCTCCATGTCTGTATCTCTGGGAGACACAGTCAGCATCACTTGCCATGCAAGTCAGGGCATTAGCAGTAATATAGGGTGGTTGCAGCAGAAACCAGGGAAATCATTTAAGGGCCTGATCTATCATGGAACCAACTTGGAAGATGGAGTTCCATCAAGGTTCAGTGGCAGTGGATCTGGAGCAGATTATTCTCTCACCATCAGCAGCCTGGAATCTGAAGATTTTGCAGACTATTACTGTGTACAGTATGCTCAGTTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCAGAAATAACTACGGTAGTTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GCTGCGAGTCACAAGG-1_contig_2": "GAGCGTATGAACCTAGCCCTGATTTCCCCAGCCTTCAGTTCCCAGATTCAGTGATCAGCCTTGAACACAGACCTGTCACCATGAAGTTGTGGCTGAACTGGATTTTCCTTGTAACACTTTTAAATGGTATCCAGTGTGAGGTGAAGCTGGTGGAGTCTGGAGGAGGCTTGGTACAGCCTGGGGGTTCTCTGAGACTCTCCTGTGCAACTTCTGGGTTCACCTTCACTGATTACTACATGAGCTGGGTCCGCCAGCCTCCAGGAAAGGCACTTGAGTGGTTGGGTTTTATTAGAAACAAAGCTAATGGTTACACAACAGAGTACAGTGCATCTGTGAAGGGTCGGTTCACCATCTCCAGAGATAATTCCCAAAGCATCCTCTATCTTCAAATGAACACCCTGAGAGCTGAGGACAGTGCCACTTATTACTGTGCAAGAGATGGGAATCCCTCTCATTACTACGGCTACGGAAAGGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "ACGAGCCGTCCAAGTT-1_contig_5": "TGGGGAATCAACCTCCTTAAAATGAATTTGCCTGTTCATCTCTTGGTGCTTCTGTTGTTCTGGATTCCTGCTTCCAGAGGTGATGTTGTGGTGACTCAAACTCCACTCTCCCTGCCTGTCAGCTTTGGAGATCAAGTTTCTATCTCTTGCAGGTCTAGTCAGAGTCTTGCAAACAGTTATGGGAACACCTATTTGTCTTGGTACCTGCACAAGCCTGGCCAGTCTCCACAGCTCCTCATCTATGGGATTTCCAACAGATTTTCTGGGGTGCCAGACAGGTTCAGTGGCAGTGGTTCAGGGACAGATTTCACACTCAAGATCAGCACAATAAAGCCTGAGGACTTGGGAATGTATTACTGCTTACAAGGTACACATCAGCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "CTCGGGAGTGACGGTA-1_contig_1": "GTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATATCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAGGCACCTCCCCCAAAAGATGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GTCCTCAGTCATCCCT-1_contig_4": "GTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "ACCGTAATCTCTGCTG-1_contig_1": "TGGGGTCTTTGCAGTGAGATATGAAATGCATCGCACCAGCATGGGCATCAAGATGGAGTCACAGATTCAGGCATTTGTATTCGTGTTTCTCTGGTTGTCTGGTGTTGACGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGAGTACTGCTGTAGCCTGGTATCAACAAAAACCAGGGCAATCTCCTAAACTACTGATTTACTGGGCATCCACCCGGCACACTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTATACTCTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCACTTTATTACTGTCAGCAACATTATAGCACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCT",
        "GCATGATAGGGCATGT-1_contig_3": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CCTTTCTAGATGTGGC-1_contig_2": "GAGACACACAAACCTGGACTCACAAGTTTTTCTCTTCAGTGACAGACACAGACATAGAACATTCACGATGTACTTGGGACTGAACTATGTATTCATAGTTTTTCTCTTAAATGGTGTCCAGAGTGAAGTGAAGCTTGAGGAGTCTGGAGGAGGCTTGGTGCAACCTGGAGGATCCATGAAACTCTCTTGTGCTGCCTCTGGATTCACTTTTAGTGACGCCTGGATGGACTGGGTCCGCCAGTCTCCAGAGAAGGGGCTTGAGTGGGTTGCTGAAATTAGAAGCAAAGCTAATAATCATGCAACATACTATGCTGAGTCTGTGAAAGGGAGGTTCACCATCTCAAGAGATGATTCCAAAAGTAGTGTCTACCTGCAAATGAACAGCTTAAGAGCTGAAGACACTGGCATTTATTACTGTACCAGATTCTCGTATGATTACGACGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAG",
        "ACGCCAGGTGTCAATC-1_contig_3": "TGGGGAATCAACCTCCTTAAAATGAATTTGCCTGTTCATCTCTTGGTGCTTCTGTTGTTCTGGATTCCTGCTTCCAGAGGTGATGTTGTGGTGACTCAAACTCCACTCTCCCTGCCTGTCAGCTTTGGAGATCAAGTTTCTATCTCTTGCAGGTCTAGTCAGAGTCTTGCAAACAGTTATGGGAACACCTATTTGTCTTGGTACCTGCACAAGCCTGGCCAGTCTCCACAGCTCCTCATCTATGGGATTTCCAACAGATTTTCTGGGGTGCCAGACAGGTTCAGTGGCAGTGGTTCAGGGACAGATTTCACACTCAAGATCAGCACAATAAAGCCTGAGGACTTGGGAATGTATTACTGCTTACAAGGTACACATCAGCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTC",
        "ACTGAGTCATGGTCAT-1_contig_1": "TGGTCATGTTAATAGGCTTTCTTATATGGGGCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GTGCTTCCAGGTCGTC-1_contig_2": "GGGGGAACAGCATCAGACAGGCAGGGGAGCAATATGGTATCAGAGACCCATGTCCTCATATTTTTGCTGCTATGGGTGTCTGGTGCTTGTGGGGAAATTGTGCTGACTGTGTCAGCAGGAGAGAGGGTCACTATCAGCTGCAAATCCAATCAGAATCTTTTATGGAGTGGAAACCAAAGGTACTGTTTGGTCTGGCACCAGTGGAAACCAGGGCAAACTCCTACACCGTTGATCACCTGGACATCTGATAGGTACTCTGGAGTCCCTGATCGTTTCATAGGCAGTGGATCTGTGACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGATGTGGCAGTTTATTTCTGTCAGCAGCATTTACACATTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CTGATCCCAGAAGCAC-1_contig_3": "GACCATGAAGCTCTAACAGACACACAAACCTGGACTCACAAGTTTTTCTCTTCAGTGACAGACACAGACATAGAACATTCACGATGTACTTGGGACTGAACTATGTATTCATAGTTTTTCTCTTAAATGGTGTCCAGAGTGAAGTGAAGCTTGAGGAGTCTGGAGGAGGCTTGGTGCAACCTGGAGGATCCATGAAACTCTCTTGTGCTGCCTCTGGATTCACTTTTAGTGACGCCTGGATGGACTGGGTCCGCCAGTCTCCAGAGAAGGGGCTTGAGTGGGTTGCTGAAATTAGAAGCAAAGCTAATAATCATGCAACATACTATGCTGAGTCTGTGAAAGGGAGGTTCACCATCTCAAGAGATGATTCCAAAAGTAGTGTCTACCTGCAAATGAACAGCTTAAGAGCTGAAGACACTGGCATTTATTACTGTACCAGGCCAAGGGATGGTTACCAGTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CTTAACTAGGGCACTA-1_contig_4": "ATACACACTCCAACAGTCATTCTTGGTCAGGAGACGTTGTAGAAATGAGACCGTCTATTCAGTTCCTGGGGCTCTTGTTGTTCTGGCTTCATGGTGCTCAGTGTGACATCCAGATGACACAGTCTCCATCCTCACTGTCTGCATCTCTGGGAGGCAAAGTCACCATCACTTGCAAGGCAAGCCAAGACATTAACAAGTATATAGCTTGGTACCAACACAAGCCTGGAAAAGGTCCTAGGCTGCTCATACATTACACATCTACATTACAGCCAGGCATCCCATCAAGGTTCAGTGGAAGTGGGTCTGGGAGAGATTATTCCTTCAGCATCAGCAACCTGGAGCCTGAAGATATTGCAACTTATTATTGTCTACAGTATGATAATCTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCT",
        "CTTTGCGGTCATCGGC-1_contig_1": "GGGGGATCATACACACTCCAACAGTCATTCTTGGTCAGGAGACGTTGTAGAAATGAGACCGTCTATTCAGTTCCTGGGGCTCTTGTTGTTCTGGCTTCATGGTGCTCAGTGTGACATCCAGATGACACAGTCTCCATCCTCACTGTCTGCATCTCTGGGAGGCAAAGTCACCATCACTTGCAAGGCAAGCCAAGACATTAACAAGTATATAGCTTGGTACCAACACAAGCCTGGAAAAGGTCCTAGGCTGCTCATACATTACACATCTACATTACAGCCAGGCATCCCATCAAGGTTCAGTGGAAGTGGGTCTGGGAGAGATTATTCCTTCAGCATCAGCAACCTGGAGCCTGAAGATATTGCAACTTATTATTGTCTACAGTATGATAATCTTCTACGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "ATTTCTGTCCCGGATG-1_contig_6": "GGGGCACATACCAGCAAGGGAGTGACCAGTTAGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGTTATACCTTCACAGACTATTCAATGCACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACTGAGACTGGTGAGCCAACATATGCAGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCTAACTGGGACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "TGCACCTGTTCCGGCA-1_contig_3": "GATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCTTGGGGCTGCTCTTCTGCCTGGTGACATTCCCAAGCTGTGTCCTATCCCAGGTGCAGCTGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATCACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGTGGTGGAAGCACAGACTATAATGCAGCTTTCATATCCAGACTGAGCATCAGCAAGGACAATTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTAATGACACAGCCATATATTACTGTGCCAGAAAACTGGGCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACCA",
        "GGTATTGAGCTTTGGT-1_contig_3": "GAACATATGTCCAATGTCCTCTCCACAGACACTGAACACACTGACTCTAACCATGGGATGGAGCTGGATCTTTCTCTTCCTCCTGTCAGGAACTGCAGGCGTCCACTCTGAGGTCCAGCTTCAGCAGTCAGGACCTGAGCTGGTGAAACCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTTCTGGATACACATTCACTGACTACAACATGCACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGATATATTTATCCTTACAATGGTGGTACTGGCTACAACCAGAAGTTCAAGAGCAAGGCCACATTGACTGTAGACAATTCCTCCAGCACAGCCTACATGGAGCTCCGCAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAGAGGGATGGTTAAACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TTATGCTAGGCTCAGA-1_contig_2": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GATCAGTGTCCAACTA-1_contig_1": "TGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GCAATCAGTGGCGAAT-1_contig_2": "TTTGATTCAGCTCAGCTGTGGGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGAC",
        "TCATTACTCTTGTATC-1_contig_3": "GGGGTTTTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "ACTGCTCGTACCGCTG-1_contig_8": "GATATGAGCCCTATCTTCTCTACAGACACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTTATCTTCTTCCTGATGGCAGTGGTTACAGGGGTCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCAGAGCTTGTGAAGCCAGGGGCCTCAGTCAAGTTGTCCTGCACAGCTTCTGGCTTCAACATTAAAGACACCTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGAAGGATTGATCCTGCGAATGGTAATACTAAATATGACCCGAAGTTCCAGGGCAAGGCCACTATAACAGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTGCTAGATATGGTAACTACGTCTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CTTAACTTCATCACCC-1_contig_4": "GACAGTAGTCTCTACAGTCACAGGAGTACACAGGGCATTGCCATGGGTTGGAGCTGTATCATCTTCTTTCTGGTAGCAACAGCTACAGGTGTGCACTCCCAGGTCCAGCTGCAGCAGTCTGGGCCTGAGCTGGTGAGGCCTGGGGTCTCAGTGAAGATTTCCTGCAAGGGTTCCGGCTACACATTCACTGATTATGCTATGCACTGGGTGAAGCAGAGTCATGCAAAGAGTCTAGAGTGGATTGGAGTTATTAGTACTTACTCTGGTAATACAAACTACAACCAGAAGTTTAAGGGCAAGGCCACAATGACTGTAGACAAATCCTCCAGCACAGCCTATATGGAACTTGCCAGATTGACATCTGAGGATTCTGCCATCTATTACTGTGCAAGATATGATGGTTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AGGTCATGTAGTGAAT-1_contig_1": "GCACTCGTTTTCTTATATGGGGACATCTCTCTCATTAGAGGTTGATCTTTGAGGAAAACAGGGTGTTGCCTAAAGGATGAAAGTGTTGAGTCTGTTGTACCTGTTGACAGCCATTCCTGGTATCCTGTCTGATGTACAGCTTCAGGAGTCAGGACCTGGCCTCGTGAAACCTTCTCAGTCTCTGTCTCTCACCTGCTCTGTCACTGGCTACTCCATCACCAGTGGTTATTACTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATAAGCTACGACGGTAGCAATAACTACAACCCATCTCTCAAAAATCGAATCTCCATCACTCGTGACACATCTAAGAACCAGTTTTTCCTGAAGTTGAATTCTGTGACTACTGAGGACACAGCTACATATTACTGTGCCTATGGTAACTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TACGGGCTCGTCGTTC-1_contig_6": "GGGACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGAGACTATTACTACGGTAGTAGCTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "AACGTTGGTATTCGTG-1_contig_4": "TCCTGACTACAGTTTTAACAGAGACATCCAAATGTGGATTTACAGGTCTTCCCATTCAGTGATCAAGTCTCAACACAAAATATCCAGCATGGAGTTGGGACTGAGCTGGGTATTTCTTGTGGCTCTTTTGAATGGTGTCCAGTGTCAGGTGCAGCTTGTAGAGACCGGGGGAGGCTTGGTGAGGCCTGGAAATTCTCTGAAACTCTCCTGTGTTACCTCGGGATTCACTTTCAGTAACTACCGGATGCACTGGCTTCGCCAGCCTCCAGGGAAGAGGCTGGAGTGGATTGCTGTAATTACAGTCAAATCTGATAATTATGGAGCAAATTATGCAGAGTCTGTGAAAGGCAGATTCACTATTTCAAGAGATGATTCAAAAAGCAGTGTCTACCTGCAGATGAACAGATTAAGAGAGGAAGACACTGCCACTTATTATTGTAGTAGAGTTTTCCATTACTACGGTAGTAGCTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GCATGATAGACAGGCT-1_contig_3": "GAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CGACTTCAGTGGCACA-1_contig_1": "TGGGGATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGTTATGGGTACTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCAAAAGTGTCAGTACATCTGGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCACAGTAGGGAGCTTCCTCCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GATGCTACACTAAGTC-1_contig_4": "AAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TCAGCTCAGAGTGACC-1_contig_4": "GGCATATGATCAGTATCCTCTTCACAGTCACTGAAAACATTGACTCTAATCATGGAATGTAACTGGATACTTCCTTTTATTCTGTCGGTAACTTCAGGGGTCTACTCAGAGGTTCAGCTCCAGCAGTCTGGGACTGTGCTGGCAAGGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACCTTTACCAGCTACTGGATGCACTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATTGGCGCTATTTATCCTGGAAATAGTGATACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCAAACTGACTGCAGTCACATCCACCAGCACTGCCTACATGGAGCTCAGCAGCCTGACAAATGAGGACTCTGCGGTCTATTACTGTACAAGATACTATAGGTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACCA",
        "CGACTTCAGTGGCACA-1_contig_5": "ATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "TGAGGGAGTCCGAGTC-1_contig_3": "GAAACAACCTATGATCAGTGTCCTCTCTACACAGTCCCTGACGACACTGACTCTAACCATGGGATGGAGCCGGATCTTTCTCTTCCTCCTGTCAATAACTGCAGGTGTCCATTGCCAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCCTCAGTGAAGATTTCCTGCAAAGCTTCTGGCTACGCATTCAGTAGCTCTTGGATGAACTGGGTGAAGCAGAGGCCTGGACAGGGTCTTGAGTGGATTGGACGGATTTATCCTGGAGATGGAGATACTAACTACAATGGGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACCTCTGTGGACTCTGCGGTCTATTTCTGTGCAAGGGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "TGGGAAGCACGTCAGC-1_contig_4": "GTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACC",
        "CGGCTAGCAGGAATGC-1_contig_3": "GCTCACAATGGAAAGGCACTGGATCTTTCTCTTCCTGTTTTCAGTAACTGCAGGTGTCCACTCCCAGGTCCAGCTTCAGCAGTCTGGGGCTGAACTGGCAAAACCTGGGGCCTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACCTTTACTAGCTACTGGATGCACTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATTGGATACATTAATCCTAGCACTGGTTATACTGAGTACAATCAGAAGTTCAAGGACAAGGCCACATTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAACTGAGCAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCCGCGACTGGGCCCCCTTTGTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "GCAAACTCATGCAATC-1_contig_4": "GACATACCAGCAAGGGAGTGACCAGTTTGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGATTGGCTGTGGAACTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACATGGCTACATATTTCTGTGCAAGGGGCTATGATTACTATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGA",
        "GCATGCGTCGCATGGC-1_contig_3": "GAATCTATGATCACTGTCCTATCCACAGTCCCTGAAGATACTGACTCTAACCATGGGATGGAGCTGGATCTTTCTCTTTCTCCTGTCAGTAACTGCAGGTGTGTTCTCTGAGGTTCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTTACTGGCTACTTTATGAACTGGGTGATGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGACGTATTAATCCTTACAATGGTGATACTTTCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAATCCTCTAGCACAGCCCACATGGAGCTCCGGAGCCTGGCATCTGAGGACTCTGCAGTCTATTATTGTGCAAGATGGGGCGGTAGTAGGGGGACTTCCTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CGAGCCACAGCAGTTT-1_contig_2": "GGGTACAATTTGCATGCCCCCATTACCACATCAGCCTCTCTGATGACATATAATGCATGCCATCAAGCTGCATGGGAAGCTGGCTATCACTCAAGGCCATGACCATGTTCTCACTAGCTCTTCTCCTCAGTCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCATGGCTATAGGAGAAAAAGTCACCATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGTACAGATTTTGTTTTTACAATTGAAAACATGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACTTGCCTACATGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGAC",
        "GTTCATTGTTCACGGC-1_contig_3": "GAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACC",
        "CCCAGTTTCATACGGT-1_contig_5": "GACATACCAGCAAGCGAGTGACCAGTTAGTCTTAAGGCACCACTTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCAAGATCAATGCGGACTGGGTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TATCTCAGTATCGCAT-1_contig_7": "GTCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAAC",
        "AAAGATGGTCTAACGT-1_contig_3": "GGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGGACTGTGTGAGGCAGGGGGATGATTTTGAGTGGTTTTCTGAGGCAGGGGGATGACTGTGAGTGATTTTCTGAGGCAGGGG",
        "CGACTTCGTCTTTCAT-1_contig_4": "GAAAACCACCCATGATCAGTATCCTCTCCACAGTCACTGAACACACTGACTCAAACCATGGAATGGTGCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCACTCCAAGGTCCAGCTGCAGCAGTCTGGAGCTGAGCTGGTGAAACCCGGGGCATCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACTGAGTATATTATACACTGGGTAAAGCAGAGGTCTGGACAGGGTCTTGAGTGGATTGGGTGGTTTTACCCTGGAAGTGGTAGTATAAAGTACAATGAGAAATTCAAGGACAAGGCCACATTGACTGCGGACAAATCCTCCAGCACAGTCTATATGGAGCTTAGTAGATTGACATCTGAAGACTCTGCGGTCTATTTCTGTGCAAGACACGAAGACCTCATTACTACGGTAGTAGAAGGTGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CTAAGACAGGCGCTCT-1_contig_5": "GGAGTTTATAGAGGAAAGTAGGCATGGAGGAATACAAAGTTCATCTAAGAGGGACTCTAGGCCTGCTGGGGAAGCCAAGAGTGTGGAACTCTAAGGTTTAGGAGTCAATCTCAGTGCTGAAGAGACAAGATGAGGAGCTGAGCTCGAAGCCAAGACGGGCAGGAAGAAGTATAGGAAACTCCTAGGATCAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCCTTGTCAAAGGCTACTTCCCTGAGCCGGTAACTGTAAAATGGAACTATGGAGCCCTGTCCAGC",
        "GTATCTTAGTGACTCT-1_contig_4": "GGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATC",
        "CCTCTGAGTTGTGGAG-1_contig_1": "GAAAACATAAGGTCACTGTTCTCACTACAGTTACTGAGCACACAGACCTCACCATGGGATGGAGCTCTATCATCCTCTTCTTGGTAGCAACAGCCTCAGGTGTCCACTCCCAGGTCCAACTGCAGCAACCTGGGTCTGAGCTGGTGAGGCCTGGAGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACATTCACCAGCTACTGGATGCACTGGGTGAAGCAGAGGCCTGGACAAGGCCTTGAGTGGATTGGAAATATTTATCCTGGTAGTGGTAGTACTAACTACGATGAGAAGTTCAAGAGCAAGGCCACACTGACTGTAGACACATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTACAAGAGACTATGATTATTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "ACGATACCACGAAAGC-1_contig_4": "GCTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAA",
        "CACAGGCGTCTCCACT-1_contig_11": "TGTTGTGACAGCCCCAGGTACATTTACTTATATAGGACCTCCCTGGGGAGATAGAATCCCAGGAGTAAAAATGCTGGATGTACACTAAGGATGGCCCCTGACACTCTGCACTGCTACCTCTGGCCCCACCAGGCAATGTTCCTGCAGAATCTCTTACCTTACTTGGCAGGGATTTTTGTCAAGGGATCTACTACTGTGTCTACTATGATTATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ACACCAAGTTCGGGCT-1_contig_1": "TGGGGATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "GCAAACTCATGCAATC-1_contig_1": "GCTGCTGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GTCAAGTTCGGAAACG-1_contig_3": "AGCATCAGACAGGCTGGGCAGCAAGATGGAATCACAGACCCAGGTCCTCATGTTTCTTCTGCTCTGGGTATCTGGTGCCTGTGCAGACATTGTGATGACACAGTCTCCATCCTCCCTGGCTATGTCAGTAGGACAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTAAATAGTAGCAATCAAAAGAACTATTTGGCCTGGTACCAGCAGAAACCAGGACAGTCTCCTAAACTTCTGGTATACTTTGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCATAGGCAGTGGATCTGGGACAGATTTCACTCTTACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGATTACTTCTGTCAGCAACATTATAGCACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CAAGATCCAGCGTAAG-1_contig_7": "GAGAGCAAGGGATGTTGTGAAGTCAGTGTTGTGACAGCCCCAGGTACATTTACTTATATAGGACCTCCCTGGGGAGATAGAATCCCAGGAGTAAAAATGCTGGATGTACACTAAGGATGGCCCCTGACACTCTGCACTGCTACCTCTGGCCCCACCAGGCAATGTTCCTGCAGAATCTCTTACCTTACTTGGCAGGGATTTTTGTCAAGGGATCTACTACTGTGTCTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TTTGGTTCATTTCACT-1_contig_6": "GCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GCGCAGTCAGCGAACA-1_contig_4": "CTTTTTGTTTCCACTTGGTTGATTTCAGCTGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGG",
        "GTCAAGTGTTCCACAA-1_contig_1": "ACAGTCATTGAAAACACTGACTCTAATCATGGAATGTAACTGGATACTTCCTTTTATTCTGTCGGTAATTTCAGGGGTCTACTCAGAGGTTCAGCTCCAGCAGTCTGGGACTGTGCTGGCAAGGCCTGGGGCTTCCGTGAAGATGTCCTGCAAGGCTTCTGGCTACAGCTTTACCAGCTACTGGATGCACTGGGTAAAACAGAGGCCTGGACAGGGTCTAGAATGGATTGGTGCTATTTATCCTGGAAATAGTGATACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCAAACTGACTGCAGTCACATCCGCCAGCACTGCCTACATGGAGCTCAGCAGCCTGACAAATGAGGACTCTGCGGTCTATTACTGTACAAGATCAACCCCTTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTACCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TTGGAACGTAAGGATT-1_contig_5": "ACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "TAGACCATCGCGCCAA-1_contig_4": "GGAATCCTCTTCTCATAGAGCCTCCATCAGAACATGGCTGTCCTGGCACTACTCCTCTGCCTGGTGGCTTTCCCAAGCTGTACCCTGTCCCAGGTGCAACTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATTACCTGCACTGTCTCTGGGTTCTCATTAACCAGCTATGATATAAGCTGGATTCGCCAGCCACCAGGAAAGGGTCTGGAGTGGCTTGGAGTAATATGGACTGGTGGAGGCACAAATTATAATTCAGCTTTCATGTCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATATATTACTGTGTAAGAGATGGGGCCGATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TCATTTGAGGGCTTCC-1_contig_2": "GGGGATCTACATCTGAAAGGCAGGTGGAGCAAGATGGAATCACAGACTCAGGTCCTCATCTCCTTGCTGTTCTGGGTATCTGGTACCTGTGGGGACATTGTGATGACACAGTCTCCATCCTCCCTGAGTGTGTCAGCAGGAGAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGTCTGTTAAACAGTGGAAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGCCTCCTAAACTGTTGATCTACGGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGAACCGATTTCACTCTTACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGAATGATCATAGTTATCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GCGCAACAGTGTGGCA-1_contig_5": "GAACATTTGTCCAATGTCCTCGCCACAGTCCCTGCAGACACTGACTCTAACCATGGGATGGAGCTGGATCTTTCTCTTCCTTCTGTCAGGAACTGCAGGTGTCCACTCTGAGATCCAGCTGCAGCAGTCTGGACCTGAGCTGATGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTAGCTACTACATGCACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGATATATTGATCCTTTCAATGGTGGTACTAGCTACAACCAGAAATTCAAGGGCAAGGCCACATTGACTGTAGACAAATCTTCCAGCACAGCCTACATGCATCTCAGCAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGCTATAGGTACGACGGGAACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GACCTGGAGATCTGCT-1_contig_3": "GGGGGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CGGACGTGTCATTAGC-1_contig_4": "GGAACAGCATCAGACAGGCAGGGGAGCAATATGGTATCAGAGACCCATGTCCTCATATTTTTGCTGCTATGGGTGTCTGGTGCTTGTGGGGAAATTGTGCTGACTGTGTCAGCAGGAGAGAGGGTCACTATCAGCTGCAAATCCAATCAGAATCTTTTATGGAGTGGAAACCAAAGGTACTGTTTGGTCTGGCACCAGTGGAAACCAGGGCAAACTCCTACACCGTTGATCACCTGGACATCTGATAGGTACTCTGGAGTCCCTGATCGTTTCATAGGCAGTGGATCTGTGACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGATGTGGCAGTTTATTTCTGTCAGCAGCATTTACACATTCCTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCAGA",
        "AGCCTAAGTGTTTGTG-1_contig_2": "ATCTGAGAGGAAAGGTAGCTTTCTGGAGGTCAGGAGACAGTGCAGAGAAGAACGAGTTACTGTGGACAGGTCTTAGATGGGGAAAGAATGAGCAAATGCAAGCATCAGAAGGGTGGATGCAATGTCCTGCCAAGGACTTACCAAGAGGATCCCCGGACAGAGCAGGCAGGTGGAGTTGACTGAGAGGACAGGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGGTTAAGTATATCCG-1_contig_3": "GGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTG",
        "AGCCTAACACTGTCGG-1_contig_2": "GGGATTGTGACTGAACAGCATCAGACAGGCAGGGGAGCAATATGGTATCAGAGACCCATGTCCTCATATTTTTGCTGCTATGGGTGTCTGGTGCTTGTGGGGAAATTGTGTTGACCCAGTCTATACCATCCCTGACTGTGTCAGCAGGAGAGAGGGTCACTATCAGCTGCAAATCCAATCAGAATCTTTTATGGAGTGGAAACCAAAGGTACTGTTTGGTCTGGCACCAGTGGAAACCAGGGCAAACTCCTACACCGTTGATCACCTGGACATCTGATAGGTACTCTGGAGTCCCTGATCGTTTCATAGGCAGTGGATCTGTGACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGATGTGGCAGTTTATTTCTGTCAGCAGCATTTACACATTCCTCCACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GAACATCGTGACCAAG-1_contig_3": "GTAGTACTTGTGAGAATAGGAGTAATTAGCCAGGAACTAAAATTCAAAGACACAATGGATTTACAGGTGCAGATTATCAGCTTCCTGCTAATCAGTGTCACAGTCATAATGTCCAGAGGAGAAAATGTGCTCACCCAGTCTCCAGCAATAATGGCTGCCTCTCTGGGGCAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTCCAGTTACTTGCACTGGTACCAGCAGAAGTCAGGCGCTTCCCCCAAACCCTTGATTCATAGGACATCCAACCTGGCTTCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCGTGGAGGCTGAAGATGATGCAACTTATTACTGCCAGCAGTGGAGTGGTTACCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACT",
        "GCCAAATTCCTGTAGA-1_contig_3": "GACAACCTATGATCAGTGTCCTCTCCACAGTCCCTGAAGACACTGACTCTAACCATGGGATTCAGCAGGATCTTTCTCTTCCTCCTGTCAATAACTACAGGTGTCCACTCCCAGGCTTATCTACAGCAGTCTGGGGCTGAGCTGGTGAGGTCTGGGGCCTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACATTTACCAGTTACAATATGCACTGGGTAAAGCAGACACCTGGACAGGGCCTGGAATGGATTGGATATATTTATCCTGGAAATGGTGGTACTAACTACAATCAGAAGTTCAAGGGCAAGGCCACATTGACTGCAGACACATCCTCCAGCACAGCCTACATGCAGATCAGCAGCCTGACATCTGAAGACTCTGCGGTCTATTTCTGTGCAAGAAGCGCAAGGGGTGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GCTCTGTCAGGTTTCA-1_contig_2": "GGGGTTCTGCATCAGAAAGGCAGGGGGATCAAGATGGAATCACAGACTCAGGTCTTCCTCTCCCTGCTGCTCTGGGTATCTGGTACCTGTGGGAACATTATGATGACACAGTCGCCATCATCTCTGGCTGTGTCTGCAGGAGAAAAGGTCACTATGAGCTGTAAGTCCAGTCAAAGTGTTTTATACAGTTCAAATCAGAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATCTACTGGGCATCCACTAGGGAATCTGGTGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTTACTCTTACCATCAGCAGTGTACAAGCTGAAGACCTGGCAGTTTATTACTGTCATCAATACCTCTCCTCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GCACTCTAGGTCATCT-1_contig_3": "TGGGGTTCTGCATCAGAAAGGCAGGGGGATCAAGATGGAATCACAGACTCAGGTCTTCCTCTCCCTGCTGCTCTGGGTATCTGGTACCTGTGGGAACATTATGATGACACAGTCGCCATCATCTCTGGCTGTGTCTGCAGGAGAAAAGGTCACTATGAGCTGTAAGTCCAGTTAAAGTGTTTTATACAGTTCAAATCAGAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CAGCAGCTCCTATTCA-1_contig_3": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGAGACAGACTCTTCTCCACAGTGTCCCCTTCATGT",
        "GTTTCTACAGCTCGCA-1_contig_3": "GGGGAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATGAGTGCCTCAGTCATAATGTCCAGGGGACAAATTGTTCTCACCCAGTCTCCAGCACTCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACTGGTACCAGCAGAAGCCAAGATCCTCCCCCAAACCCTGGATTTATCTCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCACTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AGCTCCTTCTGGCGTG-1_contig_5": "GGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCAC",
        "ATTTCTGTCCCGGATG-1_contig_1": "TGGGATCAGACAGGCAGGGGAAGCAAGATGGATTCACAGGCCCAGGTTCTTATGTTACTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTAGCTGTGTCAGTTGGAGAGAAGGTTACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TATTACCGTTGGTAAA-1_contig_1": "TGGGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGAGATCGGAAGAGC",
        "GGGAGATTCACTTCAT-1_contig_3": "GTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TACTCATCATCTACGA-1_contig_4": "AATCAACCTATGATCAGTGTCCTCTCCACAATCTCTGAAGACACTGACTCAAACCATGGGATGGAGCTGGGTCTTTCTCTTCCTCCTGTCAGGAACTGCAGGTGTCCACTGCCAGGTCCAGCTGAAGCAGTCTGGAGCTGAGCTGGTGAGGCCTGGGGCTTCAGTGAAGCTGTCCTGCAAGACTTCTGGATACATCTTCACCAGCTACTGGATTCACTGGGTAAAACAGAGGTCTGGACAGGGCCTTGAGTGGATTGCAAGGATTTATCCTGGAATTGATAATATTTACTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACTGCCTACATGCAGCTCAGCAGCCTGAAATCTGACGACTCTGCTGTCTATTTCTGTGCAAGAGGGGGGGGTTACGACTTTGACTACTGGGGCCAGGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "TCCCGATTCAACTCTT-1_contig_1": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CTACGTCGTTACAGAA-1_contig_1": "GGCTCAGAATTTATAAACCAGTATGAACTGAGCAGCATCAGACAGGCAGGGGAAGCAAGATGGATTCACAGGCCCAGGTTCTTATGTTACTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTAGCTGTGTCAGTTGGAGAGAAGGTTACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CTTAACTAGGGCACTA-1_contig_1": "TGGGGGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "AGCCTAAAGTACCGGA-1_contig_2": "GGAACATATGTCCAATGTCCTCTCCTCAGACACTGAACACACTGACTCTAACCATGGGATGGAGCTGGATCTTTCTCTTTCTCCTGTCAGGAACTGCAGGTGTCCTCTCTGAGGTCCAGCTGCAACAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGACTTCTGGATACACATTCACTGAATACACCATGCACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGAGGTATTAATCCTAACAATGGTGGTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAGTCCTCCAGCACAGCCTACATGGAGCTCCGCAGCCTGACATCTGAGGATTCTGCAGTCTATTACTGTGCAAGAGGGAGGGATGGTAACGGGGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TCTGAGAGTTCTGGTA-1_contig_1": "CTCTCAACAGCATCAGACAGGCTGGGCAGCAAGATGGAATCACAGACCCAGGTCCTCATGTTTCTTCTGCTCTGGGTATCTGGTGCCTGTGCAGACATTGTGATGACACAGTCTCCATCCTCCCTGGCTATGTCAGTAGGACAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTAAATAGTAGCAATCAAAAGAACTATTTGGCCTGGTACCAGCAGAAACCAGGACAGTCTCCTAAACTTCTGGTATACTTTGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCATAGGCAGTGGATCTGGGACAGATTTCACTCTTACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGATTACTTCTGTCAGCAACATTATAGCACTTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGG",
        "CACAAACCAACGATGG-1_contig_3": "GAATATATGATCAGTGTCTTCTCCGCTATCCCTGGACACACTGACTCTAACCATGGAATGGAGCTGGGTCTTTCTCTTCTTCCTGTCAGTAACTACAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGACGCTGAGTTGGTGAAACCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGCTACACCTTCACTGACCATGCTATTCACTGGGTGAAGCAGAAGCCTGAACAGGGCCTGGAATGGATTGGATATATTTCTCCCGGAAATGGTGATATTAAGTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACTGCCTACATGCAGCTCAACAGCCTGACATCTGAGGATTCTGCAGTGTATTTCTGTAAAAGATCTACTACGGAGACTGACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGCTTCACATCCCATC-1_contig_1": "GCTTGACTACTCAAGACTTTTTGTATCAAGTTCTCAGAATGAGGTGCCTAGCTGAGTTCCTGGGGCTGCTTGTGCTCTGGATCCCTGGAGCCATTGGGGATATTGTGATGACTCAGGCTGCACCCTCTGTACCTGTCACTCCTGGAGAGTCAGTATCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTGCATAGTAATGGCAACACTTACTTGTATTGGTTCCTGCAGAGGCCAGGCCAGTCTCCTCAGCTCCTGATATATCGGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTGGCAGTGGGTCAGGAACTGCTTTCACACTGAGAATCAGTAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTATGCAACATCTAGAATATCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "AATCCAGAGATGCGAC-1_contig_1": "GGGGTCTCAACAGCATCAGACAGGCTGGGCAGCAAGATGGAATCACAGACCCAGGTCCTCATGTTTCTTCTGCTCTGGGTATCTGGTGCCTGTGCAGACATTGTGATGACACAGTCTCCATCCTCCCTGGCTATGTCAGTAGGACAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTAAATAGTAGCAATCAAAAGAACTATTTGGCCTGGTACCAGCAGAAACCAGGACAGTCTCCTAAACTTCTGGTATACTTTGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCATAGGCAGTGGATCTGGGACAGATTTCACTCTTACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGATTACTTCTGTCAGCAACATTATAGCACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCTCAGAGGAAGGTGGAAACACGGTGAGAGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACCGCCGAAAACATAACACAATGACTAAAACCCAACCCAAAACCTATGGGGCCAGCACCTGGTCCT",
        "ATTACTCGTACTCAAC-1_contig_2": "TGGGAGTCATTCTTGGTCAGGAGACGTTGTAGAAATGAGACCGTCTATTCAGTTCCTGGGGCTCTTGTTGTTCTGGCTTCATGGTGCTCAGTGTGACATCCAGATGACACAGTCTCCATCCTCACTGTCTGCATCTCTGGGAGGCAAAGTCACCATCACTTGCAAGGCAAGCCAAGACATTAACAAGTATATAGCTTGGTACCAACACAAGCCTGGAAAAGGTCCTAGGCTGCTCATACATTACACATCTACATTACAGCCAGGCATCCCATCAAGGTTCAGTGGAAGTGGGTCTGGGAGAGATTATTCCTTCAGCATCAGCAACCTGGAGCCTGAAGATATTGCAACTTATTATTGTCTACAGTATGATAATCTGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCTCAGTCGTGTGCTT",
        "GCAAACTCATGCAATC-1_contig_5": "TCGTATTTTGCGTTTCTTATATGGGAGCTGCCAGGAAGCTAAGAAGCATCCTCTCTTCTAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCCAAAGTGTCAGTACATCTAGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCAAGTATGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATACTGCAACATATTACTGTCAGCACAGTTGGGAGATTCCTCCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCAGATCGGAAGAGC",
        "GCATGATAGACAGGCT-1_contig_2": "TATAGGGGGGTGAAACAGGAGCAGGTGCCCTTGGAGCAACAGCACATACTCTGCTGATTTGCATATGAAATAATTTTATAACAGCCCAGGCTTCTTTAAGGGCAGCTGCCAGGAGCCTAAGAAGCATCCTCTCATCTAGTTCTCAGAGATGGAGACAGACACAATCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGCTCCACTGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAAGGCCAGCCAAAGTGTTGATTATGATGGTGATAGTTATATGAACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAATCTAGAATCTGGGATCCCAGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCTGTACCAACTGTATCCATCTTCCCACCATCCAGT",
        "TTCTCCTAGGTGCTTT-1_contig_2": "GGGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GTCAAGTAGGCATTGG-1_contig_2": "GAGCAATCAGTTCCTGCCAGGACACAGTTTAGATATGAGGTTCCAGGTTCAGGTTCTGGGGCTCCTTCTGCTCTGGATATCAGGTGCCCAGTGTGATGTCCAGATAACCCAGTCTCCATCTTATCTTGCTGCATCTCCTGGAGAAACCATTACTATTAATTGCAGGGCAAGTAAGAGCATTAGCAAATATTTAGCCTGGTATCAAGAGAAACCTGGGAAAACTAATAAGCTTCTTATCTACTCTGGATCCACTTTGCAATCTGGAATTCCATCAAGGTTCAGTGGCAGTGGATCTGGTACAGATTTCACTCTCACCATCAGTAGCCTGGAGCCTGAAGATTTTGCAATGTATTACTGTCAACAGCATAATGAATACCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "AGCCTAATCTGAAAGA-1_contig_1": "TCTCTCTTCTCATGAAAACGAGTCCTGAACTAACCTTGAATCTGAAGCAAAGGGGATCAGCCCGAGATTCTCATTCAGTGATCAACACTGAACACACATCCCTTACCATGGATTTTGGGCTGATTTTTTTTATTGTTGCTCTTTTAAAAGGGGTCCAGTGTGAGGTGAAGCTTCTCGAGTCTGGAGGTGGCCTGGTGCAGCCTGGAGGATCCCTGAAACTCTCCTGTGCAGCCTCAGGATTCGATTTTAGTAGATACTGGATGAGTTGGGTCCGGCAGGCTCCAGGGAAAGGGCTAGAATGGATTGGAGAAATTAATCCAGATAGCAGTACGATAAACTATACGCCATCTCTAAAGGATAAATTCATCATCTCCAGAGACAACGCCAAAAATACGCTGTACCTGCAAATGAGCAAAGTGAGATCTGAGGACACAGCCCTTTATTATTGTGCAAGGGAAACTGGGACACGGTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCT",
        "ATCATCTGTGCTCTTC-1_contig_2": "GAACAACCTATGATCACTGTCCTCTCCACAGTCCCTGAAGACACTGACTCTAACCATGGGATGGAGCTGTATCTTTCTCTTTCTCCTGTCAGTAACTGTAGGTGTGTTCTCTGAGGTTCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTTACTGGCTACTTTATGAACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGACGTATTAATCCTTACAATGGTGATACTTTCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAATCCTCTAGCACAGCCCACATGGAGCTCCTGAGCCTGACATCTGAGGACTCTGCAGTCTATTATTGTGGAAGAGGGGGGATGATTACCTACGTGGGGCCTTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GCTGGGTTCGTGGACC-1_contig_11": "GATAGTTGATAACTCAGACACAAGTGAATGACAGATGGACCTCCATCTACTCTTATTTTAAAAAGAAGACAAACCCCACAGGCTCGAGAACTTTAGCGACTGTTTTGAGAGAAATCATTGGTCCCTGACTCAAGAGATGACTGGCAGATTGGGGATCAGAATACCCATACTCTGTGGCTAGTGTGAGGTTTAAGCCTCAGAGTCCCTGTGGTCTCTGACTGGTGCAAGGTTTTGACTAAGCGGAGCACCACAGTGCTAACTGGGGAGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "GACGGCTTCACATAGC-1_contig_6": "GATCCTCCATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCTTGGGGCTGCTCTTCTGCCTGGTGACATTCCCAAGCTGTGTCCTATCCCAGGTGCAGCTGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATCACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGTGGTGGAAGCACAGACTATAATGCAGCTTTCATATCCAGACTGAGCATCAGCAAGGACAATTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTAATGACACAGCCATATATTACTGTGCCAGAAAATGGTTACTCCCCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCC",
        "CGGTTAATCTGCCAGG-1_contig_3": "GGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCAC",
        "GGAATAATCGACCAGC-1_contig_5": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGG",
        "CAAGATCCATACCATG-1_contig_2": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTC",
        "CCTTTCTTCCGTTGCT-1_contig_6": "GAGAGGACACTCAGTCAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TGAGGGATCCGTTGTC-1_contig_2": "GGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTAGCAGGCCACTGTCACACCACTTGGGGAAAAATTGGAAATCAGACACACGAGTGTGGCTTTGTTTTCCTGGAGCTCCTCAGGGGAAGGTG",
        "CGTGAGCGTACCGAGA-1_contig_5": "CTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "ATTGGACAGCAATCTC-1_contig_1": "GCGCCCTTTTCTTATATGGGGAAAATACATCAGATCAGCATGGGCATCAAGATGGAGTCACAGACTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGATGGAGACATTGTGATGACCCAGTCTCAAAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCGTCACCTGCAAGGCCAGTCAGAATGTGGGTACTAATGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAAGCACTGATTTACTCGGCATCCTACCGGTACAGTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAATGTGCAGTCTGAAGACTTGGCAGAGTATTTCTGTCAGCAATATAACAGCTATCCTCTGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCAGG",
        "CACACAACAGTTTACG-1_contig_1": "TGGGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTTTGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CCATTCGAGTGTGAAT-1_contig_2": "GACATTGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTTGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGCCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCAGGACCTGACCTGGTGAAACCTTCTCAGTCACTTTCACTCACCTGCACTGTCACTGGCTACTCCATCACCAGTGGTTATAGCTGGCACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATACACTACAGTGGTAGCACTAACTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAATTACTACGGTAGTAGCTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAAT",
        "CTCGTACAGACACGAC-1_contig_2": "TGGGGGTCAGCCTCACACAGAACACACACAGATATGGGTGTACCCACTCAGCTCCTGTTGCTGTGGCTTACAGTCGTAGTTGTCAGATGTGACATCCAGATGACTCAGTCTCCAGCTTCACTGTCTGCATCTGTGGGAGAAACTGTCACCATCACATGTGGAGCAAGTGAGAATATTTACGGTGCTTTAAATTGGTATCAGCGGAAACAGGGAAAATCTCCTCAGCTCCTGATCTATGGTGCAACCAACTTGGCAGATGGCATGTCATCGAGGTTCAGTGGCAGTGGATCTGGTAGACAGTATTCTCTCAAGATCAGTAGCCTGCATCCTGACGATGTTGCAACGTATTACTGTCAAAATGTGTTAAGTACTCCTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "TTGACTTCAATTCCTT-1_contig_2": "TGGGGGAAATGCATCAGACCAGCATGGGCATCAAGATGGAATCACAGACTCTGGTCTTCATATCCATACTGCTCTGGTTATATGGTGCTGATGGGAACATTGTAATGACCCAATCTCCCAAATCCATGTCCATGTCAGTAGGAGAGAGGGTCACCTTGAGCTGCAAGGCCAGTGAGAATGTGGGTACTTATGTATCCTGGTATCAACAGAAACCAGAGCAGTCTCCTAAACTGCTGATATACGGGGCATCCAACCGGTACACTGGGGTCCCCGATCGCTTCACAGGCAGTGGATCTGCAACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGACCTTGCAGATTATCACTGTGGACAGAGTTACAGCTATCCTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCA",
        "AAAGATGGTTACGACT-1_contig_4": "AGTTGCTTTAAAGAGATGTACTTATAGGAAGCACAATAATTAGCCAGAGACTGAGATGGAAAACAAAATGGATTTTCAGATGCAGATTATCAGCTTGCTGCTAATCAGTGTCACAGTCATAGTGTCTAATGGAGAAATTGTGCTCACCCAGTCTCCAACCACCATGGCTGCATCTCCCGGGGAGAAGATCACTATCACCTGCAGTGCCAGCTCAAGTATAAGTTCCAATTACTTGCATTGGTATCAGCAGAAGCCAGGATTCTCCCCTAAACTCTTGATTTATAGGACATCCAATCTGGCTTCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATTGGCACCATGGAGGCTGAAGATGTTGCCACTTACTACTGCCAGCAGGGTAGTAGTATACCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCAAGTACCTAGCCACCTCGCA",
        "CTAGCCTCATTCCTCG-1_contig_7": "GAAATCAACATATGTCCAATGTCCTCTCCACAGTCCCTGAAGACACTGACTATAACTATGGGATGGACCTGGATCTTTATTTTAATCCTGTCAGTAACTACAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGAGAAGCCTGGCGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGCTACAACATGAACTGGGTGAAGCAGAGCAATGGAAAGAGCCTTGAGTGGATTGGAAATATTGATCCTTACTATGGTGGTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAAGAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGACGGTCAGACACCTCGAGTCCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGTCTGCGAGAAATCCCACCATCTACCCACTGACACTCCCACCAGCTCTGTCAAGTGACCCAGTGATAATCGGCTGCCTGATTCACGATTACTTCCCTTCCGGCACGATGAATGTGACCTGGGGAAAGAGTGGGAAGGAT",
        "CAGCCGATCTACTATC-1_contig_8": "GGGGACATATGACCAATATCCTCTCCACATTCCCTTAAGACACTGACTCTAAACATGGGATGGATCTGGATCTTTCTCTTCCTCCTGTCAGGAACTGCAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTAGTGAAGACTGGGACTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGTTACTACATACACTGGGTCAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGATATATTAGTTGTTACAATAGTGCTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTTACTGTAGACACATCCTCCAGCGCAGCCTACATGCAGTTCAACAGCCTGACATCTGAAGACTCTGCGGTCTATTACTGTGCAAGAGAGGGAGCTAACTGGGGGGACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGCCAAAACAACACCCCCATCAGTCTATCCACTGGCCCCTGGGTGTGGAGATACAACTGGTTCCTCCGTGACTCTGGGATGCCTGGTCAAGGGCTACTTCCCTGAGTCAGTGACTGTGACTTGGAACTCTGGATCCCTGTCC",
        "CTCGAAAGTCGTTGTA-1_contig_4": "GATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "ATTCTACAGTGCGATG-1_contig_8": "GAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCTTCAGTCATAATGTCCAGAGGACAAATTGTTCTCTCCCAGTCTCCAGCAATCCTGTCTGCATCTCCAGGGGAGAAGGTCACAATGACTTGCAGGGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAAACCCTGGATTTATGCCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGAGTGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CTGATCCGTACTTCTT-1_contig_3": "GGCAGTTCCTATCTCTCTCACTGGAGGCTGATTTTTGAAGAAAGGGGTTGTAGCCTAAAAGATGATGGTGTTAAGTCTTCTGTACCTGTTGACAGCCCTTCCGGGTATCCTGTCAGAGGTGCAGCTTCAGGAGTCAGGACCTAGCCTCGTGAAACCTTCTCAGACTCTGTCCCTCACCTGTTCTGTCACTGGCGACTCCATCACCAGTGGTTACTGGAACTGGATCCGGAAATTCCCAGGGAATAAACTTGAGTACATGGGGTACATAAGCTACAGTGGTAGCACTTACTACAATCCATCTCTCAAAAGTCGAATCTCCATCACTCGAGACACATCCAAGAACCAGTACTACCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGAGGGGAGGTCCTCCCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "GCATGATAGGGCATGT-1_contig_1": "TCGGTGAAACAGGAGCAGGTGCCCTTGGAGCAACAGCACATACTCTGCTGATTTGCATATGAAATAATTTTATAACAGCCCAGGCTTCTTTAAGGGCAGCTGCCAGGAGCCTAAGAAGCATCCTCTCATCTAGTTCTCAGAGATGGAGACAGACACAATCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGCTCCACTGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAAGGCCAGCCAAAGTGTTGATTATGATGGTGATAGTTATATGAACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAATCTAGAATCTGGGATCCCAGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCTTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CTGTGCTAGCCCAGCT-1_contig_5": "TGGGGATCATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGAGATTACTACGGTAGTAGCTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GGTATTGAGAGTACAT-1_contig_1": "GGGTCTTGACTACTCAAGACTTTTTGTATCAAGTTCTCAGAATGAGGTGCCTAGCTGAGTTCCTGGGGCTGCTTGTGCTCTGGATCCCTGGAGCCATTGGGGATATTGTGATGACTCAGGCTGCACCCTCTGTACCTGTCACTCCTGGAGAGTCAGTATCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTGCATAGTAATGGCAACACTTACTTGTATTGGTTCCTGCAGAGGCCAGGCCAGTCTCCTCAGCTCCTGATATATCGGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTGGCAGTGGGTCAGGAACTGCTTTCACACTGAGAATCAGTAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTATGCAACATCTAGAATATCCTTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "ACGTCAACACCACCAG-1_contig_1": "GAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGGTATGGAAAATTCATGGATGATTTTCTGGCAAGGCTTGTTAAGAGGACATATGGAAGCTGGGAAGAAGGATATTACAGTGATGTCACCACCATCCTAGGATCACCCCCACCTACACACACAGATGCAACTGGAAAATAAGGCCAACCCAAGGCTACACCCTCAGTTAATCTGTTCCCACCTTCCTCTGAAGAGCTCAAGACTAAAAAGGCCACACTGGTGTGTA",
        "CATATTCTCTGTCTAT-1_contig_1": "GGCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GCACTCTAGGTCATCT-1_contig_5": "GGGGGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GTGGGTCTCGGCCGAT-1_contig_4": "GGGAAACAACCTGTGATCAGTGTCCTCTCCACAGTCCCTGAACACACTGACTTTCACCATGGAATGGAGCGGGGTCTTTATCTTTCTCTTGTCAGTAACTGCAGATGTCCACTCCCAGGTCCAGTTGCAGCAGTCTGGAGCTGAGCTGGTAAGGCCTGGGACTTCAGTGAAGATATCCTGCAAGGCTTCTGGCTACACCTTCACTAACTACTGGCTAGGTTGGGTAAAGCAGAGGCCTGGACATGGACTTGAGTGGATTGGAGATATTTACCCTGGAGGTGGTTATACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACACATCCTCCAGCACTGCCTACATGCAGCTCAGTAGCCTGACATCTGAGGACTCTGCTGTCTATTTCTGTGCAAGACTTAATTCCTACTATAGGCCACGGGGGGGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGATGGCCAAGCCGTC-1_contig_4": "GAAGTTAAATCCCCTCTTGGGGTGTGAAAGCTCACATCTCTCTCATTAGAGGTTGATCTTTGAGGAAAACAGGGTGTTGCCTAAAGGATGAAAGTGTTGAGTCTGTTGTACCTGTTGACAGCCATTCCTGGTATCCTGTCTGATGTACAGCTTCAGGAGTCAGGACCTGGCCTCGTGAAACCTTCTCAGTCTCTGTCTCTCACCTGCTCTGTCACTGGCTACTCCATCACCAGTGGTTATTACTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATAAGCTACGACGGTAGCAATAACTACAACCCATCTCTCAAAAATCGAATCTCCATCACTCGTGACACATCTAAGAACCAGTTTTTCCTGAAGTTGAATTCTGTGACTACTGAGGACACAGCTACATATTACTGTGCAAGAGAGGGGTCTACTGGGACGGGGGATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGT",
        "AGAGCTTCAATCTACG-1_contig_5": "GGGCAGTCACATTGTGAGCCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGTGCAAGTCAGGGCATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTATTACACATCAAGTTTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGGACAGATTATTCTCTCACCATCAGCAACCTGGAACCTGAAGATATTGCCACTTACTATTGTCAGCAGTATAGTAAGCTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGT",
        "TCAGGATGTAGAAGGA-1_contig_4": "GACATACCAGCAAGGGAGTGACCAGTTAGTCTTAAGGCACCATTGAGCCCAAGTCTTAGACATCATGGATTGGCTGTGGAACTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCAACACTGGAGAGCCAACATATGCTGAAGAGTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCAAGATGCTACTATGGTCTTTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "CATCAGAAGCACAGGT-1_contig_6": "GTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGG",
        "GGAACTTGTAAACCTC-1_contig_1": "TCTATGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCTTTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCGGGCCAGTCTCCAAAGCTCCTGATCTACAGAATTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAACAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAA",
        "TTCCCAGCAGGATCGA-1_contig_2": "TCGATGCAGTACATTTTCTTATATGGGGAGATTGTGAGCCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCC",
        "GTAGTCATCTCCTATA-1_contig_1": "GCCATCCTCTTCTCATAGAGCCTCCATCAGACCATGGCTGTCCTGGAGCTGCTCCTCTCCCTGGTGACATTCCCAAGCTGTGTCCTGTCCCAGGTGCAGATGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATCACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGTGGTGGAAGCACAGACTATAATGCAGCTTTCATATCCAGACTGAGCATCAGCAAGGACAATTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTGATGACACAGCCATATACTACTGTGTCAGAAATGGGGGGCTACCTTATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TCTATTGCAAGCCCAC-1_contig_3": "GGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTGA",
        "TGTTCCGCACCTCGGA-1_contig_5": "GAGCTCTGACAGAGGAGGCCAGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTGATCAGCACTGAACACGGACCCCTCACCATGAACTTTGGGCTGAGCTTGATTTTCCTTGTCCTAATTTTAAAAGGTGTCCAGTGTGAAGTGAAGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCGCTTTCAGTAGCTATGACATGTCTTGGGTTCGCCAGACTCCGGAGAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGCCGATTCACCATCTCCAGAGACAATGCCAGGAACACCCTGTACCTGCAAATGAGCAGTCTGAGGTCTGAGGACACGGCCTTGTATTACTGTGCAAGACAGAAGTCATATGGTAACTACGAGGGGTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "CGTGAGCGTACCGAGA-1_contig_3": "GGTTTTAAAGTGATGTACTTATGAGAAGAGAAGTAATTAGCCAGGGAACAAAATTCAAATACACAATGGATTTTCTGGTGCAGATTTTCAGCTTCTTGCTAATCAGTGCCTCAGTTGCAATGTCCAGAGGAGAAAATGTGCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAAAAGGTCACCATGACCTGCAGGGCCAGCTCAAGTGTAAGTTCCAGTTACTTGCACTGGTACCAGCAGAAGTCAGGTGCCTCCCCCAAACTCTGGATTTATAGCACATCCAACTTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGTGTGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTACAGTGGTTACCCACTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "ATGTGTGAGCTAAGAT-1_contig_2": "GAGGACAATGGCAGAAAGTCACTCTCAGTGAGGATACACCATCAGCATGAGGGTCCTTGCTGAGCTCCTGGGGCTGCTGCTGTTCTGCTTTTTAGGTGTGAGATGTGACATCCAGATGAACCAGTCTCCATCCAGTCTGTCTGCATCCCTTGGAGACACAATTACCATCACTTGCCATGCCAGTCAGAACATTAATGTTTGGTTAAGCTGGTACCAGCAGAAACCAGGAAATATTCCTAAACTATTGATCTATAAGGCTTCCAACTTGCACACAGGCGTCCCATCAAGGTTTAGTGGCAGTGGATCTGGAACAGGTTTCACATTAACCATCAGCAGCCTGCAGCCTGAAGACATTGCCACTTACTACTGTCAACAGGGTCAAAGTTATCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CGAGCCACATGGTAGG-1_contig_1": "GAAAAGCAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATATCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAGGCACCTCCCCCAAAAGATGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCACCCACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GGCGACTAGCGATATA-1_contig_5": "GGAAGTACTTATGAGAATAGCAGTAATTAGCTAGGAACAAAAATTCAAAGACAAAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGTCTCAGTCATAATGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCTGGGGAGAAGGTCACCTTGACCTGCAGTGCCAGCTCAAGTGTAAGTTCCAGCTACTTGTACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAAACTCTGGATTTATAGCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCTCTTATTTCTGCCATCAGTGGAGTAGTTACCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACT",
        "GGACAAGCAGTATGCT-1_contig_2": "GAGAAAAACATGAGATCACAGTTCTCTCTACAGTTACTGAGCACACAGGACCTCACCATGGGATGGAGCTATATCATCCTCTTTTTGGTAGCAACAGCTACAGATGTCCACTCCCAGGTCCAACTGCAGCAGCCTGGGGCTGAACTGGTGAAGCCTGGGGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTGGATGCACTGGGTGAAGCAGAGGCCTGGACAAGGCCTTGAGTGGATTGGAGAGATTAATCCTAGCAACGGTCGTACTAACTACAATGAGAAGTTCAAGAGCAAGGCCACACTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGATATGGTAACTACGTTCCCTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CTACCCACACCTTGTC-1_contig_3": "GGACAGAGGCACCAAACTGTGGACTCACAAGTCTTTCCCTTCAGTGACCAACAGGGACACAGAACATTCACCATGGACTTGAGACTGAGCTGCGCTTTTATTATTGTTCTTTTAAAAGGGGTCCAGAGTGAAGTGAAGCTTGAGGAGTCTGGAGGAGGCTTGGTGCAACCTGGAGGATCCATGAAACTCTCCTGTGTAGCCTCTGGATTTACTTTCAGTAGCTACTGGATGTCTTGGGTCCGCCAGTCTCCAGAGAAGGGGCTTGAGTGGGTTGCTGAAATTAGATTGAAATCTGATAATTATGCAACACATTATGCGGAGTCTGTGAAAGGGAAGTTCACCATCTCAAGAGATGATTCCAAAAGTCGTCTCTACCTGCAAATGAACAGCTTAAGAGCTGAAGACACTGGAATTTATTACTGTACAGGGGGGTGGGGTTACCCCCTCTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "ACTATCTAGAGGTTAT-1_contig_3": "GGACATTGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTTGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGCCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCAGGACCTGACCTGGTGAAACCTTCTCAGTCACTTTCACTCACCTGCACTGTCACTGGCTACTCCATCACCAGTGGTTATAGCTGGCACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATACACTACAGTGGTAGCACTAACTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGACGGTTCACCTCGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "CCAGCGAAGTCCGGTC-1_contig_1": "GGGAGCTCTGACAGAGGAGGCCTGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACGGACCCCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCAGTGTGAAGTGATGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGCCATGTCTTGGGTTCGCCAGACTCCGGAGAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGGCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGACATAGGGACTATAGGTACGACGAGGGGGCGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ACGCCAGGTACCGGCT-1_contig_2": "GGGGAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGCATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATAACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTTCCAGCAGAAGCCAGGCACTTCTCCCAAACTCTGGATTTATAGCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGATCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAAAGGAGTAGTTACCCCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CTTAACTCACTGTTAG-1_contig_3": "GGGGACAGTAGTCTCTACAGTCACAGGAGTACACAGGGCATTGCCATGGGTTGGAGCTGTATCATCTTCTTTCTGGTAGCAACAGCTACAGGTGTGCACTCCCAGGTCCAGCTGCAGCAGTCTGGGCCTGAGGTGGTGAGGCCTGGGGTCTCAGTGAAGATTTCCTGCAAGGGTTCCGGCTACACATTCACTGATTATGCTATGCACTGGGTGAAGCAGAGTCATGCAAAGAGTCTAGAGTGGATTGGAGTTATTAGTACTTACAATGGTAATACAAACTACAACCAGAAGTTTAAGGGCAAGGCCACAATGACTGTAGACAAATCCTCCAGCACAGCCTATATGGAACTTGCCAGATTGACATCTGAGGATTCTGCCATCTATTACTGTGCAAAGGGGGATTATGGTAACCCCCTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GGCGACTAGCGATATA-1_contig_9": "GGGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCTTGGGGCTGCTCTTCTGCCTGGTGACATTCCCAAGCTGTGTCCTATCCCAGGTGCAGCTGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATCACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGTGGTGGAAGCACAGACTATAATGCAGCTTTCATATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTGATGACACAGCCATATACTACTGTGCCAGAAATTGGGGGGATGGTTACCCTTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "CTTAGGAAGTGACATA-1_contig_7": "AAAACGGCTTCACAAATCTTTTTCAAGACCACTTTCTGAGTATTCATTTTAGGAGAAATACTTTTTTTTTAAATGAATGCAATTATCTAGACTTATTTCGGTTGAACATGCTGGTTGGTGGTTGAGAGGACACTCAGTCAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CTCGTACCAATCAGAA-1_contig_3": "TCACTCTACATGTAAAGGGCTCAAGGCATCAACCTAGAACTTCCCACTAAGGAGGAAGGAAGGCAGCATGGTGGTCAGGAAGTCATCCAGAAACAGACCTCTCCAGTACTTCTTATGACCCCTCCCCCAGCAGCCTCTGTCCAGGCACTCAGAGAACAAGGGATGTTGGGAAGTCAGCCTTGTGACAGCCCCAGGCAATGTTCCTGTAGAATCTCTTACCTTACTTGGCAGAGATTTTTGTCAAGGGATCTACTACTGTGTCTACTATGATTACGCCTCTTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCAC",
        "TATTACCGTAAGGGAA-1_contig_2": "GGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCC",
        "CCGTACTCACGACTCG-1_contig_3": "GGGGATCAGACAGGCAGTGGGAGCAAGATGGATTCACAGGCCCAGGTTCTTATATTGCTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTGGCTGTGTCAGCAGGAGAGAAGGTCACTATGAGCTGCAAATCCAGTCAGAGTCTGCTCAACAGTAGAACCCGAAAGAACTACTTGGCTTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATCTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGCAAGCAATCTTATAATCTTCCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GGCGACTAGCGATATA-1_contig_3": "ATACTGTCCATTCAACAAAGCCTCCATAACCACTATCATATTTGGTACAATTTGCATGCCCCCATTACCACATCAGCCTCTCTGATGACATATAATGCATGCCATCAAGCTGCATGGGAAGCTGGCTATCACTCAAGGCCATGACCATGTTCTCACTAGCTCTTCTCCTCAGTCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCATGGCTATAGGAGAAAAAGTCACCATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGTACAGATTTTGTTTTTACAATTGAAAACATGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACTTGCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACT",
        "CAGCATATCCCTAACC-1_contig_1": "AGCCTGAAAACAACCTATGATCAGTGTCCTCTCTACACAGTCCCTGACAACACTGACTCTAACCATGGGATGGAGCCGGATCTTTCTCTTCCTCCTGTCAATAATTGCAGGTGTCCATTGCCAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAGGATATCCTGCAAGGCTTCTGGCTACACCTTCACAAGCTACTATATACACTGGGTGAAGCAGAGGCCTGGACAGGGACTTGAGTGGATTGGATGGATTTATCCTGGAAATGTTAATACTAAGTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACCTCTGAGGACTCTGCGGTCTATTTCTGTGCAAGATCCGAATGGCGCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "TCCCGATTCAACTCTT-1_contig_8": "AGCTCTGACAGAGGAGGCCAGTCCTGGATTTGAGTTCCTCACATTCAGTGATGAGCACTGAACACGGACCCCTCACCATGAATTTCGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCTGTGTGAAGTGAAGCTGGTGGAGTCTGGGGGAGGTTTAGTGCAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATACCATGTCTTGGGTTCGCCAGACTCCAGAGAAGAGGCTGGAGTGGGTCGCATACATTAGTAATGGTGGTGGTAGCACCTACTATCCAGACACTGTAAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACGGCCATGTATTACTGTGCAAGACAGGGTAGTAGTTCCCGTGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CAGATCAGTCTGGTCG-1_contig_5": "GAGGCAGAGAACTTTAGCCCTGTCTTCTTTTTTAGTGTTCAGCACTGACAATATGACATTGAACATGCTGTTGGGGCTGAAGTGGGTTTTCTTTGTTGTTTTTTATCAAGGTGTGCATTGTGAGGTGCAGCTTGTTGAGTCTGGTGGAGGATTGGTGCAGCCTAAAGGGTCATTGAAACTCTCATGTGCAGCCTCTGGATTCACCTTCAATACCTACGCCATGAACTGGGTCCGCCAGGCTCCAGGAAAGGGTTTGGAATGGGTTGCTCGCATAAGAAGTAAAAGTAATAATTATGCAACATATTATGCCGATTCAGTGAAAGACAGGTTCACCATCTCCAGAGATGATTCACAAAGCATGCTCTATCTGCAAATGAACAACTTGAAAACTGAGGACACAGCCATGTATTACTGTGTGAGAGAGGGGGAATCTACTATGATTACGCCCTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GCATGCGGTATATGAG-1_contig_10": "GAAACAACCTATGATCAGTGTCCTCTCTACACAGTCCCTGACAACACTGACTCTAACCATGGGATGGAGCCGGATCTTTCTCTTCCTCCTGTCAATAATTGCAGGTGTCCATTGCCAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAGGATATCCTGCAAGGCTTCTGGCTACACCTTCACAAGCTACTATATACACTGGGTGAAGCAGAGGCCTGGACAGGGACTTGAGTGGATTGGATGGATTTATCCTGGAAATGTTAATACTAAGTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACCTCTGAGGACTCTGCGGTCTATTTCTGTGCAAGAGAAAGAGGATTCATTACTACGGGCTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GTTAAGCCAGGTTTCA-1_contig_2": "GTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGCCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCAGA",
        "TCAATCTAGCACAGGT-1_contig_1": "TGGGGTTGTCATTGCAGTCAGGACTCAGCATGGACATGAGGGCCCCTGCTCAGTTTTTTGGGATCTTGTTGCTCTGGTTTCCAGGTATCAGATGTGACATCAAGATGACCCAGTCTCCATCCTCCATGTATGCATCGCTGGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAAAAGCTATTTAAGCTGGTACCAGCAGAAACCATGGAAATCTCCTAAGACCCTGATCTATTATGCAACAAGCTTGGCAGATGGGGTCCCATCAAGATTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTAACCATCAGCAGCCTGGAGTCTGACGATACAGCAACTTATTACTGTCTACAGCATGGTGAGAGCCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "CATGACATCAGTGCAT-1_contig_1": "TGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCT",
        "TAAACCGAGTTCGATC-1_contig_5": "GATATATGAGCCCTATCTTCTCTACAGACACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTTATCTTCTTCCTGATGGCAGTGGTTACAGGGGTCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCAGAGCTTGTGAAGCCAGGGGCCTCAGTCAAGTTGTCCTGCACAGCTTCTGGCTTCAACATTAAAGACACCTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGAAGGATTGATCCTGCGAATGGTAATACTAAATATGACCCGAAGTTCCAGGGCAAGGCCACTATAACAGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTGCTAGAAACTATAGGGACGACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CTTAGGAAGATAGTCA-1_contig_6": "GGCGTATGAACCTAGCCCTGATTTCCCCAGCCTTCAGTTCCCAGATTCAGTGATCAGCCTTGAACACAGACCTGTCACCATGAAGTTGTGGCTGAACTGGATTTTCCTTGTAACACTTTTAAATGGTATCCAGTGTGAGGTGAAGCTGGTGGAGTCTGGAGGAGGCTTGGTACAGCCTGGGGGTTCTCTGAGACTCTCCTGTGCAACTTCTGGGTTCACCTTCACTGATTACTACATGAGCTGGGTCCGCCAGCCTCCAGGAAAGGCACTTGAGTGGTTGGGTTTTATTAGAAACAAAGCTAATGGTTACACAACAGAGTACAGTGCATCTGTGAAGGGTCGGTTCACCATCTCCAGAGATAATTCCCAAAGCATCCTCTATCTTCAAATGAACACCCTGAGAGCTGAGGACAGTGCCACTTATTACTGTGCAAGGGCCTACGGTAGTAGCTACACCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGT",
        "TTGACTTCAATTCCTT-1_contig_3": "GAGCTCTGACAGAGGAGGCAGGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACGGACCCCTCACCATGAACTTTGTGCTCAGCTTGATTTTCCTTGCCCTCATTTTAAAAGGTGTCCAGTGTGAAGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGCCATGTCTTGGGTTCGCCAGACTCCGGAGAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGTCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGACATAAGAATGACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "AGCAGCCGTGGTGTAG-1_contig_5": "GACAAATCCTCCAGCACAGCCTACATGCAACTGAGCAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAATCTACTATGATTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TATCAGGTCACAATGC-1_contig_10": "GAGTGTGCAGATATGAACAGGCTTACTTCCTCATTGCTGCTGCTGATTGTCCCTGCATATGTCCTGTCCCAGGTTACTCTGAAAGAGTCTGGCCCTGGGATATTGCAGCCCTCCCAGACCCTCAGTCTGACTTGTTCTTTCTCTGGGTTTTCACTGAGCACTTCTGGTATGGGTGTGAGCTGGATTCGTCAGCCTTCAGGAAAGGGTCTGGAGTGGCTGGCACACATTTACTGGGATGATGACAAGCGCTATAACCCATCCCTGAAGAGCCGGCTCACAATCTCCAAGGATACCTCCAGCAACCAGGTATTCCTCAAGATCACCAGTGTGGACACTGCAGATACTGCCACATACTACTGTGCTCGAAGGGTTACTACGGTAGTAGCTACTAATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGCCAAAACAACACCCCCATCAGTCTATCCACTGGCCCCTGGGTGTGGAGATACAACTGGTTCCTCCGTGACTCTGGGATGCCTGGTCAAGGGC",
        "TGTGTTTTCAGGCGAA-1_contig_2": "GATTTCTCTGAGAATTTTGAAGGATAAGAGAGAACTACAACCTGTCTGTCTCAGCAGAGATCAGTAGTACCTGCATTATGGCCTGGACTCCTCTCTTCTTCTTCTTTGTTCTTCATTGCTCAGGTTCTTTCTCCCAACTTGTGCTCACTCAGTCATCTTCAGCCTCTTTCTCCCTGGGAGCCTCAGCAAAACTCACGTGCACCTTGAGTAGTCAGCACAGTACGTACACCATTGAATGGTATCAGCAACAGCCACTCAAGCCTCCTAAGTATGTGATGGAGCTTAAGAAAGATGGAAGCCACAGCACAGGTGATGGGATTCCTGATCGCTTCTCTGGATCCAGCTCTGGTGCTGATCGCTACCTTAGCATTTCCAACATCCAGCCTGAAGATGAAGCAATATACATCTGTGGTGTGGGTGATACAATTAAGGAACAATTTGTGTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCC",
        "GATTCAGTCTACTATC-1_contig_1": "GAAAGCAACATGTGTCCAATGTCCTCTCCACAGACACTGAACACACTGACTCCAACTATGGGATGGAGCTGGATCTTTCTCTTCCTCCTGTCAGGAACTGCAGGTGTCCACTCTGAGGTCCAGCTGCAACAGTCTGGACCTGAGCTGGTGAAGCCTGGAGCTTCAATGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGCTACACCATGAACTGGGTGAAGCAGAGCCATGGAAAGAACCTTGAGTGGATTGGACTTATTAATCCTTACAATGGTGGTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTAACTGTAGACAAGTCATCCAGCACAGCCTACATGGAGCTCCTCAGTCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGACCGGTAGAGGGGGGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCA",
        "CTACATTCACTAGTAC-1_contig_9": "TAGAGGTTGCTATATAGCAGGGAAACATGCAAATAAGGCCTCTCTCTTCTCATGAAAACGAGTCCTGAACTAACCTTGAATCTGAAGCAAAGGGGATCAGCCCGAGATTCTCATTCAGTGATCAACACTGAACACACATCCCTTACCATGGATTTTGGGCTGATTTTTTTTATTGTTGCTCTTTTAAAAGGGGTCCAGTGTGAGGTGAAGCTTCTCGAGTCTGGAGGTGGCCTGGTGCAGCCTGGAGGATCCCTGAAACTCTCCTGTGCAGCCTCAGGATTCGATTTTAGTAGATACTGGATGAGTTGGGTCCGGCAGGCTCCAGGGAAAGGGCTAGAATGGATTGGAGAAATTAATCCAGATAGCAGTACGATAAACTATACGCCATCTCTAAAGGATAAATTCATCATCTCCAGAGACAACGCCAAAAATACGCTGTACCTGCAAATGAGCAAAGTGAGATCTGAGGACACAGCCCTTTATTACTGTGCAAGACCGCGGGATGGTTACTACCTATATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "AGAGTGGCACAGGCCT-1_contig_2": "GGGAGACAGGCAGTGGGAGCAAGATGGATTCACAGGCCCAGGTTCTTATATTGCTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTGGCTGTGTCAGCAGGAGAGAAGGTCACTATGAGCTGCAAATCCAGTCAGAGTCTGCTCAACAGTAGAACCCGAAAGAACTACTTGGCTTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATCTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGCAAGCAATCTTATAATCTTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "ATTGGTGCAGACAAGC-1_contig_1": "TGGGATAGCAAAGTACTTATGAGAATAGTAGTAATTAGCTAGGGACCAAAGTTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATACTGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAGACTCCTGATTTATGACACATCCAACCTGGCTTCTGGAGTCCCTGTTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTTACCCACGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCGGTATCAGATGTGACATCAAGATGACCCAGTCTCCATCCTCCATGTATGCATCGCTGGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAG",
        "GGGCACTTCACCGTAA-1_contig_4": "GATCCTCATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGGGCTGCTTCTCTGCCTGGTGACTTTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCATCTCAGGGTTCTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGTAGTGATATGGAGTGATGGAAGCACAACCTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTCCAAACTGATGACACAGCCATGTACTACTGTGCCAGACAGATCTACTATGATTACGACGGAGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CGAGCACCAAATACAG-1_contig_1": "TGGGGAGAAATACATCAGGCAGGCAAGGGCATCAAGATGAAGTCACAGACCCAGGTCTTCGTATTTCTACTGCTCTGTGTGTCTGGTGCTCATGGGAGTATTGTGATGACCCAGACTCCCAAATTCCTGCTTGTATCAGCAGGAGACAGGGTTACCATAACCTGCAAGGCCAGTCAGAGTGTGAGTAATGATGTAGCTTGGTACCAACAGAAGCCAGGGCAGTCTCCTAAACTGCTGATATACTATGCATCCAATCGCTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATATGGGACGGATTTCACTTTCACCATCAGCACTGTGCAGGCTGAAGACCTGGCAGTTTATTTCTGTCAGCAGGATTATAGCTCTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCC",
        "CTACGTCGTGCACTTA-1_contig_3": "GGGGCTGAGCTGGGGTGAGCTGGGCTGAGCTGAGCTGGGTGAGCTGAGCTGAACTGAGCTGAGCTGGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCAC",
        "CCACGGAGTTATCCGA-1_contig_2": "GGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GGAATAATCGACCAGC-1_contig_6": "GGGGACATATGATCAGTATCCTCTTCACAGTCACTGAAAACATTGACTAATCATGGAAAGGAACTGGATACTTCCTTTTATTATGTCAGTAACTGCAGGTGTCTACTCACAGGTTCAGCTCCAGCAGTCTGGGCCTGAGCTGGCAAGGCCTTGGGCTTCAGTGAAGATATCCTGCCAGGCTTTCTACACCTTTTCCAGAAGGGTGTACTTTGCCATTAGGGATACCAACTACTGGATGCAGTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATCGGGGCTATTTATCCTGGAAATGGTGATACTAGTTACAATCAGAAGTTCAAGGGCAAGGCCACATTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCGCATGATGGTTACTACCCGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GTTAAGCTCGCCGTGA-1_contig_5": "ATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "GTGCGGTAGACGCAAC-1_contig_1": "CAAACTTCCTTTTCTTATATGGGGAGCTGCCAGGAGCCTAATAAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACAGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATATCCTGCAGAGCCAGTGAAAGTGTTGATAGTTATGGCAATAGTTTTATGCACTGGTACCAGCAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCGTGCATCCAACCTAGAATCTGGGATCCCTGCCAGGTTCAGTGGCAGTGGGTCTAGGACAGACTTCACCCTCACCATTAATCCTGTGGAGGCTGATGATGTTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCA",
        "GTTCATTGTTCACGGC-1_contig_2": "GGACGTTTTGGTTCTGTTTGGGTAACTTGTGTGAATTTGTGACATTTTGGCTAAATGAGCCATTCCTGGCAACCTGTGCATCAATAGAAGATCCCCCAGAAAAGAGTCAGTGTGAAAGCTGAGCGAAAAACTCGTCTTAGGCTTCTGAGACCAGTTTTGTAAGGGGAATGTAGAAGAAAGAGCTGGGCTTTTCCTCTGAATTTGGCCCATCTAGTTGGACTGGCTTCACAGGCAGGTTTTTGTAGAGAGGGGCATGTCATAGTCCTCACTGTGGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "CAGGTGCCACCTCGGA-1_contig_6": "GGGGACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGCCACTATGATGGTTACTACGCCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "TTCCCAGCAGGATCGA-1_contig_6": "GGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACC",
        "GACGGCTTCGCCGTGA-1_contig_6": "GGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGAGATCGGAAGAGC",
        "CCGTTCAAGAGCTGCA-1_contig_3": "GGGAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TGACAACAGGCGTACA-1_contig_2": "TGGGGTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCTCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TGAGGGATCCGTTGTC-1_contig_1": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGTGTACCATAGAGCACAGAAATATATTGCCTCATCCTCAGTCTGTGCCCCTGTGATGGTGAGGGCAGCCTTGTCTCCAATCAGGGAGCCTGAGAATCTGGCA",
        "CAAGATCCAGCGTAAG-1_contig_3": "TGGGGATCCTCTTCTCATAGAGCATCCATCAGAGCATGGCTGTCCTGGGGCTGCTTCTCTGCCTGGTGACTTTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCGTCTCAGGGTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGTAGTGATATGGAGTGATGGAAGCACAACCTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTCCAAACTGATGACACAGCCATGTACTACTGTGCCAGAGTACGGGATTACTACAGTGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ATAAGAGGTACCGTAT-1_contig_3": "CTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TTCTTAGAGCTAGTCT-1_contig_4": "GAGAAAACAGCATATGATCAGTGTCCTCTCCAAAGTCCTTGAACATAGACTCTAACCATGGAATGGACCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGAGCTGAGCTGATGAAGCCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTACTGGCTACACATTCAGTAGCTACTGGATAGAGTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGAGATTTTACCTGGAAGTGGTAGTACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACATTCACTGCAGATACATCCTCCAACACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCCGTCTATTACTGTGCAAGAAGGCATTTTCCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GTAACTGAGGATTCGG-1_contig_1": "GAGCTGCCAGGAGCCTAATAAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACAGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATATCCTGCAGAGCCAGTGAAAGTGTTGATAGTTATGGCAATAGTTTTATGCACTGGTACCAGCAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCGTGCATCCAACCTAGAATCTGGGATCCCTGCCAGGTTCAGTGGCAGTGGGTCTAGGACAGACTTCACCCTCACCATTAATCCTGTGGAGGCTGATGATGTTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCTCCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "CCGGGATGTCTTCTCG-1_contig_2": "GGGGACATACCAGCAAGGGAGTGACCAGTTTGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGATTGGCTGTGGAACTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACATGGCTACATATTTCTGTGCAAGAAGGGATTACTATAGGTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "ACAGCCGGTAGCGTAG-1_contig_1": "GGAGCTGCCAGGAGCCTAAGAAGCATTCTCTCTTCCAGTTCTCAGAGATGGAGAAAGACACACTCCTGCTATGGGTCCTGCTTCTCTGGGTTCCAGGTTCCACAGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAGAGCCAGCGAAAGTGTTGATAATTATGGCATTAGTTTTATGAACTGGTTCCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAACCAAGGATCCGGGGTCCCTGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCAGCCTCAACATCCATCCTATGGAGGAGGATGATACTGCAATGTATTTCTGTCAGCAAAGTAAGGAGGTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GGGCACTTCACCGTAA-1_contig_2": "GAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GATCTAGCATTGCGGC-1_contig_3": "GGGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGGGCTGCTTCTCTGCCTGGTGACTTTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCATCTCAGGGTTCTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGTAGTGATATGGAGTGATGGAAGCACAACCTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTCCAAACTGATGACACAGCCATGTACTACTGTGCCAGAACGGCCTACTATAGGTACGACGGGTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CAGCTGGCACATCTTT-1_contig_2": "TTGATCGCTTTCTTATATGGGGAAGCTGGCTATCACTCAAGGCCATGACCATGCTCTCACTAGCTCCTCTCCTCAGCCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCGTGGCTACAGGAGAAAAAGTCACTATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGCACAGATTTTGTTTTTACAATTGAAAACACGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACATGCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GCTGCTTGTCAGATAA-1_contig_4": "TGGGAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGAGACAGACTCTTCTCCACAGTGTCCCCTTCATGTGTAACT",
        "AGGGAGTTCATCACCC-1_contig_1": "GAACATATGTCCAATGTCCTCTCCACAGTCCCTGAAGACACTGACTCTAACCATGGGAGGGATCTGGATCTTTCTCTTCCTCCTGTCAGGAACTGCAGGTGCCCACTCTGAGATCCAGCTGCAGCAGACTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTATTCATTCACTGACTACATCATGCTCTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGAAATATTAATCCTTACTATGGTAGTACTAGCTACAATCTGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAATCTTCCAGCACAGCCTACATGCAGCTCAACAGTCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAACCTTCTATGATGGTTACTACGGGGGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "GTCATTTGTTAAGACA-1_contig_4": "GAGGCAGAGAACTTTAGCCCTGTCTTCTTTTTTAGTGTTCAGCACTGACAATATGACATTGAACATGCTGTTGGGGCTGAAGTGGGTTTTCTTTGTTGTTTTTTATCAAGGTGTGCATTGTGAGGTGCAGCTTGTTGAGTCTGGTGGAGGATTGGTGCAGCCTAAAGGGTCATTGAAACTCTCATGTGCAGCCTCTGGATTCACCTTCAATACCTACGCCATGAACTGGGTCCGCCAGGCTCCAGGAAAGGGTTTGGAATGGGTTGCTCGCATAAGAAGTAAAAGTAATAATTATGCAACATATTATGCCGATTCAGTGAAAGACAGGTTCACCATCTCCAGAGATGATTCACAAAGCATGCTCTATCTGCAAATGAACAACTTGAAAACTGAGGACACAGCCATGTATTACTGTGTGAGGGGTTATGGTAACTTTTTCTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCATGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCATCCAGTGAGCAGTTAACATCTGGAGGTG",
        "GTTAAGCTCGCCGTGA-1_contig_4": "GGGGCCAGGAGCCTAAGAAGCATTCTCTCTTCCAGTTCTCAGAGATGGAGAAAGACACACTCCTGCTATGGGTCCTGCTTCTCTGGGTTCCAGGTTCCACAGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAGAGCCAGCGAAAGTGTTGATAATTATGGCATTAGTTTTATGAACTGGTTCCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAACCAAGGATCCGGGGTCCCTGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCAGCCTCAACATCCATCCTATGGAGGAGGATGATACTGCAATGTATTTCTGTCAGCAAAGTAAGGAGGTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATC",
        "CGGCTAGCAGGAATGC-1_contig_1": "TGGGGGAAATACATCAGGCAGGCAAGGGCATCAAGATGAAGTCACAGACCCAGGTCTTCGTATTTCTACTGCTCTGTGTGTCTGGTGCTCATGGGAGTATTGTGATGACCCAGACTCCCAAATTCCTGCTTGTATCAGCAGGAGACAGGGTTACCATAACCTGCAAGGCCAGTCAGAGTGTGAGTAATGATGTAGCTTGGTACCAACAGAAGCCAGGGCAGTCTCCTAAACTGCTGATATACTATGCATCCAATCGCTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATATGGGACGGATTTCACTTTCACCATCAGCACTGTGCAGGCTGAAGACCTGGCAGTTTATTTCTGTCAGCAGGATTATAGCTCTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAA",
        "GCAAACTCAAACGCGA-1_contig_3": "GATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "GCAGCCATCTACTCAT-1_contig_3": "TGGGAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGA",
        "TACCTTACATACGCTA-1_contig_6": "GGTTTCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTG",
        "CAGCAGCTCCAAGTAC-1_contig_5": "GATCTGAGAGGAAAGGTAGCTTTCTGGAGGTCAGGAGACAGTGCAGAGAAGAACGAGTTACTGTGGACAGGTCTTAGATGGGGAAAGAATGAGCAAATGCAAGCATCAGAAGGGTGGATGCAATGTCCTGCCAAGGACTTACCAAGAGGATCCCCGGACAGAGCAGGCAGGTGGAGTTGACTGAGAGGACAGGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CACACAAAGCATGGCA-1_contig_1": "AGGCTGATCATATCCAGTGCAGCAATCAGTTCCTGCCAGGACACAGTTTAGATATGAGGTTCCAGGTTCAGGTTCTGGGGCTCCTTCTGCTCTGGATATCAGGTGCCCAGTGTGATGTCCAGATAACCCAGTCTCCATCTTATCTTGCTGCATCTCCTGGAGAAACCATTACTATTAATTGCAGGGCAAGTAAGAGCATTAGCAAATATTTAGCCTGGTATCAAGAGAAACCTGGGAAAACTAATAAGCTTCTTATCTACTCTGGATCCACTTTGCAATCTGGAATTCCATCAAGGTTCAGTGGCAGTGGATCTGGTACAGATTTCACTCTCACCATCAGTAGCCTGGAGCCTGAAGATTTTGCAATGTATTACTGTCAACAGCATAATGAATACCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGG",
        "TGATTTCTCTTTACAC-1_contig_1": "GGCTGAGCTGATTTGCATATGGAGTCATTATACAACATCCCAGAATTCTTTAAGGGCAGCTGCCAGGAAGCTAAGAAGCATCCTCTCTTCTAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCCAAAGTGTCAGTACATCTAGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCAAGTATGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATACTGCAACATATTACTGTCAGCACAGTTGGGAGATTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "AACTCCCTCAGCACAT-1_contig_1": "ATCTGGGGGATCTACATCTGAAAGGCAGGTGGAGCAAGATGGAATCACAGACTCAGGTCCTCATGTCCCTGCTGTTCTGGGTATCTGGTACCTGTGGGGACATTGTGATGACACAGTCTCCATCCTCCCTGACTGTGACAGCAGGAGAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGTCTGTTAAACAGTGGAAATCAAAAGAACTACTTGACCTGGTACCAGCAGAAACCAGGTCAGCCTCCTAAACTGTTGATCTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGAACAGATTTCACTCTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGAATGATTATAGTTATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCTTCCCA",
        "TACGGATAGACTACAA-1_contig_1": "GCTCTTTCTGTCTTGATTACTTTTGACTCACCATATCAAGTTCGCAGAATGAGGTTCTCTGCTCAGCTTCTGGGGCTGCTTGTGCTCTGGATCCCTGGATCCACTGCAGATATTGTGATGACGCAGGCTGCATTCTCCAATCCAGTCACTCTTGGAACATCAGCTTCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTACATAGTAATGGCATCACTTATTTGTATTGGTATCTGCAGAAGCCAGGCCAGTCTCCTCAGCTCCTGATTTATCAGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTAGCAGTGGGTCAGGAACTGATTTCACACTGAGAATCAGCAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTGCTCAAAATCTAGAACTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "ACGAGCCTCTGCAAGT-1_contig_5": "CTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "AGCATACTCTCCGGTT-1_contig_2": "TCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGAGGAGGGGGGTCCCCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACCA",
        "GCAAACTCATGCAATC-1_contig_3": "GGGAGAGTCCCTGTGGTCTCTGACTGGTGCAAGGTTTTGACTAAGCGGAGCACCACAGTGCTAACTGGGTCTAGGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGA",
        "TGCGGGTGTAGGGACT-1_contig_1": "GGGTAACTTTCTTATATGGGATTGTCTTTACAATCAGGACTCAGCATGGACATGATGGTCCTTGCTCAGTTTCTTGCATTCTTGTTGCTTTGGTTTCCAGGTGCAAGATGTGACATCCTGATGACCCAATCTCCATCCTCCATGTCTGTATCTCTGGGAGACACAGTCAGCATCACTTGCCATGCAAGTCAGGGCATTAGCAGTAATATAGGGTGGTTGCAGCAGAAACCAGGGAAATCATTTAAGGGCCTGATCTATCATGGAACCAACTTGGAAGATGGAGTTCCATCAAGGTTCAGTGGCAGTGGATCTGGAGCAGATTATTCTCTCACCATCAGCAGCCTGGAATCTGAAGATTTTGCAGACTATTACTGTGTACAGTATGCTCAGTTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "AAATGCCGTACCGGCT-1_contig_3": "GAGAAAAACATGAGATCACAGTTCTCTCTACAGTTACTGAGCACACAGGAACTCACCATGGGATGGAGCTATATCATCCTCTTTTTGGTAGCAACAGCTACAGGTGTCCACTCCCAGGTCCAACTGCAGCAGCCTGGGGCTGAACTGGTGAAGCCTGGGGCTTCAGTGAAGTTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTATATGTACTGGGTGAAGCAGAGGCCTGGACAAGGCCTTGAGTGGATTGGGGGGATTAATCCTAGCAATGGTGGTACTAACTTCAATGAGAAGTTCAAGAGCAAGGCCACACTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTACAAGTCCCCATTACTACGGTAGTAGCTACGTTCCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AGAATAGCAGTATGCT-1_contig_2": "GAGAAGCATCCTCTCTTCTAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCCAAAGTGTCAGTACATCTAGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCAAGTATGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATACTGCAACATATTACTGTCAGCACAGTTGGGAGATTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AACTTTCCAGCGAACA-1_contig_9": "GACAGTAGTCTCTACAGTCACAGGAGTACACAGGGCATTGCCATGGGTTGGAGCTGTATCATCTTCTTTCTGGTAGCAACAGCTACAGGTGTGCACTCCCAGGTCCAGCTGCAGCAGTCTGGGCCTGAGCTGGTGAGGCCTGGGGTCTCAGTGAAGATTTCCTGCAAGGGTTCCGGCTACACATTCACTGATTATGCTATGCACTGGGTGAAGCAGAGTCATGCAAAGAGTCTAGAGTGGATTGGAGTTATTAGTACTTACTCTGGTAATACAAACTACAACCAGAAGTTTAAGGGCAAGGCCACAATGACTGTAGACAAATCCTCCAGCACAGCCTATATGGAACTTGCCAGATTGACATCTGAGGATTCTGCCATCTATTACTGTGCAAGATGGCACTACGGTAGTAGCTATGACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TTCTCAAGTACCAGTT-1_contig_9": "GGGGACTGCTACCTCTGGCCCCACCAGACAATGTTCCTGCAGAACCTGTTACCTTACTTGGCAGGGATTTTTGTCAAGGGATCTACTACTGTGCCTACTATAGGTAGGGATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "GTTAAGCCAGGTTTCA-1_contig_3": "ACAGAAAAACATGAGATCACAGTTCTCTCTACAGTTACTGAGCACACAGGAACTCACCATGGGATGGAGCTATATCATCCTCTTTTTGGTAGCAACAGCAACAGGTGTCCACTCCCAGGTCCAACTGCAGCAGTCTGGGGCTGAACTGGTGAAGCCTGGGGCTTCAGTGAAGTTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTATATGTACTGGGTGAAGCAGAGGCCTGGACAAGGCCTTGAGTGGATTGGAGAGATTAATCCTAGCAATGGTGGTACTAACTTCAATGAGAAGTTCAAGAGCAAGGCCACACTGACTGTAGACAAATCCTCCAGCACAGCATACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTACAAGAAGGTACGGTAGTAGAGGCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CTTAGGAAGTGACATA-1_contig_1": "GAGCTGCCAGGAAGCTAAGAAGCATCCTCTCTTCTAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCCAAAGTGTCAGTACATCTAGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCAAGTATGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATACTGCAACATATTACTGTCAGCACAGTTGGGAGATTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAG",
        "TGGACGCCACACATGT-1_contig_2": "GAACCGACGATCAGTGTCCTCTCCAAAGTCCCTGAACACACTGACTCTAACCATGGAATGGAGTTGGATATTTCTCTTTCTCCTGTCAGGAACTGCAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTAAAGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGATACACATTCACTAGCTATGTTATGCACTGGGTGAAGCAGAAGCCTGGGCAGGGCCTTGAGTGGATTGGATATATTAATCCTTACAATGATGGTACTAAGTACAATGAGAAGTTCAAAGGCAAGGCCACACTGACTTCAGACAAATCCTCCAGCACAGCCTACATGGAGCTCAGCAGCCTGACCTCTGAGGACTCTGCGGTCTATTACTGTGCAAGATCCTCAAATCCCCACCTATGTCTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCAC",
        "CAGAATCGTGTTCGAT-1_contig_4": "ATCAACATATGTCCAATGTCCTCTCCACAGTCCCTGAAGACACTGACTATAACTATGGGATGGACCTGGATCTTTATTTTAATCCTGTCAGTAACTACAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGAGAAGCCTGGCGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGCTACAACATGAACTGGGTGAAGCAGAGCAATGGAAAGAGCCTTGAGTGGATTGGAAATATTGATCCTTACTATGGTGGTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAAGAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGAACTACTTCCTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAA",
        "GACTAACGTGTGGTTT-1_contig_4": "GAGCTCTGACAGAGGAGCCAAGCCCTGGATTCCCAGGTCCTCACATTCAGTGATCAGCACTGAACACAGACCACTCACCATGGACTCCAGGCTCAATTTAGTTTTCCTTGTCCTTATTTTAAAAGGTGTCCAGTGTGATGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGCAGCCTGGAGGGTCCCGGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTTTGGAATGCACTGGGTTCGTCAGGCTCCAGAGAAGGGGCTGGAGTGGGTCGCATACATTAGTAGTGGCAGTAGTACCATCTACTATGCAGACACAGTGAAGGGCCGATTCACCATCTCCAGAGACAATCCCAAGAACACCCTGTTCCTGCAAATGACCAGTCTAAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGGGAGGGGGATTACGAGAGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TACAGTGGTTATGTGC-1_contig_1": "GGGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTGCTGGGGCTGTTTCAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCTCCCATTTTGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGGA",
        "ACCGTAACAATCCGAT-1_contig_3": "GAGCTCTGACAGAGGAGCCAAGCCCTGGATTCCCAGGTCCTCACATTCAGTGATCAGCACTGAACACAGACCACTCACCATGGACTCCAGGCTCAATTTAGTTTTCCTTGTCCTTATTTTAAAAGGTGTCCAGTGTGATGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGCAGCCTGGAGGGTCCCGGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTTTGGAATGCACTGGGTTCGTCAGGCTCCAGAGAAGGGGCTGGAGTGGGTCGCATACATTAGTAGTGGCAGTAGTACCATCTACTATGCAGACACAGTGAAGGGCCGATTCACCATCTCCAGAGACAATCCCAAGAACACCCTGTTCCTGCAAATGACCAGTCTAAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGAGGTTACGACCTTCTTTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CAGCTGGCACATCTTT-1_contig_3": "GGCACATCTTTTGCATATCCTTTTCTATGGGGGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "CGTCACTAGGGTGTGT-1_contig_1": "GGGGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "GATCAGTTCGGGAGTA-1_contig_1": "GAGCTGCCAGGAGCCTAAGAAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACAGGTAACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATATCCTGCAGAGCCAGTGAAAGTGTTGATAGTTATGGCAATAGTTTTATGCACTGGTACCAGCAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTAGGACAGACTTCACCCTCACCATTGATCCTGTGGAGGCTGATGATGCTGCAACCTATTACTGTCAGCAAAATAATGAGGATCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCAGATCGGAAGAGC",
        "GGAACTTTCTGGAGCC-1_contig_2": "GAGCTGCCAGGAGCCTAAGAAGCATCCTCTCATCTAGTTCTCAGAGATGGAGACAGACACAATCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGCTCCACTGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAAGGCCAGCCAAAGTGTTGATTATGATGGTGATAGTTATATGAACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAATCTAGAATCTGGGATCCCAGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TATGCCCAGTCAAGGC-1_contig_8": "TGGGGGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GTCAAGTGTTCCACAA-1_contig_2": "GATACACACTCCAACAGTCATTCTTGGTCAGGAGACGTTGTAGAAATGAGACCGTCTATTCAGTTCCTGGGGCTCTTGTTGTTCTGGCTTCATGGTGCTCAGTGTGACATCCAGATGACACAGTCTCCATCCTCACTGTCTGCATCTCTGGGAGGCAAAGTCACCATCACTTGCAAGGCAAGCCAAGACATTAACAAGTATATAGCTTGGTACCAACACAAGCCTGGAAAAGGTCCTAGGCTGCTCATACATTACACATCTACATTACAGCCAGGCATCCCATCAAGGTTCAGTGGAAGTGGGTCTGGGAGAGATTATTCCTTCAGCATCAGCAACCTGGAGCCTGAAGATATTGCAACTTATTATTGTCTACAGTATGATAATCTTCTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CAGCTAAAGTCCAGGA-1_contig_4": "GACATAACAGCAAGAGAGTGTCCGGTTAGTCTCAAGGAAGACTGAGACACAGTCTTAGATATCATGGAATGGCTGTGGAACTTGCTATTTCTCATGGCAGCAGCTCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAGGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAACTGCTGGAATGCAGTGGGTGCAAAAGATGCCAGGAAAGGGTTTGAAGTGGATTGGCTGGATAAACACCCACTCTGGAGTGCCAAAATATGCAGAAGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCATATTTACAGATAAGCAACCTCAAAAATGAGGACACGGCTACGTATTTCTGTGCGAGATATGGTTACGATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "CTGTGCTAGTACATGA-1_contig_9": "GCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCC",
        "CACAGTAAGGCGATAC-1_contig_6": "GGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTG",
        "TAAACCGTCATGCATG-1_contig_3": "GGGGAGCTCTGACAGAGGAGGCCGGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACAGACACCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGTCCTTATTTTAAAAGGTGTCCAGTGTGAAGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGCCATGTCTTGGGTTCGCCAGTCTCCAGAGAAGAGGCTGGAGTGGGTCGCAGAAATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACACTGTGACGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGGAAATGAGCAGTCTGAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGGGATCCGGATTCCGACGGAGATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "AAGTCTGGTATAGGGC-1_contig_2": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCAC",
        "CAGCCGATCTACTATC-1_contig_5": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GAAGCAGCATGGTCAT-1_contig_2": "GAGAAAAACATGAGATCACAGTTCTCTCTACAGTTACTGAGCACACAGGACCTCACCATGGGATGGAGCTATATCATTTTCTTTTTGGTAGCAACAGCTACAGGTGTCCACTCCCAGGTCCAACTCCAGCAGCCTGGGGCTGAACTGGTGAAGCCTGGGGCTTCAGTGAAGTTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTGGATGCACTGGGTGAAGCTGAGGCCTGGACAAGGCTTTGAGTGGATTGGAGAGATTAATCCTAGCAATGGTGGTACTAACTACAATGAGAAGTTCAAGAGAAAGGCCACACTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTACCCGTAGCTACCGCTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CAGCATAGTTGCTCCT-1_contig_6": "GGACATTGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTTGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGCCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCAGGACCTGACCTGGTGAAACCTTCTCAGTCACTTTCACTCACCTGCACTGTCACTGGCTACTCCATCACCAGTGGTTATAGCTGGCACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATACACTACAGTGGTAGCACTAACTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGACACTATAGATACGACGGCTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCC",
        "CATATTCCAGTCACTA-1_contig_3": "GGGACTCAGCAGGAAGGCTCTGAAGCTCTGAGAGGATTTTGAACAAGTTACTGTCACAGTGAGACAGCTCGGGCTACGAACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCCTTGTCAAAGGCTACTTCCCTGAGCCGGTAACTGTAAAATGGAACTCTGGATCCCTG",
        "CATGACATCAGTGCAT-1_contig_4": "ATCTGAAGCAAAGGGGATCAGCCCGAGATTCTCATTCAGTGATCAACACTGAACACACATCCCTTACCATGGATTTTGGGCTGATTTTTTTTATTGTTGCTCTTTTAAAAGGGGTCCAGTGTGAGGTGAAGCTTCTCGAGTCTGGAGGTGGCCTGGTGCAGCCTGGAGGATCCCTGAAACTCTCCTGTGCAGCCTCAGGATTCGATTTTAGTAGATACTGGATGAGTTGGGTCCGGCAGGCTCCAGGGAAAGGGCTAGAATGGATTGGAGAAATTAATCCAGATAGCAGTACGATAAACTATACGCCATCTCTAAAGGATAAATTCATCATCTCCAGAGACAACGCCAAAAATACGCTGTACCTGCAAATGAGCAAAGTGAGATCTGAGGACACAGCCCTTTATTACTGTGCAAGACCGCTACGGGGCTACGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "TAGGCATGTTACGGAG-1_contig_6": "GACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTG",
        "GGCGACTGTCCTAGCG-1_contig_3": "GGAAAACAACATATGTCCAATGTCCTCTCCACAGGCACTGAACACACTGACTCTAACCATGGGATGGAGCTGGATCTTTCTCTTCTTCCTGTCAGGAACTGCAGGTGTCCTCTCTGAGGTCCTGCTGCAACAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATACCCTGCAAGGCTTCTGGATACACATTCACTGACTACAACATGGACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGAGATATTAATCCTAACAATGGTGGTACTATCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAGTCCTCCAGCACAGCCTACATGGAGCTCCGCAGCCTGACATCTGAGGACACTGCAGTCTATTACTGTGCAAGATTGAGGAGGTACGACGGGTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GCGCGATTCAACGGGA-1_contig_8": "ATGCAAATAGGGCCTCTTTCTCCTCATGAAACGCAGACCAACCTATCCTTGCAGTTCAGACATAGGAGCTTGGCTCTGGTTCCCAAGACCTCTCACTCACTTCTCAACATGGAGTGGGAACTGAGCTTAATTTTCATTTTTGCTCTTTTAAAAGATGTCCAGTGTGAAGTGCAGCTGTTGGAGACTGGAGGAGGCTTGGTGCAACCTGGGGGGTCACGGGGACTCTCTTGTGAAGGCTCAGGGTTTACTTTTAGTGGCTTCTGGATGAGCTGGGTTCGACAGACACCTGGGAAGACCCTGGAGTGGATTGGAGACATTAATTCTGATGGCAGTGCAATAAACTACGCACCATCCATAAAGGATCGATTCACTATCTTCAGAGACAATGACAAGAGCACCCTGTACCTGCAGATGAGCAATGTGCGATCTGAGGACACAGCCACGTATTTCTGTATGAGATATGGTAACTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TGGCTGGGTAGCCTAT-1_contig_5": "GAATGCAATTATCTAGACTTATTTCGGTTGAACATGCTGGTTGGTGGTTGAGAGGACACTCAGTCAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "GATCGATGTCCTAGCG-1_contig_6": "AGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TACGGGCTCACCTCGT-1_contig_4": "GGGAAAACATGAGATCACTGTTCTCTTTACAGTTACTGAGCACACAGGACCTCACCATGGGATGGAGCTGTATCATGCTCTTCTTGGCAGCAACAGCTACAGGTGTCCACTCCCCGGTCCAACTGCAGCAGCCTGGGGCTGAGCTTGTGAAGCCTGGGGCTTCAGTAAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTGGATGCACTGGGTGAAGCAGAGGCCTGGACGAGGCCTTGAGTGGATTGGAAGGATTGATCCTAATAGTGGTGAGTCTGCGAGAAATCCCACCATCTACCCACTGACACTCCCACCAGCTCTGTCAAGTGACCCAGTGATAATCGGCTGCCTGATTCACGATTACTTCCCTTCCGGCACGATGAATGTGACCTGGGGAAAGAGTGGGAAGGATATAACCACCGTAAACTTCCCACCTGCCCTGGCCTCTGGGGGACGGTACACCATGAGCAGCCAGT",
        "TGACAACAGGCGTACA-1_contig_1": "GGGCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TGACTTTGTTAAGGGC-1_contig_3": "CACTTCTGGCCAGATCTTATTACAGATCTTATAAAGTCAGTATGATTTTGCATAAATACATCTTCTTTTAGTGCTCCTGCTCCTTAGAGTATTATAAGAAGATCTCTATCAGGACTGTGTCATGATCCACACAAACTCAGGGAAAGCTCGAAGATGGTTTTCACACCTCAGATTCTTGGACTTATGCTTTTCTGGATTTCAGCCTCCAGAGGTGATATTGTGCTAACTCAGTCTCCAGCCACCCTGTCTGTGACTCCAGGAGATAGAGTCAGTCTTTCCTGCAGGGCCAGTCAAAGTATTAGCAACTACCTACACTGGTATCAACAAAAATCACATGAGTCTCCAAGGCTTCTCATCAAGTATGCTTCCCAGTCCATCTCTGGGATCCCCTCCAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACTCTCAGTATCAACAGTGTGGAGACTGAAGATTTTGGAATGTATTTCTGTCAACAGAGTAATAGCTGGCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CGGACTGCACATCCAA-1_contig_3": "GGGAGAGGCTCTAAAAAGCATGGCTGAGCTGAGATGGGTGGGCTTCTCTGAGCGCTTCTAAAATGCGCTAAACTGAGGTGATTACTCTGAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "ACGCCAGTCGTGGTCG-1_contig_4": "TGGGGAGAAATACATCAGGCAGGCAAGGGCATCAAGATGAAGTCACAGACCCAGGTCTTCGTATTTCTACTGCTCTGTGTGTCTGGTGCTCATGGGAGTATTGTGATGACCCAGACTCCCAAATTCCTGCTTGTATCAGCAGGAGACAGGGTTACCATAACCTGCAAGGCCAGTCAGAGTGTGAGTAATGATGTAGCTTGGTACCAACAGAAGCCAGGGCAGTCTCCTAAACTGCTGATATACTATGCATCCAATCGCTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATATGGGACGGATTTCACTTTCACCATCAGCACTGTGCAGGCTGAAGACCTGGCAGTTTATTTCTGTCAGCAGGATTATAGCTCTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "AGAGCGAAGGTGCTTT-1_contig_1": "GAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGCTGACATCAAGATGACCCAGTCTCCATCCTCCATGTATGCATCGCTGGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAAAAGCTATTTAAGC",
        "ACAGCCGGTCCGAGTC-1_contig_5": "GAGTGTCCTCTCCACAGTCCCTGAACACACTGACTCTCACCATGGAATGGAGCGGGGTCTTTATCTTTCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTCCAGCTGCAGCAGTCTGGAGCTGAGCTGGTAAGGCCTGGGACTTCAGTGAAGATGTCCTGCAAGGCTGCTGGATACACCTTCACTAACTACTGGATAGGTTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGATATTTACCCTGGAGGTGGTTATACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACACATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCCATCTATTACTGTGCAAGATCGGGGGGATTACGACGTCCATTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGTTCTGAGCACAGGT-1_contig_5": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGCCA",
        "TCAGCAATCTTTAGTC-1_contig_3": "GAACTTATGAGCACTGTTTTCTCTACAGTCACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTCATCTTCTTCCTGATGGCAGTGGTTACAGGGGTCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCTGAGCTTGTGAGGCCAGGGGCCTTAGTCAAGTTGTCCTGCAAAGCTTCTGGCTTCAACATTAAAGACTACTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGATGGATTGATCCTGAGAATGGTAATACTATATATGACCCGAAGTTCCAGGGCAAGGCCAGTATAACAGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTGCTAGAGGGAATGATGGTTACCTTTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AACTCAGGTAGAGCTG-1_contig_6": "TGGGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TGGTTCCAGATGTAAC-1_contig_1": "TGGGGTACTTATAGGAAGCACAATAATTAGCCAGAGACTGAGATGGAAAACAAAATGGATTTTCAGATGCAGATTATCAGCTTGCTGCTAATCAGTGTCACAGTCATAGTGTCTAATGGAGAAATTGTGCTCACCCAGTCTCCAACCACCATGGCTGCATCTCCCGGGGAGAAGATCACTATCACCTGCAGTGCCAGCTCAAGTATAAGTTCCAATTACTTGCATTGGTATCAGCAGAAGCCAGGATTCTCCCCTAAACTCTTGATTTATAGGACATCCAATCTGGCTTCTGGAGTCCCAGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATTGGCACCATGGAGGCTGAAGATGTTGCCACTTACTACTGCCAGCAGGGTAGTAGTATACCACTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TGACTAGCAGATGGCA-1_contig_1": "AAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCTTCAGTCATAATGTCCAGAGGACAAATTGTTCTCTCCCAGTCTCCAGCAATCCTGTCTGCATCTCCAGGGGAGAAGGTCACAATGACTTGCAGGGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAAACCCTGGATTTATGCCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGAGTGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCACGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TTGCCGTCATCCAACA-1_contig_3": "TCTGGGGGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTAG",
        "TTATGCTGTCGAGATG-1_contig_6": "GCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTAAGTTAAC",
        "GAGCAGATCAAAGTAG-1_contig_2": "TGGGCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCGC",
        "AACCGCGCAAGGTTTC-1_contig_4": "CTAATTTTTCTTATATGGGGACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGATCTGGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ACGCCAGCACCCTATC-1_contig_6": "AGAGAATGGCAGACAGAGCGGCAGACACATAAGTAGATGGCCAGTCTTAGAGGAAAGATCTACATAGTCTGTGAGGCTTTCAAAGAGGAAAGGACAGCCATGTCACAAAGCAAAAACCCTGGCTCTCCTGAGAGCCTGACGCTTCTGCCCTGTATCCTCTCAGCCCACTTGGAAGGGTCTCAAGGCATCACCACAGCGCCAACCACTAGAGGAGGAAGGAAAGCAATCTGTCAGTGAGGAAGTCACCTAGAGACAGATCTTTCCAGCAATTCTTATCACTCCTCCCTCAGCAGCCTCCATCCAGGCATTAAGAGAACAAGAGATGTTGAAATGTCAGCCTTGTGGCTGGCACAGACCCAGCATTCCATCCAGAGCATTCCTGAAGAGTTAGAAGCCCAGGAGGAAAAAGGCTGGACCTGCACTCAGGATGACCCCTGGACTCGGCCCTGCTGCCTGTGCCTCAAAGGTCCCTTACTCATGCAAAACCAGAGTCCTTACTCATCTGGGCTTTTTGTGAAGGGATCTACTACTGTGTTCATTACTACGGCTAGAGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCC",
        "CGAGCCACATGGTAGG-1_contig_2": "ACTCCTCCCTCAGCAGCCTCCATCCAGGCATTAAGAGAACAAGAGATGTTGAAATGTCAGCCTTGTGGCTGGCACAGACCCAGCATTCCATCCAGAGCATTCCTGAAGAGTTAGAAGCCCAGGAGGAAAAAGGCTGGACCTGCACTCAGGATGACCCCTGGACTCGGCCCTGCTGCCTGTGCCTCAAAGGTCCCTTACTCATGCAAAACCAGAGTCCTTACTCATCTGGGCTTTTTGTGAAGGGATCTACTACTGTGTTCATTACTACGGCTACTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACC",
        "TGGGAAGCACGTCAGC-1_contig_1": "GAATCTGATCATACACACTCCAACAGTCATTCTTGGTCAGGAGACGTTGTAGAAATGAGACCGTCTATTCAGTTCCTGGGGCTCTTGTTGTTCTGGCTTCATGGTGCTCAGTGTGACATCCAGATGACACAGTCTCCATCCTCACTGTCTGCATCTCTGGGAGGCAAAGTCACCATCACTTGCAAGGCAAGCCAAGACATTAACAAGTATATAGCTTGGTACCAACACAAGCCTGGAAAAGGTCCTAGGCTGCTCATACATTACACATCTACATTACAGCCAGGCATCCCATCAAGGTTCAGTGGAAGTGGGTCTGGGAGAGATTATTCCTTCAGCATCAGCAACCTGGAGCCTGAAGATATTGCAACTTATTATTGTCTACAGTATGATAATCTTCTATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GTAACTGGTTGGGACA-1_contig_2": "GGGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CTGATCCCAGAAGCAC-1_contig_4": "GTCACAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "CCCAGTTTCATACGGT-1_contig_2": "GGGATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGTGCAAGTCAGGGCATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTATTACACATCAAGTTTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGGACAGATTATTCTCTCACCATCAGCAACCTGGAACCTGAAGATATTGCCACTTACTATTGTCAGCAGTATAGTAAGCTTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GCGCAGTAGTTGTAGA-1_contig_3": "GGGGCTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAAC",
        "CCTCTGAGTTGTGGAG-1_contig_4": "ACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCTATGAGCTGCAAGTCCAGTCAGAGTCTGTTAAACAGTGGAAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGC",
        "GTGCTTCCAGGTCGTC-1_contig_4": "GGGCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTG",
        "ACGCAGCTCACGGTTA-1_contig_1": "TGGGGAGAAATACATCAGGCAGGCAAGGGCATCAAGATGAAGTCACAGACCCAGGTCTTCGTATTTCTACTGCTCTGTGTGTCTGGTGCTCATGGGAGTATTGTGATGACCCAGACTCCCAAATTCCTGCTTGTATCAGCAGGAGACAGGGTTACCATAACCTGCAAGGCCAGTCAGAGTGTGAGTAATGATGTAGCTTGGTACCAACAGAAGCCAGGGCAGTCTCCTAAACTGCTGATATACTATGCATCCAATCGCTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATATGGGACGGATTTCACTTTCACCATCAGCACTGTGCAGGCTGAAGACCTGGCAGTTTATTTCTGTCAGCAGGATTATAGCTCTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAA",
        "TTCGAAGAGCTAACAA-1_contig_1": "TGGGGAGAAATACATCAGGCAGGCAAGGGCATCAAGATGAAGTCACAGACCCAGGTCTTCGTATTTCTACTGCTCTGTGTGTCTGGTGCTCATGGGAGTATTGTGATGACCCAGACTCCCAAATTCCTGCTTGTATCAGCAGGAGACAGGGTTACCATAACCTGCAAGGCCAGTCAGAGTGTGAGTAATGATGTAGCTTGGTACCAACAGAAGCCAGGGCAGTCTCCTAAACTGCTGATATACTATGCATCCAATCGCTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATATGGGACGGATTTCACTTTCACCATCAGCACTGTGCAGGCTGAAGACCTGGCAGTTTATTTCTGTCAGCAGGATTATAGCTCTCCTCCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "TCAGCTCAGAGTGACC-1_contig_1": "TGGGGAACAGCATCAGACAGGCTGGGCAGCAAGATGGAATCACAGACCCAGGTCCTCATGTTTCTTCTGCTCTGGGTATCTGGTGCCTGTGCAGACATTGTGATGACACAGTCTCCATCCTCCCTGGCTATGTCAGTAGGACAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTAAATAGTAGCAATCAAAAGAACTATTTGGCCTGGTACCAGCAGAAACCAGGACAGTCTCCTAAACTTCTGGTATACTTTGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCATAGGCAGTGGATCTGGGACAGATTTCACTCTTACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGATTACTTCTGTCAGCAACATTATAGCACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CTTAACTCACTGTTAG-1_contig_4": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCA",
        "AGAGTGGTCTCCAACC-1_contig_4": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAA",
        "GCATGCGTCGCATGGC-1_contig_1": "TGGGGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTCCTTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AAGGCAGCACGGTTTA-1_contig_2": "GAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATACTGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTATAAGTTACATGCACTGGTACCAGCAGAAGCCAGGCACCTCCCCCAAAAGATGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTATTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCATCAGCGGAGTAGTTACCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCAG",
        "CATATGGCAGCTTAAC-1_contig_1": "GGGGAAGGGCAGCTGCCAGGAAGCTAAGAAGCATCCTCTCTTCTAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCCAAAGTGTCAGTACATCTAGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCAAGTATGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATACTGCAACATATTACTGTCAGCACAGTTGGGAGATTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TGGTTAGAGAGTACAT-1_contig_2": "GACCCTGCAGCTCTGACAGAGGAGGCCAGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTGATCAGCACTGAACACGGACCCCTCACCATGAACTTGGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCAGTGTGAAGTGAAGCTGGTGGAGTCTGGGGGAGGCTTAGTGCAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAACCTCTGGATTCACTTTCAGTGACTATTACATGTATTGGGTTCGCCAGACTCCAGAGAAGAGGCTGGAGTGGGTCGCATACATTAGTAATGGTGGTGGTAGCACCTATTATCCAGACACTGTAAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCCGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTGCAAGACATGGATGATTACGGGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GGGAATGGTAAGCACG-1_contig_5": "GGAACATATGTCCAATGTCCTCTCCACAGTCCCTGAAGACACTGACTCTAACCATGGGAGGGATCTGGATCTTTCTCTTCCTCCTGTCAGGAACTGCAGGTGCCCACTCTGAGATCCAGCTGCAGCAGACTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTATTCATTCACTGACTACATCATGCTCTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGAAATATTAATCCTTACTATGGTAGTACTAGCTACAATCTGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAATCTTCCAGCACAGCCTACATGCAGCTCAACAGTCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGATCGGATTCCCACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "AGAGCGAAGGTGCTTT-1_contig_5": "GGGGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "TATCAGGTCACAATGC-1_contig_5": "TGGGGCATTTCTGTCTTGACTACTCAAGACTTTTTGTATCAAGTTCTCAGAATGAGGTGCCTAGCTGAGTTCCTGGGGCTGCTTGTGCTCTGGATCCCTGGAGCCATTGGGGATATTGTGATGACTCAGGCTGCACCCTCTGTACCTGTCACTCCTGGAGAGTCAGTATCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTGCATAGTAATGGCAACACTTACTTGTATTGGTTCCTGCAGAGGCCAGGCCAGTCTCCTCAGCTCCTGATATATCGGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTGGCAGTGGGTCAGGAACTGCTTTCACACTGAGAATCAGTAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTATGCAACATCTAGAATATCCTTTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TACTCATTCATTCACT-1_contig_1": "GGGGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTCCTCACACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACA",
        "CTACATTCACTAGTAC-1_contig_6": "GCACTTACTGTTTTCTCTTTCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGAATTGAAGTCTCAACTATGAAAATCAGCAGTTCTCTGTCAGAGAAGCCCAAGCGCTTCCACGCATGCTTGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TTGGCAATCACATGCA-1_contig_7": "GTTAGTCAATCATATGATGGAAAGAGGGTAGCATTCACCTCTCTGGGACAAAGGCTGTGACTCTGGGAAAGACAAGAGAAGGGCAGGACCAAAACAGGAACAGAGACGGCTGCTTTCACAGCTTCCACATGTGAGTGGGGTCAGCAGGGAAAGGAGCTGCACGAAGAGGCCATACAAACAGCACGCATCTGTGGCCCTTCCAGATCTTTGAGTCATCCTATCACGGGAGATTGGGAAGGAGTTGACAGACCAGCCCAGGCAGTGGAAGCCTCTGTGTTAAAGAGTAAAGGTGCTTGCCTACAGCCTGGTGTCAACTAGGCAGGCCCTGGGGGGCCGGGAAGGGGCCTCCTAGACAAGCACAGGCATGTAGAGCTGCACACCCCACAGACAAACCTGAGCCCCGAGGATATCATGGAATATATCGAGAAGCCTGAGGAATGTGTTTGGCATGGACTACAGGTTGAGAGAACCAAGGAAGCTGAGCCCTGCGCCAAAACGACACCCCCATCTGTCTATCCACTGGCCCCTGGATCTGCTGCCCAAACTAACTCCATGGTGACCCTGGGATGCCTGGTCAAGGG",
        "ACAGCCGCACGAAAGC-1_contig_3": "TCACATCTCTCTCATTAGAGGTTGATCTTTGAGGAAAACAGGGTGTTGCCTAAAGGATGAAAGTGTTGAGTCTGTTGTACCTGTTGACAGCCATTCCTGGTATCCTGTCTGATGTACAGCTTCAGGAGTCAGGACCTGGCCTCGTGAAACCTTCTCAGTCTCTGTCTCTCACCTGCTCTGTCACTGGCTACTCCATCACCAGTGGTTATTACTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATAAGCTACGACGGTAGCAATAACTACAACCCATCTCTCAAAAATCGAATCTCCATCACTCGTGACACATCTAAGAACCAGTTTTTCCTGAAGTTGAATTCTGTGACTACTGAGGACACAGCTACATATTACTGTGCAAGAGAAGCCCTAGGGTACGGTAGTAGCCTCTACTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCAC",
        "CAGGTGCCACCTCGGA-1_contig_5": "GAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATATCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAGGCACCTCCCCCAAAAGATGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCACCCACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "CTAAGACAGGCGCTCT-1_contig_4": "GGGAAGCAAAGGGGATCAGCCCGAGATTCTCATTCAGTGATCAACACTGAACACACATCCCTTACCATGGATTTTGGGCTGATTTTTTTTATTGTTGCTCTTTTAAAAGGGGTCCAGTGTGAGGTGAAGCTTCTCGAGTCTGGAGGTGGCCTGGTGCAGCCTGGAGGATCCCTGAAACTCTCCTGTGCAGCCTCAGGATTCGATTTTAGTAGATACTGGATGAGTTGGGTCCGGCAGGCTCCAGGGAAAGGGCTAGAATGGATTGGAGAAATTAATCCAGATAGCAGTACGATAAACTATACGCCATCTCTAAAGGATAAATTCATCATCTCCAGAGACAACGCCAAAAATACGCTGTACCTGCAAATGAGCAAAGTGAGATCTGAGGACACAGCCCTTTATTACTGTGCAAGACCGGATAGGTTTTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "ACGCCAGCACCCTATC-1_contig_3": "AAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GCATGATAGACAGGCT-1_contig_8": "GGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TCGGTAACAATACGCT-1_contig_7": "GGGGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCAC",
        "AACTCAGCACATGACT-1_contig_6": "GGGCAAGAGGGGGATTACTCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GGTGTTATCCGTAGGC-1_contig_6": "TGGGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTTATTTCCAGCTTGGTCCCCCCTCCGAACGTGTACGGAACATGTG",
        "CTAACTTGTTCCATGA-1_contig_2": "GAGTTCTAGAGCTTCCAGGTGTGCTGTCAGGCTGCTAGTGTATGCTCTCTCTAGTTTCTTTTTGGAGGCACTCAGAGCTATGAGTTTTCCTCTTAGGACTGCCTTCATTGTGTCCCATAAGTTTGGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTG",
        "ACATGGTAGATGCCTT-1_contig_1": "GGGGTGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAA",
        "TACAGTGGTAAACACA-1_contig_1": "TGGGGAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGCATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATAACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTTCCAGCAGAAGCCAGGCACTTCTCCCAAACTCTGGATTTATAGCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGATCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAAAGGAGTAGTTACCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCG",
        "CGAGCCATCTGCCAGG-1_contig_2": "GGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CCATTCGCATCAGTAC-1_contig_4": "GGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CAGCTGGGTTCGGCAC-1_contig_3": "GGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CACACAAAGCATGGCA-1_contig_4": "ATCTGGGGGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCAC",
        "CTACATTGTATTCTCT-1_contig_3": "GAATCAACCGACGATCAGTGTCCTCTCCAAAGTCCCTGAACACACTGACTCTAACCATGGAATGGAGTTGGATATTTCTCTTTCTCCTGTCAGGAACTGCAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTAAAGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGATACACATTCACTAGCCATGTTATGCACTGGGTGAAGCAGAAGCCTGGGCAGGGCCTTGAGTGGATTGGATATATTAATCCTTACAATGATGGTACTAAGTACAATGAGAAGTTCAAAGGAAAGGCCACACTGACTTCAGACAAATCCTCCAGCACAGCCTACATGGAGTTCAGCAGCCTGACCTCTGAGGACTCTGCGGTCTATTACTGTGCAAGAGGCGGGGAACTTGGGGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TTTGTCACAAGCGTAG-1_contig_1": "GAGAGTGACTCTAGCAGTGGGATCCTGTCCTGAGTTCCCCAATCTTCACATTCAGAAATCAGCACTCAGTCCTGTCACTATGAAGTTGTGGTTAAACTGGGTTTTTCTTTTAACACTTTTACATGGTATCCAGTGTGAGGTGAAGCTGGTGGAATCTGGAGGAGGCTTGGTACAGCCTGGGGGTTCTCTGAGACTCTCCTGTGCAACTTCTGGGTTCACCTTCAGTGATTTCTACATGGAGTGGGTCCGCCAGCCTCCAGGGAAGAGACTGGAGTGGATTGCTGCAAGTAGAAACAAAGCTAATGATTATACAACAGAGTACAGTGCATCTGTGAAGGGTCGGTTCATCGTCTCCAGAGACACTTCCCAAAGCATCCTCTACCTTCAGATGAATGCCCTGAGAGCTGAGGACACTGCCATTTATTACTGTGCAAGAGATTATTACTACGGTAGTAGCTACTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTCTTGGGAGACAGCAACACCT",
        "AGGGAGTTCATCACCC-1_contig_2": "GGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGAGACAGACTCTTCTCCACAGTGTCCCCTTCATGTG",
        "CCGTTCAGTTATGTGC-1_contig_1": "GGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTAGGCCACTG",
        "TTTGGTTCATTTCACT-1_contig_1": "TGGGGAGCTGCCAGGAGCCTAAGAAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACAGGTAACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATATCCTGCAGAGCCAGTGAAAGTGTTGATAGTTATGGCAATAGTTTTATGCACTGGTACCAGCAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTAGGACAGACTTCACCCTCACCATTGATCCTGTGGAGGCTGATGATGCTGCAACCTATTACTGTCAGCAAAATAATGAGGATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCT",
        "CACAGGCGTCTCCACT-1_contig_2": "GCCGCCTCTTTTTCTTATATGGGGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GGAACTTTCTGGAGCC-1_contig_3": "GCCTATCTCTCTCACTGGAGGCTGATTTTTGAAGAAAGGGGTTGTAGCCTAAAAGATGATGGTGTTAAGTCTTCTGTACCTGTTGACAGCCCTTCCGGGTATCCTGTCAGAGGTGCAGCTTCAGGAGTCAGGACCTAGCCTCGTGAAACCTTCTCAGACTCTGTCCCTCACCTGTTCTGTCACTGGCGACTCCATCACCAGTGGTTACTGGAACTGGATCCGGAAATTCCCAGGGAATAAACTTGAGTACATGGGGTACATAAGCTACAGTGGTAGCACTTACTACAATCCATCTCTCAAAAGTCGAATCTCCATCACTCGAGACACATCCAAGAACCAGTACTACCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGATATGACAGCTCGGTCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "GCAGTTAGTCTAGTCA-1_contig_1": "GGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TAAGCGTTCACTGGGC-1_contig_1": "TGGGGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCC",
        "GTCAAGTTCGACCAGC-1_contig_4": "TGGGGCTATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGGGCTGCTTCTCTGCCTGGTGACGTTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCACCCTCACAGAGCCTGTCCATCACATGCACTGTCTCTGGGTTCTCATTATCCAGATATAGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAATGATATGGGGTGGTGGAAGCACAGACTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGAAATGGTAACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "AGAGTGGTCTCCAACC-1_contig_1": "GAAAGCAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATATCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAGGCACCTCCCCCAAAAGATGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCACCCACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCTGACTCTCTGAGGA",
        "CCATGTCTCACCTTAT-1_contig_2": "TGGGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GTCAAGTTCGGAAACG-1_contig_2": "GTCACATACCAGCAAGCGAGTGACCAGTTAGTCTTAAGGCACCACTTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCAAGATCTTCCTATGATTACCTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TTCCCAGCAGGATCGA-1_contig_4": "GGGGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGAGCTGCTCCTCTCCCTGGTGACATTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTCGTTGCGCCCTCACAGAGCCTGTCCATCACATGCACTGTCTCTGGTTTCTCATTAACCAACTCTGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGGTGATGGAAGCACAAATTATAATTCAGCTTTCAAATCCAGACTGAGCATCAGCAAGGACAATTCCAAGAGTCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCAGGTACTACTGTGCCAAACCGGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CCTCTGAGTTGTGGAG-1_contig_3": "AGTCAGAGACCAGGATCAAAAACACAATGGATTTTCATGTGCAGATTTTCAGCTTCATGCTAATCAGTGTCACAGTCATATTGTCCAGTGGAGAAATTGTGCTCACCCAGTCTCCAGCACTCATGGCTGCATCTCCAGGGGAGAAGGTCACCATCACCTGCAGTGTCAGCTCAAGTATAAGTTCCAGCAACTTGCACTGGTACCAGCAGAAGTCAGAAACCTCCCCCAAACCCTGGATTTATGGCACATCCAACCTGGCTTCTGGAGTCCCTGTTCGCTTCAGTGGCAGTGGATCTGGGACCTCTTATTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGTCAACAGTGGAGTAGTTACCCACTCACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCTATGAGCTGCAAGTCCAGTCAGAGTCTGTTAAACAGTGGAAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGC",
        "CGTCAGGAGCGAGAAA-1_contig_5": "ACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CGTCACTAGGGTGTGT-1_contig_2": "GGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTTTGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "ATTTCTGTCCCGGATG-1_contig_2": "GCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GCGACCAGTTGGTGGA-1_contig_2": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "ATCATCTGTGCTCTTC-1_contig_1": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTG",
        "GTCTCGTTCTACCTGC-1_contig_3": "GAGCATAAGATCACTGTTCTCTCTACAGTTACTGAGCACACAGGATCTCACCATGGGATGGAGCTATATCATCCTCTTCTTGTTAGCAACAGCTACATGTGTCCACTCCCAGGTCCAACTGCAGCAGCCTGGGGCTGAGCTGGTGAGGCCTGGGGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACGTTCACCAGCTACTGGATGAACTGGGTTAAGCAGAGGCCTGAGCAAGGCCTTGAGTGGATTGGAAGGATTGATCCTTACGATAGTGAAACTCACTACAATCAAAAGTTCAAGGACAAGGCCATATTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGAGGTATTACTACGGTAGTAGCTGTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGT",
        "TCCCGATTCAACTCTT-1_contig_5": "TGGGGGAAATGCATCACACCAGCATGGGCATCAAAATGGAGTCACAGATTCAGGTCTTTGTATTCGTGTTTCTCTGGTTGTCTGGTGTTGACGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGAGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGACAATCTCCTAAACTACTGATTTACTCGGCATCCTACCGGTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATCTGGGACGGATTTCACTTTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAACATTATAGTACTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "CCATTCGTCAACTCTT-1_contig_4": "GAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATATCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAGGCACCTCCCCCAAAAGATGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCACTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AACCGCGAGAAACGCC-1_contig_3": "GAGACACAAGTGAATGACAGATGGACCTCCATCTACTCTTATTTTAAAAAGAAGACAAACCCCACAGGCTCGAGAACTTTAGCGACTGTTTTGAGAGAAATCATTGGTCCCTGACTCAAGAGATGACTGGCAGATTGGGGATCAGAATACCCATACTCTGTGGCTAGTGTGAGGTTTAAGCCTCAGAGTCCCTGTGGTCTCTGACTGGTGCAAGGTTTTGACTAAGCGGAGCACCACAGTGCTAACTGGTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCATG",
        "TGAGGGAGTCCGAGTC-1_contig_1": "GGGACTTATGAGAATAGCAGTAATTAGCTAGGAACAAAAATTCAAAGACAAAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGTCTCAGTCATAATGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCTGGGGAGAAGGTCACCTTGACCTGCAGTGCCAGCTCAAGTGTAAGTTCCAGCTACTTGTACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAAACTCTGGATTTATAGCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCTCTTATTTCTGCCATCAGTGGAGTAGTTACCCACCCACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AAATGCCGTACCGGCT-1_contig_1": "GGATATGAAATGCATCACACCAGCATGGGCATCAAAATGGAGTCACAGATTCAGGTCTTTGTATTCGTGTTTCTCTGGTTGTCTGGTGTTGACGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGAGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGACAATCTCCTAAACTACTGATTTACTCGGCATCCTACCGGTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATCTGGGACGGATTTCACTTTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAACATTATAGTACTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "ACGTCAAGTGTGACCC-1_contig_2": "GTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAATACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TACTCGCAGGATGTAT-1_contig_5": "GGGGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGCCGAGACTAACAAGGCCACACTGGTGTGT",
        "GCGCGATTCAACGGGA-1_contig_1": "ATGGTTTCTTATATGGGGGTTGTCATTGCAGTCAGGACTCAGCATGGACATGAGGGCCCCTGCTCAGTTTTTTGGGATCTTGTTGCTCTGGTTTCCAGGTATCAGATGTGACATCAAGATGACCCAGTCTCCATCCTCCATGTATGCATCGCTGGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAAAAGCTATTTAAGCTGGTACCAGCAGAAACCATGGAAATCTCCTAAGACCCTGATCTATTATGCAACAAGCTTGGCAGATGGGGTCCCATCAAGATTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTAACCATCAGCAGCCTGGAGTCTGACGATACAGCAACTTATTACTGTCTACAGCATGGTGAGAGCCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "AACGTTGGTTATGCGT-1_contig_1": "TGGCCACATCCAGTGCAGTAATCAATGTCAGGTCACAGCAGAAACATGAAGTTTCCTTCTCAACTTCTGCTCTTACTGCTGTTTGGAATCCCAGGCATGATATGTGACATCCAGATGACACAATCTTCATCCTCCTTTTCTGTATCTCTAGGAGACAGAGTCACCATTACTTGCAAGGCAAGTGAGGACATATATAATCGGTTAGCCTGGTATCAGCAGAAACCAGGAAATGCTCCTAGGCTCTTAATATCTGGTGCAACCAGTTTGGAAACTGGGGTTCCTTCAAGATTCAGTGGCAGTGGATCTGGAAAGGATTACACTCTCAGCATTACCAGTCTTCAGACTGAAGATGTTGCTACTTATTACTGTCAACAGTATTGGAGTACTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "ATCGAGTGTCATATCG-1_contig_5": "GGGGGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GGCTGGTCAACACCTA-1_contig_5": "CTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCT",
        "GGATGTTGTATCGCAT-1_contig_3": "GAGCTCTGACAGAGGAGGCCTGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACGGACCCCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCAGTGTGAAGTGATGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGCCATGTCTTGGGTTCGCCAGACTCCGGAGAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGGCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAGGTCTGAGGACA",
        "CGACTTCAGTGGCACA-1_contig_6": "AACAGACACACAAACCTGGACTCACAAGTTTTTCTCTTCAGTGACAGACACAGACATAGAACATTCACGATGTACTTGGGACTGAACTATGTATTCATAGTTTTTCTCTTAAATGGTGTCCAGAGTGAAGTGAAGCTTGAGGAGTCTGGAGGAGGCTTGGTGCAACCTGGAGGATCCATGAAACTCTCTTGTGCTGCCTCTGGATTCACTTTTAGTGACGCCTGGATGGACTGGGTCCGCCAGTCTCCAGAGAAGGGGCTTGAGTGGGTTGCTGAAATTAGAAGCAAAGCTAATAATCATGCAACATACTATGCTGAGTCTGTGAAAGGGAGGTTCACCATCTCAAGAGATGATTCCAAAAGTAGTGTCTACCTGCAAATGAACAGCTTAAGAGCTGAAGACACTGGCATTTATTACTGTATAACTGGGACCATTTATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TTATGCTCACTTCGAA-1_contig_3": "GAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGTATCTGGAGTCCCTGATCGCTTCATAGGCAGTGGATCTGGGACGGATTTCACTCTGACCATCAACAGTGTGCAGGCTGAAGATCTGGCTGTTTATTACTGTCAGCAG",
        "TCGAGGCAGTGTGGCA-1_contig_5": "AAGATCACAGTAGTCTCTACAGTCACAGGAGTACACAGGGCATTGCCATGGGTTGGAGCTGTATCATCTTCTTTCTGGTAGCAACAGCTACAGGTGTGCACTCCCAGGTCCAGCTGCAGCAGTCTGGGCCTGAGGTGGTGAGGCCTGGGGTCTCAGTGAAGATTTCCTGCAAGGGTTCCGGCTACACATTCACTGATTATGCTATGCACTGGGTGAAGCAGAGTCATGCAAAGAGTCTAGAGTGGATTGGAGTTATTAGTACTTACAATGGTAATACAAACTACAACCAGAAGTTTAAGGGCAAGGCCACAATGACTGTAGACAAATCCTCCAGCACAGCCTATATGGAACTTGCCAGATTGACATCTGAGGATTCTGCCATCTATTACTGTGCAAGATTTCATTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GATGCTACACTAAGTC-1_contig_2": "GGGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATG",
        "TTTGTCACAAGCGTAG-1_contig_4": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTGATTTCCAGCTTGGTGCCTCCACCGAACGTCCACGAGGAGAGGTATTGATGACAGTAAT",
        "GATGCTAGTTGTCTTT-1_contig_6": "GCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGCTGA",
        "TTGACTTCAAGAAGAG-1_contig_4": "GAAACACCTCGGCCCTGACCCTGCAGCTCTGACAGAGGAGGCCGGTCCTGGATTCGAGTTCCTCACATTCAGTGATGAGCACTGAACACGGACCCCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCAGTGTGAAGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTGACTATTACATGTATTGGGTTCGCCAGACTCCGGAAAAGAGGCTGGAGTGGGTCGCAACCATTAGTGATGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGGCGATTCACCATCTCCAGAGACAATGCCAAGAACAACCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTGCAAGAGGGTCCTATGGTTACGACTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACCA",
        "GTCAAGTGTTCCACAA-1_contig_3": "GCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTG",
        "ACTATCTAGAGGTTAT-1_contig_1": "GAATTAGCTAGGGACCAAAATTCAAAGACAGAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATATCCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAAACCCTGGATTTATCGCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTATCATAGTTACCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTC",
        "ACGCCAGGTGTCAATC-1_contig_1": "AGCTCTATGTTAAATCCCCTCCTTGGATGTAGGGGTCACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGATGGGATACTACGGTAGTCTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CTGTGCTAGCCCAGCT-1_contig_6": "GGGTACTCACAAGTGTGCAGCCATGGGCAGGCTTACTTCTTCATTCCTGCTACTGATTGTCCCTGCATATGTCCTGTCCCAGGTTACTCTGAAAGAGTCTGGCCCTGGGATATTGCAGCCCTCCCAGACCCTCAGTCTGACTTGTTCTTTCTCTGGGTTTTCACTGAGCACTTCTGGTATGGGTGTAGGCTGGATTCGTCAGCCATCAGGGAAGGGTCTGGAGTGGCTGGCACACATTTGGTGGGATGATGTCAAGCGCTATAACCCAGCCCTGAAGAGCCGACTGACTATCTCCAAGGATACCTCCAGCAGCCAGGTATTCCTCAAGATCGCCAGTGTGGACACTGCAGATACTGCCACATACTACTGTGCTCGAATAGCTCGATTCTATTACTACGGTAGTAGCTACGGGTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CACAGGCGTTGGTTTG-1_contig_2": "GGGATCCTCTTCTCATAGAGCCTCCATCAGACCATGGCTGTCCTGGAGCTGCTCCTCTCCCTGGTGACATTCCCAAGCTGTGTCCTGTCCCAGGTGCAGATGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATCACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGTGGTGGAAGCACAGACTATAATGCAGCTTTCATATCCAGACTGAGCATCAGCAAGGACAATTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTGATGACACAGCCATATACTACTGTGTCACCCCCTACTATAGGTACGACGGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CATGGCGGTGTCGCTG-1_contig_4": "GAGTTCTAGAGCTTCCAGGTGTGCTGTCAGGCTGCTAGTGTATGCTCTCTCTAGTTTCTTTTTGGAGGCACTCAGAGCTATGAGTTTTCCTCTTAGGACTGCCTTCATTGTGTCCCATAAGTTTGGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTG",
        "TACCTTACATACGCTA-1_contig_4": "TGGGGAACCTATGATCAGTGTCATCTCCACAGTCCCTGAACACACTGACTCTAACCATGGAATGGAGGATCTTTCTCTTCATCCTGTCAGGAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGATACACATTCACTGACTATGTTATAAACTGGGTGAAGCAGAGAACTGGACAGGGCCTTGAGTGGATTGGAGAGATTTATCCTGGAAGTGGTAGTACTTACTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAACACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTTCTGTGCAAGACGCTACGCTAACTGGGAGAGAAAGGGACTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCCTTGTCAAAGGCTACTTCCCTGAGCCGGTAACTGTAAAATGGAACTATGGAGCCCTGTCCAGC",
        "CAGCATAGTTGCTCCT-1_contig_7": "GGTTCACACCACTGGGGCAAGGACTGTGAATCCTGGAAGGATGAGGCAGAGCTCAGAGCTGGGCTGTTAGAAGCACAGTGACAAACAAAAAGAGCAGCAGTCTTCTTCACGCGGCTCCACATGTGAGTGTGGTCAGCTGAGAGACTGGAGAGAGCCACAGCCTAGGGAGAGCACTGGGCCTTTCCAGAACTAATAGATGGAAGATCACAGGAGGATGGGGAGGAGTTGGCAGATCAGCCCCCCAAAAAGGGCCTGTGTGTTGCAGAGAAAAGGTGCCTACCTGCAACCTGGTGCCAACCAGGAAGAGTCCAGAGTTCTCACACACAGAAGAATGGACCAGTAAGCTGTGCACACCTACAGACAACCAGACCAAAACAACACCCCCATCAGTCTATCCACTGGCCCCTGGGTGTGGAGATACAACTGGTTCCTCCGTGACTCTGGGATGCCTGGTCAAGGGCTACTTCCCTGAGTCAGTGACTGTGACTTGGAACTCTGGATCCCTGTCCAGCAGTGTGCACACCTTCCCAGCTCTCCTGCAGTCTGGACTCTACACTATGAGCAGCTCAGTGACTGTCCCCTCCAGCACCTGGCCAAGTCAGACCGTCACCT",
        "CGTCAGGAGCGAGAAA-1_contig_1": "GGGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTAATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGCA",
        "GGTATTGAGAGTACAT-1_contig_2": "GGGGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACC",
        "GCTGGGTGTCCGAGTC-1_contig_3": "GAGTTCTAGAGCTTCCAGGTGTGCTGTCAGGCTGCTAGTGTATGCTCTCTCTAGTTTCTTTTTGGAGGCACTCAGAGCTATGAGTTTTCCTCTTAGGACTGCCTTCATTGTGTCCCATAAGTTTGGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTG",
        "CGGCTAGCACGACTCG-1_contig_2": "GGGGAAATACATCAGACCAGCATGGGCATCAAGATGGAGACACATTCTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGAAGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGGGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAACTACTGATTTACTGGGCATCCACCCGGCACACTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATTAGCAATGTGCAGTCTGAAGACTTGGCAGATTATTTCTGTCAGCAATATAGCAGCTATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TAAACCGTCATGCATG-1_contig_1": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "ACCAGTATCAATACCG-1_contig_2": "GGGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGAGACAGACTCTTCTCCACAGTGTCCCCTTCAT",
        "CCTACCATCTTTAGTC-1_contig_3": "GGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGAC",
        "GGCTCGATCCCATTAT-1_contig_4": "GCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "AGCCTAACACTGTCGG-1_contig_1": "GGGCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CTACGTCGTTACAGAA-1_contig_6": "GGGAGTTTCAATCCTGTTCTTAGCCTGTGCAGAGAGATGGGTGTGGGGTAGCATTCAGACCTTTGTGACAATTGGGTTGCAACTCCTAAGAGGAAAGATCCCAAAGCTAAGCTCCTGGAAGCTATGGACCGTCACAAAAAAGAAGGCAGCCACTTTCATATTTCCCATGCATGAGTTGTGTCAGCCGGTAAAGGAGAGTACAGTGGAAGAAGGCCTTGAGACCTGAGTCAGCAGTGGATATCAGCAAAACAGAACAACCAACAACCCATGGAAGAGTGGGAAGAAGTCGAAATACCAGACTGTGGAGGGGGTCTCTCTGATGCACAGGTAAGGTGATGGCCTGAAGAATGTCATCCAGACAGGGTCCAAGGATGCCACAGGCTGGAACATGGGCAAGTGGATCTGAACACACACAACAAAAACAACAACTACTGCTACCACCACCACCAGAGTCAGCCTCAAGGAGATGATGGGATATATCAGGATACCTAAGATAAGATGAACTTGGTCTGTCTGGAAGCTGGCAGGACCAAATTCGCTGAGTCATCAGAGAAGAGGTGGCCAGAGGAGCAAGATCTCTGCAGCAGAAATCTGCAGGACTAACAAAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCC",
        "CCATTCGCATCAGTAC-1_contig_3": "GGGCGTGAAAGACTCGCCAGAGCTAGACCTCCTTGGTGTGTGTTAACTCAGGAAGAAAAAGACGACATAAAACAACAGCCAGGACTCAGCATGGACATGAGGACCCCTGCTCAGTTTCTTGGAATCTTGTTGCTCTGGTTTCCAGGTATCAAATGTGACATCAAGATGACCCAGTCTCCATCTTCCATGTATGCATCTCTAGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAATAGCTATTTAAGCTGGTTCCAGCAGAAACCAGGGAAATCTCCTAAGACCCTGATCTATCGTGCAAACAGATTGGTAGATGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTCACCATCAGCAGCCTGGAGTATGAAGATATGGGAATTTATTATTGTCTACAGTATGATGAGTTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CGACTTCGTTTGACTG-1_contig_5": "GGGGATCCACTTACTGTTTTCTCTTTCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GTCATTTGTTAAGACA-1_contig_8": "GAATCCTGTTCTTAGCCTGTGCAGAGAGATGGGTGTGGGGTAGCATTCAGACCTTTGTGACAATTGGGTTGCAACTCCTAAGAGGAAAGATCCCAAAGCTAAGCTCCTGGAAGCTATGGACCGTCACAAAAAAGAAGGCAGTCACTTTCATATTTCCCATGCATGAGTTGTGTCAGCCGGTAAAGGAGAGTACAGTGGAAGAAGGCCTTGAGACCTGAGTCAGCAGTGGATATCAGCAAAACAGAACAACCAACAACCCATGGAAGAGTGGGAAGAAGTCGAAATACCAGACTGTGGAGGGGGTCTCTCTGATGCACAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCC",
        "CGTAGGCTCGTCCAGG-1_contig_4": "ATTTGCTTCCTTTAGTTCTAGAGCTTCCAGGTGTGCTGTCAGGCTGCTAGTGTATGCTCTCTCTAGTTTCTTTTTGGAGGCACTCAGAGCTATGAGTTTTCCTCTTAGGACTGCCTTCATTGTGTCCCATAAGTTTGGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGCCCGAGTGGTGTGACAGTGGCCTG",
        "CAAGTTGGTATAGGGC-1_contig_5": "GAGTGACTGTCTGGTTTCTCTGAGGTGAGGCTGGAATATAGGTCACCTTGAAGACTTAAGAGGGGTCCAGGGGGCTTCTGCACAGGCAGGGAACAGAATGTGGAACAATGACTTGAATGGTTGATTCTTGTGTGACACCAGGAATTGGCATAATGTCTGAGTTGCCCAGGGGTGATTCTAGTCAGACTCTGGGGTTTTTGTCGGGTATAGAGGAAAAATCCACTATTGTGATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CGGACTGGTTTGACTG-1_contig_5": "AGGCAGAGAACTTTAGCCCTGTCTTCTTTTTTAGTGTTCAGCACTGACAATATGACATTGAACATGCTGTTGGGGCTGAAGTGGGTTTTCTTTGTTGTTTTTTATCAAGGTGTGCATTGTGAGGTGCAGCTTGTTGAGTCTGGTGGAGGATTGGTGCAGCCTAAAGGGTCATTGAAACTCTCATGTGCAGCCTCTGGATTCACCTTCAATACCTACGCCATGAACTGGGTCCGCCAGGCTCCAGGAAAGGGTTTGGAATGGGTTGCTCGCATAAGAAGTAAAAGTAATAATTATGCAACATATTATGCCGATTCAGTGAAAGACAGGTTCACCATCTCCAGAGATGATTCACAAAGCATGCTCTATCTGCAAATGAACAACTTGAAAACTGAGGACACAGCCATGTATTACTGTGTGAGACATAACTGGGGCTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGT",
        "CAGCAGCTCCTATTCA-1_contig_2": "TGGGGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTCCTCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GCTTGAAGTAACGACG-1_contig_3": "GATGTACTTATGAGAAGAGAAGTAATTAGCCAGGGAACAAAATTCAAATACACAATGGATTTTCTGGTGCAGATTTTCAGCTTCTTGCTAATCAGTGCCTCAGTTGCAATGTCCAGAGGAGAAAATGTGCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAAAAGGTCACCATGACCTGCAGGGCCAGCTCAAGTGTAAGTTCCAGTTACTTGCACTGGTACCAGCAGAAGTCAGGTGCCTCCCCCAAACTCTGGATTTATAGCACATCCAACTTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGTGTGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTACAGTGGTTACCCATACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TACGGGCTCGTCGTTC-1_contig_3": "GAAAGCAAAGTACTTATGAGAATAGCAGTAATTATCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGGGGACAAATTGTTCTCACCCAGTCTCCAGCAATCCTGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACAGGTACCAGCAGAAGCCAGGATCCTCACCCAAACCCTGGATTTATGGCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGATCTGGGACCTCTTATTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAATATCATAGTTACCCACTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "TATGCCCAGTCAAGGC-1_contig_1": "GAGCTGCCAGGAGCCTAATAAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACAGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATATCCTGCAGAGCCAGTGAAAGTGTTGATAGTTATGGCAATAGTTTTATGCACTGGTACCAGCAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCGTGCATCCAACCTAGAATCTGGGATCCCTGCCAGGTTCAGTGGCAGTGGGTCTAGGACAGACTTCACCCTCACCATTAATCCTGTGGAGGCTGATGATGTTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACC",
        "CTGATAGCATGGATGG-1_contig_1": "GGGGAACAGCATCAGACAGGCTGGGCAGCAAGATGGAATCACAGACCCAGGTCCTCATGTTTCTTCTGCTCTGGGTATCTGGTGCCTGTGCAGACATTGTGATGACACAGTCTCCATCCTCCCTGGCTATGTCAGTAGGACAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTAAATAGTAGCAATCAAAAGAACTATTTGGCCTGGTACCAGCAGAAACCAGGACAGTCTCCTAAACTTCTGGTATACTTTGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCATAGGCAGTGGATCTGGGACAGATTTCACTCTTACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGATTACTTCTGTCAGCAACATTATAGCACTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CTGAAACTCCTCATTA-1_contig_3": "GAGACATCCTCAGCTCCCAGACTTCATGGCCAGAGATTTATAGGGATCCTGGCCAGCATTGCCGCTAGGTCCCTCTCTTCTATGCTTTCTTTGTCCCTCACTGGCCTCCATCTGAGATAATCCTGGAGCCCTAGCCAAGGATCATTTATTGTCAGGGGTCTAATCATTGTTGTCACAATGTGCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGAAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TCAGCTCGTCGCGAAA-1_contig_6": "GGGACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CCTAGCTCACGAAATA-1_contig_4": "GGACATCTCTCTCATTAGAGGTTGATCTTTGAGGAAAACAGGGTGTTGCCTAAAGGATGAAAGTGTTGAGTCTGTTGTACCTGTTGACAGCCATTCCTGGTATCCTGTCTGATGTACAGCTTCAGGAGTCAGGACCTGGCCTCGTGAAACCTTCTCAGTCTCTGTCTCTCACCTGCTCTGTCACTGGCTACTCCATCACCAGTGGTTATTACTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATAAGCTACGACGGTAGCAATAACTACAACCCATCTCTCAAAAATCGAATCTCCATCACTCGTGACACATCTAAGAACCAGTTTTTCCTGAAGTTGAATTCTGTGACTACTGAGGACACAGCTACATATTACTGTGCAAGACGGGGCCACTATAGGTACGACGGGGCTCCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TAAACCGAGTTCGATC-1_contig_4": "GATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCA",
        "TCTGAGAGTCACTGGC-1_contig_6": "GGGGGAAGCTAGAGTAAGTGTGGGTGTGGGGACCAGGCTGGACAGCTCTCGGGGGAGCTAGGAGTTGGATGCACAGAGGGAAGGAAGAAGGAGGCACATCAGGGTCTGAGCAGTGGAAGCCAGCAGCTGTGGTTCGAAGAAGGATGACCCTGAAGCTGAGAGGAGACCTGGGTTGGAGAAAAGACAGCCTGACAGCCAGTTTTCTCATGAATGGCAGCATATCCTAAACTCAAGGGCTCATGGCTTCTAGAAAGAAATGGGGAACTCATTGGAATCTGGAGACACTGGCACCTGTGACATAGGAGAAGGCAGGGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCC",
        "TCGGTAACAATACGCT-1_contig_8": "GGACAGTAGTCTCTACAGTCACAGGAGTACACAGGGCATTGCCATGGGTTGGAGCTGTATCATCTTCTTTCTGGTAGCAACAGCTACAGGTGTGCACTCCCAGGTCCAGCTGCAGCAGTCTGGGCCTGAGGTGGTGAGGCCTGGGGTCTCAGTGAAGATTTCCTGCAAGGGTTCCGGCTACACATTCACTGATTATGCTATGCACTGGGTGAAGCAGAGTCATGCAAAGAGTCTAGAGTGGATTGGAGTTATTAGTACTTACAATGGTAATACAAACTACAACCAGAAGTTTAAGGGCAAGGCCACAATGACTGTAGACAAATCCTCCAGCACAGCCTATATGGAACTTGCCAGATTGACATCTGAGGATTCTGCCATCTATTACTGTGCAAGCTACTATGGTAACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "AAGTCTGGTATAGGGC-1_contig_4": "GGCACATACCAGCAAGCGAGTGACCAGTTAGTCTTAAGGCACCACTTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCAAATTACTACGGTAGTAGCTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "CACAGGCGTTGGTTTG-1_contig_3": "TCCTCTTTTAACTTATTGAGTTCAACCTTTTAATTTTAGCTTGAGTAGTTCTAGTTTCCCCAAACTTAAGTTTATCGACTTCTAAAATGTATTTAGAATTCATTTTCAAAATTAGGTTATGTAAGAAATTGAAGGACTTTAGTGTCTTTAATTTCTAATATATTTAGAAAACTTCTTAAAATTACTCTATTATTCTTCCCTCTGATTATTGGTCTCCATTCAATTCTTTTCCAATACCCGAAGCATTTACAGTGACTTTGTTCATGATCTTTTTTAGTTGTTTGTTTTGCCTTACTATTAAGACTTTGACATTCTGGTCAAAACGGCTTCACAAATCTTTTTCAAGACCACTTTCTGAGTATTCATTTTAGGAGAAATACTTTTTTTTTAAATGAATGCAATTATCTAGACTTATTTCGGTTGAACATGCTGGTTGGTGGTTGAGAGGACACTCAGTCAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CAGCGACAGTGTTGAA-1_contig_9": "GACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAG",
        "GACCAATCATCGTCGG-1_contig_5": "GAGCATATGATCAGTGTCCTCTCCAAAGTCCTTGAACATAGACTCTAACCATGGAATGGACCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGAGCTGAGCTGATGAAGCCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTACTGGCTACACATTCAGTAGCTACTGGATAGAGTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGAGATTTTACCTGGAAGTGGTAGTACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACATTCACTGCAGATACATCCTCCAACACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCCGTCTATTACTGTGCAAGAGGGGGCTACGGCTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GTACGTACAAGCGATG-1_contig_3": "GAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGAGACAGACTCTTCTCCACAGTGTCCCCTTCATGTG",
        "AGAGTGGTCTCCAACC-1_contig_5": "GAACCTATGATCAGTGTCCTCTCCACAGTTCCTGAAGACACTGACTCTAACCATGGGATGGAGCTGGATCTTTCTTTTCCTCCTGTCAGGAACTGCAGGTGTCCATTGCCAGATCCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGCTACACCTTCACTGACTACTATATAAACTGGGTGAAGCAGAAGCCTGGACAGGGACTTGAGTGGATTGGATGGATTTATCCTGGAAGCGGTAATACTAAGTACAATGAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACACATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACACTGCTGTCTATTTCTGTGCAAGAAGGGACTACGGTAGTAGCTACAGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGTCCATAGGTTCCTA-1_contig_1": "GGAGCTGCCAGGAGCCTAAGAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGTTATGGGTACTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCAAAAGTGTCAGTACATCTGGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCACAGTAGGGAGCTTCCTCCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "ATTCTACAGTGCGATG-1_contig_4": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGT",
        "CCTCTGAGTTGTGGAG-1_contig_2": "GAGTCAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GATCGATGTCCTAGCG-1_contig_1": "AGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TACGGATAGACTACAA-1_contig_5": "ACAAGCCTTTCTTATATGGGGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGGGCTGCTTCTCTGCCTGGTGACGTTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACTGTCTCAGGGTTCTCATTAACCGACTATGGTGTAAGCTGGATTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGGTGGTGGAAGCACATACTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCGGGTGGGGTAACTTCTCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TACGGATAGACTACAA-1_contig_4": "GGGGGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGG",
        "TTCTACACATGCCCGA-1_contig_2": "TGCCCTACGTAGCGGATTTTCTTATATGGGCATGCCCGATACAGCGTTATTTCTTATATGGGGGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCCTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTAC",
        "CAGATCAGTCTGGTCG-1_contig_1": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTC",
        "TCAATCTAGCACAGGT-1_contig_7": "TCAATCTAGCACAGGTCCTTATATGGGGACATACCAGCAAGGGAGTGACCAGTTAGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGTTATACCTTCACAGACTATTCAATGCACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACTGAGACTGGTGAGCCAACATATGCAGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCTAGAAGGGATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AAACGGGAGTTACCCA-1_contig_1": "TGGGGGAAATACATCAGACCAGCATGGGCATCAAGATGGAGACACATTCTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGAAGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGGGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAACTACTGATTTACTGGGCATCCACCCGGCACACTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATTAGCAATGTGCAGTCTGAAGACTTGGCAGATTATTTCTGTCAGCAATATAGCAGCTATCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCA",
        "ACAGCCGCACCGCTAG-1_contig_1": "GGAAGAAGATCTTTCTCAGGGATGTGTCATGGTCCACACAAACTCAGGGAAAGTTTGAAGATGGTATCCACACCTCAGTTCCTTGTATTTTTGCTTTTCTGGATTCCAGCCTCCAGAGGTGACATCTTGCTGACTCAGTCTCCAGCCATCCTGTCTGTGAGTCCAGGAGAAAGAGTCAGTTTCTCCTGCAGGGCCAGTCAGAGCATTGGCACAAGCATACACTGGTATCAGCAAAGAACAAATGGTTCTCCAAGGCTTCTCATAAAGTATGCTTCTGAGTCTATCTCTGGGATCCCTTCCAGGTTTAGTGGCAGTGGATCAGGGACAGATTTTACTCTTAGCATCAACAGTGTGGAGTCTGAAGATATTGCAGATTATTACTGTCAACAAAGTAATAGCTGGCCATACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACA",
        "GCATACAAGACCTTTG-1_contig_2": "TGGGGAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATACTGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTATAAGTTACATGCACTGGTACCAGCAGAAGCCAGGCACCTCCCCCAAAAGATGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTATTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCATCAGCGGAGTAGTTACCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CGTGTCTAGCTGATAA-1_contig_2": "GACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGCATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATAACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTTCCAGCAGAAGCCAGGCACTTCTCCCAAACTCTGGATTTATAGCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGATCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAAAGGAGTAGTTACCCACCCCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CGGACGTGTCATTAGC-1_contig_3": "TGGGGAAAGTACTTATGAGAATAGTAGTAATTAGCTAGGGACCAAAGTTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATACTGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAGACTCCTGATTTATGACACATCCAACCTGGCTTCTGGAGTCCCTGTTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTTACCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCAGA",
        "GAACCTAGTGGTCTCG-1_contig_3": "GGGGACAGACGCACAACCCTGGACTCACAAGTCTTTCTCTTCAGTGACAAACACAGAAATAGAACATTCACCATGTACTTGGGACTGAACTGTGTATTCATAGTTTTTCTCTTAAAAGGTGTCCAGAGTGAAGTGAAGCTTGAGGAGTCTGGAGGAGGCTTGGTGCAACCTGGAGGATCCATGAAACTCTCCTGTGTTGCCTCTGGATTCACTTTCAGTAACTACTGGATGAACTGGGTCCGCCAGTCTCCAGAGAAGGGGCTTGAGTGGGTTGCTGAAATTAGATTGAAATCTAATAATTATGCAACACATTATGCGGAGTCTGTGAAAGGGAGGTTCACCATCTCAAGAGATGATTCCAAAAGTAGTGTCTACCTGCAAATGAACAACTTAAGAGCTGAAGACACTGGCATTTATTACTGTACCCTTTATTACTACGGTAGTAGCTACGTTCTTTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CTTACCGGTGGCCCTA-1_contig_2": "GGAAAACAACCTATGATCAGTGTCCTCTCCACAGTCCCTGAAGACACTGACTCTAACCATGGGATTCAGCAGGATCTTTCTCTTCCTCCTGTCAATAACTACAGGTGTCCACTCCCAGGCTTATCTACAGCAGTCTGGGGCTGAGCTGGTGAGGTCTGGGGCCTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACATTTACCAGTTACAATATGCACTGGGTAAAGCAGACACCTGGACAGGGCCTGGAATGGATTGGATATATTTATCCTGGAAATGGTGGTACTAACTACAATCAGAAGTTCAAGGGCAAGGCCACATTGACTGCAGACACATCCTCCAGCACAGCCTACATGCAGATCAGCAGCCTGACATCTGAAGACTCTGCGGTCTATTTCTGTGCAAGAGGGGATACTACGGTAGTAGCTGGGGACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TACTCATAGTGTGGCA-1_contig_6": "GCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "CAGCATATCCCTAACC-1_contig_2": "ATGGGGAAGCTGGCTATCACTCAAGGCCATGACCATGCTCTCACTAGCTCCTCTCCTCAGCCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCGTGGCTACAGGAGAAAAAGTCACTATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGCACAGATTTTGTTTTTACAATTGAAAACACGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACATGCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "TACGGATAGACTACAA-1_contig_3": "GAGGTCACAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTG",
        "CTGATCCAGTAGCGGT-1_contig_2": "GGGGAACAGCATCAGACAGGCTGGGCAGCAAGATGGAATCACAGACCCAGGTCCTCATGTTTCTTCTGCTCTGGGTATCTGGTGCCTGTGCAGACATTGTGATGACACAGTCTCCATCCTCCCTGGCTATGTCAGTAGGACAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTAAATAGTAGCAATCAAAAGAACTATTTGGCCTGGTACCAGCAGAAACCAGGACAGTCTCCTAAACTTCTGGTATACTTTGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCATAGGCAGTGGATCTGGGACAGATTTCACTCTTACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGATTACTTCTGTCAGCAACATTATAGCACTCCTCCCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTG",
        "GGAAAGCGTCCTGCTT-1_contig_4": "GACAGACAGACGCACAACCCTGGACTCACAAGTCTTTCTCTTCAGTGACAAACACAGAAATAGAACATTCACCATGTACTTGGGACTGAACTGTGTATTCATAGTTTTTCTCTTAAAAGGTGTCCAGAGTGAAGTGAAGCTTGAGGAGTCTGGAGGAGGCTTGGTGCAACCTGGAGGATCCATGAAACTCTCCTGTGTTGCCTCTGGATTCACTTTCAGTAACTACTGGATGAACTGGGTCCGCCAGTCTCCAGAGAAGGGGCTTGAGTGGGTTGCTGAAATTAGATTGAAATCTAATAATTATGCAACACATTATGCGGAGTCTGTGAAAGGGAGGTTCACCATCTCAAGAGATGATTCCAAAAGTAGTGTCTACCTGCAAATGAACAACTTAAGAGCTGAAGACACTGGCATTTATTACTGTACCAGGGCTATGGGGGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TACAGTGGTAAACACA-1_contig_2": "CATCTATTCAGGTAATCAGCAAGGGGATAAATATTCTAAGAAGGCACTAGGAGTCTGCACTTCTGGCCAGATCTTATTACAGATCTTATAAAGTCAGTATGATTTTGCATAAATACATCTTCTTTTAGTGCTCCTGCTCCTTAGAGTATTATAAGAAGATCTCTATCAGGACTGTGTCATGATCCACACAAACTCAGGGAAAGCTCGAAGATGGTTTTCACACCTCAGATTCTTGGACTTATGCTTTTCTGGATTTCAGCCTCCAGAGGTGATATTGTGCTAACTCAGTCTCCAGCCACCCTGTCTGTGACTCCAGGAGATAGAGTCAGTCTTTCCTGCAGGGCCAGTCAAAGTATTAGCAACTACCTACACTGGTATCAACAAAAATCACATGAGTCTCCAAGGCTTCTCATCAAGTATGCTTCCCAGTCCATCTCTGGGATCCCCTCCAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACTCTCAGTATCAACAGTGTGGAGACTGAAGATTTTGGAATGTATTTCTGTCAACAGAGTAACAGCTGGCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCG",
        "CGAGCCACAGCAGTTT-1_contig_4": "GACATATGACCAATATCCTCTCCACATTCCCTTAAGACACTGACTCTAAACATGGGATGGATCTGGATCTTTCTCTTCCTCCTGTCAGGAACTGCAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTAGTGAAGACTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGTTACTACATGCACTGGGTCAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGATATATTAGTTGTTACAATGGTGCTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTTACTGTAGACACATCCTCCAGCACAGCCTACATGCAGTTCAACAGCCTGACATCTGAAGACTCTGCGGTCTATTACTGTGCAAGAGAGGGGAACGGTAGTAGCTCTTTCTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TTCCCAGAGACTACAA-1_contig_5": "GGGAAAAACATGAGATCACTGTTCTCTCTACAGTTACTGAGCACACAGGACCTCACCATGGGATGGAGTTGTATCATCCTCTTCTTGGTAGCAACAGCTACAGGTGTCCACTCCCAGGTGCAACTGCAGCAGCCTGGGGCTGAGCTGGTGAAGCCTGGGGCCTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACATTTACCAGTTACAATATGCACTGGGTAAAGCAGACACCTGGACAGGGCCTGGAATGGATTGGAGCTATTTATCCAGGAAATGGTGATACTTCCTACAATCAGAAGTTCAAAGGCAAGGCCACATTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCTACGGCGGTAGTAGCCCCCCGTGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CTGTGCTAGCCCAGCT-1_contig_4": "GCTCTGATGTGTAGCACCCCAGTGTGCATTTTAGAATGATGCCAAACAGTCTTTGAAATTGACATTCAATTTTCTGTGGTCCTGCCTCTATCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGG",
        "GTACGTACAAGCGATG-1_contig_1": "GGGGAGTCAGATTGTGAGCCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCTG",
        "GAACCTAAGATCGGGT-1_contig_4": "GGGGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GAACATCAGACCTAGG-1_contig_8": "CTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TAGGCATGTTACGGAG-1_contig_12": "GAAACAACCTATGATCAGTGTCCTCTCAACAGTCCCTGAACACACTGACTCTCACCATGGAATGGAGCGGAGTCTTTATCTTTCTCCTGTCAGTAACTGCAGGTGTTCACTCCCAGGTCCAGCTGCAGCAGTCTGGAGCTGAGCTGGTAAGGCCTGGGACCTCAGTGAAGGTGTCCTGCAAGGCTTCTGGATACGCCTTCAAAAATTACTTGATAGAGTGGGTAAAGCAGAGGCCTGGACAGGGCCTTGAGTGGATTGGAGTGATTAATCTTGGAAATGGTGGTACTGACTACAATGAGAAGTTCAAGGGCAAGGCAACACTGACTGCAGACAAATCCTCCACCACTGCCTACATGGTGTTAAGTAGCCTGACATCTGATGACTCTGCGGTCTATTTCTGTGCCTGTATGATTTCGACGGCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTACAGCCAAAACAACACCCCCATCAGTCTATCCACTGGCCCCTGGGTGTGGAGATACAACTGGTTCCTCCGTGACTCTGGGATGCCTGGTCAAGGGCTACTTCCCTGAGTCAGTGACTGTGACTTGGAACTCTGGATCCCTGTCC",
        "GCATGATAGACAGGCT-1_contig_5": "AGTCAGCCTTGTGACAGCCCCCGGTACATTTACTTATATAGAACCTCCCTGGGGAGATAGAATCCCAGGAGAAAAAATGCTGGATGTACACTAAGGATGGCCCCTGACACTCTGCACTGCTACCTCTGGCCCCACCAGACAATGTTCCTGCAGAACCTGTTACCTTACTTGGCAGGGATTTTTGTCAAGGTATCTACTACTGTGGGCTTATGGTAACTCATATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ACTGAACGTACCGGCT-1_contig_6": "GGGCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCAC",
        "GTGGGTCTCGGCCGAT-1_contig_3": "GGGCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GCAGTTAAGTCATGCT-1_contig_3": "GAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATC",
        "GTAGTCATCTCCTATA-1_contig_4": "GGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGAAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGCGTGGTGACAGTGGCCTG",
        "TGACGGCTCCTTGGTC-1_contig_2": "GGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAAC",
        "GCACATAGTAGAGTGC-1_contig_5": "TGGGGTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "CGACTTCGTCTTTCAT-1_contig_1": "GGGTTTAAAGCAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATACTGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTATAAGTTACATGCACTGGTACCAGCAGAAGCCAGGCACCTCCCCCAAAAGATGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTATTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCATCAGCGGAGTAGTTACCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "ATCTGCCAGGAATCGC-1_contig_5": "GAGAAAAACATGAGATCACAGTTCTCTCTACAGTTACTGAGCACACAGGACCTCACCATGGGATGGAGCTATATCATCCTCTTTTTGGTAGCAACAGCTACAGATGTCCACTCCCAGGTCCAACTGCAGCAGCCTGGGGCTGAACTGGTGAAGCCTGGGGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTGGATGCACTGGGTGAAGCAGAGGCCTGGACAAGGCCTTGAGTGGATTGGAGAGATTAATCCTAGCAACGGTCGTACTAACTACAATGAGAAGTTCAAGAGCAAGGCCACACTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGACGGCCTATTTTTACTACGGTAGTAGCTACCCCGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CAGCTAAAGTCCAGGA-1_contig_2": "GGGGGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCAGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGCACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCTCAGAGGAAGGTGGAAACACGGTGAGAGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACCGCCGAAAACATAATGGGTGCTGTACC",
        "GGGCACTAGGAGTAGA-1_contig_1": "TGGGGAAGACGATCTCTATCAGGACAGTGTCATGAGCCACACAAACTCAGGGAAAGCTCGAAGATGGTTTTCACACCTCAGATACTTGGACTTATGCTTTTTTGGATTTCAGCCTCCAGAGGTGATATTGTGCTAACTCAGTCTCCAGCCACCCTGTCTGTGACTCCAGGAGATAGCGTCAGTCTTTCCTGCAGGGCCAGCCAAAGTATTAGCAACAACCTACACTGGTATCAACAAAAATCACATGAGTCTCCAAGGCTTCTCATCAAGTATGCTTCCCAGTCCATCTCTGGGATCCCCTCCAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACTCTCAGTATCAACAGTGTGGAGACTGAAGATTTTGGAATGTATTTCTGTCAACAGAGTAACAGCTGGCCTCAGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "TTCGAAGCATCCTTGC-1_contig_5": "GGTGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TCGGTAACAATACGCT-1_contig_6": "AAACAGAAAAACATGAGATCACAGTTCTCTCTACAGTTACTGAGCACACAGGACCTCACCATGGGATGGAGCTGTATCATCCTCTTCTTGGTAGCAACAGCTACAGGTGTCCACTCCCAGGTCCAACTGCAGCAGCCTGGGGCTGAGCTGGTGAGGCCTGGGGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACCAGCTACTGGATAAACTGGGTGAAGCAGAGGCCTGGACAAGGCCTTGAGTGGATCGGAAATATTTATCCTTCTGATAGTTATACTAACTACAATCAAAAGTTCAAGGACAAGGCCACATTGACTGTAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCCGACATCTGAGGACTCTGCGGTCTATTACTGTACAAGAGAACGGGCTGGGACTGGCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "GCCAAATGTGACCAAG-1_contig_6": "ATATCTGTCTTACTGGAGGTTGATTTTTTTTTTTTTTTTTTTTTTATGAAAAGGGGGTATAGCCTAAAGGATGAAAATGTTCACTCTTCTGTACCTGTTGACAGTCGTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCAGGACCTGGTCTGGTGAAACCTTCTCAGACAGTGTCCCTCACCTGCACTGTCACTGGCATCTCCATCACCACTGGAAATTACAGATGGAGCTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATAGGGTACATATACTACAGTGGTACCATTACCTACAATCCATCTCTCACAAGTCGAACCACCATCACTAGAGACACTTCCAAGAACCAATTCTTCCTGGAAATGAACTCTTTGACTGCTGAAGACACAGCCACATACTACTGTGCACGAGATGGGGATTACGACGGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCC",
        "AGGGAGTCAAGTAATG-1_contig_1": "TGGGGGAAATACATCAGATCAGCATGGGCATCAAGATGGAGTCACAGACTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGATGGAGACATTGTGATGACCCAGTCTCAAAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCGTCACCTGCAAGGCCAGTCAGAATGTGGGTACTAATGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAAGCACTGATTTACTCGGCATCCTACCGGTACAGTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAATGTGCAGTCTGAAGACTTGGCAGAGTATTTCTGTCAGCAATATAACAGCTATCCTCTGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCAGATCGGAAG",
        "GCACTCTCAGGGATTG-1_contig_4": "GACTAGTGTGCAGCCATGGGCAGGCTTACTTCTTCGTTCCTGCTGCTGATTGTCCCTACATATGTCCTGTCCCAGGTTATTCTGAAAGAGTCTGGCCCTGGAATATTGCAGCCCTCTCAGACCCTCAGTCTGACTTGTTCTTTCTCTGGGTTTTCACTTAGCACTTATGGTACAGCTGTGAACTGGATTCGTCAGCCTTCAGGAAAGGGTCTGGAGTGGTTGGCACAAATTGGGTCAGATGATAGCAAGCTCTATAACCCATTTCTGAAAAGCCGAATCACAATCTCCAAGGATACCTCCAACAGCCAGGTATTCCTCAAGATCACTAGTGTGGACACTGAAGATTCTGCCACATACTACTGTGCTAACAGAGTAGGGGGTAACCCGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGCCAAAACAACACCCCCATCAGTCTATCCACTGGCCCCTGGGTGTGGAGATACAACTGGTTCCTCCGTGACTCTGGGATGCCTGGTCAAGGG",
        "AGCTCCTTCTGGCGTG-1_contig_4": "GGGGCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACA",
        "AGCCTAACACTGTCGG-1_contig_8": "GTATGTGTTCTCCAACTGCTCTCTCCTGAAGTGCCTCAGATGTTTCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTG",
        "AACGTTGGTTATGCGT-1_contig_2": "GGGGACTTGTGACGTTTTGGTTCTGTTTGGGTAACTTGTGTGAATTTGTGACATTTTGGCTAAATGAGCCATTCCTGGCAACCTGTGCATCAATAGAAGATCCCCCAGAAAAGAGTCAGTGTGAAAGCTGAGCGAAAAACTCGTCTTAGGCTTCTGAGACCAGTTTTGTAAGGGGAATGTAGAAGAAAGAGCTGGGCTTTTCCTCTGAATTTGGCCCATCTAGTTGGACTGGCTTCACAGGCAGGTTTTTGTAGAGAGGGGCATGTCATAGTCCTCACTGTGGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CTGTGCTAGTACATGA-1_contig_1": "GGGGGAATGTATAAACCAGGCCTTTGGATTGTGACTGAACAGCATCAGACAGGCAGGGGAGCAATATGGTATCAGAGACCCATGTCCTCATATTTTTGCTGCTATGGGTGTCTGGTGCTTGTGGGGAAATTGTGTTGACCCAGTCTATACCATCCCTGACTGTGTCAGCAGGAGAGAGGGTCACTATCAGCTGCAAATCCAATCAGAATCTTTTATGGAGTGGAAACCAAAGGTACTGTTTGGTCTGGCACCAGTGGAAACCAGGGCAAACTCCTACACCGTTGATCACCTGGACATCTGATAGGTACTCTGGAGTCCCTGATCGTTTCATAGGCAGTGGATCTGTGACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGATGTGGCAGTTTATTTCTGTCAGCAGCATTTACACATTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "AGGCCACCATAGACTC-1_contig_3": "GGGATGAGCACTGTTTTCTCTACAGTCACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTCATCTTCTTCCTGATGGCAGTGGTTACAGGGGTCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCTGAGCTTGTGAGGCCAGGGGCCTTAGTCAAGTTGTCCTGCAAAGCTTCTGGCTTCAACATTAAAGACTACTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGATGGATTGATCCTGAGAATGGTAATACTATATATGACCCGAAGTTCCAGGGCAAGGCCAGTATAACAGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTGCTAGATCCCCTACGGTAGTAGCTCGGTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GACTAACGTGTGGTTT-1_contig_1": "TGGGAGAAATACATCAGGCAGGCAAGGGCATCAAGATGAAGTCACAGACCCAGGTCTTCGTATTTCTACTGCTCTGTGTGTCTGGTGCTCATGGGAGTATTGTGATGACCCAGACTCCCAAATTCCTGCTTGTATCAGCAGGAGACAGGGTTACCATAACCTGCAAGGCCAGTCAGAGTGTGAGTAATGATGTAGCTTGGTACCAACAGAAGCCAGGGCAGTCTCCTAAACTGCTGATATACTATGCATCCAATCGCTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATATGGGACGGATTTCACTTTCACCATCAGCACTGTGCAGGCTGAAGACCTGGCAGTTTATTTCTGTCAGCAGGATTATAGCTCTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAAC",
        "AATCGGTCATACAGCT-1_contig_4": "GGGGCACTGCTACCTCTGGCCCCACCAGACAATGTTCCTGCAGAACCTGTTACCTTACTTGGCAGGGATTTTTGTCAAGGTATCTACTACTGTGTCTACTATGGTAACTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "TGACAACGTCCGCTGA-1_contig_2": "GACCGCATATGATCAGTAACCTCTTCACAGTCACTGAAAACACTGACTCTAATCATGGAATGTAACTGGATACTTCCTTTTATTCTGTCAGTAACTTCAGGTGTCTACTCACAGGTTCAGCTCCAGCAGTCTGGGGCTGAGCTGGCAAGACCTGGGGCTTCAGTGAAGTTGTCCTGCAAGGCTTCTGGCTACACCTTTACTAGCTACTGGATGCAGTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATTGGGGCTATTTATCCTGGAGATGGTGATACTAGGTACACTCAGAAGTTCAAGGGCAAGGCCACATTGACTGCAGATAAATCCTCCAGCACAGCCTACATGCAACTCAGCAGCTTGGCATCTGAGGACTCTGCGGTCTATTACTGTGCAAGAGGGAAAGATGGTTACTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GCGGGTTTCGCACTCT-1_contig_4": "TGGGAGATGTTTCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGAC",
        "ACACCCTTCAGCTCGG-1_contig_4": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCAC",
        "TTGCCGTTCATCGATG-1_contig_3": "GGCAGAAAACAGCATATGATCAGTGTCCTCTCCAAAGTCCTTGAACATAGACTCTAACCATGGAATGGACCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGAGCTGAGCTGATGAAGCCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTACTGGCTACACATTCAGTAGCTACTGGATAGAGTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGAGATTTTACCTGGAAGTGGTAGTACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACATTCACTGCAGATACATCCTCCAACACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCCGTCTATTACTGTGCAAGAAGGGGCTATGATGGTTACTTGGAGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "ACTGTCCTCGGAAATA-1_contig_5": "GAATCCTGTTCTTAGCCTGTGCAGAGAGATGGGTGTGGGGTAGCATTCAGACCTTTGTGACAATTGGGTTGCAACTCCTAAGAGGAAAGATCCCAAAGCTAAGCTCCTGGAAGCTATGGACCGTCACAAAAAAGAAGGCAGTCACTTTCATATTTCCCATGCATGAGTTGTGTCAGCCGGTAAAGGAGAGTACAGTGGAAGAAGGCCTTGAGACCTGAGTCAGCAGTGGATATCAGCAAAACAGAACAACCAACAACCCATGGAAGAGTGGGAAGAAGTCGAAATACCAGACTGTGGAGGGGGTCTCTCTGATGCACAGGTAAGGTGATGGCCTGAAGAATGTCATCCAGACAGGGTCCAAGGATGCCACAGGCTGGAACATGGGCAAGTGGATCTGAACACACACAACAAAAACAACAACTACTGCTACCACCACCACCAG",
        "ACTGATGCAGCAGTTT-1_contig_1": "TGGGGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCAGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAA",
        "ATCCGAAGTCGTGGCT-1_contig_1": "GGTCACACCCTGTGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCTG",
        "TGACGGCAGGACAGCT-1_contig_5": "TATTACTTTCTTATATGGGACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGGGCGGGTTACTATGGTAACCTACGCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CAAGATCCAGCGTAAG-1_contig_8": "GAGAAAACAGCATATGATCAGTGTCCTCTCCAAAGTCCTTGAACATAGACTCTAACCATGGAATGGACCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGAGCTGAGCTGATGAAGCCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTACTGGCTACACATTCAGTAGCTACTGGATAGAGTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGAGATTTTACCTGGAAGTGGTAGTACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACATTCACTGCAGATACATCCTCCAACACAGCCTACATGCAACTCAGCAGCCTGACATCTGAGGACTCTGCCGTCTATTACTGTGCAAGAGATGGTAACTACGAGGGGGTCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GTGCAGCCACAACTGT-1_contig_1": "GGGGATAGTCAGGTCACACCCTGTGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCAGGAGGAGG",
        "CACAGGCGTTGGTTTG-1_contig_4": "GGGGAGCTGCCAGGAAGCTAAGAAGCATCCTCTCTTCTAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCCAAAGTGTCAGTACATCTAGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCAAGTATGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATACTGCAACATATTACTGTCAGCACAGTTGGGAGATTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TCTGGAAGTAGCTAAA-1_contig_3": "GGTTTCTTATATGGGGGTTCTGCATCAGAAAGGCAGGGGGATCAAGATGGAATCACAGACTCAGGTCTTCCTCTCCCTGCTGCTCTGGGTATCTGGTACCTGTGGGAACATTATGATGACACAGTCGCCATCATCTCTGGCTGTGTCTGCAGGAGAAAAGGTCACTATGAGCTGTAAGTCCAGTCAAAGTGTTTTATACAGTTCAAATCAGAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATCTACTGGGCATCCACTAGGGAATCTGGTGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTTACTCTTACCATCAGCAGTGTACAAGCTGAAGACCTGGCAGTTTATTACTGTCATCAATACCTCTCCTCGCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCA",
        "TAGACCATCTGTCCGT-1_contig_4": "GCTCTGACAGAGGAGGCCAGTCCTGGATTTGAGTTCCTCACATTCAGTGATGAGCACTGAACACAGACACCTCACCATGAACTTCGGGTTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCAGTGTGAAGTGAAGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGCCATGTCTTGGGTTCGCCAGACTCCAGAGAAGAGGCTGGAGTGGGTCGCATCCATTAGTAGTGGTGGTAGCACCTACTATCCAGACAGTGTGAAGGGCCGATTCACCATCTCCAGAGATAATGCCAGGAACATCCTGTACCTGCAAATGAGCAGTCTGAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGAGGCGGCTACGGTAGTAGCTACGACGCCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CATCAGACAGGACGTA-1_contig_4": "GAGACAATCTATGATCAGTGTCCTCTCTACACAGTCCCTGACGACACTGACTCCAACCATGCGATGGAGCTGGATCTTTCTCTTCCTCCTGTCAATAACTGCAGGTGTCCATTGCCAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACCTTCACAAGCTACTATATACACTGGGTGAAGCAGAGGCCTGGACAGGGACTTGAGTGGATTGGATGGATTTATCCTGGAGATGGTAGTACTAAGTACAATGAGAAGTTCAAGGGCAAGACCACACTGACTGCAGACAAATCCTCCAGCACAGCCTACATGTTGCTCAGCAGCCTGACCTCTGAGGACTCTGCGATCTATTTCTGTGCAAGGAATTGGGACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "CTACGTCGTTCCGTCT-1_contig_3": "GATATGAGCCCTATCTTCTCTACAGACACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTTATCTTCTTCCTGATGGCAGTGGTTACAGGGGTCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCAGAGCTTGTGAAGCCAGGGGCCTCAGTCAAGTTGTCCTGCACAGCTTCTGGCTTCAACATTAAAGACACCTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGAAGGATTGATCCTGCGAATGGTAATACTAAATATGACCCGAAGTTCCAGGGCAAGGCCACTATAACAGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTGCTAGATCGGGGTACGGTAGTAGCTACGAAGTGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGC",
        "GGAATAACAAGGTGTG-1_contig_1": "TGGGGAGCTTCACACATGTTACACAGAGACATGAACATGCTCACTCAGCTCCTGGGATTACTGCTGCTCTGGTTTGCAGGTGGTAAATGTGACATTCAGATGACCCAGTCTCCTGCCTCCCAGTCTGCATCTCTGGGAGAAAGTGTCACCATCACATGCCTGGCAAGTCAGACCATTGGTACATGGTTAGCATGGTATCAGCAGAAACCAGGGAAATCTCCTCAGCTCCTGATTTATGCTGCAACCAGCTTGGCAGATGGGGTCCCATCAAGGTTCAGTGGTAGTGGATCTGGCACAAAATTTTCTTTCAAGATCAGCAGCCTACAGGCTGAAGATTTTGTAAGTTATTACTGTCAACAACTTTACAGTACTCCTCTGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCTG",
        "TCAGGATGTAGAAGGA-1_contig_3": "GGGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAACAAAGACAGCACCTACACCATGAGCAGCCAGTT",
        "ATTGGTGCAGACAAGC-1_contig_5": "GCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGG",
        "TACAGTGGTAAACACA-1_contig_4": "GGAACCGACGATCAGTGTCCTCTCCAAAGTCCCTGAACACACTGACTCTAACCATGGAATGGAGTTGGATATTTCTCTTTCTCCTGTCAGGAACTGCAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTAAAGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGATACACATTCACTAGCTATGTTATGCACTGGGTGAAGCAGAAGCCTGGGCAGGGCCTTGAGTGGATTGGATATATTAATCCTTACAATGATGGTACTAAGTACAATGAGAAGTTCAAAGGCAAGGCCACACTGACTTCAGACAAATCCTCCAGCACAGCCTACATGGAGCTCAGCAGCCTGACCTCTGAGGACTCTGCGGTCTATTACTGTGCAAGCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ACCAGTATCAATACCG-1_contig_1": "TGGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGAC",
        "CACATTTAGCTCTCGG-1_contig_6": "GACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGAGCACGTCATGCCG-1_contig_1": "GAAGATCACAGTAGTCTCTACAGTCACAGGAGTACACCGGGCATTGACATGGGTTGGAGCTGTATCATCTTCTTTCTGGTAGCAACAGCTACAGGTGTGCACTCCCAGGTCCAGCTGCAGCAGTCTGGGGCTGAGCTGGTGAGGCCTGGGGTCTCAGTGAAGATTTCCTGCAAGGGTTCTGGCTACACATTCACTGATTATGCTATGCACTGGGTGAAGCAGAGTCATGCAAAGAGTCTAGAGTGGATTGGAGTTATTAGTACTTACTATGGTGATGCTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACAATGACTGTAGACAAATCCTCCAGCACAGCCTATATGGAACTTGCCAGACTGACATCTGAGGATTCTGCCATCTATTACTGTGCAAGATCCTATGGTAACTACGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CCTTCGAAGCACCGTC-1_contig_3": "GGGGAAATGCATCAGACCAGCATGGGCATCAAGATGGAATCACAGACTCTGGTCTTCATATCCATACTGCTCTGGTTATATGGTGCTGATGGGAACATTGTAATGACCCAATCTCCCAAATCCATGTCCATGTCAGTAGGAGAGAGGGTCACCTTGAGCTGCAAGGCCAGTGAGAATGTGGGTACTTATGTATCCTGGTATCAACAGAAACCAGAGCAGTCTCCTAAACTGCTGATATACGGGGCATCCAACCGGTACCCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGCAACAGATTTCACTCTGACCATCAGCAGTCTGCAGGCTGAAGACCTTGCAGATTATCACTGTGGACAGGGTTACAGCTATTATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TGACGGCAGGACAGCT-1_contig_1": "GAGATCACACCATGTGCTGGAGTCAGCCTCACACTGATCACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGATGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGTATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTAATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATGCTGCAACAAACTTAGCAGATGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTATTCCCTCAAGATCAACAGCCTGCAGTCTGAAGATTTTGGGAGTTATTACTGTCAACATTTTTGGGGTACTCCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CTACATTGTATTCTCT-1_contig_2": "GAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "AGCCTAAAGTACCGGA-1_contig_1": "GAGCCCAGGCTTCTTTAAGGGCAGCTGCCAGGAGCCTAAGAAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACAGGTAACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATATCCTGCAGAGCCAGTGAAAGTGTTGATAGTTATGGCAATAGTTTTATGCACTGGTACCAGCAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTAGGACAGACTTCACCCTCACCATTGATCCTGTGGAGGCTGATGATGCTGCAACCTATTACTGTCAGCAAAATAATGAGGATCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TGACGGCGTGTGACCC-1_contig_2": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "GGCTCGATCCCATTAT-1_contig_1": "GGATAGCAAAGTACTTATGAGAATAGTAGTAATTAGCTAGGGACCAAAGTTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATACTGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACTGGTACCAGCAGAAGCCAGGATCCTCCCCCAGACTCCTGATTTATGACACATCCAACCTGGCTTCTGGAGTCCCTGTTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTTACCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GTTCATTGTTCACGGC-1_contig_1": "TGGGGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTCCTCAGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "CCTACCATCTTTAGTC-1_contig_1": "GGGGAGAACTACAACCTGTCTGTCTCAGCAGAGATCAGTAGTACCTGCATTATGGCCTGGACTCCTCTCTTCTTCTTCTTTGTTCTTCATTGCTCAGGTTCTTTCTCCCAACTTGTGCTCACTCAGTCATCTTCAGCCTCTTTCTCCCTGGGAGCCTCAGCAAAACTCACGTGCACCTTGAGTAGTCAGCACAGTACGTACACCATTGAATGGTATCAGCAACAGCCACTCAAGCCTCCTAAGTATGTGATGGAGCTTAAGAAAGATGGAAGCCACAGCACAGGTGATGGGATTCCTGATCGCTTCTCTGGATCCAGCTCTGGTGCTGATCGCTACCTTAGCATTTCCAACATCCAGCCTGAAGATGAAGCAATATACATCTGTGGTGTGGGTGATACAATTAAGGAACAATTTGTGTAACTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGA",
        "TCGCGTTAGGAATTAC-1_contig_1": "GATCAGACAGGCAGGGGAAGCAAGATGGATTCACAGGCCCAGGTTCTTATGTTACTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTAGCTGTGTCAGTTGGAGAGAAGGTTACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCCTTGACCAGGCATCCCAGAGTCACGGAGGAACCAGTTGTATCTCC",
        "GGAAAGCCATACCATG-1_contig_3": "TGGGGAAGCTGGCTATCACTCAAGGCCATGACCATGCTCTCACTAGCTCCTCTCCTCAGCCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCGTGGCTACAGGAGAAAAAGTCACTATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGCACAGATTTTGTTTTTACAATTGAAAACACGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACATGCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "AGCAGCCCATCGGACC-1_contig_1": "CTATCTGGGGGTTGTCATTGCAGTCAGGACTCAGCATGGACATGAGGGCCCCTGCTCAGTTTTTTGGGATCTTGTTGCTCTGGTTTCCAGGTATCAGATGTGACATCAAGATGACCCAGTCTCCATCCTCCATGTATGCATCGCTGGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAAAAGCTATTTAAGCTGGTACCAGCAGAAACCATGGAAATCTCCTAAGACCCTGATCTATTATGCAACAAGCTTGGCAGATGGGGTCCCATCAAGATTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTAACCATCAGCAGCCTGGAGTCTGACGATACAGCAACTTATTACTGTCTACAGCATGGTGAGAGCCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCT",
        "TGCCAAAGTCCAGTTA-1_contig_13": "GGGCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAAC",
        "CCACGGAGTTATCCGA-1_contig_4": "GTCCCCAGTTCCATGTCCTCAGAACTTCACTATCTGTCTTGCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACCA",
        "GAACCTAAGATCGGGT-1_contig_2": "CGCGGGTTTCTTATATGGGGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "AGCGTCGGTCTGATCA-1_contig_3": "GGGGGCAAGGACTGTGAATCCTGGAAGGATGAGGCAGAGCTCAGAGCTGGGCTGTTAGAAGCACAGTGACAAACAAAAAGAGCAGCAGTCTTCTTCACGCGGCTCCACATGTGAGTGTGGTCAGCTGAGAGACTGGAGAGAGCCACAGCCTAGGGAGAGCACTGGGCCTTTCCAGAACTAATAGATGGAAGATCACAGGAGGATGGGGAGGAGTTGGCAGATCAGCCCCCCAAAAAGGGCCTGTGTGTTGCAGAGAAAAGGTGCCTACCTGCAACCTGGTGCCAACCAGGAAGAGTCCAGAGTTCTCACACACAGAAGAATGGACCAGTAAGCTGTGCACACCTACAGACAACCAGACCAAAACAACACCCCCATCAGTCTATCCACTGGCCCCTGGGTGTGGAGATACAACTGGTTCCTCCGTGACTCTGGGATGCCTGGTCAAGGG",
        "GCATACAAGGCAGTCA-1_contig_4": "GACATCCAGTGCAGTAATCAATGTCAGGTCACAGCAGAAACATGAAGTTTCCTTCTCAACTTCTGCTCTTACTGCTGTTTGGAATCCCAGGCATGATATGTGACATCCAGATGACACAATCTTCATCCTCCTTTTCTGTATCTCTAGGAGACAGAGTCACCATTACTTGCAAGGCAAGTGAGGACATATATAATCGGTTAGCCTGGTATCAGCAGAAACCAGGAAATGCTCCTAGGCTCTTAATATCTGGTGCAACCAGTTTGGAAACTGGGGTTCCTTCAAGATTCAGTGGCAGTGGATCTGGAAAGGATTACACTCTCAGCATTACCAGTCTTCAGACTGAAGATGTTGCTACTTATTACTGTCAACAGTATTGGAGTACTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAAC",
        "AGGCCACCATAGACTC-1_contig_1": "GACACCCTGTGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTG",
        "TAGTGGTCATCACAAC-1_contig_3": "AAAACAAAATATGTCCAATGTCCTCTCCACAGACACTGAACACACTGACTCTAACCATGGAATGGAGCTGGATCTTTCTCTTTCTCCTGTCAGGAACTGCAGGTGTCCTCTCTGAGGTCCAGCTGCAACAGTTTGGAGCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGCTACACATTCACTGACTACAACATGGACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGAGATATTAATCCTAACTATGATAGTACTAGCTACAACCAGAAGTTCAAGGGAAAGGCCACATTGACTGTAGACAAGTCCTCCAGCACAGCCTACATGGAGCTCCGCAGCCTGACATCTGAGGACACTGCAGTCTATTACTGTGCAAGAGAAGGGGGGTACGACGGGAACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "TATGCCCGTGTGGCTC-1_contig_3": "GAGGACTCTGCGGTCTATTACTGTGCAAGAGAGGGGACTACGGCGGGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CAGCAGCTCCTATTCA-1_contig_4": "GGGGGACAGAGGAGGCCAGTCCTGGATTTGAGTTCCTCACATTCAGTGATGAGCACTGAACACAGACACCTCACCATGAACTTCGGGTTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCAGTGTGAAGTGAAGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGCCATGTCTTGGGTTCGCCAGACTCCAGAGAAGAGGCTGGAGTGGGTCGCATCCATTAGTAGTGGTGGTAGCACCTACTATCCAGACAGTGTGAAGGGCCGATTCACCATCTCCAGAGATAATGCCAGGAACATCCTGTACCTGCAAATGAGCAGTCTGAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGAGGCCGGTCTGTGGAGTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GATTCAGCACTCTGTC-1_contig_7": "GAAAAAAAAAATACGATCAGCGTCCTCTCCACAGACACTGAAAACTCTGACTCAACATGGAAAGGCACTGGATCTTTCTCTTCCTGTTGTCAGTAACTGCAGGTGTCCACTCCCAGGTCCAGCTGCAGCAGTCTGCAGCTGAACTGGCAAGACCTGGGGCCTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACCTTTACTAGCTACACGATGCACTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATTGGATACATTAATCCTAGCAGTGGATATACTGAGTACAATCAGAAGTTCAAGGACAAGACCACATTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAACTGAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCCCCCCTCTATGATGGTTACTCCTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTT",
        "GTCCTCAGTCATCCCT-1_contig_3": "GAGCTCTGACAGAGGAGCCAAGCCCTGGATTCCCAGGTCCTCACATTCAGTGATCAGCACTGAACACAGACCACTCACCATGGACTCCAGGCTCAATTTAGTTTTCCTTGTCCTTATTTTAAAAGGTGTCCAGTGTGATGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGCAGCCTGGAGGGTCCCGGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTTTGGAATGCACTGGGTTCGTCAGGCTCCAGAGAAGGGGCTGGAGTGGGTCGCATACATTAGTAGTGGCAGTAGTACCATCTACTATGCAGACACAGTGAAGGGCCGATTCACCATCTCCAGAGACAATCCCAAGAACACCCTGTTCCTGCAAATGACCAGTCTAAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGATCCCCTAACGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CCACGGAGTTATCCGA-1_contig_1": "AGCACTGTCCAGATGAAGCTGATTTGCATGTGCTGAGATCATATTCTACTGCCCCAGAGATTTAATAATCTGATCATACACACTCCAACAGTCATTCTTGGTCAGGAGACGTTGTAGAAATGAGACCGTCTATTCAGTTCCTGGGGCTCTTGTTGTTCTGGCTTCATGGTGCTCAGTGTGACATCCAGATGACACAGTCTCCATCCTCACTGTCTGCATCTCTGGGAGGCAAAGTCACCATCACTTGCAAGGCAAGCCAAGACATTAACAAGTATATAGCTTGGTACCAACACAAGCCTGGAAAAGGTCCTAGGCTGCTCATACATTACACATCTACATTACAGCCAGGCATCCCATCAAGGTTCAGTGGAAGTGGGTCTGGGAGAGATTATTCCTTCAGCATCAGCAACCTGGAGCCTGAAGATATTGCAACTTATTATTGTCTACAGTATGATAATCTGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TGAGCATTCGGATGGA-1_contig_5": "TGGGGCTCTCTCTTCTCATGAAAACGAGTCCTGAACTAACCTTGAATCTGAAGCAAAGGGGATCAGCCCGAGATTCTCATTCAGTGATCAACACTGAACACACATCCCTTACCATGGATTTTGGGCTGATTTTTTTTATTGTTGCTCTTTTAAAAGGGGTCCAGTGTGAGGTGAAACTTCTCGAGTCTGGAGGTGGCCTGGTGCAGCCTGGAGGATCCCTGAAACTCTCCTGTGCAGCCTCAGGATTCGATTTTAGTAGATCCTGGATGAGTTGGGTCCGGCAGGCTCCAGGGAAAGGGCTAGAATGGATTGGAGAAATTAATCCACATAGCAGTACGATAAACTATACGCCATCTCTAAAGGATAAATTCATCATCTCCAGAGACAACGCCAAAAGTACGCTGTACCTGCAAATGAGCAAAGTGAGATCTGAGGACACAGCCCTTTATTACTGTGCAAGACAAGGGTACTACGGTAGAAGTGACCACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCCTTGTCAAAGGCTACTTCCCTGAGCCGGTAACTGTAAAATGGAACTCTGGATCCCTGTCC",
        "GTAACTGGTAGGGTAC-1_contig_4": "GGGGACAGACGCACAACCCTGGACTCACAAGTCTTTCTCTTCAGTGACAAACACAGAAATAGAACATTCACCATGTACTTGGGACTGAACTGTGTATTCATAGTTTTTCTCTTAAAAGGTGTCCAGAGTGAAGTGAAGCTTGAGGAGTCTGGAGGAGGCTTGGTGCAACCTGGAGGATCCATGAAACTCTCCTGTGTTGCCTCTGGATTCACTTTCAGTAACTACTGGATGAACTGGGTCCGCCAGTCTCCAGAGAAGGGGCTTGAGTGGGTTGCTGAAATTAGATTGAAATCTAATAATTATGCAACACATTATGCGGAGTCTGTGAAAGGGAGGTTCACCATCTCAAGAGATGATTCCAAAAGTAGTGTCTACCTGCAAATGAACAACTTAAGAGCTGAAGACACTGGCATTTATTACTGTACCAACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AGCCTAACATCGATGT-1_contig_1": "TGGGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACCTAGCCACCTCGCA",
        "TGGTTCCAGATGTAAC-1_contig_2": "GAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGCCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GTTTCTAAGACAAAGG-1_contig_1": "CTGTCTTATATGGGGAAATACATCAGACCAGCATGGGCATCAAGATGGAGACACATTCTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGAAGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGGGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAACTACTGATTTACTGGGCATCCACCCGGCACACTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATTAGCAATGTGCAGTCTGAAGACTTGGCAGATTATTTCTGTCAGCAATATAGCAGCTATCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GCATGATAGCTAACAA-1_contig_6": "CTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GAACATCGTGACCAAG-1_contig_2": "GACTTACTGTTTTCTCTTTCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGAATTGAAGTCTCAACTATGAAAATCAGCAGTTCTCTGTCAGAGAAGCCCAAGCGCTTCCACGCATGCTTGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGT",
        "GCCTCTAGTAGAGGAA-1_contig_1": "TGGGGAGCTGCCAGGAAGCTAAGAAGCATCCTCTCTTCTAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCCAAAGTGTCAGTACATCTAGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCAAGTATGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATACTGCAACATATTACTGTCAGCACAGTTGGGAGATTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TGACGGCAGGACAGCT-1_contig_3": "GGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGG",
        "CGCTATCCACCGGAAA-1_contig_3": "GGGGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGGGCTGCTTCTCTGCCTGGTGACGTTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACTGTCTCAGGGTTCTCATTAACCGACTATGGTGTAAGCTGGATTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGGTGGTGGAAGCACATACTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAAACAGAACAAAAATTACTACGGTAGTAGCTTCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GCTGCGAGTCACAAGG-1_contig_1": "GATAGTCAGGTCACACCCTGTGCTGGAGTCAGCCTCACACAGAACACACACAGATATGGGTGTACCCACTCAGCTCCTGTTGCTGTGGCTTACAGTCGTAGTTGTCAGATGTGACATCCAGATGACTCAGTCTCCAGCTTCACTGTCTGCATCTGTGGGAGAAACTGTCACCATCACATGTGGAGCAAGTGAGAATATTTACGGTGCTTTAAATTGGTATCAGCGGAAACAGGGAAAATCTCCTCAGCTCCTGATCTATGGTGCAACCAACTTGGCAGATGGCATGTCATCGAGGTTCAGTGGCAGTGGATCTGGTAGACAGTATTCTCTCAAGATCAGTAGCCTGCATCCTGACGATGTTGCAACGTATTACTGTCAAAATGTGTTAAGTACTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "ACGAGCCCACAAGTAA-1_contig_2": "GAGTTTCAATCCTGTTCTTAGCCTGTGCAGAGAGATGGGTGTGGGGTAGCATTCAGACCTTTGTGACAATTGGGTTGCAACTCCTAAGAGGAAAGATCCCAAAGCTAAGCTCCTGGAAGCTATGGACCGTCACAAAAAAGAAGGCAGTCACTTTCATATTTCCCATGCATGAGTTGTGTCAGCCGGTAAAGGAGAGTACAGTGGAAGAAGGCCTTGAGACCTGAGTCAGCAGTGGATATCAGCAAAACAGAACAACCAACAACCCATGGAAGAGTGGGAAGAAGTCGAAATACCAGACTGTGGAGGGGGTCTCTCTGATGCACAGGTAAGGTGATGGCCTGAAGAATGTCATCCAGACAGGGTCCAAGGATGCCACAGGCTGGAACATGGGCAAGTGGATCTGAACACACACAACAAAAACAACAACTACTGCTACCACCACCACCAGAGTCAGCCTCAAGGAGATGATGGGATATATCAGGATACCTAAGATAAGA",
        "TACCTTAGTACTTGAC-1_contig_3": "GAAACAACCTATGATCAGTGTCCTCTCTACACAGTCCCTGACGACACTGACTCTAACCATGGGATGGAGCCGGATCTTTCTCTTCCTCCTGTCAATAACTGCAGGTGTCCATTGCCAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCCTCAGTGAAGATTTCCTGCAAAGCTTCTGGCTACGCATTCAGTAGCTCTTGGATGAACTGGGTGAAGCAGAGGCCTGGACAGGGTCTTGAGTGGATTGGACGGATTTATCCTGGAGATGGAGATACTAACTACAATGGGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACCTCTGTGGACTCTGCGGTCTATTTCTGTGCAAGACTTACTACGGCTACCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CGAGCCACATGGTAGG-1_contig_3": "GAGCTCTGACAGAGGAGGCCGGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACGGACCCCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGCCCTCATTTTAAAAGGTGTCCAGTGTGAGGTGCAGCTGGTGGAGTCTGGGGGAGACTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGGCATGTCTTGGGTTCGCCAGACTCCAGACAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGGCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTGCAAGCCTCTATGATGGTTACTTCATGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACC",
        "GACTGCGGTCGTGGCT-1_contig_3": "GGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TAGACCATCGCGCCAA-1_contig_3": "GCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GGGCACTTCACCGTAA-1_contig_3": "GACCTCCCCCAGCAGCCACTGTCCAGGCACTCAGAGAGCAAGGGATGTTGGGAAGACAGACTTGGGACAGCCCCAGGTATATTTACTTATATAGGACCTCCCTGGGAGATAGAATCCCAGGAGAAAAAAGGTTGGATGTATATTAAGGATGGCCCCTAACACTCTGCACTGCTACCTCTGGCCCCACCAGACAATGTTCCTGCAGAACCTGTTACCTTACTTGGCAGGGATTTTTGTCAAGGGATCTACTACTGTGCCTACTATAGGTACTTTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACCA",
        "CTTAACTTCATCACCC-1_contig_1": "GGGGCTGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGATGCAATGTATTTCTGTGCTCTATGGTACAGCACCCATTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGG",
        "GGAAAGCCATACCATG-1_contig_4": "AGGTCACAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CCAGCGATCCTAGAAC-1_contig_3": "GGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTG",
        "TTATGCTAGGCTCAGA-1_contig_3": "GAACCTATGATCACTGTCCTCTCCACAGTCCCTGAAGACACTGACTCTAACCATGGGATGGAGCTGTATCTTTCTCTTTCTCCTGTCAGTAACTGTAGGTGTGTTCTCTGAGGTTCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATATCCTGCAAGGCTTCTGGTTACTCATTTACTGGCTACTTTATGAACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGACGTATTAATCCTTACAATGGTGATACTTTCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAATCCTCTAGCACAGCCCACATGGAGCTCCTGAGCCTGACATCTGAGGACTCTGCAGTCTATTATTGTGGAAGAGATTTCCTTTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GGCTGGTCAACACCTA-1_contig_2": "GGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GACGGCTTCACATAGC-1_contig_1": "TGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "ATGTGTGAGCCCAGCT-1_contig_4": "GCATCTCATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGAGATTTATGGTCCCTTAGCTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GCGAGAAAGACACTAA-1_contig_2": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "ACAGCCGCACCGCTAG-1_contig_3": "AAGACCACTTTCTGAGTATTCATTTTAGGAGAAATACTTTTTTTTTAAATGAATGCAATTATCTAGACTTATTTCGGTTGAACATGCTGGTTGGTGGTTGAGAGGACACTCAGTCAGTCAGTGGCGTGAAGGGCTTCTAAGCCAGTCCACATGCTCTGTGTGAACTCCCTCTGGCCCTGCTTATTGTTGAATGGGCCAAAGGTCTGAGACCAGGCTGCTGCTGGGTAGGCCTGGACTTTGGGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GGCGACTAGCGATATA-1_contig_4": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CCGTTCAAGAGCTGCA-1_contig_7": "CTCTCTTCTCATGAAAACGAGTCCTGAACTAACCTTGAATCTGAAGCAAAGGGGATCAGCCCGAGATTCTCATTCAGTGATCAACACTGAACACACATCCCTTACCATGGATTTTGGGCTGATTTTTTTTATTGTTGCTCTTTTAAAAGGGGTCCAGTGTGAGGTGAAGCTTCTCGAGTCTGGAGGTGGCCTGGTGCAGCCTGGAGGATCCCTGAAACTCTCCTGTGCAGCCTCAGGATTCGATTTTAGTAGATACTGGATGAGTTGGGTCCGGCAGGCTCCAGGGAAAGGGCTAGAATGGATTGGAGAAATTAATCCAGATAGCAGTACGATAAACTATACGCCATCTCTAAAGGATAAATTCATCATCTCCAGAGACAACGCCAAAAATACGCTGTACCTGCAAATGAGCAAAGTGAGATCTGAGGACACAGCCCTTTATTACTGTGCAAGGCTTAGTAGCTACGATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GATCGATGTCCTAGCG-1_contig_2": "GGGAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATGAGTGCCTCAGTCATAATGTCCAGGGGACAAATTGTTCTCACCCAGTCTCCAGCACTCATGTCTGCATCTCCAGGGGAGAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGTACTGGTACCAGCAGAAGCCAAGATCCTCCCCCAAACCCTGGATTTATCTCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGGTCTGGGACCTCTTACTCTCTCACAATCAGCAGCATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAGTGGAGTAGTAACCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GTTCATTGTTCACGGC-1_contig_7": "GGGTGAGCCATTTCTTATATGGGAAGGGAGTGACCAGTTAGTCTTAAGGCACCATTGAGCCCAAGTCTTAGACATCATGGATTGGCTGTGGAACTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCAACACTGGAGAGCCAACATATGCTGAAGAGTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCAAGCCCGGGAGGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AGCTCCTTCTGGCGTG-1_contig_1": "TGGGAGGTCACAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTG",
        "TGACGGCTCCTTGGTC-1_contig_4": "GGAAATACATCAGACCAGCATGGGCATCAAGATGGAGACACATTCTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGAAGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGGGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAACTACTGATTTACTGGGCATCCACCCGGCACACTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATTAGCAATGTGCAGTCTGAAGACTTGGCAGATTATTTCTGTCAGCAATATAGCAGCTATCCTCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AAAGATGGTCTAACGT-1_contig_1": "GAAAGTATTTATGAGAAGAGCAGTAATTAGCTAGGGACCAAAATTCAAAGAGAAAATGGATTTTCAGGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCTAGGGGAGGAGATCACCCTAACCTGCAGTGCCAGCTCGAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAGGCACTTCTCCCAAACTCTTGATTTATAGCACATCCAACCTGGCTTCTGGAGTCCCTTCTCGCTTCAGTGGCAGTGGGTCTGGGACCTTTTATTCTCTCACAATCAGCAGTGTGGAGGCTGAAGATGCTGCCGATTATTACTGCCATCAGTGGAGTAGTTATCCATACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGAC",
        "CCCAGTTTCATACGGT-1_contig_3": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAA",
        "CATCAGAAGCACAGGT-1_contig_5": "GGATAACTCAGACACAAGTGAATGACAGATGGACCTCCATCTACTCTTATTTTAAAAAGAAGACAAACCCCACAGGCTCGAGAACTTTAGCGACTGTTTTGAGAGAAATCATTGGTCCCTGACTCAAGAGATGACTGGCAGATTGGGGATCAGAATACCCATACTCTGTGGCTAGTGTGAGGTTTAAGCCTCAGAGTCCCTGTGGTCTCTGACTGGTGCAAGGTTTTGACTAAGCGGAGCACCACAGTGCTAACTGGGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CCTAGCTCACGAAATA-1_contig_3": "GGGGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGTGAAGGAAATGGTGCTGGGCAGGAAGTCCCGGGCCAGGCAGCCCATGGCCACCAGATTCTTATCAGACAGGGG",
        "AACTTTCGTAGCTGCC-1_contig_2": "TGGGGATTGTCATTGCAGCCAGGACTCAGCATGGACATGAGGACCCCTGCTCAGTTTCTTGGAATCTTGTTGCTCTGGTTTCCAGGTATCAAATGTGACATCAAGATGACCCAGTCTCCATCTTCCATGTATGCATCTCTAGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAATAGCTATTTAAGCTGGTTCCAGCAGAAACCAGGGAAATCTCCTAAGACCCTGATCTATCGTGCAAACAGATTGGTAGATGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTCACCATCAGCAGCCTGGAGTATGAAGATATGGGAATTTATTATTGTCTACAGTATGATGAGTTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "GACTACATCTGCGTAA-1_contig_1": "TGGGGAATCAGAACTCTTCACACCAAGTTCTCAGAATGAGGTGCTCTCTTCAGTTCCTGGGGGTGCTTATGTTCTGGATCTCTGGAGTCAGTGGGGATATTGTGATAACCCAGGATGAACTCTCCAATCCTGTCACTTCTGGAGAATCAGTTTCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTATATAAGGATGGGAAGACATACTTGAATTGGTTTCTGCAGAGACCAGGACAATCTCCTCAGCTCCTGATCTATTTGATGTCCACCCGTGCATCAGGAGTCTCAGACCGGTTTAGTGGCAGTGGGTCAGGAACAGATTTCACCCTGGAAATCAGTAGAGTGAAGGCTGAGGATGTGGGTGTGTATTACTGTCAACAACTTGTAGAGTATCCTCGACACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GCCTCTAGTAGAGGAA-1_contig_6": "GCACACAAAATAACATAAGATCACTGTTCTCTCTACAGTTACTGAGCACACAGGAACTCACCATGGGATGGAGCTGTATCATCCTCTTCTTGGCAGCAACAGCTACAGGTGTCCACTCCCAGGTCCAACTGCAGCAGTCTGGGCCTGAGCTGGTGAGGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCAGGCTATACCTTCACCAGCTACTGGATGCACTGGGTGAAACAGAGGCCTGGACAAGGCCTTGAGTGGATTGGCATGATTGATCCTTCCAATAGTGAAACTAGGTTAAATCAGAAGTTCAAGGACAAGGCCACATTGAATGTAGACAAATCCTCCAACACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAATTATGGTAACTACCTATACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CTTAACTCACTGTTAG-1_contig_1": "TTAGCTTTCTTATATGGGGATGAAATGCATCACACCAGCATGGGCATCAAAATGGAGTCACAGATTCAGGTCTTTGTATTCGTGTTTCTCTGGTTGTCTGGTGTTGACGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGAGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGACAATCTCCTAAACTACTGATTTACTCGGCATCCTACCGGTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATCTGGGACGGATTTCACTTTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAACATTATAGTACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCA",
        "CCTACCATCTTTAGTC-1_contig_6": "ACAGAGAAAAACATGAGATCACTGTTCTCTCTACAGTTACTGAGCACACAGGACCTCACCATGGGATGGAGTTGTATCATCCTCTTCTTGGTAGCAACAGCTACAGGTGTCCACTCCCAGGTGCAACTGCAGCAGCCTGGGGCTGAGCTGGTGAAGCCTGGGGCCTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACATTTACCAGTTACAATATGCACTGGGTAAAGCAGACACCTGGACAGGGCCTGGAATGGATTGGAGCTATTTATCCAGGAAATGGTGATACTTCCTACAATCAGAAGTTCAAAGGCAAGGCCACATTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTGCAAGATTGGGTGATTACGGCTATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGGCTAGCATATGAGA-1_contig_1": "TCTTATATGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "AGCAGCCGTGGTGTAG-1_contig_1": "GATTGATCCTTTCTCTTAGAGTTTTATAAGAAGGTCTTTCTCAGCTCTGTGTCATAATCCACAGAAGCTCAGGGAAAATTTGAAGATGGTGTCCACTTCTCAGCTCCTTGGACTTTTGCTTTTCTGGACTTCAGCCTCCAGATGTGACATTGTGATGACTCAGTCTCCAGCCACCCTGTCTGTGACTCCAGGAGATAGAGTCTCTCTTTCCTGCAGGGCCAGCCAGAGTATTAGCGACTACTTACACTGGTATCAACAAAAATCACATGAGTCTCCAAGGCTTCTCATCAAATATGCTTCCCAATCCATCTCTGGGATCCCCTCCAGGTTCAGTGGCAGTGGATCAGGGTCAGATTTCACTCTCAGTATCAACAGTGTGGAACCTGAAGATGTTGGAGTGTATTACTGTCAAAATGGTCACAGCTTTCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "TAGTTGGTCTGGTATG-1_contig_1": "TGGGGAAATACATCAGATCAGCATGGGCATCAAGATGGAGTCACAGACTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGATGGAGACATTGTGATGACCCAGTCTCAAAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCGTCACCTGCAAGGCCAGTCAGAATGTGGGTACTAATGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAAGCACTGATTTACTCGGCATCCTACCGGTACAGTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAATGTGCAGTCTGAAGACTTGGCAGAGTATTTCTGTCAGCAATATAACAGCTATCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCT",
        "GATCTAGGTCACCCAG-1_contig_2": "GAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCA",
        "CCCAGTTTCATACGGT-1_contig_7": "CCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "AACGTTGGTATTCGTG-1_contig_1": "GGGGGTGAAAGACTCGCCAGAGCTAGACCTCCTTGGTGTGTGTTAACTCAGGAAGAAAAAGACGACATAAAACAACAGCCAGGACTCAGCATGGACATGAGGACCCCTGCTCAGTTTCTTGGAATCTTGTTGCTCTGGTTTCCAGGTATCAAATGTGACATCAAGATGACCCAGTCTCCATCTTCCATGTATGCATCTCTAGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAATAGCTATTTAAGCTGGTTCCAGCAGAAACCAGGGAAATCTCCTAAGACCCTGATCTATCGTGCAAACAGATTGGTAGATGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTCACCATCAGCAGCCTGGAGTATGAAGATATGGGAATTTATTATTGTCTACAGTATGATGAGTTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTC",
        "AGTGAGGGTTGTCTTT-1_contig_6": "GGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CCAGCGATCCTAGAAC-1_contig_4": "TGGGGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGTCAGCAGCCTTTCTACACTGCAGTGGGTATGCAACAATGCGCATCTTGTCTCTGATTTGCTACTGATGACTGGATTTCTCATCTGTTTGCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTG",
        "AGCTCCTTCCGCTGTT-1_contig_3": "GGGACATCGCTCTCACTGGAGGCTGATCTCTGAAGATAAGGAGGTGTAGCCTAAAAGATGAGAGTGCTGATTCTTTTGTGGCTGTTCACAGCCTTTCCTGGTATCCTGTCTGATGTGCAGCTTCAGGAGTCGGGACCTGGCCTGGTGAAACCTTCTCAGTCTCTGTCCCTCACCTGCACTGTCACTGGCTACTCAATCACCAGTGATTATGCCTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAGTGGATGGGCTACATAAGCTACAGTGGTAGCACTAGCTACAACCCATCTCTCAAAAGTCGAATCTCTATCACTCGAGACACATCCAAGAACCAGTTCTTCCTGCAGTTGAATTCTGTGACTACTGAGGACACAGCCACATATTACTGTGCAAGATCAGGTACTACGGTAGTAGCCCATGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "CGTCAGGTCTGCAAGT-1_contig_1": "GGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCACTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TACAGTGGTTATGTGC-1_contig_2": "GGGACTACTCAAGACTTTTTGTATCAAGTTCTCAGAATGAGGTGCCTAGCTGAGTTCCTGGGGCTGCTTGTGCTCTGGATCCCTGGAGCCATTGGGGATATTGTGATGACTCAGGCTGCACCCTCTGTACCTGTCACGCCTGGAGAGTCAGCAGCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTGCATAGTAATGGCAACACTTATTTGTTTTGGTTCCTGCAGAGGCCAGGCCAGTCTCCTCAGCTCCTGATATATCGGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTGGCAGTGGGTCAGGAACTGCTTTCACACTGAGAATGAGTAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTATGCAACATCTAGAATATCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "CGTCAGGAGCGAGAAA-1_contig_2": "GGGACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GTAACTGGTTGGGACA-1_contig_4": "GGACAGAGGAGGCCAGTCCTGGATTTGAGTTCCTCACATTCAGTGATGAGCACTGAACACAGACACCTCACCATGAACTTCGGGTTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCAGTGTGAAGTGAAGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGCCATGTCTTGGGTTCGCCAGACTCCAGAGAAGAGGCTGGAGTGGGTCGCATCCATTAGTAGTGGTGGTAGCACCTACTATCCAGACAGTGTGAAGGGCCGATTCACCATCTCCAGAGATAATGCCAGGAACATCCTGTACCTGCAAATGAGCAGTCTGAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGAGACTATGATTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "AATCCAGGTCAACATC-1_contig_4": "GGGATGGAAATGTGAATAACCTGCCTGAAGGGCCACAGGGGAGCTGGGGCTATCAGATCACAGGGTCCCAGGTTATGCAGCTCCTGGGTCAGGGTAGGAGGTAATTCCTGAGCAGAGGAACCAAGTAGATAGGACAGATGGAGCAGTTACAGAGGAAAAGGTTTGTAGAAGCTTGAATGTCCAGGCTGAGTATCTACATGGTGACTGGAAAAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCC",
        "CAACCTCCATGCTAGT-1_contig_1": "GGGGAGCTGCCAGGAGCCTAAGAAGCATCCTCTCATCTAGTTCTCAGAGATGGAGACAGACACAATCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGCTCCACTGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAAGGCCAGCCAAAGTGTTGATTATGATGGTGATAGTTATATGAACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAATCTAGAATCTGGGATCCCAGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATGCTGCAACCTATTACTGTCAGCAAAGTAATGAGGATCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCT",
        "TGGGAAGAGAGCAATT-1_contig_1": "TGGGGGAAATGCATCACACCAGCATGGGCATCAAAATGGAGTCACAGATTCAGGTCTTTGTATTCGTGTTTCTCTGGTTGTCTGGTGTTGACGGAGACATTGTGATGACCCAGTCTCACAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCATCACCTGCAAGGCCAGTCAGGATGTGAGTACTGCTGTAGCCTGGTATCAACAGAAACCAGGACAATCTCCTAAACTACTGATTTACTCGGCATCCTACCGGTACACTGGAGTCCCTGATCGCTTCACTGGCAGTGGATCTGGGACGGATTTCACTTTCACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAACATTATAGTACTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCTCAGAGGAAGGTGGAAACACGGTGAGAGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACTGCCGAAAATAAACCACAGTGACTGAAACCCAACCC",
        "CATCAGAAGCACAGGT-1_contig_1": "GGGGTGGCAGGATTGGAAGGACTTTGGGGTGGTAGGAAGGGGATGGGAATGGGTGGTTACAAGAGAAACAAGACTGTAGTAAATAAAGCTGAAACTCAAAGCAAGCTTTCAGCATCTTTAATTGGAGACACAAACTTCAAAGAATTGAAGTCTCAACTATGAAAATCAGCAGTTCTCTGTCAGAGAAGCCCAAGCGCTTCCACGCATGCTTGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGC",
        "GATGCTACATGTTCCC-1_contig_2": "GAGGTCACAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GTCTTCGTCTAACTGG-1_contig_2": "CAAATAAGTCTTCTCTCTGCCCATGAAAAACACCTTGGCCCTGACCCTGCAGCTCTGACAGAGGAGGCCGGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACAGACACCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGTCCTTATTTTAAAAGGTGTCCAGTGTGAAGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGCCATGTCTTGGGTTCGCCAGTCTCCAGAGAAGAGGCTGGAGTGGGTCGCAGAAATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACACTGTGACGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGGAAATGAGCAGTCTGAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGGTCTTACGACACCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CAGCATAGTTGCTCCT-1_contig_3": "AGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TCTGAGAGTTCTGGTA-1_contig_5": "GGGAGCCATATATTACTGTGCCAGAAATGAGGTACGAAGGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GAGCAGATCAAAGTAG-1_contig_3": "GGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTGATTTCCAGCTTGGTGCCTCCACCGAACGTCCACGG",
        "TAGACCATCGCGCCAA-1_contig_5": "GAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGCCACT",
        "CATTATCGTGCCTGCA-1_contig_3": "TGGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAAT",
        "TTATGCTCACTTCGAA-1_contig_1": "GAGTGTGAAGCTTCACACTGTCCATTCAACAAAGCCTCCATACCCACTATCATATTTGGTACAATTTGCATGCCCCCATTACCACATCAGCCTCCCTGATGACATATAATGCATGCCATCAAGCTGCATTGGAAGCTGGCTATCACTCAAGGCCATGACCATGCTCTCACTAGCTCCTCTCCTCAGCCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCGTGGCTACAGGAGAAAAAGTCACTATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGCACAGATTTTGTTTTTACAATTGAAAACACGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACATGCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATC",
        "GTGCTTCCAGGTCGTC-1_contig_1": "TGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCTCCGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TTCTTAGCACATGACT-1_contig_2": "GAAAAACATAAGGTCACTGTTCTCACTACAGTTTCTGAGCACACAGACCTCACCATGGGATGGAGCTCTATCATCCTCTTCTTGGTAGCAACAGCCTCAGGTGTCCACTCCCAGGTCCAACTGCAGCAACCTGGGTCTGAGCTGGTGAGGCCTGGAGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACATTCACCAGCTACTGGATGCACTGGGTGAAGCAGAGGCCTGGACAAGGCCTTGCGTGGATTGGAAATATTTATCCTGGTAGTGGTAGTACTAACTACGATGAGAAGTTCAAGAGCAAGGCCACACTGACTGTAGACACATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTACTGTACAAGAGTGGGATCTACTATGACCTATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCC",
        "GACTGCGGTACAGTGG-1_contig_3": "AGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTT",
        "GACGCGTCAGTAGAGC-1_contig_2": "GACTGCTACCTCTGGCCCTACCAGACAATGTTCCTGCAGAACCTGTTACCTTACTTGGCAGGGATTTTTGTCAAGGGATCTACTACTGTGCCTACTATGGTAACTCCTTTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "CATATTCCAGTCACTA-1_contig_1": "TGGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACC",
        "GCGACCAGTTGGTGGA-1_contig_8": "CTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TGGTTCCAGATGTAAC-1_contig_7": "GACATCTCTCTCATTAGAGGTTGATCTTTGAGGAAAACAGGGTGTTGCCTAAAGGATGAAAGTGTTGAGTCTGTTGTACCTGTTGACAGCCATTCCTGGTATCCTGTCTGATGTACAGCTTCAGGAGTCAGGACCTGGCCTCGTGAAACCTTCTCAGTCTCTGTCTCTCACCTGCTCTGTCACTGGCTACTCCATCACCAGTGGTTATTACTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATAAGCTACGACGGTAGCAATAACTACAACCCATCTCTCAAAAATCGAATCTCCATCACTCGTGACACATCTAAGAACCAGTTTTTCCTGAAGTTGAATTCTGTGACTACTGAGGACACAGCTACATATTACTGTGCAAGAAGGACTCGGGAAGAGGGCCTCTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CCAATCCTCAAACAAG-1_contig_2": "GCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAAC",
        "CGTGAGCGTACCGAGA-1_contig_1": "GACATACCAGCAAGCGAGTGACCAGTTAGTCTTAAGGCACCACTTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCAAGAGGTAGTAGCTACGGCTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TTCTTAGCACATGACT-1_contig_1": "TCCCTCGATCTTTCTTATATGGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCTCCCACGTTCGGTGCTGGGTCCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "CCTTTCTAGATGTGGC-1_contig_1": "GGGGAGGACAATGGCAGAAAGTCACTCTCAGTGAGGATACACCATCAGCATGAGGGTCCTTGCTGAGCTCCTGGGGCTGCTGCTGTTCTGCTTTTTAGGTGTGAGATGTGACATCCAGATGAACCAGTCTCCATCCAGTCTGTCTGCATCCCTTGGAGACACAATTACCATCACTTGCCATGCCAGTCAGAACATTAATGTTTGGTTAAGCTGGTACCAGCAGAAACCAGGAAATATTCCTAAACTATTGATCTATAAGGCTTCCAACTTGCACACAGGCGTCCCATCAAGGTTTAGTGGCAGTGGATCTGGAACAGGTTTCACATTAACCATCAGCAGCCTGCAGCCTGAAGACATTGCCACTTACTACTGTCAACAGGGTCAAAGTTATCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GGTGTTATCCGTAGGC-1_contig_2": "TGGGGAAGCTGGCTATCACTCAAGGCCATGACCATGCTCTCACTAGCTCCTCTCCTCAGCCTTCTTCTCCTCTGTGTCTCTGATTCTAGGGCAGAAACAACTGTGACCCAGTCTCCAGCATCCCTGTCCGTGGCTACAGGAGAAAAAGTCACTATCAGATGCATAACCAGCACTGATATTGATGATGATATGAACTGGTACCAGCAGAAGCCAGGGGAACCTCCTAAGCTCCTTATTTCAGAAGGCAATACTCTTCGTCCTGGAGTCCCATCCCGATTCTCCAGCAGTGGCTATGGCACAGATTTTGTTTTTACAATTGAAAACACGCTCTCAGAAGATGTTGCAGATTACTACTGTTTGCAAAGTGATAACATGCCTCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "AGCCTAACATCGATGT-1_contig_4": "GGGGACATACCAGCAAGGGAGTGACCAGTTAGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGTTATACCTTCACAGACTATTCAATGCACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACTGAGACTGGTGAGCCAACATATGCAGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCTAGCATTACTACGGTAGTAGCCCCCTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GCATGCGTCGCATGGC-1_contig_4": "GGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTG",
        "GAAGCAGCACCAGTTA-1_contig_9": "GAGCTCTGACAGAGGAGGCCGGTCCTGGATTCGAGTTCCTCACATTCAGTGATGAGCACTGAACACGGACCCCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCAGTGTGAAGTGCAGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTGACTATTACATGTATTGGGTTCGCCAGACTCCGGAAAAGAGGCTGGAGTGGGTCGCAACCATTAGTGATGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGGCGATTCACCATCTCCAGAGACAATGCCAAGAACAACCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTGCAAGAGAAAATGGTAGGCCTTACACCGGATACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TAAGCGTTCACTGGGC-1_contig_6": "TGGGGAACCTATGATCAGTGTCCTCTCCACAGTCCCTGAACACACTGACTCTCACCATGGAATGGAGCGGGGTCTTTATCTTTCTCCTGTCAGTAACTGCAGGTGTCCACTCCCAGGTCCAGCTGCAGCAGTCTGGAGCTGAGCTGGTAAGGCCTGGGACTTCAGTGAAGATGTCCTGCAAGGCTGCTGGATACACCTTCACTAACTACTGGATAGGTTGGGTAAAGCAGAGGCCTGGACATGGCCTTGAGTGGATTGGAGATATTTACCCTGGAGGTGGTTATACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACACATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCCATCTATTACTGTGCAAGTGGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CCGTTCAGTTATGTGC-1_contig_2": "GGGGATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TGTGGTATCACCCGAG-1_contig_4": "CCTTTGATTATGAATACCAGCCCTGACTCTAAAGCTCTGACAGAGGCACCAAACTGTGGACTCACAAGTCTTTCCCTTCAGTGACCAACAGGGACACAGAACATTCACCATGGACTTGAGACTGAGCTGCGCTTTTATTATTGTTCTTTTAAAAGGGGTCCAGAGTGAAGTGAAGCTTGAGGAGTCTGGAGGAGGCTTGGTGCAACCTGGAGGATCCATGAAACTCTCCTGTGTAGCCTCTGGATTTACTTTCAGTAGCTACTGGATGTCTTGGGTCCGCCAGTCTCCAGAGAAGGGGCTTGAGTGGGTTGCTGAAATTAGATTGAAATCTGATAATTATGCAACACATTATGCGGAGTCTGTGAAAGGGAAGTTCACCATCTCAAGAGATGATTCCAAAAGTCGTCTCTACCTGCAAATGAACAGCTTAAGAGCTGAAGACACTGGAATTTATTACTGTACAGATCTAGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "ACTATCTCAGTGGAGT-1_contig_1": "GGGGAGCTGCCAGGAAGCTAAGAAGCATCCTCTCTTCTAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCCAAAGTGTCAGTACATCTAGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCAAGTATGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATACTGCAACATATTACTGTCAGCACAGTTGGGAGATTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GAAGCAGCATGGTCAT-1_contig_1": "TGGGGCATTGTAATTGAAGTCAAGACTCAGCCTGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTCAGGACATTAGCAATTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTACTACACATCAAGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGAACAGATTATTCTCTCACCATTAGCAACCTGGAGCAAGAAGATATTGCCACTTACTTTTGCCAACAGGGTAATACGCTTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "ACCAGTATCAATACCG-1_contig_4": "GAAAACAACCTGTGATCAGTGTCCTCTCCACAGTCCCTGAACACACTGACTTTCACCATGGAATGGAGCGGGGTCTTTATCTTTCTCTTGTCAGTAACTGCAGATGTCCACTCCCAGGTCCAGTTGCAGCAGTCTGGAGCTGAGCTGGTAAGGCCTGGGACTTCAGTGAAGATATCCTGCAAGGCTTCTGGCTACACCTTCACTAACTACTGGCTAGGTTGGGTAAAGCAGAGGCCTGGACATGGACTTGAGTGGATTGGAGATATTTACCCTGGAGGTGGTTATACTAACTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACACATCCTCCAGCACTGCCTACATGCAGCTCAGTAGCCTGACATCTGAGGACTCTGCTGTCTATTTCTGTGCAAGAAAGGTTAGGGGTCTCTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CGGACTGTCACATAGC-1_contig_5": "GACATATGAACACTGTTTTCTCTACAGTCACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTCATCTTCTTCCTGATGGCAGTGGTTATAGGAATCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCAGAGCTTGTGAGGTCAGGGGCCTCAGTCAAGTTGTCCTGCACAGCTTCTGGCTTCAACATTAAAGACTACTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGATGGATTGATCCTGAGAATGGTGATACTGAATATGCCCCGAAGTTCCAGGGCAAGGCCACTATGACTGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTAATGCCTACCACGTCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TGACGGCGTGTGACCC-1_contig_3": "GAGTTCTAGAGCTTCCAGGTGTGCTGTCAGGCTGCTAGTGTATGCTCTCTCTAGTTTCTTTTTGGAGGCACTCAGAGCTATGAGTTTTCCTCTTAGGACTGCCTTCATTGTGTCCCATAAGTTTGGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTG",
        "CGACTTCAGTGGCACA-1_contig_2": "GTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACCTTGGCAGAGTATTTCTGTCAGCAATATAACAGCTATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "CGAACATAGTCATGCT-1_contig_4": "TGGGAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGAT",
        "GCGCCAAAGAGTGAGA-1_contig_3": "GAGAGTCAGCCTCAAGGAGATGATGGGATATATCAGGATACCTAAGATAAGATGAACTTGGTCTGTCTGGAAGCTGGCAGGACCAAATTCGCTGAGTCATCAGAGAAGAGGTGGCCAGAGGAGCAAGATCTCTGCAGCAGAAATCTGCAGGACTAACAAAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCCTTGTCAAAGGCTACTTCCCTGAGCCGGTAACTGTAAAATGGAACTATGGAGCCCTGTCCAG",
        "GAAATGAAGGTGCAAC-1_contig_4": "GATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGAATTGAAGTCTCAACTATGAAAATCAGCAGTTCTCTGTCAGAGAAGCCCAAGCGCTTCCACGCATGCTTGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GTAACGTCAGGGTATG-1_contig_3": "GAACATATGTCCAATGTCCTCTCCACAGACACTGAACACACTGACTCTAACCATGGGATGGAGCTGGATCTTTCTCTTCCTCCTGTCAGGAACTGCAGGCGTCCACTCTGAGGTCCAGCTTCAGCAGTCAGGACCTGAGCTGGTGAAACCTGGGGCCTCAGTGAAGATATCCTGCAAGGCTTCTGGATACACATTCACTGACTACAACATGCACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGATATATTTATCCTTACAATGGTGGTACTGGCTACAACCAGAAGTTCAAGAGCAAGGCCACATTGACTGTAGACAATTCCTCCAGCACAGCCTACATGGAGCTCCGCAGCCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGATCCTCATACTATGATTACGGGGACTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GATTCAGTCTTGCCGT-1_contig_1": "GGGGGAAATACATCAGATCAGCATGGGCATCAAGATGGAGTCACAGACTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGATGGAGACATTGTGATGACCCAGTCTCAAAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCGTCACCTGCAAGGCCAGTCAGAATGTGGGTACTAATGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAAGCACTGATTTACTCGGCATCCTACCGGTACAGTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAATGTGCAGTCTGAAGACTTGGCAGAGTATTTCTGTCAGCAATATAACAGCTATCCTCTACCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AGCAGCCGTGGTGTAG-1_contig_3": "GGAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGAGACAGACTCTTCTCCACAGTGTCCCCTTCATGTGTAACT",
        "ACACCAAGTTCGGGCT-1_contig_5": "GGGATCATCATCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGCATTACTCTTCTGCCTGGTAACATTCCCAAGCTGTATCCTTTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACCGTCTCAGGGTTCTCATTAACCGGCTATGGTGTAAACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAATGATATGGGGTGATGGAAGCACAGACTATAATTCAGCTCTCAAATCCAGACTGAGCATCAGCAAGGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCAGGTACTACTGTGCCAGAGGGGATGGTTACTACCTATATTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGGTT",
        "CAGCTGGCACATCTTT-1_contig_4": "GGCTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "CCTTTCTTCCGTTGCT-1_contig_1": "GGGGAAAGTACTTATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGCATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGAGGACAAATTGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAGAAGGTCACCATAACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTTCCAGCAGAAGCCAGGCACTTCTCCCAAACTCTGGATTTATAGCACATCCAACCTGGCTTCTGGAGTCCCTGCTCGCTTCAGTGGCAGTGGATCTGGGACCTCTTACTCTCTCACAATCAGCCGAATGGAGGCTGAAGATGCTGCCACTTATTACTGCCAGCAAAGGAGTAGTTACCCACCCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GTCACAACATTTGCCC-1_contig_3": "GGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCAC",
        "CAAGATCCATACCATG-1_contig_3": "GGGATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGTCCCTACTATGGTATCAACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "AAAGTAGCAAATCCGT-1_contig_9": "GAGTGTCCTCTCCACAGTCCCTGACGACACTGACTCTAGCCATGGAATGGAGCTGGGTCTTTCTCTTCCTCCTGTCAGTAACTGCAGGTGTCCAATCCCAGGTTCAACTGCAGCAGTCTGGGGCTGAGCTGGTGAGGCCTGGGGCTTCAGTGAAGCTGTCCTGCAAGGCTTTGGGCTACACATTTACTGACTATGAAATGCACTGGGTGAAGCAGACACCTGTGCATGGCCTGGAATGGATTGGAGCTATTCATCCAGGAAGTGGTGGTACTGCCTACAATCAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACAGCCTACATGGAGCTCAGCAGCCTGACATCTGAGGACTCTGCTGTCTATTACTGTACAAGATCTGGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "ACGAGCCGTCATATCG-1_contig_1": "GTCCAACTCCAGAGCATGGTATAGCAGGAAGACATGCAAATAGGTCTTCTCTGTGCCCATGAAAAACACCTCGGCCCTGACCCTGCAGCTCTGACAGAGGAGGCCTGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTGATCAGCACTGAACACAGACCCCTCACCATGAACTTCGGGCTCAGATTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCTGTGTGACGTGAAGCTCGTGGAGTCTGGGGGAGGCTTAGTGAAGCTTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATTACATGTCTTGGGTTCGCCAGACTCCAGAGAAGAGGCTGGAGTTGGTCGCAGCCATTAATAGTAATGGTGGTAGCACCTACTATCCAGACACTGTGAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACAGCCTTGTATTACTGTGCAAGACGGGTATGGGATGGTAACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GTCTCGTTCTACCTGC-1_contig_1": "GGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGTGAGCAGTTAACATCTGGAGGTGCCTC",
        "ACGCCAGGTACCGGCT-1_contig_7": "GGCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATT",
        "TGCCCTACAGTGGAGT-1_contig_2": "GGGGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCTTGGGGCTGCTCTTCTGCCTGGTGACATTCCCAAGCTGTGTCCTATCCCAGGTGCAGCTGAAGCAGTCAGGACCTGGCCTAGTGCAGCCCTCACAGAGCCTGTCCATCACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGTGGTGGAAGCACAGACTATAATGCAGCTTTCATATCCAGACTGAGCATCAGCAAGGACAATTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTAATGACACAGCCATATATTACTGTGCCAGAAATGACGGCTCGGGCTACGAGGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GTAACGTCAGGGTATG-1_contig_1": "GGGGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCC",
        "TTGCCGTCATCCAACA-1_contig_6": "GATATCTTTCTCACTGGAGGTTGATTTCTGAAGAAAAGAGGGTGTAGCCTAAAGGATGAGAATGTTGAGTGTTCTGTACCTGTTGTCAGCCCTTCCTGGTATCCTGTCTGATGTGCAGCTTCTGGAGTCAGGACCTAGCCTGGTGAGACCTTCTCAGACACTCTCCCTTACCTGCACTGTTACTGGCTTCTCCATCAACAGTGATTGTTACTGGATCTGAATCCGGCAGTTTCCAGGAAACAAACTGGAGTACATCGGGAACACATTCTACAGTGGTATCACTTACTACAACCCATCTCTCGAAAGTCGAACGTACATAACGCGTGACACATCTAAGAACCAGTTCTCACTGAAGTTGAGTTCTGTGACTACTGAGGACACAGCCACTTACTACTGTGCGAGAGATCATCCCGTTCATTACTACGGTCACGCCTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CTGCCTAGTGGGTCAA-1_contig_4": "GGAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTATTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTG",
        "CTACGTCGTTACAGAA-1_contig_5": "GGGAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GCTGGGTGTCCGAGTC-1_contig_2": "GAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGCTCTGTTTTCAAGGTACCAGATGTGATATCCAGATGACACAGACTACATCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTG",
        "TCTATTGCAAGCCCAC-1_contig_4": "GGGAACCTATGATCAGTGTCCTCTCCACAGTCCCTGAACACACTGACTCTAACCATGGAATGGATCTGGATCTTTCTCTTCATCCTCTCAGGAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGAGCTGAGCTGGCGAGGCCCGGGGCTTCAGTGAAGCTGTCCTGCAAGGCTTCTGGCTACACCTTCACTGACTACTATATAAACTGGGTGAAGCAGAGGACTGGACAGGGCCTTGAGTGGATTGGAGAGATTTATCCTGGAAGTGGTAATACTTACTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAGCACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCAGTCTATTTCTGTGCAAGATCTAGGGGTAACACCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TATCAGGTCACAATGC-1_contig_9": "TGGGGATCTCCTCACTAGAGCCCCCATCAGAGCATGGCTGTCCTGGTGCTGTTCCTCTGCCTGGTTGCATTTCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACTTGCACTGTCTCTGGGTTTTCATTAACCAGCTATGGTGTACACTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGCTGGTGGAAGCACAAATTATAATTCGGCTCTCATGTCCAGACTGAGCATCAGCAAAGACAACTCCAAGAGCCAAGTTTTCTTAAAAATGAACAGTCTGCAAACTGATGACACAGCCATGTACTACTGTGCCAGAGACTATGGTAACTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "GGAATAATCGACCAGC-1_contig_1": "TGGGGCATTGTCATTGCAGTCAGGACTCAGCATGGACATGAGGGCTCCTGCACAGATTTTTGGCTTCTTGTTGCTCTTGTTTCCAGGTACCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCTTATCTGCCTCTCTGGGAGAAAGAGTCAGTCTCACTTGTCGGGCAAGTCAGGACATTGGTAGTAGCTTAAACTGGCTTCAGCAGGAACCAGATGGAACTATTAAACGCCTGATCTACGCCACATCCAGTTTAGATTCTGGTGTCCCCAAAAGGTTCAGTGGCAGTAGGTCTGGGTCAGATTATTCTCTCACCATCAGCAGCCTTGAGTCTGAAGATTTTGTAGACTATTACTGTCTACAATATGCTAGTTCTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "ACGTCAAGTCTCCACT-1_contig_1": "GAACTTATGAGCACTGTTTTCTCTACAGTCACTGAATCTCAAGGTCCTTACAATGAAATGCAGCTGGGTCATCTTCTTCCTGATGGCAGTGGTTACAGGGGTCAATTCAGAGGTTCAGCTGCAGCAGTCTGGGGCTGAGCTTGTGAGGCCAGGGGCCTTAGTCAAGTTGTCCTGCAAAGCTTCTGGCTTCAACATTAAAGACTACTATATGCACTGGGTGAAGCAGAGGCCTGAACAGGGCCTGGAGTGGATTGGATGGATTGATCCTGAGAATGGTAATACTATATATGACCCGAAGTTCCAGGGCAAGGCCAGTATAACAGCAGACACATCCTCCAACACAGCCTACCTGCAGCTCAGCAGCCTGACATCTGAGGACACTGCCGTCTATTACTGTGCTAGAAGGGGTTATGGTAACTACGGGGTCTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CGAACATAGTCATGCT-1_contig_1": "GGGGTCATGGTCCACACAAACTCAGGGAAAGTTTGAAGATGGTATCCACACCTCAGTTCCTTGTATTTTTGCTTTTCTGGATTCCAGCCTCCAGAGGTGACATCTTGCTGACTCAGTCTCCAGCCATCCTGTCTGTGAGTCCAGGAGAAAGAGTCAGTTTCTCCTGCAGGGCCAGTCAGAGCATTGGCACAAGCATACACTGGTATCAGCAAAGAACAAATGGTTCTCCAAGGCTTCTCATAAAGTATGCTTCTGAGTCTATCTCTGGGATCCCTTCCAGGTTTAGTGGCAGTGGATCAGGGACAGATTTTACTCTTAGCATCAACAGTGTGGAGTCTGAAGATATTGCAGATTATTACTGTCAACAAAGTAATAGCTGGCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCC",
        "CACATTTAGCTCTCGG-1_contig_1": "TGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTCCAGTGAGCAGTTAACATCTGGAGGTGCCT",
        "CGAGCACGTTCAGTAC-1_contig_3": "GGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "AGGTCATCACGGTGTC-1_contig_2": "GGAGGTCACAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "AACGTTGTCTGCAAGT-1_contig_2": "GAAGAAGCATTCTCTCTTCCAGTTCTCAGAGATGGAGAAAGACACACTCCTGCTATGGGTCCTGCTTCTCTGGGTTCCAGGTTCCACAGGTGACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATCTCCTGCAGAGCCAGCGAAAGTGTTGATAATTATGGCATTAGTTTTATGAACTGGTTCCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCTATGCTGCATCCAACCAAGGATCCGGGGTCCCTGCCAGGTTTAGTGGCAGTGGGTCTGGGACAGACTTCAGCCTCAACATCCATCCTATGGAGGAGGATGATACTGCAATGTATTTCTGTCAGCAAAGTAAGGAGGTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "TTGGAACTCCACGAAT-1_contig_4": "GAGCAACATGTGTCCAATGTCCTCTCCACAGACACTGAACACACTGACTCCAACTATGGGATGGAGCTGGATCTTTCTCTTCCTCCTGTCAGGAACTGCAGGTGTCCACTCTGAGGTCCAGCTGCAACAGTCTGGACCTGAGCTGGTGAAGCCTGGAGCTTCAATGAAGATATCCTGCAAGGCTTCTGGTTACTCATTCACTGGCTACACCATGAACTGGGTGAAGCAGAGCCATGGAAAGAACCTTGAGTGGATTGGACTTATTAATCCTTACAATGGTGGTACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCACATTAACTGTAGACAAGTCATCCAGCACAGCCTACATGGAGCTCCTCAGTCTGACATCTGAGGACTCTGCAGTCTATTACTGTGCAAGCTATGGTACTACTGGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACCA",
        "AACACGTGTTCGGGCT-1_contig_2": "ATTGTAATTGAAGTCAAGACTCAGCATGGACATGATGTCCTCTGCTCAGTTCCTTGGTCTCCTGTTGCTTTGTTTTCAAGGTACCAGATATGATATCCAGATGACACAGACTACTTCCTCCCTGTCTGCCTCTCTGGGAGACAGAGTCACCATCAGTTGCAGGGCAAGTGAGGACATTAGCACTTATTTAAACTGGTATCAGCAGAAACCAGATGGAACTGTTAAACTCCTGATCTATTACACATCAGGATTACACTCAGGAGTCCCATCAAGGTTCAGTGGCAGTGGGTCTGGGGCAGATTATTCTCTCACCATCAGCAACCTGGAACCTGAAGATATTGCCACTTACTATTGTCAGCAGTATAGTAAGCTTCCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACT",
        "AAGGTTCTCTTGAGGT-1_contig_1": "TTCTGACTAGACCTTACTACAAACCTTGTGAAGTCAGTATGATTTGCATAAGTACTTCTTCCTTCTATTGATCCTTTCTCTTAGAGTTTTATAAGAAGGTCTTTCTCAGCTCTGTGTCATAATCCACAGAAGCTCAGGGAAAATTTGAAGATGGTGTCCACTTCTCAGCTCCTTGGACTTTTGCTTTTCTGGACTTCAGCCTCCAGATGTGACATTGTGATGACTCAGTCTCCAGCCACCCTGTCTGTGACTCCAGGAGATAGAGTCTCTCTTTCCTGCAGGGCCAGCCAGAGTATTAGCGACTACTTACACTGGTATCAACAAAAATCACATGAGTCTCCAAGGCTTCTCATCAAATATGCTTCCCAATCCATCTCTGGGATCCCCTCCAGGTTCAGTGGCAGTGGATCAGGGTCAGATTTCACTCTCAGTATCAACAGTGTGGAACCTGAAGATGTTGGAGTGTATTACTGTCAAAATGGTCACAGCTTTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGA",
        "CGAGCCATCTGCCAGG-1_contig_1": "TGGGGAGCTGCCAGGAGCCTAAGAAAGCATCCTCTCTTCCAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACAGGTAACATTGTGCTGACCCAATCTCCAGCTTCTTTGGCTGTGTCTCTAGGGCAGAGGGCCACCATATCCTGCAGAGCCAGTGAAAGTGTTGATAGTTATGGCAATAGTTTTATGCACTGGTACCAGCAGAAACCAGGACAGCCACCCAAACTCCTCATCTATCTTGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTAGGACAGACTTCACCCTCACCATTGATCCTGTGGAGGCTGATGATGCTGCAACCTATTACTGTCAGCAAAATAATGAGGATCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CTGTGCTAGCCCAGCT-1_contig_1": "GGGGAGCTGCCAGGAAGCTAAGAAGCATCCTCTCTTCTAGCTCTCAGAGATGGAGACAGACACACTCCTGCTATGGGTGCTGCTGCTCTGGGTTCCAGGTTCCACTGGTGACATTGTGCTGACACAGTCTCCTGCTTCCTTAGCTGTATCTCTGGGGCAGAGGGCCACCATCTCATGCAGGGCCAGCCAAAGTGTCAGTACATCTAGCTATAGTTATATGCACTGGTACCAACAGAAACCAGGACAGCCACCCAAACTCCTCATCAAGTATGCATCCAACCTAGAATCTGGGGTCCCTGCCAGGTTCAGTGGCAGTGGGTCTGGGACAGACTTCACCCTCAACATCCATCCTGTGGAGGAGGAGGATACTGCAACATATTACTGTCAGCACAGTTGGGAGATTCCCTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "TTGCCGTTCCACTCCA-1_contig_1": "GAGGTTTCTTATATGGGGGTTGTCATTGCAGTCAGGACTCAGCATGGACATGAGGGCCCCTGCTCAGTTTTTTGGGATCTTGTTGCTCTGGTTTCCAGGTATCAGATGTGACATCAAGATGACCCAGTCTCCATCCTCCATGTATGCATCGCTGGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAAAAGCTATTTAAGCTGGTACCAGCAGAAACCATGGAAATCTCCTAAGACCCTGATCTATTATGCAACAAGCTTGGCAGATGGGGTCCCATCAAGATTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTAACCATCAGCAGCCTGGAGTCTGACGATACAGCAACTTATTACTGTCTACAGCATGGTGAGAGCCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTGAGCTCCTCAGAGGAAGGTGGAAACACGGTGAGAGTGGGAGTGGACTTGGGCTGACCTAGGACAGTGACCTTGGTTCCACCGCCGAAAACATAACACAATGACTAAAACCCAACCCAAAACCTATGGGGCCAGCACCTGGTCCTGTTCTCC",
        "TGCGGGTGTAGGGACT-1_contig_2": "GGAGGTCACAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGG",
        "AAAGTAGCAAATCCGT-1_contig_7": "GGCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTG",
        "TTATGCTGTCGAGATG-1_contig_1": "TGGGGCTGAAATGCATCAGACCAGCATGGGCATCAAGATGGAATCACAGACTCTGGTCTTCATATCCATACTGCTCTGGTTATATGGTGCTGATGGGAACATTGTAATGACCCAATCTCCCAAATCCATGTCCATGTCAGTAGGAGAGAGGGTCACCTTGAGCTGCAAGGCCAGTGAGAATGTGGGTACTTATGTATCCTGGTATCAACAGAAACCAGAGCAGTCTCCTAAACTGCTGATATACGGGGCATCCAACCGGTACACTGGGGTCCCCGATCGCTTCACAGGCAGTGGATCTGCAACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGACCTTGCAGATTATCACTGTGGACAGAGTTACAGCTATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TTGAACGGTCATCCCT-1_contig_4": "GGGGAACCTATGATCAGTGTCATCTCCACAGTCCCTGAACACACTGACTCTAACCATGGAATGGAGGATCTTTCTCTTCATCCTGTCAGGAACTGCAGGTGTCCACTCCCAGGTTCAGCTGCAGCAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGATACACATTCACTGACTATGTTATAAGCTGGGTGAAGCAGAGAACTGGACAGGGCCTTGAGTGGATTGGAGAGATTTATCCTGGAAGTGGTAGTACTTACTACAATGAGAAGTTCAAGGGCAAGGCCACACTGACTGCAGACAAATCCTCCAACACAGCCTACATGCAGCTCAGCAGCCTGACATCTGAGGACTCTGCGGTCTATTTCTGTGCAAGATTAAAGTATGGTAACTACCCTTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCAC",
        "CTGATCCAGTAGCGGT-1_contig_5": "TGGGGACATACCAGCAAGGGAGTGACCAGTTTGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGATTGGCTGTGGAACTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACATGGCTACATATTTCTGTGCAGTATGGTTACGACGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "GGTATTGAGCTTTGGT-1_contig_2": "GGGGGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "CTAGCCTCATTCCTCG-1_contig_1": "TGGGGAACAGCATCAGACAGGCTGGGCAGCAAGATGGAATCACAGACCCAGGTCCTCATGTTTCTTCTGCTCTGGGTATCTGGTGCCTGTGCAGACATTGTGATGACACAGTCTCCATCCTCCCTGGCTATGTCAGTAGGACAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTAAATAGTAGCAATCAAAAGAATTATTTGGCCTGGTACCAGCAGAAACCAGGACAGTCTCCTAAACTTCTGGTATACTTTGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCATAGGCAGTGGATCTGGGACAGATTTCACTCTTACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGATTACTTCTGTCAGCAACATTATAGCACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAG",
        "AACACGTGTTCGGGCT-1_contig_4": "AGGAAGATGAGATGTGGATAGATACTGATGACTGTGTATGTGTTCTCCAACTGCTCTCTCCTGAAGTGCCTCAGATGTTTCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GACTACACAGGATTGG-1_contig_3": "AAAAACCGCATATGATCAGTATCCTCTTCACAGTCACTGAAAACATTGACTCTAATCATGGAATGTAACTGGATACTTCCTTTTATTCTGTCGGTAACTTCAGGGGTCTACTCAGAGGTTCAGCTCCAGCAGTCTGGGACTGTGCTGGCAAGGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGCTACACCTTTACCAGCTACTGGATGCACTGGGTAAAACAGAGGCCTGGACAGGGTCTGGAATGGATTGGCGCTATTTATCCTGGAAATAGTGATACTAGCTACAACCAGAAGTTCAAGGGCAAGGCCAAACTGACTGCAGTCACATCCACCAGCACTGCCTACATGGAGCTCAGCAGCCTGACAAATGAGGACTCTGCGGTCTATTACTGTACAAGACGCTATGATTACGAGTACTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCC",
        "CGAGCACGTCATGCCG-1_contig_7": "TGGGGCTCTGAGAATTTTGAAGGATAAGAGAGAACTACAACCTGTCTGTCTCAGCAGAGATCAGTAGTACCTGCATTATGGCCTGGACTCCTCTCTTCTTCTTCTTTGTTCTTCATTGCTCAGGTTCTTTCTCCCAACTTGTGCTCACTCAGTCATCTTCAGCCTCTTTCTCCCTGGGAGCCTCAGCAAAACTCACGTGCACCTTGAGTAGTCAGCACAGTACGTACACCATTGAATGGTATCAGCAACAGCCACTCAAGCCTCCTAAGTATGTGATGGAGCTTAAGAAAGATGGAAGCCACAGCACAGGTGATGGGATTCCTGATCGCTTCTCTGGATCCAGCTCTGGTGCTGATCGCTACCTTAGCATTTCCAACATCCAGCCTGAAGATGAAGCAATATACATCTGTGGTGTGGGTGATACAATTAAGGAACAATTTGTGTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GCACTCTAGGTCATCT-1_contig_1": "TGGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CGATGTAGTAACGCGA-1_contig_3": "GCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAAT",
        "CAGTAACCATCCCATC-1_contig_3": "GAACATATGTCCAATGTCCTCTCCACAGGCACTGAACACACTGACTCTAACCATGGGATGGAGCTGGATCTTTCTCTTCTTCCTGTCAGGAACTGCAGGTGTCCTCTCTGAGGTCCTGCTGCAACAGTCTGGACCTGAGCTGGTGAAGCCTGGGGCTTCAGTGAAGATACCCTGCAAGGCTTCTGGATACACATTCACTGACTACAACATGGACTGGGTGAAGCAGAGCCATGGAAAGAGCCTTGAGTGGATTGGAGATATTAATCCTAACAATGGTGGTACTATCTACAACCAGAAGTTCAAGGGCAAGGCCACATTGACTGTAGACAAGTCCTCCAGCACAGCCTACATGGAGCTCCGCAGCCTGACATCTGAGGACACTGCAGTCTATTACTGTGCAAGATTTTCCTTGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "CTACGTCGTTACAGAA-1_contig_3": "GGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "GTGGGTCTCGGCCGAT-1_contig_5": "GACACATACCAGCAAGGGAGTGACCAGTTTGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGATTGGCTGTGGAACTTGCTATTCCTGATGGCAGCTGCCCAAAGTGCCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAAACTATGGAATGAACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACCTACACTGGAGAGCCAACATATGCTGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACATGGCTACATATTTCTGTGCAAGGGGGGAGGTACCTACCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TGTGTTTTCAGGCGAA-1_contig_1": "GGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCC",
        "GCGAGAAAGACACTAA-1_contig_6": "ATGCTTTCTTATATGGGGACAGTAGTCTCTACAGTCACAGGAGTACACAGGGCATTGCCATGGGTTGGAGCTGTATCATCTTCTTTCTGGTAGCAACAGCTACAGGTGTGCACTCCCAGGTCCAGCTGCAGCAGTCTGGGCCTGAGGTGGTGAGGCCTGGGGTCTCAGTGAAGATTTCCTGCAAGGGTTCCGGCTACACATTCACTGATTATGCTATGCACTGGGTGAAGCAGAGTCATGCAAAGAGTCTAGAGTGGATTGGAGTTATTAGTACTTACAATGGTAATACAAACTACAACCAGAAGTTTAAGGGCAAGGCCACAATGACTGTAGACAAATCCTCCAGCACAGCCTATATGGAACTTGCCAGATTGACATCTGAGGATTCTGCCATCTATTACTGTGCAATCTACTATGGTAACTACTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "GCGCAACAGTGTGGCA-1_contig_1": "TGGGGACTGATCACTCTCCTATGTTCATTTCCTCAAAATGATGAGTCCTGCCCAGTTCCTGTTTCTGTTAGTGCTCTGGATTCGGGAAACCAACGGTGATGTTGTGATGACCCAGACTCCACTCACTTTGTCGGTTACCATTGGACAACCAGCCTCCATCTCTTGCAAGTCAAGTCAGAGCCTCTTAGATAGTGATGGAAAGACATATTTGAATTGGTTGTTACAGAGGCCAGGCCAGTCTCCAAAGCGCCTAATCTATCTGGTGTCTAAACTGGACTCTGGAGTCCCTGACAGGTTCACTGGCAGTGGATCAGGGACAGATTTCACACTGAAAATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTGGCAAGGTACACATTTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCC",
        "GCGCAGTAGTTGTAGA-1_contig_1": "GGGGTTTTCAGTGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAAC",
        "GCGCGATTCAACGGGA-1_contig_4": "GGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CAGCATATCCCTAACC-1_contig_3": "GGCCCAGCCCAGCCCAGCTACTCATAATGTATAAACCAGGCCTTTGGATTGTGACTGAACAGCATCAGACAGGCAGGGGAGCAATATGGTATCAGAGACCCATGTCCTCATATTTTTGCTGCTATGGGTGTCTGGTGCTTGTGGGGAAATTGTGTTGACCCAGTCTATACCATCCCTGACTGTGTCAGCAGGAGAGAGGGTCACTATCAGCTGCAAATCCAATCAGAATCTTTTATGGAGTGGAAACCAAAGGTACTGTTTGGTCTGGCACCAGTGGAAACCAGGGCAAACTCCTACACCGTTGATCACCTGGACATCTGATAGGTACTCTGGAGTCCCTGATCGTTTCATAGGCAGTGGATCTGTGACAGATTTCACTCTGACCATCAGCAGTGTGCAGGCTGAAGATGTGGCAGTTTATTTCTGTCAGCAGCATTTACACATTCCTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GCGCAACAGTGTGGCA-1_contig_3": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACA",
        "CGGACTGTCACATAGC-1_contig_4": "TGGGGGAGTTCTAGAGCTTCCAGGTGTGCTGTCAGGCTGCTAGTGTATGCTCTCTCTAGTTTCTTTTTGGAGGCACTCAGAGCTATGAGTTTTCCTCTTAGGACTGCCTTCATTGTGTCCCATAAGTTTGGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CCAGCGATCCTAGAAC-1_contig_1": "GGGGGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAG",
        "CGACTTCGTTTGACTG-1_contig_1": "TATGGGGGACCAATATTGAAAATAATAGACTTGGTTTGTGAATTATGGCCTGGACTTCACTTATACTCTCTCTCCTGGCTCTCTGCTCAGGAGCCAGTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGGAACAGTCATACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAGCAACCGAGCTCCAGGTGTTCCTGTCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "ATGGGAGGTCGTCTTC-1_contig_1": "GATAGTCAGGTCACACCCTGTGCTGGAGTCAGCCTCACACAGAACACACACAGATATGGGTGTACCCACTCAGCTCCTGTTGCTGTGGCTTACAGTCGTAGTTGTCAGATGTGACATCCAGATGACTCAGTCTCCAGCTTCACTGTCTGCATCTGTGGGAGAAACTGTCACCATCACATGTGGAGCAAGTGAGAATATTTACGGTGCTTTAAATTGGTATCAGCGGAAACAGGGAAAATCTCCTCAGCTCCTGATCTATGGTGCAACCAACTTGGCAGATGGCATGTCATCGAGGTTCAGTGGCAGTGGATCTGGTAGACAGTATTCTCTCAAGATCAGTAGCCTGCATCCTGACGATGTTGCAACGTATTACTGTCAAAATGTGTTAAGTACTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "TACTTACGTACGCACC-1_contig_5": "GGGACATCTCTCTCATTAGAGGTTGATCTTTGAGGAAAACAGGGTGTTGCCTAAAGGATGAAAGTGTTGAGTCTGTTGTACCTGTTGACAGCCATTCCTGGTATCCTGTCTGATGTACAGCTTCAGGAGTCAGGACCTGGCCTCGTGAAACCTTCTCAGTCTCTGTCTCTCACCTGCTCTGTCACTGGCTACTCCATCACCAGTGGTTATTACTGGAACTGGATCCGGCAGTTTCCAGGAAACAAACTGGAATGGATGGGCTACATAAGCTACGACGGTAGCAATAACTACAACCCATCTCTCAAAAATCGAATCTCCATCACTCGTGACACATCTAAGAACCAGTTTTTCCTGAAGTTGAATTCTGTGACTACTGAGGACACAGCTACATATTACTGTGCAAGAGATCGTAACTACTATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "AACTCCCTCTTCATGT-1_contig_2": "GAGCTCTGACAGAGGAGGCCAGTCCTGGAATTGATTCCCAGTTCCTCACGTTCAGTGATGAGCAGTGAACACAGACCCCTCACCATGAACTTCGGGCTCAGATTGATTTTCCTTGTCCTTACTTTAAAAGGTGTCCAGTGTGACGTGAAGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATACCATGTCTTGGGTTCGCCAGACTCCGGAGAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGCCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAAGTCTGAGGACACAGCCATGTATTACTGTACAAGAGAAGGCGGGGGGTACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACC",
        "GACTGCGGTACAGTGG-1_contig_1": "GGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAA",
        "GCAGCCACACTTACGA-1_contig_5": "GGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAG",
        "TCTGAGAGTCACTGGC-1_contig_2": "GGGGGTCAATACAATCAGGACTCAGAATGAACACGAGGGCCCCTGCTGAGTTCCTTGGGTTCCTGTTGCTCTGGTTTTTAGGTGCCAGATGTGATGTCCAGATGATTCAGTCTCCATCCTCCCTGTCTGCATCTTTGGGAGACATAGTCACCATGACTTGCCAGGCAAGTCAGGGCACTAGCATTAATTTAAACTGGTTTCAGCAAAAACCAGGGAAAGCTCCTAAGCTCCTGATCTATGGTGCAAGCAACTTGGAAGATGGGGTCCCATCAAGGTTCAGTGGCAGTAGATATGGGACAGATTTCACTCTCACCATCAGCAGCCTGGAGGATGAAGATATGGCAACTTATTTCTGTCTACAGCATAGTTATCTCCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCAGATCGGAAGAGC",
        "GTCAAGTGTAAGTGGC-1_contig_4": "GGACATAACAGCAAGAGAGTGTCCGGTTAGTCTCAAGGAAGACTGAGACACAGTCTTAGATATCATGGAATGGCTGTGGAACTTGCTATTTCTCATGGCAGCAGCTCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAGGATCTCCTGCAAGGCTTCTGGGTATACCTTCACAACTGCTGGAATGCAGTGGGTGCAAAAGATGCCAGGAAAGGGTTTGAAGTGGATTGGCTGGATAAACACCCACTCTGGAGTGCCAAAATATGCAGAAGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCATATTTACAGATAAGCAACCTCAAAAATGAGGACACGGCTACGTATTTCTGTGCGAACTGGGACTTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCC",
        "TAAACCGAGTTCGATC-1_contig_1": "TGGGGAAATACATCAGATCAGCATGGGCATCAAGATGGAGTCACAGACTCAGGTCTTTGTATACATGTTGCTGTGGTTGTCTGGTGTTGATGGAGACATTGTGATGACCCAGTCTCAAAAATTCATGTCCACATCAGTAGGAGACAGGGTCAGCGTCACCTGCAAGGCCAGTCAGAATGTGGGTACTAATGTAGCCTGGTATCAACAGAAACCAGGGCAATCTCCTAAAGCACTGATTTACTCGGCATCCTACCGGTACAGTGGAGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAATGTGCAGTCTGAAGACTTGGCAGAGTATTTCTGTCAGCAATATAACAGCTATCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "TTCGGTCCAAAGGAAG-1_contig_11": "GTAGTACCTTCTGTGAACCAGTTTCAATCCTGTTCTTAGCCTGTGCAGAGAGATGGGTGTGGGGTAGCATTCAGACCTTTGTGACAATTGGGTTGCAACTCCTAAGAGGAAAGATCCCAAAGCTAAGCTCCTGGAAGCTATGGACCGTCACAAAAAAGAAGGCAGTCACTTTCATATTTCCCATGCATGAGTTGTGTCAGCCGGTAAAGGAGAGTACAGTGGAAGAAGGCCTTGAGACCTGAGTCAGCAGTGGATATCAGCAAAACAGAACAACCAACAACCCATGGAAGAGTGGGAAGAAGTCGAAATACCAGACTGTGGAGGGGGTCTCTCTGATGCACAGGTAAGGTGATGGCCTGAAGAATGTCATCCAGACAGGGTCCAAGGATGCCACAGGCTGGAACATGGGCAAGTGGATCTGAACACACACAACAAAAACAACAACTACTGCTACCACCACCACCAGAGTCAGCCTCAAGGAGATGATGGGATATATCAGGATACCTAAGATAAGATGAACTTGGTCTGTCTGGAAGCTGGCAGGACCAAATTCGCTGAGTCATCAGAGAAGAGGTGGCCAGAGGAGCAAGATCTCTGCAGCAGAAATCTGCAGGACTAACAAAGCTACAACAACAGCCCCATCTGTCTATCCCTTGGTCCCTGGCTGCAGTGACACATCTGGATCCTCGGTGACACTGGGATGCCTTGTCAAAGGCTACTTCCCTGAGCCGGTAACTGTAAAATGGAACTATGGAGCCCTGTCCAGC",
        "GCACTCTAGGTCATCT-1_contig_8": "GAAGGGAGTGACCAGTTAGTCTTAAGGCACCACTGAGCCCAAGTCTTAGACATCATGGCTTGGGTGTGGACCTTGCTATTCCTGATGGCAGCTGCCCAAAGTATCCAAGCACAGATCCAGTTGGTGCAGTCTGGACCTGAGCTGAAGAAGCCTGGAGAGACAGTCAAGATCTCCTGCAAGGCTTCTGGTTATACCTTCACAGACTATTCAATGCACTGGGTGAAGCAGGCTCCAGGAAAGGGTTTAAAGTGGATGGGCTGGATAAACACTGAGACTGGTGAGCCAACATATGCAGATGACTTCAAGGGACGGTTTGCCTTCTCTTTGGAAACCTCTGCCAGCACTGCCTATTTGCAGATCAACAACCTCAAAAATGAGGACACGGCTACATATTTCTGTGCTAGAGGGGGGGACGGAGGCTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CCTTTCTTCCGTTGCT-1_contig_2": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGCCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "GCCAAATGTGACCAAG-1_contig_2": "GACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "TCTATTGCAAGCCCAC-1_contig_1": "GGATTGACTCCAAGCACTCATGAACACTGACCAGGCATTATTTGCTTTGATATTTTCTATCTGCTTTGCATGGGTTCCAACAGCCCAGCCCAGCTGCTCAGAATTTATAAACCAAGCCTTTGCACTGTGTCTCAACAGCATCAGACAGGCTGGGCAGCAAGATGGAATCACAGACCCAGGTCCTCATGTTTCTTCTGCTCTGGGTATCTGGTGCCTGTGCAGACATTGTGATGACACAGTCTCCATCCTCCCTGGCTATGTCAGTAGGACAGAAGGTCACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTAAATAGTAGCAATCAAAAGAACTATTTGGCCTGGTACCAGCAGAAACCAGGACAGTCTCCTAAACTTCTGGTATACTTTGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCATAGGCAGTGGATCTGGGACAGATTTCACTCTTACCATCAGCAGTGTGCAGGCTGAAGACCTGGCAGATTACTTCTGTCAGCAACATTATAGCACTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCCG",
        "TTGGAACTCCACGAAT-1_contig_2": "TGGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGCCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CGGCTAGCACGACTCG-1_contig_1": "GGAGTAAAATGCTGGATGTACACTAAGGATGGCCCCTGACACTCTGCACTGCTATCTCTGGCCCCACCAGGCAATGTTCCTGTAGAATCTCTTACCTTACTTGGCAGAGATTTTTGTCAAGGGATCTACTACTGTGTCTACTATGATTACGACGGGAGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACCA",
        "GTAACTGAGGATTCGG-1_contig_3": "GGAACCGACGATCAGTGTCCTCTCCAAAGTCCCTGAACACACTGACTCTAACCATGGAATGGAGTTGGATATTTCTCTTTCTCCTGTCAGGAACTGCAGGTGTCCACTCTGAGGTCCAGCTGCAGCAGTCTGGACCTGAGCTGGTAAAGCCTGGGGCTTCAGTGAAGATGTCCTGCAAGGCTTCTGGATACACATTCACTAGCTATGTTATGCACTGGGTGAAGCAGAAGCCTGGGCAGGGCCTTGAGTGGATTGGATATATTAATCCTTACAATGATGGTACTAAGTACAATGAGAAGTTCAAAGGCAAGGCCACACTGACTTCAGACAAATCCTCCAGCACAGCCTACATGGAGCTCAGCAGCCTGACCTCTGAGGACTCTGCGGTCTATTACTGTGCAAGATGGGAACCGACCTATGATGGTTACTACGAGGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAACTGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "TACTTACGTACGCACC-1_contig_1": "GACACTCTGCGCAGCTGTCAATACAATCAGGACTCAGAATGAACACGAGGGCCCCTGCTGAGTTCCTTGGGTTCCTGTTGCTCTGGTTTTTAGGTGCCAGATGTGATGTCCAGATGATTCAGTCTCCATCCTCCCTGTCTGCATCTTTGGGAGACATAGTCACCATGACTTGCCAGGCAAGTCAGGGCACTAGCATTAATTTAAACTGGTTTCAGCAAAAACCAGGGAAAGCTCCTAAGCTCCTGATCTATGGTGCAAGCAACTTGGAAGATGGGGTCCCATCAAGGTTCAGTGGCAGTAGATATGGGACAGATTTCACTCTCACCATCAGCAGCCTGGAGGATGAAGATATGGCAACTTATTTCTGTCTACAGCATAGTTATCTCCCATTCACGTTCGGCTCGGGGACAAAGTTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GAACCTAGTGGTCTCG-1_contig_4": "TGGGGAACTCAAATTCTCTGTCTATGACAACTGCTGTATGACTATATGAAGAAATGGGATAAAGATGCTATAGTCACTCGAGGACAGATAAAGGCCTGCTCACAGAGCAAGAATGGAAAGGAGAGGAAGGCAGGTCATGGACTTCAGTTCAGAGAAGGAAGCACAGCAAATACAAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTCAGAACTGAACCTCAACCACACTTGCACCA",
        "TTGACTTCAATTCCTT-1_contig_1": "TGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTTTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCATTGTACATAGTAATGGAAACACCTATTTAGAATGGTACCTGCAGAAACCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTACTGCTTTCAAGGTTCACATGTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCA",
        "TTCGGTCCAAAGGAAG-1_contig_3": "GGGTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CATGACACACGAAGCA-1_contig_2": "TGGGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "AGCAGCCCATCGGACC-1_contig_2": "TGGGGAAGTGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGACAACAAGTACATGGCCAGCAGCTTCTTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "TCGAGGCAGTGTGGCA-1_contig_6": "GGAGGTAAGCAAAGCTGGGCTTGAGCCAAAATGAAGTAGACTGTAATGAACTGGAATGAGCTGGGCCGCTAAGCTAAACTAGGCTGGCTTAACCGAGATGAGCCAAACTGGAATGAACTTCATTAATCTAGGTTGAATAGAGCTAAACTCTACTGCCTACACTGGACTGTTCTGAGCTGAGATGAGCTGGGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "CGGCTAGCATTTCACT-1_contig_1": "GGGGACTACTCAAGACTTTTTGTATCAAGTTCTCAGAATGAGGTGCCTAGCTGAGTTCCTGGGGCTGCTTGTGCTCTGGATCCCTGGAGCCATTGGGGATATTGTGATGACTCAGGCTGCACCCTCTGTACCTGTCACTCCTGGAGAGTCAGTATCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTGCATAGTAATGGCAACACTTACTTGTATTGGTTCCTGCAGAGGCCAGGCCAGTCTCCTCAGCTCCTGATATATCGGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTGGCAGTGGGTCAGGAACTGCTTTCACACTGAGAATCAGTAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTATGCAACATCTAGAATATCCTTTCACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CCCAATCGTGTGTGCC-1_contig_1": "GACACCCTGTGCTGGAGTCAGCCTCACACAGAACACACACAGATATGGGTGTACCCACTCAGCTCCTGTTGCTGTGGCTTACAGTCGTAGTTGTCAGATGTGACATCCAGATGACTCAGTCTCCAGCTTCACTGTCTGCATCTGTGGGAGAAACTGTCACCATCACATGTGGAGCAAGTGAGAATATTTACGGTGCTTTAAATTGGTATCAGCGGAAACAGGGAAAATCTCCTCAGCTCCTGATCTATGGTGCAACCAACTTGGCAGATGGCATGTCATCGAGGTTCAGTGGCAGTGGATCTGGTAGACAGTATTCTCTCAAGATCAGTAGCCTGCATCCTGACGATGTTGCAACGTATTACTGTCAAAATGTGTTAAGTACTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCACC",
        "TCTCATAGTCTCTTAT-1_contig_11": "GGTCACAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGTGCAGCTGTGAGAGAACAGGCCCAGGTGCTTGCCCCACAGGTTTAGGGTTGGGTTTCAGTCACTGTGGTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGG",
        "AGGGAGTCAAGTAATG-1_contig_6": "TTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCC",
        "TACTTACGTACGCACC-1_contig_2": "GAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGCCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CCTACCAGTTTAGGAA-1_contig_3": "GATCCTCTTCTCATAGAGCCTCCATCAGACCATGGCTGTCCTGGCACTGCTCCTCTGCCTGGTGACATTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGGAGTCAGGACCTGGCCTGGTGGCGCCCTCACAGAGCCTGTCCATCACATGCACTGTCTCAGGGTTCTCATTAACCAGCTATGGTGTAAGCTGGGTTCGCCAGCCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTAATATGGGGTGACGGGAGCACAAATTATCATTCAGCTCTCATATCCAGACTGAGCATCAGCAAGGATAACTCCAAGAGCCAAGTTTTCTTAAAACTGAACAGTCTGCAAACTGATGACACAGCCACGTACTACTGTGCCAAAAAAAGTACCTCTAACTGGGATGCTATGGACTACTGGGGTCAAGGAACCTCAGTCACCGTCTCCTCAGGTGATAAAAAGGAACCTGACATGTTCCTCCTCTCAGAGTGCAAAGCCCCAGAGGAAAATGAAAAGATAAACCTGGGCTGTTTAGTAATTGGAAGTCAGCCACTGAAAATCAGCTGGGAGCCAAAGAAGTCAAGTATAGTTGAACATGTCTTCCCCTCTGAAATGAGAAATGGCAATTATACAATGGTCCTCCAGGTCACTGTGCTGGCCTC",
        "GGATGTTGTATCGCAT-1_contig_1": "AGGTCACACCCTGTGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTG",
        "ACCGTAAGTGTAACGG-1_contig_2": "GGGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCACTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTAC",
        "ACAGCCGGTCCGAGTC-1_contig_1": "TGGGGGTCTGACCAGTCTCCTAAAGGCGCCTCATCAAAATGAAGCTGCCTGTTCTGCTAGTGGTGCTGCTATTGTTCACGAGTCCAGCCTCAAGCAGTGATGTTGTTCTGACCCAAACTCCACTCTCTCTGCCTGTCAATATTGGAGATCAAGCCTCTATCTCTTGCAAGTCTACTAAGAGTCTTCTGAATAGTGATGGATTCACTTATTTGGACTGGTACCTGCAGAAGCCAGGCCAGTCTCCACAGCTCCTAATATATTTGGTTTCTAATCGATTTTCTGGAGTTCCAGACAGGTTCAGTGGCAGTGGGTCAGGAACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATTTGGGAGTTTATTATTGCTTCCAGAGTAACTATCTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCG",
        "TTGAACGGTCATCCCT-1_contig_2": "GGTACTCATGAGAATAGCAGTAATTAGCTAGGGACCAAAATTCAAAGACAAAATGGATTTTCAAGTGCAGATTTTCAGCTTCCTGCTAATCAGTGCCTCAGTCATAATGTCCAGAGGAGAAAATGTTCTCACCCAGTCTCCAGCAATCATGTCTGCATCTCCAGGGGAAAAGGTCACCATGACCTGCAGTGCCAGCTCAAGTGTAAGTTACATGCACTGGTACCAGCAGAAGTCAAGCACCTCCCCCAAACTCTGGATTTATGACACATCCAAACTGGCTTCTGGAGTCCCAGGTCGCTTCAGTGGCAGTGGGTCTGGAAACTCTTACTCTCTCACGATCAGCAGCATGGAGGCTGAAGATGTTGCCACTTATTACTGTTTTCAGGGGAGTGGGTACCCACTCACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "AGGTCATGTAGTGAAT-1_contig_5": "TGGGGGACTGATCAGTCTCCTCAGGCTGTCTCCTCAGGTTGCCTCCTCAAAATGAAGTTGCCTGTTAGGCTGTTGGTGCTGATGTTCTGGATTCCTGCTTCCAGCAGTGATGTTGTGATGACCCAAACTCCACTCTCCCTGCCTGTCAGTCTTGGAGATCAAGCCTCCATCTCTTGCAGATCTAGTCAGAGCCTTGTACACAGTAATGGAAACACCTATTTACATTGGTACCTGCAGAAGCCAGGCCAGTCTCCAAAGCTCCTGATCTACAAAGTTTCCAACCGATTTTCTGGGGTCCCAGACAGGTTCAGTGGCAGTGGATCAGGGACAGATTTCACACTCAAGATCAGCAGAGTGGAGGCTGAGGATCTGGGAGTTTATTTCTGCTCTCAAAGTACACATGTTCCTTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CCTTACGAGGAGTTTA-1_contig_1": "GGTGAAAGACTCGCCAGAGCTAGACCTCCTTGGTGTGTGTTAACTCAGGAAGAAAAAGACGACATAAAACAACAGCCAGGACTCAGCATGGACATGAGGACCCCTGCTCAGTTTCTTGGAATCTTGTTGCTCTGGTTTCCAGGTATCAAATGTGACATCAAGATGACCCAGTCTCCATCTTCCATGTATGCATCTCTAGGAGAGAGAGTCACTATCACTTGCAAGGCGAGTCAGGACATTAATAGCTATTTAAGCTGGTTCCAGCAGAAACCAGGGAAATCTCCTAAGACCCTGATCTATCGTGCAAACAGATTGGTAGATGGGGTCCCATCAAGGTTCAGTGGCAGTGGATCTGGGCAAGATTATTCTCTCACCATCAGCAGCCTGGAGTATGAAGATATGGGAATTTATTATTGTCTACAGTATGATGAGTTTCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCAGGCCACTGTCACACCACTCGGGGAAAAGTTGGAAATCAGACACACCAGTGTGGCTTTGTTTTCCTTG",
        "TCCCGATTCAACTCTT-1_contig_3": "TGGGGCACTCCTCTTTCTGTCTTGATTACTTTTGACTCACCATATCAAGTTCGCAGAATGAGGTTCTCTGCTCAGCTTCTGGGGCTGCTTGTGCTCTGGATCCCTGGATCCACTGCAGATATTGTGATGACGCAGGCTGCATTCTCCAATCCAGTCACTCTTGGAACATCAGCTTCCATCTCCTGCAGGTCTAGTAAGAGTCTCCTACATAGTAATGGCATCACTTATTTGTATTGGTATCTGCAGAAGCCAGGCCAGTCTCCTCAGCTCCTGATTTATCAGATGTCCAACCTTGCCTCAGGAGTCCCAGACAGGTTCAGTAGCAGTGGGTCAGGAACTGATTTCACACTGAGAATCAGCAGAGTGGAGGCTGAGGATGTGGGTGTTTATTACTGTGCTCAAAATCTAGAACTTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCC",
        "AATCGGTCATACAGCT-1_contig_3": "GGATCCTCTTCTCATAGAGCCTCCATCAGAGCATGGCTGTCCTGGTGCTGCTCCTCTGCCTGGTGACATTCCCAAGCTGTGTCCTGTCCCAGGTGCAGCTGAAGCAGTCAGGACCTAGCCTAGTGCAGCCCTCACAGAGCCTGTCCATAACCTGCACAGTCTCTGGTTTCTCATTAACTAGCTATGGTGTACACTGGGTTCGCCAGTCTCCAGGAAAGGGTCTGGAGTGGCTGGGAGTGATATGGAGAGGTGGAAGCACAGACTACAATGCAGCTTTCATGTCCAGACTGAGCATCACCAAGGACAACTCCAAGAGCCAAGTTTTCTTTAAAATGAACAGTCTGCAAGCTGATGACACTGCCATATACTACTGTGCCAAACTTGACTACTGGGGCCAAGGCACCACTCTCACAGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAGG",
        "ATTCTACAGTGCGATG-1_contig_7": "GCACTTACTGTTTTCTCTTTCATGTGAAGTGAAATGGCTGTAGCCTAATGTCCTTCTGAAATCAAGAGACACATCTTCACTTTCCAGACTGGGACCAGACTTTTGTGCCACGTTGTGACATGCACACATGCACACACAGGAAAGGACTTGGCTTGTGCCCACATGCCTTTCTTCAGGGACAAGTGGGAATGGACATAAGGAGCGTCTCATCCTTACGTGGCCTCCTTTGTCTTCCTCCTTCCTGGCCAAGACGGTGATTTACTGCCCAAAGACAAGTCCATCTGAAGCAAGAACTGACTATCGAGGACTCTGTAGTGACTGATAAACTGGGAAGAGAAAATTGGACCAGAGACTTCAAGGCAGCTCTCAGACAAGAATATTCCAGTGCCAGAATTGAAGTCTCAACTATGAAAATCAGCAGTTCTCTGTCAGAGAAGCCCAAGCGCTTCCACGCATGCTTGGAGAGGGGGTTAAGCTTTCGCAGCTACCCACTGCTCTGTTCCTCTTCAGTGAGGAGGGTTTTTGTACAGCCAGACAGTGGAGTACTACCACTGTGGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "CACACAAAGCATGGCA-1_contig_3": "GGGGACTGCTACCTCTGGCCCCACCAGACAATGTTCCTGCAGAACCTGTTACCTTACTTGGCAGGGATTTTTGTCAAGGGATCTACTACTGTGCCTACTATAGGTACGACGGAGGTTGACCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TACGGGCTCGTCGTTC-1_contig_2": "GGGAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "CACACAAAGCATGGCA-1_contig_2": "GGGAGGTCATAGCTCCACCCATTGTAGCTAGCTAGTAGTTTGATTCAGCTCAGCTGTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGG",
        "TGCCCTAGTAGAGGAA-1_contig_4": "GGGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGCAGTCAAAGTGTTTTATACAGTTCAAATCAGAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATCTACTGGGCATCCAC",
        "CATGACAAGCCAGGAT-1_contig_1": "GGGGATTGTCATTGCAGTCAGGACTCAGCATGGACATGAGGGCTCCTGCACAGATTTTTGGCTTCTTGTTGCTCTTGTTTCCAGGTACCAGATGTGACATCCAGATGACCCAGTCTCCATCCTCCTTATCTGCCTCTCTGGGAGAAAGAGTCAGTCTCACTTGTCGGGCAAGTCAGGACATTGGTAGTAGCTTAAACTGGCTTCAGCAGGAACCAGATGGAACTATTAAACGCCTGATCTACGCCACATCCAGTTTAGATTCTGGTGTCCCCAAAAGATTCAGTGGCAGTAGGTCTGGGTCAGATTATTCTCTCACCATCAGCAGCCTTGAGTCTGAAGATTTTGTAGACTATTACTGTCTACAATATGCTAGTTCTCCGTGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCTTGAACAACTTCTACCCCAAAGACATCAATGTCAAGTGGAAGATTGATGGCAGTGAACGACAAAATGGCGTCCTGAACAGTTGGACTGATCAGGACAGCAAAGACAGCACCTACAGCATGAGCAGCACCCTCACGTTGACCAAGGACGAGTATGAACGACAT",
        "GATCGCGGTCGGCTCA-1_contig_5": "GAGCTCTGACAGAGGAGGCCTGTCCTGGATTCGATTCCCAGTTCCTCACATTCAGTCAGCACTGAACACGGACCCCTCACCATGAACTTCGGGCTCAGCTTGATTTTCCTTGTCCTTGTTTTAAAAGGTGTCCAGTGTGAAGTGATGCTGGTGGAGTCTGGGGGAGGCTTAGTGAAGCCTGGAGGGTCCCTGAAACTCTCCTGTGCAGCCTCTGGATTCACTTTCAGTAGCTATGCCATGTCTTGGGTTCGCCAGACTCCGGAGAAGAGGCTGGAGTGGGTCGCAACCATTAGTAGTGGTGGTAGTTACACCTACTATCCAGACAGTGTGAAGGGGCGATTCACCATCTCCAGAGACAATGCCAAGAACACCCTGTACCTGCAAATGAGCAGTCTGAGGTCTGAGGACACGGCCATGTATTACTGTGCAAGAATTCATTACTACGGCTACTGGTACTTCGATGTCTGGGGCGCAGGGACCACGGTCACCGTCTCCTCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCAT",
        "GTCATTTGTTAAGACA-1_contig_5": "GGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCACGGGACAAACTCTTCTCCACAGTGTGACCTTCATGAGTGACCTGGCAGCTGTAACTGCTATGCCTTTCCCATGCTCTTGCTGTC",
        "GACTAACCATGCCTTC-1_contig_4": "GGTCTCCCACCCAGACCTGGGAATGTATGGTTGTGGCTTCTGCCACCCATCCACCTGGCTGCTCATGGACCAGCCAGCCTCGGTGGCTTTGAAGGAACAATTCCACACAAAGACTCTGGACCTCTCCGAAACCAGGCACCGCAAATGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCA",
        "CCGTACTTCACAAACC-1_contig_4": "GAAGATCACAGTAGTCTCTACAGTCACAGGAGTACACAGGGCATTGCCATGGGTTGGAGCTGTATCATCTTCTTTCTGGTAGCAACAGCTACAGGTGTGCACTCCCAGGTCCAGCTGCAGCAGTCTGGGCCTGAGGTGGTGAGGCCTGGGGTCTCAGTGAAGATTTCCTGCAAGGGTTCCGGCTACACATTCACTGATTATGCTATGCACTGGGTGAAGCAGAGTCATGCAAAGAGTCTAGAGTGGATTGGAGTTATTAGTACTTACAATGGTAATACAAACTACAACCAGAAGTTTAAGGGCAAGGCCACAATGACTGTAGACAAATCCTCCAGCACAGCCTATATGGAACTTGCCAGATTGACATCTGAGGATTCTGCCATCTATTACTGTGCAACTGGGACCTGGTTTGCTTACTGGGGCCAAGGGACTCTGGTCACTGTCTCTGCAGAGAGTCAGTCCTTCCCAAATGTCTTCCCCCTCGTCTCCTGCGAGAGCCCCCTGTCTGATAAGAATCTGGTGGCCATGGGCTGCCTGGCCCGGGACTTCCTGCCCAGCACCATTTCCTTCACCTGGAACTACCAGAACAACACTGAAGTCATCCAGGGTATCAGAACCTTCCCAACACTGAGGACAGGGGGCAAGTACCTAGCCACCTCGCAGGTGTTGCTGTCTCCCAAGAGCATCCTTGAAG",
        "TTCTACACATGCCCGA-1_contig_1": "GATTTCTCTGAGAATTTTGAAGGATAAGAGAGAACTACAACCTGTCTGTCTCAGCAGAGATCAGTAGTACCTGCATTATGGCCTGGACTCCTCTCTTCTTCTTCTTTGTTCTTCATTGCTCAGGTTCTTTCTCCCAACTTGTGCTCACTCAGTCATCTTCAGCCTCTTTCTCCCTGGGAGCCTCAGCAAAACTCACGTGCACCTTGAGTAGTCAGCACAGTACGTACACCATTGAATGGTATCAGCAACAGCCACTCAAGCCTCCTAAGTATGTGATGGAGCTTAAGAAAGATGGAAGCCACAGCACAGGTGATGGGATTCCTGATCGCTTCTCTGGATCCAGCTCTGGTGCTGATCGCTACCTTAGCATTTCCAACATCCAGCCTGAAGATGAAGCAATATACATCTGTGGTGTGGGTGATACAATTAAGGAACAATTTGTTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGAAGGCAAATGGTACACCTATCACCCAGGGTGTGGACACTTCAAATCCCACCAAAGAGGGCAACAAGTTCATGGCCAGCAGCTTCCTACATTTGACATCGGACCAGTGGAGATCTCACAACAGTTTTACCTGTCAAGTTACACATGAAGGGGACACTGTGGAGAAGAGTCTGTCTCCTGC",
        "GATGAAATCCTGCTTG-1_contig_1": "GATGAAATCCTGCTTGAGTGGGGCTGACCAATATTGAAAAGAATAGACCTGGTTTGTGAATTATGGCCTGGATTTCACTTATACTCTCTCTCCTGGCTCTCAGCTCAGGGGCCATTTCCCAGGCTGTTGTGACTCAGGAATCTGCACTCACCACATCACCTGGTGAAACAGTCACACTCACTTGTCGCTCAAGTACTGGGGCTGTTACAACTAGTAACTATGCCAACTGGGTCCAAGAAAAACCAGATCATTTATTCACTGGTCTAATAGGTGGTACCAACAACCGAGCTCCAGGTGTTCCTGCCAGATTCTCAGGCTCCCTGATTGGAGACAAGGCTGCCCTCACCATCACAGGGGCACAGACTGAGGATGAGGCAATATATTTCTGTGCTCTATGGTACAGCAACCATTTTATTTTCGGCAGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACACTCACCATGTTTCCACCTTCCCCTGAGGAGCTCCAGGAAAACAAAGCCACACTCGTGTGTCTGATTTCCAATTTTTCCCCAAGTGGTGTGACAGTGGCCTGG",
        "TTCTTAGAGCTAGTCT-1_contig_1": "GGGGATGAACTGAGCAGCATCAGACAGGCAGGGGAAGCAAGATGGATTCACAGGCCCAGGTTCTTATGTTACTGCTGCTATGGGTATCTGGTACCTGTGGGGACATTGTGATGTCACAGTCTCCATCCTCCCTAGCTGTGTCAGTTGGAGAGAAGGTTACTATGAGCTGCAAGTCCAGTCAGAGCCTTTTATATAGTAGCAATCAAAAGAACTACTTGGCCTGGTACCAGCAGAAACCAGGGCAGTCTCCTAAACTGCTGATTTACTGGGCATCCACTAGGGAATCTGGGGTCCCTGATCGCTTCACAGGCAGTGGATCTGGGACAGATTTCACTCTCACCATCAGCAGTGTGAAGGCTGAAGACCTGGCAGTTTATTACTGTCAGCAATATTATAGCTATCCGTACACGTTCGGAGGGGGGACCAAGCTGGAAATAAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTC",
        "AGAATAGTCTCGATGA-1_contig_2": "ACACACCCTGTGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCGGACGTTCGGTGGAGGCACCAAGCTGGAAATCAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCT",
        "CCGGTAGTCACTCTTA-1_contig_6": "CTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTA",
        "GCTGGGTTCGTGGACC-1_contig_9": "TCTGCATTTCTTATATGGGGCTCTCTGATTTTTTCCAGCATGAGCTGCAGAGATTGAGCTAGTGGTCTCAACACTTTGCATAGACTTTACACAGCTGGTGGTCTCCTCAAGCTGTCACTGGGGGCTCTTCTCTGTCCCTTCTTCTTATCCTGTGAAGATCTTCCATACCAAGTATGGAAAATTCATGAATTATTTTCTGGAAAGACTTCCTATGAGGACATATGGATCCTGGGAAGAAGGATCTTTCAGTGATGTCACCACCTTCCAAGAATTACCAGGAGCTGCATACATCACAGATGCAACTTGAGAATAAAATGCATGCAAGGTTTTTTGCATGAGTCTATATCACAGTGCTGGGTGTTCGGTGGAGGAACCAAACTGACTGTCCTAGGCCAGCCCAAGTCTTCGCCATCAGTCACCCTGTTTCCACCTTCCTCTGAAGAGCTCGAGACTAACAAGGCCACACTGGTGTGTACGATCACTGATTTCTACCCAGGTGTGGTGACAGTGGACTGGAAGGTAGATGGTACCCCTGTCACTCAGGGTATGGAGACAACCCAGCCTTCCAAACAGAGCAACAACAAGTACATGGCTAGCAGCTACCTGACCCTGACAGCAAGAGCATGGGAAAGGCATAGCAGTTACAGCTGCCAGGTCACTCATGAAGGTCACACTGTGGAGAAGAGTTTGTCCCGTGC",
        "ACGAGCCGTCATATCG-1_contig_2": "GCAGCCAGGTTTCTTATATGGGGGGCTGTAATCAGCATCACACTGAAAACACACAGACATGAGTGTGCCCACTCAGGTCCTGGGGTTGCTGCTGCTGTGGCTTACAGGTGCCAGATGTGACATCCAGATGACTCAGTCTCCAGCCTCCCTATCTGCATCTGTGGGAGAAACTGTCACCATCACATGTCGAGCAAGTGAGAATATTTACAGTTATTTAGCATGGTATCAGCAGAAACAGGGAAAATCTCCTCAGCTCCTGGTCTATAATGCAAAAACCTTAGCAGAAGGTGTGCCATCAAGGTTCAGTGGCAGTGGATCAGGCACACAGTTTTCTCTGAAGATCAACAGCCTGCAGCCTGAAGATTTTGGGAGTTATTACTGTCAACATCATTATGGTACTCCTCCGCTCACGTTCGGTGCTGGGACCAAGCTGGAGCTGAAACGGGCTGATGCTGCACCAACTGTATCCATCTTCCCACCATCCAGTGAGCAGTTAACATCTGGAGGTGCCTCAGTCGTGTGCTTCCCA",
        "ACGAGCCAGCCGATTT-1_contig_2": "GTGAGAGAACAGGACCAGGTGCTGGCCCCATAGGTTTTGGGTTGGGTTTTAGTCATTGTGTTATGTTTTCGGCGGTGGAACCAAGGTCACTGTCCTAGGTCAGCCCAAGTCCACTCCCACTCTCACCGTGTTTCCACCTTCCTCTGAGGAGCTCAAGGAAAACAAAGCCACACTGGTGTGTCTGATTTCCAACTTTTCCCCGAGTGGTGTGACAGTGGCCTGGGAAGCACACGACTGAGGCACCTCCAGATGTTAACTGCTCACTGGATGGTGGGAAGATGGATACAGTTGGTGCAGCATCAGCCCGTTTGATTTCCAGCTTGGTGCCTCCACCGAACGTCCACGGAAGCGTATTACCCTGTTGGCAAAAGTAAGTGGCAATATCTTCTTGCTCCAGGTTGCTAATGGTGAGAGAATAATCTGTTCCAGACCCACTGCCAC",
        "TTCGAAGCATCCTTGC-1_contig_1": "TGGGAGCACTGTCCAGATGAAGCTGATTTGCATGTGCTGAGATCATATTCTACTGCCCCAGAGATTTAATAATCTGATCATACACACTCCAACAGTCATTCTTGGTCAGGAG