"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BillingAlarm = void 0;
const cdk = require("@aws-cdk/core");
const cw = require("@aws-cdk/aws-cloudwatch");
const cwa = require("@aws-cdk/aws-cloudwatch-actions");
const sns = require("@aws-cdk/aws-sns");
const sub = require("@aws-cdk/aws-sns-subscriptions");
/**
 * A CDK construct that sets up email notification for when you exceed a given AWS estimated charges amount.
 *
 * Note: The email address used as SNS Topic endpoint must be manually confirmed
 * once the stack is deployed.
 *
 * @stability stable
 */
class BillingAlarm extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const billingAlarmTopic = new sns.Topic(this, 'BillingAlarmNotificationTopic', {
            topicName: 'BillingAlarmNotificationTopic',
        });
        billingAlarmTopic.addSubscription(new sub.EmailSubscription(props.email, {
            json: true,
        }));
        const billingAlarmMetric = new cw.Metric({
            metricName: 'EstimatedCharges',
            namespace: 'AWS/Billing',
            statistic: 'Maximum',
            dimensions: {
                Currency: 'USD',
            },
        }).with({
            // Evaluates the metric every 9 hours. This is needed because
            // `EstimatedCharges` metrics is updated every 6 hours
            // See: https://forums.aws.amazon.com/thread.jspa?threadID=135955
            period: cdk.Duration.hours(9),
        });
        const billingAlarm = new cw.Alarm(this, 'BillingCloudWatchAlarm', {
            alarmDescription: 'Upper monthly billing cost limit',
            comparisonOperator: cw.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            metric: billingAlarmMetric,
            threshold: props.monthlyThreshold,
        });
        const alarmAction = new cwa.SnsAction(billingAlarmTopic);
        billingAlarm.addAlarmAction(alarmAction);
    }
}
exports.BillingAlarm = BillingAlarm;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYmlsbGluZy1hbGFybS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImJpbGxpbmctYWxhcm0udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEscUNBQXFDO0FBQ3JDLDhDQUE4QztBQUM5Qyx1REFBdUQ7QUFDdkQsd0NBQXdDO0FBQ3hDLHNEQUFzRDs7Ozs7Ozs7O0FBeUJ0RCxNQUFhLFlBQWEsU0FBUSxHQUFHLENBQUMsU0FBUzs7OztJQUM3QyxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQXdCO1FBQ3BFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsTUFBTSxpQkFBaUIsR0FBZSxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxFQUFFLCtCQUErQixFQUFFO1lBQ3pGLFNBQVMsRUFBRSwrQkFBK0I7U0FDM0MsQ0FBQyxDQUFDO1FBRUgsaUJBQWlCLENBQUMsZUFBZSxDQUFDLElBQUksR0FBRyxDQUFDLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUU7WUFDdkUsSUFBSSxFQUFFLElBQUk7U0FDWCxDQUFDLENBQUMsQ0FBQztRQUVKLE1BQU0sa0JBQWtCLEdBQWMsSUFBSSxFQUFFLENBQUMsTUFBTSxDQUFDO1lBQ2xELFVBQVUsRUFBRSxrQkFBa0I7WUFDOUIsU0FBUyxFQUFFLGFBQWE7WUFDeEIsU0FBUyxFQUFFLFNBQVM7WUFDcEIsVUFBVSxFQUFFO2dCQUNWLFFBQVEsRUFBRSxLQUFLO2FBQ2hCO1NBQ0YsQ0FBQyxDQUFDLElBQUksQ0FBQztZQUNOLDZEQUE2RDtZQUM3RCxzREFBc0Q7WUFDdEQsaUVBQWlFO1lBQ2pFLE1BQU0sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7U0FDOUIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxZQUFZLEdBQWEsSUFBSSxFQUFFLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSx3QkFBd0IsRUFBRTtZQUMxRSxnQkFBZ0IsRUFBRSxrQ0FBa0M7WUFDcEQsa0JBQWtCLEVBQUUsRUFBRSxDQUFDLGtCQUFrQixDQUFDLGtDQUFrQztZQUM1RSxpQkFBaUIsRUFBRSxDQUFDO1lBQ3BCLE1BQU0sRUFBRSxrQkFBa0I7WUFDMUIsU0FBUyxFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7U0FDbEMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxXQUFXLEdBQWtCLElBQUksR0FBRyxDQUFDLFNBQVMsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDO1FBRXhFLFlBQVksQ0FBQyxjQUFjLENBQUMsV0FBVyxDQUFDLENBQUM7SUFDM0MsQ0FBQztDQUNGO0FBdENELG9DQXNDQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGN3IGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJztcbmltcG9ydCAqIGFzIGN3YSBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWR3YXRjaC1hY3Rpb25zJztcbmltcG9ydCAqIGFzIHNucyBmcm9tICdAYXdzLWNkay9hd3Mtc25zJztcbmltcG9ydCAqIGFzIHN1YiBmcm9tICdAYXdzLWNkay9hd3Mtc25zLXN1YnNjcmlwdGlvbnMnO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEJpbGxpbmdBbGFybVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IG1vbnRobHlUaHJlc2hvbGQ6IG51bWJlcjtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBlbWFpbDogc3RyaW5nO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBCaWxsaW5nQWxhcm0gZXh0ZW5kcyBjZGsuQ29uc3RydWN0IHtcbiAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBCaWxsaW5nQWxhcm1Qcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCBiaWxsaW5nQWxhcm1Ub3BpYzogc25zLklUb3BpYyA9IG5ldyBzbnMuVG9waWModGhpcywgJ0JpbGxpbmdBbGFybU5vdGlmaWNhdGlvblRvcGljJywge1xuICAgICAgdG9waWNOYW1lOiAnQmlsbGluZ0FsYXJtTm90aWZpY2F0aW9uVG9waWMnLFxuICAgIH0pO1xuXG4gICAgYmlsbGluZ0FsYXJtVG9waWMuYWRkU3Vic2NyaXB0aW9uKG5ldyBzdWIuRW1haWxTdWJzY3JpcHRpb24ocHJvcHMuZW1haWwsIHtcbiAgICAgIGpzb246IHRydWUsXG4gICAgfSkpO1xuXG4gICAgY29uc3QgYmlsbGluZ0FsYXJtTWV0cmljOiBjdy5NZXRyaWMgPSBuZXcgY3cuTWV0cmljKHtcbiAgICAgIG1ldHJpY05hbWU6ICdFc3RpbWF0ZWRDaGFyZ2VzJyxcbiAgICAgIG5hbWVzcGFjZTogJ0FXUy9CaWxsaW5nJyxcbiAgICAgIHN0YXRpc3RpYzogJ01heGltdW0nLFxuICAgICAgZGltZW5zaW9uczoge1xuICAgICAgICBDdXJyZW5jeTogJ1VTRCcsXG4gICAgICB9LFxuICAgIH0pLndpdGgoe1xuICAgICAgLy8gRXZhbHVhdGVzIHRoZSBtZXRyaWMgZXZlcnkgOSBob3Vycy4gVGhpcyBpcyBuZWVkZWQgYmVjYXVzZVxuICAgICAgLy8gYEVzdGltYXRlZENoYXJnZXNgIG1ldHJpY3MgaXMgdXBkYXRlZCBldmVyeSA2IGhvdXJzXG4gICAgICAvLyBTZWU6IGh0dHBzOi8vZm9ydW1zLmF3cy5hbWF6b24uY29tL3RocmVhZC5qc3BhP3RocmVhZElEPTEzNTk1NVxuICAgICAgcGVyaW9kOiBjZGsuRHVyYXRpb24uaG91cnMoOSksXG4gICAgfSk7XG5cbiAgICBjb25zdCBiaWxsaW5nQWxhcm06IGN3LkFsYXJtID0gbmV3IGN3LkFsYXJtKHRoaXMsICdCaWxsaW5nQ2xvdWRXYXRjaEFsYXJtJywge1xuICAgICAgYWxhcm1EZXNjcmlwdGlvbjogJ1VwcGVyIG1vbnRobHkgYmlsbGluZyBjb3N0IGxpbWl0JyxcbiAgICAgIGNvbXBhcmlzb25PcGVyYXRvcjogY3cuQ29tcGFyaXNvbk9wZXJhdG9yLkdSRUFURVJfVEhBTl9PUl9FUVVBTF9UT19USFJFU0hPTEQsXG4gICAgICBldmFsdWF0aW9uUGVyaW9kczogMSxcbiAgICAgIG1ldHJpYzogYmlsbGluZ0FsYXJtTWV0cmljLFxuICAgICAgdGhyZXNob2xkOiBwcm9wcy5tb250aGx5VGhyZXNob2xkLFxuICAgIH0pO1xuXG4gICAgY29uc3QgYWxhcm1BY3Rpb246IGN3YS5TbnNBY3Rpb24gPSBuZXcgY3dhLlNuc0FjdGlvbihiaWxsaW5nQWxhcm1Ub3BpYyk7XG5cbiAgICBiaWxsaW5nQWxhcm0uYWRkQWxhcm1BY3Rpb24oYWxhcm1BY3Rpb24pO1xuICB9XG59XG4iXX0=