"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BasicServiceMultiAZObservability = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const aws_elasticloadbalancingv2_1 = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const constructs_1 = require("constructs");
const AvailabilityAndLatencyAlarmsAndRules_1 = require("../alarmsandrules/AvailabilityAndLatencyAlarmsAndRules");
const AvailabilityZoneMapper_1 = require("../azmapper/AvailabilityZoneMapper");
const BasicServiceDashboard_1 = require("../dashboards/BasicServiceDashboard");
const AvailabilityAndLatencyMetrics_1 = require("../metrics/AvailabilityAndLatencyMetrics");
const OutlierDetectionFunction_1 = require("../outlier-detection/OutlierDetectionFunction");
const OutlierDetectionAlgorithm_1 = require("../utilities/OutlierDetectionAlgorithm");
const StackWithDynamicSource_1 = require("../utilities/StackWithDynamicSource");
/**
 * Basic observability for a service using metrics from
 * ALBs and NAT Gateways
 */
class BasicServiceMultiAZObservability extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Initialize class properties
        this.serviceName = props.serviceName;
        this.applicationLoadBalancers = props.applicationLoadBalancers;
        this.natGateways = props.natGateways;
        this._natGWZonalIsolatedImpactAlarms = {};
        this._albZonalIsolatedImpactAlarms = {};
        this.aggregateZonalIsolatedImpactAlarms = {};
        this._packetDropsPerZone = {};
        this._faultsPerZone = {};
        let outlierThreshold;
        if (!props.outlierThreshold) {
            switch (props.outlierDetectionAlgorithm) {
                case OutlierDetectionAlgorithm_1.OutlierDetectionAlgorithm.CHI_SQUARED:
                    outlierThreshold = 0.05;
                    break;
                case OutlierDetectionAlgorithm_1.OutlierDetectionAlgorithm.IQR:
                    outlierThreshold = 1.5;
                    break;
                case OutlierDetectionAlgorithm_1.OutlierDetectionAlgorithm.MAD:
                    outlierThreshold = 3;
                    break;
                case OutlierDetectionAlgorithm_1.OutlierDetectionAlgorithm.STATIC:
                    outlierThreshold = 0.7;
                    break;
                case OutlierDetectionAlgorithm_1.OutlierDetectionAlgorithm.Z_SCORE:
                    outlierThreshold = 2;
                    break;
            }
        }
        else {
            outlierThreshold = props.outlierThreshold;
        }
        // Create the AZ mapper resource to translate AZ names to ids
        this.azMapper = new AvailabilityZoneMapper_1.AvailabilityZoneMapper(this, "AvailabilityZoneMapper");
        if (props.outlierDetectionAlgorithm != OutlierDetectionAlgorithm_1.OutlierDetectionAlgorithm.STATIC) {
            let outlierDetectionStack = new StackWithDynamicSource_1.StackWithDynamicSource(this, "OutlierDetectionStack", {
                assetsBucketsParameterName: props.assetsBucketParameterName,
                assetsBucketPrefixParameterName: props.assetsBucketPrefixParameterName,
            });
            this.outlierDetectionFunction = new OutlierDetectionFunction_1.OutlierDetectionFunction(outlierDetectionStack, "OutlierDetectionFunction", {}).function;
        }
        // Create metrics and alarms for just load balancers if they were provided
        if (this.applicationLoadBalancers !== undefined &&
            this.applicationLoadBalancers != null) {
            this.doAlbMetrics(props, outlierThreshold);
        }
        // Create NAT Gateway metrics and alarms
        if (this.natGateways !== undefined && this.natGateways != null) {
            this.doNatGatewayMetrics(props, outlierThreshold);
        }
        let counter = 1;
        // Go through the ALB zonal isolated impact alarms and see if there is a NAT GW
        // isolated impact alarm for the same AZ ID, if so, create a composite alarm with both
        // otherwise create a composite alarm with just the ALB
        Object.keys(this._albZonalIsolatedImpactAlarms).forEach((az) => {
            let tmp = [];
            tmp.push(this._albZonalIsolatedImpactAlarms[az]);
            if (this._natGWZonalIsolatedImpactAlarms[az] !== undefined &&
                this._natGWZonalIsolatedImpactAlarms[az] != null) {
                tmp.push(this._natGWZonalIsolatedImpactAlarms[az]);
            }
            let availabilityZoneId = this.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(az.substring(az.length - 1));
            this.aggregateZonalIsolatedImpactAlarms[az] = new aws_cloudwatch_1.CompositeAlarm(this, "AZ" + counter++ + "AggregateIsolatedImpactAlarm", {
                compositeAlarmName: availabilityZoneId + "-aggregate-isolated-impact",
                alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...tmp),
                actionsEnabled: false,
            });
        });
        // In case there were AZs with only a NAT GW and no ALB, create a composite alarm
        // for the NAT GW metrics
        Object.keys(this._natGWZonalIsolatedImpactAlarms).forEach((az) => {
            // If we don't yet have an isolated impact alarm for this AZ, proceed
            if (this.aggregateZonalIsolatedImpactAlarms[az] === undefined ||
                this.aggregateZonalIsolatedImpactAlarms[az] == null) {
                let tmp = [];
                tmp.push(this._natGWZonalIsolatedImpactAlarms[az]);
                if (this._albZonalIsolatedImpactAlarms[az] !== undefined &&
                    this.albZonalIsolatedImpactAlarms != null) {
                    tmp.push(this.albZonalIsolatedImpactAlarms[az]);
                }
                let availabilityZoneId = this.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(az.substring(az.length - 1));
                this.aggregateZonalIsolatedImpactAlarms[az] = new aws_cloudwatch_1.CompositeAlarm(this, "AZ" + counter++ + "AggregateIsolatedImpactAlarm", {
                    compositeAlarmName: availabilityZoneId + "-aggregate-isolated-impact",
                    alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...tmp),
                    actionsEnabled: false,
                });
            }
        });
        this.albZonalIsolatedImpactAlarms = this._albZonalIsolatedImpactAlarms;
        this.natGWZonalIsolatedImpactAlarms = this._natGWZonalIsolatedImpactAlarms;
        // Should we create the dashboard
        if (props.createDashboard == true) {
            this.dashboard = new BasicServiceDashboard_1.BasicServiceDashboard(this, "BasicServiceDashboard", {
                serviceName: props.serviceName.toLowerCase(),
                zonalAggregateIsolatedImpactAlarms: this.aggregateZonalIsolatedImpactAlarms,
                zonalLoadBalancerIsolatedImpactAlarms: this.albZonalIsolatedImpactAlarms,
                zonalNatGatewayIsolatedImpactAlarms: this.natGWZonalIsolatedImpactAlarms,
                interval: props.interval,
                zonalLoadBalancerFaultRateMetrics: this._faultsPerZone,
                zonalNatGatewayPacketDropMetrics: this._packetDropsPerZone,
            }).dashboard;
        }
    }
    doAlbMetrics(props, outlierThreshold) {
        // Collect total fault count metrics per AZ
        let albZoneFaultCountMetrics = {};
        // Create fault rate alarms per AZ indicating at least 1 ALB
        // in the AZ saw a fault rate that exceeded the threshold
        let faultRatePercentageAlarms = {};
        let keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar("");
        // Iterate each ALB
        this.applicationLoadBalancers.forEach((alb) => {
            // Iterate each AZ in the VPC
            alb.vpc?.availabilityZones.forEach((az, index) => {
                let azLetter = az.substring(az.length - 1);
                let availabilityZoneId = this.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
                faultRatePercentageAlarms[azLetter] = [];
                // 5xx responses from targets
                let target5xx = alb.metrics.httpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_5XX_COUNT, {
                    dimensionsMap: {
                        AvailabilityZone: az,
                        LoadBalancer: alb
                            .loadBalancerFullName,
                    },
                    label: availabilityZoneId,
                    period: props.period,
                });
                // 5xx responses from ELB
                let elb5xx = alb.metrics.httpCodeElb(aws_elasticloadbalancingv2_1.HttpCodeElb.ELB_5XX_COUNT, {
                    dimensionsMap: {
                        AvailabilityZone: az,
                        LoadBalancer: alb
                            .loadBalancerFullName,
                    },
                    label: availabilityZoneId,
                    period: props.period,
                });
                // 2xx responses from targets
                let target2xx = alb.metrics.httpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_2XX_COUNT, {
                    dimensionsMap: {
                        AvailabilityZone: az,
                        LoadBalancer: alb
                            .loadBalancerFullName,
                    },
                    label: availabilityZoneId,
                    period: props.period,
                });
                // 3xx responses from targets
                let target3xx = alb.metrics.httpCodeTarget(aws_elasticloadbalancingv2_1.HttpCodeTarget.TARGET_3XX_COUNT, {
                    dimensionsMap: {
                        AvailabilityZone: az,
                        LoadBalancer: alb
                            .loadBalancerFullName,
                    },
                    label: availabilityZoneId,
                    period: props.period,
                });
                // 3xx responess from ELB
                let elb3xx = alb.metrics.httpCodeElb(aws_elasticloadbalancingv2_1.HttpCodeElb.ELB_3XX_COUNT, {
                    dimensionsMap: {
                        AvailabilityZone: az,
                        LoadBalancer: alb
                            .loadBalancerFullName,
                    },
                    label: availabilityZoneId,
                    period: props.period,
                });
                // Create metrics for total fault count from this ALB
                let usingMetrics = {};
                usingMetrics[`${keyPrefix}1`] = target5xx;
                usingMetrics[`${keyPrefix}2`] = elb5xx;
                if (albZoneFaultCountMetrics[azLetter] === undefined ||
                    albZoneFaultCountMetrics[azLetter] == null) {
                    albZoneFaultCountMetrics[azLetter] = [];
                }
                albZoneFaultCountMetrics[azLetter].push(new aws_cloudwatch_1.MathExpression({
                    expression: `(${keyPrefix}1 + ${keyPrefix}2)`,
                    usingMetrics: usingMetrics,
                    label: availabilityZoneId + " " + alb.loadBalancerArn + " fault count",
                    period: props.period,
                }));
                // Create metrics to calculate fault rate for this ALB
                usingMetrics = {};
                usingMetrics[`${keyPrefix}1`] = target2xx;
                usingMetrics[`${keyPrefix}2`] = target3xx;
                usingMetrics[`${keyPrefix}3`] = elb3xx;
                usingMetrics[`${keyPrefix}4`] = target5xx;
                usingMetrics[`${keyPrefix}5`] = elb5xx;
                // The ALB fault rate
                let faultRate = new aws_cloudwatch_1.MathExpression({
                    expression: `((${keyPrefix}4+${keyPrefix}5)/(${keyPrefix}1+${keyPrefix}2+${keyPrefix}3+${keyPrefix}4+${keyPrefix}5)) * 100`,
                    usingMetrics: usingMetrics,
                    label: availabilityZoneId + " " + alb.loadBalancerArn + " fault rate",
                    period: props.period,
                });
                let threshold = props.faultCountPercentageThreshold ?? 5;
                // Create a fault rate alarm for the ALB
                let faultRateAlarm = new aws_cloudwatch_1.Alarm(this, "AZ" + index + keyPrefix + "FaultRatePercentageAlarm", {
                    alarmName: availabilityZoneId + "-" + alb.loadBalancerArn + "-fault-rate",
                    actionsEnabled: false,
                    metric: faultRate,
                    evaluationPeriods: props.evaluationPeriods,
                    datapointsToAlarm: props.datapointsToAlarm,
                    threshold: threshold,
                    comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
                });
                // Add this ALB's fault rate alarm
                faultRatePercentageAlarms[azLetter].push(faultRateAlarm);
                // Get next unique key
                keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar(keyPrefix);
            });
        });
        // Iterate AZs for the ALB fault count metrics
        Object.keys(albZoneFaultCountMetrics).forEach((azLetter) => {
            let availabilityZoneId = this.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
            keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar(keyPrefix);
            let counter = 1;
            let usingMetrics = {};
            // Add each ALB's fault count metrics to the dictionary
            albZoneFaultCountMetrics[azLetter].forEach((metric) => {
                usingMetrics[`${keyPrefix}${counter++}`] = metric;
            });
            // Sum the total faults for the availability zone across all ALBs
            let totalFaultsPerZone = new aws_cloudwatch_1.MathExpression({
                expression: Object.keys(usingMetrics).join("+"),
                usingMetrics: usingMetrics,
                label: availabilityZoneId + " fault count",
                period: props.period,
            });
            keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar(keyPrefix);
            counter = 1;
            // Assign the total faults per zone to the dictionary
            this._faultsPerZone[azLetter] = totalFaultsPerZone;
        });
        keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar(keyPrefix);
        let tmp = {};
        Object.keys(this._faultsPerZone).forEach((azLetter, index) => {
            tmp[`${keyPrefix}${index}`] = this._faultsPerZone[azLetter];
        });
        // Calculate the total faults in the region by adding all AZs together
        let totalFaults = new aws_cloudwatch_1.MathExpression({
            expression: Object.keys(tmp).join("+"),
            usingMetrics: tmp,
            label: aws_cdk_lib_1.Fn.sub("${AWS::Region} fault count"),
            period: props.period,
        });
        if (props.outlierDetectionAlgorithm == OutlierDetectionAlgorithm_1.OutlierDetectionAlgorithm.STATIC) {
            // Finally, iterate back through each AZ
            Object.keys(this._faultsPerZone).forEach((azLetter, index) => {
                keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar(keyPrefix);
                let availabilityZoneId = this.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
                // Determine if AZ is an outlier for faults by exceeding
                // a static threshold
                let outlierMetrics;
                // These metrics will give the percent of faults for the AZ
                let usingMetrics = {};
                usingMetrics[`${keyPrefix}1`] = this._faultsPerZone[azLetter];
                usingMetrics[`${keyPrefix}2`] = totalFaults;
                outlierMetrics = new aws_cloudwatch_1.MathExpression({
                    expression: `${keyPrefix}1 / ${keyPrefix}2`,
                    usingMetrics: usingMetrics,
                });
                let azIsOutlierForFaults = new aws_cloudwatch_1.Alarm(this, "AZ" + index + "FaultCountOutlierAlarm", {
                    alarmName: availabilityZoneId + "-fault-count-outlier",
                    metric: outlierMetrics,
                    threshold: outlierThreshold,
                    evaluationPeriods: props.evaluationPeriods,
                    datapointsToAlarm: props.datapointsToAlarm,
                    actionsEnabled: false,
                    treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE,
                    comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                });
                // Create isolated AZ impact alarms by determining
                // if the AZ is an outlier for fault count and at least
                // one ALB exceeds the fault rate threshold provided
                this._albZonalIsolatedImpactAlarms[azLetter] = new aws_cloudwatch_1.CompositeAlarm(this, "AZ" + index + "IsolatedFaultCountImpact", {
                    compositeAlarmName: availabilityZoneId + "-isolated-fault-count-impact",
                    alarmRule: aws_cloudwatch_1.AlarmRule.allOf(azIsOutlierForFaults, aws_cloudwatch_1.AlarmRule.anyOf(...faultRatePercentageAlarms[azLetter])),
                });
            });
        }
        else {
            let allAZs = Array.from(new Set(this.applicationLoadBalancers.flatMap((x) => {
                return x.vpc.availabilityZones;
            })));
            allAZs.forEach((az, index) => {
                let azLetter = az.substring(az.length - 1);
                let availabilityZoneId = this.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
                let azIsOutlierForFaults = AvailabilityAndLatencyAlarmsAndRules_1.AvailabilityAndLatencyAlarmsAndRules.createZonalFaultRateOutlierAlarmForAlb(this, props.applicationLoadBalancers, availabilityZoneId, outlierThreshold, this.outlierDetectionFunction, props.outlierDetectionAlgorithm, this.azMapper, index, props.evaluationPeriods, props.datapointsToAlarm, "");
                // In addition to being an outlier for fault count, make sure
                // the fault count is substantial enough to trigger the alarm
                // by making sure at least 1 ALB sees packet loss that exceeds the threshold
                let azIsOutlierAndSeesImpact = new aws_cloudwatch_1.CompositeAlarm(this, "AZ" + index + "ALBIsolatedImpact", {
                    compositeAlarmName: availabilityZoneId + "-isolated-fault-count-impact",
                    alarmRule: aws_cloudwatch_1.AlarmRule.allOf(azIsOutlierForFaults, aws_cloudwatch_1.AlarmRule.anyOf(...faultRatePercentageAlarms[azLetter])),
                });
                // Record these so they can be used in dashboard or for combination
                // with AZ
                this._albZonalIsolatedImpactAlarms[azLetter] = azIsOutlierAndSeesImpact;
            });
        }
    }
    doNatGatewayMetrics(props, outlierThreshold) {
        let keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar("");
        // Collect alarms for packet drops exceeding a threshold per NAT GW
        let packetDropPercentageAlarms = {};
        // For each AZ, create metrics for each NAT GW
        Object.entries(this.natGateways).forEach((entry, index) => {
            // The number of packet drops for each NAT GW in the AZ
            let packetDropMetricsForAZ = {};
            let az = entry[0];
            let azLetter = az.substring(az.length - 1);
            let availabilityZoneId = this.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
            packetDropPercentageAlarms[azLetter] = [];
            // Iterate through each NAT GW in the current AZ
            entry[1].forEach((natgw) => {
                // Calculate packet drops
                let packetDropCount = new aws_cloudwatch_1.Metric({
                    metricName: "PacketsDropCount",
                    namespace: "AWS/NATGateway",
                    statistic: "Sum",
                    unit: aws_cloudwatch_1.Unit.COUNT,
                    label: availabilityZoneId + " packet drops",
                    dimensionsMap: {
                        NatGatewayId: natgw.attrNatGatewayId,
                    },
                    period: props.period,
                });
                // Calculate packets in from source
                let packetsInFromSourceCount = new aws_cloudwatch_1.Metric({
                    metricName: "PacketsInFromSource",
                    namespace: "AWS/NATGateway",
                    statistic: "Sum",
                    unit: aws_cloudwatch_1.Unit.COUNT,
                    label: availabilityZoneId + " packets in from source",
                    dimensionsMap: {
                        NatGatewayId: natgw.attrNatGatewayId,
                    },
                    period: props.period,
                });
                // Calculate packets in from destination
                let packetsInFromDestinationCount = new aws_cloudwatch_1.Metric({
                    metricName: "PacketsInFromDestination",
                    namespace: "AWS/NATGateway",
                    statistic: "Sum",
                    unit: aws_cloudwatch_1.Unit.COUNT,
                    label: availabilityZoneId + " packets in from destination",
                    dimensionsMap: {
                        NatGatewayId: natgw.attrNatGatewayId,
                    },
                    period: props.period,
                });
                let usingMetrics = {};
                usingMetrics[`${keyPrefix}1`] = packetDropCount;
                usingMetrics[`${keyPrefix}2`] = packetsInFromSourceCount;
                usingMetrics[`${keyPrefix}3`] = packetsInFromDestinationCount;
                // Calculate a percentage of dropped packets for the NAT GW
                let packetDropPercentage = new aws_cloudwatch_1.MathExpression({
                    expression: `(${keyPrefix}1 / (${keyPrefix}2 + ${keyPrefix}3)) * 100`,
                    usingMetrics: usingMetrics,
                    label: availabilityZoneId + " packet drop percentage",
                    period: props.period,
                });
                let threshold = props.packetLossImpactPercentageThreshold ?? 0.01;
                // Create an alarm for this NAT GW if packet drops exceed the specified threshold
                let packetDropImpactAlarm = new aws_cloudwatch_1.Alarm(this, "AZ" + (index + 1) + "PacketDropImpactAlarm", {
                    alarmName: availabilityZoneId +
                        "-" +
                        natgw.attrNatGatewayId +
                        "-packet-drop-impact",
                    actionsEnabled: false,
                    metric: packetDropPercentage,
                    threshold: threshold,
                    comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
                    evaluationPeriods: props.evaluationPeriods,
                    datapointsToAlarm: props.datapointsToAlarm,
                });
                // Collect all of the packet drop impact alarms for each
                // NAT GW in this AZ, need to know at least 1 sees substantial
                // enough impact to consider the AZ as impaired
                packetDropPercentageAlarms[azLetter].push(packetDropImpactAlarm);
                // Collect the packet drop metrics for this AZ so we can
                // add them all together and count total packet drops
                // for all NAT GWs in the AZ
                packetDropMetricsForAZ[`m${index}`] = packetDropCount;
            });
            // Create a metric that adds up all packets drops from each
            // NAT GW in the AZ
            let packetDropsInThisAZ = new aws_cloudwatch_1.MathExpression({
                expression: Object.keys(packetDropMetricsForAZ).join("+"),
                usingMetrics: packetDropMetricsForAZ,
                label: availabilityZoneId + " dropped packets",
                period: props.period,
            });
            // Record these so we can add them up
            // and get a total amount of packet drops
            // in the region across all AZs
            this._packetDropsPerZone[azLetter] = packetDropsInThisAZ;
        });
        keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar(keyPrefix);
        let tmp = {};
        Object.keys(this._packetDropsPerZone).forEach((azLetter, index) => {
            tmp[`${keyPrefix}${index}`] = this._packetDropsPerZone[azLetter];
        });
        // Calculate total packet drops for the region
        let totalPacketDrops = new aws_cloudwatch_1.MathExpression({
            expression: Object.keys(tmp).join("+"),
            usingMetrics: tmp,
            label: aws_cdk_lib_1.Fn.ref("AWS::Region") + " dropped packets",
            period: props.period,
        });
        if (props.outlierDetectionAlgorithm == OutlierDetectionAlgorithm_1.OutlierDetectionAlgorithm.STATIC) {
            // Create outlier detection alarms by comparing packet
            // drops in one AZ versus total packet drops in the region
            Object.keys(this._packetDropsPerZone).forEach((azLetter, index) => {
                let azIsOutlierForPacketDrops;
                keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar(keyPrefix);
                let availabilityZoneId = this.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
                // Determine if AZ is an outlier for faults by exceeding
                // a static threshold
                let outlierMetrics;
                // These metrics will give the percent of faults for the AZ
                let usingMetrics = {};
                usingMetrics[`${keyPrefix}1`] = this._packetDropsPerZone[azLetter];
                usingMetrics[`${keyPrefix}2`] = totalPacketDrops;
                outlierMetrics = new aws_cloudwatch_1.MathExpression({
                    expression: `(${keyPrefix}1 / ${keyPrefix}2) * 100`,
                    usingMetrics: usingMetrics,
                    label: availabilityZoneId + " percentage of dropped packets",
                });
                azIsOutlierForPacketDrops = new aws_cloudwatch_1.Alarm(this, "AZ" + (index + 1) + "NATGWDroppedPacketsOutlierAlarm", {
                    metric: outlierMetrics,
                    alarmName: availabilityZoneId + "-dropped-packets-outlier",
                    evaluationPeriods: props.evaluationPeriods,
                    datapointsToAlarm: props.datapointsToAlarm,
                    threshold: outlierThreshold,
                    actionsEnabled: false,
                    treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE,
                    comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                });
                // In addition to being an outlier for packet drops, make sure
                // the packet loss is substantial enough to trigger the alarm
                // by making sure at least 1 NAT GW sees packet loss more than 0.01%
                let azIsOutlierAndSeesImpact = new aws_cloudwatch_1.CompositeAlarm(this, "AZ" + index + "NATGWIsolatedImpact", {
                    compositeAlarmName: availabilityZoneId + "-isolated-natgw-impact",
                    alarmRule: aws_cloudwatch_1.AlarmRule.allOf(azIsOutlierForPacketDrops, aws_cloudwatch_1.AlarmRule.anyOf(...packetDropPercentageAlarms[azLetter])),
                });
                // Record these so they can be used in dashboard or for combination
                // with AZ
                this._natGWZonalIsolatedImpactAlarms[azLetter] =
                    azIsOutlierAndSeesImpact;
            });
        }
        else {
            Object.keys(props.natGateways).forEach((az, index) => {
                let azLetter = az.substring(az.length - 1);
                let availabilityZoneId = this.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(azLetter);
                // Iterate all NAT GWs in this AZ
                let azIsOutlierForPacketDrops = AvailabilityAndLatencyAlarmsAndRules_1.AvailabilityAndLatencyAlarmsAndRules.createZonalFaultRateOutlierAlarmForNatGW(this, this.natGateways, availabilityZoneId, outlierThreshold, this.outlierDetectionFunction, props.outlierDetectionAlgorithm, this.azMapper, index, props.evaluationPeriods, props.datapointsToAlarm, "");
                // In addition to being an outlier for packet drops, make sure
                // the packet loss is substantial enough to trigger the alarm
                // by making sure at least 1 NAT GW sees packet loss more than 0.01%
                let azIsOutlierAndSeesImpact = new aws_cloudwatch_1.CompositeAlarm(this, "AZ" + index + "NATGWIsolatedImpact", {
                    compositeAlarmName: availabilityZoneId + "-isolated-natgw-impact",
                    alarmRule: aws_cloudwatch_1.AlarmRule.allOf(azIsOutlierForPacketDrops, aws_cloudwatch_1.AlarmRule.anyOf(...packetDropPercentageAlarms[azLetter])),
                });
                // Record these so they can be used in dashboard or for combination
                // with AZ
                this._natGWZonalIsolatedImpactAlarms[azLetter] =
                    azIsOutlierAndSeesImpact;
            });
        }
    }
}
exports.BasicServiceMultiAZObservability = BasicServiceMultiAZObservability;
_a = JSII_RTTI_SYMBOL_1;
BasicServiceMultiAZObservability[_a] = { fqn: "multi-az-observability.BasicServiceMultiAZObservability", version: "0.0.1-alpha.10" };
//# sourceMappingURL=data:application/json;base64,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