"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OutlierDetectionFunction = void 0;
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
class OutlierDetectionFunction extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        let xrayManagedPolicy = new aws_iam_1.ManagedPolicy(this, "xrayManagedPolicy", {
            path: "/outlier-detection/",
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords",
                        "xray:GetSamplingRules",
                        "xray:GetSamplingTargets",
                        "xray:GetSamplingStatisticSummaries",
                    ],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: ["*"],
                }),
            ],
        });
        let cwManagedPolicy = new aws_iam_1.ManagedPolicy(this, "CWManagedPolicy", {
            path: "/outlier-detection/",
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: ["cloudwatch:GetMetricData", "cloduwatch:PutMetricData"],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: ["*"],
                }),
            ],
        });
        let executionRole = new aws_iam_1.Role(this, "executionRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            path: "/outlier-detection/",
            managedPolicies: [xrayManagedPolicy, cwManagedPolicy],
        });
        let sciPyLayer = new aws_lambda_1.LayerVersion(this, "SciPyLayer", {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "src/scipy-layer.zip")),
            compatibleArchitectures: [aws_lambda_1.Architecture.ARM_64],
            compatibleRuntimes: [aws_lambda_1.Runtime.PYTHON_3_12],
        });
        let monitoringLayer = new aws_lambda_1.LayerVersion(this, "MonitoringLayer", {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "../monitoring/src/monitoring-layer.zip")),
            compatibleArchitectures: [aws_lambda_1.Architecture.ARM_64],
            compatibleRuntimes: [aws_lambda_1.Runtime.PYTHON_3_12],
        });
        if (props.vpc !== undefined && props.vpc != null) {
            let sg = new aws_ec2_1.SecurityGroup(this, "OutlierDetectionSecurityGroup", {
                description: "Allow outlier detection function to communicate with CW",
                vpc: props.vpc,
                allowAllOutbound: true,
            });
            this.function = new aws_lambda_1.Function(this, "OutlierDetection", {
                runtime: aws_lambda_1.Runtime.PYTHON_3_12,
                code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "src/outlier-detection.zip")),
                handler: "index.handler",
                role: executionRole,
                architecture: aws_lambda_1.Architecture.ARM_64,
                tracing: aws_lambda_1.Tracing.ACTIVE,
                timeout: aws_cdk_lib_1.Duration.seconds(5),
                memorySize: 512,
                layers: [sciPyLayer, monitoringLayer],
                environment: {
                    REGION: aws_cdk_lib_1.Fn.ref("AWS::Region"),
                    PARTITION: aws_cdk_lib_1.Fn.ref("AWS::Partition"),
                },
                vpc: props.vpc,
                securityGroups: [sg],
                vpcSubnets: props.subnetSelection,
            });
        }
        else {
            this.function = new aws_lambda_1.Function(this, "OutlierDetection", {
                runtime: aws_lambda_1.Runtime.PYTHON_3_12,
                code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "src/outlier-detection.zip")),
                handler: "index.handler",
                role: executionRole,
                architecture: aws_lambda_1.Architecture.ARM_64,
                tracing: aws_lambda_1.Tracing.ACTIVE,
                timeout: aws_cdk_lib_1.Duration.seconds(5),
                memorySize: 512,
                layers: [sciPyLayer, monitoringLayer],
                environment: {
                    REGION: aws_cdk_lib_1.Fn.ref("AWS::Region"),
                    PARTITION: aws_cdk_lib_1.Fn.ref("AWS::Partition"),
                },
            });
        }
        aws_cdk_lib_1.Tags.of(this.function).add("cloudwatch:datasource", "custom", {
            includeResourceTypes: ["AWS::Lambda::Function"],
        });
        this.function.addPermission("invokePermission", {
            action: "lambda:InvokeFunction",
            principal: new aws_iam_1.ServicePrincipal("lambda.datasource.cloudwatch.amazonaws.com"),
            sourceAccount: aws_cdk_lib_1.Fn.ref("AWS::AccountId"),
        });
        this.logGroup = new aws_logs_1.LogGroup(this, "logGroup", {
            logGroupName: `/aws/lambda/${this.function.functionName}`,
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        new aws_iam_1.ManagedPolicy(this, "cwLogsManagedPolicy", {
            path: "/outlier-detection/",
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: ["logs:CreateLogStream", "logs:PutLogEvents"],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: [this.logGroup.logGroupArn],
                }),
            ],
            roles: [executionRole],
        });
    }
}
exports.OutlierDetectionFunction = OutlierDetectionFunction;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiT3V0bGllckRldGVjdGlvbkZ1bmN0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL291dGxpZXItZGV0ZWN0aW9uL091dGxpZXJEZXRlY3Rpb25GdW5jdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSw2QkFBNkI7QUFDN0IsNkNBQWdFO0FBQ2hFLGlEQUFvRTtBQUNwRSxpREFRNkI7QUFDN0IsdURBU2dDO0FBQ2hDLG1EQUEwRTtBQUMxRSwyQ0FBdUM7QUFJdkMsTUFBYSx3QkFDWCxTQUFRLHNCQUFTO0lBYWpCLFlBQ0UsS0FBZ0IsRUFDaEIsRUFBVSxFQUNWLEtBQW9DO1FBRXBDLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxpQkFBaUIsR0FBbUIsSUFBSSx1QkFBYSxDQUN2RCxJQUFJLEVBQ0osbUJBQW1CLEVBQ25CO1lBQ0UsSUFBSSxFQUFFLHFCQUFxQjtZQUMzQixVQUFVLEVBQUU7Z0JBQ1YsSUFBSSx5QkFBZSxDQUFDO29CQUNsQixPQUFPLEVBQUU7d0JBQ1AsdUJBQXVCO3dCQUN2QiwwQkFBMEI7d0JBQzFCLHVCQUF1Qjt3QkFDdkIseUJBQXlCO3dCQUN6QixvQ0FBb0M7cUJBQ3JDO29CQUNELE1BQU0sRUFBRSxnQkFBTSxDQUFDLEtBQUs7b0JBQ3BCLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztpQkFDakIsQ0FBQzthQUNIO1NBQ0YsQ0FDRixDQUFDO1FBQ0YsSUFBSSxlQUFlLEdBQUcsSUFBSSx1QkFBYSxDQUFDLElBQUksRUFBRSxpQkFBaUIsRUFBRTtZQUMvRCxJQUFJLEVBQUUscUJBQXFCO1lBQzNCLFVBQVUsRUFBRTtnQkFDVixJQUFJLHlCQUFlLENBQUM7b0JBQ2xCLE9BQU8sRUFBRSxDQUFDLDBCQUEwQixFQUFFLDBCQUEwQixDQUFDO29CQUNqRSxNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxLQUFLO29CQUNwQixTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7aUJBQ2pCLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksYUFBYSxHQUFVLElBQUksY0FBSSxDQUFDLElBQUksRUFBRSxlQUFlLEVBQUU7WUFDekQsU0FBUyxFQUFFLElBQUksMEJBQWdCLENBQUMsc0JBQXNCLENBQUM7WUFDdkQsSUFBSSxFQUFFLHFCQUFxQjtZQUMzQixlQUFlLEVBQUUsQ0FBQyxpQkFBaUIsRUFBRSxlQUFlLENBQUM7U0FDdEQsQ0FBQyxDQUFDO1FBRUgsSUFBSSxVQUFVLEdBQWtCLElBQUkseUJBQVksQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQ25FLElBQUksRUFBRSxpQkFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxxQkFBcUIsQ0FBQyxDQUFDO1lBQ2pFLHVCQUF1QixFQUFFLENBQUMseUJBQVksQ0FBQyxNQUFNLENBQUM7WUFDOUMsa0JBQWtCLEVBQUUsQ0FBQyxvQkFBTyxDQUFDLFdBQVcsQ0FBQztTQUMxQyxDQUFDLENBQUM7UUFFSCxJQUFJLGVBQWUsR0FBa0IsSUFBSSx5QkFBWSxDQUNuRCxJQUFJLEVBQ0osaUJBQWlCLEVBQ2pCO1lBQ0UsSUFBSSxFQUFFLGlCQUFJLENBQUMsU0FBUyxDQUNsQixJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSx3Q0FBd0MsQ0FBQyxDQUMvRDtZQUNELHVCQUF1QixFQUFFLENBQUMseUJBQVksQ0FBQyxNQUFNLENBQUM7WUFDOUMsa0JBQWtCLEVBQUUsQ0FBQyxvQkFBTyxDQUFDLFdBQVcsQ0FBQztTQUMxQyxDQUNGLENBQUM7UUFFRixJQUFJLEtBQUssQ0FBQyxHQUFHLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxHQUFHLElBQUksSUFBSSxFQUFFLENBQUM7WUFDakQsSUFBSSxFQUFFLEdBQW1CLElBQUksdUJBQWEsQ0FDeEMsSUFBSSxFQUNKLCtCQUErQixFQUMvQjtnQkFDRSxXQUFXLEVBQ1QseURBQXlEO2dCQUMzRCxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7Z0JBQ2QsZ0JBQWdCLEVBQUUsSUFBSTthQUN2QixDQUNGLENBQUM7WUFFRixJQUFJLENBQUMsUUFBUSxHQUFHLElBQUkscUJBQVEsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7Z0JBQ3JELE9BQU8sRUFBRSxvQkFBTyxDQUFDLFdBQVc7Z0JBQzVCLElBQUksRUFBRSxpQkFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSwyQkFBMkIsQ0FBQyxDQUFDO2dCQUN2RSxPQUFPLEVBQUUsZUFBZTtnQkFDeEIsSUFBSSxFQUFFLGFBQWE7Z0JBQ25CLFlBQVksRUFBRSx5QkFBWSxDQUFDLE1BQU07Z0JBQ2pDLE9BQU8sRUFBRSxvQkFBTyxDQUFDLE1BQU07Z0JBQ3ZCLE9BQU8sRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7Z0JBQzVCLFVBQVUsRUFBRSxHQUFHO2dCQUNmLE1BQU0sRUFBRSxDQUFDLFVBQVUsRUFBRSxlQUFlLENBQUM7Z0JBQ3JDLFdBQVcsRUFBRTtvQkFDWCxNQUFNLEVBQUUsZ0JBQUUsQ0FBQyxHQUFHLENBQUMsYUFBYSxDQUFDO29CQUM3QixTQUFTLEVBQUUsZ0JBQUUsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUM7aUJBQ3BDO2dCQUNELEdBQUcsRUFBRSxLQUFLLENBQUMsR0FBRztnQkFDZCxjQUFjLEVBQUUsQ0FBQyxFQUFFLENBQUM7Z0JBQ3BCLFVBQVUsRUFBRSxLQUFLLENBQUMsZUFBZTthQUNsQyxDQUFDLENBQUM7UUFDTCxDQUFDO2FBQU0sQ0FBQztZQUNOLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxxQkFBUSxDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtnQkFDckQsT0FBTyxFQUFFLG9CQUFPLENBQUMsV0FBVztnQkFDNUIsSUFBSSxFQUFFLGlCQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLDJCQUEyQixDQUFDLENBQUM7Z0JBQ3ZFLE9BQU8sRUFBRSxlQUFlO2dCQUN4QixJQUFJLEVBQUUsYUFBYTtnQkFDbkIsWUFBWSxFQUFFLHlCQUFZLENBQUMsTUFBTTtnQkFDakMsT0FBTyxFQUFFLG9CQUFPLENBQUMsTUFBTTtnQkFDdkIsT0FBTyxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztnQkFDNUIsVUFBVSxFQUFFLEdBQUc7Z0JBQ2YsTUFBTSxFQUFFLENBQUMsVUFBVSxFQUFFLGVBQWUsQ0FBQztnQkFDckMsV0FBVyxFQUFFO29CQUNYLE1BQU0sRUFBRSxnQkFBRSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUM7b0JBQzdCLFNBQVMsRUFBRSxnQkFBRSxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQztpQkFDcEM7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDO1FBRUQsa0JBQUksQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLEdBQUcsQ0FBQyx1QkFBdUIsRUFBRSxRQUFRLEVBQUU7WUFDNUQsb0JBQW9CLEVBQUUsQ0FBQyx1QkFBdUIsQ0FBQztTQUNoRCxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsUUFBUSxDQUFDLGFBQWEsQ0FBQyxrQkFBa0IsRUFBRTtZQUM5QyxNQUFNLEVBQUUsdUJBQXVCO1lBQy9CLFNBQVMsRUFBRSxJQUFJLDBCQUFnQixDQUM3Qiw0Q0FBNEMsQ0FDN0M7WUFDRCxhQUFhLEVBQUUsZ0JBQUUsQ0FBQyxHQUFHLENBQUMsZ0JBQWdCLENBQUM7U0FDeEMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLFFBQVEsR0FBRyxJQUFJLG1CQUFRLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUM3QyxZQUFZLEVBQUUsZUFBZSxJQUFJLENBQUMsUUFBUSxDQUFDLFlBQVksRUFBRTtZQUN6RCxTQUFTLEVBQUUsd0JBQWEsQ0FBQyxRQUFRO1lBQ2pDLGFBQWEsRUFBRSwyQkFBYSxDQUFDLE9BQU87U0FDckMsQ0FBQyxDQUFDO1FBRUgsSUFBSSx1QkFBYSxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRTtZQUM3QyxJQUFJLEVBQUUscUJBQXFCO1lBQzNCLFVBQVUsRUFBRTtnQkFDVixJQUFJLHlCQUFlLENBQUM7b0JBQ2xCLE9BQU8sRUFBRSxDQUFDLHNCQUFzQixFQUFFLG1CQUFtQixDQUFDO29CQUN0RCxNQUFNLEVBQUUsZ0JBQU0sQ0FBQyxLQUFLO29CQUNwQixTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQztpQkFDdkMsQ0FBQzthQUNIO1lBQ0QsS0FBSyxFQUFFLENBQUMsYUFBYSxDQUFDO1NBQ3ZCLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQTFKRCw0REEwSkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gXCJwYXRoXCI7XG5pbXBvcnQgeyBEdXJhdGlvbiwgRm4sIFJlbW92YWxQb2xpY3ksIFRhZ3MgfSBmcm9tIFwiYXdzLWNkay1saWJcIjtcbmltcG9ydCB7IElTZWN1cml0eUdyb3VwLCBTZWN1cml0eUdyb3VwIH0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1lYzJcIjtcbmltcG9ydCB7XG4gIEVmZmVjdCxcbiAgSU1hbmFnZWRQb2xpY3ksXG4gIElSb2xlLFxuICBNYW5hZ2VkUG9saWN5LFxuICBQb2xpY3lTdGF0ZW1lbnQsXG4gIFJvbGUsXG4gIFNlcnZpY2VQcmluY2lwYWwsXG59IGZyb20gXCJhd3MtY2RrLWxpYi9hd3MtaWFtXCI7XG5pbXBvcnQge1xuICBBcmNoaXRlY3R1cmUsXG4gIENvZGUsXG4gIEZ1bmN0aW9uLFxuICBJRnVuY3Rpb24sXG4gIElMYXllclZlcnNpb24sXG4gIExheWVyVmVyc2lvbixcbiAgUnVudGltZSxcbiAgVHJhY2luZyxcbn0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1sYW1iZGFcIjtcbmltcG9ydCB7IElMb2dHcm91cCwgTG9nR3JvdXAsIFJldGVudGlvbkRheXMgfSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWxvZ3NcIjtcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gXCJjb25zdHJ1Y3RzXCI7XG5pbXBvcnQgeyBJT3V0bGllckRldGVjdGlvbkZ1bmN0aW9uIH0gZnJvbSBcIi4vSU91dGxpZXJEZXRlY3Rpb25GdW5jdGlvblwiO1xuaW1wb3J0IHsgT3V0bGllckRldGVjdGlvbkZ1bmN0aW9uUHJvcHMgfSBmcm9tIFwiLi9wcm9wcy9PdXRsaWVyRGV0ZWN0aW9uRnVuY3Rpb25Qcm9wc1wiO1xuXG5leHBvcnQgY2xhc3MgT3V0bGllckRldGVjdGlvbkZ1bmN0aW9uXG4gIGV4dGVuZHMgQ29uc3RydWN0XG4gIGltcGxlbWVudHMgSU91dGxpZXJEZXRlY3Rpb25GdW5jdGlvblxue1xuICAvKipcbiAgICogVGhlIHotc2NvcmUgZnVuY3Rpb25cbiAgICovXG4gIGZ1bmN0aW9uOiBJRnVuY3Rpb247XG5cbiAgLyoqXG4gICAqIFRoZSBsb2cgZ3JvdXAgd2hlcmUgdGhlIGNhbmFydHkgbG9ncyB3aWxsIGJlIHNlbnRcbiAgICovXG4gIGxvZ0dyb3VwOiBJTG9nR3JvdXA7XG5cbiAgY29uc3RydWN0b3IoXG4gICAgc2NvcGU6IENvbnN0cnVjdCxcbiAgICBpZDogc3RyaW5nLFxuICAgIHByb3BzOiBPdXRsaWVyRGV0ZWN0aW9uRnVuY3Rpb25Qcm9wcyxcbiAgKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGxldCB4cmF5TWFuYWdlZFBvbGljeTogSU1hbmFnZWRQb2xpY3kgPSBuZXcgTWFuYWdlZFBvbGljeShcbiAgICAgIHRoaXMsXG4gICAgICBcInhyYXlNYW5hZ2VkUG9saWN5XCIsXG4gICAgICB7XG4gICAgICAgIHBhdGg6IFwiL291dGxpZXItZGV0ZWN0aW9uL1wiLFxuICAgICAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAgIFwieHJheTpQdXRUcmFjZVNlZ21lbnRzXCIsXG4gICAgICAgICAgICAgIFwieHJheTpQdXRUZWxlbWV0cnlSZWNvcmRzXCIsXG4gICAgICAgICAgICAgIFwieHJheTpHZXRTYW1wbGluZ1J1bGVzXCIsXG4gICAgICAgICAgICAgIFwieHJheTpHZXRTYW1wbGluZ1RhcmdldHNcIixcbiAgICAgICAgICAgICAgXCJ4cmF5OkdldFNhbXBsaW5nU3RhdGlzdGljU3VtbWFyaWVzXCIsXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgZWZmZWN0OiBFZmZlY3QuQUxMT1csXG4gICAgICAgICAgICByZXNvdXJjZXM6IFtcIipcIl0sXG4gICAgICAgICAgfSksXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgICk7XG4gICAgbGV0IGN3TWFuYWdlZFBvbGljeSA9IG5ldyBNYW5hZ2VkUG9saWN5KHRoaXMsIFwiQ1dNYW5hZ2VkUG9saWN5XCIsIHtcbiAgICAgIHBhdGg6IFwiL291dGxpZXItZGV0ZWN0aW9uL1wiLFxuICAgICAgc3RhdGVtZW50czogW1xuICAgICAgICBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICBhY3Rpb25zOiBbXCJjbG91ZHdhdGNoOkdldE1ldHJpY0RhdGFcIiwgXCJjbG9kdXdhdGNoOlB1dE1ldHJpY0RhdGFcIl0sXG4gICAgICAgICAgZWZmZWN0OiBFZmZlY3QuQUxMT1csXG4gICAgICAgICAgcmVzb3VyY2VzOiBbXCIqXCJdLFxuICAgICAgICB9KSxcbiAgICAgIF0sXG4gICAgfSk7XG5cbiAgICBsZXQgZXhlY3V0aW9uUm9sZTogSVJvbGUgPSBuZXcgUm9sZSh0aGlzLCBcImV4ZWN1dGlvblJvbGVcIiwge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgU2VydmljZVByaW5jaXBhbChcImxhbWJkYS5hbWF6b25hd3MuY29tXCIpLFxuICAgICAgcGF0aDogXCIvb3V0bGllci1kZXRlY3Rpb24vXCIsXG4gICAgICBtYW5hZ2VkUG9saWNpZXM6IFt4cmF5TWFuYWdlZFBvbGljeSwgY3dNYW5hZ2VkUG9saWN5XSxcbiAgICB9KTtcblxuICAgIGxldCBzY2lQeUxheWVyOiBJTGF5ZXJWZXJzaW9uID0gbmV3IExheWVyVmVyc2lvbih0aGlzLCBcIlNjaVB5TGF5ZXJcIiwge1xuICAgICAgY29kZTogQ29kZS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgXCJzcmMvc2NpcHktbGF5ZXIuemlwXCIpKSxcbiAgICAgIGNvbXBhdGlibGVBcmNoaXRlY3R1cmVzOiBbQXJjaGl0ZWN0dXJlLkFSTV82NF0sXG4gICAgICBjb21wYXRpYmxlUnVudGltZXM6IFtSdW50aW1lLlBZVEhPTl8zXzEyXSxcbiAgICB9KTtcblxuICAgIGxldCBtb25pdG9yaW5nTGF5ZXI6IElMYXllclZlcnNpb24gPSBuZXcgTGF5ZXJWZXJzaW9uKFxuICAgICAgdGhpcyxcbiAgICAgIFwiTW9uaXRvcmluZ0xheWVyXCIsXG4gICAgICB7XG4gICAgICAgIGNvZGU6IENvZGUuZnJvbUFzc2V0KFxuICAgICAgICAgIHBhdGguam9pbihfX2Rpcm5hbWUsIFwiLi4vbW9uaXRvcmluZy9zcmMvbW9uaXRvcmluZy1sYXllci56aXBcIiksXG4gICAgICAgICksXG4gICAgICAgIGNvbXBhdGlibGVBcmNoaXRlY3R1cmVzOiBbQXJjaGl0ZWN0dXJlLkFSTV82NF0sXG4gICAgICAgIGNvbXBhdGlibGVSdW50aW1lczogW1J1bnRpbWUuUFlUSE9OXzNfMTJdLFxuICAgICAgfSxcbiAgICApO1xuXG4gICAgaWYgKHByb3BzLnZwYyAhPT0gdW5kZWZpbmVkICYmIHByb3BzLnZwYyAhPSBudWxsKSB7XG4gICAgICBsZXQgc2c6IElTZWN1cml0eUdyb3VwID0gbmV3IFNlY3VyaXR5R3JvdXAoXG4gICAgICAgIHRoaXMsXG4gICAgICAgIFwiT3V0bGllckRldGVjdGlvblNlY3VyaXR5R3JvdXBcIixcbiAgICAgICAge1xuICAgICAgICAgIGRlc2NyaXB0aW9uOlxuICAgICAgICAgICAgXCJBbGxvdyBvdXRsaWVyIGRldGVjdGlvbiBmdW5jdGlvbiB0byBjb21tdW5pY2F0ZSB3aXRoIENXXCIsXG4gICAgICAgICAgdnBjOiBwcm9wcy52cGMsXG4gICAgICAgICAgYWxsb3dBbGxPdXRib3VuZDogdHJ1ZSxcbiAgICAgICAgfSxcbiAgICAgICk7XG5cbiAgICAgIHRoaXMuZnVuY3Rpb24gPSBuZXcgRnVuY3Rpb24odGhpcywgXCJPdXRsaWVyRGV0ZWN0aW9uXCIsIHtcbiAgICAgICAgcnVudGltZTogUnVudGltZS5QWVRIT05fM18xMixcbiAgICAgICAgY29kZTogQ29kZS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgXCJzcmMvb3V0bGllci1kZXRlY3Rpb24uemlwXCIpKSxcbiAgICAgICAgaGFuZGxlcjogXCJpbmRleC5oYW5kbGVyXCIsXG4gICAgICAgIHJvbGU6IGV4ZWN1dGlvblJvbGUsXG4gICAgICAgIGFyY2hpdGVjdHVyZTogQXJjaGl0ZWN0dXJlLkFSTV82NCxcbiAgICAgICAgdHJhY2luZzogVHJhY2luZy5BQ1RJVkUsXG4gICAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLnNlY29uZHMoNSksXG4gICAgICAgIG1lbW9yeVNpemU6IDUxMixcbiAgICAgICAgbGF5ZXJzOiBbc2NpUHlMYXllciwgbW9uaXRvcmluZ0xheWVyXSxcbiAgICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgICBSRUdJT046IEZuLnJlZihcIkFXUzo6UmVnaW9uXCIpLFxuICAgICAgICAgIFBBUlRJVElPTjogRm4ucmVmKFwiQVdTOjpQYXJ0aXRpb25cIiksXG4gICAgICAgIH0sXG4gICAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgICAgICBzZWN1cml0eUdyb3VwczogW3NnXSxcbiAgICAgICAgdnBjU3VibmV0czogcHJvcHMuc3VibmV0U2VsZWN0aW9uLFxuICAgICAgfSk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuZnVuY3Rpb24gPSBuZXcgRnVuY3Rpb24odGhpcywgXCJPdXRsaWVyRGV0ZWN0aW9uXCIsIHtcbiAgICAgICAgcnVudGltZTogUnVudGltZS5QWVRIT05fM18xMixcbiAgICAgICAgY29kZTogQ29kZS5mcm9tQXNzZXQocGF0aC5qb2luKF9fZGlybmFtZSwgXCJzcmMvb3V0bGllci1kZXRlY3Rpb24uemlwXCIpKSxcbiAgICAgICAgaGFuZGxlcjogXCJpbmRleC5oYW5kbGVyXCIsXG4gICAgICAgIHJvbGU6IGV4ZWN1dGlvblJvbGUsXG4gICAgICAgIGFyY2hpdGVjdHVyZTogQXJjaGl0ZWN0dXJlLkFSTV82NCxcbiAgICAgICAgdHJhY2luZzogVHJhY2luZy5BQ1RJVkUsXG4gICAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLnNlY29uZHMoNSksXG4gICAgICAgIG1lbW9yeVNpemU6IDUxMixcbiAgICAgICAgbGF5ZXJzOiBbc2NpUHlMYXllciwgbW9uaXRvcmluZ0xheWVyXSxcbiAgICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgICBSRUdJT046IEZuLnJlZihcIkFXUzo6UmVnaW9uXCIpLFxuICAgICAgICAgIFBBUlRJVElPTjogRm4ucmVmKFwiQVdTOjpQYXJ0aXRpb25cIiksXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICB9XG5cbiAgICBUYWdzLm9mKHRoaXMuZnVuY3Rpb24pLmFkZChcImNsb3Vkd2F0Y2g6ZGF0YXNvdXJjZVwiLCBcImN1c3RvbVwiLCB7XG4gICAgICBpbmNsdWRlUmVzb3VyY2VUeXBlczogW1wiQVdTOjpMYW1iZGE6OkZ1bmN0aW9uXCJdLFxuICAgIH0pO1xuXG4gICAgdGhpcy5mdW5jdGlvbi5hZGRQZXJtaXNzaW9uKFwiaW52b2tlUGVybWlzc2lvblwiLCB7XG4gICAgICBhY3Rpb246IFwibGFtYmRhOkludm9rZUZ1bmN0aW9uXCIsXG4gICAgICBwcmluY2lwYWw6IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKFxuICAgICAgICBcImxhbWJkYS5kYXRhc291cmNlLmNsb3Vkd2F0Y2guYW1hem9uYXdzLmNvbVwiLFxuICAgICAgKSxcbiAgICAgIHNvdXJjZUFjY291bnQ6IEZuLnJlZihcIkFXUzo6QWNjb3VudElkXCIpLFxuICAgIH0pO1xuXG4gICAgdGhpcy5sb2dHcm91cCA9IG5ldyBMb2dHcm91cCh0aGlzLCBcImxvZ0dyb3VwXCIsIHtcbiAgICAgIGxvZ0dyb3VwTmFtZTogYC9hd3MvbGFtYmRhLyR7dGhpcy5mdW5jdGlvbi5mdW5jdGlvbk5hbWV9YCxcbiAgICAgIHJldGVudGlvbjogUmV0ZW50aW9uRGF5cy5PTkVfV0VFSyxcbiAgICAgIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICB9KTtcblxuICAgIG5ldyBNYW5hZ2VkUG9saWN5KHRoaXMsIFwiY3dMb2dzTWFuYWdlZFBvbGljeVwiLCB7XG4gICAgICBwYXRoOiBcIi9vdXRsaWVyLWRldGVjdGlvbi9cIixcbiAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgYWN0aW9uczogW1wibG9nczpDcmVhdGVMb2dTdHJlYW1cIiwgXCJsb2dzOlB1dExvZ0V2ZW50c1wiXSxcbiAgICAgICAgICBlZmZlY3Q6IEVmZmVjdC5BTExPVyxcbiAgICAgICAgICByZXNvdXJjZXM6IFt0aGlzLmxvZ0dyb3VwLmxvZ0dyb3VwQXJuXSxcbiAgICAgICAgfSksXG4gICAgICBdLFxuICAgICAgcm9sZXM6IFtleGVjdXRpb25Sb2xlXSxcbiAgICB9KTtcbiAgfVxufVxuIl19