"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ZonalAvailabilityMetrics = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const AvailabilityAndLatencyMetrics_1 = require("./AvailabilityAndLatencyMetrics");
const AvailabilityMetricType_1 = require("../utilities/AvailabilityMetricType");
class ZonalAvailabilityMetrics {
    /**
     * Creates a zonal service level availability metrics, one metric for
     * each operation at the zonal level and the service.
     * @param props
     * @returns The metric at index 0 is the metric math expression for the whole service. The following metrics
     * are the metrics for each operation included in the request availability metric props.
     */
    static createZonalServiceAvailabilityMetrics(props) {
        let usingMetrics = {};
        let operationMetrics = [];
        let counter = 0;
        props.availabilityMetricProps.forEach((prop) => {
            let keyPrefix = (prop.keyPrefix === undefined || prop.keyPrefix == ""
                ? ""
                : prop.keyPrefix.toLowerCase() + "_") +
                prop.metricDetails.operationName.toLowerCase() +
                "_" +
                prop.metricType.toString().toLowerCase();
            let zonalOperationAvailabilityMetric = this.createZonalAvailabilityMetric(prop);
            operationMetrics.push(zonalOperationAvailabilityMetric);
            usingMetrics[`${keyPrefix}${counter++}`] =
                zonalOperationAvailabilityMetric;
        });
        let expression = "";
        switch (props.availabilityMetricProps[0].metricType) {
            case AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE:
                expression = `(${Object.keys(usingMetrics).join("+")}) / ${props.availabilityMetricProps.length}`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT:
                expression = `${Object.keys(usingMetrics).join("+")}`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT:
                expression = `${Object.keys(usingMetrics).join("+")}`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.FAULT_RATE:
                expression = `(${Object.keys(usingMetrics).join("+")}) / ${props.availabilityMetricProps.length}`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_COUNT:
                expression = `${Object.keys(usingMetrics).join("+")}`;
                break;
        }
        let math = new aws_cloudwatch_1.MathExpression({
            usingMetrics: usingMetrics,
            period: props.period,
            label: props.label,
            expression: expression,
        });
        operationMetrics.splice(0, 0, math);
        return operationMetrics;
    }
    /**
     * Creates a zonal availability metric
     * @param props
     * @returns
     */
    static createZonalAvailabilityMetric(props) {
        return AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.createAvailabilityMetric(props, props.metricDetails.metricDimensions.zonalDimensions(props.availabilityZoneId, aws_cdk_lib_1.Fn.ref("AWS::Region")));
    }
}
exports.ZonalAvailabilityMetrics = ZonalAvailabilityMetrics;
//# sourceMappingURL=data:application/json;base64,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