"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AvailabilityAndLatencyMetrics = void 0;
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const AvailabilityMetricType_1 = require("../utilities/AvailabilityMetricType");
const LatencyMetricType_1 = require("../utilities/LatencyMetricType");
/**
 * Class for creating availability and latency metrics that can be used in alarms and graphs
 */
class AvailabilityAndLatencyMetrics {
    /**
     * Increments a str by one char, for example
     * a -> b
     * z -> aa
     * ad -> ae
     *
     * This wraps at z and adds a new 'a'
     * @param str
     * @returns
     */
    static nextChar(str) {
        if (str.length == 0) {
            return "a";
        }
        let charA = str.split("");
        if (charA[charA.length - 1] === "z") {
            return (AvailabilityAndLatencyMetrics.nextChar(str.substring(0, charA.length - 1)) + "a");
        }
        else {
            return (str.substring(0, charA.length - 1) +
                String.fromCharCode(charA[charA.length - 1].charCodeAt(0) + 1));
        }
    }
    /**
     * General purpose method to create availability metrics
     * @param props
     * @param dimensions
     * @returns
     */
    static createAvailabilityMetric(props, dimensions) {
        let counter = 0;
        let key = "";
        let usingMetrics = {};
        let successKeys = [];
        let faultKeys = [];
        if (props.metricDetails.successMetricNames !== undefined &&
            props.metricType != AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT) {
            props.metricDetails.successMetricNames.forEach((successMetric) => {
                let keyPrefix = (props.keyPrefix === undefined || props.keyPrefix == ""
                    ? ""
                    : props.keyPrefix.toLowerCase() + "_") +
                    props.metricDetails.operationName.toLowerCase() +
                    "_" +
                    successMetric.toLowerCase();
                key = keyPrefix + "_" + counter++;
                successKeys.push(key);
                usingMetrics[key] = new aws_cloudwatch_1.Metric({
                    namespace: props.metricDetails.metricNamespace,
                    metricName: successMetric,
                    unit: props.metricDetails.unit,
                    period: props.metricDetails.period,
                    statistic: props.metricDetails.alarmStatistic,
                    dimensionsMap: dimensions,
                    label: successMetric,
                });
            });
        }
        if (props.metricDetails.faultMetricNames !== undefined &&
            props.metricType != AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_COUNT) {
            props.metricDetails.faultMetricNames.forEach((faultMetric) => {
                let keyPrefix = (props.keyPrefix === undefined || props.keyPrefix == ""
                    ? ""
                    : props.keyPrefix.toLowerCase() + "_") +
                    props.metricDetails.operationName.toLowerCase() +
                    "_" +
                    faultMetric.toLowerCase();
                key = keyPrefix + "_" + counter++;
                faultKeys.push(key);
                usingMetrics[key] = new aws_cloudwatch_1.Metric({
                    namespace: props.metricDetails.metricNamespace,
                    metricName: faultMetric,
                    unit: props.metricDetails.unit,
                    period: props.metricDetails.period,
                    statistic: props.metricDetails.alarmStatistic,
                    dimensionsMap: dimensions,
                    label: faultMetric,
                });
            });
        }
        let expression = "";
        switch (props.metricType) {
            case AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE:
                expression = `((${successKeys.join("+")}) / (${successKeys.join("+")}+${faultKeys.join("+")})) * 100`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT:
                expression = `${successKeys.join("+")}+${faultKeys.join("+")}`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT:
                expression = `(${faultKeys.join("+")})`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.FAULT_RATE:
                expression = `((${faultKeys.join("+")}) / (${successKeys.join("+")}+${faultKeys.join("+")})) * 100`;
                break;
            case AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_COUNT:
                expression = `(${successKeys.join("+")})`;
                break;
        }
        return new aws_cloudwatch_1.MathExpression({
            expression: expression,
            label: props.label,
            period: props.metricDetails.period,
            usingMetrics: usingMetrics,
        });
    }
    /**
     * General purpose method to create latency metrics, the reason this creates an array of metrics while the
     * equivalent availability metric method doesn't is because in availability, we can just sum the count of different
     * metric names while for latency we can't sum the count because that's not what's being measured. It allows the
     * caller to decide if they only want to take the first name, or average all of the names
     * (like SuccessLatency and BigItemSuccessLatency).
     *
     * @param props
     * @param dimensions
     * @returns
     */
    static createLatencyMetrics(props, dimensions) {
        let names;
        switch (props.metricType) {
            default:
            case LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY:
                names = props.metricDetails.successMetricNames;
                break;
            case LatencyMetricType_1.LatencyMetricType.FAULT_LATENCY:
                names = props.metricDetails.faultMetricNames;
                break;
        }
        return names.map((x) => new aws_cloudwatch_1.Metric({
            metricName: x,
            namespace: props.metricDetails.metricNamespace,
            unit: props.metricDetails.unit,
            period: props.metricDetails.period,
            statistic: props.statistic,
            dimensionsMap: dimensions,
            label: props.label,
        }));
    }
    /**
     * Takes all of the success or failure latency metric names and creates an average of those
     * names, if there's only 1 name, it just returns that metric
     * @param props
     * @param dimensions
     */
    static createAverageLatencyMetric(props, dimensions) {
        let latencyMetrics = AvailabilityAndLatencyMetrics.createLatencyMetrics(props, dimensions);
        if (latencyMetrics.length == 1) {
            return latencyMetrics[0];
        }
        else {
            let usingMetrics = {};
            latencyMetrics.forEach((metric, index) => {
                let keyPrefix = (props.keyPrefix === undefined || props.keyPrefix == ""
                    ? ""
                    : props.keyPrefix.toLowerCase() + "_") +
                    props.metricDetails.operationName.toLowerCase() +
                    "_" +
                    props.metricType.toString().toLowerCase();
                usingMetrics[keyPrefix + index] = metric;
            });
            return new aws_cloudwatch_1.MathExpression({
                expression: `(${Object.keys(usingMetrics).join("+")})/${Object.keys(usingMetrics).length}`,
                label: props.label,
                period: props.metricDetails.period,
                usingMetrics: usingMetrics,
            });
        }
    }
    /**
     * Creates a count of high latency metrics for either SuccessLatency or FaultLatency, will total
     * the count of requests that exceed a threshold you define in your statistic, like TC(200:) across
     * all metric names that are part of either Success or Fault latency.
     * @param props
     * @returns
     */
    static createLatencyCountMetric(props, dimensions) {
        let latencyMetrics = AvailabilityAndLatencyMetrics.createLatencyMetrics(props, dimensions);
        if (latencyMetrics.length == 1) {
            return latencyMetrics[0];
        }
        else {
            let usingMetrics = {};
            latencyMetrics.forEach((metric, index) => {
                let keyPrefix = (props.keyPrefix === undefined || props.keyPrefix == ""
                    ? ""
                    : props.keyPrefix.toLowerCase() + "_") +
                    props.metricDetails.operationName.toLowerCase() +
                    "_" +
                    props.metricType.toString().toLowerCase();
                usingMetrics[keyPrefix + index] = metric;
            });
            return new aws_cloudwatch_1.MathExpression({
                expression: Object.keys(usingMetrics).join("+"),
                label: props.label,
                period: props.metricDetails.period,
                usingMetrics: usingMetrics,
            });
        }
    }
}
exports.AvailabilityAndLatencyMetrics = AvailabilityAndLatencyMetrics;
//# sourceMappingURL=data:application/json;base64,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