"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationLoadBalancerMetrics = void 0;
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
class ApplicationLoadBalancerMetrics {
    /**
     * Creates a regional fault count metric using 5xx target and load balancer
     * metrics against total requests for the specified load balancer
     * @param period
     * @param loadBalancerFullName
     * @returns
     */
    static createRegionalApplicationLoadBalancerFaultRateMetric(loadBalancerFullName, period) {
        return new aws_cloudwatch_1.MathExpression({
            expression: "((m1 + m2) / m3) * 100",
            label: "Fault Rate",
            period: period,
            usingMetrics: {
                m1: new aws_cloudwatch_1.Metric({
                    metricName: "HTTPCode_Target_5XX_Count",
                    namespace: "AWS/ApplicationELB",
                    unit: aws_cloudwatch_1.Unit.COUNT,
                    period: period,
                    statistic: "Sum",
                    dimensionsMap: {
                        LoadBalancer: loadBalancerFullName,
                    },
                    label: "5xxTarget",
                }),
                m2: new aws_cloudwatch_1.Metric({
                    metricName: "HTTPCode_ELB_5XX_Count",
                    namespace: "AWS/ApplicationELB",
                    unit: aws_cloudwatch_1.Unit.COUNT,
                    period: period,
                    statistic: "Sum",
                    dimensionsMap: {
                        LoadBalancer: loadBalancerFullName,
                    },
                    label: "5xxELB",
                }),
                m3: new aws_cloudwatch_1.Metric({
                    metricName: "RequestCount",
                    namespace: "AWS/ApplicationELB",
                    unit: aws_cloudwatch_1.Unit.COUNT,
                    period: period,
                    statistic: "Sum",
                    dimensionsMap: {
                        LoadBalancer: loadBalancerFullName,
                    },
                    label: "Requests",
                }),
            },
        });
    }
    /**
     * Creates a zonal fault count metric using 5xx target and load balancer
     * metrics against total requests for the specified load balancer
     * @param loadBalancerFullName
     * @param availabilityZoneName
     * @param period
     * @returns
     */
    static createZonalApplicationLoadBalancerFaultRateMetric(loadBalancerFullName, availabilityZoneName, period) {
        return new aws_cloudwatch_1.MathExpression({
            expression: "((m1 + m2) / m3) * 100",
            label: "Fault Rate",
            period: period,
            usingMetrics: {
                m1: new aws_cloudwatch_1.Metric({
                    metricName: "HTTPCode_Target_5XX_Count",
                    namespace: "AWS/ApplicationELB",
                    unit: aws_cloudwatch_1.Unit.COUNT,
                    period: period,
                    statistic: "Sum",
                    dimensionsMap: {
                        LoadBalancer: loadBalancerFullName,
                        AvailabilityZone: availabilityZoneName,
                    },
                    label: "5xxTarget",
                }),
                m2: new aws_cloudwatch_1.Metric({
                    metricName: "HTTPCode_ELB_5XX_Count",
                    namespace: "AWS/ApplicationELB",
                    unit: aws_cloudwatch_1.Unit.COUNT,
                    period: period,
                    statistic: "Sum",
                    dimensionsMap: {
                        LoadBalancer: loadBalancerFullName,
                        AvailabilityZone: availabilityZoneName,
                    },
                    label: "5xxELB",
                }),
                m3: new aws_cloudwatch_1.Metric({
                    metricName: "RequestCount",
                    namespace: "AWS/ApplicationELB",
                    unit: aws_cloudwatch_1.Unit.COUNT,
                    period: period,
                    statistic: "Sum",
                    dimensionsMap: {
                        LoadBalancer: loadBalancerFullName,
                        AvailabilityZone: availabilityZoneName,
                    },
                    label: "Requests",
                }),
            },
        });
    }
    /**
     * Creates a regional processed bytes metric for the specified load balancer
     * @param loadBalancerFullName
     * @param period
     * @returns
     */
    static createRegionalApplicationLoadBalancerProcessedBytesMetric(loadBalancerFullName, period) {
        return new aws_cloudwatch_1.Metric({
            metricName: "ProcessedBytes",
            namespace: "AWS/ApplicationELB",
            unit: aws_cloudwatch_1.Unit.COUNT,
            period: period,
            statistic: "Sum",
            dimensionsMap: {
                LoadBalancer: loadBalancerFullName,
            },
            label: "ProcessedBytes",
        });
    }
    /**
     * Creates a zonal processed bytes metric for the specified load balancer
     * @param loadBalancerFullName
     * @param availabilityZoneName
     * @param period
     * @returns IMetric
     */
    static createZonalApplicationLoadBalancerProcessedBytesMetric(loadBalancerFullName, availabilityZoneName, period) {
        return new aws_cloudwatch_1.Metric({
            metricName: "ProcessedBytes",
            namespace: "AWS/ApplicationELB",
            unit: aws_cloudwatch_1.Unit.COUNT,
            period: period,
            statistic: "Sum",
            dimensionsMap: {
                LoadBalancer: loadBalancerFullName,
                AvailabilityZone: availabilityZoneName,
            },
            label: "ProcessedBytes",
        });
    }
}
exports.ApplicationLoadBalancerMetrics = ApplicationLoadBalancerMetrics;
//# sourceMappingURL=data:application/json;base64,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