"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceAvailabilityAndLatencyDashboard = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const ApplicationLoadBalancerMetrics_1 = require("../metrics/ApplicationLoadBalancerMetrics");
const AvailabilityAndLatencyMetrics_1 = require("../metrics/AvailabilityAndLatencyMetrics");
const RegionalAvailabilityMetrics_1 = require("../metrics/RegionalAvailabilityMetrics");
const RegionalLatencyMetrics_1 = require("../metrics/RegionalLatencyMetrics");
const ZonalAvailabilityMetrics_1 = require("../metrics/ZonalAvailabilityMetrics");
const ZonalLatencyMetrics_1 = require("../metrics/ZonalLatencyMetrics");
const AvailabilityMetricType_1 = require("../utilities/AvailabilityMetricType");
const LatencyMetricType_1 = require("../utilities/LatencyMetricType");
/**
 * Creates a service level availability and latency dashboard
 */
class ServiceAvailabilityAndLatencyDashboard extends constructs_1.Construct {
    static generateTPSWidgets(props, availabilityZoneIds) {
        let widgets = [];
        widgets.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: "**TPS Metrics**" }));
        widgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 6,
            width: 24,
            title: aws_cdk_lib_1.Fn.ref("AWS::Region") + " TPS",
            region: aws_cdk_lib_1.Fn.ref("AWS::Region"),
            left: RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalServiceAvailabilityMetrics({
                label: aws_cdk_lib_1.Fn.ref("AWS::Region") + " tps",
                period: props.service.period,
                availabilityMetricProps: props.service.operations
                    .filter((x) => x.critical)
                    .map((x) => {
                    return {
                        label: x.operationName,
                        metricDetails: x.serverSideAvailabilityMetricDetails,
                        metricType: AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT,
                    };
                }),
            }),
            statistic: "Sum",
            leftYAxis: {
                label: "TPS",
                showUnits: false,
            },
        }));
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let availabilityZoneId = availabilityZoneIds[i];
            let zonalMetricProps = {
                availabilityMetricProps: props.service.operations
                    .filter((x) => x.critical)
                    .map((x) => {
                    return {
                        availabilityZoneId: availabilityZoneId,
                        label: x.operationName,
                        metricDetails: x.serverSideAvailabilityMetricDetails,
                        metricType: AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT,
                    };
                }),
                period: props.service.period,
                label: availabilityZoneId + "tps",
            };
            widgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + " TPS",
                region: aws_cdk_lib_1.Fn.ref("AWS::Region"),
                left: ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalServiceAvailabilityMetrics(zonalMetricProps),
                statistic: "Sum",
                leftYAxis: {
                    label: "TPS",
                    showUnits: false,
                },
            }));
        }
        return widgets;
    }
    static generateServerSideAndCanaryAvailabilityWidgets(props, availabilityZoneIds) {
        let widgets = [];
        widgets.push(new aws_cloudwatch_1.TextWidget({
            height: 2,
            width: 24,
            markdown: "**Server-side Availability**\n(Each critical operation is equally weighted regardless of request volume)",
        }));
        widgets = widgets.concat(ServiceAvailabilityAndLatencyDashboard.generateAvailabilityWidgets(props, false, availabilityZoneIds));
        if (props.service.operations.filter((x) => x.critical && x.canaryMetricDetails !== undefined).length > 0) {
            widgets.push(new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: "**Canary Measured Availability**\n(Each operation is equally weighted regardless of request volume)",
            }));
            widgets = widgets.concat(ServiceAvailabilityAndLatencyDashboard.generateAvailabilityWidgets(props, true, availabilityZoneIds));
        }
        return widgets;
    }
    static generateServerSideAndCanaryLatencyWidgets(props, availabilityZoneIds) {
        let widgets = [];
        widgets.push(new aws_cloudwatch_1.TextWidget({
            height: 2,
            width: 24,
            markdown: "**Server-side Latency**\n(Each critical operation is equally weighted regardless of request volume)",
        }));
        widgets = widgets.concat(ServiceAvailabilityAndLatencyDashboard.generateLatencyMetricWidgets(props, false, availabilityZoneIds));
        if (props.service.operations.filter((x) => x.critical && x.canaryMetricDetails !== undefined).length > 0) {
            widgets.push(new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: "**Canary Measured Latency**\n(Each operation is equally weighted regardless of request volume)",
            }));
            widgets = widgets.concat(ServiceAvailabilityAndLatencyDashboard.generateLatencyMetricWidgets(props, true, availabilityZoneIds));
        }
        return widgets;
    }
    static generateAvailabilityWidgets(props, isCanary, availabilityZoneIds) {
        let widgets = [];
        widgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 6,
            width: 24,
            title: aws_cdk_lib_1.Fn.ref("AWS::Region") + " Availability",
            region: aws_cdk_lib_1.Fn.ref("AWS::Region"),
            left: RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalServiceAvailabilityMetrics({
                label: aws_cdk_lib_1.Fn.ref("AWS::Region") + " availability",
                period: props.service.period,
                availabilityMetricProps: this.createRegionalAvailabilityMetricProps(props.service.operations.filter((x) => x.critical), isCanary, AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE),
            }),
            statistic: "Sum",
            leftYAxis: {
                max: 100,
                min: 95,
                label: "Availability",
                showUnits: false,
            },
            right: RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalServiceAvailabilityMetrics({
                label: aws_cdk_lib_1.Fn.ref("AWS::Region") + " faults",
                period: props.service.period,
                availabilityMetricProps: this.createRegionalAvailabilityMetricProps(props.service.operations.filter((x) => x.critical), isCanary, AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT),
            }),
            rightYAxis: {
                label: "Faults",
                showUnits: false,
                min: 0,
                max: Math.ceil(props.service.faultCountThreshold * 1.5),
            },
            rightAnnotations: [
                {
                    color: aws_cloudwatch_1.Color.RED,
                    label: "High severity",
                    value: props.service.faultCountThreshold,
                },
            ],
        }));
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let availabilityZoneId = availabilityZoneIds[i];
            widgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + " Availability",
                region: aws_cdk_lib_1.Fn.ref("AWS::Region"),
                left: ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalServiceAvailabilityMetrics({
                    label: availabilityZoneId + " availability",
                    period: props.service.period,
                    availabilityMetricProps: this.createZonalAvailabilityMetricProps(props.service.operations.filter((x) => x.critical), availabilityZoneId, isCanary, AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE),
                }),
                statistic: "Sum",
                leftYAxis: {
                    max: 100,
                    min: 95,
                    label: "Availability",
                    showUnits: false,
                },
                right: ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalServiceAvailabilityMetrics({
                    label: availabilityZoneId + " faults",
                    period: props.service.period,
                    availabilityMetricProps: this.createZonalAvailabilityMetricProps(props.service.operations.filter((x) => x.critical), availabilityZoneId, isCanary, AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT),
                }),
                rightYAxis: {
                    label: "Faults",
                    showUnits: false,
                    min: 0,
                    max: Math.ceil(props.service.faultCountThreshold * 1.5),
                },
                rightAnnotations: [
                    {
                        color: aws_cloudwatch_1.Color.RED,
                        label: "High severity",
                        value: props.service.faultCountThreshold,
                    },
                ],
            }));
        }
        return widgets;
    }
    static generateLatencyMetricWidgets(props, isCanary, availabilityZoneIds) {
        let widgets = [];
        widgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 6,
            width: 24,
            title: aws_cdk_lib_1.Fn.ref("AWS::Region") + " Latency",
            region: aws_cdk_lib_1.Fn.ref("AWS::Region"),
            left: RegionalLatencyMetrics_1.RegionalLatencyMetrics.createRegionalServiceLatencyCountMetrics({
                label: aws_cdk_lib_1.Fn.ref("AWS::Region") + " latency",
                period: props.service.period,
                latencyMetricProps: this.createRegionalLatencyMetricProps(props.service.operations.filter((x) => x.critical), isCanary, LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY),
            }),
            statistic: "Sum",
            leftYAxis: {
                max: props.service.faultCountThreshold * 1.5,
                min: 0,
                label: "High latency count",
                showUnits: false,
            },
            leftAnnotations: [
                {
                    color: aws_cloudwatch_1.Color.RED,
                    label: "High severity",
                    value: props.service.faultCountThreshold,
                },
            ],
        }));
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let availabilityZoneId = availabilityZoneIds[i];
            widgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + " Latency",
                region: aws_cdk_lib_1.Fn.ref("AWS::Region"),
                left: ZonalLatencyMetrics_1.ZonalLatencyMetrics.createZonalServiceLatencyMetrics({
                    label: availabilityZoneId + " latency",
                    period: props.service.period,
                    latencyMetricProps: this.createZonalLatencyMetricProps(props.service.operations.filter((x) => x.critical), availabilityZoneId, isCanary, LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY),
                }),
                statistic: "Sum",
                leftYAxis: {
                    max: props.service.faultCountThreshold * 1.5,
                    min: 0,
                    label: "High latency count",
                    showUnits: false,
                },
                leftAnnotations: [
                    {
                        color: aws_cloudwatch_1.Color.RED,
                        label: "High severity",
                        value: props.service.faultCountThreshold,
                    },
                ],
            }));
        }
        return widgets;
    }
    static createRegionalAvailabilityMetricProps(criticalOperations, isCanary, metricType) {
        return criticalOperations
            .reduce((filtered, value) => {
            if (isCanary && value.canaryMetricDetails) {
                filtered.push(value.canaryMetricDetails.canaryAvailabilityMetricDetails);
            }
            else if (!isCanary) {
                filtered.push(value.serverSideAvailabilityMetricDetails);
            }
            return filtered;
        }, [])
            .map((x) => {
            return {
                label: x.operationName + " " + metricType.toString().replace("_", " "),
                metricDetails: x,
                metricType: metricType,
            };
        });
    }
    static createRegionalLatencyMetricProps(criticalOperations, isCanary, metricType) {
        return criticalOperations
            .reduce((filtered, value) => {
            if (isCanary && value.canaryMetricDetails) {
                filtered.push(value.canaryMetricDetails.canaryLatencyMetricDetails);
            }
            else if (!isCanary) {
                filtered.push(value.serverSideLatencyMetricDetails);
            }
            return filtered;
        }, [])
            .map((x) => {
            return {
                label: x.operationName + " " + metricType.toString().replace("_", " "),
                metricDetails: x,
                metricType: metricType,
                statistic: "TC(" + x.successAlarmThreshold + ":)",
            };
        });
    }
    static createZonalAvailabilityMetricProps(criticalOperations, availabilityZoneId, isCanary, metricType) {
        return criticalOperations
            .reduce((filtered, value) => {
            if (isCanary &&
                value.canaryMetricDetails !== undefined &&
                value.canaryMetricDetails != null) {
                filtered.push(value.canaryMetricDetails.canaryAvailabilityMetricDetails);
            }
            else if (!isCanary) {
                filtered.push(value.serverSideAvailabilityMetricDetails);
            }
            return filtered;
        }, [])
            .map((x) => {
            return {
                label: x.operationName + " " + metricType.toString().replace("_", " "),
                metricDetails: x,
                metricType: metricType,
                availabilityZoneId: availabilityZoneId,
            };
        });
    }
    static createZonalLatencyMetricProps(criticalOperations, availabilityZoneId, isCanary, metricType) {
        return criticalOperations
            .reduce((filtered, value) => {
            if (isCanary &&
                value.canaryMetricDetails !== undefined &&
                value.canaryMetricDetails != null) {
                filtered.push(value.canaryMetricDetails.canaryLatencyMetricDetails);
            }
            else if (!isCanary) {
                filtered.push(value.serverSideLatencyMetricDetails);
            }
            return filtered;
        }, [])
            .map((x) => {
            return {
                label: x.operationName + " " + metricType.toString().replace("_", " "),
                metricDetails: x,
                metricType: metricType,
                availabilityZoneId: availabilityZoneId,
                statistic: "TC(" + x.successAlarmThreshold + ":)",
            };
        });
    }
    static generateLoadBalancerWidgets(props, title, availabilityZoneNames) {
        let albWidgets = [];
        let loadBalancerFullName = props.service.loadBalancer.loadBalancerFullName;
        albWidgets.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: title }));
        albWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 24,
            title: aws_cdk_lib_1.Fn.sub("${AWS::Region} Fault Rate"),
            region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
            left: [
                ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.createRegionalApplicationLoadBalancerFaultRateMetric(loadBalancerFullName, props.service.period),
            ],
            leftYAxis: {
                max: 35,
                min: 0,
                label: "Fault Rate",
                showUnits: false,
            },
            leftAnnotations: [
                {
                    value: 1,
                    visible: true,
                    color: aws_cloudwatch_1.Color.RED,
                    label: "High severity",
                },
            ],
        }));
        availabilityZoneNames.forEach((availabilityZoneName) => {
            let availabilityZoneId = props.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(availabilityZoneName.substring(availabilityZoneName.length - 1));
            albWidgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + " Fault Rate",
                region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
                left: [
                    ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.createZonalApplicationLoadBalancerFaultRateMetric(loadBalancerFullName, availabilityZoneName, props.service.period),
                ],
                leftYAxis: {
                    max: 35,
                    min: 0,
                    label: "Fault Rate",
                    showUnits: false,
                },
                leftAnnotations: [
                    {
                        value: 1,
                        visible: true,
                        color: aws_cloudwatch_1.Color.RED,
                        label: "High severity",
                    },
                ],
            }));
        });
        return albWidgets;
    }
    constructor(scope, id, props) {
        super(scope, id);
        let topLevelAggregateAlarmWidgets = [];
        let availabilityZoneIds = props.service.availabilityZoneNames.map((x) => {
            return props.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(x.substring(x.length - 1));
        });
        topLevelAggregateAlarmWidgets.push(new aws_cloudwatch_1.TextWidget({
            height: 2,
            width: 24,
            markdown: "***Availability and Latency Alarms***",
        }));
        topLevelAggregateAlarmWidgets.push(new aws_cloudwatch_1.AlarmStatusWidget({
            height: 2,
            width: 24,
            alarms: [props.aggregateRegionalAlarm],
            title: "Customer Experience - Regional Aggregate Impact Alarm (measures fault count in aggregate across all critical operations)",
        }));
        let keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar("");
        let perOperationAZFaultsMetrics = [];
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let counter = 1;
            let availabilityZoneId = availabilityZoneIds[i];
            topLevelAggregateAlarmWidgets.push(new aws_cloudwatch_1.AlarmStatusWidget({
                height: 2,
                width: 8,
                alarms: [props.zonalAggregateAlarms[i]],
                title: availabilityZoneId +
                    " Zonal Isolated Impact Alarm (any critical operation in this AZ shows impact from server-side or canary)",
            }));
            let usingMetrics = {};
            props.service.operations
                .filter((x) => x.critical == true)
                .forEach((x) => {
                usingMetrics[`${keyPrefix}${counter++}`] =
                    ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
                        availabilityZoneId: availabilityZoneId,
                        metricDetails: x.serverSideAvailabilityMetricDetails,
                        label: availabilityZoneId + " " + x.operationName + " fault count",
                        metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
                        keyPrefix: keyPrefix,
                    });
            });
            let zonalFaultCount = new aws_cloudwatch_1.MathExpression({
                expression: Object.keys(usingMetrics).join("+"),
                label: availabilityZoneId + " fault count",
                usingMetrics: usingMetrics,
            });
            perOperationAZFaultsMetrics.push(zonalFaultCount);
            keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar(keyPrefix);
        }
        let azContributorWidgets = [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: "**AZ Contributors To Faults**",
            }),
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 24,
                title: "AZ Fault Count",
                period: props.service.period,
                left: perOperationAZFaultsMetrics,
            }),
        ];
        topLevelAggregateAlarmWidgets.concat(ServiceAvailabilityAndLatencyDashboard.generateTPSWidgets(props, availabilityZoneIds));
        this.dashboard = new aws_cloudwatch_1.Dashboard(this, "TopLevelDashboard", {
            dashboardName: props.service.serviceName.toLowerCase() +
                aws_cdk_lib_1.Fn.sub("-service-availability-and-latency-${AWS::Region}"),
            defaultInterval: props.interval,
            periodOverride: aws_cloudwatch_1.PeriodOverride.INHERIT,
            widgets: [
                topLevelAggregateAlarmWidgets,
                azContributorWidgets,
                ServiceAvailabilityAndLatencyDashboard.generateServerSideAndCanaryAvailabilityWidgets(props, availabilityZoneIds),
                ServiceAvailabilityAndLatencyDashboard.generateServerSideAndCanaryLatencyWidgets(props, availabilityZoneIds),
            ],
        });
        let lb = props.service.loadBalancer?.node
            .defaultChild;
        if (lb !== undefined && lb != null && lb.type == "application") {
            this.dashboard.addWidgets(...ServiceAvailabilityAndLatencyDashboard.generateLoadBalancerWidgets(props, "Application Load Balancer Metrics", props.service.availabilityZoneNames));
        }
    }
}
exports.ServiceAvailabilityAndLatencyDashboard = ServiceAvailabilityAndLatencyDashboard;
//# sourceMappingURL=data:application/json;base64,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