"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OperationAvailabilityAndLatencyDashboard = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const ContributorInsightsWidget_1 = require("./ContributorInsightsWidget");
const ApplicationLoadBalancerMetrics_1 = require("../metrics/ApplicationLoadBalancerMetrics");
const AvailabilityAndLatencyMetrics_1 = require("../metrics/AvailabilityAndLatencyMetrics");
const NetworkLoadBalancerMetrics_1 = require("../metrics/NetworkLoadBalancerMetrics");
const RegionalAvailabilityMetrics_1 = require("../metrics/RegionalAvailabilityMetrics");
const RegionalLatencyMetrics_1 = require("../metrics/RegionalLatencyMetrics");
const ZonalAvailabilityMetrics_1 = require("../metrics/ZonalAvailabilityMetrics");
const ZonalLatencyMetrics_1 = require("../metrics/ZonalLatencyMetrics");
const AvailabilityMetricType_1 = require("../utilities/AvailabilityMetricType");
const LatencyMetricType_1 = require("../utilities/LatencyMetricType");
/**
 * Creates an operation level availability and latency dashboard
 */
class OperationAvailabilityAndLatencyDashboard extends constructs_1.Construct {
    static createTopLevelAggregateAlarmWidgets(props, title, availabilityZoneIds) {
        let topLevelAggregateAlarms = [
            new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: title }),
            new aws_cloudwatch_1.AlarmStatusWidget({
                height: 2,
                width: 24,
                alarms: [props.regionalImpactAlarm],
                title: props.operation.operationName + " Regional Impact",
            }),
        ];
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let availabilityZoneId = availabilityZoneIds[i];
            topLevelAggregateAlarms.push(new aws_cloudwatch_1.AlarmStatusWidget({
                height: 2,
                width: 8,
                alarms: [props.isolatedAZImpactAlarms[i]],
                title: availabilityZoneId + " Isolated Impact",
            }));
        }
        topLevelAggregateAlarms.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: "**AZ Contributors**" }));
        let zonalServerSideHighLatencyMetrics = [];
        let zonalServerSideFaultCountMetrics = [];
        let zonalCanaryHighLatencyMetrics = [];
        let zonalCanaryFaultCountMetrics = [];
        let keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar("");
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let availabilityZoneId = availabilityZoneIds[i];
            zonalServerSideHighLatencyMetrics.push(ZonalLatencyMetrics_1.ZonalLatencyMetrics.createZonalCountLatencyMetric({
                availabilityZoneId: availabilityZoneId,
                metricDetails: props.operation.serverSideLatencyMetricDetails,
                label: availabilityZoneId + " high latency responses",
                metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
                statistic: `TC(${props.operation.serverSideLatencyMetricDetails.successAlarmThreshold}:)`,
                keyPrefix: keyPrefix,
            }));
            zonalServerSideFaultCountMetrics.push(ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
                availabilityZoneId: availabilityZoneId,
                metricDetails: props.operation.serverSideAvailabilityMetricDetails,
                label: availabilityZoneId + " fault count",
                metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
                keyPrefix: keyPrefix,
            }));
            if (props.operation.canaryMetricDetails !== undefined &&
                props.operation.canaryMetricDetails != null) {
                zonalCanaryHighLatencyMetrics.push(ZonalLatencyMetrics_1.ZonalLatencyMetrics.createZonalCountLatencyMetric({
                    availabilityZoneId: availabilityZoneId,
                    metricDetails: props.operation.canaryMetricDetails.canaryLatencyMetricDetails,
                    label: availabilityZoneId + " high latency responses",
                    metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
                    statistic: `TC(${props.operation.canaryMetricDetails.canaryLatencyMetricDetails.successAlarmThreshold}:)`,
                    keyPrefix: keyPrefix,
                }));
                zonalCanaryFaultCountMetrics.push(ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
                    availabilityZoneId: availabilityZoneId,
                    metricDetails: props.operation.canaryMetricDetails
                        .canaryAvailabilityMetricDetails,
                    label: availabilityZoneId + " fault count",
                    metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
                    keyPrefix: keyPrefix,
                }));
            }
            keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar(keyPrefix);
        }
        topLevelAggregateAlarms.push(new aws_cloudwatch_1.GraphWidget({
            height: 6,
            width: 24,
            title: "Server-side AZ Fault Contributors",
            left: zonalServerSideFaultCountMetrics,
        }));
        if (zonalCanaryFaultCountMetrics.length > 0) {
            topLevelAggregateAlarms.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 24,
                title: "Canary AZ Fault Contributors",
                left: zonalCanaryFaultCountMetrics,
            }));
        }
        topLevelAggregateAlarms.push(new aws_cloudwatch_1.GraphWidget({
            height: 6,
            width: 24,
            title: "Server-side High Latency Contributors",
            left: zonalServerSideHighLatencyMetrics,
        }));
        if (zonalCanaryHighLatencyMetrics.length > 0) {
            topLevelAggregateAlarms.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 24,
                title: "Canary High Latency Contributors",
                left: zonalCanaryHighLatencyMetrics,
            }));
        }
        topLevelAggregateAlarms.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: "**TPS Metrics**" }));
        topLevelAggregateAlarms.push(new aws_cloudwatch_1.GraphWidget({
            height: 6,
            width: 24,
            title: aws_cdk_lib_1.Fn.sub("${AWS::Region} TPS"),
            region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
            left: [
                RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalAvailabilityMetric({
                    label: aws_cdk_lib_1.Fn.ref("AWS::Region") + " tps",
                    metricDetails: props.operation.serverSideAvailabilityMetricDetails,
                    metricType: AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT,
                }),
            ],
            statistic: "Sum",
            leftYAxis: {
                label: "TPS",
                showUnits: false,
            },
        }));
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let availabilityZoneId = availabilityZoneIds[i];
            topLevelAggregateAlarms.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + " TPS",
                region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
                left: [
                    ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
                        availabilityZoneId: availabilityZoneId,
                        label: availabilityZoneId + " tps",
                        metricDetails: props.operation.serverSideAvailabilityMetricDetails,
                        metricType: AvailabilityMetricType_1.AvailabilityMetricType.REQUEST_COUNT,
                    }),
                ],
                statistic: "Sum",
                leftYAxis: {
                    label: "TPS",
                    showUnits: false,
                },
            }));
        }
        return topLevelAggregateAlarms;
    }
    static createAvailabilityWidgets(props, title) {
        let availabilityWidgets = [];
        availabilityWidgets.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: title }));
        let rowTracker = 0;
        let keyPrefix1 = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar("");
        let keyPrefix2 = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar(keyPrefix1);
        // Create regional availability and fault metrics and availability alarm widgets
        availabilityWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 24,
            title: aws_cdk_lib_1.Fn.sub("${AWS::Region} Availability"),
            region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
            left: [
                RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalAvailabilityMetric({
                    label: aws_cdk_lib_1.Fn.ref("AWS::Region") + " availability",
                    metricDetails: props.availabilityMetricDetails,
                    metricType: AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE,
                    keyPrefix: keyPrefix1,
                }),
            ],
            statistic: "Sum",
            leftYAxis: {
                max: 100,
                min: 95,
                label: "Availability",
                showUnits: false,
            },
            leftAnnotations: [
                {
                    value: props.availabilityMetricDetails.successAlarmThreshold,
                    visible: true,
                    color: aws_cloudwatch_1.Color.RED,
                    label: "High Severity",
                },
            ],
            right: [
                RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalAvailabilityMetric({
                    label: aws_cdk_lib_1.Fn.ref("AWS::Region") + " fault count",
                    metricDetails: props.availabilityMetricDetails,
                    metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
                    keyPrefix: keyPrefix2,
                }),
            ],
            rightYAxis: {
                label: "Fault Count",
                showUnits: false,
            },
        }));
        availabilityWidgets.push(new aws_cloudwatch_1.AlarmWidget({
            height: 2,
            width: 24,
            region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
            alarm: props.regionalEndpointAvailabilityAlarm,
            title: "Success Rate",
        }));
        for (let i = 0; i < props.availabilityZoneIds.length; i++) {
            let availabilityZoneId = props.availabilityZoneIds[i];
            let keyPrefix3 = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar("");
            let keyPrefix4 = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar(keyPrefix3);
            availabilityWidgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + " Availability",
                region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
                left: [
                    ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
                        availabilityZoneId: availabilityZoneId,
                        label: availabilityZoneId + " availability",
                        metricDetails: props.availabilityMetricDetails,
                        metricType: AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE,
                        keyPrefix: keyPrefix3,
                    }),
                ],
                statistic: "Sum",
                leftYAxis: {
                    max: 100,
                    min: 95,
                    label: "Availability",
                    showUnits: false,
                },
                leftAnnotations: [
                    {
                        value: props.availabilityMetricDetails.successAlarmThreshold,
                        visible: true,
                        color: aws_cloudwatch_1.Color.RED,
                        label: "High Severity",
                    },
                ],
                right: [
                    ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
                        availabilityZoneId: availabilityZoneId,
                        label: availabilityZoneId + " fault count",
                        metricDetails: props.availabilityMetricDetails,
                        metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
                        keyPrefix: keyPrefix4,
                    }),
                ],
                rightYAxis: {
                    label: "Fault Count",
                    showUnits: false,
                },
            }));
            //We're on the third one for this set, add 3 alarms
            //or if we're at the end, at the necessary amount
            //of alarms, 1, 2, or 3
            if (i % 3 == 2 || i - 1 == props.availabilityZoneIds.length) {
                for (let k = rowTracker; k <= i; k++) {
                    availabilityWidgets.push(new aws_cloudwatch_1.AlarmWidget({
                        height: 2,
                        width: 8,
                        region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
                        alarm: props.zonalEndpointAvailabilityAlarms[k],
                        title: "Success Rate",
                    }));
                }
                rowTracker += i + 1;
            }
        }
        if (!props.isCanary &&
            props.instanceContributorsToFaults !== undefined &&
            props.instanceContributorsToFaults != null) {
            availabilityWidgets.push(new ContributorInsightsWidget_1.ContributorInsightsWidget({
                height: 6,
                width: 24,
                title: "Individual Instance Contributors to Fault Count",
                insightRule: props.instanceContributorsToFaults,
                period: props.availabilityMetricDetails.period,
                legendPosition: aws_cloudwatch_1.LegendPosition.BOTTOM,
                orderStatistic: "Sum",
                accountId: aws_cdk_lib_1.Fn.ref("AWS::AccountId"),
                topContributors: 10,
            }));
        }
        return availabilityWidgets;
    }
    static createLatencyWidgets(props, title) {
        let latencyWidgets = [];
        latencyWidgets.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: title }));
        let rowTracker = 0;
        let keyPrefix = "";
        let latencyMetrics = [];
        let stats = props.latencyMetricDetails.graphedSuccessStatistics !== undefined
            ? props.latencyMetricDetails.graphedSuccessStatistics
            : ["p99"];
        let latencySuccessMetrics = stats.map((x) => {
            keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar(keyPrefix);
            return RegionalLatencyMetrics_1.RegionalLatencyMetrics.createRegionalAverageLatencyMetric({
                label: x + " Success Latency",
                metricDetails: props.latencyMetricDetails,
                metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
                statistic: x,
                keyPrefix: keyPrefix,
            });
        });
        stats =
            props.latencyMetricDetails.graphedFaultStatistics !== undefined
                ? props.latencyMetricDetails.graphedFaultStatistics
                : ["p99"];
        let latencyFaultMetrics = stats.map((x) => {
            keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar(keyPrefix);
            return RegionalLatencyMetrics_1.RegionalLatencyMetrics.createRegionalAverageLatencyMetric({
                label: x + " Fault Latency",
                metricDetails: props.latencyMetricDetails,
                metricType: LatencyMetricType_1.LatencyMetricType.FAULT_LATENCY,
                statistic: x,
                keyPrefix: keyPrefix,
            });
        });
        latencyMetrics = latencySuccessMetrics.concat(latencyFaultMetrics);
        if (latencyMetrics.length > 0) {
            latencyWidgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 8,
                width: 24,
                title: aws_cdk_lib_1.Fn.sub("${AWS::Region} Latency"),
                region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
                left: latencyMetrics,
                leftYAxis: {
                    max: props.latencyMetricDetails.successAlarmThreshold * 1.5,
                    min: 0,
                    label: "Latency",
                    showUnits: false,
                },
                leftAnnotations: [
                    {
                        value: props.latencyMetricDetails.successAlarmThreshold,
                        visible: true,
                        color: aws_cloudwatch_1.Color.RED,
                        label: "High Severity",
                    },
                ],
            }));
        }
        latencyWidgets.push(new aws_cloudwatch_1.AlarmWidget({
            height: 2,
            width: 24,
            region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
            alarm: props.regionalEndpointLatencyAlarm,
        }));
        keyPrefix = "";
        for (let i = 0; i < props.availabilityZoneIds.length; i++) {
            let availabilityZoneId = props.availabilityZoneIds[i];
            let latencyMetrics2 = [];
            let stats2 = props.latencyMetricDetails.graphedSuccessStatistics
                ? props.latencyMetricDetails.graphedSuccessStatistics
                : ["p99"];
            let zonalSuccessLatencyMetrics = stats2.map((x) => {
                keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar(keyPrefix);
                return ZonalLatencyMetrics_1.ZonalLatencyMetrics.createZonalAverageLatencyMetric({
                    label: x + " Success Latency",
                    metricDetails: props.latencyMetricDetails,
                    metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
                    statistic: x,
                    availabilityZoneId: availabilityZoneId,
                    keyPrefix: keyPrefix,
                });
            });
            stats2 = props.latencyMetricDetails.graphedFaultStatistics
                ? props.latencyMetricDetails.graphedFaultStatistics
                : ["p99"];
            let zonalFaultLatencyMetrics = stats2.map((x) => {
                keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar(keyPrefix);
                return ZonalLatencyMetrics_1.ZonalLatencyMetrics.createZonalAverageLatencyMetric({
                    label: x + " Fault Latency",
                    metricDetails: props.latencyMetricDetails,
                    metricType: LatencyMetricType_1.LatencyMetricType.FAULT_LATENCY,
                    statistic: x,
                    availabilityZoneId: availabilityZoneId,
                    keyPrefix: keyPrefix,
                });
            });
            latencyMetrics2 = zonalSuccessLatencyMetrics.concat(zonalFaultLatencyMetrics);
            if (latencyMetrics2.length > 0) {
                latencyWidgets.push(new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 8,
                    title: availabilityZoneId + " Latency",
                    region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
                    left: latencyMetrics2,
                    leftAnnotations: [
                        {
                            value: props.latencyMetricDetails.successAlarmThreshold,
                            visible: true,
                            color: aws_cloudwatch_1.Color.RED,
                            label: "High Severity",
                        },
                    ],
                    leftYAxis: {
                        max: props.latencyMetricDetails.successAlarmThreshold * 1.5,
                        min: 0,
                        label: "Latency",
                        showUnits: false,
                    },
                }));
            }
            //We're on the third one for this set, add 3 alarms
            //or if we're at the end, at the necessary amount
            //of alarms, 1, 2, or 3
            if (i % 3 == 2 || i - 1 == props.availabilityZoneIds.length) {
                for (let k = rowTracker; k <= i; k++) {
                    latencyWidgets.push(new aws_cloudwatch_1.AlarmWidget({
                        height: 2,
                        width: 8,
                        region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
                        alarm: props.zonalEndpointLatencyAlarms[k],
                    }));
                }
                rowTracker += i + 1;
            }
        }
        if (!props.isCanary &&
            props.instanceContributorsToHighLatency !== undefined &&
            props.instanceContributorsToHighLatency != null) {
            latencyWidgets.push(new ContributorInsightsWidget_1.ContributorInsightsWidget({
                height: 6,
                width: 24,
                title: "Individual Instance Contributors to High Latency",
                insightRule: props.instanceContributorsToHighLatency,
                period: props.latencyMetricDetails.period,
                legendPosition: aws_cloudwatch_1.LegendPosition.BOTTOM,
                orderStatistic: "Sum",
                accountId: aws_cdk_lib_1.Fn.ref("AWS::AccountId"),
                topContributors: 10,
            }));
        }
        return latencyWidgets;
    }
    static createApplicationLoadBalancerWidgets(props, title, availabilityZoneNames) {
        let albWidgets = [];
        let loadBalancerFullName = props.loadBalancer
            .loadBalancerFullName;
        albWidgets.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: title }));
        albWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 24,
            title: aws_cdk_lib_1.Fn.sub("${AWS::Region} Processed Bytes"),
            region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
            left: [
                ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.createRegionalApplicationLoadBalancerProcessedBytesMetric(loadBalancerFullName, props.operation.serverSideAvailabilityMetricDetails.period),
            ],
            leftYAxis: {
                label: "Processed Bytes",
                showUnits: true,
            },
        }));
        availabilityZoneNames.forEach((availabilityZoneName) => {
            let availabilityZoneId = props.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(availabilityZoneName.substring(availabilityZoneName.length - 1));
            albWidgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + " Processed Bytes",
                region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
                left: [
                    ApplicationLoadBalancerMetrics_1.ApplicationLoadBalancerMetrics.createZonalApplicationLoadBalancerProcessedBytesMetric(loadBalancerFullName, availabilityZoneName, props.operation.serverSideAvailabilityMetricDetails.period),
                ],
                leftYAxis: {
                    label: "Processed Bytes",
                    showUnits: true,
                },
            }));
        });
        return albWidgets;
    }
    static createNetworkLoadBalancerWidgets(props, title, availabilityZoneNames) {
        let nlbWidgets = [];
        let loadBalancerFullName = props.loadBalancer
            .loadBalancerFullName;
        nlbWidgets.push(new aws_cloudwatch_1.TextWidget({ height: 2, width: 24, markdown: title }));
        nlbWidgets.push(new aws_cloudwatch_1.GraphWidget({
            height: 8,
            width: 24,
            title: aws_cdk_lib_1.Fn.sub("${AWS::Region} Processed Bytes"),
            region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
            left: [
                NetworkLoadBalancerMetrics_1.NetworkLoadBalancerMetrics.createRegionalNetworkLoadBalancerProcessedBytesMetric(loadBalancerFullName, props.operation.serverSideAvailabilityMetricDetails.period),
            ],
            leftYAxis: {
                label: "Processed Bytes",
                showUnits: true,
            },
        }));
        availabilityZoneNames.forEach((availabilityZoneName) => {
            let availabilityZoneId = props.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(availabilityZoneName.substring(availabilityZoneName.length - 1));
            nlbWidgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + " Processed Bytes",
                region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
                left: [
                    NetworkLoadBalancerMetrics_1.NetworkLoadBalancerMetrics.createZonalNetworkLoadBalancerProcessedBytesMetric(loadBalancerFullName, availabilityZoneName, props.operation.serverSideAvailabilityMetricDetails.period),
                ],
                leftYAxis: {
                    label: "Processed Bytes",
                    showUnits: true,
                },
            }));
        });
        return nlbWidgets;
    }
    constructor(scope, id, props) {
        super(scope, id);
        let widgets = [];
        this.azMapper = props.azMapper;
        let availabilityZoneIds = props.operation.service.availabilityZoneNames.map((x) => {
            return this.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(x.substring(x.length - 1));
        });
        widgets.push(OperationAvailabilityAndLatencyDashboard.createTopLevelAggregateAlarmWidgets(props, "**Top Level Aggregate Alarms**", availabilityZoneIds));
        widgets.push(OperationAvailabilityAndLatencyDashboard.createAvailabilityWidgets({
            operation: props.operation,
            availabilityMetricDetails: props.operation.serverSideAvailabilityMetricDetails,
            availabilityZoneIds: availabilityZoneIds,
            isCanary: false,
            zonalEndpointAvailabilityAlarms: props.zonalEndpointServerAvailabilityAlarms,
            regionalEndpointAvailabilityAlarm: props.regionalEndpointServerAvailabilityAlarm,
            instanceContributorsToFaults: props.instanceContributorsToFaults,
        }, "**Server-side Availability**"));
        widgets.push(OperationAvailabilityAndLatencyDashboard.createLatencyWidgets({
            operation: props.operation,
            latencyMetricDetails: props.operation.serverSideLatencyMetricDetails,
            availabilityZoneIds: availabilityZoneIds,
            isCanary: false,
            zonalEndpointLatencyAlarms: props.zonalEndpointServerLatencyAlarms,
            regionalEndpointLatencyAlarm: props.regionalEndpointServerLatencyAlarm,
            instanceContributorsToHighLatency: props.instanceContributorsToHighLatency,
        }, "**Server-side Latency**"));
        let lb = props.loadBalancer?.node
            .defaultChild;
        if (lb !== undefined && lb != null) {
            if (lb.type == "application") {
                widgets.push(OperationAvailabilityAndLatencyDashboard.createApplicationLoadBalancerWidgets(props, "**Application Load Balancer Metrics**", props.operation.service.availabilityZoneNames));
            }
            else if (lb.type == "network") {
                widgets.push(OperationAvailabilityAndLatencyDashboard.createNetworkLoadBalancerWidgets(props, "**Network Load Balancer Metrics**", props.operation.service.availabilityZoneNames));
            }
        }
        if (props.operation.canaryMetricDetails !== undefined &&
            props.operation.canaryMetricDetails != null) {
            if (props.zonalEndpointCanaryAvailabilityAlarms &&
                props.zonalEndpointCanaryLatencyAlarms &&
                props.regionalEndpointCanaryAvailabilityAlarm &&
                props.regionalEndpointCanaryLatencyAlarm) {
                widgets.push(OperationAvailabilityAndLatencyDashboard.createAvailabilityWidgets({
                    operation: props.operation,
                    availabilityMetricDetails: props.operation.canaryMetricDetails
                        .canaryAvailabilityMetricDetails,
                    availabilityZoneIds: availabilityZoneIds,
                    isCanary: true,
                    zonalEndpointAvailabilityAlarms: props.zonalEndpointCanaryAvailabilityAlarms,
                    regionalEndpointAvailabilityAlarm: props.regionalEndpointCanaryAvailabilityAlarm,
                }, "**Canary Measured Availability**"));
                widgets.push(OperationAvailabilityAndLatencyDashboard.createLatencyWidgets({
                    operation: props.operation,
                    latencyMetricDetails: props.operation.canaryMetricDetails.canaryLatencyMetricDetails,
                    availabilityZoneIds: availabilityZoneIds,
                    isCanary: true,
                    zonalEndpointLatencyAlarms: props.zonalEndpointCanaryLatencyAlarms,
                    regionalEndpointLatencyAlarm: props.regionalEndpointCanaryLatencyAlarm,
                }, "**Canary Measured Latency**"));
            }
        }
        this.dashboard = new aws_cloudwatch_1.Dashboard(this, "Dashboard", {
            dashboardName: props.operation.service.serviceName.toLowerCase() +
                "-" +
                props.operation.operationName.toLowerCase() +
                aws_cdk_lib_1.Fn.sub("-operation-availability-and-latency-${AWS::Region}"),
            defaultInterval: props.interval,
            periodOverride: aws_cloudwatch_1.PeriodOverride.INHERIT,
            widgets: widgets,
        });
    }
}
exports.OperationAvailabilityAndLatencyDashboard = OperationAvailabilityAndLatencyDashboard;
//# sourceMappingURL=data:application/json;base64,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