"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BasicServiceDashboard = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
class BasicServiceDashboard extends constructs_1.Construct {
    static createLoadBalancerWidgets(alarms, metrics) {
        let widgets = [];
        widgets.push(new aws_cloudwatch_1.TextWidget({
            markdown: "Load Balancer Fault Count Metrics",
            height: 2,
            width: 24,
        }));
        let rowTracker = 0;
        Object.keys(alarms).forEach((availabilityZoneId, index) => {
            widgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + " Load Balancer Faults",
                region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
                left: [metrics[availabilityZoneId]],
                statistic: "Sum",
                leftYAxis: {
                    min: 0,
                    label: "Fault Count",
                    showUnits: false,
                },
            }));
            //We're on the third one for this set, add 3 alarms
            //or if we're at the end, at the necessary amount
            //of alarms, 1, 2, or 3
            if (index % 3 == 2 || index - 1 == Object.keys(alarms).length) {
                for (let k = rowTracker; k <= index; k++) {
                    widgets.push(new aws_cloudwatch_1.AlarmWidget({
                        height: 2,
                        width: 8,
                        region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
                        alarm: alarms[availabilityZoneId],
                    }));
                }
                rowTracker += index + 1;
            }
        });
        return widgets;
    }
    static createNatGatewayWidgets(alarms, metrics) {
        let widgets = [];
        widgets.push(new aws_cloudwatch_1.TextWidget({
            markdown: "NAT Gateway Dropped Packet Metrics",
            height: 2,
            width: 24,
        }));
        let rowTracker = 0;
        Object.keys(alarms).forEach((availabilityZoneId, index) => {
            widgets.push(new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 8,
                title: availabilityZoneId + " NAT Gateway Dropped Packets",
                region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
                left: [metrics[availabilityZoneId]],
                statistic: "Sum",
                leftYAxis: {
                    min: 0,
                    label: "Dropped packets",
                    showUnits: false,
                },
            }));
            //We're on the third one for this set, add 3 alarms
            //or if we're at the end, at the necessary amount
            //of alarms, 1, 2, or 3
            if (index % 3 == 2 || index - 1 == Object.keys(alarms).length) {
                for (let k = rowTracker; k <= index; k++) {
                    widgets.push(new aws_cloudwatch_1.AlarmWidget({
                        height: 2,
                        width: 8,
                        region: aws_cdk_lib_1.Fn.sub("${AWS::Region}"),
                        alarm: alarms[availabilityZoneId],
                    }));
                }
                rowTracker += index + 1;
            }
        });
        return widgets;
    }
    static createTopLevelAlarmWidgets(alarms) {
        let widgets = [];
        widgets.push(new aws_cloudwatch_1.TextWidget({
            markdown: "Availability Zone Isolated Impact Alarms",
            height: 2,
            width: 24,
        }));
        Object.keys(alarms).forEach((availabilityZoneId) => {
            widgets.push(new aws_cloudwatch_1.AlarmStatusWidget({
                alarms: [alarms[availabilityZoneId]],
                height: 2,
                width: 8,
                title: availabilityZoneId + " Aggregate Isolated Impact",
            }));
        });
        return widgets;
    }
    constructor(scope, id, props) {
        super(scope, id);
        let widgets = [];
        widgets.push(BasicServiceDashboard.createTopLevelAlarmWidgets(props.zonalAggregateIsolatedImpactAlarms));
        if (props.zonalLoadBalancerIsolatedImpactAlarms !== undefined &&
            props.zonalLoadBalancerFaultRateMetrics !== undefined) {
            widgets.push(BasicServiceDashboard.createLoadBalancerWidgets(props.zonalLoadBalancerIsolatedImpactAlarms, props.zonalLoadBalancerFaultRateMetrics));
        }
        if (props.zonalNatGatewayIsolatedImpactAlarms !== undefined &&
            props.zonalNatGatewayPacketDropMetrics !== undefined) {
            widgets.push(BasicServiceDashboard.createNatGatewayWidgets(props.zonalNatGatewayIsolatedImpactAlarms, props.zonalNatGatewayPacketDropMetrics));
        }
        this.dashboard = new aws_cloudwatch_1.Dashboard(this, "BasicServiceDashboard", {
            dashboardName: props.serviceName.toLowerCase() +
                aws_cdk_lib_1.Fn.sub("-service-availability-${AWS::Region}"),
            defaultInterval: props.interval,
            periodOverride: aws_cloudwatch_1.PeriodOverride.INHERIT,
            widgets: widgets,
        });
    }
}
exports.BasicServiceDashboard = BasicServiceDashboard;
//# sourceMappingURL=data:application/json;base64,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