"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CanaryFunction = void 0;
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
class CanaryFunction extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        let xrayManagedPolicy = new aws_iam_1.ManagedPolicy(this, "xrayManagedPolicy", {
            path: "/canary/",
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords",
                        "xray:GetSamplingRules",
                        "xray:GetSamplingTargets",
                        "xray:GetSamplingStatisticSummaries",
                    ],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: ["*"],
                }),
            ],
        });
        let ec2ManagedPolicy = new aws_iam_1.ManagedPolicy(this, "ec2ManagedPolicy", {
            path: "/canary/",
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        "ec2:CreateNetworkInterface",
                        "ec2:DescribeNetworkInterfaces",
                        "ec2:DeleteNetworkInterface",
                    ],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: ["*"],
                }),
            ],
        });
        let executionRole = new aws_iam_1.Role(this, "executionRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            path: "/canary/",
            managedPolicies: [xrayManagedPolicy, ec2ManagedPolicy],
        });
        /*
        let code: AssetCode = Code.fromAsset(path.join(__dirname, "src/"), {
          bundling: {
            //image: new Runtime('python3.12:latest-arm64', RuntimeFamily.PYTHON).bundlingImage,
            image: Runtime.PYTHON_3_12.bundlingImage,
            command: [
              'bash', '-c',
              'pip install --no-cache -r requirements.txt -t /asset-output && cp --archive --update . /asset-output',
            ],
            platform: 'linux/arm64',
          },
        });
        */
        let monitoringLayer = new aws_lambda_1.LayerVersion(this, "MonitoringLayer", {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "../monitoring/src/monitoring-layer.zip")),
            compatibleArchitectures: [aws_lambda_1.Architecture.ARM_64],
            compatibleRuntimes: [aws_lambda_1.Runtime.PYTHON_3_12],
        });
        if (props.vpc !== undefined && props.vpc != null) {
            let sg = new aws_ec2_1.SecurityGroup(this, "canarySecurityGroup", {
                description: "Allow canary to communicate with load balancer",
                vpc: props.vpc,
                allowAllOutbound: true,
            });
            this.function = new aws_lambda_1.Function(this, "canary", {
                runtime: aws_lambda_1.Runtime.PYTHON_3_12,
                code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "src/canary.zip")),
                handler: "index.handler",
                role: executionRole,
                architecture: aws_lambda_1.Architecture.ARM_64,
                tracing: aws_lambda_1.Tracing.ACTIVE,
                timeout: aws_cdk_lib_1.Duration.seconds(240),
                memorySize: 512,
                layers: [monitoringLayer],
                environment: {
                    REGION: aws_cdk_lib_1.Fn.ref("AWS::Region"),
                    PARTITION: aws_cdk_lib_1.Fn.ref("AWS::Partition"),
                    TIMEOUT: props.httpTimeout !== undefined
                        ? props.httpTimeout.toSeconds().toString()
                        : "2",
                    IGNORE_SSL_ERRORS: (props.ignoreTlsErrors !== undefined && props.ignoreTlsErrors == true)
                        .toString()
                        .toLowerCase(),
                },
                vpc: props.vpc,
                securityGroups: [sg],
                vpcSubnets: props.subnetSelection,
            });
        }
        else {
            this.function = new aws_lambda_1.Function(this, "canary", {
                runtime: aws_lambda_1.Runtime.PYTHON_3_12,
                code: aws_lambda_1.Code.fromAsset(path.join(__dirname, "src/canary.zip")),
                handler: "index.handler",
                role: executionRole,
                architecture: aws_lambda_1.Architecture.ARM_64,
                tracing: aws_lambda_1.Tracing.ACTIVE,
                timeout: aws_cdk_lib_1.Duration.seconds(240),
                memorySize: 512,
                layers: [monitoringLayer],
                environment: {
                    REGION: aws_cdk_lib_1.Fn.ref("AWS::Region"),
                    PARTITION: aws_cdk_lib_1.Fn.ref("AWS::Partition"),
                    TIMEOUT: props.httpTimeout !== undefined
                        ? props.httpTimeout.toSeconds().toString()
                        : "2",
                    IGNORE_SSL_ERRORS: (props.ignoreTlsErrors !== undefined && props.ignoreTlsErrors == true)
                        .toString()
                        .toLowerCase(),
                },
            });
        }
        this.function.addPermission("invokePermission", {
            action: "lambda:InvokeFunction",
            principal: new aws_iam_1.ServicePrincipal("events.amazonaws.com"),
            sourceArn: aws_cdk_lib_1.Fn.sub("arn:${AWS::Partition}:events:${AWS::Region}:${AWS::AccountId}:rule/*"),
        });
        this.logGroup = new aws_logs_1.LogGroup(this, "logGroup", {
            logGroupName: `/aws/lambda/${this.function.functionName}`,
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        new aws_iam_1.ManagedPolicy(this, "cwManagedPolicy", {
            path: "/canary/",
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: ["cloudwatch:PutMetricData"],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: ["*"],
                }),
                new aws_iam_1.PolicyStatement({
                    actions: ["logs:CreateLogStream", "logs:PutLogEvents"],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: [this.logGroup.logGroupArn],
                }),
            ],
            roles: [executionRole],
        });
    }
}
exports.CanaryFunction = CanaryFunction;
//# sourceMappingURL=data:application/json;base64,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