"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AvailabilityZoneMapper = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const constructs_1 = require("constructs");
/**
 * A construct that allows you to map AZ names to ids and back
 */
class AvailabilityZoneMapper extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /*
        let currentNode: Construct | undefined = this.node.scope;
    
        while (currentNode !== undefined && !(currentNode instanceof Stack)) {
          currentNode = currentNode.node.scope;
        }
    
        if (currentNode !== undefined) {
          (currentNode as Stack).addTransform('AWS::LanguageExtensions');
        }
        */
        let xrayManagedPolicy = new aws_iam_1.ManagedPolicy(this, "XrayManagedPolicy", {
            path: "/azmapper/",
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords",
                        "xray:GetSamplingRules",
                        "xray:GetSamplingTargets",
                        "xray:GetSamplingStatisticSummaries",
                    ],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: ["*"],
                }),
            ],
        });
        let ec2ManagedPolicy = new aws_iam_1.ManagedPolicy(this, "EC2ManagedPolicy", {
            path: "/azmapper/",
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: ["ec2:DescribeAvailabilityZones"],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: ["*"],
                }),
            ],
        });
        let executionRole = new aws_iam_1.Role(this, "executionRole", {
            assumedBy: new aws_iam_1.ServicePrincipal("lambda.amazonaws.com"),
            path: "/azmapper/",
            managedPolicies: [xrayManagedPolicy, ec2ManagedPolicy],
        });
        const file = fs.readFileSync(path.resolve(__dirname, "./../azmapper/src/index.py"), "utf-8");
        this.function = new aws_lambda_1.Function(this, "AvailabilityZoneMapperFunction", {
            runtime: aws_lambda_1.Runtime.PYTHON_3_12,
            code: aws_lambda_1.Code.fromInline(file),
            handler: "index.handler",
            role: executionRole,
            architecture: aws_lambda_1.Architecture.ARM_64,
            tracing: aws_lambda_1.Tracing.ACTIVE,
            timeout: aws_cdk_lib_1.Duration.seconds(20),
            memorySize: 512,
            environment: {
                REGION: aws_cdk_lib_1.Fn.ref("AWS::Region"),
                PARTITION: aws_cdk_lib_1.Fn.ref("AWS::Partition"),
            },
        });
        this.logGroup = new aws_logs_1.LogGroup(this, "LogGroup", {
            logGroupName: `/aws/lambda/${this.function.functionName}`,
            retention: aws_logs_1.RetentionDays.ONE_DAY,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        new aws_iam_1.ManagedPolicy(this, "CloudWatchManagedPolicy", {
            path: "/azmapper/",
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: ["cloudwatch:PutMetricData"],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: ["*"],
                }),
                new aws_iam_1.PolicyStatement({
                    actions: ["logs:CreateLogStream", "logs:PutLogEvents"],
                    effect: aws_iam_1.Effect.ALLOW,
                    resources: [
                        aws_cdk_lib_1.Fn.sub("arn:${AWS::Partition}:logs:${AWS::Region}:${AWS::AccountId}:log-group:") +
                            this.logGroup.logGroupName +
                            ":*",
                    ],
                }),
            ],
            roles: [executionRole],
        });
        this.mapper = new aws_cdk_lib_1.CustomResource(this, "AvailabilityZoneMapper", {
            serviceToken: this.function.functionArn,
            properties: props?.availabilityZoneNames !== undefined
                ? { AvailabilityZones: props.availabilityZoneNames }
                : {},
        });
    }
    /**
     * Gets the Availability Zone Id for the given Availability Zone Name in this account
     * @param availabilityZoneName
     * @returns
     */
    availabilityZoneId(availabilityZoneName) {
        return this.mapper.getAttString(availabilityZoneName);
    }
    /**
     * Gets the Availability Zone Name for the given Availability Zone Id in this account
     * @param availabilityZoneId
     * @returns
     */
    availabilityZoneName(availabilityZoneId) {
        return this.mapper.getAttString(availabilityZoneId);
    }
    /**
     * Gets the prefix for the region used with Availability Zone Ids, for example
     * in us-east-1, this returns "use1"
     * @returns
     */
    regionPrefixForAvailabilityZoneIds() {
        return this.mapper.getAttString(aws_cdk_lib_1.Fn.ref("AWS::Region"));
    }
    /**
     * Returns an array for Availability Zone Ids for the supplied Availability Zone names,
     * they are returned in the same order the names were provided
     * @param availabilityZoneNames
     * @returns
     */
    availabilityZoneIdsAsArray(availabilityZoneNames) {
        let ids = [];
        for (let i = 0; i < availabilityZoneNames.length; i++) {
            ids.push(this.availabilityZoneId(availabilityZoneNames[i]));
        }
        return ids;
    }
    /**
     * Returns a comma delimited list of Availability Zone Ids for the supplied
     * Availability Zone names. You can use this string with Fn.Select(x, Fn.Split(",", azs)) to
     * get a specific Availability Zone Id
     * @param availabilityZoneNames
     * @returns
     */
    availabilityZoneIdsAsCommaDelimitedList(availabilityZoneNames) {
        let ids = [];
        for (let i = 0; i < availabilityZoneNames.length; i++) {
            ids.push(this.availabilityZoneId(availabilityZoneNames[i]));
        }
        return ids.join(",");
    }
    /**
     * Returns a comma delimited list of Availability Zone Ids for the supplied
     * Availability Zone names. You can use this string with Fn.Select(x, Fn.Split(",", azs)) to
     * get a specific Availability Zone Id
     * @returns
     */
    allAvailabilityZoneIdsAsCommaDelimitedList() {
        return this.mapper.getAttString("AllAvailabilityZoneIds");
    }
    /**
     * Returns a reference that can be cast to a string array with all of the
     * Availability Zone Ids
     * @returns
     */
    allAvailabilityZoneIdsAsArray() {
        return this.mapper.getAtt("AllAvailabilityZoneIdsArray");
    }
    /**
     * Given a letter like "f" or "a", returns the Availability Zone Id for that
     * Availability Zone name in this account
     * @param letter
     * @returns
     */
    availabilityZoneIdFromAvailabilityZoneLetter(letter) {
        return this.mapper.getAttString(letter);
    }
    /**
     * Gets all of the Availability Zone names in this Region as a comma delimited
     * list. You can use this string with Fn.Select(x, Fn.Split(",", azs)) to
     * get a specific Availability Zone Name
     * @returns
     */
    allAvailabilityZoneNamesAsCommaDelimitedList() {
        return this.mapper.getAttString("AllAvailabilityZoneNames");
    }
}
exports.AvailabilityZoneMapper = AvailabilityZoneMapper;
_a = JSII_RTTI_SYMBOL_1;
AvailabilityZoneMapper[_a] = { fqn: "multi-az-observability.AvailabilityZoneMapper", version: "0.0.1-alpha.10" };
//# sourceMappingURL=data:application/json;base64,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