"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceAlarmsAndRules = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const AvailabilityAndLatencyMetrics_1 = require("../metrics/AvailabilityAndLatencyMetrics");
const RegionalAvailabilityMetrics_1 = require("../metrics/RegionalAvailabilityMetrics");
const AvailabilityMetricType_1 = require("../utilities/AvailabilityMetricType");
/**
 * Service level alarms and rules using critical operations
 */
class ServiceAlarmsAndRules extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.service = props.service;
        let criticalOperations = props.service.operations
            .filter((x) => x.critical == true)
            .map((x) => x.operationName);
        let counter = 1;
        this.zonalAggregateIsolatedImpactAlarms = [];
        this.zonalServerSideIsolatedImpactAlarms = [];
        let availabilityZoneIds = props.service.availabilityZoneNames.map((x) => {
            return props.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(x.substring(x.length - 1));
        });
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let availabilityZonedId = availabilityZoneIds[i];
            this.zonalAggregateIsolatedImpactAlarms.push(new aws_cloudwatch_1.CompositeAlarm(this, "AZ" + counter + "ServiceAggregateIsolatedImpactAlarm", {
                compositeAlarmName: availabilityZonedId +
                    "-" +
                    props.service.serviceName.toLowerCase() +
                    "-isolated-impact-aggregate-alarm",
                alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...Object.values(Object.entries(props.perOperationAlarmsAndRules).reduce((filtered, [key, value]) => {
                    if (criticalOperations.indexOf(key) > -1) {
                        filtered[key] = value;
                    }
                    return filtered;
                }, {})).map((x) => x.aggregateZonalAlarms[i])),
            }));
            this.zonalServerSideIsolatedImpactAlarms.push(new aws_cloudwatch_1.CompositeAlarm(this, "AZ" + counter + "ServiceServerSideIsolatedImpactAlarm", {
                compositeAlarmName: availabilityZonedId +
                    "-" +
                    props.service.serviceName.toLowerCase() +
                    "-isolated-impact-server-side-alarm",
                alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...Object.values(Object.entries(props.perOperationAlarmsAndRules).reduce((filtered, [key, value]) => {
                    if (criticalOperations.indexOf(key) > -1) {
                        filtered[key] = value;
                    }
                    return filtered;
                }, {})).map((x) => x.serverSideZonalAlarmsMap[availabilityZonedId])),
            }));
            counter++;
        }
        let keyPrefix = "";
        let regionalOperationFaultCountMetrics = {};
        props.service.operations
            .filter((x) => x.critical == true)
            .forEach((x) => {
            keyPrefix = AvailabilityAndLatencyMetrics_1.AvailabilityAndLatencyMetrics.nextChar(keyPrefix);
            regionalOperationFaultCountMetrics[keyPrefix] =
                RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalAvailabilityMetric({
                    label: x.operationName + " fault count",
                    metricDetails: x.serverSideAvailabilityMetricDetails,
                    metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
                });
        });
        let regionalFaultCount = new aws_cloudwatch_1.MathExpression({
            usingMetrics: regionalOperationFaultCountMetrics,
            expression: Object.keys(regionalOperationFaultCountMetrics).join("+"),
            label: props.service.serviceName + " fault count",
            period: props.service.period,
        });
        this.regionalFaultCountServerSideAlarm = new aws_cloudwatch_1.Alarm(this, "RegionalFaultCount", {
            alarmName: aws_cdk_lib_1.Fn.ref("AWS::Region") +
                "-" +
                props.service.serviceName.toLowerCase() +
                "-fault-count",
            datapointsToAlarm: 3,
            evaluationPeriods: 5,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            threshold: props.service.faultCountThreshold,
            alarmDescription: "Counts faults from all critical operation in the service",
            metric: regionalFaultCount,
        });
        let canaryAlarms = Object.values(Object.entries(props.perOperationAlarmsAndRules).reduce((filtered, [key, value]) => {
            if (criticalOperations.indexOf(key) > -1) {
                filtered[key] = value;
            }
            return filtered;
        }, {}))
            .reduce((filtered, value) => {
            if (value.canaryRegionalAlarmsAndRules !== undefined) {
                filtered.push(value.canaryRegionalAlarmsAndRules);
            }
            return filtered;
        }, [])
            .map((x) => x.availabilityOrLatencyAlarm);
        if (canaryAlarms !== undefined &&
            canaryAlarms !== null &&
            canaryAlarms.length > 0) {
            this.regionalAvailabilityOrLatencyCanaryAlarm = new aws_cloudwatch_1.CompositeAlarm(this, "ServiceCanaryAvailabilityOrLatencyAggregateAlarm", {
                compositeAlarmName: aws_cdk_lib_1.Fn.ref("AWS::Region") +
                    "-" +
                    props.service.serviceName.toLowerCase() +
                    "-canary-availability-or-latency-aggregate-alarm",
                alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...canaryAlarms),
            });
        }
        let canaryAvailabilityAlarms = Object.values(Object.entries(props.perOperationAlarmsAndRules).reduce((filtered, [key, value]) => {
            if (criticalOperations.indexOf(key) > -1) {
                filtered[key] = value;
            }
            return filtered;
        }, {}))
            .reduce((filtered, value) => {
            if (value.canaryRegionalAlarmsAndRules !== undefined) {
                filtered.push(value.canaryRegionalAlarmsAndRules);
            }
            return filtered;
        }, [])
            .map((x) => x.availabilityAlarm);
        if (canaryAvailabilityAlarms !== undefined &&
            canaryAvailabilityAlarms !== null &&
            canaryAvailabilityAlarms.length > 0) {
            this.regionalAvailabilityCanaryAlarm = new aws_cloudwatch_1.CompositeAlarm(this, "ServiceCanaryAvailabilityAggregateAlarm", {
                compositeAlarmName: aws_cdk_lib_1.Fn.ref("AWS::Region") +
                    "-" +
                    props.service.serviceName.toLowerCase() +
                    "-canary-availability-aggregate-alarm",
                alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...canaryAvailabilityAlarms),
            });
        }
        this.regionalAvailabilityOrLatencyServerSideAlarm = new aws_cloudwatch_1.CompositeAlarm(this, "ServiceServerSideAggregateIsolatedImpactAlarm", {
            compositeAlarmName: aws_cdk_lib_1.Fn.ref("AWS::Region") +
                "-" +
                props.service.serviceName.toLowerCase() +
                "-server-side-aggregate-alarm",
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...Object.values(Object.entries(props.perOperationAlarmsAndRules).reduce((filtered, [key, value]) => {
                if (criticalOperations.indexOf(key) > -1) {
                    filtered[key] = value;
                }
                return filtered;
            }, {}))
                .map((x) => x.serverSideRegionalAlarmsAndRules)
                .map((x) => x.availabilityOrLatencyAlarm)),
        });
        this.regionalAvailabilityServerSideAlarm = new aws_cloudwatch_1.CompositeAlarm(this, "ServiceServerSideAvailabilityAlarm", {
            compositeAlarmName: aws_cdk_lib_1.Fn.ref("AWS::Region") +
                "-" +
                props.service.serviceName.toLowerCase() +
                "-server-side-availability-alarm",
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(...Object.values(Object.entries(props.perOperationAlarmsAndRules).reduce((filtered, [key, value]) => {
                if (criticalOperations.indexOf(key) > -1) {
                    filtered[key] = value;
                }
                return filtered;
            }, {}))
                .map((x) => x.serverSideRegionalAlarmsAndRules)
                .map((x) => x.availabilityAlarm)),
        });
    }
}
exports.ServiceAlarmsAndRules = ServiceAlarmsAndRules;
//# sourceMappingURL=data:application/json;base64,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