"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OperationAlarmsAndRules = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const constructs_1 = require("constructs");
const CanaryOperationRegionalAlarmsAndRules_1 = require("./CanaryOperationRegionalAlarmsAndRules");
const CanaryOperationZonalAlarmsAndRules_1 = require("./CanaryOperationZonalAlarmsAndRules");
const ServerSideOperationRegionalAlarmsAndRules_1 = require("./ServerSideOperationRegionalAlarmsAndRules");
const ServerSideOperationZonalAlarmsAndRules_1 = require("./ServerSideOperationZonalAlarmsAndRules");
/**
 * Creates alarms and rules for an operation for both regional and zonal metrics
 */
class OperationAlarmsAndRules extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.serverSideZonalAlarmsAndRules = [];
        this.canaryZonalAlarmsAndRules = [];
        this.aggregateZonalAlarms = [];
        this.operation = props.operation;
        this.aggregateZonalAlarmsMap = {};
        this.serverSideZonalAlarmsMap = {};
        let availabilityZoneIds = props.operation.service.availabilityZoneNames.map((x) => {
            return props.azMapper.availabilityZoneIdFromAvailabilityZoneLetter(x.substring(x.length - 1));
        });
        let loadBalancerArn = "";
        if (props.loadBalancer !== undefined) {
            loadBalancerArn = props.loadBalancer
                .loadBalancerArn;
        }
        this.serverSideRegionalAlarmsAndRules =
            new ServerSideOperationRegionalAlarmsAndRules_1.ServerSideOperationRegionalAlarmsAndRules(this, props.operation.operationName + "ServerSideRegionalAlarms", {
                availabilityMetricDetails: props.operation.serverSideAvailabilityMetricDetails,
                latencyMetricDetails: props.operation.serverSideLatencyMetricDetails,
                contributorInsightRuleDetails: props.operation
                    .serverSideContributorInsightRuleDetails
                    ? props.operation.serverSideContributorInsightRuleDetails
                    : props.operation.service.defaultContributorInsightRuleDetails,
                nameSuffix: "-server",
            });
        if (props.operation.canaryMetricDetails !== undefined &&
            props.operation.canaryMetricDetails != null) {
            this.canaryRegionalAlarmsAndRules =
                new CanaryOperationRegionalAlarmsAndRules_1.CanaryOperationRegionalAlarmsAndRules(this, props.operation.operationName + "CanaryRegionalAlarms", {
                    availabilityMetricDetails: props.operation.canaryMetricDetails
                        .canaryAvailabilityMetricDetails,
                    latencyMetricDetails: props.operation.canaryMetricDetails.canaryLatencyMetricDetails,
                    nameSuffix: "-canary",
                });
            this.aggregateRegionalAlarm = new aws_cloudwatch_1.CompositeAlarm(this, props.operation.operationName + "AggregateRegionalAlarm", {
                actionsEnabled: false,
                compositeAlarmName: aws_cdk_lib_1.Fn.ref("AWS::Region") +
                    "-" +
                    props.operation.operationName.toLowerCase() +
                    "-" +
                    "aggregate-alarm",
                alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(this.serverSideRegionalAlarmsAndRules.availabilityOrLatencyAlarm, this.canaryRegionalAlarmsAndRules.availabilityOrLatencyAlarm),
            });
        }
        else {
            this.aggregateRegionalAlarm =
                this.serverSideRegionalAlarmsAndRules.availabilityOrLatencyAlarm;
        }
        let counter = 1;
        for (let i = 0; i < availabilityZoneIds.length; i++) {
            let availabilityZoneId = availabilityZoneIds[i];
            this.serverSideZonalAlarmsAndRules.push(new ServerSideOperationZonalAlarmsAndRules_1.ServerSideOperationZonalAlarmsAndRules(this, props.operation.operationName +
                "AZ" +
                counter +
                "ServerSideZonalAlarmsAndRules", {
                availabilityZoneId: availabilityZoneId,
                availabilityMetricDetails: props.operation.serverSideAvailabilityMetricDetails,
                latencyMetricDetails: props.operation.serverSideLatencyMetricDetails,
                contributorInsightRuleDetails: props.operation
                    .serverSideContributorInsightRuleDetails
                    ? props.operation.serverSideContributorInsightRuleDetails
                    : props.operation.service.defaultContributorInsightRuleDetails,
                counter: counter,
                outlierThreshold: props.outlierThreshold,
                outlierDetectionAlgorithm: props.outlierDetectionAlgorithm,
                nameSuffix: "-server",
                operation: props.operation,
                azMapper: props.azMapper,
                outlierDetectionFunction: props.outlierDetectionFunction,
            }));
            this.serverSideZonalAlarmsMap[availabilityZoneId] =
                this.serverSideZonalAlarmsAndRules[i].isolatedImpactAlarm;
            if (props.operation.canaryMetricDetails !== undefined &&
                props.operation.canaryMetricDetails != null) {
                this.canaryZonalAlarmsAndRules.push(new CanaryOperationZonalAlarmsAndRules_1.CanaryOperationZonalAlarmsAndRules(this, props.operation.operationName +
                    "AZ" +
                    counter +
                    "CanaryZonalAlarmsAndRules", {
                    availabilityZoneId: availabilityZoneId,
                    availabilityMetricDetails: props.operation.canaryMetricDetails
                        .canaryAvailabilityMetricDetails,
                    latencyMetricDetails: props.operation.canaryMetricDetails.canaryLatencyMetricDetails,
                    counter: counter,
                    outlierThreshold: props.outlierThreshold,
                    outlierDetectionAlgorithm: props.outlierDetectionAlgorithm,
                    nameSuffix: "-canary",
                    operation: props.operation,
                    azMapper: props.azMapper,
                    outlierDetectionFunction: props.outlierDetectionFunction,
                }));
                this.aggregateZonalAlarms.push(new aws_cloudwatch_1.CompositeAlarm(this, props.operation.operationName +
                    "AZ" +
                    counter +
                    "AggregateZonalIsolatedImpactAlarm", {
                    compositeAlarmName: availabilityZoneId +
                        "-" +
                        props.operation.operationName.toLowerCase() +
                        "-aggregate-isolated-az-impact",
                    alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(this.canaryZonalAlarmsAndRules[i].isolatedImpactAlarm, this.serverSideZonalAlarmsAndRules[i].isolatedImpactAlarm),
                    actionsEnabled: false,
                    alarmDescription: '{"loadBalancer":"' +
                        loadBalancerArn +
                        '","az-id":"' +
                        availabilityZoneId +
                        '"}',
                }));
            }
            else {
                this.aggregateZonalAlarms.push(this.serverSideZonalAlarmsAndRules[i].isolatedImpactAlarm);
            }
            this.aggregateZonalAlarmsMap[availabilityZoneId] =
                this.aggregateZonalAlarms[-1];
            counter++;
        }
    }
}
exports.OperationAlarmsAndRules = OperationAlarmsAndRules;
//# sourceMappingURL=data:application/json;base64,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