"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AvailabilityAndLatencyAlarmsAndRules = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_cloudwatch_1 = require("aws-cdk-lib/aws-cloudwatch");
const InsightRuleBody_1 = require("./InsightRuleBody");
const RegionalAvailabilityMetrics_1 = require("../metrics/RegionalAvailabilityMetrics");
const RegionalLatencyMetrics_1 = require("../metrics/RegionalLatencyMetrics");
const ZonalAvailabilityMetrics_1 = require("../metrics/ZonalAvailabilityMetrics");
const ZonalLatencyMetrics_1 = require("../metrics/ZonalLatencyMetrics");
const AvailabilityMetricType_1 = require("../utilities/AvailabilityMetricType");
const LatencyMetricType_1 = require("../utilities/LatencyMetricType");
/**
 * Class used to create availability and latency alarms and Contributor Insight rules
 */
class AvailabilityAndLatencyAlarmsAndRules {
    /**
     * Creates a zonal availability alarm
     * @param scope
     * @param metricDetails
     * @param availabilityZoneId
     * @param nameSuffix
     * @param counter
     * @returns
     */
    static createZonalAvailabilityAlarm(scope, metricDetails, availabilityZoneId, counter, nameSuffix) {
        return new aws_cloudwatch_1.Alarm(scope, metricDetails.operationName + "AZ" + counter + "AvailabilityAlarm", {
            alarmName: availabilityZoneId +
                "-" +
                metricDetails.operationName.toLowerCase() +
                "-success-rate" +
                nameSuffix,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            threshold: metricDetails.successAlarmThreshold,
            actionsEnabled: false,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE,
            metric: ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
                availabilityZoneId: availabilityZoneId,
                label: availabilityZoneId + " availability",
                metricDetails: metricDetails,
                metricType: AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE,
            }),
        });
    }
    /**
     * Creates a zonal latency alarm
     * @param scope
     * @param metricDetails
     * @param availabilityZoneId
     * @param nameSuffix
     * @param counter
     * @returns
     */
    static createZonalLatencyAlarm(scope, metricDetails, availabilityZoneId, counter, nameSuffix) {
        return new aws_cloudwatch_1.Alarm(scope, metricDetails.operationName + "AZ" + counter + "LatencyAlarm", {
            alarmName: availabilityZoneId +
                "-" +
                metricDetails.operationName.toLowerCase() +
                "-success-latency" +
                nameSuffix,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            threshold: metricDetails.successAlarmThreshold,
            actionsEnabled: false,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE,
            metric: ZonalLatencyMetrics_1.ZonalLatencyMetrics.createZonalAverageLatencyMetric({
                availabilityZoneId: availabilityZoneId,
                label: availabilityZoneId +
                    " " +
                    metricDetails.alarmStatistic +
                    " latency",
                metricDetails: metricDetails,
                metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
                statistic: metricDetails.alarmStatistic,
            }),
        });
    }
    /**
     * Creates a composite alarm when either latency or availability is breached in the Availabiltiy Zone
     * @param scope
     * @param operation
     * @param availabilityZoneId
     * @param nameSuffix
     * @param counter
     * @param zonalAvailabilityAlarm
     * @param zonalLatencyAlarm
     * @returns
     */
    static createZonalAvailabilityOrLatencyCompositeAlarm(scope, operationName, availabilityZoneId, counter, zonalAvailabilityAlarm, zonalLatencyAlarm, nameSuffix) {
        return new aws_cloudwatch_1.CompositeAlarm(scope, "AZ" + counter + "ZonalImpactAlarm", {
            actionsEnabled: false,
            alarmDescription: availabilityZoneId +
                " has latency or availability impact. This does not indicate it is an outlier and shows isolated impact.",
            compositeAlarmName: availabilityZoneId +
                `-${operationName.toLowerCase()}-impact-aggregate-alarm` +
                nameSuffix,
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(zonalAvailabilityAlarm, zonalLatencyAlarm),
        });
    }
    /**
     * An alarm that compares error rate in this AZ to the overall region error based only on metric data
     * @param scope
     * @param metricDetails
     * @param availabilityZoneId
     * @param nameSuffix
     * @param counter
     * @param outlierThreshold
     * @returns
     */
    static createZonalFaultRateStaticOutlierAlarm(scope, metricDetails, availabilityZoneId, counter, outlierThreshold, nameSuffix) {
        let zonalFaults = ZonalAvailabilityMetrics_1.ZonalAvailabilityMetrics.createZonalAvailabilityMetric({
            availabilityZoneId: availabilityZoneId,
            metricDetails: metricDetails,
            metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
            keyPrefix: "a",
        });
        let regionalFaults = RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalAvailabilityMetric({
            metricDetails: metricDetails,
            metricType: AvailabilityMetricType_1.AvailabilityMetricType.FAULT_COUNT,
            keyPrefix: "b",
        });
        return new aws_cloudwatch_1.Alarm(scope, "AZ" + counter + "IsolatedImpactAlarmStatic", {
            alarmName: availabilityZoneId +
                `-${metricDetails.operationName.toLowerCase()}-static-majority-errors-impact` +
                nameSuffix,
            metric: new aws_cloudwatch_1.MathExpression({
                expression: "IF(m2 > 0, (m1 / m2), 0)",
                usingMetrics: {
                    m1: zonalFaults,
                    m2: regionalFaults,
                },
                period: metricDetails.period,
                label: availabilityZoneId + " percent faults",
            }),
            threshold: outlierThreshold,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
        });
    }
    static createZonalFaultRateOutlierAlarm(scope, metricDetails, availabilityZoneId, allAvailabilityZoneIds, outlierThreshold, outlierDetectionFunction, outlierDetectionAlgorithm, counter, nameSuffix) {
        let metricDimensions = {};
        allAvailabilityZoneIds.forEach((azId) => {
            metricDimensions[azId] = [
                metricDetails.metricDimensions.zonalDimensions(azId, aws_cdk_lib_1.Fn.ref("AWS::Region")),
            ];
        });
        let str = JSON.stringify(metricDimensions)
            .replace(/[\\]/g, "\\\\")
            .replace(/[\"]/g, '\\"')
            .replace(/[\/]/g, "\\/")
            .replace(/[\b]/g, "\\b")
            .replace(/[\f]/g, "\\f")
            .replace(/[\n]/g, "\\n")
            .replace(/[\r]/g, "\\r")
            .replace(/[\t]/g, "\\t");
        let outlierMetrics = new aws_cloudwatch_1.MathExpression({
            expression: `MAX(LAMBDA("${outlierDetectionFunction.functionName}",` +
                `"${outlierDetectionAlgorithm.toString()}",` +
                `"${outlierThreshold}",` +
                `"${availabilityZoneId}",` +
                `"${str}",` +
                `"${metricDetails.metricNamespace}",` +
                `"${metricDetails.faultMetricNames.join(":")}",` +
                '"Sum",' +
                '"Count"' +
                "))",
            period: aws_cdk_lib_1.Duration.seconds(60),
        });
        return new aws_cloudwatch_1.Alarm(scope, "AZ" + counter + "FaultIsolatedImpactAlarmOutlier", {
            alarmName: availabilityZoneId +
                `-${metricDetails.operationName.toLowerCase()}-majority-errors-impact` +
                nameSuffix,
            metric: outlierMetrics,
            threshold: 1,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
        });
    }
    static createZonalFaultRateOutlierAlarmForAlb(scope, loadBalancers, availabilityZoneId, outlierThreshold, outlierDetectionFunction, outlierDetectionAlgorithm, azMapper, counter, evaluationPeriods, datapointsToAlarm, nameSuffix) {
        let metricDimensions = {};
        loadBalancers.forEach((x) => {
            x.vpc?.availabilityZones.forEach((az) => {
                let azId = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(az.substring(az.length - 1));
                if (!(azId in metricDimensions)) {
                    metricDimensions[azId] = [];
                }
                metricDimensions[azId].push({
                    AvailabilityZone: az,
                    LoadBalancer: x
                        .loadBalancerFullName,
                });
            });
        });
        let str = JSON.stringify(metricDimensions)
            .replace(/[\\]/g, "\\\\")
            .replace(/[\"]/g, '\\"')
            .replace(/[\/]/g, "\\/")
            .replace(/[\b]/g, "\\b")
            .replace(/[\f]/g, "\\f")
            .replace(/[\n]/g, "\\n")
            .replace(/[\r]/g, "\\r")
            .replace(/[\t]/g, "\\t");
        let outlierMetrics = new aws_cloudwatch_1.MathExpression({
            expression: `MAX(LAMBDA("${outlierDetectionFunction.functionName}",` +
                `"${outlierDetectionAlgorithm.toString()}",` +
                `"${outlierThreshold}",` +
                `"${availabilityZoneId}",` +
                `"${str}",` +
                '"AWS/ApplicationELB",' +
                '"HTTPCode_ELB_5XX_Count:HTTPCode_Target_5XX_Count",' +
                '"Sum",' +
                '"Count"' +
                "))",
            period: aws_cdk_lib_1.Duration.seconds(60),
        });
        return new aws_cloudwatch_1.Alarm(scope, "AZ" + counter + "AlbIsolatedImpactAlarmOutlier", {
            alarmName: availabilityZoneId + "-alb-majority-errors-impact" + nameSuffix,
            metric: outlierMetrics,
            threshold: 1,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            evaluationPeriods: evaluationPeriods,
            datapointsToAlarm: datapointsToAlarm,
        });
    }
    static createZonalFaultRateOutlierAlarmForNatGW(scope, natGateways, availabilityZoneId, outlierThreshold, outlierDetectionFunction, outlierDetectionAlgorithm, azMapper, counter, evaluationPeriods, datapointsToAlarm, nameSuffix) {
        let metricDimensions = {};
        Object.keys(natGateways).forEach((az) => {
            let azId = azMapper.availabilityZoneIdFromAvailabilityZoneLetter(az.substring(az.length - 1));
            if (!(azId in metricDimensions)) {
                metricDimensions[azId] = [];
            }
            natGateways[az].forEach((natgw) => {
                metricDimensions[azId].push({
                    NatGatewayId: natgw.attrNatGatewayId,
                });
            });
        });
        let str = JSON.stringify(metricDimensions)
            .replace(/[\\]/g, "\\\\")
            .replace(/[\"]/g, '\\"')
            .replace(/[\/]/g, "\\/")
            .replace(/[\b]/g, "\\b")
            .replace(/[\f]/g, "\\f")
            .replace(/[\n]/g, "\\n")
            .replace(/[\r]/g, "\\r")
            .replace(/[\t]/g, "\\t");
        let outlierMetrics = new aws_cloudwatch_1.MathExpression({
            expression: `MAX(LAMBDA("${outlierDetectionFunction.functionName}",` +
                `"${outlierDetectionAlgorithm.toString()}",` +
                `"${outlierThreshold}",` +
                `"${availabilityZoneId}",` +
                `"${str}",` +
                '"AWS/NATGateway",' +
                '"PacketsDropCount",' +
                '"Sum",' +
                '"Count"' +
                "))",
            period: aws_cdk_lib_1.Duration.seconds(60),
        });
        return new aws_cloudwatch_1.Alarm(scope, "AZ" + counter + "NatGWIsolatedImpactAlarmOutlier", {
            alarmName: availabilityZoneId + "-nat-gw-majority-errors-impact" + nameSuffix,
            metric: outlierMetrics,
            threshold: 1,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            evaluationPeriods: evaluationPeriods,
            datapointsToAlarm: datapointsToAlarm,
        });
    }
    static createZonalHighLatencyOutlierAlarm(scope, metricDetails, availabilityZoneId, allAvailabilityZoneIds, outlierThreshold, outlierDetectionFunction, outlierDetectionAlgorithm, counter, nameSuffix) {
        let metricDimensions = {};
        allAvailabilityZoneIds.forEach((azId) => {
            metricDimensions[azId] = [
                metricDetails.metricDimensions.zonalDimensions(azId, aws_cdk_lib_1.Fn.ref("AWS::Region")),
            ];
        });
        let str = JSON.stringify(metricDimensions)
            .replace(/[\\]/g, "\\\\")
            .replace(/[\"]/g, '\\"')
            .replace(/[\/]/g, "\\/")
            .replace(/[\b]/g, "\\b")
            .replace(/[\f]/g, "\\f")
            .replace(/[\n]/g, "\\n")
            .replace(/[\r]/g, "\\r")
            .replace(/[\t]/g, "\\t");
        let outlierMetrics = new aws_cloudwatch_1.MathExpression({
            expression: `MAX(LAMBDA("${outlierDetectionFunction.functionName}",` +
                `"${outlierDetectionAlgorithm.toString()}",` +
                `"${outlierThreshold}",` +
                `"${availabilityZoneId}",` +
                `"${str}",` +
                `"${metricDetails.metricNamespace}",` +
                `"${metricDetails.successMetricNames.join(":")}",` +
                `"TC(${metricDetails.successAlarmThreshold}:)",` +
                '"Milliseconds"' +
                "))",
            period: aws_cdk_lib_1.Duration.seconds(60),
        });
        return new aws_cloudwatch_1.Alarm(scope, metricDetails.operationName +
            "AZ" +
            counter +
            "LatencyIsolatedImpactAlarmOutlier", {
            alarmName: availabilityZoneId +
                `-${metricDetails.operationName.toLowerCase()}-majority-high-latency-impact` +
                nameSuffix,
            metric: outlierMetrics,
            threshold: 1,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
        });
    }
    static createZonalHighLatencyStaticOutlierAlarm(scope, metricDetails, availabilityZoneId, counter, outlierThreshold, nameSuffix) {
        let zonalLatency = ZonalLatencyMetrics_1.ZonalLatencyMetrics.createZonalCountLatencyMetric({
            availabilityZoneId: availabilityZoneId,
            label: availabilityZoneId +
                "-" +
                metricDetails.operationName +
                "-high-latency-requests",
            metricDetails: metricDetails,
            metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
            statistic: `TC(${metricDetails.successAlarmThreshold}:)`,
            keyPrefix: "a",
        });
        let regionalLatency = RegionalLatencyMetrics_1.RegionalLatencyMetrics.createRegionalLatencyCountMetric({
            label: aws_cdk_lib_1.Fn.ref("AWS::Region") +
                "-" +
                metricDetails.operationName +
                "-high-latency-requests",
            metricDetails: metricDetails,
            metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
            statistic: `TC(${metricDetails.successAlarmThreshold}:)`,
            keyPrefix: "b",
        });
        return new aws_cloudwatch_1.Alarm(scope, metricDetails.operationName +
            "AZ" +
            counter +
            "IsolatedImpactAlarmStatic", {
            alarmName: availabilityZoneId +
                `-${metricDetails.operationName.toLowerCase()}-static-majority-high-latency-impact` +
                nameSuffix,
            metric: new aws_cloudwatch_1.MathExpression({
                expression: "IF(m2 > 0, (m1 / m2), 0)",
                usingMetrics: {
                    m1: zonalLatency,
                    m2: regionalLatency,
                },
                period: metricDetails.period,
                label: availabilityZoneId + " percent high latency requests",
            }),
            threshold: outlierThreshold,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
        });
    }
    /**
     * An insight rule that calculates how many instances are responding to requests in
     * the specified AZ. Only useful for server-side metrics since the canary doesn't record instance id metrics.
     * @param scope
     * @param metricDetails
     * @param availabilityZoneId
     * @param logGroups
     * @param nameSuffix
     * @param counter
     * @param instanceIdPath
     * @param operationNamePath
     * @param availabilityZoneIdPath
     * @returns
     */
    static createServerSideInstancesHandlingRequestsInThisAZRule(scope, operationName, availabilityZoneId, ruleDetails, counter, nameSuffix) {
        let ruleBody = new InsightRuleBody_1.InsightRuleBody();
        ruleBody.logGroupNames = ruleDetails.logGroups.map((x) => x.logGroupName);
        ruleBody.aggregateOn = "Count";
        ruleBody.logFormat = "JSON";
        ruleBody.contribution = {
            keys: [ruleDetails.instanceIdJsonPath],
            filters: [
                {
                    Match: ruleDetails.availabilityZoneIdJsonPath,
                    In: [availabilityZoneId],
                },
                {
                    Match: ruleDetails.operationNameJsonPath,
                    In: [operationName],
                },
            ],
        };
        return new aws_cloudwatch_1.CfnInsightRule(scope, "AZ" + counter + "InstancesInTheAZRule", {
            ruleName: availabilityZoneId +
                `-${operationName.toLowerCase()}-instances-in-the-az` +
                nameSuffix,
            ruleState: "ENABLED",
            ruleBody: ruleBody.toJson(),
        });
    }
    /**
     * An insight rule that calculates the instances contributing to errors
     * in this AZ. Only useful for server-side metrics since the canary doesn't record instance id metrics.
     * @param scope
     * @param operation
     * @param availabilityZoneId
     * @param logGroups
     * @param nameSuffix
     * @param counter
     * @param instanceIdPath
     * @param operationNamePath
     * @param availabilityZoneIdPath
     * @param errorMetricPath
     * @returns
     */
    static createServerSideInstanceFaultContributorsInThisAZRule(scope, operationName, availabilityZoneId, ruleDetails, counter, nameSuffix) {
        let ruleBody = new InsightRuleBody_1.InsightRuleBody();
        ruleBody.logGroupNames = ruleDetails.logGroups.map((x) => x.logGroupName);
        ruleBody.aggregateOn = "Count";
        ruleBody.logFormat = "JSON";
        ruleBody.contribution = {
            keys: [ruleDetails.instanceIdJsonPath],
            filters: [
                {
                    Match: ruleDetails.availabilityZoneIdJsonPath,
                    In: [availabilityZoneId],
                },
                {
                    Match: ruleDetails.operationNameJsonPath,
                    In: [operationName],
                },
                {
                    Match: ruleDetails.faultMetricJsonPath,
                    GreaterThan: 0,
                },
            ],
        };
        return new aws_cloudwatch_1.CfnInsightRule(scope, "AZ" + counter + "InstanceErrorContributionRule", {
            ruleName: availabilityZoneId +
                `-${operationName.toLowerCase()}-per-instance-faults` +
                nameSuffix,
            ruleState: "ENABLED",
            ruleBody: ruleBody.toJson(),
        });
    }
    /**
     * An insight rule that calculates instances contributing to high latency in this AZ. Only
     * useful for server-side metrics since the canary doesn't record instance id metrics.
     * @param scope
     * @param metricDetails
     * @param availabilityZoneId
     * @param logGroups
     * @param nameSuffix
     * @param counter
     * @returns
     */
    static createServerSideInstanceHighLatencyContributorsInThisAZRule(scope, metricDetails, availabilityZoneId, ruleDetails, counter, nameSuffix) {
        let ruleBody = new InsightRuleBody_1.InsightRuleBody();
        ruleBody.logGroupNames = ruleDetails.logGroups.map((x) => x.logGroupName);
        ruleBody.aggregateOn = "Count";
        ruleBody.logFormat = "JSON";
        ruleBody.contribution = {
            keys: [ruleDetails.instanceIdJsonPath],
            filters: [
                {
                    Match: ruleDetails.availabilityZoneIdJsonPath,
                    In: [availabilityZoneId],
                },
                {
                    Match: ruleDetails.operationNameJsonPath,
                    In: [metricDetails.operationName],
                },
                {
                    Match: ruleDetails.successLatencyMetricJsonPath,
                    GreaterThan: metricDetails.successAlarmThreshold,
                },
            ],
        };
        return new aws_cloudwatch_1.CfnInsightRule(scope, "AZ" + counter + "LatencyContributorsRule", {
            ruleName: availabilityZoneId +
                `-${metricDetails.operationName.toLowerCase()}-per-instance-high-latency` +
                nameSuffix,
            ruleState: "ENABLED",
            ruleBody: ruleBody.toJson(),
        });
    }
    /**
     * An alarm that indicates some percentage of the instances in this AZ are producing errors. Only
     * useful for server-side metrics since the canary doesn't record instance id metrics.
     * @param scope
     * @param metricDetails
     * @param availabilityZoneId
     * @param nameSuffix
     * @param counter
     * @param outlierThreshold
     * @param instanceFaultRateContributorsInThisAZ
     * @param instancesHandlingRequestsInThisAZ
     * @returns
     */
    static createServerSideZonalMoreThanOneInstanceProducingFaultsAlarm(scope, metricDetails, availabilityZoneId, counter, outlierThreshold, instanceFaultRateContributorsInThisAZ, instancesHandlingRequestsInThisAZ, nameSuffix) {
        return new aws_cloudwatch_1.Alarm(scope, "AZ" + counter + "MoreThanOneAlarmForErrors", {
            alarmName: availabilityZoneId +
                `-${metricDetails.operationName.toLowerCase()}-multiple-instances-faults` +
                nameSuffix,
            metric: new aws_cloudwatch_1.MathExpression({
                expression: `INSIGHT_RULE_METRIC(\"${instanceFaultRateContributorsInThisAZ.attrRuleName}\", \"UniqueContributors\") / INSIGHT_RULE_METRIC(\"${instancesHandlingRequestsInThisAZ.attrRuleName}\", \"UniqueContributors\")`,
                period: metricDetails.period,
            }),
            evaluationPeriods: metricDetails.evaluationPeriods,
            threshold: outlierThreshold,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
            actionsEnabled: false,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE,
        });
    }
    /**
     * An alarm indicating more than some percentage of instances in this AZ
     * are contributing to high latency. Only useful for server-side metrics since
     * the canary doesn't record instance id metrics.
     * @param scope
     * @param metricDetails
     * @param availabilityZoneId
     * @param nameSuffix
     * @param counter
     * @param outlierThreshold
     * @param instanceHighLatencyContributorsInThisAZ
     * @param instancesHandlingRequestsInThisAZ
     * @returns
     */
    static createServerSideZonalMoreThanOneInstanceProducingHighLatencyAlarm(scope, metricDetails, availabilityZoneId, counter, outlierThreshold, instanceHighLatencyContributorsInThisAZ, instancesHandlingRequestsInThisAZ, nameSuffix) {
        return new aws_cloudwatch_1.Alarm(scope, "AZ" + counter + "MoreThanOneAlarmForHighLatency", {
            alarmName: availabilityZoneId +
                `-${metricDetails.operationName.toLowerCase()}-multiple-instances-high-latency` +
                nameSuffix,
            metric: new aws_cloudwatch_1.MathExpression({
                expression: `INSIGHT_RULE_METRIC(\"${instanceHighLatencyContributorsInThisAZ.attrRuleName}\", \"UniqueContributors\") / INSIGHT_RULE_METRIC(\"${instancesHandlingRequestsInThisAZ.attrRuleName}\", \"UniqueContributors\")`,
                period: metricDetails.period,
            }),
            evaluationPeriods: metricDetails.evaluationPeriods,
            threshold: outlierThreshold,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
            actionsEnabled: false,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE,
        });
    }
    /**
     * An alarm that indicates this AZ as an outlier
     * for availability or latency. This does not ensure that the errors
     * or latency originate from more than one instance.
     * @param scope
     * @param operation
     * @param availabilityZoneId
     * @param logGroups
     * @param nameSuffix
     * @param counter
     * @param azIsOutlierForFaultsAlarm
     * @param availabilityImpactAlarm
     * @param azIsOutlierForLatencyAlarm
     * @param latencyImpactAlarm
     * @returns
     */
    static createCanaryIsolatedAZImpactAlarm(scope, operationName, availabilityZoneId, counter, azIsOutlierForFaultsAlarm, availabilityImpactAlarm, azIsOutlierForLatencyAlarm, latencyImpactAlarm, nameSuffix) {
        return new aws_cloudwatch_1.CompositeAlarm(scope, operationName + "AZ" + counter + "IsolatedImpactAlarm" + nameSuffix, {
            compositeAlarmName: availabilityZoneId +
                `-${operationName.toLowerCase()}-isolated-impact-alarm` +
                nameSuffix,
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(aws_cloudwatch_1.AlarmRule.allOf(azIsOutlierForFaultsAlarm, availabilityImpactAlarm), aws_cloudwatch_1.AlarmRule.allOf(azIsOutlierForLatencyAlarm, latencyImpactAlarm)),
            actionsEnabled: false,
        });
    }
    /**
     * Creates the server side alarm to identify isolated single AZ
     * impact meaning that this one AZ is affected and the others aren't
     * @param scope
     * @param operation
     * @param availabilityZoneId
     * @param nameSuffix
     * @param counter
     * @param azIsOutlierForFaultsAlarm
     * @param availabilityImpactAlarm
     * @param moreThanOneInstanceContributingToFaults
     * @param azIsOutlierForLatencyAlarm
     * @param latencyImpactAlarm
     * @param moreThanOneInstanceContributingToLatency
     * @returns
     */
    static createServerSideIsolatedAZImpactAlarm(scope, operationName, availabilityZoneId, counter, azIsOutlierForFaultsAlarm, availabilityImpactAlarm, moreThanOneInstanceContributingToFaults, azIsOutlierForLatencyAlarm, latencyImpactAlarm, moreThanOneInstanceContributingToLatency, nameSuffix) {
        return new aws_cloudwatch_1.CompositeAlarm(scope, operationName + "AZ" + counter + "IsolatedImpactAlarm" + nameSuffix, {
            compositeAlarmName: availabilityZoneId +
                `-${operationName.toLowerCase()}-isolated-impact-alarm` +
                nameSuffix,
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(moreThanOneInstanceContributingToFaults === undefined ||
                moreThanOneInstanceContributingToFaults == null
                ? aws_cloudwatch_1.AlarmRule.allOf(azIsOutlierForFaultsAlarm, availabilityImpactAlarm)
                : aws_cloudwatch_1.AlarmRule.allOf(azIsOutlierForFaultsAlarm, availabilityImpactAlarm, moreThanOneInstanceContributingToFaults), moreThanOneInstanceContributingToLatency === undefined ||
                moreThanOneInstanceContributingToLatency == null
                ? aws_cloudwatch_1.AlarmRule.allOf(azIsOutlierForLatencyAlarm, latencyImpactAlarm)
                : aws_cloudwatch_1.AlarmRule.allOf(azIsOutlierForLatencyAlarm, latencyImpactAlarm, moreThanOneInstanceContributingToLatency)),
            actionsEnabled: false,
        });
    }
    /**
     * Creates an alarm that fires if either the canary or the
     * server side detect single AZ isolated impact
     * @param scope
     * @param operation
     * @param availabilityZoneId
     * @param counter
     * @param serverSideAlarm
     * @param canaryAlarm
     * @returns
     */
    static createAggregateIsolatedAZImpactAlarm(scope, operation, availabilityZoneId, counter, serverSideAlarm, canaryAlarm) {
        return new aws_cloudwatch_1.CompositeAlarm(scope, operation.operationName + "AZ" + counter + "AggregateIsolatedImpactAlarm", {
            compositeAlarmName: availabilityZoneId +
                `-${operation.operationName.toLowerCase()}-aggregate-isolated-impact-alarm`,
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(serverSideAlarm, canaryAlarm),
            actionsEnabled: false,
        });
    }
    /**
     * Creates a regional availability alarm for the operation
     * @param scope
     * @param metricDetails
     * @param nameSuffix
     * @param counter
     * @returns
     */
    static createRegionalAvailabilityAlarm(scope, metricDetails, nameSuffix) {
        return new aws_cloudwatch_1.Alarm(scope, metricDetails.operationName + "RegionalAvailabilityAlarm", {
            alarmName: aws_cdk_lib_1.Fn.ref("AWS::Region") +
                "-" +
                metricDetails.operationName.toLowerCase() +
                "-success-rate" +
                nameSuffix,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            threshold: metricDetails.successAlarmThreshold,
            actionsEnabled: false,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE,
            metric: RegionalAvailabilityMetrics_1.RegionalAvailabilityMetrics.createRegionalAvailabilityMetric({
                label: aws_cdk_lib_1.Fn.ref("AWS::Region") + " availability",
                metricDetails: metricDetails,
                metricType: AvailabilityMetricType_1.AvailabilityMetricType.SUCCESS_RATE,
            }),
        });
    }
    /**
     * Creates a regional latency alarm for the operation
     * @param scope
     * @param metricDetails
     * @param nameSuffix
     * @param counter
     * @returns
     */
    static createRegionalLatencyAlarm(scope, metricDetails, nameSuffix) {
        return new aws_cloudwatch_1.Alarm(scope, metricDetails.operationName + "RegionalLatencyAlarm", {
            alarmName: aws_cdk_lib_1.Fn.ref("AWS::Region") +
                "-" +
                metricDetails.operationName.toLowerCase() +
                "-success-latency" +
                nameSuffix,
            evaluationPeriods: metricDetails.evaluationPeriods,
            datapointsToAlarm: metricDetails.datapointsToAlarm,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            threshold: metricDetails.successAlarmThreshold,
            actionsEnabled: false,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.IGNORE,
            metric: RegionalLatencyMetrics_1.RegionalLatencyMetrics.createRegionalAverageLatencyMetric({
                label: aws_cdk_lib_1.Fn.ref("AWS::Region") +
                    " " +
                    metricDetails.alarmStatistic +
                    " latency",
                metricDetails: metricDetails,
                metricType: LatencyMetricType_1.LatencyMetricType.SUCCESS_LATENCY,
                statistic: metricDetails.alarmStatistic,
            }),
        });
    }
    /**
     * A composite alarm combining latency and availability alarms for this operation in the region
     * as measured from either the server side or canary
     * @param scope
     * @param operation
     * @param nameSuffix
     * @param regionalAvailabilityAlarm
     * @param regionalLatencyAlarm
     * @returns
     */
    static createRegionalCustomerExperienceAlarm(scope, operationName, nameSuffix, regionalAvailabilityAlarm, regionalLatencyAlarm) {
        return new aws_cloudwatch_1.CompositeAlarm(scope, operationName + "RegionalCustomerExperienceAlarm", {
            compositeAlarmName: aws_cdk_lib_1.Fn.ref("AWS::Region") +
                "-" +
                operationName.toLowerCase() +
                "-customer-experience-imact" +
                nameSuffix,
            alarmRule: aws_cloudwatch_1.AlarmRule.anyOf(regionalAvailabilityAlarm, regionalLatencyAlarm),
        });
    }
    static createRegionalInstanceContributorsToHighLatency(scope, metricDetails, ruleDetails) {
        let ruleBody = new InsightRuleBody_1.InsightRuleBody();
        ruleBody.logGroupNames = ruleDetails.logGroups.map((x) => x.logGroupName);
        ruleBody.aggregateOn = "Count";
        ruleBody.logFormat = "JSON";
        ruleBody.contribution = {
            keys: [ruleDetails.instanceIdJsonPath],
            filters: [
                {
                    Match: ruleDetails.successLatencyMetricJsonPath,
                    GreaterThan: metricDetails.successAlarmThreshold,
                },
                {
                    Match: ruleDetails.operationNameJsonPath,
                    In: [metricDetails.operationName],
                },
            ],
        };
        return new aws_cloudwatch_1.CfnInsightRule(scope, "RegionPerInstanceHighLatencyRule", {
            ruleName: aws_cdk_lib_1.Fn.ref("AWS::Region") +
                `-${metricDetails.operationName.toLowerCase()}-per-instance-high-latency-server`,
            ruleState: "ENABLED",
            ruleBody: ruleBody.toJson(),
        });
    }
    static createRegionalInstanceContributorsToFaults(scope, metricDetails, ruleDetails) {
        let ruleBody = new InsightRuleBody_1.InsightRuleBody();
        ruleBody.logGroupNames = ruleDetails.logGroups.map((x) => x.logGroupName);
        ruleBody.aggregateOn = "Count";
        ruleBody.logFormat = "JSON";
        ruleBody.contribution = {
            keys: [ruleDetails.instanceIdJsonPath],
            filters: [
                {
                    Match: ruleDetails.faultMetricJsonPath,
                    GreaterThan: 0,
                },
                {
                    Match: ruleDetails.operationNameJsonPath,
                    In: [metricDetails.operationName],
                },
            ],
        };
        return new aws_cloudwatch_1.CfnInsightRule(scope, "RegionPerInstanceErrorRule", {
            ruleName: aws_cdk_lib_1.Fn.ref("AWS::Region") +
                `-${metricDetails.operationName.toLowerCase()}-per-instance-faults-server`,
            ruleState: "ENABLED",
            ruleBody: ruleBody.toJson(),
        });
    }
}
exports.AvailabilityAndLatencyAlarmsAndRules = AvailabilityAndLatencyAlarmsAndRules;
//# sourceMappingURL=data:application/json;base64,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