"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const cloudfront_generated_1 = require("./cloudfront.generated");
class OriginAccessIdentityBase extends cdk.Resource {
    /**
     * The ARN to include in S3 bucket policy to allow CloudFront access
     */
    arn() {
        return cdk.Stack.of(this).formatArn({
            service: 'iam',
            region: '',
            account: 'cloudfront',
            resource: 'user',
            resourceName: `CloudFront Origin Access Identity ${this.originAccessIdentityName}`
        });
    }
}
/**
 * An origin access identity is a special CloudFront user that you can
 * associate with Amazon S3 origins, so that you can secure all or just some of
 * your Amazon S3 content.
 *
 * @resource AWS::CloudFront::CloudFrontOriginAccessIdentity
 */
class OriginAccessIdentity extends OriginAccessIdentityBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.resource = new cloudfront_generated_1.CfnCloudFrontOriginAccessIdentity(this, 'Resource', {
            cloudFrontOriginAccessIdentityConfig: {
                comment: (props && props.comment) || 'Allows CloudFront to reach the bucket'
            }
        });
        // physical id - OAI name
        this.originAccessIdentityName = this.getResourceNameAttribute(this.resource.ref);
        // Canonical user to whitelist in S3 Bucket Policy
        this.cloudFrontOriginAccessIdentityS3CanonicalUserId = this.resource.attrS3CanonicalUserId;
        // The principal for must be either the canonical user or a special ARN
        // with the CloudFront Origin Access Id (see `arn()` method). For
        // import/export the OAI is anyway required so the principal is constructed
        // with it. But for the normal case the S3 Canonical User as a nicer
        // interface and does not require constructing the ARN.
        this.grantPrincipal = new iam.CanonicalUserPrincipal(this.cloudFrontOriginAccessIdentityS3CanonicalUserId);
    }
    /**
     * Creates a OriginAccessIdentity by providing the OriginAccessIdentityName
     */
    static fromOriginAccessIdentityName(scope, id, originAccessIdentityName) {
        class Import extends OriginAccessIdentityBase {
            constructor(s, i) {
                super(s, i, { physicalName: originAccessIdentityName });
                this.originAccessIdentityName = originAccessIdentityName;
                this.grantPrincipal = new iam.ArnPrincipal(this.arn());
            }
        }
        return new Import(scope, id);
    }
}
exports.OriginAccessIdentity = OriginAccessIdentity;
//# sourceMappingURL=data:application/json;base64,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