import datetime
import keyword
import logging
import shlex
import subprocess
import textwrap

import medikit

logger = logging.getLogger(__name__)


def is_identifier(ident: str) -> bool:
    """Determines if string is valid Python identifier."""

    if not isinstance(ident, str):
        raise TypeError("expected str, but got {!r}".format(type(ident)))

    if not ident.isidentifier():
        return False

    if keyword.iskeyword(ident):
        return False

    return True


def format_file_content(s):
    return textwrap.dedent(s).strip() + "\n"


def get_override_warning_banner(*, prefix="# ", above=None, bellow=None):
    return "\n".join(
        filter(
            None,
            (
                above,
                textwrap.indent(
                    "\n".join(
                        (
                            "Generated by Medikit "
                            + medikit.__version__
                            + " on "
                            + str(datetime.datetime.now().date())
                            + ".",
                            "All changes will be overriden.",
                            "Edit Projectfile and run “make update” (or “medikit update”) to regenerate.",
                        )
                    ),
                    prefix=prefix,
                ),
                bellow,
            ),
        )
    )


def run_command(command, *, logger=logger):
    logger.info("Running command %s", command)
    result = subprocess.run(shlex.split(command), stdout=subprocess.PIPE)
    if result.returncode:
        raise RuntimeError(
            '"{command}" exited with status {returncode}.'.format(command=command, returncode=result.returncode)
        )
    return result.stdout.decode("utf-8").strip()
