'''
# SSM Document CDK

This library provides a code-based utility for implementing SSM Documents. The SSM Document objects can be used to print YAML/JSON documents and to mimic document processing locally.

This library abstracts SSM Documents at a high level, with each step as well as the document itself being objects. The properties needed to build these objects correlate to the settings that apply to them, making them simple to make.
This library can be used to test your document locally before deploying it to SSM.

Since the library is written in JSII, it can be exported to other languages that support JSII (Java, Python).

This is what you'd use if you wanted to:

1. The ability to test without deploying resources or executing an actual SSM on AWS.
2. Reusability of steps between documents by reusing existing items
3. Create logical higher-level groupings of reusable groups of steps ("Patterns")
4. Simple to use interface for writing documents
5. Import existing documents from a file (or string) and mimic them locally to test them.

## Usage

### Document Creation

Typescript usage (Execute AWS API Step)...
The below creates the AutomationDocument in an AWS CDK stack.

```python
import { AutomationDocument } from './automation-document';

export class HelloWorld extends Stack {
  constructor(app: Construct, id: string) {
    super(app, id);

    // Create AutomationDocument
    const myDoc = new AutomationDocument(this, "MyDoc", {
      documentFormat: DocumentFormat.JSON,
      documentName: "MyDoc",
      docInputs: [{name: "MyInput", defaultValue: "a", inputType: DataTypeEnum.STRING}]
    });

    // Define your steps...
    myDoc.addStep(new PauseStep(this, "MyPauseStep", { name: "MyPauseStep" }));

    myDoc.addStep(new ExecuteScriptStep(this, "MyExecuteStep", {
      name: "step1",
      handlerName: "my_func",
      language: ScriptLanguage.PYTHON,
      fullPathToCode: resolve("test/test_file.py"),
      // OR .inlineCode("def my_func(args, context):\n  return {'MyReturn': args['MyInput'] + '-suffix'}\n")
      outputs: [{
        outputType: DataTypeEnum.STRING,
        name: "MyFuncOut",
        selector: "$.Payload.MyReturn"
      }],
      inputs: ["MyInput"]
    }));
  }
}
```

### Document JSON/YAML Export as YAML/JSON

You can deploy the above document using CDK.
To print the above document object as a JSON (or YAML), do the following:

```python
const myDocJson = myDoc.print(); // Print YAML by setting the documentFormat to YAML
```

### Document Simulation

To run the document object in simulation mode, use the below. Simulation mode does NOT hit the SSM API, rather it mimics the execution that will happen in an SSM execution. The run happens locally and allows you to mock the calls to external services (AWS APIs for example) or to invoke those services using your local credentials.

```python
import { Simulation } from './simulation';

const myDocJson = Simulation.ofAutomation(myDoc, {}).simulate({ MyInput: "FooBar" });
```

## Patterns (High-Level Constructs)

In typical CDK style, you can assemble often used groups of steps into higher level Constructs.

Consider if you typically create AutomationDocuments that start with logging the time and end with logging the total time taken. You can create a high-level Automation Document and extend that when you implement an Automation.

See the `TimedDocument` class to see such implementation.

Or consider the case of multiple steps that are always run together such as rebooting and instance and waiting for it to be active.

The below example is copied from the `RebootInstanceAndWait` class:

```python
export class RebootInstanceAndWait extends CompositeAutomationStep {

  readonly reboot: AwsApiStep;
  readonly describe: WaitForResourceStep;

  constructor(scope: Construct, id: string, instanceId: IStringVariable) {
    super(scope, id);
    this.reboot = new AwsApiStep(this, 'RebootInstances', {
      service: 'ec2',
      pascalCaseApi: 'RebootInstances',
      apiParams: { InstanceIds: [instanceId] },
      outputs: [],
    });
    this.describe = new WaitForResourceStep(this, 'DescribeInstances', {
      service: 'ec2',
      pascalCaseApi: 'DescribeInstances',
      apiParams: { InstanceIds: [instanceId] },
      selector: '$.Reservations[0].Instances[0].State.Name',
      desiredValues: ['running'],
    });
  }

  addToDocument(doc: AutomationDocumentBuilder): void {
    doc.addStep(this.reboot);
    doc.addStep(this.describe);
  }
}
```

Now, you can use `RebootInstanceAndWait` as a step in a document and the child steps will be included.

## Existing Documents

Do you have an existing document that you want to convert to code and/or test locally using the simulation?

### Import Existing Document

Here is an example of how you can import an existing document and then simulate it locally with mocked AWS resources:

```python
// Initialize Mocks
const sleeper = new MockSleep();
const awsInvoker = new MockAwsInvoker();
awsInvoker.whenThen(
    // when invoked with...
    {awsApi: 'listBuckets', awsParams: {}, service: 'S3'},
    // then response with...
    {Owner: {ID: "BUCKET_ID"}})

// ======> Create document from file <=======
const stack: Stack = new Stack();
const myAutomationDoc = StringDocument.fromFile(stack, "MyAutomationDoc", 'test/myAutomation.json', {
                                                                        // ======================
});

// Execute simulation
const simOutput = Simulation.ofAutomation(myAutomationDoc, {
  sleepHook: sleeper,
  awsInvoker: awsInvoker
}).simulate({});

// Assert simulation result
assert.deepEqual(awsInvoker.previousInvocations, [
    { awsApi: 'listBuckets', awsParams: {}, service: 'S3' }]);
assert.deepEqual(sleeper.sleepMilliInvocations, [3000]);
assert.deepEqual(simOutput.outputs['simulationSteps'], ['MySleep', 'GetBucketId']);
```

### Import Existing Steps

You can also grab a string step (or steps) and import them as CDK step constructs.
This can be used to convert existing documents into CDK with each step defined separately.
Doing so will allow you do modify steps and reuse them in other documents.

Here's a simple example of a sleep step copy and pasted from its original yaml:

```python
StringStep.fromYaml(this, `
    name: sleep
    action: aws:sleep
    inputs:
      Duration: PT0M
`, {});
```

The above will return the CDK construct SleepStep.

## What is Planned?

This library currently contains AutomationDocument steps.

Also planned is the ability to run Command documents.

Stay tuned!

## Related Projects

* https://github.com/udondan/cdk-ssm-document

## Security

See [CONTRIBUTING](CONTRIBUTING.md#security-issue-notifications) for more information.

## License

This project is licensed under the Apache-2.0 License.
'''
import abc
import builtins
import datetime
import enum
import typing

import jsii
import publication
import typing_extensions

from ._jsii import *

import aws_cdk
import aws_cdk.aws_ssm
import aws_cdk.cx_api
import constructs


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ApiExecuteAutomationProps",
    jsii_struct_bases=[],
    name_mapping={"aws_invoker": "awsInvoker", "sleep_hook": "sleepHook"},
)
class ApiExecuteAutomationProps:
    def __init__(self, *, aws_invoker: "IAwsInvoker", sleep_hook: "ISleepHook") -> None:
        '''
        :param aws_invoker: 
        :param sleep_hook: 
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "aws_invoker": aws_invoker,
            "sleep_hook": sleep_hook,
        }

    @builtins.property
    def aws_invoker(self) -> "IAwsInvoker":
        result = self._values.get("aws_invoker")
        assert result is not None, "Required property 'aws_invoker' is missing"
        return typing.cast("IAwsInvoker", result)

    @builtins.property
    def sleep_hook(self) -> "ISleepHook":
        result = self._values.get("sleep_hook")
        assert result is not None, "Required property 'sleep_hook' is missing"
        return typing.cast("ISleepHook", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApiExecuteAutomationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ApiRunCommandProps",
    jsii_struct_bases=[],
    name_mapping={"aws_invoker": "awsInvoker", "sleep_hook": "sleepHook"},
)
class ApiRunCommandProps:
    def __init__(self, *, aws_invoker: "IAwsInvoker", sleep_hook: "ISleepHook") -> None:
        '''
        :param aws_invoker: 
        :param sleep_hook: 
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "aws_invoker": aws_invoker,
            "sleep_hook": sleep_hook,
        }

    @builtins.property
    def aws_invoker(self) -> "IAwsInvoker":
        result = self._values.get("aws_invoker")
        assert result is not None, "Required property 'aws_invoker' is missing"
        return typing.cast("IAwsInvoker", result)

    @builtins.property
    def sleep_hook(self) -> "ISleepHook":
        result = self._values.get("sleep_hook")
        assert result is not None, "Required property 'sleep_hook' is missing"
        return typing.cast("ISleepHook", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApiRunCommandProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ApproveSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"approve_hook": "approveHook"},
)
class ApproveSimulationProps:
    def __init__(self, *, approve_hook: "IApproveHook") -> None:
        '''Properties for ApproveStep.

        :param approve_hook: (Optional) Approve hook to be called to pause the execution. To mock this implementation either inject an instance of IApproveHook or use the provided MockApprove class. Default: ApproveHook instance. ApproveHook may not work in exported JSII languages. Override interface as needed.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "approve_hook": approve_hook,
        }

    @builtins.property
    def approve_hook(self) -> "IApproveHook":
        '''(Optional) Approve hook to be called to pause the execution.

        To mock this implementation either inject an instance of IApproveHook or use the provided MockApprove class.

        :default: ApproveHook instance. ApproveHook may not work in exported JSII languages. Override interface as needed.
        '''
        result = self._values.get("approve_hook")
        assert result is not None, "Required property 'approve_hook' is missing"
        return typing.cast("IApproveHook", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApproveSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AuthMethod(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.AuthMethod",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="ofBasic") # type: ignore[misc]
    @builtins.classmethod
    def of_basic(
        cls,
        user_name: "SecureVariable",
        password: "SecureVariable",
    ) -> "AuthMethod":
        '''
        :param user_name: -
        :param password: -
        '''
        return typing.cast("AuthMethod", jsii.sinvoke(cls, "ofBasic", [user_name, password]))

    @jsii.member(jsii_name="ofDigest") # type: ignore[misc]
    @builtins.classmethod
    def of_digest(
        cls,
        user_name: "SecureVariable",
        password: "SecureVariable",
    ) -> "AuthMethod":
        '''
        :param user_name: -
        :param password: -
        '''
        return typing.cast("AuthMethod", jsii.sinvoke(cls, "ofDigest", [user_name, password]))

    @jsii.member(jsii_name="ofNone") # type: ignore[misc]
    @builtins.classmethod
    def of_none(cls) -> "AuthMethod":
        return typing.cast("AuthMethod", jsii.sinvoke(cls, "ofNone", []))

    @jsii.member(jsii_name="requiredInputs") # type: ignore[misc]
    @abc.abstractmethod
    def required_inputs(self) -> typing.List[builtins.str]:
        ...

    @jsii.member(jsii_name="toEntry") # type: ignore[misc]
    @abc.abstractmethod
    def to_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        ...


class _AuthMethodProxy(AuthMethod):
    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="toEntry")
    def to_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toEntry", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AuthMethod).__jsii_proxy_class__ = lambda : _AuthMethodProxy


class AutomationDocumentBuilder(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationDocumentBuilder",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="addStep")
    def add_step(self, step: "AutomationStep") -> None:
        '''
        :param step: -
        '''
        return typing.cast(None, jsii.invoke(self, "addStep", [step]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="steps")
    def steps(self) -> typing.List["AutomationStep"]:
        return typing.cast(typing.List["AutomationStep"], jsii.get(self, "steps"))


class AutomationSimulation(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationSimulation",
):
    def __init__(
        self,
        automation_document: "AutomationDocument",
        *,
        approve_hook: typing.Optional["IApproveHook"] = None,
        aws_invoker: typing.Optional["IAwsInvoker"] = None,
        environment: typing.Optional["IEnvironment"] = None,
        input_observer: typing.Optional["IObserver"] = None,
        output_observer: typing.Optional["IObserver"] = None,
        parameter_resolver: typing.Optional["IParameterResolver"] = None,
        pause_hook: typing.Optional["IPauseHook"] = None,
        run_command_hook: typing.Optional["IRunCommandHook"] = None,
        simulation_platform: typing.Optional["Platform"] = None,
        sleep_hook: typing.Optional["ISleepHook"] = None,
        webhook: typing.Optional["IWebhook"] = None,
    ) -> None:
        '''
        :param automation_document: -
        :param approve_hook: 
        :param aws_invoker: 
        :param environment: 
        :param input_observer: 
        :param output_observer: 
        :param parameter_resolver: 
        :param pause_hook: 
        :param run_command_hook: 
        :param simulation_platform: 
        :param sleep_hook: 
        :param webhook: 
        '''
        props = SimulationProps(
            approve_hook=approve_hook,
            aws_invoker=aws_invoker,
            environment=environment,
            input_observer=input_observer,
            output_observer=output_observer,
            parameter_resolver=parameter_resolver,
            pause_hook=pause_hook,
            run_command_hook=run_command_hook,
            simulation_platform=simulation_platform,
            sleep_hook=sleep_hook,
            webhook=webhook,
        )

        jsii.create(self.__class__, self, [automation_document, props])

    @jsii.member(jsii_name="start")
    def start(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> "SimulationResult":
        '''Starts the execution of the steps by invoking the first step.

        The subsequent steps will be invoked by the steps themselves.

        :param inputs: all of the inputs necessary for the document to execute.

        :return: the outputs specified by all the steps.
        '''
        return typing.cast("SimulationResult", jsii.invoke(self, "start", [inputs]))


class AutomationSimulationBase(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationSimulationBase",
):
    def __init__(self, step: "AutomationStep") -> None:
        '''
        :param step: -
        '''
        jsii.create(self.__class__, self, [step])

    @jsii.member(jsii_name="executeStep") # type: ignore[misc]
    @abc.abstractmethod
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        ...

    @jsii.member(jsii_name="nextStep")
    def next_step(
        self,
        _inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Optional["AutomationStep"]:
        '''
        :param _inputs: -
        '''
        return typing.cast(typing.Optional["AutomationStep"], jsii.invoke(self, "nextStep", [_inputs]))


class _AutomationSimulationBaseProxy(AutomationSimulationBase):
    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AutomationSimulationBase).__jsii_proxy_class__ = lambda : _AutomationSimulationBaseProxy


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationSimulationProps",
    jsii_struct_bases=[],
    name_mapping={
        "approve_hook": "approveHook",
        "aws_invoker": "awsInvoker",
        "execute_automation_hook": "executeAutomationHook",
        "input_observer": "inputObserver",
        "output_observer": "outputObserver",
        "parameter_resolver": "parameterResolver",
        "pause_hook": "pauseHook",
        "run_command_hook": "runCommandHook",
        "sleep_hook": "sleepHook",
        "webhook": "webhook",
    },
)
class AutomationSimulationProps:
    def __init__(
        self,
        *,
        approve_hook: typing.Optional["IApproveHook"] = None,
        aws_invoker: typing.Optional["IAwsInvoker"] = None,
        execute_automation_hook: typing.Optional["IExecuteAutomationHook"] = None,
        input_observer: typing.Optional["IObserver"] = None,
        output_observer: typing.Optional["IObserver"] = None,
        parameter_resolver: typing.Optional["IParameterResolver"] = None,
        pause_hook: typing.Optional["IPauseHook"] = None,
        run_command_hook: typing.Optional["IRunCommandHook"] = None,
        sleep_hook: typing.Optional["ISleepHook"] = None,
        webhook: typing.Optional["IWebhook"] = None,
    ) -> None:
        '''
        :param approve_hook: (Optional) Approve hook to be called to pause the execution. To mock this implementation either inject an instance of IApproveHook or use the provided MockApprove class. Default: ApproveHook instance. ApproveHook may not work in exported JSII languages. Override interface as needed.
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param execute_automation_hook: Hook for simulating aws:executeAutomation. Default: - Uses AWS API to execute the document remotely.
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param parameter_resolver: (Optional) Resolver for secure strings in parameters. Required to simulate if using tokens in parameters input. Default: - Treats parameters as literal
        :param pause_hook: (Optional) Pause hook to be called to pause the execution. To mock this implemenation either inject an instance of IPauseHook or use the provided MockPause class. Default: PauseHook instance. PauseHook may not work in exported JSII languages. Override interface as needed.
        :param run_command_hook: Hook for simulating aws:runCommand. Default: - Uses AWS API to execute the document remotely.
        :param sleep_hook: (Optional) Hook to inject alternate ISleeper (to mock the sleep between failed invocations). Default: - really perform sleep using SleeperImpl class.
        :param webhook: (Optional) Hook for simulating aws:invokeWebhook. Default: - Returns 204 with an empty response
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if approve_hook is not None:
            self._values["approve_hook"] = approve_hook
        if aws_invoker is not None:
            self._values["aws_invoker"] = aws_invoker
        if execute_automation_hook is not None:
            self._values["execute_automation_hook"] = execute_automation_hook
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if parameter_resolver is not None:
            self._values["parameter_resolver"] = parameter_resolver
        if pause_hook is not None:
            self._values["pause_hook"] = pause_hook
        if run_command_hook is not None:
            self._values["run_command_hook"] = run_command_hook
        if sleep_hook is not None:
            self._values["sleep_hook"] = sleep_hook
        if webhook is not None:
            self._values["webhook"] = webhook

    @builtins.property
    def approve_hook(self) -> typing.Optional["IApproveHook"]:
        '''(Optional) Approve hook to be called to pause the execution.

        To mock this implementation either inject an instance of IApproveHook or use the provided MockApprove class.

        :default: ApproveHook instance. ApproveHook may not work in exported JSII languages. Override interface as needed.
        '''
        result = self._values.get("approve_hook")
        return typing.cast(typing.Optional["IApproveHook"], result)

    @builtins.property
    def aws_invoker(self) -> typing.Optional["IAwsInvoker"]:
        '''(Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call).

        :default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        result = self._values.get("aws_invoker")
        return typing.cast(typing.Optional["IAwsInvoker"], result)

    @builtins.property
    def execute_automation_hook(self) -> typing.Optional["IExecuteAutomationHook"]:
        '''Hook for simulating aws:executeAutomation.

        :default: - Uses AWS API to execute the document remotely.
        '''
        result = self._values.get("execute_automation_hook")
        return typing.cast(typing.Optional["IExecuteAutomationHook"], result)

    @builtins.property
    def input_observer(self) -> typing.Optional["IObserver"]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional["IObserver"], result)

    @builtins.property
    def output_observer(self) -> typing.Optional["IObserver"]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional["IObserver"], result)

    @builtins.property
    def parameter_resolver(self) -> typing.Optional["IParameterResolver"]:
        '''(Optional) Resolver for secure strings in parameters.

        Required to simulate if using tokens in parameters input.

        :default: - Treats parameters as literal
        '''
        result = self._values.get("parameter_resolver")
        return typing.cast(typing.Optional["IParameterResolver"], result)

    @builtins.property
    def pause_hook(self) -> typing.Optional["IPauseHook"]:
        '''(Optional) Pause hook to be called to pause the execution.

        To mock this implemenation either inject an instance of IPauseHook or use the provided MockPause class.

        :default: PauseHook instance. PauseHook may not work in exported JSII languages. Override interface as needed.
        '''
        result = self._values.get("pause_hook")
        return typing.cast(typing.Optional["IPauseHook"], result)

    @builtins.property
    def run_command_hook(self) -> typing.Optional["IRunCommandHook"]:
        '''Hook for simulating aws:runCommand.

        :default: - Uses AWS API to execute the document remotely.
        '''
        result = self._values.get("run_command_hook")
        return typing.cast(typing.Optional["IRunCommandHook"], result)

    @builtins.property
    def sleep_hook(self) -> typing.Optional["ISleepHook"]:
        '''(Optional) Hook to inject alternate ISleeper (to mock the sleep between failed invocations).

        :default: - really perform sleep using SleeperImpl class.
        '''
        result = self._values.get("sleep_hook")
        return typing.cast(typing.Optional["ISleepHook"], result)

    @builtins.property
    def webhook(self) -> typing.Optional["IWebhook"]:
        '''(Optional) Hook for simulating aws:invokeWebhook.

        :default: - Returns 204 with an empty response
        '''
        result = self._values.get("webhook")
        return typing.cast(typing.Optional["IWebhook"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AutomationSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AutomationStepSimulation(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationStepSimulation",
):
    def __init__(
        self,
        step: "AutomationStep",
        *,
        approve_hook: typing.Optional["IApproveHook"] = None,
        aws_invoker: typing.Optional["IAwsInvoker"] = None,
        execute_automation_hook: typing.Optional["IExecuteAutomationHook"] = None,
        input_observer: typing.Optional["IObserver"] = None,
        output_observer: typing.Optional["IObserver"] = None,
        parameter_resolver: typing.Optional["IParameterResolver"] = None,
        pause_hook: typing.Optional["IPauseHook"] = None,
        run_command_hook: typing.Optional["IRunCommandHook"] = None,
        sleep_hook: typing.Optional["ISleepHook"] = None,
        webhook: typing.Optional["IWebhook"] = None,
    ) -> None:
        '''
        :param step: -
        :param approve_hook: (Optional) Approve hook to be called to pause the execution. To mock this implementation either inject an instance of IApproveHook or use the provided MockApprove class. Default: ApproveHook instance. ApproveHook may not work in exported JSII languages. Override interface as needed.
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param execute_automation_hook: Hook for simulating aws:executeAutomation. Default: - Uses AWS API to execute the document remotely.
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param parameter_resolver: (Optional) Resolver for secure strings in parameters. Required to simulate if using tokens in parameters input. Default: - Treats parameters as literal
        :param pause_hook: (Optional) Pause hook to be called to pause the execution. To mock this implemenation either inject an instance of IPauseHook or use the provided MockPause class. Default: PauseHook instance. PauseHook may not work in exported JSII languages. Override interface as needed.
        :param run_command_hook: Hook for simulating aws:runCommand. Default: - Uses AWS API to execute the document remotely.
        :param sleep_hook: (Optional) Hook to inject alternate ISleeper (to mock the sleep between failed invocations). Default: - really perform sleep using SleeperImpl class.
        :param webhook: (Optional) Hook for simulating aws:invokeWebhook. Default: - Returns 204 with an empty response
        '''
        props = AutomationSimulationProps(
            approve_hook=approve_hook,
            aws_invoker=aws_invoker,
            execute_automation_hook=execute_automation_hook,
            input_observer=input_observer,
            output_observer=output_observer,
            parameter_resolver=parameter_resolver,
            pause_hook=pause_hook,
            run_command_hook=run_command_hook,
            sleep_hook=sleep_hook,
            webhook=webhook,
        )

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> "SimulationResult":
        '''Invokes the current step on the input and will return a SimulationResult.

        :param inputs: must contain all of the inputs declared by the current step.

        :return: a SimulationResult with the step outputs in the case of success or stacktrace in the case of failure.
        '''
        return typing.cast("SimulationResult", jsii.invoke(self, "invoke", [inputs]))

    @jsii.member(jsii_name="prependSelf")
    def _prepend_self(
        self,
        subsequent_steps: typing.Sequence[builtins.str],
    ) -> typing.List[builtins.str]:
        '''Adds this step name to the list of executed steps.

        Since the steps are invoked as a chain, the current step is prepended as invocation stack is popped.

        :param subsequent_steps: -
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "prependSelf", [subsequent_steps]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="props")
    def props(self) -> "RequiredAutomationSimulationProps":
        return typing.cast("RequiredAutomationSimulationProps", jsii.get(self, "props"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="step")
    def step(self) -> "AutomationStep":
        return typing.cast("AutomationStep", jsii.get(self, "step"))


class AwsApiSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.AwsApiSimulation",
):
    '''AutomationStep implementation of aws:executeAwsApi.'''

    def __init__(self, step: "AwsApiStep", *, aws_invoker: "IAwsInvoker") -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''Invokes the specified service (param) with the specified api (param) with the specified apiParams (param).

        This call will be invoked synchronously.
        The input variables in apiParams (param) specified using "{{INPUT}}" syntax will be replaced with the inputs.

        :param inputs: -

        :return: the AWS api response. The Output selection will take place outside of this function.
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="awsApiStep")
    def aws_api_step(self) -> "AwsApiStep":
        return typing.cast("AwsApiStep", jsii.get(self, "awsApiStep"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="awsInvoker")
    def aws_invoker(self) -> "IAwsInvoker":
        return typing.cast("IAwsInvoker", jsii.get(self, "awsInvoker"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.AwsInvocationSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"aws_invoker": "awsInvoker"},
)
class AwsInvocationSimulationProps:
    def __init__(self, *, aws_invoker: "IAwsInvoker") -> None:
        '''
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "aws_invoker": aws_invoker,
        }

    @builtins.property
    def aws_invoker(self) -> "IAwsInvoker":
        '''(Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call).

        :default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        result = self._values.get("aws_invoker")
        assert result is not None, "Required property 'aws_invoker' is missing"
        return typing.cast("IAwsInvoker", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AwsInvocationSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.BodyOrUrlProp",
    jsii_struct_bases=[],
    name_mapping={"prop_type": "propType", "value": "value"},
)
class BodyOrUrlProp:
    def __init__(self, *, prop_type: "BodyOrUrlType", value: "IStringVariable") -> None:
        '''Allow passing in a body or URL version of the property value.

        :param prop_type: Whether the body or URL was provided.
        :param value: Body or URL string.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "prop_type": prop_type,
            "value": value,
        }

    @builtins.property
    def prop_type(self) -> "BodyOrUrlType":
        '''Whether the body or URL was provided.'''
        result = self._values.get("prop_type")
        assert result is not None, "Required property 'prop_type' is missing"
        return typing.cast("BodyOrUrlType", result)

    @builtins.property
    def value(self) -> "IStringVariable":
        '''Body or URL string.'''
        result = self._values.get("value")
        assert result is not None, "Required property 'value' is missing"
        return typing.cast("IStringVariable", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BodyOrUrlProp(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@cdklabs/cdk-ssm-documents.BodyOrUrlType")
class BodyOrUrlType(enum.Enum):
    BODY = "BODY"
    URL = "URL"


class BranchSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.BranchSimulation",
):
    '''AutomationStep implementation of aws:branch https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-branch.html.'''

    def __init__(self, step: "BranchStep") -> None:
        '''
        :param step: -
        '''
        jsii.create(self.__class__, self, [step])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        _inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''noop.

        The logic performed in the branch branchStep happens in the invoke() function.

        :param _inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [_inputs]))

    @jsii.member(jsii_name="nextStep")
    def next_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Optional["AutomationStep"]:
        '''Overrides invoke because control flow of execution is different than standard steps.

        Will traverse the choices until one evaluated to true; will skip to that choice.

        :param inputs: -
        '''
        return typing.cast(typing.Optional["AutomationStep"], jsii.invoke(self, "nextStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


class ChangeInstanceStateSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ChangeInstanceStateSimulation",
):
    '''AutomationStep implemenation for aws:changeInstanceState https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-changestate.html.'''

    def __init__(
        self,
        step: "ChangeInstanceStateStep",
        *,
        aws_invoker: "IAwsInvoker",
    ) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


class Choice(metaclass=jsii.JSIIMeta, jsii_type="@cdklabs/cdk-ssm-documents.Choice"):
    '''Choices are supplied to the BranchStep to determine under which conditions to jump to which steps.'''

    def __init__(
        self,
        *,
        constant: typing.Any,
        jump_to_step_name: builtins.str,
        operation: "Operation",
        variable: typing.Union["IStringVariable", "INumberVariable", "IBooleanVariable"],
    ) -> None:
        '''
        :param constant: (Required) the constant to test against the inputToTest.
        :param jump_to_step_name: A step to jump to if this choice is evaluated to true. Must reference another step in the currently executing AutomationDocument.
        :param operation: (Required) The operation used to compare the inputToTest with the constant.
        :param variable: (Required) the input used to test using the operation with the constant.
        '''
        props = ChoiceProps(
            constant=constant,
            jump_to_step_name=jump_to_step_name,
            operation=operation,
            variable=variable,
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="asSsmEntry")
    def as_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :return: an object that can be used to print this choice into yaml/json format.
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "asSsmEntry", []))

    @jsii.member(jsii_name="evaluate")
    def evaluate(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.bool:
        '''Evaluates this choice against the provided inputs.

        The value keyed on the inputToTest will be tested against the declared constant using the Operation specified.

        :param inputs: -

        :return: true if the evaluation is true. False otherwise.
        '''
        return typing.cast(builtins.bool, jsii.invoke(self, "evaluate", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="constant")
    def constant(self) -> typing.Any:
        return typing.cast(typing.Any, jsii.get(self, "constant"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="jumpToStepName")
    def jump_to_step_name(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "jumpToStepName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="operation")
    def operation(self) -> "Operation":
        return typing.cast("Operation", jsii.get(self, "operation"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="variable")
    def variable(
        self,
    ) -> typing.Union["IStringVariable", "INumberVariable", "IBooleanVariable"]:
        return typing.cast(typing.Union["IStringVariable", "INumberVariable", "IBooleanVariable"], jsii.get(self, "variable"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ChoiceProps",
    jsii_struct_bases=[],
    name_mapping={
        "constant": "constant",
        "jump_to_step_name": "jumpToStepName",
        "operation": "operation",
        "variable": "variable",
    },
)
class ChoiceProps:
    def __init__(
        self,
        *,
        constant: typing.Any,
        jump_to_step_name: builtins.str,
        operation: "Operation",
        variable: typing.Union["IStringVariable", "INumberVariable", "IBooleanVariable"],
    ) -> None:
        '''Properties for a Choice used by the BranchStep.

        :param constant: (Required) the constant to test against the inputToTest.
        :param jump_to_step_name: A step to jump to if this choice is evaluated to true. Must reference another step in the currently executing AutomationDocument.
        :param operation: (Required) The operation used to compare the inputToTest with the constant.
        :param variable: (Required) the input used to test using the operation with the constant.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "constant": constant,
            "jump_to_step_name": jump_to_step_name,
            "operation": operation,
            "variable": variable,
        }

    @builtins.property
    def constant(self) -> typing.Any:
        '''(Required) the constant to test against the inputToTest.'''
        result = self._values.get("constant")
        assert result is not None, "Required property 'constant' is missing"
        return typing.cast(typing.Any, result)

    @builtins.property
    def jump_to_step_name(self) -> builtins.str:
        '''A step to jump to if this choice is evaluated to true.

        Must reference another step in the currently executing AutomationDocument.
        '''
        result = self._values.get("jump_to_step_name")
        assert result is not None, "Required property 'jump_to_step_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def operation(self) -> "Operation":
        '''(Required) The operation used to compare the inputToTest with the constant.'''
        result = self._values.get("operation")
        assert result is not None, "Required property 'operation' is missing"
        return typing.cast("Operation", result)

    @builtins.property
    def variable(
        self,
    ) -> typing.Union["IStringVariable", "INumberVariable", "IBooleanVariable"]:
        '''(Required) the input used to test using the operation with the constant.'''
        result = self._values.get("variable")
        assert result is not None, "Required property 'variable' is missing"
        return typing.cast(typing.Union["IStringVariable", "INumberVariable", "IBooleanVariable"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ChoiceProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CommandDocumentBuilder(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CommandDocumentBuilder",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="addStep")
    def add_step(self, step: "CommandStep") -> None:
        '''
        :param step: -
        '''
        return typing.cast(None, jsii.invoke(self, "addStep", [step]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="steps")
    def steps(self) -> typing.List["CommandStep"]:
        return typing.cast(typing.List["CommandStep"], jsii.get(self, "steps"))


class CommandSimulation(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CommandSimulation",
):
    def __init__(
        self,
        command_document: "CommandDocument",
        *,
        approve_hook: typing.Optional["IApproveHook"] = None,
        aws_invoker: typing.Optional["IAwsInvoker"] = None,
        environment: typing.Optional["IEnvironment"] = None,
        input_observer: typing.Optional["IObserver"] = None,
        output_observer: typing.Optional["IObserver"] = None,
        parameter_resolver: typing.Optional["IParameterResolver"] = None,
        pause_hook: typing.Optional["IPauseHook"] = None,
        run_command_hook: typing.Optional["IRunCommandHook"] = None,
        simulation_platform: typing.Optional["Platform"] = None,
        sleep_hook: typing.Optional["ISleepHook"] = None,
        webhook: typing.Optional["IWebhook"] = None,
    ) -> None:
        '''
        :param command_document: -
        :param approve_hook: 
        :param aws_invoker: 
        :param environment: 
        :param input_observer: 
        :param output_observer: 
        :param parameter_resolver: 
        :param pause_hook: 
        :param run_command_hook: 
        :param simulation_platform: 
        :param sleep_hook: 
        :param webhook: 
        '''
        props = SimulationProps(
            approve_hook=approve_hook,
            aws_invoker=aws_invoker,
            environment=environment,
            input_observer=input_observer,
            output_observer=output_observer,
            parameter_resolver=parameter_resolver,
            pause_hook=pause_hook,
            run_command_hook=run_command_hook,
            simulation_platform=simulation_platform,
            sleep_hook=sleep_hook,
            webhook=webhook,
        )

        jsii.create(self.__class__, self, [command_document, props])

    @jsii.member(jsii_name="start")
    def start(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> "SimulationResult":
        '''
        :param inputs: -
        '''
        return typing.cast("SimulationResult", jsii.invoke(self, "start", [inputs]))


class CommandSimulationBase(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.CommandSimulationBase",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="executeStep") # type: ignore[misc]
    @abc.abstractmethod
    def execute_step(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''
        :param inputs: -
        '''
        ...


class _CommandSimulationBaseProxy(CommandSimulationBase):
    @jsii.member(jsii_name="executeStep")
    def execute_step(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''
        :param inputs: -
        '''
        return typing.cast(None, jsii.invoke(self, "executeStep", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, CommandSimulationBase).__jsii_proxy_class__ = lambda : _CommandSimulationBaseProxy


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CommandSimulationProps",
    jsii_struct_bases=[],
    name_mapping={
        "simulation_platform": "simulationPlatform",
        "environment": "environment",
    },
)
class CommandSimulationProps:
    def __init__(
        self,
        *,
        simulation_platform: "Platform",
        environment: typing.Optional["IEnvironment"] = None,
    ) -> None:
        '''
        :param simulation_platform: The Platform used in executing the command step.
        :param environment: (Optional) Specify here the environment in which to execute the scripts. Use the DockerEnvironment to execute the commands inside the docker. You can alternatively use the LoggingEnvironment which simply logs the commands or MockEnvironment which saves them for validation. Default: LoggingEnvironment
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "simulation_platform": simulation_platform,
        }
        if environment is not None:
            self._values["environment"] = environment

    @builtins.property
    def simulation_platform(self) -> "Platform":
        '''The Platform used in executing the command step.'''
        result = self._values.get("simulation_platform")
        assert result is not None, "Required property 'simulation_platform' is missing"
        return typing.cast("Platform", result)

    @builtins.property
    def environment(self) -> typing.Optional["IEnvironment"]:
        '''(Optional) Specify here the environment in which to execute the scripts.

        Use the DockerEnvironment to execute the commands inside the docker.
        You can alternatively use the LoggingEnvironment which simply logs the commands
        or MockEnvironment which saves them for validation.

        :default: LoggingEnvironment
        '''
        result = self._values.get("environment")
        return typing.cast(typing.Optional["IEnvironment"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommandSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CommandStepSimulation(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CommandStepSimulation",
):
    def __init__(
        self,
        step: "CommandStep",
        *,
        simulation_platform: "Platform",
        environment: typing.Optional["IEnvironment"] = None,
    ) -> None:
        '''
        :param step: -
        :param simulation_platform: The Platform used in executing the command step.
        :param environment: (Optional) Specify here the environment in which to execute the scripts. Use the DockerEnvironment to execute the commands inside the docker. You can alternatively use the LoggingEnvironment which simply logs the commands or MockEnvironment which saves them for validation. Default: LoggingEnvironment
        '''
        props = CommandSimulationProps(
            simulation_platform=simulation_platform, environment=environment
        )

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> "SimulationResult":
        '''Invokes the current step on the input and will return a SimulationResult.

        :param inputs: must contain all of the inputs declared by the current step.

        :return: a SimulationResult with the step outputs in the case of success or stacktrace in the case of failure.
        '''
        return typing.cast("SimulationResult", jsii.invoke(self, "invoke", [inputs]))

    @jsii.member(jsii_name="prependSelf")
    def _prepend_self(
        self,
        future_steps: typing.Sequence[builtins.str],
    ) -> typing.List[builtins.str]:
        '''Adds this step name to the list of executed steps.

        Since the steps are invoked as a chain, the current step is prepended as invocation stack is popped.

        :param future_steps: -
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "prependSelf", [future_steps]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="props")
    def props(self) -> "RequiredCommandSimulationProps":
        return typing.cast("RequiredCommandSimulationProps", jsii.get(self, "props"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="step")
    def step(self) -> "CommandStep":
        return typing.cast("CommandStep", jsii.get(self, "step"))


class CopyImageSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CopyImageSimulation",
):
    '''AutomationStep implemenation for aws:copyImage https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-copyimage.html.'''

    def __init__(self, step: "CopyImageStep", *, aws_invoker: "IAwsInvoker") -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


class CreateImageSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CreateImageSimulation",
):
    '''AutomationStep implemenation for aws:createImage https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-create.html.'''

    def __init__(self, step: "CreateImageStep", *, aws_invoker: "IAwsInvoker") -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


class CreateStackSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CreateStackSimulation",
):
    '''AutomationStep implementation for aws:createStack https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-createstack.html.'''

    def __init__(
        self,
        step: "CreateStackStep",
        *,
        aws_invoker: "IAwsInvoker",
        parameter_resolver: "IParameterResolver",
        sleep_hook: "ISleepHook",
    ) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param parameter_resolver: (Optional) Resolver for secure strings in parameters. Required to simulate if using tokens in parameters input. Default: - Treats parameters as literal
        :param sleep_hook: (Optional) Whether to really perform a pause of the runtime. To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class. Default: SleeperImpl
        '''
        props = CreateStackSimulationProps(
            aws_invoker=aws_invoker,
            parameter_resolver=parameter_resolver,
            sleep_hook=sleep_hook,
        )

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CreateStackSimulationProps",
    jsii_struct_bases=[],
    name_mapping={
        "aws_invoker": "awsInvoker",
        "parameter_resolver": "parameterResolver",
        "sleep_hook": "sleepHook",
    },
)
class CreateStackSimulationProps:
    def __init__(
        self,
        *,
        aws_invoker: "IAwsInvoker",
        parameter_resolver: "IParameterResolver",
        sleep_hook: "ISleepHook",
    ) -> None:
        '''Properties for CreateStackStep.

        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param parameter_resolver: (Optional) Resolver for secure strings in parameters. Required to simulate if using tokens in parameters input. Default: - Treats parameters as literal
        :param sleep_hook: (Optional) Whether to really perform a pause of the runtime. To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class. Default: SleeperImpl
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "aws_invoker": aws_invoker,
            "parameter_resolver": parameter_resolver,
            "sleep_hook": sleep_hook,
        }

    @builtins.property
    def aws_invoker(self) -> "IAwsInvoker":
        '''(Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call).

        :default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        result = self._values.get("aws_invoker")
        assert result is not None, "Required property 'aws_invoker' is missing"
        return typing.cast("IAwsInvoker", result)

    @builtins.property
    def parameter_resolver(self) -> "IParameterResolver":
        '''(Optional) Resolver for secure strings in parameters.

        Required to simulate if using tokens in parameters input.

        :default: - Treats parameters as literal
        '''
        result = self._values.get("parameter_resolver")
        assert result is not None, "Required property 'parameter_resolver' is missing"
        return typing.cast("IParameterResolver", result)

    @builtins.property
    def sleep_hook(self) -> "ISleepHook":
        '''(Optional) Whether to really perform a pause of the runtime.

        To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class.

        :default: SleeperImpl
        '''
        result = self._values.get("sleep_hook")
        assert result is not None, "Required property 'sleep_hook' is missing"
        return typing.cast("ISleepHook", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CreateStackSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CreateTagsSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CreateTagsSimulation",
):
    '''AutomationStep implemenation for aws:createTags https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-createtag.html.'''

    def __init__(
        self,
        step: "CreateTagsStep",
        *,
        aws_invoker: "IAwsInvoker",
        sleep_hook: "ISleepHook",
    ) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param sleep_hook: (Optional) Whether to really perform a pause of the runtime. To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class. Default: SleeperImpl
        '''
        props = CreateTagsSimulationProps(
            aws_invoker=aws_invoker, sleep_hook=sleep_hook
        )

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CreateTagsSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"aws_invoker": "awsInvoker", "sleep_hook": "sleepHook"},
)
class CreateTagsSimulationProps:
    def __init__(self, *, aws_invoker: "IAwsInvoker", sleep_hook: "ISleepHook") -> None:
        '''Properties for CreateTagStep.

        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param sleep_hook: (Optional) Whether to really perform a pause of the runtime. To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class. Default: SleeperImpl
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "aws_invoker": aws_invoker,
            "sleep_hook": sleep_hook,
        }

    @builtins.property
    def aws_invoker(self) -> "IAwsInvoker":
        '''(Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call).

        :default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        result = self._values.get("aws_invoker")
        assert result is not None, "Required property 'aws_invoker' is missing"
        return typing.cast("IAwsInvoker", result)

    @builtins.property
    def sleep_hook(self) -> "ISleepHook":
        '''(Optional) Whether to really perform a pause of the runtime.

        To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class.

        :default: SleeperImpl
        '''
        result = self._values.get("sleep_hook")
        assert result is not None, "Required property 'sleep_hook' is missing"
        return typing.cast("ISleepHook", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CreateTagsSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DataType(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DataType",
):
    '''JSII does not allow functions or constants declared in an enum class directly.

    Therefore, interaction with DataTypeEnum happens through this class.
    '''

    def __init__(self, data_type_enum: "DataTypeEnum") -> None:
        '''
        :param data_type_enum: -
        '''
        jsii.create(self.__class__, self, [data_type_enum])

    @jsii.member(jsii_name="fromDataType") # type: ignore[misc]
    @builtins.classmethod
    def from_data_type(cls, ssm_data_type: builtins.str) -> "DataType":
        '''
        :param ssm_data_type: -
        '''
        return typing.cast("DataType", jsii.sinvoke(cls, "fromDataType", [ssm_data_type]))

    @jsii.member(jsii_name="toSsmString")
    def to_ssm_string(self) -> builtins.str:
        '''
        :return: the SSM string representation of this DataType
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "toSsmString", []))

    @jsii.member(jsii_name="validateType")
    def validate_type(self, val: typing.Any) -> builtins.bool:
        '''
        :param val: -

        :return: true if the input variable matches the associated dataTypeEnum
        '''
        return typing.cast(builtins.bool, jsii.invoke(self, "validateType", [val]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="dataTypeEnum")
    def data_type_enum(self) -> "DataTypeEnum":
        return typing.cast("DataTypeEnum", jsii.get(self, "dataTypeEnum"))


@jsii.enum(jsii_type="@cdklabs/cdk-ssm-documents.DataTypeEnum")
class DataTypeEnum(enum.Enum):
    '''DataTypes as supported by SSM Documents.'''

    STRING = "STRING"
    INTEGER = "INTEGER"
    BOOLEAN = "BOOLEAN"
    STRING_LIST = "STRING_LIST"
    STRING_MAP = "STRING_MAP"
    MAP_LIST = "MAP_LIST"


class DeleteImageSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DeleteImageSimulation",
):
    '''AutomationStep implementation for aws:deleteImage https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-delete.html.'''

    def __init__(
        self,
        step: "DeleteImageStep",
        *,
        aws_invoker: typing.Optional["IAwsInvoker"] = None,
        sleep_hook: typing.Optional["ISleepHook"] = None,
    ) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param sleep_hook: (Optional) Whether to really perform a pause of the runtime. To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class. Default: SleeperImpl
        '''
        props = DeleteImageSimulationProps(
            aws_invoker=aws_invoker, sleep_hook=sleep_hook
        )

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.DeleteImageSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"aws_invoker": "awsInvoker", "sleep_hook": "sleepHook"},
)
class DeleteImageSimulationProps:
    def __init__(
        self,
        *,
        aws_invoker: typing.Optional["IAwsInvoker"] = None,
        sleep_hook: typing.Optional["ISleepHook"] = None,
    ) -> None:
        '''Properties for DeleteImageStep.

        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param sleep_hook: (Optional) Whether to really perform a pause of the runtime. To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class. Default: SleeperImpl
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if aws_invoker is not None:
            self._values["aws_invoker"] = aws_invoker
        if sleep_hook is not None:
            self._values["sleep_hook"] = sleep_hook

    @builtins.property
    def aws_invoker(self) -> typing.Optional["IAwsInvoker"]:
        '''(Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call).

        :default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        result = self._values.get("aws_invoker")
        return typing.cast(typing.Optional["IAwsInvoker"], result)

    @builtins.property
    def sleep_hook(self) -> typing.Optional["ISleepHook"]:
        '''(Optional) Whether to really perform a pause of the runtime.

        To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class.

        :default: SleeperImpl
        '''
        result = self._values.get("sleep_hook")
        return typing.cast(typing.Optional["ISleepHook"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DeleteImageSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DeleteStackSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DeleteStackSimulation",
):
    '''AutomationStep implemenation for aws:deleteStack https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-deletestack.html.'''

    def __init__(self, step: "DeleteStackStep", *, aws_invoker: "IAwsInvoker") -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.enum(jsii_type="@cdklabs/cdk-ssm-documents.DocumentFormat")
class DocumentFormat(enum.Enum):
    JSON = "JSON"
    YAML = "YAML"


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.DocumentOutput",
    jsii_struct_bases=[],
    name_mapping={"name": "name", "output_type": "outputType"},
)
class DocumentOutput:
    def __init__(self, *, name: builtins.str, output_type: DataTypeEnum) -> None:
        '''Outputs from a document via one of the steps.

        The outputs are available for use in other documents calling this document via aws:executeAutomation.
        The outputs returned from an automation document are returned as a String list named "ouptuts".

        :param name: The step and name (in STEP.NAME format) to identify an output expected from one of the document steps. The name must therefore identify a step and an output or will fail validation.
        :param output_type: The DataType expected by this output. This will be validated in simulation mode and will also be used when printing to yaml/json.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "name": name,
            "output_type": output_type,
        }

    @builtins.property
    def name(self) -> builtins.str:
        '''The step and name (in STEP.NAME format) to identify an output expected from one of the document steps. The name must therefore identify a step and an output or will fail validation.

        Example::

            MyStep.MyOutput
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def output_type(self) -> DataTypeEnum:
        '''The DataType expected by this output.

        This will be validated in simulation mode and will also be used when printing to yaml/json.
        '''
        result = self._values.get("output_type")
        assert result is not None, "Required property 'output_type' is missing"
        return typing.cast(DataTypeEnum, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DocumentOutput(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DocumentSource(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.DocumentSource",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromArn") # type: ignore[misc]
    @builtins.classmethod
    def from_arn(cls, arn: "IStringVariable") -> "DocumentSource":
        '''
        :param arn: -
        '''
        return typing.cast("DocumentSource", jsii.sinvoke(cls, "fromArn", [arn]))

    @jsii.member(jsii_name="fromName") # type: ignore[misc]
    @builtins.classmethod
    def from_name(
        cls,
        name: "IStringVariable",
        version: typing.Optional["IStringVariable"] = None,
    ) -> "DocumentSource":
        '''
        :param name: -
        :param version: -
        '''
        return typing.cast("DocumentSource", jsii.sinvoke(cls, "fromName", [name, version]))

    @jsii.member(jsii_name="formatRequest") # type: ignore[misc]
    @abc.abstractmethod
    def format_request(self) -> typing.Mapping[builtins.str, typing.Any]:
        ...

    @jsii.member(jsii_name="requiredInputs") # type: ignore[misc]
    @abc.abstractmethod
    def required_inputs(self) -> typing.List[builtins.str]:
        ...


class _DocumentSourceProxy(DocumentSource):
    @jsii.member(jsii_name="formatRequest")
    def format_request(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatRequest", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, DocumentSource).__jsii_proxy_class__ = lambda : _DocumentSourceProxy


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.EnvironmentProps",
    jsii_struct_bases=[],
    name_mapping={"environment": "environment"},
)
class EnvironmentProps:
    def __init__(self, *, environment: "IEnvironment") -> None:
        '''
        :param environment: (Optional) Specify here the environment in which to execute the scripts. Use the DockerEnvironment to execute the commands inside the docker. You can alternatively use the LoggingEnvironment which simply logs the commands or MockEnvironment which saves them for validation. Default: LoggingEnvironment
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "environment": environment,
        }

    @builtins.property
    def environment(self) -> "IEnvironment":
        '''(Optional) Specify here the environment in which to execute the scripts.

        Use the DockerEnvironment to execute the commands inside the docker.
        You can alternatively use the LoggingEnvironment which simply logs the commands
        or MockEnvironment which saves them for validation.

        :default: LoggingEnvironment
        '''
        result = self._values.get("environment")
        assert result is not None, "Required property 'environment' is missing"
        return typing.cast("IEnvironment", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "EnvironmentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteAutomationOutputs",
    jsii_struct_bases=[],
    name_mapping={
        "execution_id": "executionId",
        "status": "status",
        "output": "output",
    },
)
class ExecuteAutomationOutputs:
    def __init__(
        self,
        *,
        execution_id: builtins.str,
        status: builtins.str,
        output: typing.Optional[typing.Sequence[builtins.str]] = None,
    ) -> None:
        '''Outputs for aws:executeAutomation.

        :param execution_id: The ID of the secondary automation.
        :param status: The status of the secondary automation.
        :param output: The output generated by the secondary automation.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "execution_id": execution_id,
            "status": status,
        }
        if output is not None:
            self._values["output"] = output

    @builtins.property
    def execution_id(self) -> builtins.str:
        '''The ID of the secondary automation.'''
        result = self._values.get("execution_id")
        assert result is not None, "Required property 'execution_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def status(self) -> builtins.str:
        '''The status of the secondary automation.'''
        result = self._values.get("status")
        assert result is not None, "Required property 'status' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def output(self) -> typing.Optional[typing.List[builtins.str]]:
        '''The output generated by the secondary automation.'''
        result = self._values.get("output")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ExecuteAutomationOutputs(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteAutomationProps",
    jsii_struct_bases=[],
    name_mapping={
        "document_name": "documentName",
        "document_version": "documentVersion",
        "max_concurrency": "maxConcurrency",
        "max_errors": "maxErrors",
        "runtime_parameters": "runtimeParameters",
        "tags": "tags",
        "target_locations": "targetLocations",
        "target_maps": "targetMaps",
        "target_parameter_name": "targetParameterName",
        "targets": "targets",
    },
)
class ExecuteAutomationProps:
    def __init__(
        self,
        *,
        document_name: builtins.str,
        document_version: typing.Optional[builtins.str] = None,
        max_concurrency: typing.Optional[builtins.str] = None,
        max_errors: typing.Optional[builtins.str] = None,
        runtime_parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        tags: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_locations: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_maps: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_parameter_name: typing.Optional[builtins.str] = None,
        targets: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
    ) -> None:
        '''Inputs for aws:executeAutomation.

        :param document_name: 
        :param document_version: 
        :param max_concurrency: 
        :param max_errors: 
        :param runtime_parameters: 
        :param tags: 
        :param target_locations: 
        :param target_maps: 
        :param target_parameter_name: 
        :param targets: 
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "document_name": document_name,
        }
        if document_version is not None:
            self._values["document_version"] = document_version
        if max_concurrency is not None:
            self._values["max_concurrency"] = max_concurrency
        if max_errors is not None:
            self._values["max_errors"] = max_errors
        if runtime_parameters is not None:
            self._values["runtime_parameters"] = runtime_parameters
        if tags is not None:
            self._values["tags"] = tags
        if target_locations is not None:
            self._values["target_locations"] = target_locations
        if target_maps is not None:
            self._values["target_maps"] = target_maps
        if target_parameter_name is not None:
            self._values["target_parameter_name"] = target_parameter_name
        if targets is not None:
            self._values["targets"] = targets

    @builtins.property
    def document_name(self) -> builtins.str:
        result = self._values.get("document_name")
        assert result is not None, "Required property 'document_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def document_version(self) -> typing.Optional[builtins.str]:
        result = self._values.get("document_version")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_concurrency(self) -> typing.Optional[builtins.str]:
        result = self._values.get("max_concurrency")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_errors(self) -> typing.Optional[builtins.str]:
        result = self._values.get("max_errors")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def runtime_parameters(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        result = self._values.get("runtime_parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def tags(
        self,
    ) -> typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]]:
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]], result)

    @builtins.property
    def target_locations(
        self,
    ) -> typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]]:
        result = self._values.get("target_locations")
        return typing.cast(typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]], result)

    @builtins.property
    def target_maps(
        self,
    ) -> typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]]:
        result = self._values.get("target_maps")
        return typing.cast(typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]], result)

    @builtins.property
    def target_parameter_name(self) -> typing.Optional[builtins.str]:
        result = self._values.get("target_parameter_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def targets(
        self,
    ) -> typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]]:
        result = self._values.get("targets")
        return typing.cast(typing.Optional[typing.List[typing.Mapping[builtins.str, typing.Any]]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ExecuteAutomationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ExecuteScriptSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteScriptSimulation",
):
    '''AutomationStep implementation for aws:executeScript https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeScript.html.'''

    def __init__(self, step: "ExecuteScriptStep") -> None:
        '''
        :param step: -
        '''
        jsii.create(self.__class__, self, [step])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''Runs the simulation.

        Nests returned object into a "Payload" key to mimic SSM behavior.
        Switch by language and execute code based on specified language.

        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="executeScriptStep")
    def execute_script_step(self) -> "ExecuteScriptStep":
        return typing.cast("ExecuteScriptStep", jsii.get(self, "executeScriptStep"))


class ExecuteStateMachineSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteStateMachineSimulation",
):
    '''AutomationStep implementation of `aws:executeStateMachine <https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeStateMachine.html>`_.'''

    def __init__(
        self,
        step: "ExecuteStateMachineStep",
        *,
        aws_invoker: "IAwsInvoker",
    ) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="executeStateMachineStep")
    def execute_state_machine_step(self) -> "ExecuteStateMachineStep":
        return typing.cast("ExecuteStateMachineStep", jsii.get(self, "executeStateMachineStep"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="props")
    def props(self) -> AwsInvocationSimulationProps:
        return typing.cast(AwsInvocationSimulationProps, jsii.get(self, "props"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.GitContentProps",
    jsii_struct_bases=[],
    name_mapping={
        "repository": "repository",
        "branch": "branch",
        "commit_id": "commitId",
        "password": "password",
        "private_ssh_key": "privateSshKey",
        "skip_host_key_checking": "skipHostKeyChecking",
        "user_name": "userName",
    },
)
class GitContentProps:
    def __init__(
        self,
        *,
        repository: "IStringVariable",
        branch: typing.Optional[builtins.str] = None,
        commit_id: typing.Optional[builtins.str] = None,
        password: typing.Optional["SecureVariable"] = None,
        private_ssh_key: typing.Optional["SecureVariable"] = None,
        skip_host_key_checking: typing.Optional["IBooleanVariable"] = None,
        user_name: typing.Optional["SecureVariable"] = None,
    ) -> None:
        '''Properties for sourceType Git.

        :param repository: The Git repository URL to the file or directory you want to download.
        :param branch: The default is master. branch parameter is required only if your SSM document is stored in a branch other than master. Supply either commitId or branch (or neither).
        :param commit_id: The default is head. To use the version of your SSM document in a commit other than the latest, specify the full commit ID. For example: "bbc1ddb94...b76d3bEXAMPLE". Supply either commitId or branch (or neither).
        :param password: The password to use when connecting to the repository you specify using HTTP.
        :param private_ssh_key: The SSH key to use when connecting to the repository you specify.
        :param skip_host_key_checking: Determines the value of the StrictHostKeyChecking option when connecting to the repository you specify. The default value is false.
        :param user_name: The username to use when connecting to the repository you specify using HTTP.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "repository": repository,
        }
        if branch is not None:
            self._values["branch"] = branch
        if commit_id is not None:
            self._values["commit_id"] = commit_id
        if password is not None:
            self._values["password"] = password
        if private_ssh_key is not None:
            self._values["private_ssh_key"] = private_ssh_key
        if skip_host_key_checking is not None:
            self._values["skip_host_key_checking"] = skip_host_key_checking
        if user_name is not None:
            self._values["user_name"] = user_name

    @builtins.property
    def repository(self) -> "IStringVariable":
        '''The Git repository URL to the file or directory you want to download.'''
        result = self._values.get("repository")
        assert result is not None, "Required property 'repository' is missing"
        return typing.cast("IStringVariable", result)

    @builtins.property
    def branch(self) -> typing.Optional[builtins.str]:
        '''The default is master.

        branch parameter is required only if your SSM document is stored in a branch other than master.
        Supply either commitId or branch (or neither).

        Example::

            main
        '''
        result = self._values.get("branch")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def commit_id(self) -> typing.Optional[builtins.str]:
        '''The default is head.

        To use the version of your SSM document in a commit other than the latest, specify the full commit ID.
        For example: "bbc1ddb94...b76d3bEXAMPLE".
        Supply either commitId or branch (or neither).
        '''
        result = self._values.get("commit_id")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def password(self) -> typing.Optional["SecureVariable"]:
        '''The password to use when connecting to the repository you specify using HTTP.'''
        result = self._values.get("password")
        return typing.cast(typing.Optional["SecureVariable"], result)

    @builtins.property
    def private_ssh_key(self) -> typing.Optional["SecureVariable"]:
        '''The SSH key to use when connecting to the repository you specify.'''
        result = self._values.get("private_ssh_key")
        return typing.cast(typing.Optional["SecureVariable"], result)

    @builtins.property
    def skip_host_key_checking(self) -> typing.Optional["IBooleanVariable"]:
        '''Determines the value of the StrictHostKeyChecking option when connecting to the repository you specify.

        The default value is false.
        '''
        result = self._values.get("skip_host_key_checking")
        return typing.cast(typing.Optional["IBooleanVariable"], result)

    @builtins.property
    def user_name(self) -> typing.Optional["SecureVariable"]:
        '''The username to use when connecting to the repository you specify using HTTP.'''
        result = self._values.get("user_name")
        return typing.cast(typing.Optional["SecureVariable"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GitContentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.GitHubContentProps",
    jsii_struct_bases=[],
    name_mapping={
        "owner": "owner",
        "path": "path",
        "repository": "repository",
        "token_info": "tokenInfo",
        "branch": "branch",
        "commit_id": "commitId",
    },
)
class GitHubContentProps:
    def __init__(
        self,
        *,
        owner: "IStringVariable",
        path: "IStringVariable",
        repository: "IStringVariable",
        token_info: "SecureVariable",
        branch: typing.Optional[builtins.str] = None,
        commit_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Properties for sourceType GitHub.

        :param owner: The repository owner.
        :param path: The path to the file or directory you want to download.
        :param repository: The name of the repository.
        :param token_info: The Systems Manager parameter (a SecureString parameter) where you store your GitHub access token information.
        :param branch: The default is master. branch parameter is required only if your SSM document is stored in a branch other than master. Supply either commitId or branch (or neither).
        :param commit_id: The default is head. To use the version of your SSM document in a commit other than the latest, specify the full commit ID. For example: "bbc1ddb94...b76d3bEXAMPLE". Supply either commitId or branch (or neither).
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "owner": owner,
            "path": path,
            "repository": repository,
            "token_info": token_info,
        }
        if branch is not None:
            self._values["branch"] = branch
        if commit_id is not None:
            self._values["commit_id"] = commit_id

    @builtins.property
    def owner(self) -> "IStringVariable":
        '''The repository owner.'''
        result = self._values.get("owner")
        assert result is not None, "Required property 'owner' is missing"
        return typing.cast("IStringVariable", result)

    @builtins.property
    def path(self) -> "IStringVariable":
        '''The path to the file or directory you want to download.'''
        result = self._values.get("path")
        assert result is not None, "Required property 'path' is missing"
        return typing.cast("IStringVariable", result)

    @builtins.property
    def repository(self) -> "IStringVariable":
        '''The name of the repository.'''
        result = self._values.get("repository")
        assert result is not None, "Required property 'repository' is missing"
        return typing.cast("IStringVariable", result)

    @builtins.property
    def token_info(self) -> "SecureVariable":
        '''The Systems Manager parameter (a SecureString parameter) where you store your GitHub access token information.'''
        result = self._values.get("token_info")
        assert result is not None, "Required property 'token_info' is missing"
        return typing.cast("SecureVariable", result)

    @builtins.property
    def branch(self) -> typing.Optional[builtins.str]:
        '''The default is master.

        branch parameter is required only if your SSM document is stored in a branch other than master.
        Supply either commitId or branch (or neither).

        Example::

            main
        '''
        result = self._values.get("branch")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def commit_id(self) -> typing.Optional[builtins.str]:
        '''The default is head.

        To use the version of your SSM document in a commit other than the latest, specify the full commit ID.
        For example: "bbc1ddb94...b76d3bEXAMPLE".
        Supply either commitId or branch (or neither).
        '''
        result = self._values.get("commit_id")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "GitHubContentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class HelloWorld(
    aws_cdk.Stack,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HelloWorld",
):
    def __init__(self, app: constructs.Construct, id: builtins.str) -> None:
        '''
        :param app: -
        :param id: -
        '''
        jsii.create(self.__class__, self, [app, id])


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.HttpContentProps",
    jsii_struct_bases=[],
    name_mapping={
        "url": "url",
        "allow_insecure_download": "allowInsecureDownload",
        "auth_method": "authMethod",
    },
)
class HttpContentProps:
    def __init__(
        self,
        *,
        url: "IStringVariable",
        allow_insecure_download: typing.Optional["IBooleanVariable"] = None,
        auth_method: typing.Optional[AuthMethod] = None,
    ) -> None:
        '''Properties for sourceType HTTP.

        :param url: The URL to the file or directory you want to download.
        :param allow_insecure_download: Determines whether a download can be performed over a connection that isn't encrypted with Secure Socket Layer (SSL) or Transport Layer Security (TLS). The default value is false. We don't recommend performing downloads without encryption. If you choose to do so, you assume all associated risks. Security is a shared responsibility between AWS and you. This is described as the shared responsibility model. To learn more, see the shared responsibility model.
        :param auth_method: Determines whether a username and password are used for authentication when connecting to the url you specify. If you specify Basic or Digest, you must provide values for the username and password parameters. To use the Digest method, SSM Agent version 3.0.1181.0 or later must be installed on your instance. The Digest method supports MD5 and SHA256 encryption.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "url": url,
        }
        if allow_insecure_download is not None:
            self._values["allow_insecure_download"] = allow_insecure_download
        if auth_method is not None:
            self._values["auth_method"] = auth_method

    @builtins.property
    def url(self) -> "IStringVariable":
        '''The URL to the file or directory you want to download.'''
        result = self._values.get("url")
        assert result is not None, "Required property 'url' is missing"
        return typing.cast("IStringVariable", result)

    @builtins.property
    def allow_insecure_download(self) -> typing.Optional["IBooleanVariable"]:
        '''Determines whether a download can be performed over a connection that isn't encrypted with Secure Socket Layer (SSL) or Transport Layer Security (TLS).

        The default value is false. We don't recommend performing downloads without encryption. If you choose to do so, you assume all associated risks.
        Security is a shared responsibility between AWS and you. This is described as the shared responsibility model. To learn more, see the shared responsibility model.
        '''
        result = self._values.get("allow_insecure_download")
        return typing.cast(typing.Optional["IBooleanVariable"], result)

    @builtins.property
    def auth_method(self) -> typing.Optional[AuthMethod]:
        '''Determines whether a username and password are used for authentication when connecting to the url you specify.

        If you specify Basic or Digest, you must provide values for the username and password parameters.
        To use the Digest method, SSM Agent version 3.0.1181.0 or later must be installed on your instance. The Digest method supports MD5 and SHA256 encryption.
        '''
        result = self._values.get("auth_method")
        return typing.cast(typing.Optional[AuthMethod], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "HttpContentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IApproveHook")
class IApproveHook(typing_extensions.Protocol):
    '''This can be used to provide a hook for approval implementation for ApproveStep.'''

    @jsii.member(jsii_name="ask")
    def ask(self, approver: builtins.str) -> builtins.bool:
        '''Ask for approval.

        :param approver: -
        '''
        ...


class _IApproveHookProxy:
    '''This can be used to provide a hook for approval implementation for ApproveStep.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IApproveHook"

    @jsii.member(jsii_name="ask")
    def ask(self, approver: builtins.str) -> builtins.bool:
        '''Ask for approval.

        :param approver: -
        '''
        return typing.cast(builtins.bool, jsii.invoke(self, "ask", [approver]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IApproveHook).__jsii_proxy_class__ = lambda : _IApproveHookProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IAutomationComponent")
class IAutomationComponent(typing_extensions.Protocol):
    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: AutomationDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        ...


class _IAutomationComponentProxy:
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IAutomationComponent"

    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: AutomationDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        return typing.cast(None, jsii.invoke(self, "addToDocument", [doc]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IAutomationComponent).__jsii_proxy_class__ = lambda : _IAutomationComponentProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IAwsInvoker")
class IAwsInvoker(typing_extensions.Protocol):
    '''Aws invoker interface.

    This can be used to provide a hook for AWS invocation of steps so as to mock AWS behavior for simulation runs.
    '''

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        *,
        aws_api: builtins.str,
        aws_params: typing.Mapping[builtins.str, typing.Any],
        service: builtins.str,
    ) -> typing.Any:
        '''Invoke AWS with the provided invocation request.

        :param aws_api: (Required) AWS api to invoke; should be referenced using lowerCamelCase.
        :param aws_params: (Required )AWS params.
        :param service: (Required) AWS service to invoke.

        :return: the AWS response object

        :see: Invocation
        '''
        ...


class _IAwsInvokerProxy:
    '''Aws invoker interface.

    This can be used to provide a hook for AWS invocation of steps so as to mock AWS behavior for simulation runs.
    '''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IAwsInvoker"

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        *,
        aws_api: builtins.str,
        aws_params: typing.Mapping[builtins.str, typing.Any],
        service: builtins.str,
    ) -> typing.Any:
        '''Invoke AWS with the provided invocation request.

        :param aws_api: (Required) AWS api to invoke; should be referenced using lowerCamelCase.
        :param aws_params: (Required )AWS params.
        :param service: (Required) AWS service to invoke.

        :return: the AWS response object

        :see: Invocation
        '''
        invocation = Invocation(
            aws_api=aws_api, aws_params=aws_params, service=service
        )

        return typing.cast(typing.Any, jsii.invoke(self, "invoke", [invocation]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IAwsInvoker).__jsii_proxy_class__ = lambda : _IAwsInvokerProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.ICommandComponent")
class ICommandComponent(typing_extensions.Protocol):
    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: CommandDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        ...


class _ICommandComponentProxy:
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.ICommandComponent"

    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: CommandDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        return typing.cast(None, jsii.invoke(self, "addToDocument", [doc]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ICommandComponent).__jsii_proxy_class__ = lambda : _ICommandComponentProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IDownloadableContent")
class IDownloadableContent(typing_extensions.Protocol):
    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> builtins.str:
        ...

    @jsii.member(jsii_name="formatSourceInfo")
    def format_source_info(self) -> typing.Mapping[builtins.str, typing.Any]:
        ...

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        ...


class _IDownloadableContentProxy:
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IDownloadableContent"

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "sourceType"))

    @jsii.member(jsii_name="formatSourceInfo")
    def format_source_info(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatSourceInfo", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IDownloadableContent).__jsii_proxy_class__ = lambda : _IDownloadableContentProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IEnvironment")
class IEnvironment(typing_extensions.Protocol):
    '''IEnvironment is an abstraction for EC2 instances.

    The implementations of this interface provides customers with
    alternatives to test their commands rather than a real instance.
    '''

    @jsii.member(jsii_name="run")
    def run(self, command: builtins.str) -> builtins.str:
        '''
        :param command: -
        '''
        ...


class _IEnvironmentProxy:
    '''IEnvironment is an abstraction for EC2 instances.

    The implementations of this interface provides customers with
    alternatives to test their commands rather than a real instance.
    '''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IEnvironment"

    @jsii.member(jsii_name="run")
    def run(self, command: builtins.str) -> builtins.str:
        '''
        :param command: -
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "run", [command]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IEnvironment).__jsii_proxy_class__ = lambda : _IEnvironmentProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IExecuteAutomationHook")
class IExecuteAutomationHook(typing_extensions.Protocol):
    '''Interface for simulating aws:executeAutomation.'''

    @jsii.member(jsii_name="execute")
    def execute(
        self,
        *,
        document_name: builtins.str,
        document_version: typing.Optional[builtins.str] = None,
        max_concurrency: typing.Optional[builtins.str] = None,
        max_errors: typing.Optional[builtins.str] = None,
        runtime_parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        tags: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_locations: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_maps: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_parameter_name: typing.Optional[builtins.str] = None,
        targets: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
    ) -> ExecuteAutomationOutputs:
        '''Simulate the aws:executeAutomation.

        :param document_name: 
        :param document_version: 
        :param max_concurrency: 
        :param max_errors: 
        :param runtime_parameters: 
        :param tags: 
        :param target_locations: 
        :param target_maps: 
        :param target_parameter_name: 
        :param targets: 
        '''
        ...


class _IExecuteAutomationHookProxy:
    '''Interface for simulating aws:executeAutomation.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IExecuteAutomationHook"

    @jsii.member(jsii_name="execute")
    def execute(
        self,
        *,
        document_name: builtins.str,
        document_version: typing.Optional[builtins.str] = None,
        max_concurrency: typing.Optional[builtins.str] = None,
        max_errors: typing.Optional[builtins.str] = None,
        runtime_parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        tags: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_locations: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_maps: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_parameter_name: typing.Optional[builtins.str] = None,
        targets: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
    ) -> ExecuteAutomationOutputs:
        '''Simulate the aws:executeAutomation.

        :param document_name: 
        :param document_version: 
        :param max_concurrency: 
        :param max_errors: 
        :param runtime_parameters: 
        :param tags: 
        :param target_locations: 
        :param target_maps: 
        :param target_parameter_name: 
        :param targets: 
        '''
        props = ExecuteAutomationProps(
            document_name=document_name,
            document_version=document_version,
            max_concurrency=max_concurrency,
            max_errors=max_errors,
            runtime_parameters=runtime_parameters,
            tags=tags,
            target_locations=target_locations,
            target_maps=target_maps,
            target_parameter_name=target_parameter_name,
            targets=targets,
        )

        return typing.cast(ExecuteAutomationOutputs, jsii.invoke(self, "execute", [props]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IExecuteAutomationHook).__jsii_proxy_class__ = lambda : _IExecuteAutomationHookProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IGenericVariable")
class IGenericVariable(typing_extensions.Protocol):
    '''Variables hold references to a value.

    There are two implementations: HardCodedValue and ExportedVariable.
    This interface allows you to reference variable outputs from previous steps via the exported step.variable().
    If you do not have a reference to a variable, you may generate your own variable using the ExportedVariable ctor.
    In the event that a hardcoded value is passed into a step, you can reference the value with a HardCodedValue.
    '''

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{Variable}} or the value.

        Example::

            {{MyVariable}}
        '''
        ...

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        ...

    @jsii.member(jsii_name="resolve")
    def resolve(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        ...

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> typing.Any:
        '''JSON.stringify(variable) will implicitly invoke this variable.'''
        ...


class _IGenericVariableProxy:
    '''Variables hold references to a value.

    There are two implementations: HardCodedValue and ExportedVariable.
    This interface allows you to reference variable outputs from previous steps via the exported step.variable().
    If you do not have a reference to a variable, you may generate your own variable using the ExportedVariable ctor.
    In the event that a hardcoded value is passed into a step, you can reference the value with a HardCodedValue.
    '''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IGenericVariable"

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{Variable}} or the value.

        Example::

            {{MyVariable}}
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [inputs]))

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> typing.Any:
        '''JSON.stringify(variable) will implicitly invoke this variable.'''
        return typing.cast(typing.Any, jsii.invoke(self, "toJSON", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IGenericVariable).__jsii_proxy_class__ = lambda : _IGenericVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IMapListVariable")
class IMapListVariable(IGenericVariable, typing_extensions.Protocol):
    '''A map list variable.'''

    @jsii.member(jsii_name="resolveToMapList")
    def resolve_to_map_list(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.List[typing.Mapping[builtins.str, typing.Any]]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        ...


class _IMapListVariableProxy(
    jsii.proxy_for(IGenericVariable) # type: ignore[misc]
):
    '''A map list variable.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IMapListVariable"

    @jsii.member(jsii_name="resolveToMapList")
    def resolve_to_map_list(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.List[typing.Mapping[builtins.str, typing.Any]]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        return typing.cast(typing.List[typing.Mapping[builtins.str, typing.Any]], jsii.invoke(self, "resolveToMapList", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IMapListVariable).__jsii_proxy_class__ = lambda : _IMapListVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.INumberVariable")
class INumberVariable(IGenericVariable, typing_extensions.Protocol):
    '''A number variable.'''

    @jsii.member(jsii_name="resolveToNumber")
    def resolve_to_number(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> jsii.Number:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        ...


class _INumberVariableProxy(
    jsii.proxy_for(IGenericVariable) # type: ignore[misc]
):
    '''A number variable.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.INumberVariable"

    @jsii.member(jsii_name="resolveToNumber")
    def resolve_to_number(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> jsii.Number:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        return typing.cast(jsii.Number, jsii.invoke(self, "resolveToNumber", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, INumberVariable).__jsii_proxy_class__ = lambda : _INumberVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IObserver")
class IObserver(typing_extensions.Protocol):
    @jsii.member(jsii_name="accept")
    def accept(self, value: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''
        :param value: -
        '''
        ...


class _IObserverProxy:
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IObserver"

    @jsii.member(jsii_name="accept")
    def accept(self, value: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "accept", [value]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IObserver).__jsii_proxy_class__ = lambda : _IObserverProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IParameterResolver")
class IParameterResolver(typing_extensions.Protocol):
    '''Resolver for secure strings in Parameters.'''

    @jsii.member(jsii_name="resolve")
    def resolve(self, input: builtins.str) -> builtins.str:
        '''Resolve the token to its value.

        :param input: -
        '''
        ...


class _IParameterResolverProxy:
    '''Resolver for secure strings in Parameters.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IParameterResolver"

    @jsii.member(jsii_name="resolve")
    def resolve(self, input: builtins.str) -> builtins.str:
        '''Resolve the token to its value.

        :param input: -
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "resolve", [input]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IParameterResolver).__jsii_proxy_class__ = lambda : _IParameterResolverProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IPauseHook")
class IPauseHook(typing_extensions.Protocol):
    '''This can be used to provide a hook for pausing implementation for PauseStep.'''

    @jsii.member(jsii_name="pause")
    def pause(self) -> None:
        ...


class _IPauseHookProxy:
    '''This can be used to provide a hook for pausing implementation for PauseStep.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IPauseHook"

    @jsii.member(jsii_name="pause")
    def pause(self) -> None:
        return typing.cast(None, jsii.invoke(self, "pause", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IPauseHook).__jsii_proxy_class__ = lambda : _IPauseHookProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IRunCommandHook")
class IRunCommandHook(typing_extensions.Protocol):
    '''Interface for simulating aws:runCommand.'''

    @jsii.member(jsii_name="execute")
    def execute(
        self,
        *,
        document_name: builtins.str,
        targets: typing.Sequence[builtins.str],
        cloud_watch_output_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        document_hash: typing.Optional[builtins.str] = None,
        document_hash_type: typing.Optional[builtins.str] = None,
        max_concurrency: typing.Optional[jsii.Number] = None,
        max_errors: typing.Optional[jsii.Number] = None,
        notification_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        output_s3_bucket_name: typing.Optional[builtins.str] = None,
        output_s3_key_prefix: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        service_role_arn: typing.Optional[builtins.str] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
    ) -> "RunCommandOutputs":
        '''Simulate the aws:runCommand.

        :param document_name: If the Command type document is owned by you or AWS, specify the name of the document. If you're using a document shared with you by a different AWS account, specify the Amazon Resource Name (ARN) of the document.
        :param targets: The instance IDs where you want the command to run. You can specify a maximum of 50 IDs. You can also use the pseudo parameter {{RESOURCE_ID}} in place of instance IDs to run the command on all instances in the target group. For more information about pseudo parameters, see `About pseudo parameters <https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-register-tasks-parameters.html>`_. Another alternative is to send commands to a fleet of instances by using the Targets parameter. The Targets parameter accepts Amazon Elastic Compute Cloud (Amazon EC2) tags. For more information about how to use the Targets parameter, see `Using targets and rate controls to send commands to a fleet <https://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html>`_.
        :param cloud_watch_output_config: (Optional) Configuration options for sending command output to Amazon CloudWatch Logs. For more information about sending command output to CloudWatch Logs, see `Configuring Amazon CloudWatch Logs for Run Command <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-rc-setting-up-cwlogs.html>`_.
        :param comment: (Optional) User-defined information about the command.
        :param document_hash: (Optional) The hash for the document.
        :param document_hash_type: (Optional) The type of the hash.
        :param max_concurrency: (Optional) The maximum concurrency.
        :param max_errors: (Optional) The maximum errors.
        :param notification_config: (Optional) The configurations for sending notifications.
        :param output_s3_bucket_name: (Optional) The name of the S3 bucket for command output responses.
        :param output_s3_key_prefix: (Optional) The prefix.
        :param parameters: (Optional) The required and optional parameters specified in the document.
        :param service_role_arn: (Optional) The ARN of the AWS Identity and Access Management (IAM) role.
        :param timeout_seconds: (Optional) The amount of time in seconds to wait for a command to deliver to the AWS Systems Manager SSM Agent on an instance. If the command isn't received by the SSM Agent on the instance before the value specified is reached, then the status of the command changes to Delivery Timed Out.
        '''
        ...


class _IRunCommandHookProxy:
    '''Interface for simulating aws:runCommand.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IRunCommandHook"

    @jsii.member(jsii_name="execute")
    def execute(
        self,
        *,
        document_name: builtins.str,
        targets: typing.Sequence[builtins.str],
        cloud_watch_output_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        document_hash: typing.Optional[builtins.str] = None,
        document_hash_type: typing.Optional[builtins.str] = None,
        max_concurrency: typing.Optional[jsii.Number] = None,
        max_errors: typing.Optional[jsii.Number] = None,
        notification_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        output_s3_bucket_name: typing.Optional[builtins.str] = None,
        output_s3_key_prefix: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        service_role_arn: typing.Optional[builtins.str] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
    ) -> "RunCommandOutputs":
        '''Simulate the aws:runCommand.

        :param document_name: If the Command type document is owned by you or AWS, specify the name of the document. If you're using a document shared with you by a different AWS account, specify the Amazon Resource Name (ARN) of the document.
        :param targets: The instance IDs where you want the command to run. You can specify a maximum of 50 IDs. You can also use the pseudo parameter {{RESOURCE_ID}} in place of instance IDs to run the command on all instances in the target group. For more information about pseudo parameters, see `About pseudo parameters <https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-register-tasks-parameters.html>`_. Another alternative is to send commands to a fleet of instances by using the Targets parameter. The Targets parameter accepts Amazon Elastic Compute Cloud (Amazon EC2) tags. For more information about how to use the Targets parameter, see `Using targets and rate controls to send commands to a fleet <https://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html>`_.
        :param cloud_watch_output_config: (Optional) Configuration options for sending command output to Amazon CloudWatch Logs. For more information about sending command output to CloudWatch Logs, see `Configuring Amazon CloudWatch Logs for Run Command <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-rc-setting-up-cwlogs.html>`_.
        :param comment: (Optional) User-defined information about the command.
        :param document_hash: (Optional) The hash for the document.
        :param document_hash_type: (Optional) The type of the hash.
        :param max_concurrency: (Optional) The maximum concurrency.
        :param max_errors: (Optional) The maximum errors.
        :param notification_config: (Optional) The configurations for sending notifications.
        :param output_s3_bucket_name: (Optional) The name of the S3 bucket for command output responses.
        :param output_s3_key_prefix: (Optional) The prefix.
        :param parameters: (Optional) The required and optional parameters specified in the document.
        :param service_role_arn: (Optional) The ARN of the AWS Identity and Access Management (IAM) role.
        :param timeout_seconds: (Optional) The amount of time in seconds to wait for a command to deliver to the AWS Systems Manager SSM Agent on an instance. If the command isn't received by the SSM Agent on the instance before the value specified is reached, then the status of the command changes to Delivery Timed Out.
        '''
        props = RunCommandProps(
            document_name=document_name,
            targets=targets,
            cloud_watch_output_config=cloud_watch_output_config,
            comment=comment,
            document_hash=document_hash,
            document_hash_type=document_hash_type,
            max_concurrency=max_concurrency,
            max_errors=max_errors,
            notification_config=notification_config,
            output_s3_bucket_name=output_s3_bucket_name,
            output_s3_key_prefix=output_s3_key_prefix,
            parameters=parameters,
            service_role_arn=service_role_arn,
            timeout_seconds=timeout_seconds,
        )

        return typing.cast("RunCommandOutputs", jsii.invoke(self, "execute", [props]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IRunCommandHook).__jsii_proxy_class__ = lambda : _IRunCommandHookProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IRunDocumentLocation")
class IRunDocumentLocation(typing_extensions.Protocol):
    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="location")
    def location(self) -> "IStringVariable":
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        ...


class _IRunDocumentLocationProxy:
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IRunDocumentLocation"

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="location")
    def location(self) -> "IStringVariable":
        return typing.cast("IStringVariable", jsii.get(self, "location"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "type"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IRunDocumentLocation).__jsii_proxy_class__ = lambda : _IRunDocumentLocationProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.ISleepHook")
class ISleepHook(typing_extensions.Protocol):
    '''This can be used to provide a hook for sleeping for SleepStep (and other places where sleep is required).'''

    @jsii.member(jsii_name="sleep")
    def sleep(self, time_millis: jsii.Number) -> None:
        '''
        :param time_millis: -
        '''
        ...


class _ISleepHookProxy:
    '''This can be used to provide a hook for sleeping for SleepStep (and other places where sleep is required).'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.ISleepHook"

    @jsii.member(jsii_name="sleep")
    def sleep(self, time_millis: jsii.Number) -> None:
        '''
        :param time_millis: -
        '''
        return typing.cast(None, jsii.invoke(self, "sleep", [time_millis]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, ISleepHook).__jsii_proxy_class__ = lambda : _ISleepHookProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IStringListVariable")
class IStringListVariable(IGenericVariable, typing_extensions.Protocol):
    '''A string list variable.'''

    @jsii.member(jsii_name="resolveToStringList")
    def resolve_to_string_list(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.List[builtins.str]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        ...


class _IStringListVariableProxy(
    jsii.proxy_for(IGenericVariable) # type: ignore[misc]
):
    '''A string list variable.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IStringListVariable"

    @jsii.member(jsii_name="resolveToStringList")
    def resolve_to_string_list(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.List[builtins.str]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "resolveToStringList", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IStringListVariable).__jsii_proxy_class__ = lambda : _IStringListVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IStringMapVariable")
class IStringMapVariable(IGenericVariable, typing_extensions.Protocol):
    '''A string map variable.'''

    @jsii.member(jsii_name="resolveToStringMap")
    def resolve_to_string_map(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        ...


class _IStringMapVariableProxy(
    jsii.proxy_for(IGenericVariable) # type: ignore[misc]
):
    '''A string map variable.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IStringMapVariable"

    @jsii.member(jsii_name="resolveToStringMap")
    def resolve_to_string_map(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "resolveToStringMap", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IStringMapVariable).__jsii_proxy_class__ = lambda : _IStringMapVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IStringVariable")
class IStringVariable(IGenericVariable, typing_extensions.Protocol):
    '''A string variable.'''

    @jsii.member(jsii_name="resolveToString")
    def resolve_to_string(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.str:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        ...


class _IStringVariableProxy(
    jsii.proxy_for(IGenericVariable) # type: ignore[misc]
):
    '''A string variable.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IStringVariable"

    @jsii.member(jsii_name="resolveToString")
    def resolve_to_string(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.str:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "resolveToString", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IStringVariable).__jsii_proxy_class__ = lambda : _IStringVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IWebhook")
class IWebhook(typing_extensions.Protocol):
    '''Hook for simulating aws:invokeWebhook.'''

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        *,
        integration_name: builtins.str,
        body: typing.Optional[builtins.str] = None,
    ) -> "InvokeWebhookResult":
        '''Invoke the web hook.

        :param integration_name: The name of the Automation integration. For example, exampleIntegration. The integration you specify must already exist.
        :param body: (Optional) The payload you want to send when your webhook integration is invoked.
        '''
        ...


class _IWebhookProxy:
    '''Hook for simulating aws:invokeWebhook.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IWebhook"

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        *,
        integration_name: builtins.str,
        body: typing.Optional[builtins.str] = None,
    ) -> "InvokeWebhookResult":
        '''Invoke the web hook.

        :param integration_name: The name of the Automation integration. For example, exampleIntegration. The integration you specify must already exist.
        :param body: (Optional) The payload you want to send when your webhook integration is invoked.
        '''
        props = InvokeWebhookProps(integration_name=integration_name, body=body)

        return typing.cast("InvokeWebhookResult", jsii.invoke(self, "invoke", [props]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IWebhook).__jsii_proxy_class__ = lambda : _IWebhookProxy


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.Input",
    jsii_struct_bases=[],
    name_mapping={
        "input_type": "inputType",
        "name": "name",
        "allowed_pattern": "allowedPattern",
        "allowed_values": "allowedValues",
        "default_value": "defaultValue",
        "description": "description",
        "max_items": "maxItems",
        "min_items": "minItems",
    },
)
class Input:
    def __init__(
        self,
        *,
        input_type: DataTypeEnum,
        name: builtins.str,
        allowed_pattern: typing.Optional[builtins.str] = None,
        allowed_values: typing.Optional[typing.Sequence[builtins.str]] = None,
        default_value: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        max_items: typing.Optional[jsii.Number] = None,
        min_items: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Properties of inputs supported by SSM documents.

        These are NOT used for declaring step inputs, rather only for document inputs.
        See https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-doc-syntax.html

        :param input_type: (Required) The DataTypeEnum of the input.
        :param name: (Required) The name of the input by which to be referenced by steps in the document.
        :param allowed_pattern: (Optional) Pattern that this input value must match. Default: undefined
        :param allowed_values: (Optional) List of allowed values that this input may be. Default: undefined
        :param default_value: (Optional) Default value to use for this input if not specified when invoking the document. Default: undefined
        :param description: (Optional) The description of the input. Default: name
        :param max_items: (Optional) Maximum number of items that this input value (list) must contain. Default: undefined
        :param min_items: (Optional) Minimum number of items that this input value (list) must contain. Default: undefined
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "input_type": input_type,
            "name": name,
        }
        if allowed_pattern is not None:
            self._values["allowed_pattern"] = allowed_pattern
        if allowed_values is not None:
            self._values["allowed_values"] = allowed_values
        if default_value is not None:
            self._values["default_value"] = default_value
        if description is not None:
            self._values["description"] = description
        if max_items is not None:
            self._values["max_items"] = max_items
        if min_items is not None:
            self._values["min_items"] = min_items

    @builtins.property
    def input_type(self) -> DataTypeEnum:
        '''(Required) The DataTypeEnum of the input.'''
        result = self._values.get("input_type")
        assert result is not None, "Required property 'input_type' is missing"
        return typing.cast(DataTypeEnum, result)

    @builtins.property
    def name(self) -> builtins.str:
        '''(Required) The name of the input by which to be referenced by steps in the document.'''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def allowed_pattern(self) -> typing.Optional[builtins.str]:
        '''(Optional) Pattern that this input value must match.

        :default: undefined
        '''
        result = self._values.get("allowed_pattern")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def allowed_values(self) -> typing.Optional[typing.List[builtins.str]]:
        '''(Optional) List of allowed values that this input may be.

        :default: undefined
        '''
        result = self._values.get("allowed_values")
        return typing.cast(typing.Optional[typing.List[builtins.str]], result)

    @builtins.property
    def default_value(self) -> typing.Optional[builtins.str]:
        '''(Optional) Default value to use for this input if not specified when invoking the document.

        :default: undefined
        '''
        result = self._values.get("default_value")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) The description of the input.

        :default: name
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_items(self) -> typing.Optional[jsii.Number]:
        '''(Optional) Maximum number of items that this input value (list) must contain.

        :default: undefined
        '''
        result = self._values.get("max_items")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def min_items(self) -> typing.Optional[jsii.Number]:
        '''(Optional) Minimum number of items that this input value (list) must contain.

        :default: undefined
        '''
        result = self._values.get("min_items")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Input(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.Invocation",
    jsii_struct_bases=[],
    name_mapping={
        "aws_api": "awsApi",
        "aws_params": "awsParams",
        "service": "service",
    },
)
class Invocation:
    def __init__(
        self,
        *,
        aws_api: builtins.str,
        aws_params: typing.Mapping[builtins.str, typing.Any],
        service: builtins.str,
    ) -> None:
        '''
        :param aws_api: (Required) AWS api to invoke; should be referenced using lowerCamelCase.
        :param aws_params: (Required )AWS params.
        :param service: (Required) AWS service to invoke.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "aws_api": aws_api,
            "aws_params": aws_params,
            "service": service,
        }

    @builtins.property
    def aws_api(self) -> builtins.str:
        '''(Required) AWS api to invoke;

        should be referenced using lowerCamelCase.

        Example::

            describeInstance
        '''
        result = self._values.get("aws_api")
        assert result is not None, "Required property 'aws_api' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def aws_params(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''(Required )AWS params.

        Example::

            { 'Filters': [{'Name': 'instance-id', 'Values': ['{{ InstanceId }}'] }] }
        '''
        result = self._values.get("aws_params")
        assert result is not None, "Required property 'aws_params' is missing"
        return typing.cast(typing.Mapping[builtins.str, typing.Any], result)

    @builtins.property
    def service(self) -> builtins.str:
        '''(Required) AWS service to invoke.

        Example::

            ec2
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Invocation(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class InvokeLambdaFunctionSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.InvokeLambdaFunctionSimulation",
):
    '''AutomationStep implemenation for aws:invokeLambdaFunction https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-lamb.html.'''

    def __init__(
        self,
        step: "InvokeLambdaFunctionStep",
        *,
        aws_invoker: IAwsInvoker,
    ) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.InvokeWebhookProps",
    jsii_struct_bases=[],
    name_mapping={"integration_name": "integrationName", "body": "body"},
)
class InvokeWebhookProps:
    def __init__(
        self,
        *,
        integration_name: builtins.str,
        body: typing.Optional[builtins.str] = None,
    ) -> None:
        '''The properties for IWebhook.Invoke.

        :param integration_name: The name of the Automation integration. For example, exampleIntegration. The integration you specify must already exist.
        :param body: (Optional) The payload you want to send when your webhook integration is invoked.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "integration_name": integration_name,
        }
        if body is not None:
            self._values["body"] = body

    @builtins.property
    def integration_name(self) -> builtins.str:
        '''The name of the Automation integration.

        For example, exampleIntegration. The integration you specify must already exist.
        '''
        result = self._values.get("integration_name")
        assert result is not None, "Required property 'integration_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def body(self) -> typing.Optional[builtins.str]:
        '''(Optional) The payload you want to send when your webhook integration is invoked.'''
        result = self._values.get("body")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InvokeWebhookProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.InvokeWebhookResult",
    jsii_struct_bases=[],
    name_mapping={"response": "response", "response_code": "responseCode"},
)
class InvokeWebhookResult:
    def __init__(self, *, response: builtins.str, response_code: jsii.Number) -> None:
        '''Response from IWebhook.Invoke.

        :param response: The text received from the webhook provider response.
        :param response_code: The HTTP status code received from the webhook provider response.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "response": response,
            "response_code": response_code,
        }

    @builtins.property
    def response(self) -> builtins.str:
        '''The text received from the webhook provider response.'''
        result = self._values.get("response")
        assert result is not None, "Required property 'response' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def response_code(self) -> jsii.Number:
        '''The HTTP status code received from the webhook provider response.'''
        result = self._values.get("response_code")
        assert result is not None, "Required property 'response_code' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InvokeWebhookResult(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class InvokeWebhookSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.InvokeWebhookSimulation",
):
    '''AutomationStep implementation for `aws:invokeWebhook <https://docs.aws.amazon.com/systems-manager/latest/userguide/invoke-webhook.html>`_.'''

    def __init__(self, step: "InvokeWebhookStep", *, webhook: IWebhook) -> None:
        '''
        :param step: -
        :param webhook: (Optional) Hook for simulating aws:invokeWebhook. Default: - Returns 204 with an empty response
        '''
        props = InvokeWebhookSimulationProps(webhook=webhook)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.InvokeWebhookSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"webhook": "webhook"},
)
class InvokeWebhookSimulationProps:
    def __init__(self, *, webhook: IWebhook) -> None:
        '''Properties for InvokeWebhookStep.

        :param webhook: (Optional) Hook for simulating aws:invokeWebhook. Default: - Returns 204 with an empty response
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "webhook": webhook,
        }

    @builtins.property
    def webhook(self) -> IWebhook:
        '''(Optional) Hook for simulating aws:invokeWebhook.

        :default: - Returns 204 with an empty response
        '''
        result = self._values.get("webhook")
        assert result is not None, "Required property 'webhook' is missing"
        return typing.cast(IWebhook, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InvokeWebhookSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IRunDocumentLocation)
class LocalRunDocument(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.LocalRunDocument",
):
    def __init__(self, document_path: IStringVariable) -> None:
        '''Specify the path to the document on the local share.

        :param document_path: -
        '''
        jsii.create(self.__class__, self, [document_path])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="location")
    def location(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "location"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "type"))


@jsii.implements(IEnvironment)
class LoggingEnvironment(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.LoggingEnvironment",
):
    '''Environment that simply logs the commands that it receives and displays them on the console.'''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="run")
    def run(self, command: builtins.str) -> builtins.str:
        '''
        :param command: -
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "run", [command]))


@jsii.implements(IApproveHook)
class MockApprove(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.MockApprove",
):
    '''Mock implementation of IApproveHook.

    Does not simulate an approval request.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="ask")
    def ask(self, _approver: builtins.str) -> builtins.bool:
        '''Ask for approval.

        :param _approver: -
        '''
        return typing.cast(builtins.bool, jsii.invoke(self, "ask", [_approver]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="timesInvoked")
    def times_invoked(self) -> jsii.Number:
        return typing.cast(jsii.Number, jsii.get(self, "timesInvoked"))

    @times_invoked.setter
    def times_invoked(self, value: jsii.Number) -> None:
        jsii.set(self, "timesInvoked", value)


@jsii.implements(IAwsInvoker)
class MockAwsInvoker(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.MockAwsInvoker",
):
    '''Mock implementation of IAwsInvoker.

    This class can be reused for testing in exported JSII languages.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        *,
        aws_api: builtins.str,
        aws_params: typing.Mapping[builtins.str, typing.Any],
        service: builtins.str,
    ) -> typing.Any:
        '''Saves the invocation to be retieved using getInvocations().

        :param aws_api: (Required) AWS api to invoke; should be referenced using lowerCamelCase.
        :param aws_params: (Required )AWS params.
        :param service: (Required) AWS service to invoke.

        :return: the next result as set by the setReturn function
        '''
        invocation = Invocation(
            aws_api=aws_api, aws_params=aws_params, service=service
        )

        return typing.cast(typing.Any, jsii.invoke(self, "invoke", [invocation]))

    @jsii.member(jsii_name="nextReturn")
    def next_return(self, aws_result: typing.Any) -> "MockAwsInvoker":
        '''Set the next return value.

        This function can be chained to return subsequent return values.
        Values are read in order they were inserted.
        The last value is used as a default if there are no other values retrieved.
        In that way this function behaves the same way as Mockito .thenReturn(val).

        :param aws_result: -
        '''
        return typing.cast("MockAwsInvoker", jsii.invoke(self, "nextReturn", [aws_result]))

    @jsii.member(jsii_name="whenThen")
    def when_then(
        self,
        when: Invocation,
        then: typing.Mapping[builtins.str, typing.Any],
    ) -> None:
        '''Allows developers to mock out responses from the AwsInvoker depending on the input that it receives.

        :param when: defines the invocation to match and return the then.
        :param then: is the value that should be returned if the above when Invocation is matched.
        '''
        return typing.cast(None, jsii.invoke(self, "whenThen", [when, then]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="previousInvocations")
    def previous_invocations(self) -> typing.List[Invocation]:
        '''All of the invocations that have been submitted to this invoker until present.'''
        return typing.cast(typing.List[Invocation], jsii.get(self, "previousInvocations"))


@jsii.implements(IEnvironment)
class MockEnvironment(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.MockEnvironment",
):
    '''Environment that simply saves commands into a previousCommands variable.

    This is useful if you want to unit test the commands that would be sent to a real environment.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="run")
    def run(self, command: builtins.str) -> builtins.str:
        '''
        :param command: -
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "run", [command]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="previousCommands")
    def previous_commands(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.get(self, "previousCommands"))

    @previous_commands.setter
    def previous_commands(self, value: typing.List[builtins.str]) -> None:
        jsii.set(self, "previousCommands", value)


@jsii.implements(IPauseHook)
class MockPause(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.MockPause",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="pause")
    def pause(self) -> None:
        return typing.cast(None, jsii.invoke(self, "pause", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="timesInvoked")
    def times_invoked(self) -> jsii.Number:
        return typing.cast(jsii.Number, jsii.get(self, "timesInvoked"))

    @times_invoked.setter
    def times_invoked(self, value: jsii.Number) -> None:
        jsii.set(self, "timesInvoked", value)


@jsii.implements(ISleepHook)
class MockSleep(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.MockSleep",
):
    '''Mock ISleeper implementation.

    Simply logs that it is sleeping and returns immediately.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="sleep")
    def sleep(self, time_millis: jsii.Number) -> None:
        '''
        :param time_millis: -
        '''
        return typing.cast(None, jsii.invoke(self, "sleep", [time_millis]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sleepMilliInvocations")
    def sleep_milli_invocations(self) -> typing.List[jsii.Number]:
        return typing.cast(typing.List[jsii.Number], jsii.get(self, "sleepMilliInvocations"))


class NameDoc(
    DocumentSource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.NameDoc",
):
    def __init__(
        self,
        name: IStringVariable,
        version: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''
        :param name: -
        :param version: -
        '''
        jsii.create(self.__class__, self, [name, version])

    @jsii.member(jsii_name="formatRequest")
    def format_request(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatRequest", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="name")
    def name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "name"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="version")
    def version(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "version"))


class NoAuthMethod(
    AuthMethod,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.NoAuthMethod",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="toEntry")
    def to_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toEntry", []))


@jsii.implements(IObserver)
class NoopObserver(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.NoopObserver",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="accept")
    def accept(self, _value: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''
        :param _value: -
        '''
        return typing.cast(None, jsii.invoke(self, "accept", [_value]))


@jsii.enum(jsii_type="@cdklabs/cdk-ssm-documents.Operation")
class Operation(enum.Enum):
    '''Operation to use for comparing a Choice's or Preconditions with provided value.

    See https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-branch.html
    or https://docs.aws.amazon.com/systems-manager/latest/userguide/document-schemas-features.html
    '''

    BOOLEAN_EQUALS = "BOOLEAN_EQUALS"
    CONTAINS = "CONTAINS"
    ENDS_WITH = "ENDS_WITH"
    STARTS_WITH = "STARTS_WITH"
    STRING_EQUALS = "STRING_EQUALS"
    EQUALS_IGNORE_CASE = "EQUALS_IGNORE_CASE"
    NUMERIC_EQUALS = "NUMERIC_EQUALS"
    NUMERIC_GREATER = "NUMERIC_GREATER"
    NUMERIC_GREATER_OR_EQUALS = "NUMERIC_GREATER_OR_EQUALS"
    NUMERIC_LESSER = "NUMERIC_LESSER"
    NUMERIC_LESSER_OR_EQUALS = "NUMERIC_LESSER_OR_EQUALS"


class OperationEvaluator(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.OperationEvaluator",
):
    def __init__(self, operation: Operation) -> None:
        '''
        :param operation: -
        '''
        jsii.create(self.__class__, self, [operation])

    @jsii.member(jsii_name="fromOperationName") # type: ignore[misc]
    @builtins.classmethod
    def from_operation_name(cls, operation_name: builtins.str) -> Operation:
        '''Converts a string to OperationType.

        :param operation_name: an operation name to return its OperationType.

        :return: an OperationType

        :throws: if operationName is not supported
        '''
        return typing.cast(Operation, jsii.sinvoke(cls, "fromOperationName", [operation_name]))

    @jsii.member(jsii_name="evaluate")
    def evaluate(self, value1: typing.Any, value2: typing.Any) -> builtins.bool:
        '''Evaluates this operation against the provided inputs.

        :param value1: -
        :param value2: -

        :return: true if the evaluation is true. False otherwise.
        '''
        return typing.cast(builtins.bool, jsii.invoke(self, "evaluate", [value1, value2]))

    @jsii.member(jsii_name="toOperationName")
    def to_operation_name(self) -> builtins.str:
        '''
        :return: the string representation used by SSM for the operation.
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "toOperationName", []))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="STRING_TO_OPERATION")
    def STRING_TO_OPERATION(cls) -> typing.Mapping[builtins.str, Operation]:
        return typing.cast(typing.Mapping[builtins.str, Operation], jsii.sget(cls, "STRING_TO_OPERATION"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="operation")
    def operation(self) -> Operation:
        return typing.cast(Operation, jsii.get(self, "operation"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.Output",
    jsii_struct_bases=[],
    name_mapping={"name": "name", "output_type": "outputType", "selector": "selector"},
)
class Output:
    def __init__(
        self,
        *,
        name: builtins.str,
        output_type: DataTypeEnum,
        selector: builtins.str,
    ) -> None:
        '''Object used to specify step output.

        :param name: The name that can be used by subsequent steps to refernce the stored value. Note that Outputs will be PREPENDED with the step name.
        :param output_type: The DataType expected by this output. This will be validated in simulation mode and will also be used when printing to yaml/json.
        :param selector: Json selector for locating the value in the json step response.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "name": name,
            "output_type": output_type,
            "selector": selector,
        }

    @builtins.property
    def name(self) -> builtins.str:
        '''The name that can be used by subsequent steps to refernce the stored value.

        Note that Outputs will be PREPENDED with the step name.
        '''
        result = self._values.get("name")
        assert result is not None, "Required property 'name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def output_type(self) -> DataTypeEnum:
        '''The DataType expected by this output.

        This will be validated in simulation mode and will also be used when printing to yaml/json.
        '''
        result = self._values.get("output_type")
        assert result is not None, "Required property 'output_type' is missing"
        return typing.cast(DataTypeEnum, result)

    @builtins.property
    def selector(self) -> builtins.str:
        '''Json selector for locating the value in the json step response.'''
        result = self._values.get("selector")
        assert result is not None, "Required property 'selector' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "Output(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IPauseHook)
class PauseImpl(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.PauseImpl",
):
    '''This IPauseHook implementation provides a real pause and wait for user input of Enter.

    This implementation does not work well on all exported JSII languages.
    Users can provide their own impl using the IPauseHook interface.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="pause")
    def pause(self) -> None:
        return typing.cast(None, jsii.invoke(self, "pause", []))


class PauseSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.PauseSimulation",
):
    '''AutomationStep implementation for aws:pause https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-pause.html.'''

    def __init__(self, step: "PauseStep", *, pause_hook: IPauseHook) -> None:
        '''
        :param step: -
        :param pause_hook: (Optional) Pause hook to be called to pause the execution. To mock this implementation either inject an instance of IPauseHook or use the provided MockPause class. Default: PauseHook instance. PauseHook may not work in exported JSII languages. Override interface as needed.
        '''
        props = PauseSimulationProps(pause_hook=pause_hook)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        _inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''May perform a real pause based on the params used during instance creation.

        :param _inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [_inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.PauseSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"pause_hook": "pauseHook"},
)
class PauseSimulationProps:
    def __init__(self, *, pause_hook: IPauseHook) -> None:
        '''Properties for PauseStep.

        :param pause_hook: (Optional) Pause hook to be called to pause the execution. To mock this implementation either inject an instance of IPauseHook or use the provided MockPause class. Default: PauseHook instance. PauseHook may not work in exported JSII languages. Override interface as needed.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "pause_hook": pause_hook,
        }

    @builtins.property
    def pause_hook(self) -> IPauseHook:
        '''(Optional) Pause hook to be called to pause the execution.

        To mock this implementation either inject an instance of IPauseHook or use the provided MockPause class.

        :default: PauseHook instance. PauseHook may not work in exported JSII languages. Override interface as needed.
        '''
        result = self._values.get("pause_hook")
        assert result is not None, "Required property 'pause_hook' is missing"
        return typing.cast(IPauseHook, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PauseSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@cdklabs/cdk-ssm-documents.Platform")
class Platform(enum.Enum):
    '''Command steps are not all applicable to all platforms.

    Each command step declares which Platforms it supports.
    That allows customers to validate their CommandDocument against a given platform prior to execution.
    '''

    LINUX = "LINUX"
    WINDOWS = "WINDOWS"
    MAC_OS = "MAC_OS"


class Platforms(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.Platforms",
):
    @jsii.member(jsii_name="toPlatform") # type: ignore[misc]
    @builtins.classmethod
    def to_platform(cls, platform_string: builtins.str) -> Platform:
        '''Converts a string to Platform.

        :param platform_string: a platform name to return its Platform type.

        :return: a Platform

        :throws: if platofrmString is not supported
        '''
        return typing.cast(Platform, jsii.sinvoke(cls, "toPlatform", [platform_string]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="STRING_TO_PLATFORM")
    def STRING_TO_PLATFORM(cls) -> typing.Mapping[builtins.str, Platform]:
        return typing.cast(typing.Mapping[builtins.str, Platform], jsii.sget(cls, "STRING_TO_PLATFORM"))


class Precondition(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.Precondition",
):
    @jsii.member(jsii_name="newPlatformPrecondition") # type: ignore[misc]
    @builtins.classmethod
    def new_platform_precondition(cls, platform: Platform) -> "Precondition":
        '''Returns a new Precondition.

        :param platform: The platform the preconditions tests against.

        :return: new Precondition with platformType EnvironmentVariable, operation and a constant
        '''
        return typing.cast("Precondition", jsii.sinvoke(cls, "newPlatformPrecondition", [platform]))

    @jsii.member(jsii_name="asSsmEntry")
    def as_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :return: an object that can be used to print this choice into yaml/json format.
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "asSsmEntry", []))

    @jsii.member(jsii_name="evaluate")
    def evaluate(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.bool:
        '''Evaluates if the precondition is met, by comparing the variable with the constant using the operator.

        :param inputs: -

        :return: true if the evaluation is true. False otherwise.
        '''
        return typing.cast(builtins.bool, jsii.invoke(self, "evaluate", [inputs]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="INJECTED_PLAYFORM_TYPE_KEY")
    def INJECTED_PLAYFORM_TYPE_KEY(cls) -> builtins.str:
        return typing.cast(builtins.str, jsii.sget(cls, "INJECTED_PLAYFORM_TYPE_KEY"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="operationType")
    def operation_type(self) -> Operation:
        return typing.cast(Operation, jsii.get(self, "operationType"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="variable1")
    def variable1(self) -> IGenericVariable:
        return typing.cast(IGenericVariable, jsii.get(self, "variable1"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="variable2")
    def variable2(self) -> IGenericVariable:
        return typing.cast(IGenericVariable, jsii.get(self, "variable2"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.PreconditionProps",
    jsii_struct_bases=[],
    name_mapping={
        "operation_type": "operationType",
        "variable1": "variable1",
        "variable2": "variable2",
    },
)
class PreconditionProps:
    def __init__(
        self,
        *,
        operation_type: Operation,
        variable1: IGenericVariable,
        variable2: IGenericVariable,
    ) -> None:
        '''The precondition parameter is used as a comparator of SSM documents inputs to determine whether a command step would be executed or not.

        See https://docs.aws.amazon.com/systems-manager/latest/userguide/document-schemas-features.html

        :param operation_type: (Required) the operation used to compare the parameter with the variable.
        :param variable1: (Required) the variable to compare against the constant.
        :param variable2: (Required) the being compared against the variable.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "operation_type": operation_type,
            "variable1": variable1,
            "variable2": variable2,
        }

    @builtins.property
    def operation_type(self) -> Operation:
        '''(Required) the operation used to compare the parameter with the variable.'''
        result = self._values.get("operation_type")
        assert result is not None, "Required property 'operation_type' is missing"
        return typing.cast(Operation, result)

    @builtins.property
    def variable1(self) -> IGenericVariable:
        '''(Required) the variable to compare against the constant.'''
        result = self._values.get("variable1")
        assert result is not None, "Required property 'variable1' is missing"
        return typing.cast(IGenericVariable, result)

    @builtins.property
    def variable2(self) -> IGenericVariable:
        '''(Required) the being compared against the variable.'''
        result = self._values.get("variable2")
        assert result is not None, "Required property 'variable2' is missing"
        return typing.cast(IGenericVariable, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PreconditionProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class PsModuleSimulation(
    CommandSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.PsModuleSimulation",
):
    def __init__(self, step: "PsModuleStep", *, environment: IEnvironment) -> None:
        '''
        :param step: -
        :param environment: (Optional) Specify here the environment in which to execute the scripts. Use the DockerEnvironment to execute the commands inside the docker. You can alternatively use the LoggingEnvironment which simply logs the commands or MockEnvironment which saves them for validation. Default: LoggingEnvironment
        '''
        props = EnvironmentProps(environment=environment)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''Installs the module specified by source then runs the specified commands.

        :param inputs: -
        '''
        return typing.cast(None, jsii.invoke(self, "executeStep", [inputs]))


@jsii.implements(IAwsInvoker)
class ReflectiveAwsInvoker(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ReflectiveAwsInvoker",
):
    '''Implementation of IAwsInvoker that executes the AWS api for real.

    If using this implementation, be sure that AWS credentials are available to the execution.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        *,
        aws_api: builtins.str,
        aws_params: typing.Mapping[builtins.str, typing.Any],
        service: builtins.str,
    ) -> typing.Any:
        '''Invoke AWS with the provided invocation request.

        :param aws_api: (Required) AWS api to invoke; should be referenced using lowerCamelCase.
        :param aws_params: (Required )AWS params.
        :param service: (Required) AWS service to invoke.
        '''
        invocation = Invocation(
            aws_api=aws_api, aws_params=aws_params, service=service
        )

        return typing.cast(typing.Any, jsii.invoke(self, "invoke", [invocation]))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.RequiredAutomationSimulationProps",
    jsii_struct_bases=[],
    name_mapping={
        "approve_hook": "approveHook",
        "aws_invoker": "awsInvoker",
        "execute_automation_hook": "executeAutomationHook",
        "input_observer": "inputObserver",
        "output_observer": "outputObserver",
        "parameter_resolver": "parameterResolver",
        "pause_hook": "pauseHook",
        "run_command_hook": "runCommandHook",
        "sleep_hook": "sleepHook",
        "webhook": "webhook",
    },
)
class RequiredAutomationSimulationProps:
    def __init__(
        self,
        *,
        approve_hook: IApproveHook,
        aws_invoker: IAwsInvoker,
        execute_automation_hook: IExecuteAutomationHook,
        input_observer: IObserver,
        output_observer: IObserver,
        parameter_resolver: IParameterResolver,
        pause_hook: IPauseHook,
        run_command_hook: IRunCommandHook,
        sleep_hook: ISleepHook,
        webhook: IWebhook,
    ) -> None:
        '''The same interface as AutomationSimulationProps but all fields are required.

        :param approve_hook: 
        :param aws_invoker: 
        :param execute_automation_hook: 
        :param input_observer: 
        :param output_observer: 
        :param parameter_resolver: 
        :param pause_hook: 
        :param run_command_hook: 
        :param sleep_hook: 
        :param webhook: 
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "approve_hook": approve_hook,
            "aws_invoker": aws_invoker,
            "execute_automation_hook": execute_automation_hook,
            "input_observer": input_observer,
            "output_observer": output_observer,
            "parameter_resolver": parameter_resolver,
            "pause_hook": pause_hook,
            "run_command_hook": run_command_hook,
            "sleep_hook": sleep_hook,
            "webhook": webhook,
        }

    @builtins.property
    def approve_hook(self) -> IApproveHook:
        result = self._values.get("approve_hook")
        assert result is not None, "Required property 'approve_hook' is missing"
        return typing.cast(IApproveHook, result)

    @builtins.property
    def aws_invoker(self) -> IAwsInvoker:
        result = self._values.get("aws_invoker")
        assert result is not None, "Required property 'aws_invoker' is missing"
        return typing.cast(IAwsInvoker, result)

    @builtins.property
    def execute_automation_hook(self) -> IExecuteAutomationHook:
        result = self._values.get("execute_automation_hook")
        assert result is not None, "Required property 'execute_automation_hook' is missing"
        return typing.cast(IExecuteAutomationHook, result)

    @builtins.property
    def input_observer(self) -> IObserver:
        result = self._values.get("input_observer")
        assert result is not None, "Required property 'input_observer' is missing"
        return typing.cast(IObserver, result)

    @builtins.property
    def output_observer(self) -> IObserver:
        result = self._values.get("output_observer")
        assert result is not None, "Required property 'output_observer' is missing"
        return typing.cast(IObserver, result)

    @builtins.property
    def parameter_resolver(self) -> IParameterResolver:
        result = self._values.get("parameter_resolver")
        assert result is not None, "Required property 'parameter_resolver' is missing"
        return typing.cast(IParameterResolver, result)

    @builtins.property
    def pause_hook(self) -> IPauseHook:
        result = self._values.get("pause_hook")
        assert result is not None, "Required property 'pause_hook' is missing"
        return typing.cast(IPauseHook, result)

    @builtins.property
    def run_command_hook(self) -> IRunCommandHook:
        result = self._values.get("run_command_hook")
        assert result is not None, "Required property 'run_command_hook' is missing"
        return typing.cast(IRunCommandHook, result)

    @builtins.property
    def sleep_hook(self) -> ISleepHook:
        result = self._values.get("sleep_hook")
        assert result is not None, "Required property 'sleep_hook' is missing"
        return typing.cast(ISleepHook, result)

    @builtins.property
    def webhook(self) -> IWebhook:
        result = self._values.get("webhook")
        assert result is not None, "Required property 'webhook' is missing"
        return typing.cast(IWebhook, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RequiredAutomationSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.RequiredCommandSimulationProps",
    jsii_struct_bases=[],
    name_mapping={
        "environment": "environment",
        "simulation_platform": "simulationPlatform",
    },
)
class RequiredCommandSimulationProps:
    def __init__(
        self,
        *,
        environment: IEnvironment,
        simulation_platform: Platform,
    ) -> None:
        '''The same interface as CommandSimulationProps but all fields are required.

        :param environment: 
        :param simulation_platform: 
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "environment": environment,
            "simulation_platform": simulation_platform,
        }

    @builtins.property
    def environment(self) -> IEnvironment:
        result = self._values.get("environment")
        assert result is not None, "Required property 'environment' is missing"
        return typing.cast(IEnvironment, result)

    @builtins.property
    def simulation_platform(self) -> Platform:
        result = self._values.get("simulation_platform")
        assert result is not None, "Required property 'simulation_platform' is missing"
        return typing.cast(Platform, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RequiredCommandSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.enum(jsii_type="@cdklabs/cdk-ssm-documents.ResponseCode")
class ResponseCode(enum.Enum):
    '''Steps report their ResponseCode using this enum.

    A successful response will contain the outputs expected.
    A failed/canceled response will contain the stackTrace.
    '''

    SUCCESS = "SUCCESS"
    CANCELED = "CANCELED"
    FAILED = "FAILED"


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.RunCommandOutputs",
    jsii_struct_bases=[],
    name_mapping={
        "command_id": "commandId",
        "status": "status",
        "output": "output",
        "response_code": "responseCode",
    },
)
class RunCommandOutputs:
    def __init__(
        self,
        *,
        command_id: builtins.str,
        status: builtins.str,
        output: typing.Optional[builtins.str] = None,
        response_code: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Outputs for aws:runCommand.

        :param command_id: The ID of the command.
        :param status: The status of the command.
        :param output: The output of the command.
        :param response_code: The response code of the command.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "command_id": command_id,
            "status": status,
        }
        if output is not None:
            self._values["output"] = output
        if response_code is not None:
            self._values["response_code"] = response_code

    @builtins.property
    def command_id(self) -> builtins.str:
        '''The ID of the command.'''
        result = self._values.get("command_id")
        assert result is not None, "Required property 'command_id' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def status(self) -> builtins.str:
        '''The status of the command.'''
        result = self._values.get("status")
        assert result is not None, "Required property 'status' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def output(self) -> typing.Optional[builtins.str]:
        '''The output of the command.'''
        result = self._values.get("output")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def response_code(self) -> typing.Optional[jsii.Number]:
        '''The response code of the command.'''
        result = self._values.get("response_code")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RunCommandOutputs(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.RunCommandProps",
    jsii_struct_bases=[],
    name_mapping={
        "document_name": "documentName",
        "targets": "targets",
        "cloud_watch_output_config": "cloudWatchOutputConfig",
        "comment": "comment",
        "document_hash": "documentHash",
        "document_hash_type": "documentHashType",
        "max_concurrency": "maxConcurrency",
        "max_errors": "maxErrors",
        "notification_config": "notificationConfig",
        "output_s3_bucket_name": "outputS3BucketName",
        "output_s3_key_prefix": "outputS3KeyPrefix",
        "parameters": "parameters",
        "service_role_arn": "serviceRoleArn",
        "timeout_seconds": "timeoutSeconds",
    },
)
class RunCommandProps:
    def __init__(
        self,
        *,
        document_name: builtins.str,
        targets: typing.Sequence[builtins.str],
        cloud_watch_output_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        document_hash: typing.Optional[builtins.str] = None,
        document_hash_type: typing.Optional[builtins.str] = None,
        max_concurrency: typing.Optional[jsii.Number] = None,
        max_errors: typing.Optional[jsii.Number] = None,
        notification_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        output_s3_bucket_name: typing.Optional[builtins.str] = None,
        output_s3_key_prefix: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        service_role_arn: typing.Optional[builtins.str] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''Inputs for aws:runCommand.

        :param document_name: If the Command type document is owned by you or AWS, specify the name of the document. If you're using a document shared with you by a different AWS account, specify the Amazon Resource Name (ARN) of the document.
        :param targets: The instance IDs where you want the command to run. You can specify a maximum of 50 IDs. You can also use the pseudo parameter {{RESOURCE_ID}} in place of instance IDs to run the command on all instances in the target group. For more information about pseudo parameters, see `About pseudo parameters <https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-register-tasks-parameters.html>`_. Another alternative is to send commands to a fleet of instances by using the Targets parameter. The Targets parameter accepts Amazon Elastic Compute Cloud (Amazon EC2) tags. For more information about how to use the Targets parameter, see `Using targets and rate controls to send commands to a fleet <https://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html>`_.
        :param cloud_watch_output_config: (Optional) Configuration options for sending command output to Amazon CloudWatch Logs. For more information about sending command output to CloudWatch Logs, see `Configuring Amazon CloudWatch Logs for Run Command <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-rc-setting-up-cwlogs.html>`_.
        :param comment: (Optional) User-defined information about the command.
        :param document_hash: (Optional) The hash for the document.
        :param document_hash_type: (Optional) The type of the hash.
        :param max_concurrency: (Optional) The maximum concurrency.
        :param max_errors: (Optional) The maximum errors.
        :param notification_config: (Optional) The configurations for sending notifications.
        :param output_s3_bucket_name: (Optional) The name of the S3 bucket for command output responses.
        :param output_s3_key_prefix: (Optional) The prefix.
        :param parameters: (Optional) The required and optional parameters specified in the document.
        :param service_role_arn: (Optional) The ARN of the AWS Identity and Access Management (IAM) role.
        :param timeout_seconds: (Optional) The amount of time in seconds to wait for a command to deliver to the AWS Systems Manager SSM Agent on an instance. If the command isn't received by the SSM Agent on the instance before the value specified is reached, then the status of the command changes to Delivery Timed Out.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "document_name": document_name,
            "targets": targets,
        }
        if cloud_watch_output_config is not None:
            self._values["cloud_watch_output_config"] = cloud_watch_output_config
        if comment is not None:
            self._values["comment"] = comment
        if document_hash is not None:
            self._values["document_hash"] = document_hash
        if document_hash_type is not None:
            self._values["document_hash_type"] = document_hash_type
        if max_concurrency is not None:
            self._values["max_concurrency"] = max_concurrency
        if max_errors is not None:
            self._values["max_errors"] = max_errors
        if notification_config is not None:
            self._values["notification_config"] = notification_config
        if output_s3_bucket_name is not None:
            self._values["output_s3_bucket_name"] = output_s3_bucket_name
        if output_s3_key_prefix is not None:
            self._values["output_s3_key_prefix"] = output_s3_key_prefix
        if parameters is not None:
            self._values["parameters"] = parameters
        if service_role_arn is not None:
            self._values["service_role_arn"] = service_role_arn
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds

    @builtins.property
    def document_name(self) -> builtins.str:
        '''If the Command type document is owned by you or AWS, specify the name of the document.

        If you're using a document shared with you by a different AWS account, specify the Amazon Resource Name (ARN) of the document.
        '''
        result = self._values.get("document_name")
        assert result is not None, "Required property 'document_name' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def targets(self) -> typing.List[builtins.str]:
        '''The instance IDs where you want the command to run. You can specify a maximum of 50 IDs.

        You can also use the pseudo parameter {{RESOURCE_ID}} in place of instance IDs to run the command on all instances in the target group. For more information about pseudo parameters, see `About pseudo parameters <https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-register-tasks-parameters.html>`_.

        Another alternative is to send commands to a fleet of instances by using the Targets parameter. The Targets parameter accepts Amazon Elastic Compute Cloud (Amazon EC2) tags. For more information about how to use the Targets parameter, see `Using targets and rate controls to send commands to a fleet <https://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html>`_.
        '''
        result = self._values.get("targets")
        assert result is not None, "Required property 'targets' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def cloud_watch_output_config(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(Optional) Configuration options for sending command output to Amazon CloudWatch Logs.

        For more information about sending command output to CloudWatch Logs, see `Configuring Amazon CloudWatch Logs for Run Command <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-rc-setting-up-cwlogs.html>`_.
        '''
        result = self._values.get("cloud_watch_output_config")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def comment(self) -> typing.Optional[builtins.str]:
        '''(Optional) User-defined information about the command.'''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def document_hash(self) -> typing.Optional[builtins.str]:
        '''(Optional) The hash for the document.'''
        result = self._values.get("document_hash")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def document_hash_type(self) -> typing.Optional[builtins.str]:
        '''(Optional) The type of the hash.'''
        result = self._values.get("document_hash_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def max_concurrency(self) -> typing.Optional[jsii.Number]:
        '''(Optional) The maximum concurrency.'''
        result = self._values.get("max_concurrency")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def max_errors(self) -> typing.Optional[jsii.Number]:
        '''(Optional) The maximum errors.'''
        result = self._values.get("max_errors")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def notification_config(
        self,
    ) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(Optional) The configurations for sending notifications.'''
        result = self._values.get("notification_config")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def output_s3_bucket_name(self) -> typing.Optional[builtins.str]:
        '''(Optional) The name of the S3 bucket for command output responses.'''
        result = self._values.get("output_s3_bucket_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_s3_key_prefix(self) -> typing.Optional[builtins.str]:
        '''(Optional) The prefix.'''
        result = self._values.get("output_s3_key_prefix")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def parameters(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''(Optional) The required and optional parameters specified in the document.'''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def service_role_arn(self) -> typing.Optional[builtins.str]:
        '''(Optional) The ARN of the AWS Identity and Access Management (IAM) role.'''
        result = self._values.get("service_role_arn")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) The amount of time in seconds to wait for a command to deliver to the AWS Systems Manager SSM Agent on an instance.

        If the command isn't received by the SSM Agent on the instance before the value specified is reached, then the status of the command changes to Delivery Timed Out.
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RunCommandProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class RunCommandSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.RunCommandSimulation",
):
    '''AutomationStep implementation of `aws:runCommand <https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-runcommand.html>`_.'''

    def __init__(
        self,
        step: "RunCommandStep",
        *,
        run_command_hook: IRunCommandHook,
    ) -> None:
        '''
        :param step: -
        :param run_command_hook: Hook for simulating aws:runCommand. Default: - Uses AWS API to execute the document remotely.
        '''
        props = RunCommandSimulationProps(run_command_hook=run_command_hook)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''May perform a real approval ask based on the params used during instance creation.

        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.RunCommandSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"run_command_hook": "runCommandHook"},
)
class RunCommandSimulationProps:
    def __init__(self, *, run_command_hook: IRunCommandHook) -> None:
        '''Properties for RunCommandStep.

        :param run_command_hook: Hook for simulating aws:runCommand. Default: - Uses AWS API to execute the document remotely.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "run_command_hook": run_command_hook,
        }

    @builtins.property
    def run_command_hook(self) -> IRunCommandHook:
        '''Hook for simulating aws:runCommand.

        :default: - Uses AWS API to execute the document remotely.
        '''
        result = self._values.get("run_command_hook")
        assert result is not None, "Required property 'run_command_hook' is missing"
        return typing.cast(IRunCommandHook, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RunCommandSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class RunInstanceSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.RunInstanceSimulation",
):
    '''AutomationStep implemenation for aws:runInstance https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-runinstance.html.'''

    def __init__(self, step: "RunInstanceStep", *, aws_invoker: IAwsInvoker) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


class RunPowerShellScriptSimulation(
    CommandSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.RunPowerShellScriptSimulation",
):
    def __init__(
        self,
        step: "RunPowerShellScriptStep",
        *,
        environment: IEnvironment,
    ) -> None:
        '''
        :param step: -
        :param environment: (Optional) Specify here the environment in which to execute the scripts. Use the DockerEnvironment to execute the commands inside the docker. You can alternatively use the LoggingEnvironment which simply logs the commands or MockEnvironment which saves them for validation. Default: LoggingEnvironment
        '''
        props = EnvironmentProps(environment=environment)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''Executes the runCommands against the environment provided in the constructor.

        :param inputs: -
        '''
        return typing.cast(None, jsii.invoke(self, "executeStep", [inputs]))


class RunShellScriptSimulation(
    CommandSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.RunShellScriptSimulation",
):
    def __init__(
        self,
        step: "RunShellScriptStep",
        *,
        environment: IEnvironment,
    ) -> None:
        '''
        :param step: -
        :param environment: (Optional) Specify here the environment in which to execute the scripts. Use the DockerEnvironment to execute the commands inside the docker. You can alternatively use the LoggingEnvironment which simply logs the commands or MockEnvironment which saves them for validation. Default: LoggingEnvironment
        '''
        props = EnvironmentProps(environment=environment)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''Executes the runCommands against the environment provided in the constructor.

        :param inputs: -
        '''
        return typing.cast(None, jsii.invoke(self, "executeStep", [inputs]))


@jsii.implements(IDownloadableContent)
class S3Content(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.S3Content",
):
    def __init__(self, *, path: IStringVariable) -> None:
        '''
        :param path: The URL to the file or directory you want to download.
        '''
        props = S3ContentProps(path=path)

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="formatSourceInfo")
    def format_source_info(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatSourceInfo", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="path")
    def path(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "path"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "sourceType"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.S3ContentProps",
    jsii_struct_bases=[],
    name_mapping={"path": "path"},
)
class S3ContentProps:
    def __init__(self, *, path: IStringVariable) -> None:
        '''Properties.json for sourceType GitHub.

        :param path: The URL to the file or directory you want to download.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "path": path,
        }

    @builtins.property
    def path(self) -> IStringVariable:
        '''The URL to the file or directory you want to download.'''
        result = self._values.get("path")
        assert result is not None, "Required property 'path' is missing"
        return typing.cast(IStringVariable, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "S3ContentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IDownloadableContent)
class SSMDocumentContent(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.SSMDocumentContent",
):
    def __init__(self, *, document_source: DocumentSource) -> None:
        '''
        :param document_source: specify one of the following The name and version of the document in the following format: name:version. Version is optional. or The ARN for the document in the following format: arn:aws:ssm:region:account_id:document/document_name
        '''
        props = SsmDocumentContentProps(document_source=document_source)

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="formatSourceInfo")
    def format_source_info(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatSourceInfo", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="documentSource")
    def document_source(self) -> DocumentSource:
        return typing.cast(DocumentSource, jsii.get(self, "documentSource"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "sourceType"))


@jsii.enum(jsii_type="@cdklabs/cdk-ssm-documents.ScriptLanguage")
class ScriptLanguage(enum.Enum):
    PYTHON = "PYTHON"


@jsii.implements(IGenericVariable)
class SecureVariable(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.SecureVariable",
):
    '''A secure string variable.

    Only supported by Command documents (only supported in downloadContent plugin).
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="ofParameter") # type: ignore[misc]
    @builtins.classmethod
    def of_parameter(cls, reference: builtins.str) -> "NonSecureVariable":
        '''
        :param reference: -
        '''
        return typing.cast("NonSecureVariable", jsii.sinvoke(cls, "ofParameter", [reference]))

    @jsii.member(jsii_name="ofSecureToken") # type: ignore[misc]
    @builtins.classmethod
    def of_secure_token(cls, secure_token: builtins.str) -> "SsmSecureVariable":
        '''
        :param secure_token: -
        '''
        return typing.cast("SsmSecureVariable", jsii.sinvoke(cls, "ofSecureToken", [secure_token]))

    @jsii.member(jsii_name="ofValue") # type: ignore[misc]
    @builtins.classmethod
    def of_value(cls, value: builtins.str) -> "HardCodedSecureVariable":
        '''
        :param value: -
        '''
        return typing.cast("HardCodedSecureVariable", jsii.sinvoke(cls, "ofValue", [value]))

    @jsii.member(jsii_name="print") # type: ignore[misc]
    @abc.abstractmethod
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{Variable}} or the value.
        '''
        ...

    @jsii.member(jsii_name="requiredInputs") # type: ignore[misc]
    @abc.abstractmethod
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        ...

    @jsii.member(jsii_name="resolve") # type: ignore[misc]
    @abc.abstractmethod
    def resolve(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        ...

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> typing.Any:
        '''JSON.stringify(variable) will implicitly invoke this variable.'''
        return typing.cast(typing.Any, jsii.invoke(self, "toJSON", []))


class _SecureVariableProxy(SecureVariable):
    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{Variable}} or the value.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, SecureVariable).__jsii_proxy_class__ = lambda : _SecureVariableProxy


class Simulation(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.Simulation",
):
    @jsii.member(jsii_name="ofAutomation") # type: ignore[misc]
    @builtins.classmethod
    def of_automation(
        cls,
        document: "AutomationDocument",
        *,
        approve_hook: typing.Optional[IApproveHook] = None,
        aws_invoker: typing.Optional[IAwsInvoker] = None,
        execute_automation_hook: typing.Optional[IExecuteAutomationHook] = None,
        input_observer: typing.Optional[IObserver] = None,
        output_observer: typing.Optional[IObserver] = None,
        parameter_resolver: typing.Optional[IParameterResolver] = None,
        pause_hook: typing.Optional[IPauseHook] = None,
        run_command_hook: typing.Optional[IRunCommandHook] = None,
        sleep_hook: typing.Optional[ISleepHook] = None,
        webhook: typing.Optional[IWebhook] = None,
    ) -> "Simulation":
        '''
        :param document: -
        :param approve_hook: (Optional) Approve hook to be called to pause the execution. To mock this implementation either inject an instance of IApproveHook or use the provided MockApprove class. Default: ApproveHook instance. ApproveHook may not work in exported JSII languages. Override interface as needed.
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param execute_automation_hook: Hook for simulating aws:executeAutomation. Default: - Uses AWS API to execute the document remotely.
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param parameter_resolver: (Optional) Resolver for secure strings in parameters. Required to simulate if using tokens in parameters input. Default: - Treats parameters as literal
        :param pause_hook: (Optional) Pause hook to be called to pause the execution. To mock this implemenation either inject an instance of IPauseHook or use the provided MockPause class. Default: PauseHook instance. PauseHook may not work in exported JSII languages. Override interface as needed.
        :param run_command_hook: Hook for simulating aws:runCommand. Default: - Uses AWS API to execute the document remotely.
        :param sleep_hook: (Optional) Hook to inject alternate ISleeper (to mock the sleep between failed invocations). Default: - really perform sleep using SleeperImpl class.
        :param webhook: (Optional) Hook for simulating aws:invokeWebhook. Default: - Returns 204 with an empty response
        '''
        props = AutomationSimulationProps(
            approve_hook=approve_hook,
            aws_invoker=aws_invoker,
            execute_automation_hook=execute_automation_hook,
            input_observer=input_observer,
            output_observer=output_observer,
            parameter_resolver=parameter_resolver,
            pause_hook=pause_hook,
            run_command_hook=run_command_hook,
            sleep_hook=sleep_hook,
            webhook=webhook,
        )

        return typing.cast("Simulation", jsii.sinvoke(cls, "ofAutomation", [document, props]))

    @jsii.member(jsii_name="ofCommand") # type: ignore[misc]
    @builtins.classmethod
    def of_command(
        cls,
        document: "CommandDocument",
        *,
        simulation_platform: Platform,
        environment: typing.Optional[IEnvironment] = None,
    ) -> "Simulation":
        '''
        :param document: -
        :param simulation_platform: The Platform used in executing the command step.
        :param environment: (Optional) Specify here the environment in which to execute the scripts. Use the DockerEnvironment to execute the commands inside the docker. You can alternatively use the LoggingEnvironment which simply logs the commands or MockEnvironment which saves them for validation. Default: LoggingEnvironment
        '''
        props = CommandSimulationProps(
            simulation_platform=simulation_platform, environment=environment
        )

        return typing.cast("Simulation", jsii.sinvoke(cls, "ofCommand", [document, props]))

    @jsii.member(jsii_name="simulate")
    def simulate(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> "DocumentResult":
        '''Synthesize before calling this function! You can use this to Synthesize: SynthUtils.synthesize(stack);

        Executes the SSM Document in simulation mode.
        This method DOES NOT result in invocation of SSM APIs.
        Rather, all steps are executed locally and mimic the behavior of SSM.
        If any inputs are not provided in this function, the specified defaults for the inputs will be used.

        :param inputs: the inputs to feed into the simulated execution.

        :return: the outputs of all the executed steps.
        '''
        return typing.cast("DocumentResult", jsii.invoke(self, "simulate", [inputs]))

    @jsii.member(jsii_name="start")
    def _start(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> "SimulationResult":
        '''Delegates the execution of the Document to the subclass (Automation, etc).

        :param inputs: a merge of the defined inputs to the document and the default values if not supplied.

        :return: the outputs that were emitted from all of the steps.
        '''
        return typing.cast("SimulationResult", jsii.invoke(self, "start", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="document")
    def document(self) -> "SsmDocument":
        return typing.cast("SsmDocument", jsii.get(self, "document"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="props")
    def props(self) -> "SimulationProps":
        return typing.cast("SimulationProps", jsii.get(self, "props"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.SimulationProps",
    jsii_struct_bases=[],
    name_mapping={
        "approve_hook": "approveHook",
        "aws_invoker": "awsInvoker",
        "environment": "environment",
        "input_observer": "inputObserver",
        "output_observer": "outputObserver",
        "parameter_resolver": "parameterResolver",
        "pause_hook": "pauseHook",
        "run_command_hook": "runCommandHook",
        "simulation_platform": "simulationPlatform",
        "sleep_hook": "sleepHook",
        "webhook": "webhook",
    },
)
class SimulationProps:
    def __init__(
        self,
        *,
        approve_hook: typing.Optional[IApproveHook] = None,
        aws_invoker: typing.Optional[IAwsInvoker] = None,
        environment: typing.Optional[IEnvironment] = None,
        input_observer: typing.Optional[IObserver] = None,
        output_observer: typing.Optional[IObserver] = None,
        parameter_resolver: typing.Optional[IParameterResolver] = None,
        pause_hook: typing.Optional[IPauseHook] = None,
        run_command_hook: typing.Optional[IRunCommandHook] = None,
        simulation_platform: typing.Optional[Platform] = None,
        sleep_hook: typing.Optional[ISleepHook] = None,
        webhook: typing.Optional[IWebhook] = None,
    ) -> None:
        '''Universe of Automation and Command simulation props.

        :param approve_hook: 
        :param aws_invoker: 
        :param environment: 
        :param input_observer: 
        :param output_observer: 
        :param parameter_resolver: 
        :param pause_hook: 
        :param run_command_hook: 
        :param simulation_platform: 
        :param sleep_hook: 
        :param webhook: 
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if approve_hook is not None:
            self._values["approve_hook"] = approve_hook
        if aws_invoker is not None:
            self._values["aws_invoker"] = aws_invoker
        if environment is not None:
            self._values["environment"] = environment
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if parameter_resolver is not None:
            self._values["parameter_resolver"] = parameter_resolver
        if pause_hook is not None:
            self._values["pause_hook"] = pause_hook
        if run_command_hook is not None:
            self._values["run_command_hook"] = run_command_hook
        if simulation_platform is not None:
            self._values["simulation_platform"] = simulation_platform
        if sleep_hook is not None:
            self._values["sleep_hook"] = sleep_hook
        if webhook is not None:
            self._values["webhook"] = webhook

    @builtins.property
    def approve_hook(self) -> typing.Optional[IApproveHook]:
        result = self._values.get("approve_hook")
        return typing.cast(typing.Optional[IApproveHook], result)

    @builtins.property
    def aws_invoker(self) -> typing.Optional[IAwsInvoker]:
        result = self._values.get("aws_invoker")
        return typing.cast(typing.Optional[IAwsInvoker], result)

    @builtins.property
    def environment(self) -> typing.Optional[IEnvironment]:
        result = self._values.get("environment")
        return typing.cast(typing.Optional[IEnvironment], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def parameter_resolver(self) -> typing.Optional[IParameterResolver]:
        result = self._values.get("parameter_resolver")
        return typing.cast(typing.Optional[IParameterResolver], result)

    @builtins.property
    def pause_hook(self) -> typing.Optional[IPauseHook]:
        result = self._values.get("pause_hook")
        return typing.cast(typing.Optional[IPauseHook], result)

    @builtins.property
    def run_command_hook(self) -> typing.Optional[IRunCommandHook]:
        result = self._values.get("run_command_hook")
        return typing.cast(typing.Optional[IRunCommandHook], result)

    @builtins.property
    def simulation_platform(self) -> typing.Optional[Platform]:
        result = self._values.get("simulation_platform")
        return typing.cast(typing.Optional[Platform], result)

    @builtins.property
    def sleep_hook(self) -> typing.Optional[ISleepHook]:
        result = self._values.get("sleep_hook")
        return typing.cast(typing.Optional[ISleepHook], result)

    @builtins.property
    def webhook(self) -> typing.Optional[IWebhook]:
        result = self._values.get("webhook")
        return typing.cast(typing.Optional[IWebhook], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.SimulationResult",
    jsii_struct_bases=[],
    name_mapping={
        "executed_steps": "executedSteps",
        "response_code": "responseCode",
        "outputs": "outputs",
        "stack_trace": "stackTrace",
    },
)
class SimulationResult:
    def __init__(
        self,
        *,
        executed_steps: typing.Sequence[builtins.str],
        response_code: ResponseCode,
        outputs: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        stack_trace: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Response object returned from steps.

        A successful response will contain the outputs expected.
        A failed/canceled response will contain the stackTrace.
        A failure will propagate up the stack unless the step is marked to succeed on failure.

        :param executed_steps: All the steps that were executed in this Simulation.
        :param response_code: 
        :param outputs: May be empty if responseCode is FAILED/CANCELLED. There are no outputs provided for Command steps or documents.
        :param stack_trace: undefined if responseCode is SUCCESS.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "executed_steps": executed_steps,
            "response_code": response_code,
        }
        if outputs is not None:
            self._values["outputs"] = outputs
        if stack_trace is not None:
            self._values["stack_trace"] = stack_trace

    @builtins.property
    def executed_steps(self) -> typing.List[builtins.str]:
        '''All the steps that were executed in this Simulation.'''
        result = self._values.get("executed_steps")
        assert result is not None, "Required property 'executed_steps' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def response_code(self) -> ResponseCode:
        result = self._values.get("response_code")
        assert result is not None, "Required property 'response_code' is missing"
        return typing.cast(ResponseCode, result)

    @builtins.property
    def outputs(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''May be empty if responseCode is FAILED/CANCELLED.

        There are no outputs provided for Command steps or documents.
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def stack_trace(self) -> typing.Optional[builtins.str]:
        '''undefined if responseCode is SUCCESS.'''
        result = self._values.get("stack_trace")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SimulationResult(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(ISleepHook)
class SleepImpl(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.SleepImpl",
):
    '''Performs a real sleep.'''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="sleep")
    def sleep(self, time_millis: jsii.Number) -> None:
        '''Synchronously sleeps for duration specified in millis.

        :param time_millis: -
        '''
        return typing.cast(None, jsii.invoke(self, "sleep", [time_millis]))


class SleepSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.SleepSimulation",
):
    '''AutomationStep implemenation for aws:sleep https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-sleep.html.'''

    def __init__(self, step: "SleepStep", *, sleep_hook: ISleepHook) -> None:
        '''
        :param step: -
        :param sleep_hook: (Optional) Whether to really perform a pause of the runtime. To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class. Default: SleeperImpl
        '''
        props = SleepSimulationProps(sleep_hook=sleep_hook)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        _inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param _inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [_inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.SleepSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"sleep_hook": "sleepHook"},
)
class SleepSimulationProps:
    def __init__(self, *, sleep_hook: ISleepHook) -> None:
        '''Properties for sleep step.

        :param sleep_hook: (Optional) Whether to really perform a pause of the runtime. To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class. Default: SleeperImpl
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "sleep_hook": sleep_hook,
        }

    @builtins.property
    def sleep_hook(self) -> ISleepHook:
        '''(Optional) Whether to really perform a pause of the runtime.

        To override sleep behavior, inject an ISleepHook impl or use the provided MockSleep class.

        :default: SleeperImpl
        '''
        result = self._values.get("sleep_hook")
        assert result is not None, "Required property 'sleep_hook' is missing"
        return typing.cast(ISleepHook, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SleepSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class SsmDocument(
    constructs.Construct,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.SsmDocument",
):
    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        assume_role: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
        doc_outputs: typing.Optional[typing.Sequence[DocumentOutput]] = None,
        document_format: typing.Optional[DocumentFormat] = None,
        document_name: typing.Optional[builtins.str] = None,
        header: typing.Optional[builtins.str] = None,
        requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.CfnTag]] = None,
        target_type: typing.Optional[builtins.str] = None,
        version_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param assume_role: (Optional) Assume role to use for this document. If provided, this value MUST be included as one of the documentInput names.
        :param description: (Optional) Description of the document. Defaults to the document name.
        :param doc_inputs: (Optional) Inputs required by the document.
        :param doc_outputs: (Optional) Outputs to be emitted from the document. The outputs are placed in a StringSet called outputs (as is done in SSM). Default: []
        :param document_format: (Optional) Specifies whether this document should be written as YAML or JSON. Default: JSON
        :param document_name: (Optional) Name of the document. Will default to the id provided for the CDK node.
        :param header: (Optional) A Header/comment to include at the start of a YAML document. JSON documents do not support headers.
        :param requires: ``AWS::SSM::Document.Requires``.
        :param tags: ``AWS::SSM::Document.Tags``.
        :param target_type: ``AWS::SSM::Document.TargetType``.
        :param version_name: ``AWS::SSM::Document.VersionName``.
        '''
        props = SsmDocumentProps(
            assume_role=assume_role,
            description=description,
            doc_inputs=doc_inputs,
            doc_outputs=doc_outputs,
            document_format=document_format,
            document_name=document_name,
            header=header,
            requires=requires,
            tags=tags,
            target_type=target_type,
            version_name=version_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="buildSsmDocument") # type: ignore[misc]
    @abc.abstractmethod
    def _build_ssm_document(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Delegates building the SSM Document to be converted to a yaml/json to the subclass (Automation etc).'''
        ...

    @jsii.member(jsii_name="documentType") # type: ignore[misc]
    @abc.abstractmethod
    def document_type(self) -> builtins.str:
        ...

    @jsii.member(jsii_name="formatInputs")
    def _format_inputs(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''The SSM Document inputs optionally specify a number of parameters including allowedValues, minItems etc.

        This function builds an object containing the relevant (declared) input parameters.
        The return object will be used to build the yaml/json representation of the document.
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatInputs", []))

    @jsii.member(jsii_name="print")
    def print(self) -> builtins.str:
        '''Synthesize before calling this function! You can use this to Synthesize: cdk.SynthUtils.synthesize(stack);

        Converts the objects define in the SSM Document (including all of the steps) to an SSM document string.
        The format is dependency on the documentFormat property provided to the class.
        The yaml can be used as is and will behave (or at least should behave) as was simulated in the runSimulation().

        :return: a string representation of this document as an SSM formatted yaml/json.
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "print", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="description")
    def description(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "description"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="docInputs")
    def doc_inputs(self) -> typing.List[Input]:
        return typing.cast(typing.List[Input], jsii.get(self, "docInputs"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="docOutputs")
    def doc_outputs(self) -> typing.List[DocumentOutput]:
        return typing.cast(typing.List[DocumentOutput], jsii.get(self, "docOutputs"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="documentName")
    def document_name(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "documentName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="props")
    def props(self) -> "SsmDocumentProps":
        return typing.cast("SsmDocumentProps", jsii.get(self, "props"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="assumeRole")
    def assume_role(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "assumeRole"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="header")
    def header(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "header"))


class _SsmDocumentProxy(SsmDocument):
    @jsii.member(jsii_name="buildSsmDocument")
    def _build_ssm_document(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Delegates building the SSM Document to be converted to a yaml/json to the subclass (Automation etc).'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "buildSsmDocument", []))

    @jsii.member(jsii_name="documentType")
    def document_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.invoke(self, "documentType", []))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, SsmDocument).__jsii_proxy_class__ = lambda : _SsmDocumentProxy


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.SsmDocumentContentProps",
    jsii_struct_bases=[],
    name_mapping={"document_source": "documentSource"},
)
class SsmDocumentContentProps:
    def __init__(self, *, document_source: DocumentSource) -> None:
        '''Properties.json for sourceType SsmDocument.

        :param document_source: specify one of the following The name and version of the document in the following format: name:version. Version is optional. or The ARN for the document in the following format: arn:aws:ssm:region:account_id:document/document_name
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "document_source": document_source,
        }

    @builtins.property
    def document_source(self) -> DocumentSource:
        '''specify one of the following The name and version of the document in the following format: name:version.

        Version is optional.
        or The ARN for the document in the following format: arn:aws:ssm:region:account_id:document/document_name
        '''
        result = self._values.get("document_source")
        assert result is not None, "Required property 'document_source' is missing"
        return typing.cast(DocumentSource, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SsmDocumentContentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.SsmDocumentProps",
    jsii_struct_bases=[],
    name_mapping={
        "assume_role": "assumeRole",
        "description": "description",
        "doc_inputs": "docInputs",
        "doc_outputs": "docOutputs",
        "document_format": "documentFormat",
        "document_name": "documentName",
        "header": "header",
        "requires": "requires",
        "tags": "tags",
        "target_type": "targetType",
        "version_name": "versionName",
    },
)
class SsmDocumentProps:
    def __init__(
        self,
        *,
        assume_role: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
        doc_outputs: typing.Optional[typing.Sequence[DocumentOutput]] = None,
        document_format: typing.Optional[DocumentFormat] = None,
        document_name: typing.Optional[builtins.str] = None,
        header: typing.Optional[builtins.str] = None,
        requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.CfnTag]] = None,
        target_type: typing.Optional[builtins.str] = None,
        version_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param assume_role: (Optional) Assume role to use for this document. If provided, this value MUST be included as one of the documentInput names.
        :param description: (Optional) Description of the document. Defaults to the document name.
        :param doc_inputs: (Optional) Inputs required by the document.
        :param doc_outputs: (Optional) Outputs to be emitted from the document. The outputs are placed in a StringSet called outputs (as is done in SSM). Default: []
        :param document_format: (Optional) Specifies whether this document should be written as YAML or JSON. Default: JSON
        :param document_name: (Optional) Name of the document. Will default to the id provided for the CDK node.
        :param header: (Optional) A Header/comment to include at the start of a YAML document. JSON documents do not support headers.
        :param requires: ``AWS::SSM::Document.Requires``.
        :param tags: ``AWS::SSM::Document.Tags``.
        :param target_type: ``AWS::SSM::Document.TargetType``.
        :param version_name: ``AWS::SSM::Document.VersionName``.
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if assume_role is not None:
            self._values["assume_role"] = assume_role
        if description is not None:
            self._values["description"] = description
        if doc_inputs is not None:
            self._values["doc_inputs"] = doc_inputs
        if doc_outputs is not None:
            self._values["doc_outputs"] = doc_outputs
        if document_format is not None:
            self._values["document_format"] = document_format
        if document_name is not None:
            self._values["document_name"] = document_name
        if header is not None:
            self._values["header"] = header
        if requires is not None:
            self._values["requires"] = requires
        if tags is not None:
            self._values["tags"] = tags
        if target_type is not None:
            self._values["target_type"] = target_type
        if version_name is not None:
            self._values["version_name"] = version_name

    @builtins.property
    def assume_role(self) -> typing.Optional[builtins.str]:
        '''(Optional) Assume role to use for this document.

        If provided, this value MUST be included as one of the documentInput names.
        '''
        result = self._values.get("assume_role")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) Description of the document.

        Defaults to the document name.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def doc_inputs(self) -> typing.Optional[typing.List[Input]]:
        '''(Optional) Inputs required by the document.'''
        result = self._values.get("doc_inputs")
        return typing.cast(typing.Optional[typing.List[Input]], result)

    @builtins.property
    def doc_outputs(self) -> typing.Optional[typing.List[DocumentOutput]]:
        '''(Optional) Outputs to be emitted from the document.

        The outputs are placed in a StringSet called outputs (as is done in SSM).

        :default: []
        '''
        result = self._values.get("doc_outputs")
        return typing.cast(typing.Optional[typing.List[DocumentOutput]], result)

    @builtins.property
    def document_format(self) -> typing.Optional[DocumentFormat]:
        '''(Optional) Specifies whether this document should be written as YAML or JSON.

        :default: JSON
        '''
        result = self._values.get("document_format")
        return typing.cast(typing.Optional[DocumentFormat], result)

    @builtins.property
    def document_name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the document.

        Will default to the id provided for the CDK node.
        '''
        result = self._values.get("document_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def header(self) -> typing.Optional[builtins.str]:
        '''(Optional) A Header/comment to include at the start of a YAML document.

        JSON documents do not support headers.
        '''
        result = self._values.get("header")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def requires(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.IResolvable, typing.List[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]]:
        '''``AWS::SSM::Document.Requires``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-requires
        '''
        result = self._values.get("requires")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.IResolvable, typing.List[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.CfnTag]]:
        '''``AWS::SSM::Document.Tags``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.CfnTag]], result)

    @builtins.property
    def target_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::SSM::Document.TargetType``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-targettype
        '''
        result = self._values.get("target_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def version_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::SSM::Document.VersionName``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-versionname
        '''
        result = self._values.get("version_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SsmDocumentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IRunDocumentLocation)
class SsmRunDocument(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.SsmRunDocument",
):
    def __init__(self, document_name: IStringVariable) -> None:
        '''specify the name of the document.

        :param document_name: -
        '''
        jsii.create(self.__class__, self, [document_name])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="location")
    def location(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "location"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="type")
    def type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "type"))


class SsmSecureVariable(
    SecureVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.SsmSecureVariable",
):
    def __init__(self, secure_token: builtins.str) -> None:
        '''
        :param secure_token: -
        '''
        jsii.create(self.__class__, self, [secure_token])

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{Variable}} or the value.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, _inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param _inputs: -
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [_inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="secureToken")
    def secure_token(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "secureToken"))


class Step(
    constructs.Construct,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.Step",
):
    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = StepProps(
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs") # type: ignore[misc]
    @abc.abstractmethod
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        ...

    @jsii.member(jsii_name="listOutputs") # type: ignore[misc]
    @abc.abstractmethod
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        ...

    @jsii.member(jsii_name="toSsmEntry") # type: ignore[misc]
    @abc.abstractmethod
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    @abc.abstractmethod
    def action(self) -> builtins.str:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="inputObserver")
    def input_observer(self) -> IObserver:
        return typing.cast(IObserver, jsii.get(self, "inputObserver"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="name")
    def name(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "name"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="outputObserver")
    def output_observer(self) -> IObserver:
        return typing.cast(IObserver, jsii.get(self, "outputObserver"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="description")
    def description(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "description"))


class _StepProxy(Step):
    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, Step).__jsii_proxy_class__ = lambda : _StepProxy


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.StepProps",
    jsii_struct_bases=[],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
    },
)
class StepProps:
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "StepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class StringDocument(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.StringDocument",
):
    '''This AutomationDocument supports declaring your document from an existing document (JSON/YAML String/File).

    Importing an existing file allows for benefiting from the simulated execution.
    The simulated execution will run locally in the same fashion that SSM Execution would run the document.
    You can supply mocks to the simulator and validate the calls and the flow of the document without running via SSM execution.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="fromFile") # type: ignore[misc]
    @builtins.classmethod
    def from_file(
        cls,
        stack: constructs.Construct,
        id: builtins.str,
        document_file_path: builtins.str,
    ) -> "AutomationDocument":
        '''Create an AutomationDocument from an existing AutomationDocument yaml or json file.

        Note: This function will deduce whether the file is written in yaml or json based on whether it has a .yaml or .yml extention.
        You can use the returned AutomationDocument to run simulations as you would other documents created using this library.

        :param stack: -
        :param id: -
        :param document_file_path: -
        '''
        return typing.cast("AutomationDocument", jsii.sinvoke(cls, "fromFile", [stack, id, document_file_path]))

    @jsii.member(jsii_name="fromJson") # type: ignore[misc]
    @builtins.classmethod
    def from_json(
        cls,
        stack: constructs.Construct,
        id: builtins.str,
        document_json: builtins.str,
    ) -> "AutomationDocument":
        '''Create an AutomationDocument from an existing json string.

        You can use the returned AutomationDocument to run simulations as you would other documents created using this library.

        :param stack: -
        :param id: -
        :param document_json: -
        '''
        return typing.cast("AutomationDocument", jsii.sinvoke(cls, "fromJson", [stack, id, document_json]))

    @jsii.member(jsii_name="fromYaml") # type: ignore[misc]
    @builtins.classmethod
    def from_yaml(
        cls,
        stack: constructs.Construct,
        id: builtins.str,
        document_yaml: builtins.str,
    ) -> "AutomationDocument":
        '''Create an AutomationDocument from an existing yaml string.

        You can use the returned AutomationDocument to run simulations as you would other documents created using this library.

        :param stack: -
        :param id: -
        :param document_yaml: -
        '''
        return typing.cast("AutomationDocument", jsii.sinvoke(cls, "fromYaml", [stack, id, document_yaml]))


@jsii.implements(IStringVariable)
class StringFormat(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.StringFormat",
):
    '''Replacement for strings using Java String format style "%s" replacements.

    Example: new StringFormat("This %s a replacement: %s", [new HardCodedValue("is"), new ExportedVariable("myInput")]);
    The supported variable strings are either:

    1. Implicit indices: "%s" where the first occurrence will match the first variable, the next will match the second...
    2. Explicit indices: Example: "%1$s"; where "%1$s" matches the first variable and "%1$s" matches the second.
       Do not combine usage of implicit and explicit indices. Choose one per StringFormat instance.
    '''

    def __init__(
        self,
        format: builtins.str,
        variables: typing.Optional[typing.Sequence[IGenericVariable]] = None,
    ) -> None:
        '''
        :param format: -
        :param variables: -
        '''
        jsii.create(self.__class__, self, [format, variables])

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{Variable}} or the value.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [inputs]))

    @jsii.member(jsii_name="resolveToString")
    def resolve_to_string(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.str:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "resolveToString", [inputs]))

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> typing.Any:
        '''JSON.stringify(variable) will implicitly invoke this variable.'''
        return typing.cast(typing.Any, jsii.invoke(self, "toJSON", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="format")
    def format(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "format"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="variables")
    def variables(self) -> typing.List[IGenericVariable]:
        return typing.cast(typing.List[IGenericVariable], jsii.get(self, "variables"))


class SynthUtils(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.SynthUtils",
):
    '''Wraps SynthUtils from @aws-cdk/assert because that package is not exported via JSII.'''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="synthesize") # type: ignore[misc]
    @builtins.classmethod
    def synthesize(
        cls,
        stack: aws_cdk.Stack,
    ) -> typing.Optional[aws_cdk.cx_api.CloudAssembly]:
        '''Wraps @aws-cdk/assert SynthUtils.synthesize(stack). Synthesizes the stack provided.

        :param stack: -
        '''
        return typing.cast(typing.Optional[aws_cdk.cx_api.CloudAssembly], jsii.sinvoke(cls, "synthesize", [stack]))


class UsernamePasswordAuthMethod(
    AuthMethod,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.UsernamePasswordAuthMethod",
):
    def __init__(
        self,
        auth_method: builtins.str,
        user_name: SecureVariable,
        password: SecureVariable,
    ) -> None:
        '''
        :param auth_method: -
        :param user_name: -
        :param password: -
        '''
        jsii.create(self.__class__, self, [auth_method, user_name, password])

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="toEntry")
    def to_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authMethod")
    def auth_method(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "authMethod"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="password")
    def password(self) -> SecureVariable:
        return typing.cast(SecureVariable, jsii.get(self, "password"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="userName")
    def user_name(self) -> SecureVariable:
        return typing.cast(SecureVariable, jsii.get(self, "userName"))


class WaitForResourceSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.WaitForResourceSimulation",
):
    '''AutomationStep impl for aws:waitForAwsResourceProperty https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-waitForAwsResourceProperty.html.'''

    def __init__(
        self,
        step: "WaitForResourceStep",
        *,
        aws_invoker: IAwsInvoker,
        sleep_hook: ISleepHook,
    ) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param sleep_hook: (Optional) Hook to inject alternate ISleeper (to mock the sleep between failed invocations). Default: - really perform sleep using SleeperImpl class.
        '''
        props = WaitForResourceSimulationProps(
            aws_invoker=aws_invoker, sleep_hook=sleep_hook
        )

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''As is the case in an SSM Automation execution, this will continue to sleep/execute until desired value is found.

        This function will throw if the timeoutSeconds is exceeded and the desired value is still not received from AWS.

        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.WaitForResourceSimulationProps",
    jsii_struct_bases=[],
    name_mapping={"aws_invoker": "awsInvoker", "sleep_hook": "sleepHook"},
)
class WaitForResourceSimulationProps:
    def __init__(self, *, aws_invoker: IAwsInvoker, sleep_hook: ISleepHook) -> None:
        '''
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        :param sleep_hook: (Optional) Hook to inject alternate ISleeper (to mock the sleep between failed invocations). Default: - really perform sleep using SleeperImpl class.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "aws_invoker": aws_invoker,
            "sleep_hook": sleep_hook,
        }

    @builtins.property
    def aws_invoker(self) -> IAwsInvoker:
        '''(Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call).

        :default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        result = self._values.get("aws_invoker")
        assert result is not None, "Required property 'aws_invoker' is missing"
        return typing.cast(IAwsInvoker, result)

    @builtins.property
    def sleep_hook(self) -> ISleepHook:
        '''(Optional) Hook to inject alternate ISleeper (to mock the sleep between failed invocations).

        :default: - really perform sleep using SleeperImpl class.
        '''
        result = self._values.get("sleep_hook")
        assert result is not None, "Required property 'sleep_hook' is missing"
        return typing.cast(ISleepHook, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "WaitForResourceSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IWebhook)
class WebhookImpl(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.WebhookImpl",
):
    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="invoke")
    def invoke(
        self,
        *,
        integration_name: builtins.str,
        body: typing.Optional[builtins.str] = None,
    ) -> InvokeWebhookResult:
        '''Invoke the web hook.

        :param integration_name: The name of the Automation integration. For example, exampleIntegration. The integration you specify must already exist.
        :param body: (Optional) The payload you want to send when your webhook integration is invoked.
        '''
        _props = InvokeWebhookProps(integration_name=integration_name, body=body)

        return typing.cast(InvokeWebhookResult, jsii.invoke(self, "invoke", [_props]))


@jsii.implements(IExecuteAutomationHook)
class ApiExecuteAutomationHook(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ApiExecuteAutomationHook",
):
    '''ExecuteAutomation implementation using AWS API.'''

    def __init__(self, aws_invoker: IAwsInvoker, sleep_hook: ISleepHook) -> None:
        '''
        :param aws_invoker: -
        :param sleep_hook: -
        '''
        jsii.create(self.__class__, self, [aws_invoker, sleep_hook])

    @jsii.member(jsii_name="execute")
    def execute(
        self,
        *,
        document_name: builtins.str,
        document_version: typing.Optional[builtins.str] = None,
        max_concurrency: typing.Optional[builtins.str] = None,
        max_errors: typing.Optional[builtins.str] = None,
        runtime_parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        tags: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_locations: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_maps: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
        target_parameter_name: typing.Optional[builtins.str] = None,
        targets: typing.Optional[typing.Sequence[typing.Mapping[builtins.str, typing.Any]]] = None,
    ) -> ExecuteAutomationOutputs:
        '''Simulate the aws:executeAutomation.

        :param document_name: 
        :param document_version: 
        :param max_concurrency: 
        :param max_errors: 
        :param runtime_parameters: 
        :param tags: 
        :param target_locations: 
        :param target_maps: 
        :param target_parameter_name: 
        :param targets: 
        '''
        props = ExecuteAutomationProps(
            document_name=document_name,
            document_version=document_version,
            max_concurrency=max_concurrency,
            max_errors=max_errors,
            runtime_parameters=runtime_parameters,
            tags=tags,
            target_locations=target_locations,
            target_maps=target_maps,
            target_parameter_name=target_parameter_name,
            targets=targets,
        )

        return typing.cast(ExecuteAutomationOutputs, jsii.invoke(self, "execute", [props]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="props")
    def props(self) -> ApiExecuteAutomationProps:
        return typing.cast(ApiExecuteAutomationProps, jsii.get(self, "props"))


@jsii.implements(IRunCommandHook)
class ApiRunCommandHook(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ApiRunCommandHook",
):
    '''RunCommand implementation using AWS API.'''

    def __init__(self, aws_invoker: IAwsInvoker, sleep_hook: ISleepHook) -> None:
        '''
        :param aws_invoker: -
        :param sleep_hook: -
        '''
        jsii.create(self.__class__, self, [aws_invoker, sleep_hook])

    @jsii.member(jsii_name="execute")
    def execute(
        self,
        *,
        document_name: builtins.str,
        targets: typing.Sequence[builtins.str],
        cloud_watch_output_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        comment: typing.Optional[builtins.str] = None,
        document_hash: typing.Optional[builtins.str] = None,
        document_hash_type: typing.Optional[builtins.str] = None,
        max_concurrency: typing.Optional[jsii.Number] = None,
        max_errors: typing.Optional[jsii.Number] = None,
        notification_config: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        output_s3_bucket_name: typing.Optional[builtins.str] = None,
        output_s3_key_prefix: typing.Optional[builtins.str] = None,
        parameters: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        service_role_arn: typing.Optional[builtins.str] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
    ) -> RunCommandOutputs:
        '''Simulate the aws:runCommand.

        :param document_name: If the Command type document is owned by you or AWS, specify the name of the document. If you're using a document shared with you by a different AWS account, specify the Amazon Resource Name (ARN) of the document.
        :param targets: The instance IDs where you want the command to run. You can specify a maximum of 50 IDs. You can also use the pseudo parameter {{RESOURCE_ID}} in place of instance IDs to run the command on all instances in the target group. For more information about pseudo parameters, see `About pseudo parameters <https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-register-tasks-parameters.html>`_. Another alternative is to send commands to a fleet of instances by using the Targets parameter. The Targets parameter accepts Amazon Elastic Compute Cloud (Amazon EC2) tags. For more information about how to use the Targets parameter, see `Using targets and rate controls to send commands to a fleet <https://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html>`_.
        :param cloud_watch_output_config: (Optional) Configuration options for sending command output to Amazon CloudWatch Logs. For more information about sending command output to CloudWatch Logs, see `Configuring Amazon CloudWatch Logs for Run Command <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-rc-setting-up-cwlogs.html>`_.
        :param comment: (Optional) User-defined information about the command.
        :param document_hash: (Optional) The hash for the document.
        :param document_hash_type: (Optional) The type of the hash.
        :param max_concurrency: (Optional) The maximum concurrency.
        :param max_errors: (Optional) The maximum errors.
        :param notification_config: (Optional) The configurations for sending notifications.
        :param output_s3_bucket_name: (Optional) The name of the S3 bucket for command output responses.
        :param output_s3_key_prefix: (Optional) The prefix.
        :param parameters: (Optional) The required and optional parameters specified in the document.
        :param service_role_arn: (Optional) The ARN of the AWS Identity and Access Management (IAM) role.
        :param timeout_seconds: (Optional) The amount of time in seconds to wait for a command to deliver to the AWS Systems Manager SSM Agent on an instance. If the command isn't received by the SSM Agent on the instance before the value specified is reached, then the status of the command changes to Delivery Timed Out.
        '''
        props = RunCommandProps(
            document_name=document_name,
            targets=targets,
            cloud_watch_output_config=cloud_watch_output_config,
            comment=comment,
            document_hash=document_hash,
            document_hash_type=document_hash_type,
            max_concurrency=max_concurrency,
            max_errors=max_errors,
            notification_config=notification_config,
            output_s3_bucket_name=output_s3_bucket_name,
            output_s3_key_prefix=output_s3_key_prefix,
            parameters=parameters,
            service_role_arn=service_role_arn,
            timeout_seconds=timeout_seconds,
        )

        return typing.cast(RunCommandOutputs, jsii.invoke(self, "execute", [props]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="awsInvoker")
    def aws_invoker(self) -> IAwsInvoker:
        return typing.cast(IAwsInvoker, jsii.get(self, "awsInvoker"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="props")
    def props(self) -> ApiRunCommandProps:
        return typing.cast(ApiRunCommandProps, jsii.get(self, "props"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sleepHook")
    def sleep_hook(self) -> ISleepHook:
        return typing.cast(ISleepHook, jsii.get(self, "sleepHook"))


@jsii.implements(IApproveHook)
class ApproveImpl(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ApproveImpl",
):
    '''This IApproveHook implementation provides a real ask and waits for user input of Enter.

    This implementation does not work well on all exported JSII languages.
    Users can provide their own impl using the IAskHook interface.
    '''

    def __init__(self) -> None:
        jsii.create(self.__class__, self, [])

    @jsii.member(jsii_name="ask")
    def ask(self, approver: builtins.str) -> builtins.bool:
        '''Ask for approval.

        :param approver: -
        '''
        return typing.cast(builtins.bool, jsii.invoke(self, "ask", [approver]))


class ApproveSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ApproveSimulation",
):
    '''AutomationStep implementation for aws:approve https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-approve.html.'''

    def __init__(self, step: "ApproveStep", *, approve_hook: IApproveHook) -> None:
        '''
        :param step: -
        :param approve_hook: (Optional) Approve hook to be called to pause the execution. To mock this implementation either inject an instance of IApproveHook or use the provided MockApprove class. Default: ApproveHook instance. ApproveHook may not work in exported JSII languages. Override interface as needed.
        '''
        props = ApproveSimulationProps(approve_hook=approve_hook)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''May perform a real approval ask based on the params used during instance creation.

        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @jsii.member(jsii_name="provideDate")
    def _provide_date(self) -> datetime.datetime:
        '''Override to mock the date the reviewer approved.'''
        return typing.cast(datetime.datetime, jsii.invoke(self, "provideDate", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="approveHook")
    def approve_hook(self) -> IApproveHook:
        return typing.cast(IApproveHook, jsii.get(self, "approveHook"))


class ArnDoc(
    DocumentSource,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ArnDoc",
):
    def __init__(self, arn: IStringVariable) -> None:
        '''
        :param arn: -
        '''
        jsii.create(self.__class__, self, [arn])

    @jsii.member(jsii_name="formatRequest")
    def format_request(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatRequest", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="arn")
    def arn(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "arn"))


class AssertAwsResourceSimulation(
    AutomationSimulationBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.AssertAwsResourceSimulation",
):
    '''AutomationStep implementation of aws:assertAwsResourceProperty.

    https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-assertAwsResourceProperty.html
    '''

    def __init__(
        self,
        step: "AssertAwsResourceStep",
        *,
        aws_invoker: IAwsInvoker,
    ) -> None:
        '''
        :param step: -
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        props = AwsInvocationSimulationProps(aws_invoker=aws_invoker)

        jsii.create(self.__class__, self, [step, props])

    @jsii.member(jsii_name="executeStep")
    def execute_step(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''If the value found matches one of the desiredValues, then this function returns.

        Otherwise it throws.

        :param inputs: -

        :return: empty object if value matches desiredValues.
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "executeStep", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


class AutomationDocument(
    SsmDocument,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationDocument",
):
    '''The AutomationDocuemnt used to both build the SSM Automation yaml/json and to use in simulation.

    The AutomationDocument will delegate execution responsibility to the AutomationSteps that it receives.
    The SsmDocument parent class contains methods to runSimulation() as well as print().
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        assume_role: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
        doc_outputs: typing.Optional[typing.Sequence[DocumentOutput]] = None,
        document_format: typing.Optional[DocumentFormat] = None,
        document_name: typing.Optional[builtins.str] = None,
        header: typing.Optional[builtins.str] = None,
        requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.CfnTag]] = None,
        target_type: typing.Optional[builtins.str] = None,
        version_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param assume_role: (Optional) Assume role to use for this document. If provided, this value MUST be included as one of the documentInput names.
        :param description: (Optional) Description of the document. Defaults to the document name.
        :param doc_inputs: (Optional) Inputs required by the document.
        :param doc_outputs: (Optional) Outputs to be emitted from the document. The outputs are placed in a StringSet called outputs (as is done in SSM). Default: []
        :param document_format: (Optional) Specifies whether this document should be written as YAML or JSON. Default: JSON
        :param document_name: (Optional) Name of the document. Will default to the id provided for the CDK node.
        :param header: (Optional) A Header/comment to include at the start of a YAML document. JSON documents do not support headers.
        :param requires: ``AWS::SSM::Document.Requires``.
        :param tags: ``AWS::SSM::Document.Tags``.
        :param target_type: ``AWS::SSM::Document.TargetType``.
        :param version_name: ``AWS::SSM::Document.VersionName``.
        '''
        props = AutomationDocumentProps(
            assume_role=assume_role,
            description=description,
            doc_inputs=doc_inputs,
            doc_outputs=doc_outputs,
            document_format=document_format,
            document_name=document_name,
            header=header,
            requires=requires,
            tags=tags,
            target_type=target_type,
            version_name=version_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="addStep")
    def add_step(self, component: IAutomationComponent) -> None:
        '''
        :param component: -
        '''
        return typing.cast(None, jsii.invoke(self, "addStep", [component]))

    @jsii.member(jsii_name="buildSsmDocument")
    def _build_ssm_document(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Delegates building the SSM Document to be converted to a yaml/json to the subclass (Automation etc).

        :return: an object that can be used to build the json/yaml string representation of this document.
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "buildSsmDocument", []))

    @jsii.member(jsii_name="collectedSteps")
    def collected_steps(self) -> typing.List["AutomationStep"]:
        return typing.cast(typing.List["AutomationStep"], jsii.invoke(self, "collectedSteps", []))

    @jsii.member(jsii_name="documentType")
    def document_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.invoke(self, "documentType", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="builder")
    def builder(self) -> AutomationDocumentBuilder:
        return typing.cast(AutomationDocumentBuilder, jsii.get(self, "builder"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationDocumentProps",
    jsii_struct_bases=[SsmDocumentProps],
    name_mapping={
        "assume_role": "assumeRole",
        "description": "description",
        "doc_inputs": "docInputs",
        "doc_outputs": "docOutputs",
        "document_format": "documentFormat",
        "document_name": "documentName",
        "header": "header",
        "requires": "requires",
        "tags": "tags",
        "target_type": "targetType",
        "version_name": "versionName",
    },
)
class AutomationDocumentProps(SsmDocumentProps):
    def __init__(
        self,
        *,
        assume_role: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
        doc_outputs: typing.Optional[typing.Sequence[DocumentOutput]] = None,
        document_format: typing.Optional[DocumentFormat] = None,
        document_name: typing.Optional[builtins.str] = None,
        header: typing.Optional[builtins.str] = None,
        requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.CfnTag]] = None,
        target_type: typing.Optional[builtins.str] = None,
        version_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''Options for AutomationDocument.

        :param assume_role: (Optional) Assume role to use for this document. If provided, this value MUST be included as one of the documentInput names.
        :param description: (Optional) Description of the document. Defaults to the document name.
        :param doc_inputs: (Optional) Inputs required by the document.
        :param doc_outputs: (Optional) Outputs to be emitted from the document. The outputs are placed in a StringSet called outputs (as is done in SSM). Default: []
        :param document_format: (Optional) Specifies whether this document should be written as YAML or JSON. Default: JSON
        :param document_name: (Optional) Name of the document. Will default to the id provided for the CDK node.
        :param header: (Optional) A Header/comment to include at the start of a YAML document. JSON documents do not support headers.
        :param requires: ``AWS::SSM::Document.Requires``.
        :param tags: ``AWS::SSM::Document.Tags``.
        :param target_type: ``AWS::SSM::Document.TargetType``.
        :param version_name: ``AWS::SSM::Document.VersionName``.
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if assume_role is not None:
            self._values["assume_role"] = assume_role
        if description is not None:
            self._values["description"] = description
        if doc_inputs is not None:
            self._values["doc_inputs"] = doc_inputs
        if doc_outputs is not None:
            self._values["doc_outputs"] = doc_outputs
        if document_format is not None:
            self._values["document_format"] = document_format
        if document_name is not None:
            self._values["document_name"] = document_name
        if header is not None:
            self._values["header"] = header
        if requires is not None:
            self._values["requires"] = requires
        if tags is not None:
            self._values["tags"] = tags
        if target_type is not None:
            self._values["target_type"] = target_type
        if version_name is not None:
            self._values["version_name"] = version_name

    @builtins.property
    def assume_role(self) -> typing.Optional[builtins.str]:
        '''(Optional) Assume role to use for this document.

        If provided, this value MUST be included as one of the documentInput names.
        '''
        result = self._values.get("assume_role")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) Description of the document.

        Defaults to the document name.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def doc_inputs(self) -> typing.Optional[typing.List[Input]]:
        '''(Optional) Inputs required by the document.'''
        result = self._values.get("doc_inputs")
        return typing.cast(typing.Optional[typing.List[Input]], result)

    @builtins.property
    def doc_outputs(self) -> typing.Optional[typing.List[DocumentOutput]]:
        '''(Optional) Outputs to be emitted from the document.

        The outputs are placed in a StringSet called outputs (as is done in SSM).

        :default: []
        '''
        result = self._values.get("doc_outputs")
        return typing.cast(typing.Optional[typing.List[DocumentOutput]], result)

    @builtins.property
    def document_format(self) -> typing.Optional[DocumentFormat]:
        '''(Optional) Specifies whether this document should be written as YAML or JSON.

        :default: JSON
        '''
        result = self._values.get("document_format")
        return typing.cast(typing.Optional[DocumentFormat], result)

    @builtins.property
    def document_name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the document.

        Will default to the id provided for the CDK node.
        '''
        result = self._values.get("document_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def header(self) -> typing.Optional[builtins.str]:
        '''(Optional) A Header/comment to include at the start of a YAML document.

        JSON documents do not support headers.
        '''
        result = self._values.get("header")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def requires(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.IResolvable, typing.List[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]]:
        '''``AWS::SSM::Document.Requires``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-requires
        '''
        result = self._values.get("requires")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.IResolvable, typing.List[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.CfnTag]]:
        '''``AWS::SSM::Document.Tags``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.CfnTag]], result)

    @builtins.property
    def target_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::SSM::Document.TargetType``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-targettype
        '''
        result = self._values.get("target_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def version_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::SSM::Document.VersionName``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-versionname
        '''
        result = self._values.get("version_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AutomationDocumentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IAutomationComponent)
class AutomationStep(
    Step,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationStep",
):
    '''Parent class for AutomationSteps.

    These steps are using in an AutomationDocument.
    You can instantiate steps using the AutomationBuilder for convenience.
    You can use these steps to simulate their execution (mimics the SSM run) AND to build their yaml/json declaration.
    Control flow of the subsequent step is determined by the currently executing step.
    The flow of the execution therefore follows a chain-of-responsibility pattern.
    The inputs received into a step AND the outputs of previous steps are merged to form inputs of subsequent steps.
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional["AutomationStep"] = None,
        on_failure: typing.Optional["AutomationStep"] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = AutomationStepProps(
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: AutomationDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        return typing.cast(None, jsii.invoke(self, "addToDocument", [doc]))

    @jsii.member(jsii_name="prepareSsmEntry")
    def _prepare_ssm_entry(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "prepareSsmEntry", [inputs]))

    @jsii.member(jsii_name="variables")
    def variables(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "variables", []))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="DEFAULT_MAX_ATTEMPTS")
    def DEFAULT_MAX_ATTEMPTS(cls) -> jsii.Number:
        return typing.cast(jsii.Number, jsii.sget(cls, "DEFAULT_MAX_ATTEMPTS"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="DEFAULT_TIMEOUT")
    def DEFAULT_TIMEOUT(cls) -> jsii.Number:
        return typing.cast(jsii.Number, jsii.sget(cls, "DEFAULT_TIMEOUT"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="isEnd")
    def is_end(self) -> builtins.bool:
        return typing.cast(builtins.bool, jsii.get(self, "isEnd"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="maxAttempts")
    def max_attempts(self) -> jsii.Number:
        return typing.cast(jsii.Number, jsii.get(self, "maxAttempts"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="timeoutSeconds")
    def timeout_seconds(self) -> jsii.Number:
        return typing.cast(jsii.Number, jsii.get(self, "timeoutSeconds"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="onCancel")
    def on_cancel(self) -> typing.Optional["AutomationStep"]:
        return typing.cast(typing.Optional["AutomationStep"], jsii.get(self, "onCancel"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="onFailure")
    def on_failure(self) -> typing.Optional["AutomationStep"]:
        return typing.cast(typing.Optional["AutomationStep"], jsii.get(self, "onFailure"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="allStepsInExecution")
    def all_steps_in_execution(self) -> typing.Optional[typing.List["AutomationStep"]]:
        return typing.cast(typing.Optional[typing.List["AutomationStep"]], jsii.get(self, "allStepsInExecution"))

    @all_steps_in_execution.setter
    def all_steps_in_execution(
        self,
        value: typing.Optional[typing.List["AutomationStep"]],
    ) -> None:
        jsii.set(self, "allStepsInExecution", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="nextStep")
    def next_step(self) -> typing.Optional["AutomationStep"]:
        return typing.cast(typing.Optional["AutomationStep"], jsii.get(self, "nextStep"))

    @next_step.setter
    def next_step(self, value: typing.Optional["AutomationStep"]) -> None:
        jsii.set(self, "nextStep", value)


class _AutomationStepProxy(
    AutomationStep, jsii.proxy_for(Step) # type: ignore[misc]
):
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, AutomationStep).__jsii_proxy_class__ = lambda : _AutomationStepProxy


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.AutomationStepProps",
    jsii_struct_bases=[StepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
    },
)
class AutomationStepProps(StepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AutomationStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AwsApiStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.AwsApiStep",
):
    '''AutomationStep implementation of aws:executeAwsApi.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        outputs: typing.Sequence[Output],
        api_params: typing.Mapping[builtins.str, typing.Any],
        pascal_case_api: builtins.str,
        service: builtins.str,
        java_script_api: typing.Optional[builtins.str] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param outputs: (Required) specify the outputs to extract from the JavaScript JSON response.
        :param api_params: (Required) API Params to submit with the request to the api. You may include variables which will be substitued for inputs during step execution as such {{INPUT}}
        :param pascal_case_api: (Required) The AWS api represented in PascalCase. This value is used as-is in the SSM yaml/json. This is used as the default for javaScriptApi (see that param).
        :param service: (Required) The AWS service to be invoked.
        :param java_script_api: (Optional) The api as represented the AWS JavaScript API. This is usually lowerCamelCase. This is used in the simulation run to execute the AWS API against the JavaScript SDK. Default: - will use the camelCaseApi param and substitute the first character for lowercase by default.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = AwsApiStepProps(
            outputs=outputs,
            api_params=api_params,
            pascal_case_api=pascal_case_api,
            service=service,
            java_script_api=java_script_api,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Derives the inputs by parsing the apiParams to find matches for inputs in double circle braces ("{{INPUT}}").

        :return: list of required inputs.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.

        :return: Outputs as specified in params
        '''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="apiParams")
    def api_params(self) -> "DictFormat":
        return typing.cast("DictFormat", jsii.get(self, "apiParams"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="javaScriptApi")
    def java_script_api(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "javaScriptApi"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="outputs")
    def outputs(self) -> typing.List[Output]:
        return typing.cast(typing.List[Output], jsii.get(self, "outputs"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="pascalCaseApi")
    def pascal_case_api(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "pascalCaseApi"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="service")
    def service(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "service"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.AwsInvocationProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "api_params": "apiParams",
        "pascal_case_api": "pascalCaseApi",
        "service": "service",
        "java_script_api": "javaScriptApi",
    },
)
class AwsInvocationProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        api_params: typing.Mapping[builtins.str, typing.Any],
        pascal_case_api: builtins.str,
        service: builtins.str,
        java_script_api: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param api_params: (Required) API Params to submit with the request to the api. You may include variables which will be substitued for inputs during step execution as such {{INPUT}}
        :param pascal_case_api: (Required) The AWS api represented in PascalCase. This value is used as-is in the SSM yaml/json. This is used as the default for javaScriptApi (see that param).
        :param service: (Required) The AWS service to be invoked.
        :param java_script_api: (Optional) The api as represented the AWS JavaScript API. This is usually lowerCamelCase. This is used in the simulation run to execute the AWS API against the JavaScript SDK. Default: - will use the camelCaseApi param and substitute the first character for lowercase by default.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "api_params": api_params,
            "pascal_case_api": pascal_case_api,
            "service": service,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if java_script_api is not None:
            self._values["java_script_api"] = java_script_api

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def api_params(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''(Required) API Params to submit with the request to the api.

        You may include variables which will be substitued for inputs during step execution as such {{INPUT}}

        Example::

            { 'VolumeIds': ['{{ EbsDescribeInstance.VolumeId }}'] }
        '''
        result = self._values.get("api_params")
        assert result is not None, "Required property 'api_params' is missing"
        return typing.cast(typing.Mapping[builtins.str, typing.Any], result)

    @builtins.property
    def pascal_case_api(self) -> builtins.str:
        '''(Required) The AWS api represented in PascalCase.

        This value is used as-is in the SSM yaml/json.
        This is used as the default for javaScriptApi (see that param).

        Example::

            DescribeInstances
        '''
        result = self._values.get("pascal_case_api")
        assert result is not None, "Required property 'pascal_case_api' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service(self) -> builtins.str:
        '''(Required) The AWS service to be invoked.

        Example::

            ec2
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def java_script_api(self) -> typing.Optional[builtins.str]:
        '''(Optional) The api as represented the AWS JavaScript API.

        This is usually lowerCamelCase.
        This is used in the simulation run to execute the AWS API against the JavaScript SDK.

        :default: - will use the camelCaseApi param and substitute the first character for lowercase by default.

        Example::

            describeInstances
        '''
        result = self._values.get("java_script_api")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AwsInvocationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class BranchStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.BranchStep",
):
    '''AutomationStep implementation of aws:branch https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-branch.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        choices: typing.Sequence[Choice],
        default_step_name: typing.Optional[builtins.str] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param choices: (Required) list of choices. The first matched choice will be used to jump to the step specified in the choice.
        :param default_step_name: (Optional) default step in all of the choices evaluate to false. Default: undefined - the next step in the chain will be invoked. See AWS Documentation for branch below.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = BranchStepProps(
            choices=choices,
            default_step_name=default_step_name,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.

        :return: all of the inputsToTest from the choices provided to the constructor
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''There is no output from branch steps.

        :return: empty list
        '''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="choices")
    def choices(self) -> typing.List[Choice]:
        return typing.cast(typing.List[Choice], jsii.get(self, "choices"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="defaultStepName")
    def default_step_name(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "defaultStepName"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.BranchStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "choices": "choices",
        "default_step_name": "defaultStepName",
    },
)
class BranchStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        choices: typing.Sequence[Choice],
        default_step_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param choices: (Required) list of choices. The first matched choice will be used to jump to the step specified in the choice.
        :param default_step_name: (Optional) default step in all of the choices evaluate to false. Default: undefined - the next step in the chain will be invoked. See AWS Documentation for branch below.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "choices": choices,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if default_step_name is not None:
            self._values["default_step_name"] = default_step_name

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def choices(self) -> typing.List[Choice]:
        '''(Required) list of choices.

        The first matched choice will be used to jump to the step specified in the choice.
        '''
        result = self._values.get("choices")
        assert result is not None, "Required property 'choices' is missing"
        return typing.cast(typing.List[Choice], result)

    @builtins.property
    def default_step_name(self) -> typing.Optional[builtins.str]:
        '''(Optional) default step in all of the choices evaluate to false.

        :default: undefined - the next step in the chain will be invoked. See AWS Documentation for branch below.
        '''
        result = self._values.get("default_step_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "BranchStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ChangeInstanceStateSimulationProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "aws_invoker": "awsInvoker",
    },
)
class ChangeInstanceStateSimulationProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        aws_invoker: IAwsInvoker,
    ) -> None:
        '''Properties for ChangeInstanceStateStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param aws_invoker: (Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call). Default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "aws_invoker": aws_invoker,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def aws_invoker(self) -> IAwsInvoker:
        '''(Optional) Use this as a hook to inject an alternate IAwsInvoker (for mocking the AWS API call).

        :default: - will perform a real invocation of the JavaScript AWS SDK using ReflectiveAwsInvoker class.
        '''
        result = self._values.get("aws_invoker")
        assert result is not None, "Required property 'aws_invoker' is missing"
        return typing.cast(IAwsInvoker, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ChangeInstanceStateSimulationProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ChangeInstanceStateStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ChangeInstanceStateStep",
):
    '''AutomationStep implemenation for aws:changeInstanceState https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-changestate.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        desired_state: "IDesiredStateVariable",
        instance_ids: IStringListVariable,
        additional_info: typing.Optional[IStringVariable] = None,
        check_state_only: typing.Optional["IBooleanVariable"] = None,
        force: typing.Optional["IBooleanVariable"] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param desired_state: The desired state. When set to running, this action waits for the Amazon EC2 state to be Running, the Instance Status to be OK, and the System Status to be OK before completing.
        :param instance_ids: The IDs of the instances.
        :param additional_info: (Optional) Reserved.
        :param check_state_only: (Optional) If false, sets the instance state to the desired state. If true, asserts the desired state using polling. Default: false
        :param force: (Optional) If set, forces the instances to stop. The instances don't have an opportunity to flush file system caches or file system metadata. If you use this option, you must perform file system check and repair procedures. This option isn't recommended for EC2 instances for Windows Server.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = ChangeInstanceStateStepProps(
            desired_state=desired_state,
            instance_ids=instance_ids,
            additional_info=additional_info,
            check_state_only=check_state_only,
            force=force,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''This step has no outputs.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="desiredState")
    def desired_state(self) -> "IDesiredStateVariable":
        return typing.cast("IDesiredStateVariable", jsii.get(self, "desiredState"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceIds")
    def instance_ids(self) -> IStringListVariable:
        return typing.cast(IStringListVariable, jsii.get(self, "instanceIds"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="additionalInfo")
    def additional_info(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "additionalInfo"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="checkStateOnly")
    def check_state_only(self) -> typing.Optional["IBooleanVariable"]:
        return typing.cast(typing.Optional["IBooleanVariable"], jsii.get(self, "checkStateOnly"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="force")
    def force(self) -> typing.Optional["IBooleanVariable"]:
        return typing.cast(typing.Optional["IBooleanVariable"], jsii.get(self, "force"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ChangeInstanceStateStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "desired_state": "desiredState",
        "instance_ids": "instanceIds",
        "additional_info": "additionalInfo",
        "check_state_only": "checkStateOnly",
        "force": "force",
    },
)
class ChangeInstanceStateStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        desired_state: "IDesiredStateVariable",
        instance_ids: IStringListVariable,
        additional_info: typing.Optional[IStringVariable] = None,
        check_state_only: typing.Optional["IBooleanVariable"] = None,
        force: typing.Optional["IBooleanVariable"] = None,
    ) -> None:
        '''Properties for ChangeInstanceStateStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param desired_state: The desired state. When set to running, this action waits for the Amazon EC2 state to be Running, the Instance Status to be OK, and the System Status to be OK before completing.
        :param instance_ids: The IDs of the instances.
        :param additional_info: (Optional) Reserved.
        :param check_state_only: (Optional) If false, sets the instance state to the desired state. If true, asserts the desired state using polling. Default: false
        :param force: (Optional) If set, forces the instances to stop. The instances don't have an opportunity to flush file system caches or file system metadata. If you use this option, you must perform file system check and repair procedures. This option isn't recommended for EC2 instances for Windows Server.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "desired_state": desired_state,
            "instance_ids": instance_ids,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if additional_info is not None:
            self._values["additional_info"] = additional_info
        if check_state_only is not None:
            self._values["check_state_only"] = check_state_only
        if force is not None:
            self._values["force"] = force

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def desired_state(self) -> "IDesiredStateVariable":
        '''The desired state.

        When set to running, this action waits for the Amazon EC2 state to be Running, the Instance Status to be OK,
        and the System Status to be OK before completing.
        '''
        result = self._values.get("desired_state")
        assert result is not None, "Required property 'desired_state' is missing"
        return typing.cast("IDesiredStateVariable", result)

    @builtins.property
    def instance_ids(self) -> IStringListVariable:
        '''The IDs of the instances.'''
        result = self._values.get("instance_ids")
        assert result is not None, "Required property 'instance_ids' is missing"
        return typing.cast(IStringListVariable, result)

    @builtins.property
    def additional_info(self) -> typing.Optional[IStringVariable]:
        '''(Optional) Reserved.'''
        result = self._values.get("additional_info")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def check_state_only(self) -> typing.Optional["IBooleanVariable"]:
        '''(Optional) If false, sets the instance state to the desired state.

        If true, asserts the desired state using polling.

        :default: false
        '''
        result = self._values.get("check_state_only")
        return typing.cast(typing.Optional["IBooleanVariable"], result)

    @builtins.property
    def force(self) -> typing.Optional["IBooleanVariable"]:
        '''(Optional) If set, forces the instances to stop.

        The instances don't have an opportunity to flush file system caches or file system metadata.
        If you use this option, you must perform file system check and repair procedures.
        This option isn't recommended for EC2 instances for Windows Server.
        '''
        result = self._values.get("force")
        return typing.cast(typing.Optional["IBooleanVariable"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ChangeInstanceStateStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CommandDocument(
    SsmDocument,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CommandDocument",
):
    '''https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html The CommandDocument will delegate execution responsibility to the CammandSteps that it receives. The SsmDocument parent class contains methods to runSimulation() as well as print().'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        assume_role: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
        doc_outputs: typing.Optional[typing.Sequence[DocumentOutput]] = None,
        document_format: typing.Optional[DocumentFormat] = None,
        document_name: typing.Optional[builtins.str] = None,
        header: typing.Optional[builtins.str] = None,
        requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.CfnTag]] = None,
        target_type: typing.Optional[builtins.str] = None,
        version_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param assume_role: (Optional) Assume role to use for this document. If provided, this value MUST be included as one of the documentInput names.
        :param description: (Optional) Description of the document. Defaults to the document name.
        :param doc_inputs: (Optional) Inputs required by the document.
        :param doc_outputs: (Optional) Outputs to be emitted from the document. The outputs are placed in a StringSet called outputs (as is done in SSM). Default: []
        :param document_format: (Optional) Specifies whether this document should be written as YAML or JSON. Default: JSON
        :param document_name: (Optional) Name of the document. Will default to the id provided for the CDK node.
        :param header: (Optional) A Header/comment to include at the start of a YAML document. JSON documents do not support headers.
        :param requires: ``AWS::SSM::Document.Requires``.
        :param tags: ``AWS::SSM::Document.Tags``.
        :param target_type: ``AWS::SSM::Document.TargetType``.
        :param version_name: ``AWS::SSM::Document.VersionName``.
        '''
        props = CommandDocumentProps(
            assume_role=assume_role,
            description=description,
            doc_inputs=doc_inputs,
            doc_outputs=doc_outputs,
            document_format=document_format,
            document_name=document_name,
            header=header,
            requires=requires,
            tags=tags,
            target_type=target_type,
            version_name=version_name,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="addStep")
    def add_step(self, component: ICommandComponent) -> None:
        '''
        :param component: -
        '''
        return typing.cast(None, jsii.invoke(self, "addStep", [component]))

    @jsii.member(jsii_name="buildSsmDocument")
    def _build_ssm_document(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Delegates building the SSM Document to be converted to a yaml/json to the subclass (Automation etc).'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "buildSsmDocument", []))

    @jsii.member(jsii_name="collectedSteps")
    def collected_steps(self) -> typing.List["CommandStep"]:
        return typing.cast(typing.List["CommandStep"], jsii.invoke(self, "collectedSteps", []))

    @jsii.member(jsii_name="documentType")
    def document_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.invoke(self, "documentType", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="builder")
    def builder(self) -> CommandDocumentBuilder:
        return typing.cast(CommandDocumentBuilder, jsii.get(self, "builder"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CommandDocumentProps",
    jsii_struct_bases=[SsmDocumentProps],
    name_mapping={
        "assume_role": "assumeRole",
        "description": "description",
        "doc_inputs": "docInputs",
        "doc_outputs": "docOutputs",
        "document_format": "documentFormat",
        "document_name": "documentName",
        "header": "header",
        "requires": "requires",
        "tags": "tags",
        "target_type": "targetType",
        "version_name": "versionName",
    },
)
class CommandDocumentProps(SsmDocumentProps):
    def __init__(
        self,
        *,
        assume_role: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
        doc_outputs: typing.Optional[typing.Sequence[DocumentOutput]] = None,
        document_format: typing.Optional[DocumentFormat] = None,
        document_name: typing.Optional[builtins.str] = None,
        header: typing.Optional[builtins.str] = None,
        requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.CfnTag]] = None,
        target_type: typing.Optional[builtins.str] = None,
        version_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param assume_role: (Optional) Assume role to use for this document. If provided, this value MUST be included as one of the documentInput names.
        :param description: (Optional) Description of the document. Defaults to the document name.
        :param doc_inputs: (Optional) Inputs required by the document.
        :param doc_outputs: (Optional) Outputs to be emitted from the document. The outputs are placed in a StringSet called outputs (as is done in SSM). Default: []
        :param document_format: (Optional) Specifies whether this document should be written as YAML or JSON. Default: JSON
        :param document_name: (Optional) Name of the document. Will default to the id provided for the CDK node.
        :param header: (Optional) A Header/comment to include at the start of a YAML document. JSON documents do not support headers.
        :param requires: ``AWS::SSM::Document.Requires``.
        :param tags: ``AWS::SSM::Document.Tags``.
        :param target_type: ``AWS::SSM::Document.TargetType``.
        :param version_name: ``AWS::SSM::Document.VersionName``.
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if assume_role is not None:
            self._values["assume_role"] = assume_role
        if description is not None:
            self._values["description"] = description
        if doc_inputs is not None:
            self._values["doc_inputs"] = doc_inputs
        if doc_outputs is not None:
            self._values["doc_outputs"] = doc_outputs
        if document_format is not None:
            self._values["document_format"] = document_format
        if document_name is not None:
            self._values["document_name"] = document_name
        if header is not None:
            self._values["header"] = header
        if requires is not None:
            self._values["requires"] = requires
        if tags is not None:
            self._values["tags"] = tags
        if target_type is not None:
            self._values["target_type"] = target_type
        if version_name is not None:
            self._values["version_name"] = version_name

    @builtins.property
    def assume_role(self) -> typing.Optional[builtins.str]:
        '''(Optional) Assume role to use for this document.

        If provided, this value MUST be included as one of the documentInput names.
        '''
        result = self._values.get("assume_role")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) Description of the document.

        Defaults to the document name.
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def doc_inputs(self) -> typing.Optional[typing.List[Input]]:
        '''(Optional) Inputs required by the document.'''
        result = self._values.get("doc_inputs")
        return typing.cast(typing.Optional[typing.List[Input]], result)

    @builtins.property
    def doc_outputs(self) -> typing.Optional[typing.List[DocumentOutput]]:
        '''(Optional) Outputs to be emitted from the document.

        The outputs are placed in a StringSet called outputs (as is done in SSM).

        :default: []
        '''
        result = self._values.get("doc_outputs")
        return typing.cast(typing.Optional[typing.List[DocumentOutput]], result)

    @builtins.property
    def document_format(self) -> typing.Optional[DocumentFormat]:
        '''(Optional) Specifies whether this document should be written as YAML or JSON.

        :default: JSON
        '''
        result = self._values.get("document_format")
        return typing.cast(typing.Optional[DocumentFormat], result)

    @builtins.property
    def document_name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the document.

        Will default to the id provided for the CDK node.
        '''
        result = self._values.get("document_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def header(self) -> typing.Optional[builtins.str]:
        '''(Optional) A Header/comment to include at the start of a YAML document.

        JSON documents do not support headers.
        '''
        result = self._values.get("header")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def requires(
        self,
    ) -> typing.Optional[typing.Union[aws_cdk.IResolvable, typing.List[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]]:
        '''``AWS::SSM::Document.Requires``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-requires
        '''
        result = self._values.get("requires")
        return typing.cast(typing.Optional[typing.Union[aws_cdk.IResolvable, typing.List[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]], result)

    @builtins.property
    def tags(self) -> typing.Optional[typing.List[aws_cdk.CfnTag]]:
        '''``AWS::SSM::Document.Tags``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-tags
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[typing.List[aws_cdk.CfnTag]], result)

    @builtins.property
    def target_type(self) -> typing.Optional[builtins.str]:
        '''``AWS::SSM::Document.TargetType``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-targettype
        '''
        result = self._values.get("target_type")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def version_name(self) -> typing.Optional[builtins.str]:
        '''``AWS::SSM::Document.VersionName``.

        :external: true
        :link: http://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ssm-document.html#cfn-ssm-document-versionname
        '''
        result = self._values.get("version_name")
        return typing.cast(typing.Optional[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommandDocumentProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(ICommandComponent)
class CommandStep(
    Step,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.CommandStep",
):
    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = CommandStepProps(
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: CommandDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        return typing.cast(None, jsii.invoke(self, "addToDocument", [doc]))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''RunCommand Steps do not have outputs.

        :return: []
        '''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="prepareSsmEntry")
    def _prepare_ssm_entry(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "prepareSsmEntry", [inputs]))

    @jsii.member(jsii_name="variables")
    def variables(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "variables", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="exitOnFailure")
    def exit_on_failure(self) -> builtins.bool:
        return typing.cast(builtins.bool, jsii.get(self, "exitOnFailure"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="exitOnSuccess")
    def exit_on_success(self) -> builtins.bool:
        return typing.cast(builtins.bool, jsii.get(self, "exitOnSuccess"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="finallyStep")
    def finally_step(self) -> builtins.bool:
        return typing.cast(builtins.bool, jsii.get(self, "finallyStep"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="markSuccessAndExitOnFailure")
    def mark_success_and_exit_on_failure(self) -> builtins.bool:
        return typing.cast(builtins.bool, jsii.get(self, "markSuccessAndExitOnFailure"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platforms")
    @abc.abstractmethod
    def platforms(self) -> typing.List[Platform]:
        ...

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="precondition")
    def precondition(self) -> typing.Optional[Precondition]:
        return typing.cast(typing.Optional[Precondition], jsii.get(self, "precondition"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="allStepsInExecution")
    def all_steps_in_execution(self) -> typing.Optional[typing.List["CommandStep"]]:
        return typing.cast(typing.Optional[typing.List["CommandStep"]], jsii.get(self, "allStepsInExecution"))

    @all_steps_in_execution.setter
    def all_steps_in_execution(
        self,
        value: typing.Optional[typing.List["CommandStep"]],
    ) -> None:
        jsii.set(self, "allStepsInExecution", value)

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="nextStep")
    def next_step(self) -> typing.Optional["CommandStep"]:
        return typing.cast(typing.Optional["CommandStep"], jsii.get(self, "nextStep"))

    @next_step.setter
    def next_step(self, value: typing.Optional["CommandStep"]) -> None:
        jsii.set(self, "nextStep", value)


class _CommandStepProxy(
    CommandStep, jsii.proxy_for(Step) # type: ignore[misc]
):
    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, CommandStep).__jsii_proxy_class__ = lambda : _CommandStepProxy


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CommandStepProps",
    jsii_struct_bases=[StepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
    },
)
class CommandStepProps(StepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
    ) -> None:
        '''
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CommandStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IAutomationComponent)
class CompositeAutomationStep(
    constructs.Construct,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.CompositeAutomationStep",
):
    def __init__(self, scope: constructs.Construct, id: builtins.str) -> None:
        '''
        :param scope: -
        :param id: -
        '''
        jsii.create(self.__class__, self, [scope, id])

    @jsii.member(jsii_name="addToDocument") # type: ignore[misc]
    @abc.abstractmethod
    def add_to_document(self, doc: AutomationDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        ...


class _CompositeAutomationStepProxy(CompositeAutomationStep):
    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: AutomationDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        return typing.cast(None, jsii.invoke(self, "addToDocument", [doc]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, CompositeAutomationStep).__jsii_proxy_class__ = lambda : _CompositeAutomationStepProxy


@jsii.implements(ICommandComponent)
class CompositeCommandStep(
    constructs.Construct,
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.CompositeCommandStep",
):
    def __init__(self, scope: constructs.Construct, id: builtins.str) -> None:
        '''
        :param scope: -
        :param id: -
        '''
        jsii.create(self.__class__, self, [scope, id])

    @jsii.member(jsii_name="addToDocument") # type: ignore[misc]
    @abc.abstractmethod
    def add_to_document(self, doc: CommandDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        ...


class _CompositeCommandStepProxy(CompositeCommandStep):
    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: CommandDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        return typing.cast(None, jsii.invoke(self, "addToDocument", [doc]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, CompositeCommandStep).__jsii_proxy_class__ = lambda : _CompositeCommandStepProxy


class ConfigureDockerStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ConfigureDockerStep",
):
    '''AutomationStep implemenation for aws:UpdateAgent https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-configuredocker.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        action: "IActionVariable",
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param action: The type of action to perform. True correlates to "Install" false correlates to "Uninstall"
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = ConfigureDockerStepProps(
            action=action,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Inputs required for this command include agentName allowDowngrade source and targetVersion if version other than latest is desired.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="installAction")
    def install_action(self) -> "IActionVariable":
        return typing.cast("IActionVariable", jsii.get(self, "installAction"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ConfigureDockerStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "action": "action",
    },
)
class ConfigureDockerStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        action: "IActionVariable",
    ) -> None:
        '''Properties for ConfigureDocker step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param action: The type of action to perform. True correlates to "Install" false correlates to "Uninstall"
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "action": action,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def action(self) -> "IActionVariable":
        '''The type of action to perform.

        True correlates to "Install" false correlates to "Uninstall"
        '''
        result = self._values.get("action")
        assert result is not None, "Required property 'action' is missing"
        return typing.cast("IActionVariable", result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ConfigureDockerStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ConfigurePackageStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ConfigurePackageStep",
):
    '''CommandStep implemenation for aws:configurePackage https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-configurepackage.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        package_action: "IActionVariable",
        package_name: "IPackageNameVariable",
        additional_arguments: typing.Optional[IStringMapVariable] = None,
        installation_type: typing.Optional["IInstallationTypeVariable"] = None,
        version: typing.Optional[IStringVariable] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param package_action: Install or uninstall a package.
        :param package_name: The name of the AWS package to install or uninstall.
        :param additional_arguments: The additional parameters to provide to your install, uninstall, or update scripts. Each parameter must be prefixed with SSM_. You can reference a Parameter Store parameter in your additional arguments by using the convention {{ssm:parameter-name}}. To use the additional parameter in your install, uninstall, or update script, you must reference the parameter as an environment variable using the syntax appropriate for the operating system. For example, in PowerShell, you reference the SSM_arg argument as $Env:SSM_arg. There is no limit to the number of arguments you define, but the additional argument input has a 4096 character limit. This limit includes all of the keys and values you define.
        :param installation_type: The type of installation to perform. If you specify Uninstall and reinstall, the package is completely uninstalled, and then reinstalled. The application is unavailable until the reinstallation is complete. If you specify In-place update, only new or changed files are added to the existing installation according you instructions you provide in an update script. The application remains available throughout the update process. The In-place update option isn't supported for AWS-published packages. Uninstall and reinstall is the default value.
        :param version: A specific version of the package to install or uninstall. If installing, the system installs the latest published version, by default. If uninstalling, the system uninstalls the currently installed version, by default. If no installed version is found, the latest published version is downloaded, and the uninstall action is run.
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = ConfigurePackageStepProps(
            package_action=package_action,
            package_name=package_name,
            additional_arguments=additional_arguments,
            installation_type=installation_type,
            version=version,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Inputs required for this command include ...'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="packageAction")
    def package_action(self) -> "IActionVariable":
        return typing.cast("IActionVariable", jsii.get(self, "packageAction"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="packageName")
    def package_name(self) -> "IPackageNameVariable":
        return typing.cast("IPackageNameVariable", jsii.get(self, "packageName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="additionalArguments")
    def additional_arguments(self) -> typing.Optional[IStringMapVariable]:
        return typing.cast(typing.Optional[IStringMapVariable], jsii.get(self, "additionalArguments"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="installationType")
    def installation_type(self) -> typing.Optional["IInstallationTypeVariable"]:
        return typing.cast(typing.Optional["IInstallationTypeVariable"], jsii.get(self, "installationType"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="version")
    def version(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "version"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ConfigurePackageStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "package_action": "packageAction",
        "package_name": "packageName",
        "additional_arguments": "additionalArguments",
        "installation_type": "installationType",
        "version": "version",
    },
)
class ConfigurePackageStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        package_action: "IActionVariable",
        package_name: "IPackageNameVariable",
        additional_arguments: typing.Optional[IStringMapVariable] = None,
        installation_type: typing.Optional["IInstallationTypeVariable"] = None,
        version: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties ConfigurePackage step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param package_action: Install or uninstall a package.
        :param package_name: The name of the AWS package to install or uninstall.
        :param additional_arguments: The additional parameters to provide to your install, uninstall, or update scripts. Each parameter must be prefixed with SSM_. You can reference a Parameter Store parameter in your additional arguments by using the convention {{ssm:parameter-name}}. To use the additional parameter in your install, uninstall, or update script, you must reference the parameter as an environment variable using the syntax appropriate for the operating system. For example, in PowerShell, you reference the SSM_arg argument as $Env:SSM_arg. There is no limit to the number of arguments you define, but the additional argument input has a 4096 character limit. This limit includes all of the keys and values you define.
        :param installation_type: The type of installation to perform. If you specify Uninstall and reinstall, the package is completely uninstalled, and then reinstalled. The application is unavailable until the reinstallation is complete. If you specify In-place update, only new or changed files are added to the existing installation according you instructions you provide in an update script. The application remains available throughout the update process. The In-place update option isn't supported for AWS-published packages. Uninstall and reinstall is the default value.
        :param version: A specific version of the package to install or uninstall. If installing, the system installs the latest published version, by default. If uninstalling, the system uninstalls the currently installed version, by default. If no installed version is found, the latest published version is downloaded, and the uninstall action is run.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "package_action": package_action,
            "package_name": package_name,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition
        if additional_arguments is not None:
            self._values["additional_arguments"] = additional_arguments
        if installation_type is not None:
            self._values["installation_type"] = installation_type
        if version is not None:
            self._values["version"] = version

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def package_action(self) -> "IActionVariable":
        '''Install or uninstall a package.'''
        result = self._values.get("package_action")
        assert result is not None, "Required property 'package_action' is missing"
        return typing.cast("IActionVariable", result)

    @builtins.property
    def package_name(self) -> "IPackageNameVariable":
        '''The name of the AWS package to install or uninstall.'''
        result = self._values.get("package_name")
        assert result is not None, "Required property 'package_name' is missing"
        return typing.cast("IPackageNameVariable", result)

    @builtins.property
    def additional_arguments(self) -> typing.Optional[IStringMapVariable]:
        '''The additional parameters to provide to your install, uninstall, or update scripts.

        Each parameter must be prefixed with SSM_.
        You can reference a Parameter Store parameter in your additional arguments by using the convention {{ssm:parameter-name}}.
        To use the additional parameter in your install, uninstall, or update script,
        you must reference the parameter as an environment variable using the syntax appropriate for the operating system.
        For example, in PowerShell, you reference the SSM_arg argument as $Env:SSM_arg.
        There is no limit to the number of arguments you define, but the additional argument input has a 4096 character limit.
        This limit includes all of the keys and values you define.
        '''
        result = self._values.get("additional_arguments")
        return typing.cast(typing.Optional[IStringMapVariable], result)

    @builtins.property
    def installation_type(self) -> typing.Optional["IInstallationTypeVariable"]:
        '''The type of installation to perform.

        If you specify Uninstall and reinstall, the package is completely uninstalled, and then reinstalled.
        The application is unavailable until the reinstallation is complete.
        If you specify In-place update, only new or changed files are added to the existing installation according you instructions you provide in an update script.
        The application remains available throughout the update process.
        The In-place update option isn't supported for AWS-published packages. Uninstall and reinstall is the default value.
        '''
        result = self._values.get("installation_type")
        return typing.cast(typing.Optional["IInstallationTypeVariable"], result)

    @builtins.property
    def version(self) -> typing.Optional[IStringVariable]:
        '''A specific version of the package to install or uninstall.

        If installing, the system installs the latest published version, by default.
        If uninstalling, the system uninstalls the currently installed version, by default.
        If no installed version is found, the latest published version is downloaded, and the uninstall action is run.
        '''
        result = self._values.get("version")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ConfigurePackageStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CopyImageStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CopyImageStep",
):
    '''AutomationStep implemenation for aws:copyImage https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-copyimage.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        image_name: IStringVariable,
        source_image_id: IStringVariable,
        source_region: IStringVariable,
        client_token: typing.Optional[IStringVariable] = None,
        encrypted: typing.Optional["IBooleanVariable"] = None,
        image_description: typing.Optional[IStringVariable] = None,
        kms_key_id: typing.Optional[IStringVariable] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param image_name: The name for the image.
        :param source_image_id: The AMI ID to copy from the source Region.
        :param source_region: The region where the source AMI exists.
        :param client_token: (Optional) A unique, case-sensitive identifier that you provide to ensure request idempotency.
        :param encrypted: (Optional) Encrypt the target AMI.
        :param image_description: (Optional) A description of the image.
        :param kms_key_id: (Optional) The full Amazon Resource Name (ARN) of the AWS KMS key to use when encrypting the snapshots of an image during a copy operation.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = CopyImageStepProps(
            image_name=image_name,
            source_image_id=source_image_id,
            source_region=source_region,
            client_token=client_token,
            encrypted=encrypted,
            image_description=image_description,
            kms_key_id=kms_key_id,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="imageName")
    def image_name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "imageName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceImageId")
    def source_image_id(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "sourceImageId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceRegion")
    def source_region(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "sourceRegion"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clientToken")
    def client_token(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "clientToken"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="encrypted")
    def encrypted(self) -> typing.Optional["IBooleanVariable"]:
        return typing.cast(typing.Optional["IBooleanVariable"], jsii.get(self, "encrypted"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="imageDescription")
    def image_description(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "imageDescription"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="kmsKeyId")
    def kms_key_id(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "kmsKeyId"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CopyImageStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "image_name": "imageName",
        "source_image_id": "sourceImageId",
        "source_region": "sourceRegion",
        "client_token": "clientToken",
        "encrypted": "encrypted",
        "image_description": "imageDescription",
        "kms_key_id": "kmsKeyId",
    },
)
class CopyImageStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        image_name: IStringVariable,
        source_image_id: IStringVariable,
        source_region: IStringVariable,
        client_token: typing.Optional[IStringVariable] = None,
        encrypted: typing.Optional["IBooleanVariable"] = None,
        image_description: typing.Optional[IStringVariable] = None,
        kms_key_id: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties for CopyImageStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param image_name: The name for the image.
        :param source_image_id: The AMI ID to copy from the source Region.
        :param source_region: The region where the source AMI exists.
        :param client_token: (Optional) A unique, case-sensitive identifier that you provide to ensure request idempotency.
        :param encrypted: (Optional) Encrypt the target AMI.
        :param image_description: (Optional) A description of the image.
        :param kms_key_id: (Optional) The full Amazon Resource Name (ARN) of the AWS KMS key to use when encrypting the snapshots of an image during a copy operation.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "image_name": image_name,
            "source_image_id": source_image_id,
            "source_region": source_region,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if client_token is not None:
            self._values["client_token"] = client_token
        if encrypted is not None:
            self._values["encrypted"] = encrypted
        if image_description is not None:
            self._values["image_description"] = image_description
        if kms_key_id is not None:
            self._values["kms_key_id"] = kms_key_id

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def image_name(self) -> IStringVariable:
        '''The name for the image.'''
        result = self._values.get("image_name")
        assert result is not None, "Required property 'image_name' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def source_image_id(self) -> IStringVariable:
        '''The AMI ID to copy from the source Region.'''
        result = self._values.get("source_image_id")
        assert result is not None, "Required property 'source_image_id' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def source_region(self) -> IStringVariable:
        '''The region where the source AMI exists.'''
        result = self._values.get("source_region")
        assert result is not None, "Required property 'source_region' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def client_token(self) -> typing.Optional[IStringVariable]:
        '''(Optional) A unique, case-sensitive identifier that you provide to ensure request idempotency.'''
        result = self._values.get("client_token")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def encrypted(self) -> typing.Optional["IBooleanVariable"]:
        '''(Optional) Encrypt the target AMI.'''
        result = self._values.get("encrypted")
        return typing.cast(typing.Optional["IBooleanVariable"], result)

    @builtins.property
    def image_description(self) -> typing.Optional[IStringVariable]:
        '''(Optional) A description of the image.'''
        result = self._values.get("image_description")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def kms_key_id(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The full Amazon Resource Name (ARN) of the AWS KMS key to use when encrypting the snapshots of an image during a copy operation.'''
        result = self._values.get("kms_key_id")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CopyImageStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CreateImageStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CreateImageStep",
):
    '''AutomationStep implemenation for aws:createImage https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-create.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        image_name: IStringVariable,
        instance_id: IStringVariable,
        block_device_mappings: typing.Optional[IStringMapVariable] = None,
        image_description: typing.Optional[IStringVariable] = None,
        no_reboot: typing.Optional["IBooleanVariable"] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param image_name: The name for the image.
        :param instance_id: The ID of the instance.
        :param block_device_mappings: (Optional) The block devices for the instance.
        :param image_description: (Optional) A description of the image.
        :param no_reboot: (Optional) By default, Amazon Elastic Compute Cloud (Amazon EC2) attempts to shut down and reboot the instance before creating the image. If the No Reboot option is set to true, Amazon EC2 doesn't shut down the instance before creating the image. When this option is used, file system integrity on the created image can't be guaranteed. If you don't want the instance to run after you create an AMI from it, first use the aws:changeInstanceState – Change or assert instance state action to stop the instance, and then use this aws:createImage action with the NoReboot option set to true.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = CreateImageStepProps(
            image_name=image_name,
            instance_id=instance_id,
            block_device_mappings=block_device_mappings,
            image_description=image_description,
            no_reboot=no_reboot,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="imageName")
    def image_name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "imageName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceId")
    def instance_id(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "instanceId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="blockDeviceMappings")
    def block_device_mappings(self) -> typing.Optional[IStringMapVariable]:
        return typing.cast(typing.Optional[IStringMapVariable], jsii.get(self, "blockDeviceMappings"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="imageDescription")
    def image_description(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "imageDescription"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="noReboot")
    def no_reboot(self) -> typing.Optional["IBooleanVariable"]:
        return typing.cast(typing.Optional["IBooleanVariable"], jsii.get(self, "noReboot"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CreateImageStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "image_name": "imageName",
        "instance_id": "instanceId",
        "block_device_mappings": "blockDeviceMappings",
        "image_description": "imageDescription",
        "no_reboot": "noReboot",
    },
)
class CreateImageStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        image_name: IStringVariable,
        instance_id: IStringVariable,
        block_device_mappings: typing.Optional[IStringMapVariable] = None,
        image_description: typing.Optional[IStringVariable] = None,
        no_reboot: typing.Optional["IBooleanVariable"] = None,
    ) -> None:
        '''Properties for CreateImageStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param image_name: The name for the image.
        :param instance_id: The ID of the instance.
        :param block_device_mappings: (Optional) The block devices for the instance.
        :param image_description: (Optional) A description of the image.
        :param no_reboot: (Optional) By default, Amazon Elastic Compute Cloud (Amazon EC2) attempts to shut down and reboot the instance before creating the image. If the No Reboot option is set to true, Amazon EC2 doesn't shut down the instance before creating the image. When this option is used, file system integrity on the created image can't be guaranteed. If you don't want the instance to run after you create an AMI from it, first use the aws:changeInstanceState – Change or assert instance state action to stop the instance, and then use this aws:createImage action with the NoReboot option set to true.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "image_name": image_name,
            "instance_id": instance_id,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if block_device_mappings is not None:
            self._values["block_device_mappings"] = block_device_mappings
        if image_description is not None:
            self._values["image_description"] = image_description
        if no_reboot is not None:
            self._values["no_reboot"] = no_reboot

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def image_name(self) -> IStringVariable:
        '''The name for the image.'''
        result = self._values.get("image_name")
        assert result is not None, "Required property 'image_name' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def instance_id(self) -> IStringVariable:
        '''The ID of the instance.'''
        result = self._values.get("instance_id")
        assert result is not None, "Required property 'instance_id' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def block_device_mappings(self) -> typing.Optional[IStringMapVariable]:
        '''(Optional) The block devices for the instance.'''
        result = self._values.get("block_device_mappings")
        return typing.cast(typing.Optional[IStringMapVariable], result)

    @builtins.property
    def image_description(self) -> typing.Optional[IStringVariable]:
        '''(Optional) A description of the image.'''
        result = self._values.get("image_description")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def no_reboot(self) -> typing.Optional["IBooleanVariable"]:
        '''(Optional) By default, Amazon Elastic Compute Cloud (Amazon EC2) attempts to shut down and reboot the instance before creating the image.

        If the No Reboot option is set to true, Amazon EC2 doesn't shut down the instance before creating the image. When this option is used, file system integrity on the created image can't be guaranteed.
        If you don't want the instance to run after you create an AMI from it, first use the aws:changeInstanceState – Change or assert instance state action to stop the instance, and then use this aws:createImage action with the NoReboot option set to true.
        '''
        result = self._values.get("no_reboot")
        return typing.cast(typing.Optional["IBooleanVariable"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CreateImageStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CreateStackStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CreateStackStep",
):
    '''AutomationStep implementation for aws:createStack https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-createstack.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        stack_name: IStringVariable,
        template: BodyOrUrlProp,
        capabilities: typing.Optional[IStringListVariable] = None,
        client_request_token: typing.Optional[IStringVariable] = None,
        notification_ar_ns: typing.Optional[IStringListVariable] = None,
        on_stack_failure: typing.Optional["IOnFailureVariable"] = None,
        parameters: typing.Optional[IMapListVariable] = None,
        resource_types: typing.Optional[IStringListVariable] = None,
        role_arn: typing.Optional[IStringVariable] = None,
        stack_policy: typing.Optional[BodyOrUrlProp] = None,
        tags: typing.Optional[IMapListVariable] = None,
        timeout_in_minutes: typing.Optional[INumberVariable] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param stack_name: The name that is associated with the stack. The name must be unique in the Region in which you're creating the stack.
        :param template: Template body or URL. For more information, see `Template Anatomy <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-anatomy.html>`_.
        :param capabilities: (Optional) A list of values that you specify before CloudFormation can create certain stacks. Some stack templates include resources that can affect permissions in your AWS account. For example, creating new AWS Identity and Access Management (IAM) users can affect permissions in your account. For those stacks, you must explicitly acknowledge their capabilities by specifying this parameter. Valid values include CAPABILITY_IAM, CAPABILITY_NAMED_IAM, and CAPABILITY_AUTO_EXPAND.
        :param client_request_token: (Optional) A unique identifier for this CreateStack request. Specify this token if you set maxAttempts in this step to a value greater than 1. By specifying this token, CloudFormation knows that you aren't attempting to create a new stack with the same name.
        :param notification_ar_ns: (Optional) The Amazon Simple Notification Service (Amazon SNS) topic ARNs for publishing stack-related events.
        :param on_stack_failure: (Optional) Determines the action to take if stack creation failed. Default: - Rollback on failure
        :param parameters: (Optional) A list of Parameter structures that specify input parameters for the stack. For more information, see the `Parameter <https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html>`_ data type.
        :param resource_types: (Optional) The template resource types that you have permissions to work with for this create stack action. For example: AWS::EC2::Instance, AWS::EC2::*, or Custom::MyCustomInstance.
        :param role_arn: (Optional) The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that CloudFormation assumes to create the stack. CloudFormation uses the role's credentials to make calls on your behalf. CloudFormation always uses this role for all future operations on the stack. As long as users have permission to operate on the stack, CloudFormation uses this role even if the users don't have permission to pass it. Ensure that the role grants the least amount of privileges. If you don't specify a value, CloudFormation uses the role that was previously associated with the stack. If no role is available, CloudFormation uses a temporary session that is generated from your user credentials.
        :param stack_policy: (Optional) Stack policy body or URL. For more information, see `Prevent Updates to Stack Resources <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/protect-stack-resources.html>`_.
        :param tags: (Optional) Key-value pairs to associate with this stack. CloudFormation also propagates these tags to the resources created in the stack. You can specify a maximum number of 10 tags.
        :param timeout_in_minutes: (Optional) The amount of time that can pass before the stack status becomes CREATE_FAILED. If DisableRollback isn't set or is set to false, the stack will be rolled back.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = CreateStackStepProps(
            stack_name=stack_name,
            template=template,
            capabilities=capabilities,
            client_request_token=client_request_token,
            notification_ar_ns=notification_ar_ns,
            on_stack_failure=on_stack_failure,
            parameters=parameters,
            resource_types=resource_types,
            role_arn=role_arn,
            stack_policy=stack_policy,
            tags=tags,
            timeout_in_minutes=timeout_in_minutes,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="stackName")
    def stack_name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "stackName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="capabilities")
    def capabilities(self) -> typing.Optional[IStringListVariable]:
        return typing.cast(typing.Optional[IStringListVariable], jsii.get(self, "capabilities"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clientRequestToken")
    def client_request_token(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "clientRequestToken"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="notificationARNs")
    def notification_ar_ns(self) -> typing.Optional[IStringListVariable]:
        return typing.cast(typing.Optional[IStringListVariable], jsii.get(self, "notificationARNs"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="onStackFailure")
    def on_stack_failure(self) -> typing.Optional["IOnFailureVariable"]:
        return typing.cast(typing.Optional["IOnFailureVariable"], jsii.get(self, "onStackFailure"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="parameters")
    def parameters(self) -> typing.Optional[IMapListVariable]:
        return typing.cast(typing.Optional[IMapListVariable], jsii.get(self, "parameters"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="resourceTypes")
    def resource_types(self) -> typing.Optional[IStringListVariable]:
        return typing.cast(typing.Optional[IStringListVariable], jsii.get(self, "resourceTypes"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "roleArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="stackPolicyBody")
    def stack_policy_body(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "stackPolicyBody"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="stackPolicyUrl")
    def stack_policy_url(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "stackPolicyUrl"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[IMapListVariable]:
        return typing.cast(typing.Optional[IMapListVariable], jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="templateBody")
    def template_body(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "templateBody"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="templateUrl")
    def template_url(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "templateUrl"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="timeoutInMinutes")
    def timeout_in_minutes(self) -> typing.Optional[INumberVariable]:
        return typing.cast(typing.Optional[INumberVariable], jsii.get(self, "timeoutInMinutes"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CreateStackStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "stack_name": "stackName",
        "template": "template",
        "capabilities": "capabilities",
        "client_request_token": "clientRequestToken",
        "notification_ar_ns": "notificationARNs",
        "on_stack_failure": "onStackFailure",
        "parameters": "parameters",
        "resource_types": "resourceTypes",
        "role_arn": "roleArn",
        "stack_policy": "stackPolicy",
        "tags": "tags",
        "timeout_in_minutes": "timeoutInMinutes",
    },
)
class CreateStackStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        stack_name: IStringVariable,
        template: BodyOrUrlProp,
        capabilities: typing.Optional[IStringListVariable] = None,
        client_request_token: typing.Optional[IStringVariable] = None,
        notification_ar_ns: typing.Optional[IStringListVariable] = None,
        on_stack_failure: typing.Optional["IOnFailureVariable"] = None,
        parameters: typing.Optional[IMapListVariable] = None,
        resource_types: typing.Optional[IStringListVariable] = None,
        role_arn: typing.Optional[IStringVariable] = None,
        stack_policy: typing.Optional[BodyOrUrlProp] = None,
        tags: typing.Optional[IMapListVariable] = None,
        timeout_in_minutes: typing.Optional[INumberVariable] = None,
    ) -> None:
        '''Properties for CreateStackStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param stack_name: The name that is associated with the stack. The name must be unique in the Region in which you're creating the stack.
        :param template: Template body or URL. For more information, see `Template Anatomy <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-anatomy.html>`_.
        :param capabilities: (Optional) A list of values that you specify before CloudFormation can create certain stacks. Some stack templates include resources that can affect permissions in your AWS account. For example, creating new AWS Identity and Access Management (IAM) users can affect permissions in your account. For those stacks, you must explicitly acknowledge their capabilities by specifying this parameter. Valid values include CAPABILITY_IAM, CAPABILITY_NAMED_IAM, and CAPABILITY_AUTO_EXPAND.
        :param client_request_token: (Optional) A unique identifier for this CreateStack request. Specify this token if you set maxAttempts in this step to a value greater than 1. By specifying this token, CloudFormation knows that you aren't attempting to create a new stack with the same name.
        :param notification_ar_ns: (Optional) The Amazon Simple Notification Service (Amazon SNS) topic ARNs for publishing stack-related events.
        :param on_stack_failure: (Optional) Determines the action to take if stack creation failed. Default: - Rollback on failure
        :param parameters: (Optional) A list of Parameter structures that specify input parameters for the stack. For more information, see the `Parameter <https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html>`_ data type.
        :param resource_types: (Optional) The template resource types that you have permissions to work with for this create stack action. For example: AWS::EC2::Instance, AWS::EC2::*, or Custom::MyCustomInstance.
        :param role_arn: (Optional) The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that CloudFormation assumes to create the stack. CloudFormation uses the role's credentials to make calls on your behalf. CloudFormation always uses this role for all future operations on the stack. As long as users have permission to operate on the stack, CloudFormation uses this role even if the users don't have permission to pass it. Ensure that the role grants the least amount of privileges. If you don't specify a value, CloudFormation uses the role that was previously associated with the stack. If no role is available, CloudFormation uses a temporary session that is generated from your user credentials.
        :param stack_policy: (Optional) Stack policy body or URL. For more information, see `Prevent Updates to Stack Resources <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/protect-stack-resources.html>`_.
        :param tags: (Optional) Key-value pairs to associate with this stack. CloudFormation also propagates these tags to the resources created in the stack. You can specify a maximum number of 10 tags.
        :param timeout_in_minutes: (Optional) The amount of time that can pass before the stack status becomes CREATE_FAILED. If DisableRollback isn't set or is set to false, the stack will be rolled back.
        '''
        if isinstance(template, dict):
            template = BodyOrUrlProp(**template)
        if isinstance(stack_policy, dict):
            stack_policy = BodyOrUrlProp(**stack_policy)
        self._values: typing.Dict[str, typing.Any] = {
            "stack_name": stack_name,
            "template": template,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if capabilities is not None:
            self._values["capabilities"] = capabilities
        if client_request_token is not None:
            self._values["client_request_token"] = client_request_token
        if notification_ar_ns is not None:
            self._values["notification_ar_ns"] = notification_ar_ns
        if on_stack_failure is not None:
            self._values["on_stack_failure"] = on_stack_failure
        if parameters is not None:
            self._values["parameters"] = parameters
        if resource_types is not None:
            self._values["resource_types"] = resource_types
        if role_arn is not None:
            self._values["role_arn"] = role_arn
        if stack_policy is not None:
            self._values["stack_policy"] = stack_policy
        if tags is not None:
            self._values["tags"] = tags
        if timeout_in_minutes is not None:
            self._values["timeout_in_minutes"] = timeout_in_minutes

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def stack_name(self) -> IStringVariable:
        '''The name that is associated with the stack.

        The name must be unique in the Region in which you're creating the stack.
        '''
        result = self._values.get("stack_name")
        assert result is not None, "Required property 'stack_name' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def template(self) -> BodyOrUrlProp:
        '''Template body or URL.

        For more information, see `Template Anatomy <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/template-anatomy.html>`_.
        '''
        result = self._values.get("template")
        assert result is not None, "Required property 'template' is missing"
        return typing.cast(BodyOrUrlProp, result)

    @builtins.property
    def capabilities(self) -> typing.Optional[IStringListVariable]:
        '''(Optional) A list of values that you specify before CloudFormation can create certain stacks.

        Some stack templates include resources that can affect permissions in your AWS account.
        For example, creating new AWS Identity and Access Management (IAM) users can affect permissions in your account.
        For those stacks, you must explicitly acknowledge their capabilities by specifying this parameter.

        Valid values include CAPABILITY_IAM, CAPABILITY_NAMED_IAM, and CAPABILITY_AUTO_EXPAND.
        '''
        result = self._values.get("capabilities")
        return typing.cast(typing.Optional[IStringListVariable], result)

    @builtins.property
    def client_request_token(self) -> typing.Optional[IStringVariable]:
        '''(Optional) A unique identifier for this CreateStack request.

        Specify this token if you set maxAttempts in this step to a value greater than 1.
        By specifying this token, CloudFormation knows that you aren't attempting to create a new stack with the same name.
        '''
        result = self._values.get("client_request_token")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def notification_ar_ns(self) -> typing.Optional[IStringListVariable]:
        '''(Optional) The Amazon Simple Notification Service (Amazon SNS) topic ARNs for publishing stack-related events.'''
        result = self._values.get("notification_ar_ns")
        return typing.cast(typing.Optional[IStringListVariable], result)

    @builtins.property
    def on_stack_failure(self) -> typing.Optional["IOnFailureVariable"]:
        '''(Optional) Determines the action to take if stack creation failed.

        :default: - Rollback on failure
        '''
        result = self._values.get("on_stack_failure")
        return typing.cast(typing.Optional["IOnFailureVariable"], result)

    @builtins.property
    def parameters(self) -> typing.Optional[IMapListVariable]:
        '''(Optional) A list of Parameter structures that specify input parameters for the stack.

        For more information, see the `Parameter <https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html>`_ data type.
        '''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[IMapListVariable], result)

    @builtins.property
    def resource_types(self) -> typing.Optional[IStringListVariable]:
        '''(Optional) The template resource types that you have permissions to work with for this create stack action.

        For example: AWS::EC2::Instance, AWS::EC2::*, or Custom::MyCustomInstance.
        '''
        result = self._values.get("resource_types")
        return typing.cast(typing.Optional[IStringListVariable], result)

    @builtins.property
    def role_arn(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that CloudFormation assumes to create the stack.

        CloudFormation uses the role's credentials to make calls on your behalf.
        CloudFormation always uses this role for all future operations on the stack.
        As long as users have permission to operate on the stack, CloudFormation uses this role even
        if the users don't have permission to pass it. Ensure that the role grants the least amount of privileges.

        If you don't specify a value, CloudFormation uses the role that was previously associated with the stack.
        If no role is available, CloudFormation uses a temporary session that is generated from your user credentials.
        '''
        result = self._values.get("role_arn")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def stack_policy(self) -> typing.Optional[BodyOrUrlProp]:
        '''(Optional) Stack policy body or URL.

        For more information, see `Prevent Updates to Stack Resources <https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/protect-stack-resources.html>`_.
        '''
        result = self._values.get("stack_policy")
        return typing.cast(typing.Optional[BodyOrUrlProp], result)

    @builtins.property
    def tags(self) -> typing.Optional[IMapListVariable]:
        '''(Optional) Key-value pairs to associate with this stack.

        CloudFormation also propagates these tags to the resources created in the stack. You can specify a maximum number of 10 tags.
        '''
        result = self._values.get("tags")
        return typing.cast(typing.Optional[IMapListVariable], result)

    @builtins.property
    def timeout_in_minutes(self) -> typing.Optional[INumberVariable]:
        '''(Optional) The amount of time that can pass before the stack status becomes CREATE_FAILED.

        If DisableRollback isn't set or is set to false, the stack will be rolled back.
        '''
        result = self._values.get("timeout_in_minutes")
        return typing.cast(typing.Optional[INumberVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CreateStackStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class CreateTagsStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.CreateTagsStep",
):
    '''AutomationStep implemenation for aws:createTags https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-createtag.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        resource_ids: IStringListVariable,
        tags: IMapListVariable,
        resource_type: typing.Optional["IResourceTypeVariable"] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param resource_ids: The IDs of the resource(s) to be tagged. If resource type isn't “EC2”, this field can contain only a single item.
        :param tags: The tags to associate with the resource(s).
        :param resource_type: (Optional) The type of resource(s) to be tagged. Valid values: EC2 | ManagedInstance | MaintenanceWindow | Parameter Default: EC2
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = CreateTagsStepProps(
            resource_ids=resource_ids,
            tags=tags,
            resource_type=resource_type,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''This step has no outputs.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="resourceIds")
    def resource_ids(self) -> IStringListVariable:
        return typing.cast(IStringListVariable, jsii.get(self, "resourceIds"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> IMapListVariable:
        return typing.cast(IMapListVariable, jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="resourceType")
    def resource_type(self) -> typing.Optional["IResourceTypeVariable"]:
        return typing.cast(typing.Optional["IResourceTypeVariable"], jsii.get(self, "resourceType"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.CreateTagsStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "resource_ids": "resourceIds",
        "tags": "tags",
        "resource_type": "resourceType",
    },
)
class CreateTagsStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        resource_ids: IStringListVariable,
        tags: IMapListVariable,
        resource_type: typing.Optional["IResourceTypeVariable"] = None,
    ) -> None:
        '''Properties for CreateTagStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param resource_ids: The IDs of the resource(s) to be tagged. If resource type isn't “EC2”, this field can contain only a single item.
        :param tags: The tags to associate with the resource(s).
        :param resource_type: (Optional) The type of resource(s) to be tagged. Valid values: EC2 | ManagedInstance | MaintenanceWindow | Parameter Default: EC2
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "resource_ids": resource_ids,
            "tags": tags,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if resource_type is not None:
            self._values["resource_type"] = resource_type

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def resource_ids(self) -> IStringListVariable:
        '''The IDs of the resource(s) to be tagged.

        If resource type isn't “EC2”, this field can contain only a single item.
        '''
        result = self._values.get("resource_ids")
        assert result is not None, "Required property 'resource_ids' is missing"
        return typing.cast(IStringListVariable, result)

    @builtins.property
    def tags(self) -> IMapListVariable:
        '''The tags to associate with the resource(s).'''
        result = self._values.get("tags")
        assert result is not None, "Required property 'tags' is missing"
        return typing.cast(IMapListVariable, result)

    @builtins.property
    def resource_type(self) -> typing.Optional["IResourceTypeVariable"]:
        '''(Optional) The type of resource(s) to be tagged.

        Valid values: EC2 | ManagedInstance | MaintenanceWindow | Parameter

        :default: EC2
        '''
        result = self._values.get("resource_type")
        return typing.cast(typing.Optional["IResourceTypeVariable"], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "CreateTagsStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DeleteImageStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DeleteImageStep",
):
    '''AutomationStep implementation for aws:deleteImage https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-delete.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        image_id: IStringVariable,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param image_id: The ID of the image to be deleted.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = DeleteImageStepProps(
            image_id=image_id,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''This step has no outputs.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="imageId")
    def image_id(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "imageId"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.DeleteImageStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "image_id": "imageId",
    },
)
class DeleteImageStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        image_id: IStringVariable,
    ) -> None:
        '''Properties for DeleteImageStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param image_id: The ID of the image to be deleted.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "image_id": image_id,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def image_id(self) -> IStringVariable:
        '''The ID of the image to be deleted.'''
        result = self._values.get("image_id")
        assert result is not None, "Required property 'image_id' is missing"
        return typing.cast(IStringVariable, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DeleteImageStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DeleteStackStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DeleteStackStep",
):
    '''AutomationStep implemenation for aws:deleteStack https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-deletestack.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        stack_name_variable: IStringVariable,
        role_arn: typing.Optional[IStringVariable] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param stack_name_variable: (Optional) Variable that is fed into this step declaring which stack to delete. Default: - StackName is the default value.
        :param role_arn: (Optional) The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that CloudFormation assumes to create the stack. CloudFormation uses the role's credentials to make calls on your behalf. CloudFormation always uses this role for all future operations on the stack. As long as users have permission to operate on the stack, CloudFormation uses this role even if the users don't have permission to pass it. Ensure that the role grants the least amount of privileges. If you don't specify a value, CloudFormation uses the role that was previously associated with the stack. If no role is available, CloudFormation uses a temporary session that is generated from your user credentials.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = DeleteStackStepProps(
            stack_name_variable=stack_name_variable,
            role_arn=role_arn,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''The input required for this step is the stackNameVariable.

        :return: the stackNameVariable as declared in the constructor
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''No outputs emitted from Delete Stack step.

        :return: []
        '''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="stackNameVariable")
    def stack_name_variable(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "stackNameVariable"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="roleArn")
    def role_arn(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "roleArn"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.DeleteStackStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "stack_name_variable": "stackNameVariable",
        "role_arn": "roleArn",
    },
)
class DeleteStackStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        stack_name_variable: IStringVariable,
        role_arn: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties for DeleteStackStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param stack_name_variable: (Optional) Variable that is fed into this step declaring which stack to delete. Default: - StackName is the default value.
        :param role_arn: (Optional) The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that CloudFormation assumes to create the stack. CloudFormation uses the role's credentials to make calls on your behalf. CloudFormation always uses this role for all future operations on the stack. As long as users have permission to operate on the stack, CloudFormation uses this role even if the users don't have permission to pass it. Ensure that the role grants the least amount of privileges. If you don't specify a value, CloudFormation uses the role that was previously associated with the stack. If no role is available, CloudFormation uses a temporary session that is generated from your user credentials.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "stack_name_variable": stack_name_variable,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if role_arn is not None:
            self._values["role_arn"] = role_arn

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def stack_name_variable(self) -> IStringVariable:
        '''(Optional) Variable that is fed into this step declaring which stack to delete.

        :default: - StackName is the default value.
        '''
        result = self._values.get("stack_name_variable")
        assert result is not None, "Required property 'stack_name_variable' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def role_arn(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The Amazon Resource Name (ARN) of an AWS Identity and Access Management (IAM) role that CloudFormation assumes to create the stack.

        CloudFormation uses the role's credentials to make calls on your behalf.
        CloudFormation always uses this role for all future operations on the stack.
        As long as users have permission to operate on the stack, CloudFormation uses this role even
        if the users don't have permission to pass it. Ensure that the role grants the least amount of privileges.

        If you don't specify a value, CloudFormation uses the role that was previously associated with the stack.
        If no role is available, CloudFormation uses a temporary session that is generated from your user credentials.
        '''
        result = self._values.get("role_arn")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DeleteStackStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IStringVariable)
class DictFormat(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DictFormat",
):
    def __init__(self, format: typing.Mapping[builtins.str, typing.Any]) -> None:
        '''
        :param format: -
        '''
        jsii.create(self.__class__, self, [format])

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{Variable}} or the value.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [inputs]))

    @jsii.member(jsii_name="resolveToDict")
    def resolve_to_dict(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''
        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "resolveToDict", [inputs]))

    @jsii.member(jsii_name="resolveToString")
    def resolve_to_string(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.str:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "resolveToString", [inputs]))

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> typing.Any:
        '''JSON.stringify(variable) will implicitly invoke this variable.'''
        return typing.cast(typing.Any, jsii.invoke(self, "toJSON", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="format")
    def format(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.get(self, "format"))


@jsii.implements(IEnvironment)
class DockerEnvironment(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DockerEnvironment",
):
    '''Provides a Docker client against which customers can execute their commands.

    This utility will not download docker images, rather will create containers from images provided.
    You can use this rather than running your run commands against a real EC2 machine.
    '''

    @jsii.member(jsii_name="fromContainer") # type: ignore[misc]
    @builtins.classmethod
    def from_container(cls, container_id: builtins.str) -> "DockerEnvironment":
        '''Use an existing container against which to run commands using the run function.

        :param container_id: -
        '''
        return typing.cast("DockerEnvironment", jsii.sinvoke(cls, "fromContainer", [container_id]))

    @jsii.member(jsii_name="fromImage") # type: ignore[misc]
    @builtins.classmethod
    def from_image(cls, image: builtins.str) -> "DockerEnvironment":
        '''Create a container from the provided image.

        The container created will be used by this instance to run commands using the run function.

        :param image: -
        '''
        return typing.cast("DockerEnvironment", jsii.sinvoke(cls, "fromImage", [image]))

    @jsii.member(jsii_name="removeContainer")
    def remove_container(self) -> None:
        '''Force removes the container associated with this instance.'''
        return typing.cast(None, jsii.invoke(self, "removeContainer", []))

    @jsii.member(jsii_name="run")
    def run(self, command: builtins.str) -> builtins.str:
        '''Runs commands against the docker specified during construction.

        This function runs synchronously.

        :param command: -

        :return: the response from the docker as a string.
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "run", [command]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="containerId")
    def container_id(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "containerId"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.DocumentResult",
    jsii_struct_bases=[SimulationResult],
    name_mapping={
        "executed_steps": "executedSteps",
        "response_code": "responseCode",
        "outputs": "outputs",
        "stack_trace": "stackTrace",
        "document_outputs": "documentOutputs",
    },
)
class DocumentResult(SimulationResult):
    def __init__(
        self,
        *,
        executed_steps: typing.Sequence[builtins.str],
        response_code: ResponseCode,
        outputs: typing.Optional[typing.Mapping[builtins.str, typing.Any]] = None,
        stack_trace: typing.Optional[builtins.str] = None,
        document_outputs: typing.Sequence[builtins.str],
    ) -> None:
        '''
        :param executed_steps: All the steps that were executed in this Simulation.
        :param response_code: 
        :param outputs: May be empty if responseCode is FAILED/CANCELLED. There are no outputs provided for Command steps or documents.
        :param stack_trace: undefined if responseCode is SUCCESS.
        :param document_outputs: 
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "executed_steps": executed_steps,
            "response_code": response_code,
            "document_outputs": document_outputs,
        }
        if outputs is not None:
            self._values["outputs"] = outputs
        if stack_trace is not None:
            self._values["stack_trace"] = stack_trace

    @builtins.property
    def executed_steps(self) -> typing.List[builtins.str]:
        '''All the steps that were executed in this Simulation.'''
        result = self._values.get("executed_steps")
        assert result is not None, "Required property 'executed_steps' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def response_code(self) -> ResponseCode:
        result = self._values.get("response_code")
        assert result is not None, "Required property 'response_code' is missing"
        return typing.cast(ResponseCode, result)

    @builtins.property
    def outputs(self) -> typing.Optional[typing.Mapping[builtins.str, typing.Any]]:
        '''May be empty if responseCode is FAILED/CANCELLED.

        There are no outputs provided for Command steps or documents.
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Optional[typing.Mapping[builtins.str, typing.Any]], result)

    @builtins.property
    def stack_trace(self) -> typing.Optional[builtins.str]:
        '''undefined if responseCode is SUCCESS.'''
        result = self._values.get("stack_trace")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def document_outputs(self) -> typing.List[builtins.str]:
        result = self._values.get("document_outputs")
        assert result is not None, "Required property 'document_outputs' is missing"
        return typing.cast(typing.List[builtins.str], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DocumentResult(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DomainJoinStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DomainJoinStep",
):
    '''CommandStep implemenation for aws:softwareInventory https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-domainJoin.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        directory_id: IStringVariable,
        directory_name: IStringVariable,
        dns_ip_addresses: IStringListVariable,
        directory_ou: typing.Optional[IStringVariable] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param directory_id: The ID of the directory.
        :param directory_name: The name of the domain.
        :param dns_ip_addresses: The IP addresses of the DNS servers.
        :param directory_ou: (Optional) The organizational unit (OU).
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = DomainJoinStepProps(
            directory_id=directory_id,
            directory_name=directory_name,
            dns_ip_addresses=dns_ip_addresses,
            directory_ou=directory_ou,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''All Inputs for this command are optional.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="directoryId")
    def directory_id(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "directoryId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="directoryName")
    def directory_name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "directoryName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="directoryOU")
    def directory_ou(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "directoryOU"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="dnsIpAddresses")
    def dns_ip_addresses(self) -> typing.Optional[IStringListVariable]:
        return typing.cast(typing.Optional[IStringListVariable], jsii.get(self, "dnsIpAddresses"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.DomainJoinStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "directory_id": "directoryId",
        "directory_name": "directoryName",
        "dns_ip_addresses": "dnsIpAddresses",
        "directory_ou": "directoryOU",
    },
)
class DomainJoinStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        directory_id: IStringVariable,
        directory_name: IStringVariable,
        dns_ip_addresses: IStringListVariable,
        directory_ou: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties of DomainJoin step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param directory_id: The ID of the directory.
        :param directory_name: The name of the domain.
        :param dns_ip_addresses: The IP addresses of the DNS servers.
        :param directory_ou: (Optional) The organizational unit (OU).
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "directory_id": directory_id,
            "directory_name": directory_name,
            "dns_ip_addresses": dns_ip_addresses,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition
        if directory_ou is not None:
            self._values["directory_ou"] = directory_ou

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def directory_id(self) -> IStringVariable:
        '''The ID of the directory.

        Example::

            "d-1234567890"
        '''
        result = self._values.get("directory_id")
        assert result is not None, "Required property 'directory_id' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def directory_name(self) -> IStringVariable:
        '''The name of the domain.

        Example::

            "example.com"
        '''
        result = self._values.get("directory_name")
        assert result is not None, "Required property 'directory_name' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def dns_ip_addresses(self) -> IStringListVariable:
        '''The IP addresses of the DNS servers.

        Example::

            ["198.51.100.1","198.51.100.2"]
        '''
        result = self._values.get("dns_ip_addresses")
        assert result is not None, "Required property 'dns_ip_addresses' is missing"
        return typing.cast(IStringListVariable, result)

    @builtins.property
    def directory_ou(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The organizational unit (OU).

        Example::

            "OU=test,DC=example,DC=com"
        '''
        result = self._values.get("directory_ou")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DomainJoinStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class DownloadContentStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DownloadContentStep",
):
    '''AutomationStep implemenation for aws:downloadContent https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-downloadContent.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        downloadable_content: IDownloadableContent,
        destination_path: typing.Optional[IStringVariable] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param downloadable_content: The information required to retrieve the content from the required source. This is a dictionary whose format changes based on the sourceType See the aws documentation for more info https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-downloadContent
        :param destination_path: (Optional) An optional local path on the instance where you want to download the file. If you don't specify a path, the content is downloaded to a path relative to your command ID.
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = DownloadContentStepProps(
            downloadable_content=downloadable_content,
            destination_path=destination_path,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Inputs required for this command includes both the sourceType and sourceInfo variables and  destinationPath if provided.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="downloadableContent")
    def downloadable_content(self) -> IDownloadableContent:
        return typing.cast(IDownloadableContent, jsii.get(self, "downloadableContent"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="destinationPath")
    def destination_path(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "destinationPath"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.DownloadContentStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "downloadable_content": "downloadableContent",
        "destination_path": "destinationPath",
    },
)
class DownloadContentStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        downloadable_content: IDownloadableContent,
        destination_path: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties.json for ps Module step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param downloadable_content: The information required to retrieve the content from the required source. This is a dictionary whose format changes based on the sourceType See the aws documentation for more info https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-downloadContent
        :param destination_path: (Optional) An optional local path on the instance where you want to download the file. If you don't specify a path, the content is downloaded to a path relative to your command ID.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "downloadable_content": downloadable_content,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition
        if destination_path is not None:
            self._values["destination_path"] = destination_path

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def downloadable_content(self) -> IDownloadableContent:
        '''The information required to retrieve the content from the required source.

        This is a dictionary whose format changes based on the sourceType
        See the aws documentation for more info
        https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-downloadContent
        '''
        result = self._values.get("downloadable_content")
        assert result is not None, "Required property 'downloadable_content' is missing"
        return typing.cast(IDownloadableContent, result)

    @builtins.property
    def destination_path(self) -> typing.Optional[IStringVariable]:
        '''(Optional) An optional local path on the instance where you want to download the file.

        If you don't specify a path, the content is downloaded to a path relative to your command ID.
        '''
        result = self._values.get("destination_path")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "DownloadContentStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ExecuteAutomationStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteAutomationStep",
):
    '''AutomationStep implementation for aws:executeAutomation https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeAutomation.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        document_name: IStringVariable,
        document_version: typing.Optional[IStringVariable] = None,
        max_concurrency: typing.Optional[IStringVariable] = None,
        max_errors: typing.Optional[IStringVariable] = None,
        runtime_parameters: typing.Optional[IStringMapVariable] = None,
        tags: typing.Optional[IMapListVariable] = None,
        target_locations: typing.Optional[IMapListVariable] = None,
        target_maps: typing.Optional[IMapListVariable] = None,
        target_parameter_name: typing.Optional[IStringVariable] = None,
        targets: typing.Optional[IMapListVariable] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param document_name: 
        :param document_version: 
        :param max_concurrency: 
        :param max_errors: 
        :param runtime_parameters: 
        :param tags: 
        :param target_locations: 
        :param target_maps: 
        :param target_parameter_name: 
        :param targets: 
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = ExecuteAutomationStepProps(
            document_name=document_name,
            document_version=document_version,
            max_concurrency=max_concurrency,
            max_errors=max_errors,
            runtime_parameters=runtime_parameters,
            tags=tags,
            target_locations=target_locations,
            target_maps=target_maps,
            target_parameter_name=target_parameter_name,
            targets=targets,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="documentName")
    def document_name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "documentName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="documentVersion")
    def document_version(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "documentVersion"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="maxConcurrency")
    def max_concurrency(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "maxConcurrency"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="maxErrors")
    def max_errors(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "maxErrors"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="runtimeParameters")
    def runtime_parameters(self) -> typing.Optional[IStringMapVariable]:
        return typing.cast(typing.Optional[IStringMapVariable], jsii.get(self, "runtimeParameters"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tags")
    def tags(self) -> typing.Optional[IMapListVariable]:
        return typing.cast(typing.Optional[IMapListVariable], jsii.get(self, "tags"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="targetLocations")
    def target_locations(self) -> typing.Optional[IMapListVariable]:
        return typing.cast(typing.Optional[IMapListVariable], jsii.get(self, "targetLocations"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="targetMaps")
    def target_maps(self) -> typing.Optional[IMapListVariable]:
        return typing.cast(typing.Optional[IMapListVariable], jsii.get(self, "targetMaps"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="targetParameterName")
    def target_parameter_name(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "targetParameterName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="targets")
    def targets(self) -> typing.Optional[IMapListVariable]:
        return typing.cast(typing.Optional[IMapListVariable], jsii.get(self, "targets"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteAutomationStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "document_name": "documentName",
        "document_version": "documentVersion",
        "max_concurrency": "maxConcurrency",
        "max_errors": "maxErrors",
        "runtime_parameters": "runtimeParameters",
        "tags": "tags",
        "target_locations": "targetLocations",
        "target_maps": "targetMaps",
        "target_parameter_name": "targetParameterName",
        "targets": "targets",
    },
)
class ExecuteAutomationStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        document_name: IStringVariable,
        document_version: typing.Optional[IStringVariable] = None,
        max_concurrency: typing.Optional[IStringVariable] = None,
        max_errors: typing.Optional[IStringVariable] = None,
        runtime_parameters: typing.Optional[IStringMapVariable] = None,
        tags: typing.Optional[IMapListVariable] = None,
        target_locations: typing.Optional[IMapListVariable] = None,
        target_maps: typing.Optional[IMapListVariable] = None,
        target_parameter_name: typing.Optional[IStringVariable] = None,
        targets: typing.Optional[IMapListVariable] = None,
    ) -> None:
        '''
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param document_name: 
        :param document_version: 
        :param max_concurrency: 
        :param max_errors: 
        :param runtime_parameters: 
        :param tags: 
        :param target_locations: 
        :param target_maps: 
        :param target_parameter_name: 
        :param targets: 
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "document_name": document_name,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if document_version is not None:
            self._values["document_version"] = document_version
        if max_concurrency is not None:
            self._values["max_concurrency"] = max_concurrency
        if max_errors is not None:
            self._values["max_errors"] = max_errors
        if runtime_parameters is not None:
            self._values["runtime_parameters"] = runtime_parameters
        if tags is not None:
            self._values["tags"] = tags
        if target_locations is not None:
            self._values["target_locations"] = target_locations
        if target_maps is not None:
            self._values["target_maps"] = target_maps
        if target_parameter_name is not None:
            self._values["target_parameter_name"] = target_parameter_name
        if targets is not None:
            self._values["targets"] = targets

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def document_name(self) -> IStringVariable:
        result = self._values.get("document_name")
        assert result is not None, "Required property 'document_name' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def document_version(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("document_version")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def max_concurrency(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("max_concurrency")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def max_errors(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("max_errors")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def runtime_parameters(self) -> typing.Optional[IStringMapVariable]:
        result = self._values.get("runtime_parameters")
        return typing.cast(typing.Optional[IStringMapVariable], result)

    @builtins.property
    def tags(self) -> typing.Optional[IMapListVariable]:
        result = self._values.get("tags")
        return typing.cast(typing.Optional[IMapListVariable], result)

    @builtins.property
    def target_locations(self) -> typing.Optional[IMapListVariable]:
        result = self._values.get("target_locations")
        return typing.cast(typing.Optional[IMapListVariable], result)

    @builtins.property
    def target_maps(self) -> typing.Optional[IMapListVariable]:
        result = self._values.get("target_maps")
        return typing.cast(typing.Optional[IMapListVariable], result)

    @builtins.property
    def target_parameter_name(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("target_parameter_name")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def targets(self) -> typing.Optional[IMapListVariable]:
        result = self._values.get("targets")
        return typing.cast(typing.Optional[IMapListVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ExecuteAutomationStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ExecuteScriptStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteScriptStep",
):
    '''AutomationStep implementation for aws:executeScript https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeScript.html.'''

    def __init__(
        self,
        stage: constructs.Construct,
        id: builtins.str,
        *,
        inputs: typing.Sequence[builtins.str],
        language: ScriptLanguage,
        full_path_to_code: typing.Optional[builtins.str] = None,
        handler_name: typing.Optional[builtins.str] = None,
        inline_code: typing.Optional[builtins.str] = None,
        outputs: typing.Optional[typing.Sequence[Output]] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param stage: -
        :param id: -
        :param inputs: (Optional) Inputs that the function needs in order to execute. Default: []
        :param language: (Required) Language used to execute the script.
        :param full_path_to_code: Full path to the code to execute. File is parsed to produce yaml/json. Simulation will execute this file using the language specified. Either this OR inclineCode must be provided.
        :param handler_name: (Optional) Function name in fullPathToCode file to use as entry point for script handler. Default: script_handler
        :param inline_code: Inline code to be executed. String will be used to produce function in yaml/json. Simulation will execute the function in this string using the language specified. Either this OR fullPathToCode must be provided.
        :param outputs: (Optional) Outputs that the function is expected to return. Be sure to prefix the selector for these outputs with "$.Payload." for executeScript step outputs. Default: []
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = ExecuteScriptStepProps(
            inputs=inputs,
            language=language,
            full_path_to_code=full_path_to_code,
            handler_name=handler_name,
            inline_code=inline_code,
            outputs=outputs,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [stage, id, props])

    @jsii.member(jsii_name="getLanguage") # type: ignore[misc]
    @builtins.classmethod
    def get_language(cls, runtime: builtins.str) -> ScriptLanguage:
        '''
        :param runtime: -
        '''
        return typing.cast(ScriptLanguage, jsii.sinvoke(cls, "getLanguage", [runtime]))

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="fullPathToCode")
    def full_path_to_code(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "fullPathToCode"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="handlerName")
    def handler_name(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "handlerName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="inputs")
    def inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.get(self, "inputs"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="language")
    def language(self) -> ScriptLanguage:
        return typing.cast(ScriptLanguage, jsii.get(self, "language"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="outputs")
    def outputs(self) -> typing.List[Output]:
        return typing.cast(typing.List[Output], jsii.get(self, "outputs"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteScriptStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "inputs": "inputs",
        "language": "language",
        "full_path_to_code": "fullPathToCode",
        "handler_name": "handlerName",
        "inline_code": "inlineCode",
        "outputs": "outputs",
    },
)
class ExecuteScriptStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        inputs: typing.Sequence[builtins.str],
        language: ScriptLanguage,
        full_path_to_code: typing.Optional[builtins.str] = None,
        handler_name: typing.Optional[builtins.str] = None,
        inline_code: typing.Optional[builtins.str] = None,
        outputs: typing.Optional[typing.Sequence[Output]] = None,
    ) -> None:
        '''Properties for ExecuteScriptStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param inputs: (Optional) Inputs that the function needs in order to execute. Default: []
        :param language: (Required) Language used to execute the script.
        :param full_path_to_code: Full path to the code to execute. File is parsed to produce yaml/json. Simulation will execute this file using the language specified. Either this OR inclineCode must be provided.
        :param handler_name: (Optional) Function name in fullPathToCode file to use as entry point for script handler. Default: script_handler
        :param inline_code: Inline code to be executed. String will be used to produce function in yaml/json. Simulation will execute the function in this string using the language specified. Either this OR fullPathToCode must be provided.
        :param outputs: (Optional) Outputs that the function is expected to return. Be sure to prefix the selector for these outputs with "$.Payload." for executeScript step outputs. Default: []
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "inputs": inputs,
            "language": language,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if full_path_to_code is not None:
            self._values["full_path_to_code"] = full_path_to_code
        if handler_name is not None:
            self._values["handler_name"] = handler_name
        if inline_code is not None:
            self._values["inline_code"] = inline_code
        if outputs is not None:
            self._values["outputs"] = outputs

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def inputs(self) -> typing.List[builtins.str]:
        '''(Optional) Inputs that the function needs in order to execute.

        :default: []
        '''
        result = self._values.get("inputs")
        assert result is not None, "Required property 'inputs' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def language(self) -> ScriptLanguage:
        '''(Required) Language used to execute the script.'''
        result = self._values.get("language")
        assert result is not None, "Required property 'language' is missing"
        return typing.cast(ScriptLanguage, result)

    @builtins.property
    def full_path_to_code(self) -> typing.Optional[builtins.str]:
        '''Full path to the code to execute.

        File is parsed to produce yaml/json.
        Simulation will execute this file using the language specified.
        Either this OR inclineCode must be provided.
        '''
        result = self._values.get("full_path_to_code")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def handler_name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Function name in fullPathToCode file to use as entry point for script handler.

        :default: script_handler
        '''
        result = self._values.get("handler_name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def inline_code(self) -> typing.Optional[builtins.str]:
        '''Inline code to be executed.

        String will be used to produce function in yaml/json.
        Simulation will execute the function in this string using the language specified.
        Either this OR fullPathToCode must be provided.
        '''
        result = self._values.get("inline_code")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def outputs(self) -> typing.Optional[typing.List[Output]]:
        '''(Optional) Outputs that the function is expected to return.

        Be sure to prefix the selector for these outputs with "$.Payload." for executeScript step outputs.

        :default: []
        '''
        result = self._values.get("outputs")
        return typing.cast(typing.Optional[typing.List[Output]], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ExecuteScriptStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ExecuteStateMachineStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteStateMachineStep",
):
    '''AutomationStep implementation of `aws:executeStateMachine <https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-executeStateMachine.html>`_.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        state_machine_arn: IStringVariable,
        execution_name: typing.Optional[IStringVariable] = None,
        input: typing.Optional[IStringVariable] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param state_machine_arn: The Amazon Resource Name (ARN) of the Step Functions state machine.
        :param execution_name: (Optional) The name of the execution.
        :param input: (Optional) A string that contains the JSON input data for the execution.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = ExecuteStateMachineStepProps(
            state_machine_arn=state_machine_arn,
            execution_name=execution_name,
            input=input,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''No outputs emitted from Delete Stack step.

        :return: []
        '''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="stateMachineArn")
    def state_machine_arn(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "stateMachineArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="executionName")
    def execution_name(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "executionName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="input")
    def input(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "input"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ExecuteStateMachineStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "state_machine_arn": "stateMachineArn",
        "execution_name": "executionName",
        "input": "input",
    },
)
class ExecuteStateMachineStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        state_machine_arn: IStringVariable,
        execution_name: typing.Optional[IStringVariable] = None,
        input: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Props for ExecuteStateMachine step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param state_machine_arn: The Amazon Resource Name (ARN) of the Step Functions state machine.
        :param execution_name: (Optional) The name of the execution.
        :param input: (Optional) A string that contains the JSON input data for the execution.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "state_machine_arn": state_machine_arn,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if execution_name is not None:
            self._values["execution_name"] = execution_name
        if input is not None:
            self._values["input"] = input

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def state_machine_arn(self) -> IStringVariable:
        '''The Amazon Resource Name (ARN) of the Step Functions state machine.'''
        result = self._values.get("state_machine_arn")
        assert result is not None, "Required property 'state_machine_arn' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def execution_name(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The name of the execution.'''
        result = self._values.get("execution_name")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def input(self) -> typing.Optional[IStringVariable]:
        '''(Optional) A string that contains the JSON input data for the execution.'''
        result = self._values.get("input")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ExecuteStateMachineStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IGenericVariable)
class GenericVariable(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.GenericVariable",
):
    '''Abstraction of SSM variables.

    Variables are printed as using this syntax: {{myVariable}}
    To resolve a variable, you must supply the available inputs and the variable will resolve the value.
    '''

    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="assertType") # type: ignore[misc]
    @abc.abstractmethod
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        ...

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{Variable}} or the value.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [inputs]))

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> typing.Any:
        '''JSON.stringify(variable) will implicitly invoke this variable.'''
        return typing.cast(typing.Any, jsii.invoke(self, "toJSON", []))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''Returns a string representation of an object.'''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="reference")
    def reference(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "reference"))


class _GenericVariableProxy(GenericVariable):
    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, GenericVariable).__jsii_proxy_class__ = lambda : _GenericVariableProxy


@jsii.implements(IDownloadableContent)
class GitContent(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.GitContent",
):
    def __init__(
        self,
        *,
        repository: IStringVariable,
        branch: typing.Optional[builtins.str] = None,
        commit_id: typing.Optional[builtins.str] = None,
        password: typing.Optional[SecureVariable] = None,
        private_ssh_key: typing.Optional[SecureVariable] = None,
        skip_host_key_checking: typing.Optional["IBooleanVariable"] = None,
        user_name: typing.Optional[SecureVariable] = None,
    ) -> None:
        '''
        :param repository: The Git repository URL to the file or directory you want to download.
        :param branch: The default is master. branch parameter is required only if your SSM document is stored in a branch other than master. Supply either commitId or branch (or neither).
        :param commit_id: The default is head. To use the version of your SSM document in a commit other than the latest, specify the full commit ID. For example: "bbc1ddb94...b76d3bEXAMPLE". Supply either commitId or branch (or neither).
        :param password: The password to use when connecting to the repository you specify using HTTP.
        :param private_ssh_key: The SSH key to use when connecting to the repository you specify.
        :param skip_host_key_checking: Determines the value of the StrictHostKeyChecking option when connecting to the repository you specify. The default value is false.
        :param user_name: The username to use when connecting to the repository you specify using HTTP.
        '''
        props = GitContentProps(
            repository=repository,
            branch=branch,
            commit_id=commit_id,
            password=password,
            private_ssh_key=private_ssh_key,
            skip_host_key_checking=skip_host_key_checking,
            user_name=user_name,
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="formatSourceInfo")
    def format_source_info(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatSourceInfo", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="repository")
    def repository(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "repository"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "sourceType"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="branch")
    def branch(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "branch"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="commitId")
    def commit_id(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "commitId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="password")
    def password(self) -> typing.Optional[SecureVariable]:
        return typing.cast(typing.Optional[SecureVariable], jsii.get(self, "password"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="privateSshKey")
    def private_ssh_key(self) -> typing.Optional[SecureVariable]:
        return typing.cast(typing.Optional[SecureVariable], jsii.get(self, "privateSshKey"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="skipHostKeyChecking")
    def skip_host_key_checking(self) -> typing.Optional["IBooleanVariable"]:
        return typing.cast(typing.Optional["IBooleanVariable"], jsii.get(self, "skipHostKeyChecking"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="userName")
    def user_name(self) -> typing.Optional[SecureVariable]:
        return typing.cast(typing.Optional[SecureVariable], jsii.get(self, "userName"))


@jsii.implements(IDownloadableContent)
class GitHubContent(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.GitHubContent",
):
    def __init__(
        self,
        *,
        owner: IStringVariable,
        path: IStringVariable,
        repository: IStringVariable,
        token_info: SecureVariable,
        branch: typing.Optional[builtins.str] = None,
        commit_id: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param owner: The repository owner.
        :param path: The path to the file or directory you want to download.
        :param repository: The name of the repository.
        :param token_info: The Systems Manager parameter (a SecureString parameter) where you store your GitHub access token information.
        :param branch: The default is master. branch parameter is required only if your SSM document is stored in a branch other than master. Supply either commitId or branch (or neither).
        :param commit_id: The default is head. To use the version of your SSM document in a commit other than the latest, specify the full commit ID. For example: "bbc1ddb94...b76d3bEXAMPLE". Supply either commitId or branch (or neither).
        '''
        props = GitHubContentProps(
            owner=owner,
            path=path,
            repository=repository,
            token_info=token_info,
            branch=branch,
            commit_id=commit_id,
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="formatSourceInfo")
    def format_source_info(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatSourceInfo", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="owner")
    def owner(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "owner"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="path")
    def path(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "path"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="repository")
    def repository(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "repository"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "sourceType"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tokenInfo")
    def token_info(self) -> SecureVariable:
        return typing.cast(SecureVariable, jsii.get(self, "tokenInfo"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="branch")
    def branch(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "branch"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="commitId")
    def commit_id(self) -> typing.Optional[builtins.str]:
        return typing.cast(typing.Optional[builtins.str], jsii.get(self, "commitId"))


class HardCodedSecureVariable(
    SecureVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HardCodedSecureVariable",
):
    '''A hard-coded string variable.

    Used when not dependent on step inputs.
    '''

    def __init__(self, val: builtins.str) -> None:
        '''
        :param val: -
        '''
        jsii.create(self.__class__, self, [val])

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{Variable}} or the value.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, _inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param _inputs: -
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [_inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="val")
    def val(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "val"))


@jsii.implements(IGenericVariable)
class HardCodedValueBase(
    metaclass=jsii.JSIIAbstractClass,
    jsii_type="@cdklabs/cdk-ssm-documents.HardCodedValueBase",
):
    def __init__(self, val: typing.Any) -> None:
        '''
        :param val: -
        '''
        jsii.create(self.__class__, self, [val])

    @jsii.member(jsii_name="assertType") # type: ignore[misc]
    @abc.abstractmethod
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        ...

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{Variable}} or the value.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, _inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param _inputs: -
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [_inputs]))

    @jsii.member(jsii_name="toJSON")
    def to_json(self) -> typing.Any:
        '''JSON.stringify(variable) will implicitly invoke this variable.'''
        return typing.cast(typing.Any, jsii.invoke(self, "toJSON", []))

    @jsii.member(jsii_name="toString")
    def to_string(self) -> builtins.str:
        '''Returns a string representation of an object.'''
        return typing.cast(builtins.str, jsii.invoke(self, "toString", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="val")
    def val(self) -> typing.Any:
        return typing.cast(typing.Any, jsii.get(self, "val"))


class _HardCodedValueBaseProxy(HardCodedValueBase):
    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the abstract class
typing.cast(typing.Any, HardCodedValueBase).__jsii_proxy_class__ = lambda : _HardCodedValueBaseProxy


@jsii.implements(IDownloadableContent)
class HttpContent(
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HttpContent",
):
    def __init__(
        self,
        *,
        url: IStringVariable,
        allow_insecure_download: typing.Optional["IBooleanVariable"] = None,
        auth_method: typing.Optional[AuthMethod] = None,
    ) -> None:
        '''
        :param url: The URL to the file or directory you want to download.
        :param allow_insecure_download: Determines whether a download can be performed over a connection that isn't encrypted with Secure Socket Layer (SSL) or Transport Layer Security (TLS). The default value is false. We don't recommend performing downloads without encryption. If you choose to do so, you assume all associated risks. Security is a shared responsibility between AWS and you. This is described as the shared responsibility model. To learn more, see the shared responsibility model.
        :param auth_method: Determines whether a username and password are used for authentication when connecting to the url you specify. If you specify Basic or Digest, you must provide values for the username and password parameters. To use the Digest method, SSM Agent version 3.0.1181.0 or later must be installed on your instance. The Digest method supports MD5 and SHA256 encryption.
        '''
        props = HttpContentProps(
            url=url,
            allow_insecure_download=allow_insecure_download,
            auth_method=auth_method,
        )

        jsii.create(self.__class__, self, [props])

    @jsii.member(jsii_name="formatSourceInfo")
    def format_source_info(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatSourceInfo", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceType")
    def source_type(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "sourceType"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="url")
    def url(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "url"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="allowInsecureDownload")
    def allow_insecure_download(self) -> typing.Optional["IBooleanVariable"]:
        return typing.cast(typing.Optional["IBooleanVariable"], jsii.get(self, "allowInsecureDownload"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="authMethod")
    def auth_method(self) -> typing.Optional[AuthMethod]:
        return typing.cast(typing.Optional[AuthMethod], jsii.get(self, "authMethod"))


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IActionVariable")
class IActionVariable(IStringVariable, typing_extensions.Protocol):
    pass


class _IActionVariableProxy(
    jsii.proxy_for(IStringVariable) # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IActionVariable"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IActionVariable).__jsii_proxy_class__ = lambda : _IActionVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IBooleanVariable")
class IBooleanVariable(IGenericVariable, typing_extensions.Protocol):
    '''A boolean variable.'''

    @jsii.member(jsii_name="resolveToBoolean")
    def resolve_to_boolean(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.bool:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        ...


class _IBooleanVariableProxy(
    jsii.proxy_for(IGenericVariable) # type: ignore[misc]
):
    '''A boolean variable.'''

    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IBooleanVariable"

    @jsii.member(jsii_name="resolveToBoolean")
    def resolve_to_boolean(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.bool:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: are the execution inputs.
        '''
        return typing.cast(builtins.bool, jsii.invoke(self, "resolveToBoolean", [inputs]))

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IBooleanVariable).__jsii_proxy_class__ = lambda : _IBooleanVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IDesiredStateVariable")
class IDesiredStateVariable(IStringVariable, typing_extensions.Protocol):
    pass


class _IDesiredStateVariableProxy(
    jsii.proxy_for(IStringVariable) # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IDesiredStateVariable"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IDesiredStateVariable).__jsii_proxy_class__ = lambda : _IDesiredStateVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IDocumentHashTypeVariable")
class IDocumentHashTypeVariable(IStringVariable, typing_extensions.Protocol):
    pass


class _IDocumentHashTypeVariableProxy(
    jsii.proxy_for(IStringVariable) # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IDocumentHashTypeVariable"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IDocumentHashTypeVariable).__jsii_proxy_class__ = lambda : _IDocumentHashTypeVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IInstallUninstallRepairVariable")
class IInstallUninstallRepairVariable(IStringVariable, typing_extensions.Protocol):
    pass


class _IInstallUninstallRepairVariableProxy(
    jsii.proxy_for(IStringVariable) # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IInstallUninstallRepairVariable"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IInstallUninstallRepairVariable).__jsii_proxy_class__ = lambda : _IInstallUninstallRepairVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IInstallationTypeVariable")
class IInstallationTypeVariable(IStringVariable, typing_extensions.Protocol):
    pass


class _IInstallationTypeVariableProxy(
    jsii.proxy_for(IStringVariable) # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IInstallationTypeVariable"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IInstallationTypeVariable).__jsii_proxy_class__ = lambda : _IInstallationTypeVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IOnFailureVariable")
class IOnFailureVariable(IStringVariable, typing_extensions.Protocol):
    pass


class _IOnFailureVariableProxy(
    jsii.proxy_for(IStringVariable) # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IOnFailureVariable"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IOnFailureVariable).__jsii_proxy_class__ = lambda : _IOnFailureVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IPackageNameVariable")
class IPackageNameVariable(IStringVariable, typing_extensions.Protocol):
    pass


class _IPackageNameVariableProxy(
    jsii.proxy_for(IStringVariable) # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IPackageNameVariable"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IPackageNameVariable).__jsii_proxy_class__ = lambda : _IPackageNameVariableProxy


@jsii.interface(jsii_type="@cdklabs/cdk-ssm-documents.IResourceTypeVariable")
class IResourceTypeVariable(IStringVariable, typing_extensions.Protocol):
    pass


class _IResourceTypeVariableProxy(
    jsii.proxy_for(IStringVariable) # type: ignore[misc]
):
    __jsii_type__: typing.ClassVar[str] = "@cdklabs/cdk-ssm-documents.IResourceTypeVariable"
    pass

# Adding a "__jsii_proxy_class__(): typing.Type" function to the interface
typing.cast(typing.Any, IResourceTypeVariable).__jsii_proxy_class__ = lambda : _IResourceTypeVariableProxy


class InvokeLambdaFunctionStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.InvokeLambdaFunctionStep",
):
    '''AutomationStep implemenation for aws:invokeLambdaFunction https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-lamb.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        function_name: IStringVariable,
        client_context: typing.Optional[IStringVariable] = None,
        invocation_type: typing.Optional[IStringVariable] = None,
        log_type: typing.Optional[IStringVariable] = None,
        payload: typing.Optional[IStringMapVariable] = None,
        qualifier: typing.Optional[IStringVariable] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param function_name: The name of the Lambda function. This function must exist.
        :param client_context: (Optional) The client-specific information.
        :param invocation_type: (Optional) The invocation type. The default value is RequestResponse.
        :param log_type: (Optional) If the default value is Tail, the invocation type must be RequestResponse. Lambda returns the last 4 KB of log data produced by your Lambda function, base64-encoded.
        :param payload: (Optional) The JSON input for your Lambda function.
        :param qualifier: (Optional) The function version or alias name.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = InvokeLambdaFunctionStepProps(
            function_name=function_name,
            client_context=client_context,
            invocation_type=invocation_type,
            log_type=log_type,
            payload=payload,
            qualifier=qualifier,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="formatInputMap")
    def format_input_map(self) -> typing.Mapping[builtins.str, typing.Any]:
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "formatInputMap", []))

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="functionName")
    def function_name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "functionName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="awsInvoker")
    def aws_invoker(self) -> typing.Optional[IAwsInvoker]:
        return typing.cast(typing.Optional[IAwsInvoker], jsii.get(self, "awsInvoker"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clientContext")
    def client_context(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "clientContext"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="invocationType")
    def invocation_type(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "invocationType"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="logType")
    def log_type(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "logType"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="payload")
    def payload(self) -> typing.Optional[IStringMapVariable]:
        return typing.cast(typing.Optional[IStringMapVariable], jsii.get(self, "payload"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="qualifier")
    def qualifier(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "qualifier"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.InvokeLambdaFunctionStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "function_name": "functionName",
        "client_context": "clientContext",
        "invocation_type": "invocationType",
        "log_type": "logType",
        "payload": "payload",
        "qualifier": "qualifier",
    },
)
class InvokeLambdaFunctionStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        function_name: IStringVariable,
        client_context: typing.Optional[IStringVariable] = None,
        invocation_type: typing.Optional[IStringVariable] = None,
        log_type: typing.Optional[IStringVariable] = None,
        payload: typing.Optional[IStringMapVariable] = None,
        qualifier: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties for InvokeLambdaFunctionStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param function_name: The name of the Lambda function. This function must exist.
        :param client_context: (Optional) The client-specific information.
        :param invocation_type: (Optional) The invocation type. The default value is RequestResponse.
        :param log_type: (Optional) If the default value is Tail, the invocation type must be RequestResponse. Lambda returns the last 4 KB of log data produced by your Lambda function, base64-encoded.
        :param payload: (Optional) The JSON input for your Lambda function.
        :param qualifier: (Optional) The function version or alias name.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "function_name": function_name,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if client_context is not None:
            self._values["client_context"] = client_context
        if invocation_type is not None:
            self._values["invocation_type"] = invocation_type
        if log_type is not None:
            self._values["log_type"] = log_type
        if payload is not None:
            self._values["payload"] = payload
        if qualifier is not None:
            self._values["qualifier"] = qualifier

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def function_name(self) -> IStringVariable:
        '''The name of the Lambda function.

        This function must exist.
        '''
        result = self._values.get("function_name")
        assert result is not None, "Required property 'function_name' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def client_context(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The client-specific information.'''
        result = self._values.get("client_context")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def invocation_type(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The invocation type.

        The default value is RequestResponse.
        '''
        result = self._values.get("invocation_type")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def log_type(self) -> typing.Optional[IStringVariable]:
        '''(Optional) If the default value is Tail, the invocation type must be RequestResponse.

        Lambda returns the last 4 KB of log data produced by your Lambda function, base64-encoded.
        '''
        result = self._values.get("log_type")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def payload(self) -> typing.Optional[IStringMapVariable]:
        '''(Optional) The JSON input for your Lambda function.'''
        result = self._values.get("payload")
        return typing.cast(typing.Optional[IStringMapVariable], result)

    @builtins.property
    def qualifier(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The function version or alias name.'''
        result = self._values.get("qualifier")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InvokeLambdaFunctionStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class InvokeWebhookStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.InvokeWebhookStep",
):
    '''AutomationStep implementation for `aws:invokeWebhook <https://docs.aws.amazon.com/systems-manager/latest/userguide/invoke-webhook.html>`_.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        integration_name: IStringVariable,
        body: typing.Optional[IStringVariable] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param integration_name: The name of the Automation integration. For example, exampleIntegration. The integration you specify must already exist.
        :param body: (Optional) The payload you want to send when your webhook integration is invoked.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = InvokeWebhookStepProps(
            integration_name=integration_name,
            body=body,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="integrationName")
    def integration_name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "integrationName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="body")
    def body(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "body"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.InvokeWebhookStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "integration_name": "integrationName",
        "body": "body",
    },
)
class InvokeWebhookStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        integration_name: IStringVariable,
        body: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties for InvokeWebhookStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param integration_name: The name of the Automation integration. For example, exampleIntegration. The integration you specify must already exist.
        :param body: (Optional) The payload you want to send when your webhook integration is invoked.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "integration_name": integration_name,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if body is not None:
            self._values["body"] = body

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def integration_name(self) -> IStringVariable:
        '''The name of the Automation integration.

        For example, exampleIntegration. The integration you specify must already exist.
        '''
        result = self._values.get("integration_name")
        assert result is not None, "Required property 'integration_name' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def body(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The payload you want to send when your webhook integration is invoked.'''
        result = self._values.get("body")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "InvokeWebhookStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IMapListVariable)
class MapListVariable(
    GenericVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.MapListVariable",
):
    '''A map list variable reference.

    Used to resolve the value from step inputs.
    '''

    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="of") # type: ignore[misc]
    @builtins.classmethod
    def of(cls, reference: builtins.str) -> "MapListVariable":
        '''
        :param reference: -
        '''
        return typing.cast("MapListVariable", jsii.sinvoke(cls, "of", [reference]))

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @jsii.member(jsii_name="resolveToMapList")
    def resolve_to_map_list(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.List[typing.Mapping[builtins.str, typing.Any]]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(typing.List[typing.Mapping[builtins.str, typing.Any]], jsii.invoke(self, "resolveToMapList", [inputs]))


class NonSecureVariable(
    SecureVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.NonSecureVariable",
):
    '''A string variable reference.

    Used to resolve the value from step inputs.
    '''

    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="print")
    def print(self) -> typing.Any:
        '''Prints the variable in a way that SSM understands.

        This is typically in the form of {{Variable}} or the value.
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "print", []))

    @jsii.member(jsii_name="requiredInputs")
    def required_inputs(self) -> typing.List[builtins.str]:
        '''The inputs that are required for determining the value of this variable.

        In the case of a single variable string, this will return a single value.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "requiredInputs", []))

    @jsii.member(jsii_name="resolve")
    def resolve(self, inputs: typing.Mapping[builtins.str, typing.Any]) -> typing.Any:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(typing.Any, jsii.invoke(self, "resolve", [inputs]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="reference")
    def reference(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "reference"))


@jsii.implements(INumberVariable)
class NumberVariable(
    GenericVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.NumberVariable",
):
    '''A number variable reference.

    Used to resolve the value from step inputs.
    '''

    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="of") # type: ignore[misc]
    @builtins.classmethod
    def of(cls, reference: builtins.str) -> "NumberVariable":
        '''
        :param reference: -
        '''
        return typing.cast("NumberVariable", jsii.sinvoke(cls, "of", [reference]))

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @jsii.member(jsii_name="resolveToNumber")
    def resolve_to_number(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> jsii.Number:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(jsii.Number, jsii.invoke(self, "resolveToNumber", [inputs]))


class PauseStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.PauseStep",
):
    '''AutomationStep implementation for aws:pause https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-pause.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = AutomationStepProps(
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''No inputs required for Pause step.

        :return: []
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''No outputs emitted from Pause step.

        :return: []
        '''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))


class PsModuleStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.PsModuleStep",
):
    '''AutomationStep implemenation for aws:psmodule https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-psModule.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        source: IStringVariable,
        run_command: typing.Optional[typing.Sequence[IStringVariable]] = None,
        source_hash: typing.Optional[IStringVariable] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        working_directory: typing.Optional[IStringVariable] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param source: The URL or local path on the instance to the application .zip file. The name of the zip file must be the name of the module to be installed.
        :param run_command: Specify the commands to run or the path to an existing script on the instance.
        :param source_hash: (Optional) The SHA256 hash of the .zip file.
        :param timeout_seconds: (Optional) The time in seconds for a command to be completed before it's considered to have failed. When the timeout is reached, Systems Manager stops the command execution.
        :param working_directory: (Optional) The path to the working directory on your instance.
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = PsModuleStepProps(
            source=source,
            run_command=run_command,
            source_hash=source_hash,
            timeout_seconds=timeout_seconds,
            working_directory=working_directory,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Inputs required for this command includes both the runCommand variables and workingDirectory if provided.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="source")
    def source(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "source"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="runCommand")
    def run_command(self) -> typing.Optional[typing.List[IStringVariable]]:
        return typing.cast(typing.Optional[typing.List[IStringVariable]], jsii.get(self, "runCommand"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceHash")
    def source_hash(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "sourceHash"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="timeoutSeconds")
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "timeoutSeconds"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="workingDirectory")
    def working_directory(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "workingDirectory"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.PsModuleStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "source": "source",
        "run_command": "runCommand",
        "source_hash": "sourceHash",
        "timeout_seconds": "timeoutSeconds",
        "working_directory": "workingDirectory",
    },
)
class PsModuleStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        source: IStringVariable,
        run_command: typing.Optional[typing.Sequence[IStringVariable]] = None,
        source_hash: typing.Optional[IStringVariable] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        working_directory: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties for ps Module step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param source: The URL or local path on the instance to the application .zip file. The name of the zip file must be the name of the module to be installed.
        :param run_command: Specify the commands to run or the path to an existing script on the instance.
        :param source_hash: (Optional) The SHA256 hash of the .zip file.
        :param timeout_seconds: (Optional) The time in seconds for a command to be completed before it's considered to have failed. When the timeout is reached, Systems Manager stops the command execution.
        :param working_directory: (Optional) The path to the working directory on your instance.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "source": source,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition
        if run_command is not None:
            self._values["run_command"] = run_command
        if source_hash is not None:
            self._values["source_hash"] = source_hash
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if working_directory is not None:
            self._values["working_directory"] = working_directory

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def source(self) -> IStringVariable:
        '''The URL or local path on the instance to the application .zip file. The name of the zip file must be the name of the module to be installed.'''
        result = self._values.get("source")
        assert result is not None, "Required property 'source' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def run_command(self) -> typing.Optional[typing.List[IStringVariable]]:
        '''Specify the commands to run or the path to an existing script on the instance.'''
        result = self._values.get("run_command")
        return typing.cast(typing.Optional[typing.List[IStringVariable]], result)

    @builtins.property
    def source_hash(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The SHA256 hash of the .zip file.'''
        result = self._values.get("source_hash")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) The time in seconds for a command to be completed before it's considered to have failed.

        When the timeout is reached, Systems Manager stops the command execution.
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def working_directory(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The path to the working directory on your instance.'''
        result = self._values.get("working_directory")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "PsModuleStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class RebootInstanceAndWait(
    CompositeAutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.RebootInstanceAndWait",
):
    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        instance_id: IStringVariable,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param instance_id: -
        '''
        jsii.create(self.__class__, self, [scope, id, instance_id])

    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: AutomationDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        return typing.cast(None, jsii.invoke(self, "addToDocument", [doc]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="describe")
    def describe(self) -> "WaitForResourceStep":
        return typing.cast("WaitForResourceStep", jsii.get(self, "describe"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="reboot")
    def reboot(self) -> AwsApiStep:
        return typing.cast(AwsApiStep, jsii.get(self, "reboot"))


class RunCommandStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.RunCommandStep",
):
    '''AutomationStep implementation of `aws:runCommand <https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-runcommand.html>`_.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        document_name: IStringVariable,
        targets: typing.Union[IStringListVariable, IMapListVariable],
        cloud_watch_output_config: typing.Optional[IStringMapVariable] = None,
        command_timeout_seconds: typing.Optional[INumberVariable] = None,
        comment: typing.Optional[IStringVariable] = None,
        document_hash: typing.Optional[IStringVariable] = None,
        document_hash_type: typing.Optional[IDocumentHashTypeVariable] = None,
        max_concurrency: typing.Optional[INumberVariable] = None,
        max_errors: typing.Optional[INumberVariable] = None,
        notification_config: typing.Optional[IStringMapVariable] = None,
        output_s3_bucket_name: typing.Optional[IStringVariable] = None,
        output_s3_key_prefix: typing.Optional[IStringVariable] = None,
        parameters: typing.Optional[IStringMapVariable] = None,
        service_role_arn: typing.Optional[IStringVariable] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param document_name: If the Command type document is owned by you or AWS, specify the name of the document. If you're using a document shared with you by a different AWS account, specify the Amazon Resource Name (ARN) of the document.
        :param targets: The instance IDs where you want the command to run. You can specify a maximum of 50 IDs. You can also use the pseudo parameter {{RESOURCE_ID}} in place of instance IDs to run the command on all instances in the target group. For more information about pseudo parameters, see `About pseudo parameters <https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-register-tasks-parameters.html>`_. Another alternative is to send commands to a fleet of instances by using the Targets parameter. The Targets parameter accepts Amazon Elastic Compute Cloud (Amazon EC2) tags. For more information about how to use the Targets parameter, see `Using targets and rate controls to send commands to a fleet <https://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html>`_.
        :param cloud_watch_output_config: (Optional) Configuration options for sending command output to Amazon CloudWatch Logs. For more information about sending command output to CloudWatch Logs, see `Configuring Amazon CloudWatch Logs for Run Command <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-rc-setting-up-cwlogs.html>`_.
        :param command_timeout_seconds: (Optional) The amount of time in seconds to wait for a command to deliver to the AWS Systems Manager SSM Agent on an instance. If the command isn't received by the SSM Agent on the instance before the value specified is reached, then the status of the command changes to Delivery Timed Out.
        :param comment: (Optional) User-defined information about the command.
        :param document_hash: (Optional) The hash for the document.
        :param document_hash_type: (Optional) The type of the hash.
        :param max_concurrency: (Optional) The maximum concurrency.
        :param max_errors: (Optional) The maximum errors.
        :param notification_config: (Optional) The configurations for sending notifications.
        :param output_s3_bucket_name: (Optional) The name of the S3 bucket for command output responses.
        :param output_s3_key_prefix: (Optional) The prefix.
        :param parameters: (Optional) The required and optional parameters specified in the document.
        :param service_role_arn: (Optional) The ARN of the AWS Identity and Access Management (IAM) role.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = RunCommandStepProps(
            document_name=document_name,
            targets=targets,
            cloud_watch_output_config=cloud_watch_output_config,
            command_timeout_seconds=command_timeout_seconds,
            comment=comment,
            document_hash=document_hash,
            document_hash_type=document_hash_type,
            max_concurrency=max_concurrency,
            max_errors=max_errors,
            notification_config=notification_config,
            output_s3_bucket_name=output_s3_bucket_name,
            output_s3_key_prefix=output_s3_key_prefix,
            parameters=parameters,
            service_role_arn=service_role_arn,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="documentName")
    def document_name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "documentName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="targets")
    def targets(self) -> typing.Union[IStringListVariable, IMapListVariable]:
        return typing.cast(typing.Union[IStringListVariable, IMapListVariable], jsii.get(self, "targets"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cloudWatchOutputConfig")
    def cloud_watch_output_config(self) -> typing.Optional[IStringMapVariable]:
        return typing.cast(typing.Optional[IStringMapVariable], jsii.get(self, "cloudWatchOutputConfig"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="commandTimeoutSeconds")
    def command_timeout_seconds(self) -> typing.Optional[INumberVariable]:
        return typing.cast(typing.Optional[INumberVariable], jsii.get(self, "commandTimeoutSeconds"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="comment")
    def comment(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "comment"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="documentHash")
    def document_hash(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "documentHash"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="documentHashType")
    def document_hash_type(self) -> typing.Optional[IDocumentHashTypeVariable]:
        return typing.cast(typing.Optional[IDocumentHashTypeVariable], jsii.get(self, "documentHashType"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="maxConcurrency")
    def max_concurrency(self) -> typing.Optional[INumberVariable]:
        return typing.cast(typing.Optional[INumberVariable], jsii.get(self, "maxConcurrency"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="maxErrors")
    def max_errors(self) -> typing.Optional[INumberVariable]:
        return typing.cast(typing.Optional[INumberVariable], jsii.get(self, "maxErrors"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="notificationConfig")
    def notification_config(self) -> typing.Optional[IStringMapVariable]:
        return typing.cast(typing.Optional[IStringMapVariable], jsii.get(self, "notificationConfig"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="outputS3BucketName")
    def output_s3_bucket_name(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "outputS3BucketName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="outputS3KeyPrefix")
    def output_s3_key_prefix(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "outputS3KeyPrefix"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="parameters")
    def parameters(self) -> typing.Optional[IStringMapVariable]:
        return typing.cast(typing.Optional[IStringMapVariable], jsii.get(self, "parameters"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="serviceRoleArn")
    def service_role_arn(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "serviceRoleArn"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.RunCommandStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "document_name": "documentName",
        "targets": "targets",
        "cloud_watch_output_config": "cloudWatchOutputConfig",
        "command_timeout_seconds": "commandTimeoutSeconds",
        "comment": "comment",
        "document_hash": "documentHash",
        "document_hash_type": "documentHashType",
        "max_concurrency": "maxConcurrency",
        "max_errors": "maxErrors",
        "notification_config": "notificationConfig",
        "output_s3_bucket_name": "outputS3BucketName",
        "output_s3_key_prefix": "outputS3KeyPrefix",
        "parameters": "parameters",
        "service_role_arn": "serviceRoleArn",
    },
)
class RunCommandStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        document_name: IStringVariable,
        targets: typing.Union[IStringListVariable, IMapListVariable],
        cloud_watch_output_config: typing.Optional[IStringMapVariable] = None,
        command_timeout_seconds: typing.Optional[INumberVariable] = None,
        comment: typing.Optional[IStringVariable] = None,
        document_hash: typing.Optional[IStringVariable] = None,
        document_hash_type: typing.Optional[IDocumentHashTypeVariable] = None,
        max_concurrency: typing.Optional[INumberVariable] = None,
        max_errors: typing.Optional[INumberVariable] = None,
        notification_config: typing.Optional[IStringMapVariable] = None,
        output_s3_bucket_name: typing.Optional[IStringVariable] = None,
        output_s3_key_prefix: typing.Optional[IStringVariable] = None,
        parameters: typing.Optional[IStringMapVariable] = None,
        service_role_arn: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties for RunCommandStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param document_name: If the Command type document is owned by you or AWS, specify the name of the document. If you're using a document shared with you by a different AWS account, specify the Amazon Resource Name (ARN) of the document.
        :param targets: The instance IDs where you want the command to run. You can specify a maximum of 50 IDs. You can also use the pseudo parameter {{RESOURCE_ID}} in place of instance IDs to run the command on all instances in the target group. For more information about pseudo parameters, see `About pseudo parameters <https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-register-tasks-parameters.html>`_. Another alternative is to send commands to a fleet of instances by using the Targets parameter. The Targets parameter accepts Amazon Elastic Compute Cloud (Amazon EC2) tags. For more information about how to use the Targets parameter, see `Using targets and rate controls to send commands to a fleet <https://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html>`_.
        :param cloud_watch_output_config: (Optional) Configuration options for sending command output to Amazon CloudWatch Logs. For more information about sending command output to CloudWatch Logs, see `Configuring Amazon CloudWatch Logs for Run Command <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-rc-setting-up-cwlogs.html>`_.
        :param command_timeout_seconds: (Optional) The amount of time in seconds to wait for a command to deliver to the AWS Systems Manager SSM Agent on an instance. If the command isn't received by the SSM Agent on the instance before the value specified is reached, then the status of the command changes to Delivery Timed Out.
        :param comment: (Optional) User-defined information about the command.
        :param document_hash: (Optional) The hash for the document.
        :param document_hash_type: (Optional) The type of the hash.
        :param max_concurrency: (Optional) The maximum concurrency.
        :param max_errors: (Optional) The maximum errors.
        :param notification_config: (Optional) The configurations for sending notifications.
        :param output_s3_bucket_name: (Optional) The name of the S3 bucket for command output responses.
        :param output_s3_key_prefix: (Optional) The prefix.
        :param parameters: (Optional) The required and optional parameters specified in the document.
        :param service_role_arn: (Optional) The ARN of the AWS Identity and Access Management (IAM) role.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "document_name": document_name,
            "targets": targets,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if cloud_watch_output_config is not None:
            self._values["cloud_watch_output_config"] = cloud_watch_output_config
        if command_timeout_seconds is not None:
            self._values["command_timeout_seconds"] = command_timeout_seconds
        if comment is not None:
            self._values["comment"] = comment
        if document_hash is not None:
            self._values["document_hash"] = document_hash
        if document_hash_type is not None:
            self._values["document_hash_type"] = document_hash_type
        if max_concurrency is not None:
            self._values["max_concurrency"] = max_concurrency
        if max_errors is not None:
            self._values["max_errors"] = max_errors
        if notification_config is not None:
            self._values["notification_config"] = notification_config
        if output_s3_bucket_name is not None:
            self._values["output_s3_bucket_name"] = output_s3_bucket_name
        if output_s3_key_prefix is not None:
            self._values["output_s3_key_prefix"] = output_s3_key_prefix
        if parameters is not None:
            self._values["parameters"] = parameters
        if service_role_arn is not None:
            self._values["service_role_arn"] = service_role_arn

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def document_name(self) -> IStringVariable:
        '''If the Command type document is owned by you or AWS, specify the name of the document.

        If you're using a document shared with you by a different AWS account, specify the Amazon Resource Name (ARN) of the document.
        '''
        result = self._values.get("document_name")
        assert result is not None, "Required property 'document_name' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def targets(self) -> typing.Union[IStringListVariable, IMapListVariable]:
        '''The instance IDs where you want the command to run. You can specify a maximum of 50 IDs.

        You can also use the pseudo parameter {{RESOURCE_ID}} in place of instance IDs to run the command on all instances in the target group. For more information about pseudo parameters, see `About pseudo parameters <https://docs.aws.amazon.com/systems-manager/latest/userguide/mw-cli-register-tasks-parameters.html>`_.

        Another alternative is to send commands to a fleet of instances by using the Targets parameter. The Targets parameter accepts Amazon Elastic Compute Cloud (Amazon EC2) tags. For more information about how to use the Targets parameter, see `Using targets and rate controls to send commands to a fleet <https://docs.aws.amazon.com/systems-manager/latest/userguide/send-commands-multiple.html>`_.
        '''
        result = self._values.get("targets")
        assert result is not None, "Required property 'targets' is missing"
        return typing.cast(typing.Union[IStringListVariable, IMapListVariable], result)

    @builtins.property
    def cloud_watch_output_config(self) -> typing.Optional[IStringMapVariable]:
        '''(Optional) Configuration options for sending command output to Amazon CloudWatch Logs.

        For more information about sending command output to CloudWatch Logs, see `Configuring Amazon CloudWatch Logs for Run Command <https://docs.aws.amazon.com/systems-manager/latest/userguide/sysman-rc-setting-up-cwlogs.html>`_.
        '''
        result = self._values.get("cloud_watch_output_config")
        return typing.cast(typing.Optional[IStringMapVariable], result)

    @builtins.property
    def command_timeout_seconds(self) -> typing.Optional[INumberVariable]:
        '''(Optional) The amount of time in seconds to wait for a command to deliver to the AWS Systems Manager SSM Agent on an instance.

        If the command isn't received by the SSM Agent on the instance before the value specified is reached, then the status of the command changes to Delivery Timed Out.
        '''
        result = self._values.get("command_timeout_seconds")
        return typing.cast(typing.Optional[INumberVariable], result)

    @builtins.property
    def comment(self) -> typing.Optional[IStringVariable]:
        '''(Optional) User-defined information about the command.'''
        result = self._values.get("comment")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def document_hash(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The hash for the document.'''
        result = self._values.get("document_hash")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def document_hash_type(self) -> typing.Optional[IDocumentHashTypeVariable]:
        '''(Optional) The type of the hash.'''
        result = self._values.get("document_hash_type")
        return typing.cast(typing.Optional[IDocumentHashTypeVariable], result)

    @builtins.property
    def max_concurrency(self) -> typing.Optional[INumberVariable]:
        '''(Optional) The maximum concurrency.'''
        result = self._values.get("max_concurrency")
        return typing.cast(typing.Optional[INumberVariable], result)

    @builtins.property
    def max_errors(self) -> typing.Optional[INumberVariable]:
        '''(Optional) The maximum errors.'''
        result = self._values.get("max_errors")
        return typing.cast(typing.Optional[INumberVariable], result)

    @builtins.property
    def notification_config(self) -> typing.Optional[IStringMapVariable]:
        '''(Optional) The configurations for sending notifications.'''
        result = self._values.get("notification_config")
        return typing.cast(typing.Optional[IStringMapVariable], result)

    @builtins.property
    def output_s3_bucket_name(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The name of the S3 bucket for command output responses.'''
        result = self._values.get("output_s3_bucket_name")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def output_s3_key_prefix(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The prefix.'''
        result = self._values.get("output_s3_key_prefix")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def parameters(self) -> typing.Optional[IStringMapVariable]:
        '''(Optional) The required and optional parameters specified in the document.'''
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[IStringMapVariable], result)

    @builtins.property
    def service_role_arn(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The ARN of the AWS Identity and Access Management (IAM) role.'''
        result = self._values.get("service_role_arn")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RunCommandStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class RunDockerActionStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.RunDockerActionStep",
):
    '''CommandStep implementation for aws:runDockerAction https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-rundockeraction.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        action: IStringVariable,
        cmd: typing.Optional[IStringVariable] = None,
        container: typing.Optional[IStringVariable] = None,
        cpu_shares: typing.Optional[IStringVariable] = None,
        env: typing.Optional[IStringVariable] = None,
        image: typing.Optional[IStringVariable] = None,
        memory: typing.Optional[IStringVariable] = None,
        publish: typing.Optional[IStringVariable] = None,
        user: typing.Optional[IStringVariable] = None,
        volume: typing.Optional[IStringListVariable] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param action: 
        :param cmd: 
        :param container: 
        :param cpu_shares: 
        :param env: 
        :param image: 
        :param memory: 
        :param publish: 
        :param user: 
        :param volume: 
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = RunDockerActionStepProps(
            action=action,
            cmd=cmd,
            container=container,
            cpu_shares=cpu_shares,
            env=env,
            image=image,
            memory=memory,
            publish=publish,
            user=user,
            volume=volume,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="dockerAction")
    def docker_action(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "dockerAction"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cmd")
    def cmd(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "cmd"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="container")
    def container(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "container"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="cpuShares")
    def cpu_shares(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "cpuShares"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="env")
    def env(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "env"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="image")
    def image(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "image"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="memory")
    def memory(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "memory"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="publish")
    def publish(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "publish"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="user")
    def user(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "user"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="volume")
    def volume(self) -> typing.Optional[IStringListVariable]:
        return typing.cast(typing.Optional[IStringListVariable], jsii.get(self, "volume"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.RunDockerActionStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "action": "action",
        "cmd": "cmd",
        "container": "container",
        "cpu_shares": "cpuShares",
        "env": "env",
        "image": "image",
        "memory": "memory",
        "publish": "publish",
        "user": "user",
        "volume": "volume",
    },
)
class RunDockerActionStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        action: IStringVariable,
        cmd: typing.Optional[IStringVariable] = None,
        container: typing.Optional[IStringVariable] = None,
        cpu_shares: typing.Optional[IStringVariable] = None,
        env: typing.Optional[IStringVariable] = None,
        image: typing.Optional[IStringVariable] = None,
        memory: typing.Optional[IStringVariable] = None,
        publish: typing.Optional[IStringVariable] = None,
        user: typing.Optional[IStringVariable] = None,
        volume: typing.Optional[IStringListVariable] = None,
    ) -> None:
        '''
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param action: 
        :param cmd: 
        :param container: 
        :param cpu_shares: 
        :param env: 
        :param image: 
        :param memory: 
        :param publish: 
        :param user: 
        :param volume: 
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "action": action,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition
        if cmd is not None:
            self._values["cmd"] = cmd
        if container is not None:
            self._values["container"] = container
        if cpu_shares is not None:
            self._values["cpu_shares"] = cpu_shares
        if env is not None:
            self._values["env"] = env
        if image is not None:
            self._values["image"] = image
        if memory is not None:
            self._values["memory"] = memory
        if publish is not None:
            self._values["publish"] = publish
        if user is not None:
            self._values["user"] = user
        if volume is not None:
            self._values["volume"] = volume

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def action(self) -> IStringVariable:
        result = self._values.get("action")
        assert result is not None, "Required property 'action' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def cmd(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("cmd")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def container(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("container")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def cpu_shares(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("cpu_shares")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def env(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("env")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def image(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("image")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def memory(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("memory")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def publish(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("publish")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def user(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("user")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def volume(self) -> typing.Optional[IStringListVariable]:
        result = self._values.get("volume")
        return typing.cast(typing.Optional[IStringListVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RunDockerActionStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class RunDocumentStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.RunDocumentStep",
):
    '''CommandStep implementation for aws:runDocument https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-rundocument.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        step_document: IRunDocumentLocation,
        document_parameters: typing.Optional[IStringMapVariable] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param step_document: Document info containing document type and document path. Can be of type LocalRunDocument or SsmRunDocument.
        :param document_parameters: Parameters for the document.
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = RunDocumentStepProps(
            step_document=step_document,
            document_parameters=document_parameters,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="stepDocument")
    def step_document(self) -> IRunDocumentLocation:
        return typing.cast(IRunDocumentLocation, jsii.get(self, "stepDocument"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="documentParameters")
    def document_parameters(self) -> typing.Optional[IStringMapVariable]:
        return typing.cast(typing.Optional[IStringMapVariable], jsii.get(self, "documentParameters"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.RunDocumentStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "step_document": "stepDocument",
        "document_parameters": "documentParameters",
    },
)
class RunDocumentStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        step_document: IRunDocumentLocation,
        document_parameters: typing.Optional[IStringMapVariable] = None,
    ) -> None:
        '''Properties for Run Document step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param step_document: Document info containing document type and document path. Can be of type LocalRunDocument or SsmRunDocument.
        :param document_parameters: Parameters for the document.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "step_document": step_document,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition
        if document_parameters is not None:
            self._values["document_parameters"] = document_parameters

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def step_document(self) -> IRunDocumentLocation:
        '''Document info containing document type and document path.

        Can be of type LocalRunDocument or SsmRunDocument.
        '''
        result = self._values.get("step_document")
        assert result is not None, "Required property 'step_document' is missing"
        return typing.cast(IRunDocumentLocation, result)

    @builtins.property
    def document_parameters(self) -> typing.Optional[IStringMapVariable]:
        '''Parameters for the document.'''
        result = self._values.get("document_parameters")
        return typing.cast(typing.Optional[IStringMapVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RunDocumentStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class RunInstanceStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.RunInstanceStep",
):
    '''AutomationStep implemenation for aws:runInstance https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-runinstance.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        image_id: IStringVariable,
        additional_info: typing.Optional[IStringVariable] = None,
        block_device_mappings: typing.Optional[IMapListVariable] = None,
        client_token: typing.Optional[IStringVariable] = None,
        disable_api_termination: typing.Optional[IBooleanVariable] = None,
        ebs_optimized: typing.Optional[IBooleanVariable] = None,
        iam_instance_profile_arn: typing.Optional[IStringVariable] = None,
        iam_instance_profile_name: typing.Optional[IStringVariable] = None,
        instance_initiated_shutdown_behavior: typing.Optional[IStringVariable] = None,
        instance_type: typing.Optional[IStringVariable] = None,
        kernel_id: typing.Optional[IStringVariable] = None,
        key_name: typing.Optional[IStringVariable] = None,
        max_instance_count: typing.Optional[INumberVariable] = None,
        min_instance_count: typing.Optional[INumberVariable] = None,
        monitoring: typing.Optional[IBooleanVariable] = None,
        network_interfaces: typing.Optional[IMapListVariable] = None,
        placement: typing.Optional[IStringMapVariable] = None,
        private_ip_address: typing.Optional[IStringVariable] = None,
        ramdisk_id: typing.Optional[IStringVariable] = None,
        security_group_ids: typing.Optional[IStringListVariable] = None,
        security_groups: typing.Optional[IStringListVariable] = None,
        subnet_id: typing.Optional[IStringVariable] = None,
        tag_specifications: typing.Optional[IMapListVariable] = None,
        user_data: typing.Optional[IStringVariable] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param image_id: The ID of the Amazon Machine Image (AMI).
        :param additional_info: (Optional) Reserved.
        :param block_device_mappings: (Optional) The block devices for the instance.
        :param client_token: (Optional) The identifier to ensure idempotency of the request.
        :param disable_api_termination: (Optional) Turns on or turns off instance API termination.
        :param ebs_optimized: (Optional) Turns on or turns off Amazon Elastic Block Store (Amazon EBS) optimization.
        :param iam_instance_profile_arn: (Optional) The Amazon Resource Name (ARN) of the AWS Identity and Access Management (IAM) instance profile for the instance.
        :param iam_instance_profile_name: (Optional) The name of the IAM instance profile for the instance.
        :param instance_initiated_shutdown_behavior: (Optional) Indicates whether the instance stops or terminates on system shutdown.
        :param instance_type: (Optional) The instance type.
        :param kernel_id: (Optional) The ID of the kernel.
        :param key_name: (Optional) The name of the key pair.
        :param max_instance_count: (Optional) The maximum number of instances to be launched.
        :param min_instance_count: (Optional) The minimum number of instances to be launched.
        :param monitoring: (Optional) Turns on or turns off detailed monitoring.
        :param network_interfaces: (Optional) The network interfaces.
        :param placement: (Optional) The placement for the instance.
        :param private_ip_address: (Optional) The primary IPv4 address.
        :param ramdisk_id: (Optional) The ID of the RAM disk.
        :param security_group_ids: (Optional) The IDs of the security groups for the instance.
        :param security_groups: (Optional) The names of the security groups for the instance.
        :param subnet_id: (Optional) The subnet ID.
        :param tag_specifications: (Optional) The tags to apply to the resources during launch. You can only tag instances and volumes at launch. The specified tags are applied to all instances or volumes that are created during launch. To tag an instance after it has been launched, use the aws:createTags – Create tags for AWS resources action.
        :param user_data: (Optional) A script provided as a string literal value. If a literal value is entered, then it must be Base64-encoded.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = RunInstanceStepProps(
            image_id=image_id,
            additional_info=additional_info,
            block_device_mappings=block_device_mappings,
            client_token=client_token,
            disable_api_termination=disable_api_termination,
            ebs_optimized=ebs_optimized,
            iam_instance_profile_arn=iam_instance_profile_arn,
            iam_instance_profile_name=iam_instance_profile_name,
            instance_initiated_shutdown_behavior=instance_initiated_shutdown_behavior,
            instance_type=instance_type,
            kernel_id=kernel_id,
            key_name=key_name,
            max_instance_count=max_instance_count,
            min_instance_count=min_instance_count,
            monitoring=monitoring,
            network_interfaces=network_interfaces,
            placement=placement,
            private_ip_address=private_ip_address,
            ramdisk_id=ramdisk_id,
            security_group_ids=security_group_ids,
            security_groups=security_groups,
            subnet_id=subnet_id,
            tag_specifications=tag_specifications,
            user_data=user_data,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Derives the inputs based on given variables.

        :return: list of required inputs.
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.

        :return: IDs and states of the new instances
        '''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="imageId")
    def image_id(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "imageId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="additionalInfo")
    def additional_info(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "additionalInfo"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="blockDeviceMappings")
    def block_device_mappings(self) -> typing.Optional[IMapListVariable]:
        return typing.cast(typing.Optional[IMapListVariable], jsii.get(self, "blockDeviceMappings"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="clientToken")
    def client_token(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "clientToken"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="disableApiTermination")
    def disable_api_termination(self) -> typing.Optional[IBooleanVariable]:
        return typing.cast(typing.Optional[IBooleanVariable], jsii.get(self, "disableApiTermination"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ebsOptimized")
    def ebs_optimized(self) -> typing.Optional[IBooleanVariable]:
        return typing.cast(typing.Optional[IBooleanVariable], jsii.get(self, "ebsOptimized"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="iamInstanceProfileArn")
    def iam_instance_profile_arn(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "iamInstanceProfileArn"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="iamInstanceProfileName")
    def iam_instance_profile_name(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "iamInstanceProfileName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceInitiatedShutdownBehavior")
    def instance_initiated_shutdown_behavior(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "instanceInitiatedShutdownBehavior"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceType")
    def instance_type(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "instanceType"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="kernelId")
    def kernel_id(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "kernelId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="keyName")
    def key_name(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "keyName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="maxInstanceCount")
    def max_instance_count(self) -> typing.Optional[INumberVariable]:
        return typing.cast(typing.Optional[INumberVariable], jsii.get(self, "maxInstanceCount"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="minInstanceCount")
    def min_instance_count(self) -> typing.Optional[INumberVariable]:
        return typing.cast(typing.Optional[INumberVariable], jsii.get(self, "minInstanceCount"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="monitoring")
    def monitoring(self) -> typing.Optional[IBooleanVariable]:
        return typing.cast(typing.Optional[IBooleanVariable], jsii.get(self, "monitoring"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkInterfaces")
    def network_interfaces(self) -> typing.Optional[IMapListVariable]:
        return typing.cast(typing.Optional[IMapListVariable], jsii.get(self, "networkInterfaces"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="placement")
    def placement(self) -> typing.Optional[IStringMapVariable]:
        return typing.cast(typing.Optional[IStringMapVariable], jsii.get(self, "placement"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="privateIpAddress")
    def private_ip_address(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "privateIpAddress"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="ramdiskId")
    def ramdisk_id(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "ramdiskId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="securityGroupIds")
    def security_group_ids(self) -> typing.Optional[IStringListVariable]:
        return typing.cast(typing.Optional[IStringListVariable], jsii.get(self, "securityGroupIds"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="securityGroups")
    def security_groups(self) -> typing.Optional[IStringListVariable]:
        return typing.cast(typing.Optional[IStringListVariable], jsii.get(self, "securityGroups"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="subnetId")
    def subnet_id(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "subnetId"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="tagSpecifications")
    def tag_specifications(self) -> typing.Optional[IMapListVariable]:
        return typing.cast(typing.Optional[IMapListVariable], jsii.get(self, "tagSpecifications"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="userData")
    def user_data(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "userData"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.RunInstanceStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "image_id": "imageId",
        "additional_info": "additionalInfo",
        "block_device_mappings": "blockDeviceMappings",
        "client_token": "clientToken",
        "disable_api_termination": "disableApiTermination",
        "ebs_optimized": "ebsOptimized",
        "iam_instance_profile_arn": "iamInstanceProfileArn",
        "iam_instance_profile_name": "iamInstanceProfileName",
        "instance_initiated_shutdown_behavior": "instanceInitiatedShutdownBehavior",
        "instance_type": "instanceType",
        "kernel_id": "kernelId",
        "key_name": "keyName",
        "max_instance_count": "maxInstanceCount",
        "min_instance_count": "minInstanceCount",
        "monitoring": "monitoring",
        "network_interfaces": "networkInterfaces",
        "placement": "placement",
        "private_ip_address": "privateIpAddress",
        "ramdisk_id": "ramdiskId",
        "security_group_ids": "securityGroupIds",
        "security_groups": "securityGroups",
        "subnet_id": "subnetId",
        "tag_specifications": "tagSpecifications",
        "user_data": "userData",
    },
)
class RunInstanceStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        image_id: IStringVariable,
        additional_info: typing.Optional[IStringVariable] = None,
        block_device_mappings: typing.Optional[IMapListVariable] = None,
        client_token: typing.Optional[IStringVariable] = None,
        disable_api_termination: typing.Optional[IBooleanVariable] = None,
        ebs_optimized: typing.Optional[IBooleanVariable] = None,
        iam_instance_profile_arn: typing.Optional[IStringVariable] = None,
        iam_instance_profile_name: typing.Optional[IStringVariable] = None,
        instance_initiated_shutdown_behavior: typing.Optional[IStringVariable] = None,
        instance_type: typing.Optional[IStringVariable] = None,
        kernel_id: typing.Optional[IStringVariable] = None,
        key_name: typing.Optional[IStringVariable] = None,
        max_instance_count: typing.Optional[INumberVariable] = None,
        min_instance_count: typing.Optional[INumberVariable] = None,
        monitoring: typing.Optional[IBooleanVariable] = None,
        network_interfaces: typing.Optional[IMapListVariable] = None,
        placement: typing.Optional[IStringMapVariable] = None,
        private_ip_address: typing.Optional[IStringVariable] = None,
        ramdisk_id: typing.Optional[IStringVariable] = None,
        security_group_ids: typing.Optional[IStringListVariable] = None,
        security_groups: typing.Optional[IStringListVariable] = None,
        subnet_id: typing.Optional[IStringVariable] = None,
        tag_specifications: typing.Optional[IMapListVariable] = None,
        user_data: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties for RunInstanceStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param image_id: The ID of the Amazon Machine Image (AMI).
        :param additional_info: (Optional) Reserved.
        :param block_device_mappings: (Optional) The block devices for the instance.
        :param client_token: (Optional) The identifier to ensure idempotency of the request.
        :param disable_api_termination: (Optional) Turns on or turns off instance API termination.
        :param ebs_optimized: (Optional) Turns on or turns off Amazon Elastic Block Store (Amazon EBS) optimization.
        :param iam_instance_profile_arn: (Optional) The Amazon Resource Name (ARN) of the AWS Identity and Access Management (IAM) instance profile for the instance.
        :param iam_instance_profile_name: (Optional) The name of the IAM instance profile for the instance.
        :param instance_initiated_shutdown_behavior: (Optional) Indicates whether the instance stops or terminates on system shutdown.
        :param instance_type: (Optional) The instance type.
        :param kernel_id: (Optional) The ID of the kernel.
        :param key_name: (Optional) The name of the key pair.
        :param max_instance_count: (Optional) The maximum number of instances to be launched.
        :param min_instance_count: (Optional) The minimum number of instances to be launched.
        :param monitoring: (Optional) Turns on or turns off detailed monitoring.
        :param network_interfaces: (Optional) The network interfaces.
        :param placement: (Optional) The placement for the instance.
        :param private_ip_address: (Optional) The primary IPv4 address.
        :param ramdisk_id: (Optional) The ID of the RAM disk.
        :param security_group_ids: (Optional) The IDs of the security groups for the instance.
        :param security_groups: (Optional) The names of the security groups for the instance.
        :param subnet_id: (Optional) The subnet ID.
        :param tag_specifications: (Optional) The tags to apply to the resources during launch. You can only tag instances and volumes at launch. The specified tags are applied to all instances or volumes that are created during launch. To tag an instance after it has been launched, use the aws:createTags – Create tags for AWS resources action.
        :param user_data: (Optional) A script provided as a string literal value. If a literal value is entered, then it must be Base64-encoded.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "image_id": image_id,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if additional_info is not None:
            self._values["additional_info"] = additional_info
        if block_device_mappings is not None:
            self._values["block_device_mappings"] = block_device_mappings
        if client_token is not None:
            self._values["client_token"] = client_token
        if disable_api_termination is not None:
            self._values["disable_api_termination"] = disable_api_termination
        if ebs_optimized is not None:
            self._values["ebs_optimized"] = ebs_optimized
        if iam_instance_profile_arn is not None:
            self._values["iam_instance_profile_arn"] = iam_instance_profile_arn
        if iam_instance_profile_name is not None:
            self._values["iam_instance_profile_name"] = iam_instance_profile_name
        if instance_initiated_shutdown_behavior is not None:
            self._values["instance_initiated_shutdown_behavior"] = instance_initiated_shutdown_behavior
        if instance_type is not None:
            self._values["instance_type"] = instance_type
        if kernel_id is not None:
            self._values["kernel_id"] = kernel_id
        if key_name is not None:
            self._values["key_name"] = key_name
        if max_instance_count is not None:
            self._values["max_instance_count"] = max_instance_count
        if min_instance_count is not None:
            self._values["min_instance_count"] = min_instance_count
        if monitoring is not None:
            self._values["monitoring"] = monitoring
        if network_interfaces is not None:
            self._values["network_interfaces"] = network_interfaces
        if placement is not None:
            self._values["placement"] = placement
        if private_ip_address is not None:
            self._values["private_ip_address"] = private_ip_address
        if ramdisk_id is not None:
            self._values["ramdisk_id"] = ramdisk_id
        if security_group_ids is not None:
            self._values["security_group_ids"] = security_group_ids
        if security_groups is not None:
            self._values["security_groups"] = security_groups
        if subnet_id is not None:
            self._values["subnet_id"] = subnet_id
        if tag_specifications is not None:
            self._values["tag_specifications"] = tag_specifications
        if user_data is not None:
            self._values["user_data"] = user_data

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def image_id(self) -> IStringVariable:
        '''The ID of the Amazon Machine Image (AMI).'''
        result = self._values.get("image_id")
        assert result is not None, "Required property 'image_id' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def additional_info(self) -> typing.Optional[IStringVariable]:
        '''(Optional) Reserved.'''
        result = self._values.get("additional_info")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def block_device_mappings(self) -> typing.Optional[IMapListVariable]:
        '''(Optional) The block devices for the instance.'''
        result = self._values.get("block_device_mappings")
        return typing.cast(typing.Optional[IMapListVariable], result)

    @builtins.property
    def client_token(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The identifier to ensure idempotency of the request.'''
        result = self._values.get("client_token")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def disable_api_termination(self) -> typing.Optional[IBooleanVariable]:
        '''(Optional) Turns on or turns off instance API termination.'''
        result = self._values.get("disable_api_termination")
        return typing.cast(typing.Optional[IBooleanVariable], result)

    @builtins.property
    def ebs_optimized(self) -> typing.Optional[IBooleanVariable]:
        '''(Optional) Turns on or turns off Amazon Elastic Block Store (Amazon EBS) optimization.'''
        result = self._values.get("ebs_optimized")
        return typing.cast(typing.Optional[IBooleanVariable], result)

    @builtins.property
    def iam_instance_profile_arn(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The Amazon Resource Name (ARN) of the AWS Identity and Access Management (IAM) instance profile for the instance.'''
        result = self._values.get("iam_instance_profile_arn")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def iam_instance_profile_name(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The name of the IAM instance profile for the instance.'''
        result = self._values.get("iam_instance_profile_name")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def instance_initiated_shutdown_behavior(self) -> typing.Optional[IStringVariable]:
        '''(Optional) Indicates whether the instance stops or terminates on system shutdown.'''
        result = self._values.get("instance_initiated_shutdown_behavior")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def instance_type(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The instance type.'''
        result = self._values.get("instance_type")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def kernel_id(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The ID of the kernel.'''
        result = self._values.get("kernel_id")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def key_name(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The name of the key pair.'''
        result = self._values.get("key_name")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def max_instance_count(self) -> typing.Optional[INumberVariable]:
        '''(Optional) The maximum number of instances to be launched.'''
        result = self._values.get("max_instance_count")
        return typing.cast(typing.Optional[INumberVariable], result)

    @builtins.property
    def min_instance_count(self) -> typing.Optional[INumberVariable]:
        '''(Optional) The minimum number of instances to be launched.'''
        result = self._values.get("min_instance_count")
        return typing.cast(typing.Optional[INumberVariable], result)

    @builtins.property
    def monitoring(self) -> typing.Optional[IBooleanVariable]:
        '''(Optional) Turns on or turns off detailed monitoring.'''
        result = self._values.get("monitoring")
        return typing.cast(typing.Optional[IBooleanVariable], result)

    @builtins.property
    def network_interfaces(self) -> typing.Optional[IMapListVariable]:
        '''(Optional) The network interfaces.'''
        result = self._values.get("network_interfaces")
        return typing.cast(typing.Optional[IMapListVariable], result)

    @builtins.property
    def placement(self) -> typing.Optional[IStringMapVariable]:
        '''(Optional) The placement for the instance.'''
        result = self._values.get("placement")
        return typing.cast(typing.Optional[IStringMapVariable], result)

    @builtins.property
    def private_ip_address(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The primary IPv4 address.'''
        result = self._values.get("private_ip_address")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def ramdisk_id(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The ID of the RAM disk.'''
        result = self._values.get("ramdisk_id")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def security_group_ids(self) -> typing.Optional[IStringListVariable]:
        '''(Optional) The IDs of the security groups for the instance.'''
        result = self._values.get("security_group_ids")
        return typing.cast(typing.Optional[IStringListVariable], result)

    @builtins.property
    def security_groups(self) -> typing.Optional[IStringListVariable]:
        '''(Optional) The names of the security groups for the instance.'''
        result = self._values.get("security_groups")
        return typing.cast(typing.Optional[IStringListVariable], result)

    @builtins.property
    def subnet_id(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The subnet ID.'''
        result = self._values.get("subnet_id")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def tag_specifications(self) -> typing.Optional[IMapListVariable]:
        '''(Optional) The tags to apply to the resources during launch.

        You can only tag instances and volumes at launch. The specified tags are applied to all instances or volumes that are created during launch.
        To tag an instance after it has been launched, use the aws:createTags – Create tags for AWS resources action.
        '''
        result = self._values.get("tag_specifications")
        return typing.cast(typing.Optional[IMapListVariable], result)

    @builtins.property
    def user_data(self) -> typing.Optional[IStringVariable]:
        '''(Optional) A script provided as a string literal value.

        If a literal value is entered, then it must be Base64-encoded.
        '''
        result = self._values.get("user_data")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RunInstanceStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class RunPowerShellScriptStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.RunPowerShellScriptStep",
):
    '''AutomationStep implementation for aws:runPowerShellScript https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-runPowerShellScript.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        run_command: typing.Sequence[IStringVariable],
        timeout_seconds: typing.Optional[jsii.Number] = None,
        working_directory: typing.Optional[IStringVariable] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param run_command: Specify the commands to run or the path to an existing script on the instance.
        :param timeout_seconds: (Optional) The time in seconds for a command to be completed before it's considered to have failed. When the timeout is reached, Systems Manager stops the command execution.
        :param working_directory: (Optional) The path to the working directory on your instance.
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = RunPowerShellScriptStepProps(
            run_command=run_command,
            timeout_seconds=timeout_seconds,
            working_directory=working_directory,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Inputs required for this command includes both the runCommand variables and workingDirectory if provided.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="runCommand")
    def run_command(self) -> typing.List[IStringVariable]:
        return typing.cast(typing.List[IStringVariable], jsii.get(self, "runCommand"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="timeoutSeconds")
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "timeoutSeconds"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="workingDirectory")
    def working_directory(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "workingDirectory"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.RunPowerShellScriptStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "run_command": "runCommand",
        "timeout_seconds": "timeoutSeconds",
        "working_directory": "workingDirectory",
    },
)
class RunPowerShellScriptStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        run_command: typing.Sequence[IStringVariable],
        timeout_seconds: typing.Optional[jsii.Number] = None,
        working_directory: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties for runPowerShellScript step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param run_command: Specify the commands to run or the path to an existing script on the instance.
        :param timeout_seconds: (Optional) The time in seconds for a command to be completed before it's considered to have failed. When the timeout is reached, Systems Manager stops the command execution.
        :param working_directory: (Optional) The path to the working directory on your instance.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "run_command": run_command,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if working_directory is not None:
            self._values["working_directory"] = working_directory

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def run_command(self) -> typing.List[IStringVariable]:
        '''Specify the commands to run or the path to an existing script on the instance.'''
        result = self._values.get("run_command")
        assert result is not None, "Required property 'run_command' is missing"
        return typing.cast(typing.List[IStringVariable], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) The time in seconds for a command to be completed before it's considered to have failed.

        When the timeout is reached, Systems Manager stops the command execution.
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def working_directory(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The path to the working directory on your instance.'''
        result = self._values.get("working_directory")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RunPowerShellScriptStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class RunShellScriptStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.RunShellScriptStep",
):
    '''AutomationStep implemenation for aws:sleep https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-sleep.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        run_command: typing.Sequence[IStringVariable],
        timeout_seconds: typing.Optional[jsii.Number] = None,
        working_directory: typing.Optional[IStringVariable] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param run_command: Specify the commands to run or the path to an existing script on the instance.
        :param timeout_seconds: (Optional) The time in seconds for a command to be completed before it's considered to have failed. When the timeout is reached, Systems Manager stops the command execution.
        :param working_directory: (Optional) The path to the working directory on your instance.
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = RunShellScriptStepProps(
            run_command=run_command,
            timeout_seconds=timeout_seconds,
            working_directory=working_directory,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Inputs required for this command includes both the runCommand variables and workingDirectory if provided.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="runCommand")
    def run_command(self) -> typing.List[IStringVariable]:
        return typing.cast(typing.List[IStringVariable], jsii.get(self, "runCommand"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="timeoutSeconds")
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        return typing.cast(typing.Optional[jsii.Number], jsii.get(self, "timeoutSeconds"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="workingDirectory")
    def working_directory(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "workingDirectory"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.RunShellScriptStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "run_command": "runCommand",
        "timeout_seconds": "timeoutSeconds",
        "working_directory": "workingDirectory",
    },
)
class RunShellScriptStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        run_command: typing.Sequence[IStringVariable],
        timeout_seconds: typing.Optional[jsii.Number] = None,
        working_directory: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties for sleep step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param run_command: Specify the commands to run or the path to an existing script on the instance.
        :param timeout_seconds: (Optional) The time in seconds for a command to be completed before it's considered to have failed. When the timeout is reached, Systems Manager stops the command execution.
        :param working_directory: (Optional) The path to the working directory on your instance.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "run_command": run_command,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if working_directory is not None:
            self._values["working_directory"] = working_directory

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def run_command(self) -> typing.List[IStringVariable]:
        '''Specify the commands to run or the path to an existing script on the instance.'''
        result = self._values.get("run_command")
        assert result is not None, "Required property 'run_command' is missing"
        return typing.cast(typing.List[IStringVariable], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) The time in seconds for a command to be completed before it's considered to have failed.

        When the timeout is reached, Systems Manager stops the command execution.
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def working_directory(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The path to the working directory on your instance.'''
        result = self._values.get("working_directory")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "RunShellScriptStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class SleepStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.SleepStep",
):
    '''AutomationStep implemenation for aws:sleep https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-sleep.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        sleep_seconds: jsii.Number,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param sleep_seconds: (Required) The amount of seconds to sleep for. May not exceed 604800. This will be used in the SSM doc as well as how long to indicate sleep to the sleepHook.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = SleepStepProps(
            sleep_seconds=sleep_seconds,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''No inputs required for Sleep step.

        :return: []
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''No outputs emitted from Sleep step.

        :return: []
        '''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sleepSeconds")
    def sleep_seconds(self) -> jsii.Number:
        return typing.cast(jsii.Number, jsii.get(self, "sleepSeconds"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.SleepStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "sleep_seconds": "sleepSeconds",
    },
)
class SleepStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        sleep_seconds: jsii.Number,
    ) -> None:
        '''Properties for sleep step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param sleep_seconds: (Required) The amount of seconds to sleep for. May not exceed 604800. This will be used in the SSM doc as well as how long to indicate sleep to the sleepHook.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "sleep_seconds": sleep_seconds,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def sleep_seconds(self) -> jsii.Number:
        '''(Required) The amount of seconds to sleep for.

        May not exceed 604800.
        This will be used in the SSM doc as well as how long to indicate sleep to the sleepHook.
        '''
        result = self._values.get("sleep_seconds")
        assert result is not None, "Required property 'sleep_seconds' is missing"
        return typing.cast(jsii.Number, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SleepStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class SoftwareInventoryStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.SoftwareInventoryStep",
):
    '''AutomationStep implemenation for aws:softwareInventory https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-softwareinventory.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        applications: typing.Optional[IStringVariable] = None,
        aws_components: typing.Optional[IStringVariable] = None,
        custom_inventory: typing.Optional[IStringVariable] = None,
        files: typing.Optional[IStringVariable] = None,
        instance_detailed_information: typing.Optional[IStringVariable] = None,
        network_config: typing.Optional[IStringVariable] = None,
        services: typing.Optional[IStringVariable] = None,
        windows_registry: typing.Optional[IStringVariable] = None,
        windows_roles: typing.Optional[IStringVariable] = None,
        windows_updates: typing.Optional[IStringVariable] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param applications: (Optional) Collect metadata for installed applications.
        :param aws_components: (Optional) Collect metadata for AWS components like amazon-ssm-agent.
        :param custom_inventory: (Optional) Collect custom inventory data. For more information about custom inventory, see Working with custom inventory
        :param files: (Optional, requires SSM Agent version 2.2.64.0 or later) Collect metadata for files, including file names, the time files were created, the time files were last modified and accessed, and file sizes, to name a few. For more information about collecting file inventory, see Working with file and Windows registry.
        :param instance_detailed_information: (Optional) Collect more instance information than is provided by the default inventory plugin (aws:instanceInformation), including CPU model, speed, and the number of cores, to name a few.
        :param network_config: (Optional) Collect metadata for network configurations.
        :param services: (Optional, Windows OS only, requires SSM Agent version 2.2.64.0 or later) Collect metadata for service configurations.
        :param windows_registry: (Optional, Windows OS only, requires SSM Agent version 2.2.64.0 or later) Collect Windows Registry keys and values. You can choose a key path and collect all keys and values recursively. You can also collect a specific registry key and its value for a specific path. Inventory collects the key path, name, type, and the value. For more information about collecting Windows Registry inventory, see Working with file and Windows registry inventory.
        :param windows_roles: (Optional, Windows OS only, requires SSM Agent version 2.2.64.0 or later) Collect metadata for Microsoft Windows role configurations.
        :param windows_updates: (Optional) Collect metadata for all Windows updates.
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = SoftwareInventoryStepProps(
            applications=applications,
            aws_components=aws_components,
            custom_inventory=custom_inventory,
            files=files,
            instance_detailed_information=instance_detailed_information,
            network_config=network_config,
            services=services,
            windows_registry=windows_registry,
            windows_roles=windows_roles,
            windows_updates=windows_updates,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''All Inputs for this command are optional.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="applications")
    def applications(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "applications"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="awsComponents")
    def aws_components(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "awsComponents"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="customInventory")
    def custom_inventory(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "customInventory"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="files")
    def files(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "files"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="instanceDetailedInformation")
    def instance_detailed_information(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "instanceDetailedInformation"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="networkConfig")
    def network_config(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "networkConfig"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="services")
    def services(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "services"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="windowsRegistry")
    def windows_registry(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "windowsRegistry"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="windowsRoles")
    def windows_roles(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "windowsRoles"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="windowsUpdates")
    def windows_updates(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "windowsUpdates"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.SoftwareInventoryStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "applications": "applications",
        "aws_components": "awsComponents",
        "custom_inventory": "customInventory",
        "files": "files",
        "instance_detailed_information": "instanceDetailedInformation",
        "network_config": "networkConfig",
        "services": "services",
        "windows_registry": "windowsRegistry",
        "windows_roles": "windowsRoles",
        "windows_updates": "windowsUpdates",
    },
)
class SoftwareInventoryStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        applications: typing.Optional[IStringVariable] = None,
        aws_components: typing.Optional[IStringVariable] = None,
        custom_inventory: typing.Optional[IStringVariable] = None,
        files: typing.Optional[IStringVariable] = None,
        instance_detailed_information: typing.Optional[IStringVariable] = None,
        network_config: typing.Optional[IStringVariable] = None,
        services: typing.Optional[IStringVariable] = None,
        windows_registry: typing.Optional[IStringVariable] = None,
        windows_roles: typing.Optional[IStringVariable] = None,
        windows_updates: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties of SoftwareInventory step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param applications: (Optional) Collect metadata for installed applications.
        :param aws_components: (Optional) Collect metadata for AWS components like amazon-ssm-agent.
        :param custom_inventory: (Optional) Collect custom inventory data. For more information about custom inventory, see Working with custom inventory
        :param files: (Optional, requires SSM Agent version 2.2.64.0 or later) Collect metadata for files, including file names, the time files were created, the time files were last modified and accessed, and file sizes, to name a few. For more information about collecting file inventory, see Working with file and Windows registry.
        :param instance_detailed_information: (Optional) Collect more instance information than is provided by the default inventory plugin (aws:instanceInformation), including CPU model, speed, and the number of cores, to name a few.
        :param network_config: (Optional) Collect metadata for network configurations.
        :param services: (Optional, Windows OS only, requires SSM Agent version 2.2.64.0 or later) Collect metadata for service configurations.
        :param windows_registry: (Optional, Windows OS only, requires SSM Agent version 2.2.64.0 or later) Collect Windows Registry keys and values. You can choose a key path and collect all keys and values recursively. You can also collect a specific registry key and its value for a specific path. Inventory collects the key path, name, type, and the value. For more information about collecting Windows Registry inventory, see Working with file and Windows registry inventory.
        :param windows_roles: (Optional, Windows OS only, requires SSM Agent version 2.2.64.0 or later) Collect metadata for Microsoft Windows role configurations.
        :param windows_updates: (Optional) Collect metadata for all Windows updates.
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition
        if applications is not None:
            self._values["applications"] = applications
        if aws_components is not None:
            self._values["aws_components"] = aws_components
        if custom_inventory is not None:
            self._values["custom_inventory"] = custom_inventory
        if files is not None:
            self._values["files"] = files
        if instance_detailed_information is not None:
            self._values["instance_detailed_information"] = instance_detailed_information
        if network_config is not None:
            self._values["network_config"] = network_config
        if services is not None:
            self._values["services"] = services
        if windows_registry is not None:
            self._values["windows_registry"] = windows_registry
        if windows_roles is not None:
            self._values["windows_roles"] = windows_roles
        if windows_updates is not None:
            self._values["windows_updates"] = windows_updates

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def applications(self) -> typing.Optional[IStringVariable]:
        '''(Optional) Collect metadata for installed applications.'''
        result = self._values.get("applications")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def aws_components(self) -> typing.Optional[IStringVariable]:
        '''(Optional) Collect metadata for AWS components like amazon-ssm-agent.'''
        result = self._values.get("aws_components")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def custom_inventory(self) -> typing.Optional[IStringVariable]:
        '''(Optional) Collect custom inventory data.

        For more information about custom inventory, see Working with custom inventory
        '''
        result = self._values.get("custom_inventory")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def files(self) -> typing.Optional[IStringVariable]:
        '''(Optional, requires SSM Agent version 2.2.64.0 or later) Collect metadata for files, including file names, the time files were created, the time files were last modified and accessed, and file sizes, to name a few. For more information about collecting file inventory, see Working with file and Windows registry.'''
        result = self._values.get("files")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def instance_detailed_information(self) -> typing.Optional[IStringVariable]:
        '''(Optional) Collect more instance information than is provided by the default inventory plugin (aws:instanceInformation), including CPU model, speed, and the number of cores, to name a few.'''
        result = self._values.get("instance_detailed_information")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def network_config(self) -> typing.Optional[IStringVariable]:
        '''(Optional) Collect metadata for network configurations.'''
        result = self._values.get("network_config")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def services(self) -> typing.Optional[IStringVariable]:
        '''(Optional, Windows OS only, requires SSM Agent version 2.2.64.0 or later) Collect metadata for service configurations.'''
        result = self._values.get("services")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def windows_registry(self) -> typing.Optional[IStringVariable]:
        '''(Optional, Windows OS only, requires SSM Agent version 2.2.64.0 or later) Collect Windows Registry keys and values. You can choose a key path and collect all keys and values recursively. You can also collect a specific registry key and its value for a specific path. Inventory collects the key path, name, type, and the value. For more information about collecting Windows Registry inventory, see Working with file and Windows registry inventory.'''
        result = self._values.get("windows_registry")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def windows_roles(self) -> typing.Optional[IStringVariable]:
        '''(Optional, Windows OS only, requires SSM Agent version 2.2.64.0 or later) Collect metadata for Microsoft Windows role configurations.'''
        result = self._values.get("windows_roles")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def windows_updates(self) -> typing.Optional[IStringVariable]:
        '''(Optional) Collect metadata for all Windows updates.'''
        result = self._values.get("windows_updates")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "SoftwareInventoryStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IStringListVariable)
class StringListVariable(
    GenericVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.StringListVariable",
):
    '''A string list variable reference.

    Used to resolve the value from step inputs.
    '''

    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="of") # type: ignore[misc]
    @builtins.classmethod
    def of(cls, reference: builtins.str) -> "StringListVariable":
        '''
        :param reference: -
        '''
        return typing.cast("StringListVariable", jsii.sinvoke(cls, "of", [reference]))

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @jsii.member(jsii_name="resolveToStringList")
    def resolve_to_string_list(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.List[builtins.str]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "resolveToStringList", [inputs]))


@jsii.implements(IStringMapVariable)
class StringMapVariable(
    GenericVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.StringMapVariable",
):
    '''A string map variable reference.

    Used to resolve the value from step inputs.
    '''

    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="of") # type: ignore[misc]
    @builtins.classmethod
    def of(cls, reference: builtins.str) -> "StringMapVariable":
        '''
        :param reference: -
        '''
        return typing.cast("StringMapVariable", jsii.sinvoke(cls, "of", [reference]))

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @jsii.member(jsii_name="resolveToStringMap")
    def resolve_to_string_map(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "resolveToStringMap", [inputs]))


class StringStep(
    CompositeAutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.StringStep",
):
    '''StringStep allows for including a step from an existing automation document verbatim.

    This is useful in that it allows developers to integrate with existing document steps.
    This step can be used just as you would use any other Step including simulation and deployment.
    '''

    @jsii.member(jsii_name="fromJson") # type: ignore[misc]
    @builtins.classmethod
    def from_json(cls, scope: constructs.Construct, json: builtins.str) -> "StringStep":
        '''Builds a step object from a json declaration.

        You may cast the step afterword to the associated Step for this action
        if you wish to gain access to action specific functionality,

        :param scope: -
        :param json: -

        :return: Step associated with the provided activity.
        '''
        return typing.cast("StringStep", jsii.sinvoke(cls, "fromJson", [scope, json]))

    @jsii.member(jsii_name="fromObject") # type: ignore[misc]
    @builtins.classmethod
    def from_object(
        cls,
        scope: constructs.Construct,
        props: typing.Mapping[builtins.str, typing.Any],
    ) -> "StringStep":
        '''Builds a step object from an object.

        You may cast the step afterword to the associated Step for this action
        if you wish to gain access to action specific functionality,

        :param scope: -
        :param props: -

        :return: Step associated with the provided activity.
        '''
        return typing.cast("StringStep", jsii.sinvoke(cls, "fromObject", [scope, props]))

    @jsii.member(jsii_name="fromYaml") # type: ignore[misc]
    @builtins.classmethod
    def from_yaml(
        cls,
        scope: constructs.Construct,
        input_yaml: builtins.str,
    ) -> "StringStep":
        '''Builds a step object from a yaml declaration.

        You may cast the step afterword to the associated Step for this action
        if you wish to gain access to action specific functionality,

        :param scope: -
        :param input_yaml: -

        :return: Step associated with the provided activity.
        '''
        return typing.cast("StringStep", jsii.sinvoke(cls, "fromYaml", [scope, input_yaml]))

    @jsii.member(jsii_name="addToDocument")
    def add_to_document(self, doc: AutomationDocumentBuilder) -> None:
        '''
        :param doc: -
        '''
        return typing.cast(None, jsii.invoke(self, "addToDocument", [doc]))


@jsii.implements(IStringVariable)
class StringVariable(
    GenericVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.StringVariable",
):
    '''A string variable reference.

    Used to resolve the value from step inputs.
    '''

    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="of") # type: ignore[misc]
    @builtins.classmethod
    def of(cls, reference: builtins.str) -> "StringVariable":
        '''
        :param reference: -
        '''
        return typing.cast("StringVariable", jsii.sinvoke(cls, "of", [reference]))

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @jsii.member(jsii_name="resolveToString")
    def resolve_to_string(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.str:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "resolveToString", [inputs]))


class TimedDocument(
    AutomationDocument,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.TimedDocument",
):
    def __init__(
        self,
        stage: constructs.Construct,
        id: builtins.str,
        *,
        assume_role: typing.Optional[builtins.str] = None,
        description: typing.Optional[builtins.str] = None,
        doc_inputs: typing.Optional[typing.Sequence[Input]] = None,
        doc_outputs: typing.Optional[typing.Sequence[DocumentOutput]] = None,
        document_format: typing.Optional[DocumentFormat] = None,
        document_name: typing.Optional[builtins.str] = None,
        header: typing.Optional[builtins.str] = None,
        requires: typing.Optional[typing.Union[aws_cdk.IResolvable, typing.Sequence[typing.Union[aws_cdk.aws_ssm.CfnDocument.DocumentRequiresProperty, aws_cdk.IResolvable]]]] = None,
        tags: typing.Optional[typing.Sequence[aws_cdk.CfnTag]] = None,
        target_type: typing.Optional[builtins.str] = None,
        version_name: typing.Optional[builtins.str] = None,
    ) -> None:
        '''
        :param stage: -
        :param id: -
        :param assume_role: (Optional) Assume role to use for this document. If provided, this value MUST be included as one of the documentInput names.
        :param description: (Optional) Description of the document. Defaults to the document name.
        :param doc_inputs: (Optional) Inputs required by the document.
        :param doc_outputs: (Optional) Outputs to be emitted from the document. The outputs are placed in a StringSet called outputs (as is done in SSM). Default: []
        :param document_format: (Optional) Specifies whether this document should be written as YAML or JSON. Default: JSON
        :param document_name: (Optional) Name of the document. Will default to the id provided for the CDK node.
        :param header: (Optional) A Header/comment to include at the start of a YAML document. JSON documents do not support headers.
        :param requires: ``AWS::SSM::Document.Requires``.
        :param tags: ``AWS::SSM::Document.Tags``.
        :param target_type: ``AWS::SSM::Document.TargetType``.
        :param version_name: ``AWS::SSM::Document.VersionName``.
        '''
        props = AutomationDocumentProps(
            assume_role=assume_role,
            description=description,
            doc_inputs=doc_inputs,
            doc_outputs=doc_outputs,
            document_format=document_format,
            document_name=document_name,
            header=header,
            requires=requires,
            tags=tags,
            target_type=target_type,
            version_name=version_name,
        )

        jsii.create(self.__class__, self, [stage, id, props])

    @jsii.member(jsii_name="collectedSteps")
    def collected_steps(self) -> typing.List[AutomationStep]:
        return typing.cast(typing.List[AutomationStep], jsii.invoke(self, "collectedSteps", []))


class UpdateAgentStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.UpdateAgentStep",
):
    '''CommandStep implementation for aws:UpdateAgent https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-updateagent.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        agent_name: typing.Optional[IStringVariable] = None,
        allow_downgrade: typing.Optional[IBooleanVariable] = None,
        source: typing.Optional[IStringVariable] = None,
        source_region: typing.Optional[builtins.str] = None,
        target_version: typing.Optional[IStringVariable] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param agent_name: EC2Config. This is the name of the agent that runs the EC2Config service. Default: EC2Config
        :param allow_downgrade: Allow the EC2Config service to be downgraded to an earlier version. If set to false, the service can be upgraded to newer versions only (default). If set to true, specify the earlier version.
        :param source: Must use one and only one of source or sourceRegion. The location where Systems Manager copies the version of EC2Config to install. You can't change this location.
        :param source_region: Must use one and only one of source or sourceRegion. If source is provided it will be used as is. If sourceRegion is provided it will populate source with https://s3.{sourceRegion}.amazonaws.com/aws-ssm-{sourceRegion}/manifest.json
        :param target_version: A specific version of the EC2Config service to install. If not specified, the service will be updated to the latest version.
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = UpdateAgentStepProps(
            agent_name=agent_name,
            allow_downgrade=allow_downgrade,
            source=source,
            source_region=source_region,
            target_version=target_version,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Inputs required for this command include agentName allowDowngrade source and targetVersion if version other than latest is desired.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="agentName")
    def agent_name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "agentName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="source")
    def source(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "source"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="allowDowngrade")
    def allow_downgrade(self) -> typing.Optional[IBooleanVariable]:
        return typing.cast(typing.Optional[IBooleanVariable], jsii.get(self, "allowDowngrade"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="targetVersion")
    def target_version(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "targetVersion"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.UpdateAgentStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "agent_name": "agentName",
        "allow_downgrade": "allowDowngrade",
        "source": "source",
        "source_region": "sourceRegion",
        "target_version": "targetVersion",
    },
)
class UpdateAgentStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        agent_name: typing.Optional[IStringVariable] = None,
        allow_downgrade: typing.Optional[IBooleanVariable] = None,
        source: typing.Optional[IStringVariable] = None,
        source_region: typing.Optional[builtins.str] = None,
        target_version: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties UpdateAgent step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param agent_name: EC2Config. This is the name of the agent that runs the EC2Config service. Default: EC2Config
        :param allow_downgrade: Allow the EC2Config service to be downgraded to an earlier version. If set to false, the service can be upgraded to newer versions only (default). If set to true, specify the earlier version.
        :param source: Must use one and only one of source or sourceRegion. The location where Systems Manager copies the version of EC2Config to install. You can't change this location.
        :param source_region: Must use one and only one of source or sourceRegion. If source is provided it will be used as is. If sourceRegion is provided it will populate source with https://s3.{sourceRegion}.amazonaws.com/aws-ssm-{sourceRegion}/manifest.json
        :param target_version: A specific version of the EC2Config service to install. If not specified, the service will be updated to the latest version.
        '''
        self._values: typing.Dict[str, typing.Any] = {}
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition
        if agent_name is not None:
            self._values["agent_name"] = agent_name
        if allow_downgrade is not None:
            self._values["allow_downgrade"] = allow_downgrade
        if source is not None:
            self._values["source"] = source
        if source_region is not None:
            self._values["source_region"] = source_region
        if target_version is not None:
            self._values["target_version"] = target_version

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def agent_name(self) -> typing.Optional[IStringVariable]:
        '''EC2Config.

        This is the name of the agent that runs the EC2Config service.

        :default: EC2Config
        '''
        result = self._values.get("agent_name")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def allow_downgrade(self) -> typing.Optional[IBooleanVariable]:
        '''Allow the EC2Config service to be downgraded to an earlier version.

        If set to false, the service can be upgraded to newer versions only (default).
        If set to true, specify the earlier version.
        '''
        result = self._values.get("allow_downgrade")
        return typing.cast(typing.Optional[IBooleanVariable], result)

    @builtins.property
    def source(self) -> typing.Optional[IStringVariable]:
        '''Must use one and only one of source or sourceRegion.

        The location where Systems Manager copies the version of EC2Config to install. You can't change this location.
        '''
        result = self._values.get("source")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def source_region(self) -> typing.Optional[builtins.str]:
        '''Must use one and only one of source or sourceRegion.

        If source is provided it will be used as is.
        If sourceRegion is provided it will populate source with https://s3.{sourceRegion}.amazonaws.com/aws-ssm-{sourceRegion}/manifest.json
        '''
        result = self._values.get("source_region")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def target_version(self) -> typing.Optional[IStringVariable]:
        '''A specific version of the EC2Config service to install.

        If not specified, the service will be updated to the latest version.
        '''
        result = self._values.get("target_version")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "UpdateAgentStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class UpdateSSMAgentStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.UpdateSSMAgentStep",
):
    '''AutomationStep implemenation for aws:UpdateSsmAgent https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-updatessmagent.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        allow_downgrade: IBooleanVariable,
        agent_name: typing.Optional[IStringVariable] = None,
        source: typing.Optional[IStringVariable] = None,
        source_region: typing.Optional[builtins.str] = None,
        target_version: typing.Optional[IStringVariable] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param allow_downgrade: Allow the SSM Agent to be downgraded to an earlier version. If set to false, the agent can be upgraded to newer versions only (default). If set to true, specify the earlier version.
        :param agent_name: amazon-ssm-agent. This is the name of the Systems Manager agent that processes requests and runs commands on the instance.
        :param source: Must use one and only one of source or sourceRegion. The location where Systems Manager copies the SSM Agent version to install. You can't change this location.
        :param source_region: Must use one and only one of source or sourceRegion. If source is provided it will be used as is. If sourceRegion is provided it will populate source with https://s3.{sourceRegion}.amazonaws.com/aws-ssm-{sourceRegion}/manifest.json
        :param target_version: A specific version of SSM Agent to install. If not specified, the agent will be updated to the latest version.
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = UpdateSSMAgentStepProps(
            allow_downgrade=allow_downgrade,
            agent_name=agent_name,
            source=source,
            source_region=source_region,
            target_version=target_version,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Inputs required for this command include agentName allowDowngrade source and targetVersion if version other than latest is desired.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="agentName")
    def agent_name(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "agentName"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="allowDowngrade")
    def allow_downgrade(self) -> IBooleanVariable:
        return typing.cast(IBooleanVariable, jsii.get(self, "allowDowngrade"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="source")
    def source(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "source"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="targetVersion")
    def target_version(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "targetVersion"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.UpdateSSMAgentStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "allow_downgrade": "allowDowngrade",
        "agent_name": "agentName",
        "source": "source",
        "source_region": "sourceRegion",
        "target_version": "targetVersion",
    },
)
class UpdateSSMAgentStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        allow_downgrade: IBooleanVariable,
        agent_name: typing.Optional[IStringVariable] = None,
        source: typing.Optional[IStringVariable] = None,
        source_region: typing.Optional[builtins.str] = None,
        target_version: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties UpdateAgent step.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param allow_downgrade: Allow the SSM Agent to be downgraded to an earlier version. If set to false, the agent can be upgraded to newer versions only (default). If set to true, specify the earlier version.
        :param agent_name: amazon-ssm-agent. This is the name of the Systems Manager agent that processes requests and runs commands on the instance.
        :param source: Must use one and only one of source or sourceRegion. The location where Systems Manager copies the SSM Agent version to install. You can't change this location.
        :param source_region: Must use one and only one of source or sourceRegion. If source is provided it will be used as is. If sourceRegion is provided it will populate source with https://s3.{sourceRegion}.amazonaws.com/aws-ssm-{sourceRegion}/manifest.json
        :param target_version: A specific version of SSM Agent to install. If not specified, the agent will be updated to the latest version.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "allow_downgrade": allow_downgrade,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition
        if agent_name is not None:
            self._values["agent_name"] = agent_name
        if source is not None:
            self._values["source"] = source
        if source_region is not None:
            self._values["source_region"] = source_region
        if target_version is not None:
            self._values["target_version"] = target_version

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def allow_downgrade(self) -> IBooleanVariable:
        '''Allow the SSM Agent to be downgraded to an earlier version.

        If set to false, the agent can be upgraded to newer versions only (default).
        If set to true, specify the earlier version.
        '''
        result = self._values.get("allow_downgrade")
        assert result is not None, "Required property 'allow_downgrade' is missing"
        return typing.cast(IBooleanVariable, result)

    @builtins.property
    def agent_name(self) -> typing.Optional[IStringVariable]:
        '''amazon-ssm-agent.

        This is the name of the Systems Manager agent that processes requests and runs commands on the instance.
        '''
        result = self._values.get("agent_name")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def source(self) -> typing.Optional[IStringVariable]:
        '''Must use one and only one of source or sourceRegion.

        The location where Systems Manager copies the SSM Agent version to install. You can't change this location.
        '''
        result = self._values.get("source")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def source_region(self) -> typing.Optional[builtins.str]:
        '''Must use one and only one of source or sourceRegion.

        If source is provided it will be used as is.
        If sourceRegion is provided it will populate source with https://s3.{sourceRegion}.amazonaws.com/aws-ssm-{sourceRegion}/manifest.json
        '''
        result = self._values.get("source_region")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def target_version(self) -> typing.Optional[IStringVariable]:
        '''A specific version of SSM Agent to install.

        If not specified, the agent will be updated to the latest version.
        '''
        result = self._values.get("target_version")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "UpdateSSMAgentStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IActionVariable)
class ActionVariable(
    StringVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ActionVariable",
):
    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="validValues")
    def valid_values(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.get(self, "validValues"))


class ApplicationsStep(
    CommandStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ApplicationsStep",
):
    '''CommandStep implemenation for aws:applications https://docs.aws.amazon.com/systems-manager/latest/userguide/ssm-plugins.html#aws-applications.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        action: IInstallUninstallRepairVariable,
        source: IStringVariable,
        parameters: typing.Optional[IStringVariable] = None,
        source_hash: typing.Optional[IStringVariable] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param action: 
        :param source: 
        :param parameters: 
        :param source_hash: 
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = ApplicationsStepProps(
            action=action,
            source=source,
            parameters=parameters,
            source_hash=source_hash,
            exit_on_failure=exit_on_failure,
            exit_on_success=exit_on_success,
            finally_step=finally_step,
            mark_success_and_exit_on_failure=mark_success_and_exit_on_failure,
            on_cancel=on_cancel,
            precondition=precondition,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="installAction")
    def install_action(self) -> IInstallUninstallRepairVariable:
        return typing.cast(IInstallUninstallRepairVariable, jsii.get(self, "installAction"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="platforms")
    def platforms(self) -> typing.List[Platform]:
        return typing.cast(typing.List[Platform], jsii.get(self, "platforms"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="source")
    def source(self) -> IStringVariable:
        return typing.cast(IStringVariable, jsii.get(self, "source"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="parameters")
    def parameters(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "parameters"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sourceHash")
    def source_hash(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "sourceHash"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ApplicationsStepProps",
    jsii_struct_bases=[CommandStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "exit_on_failure": "exitOnFailure",
        "exit_on_success": "exitOnSuccess",
        "finally_step": "finallyStep",
        "mark_success_and_exit_on_failure": "markSuccessAndExitOnFailure",
        "on_cancel": "onCancel",
        "precondition": "precondition",
        "action": "action",
        "source": "source",
        "parameters": "parameters",
        "source_hash": "sourceHash",
    },
)
class ApplicationsStepProps(CommandStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        exit_on_failure: typing.Optional[builtins.bool] = None,
        exit_on_success: typing.Optional[builtins.bool] = None,
        finally_step: typing.Optional[builtins.bool] = None,
        mark_success_and_exit_on_failure: typing.Optional[builtins.bool] = None,
        on_cancel: typing.Optional[Step] = None,
        precondition: typing.Optional[Precondition] = None,
        action: IInstallUninstallRepairVariable,
        source: IStringVariable,
        parameters: typing.Optional[IStringVariable] = None,
        source_hash: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param exit_on_failure: (Optional) Whether to exit the document execution after failed execution of this step. Finally step will be run. Default: false
        :param exit_on_success: (Optional) Whether to exit the document execution after successful execution of this step. Finally step will be run. Default: false
        :param finally_step: 
        :param mark_success_and_exit_on_failure: 
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param precondition: (Optional) A precondition to test before execution occurrs. When the precondition isn't met, the command step isn't executed. Default: undefined
        :param action: 
        :param source: 
        :param parameters: 
        :param source_hash: 
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "action": action,
            "source": source,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if exit_on_failure is not None:
            self._values["exit_on_failure"] = exit_on_failure
        if exit_on_success is not None:
            self._values["exit_on_success"] = exit_on_success
        if finally_step is not None:
            self._values["finally_step"] = finally_step
        if mark_success_and_exit_on_failure is not None:
            self._values["mark_success_and_exit_on_failure"] = mark_success_and_exit_on_failure
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if precondition is not None:
            self._values["precondition"] = precondition
        if parameters is not None:
            self._values["parameters"] = parameters
        if source_hash is not None:
            self._values["source_hash"] = source_hash

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def exit_on_failure(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after failed execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def exit_on_success(self) -> typing.Optional[builtins.bool]:
        '''(Optional) Whether to exit the document execution after successful execution of this step.

        Finally step will be run.

        :default: false
        '''
        result = self._values.get("exit_on_success")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def finally_step(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("finally_step")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def mark_success_and_exit_on_failure(self) -> typing.Optional[builtins.bool]:
        result = self._values.get("mark_success_and_exit_on_failure")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[Step]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[Step], result)

    @builtins.property
    def precondition(self) -> typing.Optional[Precondition]:
        '''(Optional) A precondition to test before execution occurrs.

        When the precondition isn't met, the command step isn't executed.

        :default: undefined
        '''
        result = self._values.get("precondition")
        return typing.cast(typing.Optional[Precondition], result)

    @builtins.property
    def action(self) -> IInstallUninstallRepairVariable:
        result = self._values.get("action")
        assert result is not None, "Required property 'action' is missing"
        return typing.cast(IInstallUninstallRepairVariable, result)

    @builtins.property
    def source(self) -> IStringVariable:
        result = self._values.get("source")
        assert result is not None, "Required property 'source' is missing"
        return typing.cast(IStringVariable, result)

    @builtins.property
    def parameters(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("parameters")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def source_hash(self) -> typing.Optional[IStringVariable]:
        result = self._values.get("source_hash")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApplicationsStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class ApproveStep(
    AutomationStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ApproveStep",
):
    '''AutomationStep implementation for aws:approve https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-approve.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        approvers: IStringListVariable,
        message: typing.Optional[IStringVariable] = None,
        min_required_approvals: typing.Optional[INumberVariable] = None,
        notification_arn: typing.Optional[IStringVariable] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param approvers: A list of AWS authenticated principals who are able to either approve or reject the action. The maximum number of approvers is 10. You can specify principals by using any of the following formats: - An AWS Identity and Access Management (IAM) user name - An IAM user ARN - An IAM role ARN - An IAM assume role user ARN
        :param message: (Optional) The information you want to include in the Amazon SNS topic when the approval request is sent. The maximum message length is 4096 characters.
        :param min_required_approvals: (Optional) The minimum number of approvals required to resume the automation. If you don't specify a value, the system defaults to one. The value for this parameter must be a positive number. The value for this parameter can't exceed the number of approvers defined by the Approvers parameter.
        :param notification_arn: (Optional) The Amazon Resource Name (ARN of an Amazon Simple Notification Service (Amazon SNS) topic for Automation approvals. When you specify an aws:approve step in a runbook, Automation sends a message to this topic letting principals know that they must either approve or reject an Automation step. The title of the Amazon SNS topic must be prefixed with "Automation".
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = ApproveStepProps(
            approvers=approvers,
            message=message,
            min_required_approvals=min_required_approvals,
            notification_arn=notification_arn,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listInputs")
    def list_inputs(self) -> typing.List[builtins.str]:
        '''Lists the inputs that are required for this step.'''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "listInputs", []))

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''Lists the outputs that will be returned from this step.'''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="provideDate")
    def _provide_date(self) -> datetime.datetime:
        '''Override to mock the date the reviewer approved.'''
        return typing.cast(datetime.datetime, jsii.invoke(self, "provideDate", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="approvers")
    def approvers(self) -> IStringListVariable:
        return typing.cast(IStringListVariable, jsii.get(self, "approvers"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="message")
    def message(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "message"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="minRequiredApprovals")
    def min_required_approvals(self) -> typing.Optional[INumberVariable]:
        return typing.cast(typing.Optional[INumberVariable], jsii.get(self, "minRequiredApprovals"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="notificationArn")
    def notification_arn(self) -> typing.Optional[IStringVariable]:
        return typing.cast(typing.Optional[IStringVariable], jsii.get(self, "notificationArn"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.ApproveStepProps",
    jsii_struct_bases=[AutomationStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "approvers": "approvers",
        "message": "message",
        "min_required_approvals": "minRequiredApprovals",
        "notification_arn": "notificationArn",
    },
)
class ApproveStepProps(AutomationStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        approvers: IStringListVariable,
        message: typing.Optional[IStringVariable] = None,
        min_required_approvals: typing.Optional[INumberVariable] = None,
        notification_arn: typing.Optional[IStringVariable] = None,
    ) -> None:
        '''Properties for ApproveStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param approvers: A list of AWS authenticated principals who are able to either approve or reject the action. The maximum number of approvers is 10. You can specify principals by using any of the following formats: - An AWS Identity and Access Management (IAM) user name - An IAM user ARN - An IAM role ARN - An IAM assume role user ARN
        :param message: (Optional) The information you want to include in the Amazon SNS topic when the approval request is sent. The maximum message length is 4096 characters.
        :param min_required_approvals: (Optional) The minimum number of approvals required to resume the automation. If you don't specify a value, the system defaults to one. The value for this parameter must be a positive number. The value for this parameter can't exceed the number of approvers defined by the Approvers parameter.
        :param notification_arn: (Optional) The Amazon Resource Name (ARN of an Amazon Simple Notification Service (Amazon SNS) topic for Automation approvals. When you specify an aws:approve step in a runbook, Automation sends a message to this topic letting principals know that they must either approve or reject an Automation step. The title of the Amazon SNS topic must be prefixed with "Automation".
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "approvers": approvers,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if message is not None:
            self._values["message"] = message
        if min_required_approvals is not None:
            self._values["min_required_approvals"] = min_required_approvals
        if notification_arn is not None:
            self._values["notification_arn"] = notification_arn

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def approvers(self) -> IStringListVariable:
        '''A list of AWS authenticated principals who are able to either approve or reject the action.

        The maximum number of approvers is 10.

        You can specify principals by using any of the following formats:

        - An AWS Identity and Access Management (IAM) user name
        - An IAM user ARN
        - An IAM role ARN
        - An IAM assume role user ARN
        '''
        result = self._values.get("approvers")
        assert result is not None, "Required property 'approvers' is missing"
        return typing.cast(IStringListVariable, result)

    @builtins.property
    def message(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The information you want to include in the Amazon SNS topic when the approval request is sent.

        The maximum message length is 4096 characters.
        '''
        result = self._values.get("message")
        return typing.cast(typing.Optional[IStringVariable], result)

    @builtins.property
    def min_required_approvals(self) -> typing.Optional[INumberVariable]:
        '''(Optional) The minimum number of approvals required to resume the automation.

        If you don't specify a value, the system defaults to one. The value for this parameter must be a positive number. The value for this parameter can't exceed the number of approvers defined by the Approvers parameter.
        '''
        result = self._values.get("min_required_approvals")
        return typing.cast(typing.Optional[INumberVariable], result)

    @builtins.property
    def notification_arn(self) -> typing.Optional[IStringVariable]:
        '''(Optional) The Amazon Resource Name (ARN of an Amazon Simple Notification Service (Amazon SNS) topic for Automation approvals.

        When you specify an aws:approve step in a runbook, Automation sends a message to this topic letting principals know that they must either approve or reject an Automation step. The title of the Amazon SNS topic must be prefixed with "Automation".
        '''
        result = self._values.get("notification_arn")
        return typing.cast(typing.Optional[IStringVariable], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "ApproveStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


class AssertAwsResourceStep(
    AwsApiStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.AssertAwsResourceStep",
):
    '''AutomationStep implementation of aws:assertAwsResourceProperty.

    https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-assertAwsResourceProperty.html
    '''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        desired_values: typing.Sequence[builtins.str],
        selector: builtins.str,
        api_params: typing.Mapping[builtins.str, typing.Any],
        pascal_case_api: builtins.str,
        service: builtins.str,
        java_script_api: typing.Optional[builtins.str] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param desired_values: (Required) Value extracted from AWS response desired to be one of these desired values.
        :param selector: (Required) Json path selector to extract value from AWS response.
        :param api_params: (Required) API Params to submit with the request to the api. You may include variables which will be substitued for inputs during step execution as such {{INPUT}}
        :param pascal_case_api: (Required) The AWS api represented in PascalCase. This value is used as-is in the SSM yaml/json. This is used as the default for javaScriptApi (see that param).
        :param service: (Required) The AWS service to be invoked.
        :param java_script_api: (Optional) The api as represented the AWS JavaScript API. This is usually lowerCamelCase. This is used in the simulation run to execute the AWS API against the JavaScript SDK. Default: - will use the camelCaseApi param and substitute the first character for lowercase by default.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = AssertAwsResourceStepProps(
            desired_values=desired_values,
            selector=selector,
            api_params=api_params,
            pascal_case_api=pascal_case_api,
            service=service,
            java_script_api=java_script_api,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @jsii.member(jsii_name="listOutputs")
    def list_outputs(self) -> typing.List[Output]:
        '''There are no outputs for this step.

        :return: empty list
        '''
        return typing.cast(typing.List[Output], jsii.invoke(self, "listOutputs", []))

    @jsii.member(jsii_name="toSsmEntry")
    def to_ssm_entry(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''Converts this step into an object to prepare for yaml/json representation of this step.'''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "toSsmEntry", []))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="desiredValues")
    def desired_values(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.get(self, "desiredValues"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="selector")
    def selector(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "selector"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.AssertAwsResourceStepProps",
    jsii_struct_bases=[AwsInvocationProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "api_params": "apiParams",
        "pascal_case_api": "pascalCaseApi",
        "service": "service",
        "java_script_api": "javaScriptApi",
        "desired_values": "desiredValues",
        "selector": "selector",
    },
)
class AssertAwsResourceStepProps(AwsInvocationProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        api_params: typing.Mapping[builtins.str, typing.Any],
        pascal_case_api: builtins.str,
        service: builtins.str,
        java_script_api: typing.Optional[builtins.str] = None,
        desired_values: typing.Sequence[builtins.str],
        selector: builtins.str,
    ) -> None:
        '''Properties for AssertAwsResourceStep.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param api_params: (Required) API Params to submit with the request to the api. You may include variables which will be substitued for inputs during step execution as such {{INPUT}}
        :param pascal_case_api: (Required) The AWS api represented in PascalCase. This value is used as-is in the SSM yaml/json. This is used as the default for javaScriptApi (see that param).
        :param service: (Required) The AWS service to be invoked.
        :param java_script_api: (Optional) The api as represented the AWS JavaScript API. This is usually lowerCamelCase. This is used in the simulation run to execute the AWS API against the JavaScript SDK. Default: - will use the camelCaseApi param and substitute the first character for lowercase by default.
        :param desired_values: (Required) Value extracted from AWS response desired to be one of these desired values.
        :param selector: (Required) Json path selector to extract value from AWS response.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "api_params": api_params,
            "pascal_case_api": pascal_case_api,
            "service": service,
            "desired_values": desired_values,
            "selector": selector,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if java_script_api is not None:
            self._values["java_script_api"] = java_script_api

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def api_params(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''(Required) API Params to submit with the request to the api.

        You may include variables which will be substitued for inputs during step execution as such {{INPUT}}

        Example::

            { 'VolumeIds': ['{{ EbsDescribeInstance.VolumeId }}'] }
        '''
        result = self._values.get("api_params")
        assert result is not None, "Required property 'api_params' is missing"
        return typing.cast(typing.Mapping[builtins.str, typing.Any], result)

    @builtins.property
    def pascal_case_api(self) -> builtins.str:
        '''(Required) The AWS api represented in PascalCase.

        This value is used as-is in the SSM yaml/json.
        This is used as the default for javaScriptApi (see that param).

        Example::

            DescribeInstances
        '''
        result = self._values.get("pascal_case_api")
        assert result is not None, "Required property 'pascal_case_api' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service(self) -> builtins.str:
        '''(Required) The AWS service to be invoked.

        Example::

            ec2
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def java_script_api(self) -> typing.Optional[builtins.str]:
        '''(Optional) The api as represented the AWS JavaScript API.

        This is usually lowerCamelCase.
        This is used in the simulation run to execute the AWS API against the JavaScript SDK.

        :default: - will use the camelCaseApi param and substitute the first character for lowercase by default.

        Example::

            describeInstances
        '''
        result = self._values.get("java_script_api")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def desired_values(self) -> typing.List[builtins.str]:
        '''(Required) Value extracted from AWS response desired to be one of these desired values.'''
        result = self._values.get("desired_values")
        assert result is not None, "Required property 'desired_values' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def selector(self) -> builtins.str:
        '''(Required) Json path selector to extract value from AWS response.'''
        result = self._values.get("selector")
        assert result is not None, "Required property 'selector' is missing"
        return typing.cast(builtins.str, result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AssertAwsResourceStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.AwsApiStepProps",
    jsii_struct_bases=[AwsInvocationProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "api_params": "apiParams",
        "pascal_case_api": "pascalCaseApi",
        "service": "service",
        "java_script_api": "javaScriptApi",
        "outputs": "outputs",
    },
)
class AwsApiStepProps(AwsInvocationProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        api_params: typing.Mapping[builtins.str, typing.Any],
        pascal_case_api: builtins.str,
        service: builtins.str,
        java_script_api: typing.Optional[builtins.str] = None,
        outputs: typing.Sequence[Output],
    ) -> None:
        '''Properties for AwsInvocation.

        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param api_params: (Required) API Params to submit with the request to the api. You may include variables which will be substitued for inputs during step execution as such {{INPUT}}
        :param pascal_case_api: (Required) The AWS api represented in PascalCase. This value is used as-is in the SSM yaml/json. This is used as the default for javaScriptApi (see that param).
        :param service: (Required) The AWS service to be invoked.
        :param java_script_api: (Optional) The api as represented the AWS JavaScript API. This is usually lowerCamelCase. This is used in the simulation run to execute the AWS API against the JavaScript SDK. Default: - will use the camelCaseApi param and substitute the first character for lowercase by default.
        :param outputs: (Required) specify the outputs to extract from the JavaScript JSON response.
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "api_params": api_params,
            "pascal_case_api": pascal_case_api,
            "service": service,
            "outputs": outputs,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if java_script_api is not None:
            self._values["java_script_api"] = java_script_api

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def api_params(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''(Required) API Params to submit with the request to the api.

        You may include variables which will be substitued for inputs during step execution as such {{INPUT}}

        Example::

            { 'VolumeIds': ['{{ EbsDescribeInstance.VolumeId }}'] }
        '''
        result = self._values.get("api_params")
        assert result is not None, "Required property 'api_params' is missing"
        return typing.cast(typing.Mapping[builtins.str, typing.Any], result)

    @builtins.property
    def pascal_case_api(self) -> builtins.str:
        '''(Required) The AWS api represented in PascalCase.

        This value is used as-is in the SSM yaml/json.
        This is used as the default for javaScriptApi (see that param).

        Example::

            DescribeInstances
        '''
        result = self._values.get("pascal_case_api")
        assert result is not None, "Required property 'pascal_case_api' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service(self) -> builtins.str:
        '''(Required) The AWS service to be invoked.

        Example::

            ec2
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def java_script_api(self) -> typing.Optional[builtins.str]:
        '''(Optional) The api as represented the AWS JavaScript API.

        This is usually lowerCamelCase.
        This is used in the simulation run to execute the AWS API against the JavaScript SDK.

        :default: - will use the camelCaseApi param and substitute the first character for lowercase by default.

        Example::

            describeInstances
        '''
        result = self._values.get("java_script_api")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def outputs(self) -> typing.List[Output]:
        '''(Required) specify the outputs to extract from the JavaScript JSON response.'''
        result = self._values.get("outputs")
        assert result is not None, "Required property 'outputs' is missing"
        return typing.cast(typing.List[Output], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "AwsApiStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IBooleanVariable)
class BooleanVariable(
    GenericVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.BooleanVariable",
):
    '''A boolean variable reference.

    Used to resolve the value from step inputs.
    '''

    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="of") # type: ignore[misc]
    @builtins.classmethod
    def of(cls, reference: builtins.str) -> "BooleanVariable":
        '''
        :param reference: -
        '''
        return typing.cast("BooleanVariable", jsii.sinvoke(cls, "of", [reference]))

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @jsii.member(jsii_name="resolveToBoolean")
    def resolve_to_boolean(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.bool:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(builtins.bool, jsii.invoke(self, "resolveToBoolean", [inputs]))


@jsii.implements(IDesiredStateVariable)
class DesiredStateVariable(
    StringVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DesiredStateVariable",
):
    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="validValues")
    def valid_values(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.get(self, "validValues"))


@jsii.implements(IDocumentHashTypeVariable)
class DocumentHashTypeVariable(
    StringVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.DocumentHashTypeVariable",
):
    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="validValues")
    def valid_values(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.get(self, "validValues"))


@jsii.implements(IBooleanVariable)
class HardCodedBoolean(
    HardCodedValueBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HardCodedBoolean",
):
    '''A hard-coded boolean variable.

    Used when not dependent on step inputs.
    '''

    def __init__(self, val: typing.Any) -> None:
        '''
        :param val: -
        '''
        jsii.create(self.__class__, self, [val])

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @jsii.member(jsii_name="resolveToBoolean")
    def resolve_to_boolean(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.bool:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(builtins.bool, jsii.invoke(self, "resolveToBoolean", [inputs]))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="FALSE")
    def FALSE(cls) -> "HardCodedBoolean":
        return typing.cast("HardCodedBoolean", jsii.sget(cls, "FALSE"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="TRUE")
    def TRUE(cls) -> "HardCodedBoolean":
        return typing.cast("HardCodedBoolean", jsii.sget(cls, "TRUE"))


@jsii.implements(IMapListVariable)
class HardCodedMapList(
    HardCodedValueBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HardCodedMapList",
):
    '''A hard-coded map list variable.

    Used when not dependent on step inputs.
    '''

    def __init__(self, val: typing.Any) -> None:
        '''
        :param val: -
        '''
        jsii.create(self.__class__, self, [val])

    @jsii.member(jsii_name="of") # type: ignore[misc]
    @builtins.classmethod
    def of(
        cls,
        val: typing.Sequence[typing.Mapping[builtins.str, typing.Any]],
    ) -> "HardCodedMapList":
        '''
        :param val: -
        '''
        return typing.cast("HardCodedMapList", jsii.sinvoke(cls, "of", [val]))

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @jsii.member(jsii_name="resolveToMapList")
    def resolve_to_map_list(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.List[typing.Mapping[builtins.str, typing.Any]]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(typing.List[typing.Mapping[builtins.str, typing.Any]], jsii.invoke(self, "resolveToMapList", [inputs]))


@jsii.implements(INumberVariable)
class HardCodedNumber(
    HardCodedValueBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HardCodedNumber",
):
    '''A hard-coded number variable.

    Used when not dependent on step inputs.
    '''

    def __init__(self, val: typing.Any) -> None:
        '''
        :param val: -
        '''
        jsii.create(self.__class__, self, [val])

    @jsii.member(jsii_name="of") # type: ignore[misc]
    @builtins.classmethod
    def of(cls, val: jsii.Number) -> "HardCodedNumber":
        '''
        :param val: -
        '''
        return typing.cast("HardCodedNumber", jsii.sinvoke(cls, "of", [val]))

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @jsii.member(jsii_name="resolveToNumber")
    def resolve_to_number(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> jsii.Number:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(jsii.Number, jsii.invoke(self, "resolveToNumber", [inputs]))


@jsii.implements(IStringVariable)
class HardCodedString(
    HardCodedValueBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HardCodedString",
):
    '''A hard-coded string variable.

    Used when not dependent on step inputs.
    '''

    def __init__(self, val: typing.Any) -> None:
        '''
        :param val: -
        '''
        jsii.create(self.__class__, self, [val])

    @jsii.member(jsii_name="of") # type: ignore[misc]
    @builtins.classmethod
    def of(cls, val: builtins.str) -> "HardCodedString":
        '''
        :param val: -
        '''
        return typing.cast("HardCodedString", jsii.sinvoke(cls, "of", [val]))

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @jsii.member(jsii_name="resolveToString")
    def resolve_to_string(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> builtins.str:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(builtins.str, jsii.invoke(self, "resolveToString", [inputs]))


@jsii.implements(IStringListVariable)
class HardCodedStringList(
    HardCodedValueBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HardCodedStringList",
):
    '''A hard-coded string list variable.

    Used when not dependent on step inputs.
    '''

    def __init__(self, val: typing.Any) -> None:
        '''
        :param val: -
        '''
        jsii.create(self.__class__, self, [val])

    @jsii.member(jsii_name="of") # type: ignore[misc]
    @builtins.classmethod
    def of(cls, val: typing.Sequence[builtins.str]) -> "HardCodedStringList":
        '''
        :param val: -
        '''
        return typing.cast("HardCodedStringList", jsii.sinvoke(cls, "of", [val]))

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @jsii.member(jsii_name="resolveToStringList")
    def resolve_to_string_list(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.List[builtins.str]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(typing.List[builtins.str], jsii.invoke(self, "resolveToStringList", [inputs]))


@jsii.implements(IStringMapVariable)
class HardCodedStringMap(
    HardCodedValueBase,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HardCodedStringMap",
):
    '''A hard-coded string map variable.

    Used when not dependent on step inputs.
    '''

    def __init__(self, val: typing.Any) -> None:
        '''
        :param val: -
        '''
        jsii.create(self.__class__, self, [val])

    @jsii.member(jsii_name="of") # type: ignore[misc]
    @builtins.classmethod
    def of(cls, val: typing.Mapping[builtins.str, typing.Any]) -> "HardCodedStringMap":
        '''
        :param val: -
        '''
        return typing.cast("HardCodedStringMap", jsii.sinvoke(cls, "of", [val]))

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @jsii.member(jsii_name="resolveToStringMap")
    def resolve_to_string_map(
        self,
        inputs: typing.Mapping[builtins.str, typing.Any],
    ) -> typing.Mapping[builtins.str, typing.Any]:
        '''Given the execution inputs, return the resolved value of this variable.

        :param inputs: -
        '''
        return typing.cast(typing.Mapping[builtins.str, typing.Any], jsii.invoke(self, "resolveToStringMap", [inputs]))


@jsii.implements(IInstallUninstallRepairVariable)
class InstallUninstallRepairVariable(
    StringVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.InstallUninstallRepairVariable",
):
    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="validValues")
    def valid_values(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.get(self, "validValues"))


@jsii.implements(IInstallationTypeVariable)
class InstallationTypeVariable(
    StringVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.InstallationTypeVariable",
):
    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="validValues")
    def valid_values(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.get(self, "validValues"))


@jsii.implements(IOnFailureVariable)
class OnFailureVariable(
    StringVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.OnFailureVariable",
):
    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="validValues")
    def valid_values(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.get(self, "validValues"))


@jsii.implements(IPackageNameVariable)
class PackageNameVariable(
    StringVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.PackageNameVariable",
):
    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="validValues")
    def valid_values(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.get(self, "validValues"))


@jsii.implements(IResourceTypeVariable)
class ResourceTypeVariable(
    StringVariable,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.ResourceTypeVariable",
):
    def __init__(self, reference: builtins.str) -> None:
        '''
        :param reference: -
        '''
        jsii.create(self.__class__, self, [reference])

    @jsii.member(jsii_name="assertType")
    def _assert_type(self, value: typing.Any) -> None:
        '''
        :param value: -
        '''
        return typing.cast(None, jsii.invoke(self, "assertType", [value]))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="validValues")
    def valid_values(self) -> typing.List[builtins.str]:
        return typing.cast(typing.List[builtins.str], jsii.get(self, "validValues"))


class WaitForResourceStep(
    AssertAwsResourceStep,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.WaitForResourceStep",
):
    '''AutomationStep impl for aws:waitForAwsResourceProperty https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-waitForAwsResourceProperty.html.'''

    def __init__(
        self,
        scope: constructs.Construct,
        id: builtins.str,
        *,
        sleep_interval_millis: typing.Optional[jsii.Number] = None,
        desired_values: typing.Sequence[builtins.str],
        selector: builtins.str,
        api_params: typing.Mapping[builtins.str, typing.Any],
        pascal_case_api: builtins.str,
        service: builtins.str,
        java_script_api: typing.Optional[builtins.str] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
    ) -> None:
        '''
        :param scope: -
        :param id: -
        :param sleep_interval_millis: (Optional) How much time to sleep after not receiving desired response from AWS SDK. This is only used for the run simulation. This is not used by SSM and is therefore not declared in the SSM yaml/json. Default: 2000 millis
        :param desired_values: (Required) Value extracted from AWS response desired to be one of these desired values.
        :param selector: (Required) Json path selector to extract value from AWS response.
        :param api_params: (Required) API Params to submit with the request to the api. You may include variables which will be substitued for inputs during step execution as such {{INPUT}}
        :param pascal_case_api: (Required) The AWS api represented in PascalCase. This value is used as-is in the SSM yaml/json. This is used as the default for javaScriptApi (see that param).
        :param service: (Required) The AWS service to be invoked.
        :param java_script_api: (Optional) The api as represented the AWS JavaScript API. This is usually lowerCamelCase. This is used in the simulation run to execute the AWS API against the JavaScript SDK. Default: - will use the camelCaseApi param and substitute the first character for lowercase by default.
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        '''
        props = WaitForResourceStepProps(
            sleep_interval_millis=sleep_interval_millis,
            desired_values=desired_values,
            selector=selector,
            api_params=api_params,
            pascal_case_api=pascal_case_api,
            service=service,
            java_script_api=java_script_api,
            is_end=is_end,
            max_attempts=max_attempts,
            on_cancel=on_cancel,
            on_failure=on_failure,
            timeout_seconds=timeout_seconds,
            description=description,
            input_observer=input_observer,
            name=name,
            output_observer=output_observer,
        )

        jsii.create(self.__class__, self, [scope, id, props])

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="action")
    def action(self) -> builtins.str:
        return typing.cast(builtins.str, jsii.get(self, "action"))

    @builtins.property # type: ignore[misc]
    @jsii.member(jsii_name="sleepIntervalMillis")
    def sleep_interval_millis(self) -> jsii.Number:
        return typing.cast(jsii.Number, jsii.get(self, "sleepIntervalMillis"))


@jsii.data_type(
    jsii_type="@cdklabs/cdk-ssm-documents.WaitForResourceStepProps",
    jsii_struct_bases=[AssertAwsResourceStepProps],
    name_mapping={
        "description": "description",
        "input_observer": "inputObserver",
        "name": "name",
        "output_observer": "outputObserver",
        "is_end": "isEnd",
        "max_attempts": "maxAttempts",
        "on_cancel": "onCancel",
        "on_failure": "onFailure",
        "timeout_seconds": "timeoutSeconds",
        "api_params": "apiParams",
        "pascal_case_api": "pascalCaseApi",
        "service": "service",
        "java_script_api": "javaScriptApi",
        "desired_values": "desiredValues",
        "selector": "selector",
        "sleep_interval_millis": "sleepIntervalMillis",
    },
)
class WaitForResourceStepProps(AssertAwsResourceStepProps):
    def __init__(
        self,
        *,
        description: typing.Optional[builtins.str] = None,
        input_observer: typing.Optional[IObserver] = None,
        name: typing.Optional[builtins.str] = None,
        output_observer: typing.Optional[IObserver] = None,
        is_end: typing.Optional[builtins.bool] = None,
        max_attempts: typing.Optional[jsii.Number] = None,
        on_cancel: typing.Optional[AutomationStep] = None,
        on_failure: typing.Optional[AutomationStep] = None,
        timeout_seconds: typing.Optional[jsii.Number] = None,
        api_params: typing.Mapping[builtins.str, typing.Any],
        pascal_case_api: builtins.str,
        service: builtins.str,
        java_script_api: typing.Optional[builtins.str] = None,
        desired_values: typing.Sequence[builtins.str],
        selector: builtins.str,
        sleep_interval_millis: typing.Optional[jsii.Number] = None,
    ) -> None:
        '''
        :param description: (Optional) description of the current step. Default: undefined
        :param input_observer: (Optional) Allows for observing the input to steps as they run. Default: NoopObserver
        :param name: (Optional) Name of the current step. The name will be prepended onto all of the outputs emitted from this step. This name will also be used to reference this step in logs. Defaults to the id of the CDK node.
        :param output_observer: (Optional) Allows for observing the output of steps as they run. Default: NoopObserver
        :param is_end: Whether to stop document execution after this step. Default: false
        :param max_attempts: (Optional) max attempts to run this step if there are failures. Default: Step.DEFAULT_MAX_ATTEMPTS
        :param on_cancel: (Optional) Step to jump to in the event that this step is cancelled. Default: undefined
        :param on_failure: (Optional) Step to jump to in the event that this step fails. Default: undefined
        :param timeout_seconds: (Optional) timeout seconds to run this step. In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step. Default: Step.DEFAULT_TIMEOUT
        :param api_params: (Required) API Params to submit with the request to the api. You may include variables which will be substitued for inputs during step execution as such {{INPUT}}
        :param pascal_case_api: (Required) The AWS api represented in PascalCase. This value is used as-is in the SSM yaml/json. This is used as the default for javaScriptApi (see that param).
        :param service: (Required) The AWS service to be invoked.
        :param java_script_api: (Optional) The api as represented the AWS JavaScript API. This is usually lowerCamelCase. This is used in the simulation run to execute the AWS API against the JavaScript SDK. Default: - will use the camelCaseApi param and substitute the first character for lowercase by default.
        :param desired_values: (Required) Value extracted from AWS response desired to be one of these desired values.
        :param selector: (Required) Json path selector to extract value from AWS response.
        :param sleep_interval_millis: (Optional) How much time to sleep after not receiving desired response from AWS SDK. This is only used for the run simulation. This is not used by SSM and is therefore not declared in the SSM yaml/json. Default: 2000 millis
        '''
        self._values: typing.Dict[str, typing.Any] = {
            "api_params": api_params,
            "pascal_case_api": pascal_case_api,
            "service": service,
            "desired_values": desired_values,
            "selector": selector,
        }
        if description is not None:
            self._values["description"] = description
        if input_observer is not None:
            self._values["input_observer"] = input_observer
        if name is not None:
            self._values["name"] = name
        if output_observer is not None:
            self._values["output_observer"] = output_observer
        if is_end is not None:
            self._values["is_end"] = is_end
        if max_attempts is not None:
            self._values["max_attempts"] = max_attempts
        if on_cancel is not None:
            self._values["on_cancel"] = on_cancel
        if on_failure is not None:
            self._values["on_failure"] = on_failure
        if timeout_seconds is not None:
            self._values["timeout_seconds"] = timeout_seconds
        if java_script_api is not None:
            self._values["java_script_api"] = java_script_api
        if sleep_interval_millis is not None:
            self._values["sleep_interval_millis"] = sleep_interval_millis

    @builtins.property
    def description(self) -> typing.Optional[builtins.str]:
        '''(Optional) description of the current step.

        :default: undefined
        '''
        result = self._values.get("description")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def input_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the input to steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("input_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def name(self) -> typing.Optional[builtins.str]:
        '''(Optional) Name of the current step.

        The name will be prepended onto all of the outputs emitted from this step.
        This name will also be used to reference this step in logs.
        Defaults to the id of the CDK node.
        '''
        result = self._values.get("name")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def output_observer(self) -> typing.Optional[IObserver]:
        '''(Optional) Allows for observing the output of steps as they run.

        :default: NoopObserver
        '''
        result = self._values.get("output_observer")
        return typing.cast(typing.Optional[IObserver], result)

    @builtins.property
    def is_end(self) -> typing.Optional[builtins.bool]:
        '''Whether to stop document execution after this step.

        :default: false
        '''
        result = self._values.get("is_end")
        return typing.cast(typing.Optional[builtins.bool], result)

    @builtins.property
    def max_attempts(self) -> typing.Optional[jsii.Number]:
        '''(Optional) max attempts to run this step if there are failures.

        :default: Step.DEFAULT_MAX_ATTEMPTS
        '''
        result = self._values.get("max_attempts")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def on_cancel(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step is cancelled.

        :default: undefined
        '''
        result = self._values.get("on_cancel")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def on_failure(self) -> typing.Optional[AutomationStep]:
        '''(Optional) Step to jump to in the event that this step fails.

        :default: undefined
        '''
        result = self._values.get("on_failure")
        return typing.cast(typing.Optional[AutomationStep], result)

    @builtins.property
    def timeout_seconds(self) -> typing.Optional[jsii.Number]:
        '''(Optional) timeout seconds to run this step.

        In a simulation run, this will only be encorced after-the-fact but execution will not be stopped mid-step.

        :default: Step.DEFAULT_TIMEOUT
        '''
        result = self._values.get("timeout_seconds")
        return typing.cast(typing.Optional[jsii.Number], result)

    @builtins.property
    def api_params(self) -> typing.Mapping[builtins.str, typing.Any]:
        '''(Required) API Params to submit with the request to the api.

        You may include variables which will be substitued for inputs during step execution as such {{INPUT}}

        Example::

            { 'VolumeIds': ['{{ EbsDescribeInstance.VolumeId }}'] }
        '''
        result = self._values.get("api_params")
        assert result is not None, "Required property 'api_params' is missing"
        return typing.cast(typing.Mapping[builtins.str, typing.Any], result)

    @builtins.property
    def pascal_case_api(self) -> builtins.str:
        '''(Required) The AWS api represented in PascalCase.

        This value is used as-is in the SSM yaml/json.
        This is used as the default for javaScriptApi (see that param).

        Example::

            DescribeInstances
        '''
        result = self._values.get("pascal_case_api")
        assert result is not None, "Required property 'pascal_case_api' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def service(self) -> builtins.str:
        '''(Required) The AWS service to be invoked.

        Example::

            ec2
        '''
        result = self._values.get("service")
        assert result is not None, "Required property 'service' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def java_script_api(self) -> typing.Optional[builtins.str]:
        '''(Optional) The api as represented the AWS JavaScript API.

        This is usually lowerCamelCase.
        This is used in the simulation run to execute the AWS API against the JavaScript SDK.

        :default: - will use the camelCaseApi param and substitute the first character for lowercase by default.

        Example::

            describeInstances
        '''
        result = self._values.get("java_script_api")
        return typing.cast(typing.Optional[builtins.str], result)

    @builtins.property
    def desired_values(self) -> typing.List[builtins.str]:
        '''(Required) Value extracted from AWS response desired to be one of these desired values.'''
        result = self._values.get("desired_values")
        assert result is not None, "Required property 'desired_values' is missing"
        return typing.cast(typing.List[builtins.str], result)

    @builtins.property
    def selector(self) -> builtins.str:
        '''(Required) Json path selector to extract value from AWS response.'''
        result = self._values.get("selector")
        assert result is not None, "Required property 'selector' is missing"
        return typing.cast(builtins.str, result)

    @builtins.property
    def sleep_interval_millis(self) -> typing.Optional[jsii.Number]:
        '''(Optional) How much time to sleep after not receiving desired response from AWS SDK.

        This is only used for the run simulation.
        This is not used by SSM and is therefore not declared in the SSM yaml/json.

        :default: 2000 millis
        '''
        result = self._values.get("sleep_interval_millis")
        return typing.cast(typing.Optional[jsii.Number], result)

    def __eq__(self, rhs: typing.Any) -> builtins.bool:
        return isinstance(rhs, self.__class__) and rhs._values == self._values

    def __ne__(self, rhs: typing.Any) -> builtins.bool:
        return not (rhs == self)

    def __repr__(self) -> str:
        return "WaitForResourceStepProps(%s)" % ", ".join(
            k + "=" + repr(v) for k, v in self._values.items()
        )


@jsii.implements(IActionVariable)
class HardCodedAction(
    HardCodedString,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HardCodedAction",
):
    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="INSTALL")
    def INSTALL(cls) -> "HardCodedAction":
        return typing.cast("HardCodedAction", jsii.sget(cls, "INSTALL"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="UNINSTALL")
    def UNINSTALL(cls) -> "HardCodedAction":
        return typing.cast("HardCodedAction", jsii.sget(cls, "UNINSTALL"))


@jsii.implements(IDesiredStateVariable)
class HardCodedDesiredState(
    HardCodedString,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HardCodedDesiredState",
):
    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="RUNNING")
    def RUNNING(cls) -> "HardCodedDesiredState":
        return typing.cast("HardCodedDesiredState", jsii.sget(cls, "RUNNING"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="STOPPED")
    def STOPPED(cls) -> "HardCodedDesiredState":
        return typing.cast("HardCodedDesiredState", jsii.sget(cls, "STOPPED"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="TERMINATED")
    def TERMINATED(cls) -> "HardCodedDesiredState":
        return typing.cast("HardCodedDesiredState", jsii.sget(cls, "TERMINATED"))


@jsii.implements(IDocumentHashTypeVariable)
class HardCodedDocumentHashType(
    HardCodedString,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HardCodedDocumentHashType",
):
    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="SHA1")
    def SHA1(cls) -> "HardCodedDocumentHashType":
        return typing.cast("HardCodedDocumentHashType", jsii.sget(cls, "SHA1"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="SHA256")
    def SHA256(cls) -> "HardCodedDocumentHashType":
        return typing.cast("HardCodedDocumentHashType", jsii.sget(cls, "SHA256"))


@jsii.implements(IInstallUninstallRepairVariable)
class HardCodedInstallUninstallRepair(
    HardCodedString,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HardCodedInstallUninstallRepair",
):
    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="INSTALL")
    def INSTALL(cls) -> "HardCodedInstallUninstallRepair":
        return typing.cast("HardCodedInstallUninstallRepair", jsii.sget(cls, "INSTALL"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="REPAIR")
    def REPAIR(cls) -> "HardCodedInstallUninstallRepair":
        return typing.cast("HardCodedInstallUninstallRepair", jsii.sget(cls, "REPAIR"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="UNINSTALL")
    def UNINSTALL(cls) -> "HardCodedInstallUninstallRepair":
        return typing.cast("HardCodedInstallUninstallRepair", jsii.sget(cls, "UNINSTALL"))


@jsii.implements(IInstallationTypeVariable)
class HardCodedInstallationType(
    HardCodedString,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HardCodedInstallationType",
):
    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="IN_PLACE_UPDATE")
    def IN_PLACE_UPDATE(cls) -> "HardCodedInstallationType":
        return typing.cast("HardCodedInstallationType", jsii.sget(cls, "IN_PLACE_UPDATE"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="UNINSTALL_AND_REINSTALL")
    def UNINSTALL_AND_REINSTALL(cls) -> "HardCodedInstallationType":
        return typing.cast("HardCodedInstallationType", jsii.sget(cls, "UNINSTALL_AND_REINSTALL"))


@jsii.implements(IOnFailureVariable)
class HardCodedOnFailure(
    HardCodedString,
    metaclass=jsii.JSIIMeta,
    jsii_type="@cdklabs/cdk-ssm-documents.HardCodedOnFailure",
):
    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="DELETE")
    def DELETE(cls) -> "HardCodedOnFailure":
        return typing.cast("HardCodedOnFailure", jsii.sget(cls, "DELETE"))

    @jsii.python.classproperty # type: ignore[misc]
    @jsii.member(jsii_name="DO_NOTHING")
    def DO_NOTHING(cls) -> "HardCodedOnFailure":
        return typing.cast("HardCodedOnFailure", jsii.sget(cls, "DO_NOTHING"))

    @jsii.py