"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.parseObservableMarble = void 0;
const ObservableMarbleToken_1 = require("./ObservableMarbleToken");
const SubscriptionMarbleToken_1 = require("./SubscriptionMarbleToken");
const tokenParseReducer_1 = require("./tokenParseReducer");
/**
 * Parse marble DSL diagram, generates array of TestMessageValue for metadata of each marble values to be scheduled into.
 *
 * @param {string} marble Marble diagram to parse
 * @param {{ [key: string]: T }} [value] Custom value for marble value
 * @param {any} [error] Custom error for marble error
 * @param {boolean} [materializeInnerObservables] Flatten inner observables in cold observable. False by default.
 * @param {number} [frameTimeFactor] Custom frametime factor for virtual time frame. 1 by default.
 */
const parseObservableMarble = (marble, value, error, materializeInnerObservables = false, frameTimeFactor = 1, maxFrame = 1000) => {
    if (marble.indexOf(SubscriptionMarbleToken_1.SubscriptionMarbleToken.UNSUBSCRIBE) !== -1) {
        throw new Error(`Observable marble cannot have unsubscription marker ${SubscriptionMarbleToken_1.SubscriptionMarbleToken.UNSUBSCRIBE}`);
    }
    const marbleTokenArray = Array.from(marble).filter(token => token !== ObservableMarbleToken_1.ObservableMarbleToken.NOOP);
    const subscriptionIndex = marbleTokenArray.join('').indexOf(SubscriptionMarbleToken_1.SubscriptionMarbleToken.SUBSCRIBE) * frameTimeFactor;
    const frameOffset = subscriptionIndex < 0 ? 0 : -subscriptionIndex;
    const values = marbleTokenArray.reduce(tokenParseReducer_1.observableTokenParseReducer(value || null, error, materializeInnerObservables, frameTimeFactor, maxFrame), {
        currentTimeFrame: frameOffset,
        messages: [],
        simultaneousGrouped: false,
        expandingTokenCount: 0,
        expandingValue: []
    });
    return values.messages;
};
exports.parseObservableMarble = parseObservableMarble;
//# sourceMappingURL=parseObservableMarble.js.map