import { Context } from '../context';
import { ContractAbstraction, ContractMethod } from '../contract';
import { OpKind, withKind } from '../operations/types';
import { WalletDelegateParams, WalletOriginateParams, WalletProvider, WalletTransferParams } from './interface';
export interface PKHOption {
    forceRefetch?: boolean;
}
export declare type WalletParamsWithKind = withKind<WalletTransferParams, OpKind.TRANSACTION> | withKind<WalletOriginateParams, OpKind.ORIGINATION> | withKind<WalletDelegateParams, OpKind.DELEGATION>;
export declare class WalletOperationBatch {
    private walletProvider;
    private context;
    private operations;
    constructor(walletProvider: WalletProvider, context: Context);
    /**
     *
     * @description Add a transaction operation to the batch
     *
     * @param params Transfer operation parameter
     */
    withTransfer(params: WalletTransferParams): this;
    /**
     *
     * @description Add a transaction operation to the batch
     *
     * @param params Transfer operation parameter
     */
    withContractCall(params: ContractMethod<Wallet>): this;
    /**
     *
     * @description Add a delegation operation to the batch
     *
     * @param params Delegation operation parameter
     */
    withDelegation(params: WalletDelegateParams): this;
    /**
     *
     * @description Add an origination operation to the batch
     *
     * @param params Origination operation parameter
     */
    withOrigination(params: WalletOriginateParams): this;
    private mapOperation;
    /**
     *
     * @description Add a group operation to the batch. Operation will be applied in the order they are in the params array
     *
     * @param params Operations parameter
     */
    with(params: WalletParamsWithKind[]): this;
    /**
     *
     * @description Submit batch operation to wallet
     *
     */
    send(): Promise<import("./operation").WalletOperation>;
}
export declare class Wallet {
    private context;
    constructor(context: Context);
    private get walletProvider();
    private _pkh?;
    /**
     * @description Retrieve the PKH of the account that is currently in use by the wallet
     *
     * @param option Option to use while fetching the PKH.
     * If forceRefetch is specified the wallet provider implementation will refetch the PKH from the wallet
     */
    pkh({ forceRefetch }?: PKHOption): Promise<string>;
    private walletCommand;
    /**
     *
     * @description Originate a new contract according to the script in parameters.
     *
     * @returns An operation handle with the result from the rpc node
     *
     * @param originateParams Originate operation parameter
     */
    originate(params: WalletOriginateParams): {
        send: () => Promise<import("./origination-operation").OriginationWalletOperation>;
    };
    /**
     *
     * @description Set the delegate for a contract.
     *
     * @returns An operation handle with the result from the rpc node
     *
     * @param delegateParams operation parameter
     */
    setDelegate(params: WalletDelegateParams): {
        send: () => Promise<import("./delegation-operation").DelegationWalletOperation>;
    };
    /**
     *
     * @description Register the current address as delegate.
     *
     * @returns An operation handle with the result from the rpc node
     *
     */
    registerDelegate(): {
        send: () => Promise<import("./delegation-operation").DelegationWalletOperation>;
    };
    /**
     *
     * @description Transfer tezos tokens from current address to a specific address or call a smart contract.
     *
     * @returns A wallet command from which we can send the operation to the wallet
     *
     * @param params operation parameter
     */
    transfer(params: WalletTransferParams): {
        send: () => Promise<import("./transaction-operation").TransactionWalletOperation>;
    };
    /**
     *
     * @description Create a batch of operation
     *
     * @returns A batch object from which we can add more operation or send a command to the wallet to execute the batch
     *
     * @param params List of operation to initialize the batch with
     */
    batch(params?: Parameters<WalletOperationBatch['with']>[0]): WalletOperationBatch;
    /**
     *
     * @description Create an smart contract abstraction for the address specified. Calling entrypoints with the returned
     * smart contract abstraction will leverage the wallet provider to make smart contract calls
     *
     * @param address Smart contract address
     */
    at<T extends ContractAbstraction<Wallet>>(address: string, contractAbstractionComposer?: (abs: ContractAbstraction<Wallet>, context: Context) => T): Promise<T>;
}
