"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const acm = require("@aws-cdk/aws-certificatemanager");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
let app;
let stack;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
});
test('minimal example renders correctly', () => {
    const origin = defaultOrigin();
    new lib_1.Distribution(stack, 'MyDist', { defaultBehavior: { origin } });
    expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            DefaultCacheBehavior: {
                ForwardedValues: { QueryString: false },
                TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                ViewerProtocolPolicy: 'allow-all',
            },
            Enabled: true,
            Origins: [{
                    DomainName: 'www.example.com',
                    Id: 'StackMyDistOrigin1D6D5E535',
                    CustomOriginConfig: {
                        OriginProtocolPolicy: 'https-only',
                    },
                }],
        },
    });
});
describe('multiple behaviors', () => {
    test('a second behavior can\'t be specified with the catch-all path pattern', () => {
        const origin = defaultOrigin();
        expect(() => {
            new lib_1.Distribution(stack, 'MyDist', {
                defaultBehavior: { origin },
                additionalBehaviors: {
                    '*': { origin },
                },
            });
        }).toThrow(/Only the default behavior can have a path pattern of \'*\'/);
    });
    test('a second behavior can be added to the original origin', () => {
        const origin = defaultOrigin();
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/*': { origin },
            },
        });
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                Origins: [{
                        DomainName: 'www.example.com',
                        Id: 'StackMyDistOrigin1D6D5E535',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    }],
            },
        });
    });
    test('a second behavior can be added to a secondary origin', () => {
        const origin = defaultOrigin();
        const origin2 = defaultOrigin('origin2.example.com');
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/*': { origin: origin2 },
            },
        });
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin20B96F3AD',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                Origins: [{
                        DomainName: 'www.example.com',
                        Id: 'StackMyDistOrigin1D6D5E535',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    },
                    {
                        DomainName: 'origin2.example.com',
                        Id: 'StackMyDistOrigin20B96F3AD',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    }],
            },
        });
    });
    test('behavior creation order is preserved', () => {
        const origin = defaultOrigin();
        const origin2 = defaultOrigin('origin2.example.com');
        const dist = new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'api/1*': { origin: origin2 },
            },
        });
        dist.addBehavior('api/2*', origin);
        expect(stack).toHaveResource('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    ForwardedValues: { QueryString: false },
                    TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                    ViewerProtocolPolicy: 'allow-all',
                },
                CacheBehaviors: [{
                        PathPattern: 'api/1*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin20B96F3AD',
                        ViewerProtocolPolicy: 'allow-all',
                    },
                    {
                        PathPattern: 'api/2*',
                        ForwardedValues: { QueryString: false },
                        TargetOriginId: 'StackMyDistOrigin1D6D5E535',
                        ViewerProtocolPolicy: 'allow-all',
                    }],
                Enabled: true,
                Origins: [{
                        DomainName: 'www.example.com',
                        Id: 'StackMyDistOrigin1D6D5E535',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    },
                    {
                        DomainName: 'origin2.example.com',
                        Id: 'StackMyDistOrigin20B96F3AD',
                        CustomOriginConfig: {
                            OriginProtocolPolicy: 'https-only',
                        },
                    }],
            },
        });
    });
});
describe('certificates', () => {
    test('should fail if using an imported certificate from outside of us-east-1', () => {
        const origin = defaultOrigin();
        const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:eu-west-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                certificate,
            });
        }).toThrow(/Distribution certificates must be in the us-east-1 region and the certificate you provided is in eu-west-1./);
    });
    test('adding a certificate renders the correct ViewerCertificate property', () => {
        const certificate = acm.Certificate.fromCertificateArn(stack, 'Cert', 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012');
        new lib_1.Distribution(stack, 'Dist', {
            defaultBehavior: { origin: defaultOrigin() },
            certificate,
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                ViewerCertificate: {
                    AcmCertificateArn: 'arn:aws:acm:us-east-1:123456789012:certificate/12345678-1234-1234-1234-123456789012',
                    SslSupportMethod: 'sni-only',
                    MinimumProtocolVersion: 'TLSv1.2_2018',
                },
            },
        });
    });
});
describe('custom error responses', () => {
    test('should fail if responsePagePath is defined but responseCode is not', () => {
        const origin = defaultOrigin();
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                errorResponses: [{
                        httpStatus: 404,
                        responsePagePath: '/errors/404.html',
                    }],
            });
        }).toThrow(/\'responseCode\' must be provided if \'responsePagePath\' is defined/);
    });
    test('should fail if only the error code is provided', () => {
        const origin = defaultOrigin();
        expect(() => {
            new lib_1.Distribution(stack, 'Dist', {
                defaultBehavior: { origin },
                errorResponses: [{ httpStatus: 404 }],
            });
        }).toThrow(/A custom error response without either a \'responseCode\' or \'errorCachingMinTtl\' is not valid./);
    });
    test('should render the array of error configs if provided', () => {
        const origin = defaultOrigin();
        new lib_1.Distribution(stack, 'Dist', {
            defaultBehavior: { origin },
            errorResponses: [{
                    httpStatus: 404,
                    responseHttpStatus: 404,
                    responsePagePath: '/errors/404.html',
                },
                {
                    httpStatus: 500,
                    ttl: core_1.Duration.seconds(2),
                }],
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                CustomErrorResponses: [
                    {
                        ErrorCode: 404,
                        ResponseCode: 404,
                        ResponsePagePath: '/errors/404.html',
                    },
                    {
                        ErrorCachingMinTTL: 2,
                        ErrorCode: 500,
                    },
                ],
            },
        });
    });
});
describe('with Lambda@Edge functions', () => {
    let lambdaFunction;
    let origin;
    beforeEach(() => {
        lambdaFunction = new lambda.Function(stack, 'Function', {
            runtime: lambda.Runtime.NODEJS,
            code: lambda.Code.fromInline('whatever'),
            handler: 'index.handler',
        });
        origin = defaultOrigin();
    });
    test('can add an edge lambdas to the default behavior', () => {
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: {
                origin,
                edgeLambdas: [
                    {
                        functionVersion: lambdaFunction.currentVersion,
                        eventType: lib_1.LambdaEdgeEventType.ORIGIN_REQUEST,
                    },
                ],
            },
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                DefaultCacheBehavior: {
                    LambdaFunctionAssociations: [
                        {
                            EventType: 'origin-request',
                            LambdaFunctionARN: {
                                Ref: 'FunctionCurrentVersion4E2B2261477a5ae8059bbaa7813f752292c0f65e',
                            },
                        },
                    ],
                },
            },
        });
    });
    test('can add an edge lambdas to additional behaviors', () => {
        new lib_1.Distribution(stack, 'MyDist', {
            defaultBehavior: { origin },
            additionalBehaviors: {
                'images/*': {
                    origin,
                    edgeLambdas: [
                        {
                            functionVersion: lambdaFunction.currentVersion,
                            eventType: lib_1.LambdaEdgeEventType.VIEWER_REQUEST,
                        },
                    ],
                },
            },
        });
        expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
            DistributionConfig: {
                CacheBehaviors: [
                    {
                        PathPattern: 'images/*',
                        LambdaFunctionAssociations: [
                            {
                                EventType: 'viewer-request',
                                LambdaFunctionARN: {
                                    Ref: 'FunctionCurrentVersion4E2B2261477a5ae8059bbaa7813f752292c0f65e',
                                },
                            },
                        ],
                    },
                ],
            },
        });
    });
    test('fails creation when attempting to add the $LATEST function version as an edge Lambda to the default behavior', () => {
        expect(() => {
            new lib_1.Distribution(stack, 'MyDist', {
                defaultBehavior: {
                    origin,
                    edgeLambdas: [
                        {
                            functionVersion: lambdaFunction.latestVersion,
                            eventType: lib_1.LambdaEdgeEventType.ORIGIN_RESPONSE,
                        },
                    ],
                },
            });
        }).toThrow(/\$LATEST function version cannot be used for Lambda@Edge/);
    });
});
test('price class is included if provided', () => {
    const origin = defaultOrigin();
    new lib_1.Distribution(stack, 'Dist', {
        defaultBehavior: { origin },
        priceClass: lib_1.PriceClass.PRICE_CLASS_200,
    });
    expect(stack).toHaveResourceLike('AWS::CloudFront::Distribution', {
        DistributionConfig: {
            PriceClass: 'PriceClass_200',
        },
    });
});
function defaultOrigin(domainName) {
    return new TestOrigin(domainName !== null && domainName !== void 0 ? domainName : 'www.example.com');
}
class TestOrigin extends lib_1.OriginBase {
    constructor(domainName, props = {}) { super(domainName, props); }
    renderCustomOriginConfig() {
        return { originProtocolPolicy: lib_1.OriginProtocolPolicy.HTTPS_ONLY };
    }
}
//# sourceMappingURL=data:application/json;base64,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