"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginBase = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 *
 * @experimental
 */
class OriginBase {
    constructor(domainName, props = {}) {
        var _a;
        validateIntInRangeOrUndefined('connectionTimeout', 1, 10, (_a = props.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds());
        validateIntInRangeOrUndefined('connectionAttempts', 1, 3, props.connectionAttempts, false);
        this.domainName = domainName;
        this.originPath = this.validateOriginPath(props.originPath);
        this.connectionTimeout = props.connectionTimeout;
        this.connectionAttempts = props.connectionAttempts;
        this.customHeaders = props.customHeaders;
    }
    /**
     * Binds the origin to the associated Distribution. Can be used to grant permissions, create dependent resources, etc.
     */
    bind(_scope, options) {
        var _a;
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return { originProperty: {
                domainName: this.domainName,
                id: options.originId,
                originPath: this.originPath,
                connectionAttempts: this.connectionAttempts,
                connectionTimeout: (_a = this.connectionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                originCustomHeaders: this.renderCustomHeaders(),
                s3OriginConfig,
                customOriginConfig,
            } };
    }
    // Overridden by sub-classes to provide S3 origin config.
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    renderCustomOriginConfig() {
        return undefined;
    }
    renderCustomHeaders() {
        if (!this.customHeaders || Object.entries(this.customHeaders).length === 0) {
            return undefined;
        }
        return Object.entries(this.customHeaders).map(([headerName, headerValue]) => {
            return { headerName, headerValue };
        });
    }
    /**
     * If the path is defined, it must start with a '/' and not end with a '/'.
     * This method takes in the originPath, and returns it back (if undefined) or adds/removes the '/' as appropriate.
     */
    validateOriginPath(originPath) {
        if (core_1.Token.isUnresolved(originPath)) {
            return originPath;
        }
        if (originPath === undefined) {
            return undefined;
        }
        let path = originPath;
        if (!path.startsWith('/')) {
            path = '/' + path;
        }
        if (path.endsWith('/')) {
            path = path.substr(0, path.length - 1);
        }
        return path;
    }
}
exports.OriginBase = OriginBase;
/**
 * Throws an error if a value is defined and not an integer or not in a range.
 */
function validateIntInRangeOrUndefined(name, min, max, value, isDuration = true) {
    if (value === undefined) {
        return;
    }
    if (!Number.isInteger(value) || value < min || value > max) {
        const seconds = isDuration ? ' seconds' : '';
        throw new Error(`${name}: Must be an int between ${min} and ${max}${seconds} (inclusive); received ${value}.`);
    }
}
//# sourceMappingURL=data:application/json;base64,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