'''
Datasets

This file contains several datasets that are useful for testing and experimenting
To import you can use the following format:
    from reliability.Datasets import automotive
    failures = automotive().failures
    right_censored = automotive().right_censored

If you would like more information on a dataset type:
    from reliability.Datasets import automotive
    print(automotive().info)
    print(help(automotive))
'''

import pandas as pd
import numpy as np


class automotive:
    '''
    This dataset is relatively small and a challenging task to fit with any distribution due to its size and shape
    It also includes mostly right censored data which makes fitting more difficult.
    Sourced (with permission) from: V.V. Krivtsov and J. W. Case (1999), Peculiarities of Censored Data Analysis in Automotive Industry Applications - SAE Technical Paper Series, # 1999-01-3220
    '''

    def __init__(self):
        self.failures = [5248, 7454, 16890, 17200, 38700, 45000, 49390, 69040, 72280, 131900]
        self.right_censored = [3961, 4007, 4734, 6054, 7298, 10190, 23060, 27160, 28690, 37100, 40060, 45670, 53000, 67000, 69630, 77350, 78470, 91680, 105700, 106300, 150400]
        rc = len(self.right_censored)
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored'], 'Value': ['automotive', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)')]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class defective_sample:
    '''
    This dataset is heavily right censored with intermixed censoring (not all censored values are greater than the largest failure)
    It exhibits the behavior of a defective sample (aka. Limited fraction defective)
    Thanks to Alexander Davis for providing this dataset.
    '''

    def __init__(self):
        self.failures = [81, 163, 56, 86, 144, 47, 121, 56, 43, 106, 54, 50, 43, 107, 107, 22, 106, 106, 106, 71, 71, 68, 65, 65, 99, 70, 23, 34, 217, 81, 95, 78, 4, 183, 117, 36, 22, 5, 106, 36, 96, 6, 32, 239, 31, 232, 91, 182, 126, 55, 43, 40, 354, 170, 91, 51, 85, 58, 20, 238, 21, 21, 14, 12, 47, 32, 252, 144, 57, 51, 91, 90, 42, 90, 56, 56, 55, 55, 55, 55, 54, 54, 30, 28, 246, 89, 83, 22, 78, 88, 87, 84, 49, 44, 22, 89, 147, 25, 82, 56, 38, 34, 115, 86, 20, 43, 179, 116, 42, 85, 36, 196,
                         146, 84, 37, 32, 3, 279, 120, 90, 34, 301, 202, 27, 150, 89, 33, 169, 91, 50, 46, 45, 44, 44, 44, 43, 247, 114, 96, 45, 45, 45, 45, 45, 44, 44, 43, 43, 43, 26, 118, 25, 14, 189, 126, 125, 122, 122, 119, 117, 115, 111, 110, 109, 108, 101, 87, 53, 53, 43, 42, 35, 25, 25, 10, 26, 18, 5, 177, 113, 87, 91, 3, 90, 88, 35, 220, 15, 242, 91, 74, 9, 5, 47, 8, 67, 58, 8, 58, 42, 3, 237, 152, 109, 91, 2, 174, 48, 56, 19, 187, 91, 178, 99, 93, 70, 61, 55, 38, 98, 63, 81, 34, 74, 74,
                         33, 6, 18, 125, 76, 211, 167, 131, 42, 4, 253, 142, 38, 72, 27, 146, 202, 27, 70, 77, 83, 15, 18, 29, 97, 25, 210, 74, 36, 70, 76, 98, 38, 65, 159, 109, 86, 71, 71, 63, 63, 97, 70, 5, 67, 16, 39, 29, 29, 106, 24, 75, 77, 734, 712, 700, 651, 645, 633, 628, 606, 584, 561, 552, 517, 503, 500, 498, 485, 478, 467, 463, 459, 456, 433, 427, 424, 423, 420, 419, 413, 412, 402, 401, 387, 385, 378, 377, 375, 370, 370, 369, 368, 364, 363, 363, 362, 362, 362, 361, 361, 359, 359, 358,
                         350, 347, 347, 346, 343, 340, 339, 336, 334, 333, 333, 332, 331, 330, 328, 325, 323, 322, 320, 319, 317, 315, 315, 312, 310, 310, 309, 306, 306, 306, 304, 303, 303, 301, 300, 300, 299, 299, 298, 298, 297, 295, 295, 294, 294, 290, 288, 288, 287, 287, 287, 287, 287, 286, 286, 283, 281, 280, 280, 279, 278, 278, 278, 277, 277, 276, 276, 275, 273, 272, 272, 272, 271, 270, 270, 270, 269, 266, 265, 264, 263, 262, 260, 260, 259, 259, 259, 259, 259, 258, 254, 253, 252, 252, 252,
                         252, 252, 252, 251, 251, 251, 251, 249, 247, 247, 247, 246, 246, 246, 246, 246, 245, 245, 242, 242, 241, 240, 240, 238, 237, 236, 235, 234, 234, 233, 233, 232, 232, 232, 231, 231, 229, 229, 229, 229, 227, 227, 226, 226, 226, 225, 224, 224, 224, 224, 223, 223, 223, 223, 223, 221, 221, 221, 221, 220, 220, 220, 220, 219, 218, 218, 218, 218, 218, 217, 216, 215, 214, 214, 214, 212, 212, 212, 212, 212, 212, 211, 211, 210, 210, 210, 210, 209, 209, 209, 209, 208, 208, 208, 207,
                         207, 207, 207, 206, 205, 205, 205, 205, 204, 203, 203, 203, 203, 203, 202, 202, 201, 201, 201, 200, 200, 199, 199, 199, 198, 197, 197, 197, 197, 197, 197, 197, 196, 196, 196, 196, 196, 196, 196, 196, 195, 194, 194, 194, 194, 194, 193, 193, 193, 192, 192, 192, 192, 192, 191, 191, 190, 190, 189, 187, 187, 186, 185, 184, 184, 184, 184, 184, 184, 183, 183, 183, 183, 183, 182, 182, 182, 181, 180, 180, 180, 180, 179, 179, 179, 179, 179, 178, 178, 177, 177, 177, 176, 176, 175,
                         175, 175, 175, 175, 175, 175, 175, 174, 174, 174, 174, 174, 174, 173, 173, 173, 173, 172, 172, 172, 172, 172, 171, 171, 171, 171, 171, 171, 170, 170, 170, 170, 170, 170, 169, 168, 168, 168, 168, 168, 168, 168, 168, 167, 167, 167, 167, 167, 166, 166, 166, 166, 166, 165, 165, 165, 164, 164, 164, 164, 163, 163, 163, 162, 162, 162, 161, 161, 161, 161, 161, 160, 160, 159, 159, 159, 158, 158, 158, 158, 157, 157, 155, 154, 154, 154, 153, 153, 153, 153, 152, 152, 151, 151, 151,
                         150, 150, 149, 148, 148, 147, 147, 147, 147, 146, 146, 144, 143, 143, 143, 143, 143, 142, 142, 142, 141, 141, 141, 140, 140, 140, 140, 140, 139, 139, 139, 138, 138, 138, 137, 137, 137, 137, 137, 136, 136, 136, 136, 136, 134, 134, 134, 134, 134, 134, 134, 133, 133, 133, 133, 133, 133, 133, 133, 133, 133, 132, 132, 132, 132, 131, 131, 130, 130, 130, 129, 129, 129, 129, 129, 129, 129, 129, 129, 129, 129, 128, 128, 128, 128, 127, 127, 126, 126, 126, 126, 125, 124, 124, 124,
                         124, 123, 123, 123, 122, 122, 122, 122, 121, 121, 121, 120, 119, 119, 119, 118, 117, 117, 116, 115, 115, 115, 115, 115, 115, 114, 114, 114, 114, 114, 113, 113, 113, 113, 113, 113, 113, 113, 112, 112, 112, 112, 112, 112, 111, 111, 111, 110, 110, 110, 110, 110, 110, 109, 109, 109, 109, 109, 109, 108, 108, 108, 108, 108, 108, 107, 107, 107, 107, 106, 106, 105, 105, 105, 105, 104, 103, 102, 102, 102, 102, 101, 101, 101, 101, 101, 101, 100, 100, 100, 100, 99, 99, 99, 99, 99,
                         98, 98, 98, 98, 98, 98, 97, 97, 97, 97, 97, 97, 97, 96, 95, 95, 95, 95, 94, 94, 94, 94, 93, 93, 93, 93, 93, 93, 93, 93, 93, 93, 92, 92, 92, 92, 92, 92, 92, 92, 92, 91, 91, 91, 91, 90, 90, 90, 89, 89, 88, 88, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 87, 86, 86, 86, 86, 86, 86, 86, 85, 85, 85, 85, 85, 85, 85, 85, 85, 84, 84, 84, 84, 84, 84, 84, 84, 84, 83, 83, 83, 83, 83, 83, 82, 82, 82, 82, 82, 81, 81, 81, 80, 80, 80, 80, 80, 80, 80, 79, 79, 79, 79, 79, 78,
                         78, 78, 78, 78, 78, 77, 77, 77, 77, 77, 77, 77, 76, 76, 76, 76, 75, 75, 74, 74, 74, 74, 73, 72, 72, 72, 72, 71, 71, 71, 71, 71, 71, 70, 70, 70, 70, 70, 70, 70, 69, 69, 69, 69, 69, 69, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 66, 66, 66, 66, 65, 65, 65, 65, 65, 64, 64, 63, 63, 63, 63, 62, 62, 62, 62, 62, 62, 61, 61, 61, 60, 60, 60, 60, 59, 59, 59, 59, 58, 58, 58, 58, 57, 57, 57, 57, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 55, 55, 55, 55, 55, 55, 54, 54, 54, 54, 54, 53,
                         53, 53, 53, 53, 53, 52, 52, 52, 52, 51, 51, 50, 50, 50, 50, 49, 49, 49, 49, 48, 48, 48, 48, 48, 48, 48, 48, 47, 47, 47, 47, 47, 47, 46, 46, 46, 46, 45, 45, 45, 44, 44, 44, 43, 43, 43, 43, 43, 43, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 41, 41, 41, 41, 41, 41, 40, 39, 39, 39, 39, 38, 38, 38, 38, 37, 37, 37, 37, 37, 37, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 35, 35, 35, 35, 34, 34, 33, 33, 33, 32, 32, 32, 32, 32, 32, 32, 32, 31, 31, 30, 30, 30, 30, 30,
                         30, 30, 29, 29, 29, 28, 28, 28, 27, 27, 27, 27, 26, 26, 26, 26, 25, 25, 25, 25, 24, 24, 24, 24, 24, 24, 23, 23, 23, 23, 23, 23, 23, 23, 23, 23, 22, 22, 22, 22, 21, 21, 21, 20, 20, 20, 20, 20, 19, 19, 19, 19, 19, 19, 18, 17, 17, 17, 17, 16, 15, 15, 15, 14, 14, 14, 14, 14, 13, 13, 12, 12, 12, 11, 10, 10, 9, 9, 8, 7, 7, 6, 6, 6, 5, 5, 5, 3, 3, 3, 2, 2, 2]
        self.right_censored = [922, 672, 580, 643, 425, 302, 38, 412, 195, 867, 61, 854, 176, 391, 364, 608, 646, 708, 432, 534, 171, 707, 377, 774, 455, 603, 209, 267, 227, 26, 253, 95, 90, 159, 350, 117, 176, 54, 88, 339, 203, 280, 335, 289, 125, 367, 43, 208, 291, 308, 323, 302, 311, 244, 94, 191, 205, 305, 240, 343, 70, 181, 193, 208, 333, 119, 238, 339, 43, 189, 176, 314, 16, 98, 13, 75, 246, 48, 152, 220, 240, 286, 6, 226, 240, 314, 125, 300, 133, 53, 150, 178, 221, 252, 329, 15, 46, 77,
                               196, 63, 127, 154, 184, 187, 166, 168, 195, 226, 265, 308, 333, 31, 152, 317, 106, 179, 228, 118, 182, 232, 301, 5, 36, 151, 207, 292, 333, 228, 260, 170, 248, 50, 169, 211, 36, 95, 102, 208, 288, 38, 44, 83, 154, 233, 253, 255, 270, 304, 92, 118, 183, 202, 204, 216, 287, 61, 271, 296, 33, 70, 270, 284, 173, 187, 248, 9, 19, 238, 20, 27, 127, 169, 191, 225, 247, 261, 16, 41, 105, 149, 237, 47, 47, 228, 265, 130, 192, 14, 59, 51, 63, 151, 147, 163, 123, 76, 160,
                               218, 162, 204, 224, 275, 21, 164, 207, 213, 244, 270, 175, 151, 199, 200, 226, 233, 249, 59, 129, 135, 204, 239, 41, 55, 115, 141, 162, 176, 198, 205, 216, 260, 100, 133, 163, 186, 233, 267, 84, 125, 199, 8, 86, 106, 130, 168, 212, 127, 127, 50, 154, 165, 209, 280, 70, 48, 63, 90, 118, 101, 185, 218, 240, 29, 114, 187, 191, 7, 7, 123, 5, 11, 49, 101, 120, 147, 186, 191, 244, 167, 86, 95, 168, 93, 143, 162, 21, 271, 6, 84, 179, 221, 133, 143, 183, 190, 211, 224,
                               225, 150, 168, 88, 245, 92, 97, 248, 86, 114, 60, 82, 85, 113, 200, 180, 61, 146, 87, 161, 204, 135, 153, 182, 75, 46, 151, 197, 11, 130, 18, 147, 112, 184, 71, 85, 150, 205, 179, 86, 78, 98, 111, 111, 129, 180, 117, 20, 87, 4, 146, 79, 198, 44, 64, 181, 25, 81, 162, 34, 134, 145, 110, 55, 45, 156, 210, 112, 179, 180, 6, 93, 9, 42, 179, 53, 56, 87, 112, 125, 77, 11, 159, 164, 94, 63, 146, 151, 163, 43, 6, 10, 31, 57, 66, 50, 95, 42, 62, 71, 114, 42, 46, 5, 12, 50,
                               63, 33, 58, 34, 56, 50, 46, 27, 22, 69, 84, 42, 83, 29, 47, 52, 65, 105, 19, 53, 4, 72, 52, 113, 49, 41, 46, 11, 30, 6, 15, 10, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 5, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 6, 7, 7, 7, 7, 7, 7, 7, 7,
                               7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 7, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 9, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 10, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 12, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 13, 14, 14, 14,
                               14, 14, 14, 14, 14, 14, 14, 14, 14, 14, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 15, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 16, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 18, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 19, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 20, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 21, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 22, 23, 23, 23,
                               23, 23, 23, 23, 23, 23, 23, 23, 23, 23, 23, 23, 24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 24, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 25, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 26, 27, 27, 27, 27, 27, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 29, 29, 29, 29, 29, 29, 29, 29, 29, 29, 29, 29, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 30, 31, 31, 31, 31, 31, 31, 31, 31, 31, 31, 31, 31, 31, 31, 31,
                               32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 32, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 33, 34, 34, 34, 34, 34, 34, 34, 34, 34, 34, 34, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 35, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 36, 37, 37, 37, 37, 37, 37, 37, 37, 37, 37, 37, 37, 37, 38, 38, 38, 38, 38, 38, 38, 38, 38, 39, 39, 39, 39, 39, 39, 39, 39, 39, 39, 39, 39, 40, 40, 40, 40,
                               40, 40, 40, 40, 40, 40, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 42, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 43, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 44, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 45, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 46, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 47, 48, 48, 48, 48,
                               48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 48, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 49, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 50, 51, 51, 51, 51, 51, 51, 51, 51, 52, 52, 52, 52, 52, 52, 52, 52, 53, 53, 53, 53, 53, 53, 53, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 54, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 55, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56, 56,
                               56, 56, 56, 56, 56, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 57, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 58, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 59, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 61, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 62, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63, 63,
                               63, 63, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 64, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 65, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 66, 67, 67, 67, 67, 67, 67, 67, 67, 67, 67, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 68, 69, 69, 69, 69, 69, 69, 69, 69, 69, 69, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 70, 71, 71, 71, 71, 71, 71, 71, 71,
                               71, 71, 71, 71, 71, 71, 71, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 72, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 73, 74, 74, 74, 74, 74, 75, 75, 75, 75, 75, 75, 75, 75, 76, 76, 76, 76, 76, 76, 76, 76, 76, 76, 76, 76, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 77, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 78, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 79, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80,
                               80, 80, 80, 80, 81, 81, 81, 81, 81, 82, 82, 82, 82, 82, 82, 82, 82, 82, 82, 82, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 83, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 84, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 85, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 86, 87, 87, 87, 87, 87, 87, 87, 87, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 88, 89, 89, 89, 89,
                               89, 89, 89, 89, 89, 89, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 91, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 92, 93, 93, 93, 93, 93, 93, 93, 93, 93, 93, 93, 93, 93, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 94, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 95, 96, 96, 96, 96, 96, 96, 96, 96, 96, 96, 96, 97, 97, 97, 97, 97, 97, 97, 97, 97,
                               97, 97, 97, 98, 98, 98, 98, 98, 98, 98, 98, 98, 98, 98, 98, 98, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 99, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 101, 101, 101, 101, 101, 101, 101, 101, 102, 102, 102, 102, 102, 102, 102, 102, 102, 102, 102, 102, 102, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 103, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 104, 105, 105, 105, 105, 105, 105, 105, 105, 105, 105,
                               105, 105, 105, 105, 105, 105, 105, 105, 105, 105, 105, 105, 106, 106, 106, 106, 106, 106, 106, 106, 106, 106, 106, 106, 106, 106, 106, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 111, 111, 111, 111, 111, 111, 111, 111, 111, 111, 111,
                               111, 111, 111, 111, 111, 111, 111, 111, 111, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 113, 113, 113, 113, 113, 113, 113, 113, 113, 113, 113, 113, 113, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 117, 117, 117, 117, 117, 117, 117,
                               117, 117, 117, 117, 117, 117, 117, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 120, 120, 120, 120, 120, 120, 120, 120, 120, 120, 120, 121, 121, 121, 121, 121, 121, 121, 121, 121, 122, 122, 122, 122, 122, 122, 122, 122, 122, 122, 122, 123, 123, 123, 123, 123, 123, 123, 123, 123, 123, 123, 123, 123, 124, 124, 124,
                               124, 124, 124, 124, 124, 124, 124, 124, 124, 124, 124, 124, 124, 124, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 125, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 126, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 127, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 129, 129, 129, 129, 129, 129, 129, 129, 129, 130, 130,
                               130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 130, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 131, 132, 132, 132, 132, 132, 132, 132, 132, 132, 132, 132, 132, 132, 132, 132, 132, 133, 133, 133, 133, 133, 133, 133, 133, 133, 133, 133, 133, 133, 133, 133, 134, 134, 134, 134, 134, 134, 134, 134, 134, 134, 134, 134, 134, 134, 134, 135, 135, 135, 135, 135, 135, 135, 135, 135, 135,
                               135, 135, 135, 135, 135, 135, 135, 136, 136, 136, 136, 136, 136, 136, 136, 136, 136, 137, 137, 137, 137, 137, 137, 137, 137, 137, 137, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 138, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 139, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 140, 141, 141, 141, 141, 141, 141, 141, 141, 141, 141,
                               141, 141, 141, 141, 141, 141, 141, 142, 142, 142, 142, 142, 142, 142, 142, 142, 143, 143, 143, 143, 143, 143, 143, 143, 143, 143, 144, 144, 144, 144, 144, 144, 144, 144, 144, 144, 144, 144, 144, 144, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 145, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 146, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147, 147,
                               147, 147, 147, 147, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 148, 149, 149, 149, 149, 149, 149, 149, 149, 150, 150, 150, 150, 150, 150, 150, 150, 150, 150, 150, 150, 150, 150, 151, 151, 151, 151, 151, 151, 151, 151, 151, 151, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 152, 153, 153, 153, 153, 153, 153, 153, 153, 153, 153, 153, 153, 153, 154, 154, 154, 154, 154, 154, 154, 154,
                               154, 154, 154, 154, 154, 154, 154, 154, 154, 154, 154, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 155, 156, 156, 156, 156, 156, 156, 156, 156, 156, 156, 156, 156, 156, 156, 156, 157, 157, 157, 157, 157, 157, 157, 157, 157, 157, 157, 157, 157, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 158, 159, 159, 159, 159, 159, 159, 159, 159, 159, 159, 160, 160, 160, 160,
                               160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 160, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 161, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 162, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 163, 164, 164, 164, 164, 164, 164, 164, 164, 164, 164, 164, 164, 164,
                               164, 164, 164, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 165, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 166, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 167, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168, 168,
                               168, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 169, 170, 170, 170, 170, 170, 170, 170, 170, 170, 170, 170, 170, 170, 170, 170, 170, 171, 171, 171, 171, 171, 171, 171, 171, 171, 171, 172, 172, 172, 172, 172, 172, 172, 172, 172, 172, 172, 172, 172, 173, 173, 173, 173, 173, 173, 173, 173, 173, 173, 173, 173, 173, 173, 174, 174, 174, 174, 174, 174, 174, 174, 174, 174, 174, 174, 174,
                               174, 174, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 175, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 176, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 177, 178, 178, 178, 178, 178, 178, 178, 178, 178, 178, 178, 178, 178, 178, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 179, 180, 180,
                               180, 180, 180, 180, 180, 180, 180, 180, 180, 180, 180, 180, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 181, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 182, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 183, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 184, 185, 185, 185, 185, 185, 185, 185, 185,
                               185, 185, 185, 185, 185, 185, 185, 185, 185, 185, 185, 185, 185, 186, 186, 186, 186, 186, 186, 186, 186, 186, 186, 187, 187, 187, 187, 187, 187, 187, 187, 187, 187, 187, 187, 187, 187, 187, 187, 187, 188, 188, 188, 188, 188, 188, 188, 188, 188, 188, 188, 188, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 189, 190, 190, 190, 190, 190, 190, 190, 190, 190, 190, 190, 190, 190, 190, 190, 190, 190, 191, 191, 191, 191, 191,
                               191, 191, 191, 191, 191, 191, 191, 191, 191, 191, 191, 191, 191, 191, 192, 192, 192, 192, 192, 192, 192, 192, 192, 192, 192, 192, 192, 192, 192, 192, 192, 193, 193, 193, 193, 193, 193, 193, 193, 193, 193, 193, 193, 193, 193, 194, 194, 194, 194, 194, 194, 194, 194, 194, 194, 194, 194, 194, 194, 194, 194, 195, 195, 195, 195, 195, 195, 195, 195, 195, 195, 195, 195, 195, 195, 195, 195, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196,
                               196, 196, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 198, 198, 198, 198, 198, 198, 198, 198, 198, 198, 198, 198, 198, 198, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 200, 200, 200, 200, 200, 200, 201, 201, 201, 201, 201, 201, 201, 201, 201, 201, 201, 201, 201, 201, 201, 201, 201, 202, 202, 202, 202, 202, 202, 202, 202, 202, 202, 202, 202, 202, 202, 203, 203, 203, 203, 203,
                               203, 203, 203, 203, 203, 203, 203, 203, 203, 203, 203, 204, 204, 204, 204, 204, 204, 204, 204, 204, 204, 204, 204, 204, 204, 204, 205, 205, 205, 205, 205, 205, 205, 205, 205, 206, 206, 206, 206, 206, 206, 206, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 207, 208, 208, 208, 208, 208, 208, 208, 208, 208, 208, 208, 208, 208, 208, 209, 209, 209, 209, 209, 209, 209, 209, 209, 209, 209, 209, 210, 210, 210, 210, 210, 210, 210, 210,
                               210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 210, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 211, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 212, 213, 213, 213, 213, 213, 213, 213, 213, 213, 213, 213, 213, 213, 213, 213, 214, 214, 214, 214, 214, 214, 214, 214, 214, 214, 214, 215, 215, 215, 215, 215, 215, 215, 215, 215, 215, 215, 215, 215,
                               215, 215, 215, 215, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 216, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 217, 218, 218, 218, 218, 218, 218, 218, 218, 218, 218, 218, 218, 218, 218, 218, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 219, 220, 220, 220, 220, 220,
                               220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 220, 221, 221, 221, 221, 221, 221, 221, 221, 221, 221, 221, 221, 221, 221, 221, 221, 221, 222, 222, 222, 222, 222, 222, 222, 222, 222, 222, 222, 222, 222, 222, 222, 223, 223, 223, 223, 223, 223, 223, 223, 223, 223, 223, 223, 223, 223, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 224, 225, 225, 225, 225, 225, 225, 225, 225, 225, 225, 225, 225, 225,
                               225, 225, 225, 226, 226, 226, 226, 226, 226, 226, 226, 226, 227, 227, 227, 227, 227, 227, 227, 227, 227, 227, 227, 227, 227, 227, 227, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 228, 229, 229, 229, 229, 229, 229, 229, 229, 229, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 230, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231, 231,
                               231, 231, 231, 231, 232, 232, 232, 232, 232, 232, 232, 232, 232, 232, 232, 232, 232, 232, 232, 232, 232, 233, 233, 233, 233, 233, 233, 233, 233, 233, 233, 233, 233, 233, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 234, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 235, 236, 236, 236, 236, 236, 236, 236, 236, 236, 237, 237, 237, 237, 237, 237, 237, 237, 237, 237, 237, 237,
                               237, 237, 237, 237, 237, 237, 237, 237, 237, 237, 237, 237, 237, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 238, 239, 239, 239, 239, 239, 239, 239, 239, 239, 239, 239, 239, 240, 240, 240, 240, 240, 240, 240, 240, 240, 240, 240, 240, 240, 241, 241, 241, 241, 241, 241, 241, 241, 241, 241, 241, 241, 241, 241, 241, 241, 242, 242, 242, 242, 242, 242, 243, 243,
                               243, 243, 243, 243, 243, 243, 243, 243, 243, 243, 243, 243, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 244, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 245, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 246, 247, 247, 247, 247, 247, 247, 247, 247, 247, 247, 247, 247, 247, 247, 247, 248, 248, 248, 248, 248, 248, 248, 248,
                               248, 248, 248, 248, 248, 248, 249, 249, 249, 249, 249, 249, 249, 249, 249, 249, 249, 249, 249, 249, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 250, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 251, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 252, 253, 253, 253, 253,
                               253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 253, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 256, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 257, 258, 258, 258,
                               258, 258, 258, 258, 258, 258, 258, 258, 258, 258, 258, 258, 258, 258, 258, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 259, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 260, 261, 261, 261, 261, 261, 261, 261, 261, 261, 261, 261, 262, 262, 262, 262, 262, 262, 262, 262, 262, 262, 262, 263, 263, 263, 263, 263, 263, 263, 263, 263, 263, 263, 263, 263, 264, 264,
                               264, 264, 264, 264, 264, 264, 264, 264, 264, 264, 264, 264, 264, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 265, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 266, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 268, 268, 268, 268, 268, 268, 268, 268, 268, 268, 269, 269, 269, 269,
                               269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 270, 270, 270, 270, 270, 270, 270, 270, 270, 270, 270, 270, 270, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 274, 274, 274, 274, 274, 274,
                               274, 274, 274, 274, 274, 274, 274, 274, 274, 274, 274, 275, 275, 275, 275, 275, 275, 275, 275, 275, 275, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 277, 278, 278, 278, 278, 278, 278, 278, 278, 278, 278, 278, 278, 278, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 279,
                               279, 279, 279, 279, 279, 279, 279, 279, 279, 279, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 280, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 281, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 282, 283, 283, 283, 283, 283, 283, 283, 283, 283, 283,
                               283, 283, 283, 283, 283, 283, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 284, 285, 285, 285, 285, 285, 285, 285, 285, 285, 285, 285, 285, 285, 285, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 286, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 287, 288, 288, 288, 288, 288, 288, 288,
                               288, 288, 288, 288, 288, 288, 288, 288, 288, 288, 288, 288, 288, 288, 288, 289, 289, 289, 289, 289, 289, 289, 289, 289, 289, 289, 289, 289, 289, 289, 290, 290, 290, 290, 290, 290, 290, 290, 290, 290, 290, 290, 291, 291, 291, 291, 291, 291, 291, 291, 291, 291, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 292, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293, 293,
                               294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 294, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 295, 296, 296, 296, 296, 296, 296, 296, 296, 296, 296, 296, 296, 296, 296, 297, 297, 297, 297, 297, 297, 297, 297, 297, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 298, 299, 299, 299, 299, 299, 299,
                               299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 303, 303, 303,
                               303, 303, 303, 303, 303, 303, 303, 303, 303, 303, 303, 303, 303, 304, 304, 304, 304, 304, 304, 304, 304, 304, 304, 304, 304, 304, 304, 304, 305, 305, 305, 305, 305, 305, 305, 305, 305, 305, 305, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 308, 308, 308, 308,
                               308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 310, 310, 310, 310, 310, 310, 310, 310, 310, 310, 310, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 312, 312, 312, 312, 312, 312, 312, 312, 312, 312, 312, 312, 312, 312,
                               313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 317, 317, 317, 317,
                               317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 319, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 320, 321, 321, 321, 321, 321, 321, 321, 321, 321, 321, 321, 321, 321, 321, 321, 321,
                               321, 321, 321, 321, 321, 321, 321, 321, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 322, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 323, 324, 324, 324, 324, 324, 324, 324, 324, 324, 324, 324, 324, 324, 324, 325, 325, 325, 325, 325, 325, 325, 325, 325, 325, 325, 325, 326, 326, 326, 326, 326, 326,
                               326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329,
                               330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 331, 331, 331, 331, 331, 331, 331, 331, 331, 331, 331, 331, 331, 331, 331, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 332, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 333, 334,
                               334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 334, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 335, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336, 336,
                               336, 336, 336, 336, 336, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 337, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 338, 339, 339, 339, 339, 339, 339, 339, 339, 339, 339, 339, 339, 339, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340,
                               340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 340, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 341, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 342, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343, 343,
                               343, 343, 343, 343, 343, 343, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 344, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 345, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 346, 347, 347, 347, 347,
                               347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 347, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 348, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 349, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350,
                               350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 350, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 351, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 352, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353,
                               353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 353, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 354, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 355, 356,
                               356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 356, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 357, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358,
                               358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 358, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 359, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 360, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361,
                               361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 361, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 362, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363,
                               363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 363, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364,
                               364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 364, 365, 365, 365, 365, 365, 365, 365, 365, 365, 365, 365, 365, 366, 366, 366, 366, 366, 366, 366, 366, 366, 366, 367, 367, 367, 367, 367, 367, 367, 367, 367, 368, 368, 368, 368, 368, 368, 368, 368, 368, 369, 369, 369, 369, 369, 369, 369, 369, 369, 369, 369, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 370, 371, 371, 371, 371, 371,
                               371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 371, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 372, 373, 373, 373, 373, 373, 373, 373, 373, 373, 373, 374, 374, 374, 374, 374, 374, 374, 374, 374, 375, 375, 375, 375, 375, 375, 375, 375, 375, 375, 376, 376, 376, 376, 376, 376, 377, 377, 377, 377, 377, 377, 377, 377, 377, 378, 378, 378, 378, 378, 378, 378, 378, 378, 378, 378, 378, 378, 378, 379, 379, 379, 379, 379, 379, 379, 379, 379, 380, 380,
                               380, 380, 380, 380, 380, 380, 380, 380, 380, 380, 381, 381, 381, 381, 381, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 382, 383, 383, 383, 383, 383, 383, 383, 383, 383, 383, 383, 383, 383, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 384, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 385, 386, 386, 386, 386, 386, 386, 386, 386, 386, 387, 387, 387, 387, 387, 387, 388, 388, 388, 388, 388,
                               388, 388, 388, 388, 388, 388, 388, 388, 388, 388, 388, 389, 389, 389, 389, 389, 389, 389, 389, 389, 389, 389, 389, 390, 390, 390, 390, 390, 390, 390, 390, 390, 391, 391, 391, 391, 391, 391, 391, 391, 391, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 392, 393, 393, 393, 393, 393, 393, 393, 393, 393, 393, 393, 394, 394, 394, 394, 394, 394, 394, 394, 394, 394, 394, 394, 395, 395, 395, 395, 395, 395, 395, 395, 396, 396, 396, 397,
                               397, 397, 398, 398, 398, 398, 398, 398, 398, 399, 399, 399, 399, 399, 399, 399, 399, 399, 399, 399, 400, 400, 400, 400, 400, 400, 401, 401, 401, 401, 401, 401, 402, 402, 402, 402, 402, 402, 403, 403, 403, 403, 403, 403, 403, 403, 403, 404, 404, 404, 404, 404, 404, 404, 404, 404, 404, 404, 405, 405, 405, 405, 405, 405, 405, 405, 405, 405, 405, 406, 406, 406, 406, 406, 406, 406, 406, 406, 406, 406, 406, 406, 406, 407, 407, 407, 407, 407, 407, 407, 407, 408, 408, 408,
                               408, 408, 408, 408, 408, 408, 408, 408, 408, 408, 408, 409, 409, 409, 409, 409, 410, 410, 410, 410, 410, 410, 410, 410, 410, 411, 411, 411, 412, 412, 412, 412, 412, 412, 412, 413, 413, 413, 413, 413, 414, 414, 414, 414, 414, 415, 415, 415, 415, 415, 416, 416, 416, 416, 416, 417, 417, 417, 417, 417, 418, 418, 418, 418, 418, 418, 418, 418, 418, 418, 418, 418, 418, 418, 418, 419, 419, 419, 420, 420, 420, 420, 420, 420, 420, 420, 420, 420, 420, 420, 420, 421, 421, 421,
                               421, 421, 421, 421, 421, 421, 421, 421, 421, 421, 421, 421, 422, 422, 422, 422, 422, 422, 422, 422, 422, 422, 422, 422, 422, 422, 422, 423, 423, 423, 423, 423, 423, 423, 423, 423, 423, 423, 424, 424, 424, 424, 424, 424, 424, 424, 424, 424, 424, 424, 425, 425, 425, 425, 425, 425, 425, 425, 426, 426, 426, 426, 426, 426, 426, 426, 426, 426, 426, 427, 427, 427, 427, 427, 427, 427, 427, 427, 427, 428, 428, 428, 428, 428, 428, 428, 428, 428, 429, 429, 429, 429, 430, 430,
                               430, 430, 430, 430, 430, 430, 431, 431, 431, 431, 431, 431, 431, 431, 431, 432, 432, 432, 433, 433, 433, 433, 433, 433, 433, 433, 433, 433, 433, 433, 434, 434, 434, 434, 434, 434, 434, 434, 434, 434, 434, 434, 434, 434, 434, 434, 435, 435, 435, 435, 435, 435, 435, 435, 435, 435, 435, 435, 435, 435, 435, 435, 435, 436, 436, 436, 436, 436, 437, 437, 437, 437, 437, 437, 437, 438, 438, 438, 438, 438, 438, 438, 438, 438, 438, 438, 438, 439, 439, 439, 439, 439, 439, 439,
                               440, 440, 440, 440, 440, 440, 440, 440, 440, 441, 441, 441, 441, 441, 441, 441, 441, 441, 441, 441, 441, 441, 442, 442, 442, 442, 442, 442, 442, 442, 442, 442, 442, 442, 443, 443, 444, 444, 444, 444, 444, 444, 445, 445, 445, 445, 445, 445, 445, 445, 445, 445, 445, 445, 445, 445, 446, 446, 446, 446, 446, 446, 446, 446, 447, 447, 447, 447, 447, 447, 447, 447, 447, 447, 447, 447, 447, 447, 447, 447, 447, 448, 448, 448, 448, 448, 448, 448, 448, 448, 448, 448, 448, 448,
                               448, 449, 449, 449, 449, 449, 449, 449, 449, 449, 449, 450, 450, 450, 450, 450, 450, 451, 451, 451, 451, 451, 451, 451, 451, 451, 452, 452, 452, 452, 452, 453, 453, 453, 453, 453, 453, 453, 454, 454, 454, 454, 454, 454, 454, 454, 454, 454, 454, 454, 455, 455, 455, 455, 455, 455, 455, 455, 455, 455, 455, 455, 455, 456, 456, 456, 456, 456, 456, 456, 456, 456, 456, 456, 456, 457, 457, 457, 458, 458, 458, 458, 458, 459, 459, 459, 459, 460, 460, 460, 460, 460, 460, 460,
                               460, 461, 461, 461, 461, 461, 461, 461, 461, 461, 461, 461, 461, 461, 462, 462, 462, 462, 462, 462, 462, 462, 462, 462, 462, 462, 462, 463, 463, 463, 463, 463, 463, 463, 463, 463, 463, 463, 463, 464, 464, 464, 464, 464, 464, 465, 465, 465, 465, 465, 466, 466, 466, 466, 466, 466, 467, 467, 467, 467, 467, 467, 467, 467, 467, 467, 468, 468, 468, 468, 468, 468, 468, 468, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469, 469,
                               469, 469, 470, 470, 470, 470, 470, 470, 470, 470, 470, 470, 470, 470, 470, 471, 471, 471, 471, 471, 472, 472, 472, 472, 472, 472, 472, 472, 472, 472, 472, 473, 473, 473, 473, 473, 473, 473, 473, 474, 474, 474, 474, 475, 475, 475, 475, 475, 475, 475, 475, 475, 475, 475, 475, 475, 475, 475, 475, 476, 476, 476, 476, 476, 476, 476, 476, 476, 476, 476, 476, 477, 477, 477, 477, 477, 477, 477, 477, 477, 477, 477, 477, 478, 478, 478, 478, 478, 478, 478, 478, 478, 479, 479,
                               479, 479, 479, 479, 479, 479, 479, 480, 480, 480, 480, 480, 481, 481, 481, 481, 481, 481, 481, 482, 482, 482, 482, 482, 482, 482, 482, 482, 482, 483, 483, 483, 483, 483, 483, 483, 484, 484, 484, 484, 484, 484, 484, 484, 484, 485, 485, 485, 485, 485, 485, 485, 485, 486, 486, 486, 486, 486, 486, 486, 486, 486, 486, 486, 486, 487, 487, 487, 487, 487, 487, 487, 487, 487, 487, 487, 487, 487, 488, 488, 488, 488, 488, 488, 488, 488, 488, 488, 488, 488, 488, 488, 488, 489,
                               489, 489, 489, 489, 489, 489, 489, 489, 489, 489, 489, 489, 489, 489, 490, 490, 490, 490, 490, 490, 490, 490, 490, 490, 490, 490, 490, 490, 491, 491, 491, 491, 491, 491, 491, 491, 491, 491, 491, 491, 491, 491, 492, 492, 492, 492, 492, 492, 493, 493, 493, 493, 493, 493, 493, 493, 493, 493, 493, 494, 494, 494, 494, 494, 494, 494, 494, 494, 494, 494, 494, 495, 495, 495, 495, 495, 495, 495, 495, 495, 495, 495, 495, 496, 496, 496, 496, 496, 496, 496, 496, 496, 496, 496,
                               496, 496, 497, 497, 497, 497, 497, 497, 497, 497, 497, 497, 497, 497, 497, 497, 497, 497, 497, 498, 498, 498, 498, 498, 498, 498, 498, 498, 498, 498, 498, 499, 499, 499, 499, 499, 499, 499, 499, 499, 499, 499, 499, 499, 500, 500, 500, 500, 500, 500, 500, 500, 500, 501, 501, 501, 501, 501, 501, 502, 502, 502, 502, 502, 502, 502, 502, 502, 503, 503, 503, 503, 503, 503, 503, 503, 503, 503, 503, 503, 503, 503, 503, 503, 503, 504, 504, 504, 504, 504, 504, 504, 504, 504,
                               504, 504, 504, 504, 504, 505, 505, 505, 505, 505, 505, 505, 506, 506, 506, 506, 506, 506, 506, 506, 506, 506, 506, 506, 506, 507, 507, 507, 507, 507, 507, 507, 507, 507, 507, 507, 508, 508, 508, 508, 508, 508, 509, 509, 509, 509, 509, 509, 509, 509, 509, 510, 510, 510, 510, 510, 510, 510, 510, 510, 510, 510, 510, 510, 510, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 511, 512, 512, 512, 512, 512, 512, 512, 512, 512, 512, 512,
                               512, 513, 513, 513, 513, 513, 513, 513, 513, 514, 514, 514, 514, 514, 514, 514, 514, 514, 514, 514, 514, 514, 514, 514, 515, 515, 515, 515, 515, 515, 515, 515, 515, 515, 515, 515, 515, 515, 515, 515, 515, 516, 516, 516, 516, 516, 516, 516, 516, 516, 516, 516, 516, 517, 517, 517, 517, 517, 517, 517, 517, 517, 517, 517, 517, 517, 517, 517, 517, 518, 518, 518, 518, 518, 518, 518, 518, 518, 518, 518, 519, 519, 519, 519, 519, 519, 519, 519, 519, 519, 519, 520, 520, 520,
                               521, 521, 521, 521, 521, 521, 521, 522, 522, 522, 522, 522, 522, 522, 523, 523, 523, 523, 523, 523, 523, 523, 523, 524, 524, 524, 524, 524, 524, 524, 524, 524, 524, 524, 524, 524, 524, 525, 525, 525, 525, 525, 525, 525, 525, 525, 525, 525, 525, 525, 525, 525, 525, 526, 526, 526, 526, 526, 526, 526, 526, 526, 526, 526, 526, 527, 527, 527, 527, 527, 528, 528, 528, 528, 528, 528, 528, 528, 529, 529, 529, 529, 529, 529, 530, 530, 530, 530, 530, 530, 530, 530, 530, 530,
                               530, 531, 531, 531, 531, 531, 531, 531, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 532, 533, 533, 533, 533, 533, 533, 533, 533, 533, 533, 533, 533, 533, 533, 533, 534, 534, 534, 534, 534, 534, 534, 534, 535, 535, 535, 535, 535, 535, 535, 535, 535, 536, 536, 536, 536, 536, 536, 536, 537, 537, 537, 537, 537, 537, 537, 537, 537, 537, 537, 538, 538, 538, 538, 538, 538, 538, 538, 538, 538, 538, 538, 538, 539, 539, 539, 539, 539,
                               539, 539, 539, 539, 539, 539, 539, 539, 539, 539, 539, 540, 540, 540, 540, 540, 540, 540, 540, 540, 541, 541, 541, 541, 541, 541, 541, 541, 541, 541, 542, 542, 542, 542, 542, 542, 542, 542, 542, 542, 542, 542, 542, 542, 542, 542, 543, 543, 543, 543, 543, 543, 543, 543, 543, 543, 544, 544, 544, 544, 544, 544, 545, 545, 545, 545, 545, 545, 545, 545, 545, 545, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 546, 547, 547, 547, 547,
                               547, 547, 547, 547, 547, 547, 548, 548, 548, 548, 548, 548, 548, 548, 548, 549, 549, 549, 549, 549, 549, 549, 549, 550, 550, 550, 550, 550, 550, 550, 550, 550, 550, 550, 550, 551, 551, 551, 551, 551, 551, 551, 551, 551, 551, 551, 551, 551, 551, 551, 552, 552, 552, 552, 552, 552, 552, 552, 552, 552, 552, 552, 552, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 553, 554, 554, 554, 554, 554, 554, 554, 554, 554, 554, 554, 554, 555,
                               555, 555, 555, 555, 555, 555, 555, 555, 555, 556, 556, 556, 556, 556, 556, 556, 557, 557, 557, 557, 557, 557, 558, 558, 558, 558, 558, 558, 558, 558, 558, 558, 558, 558, 559, 559, 559, 559, 559, 559, 559, 559, 559, 559, 560, 560, 560, 560, 560, 560, 560, 560, 560, 560, 560, 560, 560, 560, 561, 561, 561, 561, 561, 561, 561, 561, 561, 561, 561, 561, 561, 561, 562, 562, 562, 562, 562, 562, 562, 562, 562, 563, 563, 563, 563, 563, 563, 563, 563, 563, 564, 564, 564, 564,
                               564, 564, 564, 564, 564, 565, 565, 565, 565, 565, 565, 565, 565, 565, 565, 566, 566, 566, 566, 566, 566, 566, 566, 566, 566, 566, 566, 566, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 567, 568, 568, 568, 568, 568, 568, 568, 568, 568, 569, 569, 569, 569, 569, 569, 569, 570, 570, 570, 570, 570, 570, 570, 570, 570, 570, 571, 571, 571, 571, 571, 571, 571, 571, 571, 572, 572, 572, 572, 572, 572, 572, 572, 573, 573, 573, 573,
                               573, 573, 573, 573, 573, 573, 573, 574, 574, 574, 574, 574, 574, 574, 574, 574, 575, 575, 575, 575, 575, 575, 575, 575, 576, 576, 576, 576, 576, 576, 576, 576, 576, 576, 576, 576, 576, 576, 576, 577, 577, 577, 577, 577, 577, 577, 578, 578, 578, 578, 578, 578, 578, 579, 579, 579, 579, 579, 579, 579, 579, 579, 579, 580, 580, 580, 580, 580, 580, 580, 580, 580, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581, 581,
                               582, 582, 582, 582, 582, 582, 582, 582, 582, 583, 583, 583, 583, 583, 583, 583, 583, 583, 583, 584, 584, 584, 584, 584, 584, 584, 584, 584, 584, 585, 585, 585, 585, 585, 585, 585, 585, 585, 585, 585, 586, 586, 586, 586, 586, 586, 586, 586, 586, 586, 586, 587, 587, 587, 587, 587, 587, 587, 588, 588, 588, 588, 588, 588, 588, 588, 588, 588, 588, 588, 588, 588, 588, 588, 588, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589, 589,
                               590, 590, 590, 590, 590, 590, 590, 590, 590, 590, 590, 590, 590, 590, 590, 591, 591, 591, 591, 591, 591, 591, 591, 591, 591, 591, 591, 592, 592, 592, 592, 592, 592, 592, 592, 593, 593, 593, 593, 593, 594, 594, 594, 594, 594, 594, 594, 594, 594, 594, 594, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 595, 596, 596, 596, 596, 596, 596, 596, 596, 596, 596, 596, 597, 597, 597, 597, 597, 597, 597, 597, 597, 597, 597, 597,
                               598, 598, 598, 598, 598, 598, 598, 598, 598, 598, 598, 598, 598, 598, 599, 599, 599, 599, 599, 599, 599, 599, 599, 599, 599, 599, 599, 599, 599, 599, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 600, 601, 601, 601, 601, 601, 601, 601, 601, 601, 601, 601, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 602, 603, 603, 603, 603, 603, 603, 603, 603, 603, 603, 603, 604, 604, 604, 604, 604, 604, 604, 604,
                               605, 605, 605, 605, 605, 605, 605, 605, 605, 605, 605, 605, 605, 606, 606, 606, 606, 606, 606, 606, 606, 606, 606, 607, 607, 607, 607, 607, 607, 608, 608, 608, 608, 608, 608, 608, 608, 608, 608, 609, 609, 609, 609, 609, 609, 609, 609, 609, 609, 609, 609, 610, 610, 610, 610, 610, 610, 610, 610, 611, 611, 611, 611, 611, 611, 611, 611, 611, 612, 612, 612, 612, 612, 612, 612, 612, 613, 613, 613, 613, 613, 613, 613, 613, 613, 613, 613, 613, 613, 613, 613, 613, 613, 614,
                               614, 614, 614, 614, 614, 614, 614, 614, 614, 614, 614, 615, 615, 615, 615, 615, 615, 615, 615, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 616, 617, 617, 617, 617, 617, 617, 617, 617, 617, 617, 618, 618, 618, 618, 618, 618, 618, 618, 618, 618, 619, 619, 619, 619, 619, 619, 619, 619, 620, 620, 620, 620, 620, 620, 620, 620, 621, 621, 621, 621, 621, 621, 621, 621, 621, 621, 622, 622, 622, 622, 622, 622, 622, 622, 622, 622, 622,
                               622, 623, 623, 623, 623, 623, 623, 623, 623, 623, 623, 623, 623, 623, 623, 624, 624, 624, 624, 624, 624, 624, 624, 624, 624, 624, 625, 625, 625, 625, 625, 625, 625, 625, 625, 625, 625, 626, 626, 626, 626, 626, 626, 627, 627, 627, 627, 627, 627, 627, 627, 627, 627, 627, 627, 628, 628, 628, 629, 629, 629, 629, 629, 629, 629, 629, 629, 629, 629, 629, 629, 629, 629, 629, 629, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 630, 631,
                               631, 631, 631, 631, 631, 631, 631, 631, 631, 631, 631, 631, 631, 631, 632, 632, 632, 632, 632, 632, 632, 632, 632, 632, 632, 632, 632, 632, 632, 633, 633, 633, 634, 634, 634, 634, 634, 635, 635, 635, 635, 635, 635, 636, 636, 636, 636, 636, 636, 636, 636, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 637, 638, 638, 638, 638, 638, 638, 638, 638, 638, 639, 639, 639, 639, 639, 639, 639, 640, 640, 640, 640, 640, 640, 640, 641,
                               641, 641, 641, 641, 641, 641, 641, 641, 641, 642, 642, 642, 642, 642, 642, 642, 642, 642, 642, 643, 643, 643, 643, 643, 643, 643, 643, 643, 643, 643, 644, 644, 644, 644, 644, 644, 644, 644, 644, 644, 644, 644, 645, 645, 645, 645, 645, 645, 645, 645, 645, 645, 645, 645, 645, 645, 646, 646, 646, 646, 646, 646, 646, 646, 646, 646, 646, 646, 647, 647, 647, 647, 647, 647, 647, 648, 648, 648, 648, 648, 648, 648, 649, 649, 649, 649, 649, 649, 649, 649, 649, 649, 649, 649,
                               649, 649, 650, 650, 650, 650, 650, 650, 650, 650, 650, 650, 650, 650, 650, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 651, 652, 652, 652, 652, 652, 652, 652, 652, 653, 653, 653, 653, 653, 653, 653, 653, 653, 653, 653, 654, 654, 654, 654, 654, 654, 654, 655, 655, 655, 655, 655, 655, 655, 656, 656, 656, 656, 656, 656, 656, 656, 656, 656, 657, 657, 657, 657, 657, 657, 657, 657, 657, 657, 657, 657, 657, 657, 657, 658, 658, 658,
                               658, 658, 658, 658, 658, 658, 659, 659, 659, 659, 659, 659, 659, 659, 660, 660, 660, 660, 660, 660, 660, 660, 660, 660, 661, 661, 661, 661, 661, 661, 661, 662, 662, 662, 662, 662, 662, 662, 662, 662, 662, 662, 663, 663, 663, 663, 663, 663, 663, 663, 663, 663, 663, 663, 663, 663, 664, 664, 664, 664, 664, 664, 664, 664, 664, 664, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 665, 666, 666, 666, 666, 666, 666, 666, 666, 666,
                               666, 666, 667, 667, 667, 667, 667, 667, 667, 667, 667, 667, 668, 668, 668, 668, 668, 669, 669, 669, 669, 669, 669, 669, 669, 669, 669, 669, 670, 670, 670, 670, 670, 670, 670, 670, 670, 670, 670, 670, 670, 670, 671, 671, 671, 671, 671, 671, 671, 671, 671, 671, 671, 672, 672, 672, 672, 672, 672, 672, 672, 672, 672, 672, 672, 672, 672, 673, 673, 673, 673, 673, 673, 673, 673, 673, 673, 673, 673, 673, 673, 673, 673, 674, 674, 674, 674, 674, 674, 674, 674, 674, 674, 674,
                               674, 675, 675, 675, 675, 675, 675, 675, 675, 675, 676, 676, 676, 676, 676, 676, 676, 676, 676, 676, 676, 677, 677, 677, 677, 677, 677, 677, 677, 678, 678, 678, 678, 678, 678, 678, 678, 678, 678, 679, 679, 679, 679, 679, 679, 679, 679, 679, 679, 679, 680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 680, 681, 681, 681, 681, 681, 681, 681, 681, 681, 682, 682, 682, 682, 682, 682, 682, 682, 682, 683, 683, 683, 683, 683, 683, 683, 683, 683,
                               683, 683, 683, 684, 684, 684, 684, 684, 684, 684, 684, 684, 684, 685, 685, 685, 685, 685, 685, 685, 685, 685, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 686, 687, 687, 687, 687, 687, 687, 687, 687, 687, 687, 687, 688, 688, 688, 688, 688, 688, 688, 688, 689, 689, 689, 689, 689, 689, 689, 689, 689, 689, 689, 689, 689, 689, 689, 690, 690, 690, 690, 690, 690, 690, 690, 690, 691, 691, 691, 691, 691, 691, 691, 691, 692, 692,
                               692, 692, 692, 692, 692, 692, 692, 692, 693, 693, 693, 693, 693, 693, 693, 693, 693, 693, 693, 693, 693, 693, 694, 694, 694, 694, 694, 694, 694, 695, 695, 695, 695, 695, 695, 695, 695, 695, 696, 696, 696, 696, 696, 696, 696, 696, 696, 696, 697, 697, 697, 697, 697, 697, 697, 697, 697, 698, 698, 698, 698, 698, 698, 699, 699, 699, 699, 699, 699, 699, 699, 699, 699, 699, 700, 700, 700, 700, 700, 700, 700, 700, 700, 700, 700, 700, 701, 701, 701, 701, 701, 701, 701, 701,
                               701, 701, 701, 701, 701, 701, 701, 701, 701, 702, 702, 702, 702, 702, 702, 702, 702, 702, 702, 702, 703, 703, 703, 703, 703, 703, 703, 704, 704, 704, 704, 704, 704, 704, 704, 704, 704, 704, 705, 705, 705, 705, 705, 705, 705, 705, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 706, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 707, 708, 708, 708, 708, 708,
                               708, 708, 708, 708, 709, 709, 709, 709, 709, 709, 709, 709, 709, 709, 709, 709, 709, 709, 710, 710, 710, 710, 710, 710, 710, 710, 710, 710, 711, 711, 711, 711, 711, 711, 711, 711, 711, 711, 711, 711, 711, 711, 712, 712, 712, 712, 712, 712, 712, 712, 712, 712, 712, 712, 712, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 713, 714, 714, 714, 714, 714, 714, 714, 714, 714, 714, 714, 714, 714, 714, 714, 714, 714,
                               714, 714, 714, 714, 715, 715, 715, 715, 715, 715, 715, 715, 715, 715, 715, 715, 715, 715, 715, 716, 716, 716, 716, 716, 716, 716, 716, 716, 716, 716, 716, 716, 717, 717, 717, 717, 717, 717, 717, 717, 717, 717, 717, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 718, 719, 719, 719, 719, 719, 719, 719, 719, 719, 719, 719, 719, 719, 719, 720, 720, 720, 720, 720, 720, 720, 720, 720, 720, 720, 720, 720, 720, 720, 721,
                               721, 721, 721, 721, 721, 721, 721, 721, 721, 721, 721, 721, 721, 721, 721, 721, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 722, 723, 723, 723, 723, 723, 723, 723, 723, 723, 723, 723, 723, 723, 723, 723, 723, 724, 724, 724, 724, 724, 724, 724, 724, 724, 724, 724, 724, 724, 724, 725, 725, 725, 725, 725, 725, 725, 725, 725, 725, 725, 725, 725, 725, 725, 726, 726, 726, 726, 726, 726, 726, 726, 726, 727,
                               727, 727, 727, 727, 727, 727, 727, 727, 727, 727, 727, 727, 727, 727, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 728, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 729, 730, 730, 730, 730, 730, 730, 730, 730, 731, 731, 731, 731, 731, 731, 731, 731, 731, 731, 731, 732, 732, 732, 732, 733, 733, 733, 733, 734, 734, 734, 734, 734, 734,
                               734, 734, 734, 734, 735, 735, 735, 735, 735, 735, 735, 736, 736, 736, 736, 736, 736, 736, 736, 736, 736, 737, 737, 737, 737, 737, 737, 738, 738, 738, 738, 738, 738, 738, 738, 738, 738, 739, 739, 739, 739, 739, 739, 740, 740, 740, 740, 740, 740, 740, 740, 740, 740, 741, 741, 742, 742, 742, 742, 742, 742, 742, 742, 743, 743, 743, 743, 743, 743, 744, 744, 744, 744, 744, 744, 744, 744, 745, 745, 745, 745, 746, 746, 746, 746, 747, 747, 747, 747, 747, 747, 748, 748, 748,
                               748, 748, 748, 748, 749, 749, 749, 749, 749, 750, 750, 750, 750, 751, 751, 751, 751, 752, 752, 752, 753, 753, 753, 753, 753, 753, 754, 754, 754, 755, 755, 755, 755, 755, 755, 755, 755, 755, 756, 756, 756, 756, 756, 756, 757, 757, 757, 757, 757, 758, 758, 758, 758, 758, 758, 759, 759, 759, 759, 760, 760, 760, 760, 760, 760, 760, 761, 761, 761, 761, 762, 762, 763, 763, 763, 763, 764, 764, 764, 764, 764, 765, 765, 765, 765, 765, 765, 765, 765, 765, 765, 766, 766, 766,
                               766, 766, 766, 767, 767, 767, 767, 767, 767, 768, 768, 768, 769, 769, 769, 769, 769, 769, 769, 769, 769, 769, 770, 770, 770, 770, 770, 770, 770, 770, 771, 771, 771, 771, 772, 772, 772, 772, 773, 773, 773, 773, 773, 773, 773, 773, 773, 773, 773, 773, 774, 774, 774, 774, 775, 775, 775, 775, 775, 775, 776, 776, 776, 777, 777, 777, 777, 778, 778, 778, 778, 779, 779, 779, 779, 779, 779, 780, 780, 780, 780, 780, 781, 781, 781, 781, 782, 782, 782, 782, 782, 782, 782, 782,
                               782, 782, 782, 783, 783, 783, 783, 783, 784, 784, 784, 784, 784, 784, 785, 785, 785, 785, 785, 785, 785, 786, 786, 786, 786, 786, 786, 787, 787, 787, 787, 787, 788, 788, 788, 788, 789, 789, 790, 790, 790, 790, 790, 790, 791, 791, 791, 791, 791, 791, 791, 792, 792, 792, 792, 793, 793, 793, 793, 794, 794, 794, 794, 796, 796, 796, 796, 796, 797, 797, 797, 797, 797, 797, 798, 798, 798, 798, 798, 798, 798, 798, 799, 799, 799, 799, 799, 800, 800, 800, 800, 800, 801, 801,
                               801, 801, 802, 802, 802, 803, 803, 803, 803, 803, 803, 803, 804, 804, 805, 805, 805, 806, 806, 806, 807, 807, 808, 808, 809, 809, 810, 810, 810, 810, 811, 811, 811, 811, 811, 812, 812, 812, 812, 813, 813, 813, 813, 814, 814, 814, 814, 814, 814, 814, 815, 815, 815, 815, 816, 816, 816, 817, 817, 818, 818, 819, 819, 819, 819, 819, 819, 820, 820, 820, 820, 820, 822, 822, 822, 822, 822, 823, 823, 824, 824, 824, 824, 825, 825, 825, 825, 825, 825, 825, 826, 826, 826, 826,
                               826, 826, 826, 826, 826, 826, 826, 826, 827, 827, 827, 828, 828, 828, 828, 830, 830, 830, 830, 831, 831, 831, 831, 831, 831, 831, 832, 832, 832, 833, 833, 833, 833, 833, 833, 834, 834, 834, 834, 834, 834, 834, 834, 835, 835, 835, 835, 835, 835, 835, 836, 836, 836, 837, 837, 837, 837, 837, 837, 838, 839, 839, 839, 839, 840, 840, 840, 840, 841, 841, 841, 841, 841, 842, 842, 842, 842, 842, 843, 843, 844, 844, 844, 844, 845, 845, 846, 846, 846, 847, 847, 847, 847, 847,
                               848, 848, 848, 849, 849, 849, 849, 849, 849, 849, 850, 850, 850, 850, 851, 851, 851, 851, 851, 852, 852, 853, 853, 853, 853, 854, 854, 854, 854, 854, 854, 854, 854, 854, 855, 855, 855, 855, 855, 855, 856, 856, 857, 857, 857, 858, 858, 858, 858, 859, 859, 859, 860, 860, 860, 860, 860, 861, 861, 861, 862, 862, 862, 862, 862, 863, 863, 863, 863, 863, 863, 864, 864, 864, 864, 864, 865, 865, 866, 866, 866, 867, 867, 867, 867, 867, 867, 867, 868, 868, 868, 868, 868, 869,
                               869, 869, 869, 870, 870, 870, 870, 871, 871, 871, 872, 872, 872, 872, 872, 872, 873, 873, 873, 874, 874, 874, 874, 875, 875, 875, 875, 875, 875, 875, 876, 876, 877, 877, 877, 878, 878, 878, 878, 879, 879, 880, 880, 880, 881, 881, 881, 881, 882, 882, 882, 882, 883, 883, 883, 883, 883, 883, 883, 883, 884, 884, 884, 885, 885, 886, 887, 887, 887, 887, 888, 888, 888, 888, 889, 889, 889, 889, 889, 890, 890, 890, 890, 890, 891, 891, 891, 892, 892, 892, 892, 892, 892, 892,
                               893, 893, 893, 893, 894, 894, 894, 895, 895, 895, 896, 897, 897, 897, 897, 897, 897, 898, 898, 898, 898, 899, 899, 899, 899, 900, 900, 901, 901, 901, 901, 901, 902, 902, 902, 902, 902, 902, 902, 903, 903, 903, 903, 903, 903, 904, 904, 904, 904, 904, 904, 904, 905, 905, 905, 905, 906, 906, 906, 906, 906, 906, 906, 908, 908, 908, 908, 909, 909, 909, 909, 910, 910, 910, 911, 911, 912, 912, 913, 913, 913, 914, 914, 915, 915, 915, 916, 916, 916, 916, 916, 916, 916, 917,
                               917, 917, 918, 918, 919, 919, 919, 920, 920, 920, 921, 921, 921, 921, 921, 922, 922, 923, 923, 923, 923, 923, 924, 924, 924, 924, 924, 924, 924, 924, 925, 925, 925, 927, 927, 927, 928, 929, 929, 930, 930, 930, 930, 930, 931, 931, 931, 931, 931, 931, 931, 932, 932, 932, 933, 933, 933, 933, 933, 933, 934, 934, 934, 935, 935, 935, 935, 935, 935, 936, 936, 936, 937, 937, 937, 937, 937, 938, 938, 938, 938, 938, 938, 939, 939, 939, 939, 939, 940, 940, 941, 941, 942, 942,
                               942, 942, 942, 943, 944, 945, 945, 945, 945, 945, 946, 946, 948, 949, 949, 950, 951, 951, 952, 952, 952, 952, 952, 952, 952, 953, 953, 953, 953, 954, 954, 954, 955, 955, 956, 956, 957, 958, 959, 959, 960, 960, 960, 960, 961, 961, 961, 962, 962, 962, 962, 962, 962, 962, 962, 962, 963, 963, 963, 963, 964, 965, 965, 966, 966, 966, 966, 967, 968, 968, 970, 970, 970, 971, 971, 971, 971, 972, 973, 973, 973, 974, 974, 975, 975, 976, 976, 976, 976, 977, 977, 977, 978, 978,
                               978, 979, 979, 979, 980, 980, 980, 980, 980, 980, 980, 981, 981, 981, 982, 982, 982, 982, 982, 983, 984, 986, 986, 986, 987, 987, 987, 988, 988, 988, 988, 988, 988, 990, 990, 990, 991, 991, 991, 992, 992, 993, 993, 994, 994, 995, 996, 996, 997, 997, 998, 998, 999, 1000, 1000, 1001, 1001, 1001, 1001, 1001, 1001, 1002, 1002, 1002, 1004, 1004, 1005, 1005, 1007, 1007, 1007, 1007, 1008, 1009, 1009, 1012, 1013, 1013, 1013, 1014, 1014, 1015, 1015, 1015, 1016, 1016, 1016,
                               1016, 1016, 1016, 1017, 1018, 1018, 1018, 1019, 1019, 1020, 1021, 1022, 1022, 1022, 1023, 1023, 1023, 1024, 1024, 1026, 1026, 1027, 1028, 1028, 1028, 1029, 1029, 1029, 1029, 1030, 1031, 1031, 1033, 1034, 1036, 1038, 1038, 1039, 1040, 1040, 1040, 1041, 1041, 1042, 1042, 1043, 1044, 1045, 1046, 1046, 1046, 1049, 1049, 1049, 1050, 1051, 1051, 1052, 1052, 1053, 1053, 1056, 1057, 1058, 1058, 1059, 1061, 1062, 1063, 1064, 1064, 1065, 1065, 1067, 1067, 1068, 1068, 1070,
                               1071, 1071, 1071, 1074, 1075, 1075, 1075, 1076, 1076, 1077, 1078, 1079, 1080, 1081, 1081, 1081, 1083, 1083, 1084, 1086, 1086, 1090, 1090, 1091, 1092, 1092, 1092, 1094, 1094, 1120, 1139]
        rc = len(self.right_censored)
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored'], 'Value': ['defective_sample', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)')]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class electronics:
    '''
    This dataset is heavily right censored without intermixed censoring (all censored values are greater than the largest failure)
    It is very difficult to fit and requires a specific combination of initial guess (least squares) and optimizer (TNC) to achieve the lowest log-likelihood.
    Thanks to Jiwon Cha for providing this dataset.
    '''

    def __init__(self):
        time = [220, 179, 123, 146, 199, 181, 191, 216, 1, 73, 44798, 62715, 81474, 80632, 62716]
        quantity = [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 817, 823, 815, 813, 804]
        category = ['F', 'F', 'F', 'F', 'F', 'F', 'F', 'F', 'F', 'F', 'C', 'C', 'C', 'C', 'C']
        data = {'time': time, 'quantity': quantity, 'category': category}
        self.dataframe = pd.DataFrame(data, columns=['time', 'quantity', 'category'])
        self.failures = [220, 179, 123, 146, 199, 181, 191, 216, 1, 73]
        self.right_censored = np.hstack([np.ones(817) * 44798, np.ones(823) * 62715, np.ones(815) * 81474, np.ones(813) * 80632, np.ones(804) * 62716])
        rc = len(self.right_censored)
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored'], 'Value': ['defective_sample2', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)')]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class MCF_1:
    '''
    This dataset is formatted for use with the Mean Cumulative Function (MCF_parametric or MCF_nonparametric)
    It consists of failure times for 5 systems.
    It exhibits a fairly constant failure rate, appearing to be slightly increasing (beta > 1)
    '''

    def __init__(self):
        times = [
            [5, 10, 15, 17],
            [6, 13, 17, 19],
            [12, 20, 25, 26],
            [13, 15, 24],
            [16, 22, 25, 28]]
        self.times = times
        self.number_of_systems = len(times)


class MCF_2:
    '''
    This dataset is formatted for use with the Mean Cumulative Function (MCF_parametric or MCF_nonparametric)
    It consists of failure times for 56 systems.
    It exhibits an increasing failure rate at the start and a decreasing failure rate near the end.
    Due to this shape it is not fitted well by the power law model used in MCF parametric.
    '''

    def __init__(self):
        times = [
            [1276, 1317, 1449, 1475, 1475, 1476, 1489, 1492, 1511, 1568, 1575, 1581, 1581, 1582, 1583, 1585, 1588, 1600, 1607, 1613, 1617, 1708, 1716, 1718, 1733, 1737, 1746, 1756, 1763, 1827, 1829, 1935],
            [1054, 1087, 1147, 1156, 1199, 1232, 1276, 1278, 1283, 1320, 1491, 1620, 1660, 1662, 1681, 1702, 1811, 1815, 1818, 1820, 1838, 1945],
            [1803, 2024, 2032, 2033, 2039, 2304, 2418],
            [1436, 2070, 2216, 2222, 2240, 2539],
            [1441, 1799, 1960, 1961, 1964, 2064],
            [776, 830, 836, 966, 1265, 1641],
            [1845, 2132, 2436],
            [2809, 2812, 3116],
            [2102, 2886, 3495],
            [1793, 1800, 3150],
            [1895, 2217, 2301],
            [796, 1499, 2147],
            [1288, 2763],
            [2552, 2896],
            [2731, 2781],
            [1459, 2967],
            [3289],
            [3401],
            [3492],
            [2984],
            [2532],
            [2715],
            [2778],
            [2408],
            [2349],
            [2460],
            [2451],
            [2449],
            [2450],
            [2450],
            [3131],
            [3789],
            [3837],
            [3351],
            [2862],
            [2972],
            [3489],
            [1118],
            [1496],
            [711],
            [277],
            [299],
            [622],
            [253],
            [255],
            [255],
            [255],
            [255],
            [255],
            [255],
            [255],
            [255],
            [255],
            [255],
            [255],
            [151]]
        self.times = times
        self.number_of_systems = len(times)


class ALT_temperature:
    '''
    This is an accelerated life test (ALT) dataset conducted at 3 temperatures
    It should be used with an ALT probability plot
    The dataset contains mostly censored data but is easily fitted by Weibull_2P, Lognormal_2P, and Gamma_2P distributions.
    Normal_2P will fit but the ALT probability plot will not show Normal_2P is a good fit for this dataset
    Sourced from Dr. Mohammad Modarres, University of Maryland
    '''

    def __init__(self):
        self.failures = [1298, 1390, 3187, 3241, 3261, 3313, 4501, 4568, 4841, 4982, 581, 925, 1432, 1586, 2452, 2734, 2772, 4106, 4674, 283, 361, 515, 638, 854, 1024, 1030, 1045, 1767, 1777, 1856, 1951, 1964, 1951, 1964, 2884]
        self.failure_stresses = [40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 60, 60, 60, 60, 60, 60, 60, 60, 60, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80, 80]
        self.right_censored = [5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000,
                               5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000, 5000]
        self.right_censored_stresses = [40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 40, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 60, 80]

        rc = len(self.right_censored)
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_temperature', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), 3]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class ALT_temperature2:
    '''
    This is an accelerated life test (ALT) dataset conducted at 4 temperatures
    It should be used with an ALT probability plot
    This is a relatively small dataset with just 40 values, 20 of which are censored.
    Sourced from Dr. Mohammad Modarres, University of Maryland
    '''

    def __init__(self):
        self.failures = [29.1, 80.7, 47.5, 71.8, 73.7, 86.2, 29.5, 52, 56.3, 63.5, 92.5, 99.5, 26.1, 47.5, 53.4, 56.1, 61.8, 76.6, 77.6, 80.9]
        self.failure_stresses = [300, 300, 350, 350, 350, 350, 400, 400, 400, 400, 400, 400, 500, 500, 500, 500, 500, 500, 500, 500]
        self.right_censored = [100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100, 100]
        self.right_censored_stresses = [300, 300, 300, 300, 300, 300, 300, 300, 350, 350, 350, 350, 350, 350, 400, 400, 400, 400, 500, 500]

        rc = len(self.right_censored)
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_temperature2', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), 4]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class ALT_temperature3:
    '''
    This is an accelerated life test (ALT) dataset conducted at 3 temperatures
    It should be used with an ALT probability plot
    This is a relatively small dataset with just 30 values, none of which are censored.
    '''

    def __init__(self):
        self.failures = [3850, 4340, 4760, 5320, 5740, 6160, 6580, 7140, 7980, 8960, 3300, 3720, 4080, 4560, 4920, 5280, 5640, 6120, 6840, 7680, 2750, 3100, 3400, 3800, 4100, 4400, 4700, 5100, 5700, 6400]
        self.failure_stresses = [393, 393, 393, 393, 393, 393, 393, 393, 393, 393, 408, 408, 408, 408, 408, 408, 408, 408, 408, 408, 423, 423, 423, 423, 423, 423, 423, 423, 423, 423]

        rc = 0
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_temperature3', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), 3]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class ALT_load:
    '''
    This is an accelerated life test (ALT) dataset conducted at 3 loads
    It should be used with an ALT probability plot
    This is a relatively small dataset with just 20 and no censoring
    Sourced from Dr. Mohammad Modarres, University of Maryland
    '''

    def __init__(self):
        self.failures = [250, 460, 530, 730, 820, 970, 970, 1530, 160, 180, 290, 320, 390, 460, 90, 100, 150, 180, 220, 230]
        self.failure_stresses = [200, 200, 200, 200, 200, 200, 200, 200, 300, 300, 300, 300, 300, 300, 466, 466, 466, 466, 466, 466]
        rc = 0
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_load', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), 3]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class ALT_load2:
    '''
    This is an accelerated life test (ALT) dataset conducted at 3 loads
    It should be used with an ALT probability plot
    This is a relatively small dataset with just 18 values, 5 of which are censored.
    Sourced from Dr. Mohammad Modarres, University of Maryland
    '''

    def __init__(self):
        self.failures = [245, 312, 409, 110, 180, 200, 222, 50, 70, 88, 112, 140, 160]
        self.failure_stresses = [100, 100, 100, 200, 200, 200, 200, 300, 300, 300, 300, 300, 300]
        self.right_censored = [500, 500, 500, 250, 250]
        self.right_censored_stresses = [100, 100, 100, 200, 200]

        rc = len(self.right_censored)
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_load2', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), 3]}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class ALT_temperature_voltage:
    '''
    This is an accelerated life test (ALT) dataset conducted at 2 different temperatures and 2 different voltages
    The dataset is small but contains no censored values.
    It is recommended to use a dual-stress model such as the Power-Exponential model.
    '''

    def __init__(self):
        self.failures = [620, 632, 685, 822, 380, 416, 460, 596, 216, 146, 332, 400]
        self.failure_stress_temp = [348, 348, 348, 348, 348, 348, 348, 348, 378, 378, 378, 378]
        self.failure_stress_voltage = [3, 3, 3, 3, 5, 5, 5, 5, 3, 3, 3, 3]

        rc = 0
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_temperature_voltage2', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), '2 temperatures, 2 voltages']}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class ALT_temperature_voltage2:
    '''
    This is an accelerated life test (ALT) dataset conducted at 3 different temperatures and 2 different voltages
    The dataset is fairly small and the pattern of stresses make it extremely difficult to fit.
    Note that there is stress-pair that contains only a single failure.
    It is recommended to use a dual-stress model.
    '''

    def __init__(self):
        self.failures = [1200, 1205, 1509, 1560, 1780, 2390, 2490, 2690, 1466, 1536, 1698, 1784, 2689, 222, 250, 297, 354, 368]
        self.failure_stress_temp = [350, 350, 350, 350, 350, 350, 350, 350, 378, 378, 378, 378, 378, 398, 398, 398, 398, 398]
        self.failure_stress_voltage = [10, 10, 10, 10, 10, 12, 12, 12, 10, 10, 10, 10, 12, 10, 10, 10, 10, 10]

        self.right_censored = [2500, 2500, 2500, 2500, 2500, 2500, 2500, 2500]
        self.right_censored_stress_temp = [350, 350, 350, 378, 378, 378, 378, 378]
        self.right_censored_stress_voltage = [12, 12, 12, 12, 12, 12, 12, 12]

        rc = len(self.right_censored)
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_temperature_voltage', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), '3 temperatures, 2 voltages']}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df


class ALT_temperature_humidity:
    '''
    This is an accelerated life test (ALT) dataset conducted at 2 different temperatures and 2 different humidities
    The dataset is fairly small but has no censored data
    It is recommended to use a dual-stress model such as Dual-Exponential model
    '''

    def __init__(self):
        self.failures = [310, 316, 329, 411, 190, 208, 230, 298, 108, 123, 166, 200]
        self.failure_stress_temp = [378, 378, 378, 378, 378, 378, 378, 378, 398, 398, 398, 398]
        self.failure_stress_humidity = [0.4, 0.4, 0.4, 0.4, 0.8, 0.8, 0.8, 0.8, 0.4, 0.4, 0.4, 0.4]

        rc = 0
        f = len(self.failures)
        tot = f + rc
        data = {'Stat': ['Name', 'Total Values', 'Failures', 'Right Censored', 'Number of stresses'], 'Value': ['ALT_temperature_humidity', tot, str(str(f) + ' (' + str(round(f / tot * 100, 2)) + '%)'), str(str(rc) + ' (' + str(round(rc / tot * 100, 2)) + '%)'), '2 temperatures, 2 humidities']}
        df = pd.DataFrame(data, columns=['Stat', 'Value'])
        blankIndex = [''] * len(df)
        df.index = blankIndex
        self.info = df
