## rlbot-flatbuffers-py

A Python module implemented in Rust for serializing and deserializing RLBot's flatbuffers

### The goal of this project

To provide a fast, safe, and easy to use Python module for serializing and deserializing RLBot's flatbuffers.

A majority of the code is generated in the `codegen/` upon first compile and thrown into `src/python`.

This includes the code generated by `flatc` (living in `src/generated`), the Python wrapper binds to the generated Rust code, and the Python type hints (`rlbot_flatbuffers.pyi`).

### Basic usage

All classes and methods should have types hints readable by your IDE, removing the guesswork of common operations.

#### Creating

```python
import rlbot_flatbuffers as flat

desired_ball = flat.DesiredBallState(
    physics=flat.Physics(
        location=flat.Vector3Partial(z=200),
        velocity=flat.Vector3Partial(x=1500, y=1500),
        angular_velocity=flat.Vector3Partial(),
    ),
)

desired_game_info = flat.DesiredGameInfoState(
    world_gravity_z=-100,
    game_speed=2,
)

desired_game_state = flat.DesiredGameState(
    ball_state=desired_ball,
    game_info_state=desired_game_info,
)
```

In the above code, we:

- Set the ball to:
  - Location (0, 0, 200)
  - Velocity (1500, 1500, 0)
  - Angular velocity of (0, 0, 0)
- Don't set the car states
- Set the game info state:
  - World gravity to -100
  - Game speed to 2x default
  - Don't set end match or paused
- Don't set any console commands

All values are optional when creating a class and have the proper defaults.

#### Reading values

```python
import rlbot_flatbuffers as flat

def handle_packet(packet: flat.GameTickPacket):
    if packet.game_info.game_state_type not in {
        flat.GameStateType.Active,
        flat.GameStateType.Kickoff,
    }:
        # Return early if the game isn't active
        return

    # Print the ball's location
    print(packet.ball.physics.location)

    for car in packet.players:
        # Print the every car's location
        print(car.physics.location)
```

The goal of the above was to feel familiar to RLBot v4 while providing a more Pythonic interface.

- All classes (not enums and unions) implement `__match_args__` for easy destructuring via the `match`/`case` pattern.
  - Enums and unions and can still be used to match against the type,
    they just can't be destructured.
- Every class implements `__str__`, `__repr__`, and `__hash__` methods.
  - All enums also implement `__int__` and `__eq__`.
- Lists no longer have `num_x` fields accompanying them,
  they are just Python lists of the appropriate length.
