import logging
from typing import Any, Dict, List

from tornado.web import RequestHandler

logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)


def get_client_protocol(handler: RequestHandler) -> Dict[str, str]:
    """
    This is a copy of the jupyterhub-ltiauthenticator logic to get the first
    protocol value from the x-forwarded-proto list, assuming there is more than
    one value. Otherwise, this returns the value as-is.

    Extracted as a method to facilitate testing.

    Args:
        handler: a tornado.web.RequestHandler object

    Returns:
        A decoded dict with keys/values extracted from the request's arguments
    """
    if "x-forwarded-proto" in handler.request.headers:
        hops = [
            h.strip() for h in handler.request.headers["x-forwarded-proto"].split(",")
        ]
        protocol = hops[0]
    else:
        protocol = handler.request.protocol

    return protocol


def convert_request_to_dict(arguments: Dict[str, List[bytes]]) -> Dict[str, Any]:
    """
    Converts the arguments obtained from a request to a dict.

    Args:
        handler: a tornado.web.RequestHandler object

    Returns:
        A decoded dict with keys/values extracted from the request's arguments
    """
    args = {}
    for k, values in arguments.items():
        args[k] = values[0].decode()
    return args
