"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TagMutability = exports.Repository = exports.RepositoryBase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const ecr_generated_1 = require("./ecr.generated");
const lifecycle_1 = require("./lifecycle");
/**
 * Base class for ECR repository.
 *
 * Reused between imported repositories and owned repositories.
 *
 * @stability stable
 */
class RepositoryBase extends core_1.Resource {
    /**
     * The URI of this repository (represents the latest image):.
     *
     * ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY
     *
     * @stability stable
     */
    get repositoryUri() {
        return this.repositoryUriForTag();
    }
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     * ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[:TAG]
     *
     * @param tag Optional image tag.
     * @stability stable
     */
    repositoryUriForTag(tag) {
        const tagSuffix = tag ? `:${tag}` : '';
        return this.repositoryUriWithSuffix(tagSuffix);
    }
    /**
     * Returns the URL of the repository. Can be used in `docker push/pull`.
     *
     * ACCOUNT.dkr.ecr.REGION.amazonaws.com/REPOSITORY[@DIGEST]
     *
     * @param digest Optional image digest.
     * @stability stable
     */
    repositoryUriForDigest(digest) {
        const digestSuffix = digest ? `@${digest}` : '';
        return this.repositoryUriWithSuffix(digestSuffix);
    }
    /**
     * Returns the repository URI, with an appended suffix, if provided.
     * @param suffix An image tag or an image digest.
     * @private
     */
    repositoryUriWithSuffix(suffix) {
        const parts = this.stack.parseArn(this.repositoryArn);
        return `${parts.account}.dkr.ecr.${parts.region}.${this.stack.urlSuffix}/${this.repositoryName}${suffix}`;
    }
    /**
     * Define a CloudWatch event that triggers when something happens to this repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.ecr'],
            detailType: ['AWS API Call via CloudTrail'],
            detail: {
                requestParameters: {
                    repositoryName: [this.repositoryName],
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image is pushed to this repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailImagePushed(id, options = {}) {
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: ['PutImage'],
                requestParameters: {
                    imageTag: options.imageTag ? [options.imageTag] : undefined,
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event rule that can trigger a target when an image scan is completed.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onImageScanCompleted(id, options = {}) {
        var _c;
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.ecr'],
            detailType: ['ECR Image Scan'],
            detail: {
                'repository-name': [this.repositoryName],
                'scan-status': ['COMPLETE'],
                'image-tags': (_c = options.imageTags) !== null && _c !== void 0 ? _c : undefined,
            },
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule which triggers for repository events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     *
     * @stability stable
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.ecr'],
            resources: [this.repositoryArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * Grant the given principal identity permissions to perform the actions on this repository.
     *
     * @stability stable
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions,
            resourceArns: [this.repositoryArn],
            resourceSelfArns: [],
            resource: this,
        });
    }
    /**
     * Grant the given identity permissions to use the images in this repository.
     *
     * @stability stable
     */
    grantPull(grantee) {
        const ret = this.grant(grantee, 'ecr:BatchCheckLayerAvailability', 'ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage');
        iam.Grant.addToPrincipal({
            grantee,
            actions: ['ecr:GetAuthorizationToken'],
            resourceArns: ['*'],
            scope: this,
        });
        return ret;
    }
    /**
     * Grant the given identity permissions to pull and push images to this repository.
     *
     * @stability stable
     */
    grantPullPush(grantee) {
        this.grantPull(grantee);
        return this.grant(grantee, 'ecr:PutImage', 'ecr:InitiateLayerUpload', 'ecr:UploadLayerPart', 'ecr:CompleteLayerUpload');
    }
}
exports.RepositoryBase = RepositoryBase;
_a = JSII_RTTI_SYMBOL_1;
RepositoryBase[_a] = { fqn: "@aws-cdk/aws-ecr.RepositoryBase", version: "1.106.1" };
/**
 * Define an ECR repository.
 *
 * @stability stable
 */
class Repository extends RepositoryBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.repositoryName,
        });
        this.lifecycleRules = new Array();
        Repository.validateRepositoryName(this.physicalName);
        const resource = new ecr_generated_1.CfnRepository(this, 'Resource', {
            repositoryName: this.physicalName,
            // It says "Text", but they actually mean "Object".
            repositoryPolicyText: core_1.Lazy.any({ produce: () => this.policyDocument }),
            lifecyclePolicy: core_1.Lazy.any({ produce: () => this.renderLifecyclePolicy() }),
            imageScanningConfiguration: !props.imageScanOnPush ? undefined : {
                ScanOnPush: true,
            },
            imageTagMutability: props.imageTagMutability || undefined,
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.registryId = props.lifecycleRegistryId;
        if (props.lifecycleRules) {
            props.lifecycleRules.forEach(this.addLifecycleRule.bind(this));
        }
        this.repositoryName = this.getResourceNameAttribute(resource.ref);
        this.repositoryArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'ecr',
            resource: 'repository',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import a repository.
     *
     * @stability stable
     */
    static fromRepositoryAttributes(scope, id, attrs) {
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = attrs.repositoryName;
                this.repositoryArn = attrs.repositoryArn;
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    /**
     * @stability stable
     */
    static fromRepositoryArn(scope, id, repositoryArn) {
        // if repositoryArn is a token, the repository name is also required. this is because
        // repository names can include "/" (e.g. foo/bar/myrepo) and it is impossible to
        // parse the name from an ARN using CloudFormation's split/select.
        if (core_1.Token.isUnresolved(repositoryArn)) {
            throw new Error('"repositoryArn" is a late-bound value, and therefore "repositoryName" is required. Use `fromRepositoryAttributes` instead');
        }
        const repositoryName = repositoryArn.split('/').slice(1).join('/');
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = repositoryArn;
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    /**
     * @stability stable
     */
    static fromRepositoryName(scope, id, repositoryName) {
        class Import extends RepositoryBase {
            constructor() {
                super(...arguments);
                this.repositoryName = repositoryName;
                this.repositoryArn = Repository.arnForLocalRepository(repositoryName, scope);
            }
            addToResourcePolicy(_statement) {
                // dropped
                return { statementAdded: false };
            }
        }
        return new Import(scope, id);
    }
    /**
     * Returns an ECR ARN for a repository that resides in the same account/region as the current stack.
     *
     * @stability stable
     */
    static arnForLocalRepository(repositoryName, scope, account) {
        return core_1.Stack.of(scope).formatArn({
            account,
            service: 'ecr',
            resource: 'repository',
            resourceName: repositoryName,
        });
    }
    static validateRepositoryName(physicalName) {
        const repositoryName = physicalName;
        if (!repositoryName || core_1.Token.isUnresolved(repositoryName)) {
            // the name is a late-bound value, not a defined string,
            // so skip validation
            return;
        }
        const errors = [];
        // Rules codified from https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ecr-repository.html
        if (repositoryName.length < 2 || repositoryName.length > 256) {
            errors.push('Repository name must be at least 2 and no more than 256 characters');
        }
        const isPatternMatch = /^(?:[a-z0-9]+(?:[._-][a-z0-9]+)*\/)*[a-z0-9]+(?:[._-][a-z0-9]+)*$/.test(repositoryName);
        if (!isPatternMatch) {
            errors.push('Repository name must follow the specified pattern: (?:[a-z0-9]+(?:[._-][a-z0-9]+)*/)*[a-z0-9]+(?:[._-][a-z0-9]+)*');
        }
        if (errors.length > 0) {
            throw new Error(`Invalid ECR repository name (value: ${repositoryName})${os_1.EOL}${errors.join(os_1.EOL)}`);
        }
    }
    /**
     * Add a policy statement to the repository's resource policy.
     *
     * @stability stable
     */
    addToResourcePolicy(statement) {
        if (this.policyDocument === undefined) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
        return { statementAdded: false, policyDependable: this.policyDocument };
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        var _c;
        const errors = super.validate();
        errors.push(...((_c = this.policyDocument) === null || _c === void 0 ? void 0 : _c.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * Add a life cycle rule to the repository.
     *
     * Life cycle rules automatically expire images from the repository that match
     * certain conditions.
     *
     * @stability stable
     */
    addLifecycleRule(rule) {
        // Validate rule here so users get errors at the expected location
        if (rule.tagStatus === undefined) {
            rule = { ...rule, tagStatus: rule.tagPrefixList === undefined ? lifecycle_1.TagStatus.ANY : lifecycle_1.TagStatus.TAGGED };
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.TAGGED && (rule.tagPrefixList === undefined || rule.tagPrefixList.length === 0)) {
            throw new Error('TagStatus.Tagged requires the specification of a tagPrefixList');
        }
        if (rule.tagStatus !== lifecycle_1.TagStatus.TAGGED && rule.tagPrefixList !== undefined) {
            throw new Error('tagPrefixList can only be specified when tagStatus is set to Tagged');
        }
        if ((rule.maxImageAge !== undefined) === (rule.maxImageCount !== undefined)) {
            throw new Error(`Life cycle rule must contain exactly one of 'maxImageAge' and 'maxImageCount', got: ${JSON.stringify(rule)}`);
        }
        if (rule.tagStatus === lifecycle_1.TagStatus.ANY && this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY).length > 0) {
            throw new Error('Life cycle can only have one TagStatus.Any rule');
        }
        this.lifecycleRules.push({ ...rule });
    }
    /**
     * Render the life cycle policy object
     */
    renderLifecyclePolicy() {
        const stack = core_1.Stack.of(this);
        let lifecyclePolicyText;
        if (this.lifecycleRules.length === 0 && !this.registryId) {
            return undefined;
        }
        if (this.lifecycleRules.length > 0) {
            lifecyclePolicyText = JSON.stringify(stack.resolve({
                rules: this.orderedLifecycleRules().map(renderLifecycleRule),
            }));
        }
        return {
            lifecyclePolicyText,
            registryId: this.registryId,
        };
    }
    /**
     * Return life cycle rules with automatic ordering applied.
     *
     * Also applies validation of the 'any' rule.
     */
    orderedLifecycleRules() {
        var _c;
        if (this.lifecycleRules.length === 0) {
            return [];
        }
        const prioritizedRules = this.lifecycleRules.filter(r => r.rulePriority !== undefined && r.tagStatus !== lifecycle_1.TagStatus.ANY);
        const autoPrioritizedRules = this.lifecycleRules.filter(r => r.rulePriority === undefined && r.tagStatus !== lifecycle_1.TagStatus.ANY);
        const anyRules = this.lifecycleRules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY);
        if (anyRules.length > 0 && anyRules[0].rulePriority !== undefined && autoPrioritizedRules.length > 0) {
            // Supporting this is too complex for very little value. We just prohibit it.
            throw new Error("Cannot combine prioritized TagStatus.Any rule with unprioritized rules. Remove rulePriority from the 'Any' rule.");
        }
        const prios = prioritizedRules.map(r => r.rulePriority);
        let autoPrio = (prios.length > 0 ? Math.max(...prios) : 0) + 1;
        const ret = new Array();
        for (const rule of prioritizedRules.concat(autoPrioritizedRules).concat(anyRules)) {
            ret.push({
                ...rule,
                rulePriority: (_c = rule.rulePriority) !== null && _c !== void 0 ? _c : autoPrio++,
            });
        }
        // Do validation on the final array--might still be wrong because the user supplied all prios, but incorrectly.
        validateAnyRuleLast(ret);
        return ret;
    }
}
exports.Repository = Repository;
_b = JSII_RTTI_SYMBOL_1;
Repository[_b] = { fqn: "@aws-cdk/aws-ecr.Repository", version: "1.106.1" };
function validateAnyRuleLast(rules) {
    const anyRules = rules.filter(r => r.tagStatus === lifecycle_1.TagStatus.ANY);
    if (anyRules.length === 1) {
        const maxPrio = Math.max(...rules.map(r => r.rulePriority));
        if (anyRules[0].rulePriority !== maxPrio) {
            throw new Error(`TagStatus.Any rule must have highest priority, has ${anyRules[0].rulePriority} which is smaller than ${maxPrio}`);
        }
    }
}
/**
 * Render the lifecycle rule to JSON
 */
function renderLifecycleRule(rule) {
    var _c, _d;
    return {
        rulePriority: rule.rulePriority,
        description: rule.description,
        selection: {
            tagStatus: rule.tagStatus || lifecycle_1.TagStatus.ANY,
            tagPrefixList: rule.tagPrefixList,
            countType: rule.maxImageAge !== undefined ? "sinceImagePushed" /* SINCE_IMAGE_PUSHED */ : "imageCountMoreThan" /* IMAGE_COUNT_MORE_THAN */,
            countNumber: (_d = (_c = rule.maxImageAge) === null || _c === void 0 ? void 0 : _c.toDays()) !== null && _d !== void 0 ? _d : rule.maxImageCount,
            countUnit: rule.maxImageAge !== undefined ? 'days' : undefined,
        },
        action: {
            type: 'expire',
        },
    };
}
/**
 * The tag mutability setting for your repository.
 *
 * @stability stable
 */
var TagMutability;
(function (TagMutability) {
    TagMutability["MUTABLE"] = "MUTABLE";
    TagMutability["IMMUTABLE"] = "IMMUTABLE";
})(TagMutability = exports.TagMutability || (exports.TagMutability = {}));
//# sourceMappingURL=data:application/json;base64,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