import pickle
from parse import compile
from transliterate import translit
from typing import Set, List, Dict
chars_to_remove = {
    '!',
    '"',
    '#',
    '%',
    '&',
    '(',
    ')',
    '*',
    '+',
    ',',
    '-',
    '.',
    '/',
    ':',
    ';',
    '<',
    '=',
    '>',
    '?',
    '[',
    ']',
    '_',
    '`',
    '«',
    '°',
    '²',
    '³',
    'µ',
    '·',
    '»',
    '½',
    '‑',
    '–',
    '‘',
    '’',
    '“',
    '”',
    '„',
    '•',
    '…',
    '‰',
    '″',
    '₂',
    '₃',
    '€',
    '™',
    '→',
    '−',
    '∕',
    '😀',
    '😉',
    '🙁',
    '🙂'

}


def is_alpha(token: str) -> bool:
    """Checks if the input string is strictly lowercase without numerals.

    Args:
        token (str): Input text.

    Returns:
        bool: Result of checking.
    """    
    import re
    pattern = "^[a-zšđčćž]+$"
    compiled_pattern = re.compile(pattern)
    return bool(compiled_pattern.match(token))

def preprocess(s: str) -> str:
    """Removes unusual characters and lowercases the string.

    Args:
        s (str): input string.

    Returns:
        str: output string.
    """
    for c in chars_to_remove:
        s = s.replace(c, "")
    s = s.casefold()
    return s


def count_variants(s: str, lex: dict):
    """Counts the variant specific words in the preprocessed input string based on the lexicon lex.

    Returns tuple (counts, per_token_breakdown).
    Counts look like this:
        {"A":3, "B":0}.
    per_token is a dictionary with all the words detected, their counts and their variant:
        {"word1":
            {"count":3, "variant":"A"}
        }

    Args:
        s (str): Input string.
        lex (dict): Lexicon.

    Returns:
        results (tuple): (counts, per_token).
    """
    counts = dict()
    per_token = dict()
    for word in preprocess(s).split():
        if not is_alpha(word):
            continue
        variant = lex.get(word, None)
        if not variant:
            continue
        counts[variant] = counts.get(variant, 0) + 1
        if word in per_token.keys():
            per_token[word]["count"] += 1
        else:
            per_token[word] = {"variant": variant, "count": 1}
    return counts, per_token



def counts_to_category(counts: dict) -> str:
    """Discretizes counts like {"A": 2, "B":0} to
    categories A, B, MIX, UNK.

    Args:
        counts (dict): result of count_variants function.

    Returns:
        str: category.
    """    
    A = counts.get("A", 0)
    B = counts.get("B",0)

    if A > 2*B:
        return "A"
    elif B > 2*A:
        return "B"
    elif A == B == 0:
        return "UNK"
    else:
        return "MIX"

def get_variant(text: str) -> str:
    """Quick way to classify text. 

    Loads the lexicon, preprocesses the string. Returns the predicted 
    category {'A', 'B', 'UNK', 'MIX'}.

    Args:
        text (str): input string.

    Returns:
        str: category {'A', 'B', 'UNK', 'MIX'}
    """    
    variant_detector_count = count_variants(text, lex)[0]
    return counts_to_category(variant_detector_count) 


lex = {'abettor': 'A', 'abetter': 'B', 'abettors': 'A', 'abetters': 'B', "abettor's": 'A', "abetter's": 'B', 'abridgment': 'A', 'abridgement': 'B', 'abridgments': 'A', 'abridgements': 'B', "abridgment's": 'A', "abridgement's": 'B', 'accouter': 'A', 'accoutre': 'B', 'accoutered': 'A', 'accoutred': 'B', 'accoutering': 'A', 'accoutring': 'B', 'accouters': 'A', 'accoutres': 'B', 'accouterment': 'A', 'accoutrement': 'B', 'accouterments': 'A', 'accoutrements': 'B', "accouterment's": 'A', "accoutrement's": 'B', 'acknowledgment': 'A', 'acknowledgement': 'B', 'acknowledgments': 'A', 'acknowledgements': 'B', "acknowledgment's": 'A', "acknowledgement's": 'B', 'adapter': 'A', 'adaptor': 'B', 'adapters': 'A', 'adaptors': 'B', "adapter's": 'A', "adaptor's": 'B', 'aging': 'A', 'ageing': 'B', 'agings': 'A', 'ageings': 'B', "aging's": 'A', "ageing's": 'B', 'airfoil': 'A', 'aerofoil': 'B', 'airfoils': 'A', 'aerofoils': 'B', "airfoil's": 'A', "aerofoil's": 'B', 'airplane': 'A', 'aeroplane': 'B', 'airplanes': 'A', 'aeroplanes': 'B', "airplane's": 'A', "aeroplane's": 'B', 'aluminum': 'A', 'aluminium': 'B', 'aluminums': 'A', 'aluminiums': 'B', "aluminum's": 'A', "aluminium's": 'B', 'ambiance': 'A', 'ambience': 'B', 'ambiances': 'A', 'ambiences': 'B', "ambiance's": 'A', "ambience's": 'B', 'amenorrhea': 'A', 'amenorrhoea': 'B', 'amenorrheas': 'A', 'amenorrhoeas': 'B', "amenorrhea's": 'A', "amenorrhoea's": 'B', 'amphitheater': 'A', 'amphitheatre': 'B', 'amphitheaters': 'A', 'amphitheatres': 'B', "amphitheater's": 'A', "amphitheatre's": 'B', 'analyzable': 'A', 'analysable': 'B', 'analyzabler': 'A', 'analysabler': 'B', 'analyzablest': 'A', 'analysablest': 'B', 'analyze': 'A', 'analyse': 'B', 'analyzed': 'A', 'analysed': 'B', 'analyzing': 'A', 'analysing': 'B', 'analyzes': 'A', 'analyzer': 'A', 'analyser': 'B', 'analyzers': 'A', 'analysers': 'B', "analyzer's": 'A', "analyser's": 'B', 'anemia': 'A', 'anaemia': 'B', 'anemias': 'A', 'anaemias': 'B', "anemia's": 'A', "anaemia's": 'B', 'anemic': 'A', 'anaemic': 'B', 'anemicer': 'A', 'anaemicer': 'B', 'anemicest': 'A', 'anaemicest': 'B', 'anemics': 'A', 'anaemics': 'B', 'anesthesia': 'A', 'anaesthesia': 'B', 'anesthesias': 'A', 'anaesthesias': 'B', "anesthesia's": 'A', "anaesthesia's": 'B', 'anesthesiologist': 'A', 'anaesthesiologist': 'B', 'anesthesiologists': 'A', 'anaesthesiologists': 'B', "anesthesiologist's": 'A', "anaesthesiologist's": 'B', 'anesthesiology': 'A', 'anaesthesiology': 'B', 'anesthesiologies': 'A', 'anaesthesiologies': 'B', "anesthesiology's": 'A', "anaesthesiology's": 'B', 'anesthetic': 'A', 'anaesthetic': 'B', 'anestheticer': 'A', 'anaestheticer': 'B', 'anestheticest': 'A', 'anaestheticest': 'B', 'anesthetics': 'A', 'anaesthetics': 'B', "anesthetic's": 'A', "anaesthetic's": 'B', 'anesthetist': 'A', 'anaesthetist': 'B', 'anesthetists': 'A', 'anaesthetists': 'B', "anesthetist's": 'A', "anaesthetist's": 'B', 'apnea': 'A', 'apnoea': 'B', 'apneas': 'A', 'apnoeas': 'B', "apnea's": 'A', "apnoea's": 'B', 'appall': 'A', 'appal': 'B', 'appalls': 'A', 'appals': 'B', 'appareled': 'A', 'apparelled': 'B', 'appareling': 'A', 'apparelling': 'B', 'arbor': 'A', 'arbour': 'B', 'arbors': 'A', 'arbours': 'B', 'arbores': 'A', 'arboures': 'B', "arbor's": 'A', "arbour's": 'B', 'arbored': 'A', 'arboured': 'B', 'ardor': 'A', 'ardour': 'B', 'ardors': 'A', 'ardours': 'B', "ardor's": 'A', "ardour's": 'B', 'armor': 'A', 'armour': 'B', 'armored': 'A', 'armoured': 'B', 'armoring': 'A', 'armouring': 'B', 'armors': 'A', 'armours': 'B', "armor's": 'A', "armour's": 'B', 'armorer': 'A', 'armourer': 'B', 'armorers': 'A', 'armourers': 'B', "armorer's": 'A', "armourer's": 'B', 'armory': 'A', 'armoury': 'B', 'armories': 'A', 'armouries': 'B', "armory's": 'A', "armoury's": 'B', 'artifact': 'A', 'artefact': 'B', "artifact's": 'A', "artefact's": 'B', 'artifacts': 'A', 'artefacts': 'B', 'asafetida': 'A', 'asafoetida': 'B', 'asafetidas': 'A', 'asafoetidas': 'B', "asafetida's": 'A', "asafoetida's": 'B', 'asshole': 'A', 'arsehole': 'B', 'assholes': 'A', 'arseholes': 'B', "asshole's": 'A', "arsehole's": 'B', 'ax': 'A', 'axe': 'B', "ax's": 'A', "axe's": 'B', 'balk': 'A', 'baulk': 'B', "balk's": 'A', "baulk's": 'B', 'balked': 'A', 'baulked': 'B', 'balking': 'A', 'baulking': 'B', 'balks': 'A', 'baulks': 'B', 'barreled': 'A', 'barrelled': 'B', 'barreling': 'A', 'barrelling': 'B', 'bedeviled': 'A', 'bedevilled': 'B', 'bedeviling': 'A', 'bedevilling': 'B', 'behavior': 'A', 'behaviour': 'B', 'behaviors': 'A', 'behaviours': 'B', "behavior's": 'A', "behaviour's": 'B', 'behavioral': 'A', 'behavioural': 'B', 'behavioraler': 'A', 'behaviouraler': 'B', 'behavioralest': 'A', 'behaviouralest': 'B', 'behaviorally': 'A', 'behaviourally': 'B', 'behaviorism': 'A', 'behaviourism': 'B', 'behaviorisms': 'A', 'behaviourisms': 'B', "behaviorism's": 'A', "behaviourism's": 'B', 'behaviorist': 'A', 'behaviourist': 'B', 'behaviorists': 'A', 'behaviourists': 'B', "behaviorist's": 'A', "behaviourist's": 'B', 'behavioristic': 'A', 'behaviouristic': 'B', 'behavioristicer': 'A', 'behaviouristicer': 'B', 'behavioristicest': 'A', 'behaviouristicest': 'B', 'behavioristics': 'A', 'behaviouristics': 'B', 'behoove': 'A', 'behove': 'B', 'behooved': 'A', 'behoved': 'B', 'behooving': 'A', 'behoving': 'B', 'behooves': 'A', 'behoves': 'B', "behoove's": 'A', "behove's": 'B', 'behoovings': 'A', 'behovings': 'B', "behooving's": 'A', "behoving's": 'B', 'bejeweled': 'A', 'bejewelled': 'B', 'bejeweling': 'A', 'bejewelling': 'B', 'belabor': 'A', 'belabour': 'B', 'belabored': 'A', 'belaboured': 'B', 'belaboring': 'A', 'belabouring': 'B', 'belabors': 'A', 'belabours': 'B', "belabor's": 'A', "belabour's": 'B', 'beveled': 'A', 'bevelled': 'B', 'beveler': 'A', 'beveller': 'B', 'bevelers': 'A', 'bevellers': 'B', "beveler's": 'A', "beveller's": 'B', 'beveling': 'A', 'bevelling': 'B', 'bevelings': 'A', 'bevellings': 'B', 'bister': 'A', 'bistre': 'B', 'bisters': 'A', 'bistres': 'B', "bister's": 'A', "bistre's": 'B', 'blamable': 'A', 'blameable': 'B', 'blond': 'A', 'blonde': 'B', 'blonds': 'A', 'blondes': 'B', "blond's": 'A', "blonde's": 'B', 'blowzy': 'A', 'blowsy': 'B', 'blowzier': 'A', 'blowsier': 'B', 'blowziest': 'A', 'blowsiest': 'B', 'breathalyze': 'A', 'breathalyse': 'B', 'breathalyzed': 'A', 'breathalysed': 'B', 'breathalyzing': 'A', 'breathalysing': 'B', 'breathalyzes': 'A', 'breathalyzer': 'A', 'breathalyser': 'B', "breathalyzer's": 'A', "breathalyser's": 'B', 'breathalyzers': 'A', 'breathalysers': 'B', 'brier': 'A', 'briar': 'B', 'briers': 'A', 'briars': 'B', "brier's": 'A', "briar's": 'B', 'caftan': 'A', 'kaftan': 'B', 'caftans': 'A', 'kaftans': 'B', "caftan's": 'A', "kaftan's": 'B', 'caliber': 'A', 'calibre': 'B', 'calibers': 'A', 'calibres': 'B', "caliber's": 'A', "calibre's": 'B', 'canceled': 'A', 'cancelled': 'B', 'canceling': 'A', 'cancelling': 'B', 'canceler': 'A', 'canceller': 'B', 'cancelers': 'A', 'cancellers': 'B', "canceler's": 'A', "canceller's": 'B', 'candor': 'A', 'candour': 'B', 'candors': 'A', 'candours': 'B', "candor's": 'A', "candour's": 'B', 'caroled': 'A', 'carolled': 'B', 'caroling': 'A', 'carolling': 'B', 'caroler': 'A', 'caroller': 'B', 'carolers': 'A', 'carollers': 'B', "caroler's": 'A', "caroller's": 'B', 'catalog': 'A', 'catalogue': 'B', 'cataloged': 'A', 'catalogued': 'B', 'cataloging': 'A', 'cataloguing': 'B', 'catalogs': 'A', 'catalogues': 'B', "catalog's": 'A', "catalogue's": 'B', 'cataloger': 'A', 'cataloguer': 'B', 'catalogers': 'A', 'cataloguers': 'B', "cataloger's": 'A', "cataloguer's": 'B', 'catalyze': 'A', 'catalyse': 'B', 'catalyzed': 'A', 'catalysed': 'B', 'catalyzing': 'A', 'catalysing': 'B', 'catalyzes': 'A', 'caviled': 'A', 'cavilled': 'B', 'caviling': 'A', 'cavilling': 'B', 'cavilings': 'A', 'cavillings': 'B', 'caviler': 'A', 'caviller': 'B', 'cavilers': 'A', 'cavillers': 'B', "caviler's": 'A', "caviller's": 'B', 'celom': 'A', 'coelom': 'B', 'coeloms': 'B', 'celoms': 'A', "celom's": 'A', "coelom's": 'B', 'cenobite': 'A', 'coenobite': 'B', 'cenobites': 'A', 'coenobites': 'B', "cenobite's": 'A', "coenobite's": 'B', 'cenobitic': 'A', 'coenobitic': 'B', 'cenobitical': 'A', 'coenobitical': 'B', 'center': 'A', 'centre': 'B', 'centered': 'A', 'centred': 'B', 'centering': 'A', 'centring': 'B', "center's": 'A', "centre's": 'B', 'centerings': 'A', 'centrings': 'B', "centering's": 'A', "centring's": 'B', 'centers': 'A', 'centres': 'B', 'centerboard': 'A', 'centreboard': 'B', 'centerboards': 'A', 'centreboards': 'B', "centerboard's": 'A', "centreboard's": 'B', 'centerfold': 'A', 'centrefold': 'B', 'centerfolds': 'A', 'centrefolds': 'B', "centerfold's": 'A', "centrefold's": 'B', 'centerpiece': 'A', 'centrepiece': 'B', 'centerpieces': 'A', 'centrepieces': 'B', "centerpiece's": 'A', "centrepiece's": 'B', 'centiliter': 'A', 'centilitre': 'B', 'centiliters': 'A', 'centilitres': 'B', "centiliter's": 'A', "centilitre's": 'B', 'centimeter': 'A', 'centimetre': 'B', 'centimeters': 'A', 'centimetres': 'B', "centimeter's": 'A', "centimetre's": 'B', 'cesium': 'A', 'caesium': 'B', 'cesiums': 'A', 'caesiums': 'B', "cesium's": 'A', "caesium's": 'B', 'cesural': 'A', 'caesural': 'B', 'channeled': 'A', 'channelled': 'B', 'channeling': 'A', 'channelling': 'B', 'chaperon': 'A', 'chaperone': 'B', 'chaperons': 'A', 'chaperones': 'B', "chaperon's": 'A', "chaperone's": 'B', 'checkbook': 'A', 'chequebook': 'B', 'checkbooks': 'A', 'chequebooks': 'B', "checkbook's": 'A', "chequebook's": 'B', 'checkerboard': 'A', 'chequerboard': 'B', 'checkerboards': 'A', 'chequerboards': 'B', "checkerboard's": 'A', "chequerboard's": 'B', 'chilli': 'B', 'chili': 'A', "chilli's": 'B', "chili's": 'A', 'chiseled': 'A', 'chiselled': 'B', 'chiseling': 'A', 'chiselling': 'B', 'chiseler': 'A', 'chiseller': 'B', 'chiselers': 'A', 'chisellers': 'B', "chiseler's": 'A', "chiseller's": 'B', 'chocolaty': 'A', 'chocolatey': 'B', 'clamor': 'A', 'clamour': 'B', 'clamored': 'A', 'clamoured': 'B', 'clamoring': 'A', 'clamouring': 'B', 'clamors': 'A', 'clamours': 'B', "clamor's": 'A', "clamour's": 'B', 'clangor': 'A', 'clangour': 'B', 'clangored': 'A', 'clangoured': 'B', 'clangoring': 'A', 'clangouring': 'B', 'clangors': 'A', 'clangours': 'B', "clangor's": 'A', "clangour's": 'B', 'clarinetist': 'A', 'clarinettist': 'B', 'clarinetists': 'A', 'clarinettists': 'B', "clarinetist's": 'A', "clarinettist's": 'B', 'collectible': 'A', 'collectable': 'B', 'collectibles': 'A', 'collectables': 'B', "collectible's": 'A', "collectable's": 'B', 'color': 'A', 'colour': 'B', 'colored': 'A', 'coloured': 'B', "colored's": 'A', "coloured's": 'B', 'colorer': 'A', 'colourer': 'B', "color's": 'A', "colour's": 'B', 'coloreds': 'A', 'coloureds': 'B', 'colorers': 'A', 'colourers': 'B', "colorer's": 'A', "colourer's": 'B', 'coloring': 'A', 'colouring': 'B', 'colorings': 'A', 'colourings': 'B', "coloring's": 'A', "colouring's": 'B', 'colorrest': 'A', 'colourest': 'B', 'colors': 'A', 'colours': 'B', 'colorable': 'A', 'colourable': 'B', 'colorabler': 'A', 'colourabler': 'B', 'colorables': 'A', 'colourables': 'B', 'colorablest': 'A', 'colourablest': 'B', "colorable's": 'A', "colourable's": 'B', 'colorant': 'A', 'colourant': 'B', 'colorants': 'A', 'colourants': 'B', "colorant's": 'A', "colourant's": 'B', 'colorful': 'A', 'colourful': 'B', 'colorfuler': 'A', 'colourfuler': 'B', 'colorfulest': 'A', 'colourfulest': 'B', 'colorfully': 'A', 'colourfully': 'B', 'colorfullier': 'A', 'colourfullier': 'B', 'colorfulliest': 'A', 'colourfulliest': 'B', 'colorfulness': 'A', 'colourfulness': 'B', 'colorfulnesses': 'A', 'colourfulnesses': 'B', "colorfulness's": 'A', "colourfulness's": 'B', 'colorist': 'A', 'colourist': 'B', 'colorists': 'A', 'colourists': 'B', "colorist's": 'A', "colourist's": 'B', 'colorless': 'A', 'colourless': 'B', 'colorlesser': 'A', 'colourlesser': 'B', 'colorlessest': 'A', 'colourlessest': 'B', 'colorlessly': 'A', 'colourlessly': 'B', 'colorlesslier': 'A', 'colourlesslier': 'B', 'colorlessliest': 'A', 'colourlessliest': 'B', 'colorway': 'A', 'colourway': 'B', 'colorways': 'A', 'colourways': 'B', 'councilor': 'A', 'councillor': 'B', 'councilors': 'A', 'councillors': 'B', "councilor's": 'A', "councillor's": 'B', 'councilorship': 'A', 'councillorship': 'B', 'councilorships': 'A', 'councillorships': 'B', "councilorship's": 'A', "councillorship's": 'B', 'counseled': 'A', 'counselled': 'B', 'counseling': 'A', 'counselling': 'B', 'counselor': 'A', 'counsellor': 'B', 'counselors': 'A', 'counsellors': 'B', "counselor's": 'A', "counsellor's": 'B', 'cozily': 'A', 'cosily': 'B', 'coziness': 'A', 'cosiness': 'B', "coziness's": 'A', "cosiness's": 'B', 'cozy': 'A', 'cosy': 'B', 'cozied': 'A', 'cosied': 'B', 'cozier': 'A', 'cosier': 'B', 'coziest': 'A', 'cosiest': 'B', 'cozying': 'A', 'cosying': 'B', 'cozies': 'A', 'cosies': 'B', "cozy's": 'A', "cosy's": 'B', 'crenelate': 'A', 'crenellate': 'B', 'crenelated': 'A', 'crenellated': 'B', 'crenelater': 'A', 'crenellater': 'B', 'crenelates': 'A', 'crenellates': 'B', 'crenelatest': 'A', 'crenellatest': 'B', 'crenelating': 'A', 'crenellating': 'B', 'crosier': 'A', 'crozier': 'B', 'crosiers': 'A', 'croziers': 'B', "crosier's": 'A', "crozier's": 'B', 'crueler': 'A', 'crueller': 'B', 'cruelest': 'A', 'cruellest': 'B', 'cudgeled': 'A', 'cudgelled': 'B', 'cudgeling': 'A', 'cudgelling': 'B', 'cudgelings': 'A', 'cudgellings': 'B', 'cupelation': 'A', 'cupellation': 'B', 'tsar': 'B', 'czar': 'A', 'tsars': 'B', 'czars': 'A', "tsar's": 'B', "czar's": 'A', 'deciliter': 'A', 'decilitre': 'B', 'deciliters': 'A', 'decilitres': 'B', "deciliter's": 'A', "decilitre's": 'B', 'decimeter': 'A', 'decimetre': 'B', 'decimeters': 'A', 'decimetres': 'B', "decimeter's": 'A', "decimetre's": 'B', 'defense': 'A', 'defence': 'B', 'defensed': 'A', 'defenced': 'B', 'defensing': 'A', 'defencing': 'B', 'defenses': 'A', 'defences': 'B', "defense's": 'A', "defence's": 'B', 'defenseless': 'A', 'defenceless': 'B', 'defenselesser': 'A', 'defencelesser': 'B', 'defenselessest': 'A', 'defencelessest': 'B', 'defenselessness': 'A', 'defencelessness': 'B', 'defenselessnesses': 'A', 'defencelessnesses': 'B', "defenselessness's": 'A', "defencelessness's": 'B', 'demeanor': 'A', 'demeanour': 'B', 'demeanors': 'A', 'demeanours': 'B', "demeanor's": 'A', "demeanour's": 'B', 'dentin': 'A', 'dentine': 'B', 'dentins': 'A', 'dentines': 'B', "dentin's": 'A', "dentine's": 'B', 'deviled': 'A', 'devilled': 'B', 'deviling': 'A', 'devilling': 'B', 'dialed': 'A', 'dialled': 'B', 'dialing': 'A', 'dialling': 'B', 'dialings': 'A', 'diallings': 'B', "dialing's": 'A', "dialling's": 'B', 'dialyze': 'A', 'dialyse': 'B', 'dialyzed': 'A', 'dialysed': 'B', 'dialyzing': 'A', 'dialysing': 'B', 'dialyzes': 'A', 'diarrhea': 'A', 'diarrhoea': 'B', 'diarrheas': 'A', 'diarrhoeas': 'B', "diarrhea's": 'A', "diarrhoea's": 'B', 'diarrheal': 'A', 'diarrhoeal': 'B', 'diarrhealer': 'A', 'diarrhoealer': 'B', 'diarrhealest': 'A', 'diarrhoealest': 'B', 'dike': 'A', 'dyke': 'B', 'diked': 'A', 'dyked': 'B', 'diking': 'A', 'dyking': 'B', 'dikes': 'A', 'dykes': 'B', "dike's": 'A', "dyke's": 'B', 'diopter': 'A', 'dioptre': 'B', 'diopters': 'A', 'dioptres': 'B', "diopter's": 'A', "dioptre's": 'B', 'discolor': 'A', 'discolour': 'B', 'discolored': 'A', 'discoloured': 'B', "discolor's": 'A', "discolour's": 'B', 'discoloreds': 'A', 'discoloureds': 'B', "discolored's": 'A', "discoloured's": 'B', 'discoloring': 'A', 'discolouring': 'B', 'discolors': 'A', 'discolours': 'B', 'disemboweled': 'A', 'disembowelled': 'B', 'disemboweling': 'A', 'disembowelling': 'B', 'disfavor': 'A', 'disfavour': 'B', 'disfavored': 'A', 'disfavoured': 'B', 'disfavoring': 'A', 'disfavouring': 'B', 'disfavors': 'A', 'disfavours': 'B', "disfavor's": 'A', "disfavour's": 'B', 'disheveled': 'A', 'dishevelled': 'B', 'disheveling': 'A', 'dishevelling': 'B', 'dishonor': 'A', 'dishonour': 'B', 'dishonored': 'A', 'dishonoured': 'B', 'dishonoring': 'A', 'dishonouring': 'B', 'dishonors': 'A', 'dishonours': 'B', "dishonor's": 'A', "dishonour's": 'B', 'dishonorable': 'A', 'dishonourable': 'B', 'dishonorabler': 'A', 'dishonourabler': 'B', 'dishonorables': 'A', 'dishonourables': 'B', 'dishonorablest': 'A', 'dishonourablest': 'B', "dishonorable's": 'A', "dishonourable's": 'B', 'dishonorableness': 'A', 'dishonourableness': 'B', 'dishonorablenesses': 'A', 'dishonourablenesses': 'B', "dishonorableness's": 'A', "dishonourableness's": 'B', 'dishonorably': 'A', 'dishonourably': 'B', 'dishonorablier': 'A', 'dishonourablier': 'B', 'dishonorablies': 'A', 'dishonourablies': 'B', 'dishonorabliest': 'A', 'dishonourabliest': 'B', 'distill': 'A', 'distil': 'B', 'distills': 'A', 'distils': 'B', 'dolor': 'A', 'dolour': 'B', 'dolors': 'A', 'dolours': 'B', "dolor's": 'A', "dolour's": 'B', 'downdraft': 'A', 'downdraught': 'B', "downdraft's": 'A', "downdraught's": 'B', 'draftily': 'A', 'draughtily': 'B', 'draftilier': 'A', 'draughtilier': 'B', 'draftiliest': 'A', 'draughtiliest': 'B', 'draftiness': 'A', 'draughtiness': 'B', 'draftinesses': 'A', 'draughtinesses': 'B', "draftiness's": 'A', "draughtiness's": 'B', 'draftsman': 'A', 'draughtsman': 'B', 'draftsmen': 'A', 'draughtsmen': 'B', "draftsman's": 'A', "draughtsman's": 'B', 'draftsmanship': 'A', 'draughtsmanship': 'B', 'draftsmanships': 'A', 'draughtsmanships': 'B', "draftsmanship's": 'A', "draughtsmanship's": 'B', 'draftswoman': 'A', 'draughtswoman': 'B', 'draftswomen': 'A', 'draughtswomen': 'B', "draftswoman's": 'A', "draughtswoman's": 'B', 'drafty': 'A', 'draughty': 'B', 'draftier': 'A', 'draughtier': 'B', 'draftiest': 'A', 'draughtiest': 'B', 'driveled': 'A', 'drivelled': 'B', 'driveling': 'A', 'drivelling': 'B', 'driveler': 'A', 'driveller': 'B', 'drivelers': 'A', 'drivellers': 'B', "driveler's": 'A', "driveller's": 'B', 'dryly': 'A', 'drily': 'B', 'dueled': 'A', 'duelled': 'B', 'dueling': 'A', 'duelling': 'B', 'duelings': 'A', 'duellings': 'B', 'dueler': 'A', 'dueller': 'B', 'duelers': 'A', 'duellers': 'B', "dueler's": 'A', "dueller's": 'B', 'duelist': 'A', 'duellist': 'B', 'duelists': 'A', 'duellists': 'B', "duelist's": 'A', "duellist's": 'B', 'dyspnea': 'A', 'dyspnoea': 'B', 'dyspneas': 'A', 'dyspnoeas': 'B', "dyspnea's": 'A', "dyspnoea's": 'B', 'edema': 'A', 'oedema': 'B', 'edemas': 'A', 'oedemas': 'B', "edema's": 'A', "oedema's": 'B', 'edematous': 'A', 'oedematous': 'B', 'enameled': 'A', 'enamelled': 'B', 'enameling': 'A', 'enamelling': 'B', 'enamelings': 'A', 'enamellings': 'B', 'enameler': 'A', 'enameller': 'B', 'enamelers': 'A', 'enamellers': 'B', "enameler's": 'A', "enameller's": 'B', 'enamor': 'A', 'enamour': 'B', 'enamored': 'A', 'enamoured': 'B', "enamor's": 'A', "enamour's": 'B', 'enamoreds': 'A', 'enamoureds': 'B', "enamored's": 'A', "enamoured's": 'B', 'enamoring': 'A', 'enamouring': 'B', 'enamors': 'A', 'enamours': 'B', 'endeavor': 'A', 'endeavour': 'B', 'endeavored': 'A', 'endeavoured': 'B', 'endeavoring': 'A', 'endeavouring': 'B', 'endeavors': 'A', 'endeavours': 'B', "endeavor's": 'A', "endeavour's": 'B', 'enological': 'A', 'oenological': 'B', 'enologist': 'A', 'oenologist': 'B', 'enologists': 'A', 'oenologists': 'B', 'enroll': 'A', 'enrol': 'B', 'enrolls': 'A', 'enrols': 'B', 'enrollment': 'A', 'enrolment': 'B', 'enrollments': 'A', 'enrolments': 'B', "enrollment's": 'A', "enrolment's": 'B', 'enthrall': 'A', 'enthral': 'B', 'enthralls': 'A', 'enthrals': 'B', 'enthrallment': 'A', 'enthralment': 'B', 'enthrallments': 'A', 'enthralments': 'B', "enthrallment's": 'A', "enthralment's": 'B', 'eon': 'A', 'aeon': 'B', 'eons': 'A', 'aeons': 'B', "eon's": 'A', "aeon's": 'B', 'epaulet': 'A', 'epaulette': 'B', 'epaulets': 'A', 'epaulettes': 'B', "epaulet's": 'A', "epaulette's": 'B', 'epicenter': 'A', 'epicentre': 'B', 'epicenters': 'A', 'epicentres': 'B', "epicenter's": 'A', "epicentre's": 'B', 'equaled': 'A', 'equalled': 'B', 'equaling': 'A', 'equalling': 'B', 'esophagus': 'A', 'oesophagus': 'B', 'esthetician': 'A', 'aesthetician': 'B', 'estheticians': 'A', 'aestheticians': 'B', "esthetician's": 'A', "aesthetician's": 'B', 'estival': 'A', 'aestival': 'B', 'estivate': 'A', 'aestivate': 'B', 'estivated': 'A', 'aestivated': 'B', 'estivating': 'A', 'aestivating': 'B', 'estivates': 'A', 'aestivates': 'B', 'estivation': 'A', 'aestivation': 'B', 'estivations': 'A', 'aestivations': 'B', "estivation's": 'A', "aestivation's": 'B', 'estrogen': 'A', 'oestrogen': 'B', 'estrogens': 'A', 'oestrogens': 'B', "estrogen's": 'A', "oestrogen's": 'B', 'estrogenic': 'A', 'oestrogenic': 'B', 'estrous': 'A', 'oestrous': 'B', 'estrus': 'A', 'oestrus': 'B', 'estruses': 'A', 'oestruses': 'B', "estrus's": 'A', "oestrus's": 'B', 'etiologically': 'A', 'aetiologically': 'B', 'etiology': 'A', 'aetiology': 'B', "etiology's": 'A', "aetiology's": 'B', 'eviler': 'A', 'eviller': 'B', 'evilest': 'A', 'evillest': 'B', 'favor': 'A', 'favour': 'B', 'favored': 'A', 'favoured': 'B', "favor's": 'A', "favour's": 'B', 'favoreds': 'A', 'favoureds': 'B', "favored's": 'A', "favoured's": 'B', 'favoring': 'A', 'favouring': 'B', 'favorings': 'A', 'favourings': 'B', "favoring's": 'A', "favouring's": 'B', 'favors': 'A', 'favours': 'B', 'favorable': 'A', 'favourable': 'B', 'favorabler': 'A', 'favourabler': 'B', 'favorables': 'A', 'favourables': 'B', 'favorablest': 'A', 'favourablest': 'B', "favorable's": 'A', "favourable's": 'B', 'favorableness': 'A', 'favourableness': 'B', 'favorablenesses': 'A', 'favourablenesses': 'B', "favorableness's": 'A', "favourableness's": 'B', 'favorably': 'A', 'favourably': 'B', 'favorablier': 'A', 'favourablier': 'B', 'favorabliest': 'A', 'favourabliest': 'B', 'favorite': 'A', 'favourite': 'B', 'favoriter': 'A', 'favouriter': 'B', 'favorites': 'A', 'favourites': 'B', 'favoritest': 'A', 'favouritest': 'B', "favorite's": 'A', "favourite's": 'B', 'favoritism': 'A', 'favouritism': 'B', 'favoritisms': 'A', 'favouritisms': 'B', "favoritism's": 'A', "favouritism's": 'B', 'fecal': 'A', 'faecal': 'B', 'feces': 'A', 'faeces': 'B', "feces's": 'A', "faeces's": 'B', 'fervor': 'A', 'fervour': 'B', 'fervors': 'A', 'fervours': 'B', "fervor's": 'A', "fervour's": 'B', 'fiber': 'A', 'fibre': 'B', 'fibers': 'A', 'fibres': 'B', "fiber's": 'A', "fibre's": 'B', 'fiberboard': 'A', 'fibreboard': 'B', 'fiberboards': 'A', 'fibreboards': 'B', "fiberboard's": 'A', "fibreboard's": 'B', 'fiberfill': 'A', 'fibrefill': 'B', 'fiberfills': 'A', 'fibrefills': 'B', "fiberfill's": 'A', "fibrefill's": 'B', 'fiberglass': 'A', 'fibreglass': 'B', 'fiberglasses': 'A', 'fibreglasses': 'B', "fiberglass's": 'A', "fibreglass's": 'B', 'fiberless': 'A', 'fibreless': 'B', 'fiberlesser': 'A', 'fibrelesser': 'B', 'fiberlessest': 'A', 'fibrelessest': 'B', 'flanneled': 'A', 'flannelled': 'B', 'flanneling': 'A', 'flannelling': 'B', 'flavor': 'A', 'flavour': 'B', 'flavored': 'A', 'flavoured': 'B', 'flavoring': 'A', 'flavouring': 'B', "flavor's": 'A', "flavour's": 'B', 'flavorings': 'A', 'flavourings': 'B', "flavoring's": 'A', "flavouring's": 'B', 'flavors': 'A', 'flavours': 'B', 'flavorful': 'A', 'flavourful': 'B', 'flavorfuler': 'A', 'flavourfuler': 'B', 'flavorfulest': 'A', 'flavourfulest': 'B', 'flavorless': 'A', 'flavourless': 'B', 'flavorlesser': 'A', 'flavourlesser': 'B', 'flavorlessest': 'A', 'flavourlessest': 'B', 'flavorsome': 'A', 'flavoursome': 'B', 'flavorsomer': 'A', 'flavoursomer': 'B', 'flavorsomest': 'A', 'flavoursomest': 'B', 'flutist': 'A', 'flautist': 'B', "flutist's": 'A', "flautist's": 'B', 'flutists': 'A', 'flautists': 'B', 'fogy': 'A', 'fogey': 'B', 'fogies': 'A', 'fogeys': 'B', "fogy's": 'A', "fogey's": 'B', 'fontanel': 'A', 'fontanelle': 'B', 'fontanels': 'A', 'fontanelles': 'B', "fontanel's": 'A', "fontanelle's": 'B', 'forgather': 'A', 'foregather': 'B', 'forgathered': 'A', 'foregathered': 'B', 'forgathering': 'A', 'foregathering': 'B', 'forgathers': 'A', 'foregathers': 'B', 'fueled': 'A', 'fuelled': 'B', 'fueling': 'A', 'fuelling': 'B', 'fulfill': 'A', 'fulfil': 'B', 'fulfills': 'A', 'fulfils': 'B', 'fulfillment': 'A', 'fulfilment': 'B', 'fulfillments': 'A', 'fulfilments': 'B', "fulfillment's": 'A', "fulfilment's": 'B', 'funneled': 'A', 'funnelled': 'B', 'funneling': 'A', 'funnelling': 'B', 'furor': 'A', 'furore': 'B', 'furors': 'A', 'furores': 'B', "furor's": 'A', "furore's": 'B', 'gamboled': 'A', 'gambolled': 'B', 'gamboling': 'A', 'gambolling': 'B', 'gemology': 'A', 'gemmology': 'B', 'gemologies': 'A', 'gemmologies': 'B', "gemology's": 'A', "gemmology's": 'B', 'glycerin': 'A', 'glycerine': 'B', 'glycerins': 'A', 'glycerines': 'B', "glycerin's": 'A', "glycerine's": 'B', 'gobbledygook': 'A', 'gobbledegook': 'B', 'gobbledygooks': 'A', 'gobbledegooks': 'B', "gobbledygook's": 'A', "gobbledegook's": 'B', 'goiter': 'A', 'goitre': 'B', 'goiters': 'A', 'goitres': 'B', "goiter's": 'A', "goitre's": 'B', 'gonorrhea': 'A', 'gonorrhoea': 'B', 'gonorrheas': 'A', 'gonorrhoeas': 'B', "gonorrhea's": 'A', "gonorrhoea's": 'B', 'gonorrheal': 'A', 'gonorrhoeal': 'B', 'gonorrhealer': 'A', 'gonorrhoealer': 'B', 'gonorrhealest': 'A', 'gonorrhoealest': 'B', 'graveled': 'A', 'gravelled': 'B', 'graveling': 'A', 'gravelling': 'B', 'gray': 'A', 'grey': 'B', 'grayed': 'A', 'greyed': 'B', 'grayer': 'A', 'greyer': 'B', 'grayest': 'A', 'greyest': 'B', 'graying': 'A', 'greying': 'B', 'grays': 'A', 'greys': 'B', "gray's": 'A', "grey's": 'B', 'grayish': 'A', 'greyish': 'B', 'grayness': 'A', 'greyness': 'B', "grayness's": 'A', "greyness's": 'B', 'grecism': 'A', 'graecism': 'B', 'grecisms': 'A', 'graecisms': 'B', "grecism's": 'A', "graecism's": 'B', 'groveled': 'A', 'grovelled': 'B', 'groveling': 'A', 'grovelling': 'B', 'groveler': 'A', 'groveller': 'B', 'grovelers': 'A', 'grovellers': 'B', "groveler's": 'A', "groveller's": 'B', 'grovelingly': 'A', 'grovellingly': 'B', 'grovelinglier': 'A', 'grovellinglier': 'B', 'grovelingliest': 'A', 'grovellingliest': 'B', 'grueled': 'A', 'gruelled': 'B', 'grueling': 'A', 'gruelling': 'B', 'gruelings': 'A', 'gruellings': 'B', "grueling's": 'A', "gruelling's": 'B', 'gruelingly': 'A', 'gruellingly': 'B', 'gynecological': 'A', 'gynaecological': 'B', 'gynecologicaler': 'A', 'gynaecologicaler': 'B', 'gynecologicalest': 'A', 'gynaecologicalest': 'B', 'gynecologicals': 'A', 'gynaecologicals': 'B', "gynecological's": 'A', "gynaecological's": 'B', 'gynecologist': 'A', 'gynaecologist': 'B', 'gynecologists': 'A', 'gynaecologists': 'B', "gynecologist's": 'A', "gynaecologist's": 'B', 'gynecology': 'A', 'gynaecology': 'B', 'gynecologies': 'A', 'gynaecologies': 'B', "gynecology's": 'A', "gynaecology's": 'B', 'gynecomastia': 'A', 'gynaecomastia': 'B', 'gynecomastias': 'A', 'gynaecomastias': 'B', "gynecomastia's": 'A', "gynaecomastia's": 'B', 'hankie': 'A', 'hanky': 'B', "hankie's": 'A', "hanky's": 'B', 'harbor': 'A', 'harbour': 'B', 'harbored': 'A', 'harboured': 'B', 'harboring': 'A', 'harbouring': 'B', 'harbors': 'A', 'harbours': 'B', "harbor's": 'A', "harbour's": 'B', 'hectoliter': 'A', 'hectolitre': 'B', 'hectoliters': 'A', 'hectolitres': 'B', "hectoliter's": 'A', "hectolitre's": 'B', 'hectometer': 'A', 'hectometre': 'B', 'hectometers': 'A', 'hectometres': 'B', "hectometer's": 'A', "hectometre's": 'B', 'hemal': 'A', 'haemal': 'B', 'hematite': 'A', 'haematite': 'B', 'hematites': 'A', 'haematites': 'B', "hematite's": 'A', "haematite's": 'B', 'hematocele': 'A', 'haematocele': 'B', "hematocele's": 'A', "haematocele's": 'B', 'hematologic': 'A', 'haematologic': 'B', 'hematological': 'A', 'haematological': 'B', 'hematologist': 'A', 'haematologist': 'B', 'hematologists': 'A', 'haematologists': 'B', "hematologist's": 'A', "haematologist's": 'B', 'hematology': 'A', 'haematology': 'B', 'hematologies': 'A', 'haematologies': 'B', "hematology's": 'A', "haematology's": 'B', 'hematoxylin': 'A', 'haematoxylin': 'B', 'hematoxylins': 'A', 'haematoxylins': 'B', "hematoxylin's": 'A', "haematoxylin's": 'B', 'hematuria': 'A', 'haematuria': 'B', 'hematurias': 'A', 'haematurias': 'B', "hematuria's": 'A', "haematuria's": 'B', 'hemodialysis': 'A', 'haemodialysis': 'B', "hemodialysis's": 'A', "haemodialysis's": 'B', 'hemoglobin': 'A', 'haemoglobin': 'B', "hemoglobin's": 'A', "haemoglobin's": 'B', 'hemophilia': 'A', 'haemophilia': 'B', "hemophilia's": 'A', "haemophilia's": 'B', 'hemophiliac': 'A', 'haemophiliac': 'B', 'hemophiliacs': 'A', 'haemophiliacs': 'B', "hemophiliac's": 'A', "haemophiliac's": 'B', 'hemophilic': 'A', 'haemophilic': 'B', 'hemorrhage': 'A', 'haemorrhage': 'B', "hemorrhage's": 'A', "haemorrhage's": 'B', 'hemorrhaged': 'A', 'haemorrhaged': 'B', 'hemorrhaging': 'A', 'haemorrhaging': 'B', 'hemorrhages': 'A', 'haemorrhages': 'B', 'hemorrhoid': 'A', 'haemorrhoid': 'B', 'hemorrhoids': 'A', 'haemorrhoids': 'B', 'hemorrhoidal': 'A', 'haemorrhoidal': 'B', 'honor': 'A', 'honour': 'B', 'honored': 'A', 'honoured': 'B', 'honoring': 'A', 'honouring': 'B', 'honors': 'A', 'honours': 'B', "honor's": 'A', "honour's": 'B', 'honorable': 'A', 'honourable': 'B', 'honorabler': 'A', 'honourabler': 'B', 'honorables': 'A', 'honourables': 'B', 'honorablest': 'A', 'honourablest': 'B', "honorable's": 'A', "honourable's": 'B', 'honorableness': 'A', 'honourableness': 'B', 'honorablenesses': 'A', 'honourablenesses': 'B', "honorableness's": 'A', "honourableness's": 'B', 'honorably': 'A', 'honourably': 'B', 'honorablier': 'A', 'honourablier': 'B', 'honorablies': 'A', 'honourablies': 'B', 'honorabliest': 'A', 'honourabliest': 'B', 'humor': 'A', 'humour': 'B', 'humored': 'A', 'humoured': 'B', 'humoring': 'A', 'humouring': 'B', 'humors': 'A', 'humours': 'B', "humor's": 'A', "humour's": 'B', 'humorless': 'A', 'humourless': 'B', 'humorlesser': 'A', 'humourlesser': 'B', 'humorlessest': 'A', 'humourlessest': 'B', 'humorlessness': 'A', 'humourlessness': 'B', 'humorlessnesses': 'A', 'humourlessnesses': 'B', "humorlessness's": 'A', "humourlessness's": 'B', 'hydrolyze': 'A', 'hydrolyse': 'B', 'hydrolyzed': 'A', 'hydrolysed': 'B', 'hydrolyzing': 'A', 'hydrolysing': 'B', 'hydrolyzes': 'A', 'imperiled': 'A', 'imperilled': 'B', 'imperiling': 'A', 'imperilling': 'B', 'indraft': 'A', 'indraught': 'B', 'indrafts': 'A', 'indraughts': 'B', "indraft's": 'A', "indraught's": 'B', 'initialed': 'A', 'initialled': 'B', 'initialing': 'A', 'initialling': 'B', 'installment': 'A', 'instalment': 'B', 'installments': 'A', 'instalments': 'B', "installment's": 'A', "instalment's": 'B', 'instill': 'A', 'instil': 'B', 'instills': 'A', 'instils': 'B', 'ischemia': 'A', 'ischaemia': 'B', 'ischemias': 'A', 'ischaemias': 'B', "ischemia's": 'A', "ischaemia's": 'B', 'ischemic': 'A', 'ischaemic': 'B', 'jeweled': 'A', 'jewelled': 'B', 'jeweling': 'A', 'jewelling': 'B', 'jeweler': 'A', 'jeweller': 'B', 'jewelers': 'A', 'jewellers': 'B', "jeweler's": 'A', "jeweller's": 'B', 'jewelry': 'A', 'jewellery': 'B', "jewelry's": 'A', "jewellery's": 'B', 'judgment': 'A', 'judgement': 'B', 'judgments': 'A', 'judgements': 'B', "judgment's": 'A', "judgement's": 'B', 'judgmental': 'A', 'judgemental': 'B', 'judgmentaler': 'A', 'judgementaler': 'B', 'judgmentalest': 'A', 'judgementalest': 'B', 'kenneled': 'A', 'kennelled': 'B', 'kenneling': 'A', 'kennelling': 'B', 'kiloliter': 'A', 'kilolitre': 'B', 'kiloliters': 'A', 'kilolitres': 'B', "kiloliter's": 'A', "kilolitre's": 'B', 'kilometer': 'A', 'kilometre': 'B', 'kilometers': 'A', 'kilometres': 'B', "kilometer's": 'A', "kilometre's": 'B', 'kindergartner': 'A', 'kindergartener': 'B', 'kindergartners': 'A', 'kindergarteners': 'B', "kindergartner's": 'A', "kindergartener's": 'B', 'kinesthesia': 'A', 'kinaesthesia': 'B', 'kinesthesias': 'A', 'kinaesthesias': 'B', "kinesthesia's": 'A', "kinaesthesia's": 'B', 'kinesthetic': 'A', 'kinaesthetic': 'B', 'kinestheticer': 'A', 'kinaestheticer': 'B', 'kinestheticest': 'A', 'kinaestheticest': 'B', 'kinesthetics': 'A', 'kinaesthetics': 'B', 'labeled': 'A', 'labelled': 'B', 'labeling': 'A', 'labelling': 'B', 'labeler': 'A', 'labeller': 'B', 'labelers': 'A', 'labellers': 'B', "labeler's": 'A', "labeller's": 'B', 'labor': 'A', 'labour': 'B', 'labored': 'A', 'laboured': 'B', 'laboring': 'A', 'labouring': 'B', "labor's": 'A', "labour's": 'B', 'laborings': 'A', 'labourings': 'B', "laboring's": 'A', "labouring's": 'B', 'labors': 'A', 'labours': 'B', 'laborer': 'A', 'labourer': 'B', 'laborers': 'A', 'labourers': 'B', "laborer's": 'A', "labourer's": 'B', 'lackluster': 'A', 'lacklustre': 'B', 'lacklusterrer': 'A', 'lacklustrer': 'B', 'lacklusterrest': 'A', 'lacklustrest': 'B', 'lacklusters': 'A', 'lacklustres': 'B', "lackluster's": 'A', "lacklustre's": 'B', 'leukemia': 'A', 'leukaemia': 'B', 'leukemias': 'A', 'leukaemias': 'B', "leukemia's": 'A', "leukaemia's": 'B', 'leveled': 'A', 'levelled': 'B', 'leveler': 'A', 'leveller': 'B', 'levelers': 'A', 'levellers': 'B', "leveler's": 'A', "leveller's": 'B', 'levelest': 'A', 'levellest': 'B', 'leveling': 'A', 'levelling': 'B', "leveling's": 'A', "levelling's": 'B', 'libeled': 'A', 'libelled': 'B', 'libeling': 'A', 'libelling': 'B', 'libeler': 'A', 'libeller': 'B', 'libelers': 'A', 'libellers': 'B', "libeler's": 'A', "libeller's": 'B', 'libelous': 'A', 'libellous': 'B', 'libelouser': 'A', 'libellouser': 'B', 'libelousest': 'A', 'libellousest': 'B', 'likability': 'A', 'likeability': 'B', 'likabilities': 'A', 'likeabilities': 'B', "likability's": 'A', "likeability's": 'B', 'likable': 'A', 'likeable': 'B', 'likableness': 'A', 'likeableness': 'B', 'likablenesses': 'A', 'likeablenesses': 'B', "likableness's": 'A', "likeableness's": 'B', 'lissome': 'A', 'lissom': 'B', 'lissomeness': 'A', 'lissomness': 'B', 'lissomenesses': 'A', 'lissomnesses': 'B', 'liter': 'A', 'litre': 'B', 'liters': 'A', 'litres': 'B', "liter's": 'A', "litre's": 'B', 'livable': 'A', 'liveable': 'B', 'logorrhea': 'A', 'logorrhoea': 'B', 'logorrheas': 'A', 'logorrhoeas': 'B', "logorrhea's": 'A', "logorrhoea's": 'B', 'louver': 'A', 'louvre': 'B', 'louvered': 'A', 'louvred': 'B', 'louvers': 'A', 'louvres': 'B', "louver's": 'A', "louvre's": 'B', 'luster': 'A', 'lustre': 'B', 'lustered': 'A', 'lustred': 'B', 'lustering': 'A', 'lustring': 'B', "luster's": 'A', "lustre's": 'B', "lustering's": 'A', "lustring's": 'B', 'lusters': 'A', 'lustres': 'B', 'lusterless': 'A', 'lustreless': 'B', 'lusterlesser': 'A', 'lustrelesser': 'B', 'lusterlessest': 'A', 'lustrelessest': 'B', 'lusterware': 'A', 'lustreware': 'B', 'lusterwares': 'A', 'lustrewares': 'B', "lusterware's": 'A', "lustreware's": 'B', 'mandean': 'A', 'mandaean': 'B', 'mandeans': 'A', 'mandaeans': 'B', "mandean's": 'A', "mandaean's": 'B', 'manoeuvre': 'B', 'maneuver': 'A', 'manoeuvred': 'B', 'maneuvered': 'A', 'manoeuvring': 'B', 'maneuvering': 'A', 'manoeuvrings': 'B', 'maneuverings': 'A', 'manoeuvres': 'B', 'maneuvers': 'A', "manoeuvre's": 'B', "maneuver's": 'A', 'maneuverability': 'A', 'manoeuvrability': 'B', 'maneuverabilities': 'A', 'manoeuvrabilities': 'B', "maneuverability's": 'A', "manoeuvrability's": 'B', 'maneuverable': 'A', 'manoeuvrable': 'B', 'maneuverabler': 'A', 'manoeuvrabler': 'B', 'maneuverablest': 'A', 'manoeuvrablest': 'B', 'maneuverer': 'A', 'manoeuvrer': 'B', 'maneuverers': 'A', 'manoeuvrers': 'B', "maneuverer's": 'A', "manoeuvrer's": 'B', 'manicheism': 'A', 'manichaeism': 'B', 'manicheisms': 'A', 'manichaeisms': 'B', "manicheism's": 'A', "manichaeism's": 'B', 'marshaled': 'A', 'marshalled': 'B', 'marshaling': 'A', 'marshalling': 'B', 'marveled': 'A', 'marvelled': 'B', 'marveling': 'A', 'marvelling': 'B', 'marvelous': 'A', 'marvellous': 'B', 'marvelouser': 'A', 'marvellouser': 'B', 'marvelousest': 'A', 'marvellousest': 'B', 'marvelously': 'A', 'marvellously': 'B', 'marvelouslier': 'A', 'marvellouslier': 'B', 'marvelousliest': 'A', 'marvellousliest': 'B', 'marvelousness': 'A', 'marvellousness': 'B', 'marvelousnesses': 'A', 'marvellousnesses': 'B', "marvelousness's": 'A', "marvellousness's": 'B', 'meager': 'A', 'meagre': 'B', 'meagerrer': 'A', 'meagrer': 'B', 'meagerrest': 'A', 'meagrest': 'B', 'medalist': 'A', 'medallist': 'B', 'medalists': 'A', 'medallists': 'B', "medalist's": 'A', "medallist's": 'B', 'meow': 'A', 'miaow': 'B', 'meowed': 'A', 'miaowed': 'B', 'meowing': 'A', 'miaowing': 'B', 'meows': 'A', 'miaows': 'B', "meow's": 'A', "miaow's": 'B', 'metacenter': 'A', 'metacentre': 'B', 'metacenters': 'A', 'metacentres': 'B', "metacenter's": 'A', "metacentre's": 'B', 'metaled': 'A', 'metalled': 'B', 'metaling': 'A', 'metalling': 'B', 'milliliter': 'A', 'millilitre': 'B', 'milliliters': 'A', 'millilitres': 'B', "milliliter's": 'A', "millilitre's": 'B', 'millimeter': 'A', 'millimetre': 'B', 'millimeters': 'A', 'millimetres': 'B', "millimeter's": 'A', "millimetre's": 'B', 'misbehavior': 'A', 'misbehaviour': 'B', 'misbehaviors': 'A', 'misbehaviours': 'B', "misbehavior's": 'A', "misbehaviour's": 'B', 'misdemeanor': 'A', 'misdemeanour': 'B', 'misdemeanors': 'A', 'misdemeanours': 'B', "misdemeanor's": 'A', "misdemeanour's": 'B', 'misjudgment': 'A', 'misjudgement': 'B', 'misjudgments': 'A', 'misjudgements': 'B', "misjudgment's": 'A', "misjudgement's": 'B', 'miter': 'A', 'mitre': 'B', 'mitered': 'A', 'mitred': 'B', 'mitering': 'A', 'mitring': 'B', 'miters': 'A', 'mitres': 'B', "miter's": 'A', "mitre's": 'B', 'modeled': 'A', 'modelled': 'B', 'modeler': 'A', 'modeller': 'B', 'modelers': 'A', 'modellers': 'B', "modeler's": 'A', "modeller's": 'B', 'modeling': 'A', 'modelling': 'B', 'modelings': 'A', 'modellings': 'B', "modeling's": 'A', "modelling's": 'B', 'moggy': 'A', 'moggie': 'B', "moggy's": 'A', "moggie's": 'B', 'mold': 'A', 'mould': 'B', 'molded': 'A', 'moulded': 'B', 'molding': 'A', 'moulding': 'B', "mold's": 'A', "mould's": 'B', 'moldings': 'A', 'mouldings': 'B', "molding's": 'A', "moulding's": 'B', 'molds': 'A', 'moulds': 'B', 'molder': 'A', 'moulder': 'B', "molder's": 'A', "moulder's": 'B', 'moldered': 'A', 'mouldered': 'B', 'moldering': 'A', 'mouldering': 'B', 'molders': 'A', 'moulders': 'B', 'moldy': 'A', 'mouldy': 'B', 'moldier': 'A', 'mouldier': 'B', 'moldiest': 'A', 'mouldiest': 'B', 'mollusk': 'A', 'mollusc': 'B', 'mollusks': 'A', 'molluscs': 'B', "mollusk's": 'A', "mollusc's": 'B', 'molt': 'A', 'moult': 'B', "molt's": 'A', "moult's": 'B', 'molted': 'A', 'moulted': 'B', 'molting': 'A', 'moulting': 'B', 'molts': 'A', 'moults': 'B', 'mustache': 'A', 'moustache': 'B', 'mustaches': 'A', 'moustaches': 'B', "mustache's": 'A', "moustache's": 'B', 'mustached': 'A', 'moustached': 'B', 'myna': 'A', 'mynah': 'B', 'mynas': 'A', 'mynahes': 'B', "myna's": 'A', "mynah's": 'B', 'mythopeic': 'A', 'mythopoeic': 'B', 'myxedema': 'A', 'myxoedema': 'B', 'myxedemas': 'A', 'myxoedemas': 'B', "myxedema's": 'A', "myxoedema's": 'B', 'nanometer': 'A', 'nanometre': 'B', 'nanometers': 'A', 'nanometres': 'B', "nanometer's": 'A', "nanometre's": 'B', 'neighbor': 'A', 'neighbour': 'B', 'neighbored': 'A', 'neighboured': 'B', 'neighboring': 'A', 'neighbouring': 'B', "neighbor's": 'A', "neighbour's": 'B', 'neighborings': 'A', 'neighbourings': 'B', 'neighbors': 'A', 'neighbours': 'B', 'neighborhood': 'A', 'neighbourhood': 'B', 'neighborhoods': 'A', 'neighbourhoods': 'B', "neighborhood's": 'A', "neighbourhood's": 'B', 'neighborliness': 'A', 'neighbourliness': 'B', 'neighborlinesses': 'A', 'neighbourlinesses': 'B', "neighborliness's": 'A', "neighbourliness's": 'B', 'neighborly': 'A', 'neighbourly': 'B', 'neighborlier': 'A', 'neighbourlier': 'B', 'neighborliest': 'A', 'neighbourliest': 'B', 'niter': 'A', 'nitre': 'B', 'niters': 'A', 'nitres': 'B', "niter's": 'A', "nitre's": 'B', 'nitroglycerin': 'A', 'nitroglycerine': 'B', 'nitroglycerins': 'A', 'nitroglycerines': 'B', "nitroglycerin's": 'A', "nitroglycerine's": 'B', 'numskull': 'A', 'numbskull': 'B', 'numskulls': 'A', 'numbskulls': 'B', "numskull's": 'A', "numbskull's": 'B', 'ocher': 'A', 'ochre': 'B', 'ochered': 'A', 'ochred': 'B', 'ochering': 'A', 'ochring': 'B', 'ocherrer': 'A', 'ochrer': 'B', 'ocherrest': 'A', 'ochrest': 'B', 'ochers': 'A', 'ochres': 'B', "ocher's": 'A', "ochre's": 'B', 'ocherous': 'A', 'ochreous': 'B', 'ocherouser': 'A', 'ochreouser': 'B', 'ocherousest': 'A', 'ochreousest': 'B', 'odor': 'A', 'odour': 'B', 'odors': 'A', 'odours': 'B', "odor's": 'A', "odour's": 'B', 'odorless': 'A', 'odourless': 'B', 'odorlesser': 'A', 'odourlesser': 'B', 'odorlessest': 'A', 'odourlessest': 'B', 'offense': 'A', 'offence': 'B', 'offenses': 'A', 'offences': 'B', "offense's": 'A', "offence's": 'B', 'omelet': 'A', 'omelette': 'B', 'omelets': 'A', 'omelettes': 'B', "omelet's": 'A', "omelette's": 'B', 'organdy': 'A', 'organdie': 'B', "organdy's": 'A', "organdie's": 'B', 'orthopedic': 'A', 'orthopaedic': 'B', 'orthopedicer': 'A', 'orthopaedicer': 'B', 'orthopedicest': 'A', 'orthopaedicest': 'B', 'orthopedics': 'A', 'orthopaedics': 'B', "orthopedic's": 'A', "orthopaedic's": 'B', "orthopedics's": 'A', "orthopaedics's": 'B', 'orthopedist': 'A', 'orthopaedist': 'B', 'orthopedists': 'A', 'orthopaedists': 'B', "orthopedist's": 'A', "orthopaedist's": 'B', 'outmaneuver': 'A', 'outmanoeuvre': 'B', 'outmaneuvered': 'A', 'outmanoeuvred': 'B', 'outmaneuvering': 'A', 'outmanoeuvring': 'B', 'outmaneuvers': 'A', 'outmanoeuvres': 'B', 'oversize': 'A', 'oversized': 'B', 'pajama': 'A', 'pyjama': 'B', 'pajamas': 'A', 'pyjamas': 'B', "pajama's": 'A', "pyjama's": 'B', "pajamas's": 'A', "pyjamas's": 'B', 'paleoclimatology': 'A', 'palaeoclimatology': 'B', 'paleoclimatologies': 'A', 'palaeoclimatologies': 'B', "paleoclimatology's": 'A', "palaeoclimatology's": 'B', 'paleoecology': 'A', 'palaeoecology': 'B', 'paleoecologies': 'A', 'palaeoecologies': 'B', "paleoecology's": 'A', "palaeoecology's": 'B', 'paleogeography': 'A', 'palaeogeography': 'B', 'paleogeographies': 'A', 'palaeogeographies': 'B', 'paleolithic': 'A', 'palaeolithic': 'B', 'paleontological': 'A', 'palaeontological': 'B', 'paleontologist': 'A', 'palaeontologist': 'B', "paleontologist's": 'A', "palaeontologist's": 'B', 'paleontologists': 'A', 'palaeontologists': 'B', 'paleontology': 'A', 'palaeontology': 'B', "paleontology's": 'A', "palaeontology's": 'B', 'paneled': 'A', 'panelled': 'B', 'paneling': 'A', 'panelling': 'B', 'panelings': 'A', 'panellings': 'B', "paneling's": 'A', "panelling's": 'B', 'panelist': 'A', 'panellist': 'B', 'panelists': 'A', 'panellists': 'B', "panelist's": 'A', "panellist's": 'B', 'paralyze': 'A', 'paralyse': 'B', 'paralyzed': 'A', 'paralysed': 'B', 'paralyzing': 'A', 'paralysing': 'B', 'paralyzes': 'A', 'parceled': 'A', 'parcelled': 'B', 'parceling': 'A', 'parcelling': 'B', "parceling's": 'A', "parcelling's": 'B', 'parlor': 'A', 'parlour': 'B', 'parlors': 'A', 'parlours': 'B', "parlor's": 'A', "parlour's": 'B', 'parlormaid': 'A', 'parlourmaid': 'B', 'pedaled': 'A', 'pedalled': 'B', 'pedaler': 'A', 'pedaller': 'B', 'pedalers': 'A', 'pedallers': 'B', "pedaler's": 'A', "pedaller's": 'B', 'pedaling': 'A', 'pedalling': 'B', 'peddler': 'A', 'pedlar': 'B', 'peddlers': 'A', 'pedlars': 'B', "peddler's": 'A', "pedlar's": 'B', 'pediatric': 'A', 'paediatric': 'B', 'pediatrics': 'A', 'paediatrics': 'B', 'paediatrician': 'B', 'pediatrician': 'A', 'paediatricians': 'B', 'pediatricians': 'A', "paediatrician's": 'B', "pediatrician's": 'A', 'pedophile': 'A', 'paedophile': 'B', 'pedophiles': 'A', 'paedophiles': 'B', 'pedophilia': 'A', 'paedophilia': 'B', 'pedophilias': 'A', 'paedophilias': 'B', 'penciled': 'A', 'pencilled': 'B', 'penciling': 'A', 'pencilling': 'B', 'pencilings': 'A', 'pencillings': 'B', 'periled': 'A', 'perilled': 'B', 'periling': 'A', 'perilling': 'B', 'philter': 'A', 'philtre': 'B', "philter's": 'A', "philtre's": 'B', 'philters': 'A', 'philtres': 'B', 'piaster': 'A', 'piastre': 'B', 'piasters': 'A', 'piastres': 'B', "piaster's": 'A', "piastre's": 'B', 'plow': 'A', 'plough': 'B', 'plowed': 'A', 'ploughed': 'B', 'plowing': 'A', 'ploughing': 'B', "plow's": 'A', "plough's": 'B', "plowing's": 'A', "ploughing's": 'B', 'plows': 'A', 'ploughs': 'B', 'plowable': 'A', 'ploughable': 'B', 'plower': 'A', 'plougher': 'B', 'plowers': 'A', 'ploughers': 'B', "plower's": 'A', "plougher's": 'B', 'plowman': 'A', 'ploughman': 'B', 'plowmen': 'A', 'ploughmen': 'B', "plowman's": 'A', "ploughman's": 'B', 'plowshare': 'A', 'ploughshare': 'B', 'plowshares': 'A', 'ploughshares': 'B', "plowshare's": 'A', "ploughshare's": 'B', 'pommeled': 'A', 'pommelled': 'B', 'pommeling': 'A', 'pommelling': 'B', 'porer': 'A', 'pourer': 'B', 'porers': 'A', 'pourers': 'B', 'practicer': 'A', 'practiser': 'B', 'practicers': 'A', 'practisers': 'B', "practicer's": 'A', "practiser's": 'B', 'prejudgment': 'A', 'prejudgement': 'B', 'prejudgments': 'A', 'prejudgements': 'B', "prejudgment's": 'A', "prejudgement's": 'B', 'pretense': 'A', 'pretence': 'B', 'pretenses': 'A', 'pretences': 'B', "pretense's": 'A', "pretence's": 'B', 'preterit': 'A', 'preterite': 'B', 'preterits': 'A', 'preterites': 'B', "preterit's": 'A', "preterite's": 'B', 'psychoanalyze': 'A', 'psychoanalyse': 'B', 'psychoanalyzed': 'A', 'psychoanalysed': 'B', 'psychoanalyzing': 'A', 'psychoanalysing': 'B', 'psychoanalyzes': 'A', 'pummeled': 'A', 'pummelled': 'B', 'pummeling': 'A', 'pummelling': 'B', 'pyorrhea': 'A', 'pyorrhoea': 'B', 'pyorrheas': 'A', 'pyorrhoeas': 'B', "pyorrhea's": 'A', "pyorrhoea's": 'B', 'pyrolyze': 'A', 'pyrolyse': 'B', 'pyrolyzed': 'A', 'pyrolysed': 'B', 'pyrolyzing': 'A', 'pyrolysing': 'B', 'pyrolyzes': 'A', "pyrolyze's": 'A', "pyrolyse's": 'B', 'quarreled': 'A', 'quarrelled': 'B', 'quarreling': 'A', 'quarrelling': 'B', 'quarreler': 'A', 'quarreller': 'B', 'quarrelers': 'A', 'quarrellers': 'B', "quarreler's": 'A', "quarreller's": 'B', 'rancor': 'A', 'rancour': 'B', 'rancors': 'A', 'rancours': 'B', "rancor's": 'A', "rancour's": 'B', 'reanalyze': 'A', 'reanalyse': 'B', 'reanalyzed': 'A', 'reanalysed': 'B', 'reanalyzing': 'A', 'reanalysing': 'B', 'reanalyzes': 'A', 'reconnoiter': 'A', 'reconnoitre': 'B', 'reconnoitered': 'A', 'reconnoitred': 'B', 'reconnoitering': 'A', 'reconnoitring': 'B', 'reconnoiters': 'A', 'reconnoitres': 'B', "reconnoiter's": 'A', "reconnoitre's": 'B', 'redialed': 'A', 'redialled': 'B', 'redialing': 'A', 'redialling': 'B', 'refueled': 'A', 'refuelled': 'B', 'refueling': 'A', 'refuelling': 'B', 'remodeled': 'A', 'remodelled': 'B', 'remodeling': 'A', 'remodelling': 'B', 'remold': 'A', 'remould': 'B', 'remolded': 'A', 'remoulded': 'B', 'remolding': 'A', 'remoulding': 'B', 'remolds': 'A', 'remoulds': 'B', 'reveled': 'A', 'revelled': 'B', 'reveling': 'A', 'revelling': 'B', 'revelings': 'A', 'revellings': 'B', 'reveler': 'A', 'reveller': 'B', 'revelers': 'A', 'revellers': 'B', "reveler's": 'A', "reveller's": 'B', 'rigor': 'A', 'rigour': 'B', 'rigors': 'A', 'rigours': 'B', "rigor's": 'A', "rigour's": 'B', 'rivaled': 'A', 'rivalled': 'B', 'rivaling': 'A', 'rivalling': 'B', 'roweled': 'A', 'rowelled': 'B', 'roweling': 'A', 'rowelling': 'B', 'rumor': 'A', 'rumour': 'B', 'rumored': 'A', 'rumoured': 'B', 'rumoring': 'A', 'rumouring': 'B', 'rumors': 'A', 'rumours': 'B', "rumor's": 'A', "rumour's": 'B', 'saber': 'A', 'sabre': 'B', 'sabered': 'A', 'sabred': 'B', 'sabering': 'A', 'sabring': 'B', 'sabers': 'A', 'sabres': 'B', "saber's": 'A', "sabre's": 'B', 'salable': 'A', 'saleable': 'B', 'salabler': 'A', 'saleabler': 'B', 'salablest': 'A', 'saleablest': 'B', 'saltpeter': 'A', 'saltpetre': 'B', 'saltpeters': 'A', 'saltpetres': 'B', "saltpeter's": 'A', "saltpetre's": 'B', 'sandaled': 'A', 'sandalled': 'B', 'sandaling': 'A', 'sandalling': 'B', 'savior': 'A', 'saviour': 'B', 'saviors': 'A', 'saviours': 'B', "savior's": 'A', "saviour's": 'B', 'savor': 'A', 'savour': 'B', 'savored': 'A', 'savoured': 'B', 'savoring': 'A', 'savouring': 'B', 'savors': 'A', 'savours': 'B', "savor's": 'A', "savour's": 'B', 'savorily': 'A', 'savourily': 'B', 'savorilies': 'A', 'savourilies': 'B', 'savoriness': 'A', 'savouriness': 'B', 'savorinesses': 'A', 'savourinesses': 'B', "savoriness's": 'A', "savouriness's": 'B', 'savorless': 'A', 'savourless': 'B', 'savorlesser': 'A', 'savourlesser': 'B', 'savorlessest': 'A', 'savourlessest': 'B', 'savory': 'A', 'savoury': 'B', 'savorier': 'A', 'savourier': 'B', 'savories': 'A', 'savouries': 'B', 'savoriest': 'A', 'savouriest': 'B', "savory's": 'A', "savoury's": 'B', 'scepter': 'A', 'sceptre': 'B', 'sceptered': 'A', 'sceptred': 'B', 'sceptering': 'A', 'sceptring': 'B', 'scepters': 'A', 'sceptres': 'B', "scepter's": 'A', "sceptre's": 'B', 'seborrhea': 'A', 'seborrhoea': 'B', 'seborrheas': 'A', 'seborrhoeas': 'B', "seborrhea's": 'A', "seborrhoea's": 'B', 'seborrheic': 'A', 'seborrhoeic': 'B', 'septicemia': 'A', 'septicaemia': 'B', 'septicemias': 'A', 'septicaemias': 'B', "septicemia's": 'A', "septicaemia's": 'B', 'septicemic': 'A', 'septicaemic': 'B', 'sepulcher': 'A', 'sepulchre': 'B', 'sepulchered': 'A', 'sepulchred': 'B', 'sepulchering': 'A', 'sepulchring': 'B', 'sepulchers': 'A', 'sepulchres': 'B', "sepulcher's": 'A', "sepulchre's": 'B', 'shoveled': 'A', 'shovelled': 'B', 'shoveling': 'A', 'shovelling': 'B', 'shoveler': 'A', 'shoveller': 'B', 'shovelers': 'A', 'shovellers': 'B', "shoveler's": 'A', "shoveller's": 'B', 'shriveled': 'A', 'shrivelled': 'B', 'shriveling': 'A', 'shrivelling': 'B', 'signaled': 'A', 'signalled': 'B', 'signaler': 'A', 'signaller': 'B', 'signalers': 'A', 'signallers': 'B', "signaler's": 'A', "signaller's": 'B', 'signaling': 'A', 'signalling': 'B', 'sizable': 'A', 'sizeable': 'B', 'sizably': 'A', 'sizeably': 'B', 'skeptic': 'A', 'sceptic': 'B', 'skepticer': 'A', 'scepticer': 'B', 'skepticest': 'A', 'scepticest': 'B', 'skeptics': 'A', 'sceptics': 'B', "skeptic's": 'A', "sceptic's": 'B', 'skeptical': 'A', 'sceptical': 'B', 'skepticaler': 'A', 'scepticaler': 'B', 'skepticalest': 'A', 'scepticalest': 'B', 'skeptically': 'A', 'sceptically': 'B', 'skepticism': 'A', 'scepticism': 'B', 'skepticisms': 'A', 'scepticisms': 'B', "skepticism's": 'A', "scepticism's": 'B', 'slier': 'A', 'slyer': 'B', 'sliest': 'A', 'slyest': 'B', 'smolder': 'A', 'smoulder': 'B', 'smoldered': 'A', 'smouldered': 'B', 'smoldering': 'A', 'smouldering': 'B', 'smolders': 'A', 'smoulders': 'B', "smolder's": 'A', "smoulder's": 'B', 'sniveled': 'A', 'snivelled': 'B', 'sniveling': 'A', 'snivelling': 'B', 'snivelings': 'A', 'snivellings': 'B', 'sniveler': 'A', 'sniveller': 'B', 'snivelers': 'A', 'snivellers': 'B', "sniveler's": 'A', "sniveller's": 'B', 'snorkeled': 'A', 'snorkelled': 'B', 'snorkeling': 'A', 'snorkelling': 'B', 'snowplow': 'A', 'snowplough': 'B', "snowplow's": 'A', "snowplough's": 'B', 'snowplows': 'A', 'snowploughs': 'B', 'somber': 'A', 'sombre': 'B', 'somberer': 'A', 'sombrer': 'B', 'somberest': 'A', 'sombrest': 'B', 'somberly': 'A', 'sombrely': 'B', 'somberness': 'A', 'sombreness': 'B', 'sombernesses': 'A', 'sombrenesses': 'B', "somberness's": 'A', "sombreness's": 'B', 'specialty': 'A', 'speciality': 'B', 'specialties': 'A', 'specialities': 'B', "specialty's": 'A', "speciality's": 'B', 'specter': 'A', 'spectre': 'B', 'specters': 'A', 'spectres': 'B', "specter's": 'A', "spectre's": 'B', 'spilled': 'A', 'spilt': 'B', 'spiraled': 'A', 'spiralled': 'B', 'spiraling': 'A', 'spiralling': 'B', 'spirea': 'A', 'spiraea': 'B', 'spireas': 'A', 'spiraeas': 'B', "spirea's": 'A', "spiraea's": 'B', 'spirochete': 'A', 'spirochaete': 'B', 'spirochetes': 'A', 'spirochaetes': 'B', "spirochete's": 'A', "spirochaete's": 'B', 'splendor': 'A', 'splendour': 'B', 'splendors': 'A', 'splendours': 'B', "splendor's": 'A', "splendour's": 'B', 'squirreled': 'A', 'squirrelled': 'B', 'squirreling': 'A', 'squirrelling': 'B', 'stenciled': 'A', 'stencilled': 'B', 'stenciling': 'A', 'stencilling': 'B', 'succor': 'A', 'succour': 'B', 'succored': 'A', 'succoured': 'B', 'succoring': 'A', 'succouring': 'B', 'succors': 'A', 'succours': 'B', "succor's": 'A', "succour's": 'B', 'sulfate': 'A', 'sulphate': 'B', "sulfate's": 'A', "sulphate's": 'B', 'sulfates': 'A', 'sulphates': 'B', 'sulfide': 'A', 'sulphide': 'B', "sulfide's": 'A', "sulphide's": 'B', 'sulfides': 'A', 'sulphides': 'B', 'sulfur': 'A', 'sulphur': 'B', 'sulfured': 'A', 'sulphured': 'B', 'sulfuring': 'A', 'sulphuring': 'B', 'sulfurs': 'A', 'sulphurs': 'B', "sulfur's": 'A', "sulphur's": 'B', 'sulfuric': 'A', 'sulphuric': 'B', 'sulfurous': 'A', 'sulphurous': 'B', 'sweepstake': 'B', 'sweepstakes': 'A', "sweepstake's": 'B', "sweepstakes's": 'A', 'swiveled': 'A', 'swivelled': 'B', 'swiveling': 'A', 'swivelling': 'B', 'tambura': 'A', 'tamboura': 'B', 'tamburas': 'A', 'tambouras': 'B', "tambura's": 'A', "tamboura's": 'B', 'tasseled': 'A', 'tasselled': 'B', 'tasseling': 'A', 'tasselling': 'B', 'teetotaler': 'A', 'teetotaller': 'B', 'teetotalers': 'A', 'teetotallers': 'B', "teetotaler's": 'A', "teetotaller's": 'B', 'theater': 'A', 'theatre': 'B', 'theaters': 'A', 'theatres': 'B', "theater's": 'A', "theatre's": 'B', 'theatergoer': 'A', 'theatregoer': 'B', 'theatergoers': 'A', 'theatregoers': 'B', "theatergoer's": 'A', "theatregoer's": 'B', 'thralldom': 'A', 'thraldom': 'B', 'thralldoms': 'A', 'thraldoms': 'B', "thralldom's": 'A', "thraldom's": 'B', 'thruway': 'A', 'throughway': 'B', 'thruways': 'A', 'throughways': 'B', "thruway's": 'A', "throughway's": 'B', 'tidbit': 'A', 'titbit': 'B', 'tidbits': 'A', 'titbits': 'B', "tidbit's": 'A', "titbit's": 'B', 'tinseled': 'A', 'tinselled': 'B', 'tinseling': 'A', 'tinselling': 'B', 'titer': 'A', 'titre': 'B', 'titers': 'A', 'titres': 'B', "titer's": 'A', "titre's": 'B', 'totaled': 'A', 'totalled': 'B', 'totaling': 'A', 'totalling': 'B', 'toweled': 'A', 'towelled': 'B', 'toweling': 'A', 'towelling': 'B', 'towelings': 'A', 'towellings': 'B', "toweling's": 'A', "towelling's": 'B', 'toxemia': 'A', 'toxaemia': 'B', 'toxemias': 'A', 'toxaemias': 'B', "toxemia's": 'A', "toxaemia's": 'B', 'toxemic': 'A', 'toxaemic': 'B', 'trammeled': 'A', 'trammelled': 'B', 'trammeling': 'A', 'trammelling': 'B', 'tranquility': 'A', 'tranquillity': 'B', 'tranquilities': 'A', 'tranquillities': 'B', "tranquility's": 'A', "tranquillity's": 'B', 'traveled': 'A', 'travelled': 'B', 'traveler': 'A', 'traveller': 'B', 'travelers': 'A', 'travellers': 'B', "traveler's": 'A', "traveller's": 'B', 'traveling': 'A', 'travelling': 'B', 'travelings': 'A', 'travellings': 'B', "traveling's": 'A', "travelling's": 'B', 'trialed': 'A', 'trialled': 'B', 'trialing': 'A', 'trialling': 'B', 'troweled': 'A', 'trowelled': 'B', 'troweling': 'A', 'trowelling': 'B', 'tularemia': 'A', 'tularaemia': 'B', 'tularemias': 'A', 'tularaemias': 'B', 'tumbrel': 'A', 'tumbril': 'B', 'tumbrels': 'A', 'tumbrils': 'B', "tumbrel's": 'A', "tumbril's": 'B', 'tumor': 'A', 'tumour': 'B', 'tumors': 'A', 'tumours': 'B', "tumor's": 'A', "tumour's": 'B', 'tunneled': 'A', 'tunnelled': 'B', 'tunneling': 'A', 'tunnelling': 'B', 'tunnelings': 'A', 'tunnellings': 'B', 'tunneler': 'A', 'tunneller': 'B', 'tunnelers': 'A', 'tunnellers': 'B', "tunneler's": 'A', "tunneller's": 'B', 'unbeknownst': 'A', 'unbeknown': 'B', 'uncolored': 'A', 'uncoloured': 'B', 'uncoloredder': 'A', 'uncolouredder': 'B', 'uncoloreddest': 'A', 'uncoloureddest': 'B', 'uncoloreds': 'A', 'uncoloureds': 'B', 'unequaled': 'A', 'unequalled': 'B', 'unequaledder': 'A', 'unequalledder': 'B', 'unequaleddest': 'A', 'unequalleddest': 'B', 'unfavorable': 'A', 'unfavourable': 'B', 'unfavorabler': 'A', 'unfavourabler': 'B', 'unfavorables': 'A', 'unfavourables': 'B', 'unfavorablest': 'A', 'unfavourablest': 'B', "unfavorable's": 'A', "unfavourable's": 'B', 'unfavorably': 'A', 'unfavourably': 'B', 'unfavorablier': 'A', 'unfavourablier': 'B', 'unfavorabliest': 'A', 'unfavourabliest': 'B', 'unflavored': 'A', 'unflavoured': 'B', 'unflavoredder': 'A', 'unflavouredder': 'B', 'unflavoreddest': 'A', 'unflavoureddest': 'B', 'unlabeled': 'A', 'unlabelled': 'B', 'unlabeledder': 'A', 'unlabelledder': 'B', 'unlabeleddest': 'A', 'unlabelleddest': 'B', 'unpracticed': 'A', 'unpractised': 'B', 'unpracticedder': 'A', 'unpractisedder': 'B', 'unpracticeddest': 'A', 'unpractiseddest': 'B', 'unraveled': 'A', 'unravelled': 'B', 'unraveling': 'A', 'unravelling': 'B', 'unrivaled': 'A', 'unrivalled': 'B', 'unrivaledder': 'A', 'unrivalledder': 'B', 'unrivaleddest': 'A', 'unrivalleddest': 'B', 'unsavoriness': 'A', 'unsavouriness': 'B', 'unsavorinesses': 'A', 'unsavourinesses': 'B', "unsavoriness's": 'A', "unsavouriness's": 'B', 'unsavory': 'A', 'unsavoury': 'B', 'unsavorier': 'A', 'unsavourier': 'B', 'unsavories': 'A', 'unsavouries': 'B', 'unsavoriest': 'A', 'unsavouriest': 'B', "unsavory's": 'A', "unsavoury's": 'B', 'unshakable': 'A', 'unshakeable': 'B', 'untrammeled': 'A', 'untrammelled': 'B', 'untrammeledder': 'A', 'untrammelledder': 'B', 'untrammeleddest': 'A', 'untrammelleddest': 'B', 'updraft': 'A', 'updraught': 'B', 'updrafts': 'A', 'updraughts': 'B', "updraft's": 'A', "updraught's": 'B', 'uremia': 'A', 'uraemia': 'B', 'uremias': 'A', 'uraemias': 'B', "uremia's": 'A', "uraemia's": 'B', 'uremic': 'A', 'uraemic': 'B', 'valor': 'A', 'valour': 'B', 'valors': 'A', 'valours': 'B', "valor's": 'A', "valour's": 'B', 'vapor': 'A', 'vapour': 'B', 'vapored': 'A', 'vapoured': 'B', 'vaporing': 'A', 'vapouring': 'B', "vapor's": 'A', "vapour's": 'B', 'vaporings': 'A', 'vapourings': 'B', "vaporing's": 'A', "vapouring's": 'B', 'vapors': 'A', 'vapours': 'B', 'vaporware': 'A', 'vapourware': 'B', 'varicolored': 'A', 'varicoloured': 'B', 'varicoloredder': 'A', 'varicolouredder': 'B', 'varicoloreddest': 'A', 'varicoloureddest': 'B', 'varicoloreds': 'A', 'varicoloureds': 'B', "varicolored's": 'A', "varicoloured's": 'B', 'victualed': 'A', 'victualled': 'B', 'victualing': 'A', 'victualling': 'B', 'victualer': 'A', 'victualler': 'B', 'victualers': 'A', 'victuallers': 'B', "victualer's": 'A', "victualler's": 'B', 'vigor': 'A', 'vigour': 'B', 'vigors': 'A', 'vigours': 'B', "vigor's": 'A', "vigour's": 'B', 'watercolor': 'A', 'watercolour': 'B', 'watercolored': 'A', 'watercoloured': 'B', 'watercoloring': 'A', 'watercolouring': 'B', 'watercolors': 'A', 'watercolours': 'B', "watercolor's": 'A', "watercolour's": 'B', 'watercolorist': 'A', 'watercolourist': 'B', 'watercolorists': 'A', 'watercolourists': 'B', "watercolorist's": 'A', "watercolourist's": 'B', 'weaseled': 'A', 'weaselled': 'B', 'weaseling': 'A', 'weaselling': 'B', 'whir': 'A', 'whirr': 'B', 'whirs': 'A', 'whirrs': 'B', "whir's": 'A', "whirr's": 'B', 'willful': 'A', 'wilful': 'B', 'willfuler': 'A', 'wilfuler': 'B', 'willfulest': 'A', 'wilfulest': 'B', 'willfully': 'A', 'wilfully': 'B', 'willfullier': 'A', 'wilfullier': 'B', 'willfulliest': 'A', 'wilfulliest': 'B', 'willfulness': 'A', 'wilfulness': 'B', "willfulness's": 'A', "wilfulness's": 'B', 'woolen': 'A', 'woollen': 'B', 'woolenner': 'A', 'woollenner': 'B', 'woolennest': 'A', 'woollennest': 'B', 'woolens': 'A', 'woollens': 'B', "woolen's": 'A', "woollen's": 'B', 'worshiped': 'A', 'worshipped': 'B', 'worshiping': 'A', 'worshipping': 'B', 'worshiper': 'A', 'worshipper': 'B', 'worshipers': 'A', 'worshippers': 'B', "worshiper's": 'A', "worshipper's": 'B', 'yodeled': 'A', 'yodelled': 'B', 'yodeling': 'A', 'yodelling': 'B', 'yodeler': 'A', 'yodeller': 'B', 'yodelers': 'A', 'yodellers': 'B', "yodeler's": 'A', "yodeller's": 'B', 'anyplace': 'A', "ashcan's": 'A', "garbage can's": 'A', "trashcan's": 'A', "dustbin's": 'B', 'ashcan': 'A', 'garbage can': 'A', 'trashcan': 'A', 'dustbin': 'B', 'ashcans': 'A', 'garbage cans': 'A', 'trashcans': 'A', 'dustbins': 'B', 'barrister': 'B', "barrister's": 'B', 'barristers': 'B', 'automobile': 'A', "automobile's": 'A', 'automobiles': 'A', 'beeper': 'A', 'bleeper': 'B', "beeper's": 'A', "bleeper's": 'B', 'beepers': 'A', 'bleepers': 'B', 'beltway': 'A', 'ring road': 'B', "beltway's": 'A', "ring road's": 'B', 'beltways': 'A', 'ring roads': 'B', 'billfold': 'A', "billfold's": 'A', 'billfolds': 'A', 'bookstore': 'A', 'bookshop': 'B', 'candy': 'A', 'clothespin': 'A', 'clothes peg': 'B', 'diaper': 'A', 'nappy': 'B', "diaper's": 'A', "nappy's": 'B', 'diapers': 'A', 'nappies': 'B', 'dumpster': 'A', 'elevator': 'A', "elevator's": 'A', 'elevators': 'A', 'eraser': 'A', "eraser's": 'A', 'erasers': 'A', 'flashlight': 'A', "flashlight's": 'A', 'flashlights': 'A', 'motorway': 'B', 'motorways': 'B', 'garbage collector': 'A', 'dustman': 'B', "garbage collector's": 'A', "dustman's": 'B', 'garbage collectors': 'A', 'dustmans': 'B', "gasoline's": 'A', "petrol's": 'B', 'gasoline': 'A', 'petrol': 'B', 'gearshift': 'A', 'gear-lever': 'B', "gearshift's": 'A', "gear-lever's": 'B', 'gearshifts': 'A', 'gear-levers': 'B', 'mailman': 'A', 'postman': 'B', "mailman's": 'A', "postman's": 'B', 'mailmans': 'A', 'postmans': 'B', 'math': 'A', 'maths': 'B', 'movie': 'A', "movie's": 'A', 'movies': 'A', 'overpass': 'A', "overpass's": 'A', 'overpasses': 'A', 'railway': 'B', "railway's": 'B', 'railways': 'B', 'pram': 'B', 'thumbtack': 'A', 'drawing-pin': 'B', "thumbtack's": 'A', "drawing-pin's": 'B', 'thumbtacks': 'A', 'drawing-pins': 'B', "traffic circle's": 'A', "roundabout's": 'B', 'traffic circles': 'A', 'roundabouts': 'B', 'lorry': 'B', "lorry's": 'B', 'lorries': 'B', 'turnpike': 'A', 'toll motorway': 'B', "turnpike's": 'A', "toll motorway's": 'B', 'turnpikes': 'A', 'toll motorways': 'B', 'undershirt': 'A', "undershirt's": 'A', 'undershirts': 'A', 'vacation': 'A', "vacation's": 'A', 'vacations': 'A', "whiskey's": 'A', "whisky's": 'B', 'whiskey': 'A', 'whisky': 'B', 'whiskeys': 'A', 'whiskys': 'B', 'windshield': 'A', "windshield's": 'A', 'windshields': 'A', 'spanner': 'B', "spanner's": 'B', 'spanners': 'B', 'zip code': 'A', 'postcode': 'B', "zip code's": 'A', "postcode's": 'B', 'zip codes': 'A', 'postcodes': 'B', 'zipper': 'A', 'zippered': 'A', 'zippering': 'A'}
