"""Centralized path management for the application.

WHY: All path resolution goes through this module to ensure portability
and configurability across different environments.
"""
from __future__ import annotations

import os
from functools import lru_cache
from pathlib import Path


@lru_cache(maxsize=1)
def get_project_root() -> Path:
    """Return project root directory.

    Resolution order:
    1. IDEALISTA_PROJECT_ROOT environment variable
    2. Directory containing pyproject.toml (walking up from this file)
    3. Current working directory
    """
    if env_root := os.getenv("IDEALISTA_PROJECT_ROOT"):
        return Path(env_root).resolve()

    current = Path(__file__).resolve().parent
    while current != current.parent:
        if (current / "pyproject.toml").exists():
            return current
        current = current.parent

    return Path.cwd()


def get_output_dir() -> Path:
    """Return output directory, creating if needed."""
    output_dir = Path(os.getenv("IDEALISTA_OUTPUT_DIR", get_project_root() / "output"))
    output_dir.mkdir(parents=True, exist_ok=True)
    return output_dir


def get_cache_dir() -> Path:
    """Return cache directory, creating if needed."""
    cache_dir = Path(os.getenv("IDEALISTA_CACHE_DIR", get_output_dir() / "cache"))
    cache_dir.mkdir(parents=True, exist_ok=True)
    return cache_dir


def get_config_dir() -> Path:
    """Return configuration directory."""
    return Path(os.getenv("IDEALISTA_CONFIG_DIR", get_project_root() / "configs"))


def get_env_file() -> Path | None:
    """Return path to .env file if it exists.

    Resolution order: .env.local, .env
    """
    root = get_project_root()
    for name in (".env.local", ".env"):
        env_path = root / name
        if env_path.exists():
            return env_path
    return None


def ensure_output_structure() -> dict[str, Path]:
    """Ensure all required output directories exist.

    Returns a dictionary of directory names to their paths.
    """
    output_dir = get_output_dir()
    cache_dir = get_cache_dir()

    directories = {
        "output": output_dir,
        "cache": cache_dir,
        "sessions": cache_dir / "sessions",
        "agents": cache_dir / "agents",
        "listings": cache_dir / "listings",
        "temp": output_dir / "temp",
    }

    for directory in directories.values():
        directory.mkdir(parents=True, exist_ok=True)

    return directories
