"""Country configuration management for multi-country Idealista scraping."""
from __future__ import annotations

import json
import os
from dataclasses import dataclass, field
from functools import lru_cache
from typing import TYPE_CHECKING, Optional

if TYPE_CHECKING:
    from pathlib import Path

from .paths import get_config_dir


@dataclass
class RegionInfo:
    """Region information for a country."""
    slug: str
    name: str
    code: str


@dataclass(frozen=False)
class CountryConfig:
    """Configuration for an Idealista country domain."""

    code: str
    domain: str
    language: str
    currency: str
    property_types: dict[str, str] = field(default_factory=dict)
    regions_endpoint: Optional[str] = None
    agents_endpoint: Optional[str] = None
    agents_url_pattern: Optional[str] = None
    regions: list[dict] = field(default_factory=list)

    @property
    def base_url(self) -> str:
        """Full base URL with language path."""
        return f"{self.domain}/{self.language}"

    @property
    def has_static_regions(self) -> bool:
        """Check if country has static region list (no API discovery)."""
        return len(self.regions) > 0

    def get_regions(self) -> list[RegionInfo]:
        """Get list of regions for this country."""
        return [RegionInfo(**r) for r in self.regions]

    def get_property_type_path(self, prop_type: str) -> str:
        """Get URL path segment for property type (rental/sale)."""
        return self.property_types.get(prop_type, prop_type)

    def build_listing_url(self, region: str, prop_type: str = "rental") -> str:
        """Build full listing URL for a region and property type."""
        type_path = self.get_property_type_path(prop_type)
        return f"{self.base_url}/{type_path}/{region}/"

    def build_agents_url(self, region_slug: str) -> Optional[str]:
        """Build URL for agency listing page for a region."""
        if self.agents_url_pattern:
            return f"{self.domain}{self.agents_url_pattern.format(region=region_slug)}"
        return None

    def build_ajax_url(self, path: str) -> str:
        """Build AJAX endpoint URL."""
        return f"{self.domain}{path}"


@lru_cache(maxsize=1)
def _load_countries_data() -> dict:
    """Load raw countries data from JSON config file."""
    config_path = get_config_dir() / "countries.json"
    if not config_path.exists():
        raise FileNotFoundError(f"Countries config not found: {config_path}")
    with open(config_path, encoding="utf-8") as f:
        return json.load(f)


def load_countries() -> dict[str, CountryConfig]:
    """Load all country configurations."""
    data = _load_countries_data()
    return {name: CountryConfig(**config) for name, config in data.items()}


def get_country(name: str | None = None) -> CountryConfig:
    """Get country configuration by name.

    Args:
        name: Country name (spain, portugal). If None, uses IDEALISTA_DEFAULT_COUNTRY
              env var or defaults to 'spain'.

    Returns:
        CountryConfig for the specified country.

    Raises:
        ValueError: If country name is not found.
    """
    if name is None:
        name = os.getenv("IDEALISTA_DEFAULT_COUNTRY", "spain")

    countries = load_countries()
    name_lower = name.lower()

    if name_lower not in countries:
        available = ", ".join(countries.keys())
        raise ValueError(f"Unknown country: {name}. Available: {available}")

    return countries[name_lower]


def list_available_countries() -> list[str]:
    """Get list of available country names."""
    return list(load_countries().keys())


def clear_cache() -> None:
    """Clear the cached countries data. Useful for testing."""
    _load_countries_data.cache_clear()
