"""
Configuration utilities with caching and centralized settings.

Provides ScraperSettings singleton for environment-based configuration.
Uses centralized path management for portable env file resolution.
"""
from __future__ import annotations

import os
from dataclasses import dataclass
from functools import lru_cache
from pathlib import Path
from typing import Any, Optional

from dotenv import load_dotenv

from .paths import get_env_file, get_config_dir

# Load environment variables using centralized path resolution
if _env_file := get_env_file():
    load_dotenv(_env_file)


def _parse_bool(value: str, default: bool = False) -> bool:
    """Parse string to boolean, handling various truthy/falsy values."""
    if value is None:
        return default
    return value.lower() in ("true", "1", "yes", "on")


def _parse_int(value: str, default: int) -> int:
    """Parse string to int with default fallback."""
    if value is None:
        return default
    try:
        return int(value)
    except (ValueError, TypeError):
        return default


def _parse_float(value: str, default: float) -> float:
    """Parse string to float with default fallback."""
    if value is None:
        return default
    try:
        return float(value)
    except (ValueError, TypeError):
        return default


@dataclass(frozen=True)
class ScraperSettings:
    """
    Centralized configuration for the Idealista scraper.

    All values are loaded from environment variables with sensible defaults.
    This class is immutable (frozen=True) to prevent accidental modification.

    ASP disabled by default based on test results showing it's not required.
    """

    # ASP (Anti-Scraping Protection) settings
    asp_enabled: bool
    asp_escalate_on_block: bool

    # Concurrency settings
    max_concurrency: int
    max_workers: int

    # Cache settings
    cache_enabled: bool
    cache_ttl: int

    # Batch processing
    batch_size: int
    s3_batch_size: int
    s3_connection_limit: int

    # Timeouts (seconds)
    client_total_timeout: int
    client_connect_timeout: int

    # Optional rate limiting delays (seconds)
    request_delay: float
    error_delay: float


@lru_cache(maxsize=1)
def get_settings() -> ScraperSettings:
    """
    Get the singleton ScraperSettings instance.

    Values are loaded from environment variables with defaults optimized
    for cost savings (ASP disabled, reasonable concurrency limits).

    Returns:
        ScraperSettings instance with all configuration values.
    """
    return ScraperSettings(
        # ASP disabled by default - tests show it's not required for Idealista
        asp_enabled=_parse_bool(os.getenv("SCRAPFLY_ASP_ENABLED"), default=False),
        asp_escalate_on_block=_parse_bool(
            os.getenv("SCRAPFLY_ASP_ESCALATE_ON_BLOCK"), default=True
        ),
        # Concurrency
        max_concurrency=_parse_int(os.getenv("SCRAPFLY_MAX_CONCURRENCY"), default=50),
        max_workers=_parse_int(os.getenv("SCRAPFLY_MAX_WORKERS"), default=15),
        # Caching
        cache_enabled=_parse_bool(os.getenv("SCRAPFLY_CACHE_ENABLED"), default=True),
        cache_ttl=_parse_int(os.getenv("SCRAPFLY_CACHE_TTL"), default=3600),
        # Batch processing
        batch_size=_parse_int(os.getenv("BATCH_SIZE"), default=100),
        s3_batch_size=_parse_int(os.getenv("S3_BATCH_SIZE"), default=100),
        s3_connection_limit=_parse_int(os.getenv("S3_CONNECTION_LIMIT"), default=1000),
        # Timeouts
        client_total_timeout=_parse_int(
            os.getenv("CLIENT_TOTAL_TIMEOUT"), default=3600
        ),
        client_connect_timeout=_parse_int(
            os.getenv("CLIENT_CONNECT_TIMEOUT"), default=60
        ),
        # Rate limiting - disabled by default since Scrapfly handles this
        request_delay=_parse_float(os.getenv("REQUEST_DELAY_SECONDS"), default=0.0),
        error_delay=_parse_float(os.getenv("DELAY_AFTER_ERROR_SECONDS"), default=2.0),
    )


def reload_settings() -> ScraperSettings:
    """
    Force reload of settings from environment.

    Clears the cache and returns a fresh ScraperSettings instance.
    Useful after modifying environment variables at runtime.
    """
    get_settings.cache_clear()
    return get_settings()


# Module-level singleton for convenient access
settings = get_settings()

# Re-export from cache module for convenience
try:
    from ..cache.config_loader import (
        get_regex_patterns,
        get_settings_config,
        get_data_structure_config,
        get_views_config,
        load_config,
    )
except ImportError:
    # Fallback if cache module not available
    import json

    _cache = {}

    def load_config(name: str) -> dict:
        if name not in _cache:
            path = get_config_dir() / name
            if path.exists():
                with open(path, "r") as f:
                    _cache[name] = json.load(f)
            else:
                _cache[name] = {}
        return _cache[name]

    def get_regex_patterns() -> dict:
        return load_config("regex_patterns.json")

    def get_settings_config() -> dict:
        return load_config("settings_config.json")

    def get_data_structure_config() -> dict:
        return load_config("data_structure_config.json")

    def get_views_config() -> dict:
        return load_config("views_config.json")


def get_env(key: str, default: Optional[str] = None) -> Optional[str]:
    """Get environment variable with optional default."""
    return os.getenv(key, default)


def require_env(key: str) -> str:
    """Get required environment variable, raise if not set."""
    value = os.getenv(key)
    if value is None:
        raise ValueError(f"Required environment variable not set: {key}")
    return value


def get_scrapfly_key() -> str:
    """Get Scrapfly API key from environment."""
    return require_env("SCRAPFLY_KEY")


def get_aws_credentials() -> dict:
    """Get AWS credentials from environment."""
    return {
        "access_key_id": get_env("AWS_ACCESS_KEY_ID"),
        "secret_access_key": get_env("AWS_SECRET_ACCESS_KEY"),
        "region": get_env("AWS_REGION", "eu-west-1"),
    }
