"""
Property data transformation module.

This module provides functionality to transform cleaned HTML content
into MongoDB-compatible format for property listings.
"""

import uuid
from datetime import datetime, timezone
import logging
import json
from pathlib import Path
from typing import Dict, Optional, Any

from ..utils.geo import parse_location_html, get_location_details

# Set up logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

# Constants for config files
CONFIG_DIR = Path(__file__).parent.parent.parent / 'configs'


def get_utc_now() -> str:
    """Get current UTC timestamp in ISO format."""
    return datetime.now(timezone.utc).isoformat()


def load_agent_info(agents_file: str, agency_ref: str) -> Optional[Dict]:
    """Load agent information from property_agents.jsonl file."""
    try:
        # Look for agents file relative to project root
        project_root = Path(__file__).parent.parent.parent
        agents_path = project_root / agents_file
        with open(agents_path, 'r', encoding='utf-8') as f:
            for line in f:
                if line.strip():
                    agent = json.loads(line)
                    if agent.get('firstName') == agency_ref:
                        return agent
        return None
    except Exception as e:
        logger.error(f"Error loading agent info: {str(e)}")
        return None


def load_config(filename: str) -> Dict:
    """Load configuration from JSON file."""
    try:
        with open(CONFIG_DIR / filename, 'r', encoding='utf-8') as f:
            return json.load(f)
    except Exception as e:
        logger.error(f"Error loading config {filename}: {str(e)}")
        return {}


def transform_to_mongodb_format(
    content: Dict[str, Any],
    agent_info: Optional[Dict],
    advertiser_name: Optional[str],
    development_name: Optional[str]
) -> Optional[Dict[str, Any]]:
    """
    Transform cleaned content to MongoDB format.

    Args:
        content: Cleaned HTML content dictionary
        agent_info: Agent information dictionary
        advertiser_name: Name of the advertiser
        development_name: Name of the development

    Returns:
        Transformed data dictionary ready for MongoDB insertion
    """
    try:
        # Load configurations
        views_config = load_config('views_config.json')
        settings_config = load_config('settings_config.json')
        data_structure = load_config('data_structure_config.json')

        # Get base structure
        transformed_data = data_structure.get('base_structure', {}).copy()

        # Generate IDs
        main_id = str(uuid.uuid4())
        owner_details_id = str(uuid.uuid4())
        pricing_id = str(uuid.uuid4())

        # Get location details
        location_data = parse_location_html(content.get('html_content', ''))
        country, province, municipality, urbanisation, address = get_location_details(location_data)

        # Extract utag_data with safe defaults
        utag_data = content.get('utag_data', {})
        ad_data = utag_data.get('ad', {}) if utag_data else {}
        characteristics = ad_data.get('characteristics', {}) if ad_data else {}
        condition_data = ad_data.get('condition', {}) if ad_data else {}

        # Map fields with safe defaults
        total_units = int(ad_data.get('builtType', 0)) if ad_data else 0
        price = float(ad_data.get('price', 0)) if ad_data else 0
        room_number = int(characteristics.get('roomNumber', 0)) if characteristics else 0
        bath_number = int(characteristics.get('bathNumber', 0)) if characteristics else 0
        has_parking = int(characteristics.get('hasParking', 0)) if characteristics else 0
        has_garden = int(characteristics.get('hasGarden', 0)) if characteristics else 0
        has_pool = int(characteristics.get('hasSwimmingPool', 0)) if characteristics else 0
        has_terrace = int(characteristics.get('hasTerrace', 0)) if characteristics else 0
        constructed_area = float(characteristics.get('constructedArea', 0)) if characteristics else 0

        # Map condition
        property_condition = []
        if condition_data:
            if condition_data.get('isNewDevelopment') == '1':
                property_condition.append("New Construction")
            if condition_data.get('isNeedsRenovating') == '1':
                property_condition.extend(["Renovation Required", "Restoration Required"])
            if condition_data.get('isGoodCondition') == '1':
                property_condition.extend(["Excellent", "Good"])

        # Handle duplicate image tags
        image_tags_dict = {}
        for tag in content.get('image_tags', []):
            cleaned_tag = tag.lower().strip()
            image_tags_dict[cleaned_tag] = True
        unique_image_tags = list(image_tags_dict.keys())

        # Create combined text for feature detection
        description = content.get('description', '').lower()
        features_text = ' '.join(content.get('basic_features', [])).lower()
        combined_text = f"{description} {features_text}"

        # Views Detection using config
        views = []
        for view in views_config.get('view_options', []):
            if view.lower() in combined_text:
                views.append(view)
                logger.info(f"Found view: {view}")

        # Setting Detection using config
        setting = []
        setting_keywords = settings_config.get('setting_keywords', {})
        for setting_type, keywords in setting_keywords.items():
            if any(keyword in combined_text for keyword in keywords):
                setting.append(setting_type)
                logger.info(f"Found setting: {setting_type}")

        # Determine construction status
        construction_status = 'completed'
        if condition_data.get('isNewDevelopment') == '1':
            construction_status = 'under-construction'

        # Generate system reference
        system_reference = f"P{content.get('propertyId')}" if content.get('propertyId') else None

        # Format image paths with system reference
        formatted_images = [f"{system_reference}/{img}" for img in content.get('images', [])]

        # Update the base structure with processed data
        transformed_data.update({
            "_id": {"$oid": main_id},
            "systemReference": system_reference,
            "owner": {"$oid": agent_info.get('owner', {}).get('$oid')} if agent_info else None,
            "superAgent": {"$oid": agent_info.get('superAgent', {}).get('$oid')} if agent_info else None,
            "country": country or "Spain",
            "province": province,
            "municipality": municipality,
            "urbanisation": urbanisation,
            "address": address,
            "developmentName": development_name,
            "completedYear": content.get('completedYear'),
            "description": content.get('description'),
            "orientation": content.get('orientation', []),
            "additionalFeatures": content.get('basic_features', []),
            "images": formatted_images,
            "imageTags": unique_image_tags,
            "ownerDetails": {
                "_id": {"$oid": owner_details_id},
                "name": advertiser_name,
                "email": agent_info.get('email') if agent_info else None,
                "phone": agent_info.get('phone') if agent_info else None,
                "createdAt": {"$date": get_utc_now()},
                "updatedAt": {"$date": get_utc_now()}
            },
            "bedsIncluded": [str(room_number)] if room_number > 0 else [],
            "constructionStatus": construction_status,
            "units": [{
                "beds": room_number,
                "baths": bath_number,
                "builtArea": constructed_area,
                "utilArea": content.get('utilArea', 0),
                "unitPlotSize": 0,
                "parking": has_parking,
                "garden": has_garden,
                "pool": has_pool,
                "terrace": has_terrace,
                "floor": content.get('floor'),
                "pricing": {
                    "sales": {
                        "_id": {"$oid": pricing_id},
                        "price": price
                    }
                }
            }],
            "propertyUrl": content.get('propertyUrl'),
            "totalUnits": total_units,
            "condition": property_condition,
            "developer": utag_data.get('agency', {}).get('name') if utag_data else None,
            "developmentWebsite": content.get('development_website'),
            "viewingContactDetails": {
                "name": advertiser_name,
                "email": agent_info.get('email') if agent_info else None,
                "phone": agent_info.get('phone') if agent_info else None,
                "salesOfficeAddress": province
            },
            "startingPrice": price,
            "rentedPrice": price,
            "climateControl": content.get('climateControl', []),
            "views": views,
            "setting": setting
        })

        return transformed_data

    except Exception as e:
        logger.error(f"Error transforming to MongoDB format: {str(e)}")
        return None
