"""
Session Manager module for Scrapfly client sessions.

This module manages sessions for Scrapfly clients, providing session
persistence, reuse, and tracking for efficient API usage.
"""

from dataclasses import dataclass
from datetime import datetime
import uuid
import json
import asyncio
from pathlib import Path
import aiofiles
from typing import Dict, Optional, Set
from loguru import logger

from ..utils.paths import get_cache_dir


@dataclass
class SessionInfo:
    """Track information about a session."""
    session_id: str
    worker_key: str
    client_key: str
    in_use: bool = False
    created_at: datetime = None
    last_used: datetime = None
    property_type: str = None  # 'rental' or 'resale'


class SessionManager:
    """Manages sessions for Scrapfly clients."""

    def __init__(self, cache_dir: Path | str | None = None):
        self.cache_dir = Path(cache_dir) if cache_dir else get_cache_dir() / "sessions"
        self.cache_dir.mkdir(parents=True, exist_ok=True)
        self.sessions: Dict[str, SessionInfo] = {}
        self.lock = asyncio.Lock()
        self.sessions_file = self.cache_dir / "sessions.jsonl"
        self.load_cached_sessions()

    def load_cached_sessions(self):
        """Load sessions from JSONL file."""
        try:
            if self.sessions_file.exists():
                with open(self.sessions_file, 'r') as f:
                    for line in f:
                        if line.strip():
                            session_data = json.loads(line)
                            session = SessionInfo(
                                session_id=session_data['session_id'],
                                worker_key=session_data['worker_key'],
                                client_key=session_data['client_key'],
                                in_use=False,  # Reset to false on load
                                created_at=datetime.fromisoformat(session_data['created_at']),
                                last_used=datetime.fromisoformat(session_data['last_used']),
                                property_type=session_data['property_type']
                            )
                            self.sessions[session.session_id] = session
                logger.info(f"Loaded {len(self.sessions)} cached sessions")
        except Exception as e:
            logger.error(f"Error loading cached sessions: {e}")

    async def save_session(self, session: SessionInfo):
        """Save single session to JSONL file."""
        try:
            session_data = {
                'session_id': session.session_id,
                'worker_key': session.worker_key,
                'client_key': session.client_key,
                'in_use': session.in_use,
                'created_at': session.created_at.isoformat(),
                'last_used': session.last_used.isoformat(),
                'property_type': session.property_type
            }

            async with aiofiles.open(self.sessions_file, 'a') as f:
                await f.write(json.dumps(session_data) + '\n')
            logger.debug(f"Saved session {session.session_id}")

        except Exception as e:
            logger.error(f"Error saving session: {e}")

    async def get_session(
        self,
        worker_key: str,
        client_key: str,
        property_type: str
    ) -> str:
        """Get an available session or create a new one."""
        logger.info(f"Getting session for worker {worker_key} - {property_type}")
        try:
            async with self.lock:
                # First try to find an existing free session
                for session in self.sessions.values():
                    if (
                        session.worker_key == worker_key
                        and not session.in_use
                        and session.property_type == property_type
                    ):
                        session.in_use = True
                        session.last_used = datetime.now()
                        await self.save_session(session)
                        logger.info(f"Reusing session {session.session_id}")
                        return session.session_id

                # Create new session if none available
                session_id = str(uuid.uuid4())
                new_session = SessionInfo(
                    session_id=session_id,
                    worker_key=worker_key,
                    client_key=client_key,
                    in_use=True,
                    created_at=datetime.now(),
                    last_used=datetime.now(),
                    property_type=property_type
                )
                self.sessions[session_id] = new_session
                await self.save_session(new_session)
                logger.info(f"Created new session {session_id}")
                return session_id

        except Exception as e:
            logger.error(f"Error getting session: {e}")
            raise

    async def release_session(self, session_id: str):
        """Mark a session as no longer in use."""
        logger.info(f"Releasing session {session_id}")
        async with self.lock:
            if session_id in self.sessions:
                self.sessions[session_id].in_use = False
                self.sessions[session_id].last_used = datetime.now()
                await self.save_session(self.sessions[session_id])
                logger.info(f"Session {session_id} released")
            else:
                logger.warning(f"Attempted to release unknown session {session_id}")

    def get_worker_sessions(self, worker_key: str) -> Set[str]:
        """Get all sessions associated with a worker."""
        return {
            s.session_id for s in self.sessions.values()
            if s.worker_key == worker_key
        }
